/* $Id: eris_ifu_utils.c,v 1.12 2013-03-25 11:46:49 cgarcia Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2013-03-25 11:46:49 $
 * $Revision: 1.12 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <string.h>
#include <gsl/gsl_interp.h>
#include "eris_ifu_dfs.h"
#include "eris_ifu_debug.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_functions.h"
#include "eris_ifu_distortion_static.h"
#include "eris_ifu_flat_static.h"
#include "eris_ifu_error.h"
#include "eris_utils.h"

/**
 * @defgroup eris_ifu_functions IFU Helper Functions
 *
 * This module provides a comprehensive set of utility functions for IFU data processing,
 * including:
 * - Instrument identification and configuration
 * - Raw exposure loading with noise calculation
 * - Image corrections (line, column, cosmic ray, saturation detection)
 * - Bad pixel mask creation and correction (2D and 3D methods)
 * - Polynomial fitting and interpolation (1D)
 * - Gaussian fitting for line and slitlet detection
 * - Image warping with polynomial transformations
 * - Parameter list management for recipes
 * - Mathematical utilities (mean calculation, NaN/Inf filtering)
 *
 * These functions support both ERIS and SINFONI instruments and are used throughout
 * the pipeline for data reduction tasks.
 *
 * @{
 */

/*----------------------------------------------------------------------------*/
/**
  @brief    Get instrument identifier from a CPL frame
  @param    frame    Input CPL frame
  @return   Instrument identifier (SPIFFI, SPIFFIER, OTHER_INSTRUMENT, or UNSET_INSTRUMENT on error)
  @note     Reads the INSTRUME keyword from the FITS header (extension 0)
  @note     Returns UNSET_INSTRUMENT if frame is NULL or on error
  @see      eris_ifu_get_instrument
 */
/*----------------------------------------------------------------------------*/
ifsInstrument eris_ifu_get_instrument_frame(cpl_frame *frame)
{
    ifsInstrument       instrument = UNSET_INSTRUMENT;
	const char          *filename   = NULL;
	cpl_propertylist    *header     = NULL;

	TRY
	{
        if (frame == NULL) {
            BRK_WITH_ERROR(CPL_ERROR_NULL_INPUT);
        }

        filename = cpl_frame_get_filename(frame);
		header = cpl_propertylist_load(filename, 0);
		instrument = eris_ifu_get_instrument(header);
		CHECK_ERROR_STATE();
	}
	CATCH
	{
		instrument = UNSET_INSTRUMENT;
	}

	eris_ifu_free_propertylist(&header);
	eris_check_error_code("eris_ifu_get_instrument_frame");
	return instrument;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create an HDRL image from a CPL image with calculated noise
  @param    cplImage    Input CPL image (data only)
  @return   HDRL image containing data and noise, or NULL on error
  @note     Noise is calculated using detector gain (2.10) and readout noise (0.0)
  @note     The returned HDRL image must be freed with hdrl_image_delete()
  @note     Formula: sigma = sqrt(counts * gain + readnoise^2) / gain
  @see      eris_ifu_calc_noise_map
 */
/*----------------------------------------------------------------------------*/
hdrl_image * eris_ifu_raw_hdrl_image(const cpl_image *cplImage)
{
	hdrl_image *image = NULL;
	cpl_image *noiseImage = NULL;

	cpl_ensure(cplImage, CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{
		noiseImage = eris_ifu_calc_noise_map(cplImage, 2.10, 0.);
		image = hdrl_image_create (cplImage, noiseImage);
	}
	CATCH
	{
	}

	cpl_image_delete(noiseImage);
	eris_check_error_code("eris_ifu_raw_hdrl_image");
	return image;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Calculate a noise map for detector data
  @param    data        Input image data
  @param    gain        Detector gain [e-/ADU] (must be > 0)
  @param    readnoise   Read-out noise [e-] (must be >= 0)
  @return   Noise image, or NULL on error
  @note     Formula: sigma = sqrt(counts * gain + readnoise^2) / gain
  @note     Assumes Poisson noise for photon counts plus Gaussian readout noise
  @note     The returned image must be freed with cpl_image_delete()
 */
/*----------------------------------------------------------------------------*/
cpl_image* eris_ifu_calc_noise_map(const cpl_image *data,
		double gain,
		double readnoise)
{
	cpl_image *noiseImg = NULL;

	cpl_ensure(data,                CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(gain > 0.0,          CPL_ERROR_ILLEGAL_INPUT, NULL);
	cpl_ensure(readnoise >= 0.0,    CPL_ERROR_ILLEGAL_INPUT, NULL);

	TRY
	{
		BRK_IF_NULL(
				noiseImg = cpl_image_duplicate(data));

		/* calculate initial noise estimate
            sigma = sqrt(counts * gain + readnoise^2) / gain */
		cpl_image_abs(noiseImg);
		cpl_image_multiply_scalar(noiseImg, gain);
		cpl_image_add_scalar(noiseImg, readnoise * readnoise);
		cpl_image_power(noiseImg, 0.5);
		cpl_image_divide_scalar(noiseImg, gain);
	}
	CATCH
	{
		CATCH_MSG();
		eris_ifu_free_image(&noiseImg);
	}
	eris_check_error_code("eris_ifu_calc_noise_map");
	return noiseImg;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Load raw exposures from a frameset with optional corrections
  @param    frameset                 CPL frameset containing detector FITS files
  @param    exposureCorrectionMode   Bit flags for corrections to apply
  @return   HDRL imagelist containing data and noise images, or NULL on error
  @note     Applies corrections based on exposureCorrectionMode flags (bitwise OR):
            - LINE_EXPOSURE_CORRECTION: Line/row correction
            - COLUMN_EXPOSURE_CORRECTION: Column correction
            - COSMIC_RAY_EXPOSURE_CORRECTION: Cosmic ray correction (not implemented)
            - COSMIC_RAY_EXPOSURE_DETECTION: Cosmic ray detection (not implemented)
  @note     The returned imagelist must be freed with hdrl_imagelist_delete()
  @see      eris_ifu_load_exposure_frame, eris_ifu_load_exposure_file
 */
/*----------------------------------------------------------------------------*/
hdrl_imagelist* eris_ifu_load_exposure_frameset(const cpl_frameset *frameset,
		int exposureCorrectionMode) {
	hdrl_imagelist *imageList = NULL;
	hdrl_image *tmpImage = NULL;
	const cpl_frame *frame = NULL;
	cpl_size frameCnt;
	cpl_ensure(frameset, CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{
		frameCnt = cpl_frameset_get_size(frameset);
		imageList = hdrl_imagelist_new();
		for (cpl_size ix=0; ix<frameCnt; ix++) {
			frame = cpl_frameset_get_position_const(frameset, ix);
			tmpImage = eris_ifu_load_exposure_frame(
					frame, exposureCorrectionMode, NULL);
			hdrl_imagelist_set(imageList, tmpImage,
					hdrl_imagelist_get_size(imageList));
		}
	}
	CATCH
	{
		imageList = NULL;
	}
	eris_check_error_code("eris_ifu_load_exposure_frameset");
	return imageList;

}

/*----------------------------------------------------------------------------*/
/**
  @brief    Load a raw detector exposure from a frame with optional corrections
  @param    frame                    CPL frame holding the detector FITS file info
  @param    exposureCorrectionMode   Bit flags for corrections to apply
  @param    dqi                      (Optional) Data quality image to populate with DQI flags
  @return   HDRL image containing data and noise, or NULL on error
  @note     Wrapper function that extracts filename from frame and calls eris_ifu_load_exposure_file()
  @note     The returned HDRL image must be freed with hdrl_image_delete()
  @see      eris_ifu_load_exposure_file, eris_ifu_load_exposure_frameset
 */
/*----------------------------------------------------------------------------*/
hdrl_image* eris_ifu_load_exposure_frame(const cpl_frame *frame,
		int exposureCorrectionMode,
		cpl_image *dqi) {

	hdrl_image *image = NULL;
	const char *filename;

	cpl_ensure(frame, CPL_ERROR_NULL_INPUT, NULL);
	TRY
	{
		filename = cpl_frame_get_filename(frame);
		image = eris_ifu_load_exposure_file(
				filename, exposureCorrectionMode, dqi);
	}
	CATCH
	{
		image = NULL;
	}
	eris_check_error_code("eris_ifu_load_exposure_frame");
	return image;

}

/*----------------------------------------------------------------------------*/
//TODO: why should one set flagged pixels to 1 in the image? Isn't sufficient
//to flag them? That is also a different convention from the one of using NaN
/**
  @brief    Flag detector border pixels as bad in image and optionally set to 1
  @param    data        Input/output image
  @param    add_ones    If TRUE, set flagged pixels to 1.0
  @param    dqi         (Optional) Data quality image to update with ERIS_DQI_BP flag
  @return   CPL_ERROR_NONE on success, otherwise an error code
  @note     Flags ERIS_IFU_DETECTOR_BP_BORDER pixels on all four edges
  @note     Border width is defined by ERIS_IFU_DETECTOR_BP_BORDER (typically 4 pixels)
  @note     TODO: Setting flagged pixels to 1 is inconsistent with NaN convention elsewhere
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_add_badpix_border(cpl_image *data, cpl_boolean add_ones,
		cpl_image *dqi)
{
	cpl_error_code  err = CPL_ERROR_NONE;
	cpl_size        x   = 0;
	cpl_size        y   = 0;

	cpl_ensure_code(data, CPL_ERROR_NULL_INPUT);

	TRY
	{
		for (x = 1; x <= ERIS_IFU_DETECTOR_SIZE_X; x++) {
			/* reject lower border*/
			for (y = 1;
					y <= ERIS_IFU_DETECTOR_BP_BORDER;
					y++)
			{
				cpl_image_reject(data, x, y);
				if (dqi != NULL) {
					cpl_image_set(dqi, x, y, ERIS_DQI_BP);
				}
				if (add_ones) {
					cpl_image_set(data, x, y, 1.);
				}
			}

			/* reject upper border*/
			for (y = ERIS_IFU_DETECTOR_SIZE_Y-ERIS_IFU_DETECTOR_BP_BORDER+1;
					y <= ERIS_IFU_DETECTOR_SIZE_Y;
					y++)
			{
				cpl_image_reject(data, x, y);
				if (dqi != NULL) {
					cpl_image_set(dqi, x, y, ERIS_DQI_BP);
				}
				if (add_ones) {
					cpl_image_set(data, x, y, 1.);
				}
			}
		}
		CHECK_ERROR_STATE();

		for (y = ERIS_IFU_DETECTOR_BP_BORDER+1; y <= ERIS_IFU_DETECTOR_SIZE_Y-ERIS_IFU_DETECTOR_BP_BORDER; y++) {
			/* reject remaining left border*/
			for (x = 1; x <= ERIS_IFU_DETECTOR_BP_BORDER; x++) {
				cpl_image_reject(data, x, y);
				if (dqi != NULL) {
					cpl_image_set(dqi, x, y, ERIS_DQI_BP);
				}
				if (add_ones) {
					cpl_image_set(data, x, y, 1.);
				}
			}

			/* reject remaining right border*/
			for (x = ERIS_IFU_DETECTOR_SIZE_X-ERIS_IFU_DETECTOR_BP_BORDER+1;
					x <= ERIS_IFU_DETECTOR_SIZE_X;
					x++)
			{
				cpl_image_reject(data, x, y);
				if (dqi != NULL) {
					cpl_image_set(dqi, x, y, ERIS_DQI_BP);
				}
				if (add_ones) {
					cpl_image_set(data, x, y, 1.);
				}
			}
		}
		CHECK_ERROR_STATE();
	}
	CATCH
	{
		err = cpl_error_get_code();
	}
	eris_check_error_code("eris_ifu_add_badpix_border");
	return err;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Load a raw detector exposure from file with corrections and noise
  @param    filename                 Path to detector exposure FITS file
  @param    exposureCorrectionMode   Bit flags specifying corrections to apply
  @param    dqi                      (Optional) Data quality image to populate
  @return   HDRL image containing data and noise, or NULL on error
  
  @note     The exposureCorrectionMode parameter is bitwise OR of:
            - LINE_EXPOSURE_CORRECTION: Correct bad lines (reference pixel median)
            - COLUMN_EXPOSURE_CORRECTION: Correct bad columns
            - COSMIC_RAY_EXPOSURE_CORRECTION: Cosmic ray correction (not implemented)
            - COSMIC_RAY_EXPOSURE_DETECTION: Cosmic ray detection (not implemented)
  
  @note     Automatic processing includes:
            - Border pixel flagging (ERIS_DQI_BP = 16)
            - Saturation detection (ERIS_DQI_SAT = 8)
            - Noise calculation from gain and readout noise
  
  @note     Gain and readout noise are read from FITS header keywords:
            - SPIFFIER (ERIS): Uses FHDR_DET_CHIP_GAIN and FHDR_DET_CHIP_RON
            - SPIFFI (SINFONI): Uses defaults (gain=2.1, RON=0.0)
  
  @note     The returned HDRL image must be freed with hdrl_image_delete()
  @note     DQI bit flags: bit 3 (value 8) = saturated, bit 4 (value 16) = border
  @note     TODO: Saturation value -45727232.0 is hard-coded, should use #define
  @see      eris_ifu_exposure_line_correction, eris_ifu_exposure_column_correction
 */
/*----------------------------------------------------------------------------*/
hdrl_image* eris_ifu_load_exposure_file(const char *filename,
		int exposureCorrectionMode,
		cpl_image *dqi)
{
	hdrl_image *image = NULL;
	cpl_image *dataImage = NULL;
	cpl_image *noiseImage = NULL;
	cpl_propertylist *header = NULL;
	ifsInstrument instrument = UNSET_INSTRUMENT;
	double gain;
	double ron;

	cpl_ensure(filename, CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{
		dataImage = cpl_image_load(filename, CPL_TYPE_UNSPECIFIED, 0, 0);
		eris_ifu_add_badpix_border(dataImage, CPL_FALSE, dqi);
		eris_ifu_saturation_detection(dataImage, dqi);

		// ==========
				// correct data image
				// ==========
		if ((exposureCorrectionMode & LINE_EXPOSURE_CORRECTION) != 0) {
			eris_ifu_exposure_line_correction(dataImage);
			//            cpl_image_save(dataImage,"line_corr_img.fits",
			//                    CPL_TYPE_FLOAT,NULL,CPL_IO_CREATE);
		}
		if ((exposureCorrectionMode & COLUMN_EXPOSURE_CORRECTION) != 0) {
			eris_ifu_exposure_column_correction(dataImage);
			//            cpl_image_save(dataImage,"col_corr_img.fits",
			//                    CPL_TYPE_FLOAT,NULL,CPL_IO_CREATE);
		}

		// ==========
		// fill noise image
		// ==========
		// get detector gain and readout noise parameters from FITS header
		header = cpl_propertylist_load(filename, 0);
		instrument = eris_ifu_get_instrument(header);
		CHECK_ERROR_STATE();
		if (instrument == SPIFFI) {
			//FITS header keywords are missing for SPIFFI, set defaults
			gain = 2.1;
			ron = 0.0;
		} else if (instrument == SPIFFIER) {
			gain = cpl_propertylist_get_double(header, FHDR_DET_CHIP_GAIN);
			ron = cpl_propertylist_get_double(header, FHDR_DET_CHIP_RON);
			CHECK_ERROR_STATE();
		} else {
			gain = 0.;
			ron = 0.;
		}
		noiseImage = eris_ifu_calc_noise_map(dataImage, gain, ron);
		image = hdrl_image_create(dataImage, noiseImage);

	}
	CATCH
	{
		CATCH_MSGS();
		eris_ifu_free_hdrl_image(&image);
	}

	eris_ifu_free_propertylist(&header);
	eris_ifu_free_image(&dataImage);
	eris_ifu_free_image(&noiseImage);
	eris_check_error_code("eris_ifu_load_exposure_file");
	return image;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Detect cosmic ray hits using LA-cosmic algorithm
  @param    image                    Input HDRL image
  @param    exposureCorrectionMode   Bit flags controlling detection
  @param    laCosmicParams           HDRL LA-cosmic parameters
  @param    maskImage                If true, apply mask to input image
  @return   Mask with detected cosmic ray pixels flagged, or NULL on error
  @note     Only performs detection if COSMIC_RAY_EXPOSURE_DETECTION flag is set
  @note     Uses HDRL lacosmic edge detection algorithm
  @note     Prints number of detected cosmic ray hits to log
  @note     The returned mask must be freed with cpl_mask_delete()
  @see      hdrl_lacosmic_edgedetect
 */
/*----------------------------------------------------------------------------*/
cpl_mask* eris_ifu_detect_crh(hdrl_image* image,
		int exposureCorrectionMode,
		hdrl_parameter *laCosmicParams,
		bool maskImage)
{
	cpl_mask *mask = NULL;
	cpl_ensure(image, CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(laCosmicParams, CPL_ERROR_NULL_INPUT, NULL);


	TRY
	{
		if ((exposureCorrectionMode & COSMIC_RAY_EXPOSURE_DETECTION) &&
				(laCosmicParams != NULL)) {
			mask = hdrl_lacosmic_edgedetect(image, laCosmicParams);
			cpl_msg_info(__func__,"CRH count %lld",cpl_mask_count(mask));
			if (maskImage) {
				BRK_IF_ERROR(
						hdrl_image_reject_from_mask(image, mask));
			}
			//            if (loadingCnt == 0) {
			//                cpl_mask_save(mask,"crh.fits", NULL, CPL_IO_CREATE);
			//            } else {
			//                cpl_mask_save(mask,"crh.fits", NULL, CPL_IO_EXTEND);
			//            }
			//            loadingCnt++;
		}
	}
	CATCH
	{
		CATCH_MSGS();
		eris_ifu_free_mask(&mask);
	}
	eris_check_error_code("eris_ifu_detect_crh");
	return mask;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Detect saturated pixels and flag them as bad
  @param    image    Input/output image
  @param    dqi      (Optional) Data quality image to update with ERIS_DQI_SAT flag
  @return   CPL_ERROR_NONE on success, otherwise an error code
  @note     Saturation threshold is hard-coded as -45727232.0f (TODO: should use #define)
  @note     Saturated pixels are flagged in the image's bad pixel mask
  @note     If dqi is provided, ERIS_DQI_SAT (value 8) is added to saturated pixels
  @note     Prints debug message if more than 50 saturated pixels found
  @note     TODO: Function expects float image but should work with double
  @note     TODO: Hard-coded threshold value should be a parameter or constant
 */
/*----------------------------------------------------------------------------*/
/* TODO: this function expects a floating point image, should rather work with double
 *       filling of dqi to be understood: with real data works as implemented but code it's strange
 */

cpl_error_code eris_ifu_saturation_detection(cpl_image *image, cpl_image *dqi) {

	cpl_ensure_code(image, CPL_ERROR_NULL_INPUT);
	//TODO: you should not hard-code numbers, rather use a #define
	const float satPixelValue = -45727232.0f;
	const float satPixelLevel = satPixelValue + 1.f;

	cpl_size nx = cpl_image_get_size_x(image);
	cpl_size ny = cpl_image_get_size_y(image);
	const float *data = cpl_image_get_data_float_const(image);
	cpl_mask *mask = cpl_image_get_bpm(image);
	cpl_binary *bp = cpl_mask_get_data(mask);
	//int* qp = cpl_image_get_data_int(dqi);

	int nSat = 0;
	for (cpl_size iy=0; iy<ny; iy++) {
		for (cpl_size ix=0; ix<nx; ix++) {
		if (data[ix+iy*nx] <= satPixelLevel) {
			if (dqi != NULL) {
				cpl_image_add_scalar(dqi, ERIS_DQI_SAT);
				//qp[ix+iy*nx] = ERIS_DQI_SAT;
			}
            bp[ix+iy*nx] = BAD_PIX;
			nSat++;
		}
		}
	}
	cpl_image_set_bpm(image, mask);
	//TODO: you should not hard-code numbers, rather use a #define
	if (nSat > 50) {
		cpl_msg_debug(__FUNCTION__, "Found %d saturated pixels", nSat);
	}
	eris_check_error_code("eris_ifu_saturation_detection");
	return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Perform line/row correction on raw detector image
  @param    image    Input/output detector image
  @return   CPL_ERROR_NONE on success, otherwise an error code
  
  @note     The read-out electronics subtracts the mean of 4 reference pixels at
            left and right borders to remove row-dependent bias. This function
            replaces the mean with the median to reduce impact of hot pixels.
  
  @note     Processing:
            - Extracts 4 leftmost and 4 rightmost pixels for each row
            - Calculates mean and median of these 8 reference pixels
            - Adds (mean - median) correction to all non-border pixels in the row
  
  @note     Border size is 4 pixels (hard-coded constant)
  @note     Supports both float and double image types
  @note     Image must have width >= 8 pixels (2 × border size)
  @see      eris_ifu_exposure_column_correction
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_exposure_line_correction(cpl_image *image)
{
	cpl_error_code retVal = CPL_ERROR_NONE;
	cpl_size nCols;
	cpl_size nRows;
	cpl_type type;
	double *dData = NULL;
	float *fData = NULL;
	cpl_vector *borderVector = NULL;
	double *borderData;
	double borderMean;
	double borderMedian;
	float fborderMean;
	float fborderMedian;
	const int borderSize = 4;

	cpl_ensure_code(image, CPL_ERROR_NULL_INPUT);
	TRY
	{
		BRK_IF_NULL(
				borderVector = cpl_vector_new(borderSize*2));
		borderData = cpl_vector_get_data(borderVector);

		nCols = cpl_image_get_size_x(image);
		nRows = cpl_image_get_size_y(image);
		if (nCols < borderSize*2) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
                    "CPL image too small");
		}
		type = cpl_image_get_type(image);
		if (type == CPL_TYPE_FLOAT) {
			fData = cpl_image_get_data_float(image);
		} else if (type == CPL_TYPE_DOUBLE) {
			dData = cpl_image_get_data_double(image);
		} else {
            BRK_WITH_ERROR_MSG(CPL_ERROR_INVALID_TYPE,
					"CPL image type must be float or double");
		}

		for (cpl_size row=0; row<nRows; row++) {
			for (int col=0; col<4; col++) {
				if (type == CPL_TYPE_FLOAT) {
					borderData[col] = fData[col+row*nCols];
					borderData[col+borderSize] = fData[nCols-1-col+row*nCols];
				} else {
					borderData[col] = dData[col+row*nCols];
					borderData[col+borderSize] = dData[nCols-1-col+row*nCols];
				}
			}
			borderMean = cpl_vector_get_mean(borderVector);
			borderMedian = cpl_vector_get_median(borderVector);
			if (type == CPL_TYPE_FLOAT) {
				fborderMean = (float) borderMean;
				fborderMedian = (float) borderMedian;
			}
			for (int col=borderSize; col<(nCols-borderSize); col++) {
				if (type == CPL_TYPE_FLOAT) {
					fData[col+row*nCols] = fData[col+row*nCols] +
							fborderMean - fborderMedian;
				} else {
					dData[col+row*nCols] = dData[col+row*nCols] +
							borderMean - borderMedian;
				}
			}
		}
		CHECK_ERROR_STATE();
	}
	CATCH
	{
		retVal = cpl_error_get_code();
	}
	eris_ifu_free_vector(&borderVector);
	eris_check_error_code("eris_ifu_exposure_line_correction");
	return retVal;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Perform column correction on raw detector image
  @param    image    Input/output detector image
  @return   CPL_ERROR_NONE on success, otherwise an error code
  
  @note     The detector is read out vertically, so artifacts in vertical direction
            can be due to read-out problems. Sometimes alternating bright/dark
            column patterns are visible.
  
  @note     Processing:
            - For each column, extracts 4 top and 4 bottom reference pixels
            - Calculates median of these 8 reference pixels
            - Subtracts median from all non-border pixels in the column
  
  @note     Border size is 4 pixels (hard-coded constant)
  @note     Supports both float and double image types
  @note     Image must have height >= 8 pixels (2 × border size)
  @see      eris_ifu_exposure_line_correction
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_exposure_column_correction(cpl_image *image)
{
	cpl_error_code retVal = CPL_ERROR_NONE;
	cpl_size nCols;
	cpl_size nRows;
	cpl_type type;
	double *dData = NULL;
	float *fData= NULL;
	cpl_vector *borderVector = NULL;
	double *borderData;
	double borderMedian;
	float fborderMedian;
	const int borderSize = 4;

	cpl_ensure_code(image, CPL_ERROR_NULL_INPUT);
	TRY
	{
		BRK_IF_NULL(
				borderVector = cpl_vector_new(borderSize*2));
		borderData = cpl_vector_get_data(borderVector);

		nCols = cpl_image_get_size_x(image);
		nRows = cpl_image_get_size_y(image);
		if (nRows < borderSize*2) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
					"CPL image too small");
		}
		type = cpl_image_get_type(image);
		if (type == CPL_TYPE_FLOAT) {
			fData = cpl_image_get_data_float(image);
		} else if (type == CPL_TYPE_DOUBLE) {
			dData = cpl_image_get_data_double(image);
		} else {
            BRK_WITH_ERROR_MSG(CPL_ERROR_INVALID_TYPE,
					"CPL image type must be float or double");
		}

		for (cpl_size col=0; col<nCols; col++) {
			for (int row=0; row<4; row++) {
				if (type == CPL_TYPE_FLOAT) {
					borderData[row] = fData[col + row*nCols];
					borderData[row+borderSize] = fData[col+(nRows-row-1)*nCols];
				} else {
					borderData[row] = dData[col + row*nCols];
					borderData[row+borderSize] = dData[col+(nRows-row-1)*nCols];
				}
			}
			borderMedian = cpl_vector_get_median(borderVector);
			if (type == CPL_TYPE_FLOAT) {
				fborderMedian = (float) borderMedian;
			}
			for (int row=borderSize; row<(nCols-borderSize); row++) {
				if (type == CPL_TYPE_FLOAT) {
					fData[col + row*nCols] = fData[col + row*nCols] -
							fborderMedian;
				} else {
					dData[col + row*nCols] = dData[col + row*nCols] -
							borderMedian;
				}
			}
		}
		CHECK_ERROR_STATE();
	}
	CATCH
	{
		retVal = cpl_error_get_code();
	}
	eris_ifu_free_vector(&borderVector);
	eris_check_error_code("eris_ifu_exposure_column_correction");
	return retVal;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create 2D and/or 3D bad pixel masks using HDRL algorithms
  @param    pl              Recipe parameter list
  @param    recipe_name     Recipe name for parameter lookup
  @param    master_img      Master image for 2D BPM computation
  @param    imglist_on      Image list for 3D BPM computation
  @param    bpm2dMask       (Output) 2D bad pixel mask
  @param    bpm3dMask       (Output) 3D bad pixel mask
  @return   CPL_ERROR_NONE on success, otherwise an error code
  
  @note     BPM method is controlled by parameter eris.<recipe>.bpm.method:
            - "2": Only 2D bad pixel detection
            - "3": Only 3D bad pixel detection
            - "2d3d": Both 2D and 3D detection
  
  @note     2D method detects spatial outliers in master_img
  @note     3D method detects temporal outliers across imglist_on
  @note     Both masks are combined and applied to master_img
  @note     Output masks must be freed by caller with cpl_mask_delete()
  @note     Logs number of detected bad pixels for each method
  @see      hdrl_bpm_2d_compute, hdrl_bpm_3d_compute
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_calc_bpm(const cpl_parameterlist    *pl,
		const char                  *recipe_name,
		hdrl_image                  *master_img,
		const hdrl_imagelist        *imglist_on,
		cpl_mask                    **bpm2dMask,
		cpl_mask                    **bpm3dMask)
{
	cpl_error_code      err                 = CPL_ERROR_NONE;
	const cpl_parameter *param              = NULL;
	const char          *bpmMethod          = NULL;
	char                *parName            = NULL;
	hdrl_parameter      *bpm2dParam         = NULL;
	hdrl_parameter      *bpm3dParam         = NULL;
	cpl_image           *bpm3dImg           = NULL;
	cpl_imagelist       *bpm3dImageList     = NULL;
	cpl_mask            *masterBpmMask      = NULL;

	cpl_ensure_code(pl, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(recipe_name, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(master_img, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(imglist_on, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(bpm2dMask, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(bpm3dMask, CPL_ERROR_NULL_INPUT);

	TRY
	{
		parName = cpl_sprintf("eris.%s.bpm.method", recipe_name);
		param = cpl_parameterlist_find_const(pl, parName);
		eris_ifu_free_string(&parName);

		bpmMethod = cpl_parameter_get_string(param);

		if (strpbrk(bpmMethod, "2") != NULL)
		{
			parName = cpl_sprintf("eris.%s.2dBadPix", recipe_name);
			bpm2dParam = hdrl_bpm_2d_parameter_parse_parlist(pl, parName);
			eris_ifu_free_string(&parName);

			cpl_msg_info(cpl_func, "Generating 2D bad pixel mask...");
			*bpm2dMask = hdrl_bpm_2d_compute(master_img, bpm2dParam);
			CHECK_ERROR_STATE();
		}

		if (strpbrk(bpmMethod, "3") != NULL)
		{
			parName = cpl_sprintf("eris.%s.3dBadPix", recipe_name);
			bpm3dParam = hdrl_bpm_3d_parameter_parse_parlist(pl, parName);
			eris_ifu_free_string(&parName);

			cpl_msg_info(cpl_func, "Generate 3D bad pixel mask");
			bpm3dImageList = hdrl_bpm_3d_compute(imglist_on, bpm3dParam);

			CHECK_ERROR_STATE();
			int imglistCnt = (int) cpl_imagelist_get_size(bpm3dImageList);
			cpl_image *patternImg = NULL;

			bpm3dImg = cpl_image_duplicate(cpl_imagelist_get(bpm3dImageList, 0));
			patternImg = cpl_image_duplicate(
					cpl_imagelist_get(bpm3dImageList, 0));
			for (int i = 1; i < imglistCnt; i++) {
				cpl_image *tmpImg = cpl_imagelist_get(bpm3dImageList, i);
				cpl_image_add(bpm3dImg, tmpImg);
				cpl_image_multiply_scalar(tmpImg, (double) (1<<i));
				cpl_image_add(patternImg, tmpImg);
			}

			parName = cpl_sprintf("eris.%s.product_depth", recipe_name);
			param = cpl_parameterlist_find_const(pl, parName);
			eris_ifu_free_string(&parName);
			int pd = cpl_parameter_get_int(param);
			CHECK_ERROR_STATE();
			if (pd >= PD_DEBUG) {
				cpl_image_save(bpm3dImg,ERIS_IFU_PRO_DARK_DBG_3D_FN".fits",
						CPL_TYPE_INT, NULL, CPL_IO_CREATE);
				cpl_image_save(patternImg,ERIS_IFU_PRO_DARK_DBG_3D_FN".fits",
						CPL_TYPE_INT, NULL, CPL_IO_EXTEND);
			}

			*bpm3dMask = cpl_mask_threshold_image_create(bpm3dImg,
					1.5, imglistCnt + 0.5);
			cpl_image_delete(patternImg);
		}

		cpl_msg_info(cpl_func, "Generating master bad pixel mask...");
		masterBpmMask = hdrl_image_get_mask(master_img);
		if (*bpm2dMask != NULL) {
			cpl_msg_info(cpl_func,"Number of 2D bad pixels: %lld",
					cpl_mask_count(*bpm2dMask));
			cpl_mask_or(masterBpmMask, *bpm2dMask);
		}
		if (bpm3dImg != NULL) {
			cpl_msg_info(cpl_func,"Number of 3D bad pixels: %lld",
					cpl_mask_count(*bpm3dMask));
			cpl_mask_or(masterBpmMask, *bpm3dMask);
		}
		hdrl_image_reject_from_mask(master_img, masterBpmMask);
	}
	CATCH
	{
		err = cpl_error_get_code();

		cpl_mask_delete(*bpm2dMask);  // not using eris_ifu_free_image() here
		cpl_mask_delete(*bpm3dMask);  // because arg is allocated outside of function

		eris_ifu_free_mask(&masterBpmMask);
		masterBpmMask = cpl_mask_new(hdrl_image_get_size_x(master_img),
				hdrl_image_get_size_y(master_img));
		hdrl_image_reject_from_mask(master_img, masterBpmMask);
	}

	eris_ifu_free_hdrl_parameter(&bpm2dParam);
	eris_ifu_free_hdrl_parameter(&bpm3dParam);
	//eris_ifu_free_mask(&masterBpmMask);
	eris_ifu_free_image(&bpm3dImg);
	eris_ifu_free_imagelist(&bpm3dImageList);
	eris_check_error_code("eris_ifu_calc_bpm");
	return err;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Warp an HDRL image using 2D polynomial transformations
  @param    hdrlInImg    Input HDRL image (data and error)
  @param    poly_u       Polynomial transformation for x-coordinates
  @param    poly_v       Polynomial transformation for y-coordinates
  @return   Warped HDRL image, or NULL on error
  
  @note     Warps both data and error images using polynomial distortion maps
  @note     Uses tanh kernel profile with default width for interpolation
  @note     Post-processing: Marks pixels with zero data values as bad
  @note     The returned HDRL image must be freed with hdrl_image_delete()
  @note     Polynomial format: output = poly(input_x, input_y)
  @see      cpl_image_warp_polynomial
 */
/*----------------------------------------------------------------------------*/
hdrl_image *eris_ifu_warp_polynomial_image(const hdrl_image     *hdrlInImg,
		const cpl_polynomial *poly_u,
		const cpl_polynomial *poly_v)
{
	hdrl_image      *hdrlOutImg = NULL;
	const cpl_image *inImgData  = NULL,
			*inImgErr   = NULL;
	cpl_image       *outImgData = NULL,
			*outImgErr  = NULL;
	cpl_vector      *profile    = NULL;
	cpl_size        nx          = 0,
			ny          = 0;
	cpl_type        type        = CPL_TYPE_UNSPECIFIED;

	cpl_ensure(hdrlInImg, CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(poly_u, CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(poly_v, CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{
		inImgData = hdrl_image_get_image_const(hdrlInImg);
		inImgErr = hdrl_image_get_error_const(hdrlInImg);

		nx = cpl_image_get_size_x(inImgData);
		ny = cpl_image_get_size_y(inImgData);
		type = cpl_image_get_type(inImgData);
		CHECK_ERROR_STATE();

		outImgData = cpl_image_new(nx, ny, type);
		outImgErr = cpl_image_new(nx, ny, type);
		profile = cpl_vector_new(CPL_KERNEL_DEF_SAMPLES);
		cpl_vector_fill_kernel_profile(profile, CPL_KERNEL_TANH,
				CPL_KERNEL_DEF_WIDTH);
		cpl_image_warp_polynomial(outImgData, inImgData,
				poly_u, poly_v,
				profile, CPL_KERNEL_DEF_WIDTH,
				profile, CPL_KERNEL_DEF_WIDTH);
		cpl_image_warp_polynomial(outImgErr, inImgErr,
				poly_u, poly_v,
				profile, CPL_KERNEL_DEF_WIDTH,
				profile, CPL_KERNEL_DEF_WIDTH);

		hdrlOutImg = hdrl_image_create(outImgData, outImgErr);

		// there can still be inconsistencies:
		// warped data can have a value of zero and not be marked as bad
		// so here we will mark zero data values as bad in the bpm
		cpl_image  *tmpImg  = hdrl_image_get_image(hdrlOutImg);
		double     *ptmpImg = cpl_image_get_data(tmpImg);
		cpl_mask   *tmpMask = cpl_image_get_bpm(tmpImg);
		cpl_binary *ptmpMask = cpl_mask_get_data(tmpMask);

		int w = (int)hdrl_image_get_size_x(hdrlOutImg),
				h = (int)hdrl_image_get_size_y(hdrlOutImg);

		for (int y = 0; y < h; y++) {
			for (int x = 0; x < w; x++) {
				if (ptmpImg[y*w+x] == 0) {
                    ptmpMask[y*w+x] = BAD_PIX;
				}
			}
		}
	}
	CATCH
	{
		eris_ifu_free_hdrl_image(&hdrlOutImg);
	}
	eris_ifu_free_image(&outImgData);
	eris_ifu_free_image(&outImgErr);
	eris_ifu_free_vector(&profile);
	eris_check_error_code("eris_ifu_warp_polynomial_image");
	return hdrlOutImg;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add standard recipe parameters to a parameter list
  @param    pl           Parameter list to which to append parameters
  @param    recipename   Recipe name (without "eris." prefix)
  @return   CPL_ERROR_NONE on success, otherwise an error code
  
  @note     Adds the following standard parameters:
            - instrument: ERIS, SINFONI, or NONE (default: ERIS)
            - product_depth: Product output depth (default: 0)
            - line_corr: Enable line correction (default: FALSE)
            - col_corr: Enable column correction (default: FALSE)
            - crh_corr: Enable cosmic ray hit correction (default: FALSE)
            - crh_detection: Enable cosmic ray hit detection (default: FALSE)
            - crh.*: LA-cosmic parameters (created with defaults: 5, 2., 3)
            - pixel_saturation: Saturation level (default: 18000.0)
  
  @note     All parameters are prefixed with "eris.<recipename>."
  @note     CLI aliases are set without the "eris.<recipename>." prefix
  @see      eris_ifu_fetch_std_param, hdrl_lacosmic_parameter_create
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_add_std_params(
		cpl_parameterlist *pl,
		const char *recipename)
{

	cpl_ensure_code(pl,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(recipename,CPL_ERROR_NULL_INPUT);


	cpl_error_code  retVal = CPL_ERROR_NONE;
	cpl_parameter   *p = NULL;
	char *pName = NULL;
	char *recname_full = NULL;
	hdrl_parameter    *p_hdrl = NULL;
	cpl_parameterlist *tmp_pl = NULL;

	TRY
	{
		recname_full = cpl_sprintf("eris.%s", recipename);
		pName = cpl_sprintf("%s.%s", recname_full, "instrument");
		p = cpl_parameter_new_enum(pName, CPL_TYPE_STRING,
				"Specifies the VLT instrument "
				"{ERIS,SINFONI,NONE}",
				recname_full, "ERIS", 3, "ERIS",
				"SINFONI", "NONE");
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "instrument");
		cpl_parameterlist_append(pl, p);
		p = NULL;
		eris_ifu_free_string(&pName);

		//        productDepthType pdMin = PD_SCIENCE;
		//        productDepthType pdMax = PD_DEBUG;
		pName = cpl_sprintf("%s.%s", recname_full, "product_depth");
		p = cpl_parameter_new_value(pName, CPL_TYPE_INT,
				"Specifies the product output depth "
				"(>0 for auxiliary products)",
				recname_full, 0);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
				"product_depth");
		cpl_parameterlist_append(pl, p);
		p = NULL;
		eris_ifu_free_string(&pName);

		pName = cpl_sprintf("%s.%s", recname_full, "line_corr");
		p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
				"If TRUE raw exposure image line "
				"correction will be applied",
				recname_full,  CPL_FALSE);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
				"line_corr");
		cpl_parameterlist_append(pl, p);
		p = NULL;
		eris_ifu_free_string(&pName);

		pName = cpl_sprintf("%s.%s", recname_full, "col_corr");
		p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
				"If TRUE raw exposure image column "
				"correction will be applied",
				recname_full,  CPL_FALSE);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
				"col_corr");
		cpl_parameterlist_append(pl, p);
		p = NULL;
		eris_ifu_free_string(&pName);

		pName = cpl_sprintf("%s.%s", recname_full, "crh_corr");
		p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
				"If TRUE raw exposure image cosmic ray "
				"hit correction will be applied",
				recname_full,  CPL_FALSE);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
				"crh_corr");
		cpl_parameterlist_append(pl, p);
		p = NULL;
		eris_ifu_free_string(&pName);

		pName = cpl_sprintf("%s.%s", recname_full, "crh_detection");
		p = cpl_parameter_new_value(pName, CPL_TYPE_BOOL,
				"If TRUE raw exposure image cosmic ray "
				"hit detection will be applied",
				recname_full,  CPL_FALSE);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
				"crh_detection");
		cpl_parameterlist_append(pl, p);
		p = NULL;
		eris_ifu_free_string(&pName);

		p_hdrl = hdrl_lacosmic_parameter_create(5, 2., 3);
		tmp_pl =  hdrl_lacosmic_parameter_create_parlist(recname_full, "crh",
				p_hdrl);
		eris_ifu_parameterlist_append_list(pl, tmp_pl);


		pName = cpl_sprintf("%s.%s", recname_full, "pixel_saturation");
		p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
				"Pixel saturation level ",
				recname_full,  18000.);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "pixel_saturation");

		cpl_parameterlist_append(pl, p);
		p = NULL;



	}
	CATCH
	{
		CATCH_MSGS();
		retVal = cpl_error_get_code();
	}

	eris_ifu_free_hdrl_parameter(&p_hdrl);
	eris_ifu_free_parameterlist(&tmp_pl);
	eris_ifu_free_parameter(&p);
	eris_ifu_free_string(&pName);
	eris_ifu_free_string(&recname_full);
	eris_check_error_code("eris_ifu_add_std_params");
	return retVal;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Free memory allocated for stdParamStruct
  @param    stdParams    Pointer to stdParamStruct to free
  @note     Only frees the crh_detection HDRL parameter within the structure
  @note     The structure itself is not freed, only its internal hdrl_parameter
  @note     Safe to call with NULL pointer
  @see      eris_ifu_fetch_std_param, eris_ifu_add_std_params
 */
/*----------------------------------------------------------------------------*/
void eris_ifu_free_std_param(struct stdParamStruct * stdParams)    {

	if(stdParams != NULL) {
		hdrl_parameter_delete(stdParams->crh_detection);
		stdParams=NULL;
	}


	eris_check_error_code("eris_ifu_free_std_param");
	return;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Fetch standard parameters from parameter list into structure
  @param    parlist      Input parameter list
  @param    recipename   Recipe name (without "eris." prefix)
  @param    stdParams    (Output) Structure to populate with parameter values
  @return   CPL_ERROR_NONE on success, otherwise an error code
  
  @note     Populates stdParams structure with:
            - productDepth: Product depth level
            - line_corr, col_corr, crh_corr, crh_det: Correction flags
            - rawImageCorrectionMask: Bitwise OR of enabled correction flags
            - crh_detection: LA-cosmic HDRL parameter
            - instrument: Instrument identifier (SPIFFI, SPIFFIER, OTHER_INSTRUMENT)
  
  @note     rawImageCorrectionMask bits:
            - LINE_EXPOSURE_CORRECTION
            - COLUMN_EXPOSURE_CORRECTION
            - COSMIC_RAY_EXPOSURE_CORRECTION
            - COSMIC_RAY_EXPOSURE_DETECTION
  
  @note     Instrument mapping: "SINFONI"→SPIFFI, "ERIS"→SPIFFIER, other→OTHER_INSTRUMENT
  @see      eris_ifu_add_std_params, eris_ifu_free_std_param
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_fetch_std_param(
		const cpl_parameterlist *parlist,
		const char *recipename,
		struct stdParamStruct *stdParams)
{

	cpl_ensure_code(parlist,CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(recipename,CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(stdParams,CPL_ERROR_NULL_INPUT);

	cpl_error_code  retVal = CPL_ERROR_NONE;
	cpl_parameter   *p = NULL;
	char *pName = NULL;
	const char *instrument = NULL;
	char *recname_full = NULL;
	char *tmp_str = NULL;

	TRY
	{
		recname_full = cpl_sprintf("eris.%s", recipename);
		pName = cpl_sprintf("%s.%s", recname_full, "product_depth");
		stdParams->productDepth = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist, pName));
		CHECK_ERROR_STATE();
		eris_ifu_free_string(&pName);

		stdParams->rawImageCorrectionMask = 0;
		pName = cpl_sprintf("%s.%s", recname_full, "line_corr");
		stdParams->line_corr = cpl_parameter_get_bool(
				cpl_parameterlist_find_const(parlist, pName));
		CHECK_ERROR_STATE();
		eris_ifu_free_string(&pName);
		if (stdParams->line_corr) {
			stdParams->rawImageCorrectionMask |= LINE_EXPOSURE_CORRECTION;
		}

		pName = cpl_sprintf("%s.%s", recname_full, "col_corr");
		stdParams->col_corr = cpl_parameter_get_bool(
				cpl_parameterlist_find_const(parlist, pName));
		CHECK_ERROR_STATE();
		eris_ifu_free_string(&pName);
		if (stdParams->col_corr) {
			stdParams->rawImageCorrectionMask |= COLUMN_EXPOSURE_CORRECTION;
		}

		pName = cpl_sprintf("%s.%s", recname_full, "crh_corr");
		stdParams->crh_corr = cpl_parameter_get_bool(
				cpl_parameterlist_find_const(parlist, pName));
		CHECK_ERROR_STATE();
		eris_ifu_free_string(&pName);
		if (stdParams->crh_corr) {
			stdParams->rawImageCorrectionMask |= COSMIC_RAY_EXPOSURE_CORRECTION;
		}

		pName = cpl_sprintf("%s.%s", recname_full, "crh_detection");
		stdParams->crh_det = cpl_parameter_get_bool(
				cpl_parameterlist_find_const(parlist, pName));
		CHECK_ERROR_STATE();
		eris_ifu_free_string(&pName);
		if (stdParams->crh_det) {
			stdParams->rawImageCorrectionMask |= COSMIC_RAY_EXPOSURE_DETECTION;
		}

		tmp_str = cpl_sprintf("%s.crh", recname_full);

		stdParams->crh_detection =
				hdrl_lacosmic_parameter_parse_parlist(parlist, tmp_str);
		eris_ifu_free_string(&tmp_str);
		CHECK_ERROR_STATE();

		pName = cpl_sprintf("%s.%s", recname_full, "instrument");
		instrument = cpl_parameter_get_string(
				cpl_parameterlist_find_const(parlist, pName));
		CHECK_ERROR_STATE();
		eris_ifu_free_string(&pName);
		if (strcmp(instrument, "SINFONI") == 0) {
			stdParams->instrument = SPIFFI;
		}
		else if(strcmp(instrument, "ERIS") == 0) {
			stdParams->instrument = SPIFFIER;
		}
		else {
			stdParams->instrument = OTHER_INSTRUMENT;
		}
	}
	CATCH
	{
		CATCH_MSGS();
		retVal = cpl_error_get_code();
	}

	eris_ifu_free_parameter(&p);
	eris_ifu_free_string(&pName);
	eris_ifu_free_string(&tmp_str);
	eris_ifu_free_string(&recname_full);
	eris_check_error_code("eris_ifu_fetch_std_param");
	return retVal;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add common configuration parameters for all recipes
  @param    pl         Parameter list to which to append
  @param    recname    Recipe name (without "eris." prefix)
  @return   CPL_ERROR_NONE on success, otherwise an error code
  @note     Adds:
            - instrument: {ERIS, SINFONI, NONE} (default: ERIS)
            - product_depth: Output depth for auxiliary products (default: 0)
  @note     Parameters are prefixed with "eris.<recname>."
  @note     CLI aliases are set without prefix
  @see      eris_parlist_config_add_bpm, eris_parlist_config_add_flat
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_parlist_config_add_all_recipes(cpl_parameterlist *pl,
		const char* recname)
{
	char                *tmp_str1       = NULL,
			*recname_full   = NULL;
	cpl_error_code      err             = CPL_ERROR_NONE;
	cpl_parameter       *p              = NULL;

	cpl_ensure_code(pl, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(recname, CPL_ERROR_NULL_INPUT);

	TRY
	{
		recname_full = cpl_sprintf("eris.%s", recname);

		/* --instrument */
		tmp_str1 = cpl_sprintf("%s%s%s", "eris.", recname, ".instrument");
		p = cpl_parameter_new_enum(tmp_str1, CPL_TYPE_STRING,
				"Specifies the VLT instrument {ERIS,SINFONI,NONE}",
				recname_full, INSTRUMENT_ERIS, 3,
				INSTRUMENT_ERIS, INSTRUMENT_SINFONI, INSTRUMENT_NONE);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "instrument");
		cpl_parameterlist_append(pl, p);
		eris_ifu_free_string(&tmp_str1);

		/* -- product_depth */
		tmp_str1 = cpl_sprintf("%s%s%s", "eris.", recname, ".product_depth");
		p = cpl_parameter_new_value(tmp_str1, CPL_TYPE_INT,
				"Specifies the product output depth "
				"instrument (>0 for auxillary products)",
				recname_full, 0);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
				"product_depth");
		cpl_parameterlist_append(pl, p);
		eris_ifu_free_string(&tmp_str1);
	}
	CATCH
	{
		CATCH_MSGS();
		err = cpl_error_get_code();
		eris_ifu_free_parameter(&p);
		eris_ifu_free_string(&tmp_str1);
	}
	eris_ifu_free_string(&tmp_str1);
	eris_ifu_free_string(&recname_full);
	eris_check_error_code("eris_parlist_config_add_all_recipes");
	return err;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add bad pixel mask configuration parameters to parameter list
  @param    pl         Parameter list to which to append
  @param    recname    Recipe name (without "eris." prefix)
  @return   CPL_ERROR_NONE on success, otherwise an error code
  
  @note     Adds:
            - bpm.method: {2d, 3d, 2d3d, none} - selects BPM detection method
            - collapse.*: HDRL collapse parameters (median, sigclip, minmax)
            - 2dBadPix.*: HDRL 2D BPM parameters (filter or Legendre smoothing)
            - 3dBadPix.*: HDRL 3D BPM parameters
  
  @note     Default method: "2d" for distortion recipe, "2d3d" for others
  @note     Default 2D method: "LEGENDRE" for distortion, "FILTER" for others
  @note     Parameters use HDRL library parameter creation functions
  @see      eris_ifu_calc_bpm, hdrl_bpm_2d_compute, hdrl_bpm_3d_compute
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_parlist_config_add_bpm(cpl_parameterlist *pl,
		const char* recname)
{
	char                *tmp_str        = NULL,
			*recname_full   = NULL,
			*tmp_def_meth   = NULL,
			*tmp_meth       = NULL;
	cpl_error_code      err             = CPL_ERROR_NONE;
	cpl_parameterlist   *tmp_pl         = NULL;
	cpl_parameter       *p              = NULL;
	hdrl_parameter      *p_hdrl1        = NULL;
	hdrl_parameter      *p_hdrl2        = NULL;

	cpl_ensure_code(pl, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(recname, CPL_ERROR_NULL_INPUT);

	TRY
	{
		recname_full = cpl_sprintf("eris.%s", recname);

		/* set default for bpm-method depending on recipe */
		if (strcmp(recname, REC_NAME_DISTORTION) == 0) {
			tmp_def_meth = cpl_sprintf("2d");
			tmp_meth = cpl_sprintf("LEGENDRE");
		} else {
			tmp_def_meth = cpl_sprintf("2d3d");
			tmp_meth = cpl_sprintf("FILTER");
		}

		/* --bpm.method */
		tmp_str = cpl_sprintf("%s%s", recname_full, ".bpm.method");
		p = cpl_parameter_new_enum(tmp_str, CPL_TYPE_STRING,
				"Specifies the VLT instrument {2d,3d,2d3d}",
				recname_full, tmp_def_meth, 4,
				"2d", "3d", "2d3d", "none");
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "bpm.method");
		cpl_parameterlist_append(pl, p);
		eris_ifu_free_string(&tmp_str);

		/* --collapse.sigclip/minmax (HDRL) */
		hdrl_parameter * mode_def =
				hdrl_collapse_mode_parameter_create(10., 1., 0., HDRL_MODE_MEDIAN, 0);
		p_hdrl1 = hdrl_collapse_sigclip_parameter_create(3., 3., 5);
		p_hdrl2 = hdrl_collapse_minmax_parameter_create(1., 1.);
		tmp_pl = hdrl_collapse_parameter_create_parlist(recname_full,
				"collapse", "MEDIAN",
				p_hdrl1, p_hdrl2, mode_def);
		eris_ifu_parameterlist_append_list(pl, tmp_pl);
		eris_ifu_free_hdrl_parameter(&p_hdrl1);
		eris_ifu_free_hdrl_parameter(&p_hdrl2);
		eris_ifu_free_parameterlist(&tmp_pl);
		hdrl_parameter_delete(mode_def);
		/* --2dBadPix.* (HDRL) */
		p_hdrl1 = hdrl_bpm_2d_parameter_create_filtersmooth(
				10., 10., 3, CPL_FILTER_MEDIAN, CPL_BORDER_NOP, 5, 5);
		// values from HDRL-Demo
		// 3, 3, 5, CPL_FILTER_MEDIAN, CPL_BORDER_FILTER, 3, 3));
		p_hdrl2 = hdrl_bpm_2d_parameter_create_legendresmooth(
				10., 10., 3, 20, 20, 10, 10, 2, 2);
		// values from HDRL-Demo
		// 3, 3, 5, 20, 20, 11, 11, 3, 3));
		tmp_pl = hdrl_bpm_2d_parameter_create_parlist(recname_full,
				"2dBadPix", tmp_meth,
				p_hdrl1, p_hdrl2);
		eris_ifu_parameterlist_append_list(pl, tmp_pl);
		eris_ifu_free_hdrl_parameter(&p_hdrl1);
		eris_ifu_free_hdrl_parameter(&p_hdrl2);
		eris_ifu_free_parameterlist(&tmp_pl);

		/* --3dBadPix.* (HDRL) */
		p_hdrl1 = hdrl_bpm_3d_parameter_create(12., 12.,
				HDRL_BPM_3D_THRESHOLD_RELATIVE);
		tmp_pl = hdrl_bpm_3d_parameter_create_parlist(recname_full,
				"3dBadPix", p_hdrl1);
		eris_ifu_parameterlist_append_list(pl, tmp_pl);
		eris_ifu_free_hdrl_parameter(&p_hdrl1);
		eris_ifu_free_parameterlist(&tmp_pl);
	}
	CATCH
	{
		CATCH_MSGS();
		err = cpl_error_get_code();
		eris_ifu_free_parameter(&p);
		eris_ifu_free_string(&tmp_str);
		eris_ifu_free_hdrl_parameter(&p_hdrl1);
		eris_ifu_free_hdrl_parameter(&p_hdrl2);
		eris_ifu_free_parameterlist(&tmp_pl);
	}
	eris_ifu_free_string(&tmp_str);
	eris_ifu_free_string(&recname_full);
	eris_ifu_free_string(&tmp_def_meth);
	eris_ifu_free_string(&tmp_meth);
	eris_check_error_code("eris_parlist_config_add_bpm");
	return err;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add flat field configuration parameters to parameter list
  @param    pl         Parameter list to which to append
  @param    recname    Recipe name (without "eris." prefix)
  @return   CPL_ERROR_NONE on success, otherwise an error code
  
  @note     Adds:
            - mode: Flat field mode {fast, hdrl, segment} (default: fast)
            - flat_lo.*: Low-frequency flat parameters (HDRL, filter size 5×5)
            - flat_hi.*: High-frequency flat parameters (HDRL, filter size 7×7)
            - qc.fpn.{xmin,xmax,ymin,ymax}{1,2}: Fixed pattern noise ROI parameters
  
  @note     FPN region 1 default: [512:1536, 512:1536]
  @note     FPN region 2 default: [1350:1390, 1000:1200]
  @note     Parameters use HDRL library flat field creation functions
  @see      eris_ifu_calc_flat, eris_ifu_flat_calc_qc_pre
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_parlist_config_add_flat(cpl_parameterlist *pl,
		const char* recname)
{
	cpl_error_code      err             = CPL_ERROR_NONE;
	hdrl_parameter      *p_hdrl         = NULL;
	cpl_parameterlist   *tmp_pl         = NULL;
	cpl_parameter       *p              = NULL;
	char                *tmp_str        = NULL,
			*recname_full   = NULL;

	cpl_ensure_code(pl, CPL_ERROR_NULL_INPUT);

	TRY
	{
		recname_full = cpl_sprintf("eris.%s", recname);

		/* add mode parameter */
		tmp_str = cpl_sprintf("%s%s", recname_full, ".mode");
		// set default to segment, when this mode is implemented
		p = cpl_parameter_new_enum(tmp_str,
				CPL_TYPE_STRING,
				"Mode of flat-calculation",
				recname,
				flatModes[0], 3,
				flatModes[0],flatModes[1],flatModes[2]);
		//        if (strcmp(recname, REC_NAME_DISTORTION)==0) {
		//        BRK_IF_NULL(
		//            /* fast-mode for distortion */
		//            p = cpl_parameter_new_enum(tmp_str,
		//                                    CPL_TYPE_STRING,
		//                                    "Mode of flat-calculation",
		//                                    recname,
		//                                    flatModes[2], 3,
		//                                    flatModes[0],flatModes[1],flatModes[2]));
		//        } else {
		//            BRK_IF_NULL(
		//                /* hdrl-mode for flat */
		//                p = cpl_parameter_new_enum(tmp_str,
		//                                        CPL_TYPE_STRING,
		//                                        "Mode of flat-calculation",
		//                                        recname,
		//                                        flatModes[1], 3,
		//                                        flatModes[0],flatModes[1],flatModes[2]));
		//        }

		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,"flat.mode");
		cpl_parameterlist_append(pl, p);
		eris_ifu_free_string(&tmp_str);

		/* --flat_lo.* (HDRL) */
		p_hdrl = hdrl_flat_parameter_create(5, 5, HDRL_FLAT_FREQ_LOW);
		tmp_pl =  hdrl_flat_parameter_create_parlist(recname_full, "flat_lo",
				p_hdrl);
		eris_ifu_parameterlist_append_list(pl, tmp_pl);
		eris_ifu_free_hdrl_parameter(&p_hdrl);
		eris_ifu_free_parameterlist(&tmp_pl);

		/* --flat_hi.* (HDRL) */
		p_hdrl = hdrl_flat_parameter_create(7, 7, HDRL_FLAT_FREQ_HIGH);
		tmp_pl =  hdrl_flat_parameter_create_parlist(recname_full, "flat_hi",
				p_hdrl);
		eris_ifu_parameterlist_append_list(pl, tmp_pl);
		eris_ifu_free_hdrl_parameter(&p_hdrl);
		eris_ifu_free_parameterlist(&tmp_pl);

		/* QC FPN */
		tmp_str = cpl_sprintf("%s%s", recname_full, ".qc.fpn.xmin1");
		p = cpl_parameter_new_range(tmp_str, CPL_TYPE_INT,
				"Fixed Pattern Noise: qc_fpn_xmin1",
				recname_full,
				512, 1, ERIS_IFU_DETECTOR_SIZE_X);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "qc_fpn_xmin1");
		cpl_parameterlist_append(pl, p);
		eris_ifu_free_string(&tmp_str);

		tmp_str = cpl_sprintf("%s%s", recname_full, ".qc.fpn.xmax1");
		p = cpl_parameter_new_range(tmp_str, CPL_TYPE_INT,
				"Fixed Pattern Noise: qc_fpn_xmax1",
				recname_full,
				1536, 1, ERIS_IFU_DETECTOR_SIZE_X);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "qc_fpn_xmax1");
		cpl_parameterlist_append(pl, p);
		eris_ifu_free_string(&tmp_str);

		tmp_str = cpl_sprintf("%s%s", recname_full, ".qc.fpn.ymin1");
		p = cpl_parameter_new_range(tmp_str, CPL_TYPE_INT,
				"Fixed Pattern Noise: qc_fpn_ymin1",
				recname_full,
				512, 1, ERIS_IFU_DETECTOR_SIZE_Y);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "qc_fpn_ymin1");
		cpl_parameterlist_append(pl, p);
		eris_ifu_free_string(&tmp_str);

		tmp_str = cpl_sprintf("%s%s", recname_full, ".qc.fpn.ymax1");
		p = cpl_parameter_new_range(tmp_str, CPL_TYPE_INT,
				"Fixed Pattern Noise: qc_fpn_ymax1",
				recname_full,
				1536, 1, ERIS_IFU_DETECTOR_SIZE_Y);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "qc_fpn_ymax1");
		cpl_parameterlist_append(pl, p);
		eris_ifu_free_string(&tmp_str);

		tmp_str = cpl_sprintf("%s%s", recname_full, ".qc.fpn.xmin2");
		p = cpl_parameter_new_range(tmp_str, CPL_TYPE_INT,
				"Fixed Pattern Noise: qc_fpn_xmin2",
				recname_full,
				1350, 1, ERIS_IFU_DETECTOR_SIZE_X);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "qc_fpn_xmin2");
		cpl_parameterlist_append(pl, p);
		eris_ifu_free_string(&tmp_str);

		tmp_str = cpl_sprintf("%s%s", recname_full, ".qc.fpn.xmax2");
		p = cpl_parameter_new_range(tmp_str, CPL_TYPE_INT,
				"Fixed Pattern Noise: qc_fpn_xmax2",
				recname_full,
				1390, 1, ERIS_IFU_DETECTOR_SIZE_X);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "qc_fpn_xmax2");
		cpl_parameterlist_append(pl, p);
		eris_ifu_free_string(&tmp_str);

		tmp_str = cpl_sprintf("%s%s", recname_full, ".qc.fpn.ymin2");
		p = cpl_parameter_new_range(tmp_str, CPL_TYPE_INT,
				"Fixed Pattern Noise: qc_fpn_ymin2",
				recname_full,
				1000, 1, ERIS_IFU_DETECTOR_SIZE_Y);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "qc_fpn_ymin2");
		cpl_parameterlist_append(pl, p);
		eris_ifu_free_string(&tmp_str);

		tmp_str = cpl_sprintf("%s%s", recname_full, ".qc.fpn.ymax2");
		p = cpl_parameter_new_range(tmp_str, CPL_TYPE_INT,
				"Fixed Pattern Noise: qc_fpn_ymax2",
				recname_full,
				1200, 1, ERIS_IFU_DETECTOR_SIZE_Y);
		cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "qc_fpn_ymax2");
		cpl_parameterlist_append(pl, p);
		eris_ifu_free_string(&tmp_str);
	}
	CATCH
	{
		CATCH_MSGS();
		err = cpl_error_get_code();
		eris_ifu_free_hdrl_parameter(&p_hdrl);
		eris_ifu_free_parameterlist(&tmp_pl);
	}
	eris_ifu_free_string(&tmp_str);
	eris_ifu_free_string(&recname_full);
	eris_check_error_code("eris_parlist_config_add_flat");
	return err;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Fit a 1D polynomial to vector data
  @param    x          X-coordinate vector
  @param    y          Y-value vector to fit
  @param    degree     Polynomial degree (must be > 0)
  @return   Vector of fit coefficients (length = degree + 1), or NULL on error
  
  @note     Fit formula: y(x) = coeff[0] + coeff[1]*x + coeff[2]*x^2 + ...
  @note     Uses CPL polynomial fitting with no sample symmetry
  @note     Minimum degree is 0 (constant), maximum is the specified degree
  @note     The returned vector must be freed with cpl_vector_delete()
  @note     Coefficient ordering: [constant, linear, quadratic, cubic, ...]
  @see      cpl_polynomial_fit, eris_ifu_1d_polynomial_fit
 */
/*----------------------------------------------------------------------------*/
cpl_vector* eris_ifu_polyfit_1d(const cpl_vector *x,
		const cpl_vector *y,
		const int degree)
{
	cpl_vector          *fit_par    = NULL;
	cpl_polynomial      *poly_coeff = NULL;
	cpl_matrix          *x_matrix   = NULL;
	double              *pfit_par   = NULL,
			*px          = NULL;
	cpl_size            k           = 0,
			mindeg1d    = 0,    //1,
			maxdeg1d    = degree;

	const cpl_boolean   sampsym     = CPL_FALSE;

	cpl_ensure(x, CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(y, CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(degree > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

	TRY
	{

		//
		//  setup data for fitting
		//
		poly_coeff = cpl_polynomial_new(1);

		// put x-vector into a matrix (cast away constness of x: is okay since
		// data is wrapped into a matrix which is passed as const again)
		px = cpl_vector_get_data((cpl_vector*)x);
		x_matrix = cpl_matrix_wrap(1, cpl_vector_get_size(x), px);

		//
		// fit 1d data
		//
		cpl_polynomial_fit(poly_coeff,
				x_matrix,
				&sampsym,
				y,
				NULL,
				CPL_FALSE,
				&mindeg1d,
				&maxdeg1d);

		cpl_matrix_unwrap(x_matrix); x_matrix = NULL;
		CHECK_ERROR_STATE();

		//
		// put fit coefficients into a vector to return
		//
		fit_par = cpl_vector_new(degree + 1);
		pfit_par = cpl_vector_get_data(fit_par);

		for(k = 0; k <= degree; k++) {
			pfit_par[k] = cpl_polynomial_get_coeff(poly_coeff, &k);
		}
	}
	CATCH
	{
		eris_ifu_free_vector(&fit_par);
	}

	cpl_matrix_unwrap(x_matrix); x_matrix = NULL;
	cpl_polynomial_delete(poly_coeff); poly_coeff = NULL;
	eris_check_error_code("eris_ifu_polyfit_1d");
	return fit_par;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Fit a 1D polynomial to arrays of data points
  @param    nPoints    Number of data points
  @param    xdata      X-coordinates array
  @param    ydata      Y-values array
  @param    degree     Polynomial degree
  @return   Fitted polynomial object, or NULL on error
  
  @note     If nPoints < degree+1, returns polynomial with all coefficients set to 0
  @note     Uses CPL polynomial fitting without sample symmetry
  @note     Arrays are wrapped (not copied) for efficiency
  @note     The returned polynomial must be freed with cpl_polynomial_delete()
  @see      eris_ifu_polyfit_1d, cpl_polynomial_fit
 */
/*----------------------------------------------------------------------------*/
cpl_polynomial * eris_ifu_1d_polynomial_fit(
		int nPoints,
		double *xdata,
		double *ydata,
		int degree)
{
	cpl_polynomial *fit = NULL;
	cpl_matrix *x_pos = NULL;
	cpl_vector *values = NULL;
	TRY
	{
		fit = cpl_polynomial_new(1);
		if (nPoints < degree+1) {
			for (cpl_size power=0; power<=degree; power++) {
				cpl_polynomial_set_coeff(fit, &power, 0.0);
			}

		} else {
			x_pos = cpl_matrix_wrap(1, nPoints, xdata);
			values = cpl_vector_wrap(nPoints, ydata);
			const cpl_boolean sampsym = CPL_FALSE;
			const cpl_size maxdeg1d = degree;
			cpl_polynomial_fit(fit, x_pos, &sampsym, values, NULL,
					CPL_FALSE, NULL, &maxdeg1d);
		}
	}
	CATCH
	{
		CATCH_MSGS();
		if (fit != NULL) {
			cpl_polynomial_delete(fit);
		}
		fit = NULL;
	}

	if (x_pos != NULL) {
		cpl_matrix_unwrap(x_pos);
	}
	if (values != NULL) {
		cpl_vector_unwrap(values);
	}
	eris_check_error_code("eris_ifu_1d_polynomial_fit");
	return fit;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Perform 1D interpolation using GSL routines
  @param    xIn          X-values of input data points
  @param    yIn          Y-values of input data points
  @param    nIn          Number of input points
  @param    xOut         X-values for output (interpolation points)
  @param    yOut         (Output) Interpolated Y-values
  @param    nOut         Number of output points
  @param    interType    Interpolation type (see below)
  @return   CPL_ERROR_NONE on success, otherwise an error code
  
  @note     interType values:
            - > 2: Polynomial interpolation with degree = interType-1 (uses moving window)
            - 2: Linear interpolation
            - -1: Cubic spline with natural boundary conditions
            - -2: Cubic spline with periodic boundary conditions
            - -3: Akima spline with natural boundary conditions
            - -4: Akima spline with periodic boundary conditions
            - -5: Steffen's monotonic interpolation
  
  @note     NaN and Inf values in input are automatically removed before interpolation
  @note     GSL errors (except GSL_EDOM) are converted to CPL errors
  @note     For polynomial interpolation, uses a moving window of interType points
  @see      remove_2nans, gsl_interp_eval_e
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_1d_interpolation(
		double *xIn,
		double *yIn,
		int     nIn,
		double *xOut,
		double *yOut,
		int     nOut,
		const int interType)
{
	cpl_error_code retVal = CPL_ERROR_NONE;
	const gsl_interp_type *interp;
	int gsl_status;
	gsl_error_handler_t * gslErrorHandler;
	double *xIn2 = NULL;
	double *yIn2 = NULL;
	int     nIn2 = 0;

	TRY
	{
		gslErrorHandler = gsl_set_error_handler_off();

		ASSURE((xIn != NULL && yIn != NULL && xOut != NULL &&
				yOut != NULL),
				CPL_ERROR_NULL_INPUT, "no NULL pointer allowed for arrays");
		ASSURE((nIn > 0 && nOut > 0),
				CPL_ERROR_ILLEGAL_INPUT, "invalid array size input");

		remove_2nans(nIn, xIn, yIn, &nIn2, &xIn2, &yIn2);

		gsl_interp_accel *acc = gsl_interp_accel_alloc();
		if (interType > 2){
			const int nPoints = interType;
			interp = gsl_interp_polynomial;
			gsl_interp *workspace = gsl_interp_alloc(interp, nPoints);
			size_t sx;
			for (int rx = 0; rx < nOut; rx++) {
				sx = gsl_interp_accel_find(acc, xIn2, nIn2,
						xOut[rx]);
				if (sx > (size_t) (nPoints / 2)) {
					sx -= nPoints / 2;
				}
				if (((cpl_size) sx + nPoints) > nIn2) {
					sx = nIn2 - nPoints;
				}
				gsl_status = gsl_interp_init(workspace, &xIn2[sx], &yIn2[sx],
						nPoints);
				if (gsl_status != 0) {
					BRK_WITH_ERROR_MSG(CPL_ERROR_UNSPECIFIED,
							"GSL interpolation routine returned error %d: %s",
							gsl_status, gsl_strerror(gsl_status));
				}
				gsl_status = gsl_interp_eval_e(workspace, &xIn2[sx],
						&yIn2[sx], xOut[rx], acc, &yOut[rx]);
				if (gsl_status != 0 && gsl_status != GSL_EDOM) {
					BRK_WITH_ERROR_MSG(CPL_ERROR_UNSPECIFIED,
							"GSL interpolation routine returned error %d: %s",
							gsl_status, gsl_strerror(gsl_status));
				}
			}
			gsl_interp_free(workspace);
		} else {
			switch (interType) {
			case 2:
				interp = gsl_interp_linear;
				break;
			case -1:
				interp = gsl_interp_cspline;
				break;
			case -2:
				interp = gsl_interp_cspline_periodic;
				break;
			case -3:
				interp = gsl_interp_akima;
				break;
			case -4:
				interp = gsl_interp_akima_periodic;
				break;
			case -5:
				interp = gsl_interp_steffen;
				break;
			default:
				BRK_WITH_ERROR_MSG(CPL_ERROR_UNSUPPORTED_MODE,
						"unknown interpolation type %d", interType);
			}
			gsl_interp *workspace = gsl_interp_alloc(interp, nIn2);

			gsl_status = gsl_interp_init(workspace, xIn2, yIn2, nIn2);
			if (gsl_status != 0) {
				BRK_WITH_ERROR_MSG(CPL_ERROR_UNSPECIFIED,
						"GSL interpolation routine returned error %d: %s",
						gsl_status, gsl_strerror(gsl_status));
			}
			for (int rx = 0; rx < nOut; rx++) {
				gsl_status = gsl_interp_eval_e(workspace, xIn2, yIn2,
						xOut[rx], acc, &yOut[rx]);
				if (gsl_status != 0 && gsl_status != GSL_EDOM) {
					BRK_WITH_ERROR_MSG(CPL_ERROR_UNSPECIFIED,
							"GSL interpolation routine returned error %d: %s",
							gsl_status, gsl_strerror(gsl_status));
				}
			}
			gsl_interp_free(workspace);
		}
		gsl_interp_accel_free(acc);

	}
	CATCH
	{
		retVal = cpl_error_get_code();
	}
	gsl_set_error_handler(gslErrorHandler);
	eris_ifu_free_double_array(&xIn2);
	eris_ifu_free_double_array(&yIn2);
	eris_check_error_code("eris_ifu_1d_interpolation");
	return retVal;
}

void remove_2nans(int size_in, double *xin, double *yin, int *size_out, double **xout, double **yout) {
	int no_valids = 0,
			ox        = 0;

	for (int i = 0; i < size_in; i++) {
		if ((! eris_ifu_is_nan_or_inf(xin[i])) && (! eris_ifu_is_nan_or_inf(yin[i]))) {
			//        if ((kmclipm_is_nan_or_inf(xin[i]) == 0) && (kmclipm_is_nan_or_inf(yin[i]) == 0)) {
			no_valids++;
		}
	}

	*size_out = no_valids;
	*xout =  (double *) cpl_calloc(no_valids, sizeof(double));
	*yout =  (double *) cpl_calloc(no_valids, sizeof(double));

	ox = 0;
	for (int i = 0; i < size_in; i++) {
		if ((! eris_ifu_is_nan_or_inf(xin[i])) && (! eris_ifu_is_nan_or_inf(yin[i]))) {
			//        if ((kmclipm_is_nan_or_inf(xin[i]) == 0) && (kmclipm_is_nan_or_inf(yin[i]) == 0)) {
			(*xout)[ox] = xin[i];
			(*yout)[ox] = yin[i];
			ox++;
		}
	}
	eris_check_error_code("remove_2nans");
	return;
}


// now in eris_ifu_vector.h/.c
//
///**
//  @brief    Compute the mean value of vector elements ignoring NaN values.
//  @param    v  Input const cpl_vector
//  @return   Mean value of vector elements or undefined on error.
// */
//double eris_ifu_vector_get_mean(const cpl_vector *v)
//{
//    double          sum     = 0.,
//                    mean    = 0.;
//    const double    *pv     = NULL;
//    int             n       = 0;

//    cpl_ensure(v != NULL, CPL_ERROR_NULL_INPUT, -1.0);

//    TRY
//    {
//        BRK_IF_NULL(
//            pv = cpl_vector_get_data_const(v));

//        for (int i = 0; i < cpl_vector_get_size(v); i++) {
//            if (!isnan(pv[i])) {
//                sum += pv[i];
//                n++;
//            }
//        }
//        mean = sum / (double)n;
//    }
//    CATCH
//    {
//        CATCH_MSGS();
//        mean = 0./0.;
//    }
//    return mean;
//}

/**
  @brief    Calculate the mean of an image ignoring NaN values.
  @param    image   Image to be averaged.
  @return   The mean.

  Possible cpl_error_code set in this function:
  @li CPL_ERROR_NULL_INPUT    if @c data is NULL.
 */
double eris_ifu_image_get_mean(const cpl_image *image)
{
	cpl_size        nr_valid_pix    = 0,
			nx              = 0,
			ny              = 0;
	const double    *pimg           = NULL;
	double          mean            = 0.0,
			sum             = 0.0;

	cpl_ensure(image != NULL, CPL_ERROR_NULL_INPUT, -1.0);

	TRY
	{
		nx = cpl_image_get_size_x(image);
		ny = cpl_image_get_size_y(image);
		nr_valid_pix = nx * ny;

		BRK_IF_NULL(
				pimg = cpl_image_get_data_double_const(image));

		for (int j = 0; j < ny; j++) {
			for (int i = 0; i < nx; i++) {
				if (isnan(pimg[i+j*nx]))
					nr_valid_pix--;
				else
					sum += pimg[i+j*nx];
			}
		}
		mean = sum / (double) nr_valid_pix;
	}
	CATCH
	{
		CATCH_MSGS();
		mean = 0.0;
	}
	eris_check_error_code("eris_ifu_image_get_mean");
	return mean;
}

cpl_error_code eris_ifu_line_gauss_fit(
		const cpl_vector *yIn,
		int center,
		int range,
		struct gaussParStruct *gaussPar)
{
	cpl_error_code  retVal      = CPL_ERROR_NONE;
	int             start       = 0;
	double          pos         = 0.,
			sigma       = 0.,
			area        = 0.,
			offset      = 0.,
			mserr       = 0.,
			*yInData    = NULL,
			*pxVec      = NULL;
	const double    *pyVec      = NULL;
	cpl_vector      *xVec       = NULL,
			*yVec       = NULL,
			*yIn2       = NULL;

	cpl_ensure_code(yIn, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(cpl_vector_get_size(yIn) == ERIS_IFU_DETECTOR_SIZE,
			CPL_ERROR_ILLEGAL_INPUT);

	TRY
	{
		try_again_fit:
		BRK_IF_NULL(
				yIn2 = cpl_vector_duplicate(yIn));
		BRK_IF_NULL(
				yInData = cpl_vector_get_data(yIn2));

		// calc start position in vector-data
		start = center-(range/2);

		if (start < 0) {
			// no negative starting points!
			start = 0;
			range -= start;
		} else if (start >= ERIS_IFU_DETECTOR_SIZE_X-ERIS_IFU_DETECTOR_BP_BORDER) {
			start = ERIS_IFU_DETECTOR_SIZE_X - ERIS_IFU_DETECTOR_BP_BORDER - 1;
		}
		if (start + range >= ERIS_IFU_DETECTOR_SIZE_X-ERIS_IFU_DETECTOR_BP_BORDER) {
			// start position + range to fit are larger than input data!
			// decrease range accordingly
			range = ERIS_IFU_DETECTOR_SIZE_X-ERIS_IFU_DETECTOR_BP_BORDER - start;
		}
		BRK_IF_NULL(
				yVec = cpl_vector_wrap(range, &yInData[start]));

		BRK_IF_NULL(
				pyVec = cpl_vector_get_data_const(yVec));

		BRK_IF_NULL(
				xVec = cpl_vector_new(range));
		BRK_IF_NULL(
				pxVec = cpl_vector_get_data(xVec));
		for (int ix=0; ix<range; ix++) {
			pxVec[ix] = start + ix;
		}

		cpl_errorstate prestate = cpl_errorstate_get();
		retVal = cpl_vector_fit_gaussian(xVec, NULL, yVec, NULL, CPL_FIT_ALL,
				&pos, &sigma, &area, &offset, &mserr, NULL, NULL);

		gaussPar->errorCode = (int) retVal;
		if ((retVal == CPL_ERROR_NONE) ||
				((retVal == CPL_ERROR_CONTINUE) &&
						(((pos > 2042) && (pos < ERIS_IFU_DETECTOR_SIZE_X-ERIS_IFU_DETECTOR_BP_BORDER)) ||
								((pos < 7) && (pos >= ERIS_IFU_DETECTOR_BP_BORDER)))))
		{
			// retVal is perhaps bad, but fit is close enough for continuing
			if (retVal == CPL_ERROR_CONTINUE) {
				cpl_errorstate_set(prestate);
				retVal = CPL_ERROR_NONE;
			}

			gaussPar->x0     = pos;
			gaussPar->sigma  = sigma;
			gaussPar->area   = area;
			gaussPar->offset = offset;
			gaussPar->peak   = area / sqrt(2 * CPL_MATH_PI * sigma * sigma);
			gaussPar->mse    = mserr;

		} else {
			gaussPar->x0     = 0.;
			gaussPar->sigma  = 0.;
			gaussPar->area   = 0.;
			gaussPar->offset = 0.;
			gaussPar->peak   = 0.;
			gaussPar->mse    = 0.;
		}

		if (retVal == CPL_ERROR_CONTINUE) {
			// Recover from "The iterative process did not converge" error
			cpl_errorstate_set(prestate);
			retVal = CPL_ERROR_NONE;

			const int decrement = 9;
			if ((range - decrement) > 2) {
				range -= decrement; // decrease range,
				// in doubt that some artifact is disturbing the fit
				cpl_vector_unwrap(yVec); yVec = NULL;
				eris_ifu_free_vector(&xVec);
				eris_ifu_free_vector(&yIn2);

				goto try_again_fit;
			}
		}

		if (range <= GAUSS_PAR_RANGE_MAX) {
			for (int ix=0; ix<range; ix++) {
				gaussPar->range     = range;
				gaussPar->xdata[ix] = pxVec[ix];
				gaussPar->ydata[ix] = pyVec[ix];
			}
		}
	}
	CATCH
	{
		gaussPar->x0     = 0.;
		gaussPar->sigma  = 0.;
		gaussPar->area   = 0.;
		gaussPar->offset = 0.;
		gaussPar->peak   = 0.;
		gaussPar->mse    = 0.;
		if (range <= GAUSS_PAR_RANGE_MAX) {
			for (int ix=0; ix<range; ix++) {
				gaussPar->range     = 0.;
				gaussPar->xdata[ix] = 0.;
				gaussPar->ydata[ix] = 0.;
			}
		}
	}
	eris_ifu_free_vector(&xVec);
	eris_ifu_free_vector(&yIn2);
	if (yVec != NULL) {
		cpl_vector_unwrap(yVec); yVec = NULL;
	}
	eris_check_error_code("eris_ifu_line_gauss_fit");
	return retVal;
}

/**
   takes horizontal chuck of an image, collapses it in y
 */
cpl_vector* eris_ifu_calc_centers_collapse_chunk(const cpl_image* img,
		int chunk_center,
		int height)
{
	cpl_size    nx              = 0,
			ny              = 0,
			chunk_top       = 0,
			chunk_bottom    = 0;
	double      *pcollapsed_img = NULL;
	cpl_vector  *tmp            = NULL,
			*collapsed      = NULL;
	cpl_image   *collapsed_img  = NULL;

	cpl_ensure(chunk_center > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
	cpl_ensure(height > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
	// height should be even
	cpl_ensure(height % 2 == 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
	cpl_ensure(img, CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{
		nx = cpl_image_get_size_x(img);
		ny = cpl_image_get_size_y(img);

		/* lower and upper limit of chunk in y */
		chunk_bottom = chunk_center - height/2;
		chunk_top    = ny - (chunk_center + height/2);

		/* collapse image in y and convert to vector */
		collapsed_img = cpl_image_collapse_median_create(img, 0,
				chunk_bottom,
				chunk_top);
		pcollapsed_img = cpl_image_get_data_double(collapsed_img);
		tmp = cpl_vector_wrap(nx, pcollapsed_img);
		collapsed = cpl_vector_duplicate(tmp);
	}
	CATCH
	{
		CATCH_MSGS();
		eris_ifu_free_vector(&collapsed);
	}

	cpl_vector_unwrap(tmp); tmp = NULL;
	eris_ifu_free_image(&collapsed_img);
	eris_check_error_code("eris_ifu_calc_centers_collapse_chunk");
	return collapsed;
}

/**
   collapses image in y and converts to vector
 */
cpl_vector* eris_ifu_image_collapse(const cpl_image* img)
{
	cpl_size    nx              = 0;
	double      *pcollapsed_img = NULL;
	cpl_vector  *tmp            = NULL,
			*collapsed      = NULL;
	cpl_image   *collapsed_img  = NULL;

	cpl_ensure(img, CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{
		nx = cpl_image_get_size_x(img);

		/* collapse image in y and convert to vector */
		collapsed_img = cpl_image_collapse_median_create(img, 0, 0, 0);
		pcollapsed_img = cpl_image_get_data_double(collapsed_img);
		tmp = cpl_vector_wrap(nx, pcollapsed_img);
		collapsed = cpl_vector_duplicate(tmp);
	}
	CATCH
	{
		CATCH_MSGS();
		eris_ifu_free_vector(&collapsed);
	}

	cpl_vector_unwrap(tmp); tmp = NULL;
	eris_ifu_free_image(&collapsed_img);
	eris_check_error_code("eris_ifu_image_collapse");
	return collapsed;
}

/**
 * @brief eris_ifu_dist_slitpos_gauss
 * @param profile_x       In y collapsed profile across detector
 * @param left_edge_pos   (return) position of left edge ()
 * @param right_edge_pos  (return) position of right edge
 * @param llx             offset in x for specific slitlet
 * @param productDepth    level of verbosity
 * @return                Error in case of error
 *
 * The function extracts a small area around the left ascending edge and another
 * around the descending right edge. The y-values are subtracted in order th get
 * a peak at the edge. The peak will be fitted using a gauss-funtion.
 */
cpl_error_code eris_ifu_slitpos_gauss(const cpl_image *profile_x,
		double     *left_edge_pos,
		double     *right_edge_pos,
		int        llx,
		int        productDepth)
{
	cpl_error_code  err         = CPL_ERROR_NONE;
	cpl_vector     *x_left      = NULL,
			*x_right     = NULL,
			*y_left      = NULL,
			*y_right     = NULL,
			*y_left2     = NULL,
			*y_right2    = NULL,
			*profile_x_v = NULL,
			*tmpVector   = NULL;
	int             s           = 0,
			s2          = 0;
	double          *px_left    = NULL,
			*px_right   = NULL,
			*py_left    = NULL,
			*py_right   = NULL,
			x0          = 0.,
			sigma       = 0.,
			area        = 0.,
			offset      = 0.,
			median      = 0.;
	const double    *pprofile_x = NULL;
	cpl_size        maxpos      = 0,
			minpos      = 0,
			inBetween   = 0;

	cpl_ensure_code(profile_x, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(left_edge_pos, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(right_edge_pos, CPL_ERROR_NULL_INPUT);

	TRY
	{
		s = (int) cpl_image_get_size_x(profile_x);
		s2 = (int)(s/2);

		// create x-Vector (profile_x goes into y :-)
        		// one for the left edge with 0-47
				// one for the right edge with 47-95
				// split profile_x into two and convert to vector
				x_left  = cpl_vector_new(s2);
		x_right = cpl_vector_new(s2);
		y_left  = cpl_vector_new(s2);
		y_right = cpl_vector_new(s2);

		px_left  = cpl_vector_get_data(x_left);
		px_right = cpl_vector_get_data(x_right);
		py_left  = cpl_vector_get_data(y_left);
		py_right = cpl_vector_get_data(y_right);

		//erw
		//        BRK_IF_NULL(
		//            pprofile_x = cpl_image_get_data_double_const(profile_x));
		tmpVector = cpl_vector_new_from_image_row(profile_x,1);
		profile_x_v = cpl_vector_filter_lowpass_create(
				tmpVector, CPL_LOWPASS_LINEAR, 2);
		eris_ifu_free_vector(&tmpVector);
		pprofile_x = cpl_vector_get_data(profile_x_v);

		for (int i = 0; i < s2; i++) {
			px_left[i] = i;
			// subtract left value from right value (we need a positive peak)
			py_left[i] = pprofile_x[i+1] - pprofile_x[i];
			px_right[i] = i+s2+1;
			// subtract right value from left value (we need a positive peak)
			py_right[i] = pprofile_x[i+s2-1] - pprofile_x[i+s2];
		}

		y_left2 = cpl_vector_filter_lowpass_create(y_left,
				CPL_LOWPASS_GAUSSIAN, 5);
		y_right2 = cpl_vector_filter_lowpass_create(y_right,
				CPL_LOWPASS_GAUSSIAN, 5);

		// check that left differences don't start with a negative peak
		if (cpl_vector_get_min(y_left2) < -100) {
			maxpos = cpl_vector_get_maxpos(y_left2);
			cpl_vector *tmpVec = cpl_vector_extract(y_left2,0,maxpos,1);
			minpos = cpl_vector_get_minpos(tmpVec);
			cpl_vector_delete(tmpVec);
			median = cpl_vector_get_median_const(y_left2);
			if (cpl_vector_get(y_left2,minpos < median-10.)) {
				//            if (maxpos > minpos) {
				// negative peak is to the left --> arc lamp from other slitlet
				inBetween = (cpl_size) (((double) (maxpos + minpos) / 2.) + .5);
				for (cpl_size vx = 0; vx < inBetween; vx++) {
					if (cpl_vector_get(y_left2, vx) < -10.) {
						cpl_vector_set(y_left2, vx, median);
					}
				}
				minpos = cpl_vector_get_minpos(y_left2);
				if (cpl_vector_get(y_left2,minpos < median-10.)) {
					inBetween = (cpl_size) (((double)(maxpos + minpos) / 2.) + .5);
					for (cpl_size vx = inBetween;
							vx < cpl_vector_get_size(y_left2); vx++) {
						if (cpl_vector_get(y_left2, vx) < -10.) {
							cpl_vector_set(y_left2, vx, median);
						}
					}

				}

				//BRK_WITH_ERROR(CPL_ERROR_EOL);
			} else {
				cpl_size vSize = cpl_vector_get_size(y_left2);
				for (cpl_size vx = maxpos; vx < vSize; vx++) {
					if (cpl_vector_get(y_left2, vx) < -10.) {
						cpl_vector_set(y_left2, vx, median);
					}
				}
			}
		}
		CHECK_ERROR_STATE();

		// check that right differences don't end with a negative peak
		if (cpl_vector_get_min(y_right2) < -100.) {
			maxpos = cpl_vector_get_maxpos(y_right2);
			minpos = cpl_vector_get_minpos(y_right2);
			median = cpl_vector_get_median_const(y_right2);
			if (maxpos < minpos) {
				// negative peak is to the right--> arc lamp from other slitlet
				cpl_size vSize = cpl_vector_get_size(y_right2);
				inBetween = (cpl_size) (((double) (maxpos + minpos) / 2.) + .5);
				for (cpl_size vx = inBetween; vx < vSize; vx++) {
					if (cpl_vector_get(y_left2, vx) < -10.) {
						cpl_vector_set(y_left2, vx, median);
					}
				}
				BRK_WITH_ERROR(CPL_ERROR_EOL);
			} else {
				for (cpl_size vx=maxpos; vx<=0; vx--) {
					if (cpl_vector_get(y_right2, vx) < -10.) {
						cpl_vector_set(y_right2, vx, median);
					}
				}
			}
		}
		CHECK_ERROR_STATE();

		if (productDepth & 8) {
			cpl_propertylist *pl = NULL;
			eris_ifu_save_vector_dbg(x_left,
					"eris_ifu_distortion_dbg_slitpos_profile_left_xy.fits",
					CPL_IO_CREATE, pl);
			eris_ifu_save_vector_dbg(y_left,
					"eris_ifu_distortion_dbg_slitpos_profile_left_xy.fits",
					CPL_IO_EXTEND, pl);
			eris_ifu_save_vector_dbg(x_right,
					"eris_ifu_distortion_dbg_slitpos_profile_right_xy.fits",
					CPL_IO_CREATE, pl);
			eris_ifu_save_vector_dbg(y_right,
					"eris_ifu_distortion_dbg_slitpos_profile_right_xy.fits",
					CPL_IO_EXTEND, pl);
			eris_ifu_save_vector_dbg(y_left2,
					"eris_ifu_distortion_dbg_slitpos_profile_left_xy.fits",
					CPL_IO_EXTEND, pl);
			eris_ifu_save_vector_dbg(y_right2,
					"eris_ifu_distortion_dbg_slitpos_profile_right_xy.fits",
					CPL_IO_EXTEND, pl);
		}


		// fit gauss-function to left edge
		//        cpl_msg_debug(cpl_func, "    === FIT LEFT EDGE ======"
		//                                "=======================");
        err = eris_ifu_fit_gauss(x_left, y_left2, &x0, &sigma, &area, &offset);
        if (err != CPL_ERROR_NONE) {
            eris_ifu_free_vector(&profile_x_v);
            eris_ifu_free_vector(&tmpVector);
            eris_ifu_free_vector(&x_left);
            eris_ifu_free_vector(&x_right);
            eris_ifu_free_vector(&y_left);
            eris_ifu_free_vector(&y_right);
            eris_ifu_free_vector(&y_left2);
            eris_ifu_free_vector(&y_right2);
            return err;
        }

		if (sigma > 25) {
			BRK_WITH_ERROR(CPL_ERROR_EOL);
		}
		// correct half pixel because of subtraction of y-values at beginning
		*left_edge_pos = llx + x0 + 0.5;

		// fit gauss-function to right edge
		//        cpl_msg_debug(cpl_func, "    === FIT RIGHT EDGE ============================");
        err = eris_ifu_fit_gauss(x_right, y_right2, &x0, &sigma, &area, &offset);
        if (err != CPL_ERROR_NONE) {
            eris_ifu_free_vector(&profile_x_v);
            eris_ifu_free_vector(&tmpVector);
            eris_ifu_free_vector(&x_left);
            eris_ifu_free_vector(&x_right);
            eris_ifu_free_vector(&y_left);
            eris_ifu_free_vector(&y_right);
            eris_ifu_free_vector(&y_left2);
            eris_ifu_free_vector(&y_right2);
            return err;
        }

		if (sigma > 25) {
			BRK_WITH_ERROR(CPL_ERROR_EOL);
		}
		// correct half pixel because of subtraction of y-values at beginning
		*right_edge_pos = llx + x0 - 0.5;

		//        cpl_msg_debug(cpl_func, "left/right edge : %g / %g",
		//                      *left_edge_pos, *right_edge_pos);
	}
	CATCH
	{
		err = cpl_error_get_code();
		if (err != CPL_ERROR_EOL) {
			CATCH_MSGS();
		}
	}
	eris_ifu_free_vector(&profile_x_v);
	eris_ifu_free_vector(&tmpVector);
	eris_ifu_free_vector(&x_left);
	eris_ifu_free_vector(&x_right);
	eris_ifu_free_vector(&y_left);
	eris_ifu_free_vector(&y_right);
	eris_ifu_free_vector(&y_left2);
	eris_ifu_free_vector(&y_right2);
	eris_check_error_code("eris_ifu_slitpos_gauss");
	return err;
}

/**
 * @brief eris_ifu_bpm_correction
 * @param image       2D hdrl image to be corrected  
 * @param badPixelMaskImg Master bad pixel image 
 * @return                Error in case of error
 *
 * Correct the bad pixels of the image by taking the average of the 4 neighboring pixels 
 */

cpl_error_code eris_ifu_bpm_correction(hdrl_image *himg,
		hdrl_image *badPixelMaskImg)
{
	cpl_mask *m = cpl_mask_new(3,3); // 3x3 mask for filtering; use a cross pattern
    cpl_mask_set(m, 1,2, BAD_PIX);
    cpl_mask_set(m, 2,1, BAD_PIX);
    cpl_mask_set(m, 2,3, BAD_PIX);
    cpl_mask_set(m, 3,2, BAD_PIX);

	cpl_image *filterd_image = cpl_image_duplicate(hdrl_image_get_image_const(himg));
	cpl_image_filter_mask(filterd_image, hdrl_image_get_image_const(himg), m, CPL_FILTER_AVERAGE,
			CPL_BORDER_FILTER);

	cpl_image *filterd_error = cpl_image_duplicate(hdrl_image_get_error_const(himg));
	cpl_image *variance = cpl_image_duplicate(hdrl_image_get_error_const(himg));
	cpl_image_power(variance, 2.0);
	cpl_image_filter_mask(filterd_error, variance, m, CPL_FILTER_AVERAGE,
			CPL_BORDER_FILTER);
	cpl_image_delete(variance);

	cpl_image_divide_scalar(filterd_error, (double) cpl_mask_count(m));
	cpl_image_power(filterd_error, 0.5);

	if (badPixelMaskImg == NULL){
		cpl_image *badPixelMaskImg1 = cpl_image_new_from_mask(hdrl_image_get_mask_const(himg));
		cpl_image *badPixelMaskImg2 = cpl_image_cast(badPixelMaskImg1, CPL_TYPE_DOUBLE);
		cpl_image_delete(badPixelMaskImg1);

		badPixelMaskImg = hdrl_image_new(
				hdrl_image_get_size_x(himg),
				hdrl_image_get_size_y(himg));
		hdrl_image_insert(badPixelMaskImg,
				badPixelMaskImg2, NULL, 1, 1);

		cpl_image_delete(badPixelMaskImg2);
	}

	cpl_image *filtered_mask = cpl_image_duplicate(hdrl_image_get_image_const(badPixelMaskImg));

    cpl_mask_set(m, 2, 2, GOOD_PIX);
	cpl_image_filter_mask(filtered_mask, hdrl_image_get_image_const(badPixelMaskImg), m, CPL_FILTER_AVERAGE,
			CPL_BORDER_FILTER);


	// Get the places to be corrected
	double corr_thesh = 0.4;
	cpl_image_threshold(filtered_mask, corr_thesh, corr_thesh, 1.0, 0.0);
	cpl_image_multiply(filtered_mask, hdrl_image_get_image_const(badPixelMaskImg)); // To be corrected set to 1
	cpl_mask *corr_mask_t = cpl_mask_threshold_image_create(filtered_mask,-0.1, 0.1);
	cpl_mask_not(corr_mask_t);
	cpl_msg_info(__func__, "No. bad pixels to be corrected: %d", (int)cpl_mask_count(corr_mask_t));
	cpl_image_multiply(filterd_image, filtered_mask);
	cpl_image_multiply(filterd_error, filtered_mask);

	cpl_image_subtract_scalar(filtered_mask, 1);
	cpl_image_multiply_scalar(filtered_mask, -1); // To be kept: set to 1;
	cpl_mask *corr_mask = cpl_mask_threshold_image_create(filtered_mask,-0.1, 0.1);
	cpl_mask_not(corr_mask);
	cpl_msg_debug(__func__, "No. bad pixels before correction: %d", (int)cpl_image_count_rejected(hdrl_image_get_image(himg)));
	cpl_mask_and(hdrl_image_get_mask(himg), corr_mask);
	cpl_mask_and(cpl_image_get_bpm(hdrl_image_get_error(himg)), corr_mask);
	cpl_msg_debug(__func__, "No. bad pixels after correction: %d", (int)cpl_image_count_rejected(hdrl_image_get_image(himg)));

	cpl_image_multiply(hdrl_image_get_image(himg), filtered_mask);
	cpl_image_add(hdrl_image_get_image(himg), filterd_image);

	cpl_image_multiply(hdrl_image_get_error(himg), filtered_mask);
	cpl_image_add(hdrl_image_get_error(himg), filterd_error);

	cpl_image_multiply(hdrl_image_get_image(badPixelMaskImg), filtered_mask);
	cpl_mask_delete(m);
	cpl_image_delete(filtered_mask);
	cpl_image_delete(filterd_error);
	cpl_image_delete(filterd_image);
	cpl_mask_delete(corr_mask);
	cpl_mask_delete(corr_mask_t);
	eris_check_error_code("eris_ifu_bpm_correction");
	return cpl_error_get_code();
}

double eris_ifu_auto_derot_corr(double alt, double rot)
{
	double za = 90.0-alt;
	double offset = -0.000379*za + 0.000500*pow(za,2);
	double alt_scale = 0.0467*za - 0.000265*pow(za,2);
	double pi = 3.14159265359;
	double derot_corr = offset + alt_scale * cos((rot - 15.4)*pi/180);
	eris_check_error_code("eris_ifu_auto_derot_corr");
	return derot_corr;
}
/**@}*/
