/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
 * @file eris_ifu_wavecal_static.c
 * @brief Wavelength Calibration Functions for ERIS IFU
 * @ingroup eris_ifu
 *
 * This module provides comprehensive wavelength calibration functionality for the ERIS
 * IFU pipeline. It implements the complete wavelength solution pipeline including:
 * - Arc lamp image loading and preprocessing
 * - Dark frame subtraction and image collapsing
 * - Reference arc line loading from calibration tables
 * - Initial wavelength fit generation (first fit)
 * - Line identification and Gaussian fitting
 * - 2D polynomial wavelength solution computation
 * - Coefficient smoothing across detector columns
 * - Wavelength error and spectral resolution calculation
 * - Arc image resampling to wavelength coordinates
 * - QC parameter computation and product saving
 *
 * The wavelength calibration uses arc lamp spectra (Ar, Ne, Kr, Xe) to establish
 * the wavelength-pixel relationship across the detector. A polynomial fit is performed
 * for each detector column, with coefficients smoothed spatially to reduce noise.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdbool.h>
#include <string.h>
#include <math.h>
#include <eris_utils.h>
#include "eris_ifu_error.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_functions.h"
#include "eris_ifu_dfs.h"
#include "eris_ifu_wavecal_static.h"
#include "eris_ifu_debug.h"

/**
 * @brief Load and preprocess arc lamp images for wavelength calibration
 *
 * This function loads raw arc lamp frames, identifies lamp states and DITs,
 * sorts frames by lamp configuration, and collapses multiple frames with
 * the same lamp state. Dark frames are automatically identified and subtracted.
 * Returns a list of preprocessed arc images ready for wavelength calibration.
 *
 * @param arcFrames        Input frameset containing arc and dark frames
 * @param exposureCorrectionMode  Exposure correction method (CDS, UTR, etc.)
 * @param arcImgCnt        Output: number of arc images produced
 * @param arcImages        Output: collapsed arc images (one per lamp state)
 * @param lampStates       Output: array of lamp state codes for each image
 * @param band             Output: instrument band (must be consistent across frames)
 * @param scale            Output: pre-optics scale (must be consistent)
 * @param instrument       Output: instrument identifier (SPIFFI/SPIFFIER)
 * @param saturation_threshold  Threshold for pixel saturation detection
 * @param qclog            Output: QC log table with frame statistics
 *
 * @return CPL_ERROR_NONE on success, error code otherwise
 *
 * @note Frames with different lamp states and DITs are processed separately
 * @note Dark frames (lamp state = 0) are used for dark subtraction
 * @note QC parameters include mean/median/max flux and number of saturated pixels
 */
cpl_error_code eris_ifu_wave_get_arc_images (
        cpl_frameset *arcFrames,
        int exposureCorrectionMode,
        int *arcImgCnt,
        hdrl_imagelist **arcImages,
        int **lampStates,
        ifsBand *band,
        ifsPreopticsScale *scale,
        ifsInstrument *instrument,
        double saturation_threshold,
        cpl_table** qclog)
{
    cpl_error_code  err = CPL_ERROR_NONE;
    hdrl_imagelist *arcImagesLocal = NULL;
    int *lampStatesLocal = NULL;
    hdrl_image *image = NULL;
    cpl_table *arcFrameTable = NULL;
    cpl_table *darkTable = NULL;
    cpl_table *litTable = NULL;
    const cpl_frame *frame = NULL;
    const char *filename;
    cpl_propertylist *header = NULL;
    int lamps = 0;
    float dit;
//    cpl_size nDarks;
    cpl_size nLits;
    ifsBand currBand;
    ifsPreopticsScale currScale;
    // four different lamps make a maximum number of 16 different states
    const int MaxLampStateSetSize = 16;

    TRY
    {
        ASSURE(arcFrames != NULL,
                            CPL_ERROR_NULL_INPUT,
                            "Null frameset (ARC frames)!");

        ASSURE(arcImages != NULL, CPL_ERROR_NULL_INPUT, "Null arcImages!");

        ASSURE(qclog != NULL, CPL_ERROR_NULL_INPUT, "Null qclog!");
        int frameCnt = (int) cpl_frameset_get_size(arcFrames);
        if (frameCnt == 0) {
            SET_ERROR_MSG(CPL_ERROR_NULL_INPUT,
                "input frameset (ARC frames) is empty");
        }
        BRK_IF_NULL(arcImagesLocal = hdrl_imagelist_new());
        lampStatesLocal = cpl_calloc(MaxLampStateSetSize, sizeof(int));
        *lampStates = lampStatesLocal;
        *band = UNDEFINED_BAND;
        *instrument = UNSET_INSTRUMENT;

        // Create table
        BRK_IF_NULL(
                arcFrameTable = cpl_table_new(frameCnt));
        cpl_table_new_column(arcFrameTable, ERIS_IFU_ARCFRAMR_IDX,
                CPL_TYPE_INT);
        cpl_table_new_column(arcFrameTable, ERIS_IFU_ARCFRAME_LAMP,
                CPL_TYPE_INT);
        cpl_table_new_column(arcFrameTable, ERIS_IFU_ARCFRAME_DIT,
                CPL_TYPE_FLOAT);
        cpl_table_new_column(arcFrameTable, ERIS_IFU_ARCFRAME_FILE,
                CPL_TYPE_STRING);
        CHECK_ERROR_STATE();

        for (int i=0 ; i<frameCnt ; i++) {
            frame = cpl_frameset_get_position_const(arcFrames, i);
            cpl_msg_info("wavecal","Tag is %s",cpl_frame_get_tag(frame));
            BRK_IF_NULL(
                filename = cpl_frame_get_filename(frame));
            BRK_IF_NULL(
                header = cpl_propertylist_load(filename, 0));
            lamps = eris_ifu_get_callamp_status(header);
            dit = eris_ifu_get_dit(header);
            if (*band == UNDEFINED_BAND) {
                *band = eris_ifu_get_band(header);
            } else {
                currBand = eris_ifu_get_band(header);
                if (*band != currBand) {
                    BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
                                    "different instrument band settings"
                                    " in ARC frames");
                }
            }
            if (*scale == UNDEFINED_SCALE) {
                *scale = eris_ifu_get_preopticsScale(header);
            } else {
                currScale = eris_ifu_get_preopticsScale(header);
                if (*scale != currScale) {
                    BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
                                    "different instrument pre-optics settings"
                                    " in ARC frames");
                }
            }
            if (*instrument == UNSET_INSTRUMENT) {
                *instrument = eris_ifu_get_instrument(header);
            }
            CHECK_ERROR_STATE();
            cpl_table_set_int(arcFrameTable, ERIS_IFU_ARCFRAMR_IDX, i, i);
            cpl_table_set_int(arcFrameTable, ERIS_IFU_ARCFRAME_LAMP, i, lamps);
            cpl_table_set_float(arcFrameTable, ERIS_IFU_ARCFRAME_DIT, i, dit);
            cpl_table_set_string(arcFrameTable, ERIS_IFU_ARCFRAME_FILE, i, filename);
            CHECK_ERROR_STATE();
            eris_ifu_free_propertylist(&header);
        }

        cpl_propertylist *refList = NULL;
        BRK_IF_NULL(
                refList = cpl_propertylist_new());
        BRK_IF_ERROR(
                cpl_propertylist_append_bool(refList,ERIS_IFU_ARCFRAME_LAMP,0));
        BRK_IF_ERROR(
                cpl_propertylist_append_bool(refList,ERIS_IFU_ARCFRAME_DIT,0));
        BRK_IF_ERROR(
                cpl_table_sort(arcFrameTable, refList));
        cpl_propertylist_delete(refList);
        cpl_msg_debug(cpl_func, "Input arc/dark frames sorted by lamps/dit");
        if (cpl_msg_get_level() <= CPL_MSG_DEBUG) {
            cpl_table_dump(arcFrameTable, 0, frameCnt, stdout);
        }
        CHECK_ERROR_STATE();

        BRK_IF_ERROR(
                cpl_table_unselect_all(arcFrameTable));
        cpl_size nDarks = cpl_table_or_selected_int(arcFrameTable, ERIS_IFU_ARCFRAME_LAMP,
            CPL_EQUAL_TO, 0);
        BRK_IF_NULL(
                darkTable = cpl_table_extract_selected(arcFrameTable));
        nLits = cpl_table_not_selected(arcFrameTable);
        BRK_IF_NULL(
                litTable = cpl_table_extract_selected(arcFrameTable));
        cpl_table_dump(darkTable, 0, nDarks, stdout);
        cpl_table_dump(litTable, 0, nLits, stdout);
        CHECK_ERROR_STATE();

        float nDit = -1.;
        float lDit;
        int nLampState = 0;
        int lLampState;
        int isNull;
        cpl_table* qc_log_tmp;
        *qclog = eris_qclog_init();
        for (int i=0 ; i<nLits ; i++) {
            lLampState = cpl_table_get_int(litTable, ERIS_IFU_ARCFRAME_LAMP, i, &isNull);
            lDit = cpl_table_get_float(litTable, ERIS_IFU_ARCFRAME_DIT, i, &isNull);
            if (lDit != nDit || lLampState != nLampState) {
                nLampState = lLampState;
                nDit = lDit;
                char *lampString = eris_ifu_get_lampString(nLampState);
                cpl_msg_info(cpl_func,"New lamp/dit state: %d (%s) %f",
                    nLampState, lampString, nDit);
                eris_ifu_free_string(&lampString);
                BRK_IF_NULL(
                        image = eris_ifu_wave_collapse_arc_images(
                                darkTable, litTable, lLampState, lDit,
                                exposureCorrectionMode, saturation_threshold, &qc_log_tmp));
                BRK_IF_ERROR(
                        hdrl_imagelist_set(arcImagesLocal, image,
                            hdrl_imagelist_get_size(arcImagesLocal)));
                *lampStatesLocal = lLampState;
                lampStatesLocal++;
                
                cpl_size size = cpl_table_get_nrow(*qclog);
                cpl_table_insert(*qclog, qc_log_tmp, size);
                cpl_table_delete(qc_log_tmp);
            }
        }
        //hdrl_imagelist_delete(arcImagesLocal);
    }
    CATCH
    {
        err = cpl_error_get_code();
        eris_ifu_free_propertylist(&header);
        hdrl_imagelist_delete(arcImagesLocal);
    }
    eris_ifu_free_table(&arcFrameTable);
    eris_ifu_free_table(&darkTable);
    eris_ifu_free_table(&litTable);

    *arcImages = arcImagesLocal;
    *arcImgCnt = (int) hdrl_imagelist_get_size(arcImagesLocal);
    return err;
}

/**
 * @brief Collapse multiple arc images with dark subtraction
 *
 * This function processes arc lamp frames with a specific lamp state and DIT.
 * It collapses multiple frames using median (>=3 frames) or mean (1-2 frames)
 * combination. Dark frames with matching DIT are collapsed and subtracted.
 * QC parameters are computed for all frames and the final difference image.
 *
 * @param darkTable        Table of dark frame information
 * @param litTable         Table of illuminated arc frame information
 * @param lampState        Lamp state code to process
 * @param dit              DIT (detector integration time) to match
 * @param exposureCorrectionMode  Exposure correction method
 * @param threshold        Saturation threshold for pixel flagging
 * @param qclog            Output: QC log table with frame statistics
 *
 * @return Collapsed and dark-subtracted arc image, or NULL on error
 *
 * @note Uses median combination for 3+ frames, mean for 1-2 frames
 * @note QC parameters: mean flux, median flux, max flux, saturated pixel count
 */
hdrl_image *eris_ifu_wave_collapse_arc_images(
        cpl_table *darkTable,
        cpl_table *litTable,
        int lampState,
        float dit,
        int exposureCorrectionMode,
        double threshold,
        cpl_table** qclog)
{
    hdrl_image *arcImage = NULL;
    hdrl_image *tmpImage = NULL;
    hdrl_image *litImage = NULL;
    hdrl_image *darkImage = NULL;
    hdrl_imagelist *litImageList = NULL;
    hdrl_imagelist *darkImageList = NULL;
    cpl_image *contribMap = NULL;
    char* name = NULL;
    cpl_size nDarks;
    cpl_size nLits;
    const char *filename;
    hdrl_value mean = {0., 0.};
    hdrl_value median = {0., 0.};
    double maximum = 0.;
    int npixsat = 0;
    //double threshold = 18000.;
    cpl_image* image = NULL;



    TRY
    {
        *qclog = NULL;
        ASSURE((darkTable != NULL) && (litTable != NULL),
                            CPL_ERROR_NULL_INPUT,
                            "One or both input tables are NULL");
         BRK_IF_ERROR(
                cpl_table_unselect_all(darkTable));
        BRK_IF_ERROR(
                cpl_table_unselect_all(litTable));
        nDarks = cpl_table_or_selected_float(darkTable, ERIS_IFU_ARCFRAME_DIT,
            CPL_EQUAL_TO, dit);
        nLits = cpl_table_or_selected_float(litTable, ERIS_IFU_ARCFRAME_DIT,
            CPL_EQUAL_TO, dit);
        nLits = cpl_table_and_selected_int(litTable, ERIS_IFU_ARCFRAME_LAMP,
            CPL_EQUAL_TO, lampState);
        CHECK_ERROR_STATE();
        ASSURE(nLits != 0,
                CPL_ERROR_DATA_NOT_FOUND,
                "No frames with the lamp-on pattern %d found", lampState);

        BRK_IF_NULL(
            litImageList = hdrl_imagelist_new());
        *qclog = eris_qclog_init();
        for (int ix=0; ix<cpl_table_get_nrow(litTable); ix++) {
            if (cpl_table_is_selected(litTable, ix)) {
                filename = cpl_table_get_string(litTable, ERIS_IFU_ARCFRAME_FILE, ix);
                BRK_IF_NULL(
                    tmpImage = eris_ifu_load_exposure_file(filename,
                            exposureCorrectionMode, NULL));

                mean = hdrl_image_get_mean(tmpImage);
                median = hdrl_image_get_median(tmpImage);
                maximum = cpl_image_get_max(hdrl_image_get_image(tmpImage));
                image = cpl_image_duplicate(hdrl_image_get_image(tmpImage));
                cpl_image_threshold(image, threshold, threshold, 0, 1);
                npixsat = (int) cpl_image_get_flux(image);

                name = cpl_sprintf("QC FRMON%d MEANFLUX",ix);
                eris_qclog_add_double(*qclog, name, mean.data, "[ADU] Average of flux");
                cpl_free(name); name = cpl_sprintf("QC FRMON%d MEDIANFLUX",ix);
                eris_qclog_add_double(*qclog, name, median.data, "[ADU] Median of flux");
                cpl_free(name); name = cpl_sprintf("QC FRMON%d MAXFLUX",ix);
                eris_qclog_add_double(*qclog, name, maximum, "[ADU] Max of flux");
                cpl_free(name); name = cpl_sprintf("QC FRMON%d NPIXSAT",ix);
                eris_qclog_add_int(*qclog, name, npixsat, "Number of saturated pixels");
                cpl_free(name); name = NULL;

                BRK_IF_ERROR(
                    hdrl_imagelist_set(litImageList, tmpImage,
                            hdrl_imagelist_get_size(litImageList)));
                cpl_image_delete(image);
            }
        }
        BRK_IF_NULL(
            darkImageList = hdrl_imagelist_new());
        for (int ix=0; ix<cpl_table_get_nrow(darkTable); ix++) {
            if (cpl_table_is_selected(darkTable, ix)) {
                filename = cpl_table_get_string(darkTable, ERIS_IFU_ARCFRAME_FILE, ix);
/*
                BRK_IF_NULL(
                    image = cpl_image_load(filename, CPL_TYPE_DOUBLE, 0, 0));
                BRK_IF_NULL(
                        tmpImage = eris_ifu_raw_hdrl_image(image));
*/
                BRK_IF_NULL(
                    tmpImage = eris_ifu_load_exposure_file(filename,
                            exposureCorrectionMode, NULL));

                mean = hdrl_image_get_mean(tmpImage);
                median = hdrl_image_get_median(tmpImage);
                maximum = cpl_image_get_max(hdrl_image_get_image(tmpImage));
                image = cpl_image_duplicate(hdrl_image_get_image(tmpImage));
                cpl_image_threshold(image, threshold, threshold, 0, 1);
                npixsat = (int) cpl_image_get_flux(image);

                name = cpl_sprintf("QC FRMOFF%d MEANFLUX",ix);
                eris_qclog_add_double(*qclog, name, mean.data, "[ADU] Average of flux");
                cpl_free(name); name = cpl_sprintf("QC FRMOFF%d MEDIANFLUX",ix);
                eris_qclog_add_double(*qclog, name, median.data, "[ADU] Median of flux");
                cpl_free(name); name = cpl_sprintf("QC FRMOFF%d MAXFLUX",ix);
                eris_qclog_add_double(*qclog, name, maximum, "[ADU] Max of flux");
                cpl_free(name); name = cpl_sprintf("QC FRMOFF%d NPIXSAT",ix);
                eris_qclog_add_int(*qclog, name, npixsat, "Number of saturated pixels");
                cpl_free(name); name = NULL;
                BRK_IF_ERROR(
                    hdrl_imagelist_set(darkImageList, tmpImage,
                            hdrl_imagelist_get_size(darkImageList)));
                cpl_image_delete(image);
            }
        }

        if (nLits == 1) {
            BRK_IF_NULL(
                litImage = hdrl_image_duplicate(
                    hdrl_imagelist_get(litImageList, 0)));
        } else if (nLits >= 3) {
            BRK_IF_ERROR(
                hdrl_imagelist_collapse_median(
                    litImageList, &litImage, &contribMap));
        } else {
            BRK_IF_ERROR(
                            hdrl_imagelist_collapse_mean(
                                litImageList, &litImage, &contribMap));
        }

        arcImage = litImage;
        litImage = NULL;
        if (nDarks != 0) {
            if (nDarks == 1) {
                BRK_IF_NULL(
                    darkImage = hdrl_image_duplicate(
                        hdrl_imagelist_get(darkImageList, 0)));
            } else if (nDarks >= 3) {
                BRK_IF_ERROR(
                    hdrl_imagelist_collapse_median(
                        darkImageList, &darkImage, &contribMap));
            } else {
                BRK_IF_ERROR(
                                hdrl_imagelist_collapse_mean(
                                    darkImageList, &darkImage, &contribMap));
            }
            BRK_IF_ERROR(
                    hdrl_image_sub_image(arcImage, darkImage));

            mean = hdrl_image_get_mean(arcImage);
            median = hdrl_image_get_median(arcImage);
            maximum = cpl_image_get_max(hdrl_image_get_image(arcImage));
            image = cpl_image_duplicate(hdrl_image_get_image(arcImage));
            cpl_image_threshold(image, threshold, threshold, 0, 1);
            npixsat = (int) cpl_image_get_flux(image);

            name = cpl_sprintf("QC FRMDIFF MEANFLUX");
            eris_qclog_add_double(*qclog, name, mean.data, "[ADU] Average of flux");
            cpl_free(name); name = cpl_sprintf("QC FRMDIFF MEDIANFLUX");
            eris_qclog_add_double(*qclog, name, median.data, "[ADU] Median of flux");
            cpl_free(name); name = cpl_sprintf("QC FRMDIFF MAXFLUX");
            eris_qclog_add_double(*qclog, name, maximum, "[ADU] Max of flux");
            cpl_free(name); name = cpl_sprintf("QC FRMDIFF NPIXSAT");
            eris_qclog_add_int(*qclog, name, npixsat, "Number of saturated pixels");
            cpl_free(name); name = NULL;
            cpl_image_delete(image);
        }

   }
     CATCH
     {
         cpl_free(name);
         hdrl_image_delete(litImage);
         cpl_table_delete(*qclog);
         *qclog = NULL;
         hdrl_image_delete(arcImage);
         arcImage = NULL;
     }
     eris_ifu_free_hdrl_imagelist(&litImageList);
     eris_ifu_free_hdrl_imagelist(&darkImageList);
     eris_ifu_free_hdrl_image(&darkImage);
     return arcImage;
}

//static cpl_error_code
//eris_compute_residuals_fit(cpl_table* tab, cpl_table* qclog)
//{
//    /* compute QC parameters */
//    cpl_table* xtab = NULL;
//    double mean = 0;
//    double median = 0;
//    double stddev = 0;
//    double min = 0;
//    double max = 0;
//    const char* key_name;
//    const char* key_help;

//    for (int sIdx = 0; sIdx < SLITLET_CNT; sIdx++) {
//    	cpl_table_and_selected_int(tab, "slitlet", CPL_EQUAL_TO, sIdx );
//    	xtab = cpl_table_extract_selected(tab);

//    	mean = cpl_table_get_column_mean(xtab, "wavelengthError");
//    	key_name = cpl_sprintf("QC WFIT SLICE%d MEAN",sIdx);
//    	key_help = cpl_sprintf("[um] Mean error of wave fit for slice %d",sIdx);
//    	eris_qclog_add_double(qclog, key_name, mean, key_help);


//    	median = cpl_table_get_column_median(xtab, "wavelengthError");
//    	key_name = cpl_sprintf("QC WFIT SLICE%d MEDIAN",sIdx);
//    	key_help = cpl_sprintf("[um] Median error of wave fit for slice %d",sIdx);
//    	eris_qclog_add_double(qclog, key_name, median, key_help);

//    	stddev = cpl_table_get_column_stdev(xtab, "wavelengthError");
//    	key_name = cpl_sprintf("QC WFIT SLICE%d STDEV",sIdx);
//    	key_help = cpl_sprintf("[um] Stdev error of wave fit for slice %d",sIdx);
//    	eris_qclog_add_double(qclog, key_name, stddev, key_help);


//    	min    = cpl_table_get_column_min(xtab, "wavelengthError");
//    	key_name = cpl_sprintf("QC WFIT SLICE%d MIN",sIdx);
//    	key_help = cpl_sprintf("[um] Minimum error of wave fit for slice %d",sIdx);
//    	eris_qclog_add_double(qclog, key_name, min, key_help);


//    	max    = cpl_table_get_column_max(xtab, "wavelengthError");
//    	key_name = cpl_sprintf("QC WFIT SLICE%d MAX",sIdx);
//    	key_help = cpl_sprintf("[um] Maximum error of wave fit for slice %d",sIdx);
//    	eris_qclog_add_double(qclog, key_name, max, key_help);

//    	cpl_table_select_all(tab);
//    	cpl_table_delete(xtab);

//    }
//    cpl_table_select_all(tab);
//	mean = cpl_table_get_column_mean(tab, "wavelengthError");
//	key_name = cpl_sprintf("QC WFIT MEAN");
//	key_help = cpl_sprintf("[um] Mean error of wave fit");
//	eris_qclog_add_double(qclog, key_name, mean, key_help);


//	median = cpl_table_get_column_median(tab, "wavelengthError");
//	key_name = cpl_sprintf("QC WFIT MEDIAN");
//	key_help = cpl_sprintf("[um] Median error of wave fit");
//	eris_qclog_add_double(qclog, key_name, median, key_help);


//	stddev = cpl_table_get_column_stdev(tab, "wavelengthError");
//	key_name = cpl_sprintf("QC WFIT STDEV");
//	key_help = cpl_sprintf("[um] Stdev error of wave fit");
//	eris_qclog_add_double(qclog, key_name, stddev, key_help);

//	min    = cpl_table_get_column_min(tab, "wavelengthError");
//	key_name = cpl_sprintf("QC WFIT MIN");
//	key_help = cpl_sprintf("[um] Minimum error of wave fit");
//	eris_qclog_add_double(qclog, key_name, min, key_help);

//	max    = cpl_table_get_column_max(tab, "wavelengthError");
//	key_name = cpl_sprintf("QC WFIT MAX");
//	key_help = cpl_sprintf("[um] Maximum error of wave fit");
//	eris_qclog_add_double(qclog, key_name, max, key_help);
//    cpl_table_save(qclog,NULL, NULL, "qclog.fits", CPL_IO_CREATE);

//	return cpl_error_get_code();
//}

/**
 * @brief Generate wavelength calibration image from arc lamp data
 *
 * This is the main wavelength calibration function. It processes arc lamp images
 * to create a 2D wavelength calibration image where each pixel value represents
 * the wavelength at that detector position. The processing includes:
 * - Loading reference arc line wavelengths
 * - Collapsing spectra for each slitlet
 * - Computing initial wavelength fit (first fit)
 * - Fitting all reference lines in each detector column
 * - Smoothing polynomial coefficients spatially
 * - Computing wavelength errors and spectral resolution
 * - Saving resampled arc images and fitting tables
 *
 * @param arcImagesCnt     Number of arc images to process
 * @param arcImages        List of preprocessed arc images
 * @param lampStates       Array of lamp state codes for each image
 * @param band             Instrument band
 * @param instrument       Instrument identifier (SPIFFI/SPIFFIER)
 * @param waveSetup        Wavelength setup parameters (ranges, sigma)
 * @param refLineTableFileName  Path to reference line table
 * @param firstFitTableFileName Path to first fit table
 * @param slitPos          Slitlet positions (if NULL, uses default)
 * @param tables           Structure containing fitting tables
 * @param productDepth     Bitmask for intermediate product saving
 * @param fs               Frameset for product headers
 * @param parlist          Parameter list for product headers
 * @param qclog            QC log table
 *
 * @return 2D wavelength calibration image, or NULL on error
 *
 * @note The output image has dimensions ERIS_IFU_DETECTOR_SIZE_X × ERIS_IFU_DETECTOR_SIZE_Y
 * @note Each pixel value is the wavelength in micrometers at that position
 * @note QC parameters include central wavelength and its error
 */
cpl_image * eris_ifu_wave_get_calImg(
        int arcImagesCnt,
        hdrl_imagelist *arcImages,
        int *lampStates,
        ifsBand band,
        ifsInstrument instrument,
        struct waveSetupStruct waveSetup,
        const char* refLineTableFileName,
        const char* firstFitTableFileName,
        cpl_bivector *slitPos,
        struct waveTablesStruct *tables,
        int productDepth,
        cpl_frameset *fs,
        const cpl_parameterlist* parlist,
        cpl_table* qclog)
{
    int arcImgCnt;
    cpl_image *waveCalImg = NULL;
    const cpl_image **dataImg = NULL;
    cpl_bivector **refLines = NULL;
    hdrl_image *arcImg = NULL;
    int lampStatus = 0;
    double *slitletStart;
    double *slitletEnd;
    cpl_image **collapsedSpectra = NULL;
    cpl_vector *collapsedSpectrum = NULL;
//    const double *collapsedSpectrumData;
    cpl_size nRows;
    int center;
    cpl_polynomial *firstFit = NULL;
    cpl_table *firstFitTable = NULL;
    cpl_polynomial *allFits[ERIS_IFU_DETECTOR_SIZE_Y];
    const hdrl_image *tmpImg;
    enum multiArcModes {SUPERIMPOSE, SEPARATE};
    int multiArcMode = SEPARATE;

    TRY
    {
        for (int ix=0; ix<ERIS_IFU_DETECTOR_SIZE_Y; ix++) {
            allFits[ix] = NULL;
        }


        // If there are more than one arc images two possibilities exist:
        //  a) superimpose all arc images as they were illuminated by a
        //     combination of all lamp states. This can be used if the
        //     different lamps intensities need different DITs but no severe
        //     line blending occurs.
        //  b) work on each arc image independent and combine the lambda/pos
        //     fits at the end.

        if (multiArcMode == SUPERIMPOSE) {
            arcImgCnt = 1;
        } else { // SEPARATE
            arcImgCnt = arcImagesCnt;
        }

        BRK_IF_NULL(
            dataImg = cpl_calloc(arcImgCnt, sizeof(cpl_image *)));
        BRK_IF_NULL(
            refLines = cpl_calloc(arcImgCnt, sizeof(cpl_bivector *)));
        BRK_IF_NULL(
            collapsedSpectra = cpl_calloc(arcImgCnt, sizeof(cpl_image *)));

        if (multiArcMode == SUPERIMPOSE) { //option a)
            BRK_IF_NULL(
                arcImg = hdrl_imagelist_get(arcImages, 0));
            lampStatus = lampStates[0];
            for (int aix = 1; aix< arcImgCnt; aix++) {
                lampStatus = lampStatus | lampStates[aix];
                BRK_IF_NULL(
                        tmpImg = hdrl_imagelist_get_const(arcImages, aix));
                BRK_IF_ERROR(
                        hdrl_image_add_image(arcImg, tmpImg));
            }
            lampStates[0] = lampStatus;

            BRK_IF_NULL(
                dataImg[0] = hdrl_image_get_image(arcImg));

        } else { // SEPARATE
            for (int ix = 0; ix < arcImgCnt; ix++) {
                BRK_IF_NULL(
                        tmpImg = hdrl_imagelist_get_const(arcImages, ix));
                BRK_IF_NULL(
                        dataImg[ix] = hdrl_image_get_image_const(tmpImg));
            }
        }

        char* pipe_id = cpl_sprintf("%s%s%s", PACKAGE, "/", PACKAGE_VERSION);
        cpl_propertylist *applist = cpl_propertylist_new();
        eris_pfits_put_qc(applist, qclog);
        cpl_propertylist_update_string(applist, CPL_DFS_PRO_CATG,
                         ERIS_IFU_PRO_WAVE_LAMP_STACKED);
        cpl_dfs_save_propertylist(fs, NULL, parlist, fs, NULL,
                                 "eris_ifu_wavecal", applist, NULL,
                                 pipe_id, ERIS_IFU_PRO_WAVE_LAMP_STACKED_FN);

        //BRK_IF_ERROR(
        //        cpl_image_save(NULL,ERIS_IFU_PRO_WAVE_LAMP_STACKED_FN,
        //            CPL_TYPE_DOUBLE,NULL,CPL_IO_CREATE));
        for (int ix = 0; ix < arcImgCnt; ix++) {
            BRK_IF_NULL(
                    refLines[ix] = eris_ifu_wave_get_refLines(
                        refLineTableFileName, instrument, lampStates[ix]));
            /* AMo: save as final product as requested by QC: PIPE-10189 */
            eris_ifu_save_image_dbg(dataImg[ix],
                                   ERIS_IFU_PRO_WAVE_LAMP_STACKED_FN,
                                   CPL_IO_EXTEND, NULL);
        }
        cpl_propertylist_delete(applist);
        cpl_free(pipe_id);
        nRows = cpl_image_get_size_y(dataImg[0]);
        if (slitPos == NULL) {
            BRK_IF_NULL(
                slitletStart = cpl_calloc(SLITLET_CNT, sizeof(double)));
            BRK_IF_NULL(
                slitletEnd = cpl_calloc(SLITLET_CNT, sizeof(double)));
            for (int mx = 0; mx <SLITLET_CNT; mx++){
                double tmp = (double) (SLITLET_WIDTH * mx);
                slitletStart[mx] = tmp;
                slitletEnd[mx] = tmp + SLITLET_WIDTH -1;
           }
        } else {
            BRK_IF_NULL(
                slitletStart = cpl_bivector_get_x_data(slitPos));
            BRK_IF_NULL(
                    slitletEnd = cpl_bivector_get_y_data(slitPos));
        }

        for (int aIdx = 0; aIdx < arcImgCnt; aIdx++) {
            for (int sIdx = 0; sIdx < SLITLET_CNT; sIdx++) {
                center = (int) (
                        slitletStart[sIdx] +
                        (slitletEnd[sIdx] - slitletStart[sIdx]) / 2.);
                collapsedSpectrum = eris_ifu_wave_collapse_slitlet(
                    dataImg[aIdx], center);
                eris_ifu_wave_save_spectrum(collapsedSpectra, aIdx,
                        collapsedSpectrum, sIdx, nRows,
                        lampStates[aIdx], band, instrument,
                        refLines[aIdx], productDepth);
                eris_ifu_free_vector(&collapsedSpectrum);
            }
        }

        BRK_IF_NULL(
                firstFitTable = eris_ifu_wave_get_firstFitTable(
                firstFitTableFileName, instrument,  band));
        cpl_msg_debug(cpl_func, "First fit table:");
        if (cpl_msg_get_level() <= CPL_MSG_DEBUG) {
            cpl_table_dump(firstFitTable, 0,
                cpl_table_get_nrow(firstFitTable), stdout);
        }

        int firstFitOffset;
        int firstFitMasterOffset;
        BRK_IF_NULL(
            firstFit = eris_ifu_get_first_fit(collapsedSpectra,
                lampStates, arcImgCnt, 16, 0, &firstFitMasterOffset,
                waveSetup, firstFitTable,
                tables->slitletFitting, tables->slitletCoeff));
        cpl_msg_info(__func__,"FirstFit Master Offset %d", firstFitMasterOffset);

        for (int sIdx = 0; sIdx < SLITLET_CNT; sIdx++) {
                cpl_polynomial_delete(firstFit);
                BRK_IF_NULL(
                    firstFit = eris_ifu_get_first_fit(collapsedSpectra,
                        lampStates, arcImgCnt, sIdx, firstFitMasterOffset, &firstFitOffset,
                        waveSetup, firstFitTable,
                        tables->slitletFitting, tables->slitletCoeff));

                eris_ifu_fit_all_lines(REC_NAME_WAVECAL, sIdx, slitletStart, slitletEnd,
                        arcImgCnt, dataImg, refLines,
                        waveSetup, firstFit, allFits,
                        tables->columnFitting, tables->columnCoeffRaw,
                        tables->columnCoeffSmoothed, tables->smoothingCoeff);

        }
        cpl_free(slitletEnd);
        cpl_free(slitletStart);
        BRK_IF_NULL(firstFitTable);

        // create wave cal image
        BRK_IF_NULL(
            waveCalImg = cpl_image_new(ERIS_IFU_DETECTOR_SIZE_X,
                ERIS_IFU_DETECTOR_SIZE_Y, CPL_TYPE_DOUBLE));
        cpl_image* waveCalErr = cpl_image_new(ERIS_IFU_DETECTOR_SIZE_X,
                       ERIS_IFU_DETECTOR_SIZE_Y, CPL_TYPE_DOUBLE);
        double errs = 0;
        double wave = 0;
        for (int col=0; col<ERIS_IFU_DETECTOR_SIZE_X; col++) {
            for (int row=0; row<ERIS_IFU_DETECTOR_SIZE_Y; row++) {
            	wave = cpl_polynomial_eval_1d(allFits[col], row, &errs);
                BRK_IF_ERROR(
                    cpl_image_set(waveCalImg, col+1, row+1, wave));
                cpl_image_set(waveCalErr, col+1, row+1, errs);
            }
        }

        cpl_image* img = cpl_image_extract(waveCalImg,1,1024,2048,1024);

        cpl_image* err = cpl_image_extract(waveCalErr,1,1024,2048,1024);

        double wcen = cpl_image_get_mean(img);
        double werr = fabs(cpl_image_get_mean(err));
        cpl_msg_warning(cpl_func,"Central wave: %g, %g", wcen, werr);
        cpl_image_delete(img);
        cpl_image_delete(err);
        char* key_name = NULL;
        char* key_help = NULL;
        key_name = cpl_sprintf("QC WCEN VALUE");
        key_help = cpl_sprintf("[um] Central wavelength value");
        eris_qclog_add_double(qclog, key_name, wcen, key_help);

        cpl_free(key_name);
        cpl_free(key_help);

        key_name = cpl_sprintf("QC WCEN ERR");
        key_help = cpl_sprintf("[um] Error on central wavelength value");
        eris_qclog_add_double(qclog, key_name, werr, key_help);
        cpl_free(key_name);
        cpl_free(key_help);


        //cpl_image_save(waveCalErr,"waveCalErr.fits",CPL_TYPE_DOUBLE,NULL,CPL_IO_DEFAULT);
        cpl_image_delete(waveCalErr);
        /* AMo: save as final product as requested by QC: PIPE-10189 */
        eris_ifu_save_resampled_arc_images(
        		arcImgCnt,
				dataImg,
				waveCalImg,
				band,
				lampStates, fs, parlist);

        // calculate position errors and store them in FIT table
        BRK_IF_ERROR(
                eris_ifu_wave_pos_err(allFits, tables->columnFitting, band));
          /* TODO: temporarily commented out as seems same infor is computed elsewhere
            eris_compute_residuals_fit(tables->columnFitting,qclog);
          */
    }
    CATCH
    {
        waveCalImg = NULL;
    }
    for (int ix=0; ix<ERIS_IFU_DETECTOR_SIZE_Y; ix++) {
        eris_ifu_free_polynomial(&allFits[ix]);
    }
    if (dataImg != NULL) {
        cpl_free(dataImg);
    }
    if (refLines != NULL) {
        for (int ix=0; ix < arcImgCnt; ix++) {
            eris_ifu_free_bivector(&refLines[ix]);
        }
        cpl_free(refLines);
    }
    if (collapsedSpectra != NULL) {
        for (int ix=0; ix < arcImgCnt; ix++) {
            eris_ifu_free_image(&collapsedSpectra[ix]);
        }
        cpl_free(collapsedSpectra);
    }
    eris_ifu_free_vector(&collapsedSpectrum);
    eris_ifu_free_polynomial(&firstFit);
    eris_ifu_free_table(&firstFitTable);
    return waveCalImg;
}

/**
 * @brief Load reference arc line wavelengths for specified lamp configuration
 *
 * Loads reference arc line wavelengths and intensities from a FITS table for
 * the specified instrument and lamp state. The table contains multiple extensions,
 * one for each instrument/lamp combination. Lines marked as ignored are skipped.
 * Wavelengths in nanometers are automatically converted to micrometers.
 *
 * @param refLineTableFileName  Path to reference line FITS table
 * @param instrument           Instrument identifier (SPIFFI/SPIFFIER)
 * @param lampState            Lamp state code (combination of AR, NE, KR, XE lamps)
 *
 * @return Bivector containing wavelengths (x) and intensities (y), or NULL on error
 *
 * @note Wavelengths > 100 are assumed to be in nm and converted to µm
 * @note Lines with ignored flag = 1 are excluded
 */
cpl_bivector *eris_ifu_wave_get_refLines(
        const char * refLineTableFileName,
        ifsInstrument instrument,
        int lampState)
{
    cpl_bivector *refLines = NULL;
    cpl_propertylist *header = NULL;
    cpl_size exCnt = 0;
    const char *instrumentHdr;
    const char *lampHdr;
    const char *instrString;
    char lampString[9] = "";
    cpl_vector *wavelength = NULL;
    cpl_vector *intensity = NULL;
    cpl_table *refLineTable = NULL;
    cpl_size nRows;
    int ignored;
    int vx=0;
    double tmp;

    cpl_ensure(refLineTableFileName, CPL_ERROR_NULL_INPUT, NULL);
    if (access(refLineTableFileName, F_OK)) {
        	            cpl_msg_error(cpl_func, "File %s was not found",
        	                            refLineTableFileName);
        	            cpl_error_set(cpl_func, CPL_ERROR_FILE_NOT_FOUND);
        	            cpl_ensure(CPL_FALSE, CPL_ERROR_FILE_NOT_FOUND, NULL);
    }
    TRY
    {
        switch(instrument) {
            case SPIFFIER:
                instrString = ERIS_IFU_REFTABLE_SPIFFIER_VAL;
                break;
            case SPIFFI:
                instrString = ERIS_IFU_REFTABLE_SPIFFI_VAL;
                break;
            default:
                instrString = "";
                break;
        }
        if ((lampState & AR_LAMP) != 0) {
            strcat(lampString, ERIS_IFU_REFTABLE_ARGON_VAL);
        }
        if ((lampState & NE_LAMP) != 0) {
            strcat(lampString, ERIS_IFU_REFTABLE_NEON_VAL);
        }
        if ((lampState & KR_LAMP) != 0) {
            strcat(lampString, ERIS_IFU_REFTABLE_KRYPTON_VAL);
        }
        if ((lampState & XE_LAMP) != 0) {
            strcat(lampString, ERIS_IFU_REFTABLE_XEON_VAL);
        }
        exCnt = cpl_fits_count_extensions(refLineTableFileName);
        CHECK_ERROR_STATE();
        for (int ex=1; ex<exCnt+1; ex++) {
            BRK_IF_NULL(
                header = cpl_propertylist_load(refLineTableFileName, ex));
            BRK_IF_NULL(
                instrumentHdr = cpl_propertylist_get_string(
                    header, ERIS_IFU_REFTABLE_INSTR_HDR));
            BRK_IF_NULL(
                lampHdr = cpl_propertylist_get_string(
                    header, ERIS_IFU_REFTABLE_LAMPS_HDR));
            if ((strcmp(instrumentHdr, instrString) == 0) &&
                (strcmp(lampHdr, lampString) == 0)) {

                BRK_IF_NULL(
                    refLineTable = cpl_table_load(refLineTableFileName, ex, 0));
                nRows = cpl_table_get_nrow(refLineTable);
                BRK_IF_NULL(
                    wavelength = cpl_vector_new(nRows));
                BRK_IF_NULL(
                    intensity = cpl_vector_new(nRows));
                for (int rx=0; rx<nRows; rx++) {
                    ignored = cpl_table_get_int(
                        refLineTable, ERIS_IFU_REFTABLE_IGNORED_COLUMN,
                        rx, NULL);
                    if (ignored == 0) {
                        tmp = cpl_table_get_double(
                            refLineTable, ERIS_IFU_REFTABLE_LAMBDA_COLUMN, rx,
                            NULL);
                        CHECK_ERROR_STATE();
                        if (tmp > 100.) {
                            tmp /= 1000.;
                        }
                        BRK_IF_ERROR(
                            cpl_vector_set(wavelength, vx, tmp));

                        tmp = cpl_table_get_double(
                            refLineTable, ERIS_IFU_REFTABLE_INTENSITY_COLUMN,
                            rx, NULL);
                        CHECK_ERROR_STATE();
                        BRK_IF_ERROR(
                            cpl_vector_set(intensity, vx, tmp));

                        vx++;
                    }
                }
                eris_ifu_free_propertylist(&header);
                break;
            }
            eris_ifu_free_propertylist(&header);
        }
        if (wavelength == NULL) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
                "There is no extension in reference lines file %s "
                "for instrument %s and lamps %s",
                refLineTableFileName, instrString, lampString);

        }
        if (vx > 0) {
            BRK_IF_NULL(
                refLines = cpl_bivector_wrap_vectors(
                    cpl_vector_extract(wavelength, 0, vx-1, 1),
                    cpl_vector_extract(intensity, 0, vx-1, 1)));
        }
    }
    CATCH
    {
        refLines = NULL;
        eris_ifu_free_propertylist(&header);
    }
    eris_ifu_free_vector(&wavelength);
    eris_ifu_free_vector(&intensity);
    eris_ifu_free_table(&refLineTable);

    return refLines;
}

/**
 * @brief Collapse slitlet spatially to create 1D spectrum
 *
 * Extracts a 1D spectrum from an arc image by collapsing a slitlet in the
 * spatial direction. Uses a median of pixels within ±10 pixels of the
 * specified center column to reduce noise while preserving line shapes.
 *
 * @param dataImg  Input arc lamp image
 * @param center   Center column of slitlet to collapse
 *
 * @return 1D spectrum vector (length = image height), or NULL on error
 *
 * @note Uses 21-pixel wide window (center ± 10 pixels)
 * @note Invalid pixels are replaced with zero before median calculation
 */
cpl_vector * eris_ifu_wave_collapse_slitlet(const cpl_image *dataImg,
                                            int center)
{
    cpl_vector *spec = NULL;
    cpl_size nRows;
    cpl_vector *slitletRow = NULL;
    double *slitletRowData = NULL;
    int isValid;
    int startCol;
    int endCol;
    double tmpData;

    const int halfWidth = 10;

    startCol = center - halfWidth;
    endCol = center + halfWidth;
    TRY
    {
        nRows = cpl_image_get_size_y(dataImg);
        BRK_IF_NULL(
            spec = cpl_vector_new(nRows));

        BRK_IF_NULL(
            slitletRow = cpl_vector_new(halfWidth * 2 + 1));
        BRK_IF_NULL(
            slitletRowData = cpl_vector_get_data(slitletRow));

        for (int row=0; row<nRows; row++) {
            for (int col=startCol; col<endCol+1; col++) {
                tmpData = cpl_image_get(dataImg, col+1, row+1, &isValid);
                CHECK_ERROR_STATE();
                if (isValid != 0) {
                    tmpData = 0;
                }
                slitletRowData[col-startCol] = tmpData;
            }
            tmpData = cpl_vector_get_median(slitletRow);
            BRK_IF_ERROR(
                    cpl_vector_set(spec, row, tmpData));
        }
    }
    CATCH
    {
        spec = NULL;
    }
    eris_ifu_free_vector(&slitletRow);
    return spec;
}

/**
 * @brief Generate initial wavelength fit for a slitlet
 *
 * Generates the first polynomial fit relating wavelength to pixel position for
 * a specific slitlet. This initial fit is used as a starting point for finding
 * lines in all detector columns. The function:
 * - Loads reference line positions from firstFitTable
 * - Performs Gaussian fits to locate lines in collapsed spectra
 * - Fits a polynomial (wavelength vs. position) to the found lines
 * - Computes position offset between expected and found line centers
 *
 * @param spectra          Array of collapsed spectra images (one per lamp state)
 * @param lampStates       Array of lamp state codes
 * @param spectrumCnt      Number of spectra
 * @param slitlet          Slitlet number to process
 * @param ffOffsetIn       Input pixel offset from previous fit
 * @param ffOffsetOut      Output: computed pixel offset for this slitlet
 * @param waveSetup        Wavelength setup parameters
 * @param firstFitTable    Table of reference line positions
 * @param fittingDumpTable Table to store fit results
 * @param coeffDumpTable   Table to store polynomial coefficients
 *
 * @return Polynomial fit (wavelength = f(pixel)), or NULL on error
 *
 * @note Uses FIRST_FIT_DEGREE polynomial order
 * @note Offset is computed as mean of (found - expected) positions
 */
cpl_polynomial * eris_ifu_get_first_fit(
        cpl_image** spectra,
        int *lampStates,
        int spectrumCnt,
        int slitlet,
        int ffOffsetIn,
        int *ffOffsetOut,
        struct waveSetupStruct waveSetup,
        cpl_table *firstFitTable,
        cpl_table *fittingDumpTable,
        cpl_table *coeffDumpTable)
{
    cpl_polynomial *fit = NULL;
    cpl_table *blockTable = NULL;
    cpl_table *tmpTable = NULL;
    cpl_size rowCnt;
    cpl_vector *spectrum = NULL;
    int isValid;
    int position;
    double wavelength;
    const int fitLength = 40;
    double xData[fitLength];
    double yData[fitLength];
    int fx;
    struct gaussParStruct gaussPar;
    cpl_size dumpRow;
    double ffOffsets = 0.;
    int ffOffsetCnt = 0;

    const int slitlet2collapsed[SLITLET_CNT] = {
             0, 15,  1, 16,  2, 17,  3, 18,  4, 19,  5, 20,  6, 21,  7,
            30, 31,
            22,  8, 23,  9, 24, 10, 25, 11, 26, 12, 27, 13, 28, 14, 29};

    TRY
    {
        BRK_IF_ERROR(
            cpl_table_unselect_all(firstFitTable));

        rowCnt = cpl_table_or_selected_int(firstFitTable,
            ERIS_IFU_FIRSTFIT_BLOCK, CPL_EQUAL_TO, slitlet_block[slitlet]);
        CHECK_ERROR_STATE();
        if (rowCnt < 1) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_DATA_NOT_FOUND,
                "no records found in firstFits table with block %d",
                slitlet_block[slitlet]);
        }
        blockTable = cpl_table_extract_selected(firstFitTable);

        fx = 0;

        for (int aIdx = 0; aIdx < spectrumCnt; aIdx++) {

            BRK_IF_ERROR(
                cpl_table_unselect_all(blockTable));
            char* lampString = eris_ifu_get_lampString(lampStates[aIdx]);
            rowCnt = cpl_table_or_selected_string(blockTable,
                ERIS_IFU_FIRSTFIT_LAMPS, CPL_EQUAL_TO, lampString);
            cpl_free(lampString);
            tmpTable = cpl_table_extract_selected(blockTable);
            dumpRow = cpl_table_get_nrow(fittingDumpTable);
            cpl_table_set_size(fittingDumpTable, dumpRow+rowCnt);

            if (slitlet < 15) {

            }
            BRK_IF_NULL(
                spectrum = cpl_vector_new_from_image_column(
                    spectra[aIdx], slitlet2collapsed[slitlet]+1));

            for (int rx = 0; rx < rowCnt; rx++) {
                position = cpl_table_get_int(tmpTable,
                    ERIS_IFU_FIRSTFIT_POSITION, rx, &isValid);
                wavelength = cpl_table_get_double(tmpTable,
                    ERIS_IFU_FIRSTFIT_WAVELENGTH, rx, &isValid);

                if (CPL_ERROR_NONE == eris_ifu_line_gauss_fit(
                    spectrum, position+ffOffsetIn, waveSetup.s_range, &gaussPar)) {
                    if ((gaussPar.x0 != 0.) &&
                            (gaussPar.sigma < 3. || gaussPar.peak > 8000.)){
                        if (fx < fitLength) {
                            if (wavelength > 100.) {
                                wavelength /= 1000.;
                            }
                            xData[fx] = wavelength;
                            yData[fx] = gaussPar.x0;
                            fx++;
                        }
                        ffOffsets += gaussPar.x0 - position;
                        ffOffsetCnt++;
                    } else {
                        int range = waveSetup.s_range;
                        int start = position-(range/2);
                        if (range <= GAUSS_PAR_RANGE_MAX) {
                            gaussPar.range = range;
                            for (int ix=0; ix<range; ix++) {
                                gaussPar.xdata[ix] = start+ix;
                                gaussPar.ydata[ix] = cpl_vector_get(spectrum,start+ix);
                            }
                        }

                        const char *lamp = cpl_table_get_string(tmpTable,
                                ERIS_IFU_FIRSTFIT_LAMPS, rx);
                        cpl_msg_warning(__func__,
                            "First line fit failed for "
                            "slitlet %d at row %d wavelength %.3f for lamp %s",
                            slitlet, position, wavelength, lamp);
                    }
                }
                BRK_IF_ERROR(
                        eris_ifu_wave_fill_fitting_table(fittingDumpTable,
                            (int) dumpRow,
                            slitlet, slitlet_block[slitlet], aIdx,
                            position, wavelength,
                            &gaussPar));
                dumpRow++;
            }
            eris_ifu_free_vector(&spectrum);
            eris_ifu_free_table(&tmpTable);
            *ffOffsetOut = (int) (ffOffsets/ffOffsetCnt + .5);
        }
        BRK_IF_NULL(
                fit = eris_ifu_1d_polynomial_fit(fx, xData, yData,
                    FIRST_FIT_DEGREE));
        eris_ifu_wave_fill_coeff_table(coeffDumpTable, slitlet, slitlet, fx,
                fit);
    }

    CATCH
    {
        fit = NULL;
    }
    eris_ifu_free_table(&tmpTable);
    eris_ifu_free_table(&blockTable);
    eris_ifu_free_vector(&spectrum);
    BRK_IF_ERROR(
        cpl_table_unselect_all(firstFitTable));
    return fit;
}

/**
 * @brief Fit all reference lines in a slitlet across detector columns
 *
 * This function processes a single slitlet, fitting all reference arc lines
 * in each detector column within the slitlet. For each column:
 * - Uses firstFit to predict line positions
 * - Performs Gaussian fits to locate actual line centers
 * - Applies quality filters (sigma, offset, position deviation)
 * - Fits polynomial wavelength solution for the column
 * - Stores fit results and coefficients in output tables
 *
 * After all columns are processed, polynomial coefficients are smoothed
 * spatially to reduce noise and improve wavelength solution stability.
 *
 * @param recipe_name      Recipe name (REC_NAME_WAVECAL or REC_NAME_DISTORTION)
 * @param sIdx             Slitlet index to process
 * @param slitletStart     Array of slitlet start positions
 * @param slitletEnd       Array of slitlet end positions
 * @param arcImgCnt        Number of arc images
 * @param dataImg          Array of arc lamp images
 * @param refLines         Array of reference line bivectors (per image)
 * @param waveSetup        Wavelength setup parameters
 * @param firstFit         Initial wavelength fit polynomial
 * @param allFits          Output: array of polynomial fits for all columns
 * @param dumpTable        Output: table of line fit results
 * @param columnCoeffRawTable      Output: raw polynomial coefficients
 * @param columnCoeffSmoothedTable Output: smoothed polynomial coefficients
 * @param smoothingCoeffTable      Output: smoothing polynomial coefficients
 *
 * @return CPL_ERROR_NONE on success, error code otherwise
 *
 * @note Column fit uses COLUMN_FIT_DEGREE polynomial order
 * @note Lines are rejected if: sigma >= waveSetup.sigma, |position deviation| > 5 pixels,
 *       or offset < -10 - peak/300
 */
cpl_error_code eris_ifu_fit_all_lines(
        const char *recipe_name,
        int sIdx,
        const double *slitletStart,
        const double *slitletEnd,
        int arcImgCnt,
        const cpl_image **dataImg,
        cpl_bivector **refLines,
        struct waveSetupStruct waveSetup,
        cpl_polynomial *firstFit,
        cpl_polynomial *allFits[ERIS_IFU_DETECTOR_SIZE_Y],
        cpl_table *dumpTable,
        cpl_table *columnCoeffRawTable,
        cpl_table *columnCoeffSmoothedTable,
        cpl_table *smoothingCoeffTable)
{
    cpl_error_code err = CPL_ERROR_NONE;
    int start, end;
    double s1, s2, e1, e2;
    cpl_vector *spectrum = NULL;
    double *refLinelambda;
    int firstGuessPos;
    cpl_size nRefLines;
    cpl_size nSpectrum;
    cpl_error_code fitErr;
    struct gaussParStruct gaussPar;
    cpl_size dumpRow = 0;
    const int fitLength = 2048;
    double xData[fitLength];
    double yData[fitLength];
    int fx;

    cpl_ensure_code(recipe_name, CPL_ERROR_NULL_INPUT);

    TRY
    {
        if (strcmp(recipe_name, REC_NAME_DISTORTION) == 0) {
            start = (int) slitletStart[sIdx];
            end = (int) slitletEnd[sIdx];
        } else {
            s2 = slitletStart[sIdx];
            if (sIdx == 0) {
                    s1 = -.5;
            } else {
                    s1 = slitletEnd[sIdx-1];
            }
            e1 = slitletEnd[sIdx];
            if (sIdx == SLITLET_CNT-1) {
                    e2 = 2047.;
            } else {
                    e2 = slitletStart[sIdx+1];
            }
    //        start = 1 + (int) floor(((s1 + s2) /2.0));
    //        end = (int) ((e1 + e2) /2.0);
            if (sIdx == 0) {
                start = 0;
            } else {
                start = (int) (((s1 + s2) / 2.0) + .5) + 1;
            }
            if (sIdx == SLITLET_CNT-1) {
                end = 2047;
            } else {
                end   = (int) (((e1 + e2) / 2.0) + .5);
            }
            cpl_msg_debug(cpl_func,
               "Slitlet %2d: start: %4d end: %4d (%7.2f - %7.2f ... %7.2f - %7.2f)",
               sIdx, start, end, s1, s2, e1, e2);
        }

        for (int cx = start; cx <= end; cx++) {
            fx = 0;
            for (int ax = 0; ax< arcImgCnt; ax++) {
                nRefLines = cpl_bivector_get_size(refLines[ax]);
                refLinelambda = cpl_bivector_get_x_data(refLines[ax]);

                dumpRow = cpl_table_get_nrow(dumpTable);
                cpl_table_set_size(dumpTable, dumpRow + nRefLines);

                BRK_IF_NULL(
                    spectrum = cpl_vector_new_from_image_column(dataImg[ax], cx+1));
                nSpectrum = cpl_vector_get_size(spectrum);
                for (int rx=0; rx<nRefLines; rx++) {
                    firstGuessPos = (int) (0.5 +
                            cpl_polynomial_eval_1d(firstFit, refLinelambda[rx], NULL));
                    if (firstGuessPos > 4+waveSetup.c_range/2 &&
                            firstGuessPos < (nSpectrum - 4 - waveSetup.c_range/2)) {
                        fitErr = eris_ifu_line_gauss_fit(
                            spectrum, firstGuessPos, waveSetup.c_range, &gaussPar);
                        if ((fitErr == CPL_ERROR_NONE) &&
                                (gaussPar.offset > -10. - gaussPar.peak / 300. ) &&
                                (gaussPar.sigma < waveSetup.sigma) &&
                                (fabs(gaussPar.x0 - firstGuessPos) < 5.01)) {
                            yData[fx] = refLinelambda[rx];
                            xData[fx] = gaussPar.x0;
                            fx++;
                        } else {
                            if (fitErr == CPL_ERROR_NONE) { //wierd looking fit
                                gaussPar.errorCode = -2; //for dump table only
                            }
                        }
                    } else {
                        // search outside spectra
                        gaussPar.errorCode = -1;
                        gaussPar.x0 = 0.;
                        gaussPar.sigma = 0.;
                        gaussPar.area = 0.;
                        gaussPar.offset = 0.;
                        gaussPar.peak = 0.;
                        gaussPar.mse = 0.;
                        gaussPar.range = 0;
                    }
                    if (gaussPar.errorCode != -1) {
                        BRK_IF_ERROR(
                            eris_ifu_wave_fill_fitting_table(dumpTable, (int) dumpRow,
                                sIdx, cx, ax, firstGuessPos, refLinelambda[rx],
                                &gaussPar));
                        dumpRow++;
                    }
                }
                eris_ifu_free_vector(&spectrum);
                cpl_table_set_size(dumpTable, dumpRow);
            }
            if (fx > COLUMN_FIT_DEGREE+1) {
            //if (fx > (/2)) {
                BRK_IF_NULL(
                    allFits[cx] = eris_ifu_1d_polynomial_fit(fx, xData, yData,
                        COLUMN_FIT_DEGREE));
            } else {
                cpl_msg_warning(cpl_func,
                    "Number of found lines (=%d) too small for column %d",
                    fx, cx);
                 //mark fit as failed
                allFits[cx] = cpl_polynomial_new(1);
                cpl_size tmpPower = 0;
                cpl_polynomial_set_coeff(allFits[cx], &tmpPower, 0.0);
             }
            eris_ifu_wave_fill_coeff_table(columnCoeffRawTable, cx, cx,
                    fx, allFits[cx]);

            eris_ifu_free_vector(&spectrum);
        }

        BRK_IF_ERROR(
                eris_ifu_wave_smooth_coeffs(sIdx, start, end,
                    COLUMN_FIT_DEGREE, allFits,
                    columnCoeffSmoothedTable, smoothingCoeffTable));
    }
    CATCH
    {
    }
    return err;

}

/**
 * @brief Smooth polynomial coefficients across detector columns
 *
 * Smooths wavelength polynomial coefficients spatially across columns within
 * a slitlet to reduce noise and improve wavelength solution stability. For
 * each polynomial coefficient:
 * - Sorts columns by central wavelength to handle edge cases
 * - Fits a smoothing polynomial through coefficient values
 * - Rejects 2 outliers from each end before fitting
 * - Replaces original coefficients with smoothed values
 *
 * This spatial smoothing assumes wavelength solutions vary slowly across
 * neighboring columns, which is generally valid for well-aligned spectrographs.
 *
 * @param sIdx             Slitlet index
 * @param start            First column in slitlet
 * @param end              Last column in slitlet
 * @param fitDegree        Degree of wavelength polynomial (per column)
 * @param allFits          Array of polynomial fits (modified in place)
 * @param columnCoeffSmoothedTable Output: smoothed polynomial coefficients
 * @param smoothingCoeffTable      Output: smoothing polynomial coefficients
 *
 * @return CPL_ERROR_NONE on success, error code otherwise
 *
 * @note Uses SMOOTH_FIT_DEGREE polynomial for smoothing
 * @note Removes 4 outliers total (2 from each end) before fitting
 */
cpl_error_code eris_ifu_wave_smooth_coeffs(
        int sIdx,
        int start,
        int end,
        int fitDegree,
        cpl_polynomial *allFits[ERIS_IFU_DETECTOR_SIZE_Y],
        cpl_table *columnCoeffSmoothedTable,
        cpl_table *smoothingCoeffTable)
{
    cpl_error_code retVal = CPL_ERROR_NONE;
    int nc;
    const int fitLength = 2048;
    double cData[fitLength];
    double xData[fitLength];
    double yData[fitLength];
    cpl_polynomial *smoothFit = NULL;
    cpl_size firstCoeff = 0;

    TRY
    {
        nc = 0;
        for (int cx = start; cx <= end; cx++) {
            if (cpl_polynomial_get_coeff(allFits[cx], &firstCoeff) != 0.) {
                cData[nc] = cx + .1;
                yData[nc] = cpl_polynomial_eval_1d( allFits[cx], 1024., NULL);
                nc++;
           }
        }
        //sort cData/yData pairs by yData values
        cpl_vector *xVec = cpl_vector_wrap(nc, cData);
        cpl_vector *yVec = cpl_vector_wrap(nc, yData);
        cpl_bivector *biVec = cpl_bivector_wrap_vectors(xVec, yVec);
        cpl_bivector_sort(biVec, biVec, CPL_SORT_ASCENDING, CPL_SORT_BY_Y);
        cpl_bivector_unwrap_vectors(biVec);
        cpl_vector_unwrap(xVec);
        cpl_vector_unwrap(yVec);

        for (cpl_size power = 0; power <= fitDegree; power++) {
            for (int ix = 0; ix < nc; ix++) {
                int cx = (int) cData[ix];
                xData[ix] = cx;
                yData[ix] = cpl_polynomial_get_coeff(allFits[cx], &power);
            }
            // reject outliers
            smoothFit = eris_ifu_1d_polynomial_fit(nc - 4, &xData[2], &yData[2],
                SMOOTH_FIT_DEGREE);
            eris_ifu_wave_fill_coeff_table(smoothingCoeffTable,
                    (COLUMN_FIT_DEGREE + 1) * sIdx + (int) power,
                    sIdx, (int) power, smoothFit);
            for (int cx = start; cx <= end; cx++) {
                cpl_polynomial_set_coeff(allFits[cx], &power,
                    cpl_polynomial_eval_1d(smoothFit, (double) cx, NULL));
            }
            eris_ifu_free_polynomial(&smoothFit);
        }

        for (int cx = start; cx <= end; cx++) {
            eris_ifu_wave_fill_coeff_table(columnCoeffSmoothedTable, cx, cx, 0,
                    allFits[cx]);
        }
    }
    CATCH {
    }

    return retVal;
}

/**
 * @brief Compute wavelength errors and spectral resolution
 *
 * For each successfully fitted reference line, this function:
 * - Evaluates the wavelength polynomial at the fitted line position
 * - Computes wavelength error (reference - fitted wavelength)
 * - Calculates spectral resolution R = λ/(FWHM × dispersion)
 * - Updates the fitting table with these derived quantities
 *
 * The spectral resolution uses the Gaussian sigma from line fits converted
 * to FWHM (2.355σ) and multiplied by the band dispersion.
 *
 * @param allFits   Array of wavelength polynomials for all columns
 * @param fitTable  Fitting table to update (modified in place)
 * @param band      Instrument band (for dispersion lookup)
 *
 * @return CPL_ERROR_NONE on success, error code otherwise
 *
 * @note Only processes rows with errorCode == 0 (successful fits)
 * @note Resolution = wavelength / (sigma × dispersion × 2.355)
 */
cpl_error_code eris_ifu_wave_pos_err(
        cpl_polynomial *allFits[ERIS_IFU_DETECTOR_SIZE_Y],
        cpl_table *fitTable,
		ifsBand band)
{
    cpl_error_code retVal = CPL_ERROR_NONE;
    int col;
    double val;
    double wave;
    double waveError;

    TRY
    {
        for (cpl_size row=0; row<cpl_table_get_nrow(fitTable); row++) {
            if (cpl_table_get_int(fitTable, ERIS_IFU_FITTABLE_ERRORCODE, row, NULL) != 0) {
                continue;
            }
            col = cpl_table_get_int(fitTable, "index", row, NULL);
            val = cpl_table_get_double(fitTable, "x0", row, NULL);
            wave = cpl_polynomial_eval_1d(allFits[col], val, NULL);
            waveError = cpl_table_get_double(fitTable, ERIS_IFU_FITTABLE_WAVELENGTH, row, NULL)-
                    wave;
            cpl_table_set_double(fitTable, "wavelengthFit", row, wave);
            cpl_table_set_double(fitTable, "wavelengthError", row, waveError);
            CHECK_ERROR_STATE();
        }
        double dispersion;
        eris_ifu_get_dispersion(band, &dispersion);
        double convFactor = 2. * sqrt(2. * log(2.));
        cpl_table_duplicate_column(fitTable,"resol",fitTable,"wavelengthFit");
        cpl_table_divide_columns(fitTable,"resol","sigma");
        cpl_table_divide_scalar(fitTable,"resol", dispersion);
        cpl_table_divide_scalar(fitTable,"resol", convFactor);

    }
    CATCH {
    }
    return retVal;

}

/**
 * @brief Save collapsed spectrum to product file
 *
 * Organizes collapsed slitlet spectra into an image for saving. Spectra are
 * arranged by slitlet block to match the detector geometry. On first call
 * (sIdx=0), initializes the output image. On last call (sIdx=SLITLET_CNT-1),
 * saves the complete image to FITS with appropriate headers including lamp
 * state, band, and instrument information.
 *
 * @param collapsedSpectra Array of spectrum images (one per arc image)
 * @param aIdx             Arc image index
 * @param collapsedSpectrum Current collapsed spectrum to add
 * @param sIdx             Slitlet index
 * @param nRows            Number of rows in spectrum
 * @param lampStatus       Lamp state code
 * @param band             Instrument band
 * @param instrument       Instrument identifier
 * @param refLines         Reference line bivector
 * @param productDepth     Bitmask controlling product saving (bit 0)
 *
 * @note Product saved only if productDepth & 1 is true
 * @note Reference line wavelengths and intensities saved as extensions
 */
void eris_ifu_wave_save_spectrum(
        cpl_image **collapsedSpectra,
        int aIdx,
        cpl_vector *collapsedSpectrum,
        int sIdx,
        cpl_size nRows,
        int lampStatus,
        ifsBand band,
        ifsInstrument instrument,
        cpl_bivector *refLines,
        int productDepth)
{
    const double *collapsedSpectrumData = NULL;
    static int slitletBlockIdx[SLITLET_BLOCK_CNT];
    int columnIdx = 0;

    TRY
    {
        if (aIdx == 0) {
        	/* TODO: One should save product in standard way.
        	 * But this is not yet working
        	 *
        	cpl_dfs_save_propertylist(fs, NULL, parlist, fs,
        	                          NULL, "eris_ifu_wavecal", NULL, NULL,
        	                          PACKAGE "/" PACKAGE_VERSION,
									  ERIS_IFU_PRO_WAVE_COLLAP_SPECTR_FN);
									  */
        	if((productDepth & 1)) {
                    cpl_image_save(NULL, ERIS_IFU_PRO_WAVE_COLLAP_SPECTR_FN,
                        CPL_TYPE_DOUBLE,NULL,CPL_IO_CREATE);
        	}
        }
        if (sIdx == 0) {
            for (int i=0; i<SLITLET_BLOCK_CNT; i++) {
                slitletBlockIdx[i] = 0;
            }
            BRK_IF_NULL(
                    collapsedSpectra[aIdx] = cpl_image_new(SLITLET_CNT, nRows,
                        CPL_TYPE_DOUBLE));
        }
        collapsedSpectrumData =
                cpl_vector_get_data_const(collapsedSpectrum);
        switch (slitlet_block[sIdx]) {
            case 0:
                columnIdx = slitletBlockIdx[0];
                break;
            case 1:
                columnIdx = slitletBlockIdx[1] +
                SLITLET_BLOCK0_CNT;
                break;
            case 2:
                columnIdx = slitletBlockIdx[2] +
                SLITLET_BLOCK0_CNT + SLITLET_BLOCK1_CNT;
                break;
            case 3:
                columnIdx = slitletBlockIdx[3] +
                SLITLET_BLOCK0_CNT + SLITLET_BLOCK1_CNT +SLITLET_BLOCK2_CNT;
                break;
            default:
                columnIdx = -1;
                break;
        }
        slitletBlockIdx[slitlet_block[sIdx]]++;
        for (int row=0; row<nRows; row++) {
            BRK_IF_ERROR(
                cpl_image_set(collapsedSpectra[aIdx], columnIdx+1, row+1,
                    collapsedSpectrumData[row]));
        }
        if ((sIdx >= SLITLET_CNT-1) && ((productDepth & 1) != 0)) {
            const char *instrumentString;
            char *lampString;
            if (instrument == SPIFFI) {
                instrumentString = "SINFONI";
            } else {
                instrumentString = "ERIS";
            }
            cpl_propertylist *hdr = cpl_propertylist_new();
            BRK_IF_ERROR(
                cpl_propertylist_update_string(hdr, CPL_DFS_PRO_CATG,
                    ERIS_IFU_PRO_WAVE_COLLAP_SPECTR ));
            cpl_propertylist_append_string(hdr,"INSTRUME", instrumentString);
            lampString = eris_ifu_get_lampString(lampStatus);
            cpl_propertylist_append_string(hdr,"LAMPS", lampString);
            cpl_propertylist_append_string(hdr,"BAND",
                    eris_ifu_get_bandString(band));
            cpl_image_save(collapsedSpectra[aIdx], ERIS_IFU_PRO_WAVE_COLLAP_SPECTR_FN,
                    CPL_TYPE_UNSPECIFIED, hdr, CPL_IO_EXTEND);
            eris_ifu_free_propertylist(&hdr);
            cpl_free(lampString);
            cpl_vector_save(cpl_bivector_get_x(refLines), ERIS_IFU_PRO_WAVE_COLLAP_SPECTR_FN,
                    CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
            cpl_vector_save(cpl_bivector_get_y(refLines), ERIS_IFU_PRO_WAVE_COLLAP_SPECTR_FN,
                    CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
        }
    }
    CATCH
    {

    }
    return;
}

/**
 * @brief Load first fit table for instrument and band
 *
 * Loads the table of initial fit parameters (reference line positions)
 * for the specified instrument and spectral band. The table contains
 * expected pixel positions of strong arc lines used to bootstrap the
 * wavelength calibration.
 *
 * @param firstFitTableFileName Path to first fit FITS table
 * @param instrument            Instrument identifier (SPIFFI/SPIFFIER)
 * @param band                  Instrument band
 *
 * @return Extracted table for specified instrument/band, or NULL on error
 *
 * @note Returns only rows matching instrument and band
 * @note Table must contain ERIS_IFU_FIRSTFIT_INSTRUMENT and _BAND columns
 */
cpl_table *eris_ifu_wave_get_firstFitTable(
        const char* firstFitTableFileName,
        ifsInstrument instrument,
//        int lampStatus,
        ifsBand band)
{
    cpl_table *firstFitTable = NULL;
    cpl_table *fullTable = NULL;
    cpl_size rowCnt;
    const char *instrumentString;
    const char* bandString;

    TRY
    {
        if (instrument == SPIFFI) {
            instrumentString = "SINFONI";
        } else {
            instrumentString = "ERIS";
        }
        bandString = eris_ifu_get_bandString(band);

        BRK_IF_NULL(
            fullTable = cpl_table_load(firstFitTableFileName, 1, 0));

        rowCnt = cpl_table_and_selected_string(fullTable,
            ERIS_IFU_FIRSTFIT_INSTRUMENT, CPL_EQUAL_TO, instrumentString);
        CHECK_ERROR_STATE();
        if (rowCnt < 1) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_DATA_NOT_FOUND,
                "no records found in firstFits table with instrument %s",
                instrumentString);
        }

        char *regExp = cpl_sprintf("^%s$", bandString);
        rowCnt = cpl_table_and_selected_string(fullTable,
                ERIS_IFU_FIRSTFIT_BAND, CPL_EQUAL_TO, regExp);
        CHECK_ERROR_STATE();
        eris_ifu_free_string(&regExp);
        if (rowCnt < 1) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_DATA_NOT_FOUND,
                "no records found in firstFits table with band %s",
                bandString);
        }
        firstFitTable = cpl_table_extract_selected(fullTable);
    }
    CATCH
    {
    }
    eris_ifu_free_table(&fullTable);

    return firstFitTable;
}

/**
 * @brief Get spectral dispersion for instrument band
 *
 * Returns the spectral dispersion (micrometers per pixel) for the
 * specified instrument band. Used for wavelength axis calibration
 * and spectral resolution calculations.
 *
 * @param band        Instrument band identifier
 * @param dispersion  Output: dispersion in µm/pixel
 *
 * @return CPL_ERROR_NONE on success, error code otherwise
 *
 * @note Dispersion values are hardcoded constants defined in header
 */
cpl_error_code
eris_ifu_get_dispersion(ifsBand band, double* dispersion)
{

    switch (band) {
    case J_SPIFFI:
        *dispersion = DISPERSION_J_SPIFFI;
        break;
    case H_SPIFFI:
        *dispersion = DISPERSION_H_SPIFFI;
        break;
    case K_SPIFFI:
        *dispersion = DISPERSION_K_SPIFFI;
        break;
    case HK_SPIFFI:
        *dispersion = DISPERSION_HK_SPIFFI;
        break;
    case J_LOW:
        *dispersion = DISPERSION_J_LOW;
        break;
    case J_SHORT:
        *dispersion = DISPERSION_J_SHORT;
        break;
    case J_MIDDLE:
        *dispersion = DISPERSION_J_MIDDLE;
        break;
    case J_LONG:
        *dispersion = DISPERSION_J_LONG;
        break;
    case H_LOW:
        *dispersion = DISPERSION_H_LOW;
        break;
    case H_SHORT:
        *dispersion = DISPERSION_H_SHORT;
        break;
    case H_MIDDLE:
        *dispersion = DISPERSION_H_MIDDLE;
        break;
    case H_LONG:
        *dispersion = DISPERSION_H_LONG;
        break;
    case K_LOW:
        *dispersion = DISPERSION_K_LOW;
        break;
    case K_SHORT:
        *dispersion = DISPERSION_K_SHORT;
        break;
    case K_MIDDLE:
        *dispersion = DISPERSION_K_MIDDLE;
        break;
    case K_LONG:
        *dispersion = DISPERSION_K_LONG;
        break;
    default:

        break;
    }

    return cpl_error_get_code();
}

cpl_error_code
eris_ifu_get_central_lambda(ifsBand band, double* centralLambda)
{

    switch (band) {
    case J_SPIFFI:
        *centralLambda = CENTRALLAMBDA_J_SPIFFI;
        break;
    case H_SPIFFI:
        *centralLambda = CENTRALLAMBDA_H_SPIFFI;
        break;
    case K_SPIFFI:
        *centralLambda = CENTRALLAMBDA_K_SPIFFI;
        break;
    case HK_SPIFFI:
        *centralLambda = CENTRALLAMBDA_HK_SPIFFI;
        break;
    case J_LOW:
        *centralLambda = CENTRALLAMBDA_J_LOW;
        break;
    case J_SHORT:
        *centralLambda = CENTRALLAMBDA_J_SHORT;
        break;
    case J_MIDDLE:
        *centralLambda = CENTRALLAMBDA_J_MIDDLE;
        break;
    case J_LONG:
        *centralLambda = CENTRALLAMBDA_J_LONG;
        break;
    case H_LOW:
        *centralLambda = CENTRALLAMBDA_H_LOW;
        break;
    case H_SHORT:
        *centralLambda = CENTRALLAMBDA_H_SHORT;
        break;
    case H_MIDDLE:
        *centralLambda = CENTRALLAMBDA_H_MIDDLE;
        break;
    case H_LONG:
        *centralLambda = CENTRALLAMBDA_H_LONG;
        break;
    case K_LOW:
        *centralLambda = CENTRALLAMBDA_K_LOW;
        break;
    case K_SHORT:
        *centralLambda = CENTRALLAMBDA_K_SHORT;
        break;
    case K_MIDDLE:
        *centralLambda = CENTRALLAMBDA_K_MIDDLE;
        break;
    case K_LONG:
        *centralLambda = CENTRALLAMBDA_K_LONG;
        break;
    default:

        break;
    }

    return cpl_error_get_code();
}


/**
 * @brief Resample arc image to uniform wavelength grid
 *
 * Resamples an arc lamp image from detector coordinates (pixel, row) to
 * wavelength coordinates (pixel, wavelength) using the wavelength calibration
 * image. Each detector column is interpolated from its non-uniform wavelength
 * sampling to a uniform grid with constant dispersion. This enables direct
 * visualization of arc line positions and wavelength coverage.
 *
 * The wavelength axis is constructed with:
 * - Constant dispersion from band parameters
 * - Center at central wavelength
 * - Range covering [minLambda, maxLambda] from wavecal image
 *
 * @param arcImg       Input arc lamp image (detector coordinates)
 * @param waveCalImg   Wavelength calibration image
 * @param band         Instrument band (for dispersion)
 * @param minLambda    Minimum wavelength in output grid (µm)
 * @param maxLambda    Maximum wavelength in output grid (µm)
 * @param plist        Property list to update with WCS keywords
 * @param axisNumber   FITS axis number for wavelength (usually 2)
 *
 * @return Resampled arc image in wavelength coordinates, or NULL on error
 *
 * @note Output image has same width as input, height determined by wavelength range
 * @note Uses 1D interpolation (degree -5) for resampling
 * @note Updates FITS WCS keywords (CTYPE, CUNIT, CRPIX, CRVAL, CDELT, CD matrix)
 * @note Columns that fail to resample are flagged but processing continues
 */
cpl_image *eris_ifu_wave_resampled_arc_image(
        const cpl_image *arcImg,
        const cpl_image *waveCalImg,
        ifsBand band,
        double minLambda,
        double maxLambda,
        cpl_propertylist *plist,
        int axisNumber)
{
    cpl_image *resampledImage = NULL;
    cpl_vector *arcImgCol = NULL;
    cpl_vector *calImgCol = NULL;
    double  dispersion =0;
    double  centralLambda = 0;
    double  *lambdaIn = NULL;
    double  *lambdaOut = NULL;
    double  *aduIn = NULL;
    double  *aduOut = NULL;
    cpl_size origNx;
    cpl_size origNy;
    char    *propName = NULL;

    TRY
    {
        ASSURE(arcImg != NULL,
            CPL_ERROR_NULL_INPUT, "no arc image provided");
        ASSURE(waveCalImg != NULL,
            CPL_ERROR_NULL_INPUT, "no wavelength calibration image provided");
        origNx = cpl_image_get_size_x(arcImg);
        origNy = cpl_image_get_size_y(arcImg);
        ASSURE(cpl_image_get_size_x(waveCalImg) == origNx,
            CPL_ERROR_INCOMPATIBLE_INPUT,
            "arc and wavecal image have different size");
        ASSURE(cpl_image_get_size_y(waveCalImg) == origNy,
            CPL_ERROR_INCOMPATIBLE_INPUT,
            "arc and wavecal image have different size");

        eris_ifu_get_dispersion(band, &dispersion);
        eris_ifu_get_central_lambda(band, &centralLambda);

        cpl_size nRows = (cpl_size) ((maxLambda - minLambda) / dispersion + .5);
        cpl_size centerRow = nRows / 2;
        double startLambda = centralLambda - dispersion * (double) centerRow;

        BRK_IF_NULL(
            resampledImage = cpl_image_new(origNx, nRows, CPL_TYPE_FLOAT));
        BRK_IF_NULL(
            lambdaIn = cpl_malloc(sizeof(double) * origNy));
        BRK_IF_NULL(
            aduIn = cpl_malloc(sizeof(double) * origNy));
        BRK_IF_NULL(
            lambdaOut = cpl_malloc(sizeof(double) * nRows));
        BRK_IF_NULL(
            aduOut = cpl_malloc(sizeof(double) * nRows));

        // populate wavelength axis for resampled image
        for (int rx=0; rx<nRows; rx++) {
            lambdaOut[rx] = startLambda + dispersion * (double) rx;
        }

        cpl_errorstate  errorState;
        bool errorHappend = false;

        for (int cx=0; cx<origNx; cx++) {
            // copy arc and wavecal columns and swap the columns
            arcImgCol = cpl_vector_new_from_image_column(arcImg, cx+1);
            calImgCol = cpl_vector_new_from_image_column(waveCalImg, cx+1);
            double *arcImgColData = cpl_vector_get_data(arcImgCol);
            double *calImgColData = cpl_vector_get_data(calImgCol);
            for (int rx=0; rx<origNy; rx++) {
                lambdaIn[rx] = calImgColData[origNy-rx-1];
                aduIn[rx] = arcImgColData[origNy-rx-1];
            }

            int nDegree = -5;
/*
            BRK_IF_ERROR(
                eris_ifu_1d_interpolation(lambdaIn, aduIn, (int) origNy,
                    lambdaOut, aduOut, (int) nRows, nDegree));
*/
            errorState = cpl_errorstate_get();
            if (eris_ifu_1d_interpolation(lambdaIn, aduIn, (int) origNy,
                    lambdaOut, aduOut, (int) nRows, nDegree)
                        != CPL_ERROR_NONE) {
                errorHappend = true;
                cpl_errorstate_set(errorState);
            }
            for (int rx = 0; rx < nRows; rx++) {
                cpl_image_set(resampledImage, cx + 1, rx + 1, aduOut[rx]);
            }
            eris_ifu_free_vector(&arcImgCol);
            eris_ifu_free_vector(&calImgCol);
        }
        if (errorHappend) {
            cpl_msg_error(__func__,
                "One or more columns of the arc images could not be resampled");
        }

        propName = cpl_sprintf("CTYPE%d", axisNumber);
        cpl_propertylist_update_string(plist, propName, "WAVE");
        eris_ifu_free_string(&propName);
        propName = cpl_sprintf("CUNIT%d", axisNumber);
        cpl_propertylist_update_string(plist, propName, "um");
        eris_ifu_free_string(&propName);
        propName = cpl_sprintf("CRPIX%d", axisNumber);
        cpl_propertylist_update_double(plist, propName, 1.);
        eris_ifu_free_string(&propName);
        propName = cpl_sprintf("CRVAL%d", axisNumber);
        cpl_propertylist_update_double(plist, propName, lambdaOut[0]);
        eris_ifu_free_string(&propName);
        propName = cpl_sprintf("CDELT%d", axisNumber);
        cpl_propertylist_update_double(plist, propName, dispersion);
        eris_ifu_free_string(&propName);
        for (int ix=1; ix<=axisNumber; ix++) {
            propName = cpl_sprintf("CD%d_%d", axisNumber, ix);
            if (ix == axisNumber) {
                cpl_propertylist_update_double(plist, propName, dispersion);
            } else {
                cpl_propertylist_update_double(plist, propName, 0.);
            }
            eris_ifu_free_string(&propName);
        }
        CHECK_ERROR_STATE();

    }
    CATCH
    {
        eris_ifu_free_vector(&arcImgCol);
        eris_ifu_free_vector(&calImgCol);
        resampledImage = NULL;
    }
    eris_ifu_free_double_array(&lambdaIn);
    eris_ifu_free_double_array(&aduIn);
    eris_ifu_free_double_array(&lambdaOut);
    eris_ifu_free_double_array(&aduOut);
    return resampledImage;
}

/**
 * @brief Save resampled arc images as FITS product
 *
 * Resamples all arc lamp images to wavelength coordinates and saves them
 * as a multi-extension FITS file. Each extension contains one resampled
 * arc image with complete WCS information. This product is useful for
 * verifying wavelength calibration quality and identifying problematic
 * spectral regions.
 *
 * @param arcImgCnt  Number of arc images
 * @param dataImg    Array of arc lamp images
 * @param waveCalImg Wavelength calibration image
 * @param band       Instrument band
 * @param lampStates Array of lamp state codes
 * @param frameset   Frameset for product header
 * @param parlist    Parameter list for product header
 *
 * @return CPL_ERROR_NONE on success, error code otherwise
 *
 * @note Saved to ERIS_IFU_PRO_WAVE_LAMP_RESAMPLED_FN
 * @note Each extension includes lamp state in LAMPS keyword
 */
cpl_error_code eris_ifu_save_resampled_arc_images(
        int arcImgCnt,
        const cpl_image **dataImg,
        const cpl_image *waveCalImg,
        ifsBand band,
        int *lampStates,
    cpl_frameset* frameset,
    const cpl_parameterlist* parlist) {

    cpl_error_code retVal = CPL_ERROR_NONE;
    cpl_propertylist *plist = NULL;
    cpl_image *resampledImg = NULL;
    cpl_image *tmpImg = NULL;
    double minLambda = 0.;
    double maxLambda = 0.;

    TRY
    {
        BRK_IF_NULL(
            plist = cpl_propertylist_new());
        BRK_IF_ERROR(
            cpl_propertylist_update_string(plist, CPL_DFS_PRO_CATG,
                    ERIS_IFU_PRO_WAVE_LAMP_RESAMPLED));

        /* AMO: not necessary as header is set later by cpl_dfs_save_image , else one duplicate products
        eris_setup_product_header(ERIS_IFU_PRO_WAVE_LAMP_RESAMPLED_FN,
                                  ERIS_IFU_PRO_WAVE_LAMP_RESAMPLED,
                  CPL_FRAME_TYPE_IMAGE, "eris_ifu_wavecal",
                      frameset, parlist, plist);
                      */
        BRK_IF_NULL(
            tmpImg = cpl_image_new(1, 1, CPL_TYPE_INT));
        /* Save ERIS_IFU_PRO_WAVE_LAMP_RESAMPLED_FN as final product (PIPE-10189) */
        char* pipe_id = cpl_sprintf("%s%s%s", PACKAGE, "/", PACKAGE_VERSION);
        cpl_dfs_save_image(frameset, NULL, parlist, frameset, NULL, tmpImg,
                               CPL_TYPE_UNSPECIFIED, REC_NAME_WAVECAL, plist, "RADECSYS",
                               pipe_id, ERIS_IFU_PRO_WAVE_LAMP_RESAMPLED_FN);
        cpl_free(pipe_id);
        maxLambda = cpl_image_get_max(waveCalImg);
        minLambda = cpl_image_get_min(waveCalImg);

        for (int ix = 0; ix < arcImgCnt; ix++) {

            // create resampled arc image
            BRK_IF_NULL(
                resampledImg =
                        eris_ifu_wave_resampled_arc_image(dataImg[ix],
                            waveCalImg, band, minLambda, maxLambda, plist, 2));

            cpl_propertylist_update_string(plist, "CTYPE1", "PIXEL");
            cpl_propertylist_update_string(plist, "CUNIT1", "PIXEL");
            cpl_propertylist_update_double(plist, "CRPIX1", 1.);
            cpl_propertylist_update_double(plist, "CRVAL1", 1.);
            cpl_propertylist_update_double(plist, "CDELT1", 1.);
            char* lamp_string = eris_ifu_get_lampString(lampStates[ix]);
            cpl_propertylist_update_string(plist, "LAMPS",lamp_string);
            cpl_free(lamp_string);
            eris_ifu_save_image_dbg(resampledImg,
            		ERIS_IFU_PRO_WAVE_LAMP_RESAMPLED_FN, CPL_IO_EXTEND,plist);
            eris_ifu_free_image(&resampledImg);
        }

    }
    CATCH
    {
        retVal = cpl_error_get_code();
        CATCH_MSG();
        CATCH_MSGS();
        RECOVER();
    }

    eris_ifu_free_image(&tmpImg);
    //eris_ifu_free_image(&resampledImg);
    eris_ifu_free_propertylist(&plist);

    return retVal;
}

/**
 * @brief Read wavelength setup parameters from configuration file
 *
 * Loads band-specific wavelength calibration parameters from a FITS table,
 * including search ranges for line fitting and expected line width (FWHM).
 * These parameters control the wavelength calibration algorithm behavior.
 *
 * @param filename   Path to wavelength setup FITS table
 * @param band       Instrument band to load parameters for
 * @param waveSetup  Output: structure filled with setup parameters
 *
 * @return CPL_ERROR_NONE on success, error code otherwise
 *
 * @note Parameters include: fwhm, s_range, c_range, ri_dispersion, ri_centrallambda
 * @note Sigma is computed from FWHM as: sigma = fwhm / (2 × sqrt(2 ln 2))
 * @note Table must contain exactly one row matching the band name
 */
cpl_error_code eris_ifu_read_wave_setup(
        const char* filename,
        ifsBand band,
        struct waveSetupStruct *waveSetup)
{
    cpl_error_code retVal = CPL_ERROR_NONE;
    cpl_table   *table = NULL;
    cpl_size    row;
    const char* bandName;
    const char* rowBandName;
    bool        found = FALSE;
    bool        invalidColumnFound = FALSE;
    int         n;

    TRY
    {
        /* Rough init in case table load fails */
        (void)memset(waveSetup, 0, sizeof(*waveSetup));
        bandName = eris_ifu_get_bandString(band);
        BRK_IF_NULL(
            table = cpl_table_load(filename, 1, 0));
        for (row = 0; row<cpl_table_get_nrow(table); row++) {
            rowBandName = cpl_table_get_string(table, "band", row);
            CHECK_ERROR_STATE();
            if (strcmp(bandName, rowBandName) == 0){
                if (found) {
                    BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
                        "There are more than row in the WAVE_SETUP with "
                        "the same band name %s", bandName);
                }
                found = TRUE;
                waveSetup->fwhm =
                        cpl_table_get_double(table, "fwhm", row, &n);
                if (n == 1) {
                    invalidColumnFound = TRUE;
                }
                waveSetup->s_range =
                        cpl_table_get_int(table, "s_range", row, &n);
                if (n == 1) {
                    invalidColumnFound = TRUE;
                }
                waveSetup->c_range =
                        cpl_table_get_int(table, "c_range", row, &n);
                if (n == 1) {
                    invalidColumnFound = TRUE;
                }
                waveSetup->ri_dispersion =
                        cpl_table_get_double(table, "ri_dispersion", row, &n);
                if (n == 1) {
                    invalidColumnFound = TRUE;
                }
                waveSetup->ri_centralLambda =
                        cpl_table_get_double(table, "ri_centrallambda", row,&n);
                CHECK_ERROR_STATE();
                double convFactor = 2. * sqrt(2. * log(2.));
                waveSetup->sigma = waveSetup->fwhm / convFactor;
            }
        }
        if (! found) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
                "There is no row in the WAVE_SETUP with the band name %s",
                bandName);
        }
        if (invalidColumnFound) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
                "There is an invalid column in the WAVE_SETUP "
                "with the band name %s",
                bandName);
        }
    }
    CATCH
    {
    }
    eris_ifu_free_table(&table);
    return retVal;
}

/**
 * @brief Initialize wavelength fitting tables
 *
 * Creates all tables needed to store wavelength calibration results:
 * - slitletFitting: Gaussian fit results for first fit lines
 * - columnFitting: Gaussian fit results for all column fits
 * - slitletCoeff: Polynomial coefficients for first fits
 * - columnCoeffRaw: Raw polynomial coefficients per column
 * - columnCoeffSmoothed: Smoothed polynomial coefficients
 * - smoothingCoeff: Coefficients of smoothing polynomials
 *
 * @param tables  Output: structure with pointers to created tables
 *
 * @return CPL_ERROR_NONE on success, error code otherwise
 *
 * @note All table pointers are initialized even on error (may be NULL)
 * @note Tables must be freed with eris_ifu_wave_free_tables()
 */
cpl_error_code eris_ifu_wave_init_tables(
        struct waveTablesStruct *tables)
{
	cpl_ensure_code(tables, CPL_ERROR_NULL_INPUT);
    cpl_error_code retVal = CPL_ERROR_NONE;

    TRY
    {
        tables->slitletFitting = NULL;
        tables->columnFitting = NULL;
        tables->slitletCoeff = NULL;
        tables->columnCoeffRaw = NULL;
        tables->columnCoeffSmoothed = NULL;
        tables->smoothingCoeff = NULL;

        tables->slitletFitting = eris_ifu_wave_create_fitting_table();
        tables->columnFitting = eris_ifu_wave_create_fitting_table();
        tables->slitletCoeff = eris_ifu_wave_create_coeff_table(
                     SLITLET_CNT, FIRST_FIT_DEGREE);
        tables->columnCoeffRaw = eris_ifu_wave_create_coeff_table(
                    ERIS_IFU_DETECTOR_SIZE_X, COLUMN_FIT_DEGREE);
        tables->columnCoeffSmoothed = eris_ifu_wave_create_coeff_table(
                    ERIS_IFU_DETECTOR_SIZE_X, COLUMN_FIT_DEGREE);
        tables->smoothingCoeff = eris_ifu_wave_create_coeff_table(
                    SLITLET_CNT * (COLUMN_FIT_DEGREE + 1), SMOOTH_FIT_DEGREE);
        CHECK_ERROR_STATE();

    }
    CATCH
    {
    }
    return retVal;
}

/**
 * @brief Create fitting table for line fit results
 *
 * Creates a table structure to store Gaussian fit results for arc lines.
 * The table includes columns for:
 * - Position information (slitlet, index/column, arc image index, position)
 * - Reference wavelength
 * - Fit results (x0, sigma, area, offset, mse, error code)
 * - Derived quantities (wavelength fit, wavelength error, resolution)
 * - Fit data (xdata, ydata arrays for plotting/debugging)
 *
 * @return New fitting table, or NULL on error
 *
 * @note Table is created with 0 rows; rows are added as needed
 * @note Array columns (xdata, ydata) have size GAUSS_PAR_RANGE_MAX
 */
cpl_table *eris_ifu_wave_create_fitting_table(void)
{

    cpl_table *table = NULL;

    TRY
    {
        BRK_IF_NULL(
            table = cpl_table_new(0));
        cpl_table_new_column(table, ERIS_IFU_FITTABLE_SLITLET, CPL_TYPE_INT);
        cpl_table_new_column(table, ERIS_IFU_FITTABLE_INDEX, CPL_TYPE_INT);
        cpl_table_new_column(table, "imgIdx", CPL_TYPE_INT);
        cpl_table_new_column(table, ERIS_IFU_FITTABLE_POSITION, CPL_TYPE_INT);
        cpl_table_new_column(table, ERIS_IFU_FITTABLE_WAVELENGTH, CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, ERIS_IFU_FITTABLE_ERRORCODE, CPL_TYPE_INT);
        cpl_table_new_column(table, "x0", CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, "sigma", CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, "area", CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, "offset", CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, "mse", CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, "wavelengthFit", CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, "wavelengthError", CPL_TYPE_DOUBLE);
        cpl_table_new_column(table, "range", CPL_TYPE_INT);
        cpl_table_new_column_array(table, "xdata", CPL_TYPE_DOUBLE,
                GAUSS_PAR_RANGE_MAX);
        cpl_table_new_column_array(table, "ydata", CPL_TYPE_DOUBLE,
                GAUSS_PAR_RANGE_MAX);
        CHECK_ERROR_STATE();
    }
    CATCH
    {

    }
    return table;

}

/**
 * @brief Fill fitting table row with Gaussian fit results
 *
 * Populates a single row of the fitting table with line fit information
 * including position, wavelength, Gaussian parameters, and fit data.
 *
 * @param table      Fitting table to update
 * @param row        Row index to fill
 * @param slitlet    Slitlet number
 * @param index      Column or block index
 * @param arcImgIdx  Arc image index
 * @param position   Expected line position (pixels)
 * @param wavelength Reference wavelength (µm)
 * @param gaussPar   Gaussian fit parameters structure
 *
 * @return CPL_ERROR_NONE on success, error code otherwise
 *
 * @note Array data beyond gaussPar->range is zeroed for reproducible output
 */
cpl_error_code eris_ifu_wave_fill_fitting_table(
        cpl_table *table,
        int row,
        int slitlet,
        int index,
        int arcImgIdx,
        int position,
        double wavelength,
        struct gaussParStruct *gaussPar)
{
    cpl_error_code retVal = CPL_ERROR_NONE;
    cpl_array *xArray;
    cpl_array *yArray;

    TRY
    {
        cpl_table_set_int(table, ERIS_IFU_FITTABLE_SLITLET, row, slitlet);
        cpl_table_set_int(table, ERIS_IFU_FITTABLE_INDEX, row, index);
        cpl_table_set_int(table, "imgIdx", row, arcImgIdx);
        cpl_table_set_int(table, ERIS_IFU_FITTABLE_POSITION, row, position);
        cpl_table_set_double(table, ERIS_IFU_FITTABLE_WAVELENGTH, row, wavelength);
        cpl_table_set_int(table, ERIS_IFU_FITTABLE_ERRORCODE, row, gaussPar->errorCode);
        cpl_table_set_double(table, "x0", row, gaussPar->x0);
        cpl_table_set_double(table, "sigma", row, gaussPar->sigma);
        cpl_table_set_double(table, "area", row, gaussPar->area);
        cpl_table_set_double(table, "offset", row, gaussPar->offset);
        cpl_table_set_double(table, "mse", row, gaussPar->mse);
        cpl_table_set_int(table, "range", row, gaussPar->range);
        CHECK_ERROR_STATE();
        // clear data beyond range to allow reference testing;
        for (int ix=gaussPar->range; ix<GAUSS_PAR_RANGE_MAX; ix++) {
            gaussPar->xdata[ix] = 0.0;
            gaussPar->ydata[ix] = 0.0;
        }
        BRK_IF_NULL(
                xArray = cpl_array_wrap_double(gaussPar->xdata,
                    GAUSS_PAR_RANGE_MAX));
        BRK_IF_NULL(
                yArray = cpl_array_wrap_double(gaussPar->ydata,
                    GAUSS_PAR_RANGE_MAX));
        cpl_table_set_array(table, "xdata", row, xArray);
        cpl_table_set_array(table, "ydata", row, yArray);
        cpl_array_unwrap(xArray);
        cpl_array_unwrap(yArray);

    }
    CATCH
    {

    }
    return retVal;
}

/**
 * @brief Create coefficient table for polynomial fits
 *
 * Creates a table to store polynomial coefficients with columns for:
 * - slitlet: Slitlet or column identifier
 * - nlines: Number of lines used in fit
 * - degree: Polynomial degree
 * - coeffs: Array of polynomial coefficients
 *
 * @param size             Number of rows to create
 * @param polynomialDegree Maximum polynomial degree (determines array size)
 *
 * @return New coefficient table, or NULL on error
 */
cpl_table *eris_ifu_wave_create_coeff_table(
        int size, int polynomialDegree)
{

    cpl_table *table = NULL;

    TRY
    {
        BRK_IF_NULL(
            table = cpl_table_new(size));
        cpl_table_new_column(table, "slitlet", CPL_TYPE_INT);
        cpl_table_new_column(table, "nlines", CPL_TYPE_INT);
        cpl_table_new_column(table, "degree", CPL_TYPE_INT);
        cpl_table_new_column_array(table, "coeffs", CPL_TYPE_DOUBLE,
                polynomialDegree+1);
        CHECK_ERROR_STATE();
    }
    CATCH
    {

    }
    return table;

}


/**
 * @brief Fill coefficient table row with polynomial data
 *
 * Extracts polynomial coefficients and stores them in a table row along
 * with metadata (index, number of lines used, polynomial degree).
 *
 * @param table      Coefficient table to update
 * @param row        Row index to fill
 * @param idx        Slitlet or column identifier
 * @param nLines     Number of lines used in polynomial fit
 * @param polynomial Polynomial to extract coefficients from
 *
 * @return CPL_ERROR_NONE on success, error code otherwise
 */
cpl_error_code eris_ifu_wave_fill_coeff_table(
        cpl_table *table,
        int row,
        int idx,
        int nLines,
        cpl_polynomial *polynomial)
{
    cpl_error_code retVal = CPL_ERROR_NONE;
    cpl_array *array;
    cpl_size arrayDim;
    cpl_size degree;

    TRY
    {
        degree = cpl_polynomial_get_degree(polynomial);
        arrayDim = cpl_table_get_column_dimension (table, "coeffs", 0);
        array = cpl_array_new(arrayDim, CPL_TYPE_DOUBLE);
        CHECK_ERROR_STATE();

        cpl_table_set_int(table, "slitlet", row, idx);
        cpl_table_set_int(table, "nlines", row, nLines);
        cpl_table_set_int(table, "degree", row, (int) degree);
        for (cpl_size ix=0; ix<=degree; ix++) {
            cpl_array_set_double(array, ix,
                cpl_polynomial_get_coeff(polynomial, &ix));
        }
        cpl_table_set_array(table, "coeffs", row, array);
        cpl_array_delete(array);

    }CATCH {
        CATCH_MSGS();
    }
    return retVal;
}

/**
 * @brief Save all wavelength fitting tables to FITS file
 *
 * Saves all wavelength calibration fitting tables as extensions in a
 * single FITS file. The primary header contains metadata (instrument,
 * band, setup parameters), and each table is saved as a separate extension
 * with descriptive EXTNAME keywords.
 *
 * Tables saved (if not NULL):
 * - slitletFittingTable: First fit line results
 * - columnFittingTable: Column fit line results
 * - slitletCoeff: First fit polynomial coefficients
 * - columnCoeffRaw: Raw column polynomial coefficients
 * - columnCoeffSmoothed: Smoothed column coefficients
 * - smoothingCoeff: Smoothing polynomial coefficients
 *
 * @param tables      Structure containing all fitting tables
 * @param instrument  Instrument identifier
 * @param band        Instrument band
 * @param waveSetup   Wavelength setup parameters
 * @param frameset    Frameset for product header
 * @param parlist     Parameter list for product header
 * @param recipe_name Recipe name for product header
 *
 * @return CPL_ERROR_NONE on success, error code otherwise
 *
 * @note Saved to ERIS_IFU_PRO_WAVE_FIT_TABLES_FN
 * @note Creates empty FITS file if no tables are provided
 */
cpl_error_code eris_ifu_wave_save_fitting_tables(
        struct waveTablesStruct *tables,
        ifsInstrument           instrument,
        ifsBand                 band,
        struct waveSetupStruct waveSetup,
    cpl_frameset* frameset,
    const cpl_parameterlist* parlist,
    const char* recipe_name)
{
    cpl_error_code retVal = CPL_ERROR_NONE;
    cpl_propertylist *phdr = NULL;
    cpl_propertylist *hdr = NULL;
    const char* filename = ERIS_IFU_PRO_WAVE_FIT_TABLES_FN;
    const char* extname = "EXTNAME";
    const char *instrString;
    const char *bandString;
    int ioMode = CPL_IO_CREATE;

    TRY
    {
        switch (instrument) {
        case SPIFFIER:
            instrString = "ERIS";
            break;
        case SPIFFI:
            instrString = "SINFONI";
            break;
        default:
            instrString = "UNKNOWN";
            break;
        }
        bandString = eris_ifu_get_bandString(band);

        BRK_IF_NULL(
            hdr = cpl_propertylist_new());
        BRK_IF_NULL(
            phdr = cpl_propertylist_new());
        BRK_IF_ERROR(
                cpl_propertylist_update_string(phdr, CPL_DFS_PRO_CATG,
                    ERIS_IFU_PRO_WAVE_FIT_TABLES));
        BRK_IF_ERROR(
                cpl_propertylist_update_string(phdr, "INSTRUME", instrString));
        BRK_IF_ERROR(
                cpl_propertylist_update_string(phdr, "BAND", bandString));
        BRK_IF_ERROR(
                cpl_propertylist_update_int(phdr, "SRANGE", waveSetup.s_range));
        BRK_IF_ERROR(
                cpl_propertylist_update_int(phdr, "CRANGE", waveSetup.c_range));
        /* AMO added to have proper FITS header */
        eris_setup_product_header(filename, ERIS_IFU_PRO_WAVE_FIT_TABLES,
                  CPL_FRAME_TYPE_IMAGE, recipe_name,
                              frameset, parlist, phdr);

        if (tables->slitletFitting != NULL) {
            BRK_IF_ERROR(
                cpl_propertylist_update_string(
                    hdr, extname, "slitletFittingTable"));
            BRK_IF_ERROR(
                    cpl_table_name_column(tables->slitletFitting, "index", "block"));
            BRK_IF_ERROR(
                    cpl_table_save(tables->slitletFitting,
                        phdr, hdr, filename, ioMode));
            ioMode = CPL_IO_EXTEND;
        }

        if (tables->columnFitting != NULL) {
            BRK_IF_ERROR(
                cpl_propertylist_update_string(
                    hdr, extname, "columnFittingTable"));
            BRK_IF_ERROR(
                    cpl_table_name_column(tables->columnFitting, "index", "column"));
            BRK_IF_ERROR(
                    cpl_table_save(tables->columnFitting,
                        phdr, hdr, filename, ioMode));
            ioMode = CPL_IO_EXTEND;
        }

        if (tables->columnCoeffRaw != NULL) {
            BRK_IF_ERROR(
                cpl_propertylist_update_string(
                    hdr, extname, "slitletCoeff"));
            BRK_IF_ERROR(
                    cpl_table_save(tables->slitletCoeff,
                        phdr, hdr, filename, ioMode));

            BRK_IF_ERROR(
                    cpl_propertylist_update_string(
                        hdr, extname, "columnCoeffRaw"));
            BRK_IF_ERROR(
                    cpl_table_save(tables->columnCoeffRaw,
                        phdr, hdr, filename, ioMode));
            ioMode = CPL_IO_EXTEND;
        }

        if (tables->columnCoeffSmoothed != NULL) {
            BRK_IF_ERROR(
                cpl_propertylist_update_string(
                    hdr, extname, "columnCoeffSmoothed"));
            BRK_IF_ERROR(
                    cpl_table_name_column(tables->columnCoeffSmoothed, "nlines", "dummy"));
            BRK_IF_ERROR(
                    cpl_table_save(tables->columnCoeffSmoothed,
                        phdr, hdr, filename, ioMode));
            ioMode = CPL_IO_EXTEND;
        }

        if (tables->smoothingCoeff != NULL) {
            BRK_IF_ERROR(
                cpl_propertylist_update_string(
                    hdr, extname, "smoothingCoeff"));
            BRK_IF_ERROR(
                    cpl_table_name_column(tables->smoothingCoeff, "nlines", "order"));
            BRK_IF_ERROR(
                    cpl_table_save(tables->smoothingCoeff,
                        phdr, hdr, filename, ioMode));
            ioMode = CPL_IO_EXTEND;
       }

       // check whether no table was written
       if (ioMode == CPL_IO_CREATE) {
           // create an "empty" FITS file
           BRK_IF_ERROR(
               cpl_image_save(NULL, filename, CPL_TYPE_FLOAT, phdr, ioMode));
       }

    }
    CATCH
    {

    }
    eris_ifu_free_propertylist(&hdr);
    eris_ifu_free_propertylist(&phdr);
    return retVal;
}

/**
 * @brief Free all wavelength fitting tables
 *
 * Deletes all tables in the waveTablesStruct and sets pointers to NULL.
 * Safe to call even if some tables are already NULL.
 *
 * @param tables  Structure containing table pointers to free
 */
void eris_ifu_wave_free_tables(
        struct waveTablesStruct *tables)
{
    eris_ifu_free_table(&tables->slitletFitting);
    eris_ifu_free_table(&tables->columnFitting);
    eris_ifu_free_table(&tables->slitletCoeff);
    eris_ifu_free_table(&tables->columnCoeffRaw);
    eris_ifu_free_table(&tables->columnCoeffSmoothed);
    eris_ifu_free_table(&tables->smoothingCoeff);
    return;
}

/**
 * @brief Clear wavelength table pointers without freeing
 *
 * Sets all table pointers in the structure to NULL without freeing
 * the tables. Use when tables have been transferred elsewhere or
 * when re-initializing the structure.
 *
 * @param tables  Structure to clear
 */
void eris_ifu_wave_clear_tables(
        struct waveTablesStruct *tables)
{
    tables->slitletFitting = NULL;
    tables->columnFitting = NULL;
    tables->slitletCoeff = NULL;
    tables->columnCoeffRaw = NULL;
    tables->columnCoeffSmoothed = NULL;
    tables->smoothingCoeff = NULL;
   return;
}

/**@}*/
