/*
 *  This file is part of the SKYCORR software package.
 *  Copyright (C) 2009-2013 European Southern Observatory
 *
 *  This programme is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This programme is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this programme. If not, see <http://www.gnu.org/licenses/>.
 */


/*!
 * \ingroup sky_correction
 */

/**@{*/

/*!
 * \file sc_skycorr.c
 *
 * Routines for handling CMPFIT
 *
 * \author Stefan Noll & ESO In-Kind Team Innsbruck
 * \since  18 Feb 2011
 * \date   08 Sep 2013
 */


/*****************************************************************************
 *                                 INCLUDES                                  *
 ****************************************************************************/

#include <sc_skycorr.h>


/*****************************************************************************
 *                                  CODE                                     *
 ****************************************************************************/

cpl_error_code sc_skycorr(cpl_parameterlist *parlist,
                          cpl_table *scispec,
                          cpl_table *skyspec,
                          int product_depth)
{
    /*!
     * \callgraph
     *
     * This is the top-level routine of the sky correction code that subtracts
     * an optimised 1D sky spectrum from an 1D science spectrum. The
     * optimisation process is a CMPFIT-based fitting procedure that adapts
     * airglow lines in the sky spectrum to those in the science spectrum.
     * Object-related lines in the science spectrum should not be affected by
     * this procedure. The sky continuum is taken from the sky spectrum
     * without optimisation, since a separation of object and sky continuum in
     * the science spectrum is not possible. All required information for the
     * sky correction procedure is provided by an input ASCII parameter file.
     *
     * \b INPUT:
     * \param parfile_in  name of parameter file
     *
     * \b ERRORS:
     * - Error in subroutine
     * - see subroutines
     */

    cpl_error_code status = CPL_ERROR_NONE;
    cpl_table *groups, *linetab;
    char errtxt[SC_MAXLEN];

    /* Read line groups from ASCII file and adapt it to observing
       conditions */
    groups = cpl_table_new(0);
    if ((status = sc_lines(groups, parlist)) != CPL_ERROR_NONE) {
        cpl_table_delete(groups);
        return status;
    }

    /* Subtraction of continuum from science and sky spectrum and estimation
       of line FWHM in sky spectrum */

    /* Prepare science spectrum */
//    cpl_msg_debug(cpl_func, "Science spectrum:");
    linetab = cpl_table_new(0);
    sc_specdiss_init_linetab(linetab);
    sc_skycorr_subcont(scispec, linetab, parlist, groups);
    cpl_table_delete(linetab);
    status = cpl_error_get_code();
    if (status != CPL_ERROR_NONE) {
        sprintf(errtxt, "%s: error while separating lines and continuum of "
                "science spectrum", SC_ERROR_EIS_TXT);
        cpl_error_set_message(cpl_func, (cpl_error_code)SC_ERROR_EIS, "%s", errtxt);
        return status;  // agudo
    }

    /* Prepare sky spectrum */
//    cpl_msg_debug(cpl_func, "Sky spectrum:");
    linetab = cpl_table_new(0);
    sc_specdiss_init_linetab(linetab);
    status = sc_skycorr_subcont(skyspec, linetab, parlist, groups);
    cpl_table_delete(linetab); linetab = NULL;
    if (status != CPL_ERROR_NONE) {
        sprintf(errtxt, "%s: error while separating lines and continuum of "
                "sky spectrum", SC_ERROR_EIS_TXT);
        cpl_error_set_message(cpl_func, (cpl_error_code)SC_ERROR_EIS, "%s", errtxt);
        cpl_table_delete(groups);
        return status;  // agudo
    }

    /* CMPFIT-based fitting procedure to adapt the sky line spectrum to the
       science line spectrum */
    status = sc_skycorr_fit(scispec, skyspec, groups, parlist, product_depth);
    if (status != CPL_ERROR_NONE) {
        cpl_table_delete(groups);
        return status;
    }

    /* Free allocated memory */
    cpl_table_delete(groups);

    /* Perform sky subtraction */
    status = sc_skycorr_subsky(scispec);

    return status;
}

cpl_error_code sc_skycorr_subcont(cpl_table *spec, cpl_table *linetab,
                                  cpl_parameterlist *parlist,
                                  const cpl_table *groups)
{
    /*!
     * \callgraph
     *
     * Subtracts continuum from a spectrum and derives FWHM of spectral lines
     * in this spectrum.
     *
     * The FWHM is derived by an iterative procedure that identifies isolated
     * lines, subtracts the continuum from these lines, and derives a clipped
     * mean FWHM. Then, the resulting line width is used for another line
     * search and so on. The procedure stops either if the relative change of
     * the FWHM between two iterations is below the 'ltol' value from the
     * input parameter list or after the 10th iteration.
     *
     * The list of line pixels identified by means of the first derivative of
     * the line flux is supplemented by pixels belonging to lines taken from
     * the airglow line lists. The number of pixels per line depends on the
     * FWHM derived before. Only lines are considered that have peak fluxes
     * above a threshold. If desired (parameter 'fluxlim' = -1), the flux
     * limit will be searched by an iterative procedure which tries to find
     * the best compromise between sufficient continuum pixels and a high
     * number of considered lines. In the case of a positive value for
     * 'fluxlim', this value is multiplied by the median flux of the already
     * identified lines in order to obtain the actual threshold. Factors in
     * the order of 0.01 are expected to work for most kind of spectra.
     *
     * \b INPUT:
     * \param spec     CPL table with spectrum
     * \param linetab  CPL table for information on detected lines
     * \param parlist  general CPL parameter list
     * \param groups   CPL table with airglow line information
     *
     * \b OUTPUT:
     * \param spec     spectrum with separation of lines and continuum
     * \param linetab  CPL table with information on detected lines
     * \param parlist  parameter list with updated line FWHM and relative line
     *                 flux limit
     *
     * \b ERRORS:
     * - see subroutines
     */

    cpl_error_code status = CPL_ERROR_NONE;
    cpl_parameter *p;
    //cpl_parameterlist *plottags;
    cpl_array *mask;
    int it = 0, i = 0, n = 10, nrow = 0, j = 0, niso = 0, search = 0;
    double fwhm0 = 0., fwhm = 1000., eps = 1e-2, tmp = 0., rms = 0.;
    double fluxlim = 0., minfluxlim = 0.005, maxfluxlim = 0.1;

    /* Print info message */
//    cpl_msg_debug(cpl_func, "Identify lines, estimate FWHM, "
//                 "and subtract continuum");

    /* Get FWHM from parameter list */
    p = cpl_parameterlist_find(parlist, "fwhm");
    fwhm0 = cpl_parameter_get_double(p);

    /* Take FWHM from parameter list as start value if already fitted */
    p = cpl_parameterlist_find(parlist, "iteration");
    it = cpl_parameter_get_int(p);
    if (it > 0) {
        fwhm = fwhm0;
    }
if (cpl_error_get_code() != CPL_ERROR_NONE) {
    return cpl_error_get_code();
}
    /* Get convergence criterion for line width */
    p = cpl_parameterlist_find(parlist, "ltol");
    eps = cpl_parameter_get_double(p);

    /* Iterative line identification and FWHM estimation */
    for (tmp = fwhm, i = 0; i < n; i++, tmp = fwhm) {
        /* Find emission lines in spectrum */
        status = sc_specdiss_find_emissionlines(spec, linetab, parlist, groups);
        if (status != CPL_ERROR_NONE) {
            return status;
        }

        /* Subtract continuum in spectrum */
        status = sc_contsub(spec);
        if (status != CPL_ERROR_NONE) {
            return status;
        }

        /* Count isolated lines */
        nrow = cpl_table_get_nrow(linetab);
        for (j = 0, niso = 0; j < nrow; j++) {
            if (cpl_table_get_int(linetab, "isol_flag", j, NULL) != 0) {
                niso++;
            }
        }

        /* No isolated lines -> break */
        if (niso == 0) {
            if (it == 0) {
                fwhm = fwhm0;
            }
            cpl_msg_warning(cpl_func, "No isolated lines found -> "
                            "Take initial FWHM");
            break;
        }

        /* Estimate FWHM of lines in spectrum */
        status = sc_fwhmest(&fwhm, &rms, spec, linetab, parlist);
        if (status != CPL_ERROR_NONE) {
            return status;
        }

        /* Update FWHM in parlist */
        p = cpl_parameterlist_find(parlist, "fwhm");
        cpl_parameter_set_double(p, fwhm);

        /* Check convergence */
        if (fabs((fwhm - tmp) / tmp) < eps) {
            break;
        }
    }

    /* Save pixel class column */
    mask = cpl_array_new(0, CPL_TYPE_INT);
    status = sc_basic_col2arr(mask, spec, "class");
    if (status != CPL_ERROR_NONE) {
        return status;
    }

    /* Get relative peak flux limit from parameter list */
    p = cpl_parameterlist_find(parlist, "fluxlim");
    fluxlim = cpl_parameter_get_double(p);

    /* Check whether automatic search is desired */
    if (fluxlim < 0) {
        fluxlim = minfluxlim;
        search = 1;
    }

    /* Get line and continuum pixels (iterative approach if desired) */
    do {
        /* Get initial pixel class column */
        status = sc_basic_arr2col(spec, "class", mask);
        if (status != CPL_ERROR_NONE) {
            cpl_array_delete(mask);
            return status;
        }

        /* Identify continuum windows by means of airglow line list */
        status = sc_contsub_identcont(spec, groups, linetab, fluxlim,
                                      parlist);
        if (status != CPL_ERROR_NONE) {
            cpl_array_delete(mask);
            return status;
        }

        /* Modify lower flux limit for line peaks */
        fluxlim *= 2;
    } while (sc_contsub_check(spec) != 0 && fluxlim <= maxfluxlim && search == 1);

    status = cpl_error_get_code();
    if (status != CPL_ERROR_NONE) {
        return status;
    }

    /* Info message on fluxlim parameter */
//    if (search == 1) {
//        cpl_msg_debug(cpl_func, "Search for line threshold: FLUXLIM = %g",
//                     fluxlim / 2);
//    }

    /* Avoid identification of lines in the thermal IR */
    status = sc_contsub_skipthermir(spec);
    if (status != CPL_ERROR_NONE)       {
        return status;
    }

    /* Cleanup */
    cpl_array_delete(mask); mask = NULL;

    /* Plotting spectrum with identified line peaks */
    //plottags=cpl_parameterlist_new();
    //sc_setplottags_single_spec(plottags, "lambda", "cflux", "CFLUX before",
    //                           "lambda", "cflux", parlist);
    //sc_plot_single_spec(spec, plottags);
    //cpl_parameterlist_delete(plottags);

    /* Subtract continuum in spectrum */
    status = sc_contsub(spec);
    if (status != CPL_ERROR_NONE) {
        return status;
    }

    return status;
}

cpl_error_code sc_skycorr_fit(cpl_table *scispec, cpl_table *skyspec,
                              cpl_table *groups, cpl_parameterlist *parlist,
                              int product_depth)
{
    /*!
     * \callgraph
     *
     * CMPFIT-based fitting procedure to adapt the sky line spectrum to the
     * science line spectrum.
     *
     * \b INPUT:
     * \param scispec  CPL table with science spectrum
     * \param skyspec  CPL table with sky spectrum
     * \param groups   CPL table with airglow line information
     * \param parlist  general CPL parameter list
     *
     * \b OUTPUT:
     * \param scispec  science and best-fit modified sky spectrum
     * \param skyspec  sky spectrum with line group weights
     * \param groups   modified line table
     * \param parlist  parameter list with additional entries
     *
     * \b ERRORS:
     * - see ::sc_mpfit
     */

    cpl_error_code status = CPL_ERROR_NONE;
    cpl_table *fitpar;
    mp_result result;
    int nsci = 0;
//    double ts = 0., te = 0., runtime = 0.;

    /* Create columns for modification of sky spectrum */
    cpl_table_new_column(skyspec, "mlambda", CPL_TYPE_DOUBLE);
    cpl_table_new_column(skyspec, "mlflux", CPL_TYPE_DOUBLE);
    if (cpl_table_has_column(skyspec, "dflux") == 1) {
        cpl_table_new_column(skyspec, "mdflux", CPL_TYPE_DOUBLE);
    }
    cpl_table_new_column(skyspec, "mweight", CPL_TYPE_DOUBLE);
    cpl_table_new_column(skyspec, "mpix", CPL_TYPE_INT);
    cpl_table_new_column(skyspec, "dpix", CPL_TYPE_DOUBLE);
    cpl_table_new_column(skyspec, "frat", CPL_TYPE_DOUBLE);

    /* Create columns for comparison of science and sky spectrum */
    cpl_table_new_column(scispec, "mcflux", CPL_TYPE_DOUBLE);
    cpl_table_new_column(scispec, "mlflux", CPL_TYPE_DOUBLE);
    cpl_table_new_column(scispec, "mflux", CPL_TYPE_DOUBLE);
    if (cpl_table_has_column(skyspec, "dflux") == 1) {
        cpl_table_new_column(scispec, "mdflux", CPL_TYPE_DOUBLE);
    }
    if (cpl_table_has_column(skyspec, "mask") == 1) {
        cpl_table_new_column(scispec, "mmask", CPL_TYPE_INT);
    }
    cpl_table_new_column(scispec, "mweight", CPL_TYPE_DOUBLE);
    cpl_table_new_column(scispec, "sigclip", CPL_TYPE_INT);
    cpl_table_new_column(scispec, "cweight", CPL_TYPE_DOUBLE);
    cpl_table_new_column(scispec, "dev", CPL_TYPE_DOUBLE);

    /* Initialise sigclip column with 1 (= ejected) */
    nsci = cpl_table_get_nrow(scispec);
    cpl_table_fill_column_window(scispec, "sigclip", 0, nsci, 1);

    /* Initialise fit parameter table */
    fitpar = cpl_table_new(0);

    /* Prepare the line group weights of each pixel of the input sky
       spectrum */
    if ((status = sc_weights(skyspec, fitpar, groups, parlist)) !=
        CPL_ERROR_NONE) {
        cpl_table_delete(fitpar);
        return status;
    }

    /* Start run time measurement */
//    ts = cpl_test_get_walltime();

    /* Start CMPFIT and compare resulting deviations */
    status = sc_mpfit(&result, scispec, skyspec, fitpar, parlist);

    /* Get CMPFIT run time in s */
//    te = cpl_test_get_walltime();
//    runtime = te - ts;

    /* Print fit results */
//    cpl_msg_debug(cpl_func, "FIT RESULTS:");
//    cpl_msg_debug(cpl_func, "status: %d", result.status);
//    if (status == CPL_ERROR_NONE) {
//        cpl_msg_debug(cpl_func, "npar: %d", result.npar);
//        cpl_msg_debug(cpl_func, "npix: %d", result.nfunc);
//        cpl_msg_debug(cpl_func, "niter: %d", result.niter);
//        cpl_msg_debug(cpl_func, "nfev: %d", result.nfev);
//        cpl_msg_debug(cpl_func, "fittime: %.2f s", runtime);
//        cpl_msg_debug(cpl_func, "orignorm: %.3e", result.orignorm);
//        cpl_msg_debug(cpl_func, "bestnorm: %.3e", result.bestnorm);
//    }
    if ((product_depth == 3) && (result.status != 1)) {
        cpl_msg_debug(cpl_func, ">>>>>       skycorr fit result: %d", result.status);
    }
//    if (result.status == MP_OK_DIR) {
//        cpl_msg_info(cpl_func, "############# got status = MP_OK_DIR -> should be fixed");
//        cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_OUTPUT);
//        status = CPL_ERROR_ILLEGAL_OUTPUT;
//    } else if (result.status <= 0) {
//        cpl_msg_info(cpl_func, "############# got status <= 0 -> bad, not fixed");
//        cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_OUTPUT);
//        status = CPL_ERROR_ILLEGAL_OUTPUT;
//    } else if (result.status > 1) {
//        cpl_msg_info(cpl_func, "############# got status > 1 -> what? bad? ok?");
////        cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_OUTPUT);
////        status = CPL_ERROR_ILLEGAL_OUTPUT;
//    }

    /* Free allocated memory */
    cpl_table_delete(fitpar);
    sc_mpfit_freememresult(&result);

    return status;
}


cpl_error_code sc_skycorr_subsky(cpl_table *scispec)
{
    /*!
     * Subtracts sky continuum and fit-optimised sky line flux from science
     * spectrum. The resulting sky-corrected flux is written into the new
     * column "scflux". If an error occurred before, the flux correction is
     * not carried out. If flux error and mask column exist in the input
     * spectra, the effect of sky subtraction on these columns is also
     * considered. The resulting columns are labelled "scdflux" and "scmask"
     * in this case.
     *
     * \b INPUT:
     * \param scispec  CPL table with science spectrum and best-fit modified
     *                 sky spectrum
     *
     * \b OUTPUT:
     * \param scispec  data table with column for sky-subtracted flux
     *
     * \b ERRORS:
     * - none
     */

    cpl_error_code status = CPL_ERROR_NONE;
    int nrow = 0, i = 0;
    int *mmask = NULL, *mask = NULL, *scmask = NULL;
    double *mweight = NULL, *dflux = NULL, *mdflux = NULL, *scdflux = NULL;

    /* Get number of rows in science spectrum table */
    nrow = cpl_table_get_nrow(scispec);

    /* Calculate combined sky line and continuum flux */
    cpl_table_fill_column_window(scispec, "mflux", 0, nrow, 0.);
    cpl_table_add_columns(scispec, "mflux", "mcflux");
    cpl_table_add_columns(scispec, "mflux", "mlflux");

    /* Derive mask for sky spectrum if present */
    if (cpl_table_has_column(scispec, "mask") == 1) {
        cpl_table_fill_column_window(scispec, "mmask", 0, nrow, 0);
        mweight = cpl_table_get_data_double(scispec, "mweight");
        mmask = cpl_table_get_data_int(scispec, "mmask");
        for (i = 0; i < nrow; i++) {
            if (mweight[i] == 0.) {
                mmask[i] = 0;
            } else {
                mmask[i] = 1;
            }
        }
    }

    /* Create column for sky-subtracted flux and initialise it with
       uncorrected flux */
    cpl_table_duplicate_column(scispec, "scflux", scispec, "flux");

    /* Create column for flux errors after sky subtraction and initialise it
       with input flux errors if present */
    if (cpl_table_has_column(scispec, "dflux") == 1) {
         cpl_table_duplicate_column(scispec, "scdflux", scispec, "dflux");
    }

    /* Create column for mask after sky subtraction and initialise it with
       input mask if present */
    if (cpl_table_has_column(scispec, "mask") == 1) {
         cpl_table_duplicate_column(scispec, "scmask", scispec, "mask");
    }

    /* Leave routine in the case of errors */
    if ((status = cpl_error_get_code()) != CPL_ERROR_NONE) {
        return status;
    }

    /* Subtract sky continuum and best-fit sky line flux */
    cpl_table_subtract_columns(scispec, "scflux", "mflux");

    /* Derive errors for sky-subtracted flux if errors are present in the
       input data */
    if (cpl_table_has_column(scispec, "dflux") == 1) {
        dflux = cpl_table_get_data_double(scispec, "dflux");
        mdflux = cpl_table_get_data_double(scispec, "mdflux");
        scdflux = cpl_table_get_data_double(scispec, "scdflux");
        for (i = 0; i < nrow; i++) {
            scdflux[i] = sqrt(dflux[i] * dflux[i] + mdflux[i] * mdflux[i]);
        }
    }

    /* Derive combined mask if masks are present in the input data */
    if (cpl_table_has_column(scispec, "mask") == 1) {
        mask = cpl_table_get_data_int(scispec, "mask");
        mmask = cpl_table_get_data_int(scispec, "mmask");
        scmask = cpl_table_get_data_int(scispec, "scmask");
        for (i = 0; i < nrow; i++) {
            if (mask[i] == 1 && mmask[i] == 1) {
                scmask[i] = 1;
            } else {
                scmask[i] = 0;
            }
        }
    }

    return cpl_error_get_code();
}

/**@}*/
