/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <float.h>
#include <math.h>
#include <string.h>
#include "eris_nix_test_defs.h"
#include "eris_nix_master_wave.h"
#include <hdrl.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_master_wave_test  Unit test of eris_nix_master_wave
 *
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                   Defines
 -----------------------------------------------------------------------------*/

#define SMALL_TEST_SIZE_X 64
#define SMALL_TEST_SIZE_Y 64

/*-----------------------------------------------------------------------------
                                Helper Functions
 -----------------------------------------------------------------------------*/

/**
 * @brief Create a test hdrl_image filled with a constant value
 */
static hdrl_image * create_test_hdrl_image(int nx, int ny, double value) {
    hdrl_image * himage = hdrl_image_new(nx, ny);
    cpl_image_fill_window(hdrl_image_get_image(himage), 1, 1, nx, ny, value);
    cpl_image_fill_window(hdrl_image_get_error(himage), 1, 1, nx, ny, sqrt(value));
    return himage;
}

/**
 * @brief Create a test mask (all good pixels)
 */
static cpl_mask * create_test_mask(int nx, int ny) {
    cpl_mask * mask = cpl_mask_new(nx, ny);
    /* All pixels are good (0) by default */
    return mask;
}

/**
 * @brief Create a test confidence image
 */
static cpl_image * create_test_confidence(int nx, int ny, double value) {
    cpl_image * confidence = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(confidence, 1, 1, nx, ny, value);
    return confidence;
}

/**
 * @brief Create a test propertylist with some keywords
 */
static cpl_propertylist * create_test_plist(void) {
    cpl_propertylist * plist = cpl_propertylist_new();
    cpl_propertylist_update_string(plist, "INSTRUME", "ERIS");
    cpl_propertylist_update_string(plist, "ESO DPR CATG", "CALIB");
    cpl_propertylist_update_double(plist, "MJD-OBS", 60000.0);
    cpl_propertylist_update_string(plist, "ESO INS2 NXFW NAME", "Ks");
    return plist;
}

/*-----------------------------------------------------------------------------
                                Test Functions
 -----------------------------------------------------------------------------*/

/**
 * @brief Test en_master_wave_create with NULL required inputs
 */
static void test_create_null_raw_wave_image(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    
    cpl_mask * bpm = create_test_mask(nx, ny);
    cpl_image * confidence = create_test_confidence(nx, ny, 100.0);
    
    /* NULL raw_wave_image should fail */
    master_wave * result = en_master_wave_create(NULL, bpm, confidence,
                                                  NULL, NULL, NULL,
                                                  NULL, NULL);
    cpl_test_null(result);
    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    
    cpl_mask_delete(bpm);
    cpl_image_delete(confidence);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_master_wave_create with NULL raw_wave_bpm
 */
static void test_create_null_raw_wave_bpm(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, 1.0);
    cpl_image * confidence = create_test_confidence(nx, ny, 100.0);
    
    /* NULL raw_wave_bpm should fail */
    master_wave * result = en_master_wave_create(raw_wave, NULL, confidence,
                                                  NULL, NULL, NULL,
                                                  NULL, NULL);
    cpl_test_null(result);
    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    
    hdrl_image_delete(raw_wave);
    cpl_image_delete(confidence);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_master_wave_create with NULL raw_wave_confidence
 */
static void test_create_null_raw_wave_confidence(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, 1.0);
    cpl_mask * bpm = create_test_mask(nx, ny);
    
    /* NULL raw_wave_confidence should fail */
    master_wave * result = en_master_wave_create(raw_wave, bpm, NULL,
                                                  NULL, NULL, NULL,
                                                  NULL, NULL);
    cpl_test_null(result);
    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(bpm);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_master_wave_create with pre-existing error
 */
static void test_create_preexisting_error(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, 1.0);
    cpl_mask * bpm = create_test_mask(nx, ny);
    cpl_image * confidence = create_test_confidence(nx, ny, 100.0);
    
    /* Set pre-existing error */
    cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
    
    master_wave * result = en_master_wave_create(raw_wave, bpm, confidence,
                                                  NULL, NULL, NULL,
                                                  NULL, NULL);
    cpl_test_null(result);
    
    cpl_error_reset();
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(bpm);
    cpl_image_delete(confidence);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_master_wave_create with minimal required inputs
 */
static void test_create_minimal_inputs(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, 2.2);
    cpl_mask * bpm = create_test_mask(nx, ny);
    cpl_image * confidence = create_test_confidence(nx, ny, 100.0);
    
    /* Create with only required inputs */
    master_wave * result = en_master_wave_create(raw_wave, bpm, confidence,
                                                  NULL, NULL, NULL,
                                                  NULL, NULL);
    cpl_test_nonnull(result);
    
    /* Verify required fields are set */
    cpl_test_nonnull(result->raw_wave_image);
    cpl_test_nonnull(result->raw_wave_bpm);
    cpl_test_nonnull(result->raw_wave_confidence);
    
    /* Verify optional fields are NULL */
    cpl_test_null(result->calibrated_wave_image);
    cpl_test_null(result->calibrated_wave_bpm);
    cpl_test_null(result->calibrated_wave_confidence);
    cpl_test_null(result->filename);
    cpl_test_null(result->plist);
    
    /* Verify data is duplicated (not just pointer copy) */
    cpl_test(result->raw_wave_image != raw_wave);
    cpl_test(result->raw_wave_bpm != bpm);
    cpl_test(result->raw_wave_confidence != confidence);
    
    /* Verify dimensions */
    cpl_test_eq(hdrl_image_get_size_x(result->raw_wave_image), nx);
    cpl_test_eq(hdrl_image_get_size_y(result->raw_wave_image), ny);
    cpl_test_eq(cpl_mask_get_size_x(result->raw_wave_bpm), nx);
    cpl_test_eq(cpl_mask_get_size_y(result->raw_wave_bpm), ny);
    
    /* Cleanup */
    en_master_wave_delete(result);
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(bpm);
    cpl_image_delete(confidence);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_master_wave_create with all inputs provided
 */
static void test_create_all_inputs(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    
    /* Create all inputs */
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, 2.2);
    cpl_mask * raw_bpm = create_test_mask(nx, ny);
    cpl_image * raw_confidence = create_test_confidence(nx, ny, 100.0);
    
    hdrl_image * cal_wave = create_test_hdrl_image(nx, ny, 2.18);
    cpl_mask * cal_bpm = create_test_mask(nx, ny);
    cpl_image * cal_confidence = create_test_confidence(nx, ny, 95.0);
    
    const char * filename = "test_master_wave.fits";
    cpl_propertylist * plist = create_test_plist();
    
    /* Create with all inputs */
    master_wave * result = en_master_wave_create(raw_wave, raw_bpm, raw_confidence,
                                                  cal_wave, cal_bpm, cal_confidence,
                                                  filename, plist);
    cpl_test_nonnull(result);
    
    /* Verify all fields are set */
    cpl_test_nonnull(result->raw_wave_image);
    cpl_test_nonnull(result->raw_wave_bpm);
    cpl_test_nonnull(result->raw_wave_confidence);
    cpl_test_nonnull(result->calibrated_wave_image);
    cpl_test_nonnull(result->calibrated_wave_bpm);
    cpl_test_nonnull(result->calibrated_wave_confidence);
    cpl_test_nonnull(result->filename);
    cpl_test_nonnull(result->plist);
    
    /* Verify data is duplicated */
    cpl_test(result->raw_wave_image != raw_wave);
    cpl_test(result->calibrated_wave_image != cal_wave);
    cpl_test(result->plist != plist);
    
    /* Verify filename content */
    cpl_test_eq_string(result->filename, filename);
    
    /* Verify propertylist content */
    const char * instrume = cpl_propertylist_get_string(result->plist, "INSTRUME");
    cpl_test_eq_string(instrume, "ERIS");
    
    /* Cleanup */
    en_master_wave_delete(result);
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(raw_bpm);
    cpl_image_delete(raw_confidence);
    hdrl_image_delete(cal_wave);
    cpl_mask_delete(cal_bpm);
    cpl_image_delete(cal_confidence);
    cpl_propertylist_delete(plist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_master_wave_create verifies pixel values are copied
 */
static void test_create_pixel_values(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    const double test_value = 2.15;
    const double conf_value = 99.0;
    
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, test_value);
    cpl_mask * bpm = create_test_mask(nx, ny);
    cpl_image * confidence = create_test_confidence(nx, ny, conf_value);
    
    master_wave * result = en_master_wave_create(raw_wave, bpm, confidence,
                                                  NULL, NULL, NULL,
                                                  NULL, NULL);
    cpl_test_nonnull(result);
    
    /* Verify pixel values are copied correctly */
    int reject = 0;
    hdrl_value val = hdrl_image_get_pixel(result->raw_wave_image, 32, 32, &reject);
    cpl_test_eq(reject, 0);
    cpl_test_abs(val.data, test_value, DBL_EPSILON);
    
    double conf_val = cpl_image_get(result->raw_wave_confidence, 32, 32, &reject);
    cpl_test_eq(reject, 0);
    cpl_test_abs(conf_val, conf_value, DBL_EPSILON);
    
    /* Cleanup */
    en_master_wave_delete(result);
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(bpm);
    cpl_image_delete(confidence);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_master_wave_delete with NULL pointer
 */
static void test_delete_null(void) {
    /* Should not crash or set error */
    en_master_wave_delete(NULL);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_master_wave_delete with valid pointer
 */
static void test_delete_valid(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, 2.2);
    cpl_mask * bpm = create_test_mask(nx, ny);
    cpl_image * confidence = create_test_confidence(nx, ny, 100.0);
    cpl_propertylist * plist = create_test_plist();
    
    master_wave * result = en_master_wave_create(raw_wave, bpm, confidence,
                                                  NULL, NULL, NULL,
                                                  "test.fits", plist);
    cpl_test_nonnull(result);
    
    /* Delete should not crash or set error */
    en_master_wave_delete(result);
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Cleanup original inputs */
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(bpm);
    cpl_image_delete(confidence);
    cpl_propertylist_delete(plist);
}

/**
 * @brief Test en_master_wave_delete with all fields populated
 */
static void test_delete_all_fields(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    
    /* Create all inputs */
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, 2.2);
    cpl_mask * raw_bpm = create_test_mask(nx, ny);
    cpl_image * raw_confidence = create_test_confidence(nx, ny, 100.0);
    
    hdrl_image * cal_wave = create_test_hdrl_image(nx, ny, 2.18);
    cpl_mask * cal_bpm = create_test_mask(nx, ny);
    cpl_image * cal_confidence = create_test_confidence(nx, ny, 95.0);
    
    cpl_propertylist * plist = create_test_plist();
    
    master_wave * result = en_master_wave_create(raw_wave, raw_bpm, raw_confidence,
                                                  cal_wave, cal_bpm, cal_confidence,
                                                  "test_wave.fits", plist);
    cpl_test_nonnull(result);
    
    /* Delete with all fields populated should work */
    en_master_wave_delete(result);
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Cleanup original inputs */
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(raw_bpm);
    cpl_image_delete(raw_confidence);
    hdrl_image_delete(cal_wave);
    cpl_mask_delete(cal_bpm);
    cpl_image_delete(cal_confidence);
    cpl_propertylist_delete(plist);
}

/**
 * @brief Test multiple create/delete cycles for memory leaks
 */
static void test_create_delete_cycle(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    const int ncycles = 10;
    
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, 2.2);
    cpl_mask * bpm = create_test_mask(nx, ny);
    cpl_image * confidence = create_test_confidence(nx, ny, 100.0);
    
    /* Multiple create/delete cycles */
    for (int i = 0; i < ncycles; i++) {
        master_wave * result = en_master_wave_create(raw_wave, bpm, confidence,
                                                      NULL, NULL, NULL,
                                                      NULL, NULL);
        cpl_test_nonnull(result);
        en_master_wave_delete(result);
    }
    
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Cleanup */
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(bpm);
    cpl_image_delete(confidence);
}

/**
 * @brief Test en_master_wave_create with very small image (1x1)
 */
static void test_create_tiny_image(void) {
    const int nx = 1;
    const int ny = 1;
    const double test_value = 2.5;
    
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, test_value);
    cpl_mask * bpm = create_test_mask(nx, ny);
    cpl_image * confidence = create_test_confidence(nx, ny, 100.0);
    
    master_wave * result = en_master_wave_create(raw_wave, bpm, confidence,
                                                  NULL, NULL, NULL,
                                                  NULL, NULL);
    cpl_test_nonnull(result);
    
    /* Verify dimensions */
    cpl_test_eq(hdrl_image_get_size_x(result->raw_wave_image), 1);
    cpl_test_eq(hdrl_image_get_size_y(result->raw_wave_image), 1);
    
    /* Verify pixel value */
    int reject = 0;
    hdrl_value val = hdrl_image_get_pixel(result->raw_wave_image, 1, 1, &reject);
    cpl_test_eq(reject, 0);
    cpl_test_abs(val.data, test_value, DBL_EPSILON);
    
    en_master_wave_delete(result);
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(bpm);
    cpl_image_delete(confidence);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_master_wave_create with larger image
 */
static void test_create_larger_image(void) {
    const int nx = 256;
    const int ny = 256;
    
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, 2.2);
    cpl_mask * bpm = create_test_mask(nx, ny);
    cpl_image * confidence = create_test_confidence(nx, ny, 100.0);
    
    master_wave * result = en_master_wave_create(raw_wave, bpm, confidence,
                                                  NULL, NULL, NULL,
                                                  NULL, NULL);
    cpl_test_nonnull(result);
    
    /* Verify dimensions */
    cpl_test_eq(hdrl_image_get_size_x(result->raw_wave_image), nx);
    cpl_test_eq(hdrl_image_get_size_y(result->raw_wave_image), ny);
    cpl_test_eq(cpl_image_get_size_x(result->raw_wave_confidence), nx);
    cpl_test_eq(cpl_image_get_size_y(result->raw_wave_confidence), ny);
    
    en_master_wave_delete(result);
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(bpm);
    cpl_image_delete(confidence);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_master_wave_create with non-square image
 */
static void test_create_nonsquare_image(void) {
    const int nx = 128;
    const int ny = 64;
    
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, 2.2);
    cpl_mask * bpm = create_test_mask(nx, ny);
    cpl_image * confidence = create_test_confidence(nx, ny, 100.0);
    
    master_wave * result = en_master_wave_create(raw_wave, bpm, confidence,
                                                  NULL, NULL, NULL,
                                                  NULL, NULL);
    cpl_test_nonnull(result);
    
    /* Verify dimensions */
    cpl_test_eq(hdrl_image_get_size_x(result->raw_wave_image), nx);
    cpl_test_eq(hdrl_image_get_size_y(result->raw_wave_image), ny);
    
    en_master_wave_delete(result);
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(bpm);
    cpl_image_delete(confidence);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_master_wave_create with bad pixels in mask
 */
static void test_create_with_bad_pixels(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, 2.2);
    cpl_mask * bpm = create_test_mask(nx, ny);
    cpl_image * confidence = create_test_confidence(nx, ny, 100.0);
    
    /* Mark some pixels as bad */
    cpl_mask_set(bpm, 10, 10, CPL_BINARY_1);
    cpl_mask_set(bpm, 20, 20, CPL_BINARY_1);
    cpl_mask_set(bpm, 30, 30, CPL_BINARY_1);
    
    master_wave * result = en_master_wave_create(raw_wave, bpm, confidence,
                                                  NULL, NULL, NULL,
                                                  NULL, NULL);
    cpl_test_nonnull(result);
    
    /* Verify bad pixels are copied */
    cpl_test_eq(cpl_mask_get(result->raw_wave_bpm, 10, 10), CPL_BINARY_1);
    cpl_test_eq(cpl_mask_get(result->raw_wave_bpm, 20, 20), CPL_BINARY_1);
    cpl_test_eq(cpl_mask_get(result->raw_wave_bpm, 30, 30), CPL_BINARY_1);
    
    /* Verify good pixels are still good */
    cpl_test_eq(cpl_mask_get(result->raw_wave_bpm, 5, 5), CPL_BINARY_0);
    
    en_master_wave_delete(result);
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(bpm);
    cpl_image_delete(confidence);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_master_wave_create with variable confidence values
 */
static void test_create_variable_confidence(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, 2.2);
    cpl_mask * bpm = create_test_mask(nx, ny);
    cpl_image * confidence = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    
    /* Set varying confidence values */
    cpl_image_fill_window(confidence, 1, 1, nx/2, ny, 100.0);
    cpl_image_fill_window(confidence, nx/2+1, 1, nx, ny, 50.0);
    
    master_wave * result = en_master_wave_create(raw_wave, bpm, confidence,
                                                  NULL, NULL, NULL,
                                                  NULL, NULL);
    cpl_test_nonnull(result);
    
    /* Verify confidence values are copied correctly */
    int reject = 0;
    double conf_high = cpl_image_get(result->raw_wave_confidence, 10, 10, &reject);
    cpl_test_abs(conf_high, 100.0, DBL_EPSILON);
    
    double conf_low = cpl_image_get(result->raw_wave_confidence, nx-5, 10, &reject);
    cpl_test_abs(conf_low, 50.0, DBL_EPSILON);
    
    en_master_wave_delete(result);
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(bpm);
    cpl_image_delete(confidence);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test that error image values are correctly copied
 */
static void test_create_error_values(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    const double data_value = 4.0;
    const double error_value = 2.0;  /* sqrt(4) = 2 */
    
    hdrl_image * raw_wave = hdrl_image_new(nx, ny);
    cpl_image_fill_window(hdrl_image_get_image(raw_wave), 1, 1, nx, ny, data_value);
    cpl_image_fill_window(hdrl_image_get_error(raw_wave), 1, 1, nx, ny, error_value);
    
    cpl_mask * bpm = create_test_mask(nx, ny);
    cpl_image * confidence = create_test_confidence(nx, ny, 100.0);
    
    master_wave * result = en_master_wave_create(raw_wave, bpm, confidence,
                                                  NULL, NULL, NULL,
                                                  NULL, NULL);
    cpl_test_nonnull(result);
    
    /* Verify error values are copied correctly */
    int reject = 0;
    hdrl_value val = hdrl_image_get_pixel(result->raw_wave_image, 32, 32, &reject);
    cpl_test_abs(val.data, data_value, DBL_EPSILON);
    cpl_test_abs(val.error, error_value, DBL_EPSILON);
    
    en_master_wave_delete(result);
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(bpm);
    cpl_image_delete(confidence);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test that modifying original data doesn't affect master_wave
 */
static void test_create_data_independence(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    const double original_value = 2.2;
    const double modified_value = 9.9;
    
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, original_value);
    cpl_mask * bpm = create_test_mask(nx, ny);
    cpl_image * confidence = create_test_confidence(nx, ny, 100.0);
    
    master_wave * result = en_master_wave_create(raw_wave, bpm, confidence,
                                                  NULL, NULL, NULL,
                                                  NULL, NULL);
    cpl_test_nonnull(result);
    
    /* Modify the original data */
    cpl_image_fill_window(hdrl_image_get_image(raw_wave), 1, 1, nx, ny, modified_value);
    cpl_image_fill_window(confidence, 1, 1, nx, ny, 0.0);
    cpl_mask_set(bpm, 1, 1, CPL_BINARY_1);
    
    /* Verify master_wave data is unchanged */
    int reject = 0;
    hdrl_value val = hdrl_image_get_pixel(result->raw_wave_image, 32, 32, &reject);
    cpl_test_abs(val.data, original_value, DBL_EPSILON);
    
    double conf_val = cpl_image_get(result->raw_wave_confidence, 32, 32, &reject);
    cpl_test_abs(conf_val, 100.0, DBL_EPSILON);
    
    cpl_test_eq(cpl_mask_get(result->raw_wave_bpm, 1, 1), CPL_BINARY_0);
    
    en_master_wave_delete(result);
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(bpm);
    cpl_image_delete(confidence);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_master_wave_create with partial calibrated inputs
 */
static void test_create_partial_calibrated(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, 2.2);
    cpl_mask * raw_bpm = create_test_mask(nx, ny);
    cpl_image * raw_confidence = create_test_confidence(nx, ny, 100.0);
    
    /* Only provide calibrated image, not bpm or confidence */
    hdrl_image * cal_wave = create_test_hdrl_image(nx, ny, 2.18);
    
    master_wave * result = en_master_wave_create(raw_wave, raw_bpm, raw_confidence,
                                                  cal_wave, NULL, NULL,
                                                  NULL, NULL);
    cpl_test_nonnull(result);
    
    /* Verify calibrated image is set, others are NULL */
    cpl_test_nonnull(result->calibrated_wave_image);
    cpl_test_null(result->calibrated_wave_bpm);
    cpl_test_null(result->calibrated_wave_confidence);
    
    en_master_wave_delete(result);
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(raw_bpm);
    cpl_image_delete(raw_confidence);
    hdrl_image_delete(cal_wave);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_master_wave_create with only calibrated bpm
 */
static void test_create_only_calibrated_bpm(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, 2.2);
    cpl_mask * raw_bpm = create_test_mask(nx, ny);
    cpl_image * raw_confidence = create_test_confidence(nx, ny, 100.0);
    
    /* Only provide calibrated bpm */
    cpl_mask * cal_bpm = create_test_mask(nx, ny);
    cpl_mask_set(cal_bpm, 15, 15, CPL_BINARY_1);
    
    master_wave * result = en_master_wave_create(raw_wave, raw_bpm, raw_confidence,
                                                  NULL, cal_bpm, NULL,
                                                  NULL, NULL);
    cpl_test_nonnull(result);
    
    /* Verify only calibrated bpm is set */
    cpl_test_null(result->calibrated_wave_image);
    cpl_test_nonnull(result->calibrated_wave_bpm);
    cpl_test_null(result->calibrated_wave_confidence);
    
    /* Verify bad pixel was copied */
    cpl_test_eq(cpl_mask_get(result->calibrated_wave_bpm, 15, 15), CPL_BINARY_1);
    
    en_master_wave_delete(result);
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(raw_bpm);
    cpl_image_delete(raw_confidence);
    cpl_mask_delete(cal_bpm);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_master_wave_create with long filename
 */
static void test_create_long_filename(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, 2.2);
    cpl_mask * bpm = create_test_mask(nx, ny);
    cpl_image * confidence = create_test_confidence(nx, ny, 100.0);
    
    /* Create a long filename */
    const char * long_filename = "/very/long/path/to/some/deeply/nested/directory/"
                                  "with/many/subdirectories/test_master_wave_file.fits";
    
    master_wave * result = en_master_wave_create(raw_wave, bpm, confidence,
                                                  NULL, NULL, NULL,
                                                  long_filename, NULL);
    cpl_test_nonnull(result);
    cpl_test_eq_string(result->filename, long_filename);
    
    en_master_wave_delete(result);
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(bpm);
    cpl_image_delete(confidence);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_master_wave_create with empty propertylist
 */
static void test_create_empty_plist(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, 2.2);
    cpl_mask * bpm = create_test_mask(nx, ny);
    cpl_image * confidence = create_test_confidence(nx, ny, 100.0);
    cpl_propertylist * plist = cpl_propertylist_new();  /* Empty */
    
    master_wave * result = en_master_wave_create(raw_wave, bpm, confidence,
                                                  NULL, NULL, NULL,
                                                  NULL, plist);
    cpl_test_nonnull(result);
    cpl_test_nonnull(result->plist);
    cpl_test_eq(cpl_propertylist_get_size(result->plist), 0);
    
    en_master_wave_delete(result);
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(bpm);
    cpl_image_delete(confidence);
    cpl_propertylist_delete(plist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_master_wave_create with many propertylist entries
 */
static void test_create_large_plist(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    const int nprops = 50;
    
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, 2.2);
    cpl_mask * bpm = create_test_mask(nx, ny);
    cpl_image * confidence = create_test_confidence(nx, ny, 100.0);
    cpl_propertylist * plist = cpl_propertylist_new();
    
    /* Add many properties */
    for (int i = 0; i < nprops; i++) {
        char * key = cpl_sprintf("KEYWORD%03d", i);
        cpl_propertylist_update_double(plist, key, (double)i);
        cpl_free(key);
    }
    
    master_wave * result = en_master_wave_create(raw_wave, bpm, confidence,
                                                  NULL, NULL, NULL,
                                                  NULL, plist);
    cpl_test_nonnull(result);
    cpl_test_eq(cpl_propertylist_get_size(result->plist), nprops);
    
    /* Verify some properties */
    cpl_test_abs(cpl_propertylist_get_double(result->plist, "KEYWORD000"), 0.0, DBL_EPSILON);
    cpl_test_abs(cpl_propertylist_get_double(result->plist, "KEYWORD025"), 25.0, DBL_EPSILON);
    cpl_test_abs(cpl_propertylist_get_double(result->plist, "KEYWORD049"), 49.0, DBL_EPSILON);
    
    en_master_wave_delete(result);
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(bpm);
    cpl_image_delete(confidence);
    cpl_propertylist_delete(plist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_master_wave_create with different calibrated values
 */
static void test_create_calibrated_different_values(void) {
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    const double raw_value = 2.2;
    const double cal_value = 2.18;
    
    hdrl_image * raw_wave = create_test_hdrl_image(nx, ny, raw_value);
    cpl_mask * raw_bpm = create_test_mask(nx, ny);
    cpl_image * raw_confidence = create_test_confidence(nx, ny, 100.0);
    
    hdrl_image * cal_wave = create_test_hdrl_image(nx, ny, cal_value);
    cpl_mask * cal_bpm = create_test_mask(nx, ny);
    cpl_image * cal_confidence = create_test_confidence(nx, ny, 95.0);
    
    master_wave * result = en_master_wave_create(raw_wave, raw_bpm, raw_confidence,
                                                  cal_wave, cal_bpm, cal_confidence,
                                                  NULL, NULL);
    cpl_test_nonnull(result);
    
    /* Verify raw and calibrated have different values */
    int reject = 0;
    hdrl_value raw_val = hdrl_image_get_pixel(result->raw_wave_image, 32, 32, &reject);
    hdrl_value cal_val = hdrl_image_get_pixel(result->calibrated_wave_image, 32, 32, &reject);
    
    cpl_test_abs(raw_val.data, raw_value, DBL_EPSILON);
    cpl_test_abs(cal_val.data, cal_value, DBL_EPSILON);
    cpl_test(fabs(raw_val.data - cal_val.data) > 0.01);
    
    double raw_conf = cpl_image_get(result->raw_wave_confidence, 32, 32, &reject);
    double cal_conf = cpl_image_get(result->calibrated_wave_confidence, 32, 32, &reject);
    
    cpl_test_abs(raw_conf, 100.0, DBL_EPSILON);
    cpl_test_abs(cal_conf, 95.0, DBL_EPSILON);
    
    en_master_wave_delete(result);
    hdrl_image_delete(raw_wave);
    cpl_mask_delete(raw_bpm);
    cpl_image_delete(raw_confidence);
    hdrl_image_delete(cal_wave);
    cpl_mask_delete(cal_bpm);
    cpl_image_delete(cal_confidence);
    cpl_test_error(CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_nix_master_wave module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    /* Test en_master_wave_create with NULL required inputs */
    test_create_null_raw_wave_image();
    test_create_null_raw_wave_bpm();
    test_create_null_raw_wave_confidence();
    
    /* Test en_master_wave_create with pre-existing error */
    test_create_preexisting_error();
    
    /* Test en_master_wave_create with valid inputs */
    test_create_minimal_inputs();
    test_create_all_inputs();
    test_create_pixel_values();
    
    /* Test en_master_wave_delete */
    test_delete_null();
    test_delete_valid();
    test_delete_all_fields();
    
    /* Test create/delete cycles */
    test_create_delete_cycle();
    
    /* Extended tests - different image sizes */
    test_create_tiny_image();
    test_create_larger_image();
    test_create_nonsquare_image();
    
    /* Extended tests - edge cases */
    test_create_with_bad_pixels();
    test_create_variable_confidence();
    test_create_error_values();
    test_create_data_independence();
    
    /* Extended tests - partial calibrated inputs */
    test_create_partial_calibrated();
    test_create_only_calibrated_bpm();
    
    /* Extended tests - filename and propertylist */
    test_create_long_filename();
    test_create_empty_plist();
    test_create_large_plist();
    
    /* Extended tests - calibrated values */
    test_create_calibrated_different_values();

    return cpl_test_end(0);
}

/**@}*/

