/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author$
 * $Date$
 * $Revision$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "eris_nix_test_defs.h"
#include <eris_nix_match.h>
#include <hdrl_utils.h>
#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_match_test  Unit test of eris_nix_match
 *
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                       Prototypes
 -----------------------------------------------------------------------------*/

static void eris_nix_match_test(void);

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Textual representation of CPL frame group
  @param    group     to convert
  @return   textual representation
 */
/*----------------------------------------------------------------------------*/
static void eris_nix_match_test(void)
{
    cpl_error_code code = CPL_ERROR_NONE;

    {    /* start tests for enm_correct_crpix */

    /* ..null call */

    code = enm_correct_crpix(NULL, NULL, NULL, NULL, NULL); 
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* ..valid call with null matched_stds */

    hdrl_image * himage = hdrl_image_new(TEST_SIZE_X,
                                         TEST_SIZE_Y);
    cpl_image * confidence = cpl_image_new(TEST_SIZE_X,
                                           TEST_SIZE_Y,
                                           CPL_TYPE_DOUBLE);
    cpl_image_fill_window(confidence,
                          1, 1, TEST_SIZE_X, TEST_SIZE_Y,
                          100.0);
    cpl_propertylist * plist = cpl_propertylist_new();
    cpl_propertylist_update_double(plist, "CRPIX1", 1024.0);
    cpl_propertylist_update_double(plist, "CRPIX2", 1030.0);

    located_image * limage = enu_located_image_new(
                                       himage,
                                       NULL,
                                       confidence,
                                       NULL, NULL,
                                       plist,
                                       NULL, NULL, NULL, NULL, NULL);

    cpl_matrix * xy_offset = NULL;
    code = enm_correct_crpix("test", "test1", NULL, limage, &xy_offset); 
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq_ptr(xy_offset, NULL);

    /* ..valid call with filled matched_stds */

    cpl_table * matched_stds = cpl_table_new(1);

    cpl_table_new_column(matched_stds, "X_coordinate", CPL_TYPE_DOUBLE);
    cpl_table_new_column(matched_stds, "Y_coordinate", CPL_TYPE_DOUBLE);
    cpl_table_new_column(matched_stds, "xpredict", CPL_TYPE_DOUBLE);
    cpl_table_new_column(matched_stds, "ypredict", CPL_TYPE_DOUBLE);

    cpl_table_set_double(matched_stds, "X_coordinate", 0, 14.0);
    cpl_table_set_double(matched_stds, "Y_coordinate", 0, 10.0);
    cpl_table_set_double(matched_stds, "xpredict", 0, 9.0);
    cpl_table_set_double(matched_stds, "ypredict", 0, 7.0);

    code = enm_correct_crpix("test", "test1", matched_stds, limage, &xy_offset); 
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    double crpix1 = cpl_propertylist_get_double(limage->plist, "CRPIX1");
    double crpix2 = cpl_propertylist_get_double(limage->plist, "CRPIX2");
    cpl_test_leq(crpix1-1029.0, 1e-6);
    cpl_test_leq(crpix2-1033.0, 1e-6);

    double xcorr = cpl_matrix_get(xy_offset, 0, 0);
    double ycorr = cpl_matrix_get(xy_offset, 0, 1);
    cpl_test_leq(xcorr-5.0, 1e-6);
    cpl_test_leq(ycorr-3.0, 1e-6);

    const char * wcs_method = cpl_propertylist_get_string(limage->plist,
                                                          "ESO WCS_METHOD");
    const char * wcs_cat = cpl_propertylist_get_string(limage->plist,
                                                       "WCS_CAT");
    cpl_test_eq_string(wcs_method, "test");
    cpl_test_eq_string(wcs_cat, "test1");

    /* tidy up */

    cpl_matrix_delete(xy_offset);
    cpl_table_delete(matched_stds);
    enu_located_image_delete(limage);
    
    }    /* end tests for enm_correct_crpix */

    {    /* start tests for enm_try_association */

    /* ..null call */

    code = enm_try_association(NULL, NULL, 0, NULL, NULL, 0, 0, 0, NULL); 
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    double xstd[3] = {0.0, 1.0, 2.0};
    double ystd[3] = {0.0, 1.0, 4.0};
    double xobj[2] = {0.0, 1.0};
    double yobj[2] = {0.0, 3.0};

    cpl_matrix * distances = NULL;

    code = enm_try_association(xstd, ystd, 3,
                               xobj, yobj, 2,
                               2, 1,
                               &distances);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    //cpl_matrix_dump(distances, NULL);
    double dist = cpl_matrix_get(distances, 1, 0);
    cpl_test_leq(dist, 1e-6);
    dist = cpl_matrix_get(distances, 2, 1);
    cpl_test_leq(dist, 1e-6);

    cpl_matrix_delete(distances);

    }    /* end tests for enm_try_association */

    {    /* start tests for enm_try_all_associations */

    /* ..null call */

    code = enm_try_all_associations(NULL, NULL, 0.0, NULL); 
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_table * objtab = cpl_table_new(2);
    cpl_table_new_column(objtab, "X_coordinate", CPL_TYPE_DOUBLE);
    cpl_table_new_column(objtab, "Y_coordinate", CPL_TYPE_DOUBLE);

    cpl_table_set_double(objtab, "X_coordinate", 0, 0.0);
    cpl_table_set_double(objtab, "Y_coordinate", 0, 0.0);
    cpl_table_set_double(objtab, "X_coordinate", 1, 1.0);
    cpl_table_set_double(objtab, "Y_coordinate", 1, 3.0);

    cpl_table * stdtab = cpl_table_new(3);
    cpl_table_new_column(stdtab, "xpredict", CPL_TYPE_DOUBLE);
    cpl_table_new_column(stdtab, "ypredict", CPL_TYPE_DOUBLE);

    cpl_table_set_double(stdtab, "xpredict", 0, 0.0);
    cpl_table_set_double(stdtab, "ypredict", 0, 0.0);
    cpl_table_set_double(stdtab, "xpredict", 1, 1.0);
    cpl_table_set_double(stdtab, "ypredict", 1, 1.0);
    cpl_table_set_double(stdtab, "xpredict", 2, 2.0);
    cpl_table_set_double(stdtab, "ypredict", 2, 4.0);

    cpl_table * matched = NULL;

    code = enm_try_all_associations(objtab,
                                    stdtab,
                                    0.5,
                                    &matched); 
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_table_dump(matched, 0, 100, NULL);

    cpl_size nrow = cpl_table_get_nrow(matched);
    cpl_test_eq(nrow, 2);

    double * predict_data = cpl_table_get_data_double(matched, "xpredict");
    cpl_array * predict = cpl_array_wrap_double(predict_data, nrow);
    double predict_data_good[] = {1.0, 2.0};
    cpl_array * predict_good = cpl_array_wrap_double(predict_data_good, 2);

    cpl_test_array_abs(predict, predict_good, 1e-6);
    cpl_array_unwrap(predict);
    cpl_array_unwrap(predict_good);

    predict_data = cpl_table_get_data_double(matched, "ypredict");
    predict = cpl_array_wrap_double(predict_data, nrow);
    predict_data_good[0] = 1.0;
    predict_data_good[1] = 4.0;
    predict_good = cpl_array_wrap_double(predict_data_good, 2);

    cpl_test_array_abs(predict, predict_good, 1e-6);
    cpl_array_unwrap(predict);
    cpl_array_unwrap(predict_good);

    predict_data = cpl_table_get_data_double(matched, "X_coordinate");
    predict = cpl_array_wrap_double(predict_data, nrow);
    predict_data_good[0] = 0.0;
    predict_data_good[1] = 1.0;
    predict_good = cpl_array_wrap_double(predict_data_good, 2);

    cpl_test_array_abs(predict, predict_good, 1e-6);
    cpl_array_unwrap(predict);
    cpl_array_unwrap(predict_good);

    predict_data = cpl_table_get_data_double(matched, "Y_coordinate");
    predict = cpl_array_wrap_double(predict_data, nrow);
    predict_data_good[0] = 0.0;
    predict_data_good[1] = 3.0;
    predict_good = cpl_array_wrap_double(predict_data_good, 2);

    cpl_test_array_abs(predict, predict_good, 1e-6);
    cpl_array_unwrap(predict);
    cpl_array_unwrap(predict_good);

    cpl_table_delete(objtab);
    cpl_table_delete(stdtab);
    cpl_table_delete(matched);

    }    /* end tests for enm_try_all_associations */

}


/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_nix_match module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    eris_nix_match_test();

    return cpl_test_end(0);
}

/**@}*/
