/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author$
 * $Date$
 * $Revision$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "eris_nix_test_defs.h"
#include <eris_nix_utils.h>
#include <hdrl_utils.h>
#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_utils_test  Unit test of eris_nix_utils
 *
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                       Prototypes
 -----------------------------------------------------------------------------*/

static void eris_nix_utils_test(void);

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Textual representation of CPL frame group
  @param    group     to convert
  @return   textual representation
 */
/*----------------------------------------------------------------------------*/
static void eris_nix_utils_test(void)
{

    located_image * limage_global = NULL;


    {    /* start tests for enu_basic_calibrate */

    /* ..null call */

    cpl_error_code code = enu_basic_calibrate(NULL, CPL_FALSE, NULL, NULL, 
                                              NULL, NULL, NULL, NULL,
                                              0, "noop", 0.0, -1, -1); 
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* ..removal of read offsets and edge blanking */

    /* ....create a test located_image
           all {10, 10} with one channel {20, 10}
           top row + {1, 0}, next row down - {1, 0} */
    
    hdrl_image * himage1 = hdrl_image_new(TEST_SIZE_X,
                                          TEST_SIZE_Y);
    cpl_image_fill_window(hdrl_image_get_image(himage1),
                          1, 1, TEST_SIZE_X, TEST_SIZE_Y,
                          10.0);
    cpl_image_fill_window(hdrl_image_get_image(himage1),
                          1, TEST_SIZE_Y, TEST_SIZE_X, TEST_SIZE_Y,
                          11.0);
    cpl_image_fill_window(hdrl_image_get_image(himage1),
                          1, TEST_SIZE_Y-1, TEST_SIZE_X, TEST_SIZE_Y-1,
                          9.0);
    cpl_image_fill_window(hdrl_image_get_error(himage1),
                          1, 1, TEST_SIZE_X, TEST_SIZE_Y,
                          10.0);
    cpl_image_fill_window(hdrl_image_get_image(himage1),
                          65, 1, 128, TEST_SIZE_Y,
                          20.0);
    cpl_image_fill_window(hdrl_image_get_image(himage1),
                          65, TEST_SIZE_Y, 128, TEST_SIZE_Y,
                          21.0);
    cpl_image_fill_window(hdrl_image_get_image(himage1),
                          65, TEST_SIZE_Y-1, 128, TEST_SIZE_Y-1,
                          19.0);

    cpl_image * confidence1 = cpl_image_new(TEST_SIZE_X,
                                            TEST_SIZE_Y,
                                            CPL_TYPE_DOUBLE);
    cpl_image_fill_window(confidence1,
                          1, 1, TEST_SIZE_X, TEST_SIZE_Y,
                          100.0);

    cpl_propertylist * plist1 = cpl_propertylist_new();
    cpl_propertylist_update_string(plist1, "INSTRUME", "ERIS");
    cpl_propertylist_update_string(plist1, "ESO DPR CATG", "CALIB");
    cpl_propertylist_update_string(plist1,
                                   "ESO DET READ CURNAME",
                                   "SLOW_GR_UTR");
    cpl_propertylist_update_int(plist1, "ESO DET SEQ1 WIN ROT", 0);
    cpl_propertylist_update_int(plist1, "ESO DET SEQ1 WIN STRX", 1);
    cpl_propertylist_update_int(plist1, "ESO DET SEQ1 WIN STRY", 1);
    cpl_propertylist_update_int(plist1, "ESO DET SEQ1 WIN NX", TEST_SIZE_X);
    cpl_propertylist_update_int(plist1, "ESO DET SEQ1 WIN NY", TEST_SIZE_Y);
    cpl_propertylist_update_int(plist1, "ESO DET CHIP NX", TEST_SIZE_X);
    cpl_propertylist_update_int(plist1, "ESO DET CHIP NY", TEST_SIZE_Y);
    cpl_propertylist_update_double(plist1, "CRPIX1", TEST_SIZE_X / 2.0);
    cpl_propertylist_update_double(plist1, "CRPIX2", TEST_SIZE_Y / 2.0);
    cpl_propertylist_append_double(plist1, "MJD-OBS", 60005);
    located_image * limage1 = enu_located_image_new(
                                       himage1,
                                       NULL,
                                       confidence1,
                                       NULL, NULL,
                                       plist1,
                                       NULL, NULL, NULL, NULL, NULL);
    located_image * limage1_copy = enu_located_image_duplicate(limage1);

    /* ....check removal of read offsets */

    /* ......full chip MJD before DITDELAY fixed so don't use bottom rows*/

    cpl_propertylist_update_double(plist1, "MJD-OBS", 59000);
    code = enu_basic_calibrate(limage1,
                               CPL_TRUE, NULL, NULL, NULL, NULL, NULL, NULL,
                               0, "noop", 0.0, -1, -1); 
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    int reject;
    hdrl_value val = hdrl_image_get_pixel(limage1->himage, 512, 512, &reject);
    cpl_test_leq(fabs(val.data - 0.0), 1e-6);
    cpl_test_leq(fabs(val.error - sqrt(100.0 + (64.0 + 64.0) / 255)), 1e-6);
    cpl_test_eq((int)hdrl_image_count_rejected(limage1->himage),
                TEST_SIZE_X * 8 + (TEST_SIZE_Y - 8) * 8);

    /* ......full chip MJD after DITDELAY fixed so do use bottom rows*/
    
    enu_located_image_delete(limage1);
    limage1 = enu_located_image_duplicate(limage1_copy);
    cpl_propertylist_update_double(limage1->plist, "MJD-OBS", 60010);

    code = enu_basic_calibrate(limage1,
                               CPL_TRUE, NULL, NULL, NULL, NULL, NULL, NULL,
                               0, "noop", 0.0, -1, -1); 
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    val = hdrl_image_get_pixel(limage1->himage, 512, 512, &reject);
    cpl_test_leq(fabs(val.data - 0.0), 1e-6);
    cpl_test_leq(fabs(val.error - sqrt(100.0 + (64.0 + 64.0) / 511)), 1e-6);
    cpl_test_eq((int)hdrl_image_count_rejected(limage1->himage),
                TEST_SIZE_X * 8 + (TEST_SIZE_Y - 8) * 8);

    /* ......windowed chip, MJD after DITDELAY fixed so do use bottom rows
             make nominal chip 2048 x 2048 and test image is top right
             corner of that - result should be same as non-windowed
             when not looking at the bottom rows because the bottom
             rows aren't in the window */
    
    enu_located_image_delete(limage1);
    limage1 = enu_located_image_duplicate(limage1_copy);
    cpl_propertylist_update_double(limage1->plist, "MJD-OBS", 60010);
    cpl_propertylist_update_int(limage1->plist, "ESO DET SEQ1 WIN STRX", TEST_SIZE_X+1);
    cpl_propertylist_update_int(limage1->plist, "ESO DET SEQ1 WIN STRY", TEST_SIZE_Y+1);
    cpl_propertylist_update_int(limage1->plist, "ESO DET SEQ1 WIN NX", TEST_SIZE_X);
    cpl_propertylist_update_int(limage1->plist, "ESO DET SEQ1 WIN NY", TEST_SIZE_Y);
    cpl_propertylist_update_int(limage1->plist, "ESO DET CHIP NX", 2 * TEST_SIZE_X);
    cpl_propertylist_update_int(limage1->plist, "ESO DET CHIP NY", 2 * TEST_SIZE_Y);

    code = enu_basic_calibrate(limage1,
                               CPL_TRUE, NULL, NULL, NULL, NULL, NULL, NULL,
                               0, "noop", 0.0, -1, -1); 
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    val = hdrl_image_get_pixel(limage1->himage, 512, 512, &reject);
    cpl_test_leq(fabs(val.data - 0.0), 1e-6);
    cpl_test_leq(fabs(val.error - sqrt(100.0 + (64.0 + 64.0) / 255)), 1e-6);
    cpl_test_eq((int)hdrl_image_count_rejected(limage1->himage),
                TEST_SIZE_X * 4 + (TEST_SIZE_Y - 4) * 4);

    enu_located_image_delete(limage1);
    enu_located_image_delete(limage1_copy);


    /* test other basic reduction steps, in order */

    hdrl_image * himage = hdrl_image_new(TEST_SIZE_X,
                                         TEST_SIZE_Y);
    cpl_image_fill_window(hdrl_image_get_image(himage),
                          1, 1, TEST_SIZE_X, TEST_SIZE_Y,
                          20.0);
    cpl_image_fill_window(hdrl_image_get_error(himage),
                          1, 1, TEST_SIZE_X, TEST_SIZE_Y,
                          4.0);

    cpl_image * confidence2 = cpl_image_new(TEST_SIZE_X,
                                            TEST_SIZE_Y,
                                            CPL_TYPE_DOUBLE);
    cpl_image_fill_window(confidence2,
                          1, 1, TEST_SIZE_X, TEST_SIZE_Y,
                          5.0);

    hdrl_image * bkg = hdrl_image_duplicate(himage);
    cpl_image * bkg_confidence = cpl_image_duplicate(confidence2);

    cpl_propertylist * plist2 = cpl_propertylist_new();
    cpl_propertylist_update_string(plist2, "INSTRUME", "ERIS");
    cpl_propertylist_update_string(plist2, "ESO DPR CATG", "CALIB");
    cpl_propertylist_update_double(plist2, "MJD-OBS", 60010);
    cpl_propertylist_update_int(plist2, "ESO DET NDIT", 9);
    cpl_propertylist_update_double(plist2, "ESO DET DIT", 15.0);
    cpl_propertylist_update_int(plist2, "ESO DET NDSAMPLES", 7);
    cpl_propertylist_update_string(plist2, "ESO DET READ CURNAME",
                                   "SLOW_UP_THE_RAMP");
    cpl_propertylist_update_int(plist2, "ESO DET SEQ1 WIN ROT", 0);
    cpl_propertylist_update_int(plist2, "ESO DET SEQ1 WIN STRX", 1);
    cpl_propertylist_update_int(plist2, "ESO DET SEQ1 WIN STRY", 1);
    cpl_propertylist_update_int(plist2, "ESO DET SEQ1 WIN NX", TEST_SIZE_X);
    cpl_propertylist_update_int(plist2, "ESO DET SEQ1 WIN NY", TEST_SIZE_Y);
    cpl_propertylist_update_int(plist2, "ESO DET CHIP NX", TEST_SIZE_X);
    cpl_propertylist_update_int(plist2, "ESO DET CHIP NY", TEST_SIZE_Y);
    cpl_propertylist_update_string(plist2, "ESO PRO CATG", "CALIB_JITTER");

    cpl_frame * frame2 = cpl_frame_new();
    cpl_frame_set_filename(frame2, "nix_test_master_dark.fits");
    cpl_frame_set_tag(frame2, "JITTER");
    cpl_frame_set_group(frame2, CPL_FRAME_GROUP_RAW);
    cpl_frame_set_level(frame2, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(frame2, CPL_FRAME_TYPE_IMAGE);

    located_image * limage = enu_located_image_new(
                                       himage,
                                       NULL,
                                       confidence2,
                                       bkg,
                                       bkg_confidence,
                                       plist2,
                                       NULL, NULL, NULL, NULL,
                                       frame2);

    /* make global copy of data */

    limage_global = enu_located_image_duplicate(limage);

    /* read test master dark - created by eris_nix_master_dark-test.c */

    cpl_frameset * frameset = cpl_frameset_new();
    cpl_frameset * used = cpl_frameset_new();
    cpl_frame * frame3 = cpl_frame_new();
    cpl_frame_set_filename(frame3, "nix_test_master_dark.fits");
    cpl_frame_set_tag(frame3, "MASTER_DARK");
    cpl_frame_set_group(frame3, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame3, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(frame3, CPL_FRAME_TYPE_IMAGE);
    cpl_frameset_insert(frameset, frame3);
    cpl_frameset_insert(used, cpl_frame_duplicate(frame3));
    master_dark * m_dark = en_master_dark_load_from_frameset(frameset,
                                                             "MASTER_DARK",
                                                             used);

    /* and a master dark with mismatching properties */ 

    master_dark * bad_dark = en_master_dark_load_from_frameset(frameset,
                                                               "MASTER_DARK",
                                                               used);
    cpl_propertylist_update_double(bad_dark->plist,
                                   "ESO DET DIT",
                                   20.0);

    /* read test master_flat_hifreq - created by 
       eris_nix_master_flat-test.c */

    cpl_frame * frame4 = cpl_frame_new();
    cpl_frame_set_filename(frame4, "nix_test_master_flat_lamp_hifreq.fits");
    cpl_frame_set_tag(frame4, "MASTER_FLAT_HIFREQ");
    cpl_frame_set_group(frame4, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame4, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(frame4, CPL_FRAME_TYPE_IMAGE);
    cpl_frameset_insert(frameset, frame4);
    master_flat * m_flat_hifreq = en_master_flat_load_from_frameset(frameset,
                                       "MASTER_FLAT_HIFREQ",
                                       used,
                                       CPL_TRUE);

    /* read test master_bpm - made by eris_nix_master_bpm-test.c */

    cpl_frame * frame5 = cpl_frame_new();
    cpl_frame_set_filename(frame5, "nix_test_master_bpm.fits");
    cpl_frame_set_tag(frame5, "MASTER_BPM");
    cpl_frame_set_group(frame5, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame5, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(frame5, CPL_FRAME_TYPE_IMAGE);
    cpl_frameset_insert(frameset, frame5);
    master_bpm * m_bpm = en_master_bpm_load_from_frameset(frameset,
                                                          "MASTER_BPM",
                                                          used,
                                                          CPL_TRUE);

    /* read in demo gain linearity - created by 
       eris_nix_gain_linearity-test */

    cpl_frame * frame6 = cpl_frame_new();
    cpl_frame_set_filename(frame6, "nix_test_lin_coeffs_cube.fits");
    cpl_frame_set_tag(frame6, "LIN_COEFFS");
    cpl_frame_set_group(frame6, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame6, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(frame6, CPL_FRAME_TYPE_IMAGE);
    cpl_frameset_insert(frameset, frame6);

    cpl_frame * frame7 = cpl_frame_new();
    cpl_frame_set_filename(frame7, "nix_test_lin_bpm.fits");
    cpl_frame_set_tag(frame7, "LIN_BPM");
    cpl_frame_set_group(frame7, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame7, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(frame7, CPL_FRAME_TYPE_IMAGE);
    cpl_frameset_insert(frameset, frame7);

    cpl_frame * frame8 = cpl_frame_new();
    cpl_frame_set_filename(frame8, "nix_test_gain_table.fits");
    cpl_frame_set_tag(frame8, "GAIN_TABLE");
    cpl_frame_set_group(frame8, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame8, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(frame8, CPL_FRAME_TYPE_IMAGE);
    cpl_frameset_insert(frameset, frame8);

    gain_linearity * m_gl = engl_gain_linearity_load_from_frameset(
                                       frameset,
                                       "GAIN_TABLE",
                                       "LIN_COEFFS",
                                       "LIN_BPM",
                                       1, used);

    /* check with no calibration applied */

    code = enu_basic_calibrate(limage,
                               CPL_FALSE,
                               NULL, NULL, NULL, NULL, NULL, NULL,
                               0, "noop", 0.0, -1, -1); 
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* check dark subtraction */

    code = enu_basic_calibrate(limage,
                               CPL_FALSE,
                               NULL,
                               bad_dark,
                               NULL, NULL, NULL, NULL,
                               0, "noop", 0.0, -1, -1); 
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);
    code = enu_basic_calibrate(limage,
                               CPL_FALSE,
                               NULL, 
                               m_dark,
                               NULL, NULL, NULL, NULL,
                               0, "noop", NAN, -1, -1); 
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    val = hdrl_image_get_pixel(limage->himage,
                               10, 10,
                               NULL);
    cpl_test_leq(fabs(val.data - 13.0), DBL_EPSILON);
    cpl_test_leq(fabs(val.error - 5.0), DBL_EPSILON);

    /* check dark subtraction, divide by flatfield 1 */

    code = enu_basic_calibrate(limage,
                               CPL_FALSE,
                               NULL,
                               m_dark,
                               NULL,
                               m_flat_hifreq,
                               NULL, NULL,
                               0, "noop", 0.0, -1, -1); 

    cpl_test_eq_error(code, CPL_ERROR_NONE);

    double cdata1 = 20.0 - 7.0 - 7.0;
    double cdata = cdata1 / 2.0;
    double cerror = pow(cdata,2) * (  
                    (pow(4.0,2) + pow(3.0,2) + pow(3.0,2)) / pow(cdata1,2) +
                    pow(0.5/2.0, 2));
    cerror = sqrt(cerror);

    val = hdrl_image_get_pixel(limage->himage,
                               10, 10,
                               NULL);
    cpl_test_leq(fabs(val.data - cdata), DBL_EPSILON);
    cpl_test_leq(fabs(val.error - cerror), DBL_EPSILON);

    /* check dark subtraction, divide by flatfield 1, attach master bpm */
    /* set all bits in flag_mask so that all faults count */ 

    int flag_mask =0;
    flag_mask = ~flag_mask;
    cpl_image_fill_window(limage->confidence, 1, 1, TEST_SIZE_X, TEST_SIZE_Y, 100.0);

    code = enu_basic_calibrate(limage,
                               CPL_TRUE,
                               NULL,
                               m_dark,
                               NULL,
                               m_flat_hifreq,
                               NULL,
                               m_bpm,
                               flag_mask,
                               "set_NaN",
                               0.0,
                               -1, -1); 
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    int count = cpl_mask_count(cpl_image_get_bpm_const(
                               hdrl_image_get_image_const(limage->himage)));

    /* this test assumes read channels / detector blanking has been done */

    cpl_test_eq(count, TEST_SIZE_X * TEST_SIZE_Y - 
                (TEST_SIZE_X-8) * (TEST_SIZE_Y-8) + 3);


    /* check dark subtraction, linearise, divide by flatfield 1, 
       attach bpm */

    /* first, reset data to avoid next dark-subtract going -ve */

    enu_located_image_delete(limage);
    limage = enu_located_image_duplicate(limage_global);
    code = enu_basic_calibrate(limage,
                               CPL_FALSE,
                               NULL,
                               m_dark,
                               m_gl,
                               m_flat_hifreq,
                               NULL,
                               m_bpm,
                               flag_mask,
                               "set_NaN",
                               0.0,
                               -1, -1); 
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    val = hdrl_image_get_pixel(limage->himage,
                               10, 10,
                               NULL);
    /*cpl_msg_info(cpl_func, "val %20.16f %20.16f", val.data, val.error);*/
    cpl_test_leq(fabs(val.data - 0.4333629911027519), DBL_EPSILON);
    cpl_test_leq(fabs(val.error - 0.1094107076228141), DBL_EPSILON);

    /* for cube, check dark subtraction, linearise, divide by flatfield 1, 
       attach bpm */

    hdrl_imagelist * himagelist = hdrl_imagelist_new();
    hdrl_imagelist_set(himagelist,
                       hdrl_image_duplicate(limage_global->himage),
                       0);
    hdrl_imagelist_set(himagelist,
                       hdrl_image_duplicate(limage_global->himage),
                       1);
    located_image * limage2 = enu_located_image_new(
                                       NULL,
                                       himagelist,
                                       cpl_image_duplicate(limage_global->confidence),
                                       NULL,
                                       NULL,
                                       cpl_propertylist_duplicate(limage_global->plist),
                                       NULL, NULL, NULL, NULL,
                                       cpl_frame_duplicate(limage_global->frame));

    code = enu_basic_calibrate(limage2,
                               CPL_FALSE,
                               NULL,
                               m_dark,
                               m_gl,
                               m_flat_hifreq,
                               NULL,
                               m_bpm,
                               flag_mask,
                               "set_NaN",
                               0.0,
                               -1, -1); 
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    hdrl_image * himage2 = hdrl_imagelist_get(limage2->himagelist, 0);
    val = hdrl_image_get_pixel(himage2,
                               10, 10,
                               NULL);
    count = hdrl_image_count_rejected(himage2);
    /*cpl_msg_info(cpl_func, "val %20.16f %20.16f count %d", val.data, val.error,
                 (int)count);*/

    cpl_test_leq(fabs(val.data - 0.4333629911027519), DBL_EPSILON);
    cpl_test_leq(fabs(val.error - 0.1094107076228141), DBL_EPSILON);
    cpl_test_eq(count, 3);

    /* tidy up */

    enu_located_image_delete(limage);
    enu_located_image_delete(limage2);
    en_master_dark_delete(m_dark); m_dark = NULL;
    en_master_dark_delete(bad_dark); bad_dark = NULL;
    en_master_flat_delete(m_flat_hifreq); m_flat_hifreq = NULL;
    en_master_bpm_delete(m_bpm); m_bpm = NULL;
    engl_gain_linearity_delete(m_gl);

    cpl_frameset_delete(frameset); frameset = NULL;
    cpl_frameset_delete(used); used = NULL;
    remove("nix_test_gain_table.fits");
    remove("nix_test_lin_bpm.fits");
    remove("nix_test_lin_coeffs_cube.fits");
    remove("nix_test_master_bpm.fits");
    remove("nix_test_master_dark.fits");
    remove("nix_test_master_flat_lamp_hifreq.fits");

    }    /* end tests for enu_basic_calibrate */


    {    /* start tests for repreface */

    char * ctest = enu_repreface("old.older.very_old.ERIS.20blah",
                                 "new");
    cpl_test_eq_string(ctest, "new.ERIS.20blah");
    cpl_free(ctest);

    }    /* end tests for repreface */


    {    /* start tests for enu_calc_pixel_coords */

    /* test files are written by enu_calc_pixel_coords by commented-out 
       code. The wcs plist is incomplete and needs NAXIS, NAXIS1, NAXIS2
       added to be functional */



    cpl_propertylist *  enu_calc_pixel_coords_test_wcs = cpl_propertylist_load(
                        DATADIR "/enu_calc_pixel_coords_test_wcs.fits", 0);
    cpl_propertylist_erase_regexp(enu_calc_pixel_coords_test_wcs, 
                                  CPL_WCS_REGEXP, 1);
    cpl_propertylist_update_int(enu_calc_pixel_coords_test_wcs, "NAXIS", 2);
    cpl_propertylist_update_int(enu_calc_pixel_coords_test_wcs, "NAXIS1",
                                2048);
    cpl_propertylist_update_int(enu_calc_pixel_coords_test_wcs, "NAXIS2",
                                2048);

    /* ..null inputs */

    enu_calc_pixel_coords(NULL, NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    enu_calc_pixel_coords(NULL, enu_calc_pixel_coords_test_wcs);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* ..valid input from test files */

    cpl_table * enu_calc_pixel_coords_test_table = cpl_table_load(
                DATADIR "/enu_calc_pixel_coords_test_input_table.fits",
                1, 0);
    enu_calc_pixel_coords(enu_calc_pixel_coords_test_table,
                          enu_calc_pixel_coords_test_wcs);
    cpl_test_error(CPL_ERROR_NONE);

    /* ..compare output with test file */

    cpl_table * enu_calc_pixel_coords_compare_table = cpl_table_load(
                DATADIR "/enu_calc_pixel_coords_test_output_table.fits",
                1, 0);
    cpl_size nrow = cpl_table_get_nrow(enu_calc_pixel_coords_test_table);

    /* ..X_coordinate */

    double * x_coord = cpl_table_get_data_double(
                                       enu_calc_pixel_coords_test_table,
                                       "X_coordinate");
    cpl_array * x_coord_array = cpl_array_wrap_double(x_coord, nrow);
    double * x_coord_compare = cpl_table_get_data_double(
                                       enu_calc_pixel_coords_compare_table,
                                       "X_coordinate");
    cpl_array * x_coord_compare_array = cpl_array_wrap_double(x_coord_compare,
                                                              nrow);
    cpl_test_array_abs(x_coord_array, x_coord_compare_array, 1.0e-6);

    /* ..Y_coordinate */

    double * y_coord = cpl_table_get_data_double(
                                       enu_calc_pixel_coords_test_table,
                                       "Y_coordinate");
    cpl_array * y_coord_array = cpl_array_wrap_double(y_coord, nrow);
    double * y_coord_compare = cpl_table_get_data_double(
                                       enu_calc_pixel_coords_compare_table,
                                       "Y_coordinate");
    cpl_array * y_coord_compare_array = cpl_array_wrap_double(y_coord_compare,
                                                              nrow);
    cpl_test_array_abs(y_coord_array, y_coord_compare_array, 1.0e-6);

    cpl_array_unwrap(x_coord_array);
    cpl_array_unwrap(x_coord_compare_array);
    cpl_array_unwrap(y_coord_array);
    cpl_array_unwrap(y_coord_compare_array);
    cpl_propertylist_delete(enu_calc_pixel_coords_test_wcs);
    cpl_table_delete(enu_calc_pixel_coords_test_table);
    cpl_table_delete(enu_calc_pixel_coords_compare_table);

    }    /* end tests for enu_calc_pixel_coords */


    {    /* start test of enu_check_wcs */

    /* ..null inputs */

    enu_check_wcs(NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* ..invalid input from test image list */

    located_image * limage3 = enu_located_image_new(
                                       NULL,
                                       NULL,
                                       NULL,
                                       NULL,
                                       NULL,
                                       cpl_propertylist_new(),
                                       NULL, NULL, NULL, NULL, NULL);

    enu_check_wcs(limage3);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_propertylist_update_string(limage3->plist, "CTYPE1",
                                   "PIXEL ");
    cpl_propertylist_update_string(limage3->plist, "CTYPE2",
                                   "PIXEL ");
    enu_check_wcs(limage3);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    /* ..valid input */
    cpl_propertylist_update_string(limage3->plist, "CTYPE1",
                                   "RA---TAN");
    cpl_propertylist_update_string(limage3->plist, "CTYPE2",
                                   "DEC--TAN");
    enu_check_wcs(limage3);
    cpl_test_error(CPL_ERROR_NONE);

    enu_located_image_delete(limage3);

    }    /* end test of enu_check_wcs */

    {    /* start test of enu_calc_maglim */

    /* ..null inputs */

    enu_calc_maglim(NULL, 0.0, 0.0, NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* ..valid input */

    located_image * limage = enu_located_image_duplicate(limage_global);
    double abmaglim = 0.0;

    enu_calc_maglim(limage, 15.0, 10.0, &abmaglim);
    cpl_test_error(CPL_ERROR_NONE);

    enu_located_image_delete(limage);

    }    /* end test of enu_calc_maglim */

    {    /* start test of enu_modify_CD_matrix */

    /* ..null inputs */

    enu_modify_CD_matrix(NULL, NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* ..invalid input from test image list */

    cpl_table * wcs_refine = cpl_table_new(1);

    located_image * limage3 = enu_located_image_new(
                                       NULL,
                                       NULL,
                                       NULL,
                                       NULL,
                                       NULL,
                                       cpl_propertylist_new(),
                                       NULL, NULL, NULL, NULL, NULL);

    enu_modify_CD_matrix(limage3, wcs_refine);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    cpl_table_new_column(wcs_refine, "camera", CPL_TYPE_STRING);
    cpl_table_new_column(wcs_refine, "CD1_1", CPL_TYPE_DOUBLE);
    cpl_table_new_column(wcs_refine, "CD1_2", CPL_TYPE_DOUBLE);
    cpl_table_new_column(wcs_refine, "CD2_1", CPL_TYPE_DOUBLE);
    cpl_table_new_column(wcs_refine, "CD2_2", CPL_TYPE_DOUBLE);

    cpl_table_set_string(wcs_refine, "camera", 0, "27mas-JHK");
    cpl_table_set_double(wcs_refine, "CD1_1", 0, 41.0);
    cpl_table_set_double(wcs_refine, "CD1_2", 0, 42.0);
    cpl_table_set_double(wcs_refine, "CD2_1", 0, 43.0);
    cpl_table_set_double(wcs_refine, "CD2_2", 0, 44.0);

    cpl_propertylist_update_string(limage3->plist,
                                   "ESO INS2 NXCW NAME",
                                   "blah");

    enu_modify_CD_matrix(limage3, wcs_refine);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    /* ..valid input */

    cpl_propertylist_update_string(limage3->plist,
                                   "ESO INS2 NXCW NAME",
                                   "27mas-JHK");
    enu_modify_CD_matrix(limage3, wcs_refine);
    cpl_test_error(CPL_ERROR_NONE);

    double test = cpl_propertylist_get_double(limage3->plist, "CD1_1");
    cpl_test_leq(fabs(test - 41.0), 1e-6);
    test = cpl_propertylist_get_double(limage3->plist, "CD2_1");
    cpl_test_leq(fabs(test - 43.0), 1e-6);
    test = cpl_propertylist_get_double(limage3->plist, "CD1_2");
    cpl_test_leq(fabs(test - 42.0), 1e-6);
    test = cpl_propertylist_get_double(limage3->plist, "CD2_2");
    cpl_test_leq(fabs(test - 44.0), 1e-6);

    enu_located_image_delete(limage3);
    cpl_table_delete(wcs_refine);

    }    /* end test of enu_modify_CD_matrix */

    /* tidy up */

    enu_located_image_delete(limage_global);


    {    /* start test of enu_get_window_info */

    /* ..null inputs */

    enu_get_window_info(NULL,
                        NULL,
                        NULL,
                        NULL,
                        NULL,
                        NULL,
                        NULL,
                        NULL,
                        NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* ..valid inputs */

    cpl_size nxw = -1;
    cpl_size nyw = -1;
    int rotw = -1;
    cpl_size strxw = -1;
    cpl_size stryw = -1;
    cpl_size nx_chipw = -1;
    cpl_size ny_chipw = -1;
    cpl_boolean windowed = CPL_FALSE;

    cpl_propertylist * plistw = cpl_propertylist_new();

    cpl_propertylist_update_string(plistw, "INSTRUME", "ERIS");
    cpl_propertylist_update_int(plistw, "ESO DET SEQ1 WIN NX", 500);
    cpl_propertylist_update_int(plistw, "ESO DET SEQ1 WIN NY", 400);
    cpl_propertylist_update_int(plistw, "ESO DET SEQ1 WIN ROT", 0);
    cpl_propertylist_update_int(plistw, "ESO DET SEQ1 WIN STRX", 600);
    cpl_propertylist_update_int(plistw, "ESO DET SEQ1 WIN STRY", 700);
    cpl_propertylist_update_int(plistw, "ESO DET CHIP NX", 2048);
    cpl_propertylist_update_int(plistw, "ESO DET CHIP NY", 2048);

    enu_get_window_info(&nxw,
                        &nyw,
                        &rotw,
                        &strxw,
                        &stryw,
                        &nx_chipw,
                        &ny_chipw,
                        &windowed,
                        plistw);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_eq(nxw, 500);
    cpl_test_eq(nyw, 400);
    cpl_test_eq(rotw, 0);
    cpl_test_eq(strxw, 600);
    cpl_test_eq(stryw, 700);
    cpl_test_eq(nx_chipw, 2048);
    cpl_test_eq(ny_chipw, 2048);
    cpl_test_eq(windowed, CPL_TRUE);

    /* ..check for corrupted result when result variables 
         duplicated */

    cpl_size ignore = -1;
    int iignore = -1;

    cpl_propertylist_update_int(plistw, "ESO DET SEQ1 WIN NX", 2048);
    cpl_propertylist_update_int(plistw, "ESO DET SEQ1 WIN NY", 2048);
    cpl_propertylist_update_int(plistw, "ESO DET SEQ1 WIN ROT", 0);
    cpl_propertylist_update_int(plistw, "ESO DET SEQ1 WIN STRX", 1);
    cpl_propertylist_update_int(plistw, "ESO DET SEQ1 WIN STRY", 1);
    cpl_propertylist_update_int(plistw, "ESO DET CHIP NX", 2048);
    cpl_propertylist_update_int(plistw, "ESO DET CHIP NY", 2048);

    enu_get_window_info(&ignore,
                        &ignore,
                        &iignore,
                        &ignore,
                        &ignore,
                        &ignore,
                        &ignore,
                        &windowed,
                        plistw);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(windowed, CPL_FALSE);


    cpl_propertylist_delete(plistw);

    }    /* end tests for enu_calc_pixel_coords */


    {    /* start test of enu_get_filter and enu_get_filter_wavelength */

    /* ..null inputs */

    const char * filter = enu_get_filter(NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    double lambda = enu_get_filter_wavelength(NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* ..valid inputs */

    cpl_propertylist * plist = cpl_propertylist_new();
    cpl_propertylist_update_string(plist, "INSTRUME", "ERIS");
    cpl_propertylist_update_string(plist, "ESO INS2 NXFW NAME", "Ks");

    filter = enu_get_filter(plist);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq_string(filter, "Ks");

    lambda = enu_get_filter_wavelength("J");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(lambda - 1.28), 1e-6);

    lambda = enu_get_filter_wavelength("H");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(lambda - 1.66), 1e-6);

    lambda = enu_get_filter_wavelength("Ks");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(lambda - 2.18), 1e-6);

    lambda = enu_get_filter_wavelength("Short-Lp");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(lambda - 3.32), 1e-6);

    lambda = enu_get_filter_wavelength("L-Broad");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(lambda - 3.57), 1e-6);

    lambda = enu_get_filter_wavelength("Lp");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(lambda - 3.79), 1e-6);

    lambda = enu_get_filter_wavelength("Mp");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(lambda - 4.78), 1e-6);

    lambda = enu_get_filter_wavelength("Pa-b");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(lambda - 1.282), 1e-6);

    lambda = enu_get_filter_wavelength("Fe-II");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(lambda - 1.644), 1e-6);

    lambda = enu_get_filter_wavelength("H2-cont");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(lambda - 2.068), 1e-6);

    lambda = enu_get_filter_wavelength("H2-1-0S");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(lambda - 2.120), 1e-6);

    lambda = enu_get_filter_wavelength("Br-g");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(lambda - 2.172), 1e-6);

    lambda = enu_get_filter_wavelength("K-peak");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(lambda - 2.198), 1e-6);

    lambda = enu_get_filter_wavelength("IB-2.42");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(lambda - 2.420), 1e-6);

    lambda = enu_get_filter_wavelength("IB-2.48");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(lambda - 2.479), 1e-6);

    lambda = enu_get_filter_wavelength("Br-a-cont");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(lambda - 3.965), 1e-6);

    lambda = enu_get_filter_wavelength("Br-a");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(lambda - 4.051), 1e-6);

    cpl_propertylist_delete(plist);

    }    /* end tests for enu_get_filter and enu_get_filter_wavelength */

    {    /* start test of enu_get_filter and enu_get_ra_dec */

    /* ..null inputs */

    enu_get_ra_dec(NULL, NULL, NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* ..valid inputs */

    /* ..Create the WCS header */

    #define NSK 2
    #define NDK 8
    #define NIK 3

    const char * skeys[NSK] = {"CTYPE1", "CTYPE2"};
    const char * dkeys[NDK] = {"CRVAL1", "CRVAL2", "CRPIX1", "CRPIX2", "CD1_1",
                               "CD1_2",  "CD2_1",  "CD2_2"};
    const char * ikeys[NIK] = {"NAXIS", "NAXIS1", "NAXIS2"};

    const char * svals[NSK] = {"RA---TAN", "DEC--TAN"};
    const double dvals[NDK] = {5.57368333333,
                               -72.0576388889,
                               1024.0,
                               1024.0,
                               0.0,
                               10.0E-05,
                               10.0E-05,
                               0.0};
    const int ivals[NIK] = {2, 2048, 2048};

    cpl_propertylist * pl = cpl_propertylist_new();
    for (cpl_size i = 0; i < NSK; i++)
        cpl_propertylist_append_string(pl, skeys[i], svals[i]);
    for (cpl_size i = 0; i < NDK; i++)
        cpl_propertylist_append_double(pl, dkeys[i], dvals[i]);
    for (cpl_size i = 0; i < NIK; i++)
        cpl_propertylist_append_int(pl, ikeys[i], ivals[i]);

    /* ..and the wcs */

    cpl_wcs * wcs = cpl_wcs_new_from_propertylist(pl);
    double ra = 0.0;
    double dec = 0.0;

    /* ..the RA, dec should just come out as the reference values for this
         simple wcs */
    enu_get_ra_dec(wcs, &ra, &dec);

    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(ra - 5.5736833333), 1e-6);
    cpl_test_leq(fabs(dec - -72.0576388889), 1e-6);

    cpl_propertylist_delete(pl);
    cpl_wcs_delete(wcs);

    }    /* end tests for enu_get_filter and enu_get_ra_dec */

    {    /* start test of enu_get_rcore_and_mesh_size */

    /* ..null inputs */

    enu_get_rcore_and_mesh_size(NULL,
                                NULL,
                                NULL,
                                NULL,
                                NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* ..real inputs but with empty parameter list */

    cpl_parameterlist * parlist = cpl_parameterlist_new();
    cpl_propertylist * plistw = cpl_propertylist_new();
    double core_radius = -1.0;
    int mesh_size = -1;
    enu_get_rcore_and_mesh_size("TEST",
                                parlist,
                                plistw,
                                &core_radius,
                                &mesh_size);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* ..with OK parlist but empty propertylist - should work and
         return parameter defaults */

    cpl_parameterlist_delete(parlist);
    hdrl_parameter * catalogue_defaults = hdrl_catalogue_parameter_create(
                                       20, 3.0, CPL_TRUE,
                                       5.0, CPL_TRUE, 128, 5.0, 3.0,
                                       1000, HDRL_CATALOGUE_ALL);
    parlist = hdrl_catalogue_parameter_create_parlist(
                                "TEST",
                                "catalogue",
                                catalogue_defaults);
    cpl_parameter * p = cpl_parameter_new_enum(
                                       "TEST.catalogue.ao-params",
                                       CPL_TYPE_STRING,
                                       "Default catalogue.core-radius and "
                                       "catalogue.mesh-size depending on "
                                       "AOMODE, or not",
                                       "TEST",
                                       "auto", 2, "auto", "user");
    cpl_parameterlist_append(parlist, p);

    enu_get_rcore_and_mesh_size("TEST",
                                parlist,
                                plistw,
                                &core_radius,
                                &mesh_size);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(core_radius - 5.0), 1e-6);
    cpl_test_eq(mesh_size, 128);

    /* ..with OK parlist and propertylist - should return AO defaults */

    cpl_propertylist_update_string(plistw, "ESO OBS AOMODE", "FULL_AO");
    enu_get_rcore_and_mesh_size("TEST",
                                parlist,
                                plistw,
                                &core_radius,
                                &mesh_size);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(core_radius - 10.0), 1e-6);
    cpl_test_eq(mesh_size, 64);

    cpl_propertylist_update_string(plistw, "ESO OBS AOMODE", "NO_AO");
    enu_get_rcore_and_mesh_size("TEST",
                                parlist,
                                plistw,
                                &core_radius,
                                &mesh_size);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(core_radius - 25.0), 1e-6);
    cpl_test_eq(mesh_size, 128);

    /* ..with OK parlist but and propertylist but with ao-params
         set to 'user' - should return parameter defaults */
    /* Reset 2 ao-params to defaults, as previous calls update it on the fly */
    char * param_name = NULL;
    cpl_msg_warning(cpl_func, "test user %s", cpl_error_get_message());
    param_name = cpl_sprintf("TEST.catalogue.obj.core-radius");
    p = (cpl_parameter*) cpl_parameterlist_find_const(parlist, param_name);
    cpl_parameter_set_double(p, 5.0);
    cpl_free(param_name);

    param_name = cpl_sprintf("TEST.catalogue.bkg.mesh-size");
    p = (cpl_parameter*) cpl_parameterlist_find_const(parlist, param_name);
    cpl_parameter_set_int(p, 128);
    cpl_free(param_name);
    
    param_name = cpl_sprintf("TEST.catalogue.ao-params");
    p = (cpl_parameter*) cpl_parameterlist_find_const(parlist, param_name);
    cpl_parameter_set_string(p, "user");
    cpl_free(param_name);
   
    enu_get_rcore_and_mesh_size("TEST",
                                parlist,
                                plistw,
                                &core_radius,
                                &mesh_size);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_leq(fabs(core_radius - 5.0), 1e-6);
    cpl_test_eq(mesh_size, 128);

    /* clean up */

    hdrl_parameter_delete(catalogue_defaults);
    cpl_parameterlist_delete(parlist);
    cpl_propertylist_delete(plistw);

    }    /* end tests for enu_get_rcore_and_mesh_size */
}



/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_nix_utils module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    eris_nix_utils_test();

    return cpl_test_end(0);
}

/**@}*/
