/* -*- Mode: C; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set sw=2 sts=2 et cin: */
/*
 * This file is part of the MUSE Instrument Pipeline
 * Copyright (C) 2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*
 *                             Includes                                       *
 *----------------------------------------------------------------------------*/
#include <string.h>
#include <eris_pfits.h>
#include "eris_ifu_sdp.h"
#include "eris_ifu_dfs.h"
#include <math.h>
#include <cpl.h>
#include <eris_utils.h>
#include <eris_ifu_utils.h>
#include <eris_ifu_wavecal_static.h>
/*
#include "muse_dfs.h"
#include "muse_pfits.h"
#include "muse_rtcdata.h"
#include "muse_pixtable.h"
#include "muse_cplwrappers.h"
#include "muse_wcs.h"
#include "muse_flux.h"
#include "muse_utils.h"
 */

/*----------------------------------------------------------------------------*
 *                              Defines                                       *
 *----------------------------------------------------------------------------*/

#define KEY_ASSON                    "ASSON"
#define KEY_ASSON_COMMENT            "Associated file name"
#define KEY_ASSOC                    "ASSON"
#define KEY_ASSOC_COMMENT            "Associated file category"
#define KEY_ASSOM                    "ASSOM"
#define KEY_ASSOM_COMMENT            "Associated file datamd5"

#define KEY_DEC                      "DEC"
#define KEY_DEC_COMMENT              "[deg] Image center (J2000)"
#define KEY_EXPTIME                  "EXPTIME"
#define KEY_EXPTIME_COMMENT          "[s] Total integration time per pixel"
#define KEY_FLUXCAL                  "FLUXCAL"
#define KEY_FLUXCAL_COMMENT          "Type of flux calibration (ABSOLUTE or UNCALIBRATED)"
#define KEY_FLUXCAL_VALUE_FALSE      "UNCALIBRATED"
#define KEY_FLUXCAL_VALUE_TRUE       "ABSOLUTE"
#define KEY_MJDOBS                   "MJD-OBS"
#define KEY_MJDOBS_COMMENT           "[d] Start of observations (days)"
#define KEY_MJDEND                   "MJD-END"
#define KEY_MJDEND_COMMENT           "[d] End of observations (days)"
#define KEY_OBID                     "OBID"
#define KEY_OBID1                    "OBID1"
#define KEY_OBID_COMMENT             "Observation block ID"
#define KEY_OBSTECH                  "OBSTECH"
#define KEY_OBSTECH_COMMENT          "Technique for observation"
#define KEY_PROCSOFT                 "PROCSOFT"
#define KEY_PROCSOFT_COMMENT         "ESO pipeline version"
#define KEY_PRODCATG                 "PRODCATG"
#define KEY_PRODCATG_COMMENT         "Data product category"
#define KEY_PRODCATG_VALUE_IFS_CUBE  "SCIENCE.CUBE.IFS"
#define KEY_PRODCATG_VALUE_FOV_IMAGE "ANCILLARY.IMAGE"
#define KEY_PROG_ID                  "PROG_ID"
#define KEY_PROG_ID_COMMENT          "ESO programme identification"
#define KEY_PROG_ID_VALUE_MULTIPLE   "MULTI"
#define KEY_PROGID                   "PROGID"
#define KEY_PROGID_COMMENT           KEY_PROG_ID_COMMENT
#define KEY_PROV                     "PROV"
#define KEY_PROV_COMMENT             "Originating raw science file"
#define KEY_RA                       "RA"
#define KEY_RA_COMMENT               "[deg] Image center (J2000)"
#define KEY_REFERENC                 "REFERENC"
#define KEY_REFERENC_COMMENT         "Reference publication"
#define KEY_TEXPTIME                 "TEXPTIME"
#define KEY_TEXPTIME_COMMENT         "[s] Total integration time of all exposures"
#define KEY_WAVELMIN                 "WAVELMIN"
#define KEY_WAVELMIN_COMMENT         "[nm] Minimum wavelength"
#define KEY_WAVELMAX                 "WAVELMAX"
#define KEY_WAVELMAX_COMMENT         "[nm] Maximum wavelength"
#define KEY_SKY_RES                  "SKY_RES"
#define KEY_SKY_RES_COMMENT          "[arcsec] FWHM effective spatial resolution"
#define KEY_SKY_RERR                 "SKY_RERR"
#define KEY_SKY_RERR_COMMENT         "[arcsec] Error of SKY_RES"
#define KEY_SPEC_RES                 "SPEC_RES"
#define KEY_SPEC_RES_COMMENT         "Spectral resolving power at central wavelength"
#define KEY_SPEC_ERR                 "CRDER3"
#define KEY_SPEC_ERR_COMMENT         "[angstrom] Random error in spectral coordinate"
#define KEY_SPECSYS                  "SPECSYS"
#define KEY_SPECSYS_COMMENT          "Frame of reference for spectral coordinates."
#define KEY_PIXNOISE                 "PIXNOISE"
#define KEY_PIXNOISE_COMMENT         "[erg.s**(-1).cm**(-2).angstrom**(-1)] pixel-to-pixel noise"
#define KEY_ABMAGLIM                 "ABMAGLIM"
#define KEY_ABMAGLIM_COMMENT         "5-sigma magnitude limit for point sources"
#define KEY_NCOMBINE                 "NCOMBINE"
#define KEY_NCOMBINE_COMMENT         "No. of combined raw science data files"

/* A regular expression matching SDP IFS standard keywords to be cleared. */
#define CLEAN_KEYS_REGEXP \
		"^(" KEY_MJDEND "|" \
				KEY_PROCSOFT "|" \
				KEY_PRODCATG "|" \
				KEY_PROG_ID "|" \
				KEY_PROGID "[0-9]+|" \
				KEY_OBID "[0-9]+|" \
				KEY_OBSTECH "|" \
				KEY_FLUXCAL "|" \
				KEY_TEXPTIME "|" \
				KEY_WAVELMIN "|" \
				KEY_WAVELMAX "|" \
				KEY_SKY_RES "|" \
				KEY_SKY_RERR "|" \
				KEY_SPEC_RES "|" \
				KEY_PIXNOISE "|" \
				KEY_ABMAGLIM "|" \
				KEY_REFERENC "|" \
				KEY_NCOMBINE "|" \
				KEY_PROV "[0-9]+|" \
				KEY_ASSON "[0-9]+" ")$"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_ifu_idp   ESO Science Data Product (SDP) Format Support
 *
 * This module provides functionality for creating ESO Science Data Product (SDP)
 * compliant FITS headers and managing Phase 3 metadata for ERIS IFU data products.
 * 
 * The module handles:
 * - Collection of observation metadata (OBIDs, program IDs, exposure times, MJDs)
 * - Computation of data quality metrics (sky resolution, pixel noise, AB magnitude limits)
 * - FWHM calculation with wavelength and airmass corrections
 * - Spectral resolution determination from wavelength calibration
 * - Provenance tracking from raw data to final products
 * - Generation of Phase 3 compliant FITS keywords
 * - Wavelength range determination (WAVELMIN/WAVELMAX in nm)
 * - Flux calibration status (FLUXCAL: ABSOLUTE or UNCALIBRATED)
 * - Associated file management (ASSON keywords)
 * 
 * The SDP properties structure aggregates all required metadata which is then
 * written to FITS headers using standardized ESO keywords and formatting.
 * 
 * @{
 */
/*----------------------------------------------------------------------------*/

/* Lookup table for spectral resolution vs. wavelength */

typedef struct {
	/* Wavelength in Angstrom at which the spectral resolution was measured. */
	double lambda;
	/* Measured spectral resolution. */
	double R;
} eris_ifu_specres_lut_entry;

/* The last entry in the data tables must have a zero wavelength entry! */

//static const eris_ifu_specres_lut_entry
//eris_ifu_specres_data_wfm[] =
//{
//		{4650.0000, 1674.77},
//		{4810.3448, 1769.45},
//		{4970.6897, 1864.50},
//		{5131.0345, 1959.52},
//		{5291.3793, 2054.19},
//		{5451.7242, 2148.34},
//		{5612.0690, 2241.81},
//		{5772.4138, 2334.46},
//		{5932.7587, 2426.18},
//		{6093.1035, 2516.84},
//		{6253.4483, 2606.28},
//		{6413.7932, 2694.36},
//		{6574.1380, 2780.89},
//		{6734.4828, 2865.70},
//		{6894.8277, 2948.59},
//		{7055.1725, 3029.32},
//		{7215.5173, 3107.70},
//		{7375.8622, 3183.46},
//		{7536.2070, 3256.36},
//		{7696.5518, 3326.12},
//		{7856.8967, 3392.45},
//		{8017.2415, 3455.01},
//		{8177.5863, 3513.44},
//		{8337.9312, 3567.37},
//		{8498.2760, 3616.34},
//		{8658.6208, 3659.87},
//		{8818.9657, 3697.42},
//		{8979.3105, 3728.38},
//		{9139.6553, 3752.11},
//		{9300.0002, 3767.93},
//		{   0.0000 ,   0.00}
//};

/* XXX: Dummy values from the User Manual for the NFM. Update for production. */
//static const eris_ifu_specres_lut_entry
//eris_ifu_specres_data_nfm[] =
//{
//		{4800.0000, 1740.00},
//		{9300.0000, 3450.00},
//		{   0.0000 ,   0.00}
//};

/* Instrument properties and reference values */

/* Number of detector readout ports */
//static const unsigned int kErisNumQuadrants = 4;

/* Nominal field size [arcsec]  */
//static const double kErisFovSizeWFM = 60.00;
//static const double kErisFovSizeNFM =  7.43;

/* Nominal pixel scale wfm, nfm */
//static const double kErisPixscaleWFM = 0.2;
//static const double kErisPixscaleNFM = 0.025;

/* Reference wavelength [Angstrom] */
//static const double kErisLambdaRef = 7000.;

/* Instrument response at reference wavelength */
//static const double kErisResponseRef = 41.2;

/* Typical error estimated from unresolved sources [arcsec] */
//static const double kErisExtraErrorIQE = 0.3;

/* Typical values [arcsec] for the effective spatial resolution *
 * and its uncertainty, given by the median and the standard    *
 * deviation of previous measurements.                          */
//static const double kErisSkyResWFM      = 0.853823;
//static const double kErisSkyResErrorWFM = 0.495547;
//static const double kErisSkyResNFM      = 0.15;
//static const double kErisSkyResErrorNFM = 0.05;


/* Unit conversion factors */

static const double day2sec     = 86400.0;     /* Seconds per day       */
//static const double m2nm        = 1.e9;        /* meter to nanometer    */
//static const double nm2Angstrom = 10.;         /* nanometer to Angstrom */
//static const double deg2as      = 3600.;       /* degrees to arcseconds */


/*----------------------------------------------------------------------------*
 *                              Functions                                     *
 *----------------------------------------------------------------------------*/


///*----------------------------------------------------------------------------*/
///**
//  @private
//  @brief Compute spectral resolution for the central wavelength.
//  @param aLambda      Wavelength in Angstrom.
//  @param aSpecresLut  Spectral resolution look-up table.
//  @return The interpolated spectral resolution

//  The function computes the spectral resolution at the wavelength @em aLambda,
//  by interpolating the data points given by the look-up table @em aSpecresLut.
//  For wavelengths outside of the wavelength range of the table, the lower, or
//  the upper boundary value is returned, respectively.
// */
///*----------------------------------------------------------------------------*/
//static double
//eris_ifu_idp_compute_specres(double aLambda, const eris_ifu_specres_lut_entry *aSpecresLut)
//{

//	/* Locate the appropriate wavelength bin in the lut and compute the *
//	 * spectral by linear interpolation.                                */

//	cpl_size i = 0;
//	while ((aSpecresLut[i].lambda > 0.) && (aLambda > aSpecresLut[i].lambda)) {
//		++i;
//	}

//	if (i == 0) {
//		return aSpecresLut[0].R;
//	}
//	else if (aSpecresLut[i].lambda == 0.) {
//		return aSpecresLut[i - 1].R;
//	}

//	double t = (aLambda - aSpecresLut[i - 1].lambda) /
//			(aSpecresLut[i].lambda - aSpecresLut[i - 1].lambda);

//	return (1 - t) * aSpecresLut[i - 1].R + t * aSpecresLut[i].R;
//}

///*----------------------------------------------------------------------------*/
///**
//  @private
//  @brief Estimate image quality at a given wavelength from the seeing
//         and the airmass.
//  @param aLambda   Wavelength in Angstrom at which the image quality is
//                   computed.
//  @param aSeeing   The seeing as seen by the telescope and corrected for
//                   airmass.
//  @param aAirmass  Airmass at which the observation was carried out.
//  @return The estimate of the image quality at the given wavelength.

//  The returned image quality estimate is in units of arcsec.
// */
///*----------------------------------------------------------------------------*/
//static double
//eris_ifu_idp_compute_iqe(double aLambda, double aSeeing, double aAirmass)
//{

//	const double rad2as = 180. / CPL_MATH_PI * 3600.;  /* radians to arcsec */


//	/* Wavefront outer scale at Paranal [meters] */
//	const double L0 = 23.;

//	/* UT M1 diameter [meters] from UT M1 vignetted area */
//	const double D  = 8.1175365721399437;

//	/* Kolb factor (cf. ESO Technical Report 12) */
//	const double Fkolb = 1. / (1. + 300. * D / L0) - 1.;


//	/* Scaled wavelength: lambda over lambda_ref */
//	double lambda = aLambda / 5000.;

//	/* Fried parameter */
//	double r0 = 0.976 * 5.e-7 / aSeeing * rad2as *
//			pow(lambda, 1.2) * pow(aAirmass, -0.6);

//	double iqe = aSeeing * pow(lambda, -0.2) * pow(aAirmass, 0.6);
//	iqe *= sqrt(1. + Fkolb * 2.183 * pow(r0 / L0, 0.356));

//	return iqe;
//}

///*----------------------------------------------------------------------------*/
///**
//  @private
//  @brief Compute the effective spatial resolution from the Strehl ratio.
//  @param aSkyres      Computed effective spatial resolution.
//  @param aSkyresError Error of the computed effective spatial resolution.
//  @param aStrehl      Strehl ratio to convert.
//  @param aStrehlError Error of the Strehl ratio.
//  @return Nothing.

//  Converts the given input Strehl ratio into an effective spatial resolution
//  valid for a MUSE NFM observation. The conversion uses an empirical model
//  which is only valid for the MUSE NFM mode!

//  The computed spatial resolution and its uncertainty are stored at the
//  location referenced by @em aSkyres and @em aSkyresError.
// */
///*----------------------------------------------------------------------------*/
//static void
//eris_ifu_idp_compute_skyres_from_strehl(double *aSkyres, double *aSkyresError,
//		double aStrehl, double aStrehlError)
//{
//	/* Parameter values of the empirically determined model used to convert *
//	 * a Strehl ratio into an effective spatial resolution for the NFM      *
//	 * instrument mode.                                                     */
//	const double rms = 0.01;
//	const double parameter[3] = {-0.71985411, 0.81622807, -0.01941496};

//	const double strehl_min = 1.;
//	const double strehl_max = 80.;

//	/* The uncertainty of the Strehl ratio is actually not used in the *
//	 * following model implementation. Keep it to be prepared.         */
//	CPL_UNUSED(aStrehlError);

//	/* Keep the input strehl ratio to the model within the limits */
//	double strehl = CPL_MIN(CPL_MAX(aStrehl, strehl_min), strehl_max);

//	double skyres = parameter[0] + parameter[1] * pow(strehl, parameter[2]);
//	double skyrerr = parameter[1] * parameter[2] *
//			pow(strehl, (parameter[2] - 1.));
//	skyrerr = sqrt(skyrerr * skyrerr + rms * rms);

//	*aSkyres = skyres;
//	*aSkyresError = skyrerr;
//	return;
//}

/*----------------------------------------------------------------------------*/
/**
 * @brief Compute wavelength-corrected FWHM from ASM data
 * 
 * Calculates the seeing FWHM corrected to the central wavelength of the observation
 * band. The correction accounts for wavelength-dependent seeing using a power law
 * with exponent 0.2. Also computes DIMM seeing with airmass and wavelength corrections.
 * 
 * The function retrieves IA_FWHM (image analyzer FWHM) from the ASM (Acquisition and
 * Setup Module) extension in the first raw frame, or falls back to header keywords.
 * 
 * @param set  Frameset containing raw observation frames with ASM data
 * 
 * @return Wavelength-corrected FWHM in arcseconds, or 0 on error
 * 
 * @note Correction formula: FWHM_corr = FWHM_measured * (0.5/lambda_central)^0.2
 * @note IA_FWHM is read from ASM table column or "ESO TEL IA FWHM" header keyword
 * @note DIMM seeing is read from ASM table or computed from FWHM START/END keywords
 * @note Also computes DIMM correction with Fried parameter and outer scale effects
 * @note Pixel scale is determined from instrument pre-optics configuration
 */
/*----------------------------------------------------------------------------*/

static double
eris_ifu_sdp_compute_fwhm(cpl_frameset* set)
{
	cpl_ensure(set != NULL, CPL_ERROR_NULL_INPUT, 0);
	//	double abmaglimit = 0;
	cpl_frame* frm = NULL;
	//	cpl_size nexposures  = cpl_frameset_count_tags(set, ERIS_IFU_RAW_OBJ);
	cpl_size kexposure = 0;
	const char* fname;
	cpl_propertylist* phead;
	cpl_size next = 0;

	frm = cpl_frameset_get_position(set, kexposure);
	next = cpl_frame_get_nextensions(frm);
	fname = cpl_frame_get_filename(frm);
	phead = cpl_propertylist_load(fname, 0);
	cpl_msg_info(cpl_func,"fname: %s next: %lld",fname, next);
	cpl_table* asm_data;
	double IA_fwhm_corr = 0;
	double IA_fwhm_corr_pix = 0;
	if (next > 0) {
		double IA_fwhm = 0;
		double dimm = 0;
		ifsBand bandId = eris_ifu_get_band(phead);
		asm_data = cpl_table_load(fname, 1, 0);
		if(cpl_table_has_column(asm_data,"IA_FWHM")) {
			IA_fwhm = cpl_table_get_column_mean(asm_data,"IA_FWHM");
		} else if (cpl_propertylist_has(phead,"ESO TEL IA FWHM")) {
			IA_fwhm = cpl_propertylist_get_double(phead,"ESO TEL IA FWHM");
		}
		if(cpl_table_has_column(asm_data,"DIMM_SEEING")) {
			  dimm = cpl_table_get_column_mean(asm_data,"DIMM_SEEING");
		} else {
			double airm_start = 0;
			double airm_end = 0;
			
			if (cpl_propertylist_has(phead,"ESO TEL AMBI FWHM START")) {
			  airm_start = eris_pfits_get_fwhm_start(phead);
			}
			
			if (cpl_propertylist_has(phead,"ESO TEL AMBI FWHM END")) {
				airm_end = eris_pfits_get_fwhm_end(phead);
			}
			
			if(airm_start > 0 && airm_end > 0) {
				dimm = 0.5 * (airm_start + airm_end);
			} else if (airm_start > 0 && airm_end == 0) {
				dimm = airm_start;
			} else if (airm_start > 0 && airm_end == 0) {
				dimm = airm_end;
			}
			
		}
		cpl_table_delete(asm_data);
		/* compute the wavelength corrected FWHM values from the ASM extension */
		double lambda_c = 1;
		eris_ifu_get_central_lambda(bandId, &lambda_c);
		ifsPreopticsScale  scale = eris_ifu_get_preopticsScale(phead);
		double pix_scale = 1;
		switch(scale){
		case S25MAS: pix_scale = 0.025; break;
		case S100MAS:pix_scale = 0.100; break;
		case S250MAS:pix_scale = 0.250; break;
		default:
			cpl_msg_error(cpl_func,"scale not found");
			break;
		}


		IA_fwhm_corr = IA_fwhm * pow((0.500/lambda_c), 0.2 );
		IA_fwhm_corr_pix = IA_fwhm_corr / pix_scale;
		cpl_msg_info(cpl_func,"IA_fwhm_corr: %g",IA_fwhm_corr);
		cpl_msg_info(cpl_func,"IA_fwhm_corr_pix: %g",IA_fwhm_corr_pix);
		double airmass = eris_pfits_get_airmass(phead);

		double dimm_corr = dimm *
				pow((0.500/lambda_c),0.2) *
				pow(airmass,(3./5.)) *
				(1. - 78.08 *
						(pow( (lambda_c / 1.e6), 0.4) *
								(pow(airmass, -0.2) / pow(dimm, (1./3.)))
						));
		cpl_msg_info(cpl_func,"dimm_corr: %g",dimm_corr);
		double dimm_corr_pix = dimm_corr/pix_scale;
		cpl_msg_info(cpl_func,"dimm_corr_pix: %g",dimm_corr_pix);
	}

	cpl_propertylist_delete(phead);
	eris_check_error_code("eris_ifu_sdp_compute_fwhm");
	return IA_fwhm_corr;


	//abmaglimit = -2.5 * log10(2. * sdev / kMuseFluxUnitFactor) + zeropoint;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Compute pixel-to-pixel noise from flux-calibrated cube
 * 
 * Calculates the standard deviation of pixel values in a flux-calibrated cube
 * after masking out objects. This represents the sky background noise level
 * and is used for the PIXNOISE SDP keyword.
 * 
 * The function searches for various flux-calibrated cube products in the frameset,
 * loads the first plane, creates an object mask using morphological filtering,
 * and computes the standard deviation of the masked (sky-only) pixels.
 * 
 * @param set  Frameset containing flux-calibrated cube products
 * 
 * @return Pixel-to-pixel noise in flux units, or 0 on error
 * 
 * @note Searches for products in priority order: OBJ_CUBE_COADD_FLUXCAL_MEAN,
 *       OBJ_CUBE_MEAN_FLUXCAL, DAR_CUBE_FLUXCAL_MEAN, STD_FLUX_CUBE variants, TWK_CUBE
 * @note Uses quality image (extension 3) to mask bad pixels
 * @note Object masking uses 3x3 kernel for detection
 * @note Returns standard deviation of sky pixels after object masking
 */
/*----------------------------------------------------------------------------*/

static double
eris_ifu_sdp_compute_pixnoise(cpl_frameset* set) {

	cpl_ensure(set != NULL, CPL_ERROR_NULL_INPUT, 0);
	cpl_frame* frm = NULL;
	//cpl_frameset_dump(set,stdout);
	cpl_error_code error_code = cpl_error_get_code();
	cpl_error_set(cpl_func, error_code);

	if(NULL != cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_OBJ_CUBE_COADD_FLUXCAL_MEAN)) {
		frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_OBJ_CUBE_COADD_FLUXCAL_MEAN);
	} else if(NULL != cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_OBJ_CUBE_MEAN_FLUXCAL)) {
		frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_OBJ_CUBE_MEAN_FLUXCAL);
	} else if(NULL != cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_OBJ_DAR_CUBE_FLUXCAL_MEAN)) {
		frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_OBJ_DAR_CUBE_FLUXCAL_MEAN);
	} else if(NULL != cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_FLUXCAL_MEAN)) {
		frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_FLUXCAL_MEAN);
	} else if(NULL != cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_FLUX_STD_CUBE_COADD_FLUXCAL_MEAN)) {
		frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_FLUX_STD_CUBE_COADD_FLUXCAL_MEAN);
	} else if(NULL != cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_TWK_OBJ_CUBE_COADD_FLUXCAL_MEAN)) {
		frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_TWK_OBJ_CUBE_COADD_FLUXCAL_MEAN);
	}

	const char* fname = cpl_frame_get_filename(frm);
    cpl_msg_warning(cpl_func,"fname: %s",fname);
	cpl_image* data = cpl_image_load(fname, CPL_TYPE_DOUBLE, 0, 1);
	cpl_image* errs = cpl_image_load(fname, CPL_TYPE_DOUBLE, 0, 2);
	cpl_image* qual = cpl_image_load(fname, CPL_TYPE_INT, 0, 3);

	cpl_mask* bpm = cpl_mask_threshold_image_create(qual, DBL_MIN, 0.9);
	cpl_image_reject_from_mask(data, bpm);
	cpl_mask_delete(bpm);
	//cpl_image_set_bpm(errs, bpm);

	hdrl_image* hima = hdrl_image_create(data, errs);

	cpl_image_delete(data);
	cpl_image_delete(errs);
	cpl_image_delete(qual);

	eris_ifu_mask_nans_in_hdrlimage(&hima);
	/* TODO: which of the two option should we use to mask the object? */
	//cpl_mask* obj_mask = eris_ifu_hima_get_obj_mask_percent(hima, 0.5);
	cpl_mask* obj_mask = eris_ifu_hima_get_obj_mask(hima, 3, 3);

	hdrl_image_reject_from_mask(hima, obj_mask);

	double pixnoise = hdrl_image_get_stdev(hima);

	hdrl_image_delete(hima);


	cpl_mask_delete(obj_mask);
	eris_check_error_code("eris_ifu_sdp_compute_pixnoise");
	return pixnoise;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Load flux-calibrated mean object cube as HDRL image
 * 
 * Searches for and loads the first plane of various flux-calibrated mean cube
 * products from the frameset. Creates an HDRL image with data, error, and
 * bad pixel mask, then applies NaN masking and object detection.
 * 
 * This is used for AB magnitude limit calculations where the mean collapsed
 * image provides better signal-to-noise than individual planes.
 * 
 * @param set  Frameset containing flux-calibrated cube products
 * 
 * @return Pointer to hdrl_image with data/error/mask, or NULL on error
 * 
 * @note The returned hdrl_image must be freed with hdrl_image_delete()
 * @note Searches for products in priority order: DAR_CUBE, OBJ_CUBE_COADD,
 *       STD_CUBE variants, TWK_CUBE
 * @note Loads first plane (extension 1=data, 2=errors, 3=quality) only
 * @note Object masking uses 1x3 kernel for detection but does NOT reject objects
 * @note NaN values are masked before object detection
 */
/*----------------------------------------------------------------------------*/

static hdrl_image*
eris_ifu_sdp_get_obj_mean(cpl_frameset* set) {

	cpl_ensure(set != NULL, CPL_ERROR_NULL_INPUT, NULL);
	cpl_frame* frm = NULL;
	//cpl_frameset_dump(set,stdout);
	cpl_error_code error_code = cpl_error_get_code();
	cpl_error_set(cpl_func, error_code);
	cpl_msg_warning(cpl_func,"search input frame");

	if(NULL != cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_OBJ_DAR_CUBE_FLUXCAL_MEAN)) {
		frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_OBJ_DAR_CUBE_FLUXCAL_MEAN);
	} else if(NULL != cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_OBJ_CUBE_COADD_FLUXCAL_MEAN)) {
		frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_OBJ_CUBE_COADD_FLUXCAL_MEAN);
	} else if(NULL != cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_STD_CUBE_COADD_FLUXCAL_MEAN)) {
		frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_STD_CUBE_COADD_FLUXCAL_MEAN);
	} else if(NULL != cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_FLUX_STD_CUBE_COADD_FLUXCAL_MEAN)) {
		frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_FLUX_STD_CUBE_COADD_FLUXCAL_MEAN);
	} else if(NULL != cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_FLUXCAL_MEAN)) {
		frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_FLUXCAL_MEAN);
	} else if(NULL != cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_TWK_OBJ_CUBE_COADD_FLUXCAL_MEAN)) {
		frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_TWK_OBJ_CUBE_COADD_FLUXCAL_MEAN);
	}

	const char* fname = cpl_frame_get_filename(frm);

	cpl_image* data = cpl_image_load(fname, CPL_TYPE_DOUBLE, 0, 1);
	cpl_image* errs = cpl_image_load(fname, CPL_TYPE_DOUBLE, 0, 2);
	cpl_image* qual = cpl_image_load(fname, CPL_TYPE_INT, 0, 3);
	cpl_mask* bpm = cpl_mask_threshold_image_create(qual, DBL_MIN, 0.9);

	cpl_image_set_bpm(data, bpm);
	//cpl_image_set_bpm(errs, bpm);

	hdrl_image* hima = hdrl_image_create(data, errs);

	eris_ifu_mask_nans_in_hdrlimage(&hima);

	/* TODO: which of the two option should we use to mask the object? */
	//cpl_mask* obj_mask = eris_ifu_hima_get_obj_mask_percent(hima, 0.5);
	cpl_mask* obj_mask = eris_ifu_hima_get_obj_mask(hima, 1, 3);
	//hdrl_image_reject_from_mask(hima, obj_mask);

	cpl_image_delete(data);
	cpl_image_delete(errs);
	cpl_image_delete(qual);
	cpl_mask_delete(obj_mask);

	eris_check_error_code("eris_ifu_sdp_get_obj_mean");
	return hima;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Compute photometric zero point for a given ERIS band
 * 
 * Returns the AB magnitude zero point for the specified observing band.
 * These values are derived from Mark Casali's formula:
 * ZP = AB_mag + 2.5 * log10(flux_max) - aperture_correction
 * 
 * The zero points are calibrated for each ERIS IFU band configuration
 * (J, H, K in LOW, SHORT, MIDDLE, LONG variants).
 * 
 * @param band  ERIS IFU band identifier
 * 
 * @return AB magnitude zero point, or 2.0 if band is undefined
 * 
 * @note Zero points are negative values (typically -23 to -25)
 * @note J_LOW uses mean of other J bands (value not individually calibrated)
 * @note Undefined bands return default value of 2.0
 */
/*----------------------------------------------------------------------------*/

static double
eris_ifu_sdp_compute_zp(ifsBand band)
{
	double ZP = 0;
	switch(band) {
	/* OLD
	case J_LOW:    ZP = -23.05987; break; // no result??
	case J_SHORT:  ZP = -22.720572007420277; break;
	case J_MIDDLE: ZP = -23.059874978944993; break;
	case J_LONG:   ZP = -23.40258436053865; break;
	case H_LOW:    ZP = -23.36106526995602; break;
	case H_SHORT:  ZP = -23.539146965; break;
	case H_MIDDLE: ZP = -23.911951487414118; break;
	case H_LONG:   ZP = -23.9152376244; break;
	case K_LOW:    ZP = -24.516976313167717; break;
	case K_SHORT:  ZP = -24.2690232517; break;
	case K_MIDDLE: ZP = -24.413665245625793; break;
	case K_LONG:   ZP = -24.644367831; break;
	default: ZP = 2; break;
	*/
	/* Using Mark's formula:zp = ab_mag + 2.5 * math.log10(f3_max) - apcor3 */
	case J_LOW:    ZP = -23.564080671; break; //not available take mean of other band values
	case J_SHORT:  ZP = -23.370149709; break;  //
	case J_MIDDLE: ZP = -23.575634864; break; //
	case J_LONG:   ZP = -23.746457445; break; //
	case H_LOW:    ZP = -23.764826331; break; //
	case H_SHORT:  ZP = -23.961968187; break; //
	case H_MIDDLE: ZP = -24.238252577; break; //
	case H_LONG:   ZP = -24.396966217; break; //
	case K_LOW:    ZP = -24.89135131; break;//
	case K_SHORT:  ZP = -24.730943402; break; //
	case K_MIDDLE: ZP = -24.8542167693; break;//
	case K_LONG:   ZP = -25.24818993; break;//
	default: ZP = 2; break;


	}
	return ZP;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Compute 5-sigma AB magnitude limit for point sources
 * 
 * Calculates the detection limit for point sources as an AB magnitude,
 * representing the faintest sources detectable at 5-sigma significance.
 * Uses the sky RMS, PSF FWHM, and band-specific zero point.
 * 
 * The formula accounts for:
 * - Sky noise (RMS per pixel)
 * - PSF size (FWHM determines source aperture)
 * - Aperture area (sqrt(pi) factor)
 * - 5-sigma detection threshold
 * - Factor of 2 in flux for 5-sigma detection
 * 
 * @param sky_rms  Sky background RMS in flux units
 * @param fwhm     Point source FWHM in pixels
 * @param band     ERIS IFU band for zero point lookup
 * 
 * @return 5-sigma AB magnitude limit
 * 
 * @note Formula: -2.5*log10(5 * FWHM * sky_rms * sqrt(pi) / (2*sqrt(log(4)))) - 2.5*log10(2) + ZP
 * @note Higher (more positive) values indicate deeper observations
 * @note Typical values range from ~22 to ~26 AB mag depending on conditions
 */
/*----------------------------------------------------------------------------*/

static double
eris_ifu_sdp_compute_abmaglimit(const double  sky_rms, const double fwhm,
		ifsBand band) {

	double abmaglim = 0;
	//ABmaglim = -2.5*log10(5. * FWHM * sky_rms * sqrt(pi)/(2. * sqrt(log(4)))) - 2.5 * log10(2.) + ZP
	double ZP = eris_ifu_sdp_compute_zp(band);
	abmaglim = -2.5*log10(5. * fwhm * sky_rms * sqrt(CPL_MATH_PI) /
			(2. * sqrt(log(4)))) - 2.5 * log10(2.) + ZP;

	eris_check_error_code("eris_ifu_sdp_compute_abmaglimit");
	return abmaglim;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Allocate and initialize new SDP properties structure
 * 
 * Creates a new eris_ifu_sdp_properties structure with all fields initialized
 * to zero/NULL. This structure holds all metadata required for ESO Science
 * Data Product headers.
 * 
 * @return Pointer to allocated structure, or NULL on allocation failure
 * 
 * @note The returned structure must be freed with eris_ifu_sdp_properties_delete()
 * @note All pointer fields are initialized to NULL
 * @note All numeric fields are initialized to 0
 */
/*----------------------------------------------------------------------------*/

eris_ifu_sdp_properties *
eris_ifu_sdp_properties_new(void) {
	eris_ifu_sdp_properties *properties = cpl_calloc(1, sizeof *properties);
	return properties;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Free SDP properties structure and all contained data
 * 
 * Deallocates an eris_ifu_sdp_properties structure including all dynamically
 * allocated members (arrays, property lists, strings).
 * 
 * @param aProperties  Pointer to SDP properties structure to free
 * 
 * @note Safe to call with NULL pointer (no-op)
 * @note Frees: obid array, progid array, prov propertylist, asson array,
 *       assoc array (deprecated), prodcatg, specsys, obstech, referenc strings
 * @note procsoft string is NOT freed (points to header data)
 */
/*----------------------------------------------------------------------------*/

void
eris_ifu_sdp_properties_delete(eris_ifu_sdp_properties *aProperties)
{
	if (aProperties) {
		cpl_array_delete(aProperties->obid);
		cpl_array_delete(aProperties->progid);
		cpl_propertylist_delete(aProperties->prov);
		cpl_array_delete(aProperties->asson);

		/* TODO: Remove deallocator call for assoc once it has been *
		 * removed from the interface.                              */
		cpl_array_delete(aProperties->assoc);
		cpl_free((char *)aProperties->prodcatg);
		//cpl_free((char *)aProperties->procsoft);
		cpl_free((char *)aProperties->specsys);
		cpl_free((char *)aProperties->obstech);
		cpl_free((char *)aProperties->referenc);
	}
	cpl_free(aProperties);
	return;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Get ORIGFILE keyword value from FITS header
 * 
 * Retrieves the original filename from the FITS header ORIGFILE keyword.
 * This tracks the name of the file as originally created by the instrument.
 * 
 * @param aHeaders  Property list (FITS header) to query
 * 
 * @return Pointer to ORIGFILE string value, or NULL on error
 * 
 * @note The returned pointer is to internal data - do not free
 * @note Sets CPL error if keyword is missing or wrong type
 */
/*----------------------------------------------------------------------------*/

const char *
eris_ifu_pfits_get_origfile(const cpl_propertylist *aHeaders)
{
	const char *value = cpl_propertylist_get_string(aHeaders, "ORIGFILE");
	cpl_ensure(value, cpl_error_get_code(), NULL);
	return value;
}

///*----------------------------------------------------------------------------*/
///**
//  @private
//  @brief Estimate image quality at a given wavelength from the seeing
//         and the airmass.
//  @param aLambda   Wavelength in Angstrom at which the image quality is
//                   computed.
//  @param aSeeing   The seeing as seen by the telescope and corrected for
//                   airmass.
//  @param aAirmass  Airmass at which the observation was carried out.
//  @return The estimate of the image quality at the given wavelength.

//  The returned image quality estimate is in units of arcsec.
// */
///*----------------------------------------------------------------------------*/
//static double
//eris_ifu_sdp_compute_iqe(double aLambda, double aSeeing, double aAirmass)
//{

//	const double rad2as = 180. / CPL_MATH_PI * 3600.;  /* radians to arcsec */


//	/* Wavefront outer scale at Paranal [meters] */
//	const double L0 = 23.;

//	/* UT M1 diameter [meters] from UT M1 vignetted area */
//	const double D  = 8.1175365721399437;

//	/* Kolb factor (cf. ESO Technical Report 12) */
//	const double Fkolb = 1. / (1. + 300. * D / L0) - 1.;


//	/* Scaled wavelength: lambda over lambda_ref */
//	double lambda = aLambda / 5000.;

//	/* Fried parameter */
//	double r0 = 0.976 * 5.e-7 / aSeeing * rad2as *
//			pow(lambda, 1.2) * pow(aAirmass, -0.6);

//	double iqe = aSeeing * pow(lambda, -0.2) * pow(aAirmass, 0.6);
//	iqe *= sqrt(1. + Fkolb * 2.183 * pow(r0 / L0, 0.356));

//	return iqe;
//}

static cpl_error_code
eris_ifu_get_sdp_frames(cpl_frameset* set, const cpl_parameterlist* parlist,
		const char* recipe_name, eris_ifu_sdp_properties *properties,
		cpl_frameset** raws_obj, cpl_frameset** raws_sky, cpl_frameset** pros_obj)
{

	cpl_ensure(set != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(parlist != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(recipe_name != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(properties != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

	char* param_name = cpl_sprintf("eris.%s.dar-corr", recipe_name);
	cpl_boolean is_dar_cor =
			cpl_parameter_get_bool(cpl_parameterlist_find_const(parlist, param_name));
	cpl_free(param_name);


	param_name = cpl_sprintf("eris.%s.sky_tweak", recipe_name);
	int is_sky_tweak =
			cpl_parameter_get_int(cpl_parameterlist_find_const(parlist, param_name));
	cpl_free(param_name);

    cpl_size nexposures = 0;
    cpl_frame* frm = NULL;
    //cpl_frameset_dump(set, stdout);
    //exit(0);
	if (strstr(recipe_name, "jitter") != NULL) {

		*raws_obj = eris_dfs_extract_frames_with_tag (set, ERIS_IFU_RAW_OBJ);
		if(is_dar_cor) {
			*pros_obj = eris_dfs_extract_frames_with_tag (set, ERIS_IFU_PRO_JITTER_DAR_CUBE);
		} else {
			*pros_obj = eris_dfs_extract_frames_with_tag (set, ERIS_IFU_PRO_JITTER_OBJ_CUBE);
		}

		if(is_sky_tweak > 0) {
			*pros_obj = eris_dfs_extract_frames_with_tag (set, ERIS_IFU_PRO_JITTER_TWK_CUBE);
		} else {
			*pros_obj = eris_dfs_extract_frames_with_tag (set, ERIS_IFU_PRO_JITTER_OBJ_CUBE);
		}
		if(*pros_obj != NULL) {
			// same PRO.CATG independently from DAR correction
			if(is_dar_cor) {
				if(NULL == (frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_OBJ_DAR_CUBE_FLUXCAL_MEAN))) {
					frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_OBJ_CUBE_COADD_FLUXCAL_MEAN);
				}
			} else {
				frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_OBJ_CUBE_COADD_FLUXCAL_MEAN);
			}
			if(frm != NULL) {
				cpl_frameset_insert(*pros_obj, cpl_frame_duplicate(frm));
			}


			if(is_sky_tweak > 0) {
				if(NULL == (frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_OBJ_DAR_CUBE_FLUXCAL_MEAN))) {
					frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_OBJ_CUBE_COADD_FLUXCAL_MEAN);
				}
			} else {
				frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_OBJ_CUBE_COADD_FLUXCAL_MEAN);
			}
			if(frm != NULL) {
				cpl_frameset_insert(*pros_obj, cpl_frame_duplicate(frm));
			}


		}


		if(*pros_obj != NULL) {
			// same PRO.CATG independently from DAR correction
			if(is_sky_tweak > 0) {
				if(NULL == (frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_TWK_OBJ_CUBE_COADD_FLUXCAL_MEAN))) {
					frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_TWK_OBJ_CUBE_COADD_FLUXCAL_MEAN);
				}

				if(frm != NULL) {
					cpl_frameset_insert(*pros_obj, cpl_frame_duplicate(frm));
				}
			}
		}

		if(*pros_obj == NULL) {
			*pros_obj = eris_dfs_extract_frames_with_tag (set, ERIS_IFU_PRO_JITTER_TWK_CUBE);
		}

		nexposures = cpl_frameset_count_tags(*raws_obj, ERIS_IFU_RAW_OBJ);
	} else if (strstr(recipe_name, "stdstar") != NULL) {
		*raws_obj = eris_dfs_extract_frames_with_tag (set, ERIS_IFU_RAW_STD_FLUX);
		/* for STD stars PRO.CATG is independent from dar correction */
		if(is_dar_cor) {
			*pros_obj = eris_dfs_extract_frames_with_tag (set, ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE);
		} else {
			*pros_obj = eris_dfs_extract_frames_with_tag (set, ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE);
		}

		if(*pros_obj != NULL) {
			if(is_dar_cor) {
				frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_FLUXCAL_MEAN);
			} else {
				frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_FLUXCAL_MEAN);
			}
			if(frm != NULL) {
				cpl_frameset_insert(*pros_obj, cpl_frame_duplicate(frm));
			}
		}

		if(pros_obj != NULL) {
			if(is_dar_cor) {
				frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_FLUX_STD_CUBE_COADD_FLUXCAL_MEAN);
			} else {
				frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_FLUX_STD_CUBE_COADD_FLUXCAL_MEAN);
			}
			if(frm != NULL) {
				cpl_frameset_insert(*pros_obj, cpl_frame_duplicate(frm));
			}
		}

		if(*pros_obj != NULL) {
			if(is_dar_cor) {
				frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_STD_CUBE_COADD_FLUXCAL_MEAN);
			} else {
				frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_STD_CUBE_COADD_FLUXCAL_MEAN);
			}
			if(frm != NULL) {
				cpl_frameset_insert(*pros_obj, cpl_frame_duplicate(frm));
			}
		}
		nexposures = cpl_frameset_count_tags(*raws_obj, ERIS_IFU_RAW_STD_FLUX);
	}

	*raws_sky =
			eris_dfs_extract_frames_with_tag(set, ERIS_IFU_PRO_JITTER_SKY_CUBE);

	properties->ncombine = nexposures;


    //cpl_frameset_dump(set,stdout);
	frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_EXPOSURE_MAP);
	if(frm != NULL) {
		cpl_frameset_insert(*pros_obj, cpl_frame_duplicate(frm));
	}


	eris_check_error_code("eris_ifu_get_sdp_frames");
	return cpl_error_get_code();

}

/*----------------------------------------------------------------------------*/
/**
 * @brief Collect all SDP metadata from cube, frameset, and parameters
 * 
 * Main function for gathering ESO Science Data Product metadata from various
 * sources. Processes raw and calibrated data to compute quality metrics and
 * collect observation information required for Phase 3 compliant headers.
 * 
 * The function performs:
 * - Frame organization (raw/processed object/sky/std frames)
 * - Observation metadata collection (OBIDs, program IDs, MJDs, exposure times)
 * - FWHM computation from ASM data with wavelength correction
 * - Exposure time calculation from exposure map
 * - Provenance tracking (ARCFILE/ANCESTOR/raw filenames)
 * - Pixel noise computation from flux-calibrated cubes
 * - Spectral resolution from wavelength calibration
 * - AB magnitude limit calculation using HDRL
 * - Wavelength range determination from cube WCS
 * - Associated file list creation (ASSON keywords)
 * 
 * @param aCube       HDRL resampling result with cube and header
 * @param set         Complete frameset with raw and processed data
 * @param parlist     Recipe parameters
 * @param recipe_name Recipe name ("jitter" or "stdstar")
 * 
 * @return Pointer to populated eris_ifu_sdp_properties structure, or NULL on error
 * 
 * @note The returned structure must be freed with eris_ifu_sdp_properties_delete()
 * @note FWHM calculation differs for jitter (from ASM) vs stdstar (hardcoded 7.18")
 * @note Spectral resolution is obtained from wavelength calibration map
 * @note AB magnitude limit is computed using both HDRL method and simple formula
 * @note For single object cubes, first product in associated file list is skipped
 * @note Flux calibration status is set to ABSOLUTE (CPL_TRUE)
 * @note SPECSYS is set to "TOPOCENT" (topocentric reference frame)
 * @note Wavelength range is converted from microns to nm for FITS keywords
 * @note Fallback wavelength error of 0.026 Angstrom is used
 */
/*----------------------------------------------------------------------------*/

eris_ifu_sdp_properties *
eris_ifu_sdp_properties_collect(hdrl_resample_result *aCube, cpl_frameset* set,
		const cpl_parameterlist* parlist, const char* recipe_name)
{

	cpl_ensure(aCube != NULL, CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(set != NULL, CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(parlist != NULL, CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(recipe_name != NULL, CPL_ERROR_NULL_INPUT, NULL);

//	cpl_errorstate _status = cpl_errorstate_get();
	cpl_frameset* raws_obj = NULL;
	cpl_frameset* raws_sky = NULL;
	cpl_frameset* pros_obj = NULL;
	cpl_propertylist* header = aCube->header;
	cpl_frame*  frm;
	int naxis3 = hdrl_imagelist_get_size(aCube->himlist);
	eris_ifu_sdp_properties *properties = eris_ifu_sdp_properties_new();

	properties->wlerror = 0.026;     /* Fallback random error estimate of wavelength */

	eris_ifu_get_sdp_frames(set, parlist, recipe_name, properties, &raws_obj,
			&raws_sky, &pros_obj);
//    cpl_size nsky = cpl_frameset_get_size(raws_sky);
	if(raws_sky != NULL) {
	cpl_frameset_join(pros_obj, raws_sky);
	}
cpl_frameset_delete(raws_sky);
	cpl_size nexposures  = properties->ncombine;
	/* Collect data from the raw data files */
		//  cpl_errorstate status = cpl_errorstate_get();
	properties->obid    = cpl_array_new(nexposures, CPL_TYPE_LONG);
	properties->progid  = cpl_array_new(nexposures, CPL_TYPE_STRING);

	cpl_msg_info(cpl_func,"obid array size: %lld",
			cpl_array_get_size(properties->obid));

	cpl_msg_info(cpl_func,"progid array size: %lld",
			cpl_array_get_size(properties->progid));

	properties->prov    = cpl_propertylist_new();

	const char* fname = NULL;
	cpl_propertylist* phead  = NULL;
	cpl_propertylist* pchead  = NULL;
	double exptime = 0;
	double dit = 0;
	int ndit = 0;
	//  int npixel = 64 * 64;
	properties->texptime = 0;
	double ia_fwhm_corr = 0;
	for(cpl_size kexposure = 0; kexposure < properties->ncombine; kexposure++) {

		frm = cpl_frameset_get_position(raws_obj, kexposure);
		fname = cpl_frame_get_filename(frm);
		phead = cpl_propertylist_load(fname, 0);

		const char *progid = eris_pfits_get_progid(phead);

		long obid = eris_pfits_get_obsid(phead);
		cpl_msg_info(cpl_func,"obid: %ld",obid);
		dit = eris_pfits_get_dit(phead);
		ndit = eris_pfits_get_ndit(phead);
		exptime = dit  * ndit;

		double mjd     = eris_pfits_get_mjdobs(phead);

		double endtime = mjd + exptime / day2sec;

		properties->texptime += exptime;
		properties->mjd_end = CPL_MAX(endtime, properties->mjd_end);

		/* Get image quality SKY_RES (at the reference wavelength)
		 * and the associated error for this exposure                      */


		cpl_msg_info(cpl_func,"compute FWHM");
		if (strstr(recipe_name, "jitter") != NULL) {
		    ia_fwhm_corr = eris_ifu_sdp_compute_fwhm(raws_obj);
		} else {
			/* in STD stars miss a FITS extension required to determine FWHM */
			ia_fwhm_corr=7.18;
		}

		properties->skyres += ia_fwhm_corr;
		cpl_msg_info(cpl_func, "compute exposure time");
		frm = cpl_frameset_find(set, ERIS_IFU_PRO_JITTER_EXPOSURE_MAP);
		cpl_image* expt = cpl_image_load(cpl_frame_get_filename(frm),
				CPL_TYPE_DOUBLE, 0, 0);
		cpl_mask* bpm = cpl_image_get_bpm(expt);
		cpl_mask_threshold_image(bpm, expt, 0, DBL_MAX, CPL_BINARY_0);
		cpl_image_set_bpm(expt, bpm);
		properties->exptime = cpl_image_get_median(expt);
		cpl_image_delete(expt);

		/* Observation ID, program ID, etc. */
		cpl_array_set_long(properties->obid, kexposure, obid);
		if (progid) {
			cpl_array_set_string(properties->progid, kexposure, progid);
		}

		/* provenance */
		unsigned int nraw = cpl_propertylist_get_size(properties->prov);
		const char *prov = eris_pfits_get_ancestor(phead);

		if (!prov) {
			prov = eris_pfits_get_arcfile(phead);
			if (!prov) {
				prov = eris_ifu_pfits_get_origfile(phead);
			}
		}

		if (prov) {
			char *name = cpl_sprintf("PROV%-u", ++nraw);
			cpl_propertylist_append_string(properties->prov, name, prov);
			cpl_free(name);
		} else {
			unsigned int iraw = 1;
			prov =eris_pfits_get_raw_filename(phead, iraw);
			while (prov) {
				char *name = cpl_sprintf("PROV%-u", ++nraw);
				cpl_propertylist_append_string(properties->prov, name, prov);
				prov =eris_pfits_get_raw_filename(phead, ++iraw);
				cpl_free(name);
			}
		}

		cpl_propertylist_delete(phead);
	}
	properties->skyres /= properties->ncombine;

	//ASSON WAS HERE

	/* ABMAGLIM */
	double abmaglim = 0.0;

	cpl_msg_info(cpl_func,"compute abmaglim");

	double pixnoise = 1;
	//cpl_frameset_dump(pros_obj,stdout);
	pixnoise = eris_ifu_sdp_compute_pixnoise(pros_obj);

	properties->pixnoise = pixnoise;
	cpl_size nobj = cpl_frameset_get_size(raws_obj);
	cpl_frameset_delete(raws_obj);
	cpl_msg_info(cpl_func,"obid array size: %lld",
			cpl_array_get_size(properties->obid));

	cpl_frame* wave_frm = cpl_frameset_find(set, ERIS_IFU_CALIB_WAVE_MAP);
	fname = cpl_frame_get_filename(wave_frm);
	phead = cpl_propertylist_load(fname, 0);
	//double resol_med = cpl_propertylist_get_double(phead, "ESO QC RESOL MED");
	ifsBand bandId = eris_ifu_get_band(phead);
	double resol_med = eris_ifu_get_band_resolution(bandId);
	cpl_propertylist_delete(phead);

	hdrl_image* hima_obj_mean = eris_ifu_sdp_get_obj_mean(pros_obj);

	cpl_image*  ima_obj_mean = hdrl_image_get_image(hima_obj_mean);

	double zp = eris_ifu_sdp_compute_zp(bandId);
	//double fwhm = 10;
	const cpl_size kernel_size_x = 3;
	const cpl_size kernel_size_y = 3;

	/*Mode parameters*/
	double histo_min = 0.;
	double histo_max = 0.;
	double bin_size = 0.;
	cpl_size error_niter = 0;
	hdrl_mode_type mode_method = HDRL_MODE_MEDIAN;
	/*
	  hdrl_mode_parameter_parse_parlist(parlist, "eris_ifu_jitter", &histo_min, &histo_max,
	                    &bin_size, &mode_method, &error_niter);
	 */

	hdrl_parameter * mode_parameter = hdrl_collapse_mode_parameter_create(histo_min,
			histo_max, bin_size, mode_method, error_niter);

	hdrl_maglim_compute(ima_obj_mean, zp, ia_fwhm_corr, kernel_size_x, kernel_size_y,
			HDRL_IMAGE_EXTEND_NEAREST, mode_parameter,&abmaglim);
	hdrl_parameter_delete(mode_parameter);
	hdrl_image_delete(hima_obj_mean);
	cpl_msg_info(cpl_func,"HDRL computed abmaglim: %g", abmaglim);
	abmaglim = eris_ifu_sdp_compute_abmaglimit(pixnoise, ia_fwhm_corr, bandId);
	properties->abmaglimit = abmaglim;
	cpl_msg_info(cpl_func,"LODO computed abmaglim: %g", abmaglim);

	/* The following IDP properties are simply hard coded here since *
	 * not supposed to change.                                      */
	properties->prodcatg = cpl_strdup(KEY_PRODCATG_VALUE_IFS_CUBE);
	properties->obstech  = cpl_strdup("IFU");

	char *key = cpl_sprintf("ESO PRO REC1 PIPE ID");
	if(cpl_propertylist_has(header, key)) {
		properties->procsoft = cpl_propertylist_get_string(header, key);
	}
	cpl_free(key);
	properties->referenc = NULL;
	if(cpl_propertylist_has(header, KEY_PROG_ID) ){
		cpl_propertylist_get_string(header, KEY_PROG_ID);
	}
	properties->fluxcal = CPL_TRUE;

	/* TODO: ASSON */
	cpl_size npros  = cpl_frameset_get_size(pros_obj);
	//cpl_frameset_dump(pros_obj, stdout);
	properties->asson = cpl_array_new(npros, CPL_TYPE_STRING);
	cpl_size kstart = 0;
	if (nobj == 1) {
		/* If nobj == 1 3D cube object (ADU) (1st of list) is not associated */
		properties->nobj = 1;
		kstart = 1;
	}
	for(cpl_size kexposure = kstart; kexposure < npros; kexposure++) {
		frm = cpl_frameset_get_position(pros_obj, kexposure);
		fname = cpl_frame_get_filename(frm);
		pchead = cpl_propertylist_load(fname, 0);
		cpl_array_set_string(properties->asson, kexposure,
				eris_pfits_get_pipefile(pchead));
		cpl_propertylist_delete(pchead);
	}





	properties->specres = resol_med;
	properties->specsys = cpl_sprintf("TOPOCENT");
	//properties->obid = eris_pfits_get_obsid(header);
	//  double wstart = 0;
	//  double wend = 100;
	if(cpl_propertylist_has(header,"CRVAL3")) {
		properties->wlenrange[0] = cpl_propertylist_get_double(header,"CRVAL3");
	}

	double cd3_3 = 0;
	if(cpl_propertylist_has(header,"CD3_3")) {
		cd3_3 = cpl_propertylist_get_double(header,"CD3_3");
	}
	properties->wlenrange[1] = properties->wlenrange[0] + cd3_3 * (double)(naxis3 -1.);

	/* TODO
  mandatory keyword PIXNOISE is undefined.
  mandatory keyword SKY_RES is undefined.
  mandatory keyword ABMAGLIM is undefined.
  mandatory keyword ext:HDUCLASS is undefined in HDU0.
  mandatory keyword ext:HDUCLASS is undefined in HDU1.
  mandatory keyword ext:HDUCLASS is undefined in HDU2.
  mandatory keyword ext:HDUDOC is undefined in HDU0.
  mandatory keyword ext:HDUDOC is undefined in HDU1.
  mandatory keyword ext:HDUDOC is undefined in HDU2.
  mandatory keyword ext:HDUVERS is undefined in HDU0.
  mandatory keyword ext:HDUVERS is undefined in HDU1.
  mandatory keyword ext:HDUVERS is undefined in HDU2.
  mandatory keyword ext:HDUCLAS1 is undefined in HDU0.
  mandatory keyword ext:HDUCLAS1 is undefined in HDU1.
  mandatory keyword ext:HDUCLAS1 is undefined in HDU2.

3. The keywords CDELT are not supported by the SDP standard. They are in every extension.

4. ORIGFILE must be recorded in the primary header, and not duplicated in any extension.
	 */
	if (pros_obj) cpl_frameset_delete(pros_obj);

	//if (raws_obj) cpl_frameset_delete(raws_obj);
	eris_check_error_code("eris_ifu_sdp_properties_collect");
	return properties;
}

/*----------------------------------------------------------------------------*/
/* Helper functions for quicksort */
/*----------------------------------------------------------------------------*/

/** @private Comparison function for ascending double sort */
static int cmp_double_asc(const void *p1, const void *p2) {
	double d = (*(const double *)p1 - *(const double *)p2);
	return (d  < 0)?-1:(d>0)?1:0;
}

/** @private Comparison function for descending double sort */
static int cmp_double_desc(const void *p1, const void *p2) {
	double d = (*(const double *)p1 - *(const double *)p2);
	return (d  < 0)?1:(d>0)?-1:0;
}

/** @private Comparison function for ascending float sort */
static int cmp_float_asc(const void *p1, const void *p2) {
	float d = (*(const float *)p1 - *(const float *)p2);
	return (d  < 0)?-1:(d>0)?1:0;
}

/** @private Comparison function for descending float sort */
static int cmp_float_desc(const void *p1, const void *p2) {
	float d = (*(const float *)p1 - *(const float *)p2);
	return (d  < 0)?1:(d>0)?-1:0;
}

/** @private Comparison function for ascending int sort */
static int cmp_int_asc(const void *p1, const void *p2) {
	return (*(const int *)p1 - *(const int *)p2);
}

/** @private Comparison function for descending int sort */
static int cmp_int_desc(const void *p1, const void *p2) {
	return (*(const int *)p2 - *(const int *)p1);
}

/** @private Comparison function for ascending long sort */
static int cmp_long_asc(const void *p1, const void *p2) {
	return (*(const long *)p1 - *(const long *)p2);
}

/** @private Comparison function for descending long sort */
static int cmp_long_desc(const void *p1, const void *p2) {
	return (*(const long *)p2 - *(const long *)p1);
}

/** @private Comparison function for ascending string sort */
static int cmp_string_asc(const void *p1, const void *p2) {
	return strcmp(*(const char **)p1, *(const char **)p2);
}

/** @private Comparison function for descending string sort */
static int cmp_string_desc(const void *p1, const void *p2) {
	return strcmp(*(const char **)p2, *(const char **)p1);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Sort CPL array in place using quicksort
 * 
 * Sorts the elements of a CPL array in ascending or descending order using
 * the standard library quicksort function. Supports arrays of type double,
 * float, int, long, and string.
 * 
 * @param aArray  Array to sort (modified in place)
 * @param aOrder  Sort order: CPL_TRUE for ascending, CPL_FALSE for descending
 * 
 * @return CPL_ERROR_NONE on success, error code otherwise
 * 
 * @note The array must not contain invalid values
 * @note The array is sorted in place - original order is lost
 * @note String arrays are sorted lexicographically
 * @note Returns CPL_ERROR_ILLEGAL_INPUT for unsupported array types
 */
/*----------------------------------------------------------------------------*/

cpl_error_code
eris_ifu_cplarray_sort(cpl_array *aArray, cpl_boolean aOrder)
{
	cpl_ensure_code(aArray != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(!cpl_array_has_invalid(aArray), CPL_ERROR_NULL_INPUT);

	cpl_size n = cpl_array_get_size(aArray);
	if (cpl_array_get_type(aArray) == CPL_TYPE_DOUBLE) {
		double *d = cpl_array_get_data_double(aArray);
		qsort(d, n, sizeof(double), (aOrder)?cmp_double_asc:cmp_double_desc);
		return CPL_ERROR_NONE;
	} else if (cpl_array_get_type(aArray) == CPL_TYPE_FLOAT) {
		float *d = cpl_array_get_data_float(aArray);
		qsort(d, n, sizeof(float), (aOrder)?cmp_float_asc:cmp_float_desc);
		return CPL_ERROR_NONE;
	} else if (cpl_array_get_type(aArray) == CPL_TYPE_INT) {
		int *d = cpl_array_get_data_int(aArray);
		qsort(d, n, sizeof(int), (aOrder)?cmp_int_asc:cmp_int_desc);
		return CPL_ERROR_NONE;
	} else if (cpl_array_get_type(aArray) == CPL_TYPE_LONG) {
		long *d = cpl_array_get_data_long(aArray);
		qsort(d, n, sizeof(long), (aOrder)?cmp_long_asc:cmp_long_desc);
		return CPL_ERROR_NONE;
	} else if (cpl_array_get_type(aArray) == CPL_TYPE_STRING) {
		char **d = cpl_array_get_data_string(aArray);
		qsort(d, n, sizeof(char *), (aOrder)?cmp_string_asc:cmp_string_desc);
		return CPL_ERROR_NONE;
	} else {
		return CPL_ERROR_ILLEGAL_INPUT;
	}
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Update FITS header with ESO Science Data Product keywords
 * 
 * Writes all collected SDP metadata to the FITS header in ESO Phase 3 compliant
 * format. This includes observation information, data quality metrics, provenance,
 * wavelength range, flux calibration status, and associated files.
 * 
 * The function:
 * - Clears existing SDP keywords using regex pattern
 * - Validates input arrays match expected sizes
 * - Writes spatial center (RA, DEC)
 * - Writes temporal information (MJD-OBS, MJD-END, EXPTIME, TEXPTIME, NCOMBINE)
 * - Writes observation blocks (OBID1, OBID2, ...) removing duplicates
 * - Writes program IDs (PROG_ID, PROGID1, ...) with MULTI for multiple programs
 * - Writes provenance (PROV1, PROV2, ...)
 * - Writes associated files (ASSON1, ASSON2, ...)
 * - Writes product metadata (PRODCATG, PROCSOFT, OBSTECH, FLUXCAL)
 * - Writes wavelength info (WAVELMIN, WAVELMAX in nm, SPEC_RES, SPECSYS)
 * - Writes quality metrics (SKY_RES, PIXNOISE, ABMAGLIM)
 * - Writes reference (REFERENC)
 * 
 * @param aHeader     FITS header propertylist to update (modified in place)
 * @param aProperties SDP properties structure with all metadata
 * 
 * @return CPL_ERROR_NONE on success, error code otherwise
 * 
 * @note Header is validated: obid/progid arrays must match ncombine size
 * @note OBID values are sorted and duplicates removed before writing
 * @note Program IDs: single program writes PROG_ID directly, multiple writes "MULTI" + PROGIDn
 * @note Wavelength range is converted from microns to nanometers (* 1000)
 * @note SKY_RES comment indicates "measured" or "default" based on sign
 * @note Negative skyres values indicate defaults, absolute value is written
 * @note Associated files list skips first entry if nobj==1 (uncalibrated cube)
 */
/*----------------------------------------------------------------------------*/

cpl_error_code
eris_ifu_sdp_properties_update(cpl_propertylist *aHeader,
		const eris_ifu_sdp_properties *aProperties)
{
	cpl_ensure(aHeader && aProperties, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

	/* TODO */
	cpl_ensure(cpl_array_get_size(aProperties->obid) == aProperties->ncombine,
			CPL_ERROR_ILLEGAL_INPUT, CPL_ERROR_ILLEGAL_INPUT);

	cpl_ensure(cpl_array_get_size(aProperties->progid) == aProperties->ncombine,
			CPL_ERROR_ILLEGAL_INPUT, CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure(cpl_propertylist_get_size(aProperties->prov) >= aProperties->ncombine,
			CPL_ERROR_ILLEGAL_INPUT, CPL_ERROR_ILLEGAL_INPUT);


	cpl_propertylist_erase_regexp(aHeader, CLEAN_KEYS_REGEXP, 0);

	cpl_propertylist_update_double(aHeader, KEY_RA, aProperties->fovcenter[0]);
	cpl_propertylist_set_comment(aHeader, KEY_RA, KEY_RA_COMMENT);
	cpl_propertylist_update_double(aHeader, KEY_DEC, aProperties->fovcenter[1]);
	cpl_propertylist_set_comment(aHeader, KEY_DEC, KEY_DEC_COMMENT);
	cpl_propertylist_update_double(aHeader, KEY_EXPTIME, aProperties->exptime);
	cpl_propertylist_set_comment(aHeader, KEY_EXPTIME, KEY_EXPTIME_COMMENT);

	cpl_propertylist_insert_after_double(aHeader, KEY_EXPTIME,
			KEY_TEXPTIME, aProperties->texptime);
	cpl_propertylist_set_comment(aHeader, KEY_TEXPTIME, KEY_TEXPTIME_COMMENT);

	cpl_propertylist_insert_after_int(aHeader, KEY_TEXPTIME,
			KEY_NCOMBINE, aProperties->ncombine);
	cpl_propertylist_set_comment(aHeader, KEY_NCOMBINE, KEY_NCOMBINE_COMMENT);

	cpl_propertylist_set_comment(aHeader, KEY_MJDOBS, KEY_MJDOBS_COMMENT);
	cpl_propertylist_insert_after_double(aHeader, KEY_MJDOBS,
			KEY_MJDEND, aProperties->mjd_end);
	cpl_propertylist_set_comment(aHeader, KEY_MJDEND, KEY_MJDEND_COMMENT);

	/* TODO */
	cpl_array *obids = cpl_array_duplicate(aProperties->obid);

	eris_ifu_cplarray_sort(obids, CPL_TRUE);

	// Add observation block IDs, skipping duplicates
	long obid = cpl_array_get_long(obids, 0, NULL);
	cpl_msg_warning(cpl_func,"obid: %ld",obid);
	cpl_propertylist_update_long(aHeader, KEY_OBID1, obid);
	cpl_propertylist_set_comment(aHeader, KEY_OBID1, KEY_OBID_COMMENT);

	if (aProperties->ncombine > 1) {
		unsigned int ik = 1;
		cpl_size idx;
		for (idx = 1; idx < aProperties->ncombine; ++idx) {
			long _obid = cpl_array_get_long(obids, idx, NULL);
			if (_obid != obid) {
				char *key = cpl_sprintf(KEY_OBID "%-u", ++ik);
				cpl_propertylist_update_long(aHeader, key, _obid);
				cpl_propertylist_set_comment(aHeader, key, KEY_OBID_COMMENT);
				cpl_free(key);
				obid = _obid;
			}
		}
	}
	cpl_array_delete(obids);

	/* Add the ESO program ID(s). If the product is made from a single program *
	 * PROG_ID is the ID string, otherwise it set to the special value MULTI,  *
	 * followed by a list of all IDs, skipping duplicate ID values             */

	/* TODO */
	if(aProperties->progid != NULL) {

	cpl_array *progids = cpl_array_duplicate(aProperties->progid);

	/* TODO */
	eris_ifu_cplarray_sort(progids, CPL_TRUE);

	const char *progid = cpl_array_get_string(progids, 0);

	if (aProperties->ncombine > 1) {
		unsigned int nprogid = 1;
		cpl_size idx;
		for (idx = 1; idx < aProperties->ncombine; ++idx) {
			const char *_progid = cpl_array_get_string(progids, idx);
			if (strcmp(_progid, progid) != 0) {
				progid = _progid;
				++nprogid;
			}
		}

		progid = cpl_array_get_string(progids, 0);
		if (nprogid == 1) {
			cpl_propertylist_update_string(aHeader, KEY_PROG_ID, progid);
		} else {
			cpl_propertylist_update_string(aHeader, KEY_PROG_ID,
					KEY_PROG_ID_VALUE_MULTIPLE);
			cpl_propertylist_update_string(aHeader, KEY_PROGID "1", progid);
			cpl_propertylist_set_comment(aHeader, KEY_PROGID "1", KEY_PROGID_COMMENT);

			unsigned int ik = 1;
			for (idx = 1; idx < aProperties->ncombine; ++idx) {
				const char *_progid = cpl_array_get_string(progids, idx);
				if (strcmp(_progid, progid) != 0) {
					char *key = cpl_sprintf(KEY_PROGID "%-u", ++ik);
					cpl_propertylist_update_string(aHeader, key, _progid);
					cpl_propertylist_set_comment(aHeader, key, KEY_PROGID_COMMENT);
					cpl_free(key);
					progid = _progid;
				}
			}
		}
		cpl_propertylist_set_comment(aHeader, KEY_PROG_ID, KEY_PROG_ID_COMMENT);

	} else {
		cpl_propertylist_update_string(aHeader, KEY_PROG_ID, progid);
		cpl_propertylist_set_comment(aHeader, KEY_PROG_ID, KEY_PROG_ID_COMMENT);
	}
	cpl_array_delete(progids);
	}


	/* Add raw data information */
	cpl_propertylist_append(aHeader, aProperties->prov);

	/* Add ancillary products file information */
	/* TODO */
	cpl_size idx_start = 0;
	if(aProperties->nobj == 1) {
		idx_start = 1;
	}

	for (cpl_size idx = idx_start; idx < cpl_array_get_size(aProperties->asson); ++idx) {
		char *name = cpl_sprintf(KEY_ASSON "%-" CPL_SIZE_FORMAT, idx - idx_start + 1 );
		cpl_propertylist_update_string(aHeader, name,
				cpl_array_get_string(aProperties->asson, idx));
		cpl_free(name);
	}


	cpl_propertylist_update_string(aHeader, KEY_PRODCATG, aProperties->prodcatg);
	cpl_propertylist_set_comment(aHeader, KEY_PRODCATG, KEY_PRODCATG_COMMENT);

	cpl_propertylist_update_string(aHeader, KEY_PROCSOFT, aProperties->procsoft);
	cpl_propertylist_set_comment(aHeader, KEY_PROCSOFT, KEY_PROCSOFT_COMMENT);

	cpl_propertylist_update_string(aHeader, KEY_OBSTECH, aProperties->obstech);
	cpl_propertylist_set_comment(aHeader, KEY_OBSTECH, KEY_OBSTECH_COMMENT);

	const char *fluxcal = KEY_FLUXCAL_VALUE_TRUE;
	if (aProperties->fluxcal == CPL_FALSE) {
		fluxcal = KEY_FLUXCAL_VALUE_FALSE;
	}

	cpl_propertylist_update_string(aHeader, KEY_FLUXCAL, fluxcal);
	cpl_propertylist_set_comment(aHeader, KEY_FLUXCAL, KEY_FLUXCAL_COMMENT);
    double um2nm = 1000.;
	cpl_propertylist_insert_after_double(aHeader, KEY_FLUXCAL, KEY_WAVELMIN,
			aProperties->wlenrange[0] * um2nm);
	cpl_propertylist_set_comment(aHeader, KEY_WAVELMIN, KEY_WAVELMIN_COMMENT);
	cpl_propertylist_insert_after_double(aHeader, KEY_WAVELMIN, KEY_WAVELMAX,
			aProperties->wlenrange[1] * um2nm);
	cpl_propertylist_set_comment(aHeader, KEY_WAVELMAX, KEY_WAVELMAX_COMMENT);
	cpl_propertylist_insert_after_double(aHeader, KEY_WAVELMAX, KEY_SPEC_RES,
			aProperties->specres);
	cpl_propertylist_set_comment(aHeader, KEY_SPEC_RES, KEY_SPEC_RES_COMMENT);
	cpl_propertylist_insert_after_string(aHeader, KEY_SPEC_RES, KEY_SPECSYS,
			aProperties->specsys);
	cpl_propertylist_set_comment(aHeader, KEY_SPECSYS, KEY_SPECSYS_COMMENT);

	/* If any of the properties 'skyres' and 'skyrerr' is negative, the value *
	 * is a default rather than a measured quantity. Indicate this in the     *
	 * comment. The actual value to be written to the header is the absolute  *
	 * value of the quantity.                                                 */

	cpl_propertylist_insert_after_double(aHeader, KEY_SPEC_RES, KEY_SKY_RES,
			fabs(aProperties->skyres));

	const char *method_qualifier = (aProperties->skyres < 0.) ?
			"default" : "measured";
	char *comment = cpl_sprintf(KEY_SKY_RES_COMMENT " (%s)", method_qualifier);
	cpl_propertylist_set_comment(aHeader, KEY_SKY_RES, comment);
	cpl_free(comment);
	/* TODO
  cpl_propertylist_insert_after_double(aHeader, KEY_SKY_RES, KEY_SKY_RERR,
                                       fabs(aProperties->skyrerr));

  method_qualifier = (aProperties->skyrerr < 0.) ? "default" : "measured";
  comment = cpl_sprintf(KEY_SKY_RERR_COMMENT " (%s)", method_qualifier);
  cpl_propertylist_set_comment(aHeader, KEY_SKY_RERR, comment);
  cpl_free(comment);
	 */
	cpl_propertylist_insert_after_double(aHeader, KEY_SKY_RES, KEY_PIXNOISE,
			aProperties->pixnoise);
	cpl_propertylist_set_comment(aHeader, KEY_PIXNOISE, KEY_PIXNOISE_COMMENT);

	/* TODO  */
	cpl_propertylist_insert_after_double(aHeader, KEY_WAVELMAX, KEY_ABMAGLIM,
			aProperties->abmaglimit);
	cpl_propertylist_set_comment(aHeader, KEY_ABMAGLIM, KEY_ABMAGLIM_COMMENT);

	const char *reference = "";
	if (aProperties->referenc) {
		reference = aProperties->referenc;
	}
	cpl_propertylist_update_string(aHeader, KEY_REFERENC, reference);
	cpl_propertylist_set_comment(aHeader, KEY_REFERENC, KEY_REFERENC_COMMENT);

	/* TODO: NEEDED??
  cpl_propertylist_insert_after_double(aHeader, KEY_REFERENC, KEY_SPEC_ERR,
                                       aProperties->wlerror);
  cpl_propertylist_set_comment(aHeader, KEY_SPEC_ERR, KEY_SPEC_ERR_COMMENT);
	 */

	/* Update data unit values to IDP standards */
	/* TODO
  if (!strncmp(eris_ifu_pfits_get_cunit(aHeader, 3), "Angstrom", 9)) {
    cpl_propertylist_update_string(aHeader, "CUNIT3",
        kErisIdpWavelengthUnit);
  }
	 */
	/*
  if (!strncmp(eris_ifu_pfits_get_bunit(aHeader),
        kErisFluxUnitString, strlen(kErisFluxUnitString) + 1)) {
    cpl_propertylist_update_string(aHeader, "BUNIT", kErisIdpFluxDataUnit);
  }
	 */

	/* Fallback required for IDPs */
	/* TODO: not required but would be nice to have them. In such a case we need
	 * to add correct values

  if (!cpl_propertylist_has(aHeader, "CSYER1")) {
    cpl_propertylist_update_double(aHeader, "CSYER1", -1.);
    cpl_propertylist_set_comment(aHeader, "CSYER1",
                                 "[deg] Systematic error in coordinate");
  }
  if (!cpl_propertylist_has(aHeader, "CSYER2")) {
    cpl_propertylist_update_double(aHeader, "CSYER2", -1.);
    cpl_propertylist_set_comment(aHeader, "CSYER2",
                                 "[deg] Systematic error in coordinate");
  }
	 */

	eris_check_error_code("eris_ifu_sdp_properties_update");
	return CPL_ERROR_NONE;
}


/**@}*/
