/* $Id: eris_utils.c,v 1.12 2013-03-25 11:46:49 cgarcia Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2013-03-25 11:46:49 $
 * $Revision: 1.12 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <string.h>
#include "eris_utils.h"
#include <math.h>
/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_utils     Miscellaneous Utilities
 */
/*----------------------------------------------------------------------------*/

/**@{*/
/**
  @brief    handle CPL errors

  @param    func_id    input function name
  @return   cpl_error_code

       Example function to handle errors. In case of error this function
       displays the function caller name, the error message, the line of
       code at which the error occurs
 */
cpl_error_code
eris_check_error_code(const char* func_id) {

   if(cpl_error_get_code() != CPL_ERROR_NONE) {
      cpl_msg_info(cpl_func,"Function %s has an error",func_id);
      cpl_msg_info(cpl_func, "%s",(const char*) cpl_error_get_message());
      cpl_msg_info(cpl_func, "%s",(const char*) cpl_error_get_where());
      //exit(0);
   }
   return cpl_error_get_code();
}
/*---------------------------------------------------------------------------*/
/* AMO added (useful to debug)
 @brief     Print recipe status
 @param     val reference point
 @return    status
 */
/*---------------------------------------------------------------------------*/
int eris_print_rec_status(const int val) {
	if(cpl_error_get_code() != CPL_ERROR_NONE) {
		cpl_msg_error(cpl_func, "Recipe status at %d",val);
		cpl_msg_error(cpl_func, "%s",(const char*) cpl_error_get_message());
		cpl_msg_error(cpl_func, "%s",(const char*) cpl_error_get_where());
		//exit(0);
		return -1;
	}
	return 0;
}


/*---------------------------------------------------------------------------*/
/* AMO added
 * TODO: plist not used. Why?
 @brief     function to create proper FITS header
 @param     filename of product to be saved
 @param     pro_catg of product to be saved
 @param     frame_type  frame type
 @param     frameset input frameset
 @param     parlist input parameterlist
 @param     plist
 @return    status
 */
/*---------------------------------------------------------------------------*/
cpl_error_code eris_setup_product_header(const char* filename,
					    const char* pro_catg,
					    cpl_frame_type frame_type,
					    const char* recipe_name,
					    cpl_frameset* frameset,
					    const cpl_parameterlist* parlist,
					    cpl_propertylist *plist)
{

  cpl_ensure_code(filename, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(pro_catg, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(recipe_name, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(plist, CPL_ERROR_NULL_INPUT);

  cpl_frame* frame = cpl_frame_new();
  cpl_frame_set_filename(frame, filename);
  cpl_frame_set_type(frame, frame_type);
  cpl_frame_set_tag(frame, pro_catg);
  cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);
  cpl_frame_set_level(frame, CPL_FRAME_LEVEL_FINAL);
  cpl_frameset_insert(frameset, frame);
  cpl_dfs_setup_product_header(plist, frame, frameset, parlist,
                                 recipe_name, PACKAGE "/" PACKAGE_VERSION,
                                 CPL_DFS_PRO_DID, NULL);

  eris_check_error_code("eris_setup_product_header");
  return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the pipeline copyright and license
  @return   The copyright and license string

  The function returns a pointer to the statically allocated license string.
  This string should not be modified using the returned pointer.
 */
/*----------------------------------------------------------------------------*/
const char * eris_get_license(void)
{
    const char * eris_license = cpl_get_license("ERIS", "2017");

    return eris_license;
}
long
eris_image_get_threshpix(cpl_image* img, double threshold,
		cpl_boolean thresh_is_min)
{
	double* data = cpl_image_get_data_double(img);
	cpl_size sx = cpl_image_get_size_x(img);
	cpl_size sy = cpl_image_get_size_y(img);
	long npix = sx * sy;
    long nthreshold = 0;
    //cpl_msg_info(cpl_func,"sx: %lld sy: %lld npix: %ld", sx, sy, npix);

    if(thresh_is_min) {
    	/* count pixels above saturation */
    	for(long i = 0; i < npix; i++) {
    		if(data[i] > threshold) {
    			nthreshold++;
    		}
    	}
    	//cpl_msg_info(cpl_func,"count pixels above saturation: %ld",nthreshold);
    } else {
    	//cpl_msg_info(cpl_func,"negative threshold: %g",threshold);
    	/* count pixels below saturation */
    	//cpl_msg_info(cpl_func,"count pixels below saturation: %ld",nthreshold);
    	for(long i = 0; i < npix; i++) {
    		if(data[i] <= threshold) {
    			nthreshold++;
    		}
    	}
    	//cpl_msg_info(cpl_func,"count pixels below saturation: %ld",nthreshold);
    }

	return nthreshold;
}

cpl_error_code
eris_image_flag_threshpix(cpl_image** img, double threshold,
		cpl_boolean thresh_is_min)
{

	cpl_ensure_code(img != NULL, CPL_ERROR_NULL_INPUT);

	double* data = cpl_image_get_data_double(*img);
	cpl_size sx = cpl_image_get_size_x(*img);
	cpl_size sy = cpl_image_get_size_y(*img);
	cpl_mask* mask = cpl_image_get_bpm(*img);
	cpl_binary* pbpm = cpl_mask_get_data(mask);
	long npix = sx * sy;

    //cpl_msg_info(cpl_func,"sx: %lld sy: %lld npix: %ld", sx, sy, npix);

    if(thresh_is_min) {
    	/* count pixels above saturation */
    	for(long i = 0; i < npix; i++) {
    		if(data[i] > threshold) {
    			pbpm[i] = CPL_BINARY_1;
    		}
    	}

    } else {
    	//cpl_msg_info(cpl_func,"negative threshold: %g",threshold);
    	/* count pixels below saturation */

    	for(long i = 0; i < npix; i++) {
    		if(data[i] <= threshold) {
    			pbpm[i] = CPL_BINARY_1;

    		}
    	}

    }

	return cpl_error_get_code();
}
/*---------------------------------------------------------------------------*/
/**
   @brief    Extract frames of user given tag
   @param    input the input set of files
   @param    rtag the frame tag to be selected

   @return   cpl_frameset with frames of user given tag
*/
/*---------------------------------------------------------------------------*/
cpl_frameset*
eris_dfs_extract_frames_with_tag (cpl_frameset * input, const char* rtag)
{

	cpl_ensure(input != NULL, CPL_ERROR_NULL_INPUT, NULL);

	cpl_frame *cur_frame = NULL;

	cpl_size nfrm = 0;
	const char* tag = NULL;
	cpl_frameset* out = cpl_frameset_new();
	/* Loop on frames */
	nfrm = cpl_frameset_get_size(input);
	for (cpl_size i = 0; i < nfrm; i++) {
		cur_frame=cpl_frameset_get_position(input,i);
		tag = cpl_frame_get_tag(cur_frame);
		/* RAW frames */
		if (strcmp(tag, rtag) == 0) {
			cpl_frameset_insert (out, cpl_frame_duplicate(cur_frame));
		}
	}

	return out;

}
/*---------------------------------------------------------------------------*/
/**
   @brief    split input sof in groups: raw and calib
   @param    input the input set of files
   @param    raws the raws file of the input set of files
   @return   cpl error code
*/
/*---------------------------------------------------------------------------*/
cpl_error_code
eris_dfs_extract_raw_frames (cpl_frameset * input, cpl_frameset * raws)
{
	cpl_frame *cur_frame = NULL;

	cpl_size nfrm = 0;
	/* Loop on frames */
	nfrm = cpl_frameset_get_size(input);
	for (cpl_size i = 0; i < nfrm; i++) {
		cur_frame=cpl_frameset_get_position(input,i);
		/* RAW frames */
		if (cpl_frame_get_group (cur_frame) == CPL_FRAME_GROUP_RAW) {
			cpl_frameset_insert (raws, cpl_frame_duplicate(cur_frame));
		}
	}

	return cpl_error_get_code();

}
/*---------------------------------------------------------------------------*/
/**
   @brief    split input sof in groups: raw and calib
   @param    input the input set of files
   @param    calibs the calibration input set of files
   @return   cpl error code
*/
/*---------------------------------------------------------------------------*/
cpl_error_code
eris_dfs_extract_cal_frames (cpl_frameset * input, cpl_frameset * calibs)
{
	cpl_frame *cur_frame = NULL;

	cpl_size nfrm = 0;
	/* Loop on frames */
	nfrm = cpl_frameset_get_size(input);
	for (cpl_size i = 0; i < nfrm; i++) {
		cur_frame=cpl_frameset_get_position(input,i);
		/* RAW frames */
		if (cpl_frame_get_group (cur_frame) == CPL_FRAME_GROUP_CALIB) {
			cpl_frameset_insert (calibs, cpl_frame_duplicate(cur_frame));
		}
	}

	return cpl_error_get_code();

}

cpl_error_code
eris_dfs_check_input_tags(cpl_frameset * input, const char** tags, const int n,
		const int mode)
{
    cpl_ensure_code(input, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(tags, CPL_ERROR_NULL_INPUT);
    int found = 0;
	for(cpl_size i = 0; i < n; i++) {
		found = cpl_frameset_count_tags(input, tags[i]);
		if(found == 0) {
			if(mode == 1) {
			    cpl_msg_error(cpl_func,"Missing required input FITS file with tag: %s", tags[i]);
			    return CPL_ERROR_ILLEGAL_INPUT;
			} else if (mode == 0) {
				cpl_msg_info(cpl_func,"Missing optional input FITS file with tag: %s", tags[i]);
				return CPL_ERROR_NONE;
			}

		}
	}

	return cpl_error_get_code();
}
/*
 * @brief    count number of pixels that are outside threshold
 * @param    img the input image
 * @param    threshold for QC parameters computation
 * @param    thresh_is_min check if the threshold is min or max
 * @return   cpl error code
 */

static long
eris_persistence_get_threshpix(cpl_image* img, double threshold,
		cpl_boolean thresh_is_min)
{
	double* data = cpl_image_get_data_double(img);
	cpl_size sx = cpl_image_get_size_x(img);
	cpl_size sy = cpl_image_get_size_y(img);
	long npix = sx * sy;
    long nthreshold = 0;
    //cpl_msg_info(cpl_func,"sx: %lld sy: %lld npix: %ld", sx, sy, npix);

    if(thresh_is_min) {
    	/* count pixels above saturation */
    	for(long i = 0; i < npix; i++) {
    		if(data[i] > threshold) {
    			nthreshold++;
    		}
    	}
    } else {
    	//cpl_msg_info(cpl_func,"negative threshold: %g",threshold);
    	/* count pixels above saturation */
    	for(long i = 0; i < npix; i++) {
    		if(data[i] <= threshold) {
    			nthreshold++;
    		}
    	}
    }

	return nthreshold;
}
/*
 * @brief    count number of pixels that are outside threshold
 * @param    frame_name the input frame filename
 * @param    frm_mdark the input dark frame
 * @param    threshold for QC parameters computation
 * @param    i   extension id
 * @param    propertylist where QC params are stored
 * @note  image is expected in extention
 * @return   cpl error code
 */
cpl_error_code
eris_get_thresh_pix_qc_for_image(const char* frame_name, cpl_frame* frm_mdark,
		const double threshold, const long long i, cpl_propertylist* applist)
{

	cpl_ensure_code(frame_name, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(frm_mdark, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(applist, CPL_ERROR_NULL_INPUT);

	cpl_msg_info(cpl_func,"Image case");

	long int nthreshold = 0;
	int plane_nthresh_max = 0;

	const char*  fname = cpl_frame_get_filename(frm_mdark);

	cpl_image* mdark = cpl_image_load(fname, CPL_TYPE_DOUBLE, 0, 1);
	cpl_size sx = cpl_image_get_size_x(mdark);
	cpl_size sy = cpl_image_get_size_y(mdark);
	cpl_size npix = sx * sy;

	cpl_image* img = cpl_image_load(frame_name, CPL_TYPE_DOUBLE, 0, 0 );

	cpl_image_subtract(img, mdark);

	nthreshold = eris_persistence_get_threshpix(img, threshold, CPL_TRUE);

	cpl_msg_info(cpl_func,"Frame %lld Threshold pixels: %ld", i, nthreshold);

	cpl_image_delete(img);
	cpl_image_delete(mdark);

	/* Add a QC parameter */
	char* keyname;
	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NTHRESH",i);
	cpl_propertylist_append_int(applist, keyname, nthreshold);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NTHRESH MAX",i);
	cpl_propertylist_append_int(applist, keyname, nthreshold);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NTHRESH TOT",i);
	cpl_propertylist_append_int(applist, keyname, nthreshold);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 BPTHRESH FRAC",i);
	cpl_propertylist_append_double(applist, keyname, (double)nthreshold / npix);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NTHRESH AVG",i);
	cpl_propertylist_append_int(applist, keyname, nthreshold);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 SLICE NTHRESH MAX",i);
	cpl_propertylist_append_int(applist, keyname, plane_nthresh_max);
	cpl_free(keyname);


	return cpl_error_get_code();
}

/*
 * @brief    count number of pixels that are outside threshold
 * @param    frame_name the input frame filename
 * @param    frm_mdark the input dark frame
 * @param    saturation for QC parameters computation
 * @param    saturation_negative for QC parameters computation
 * @param    i   extension id
 * @param    propertylist where QC params are stored
 * @note  image is expected in extention
 * @return   cpl error code
 */

cpl_error_code
eris_get_sat_pix_qc_for_image(const char* frame_name, cpl_frame* frm_mdark,
		const double saturation, const double saturation_negative,
		const long long i, cpl_propertylist* applist)
{

	cpl_ensure_code(frame_name, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(frm_mdark, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(applist, CPL_ERROR_NULL_INPUT);

	cpl_msg_info(cpl_func,"Image case");
	int nsat = 0;
	long int nsat_neg = 0;
	int plane_nsat_max = 0;

	const char*  fname = cpl_frame_get_filename(frm_mdark);

	cpl_image* mdark = cpl_image_load(fname, CPL_TYPE_DOUBLE, 0, 1);
	cpl_size sx = cpl_image_get_size_x(mdark);
	cpl_size sy = cpl_image_get_size_y(mdark);
	cpl_size npix = sx * sy;

	cpl_image* img = cpl_image_load(frame_name, CPL_TYPE_DOUBLE, 0, 0 );
	cpl_image_subtract(img, mdark);

	nsat = eris_persistence_get_threshpix(img, saturation,
			CPL_TRUE);
	cpl_msg_info(cpl_func,"Frame %lld saturation pixels above positive threshold: %ld", i, nsat_neg);

	nsat_neg = eris_persistence_get_threshpix(img, saturation_negative, CPL_FALSE);
	cpl_msg_info(cpl_func,"Frame %lld Saturation pixels below negative threshold: %ld", i, nsat_neg);

	nsat += nsat_neg; // combine two sources of saturation pixels.
	cpl_image_delete(img);
	cpl_image_delete(mdark);

	/* Add a QC parameter */
	char* keyname;
	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NSAT",i);
	cpl_propertylist_append_int(applist, keyname, nsat);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NSAT MAX",i);
	cpl_propertylist_append_int(applist, keyname, nsat);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NSAT TOT",i);
	cpl_propertylist_append_int(applist, keyname, nsat);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 BPSAT FRAC",i);
	cpl_propertylist_append_double(applist, keyname, (double)nsat / npix);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NSAT AVG",i);
	cpl_propertylist_append_int(applist, keyname, nsat);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLICE NSAT MAX",i);
	cpl_propertylist_append_int(applist, keyname, plane_nsat_max);
	cpl_free(keyname);

	return cpl_error_get_code();
}



cpl_error_code
eris_get_sat_qc_for_image(const char* frame_name, cpl_frame* frm_mdark,
		const double saturation, const double saturation_negative,
		const double threshold, const long long i, cpl_propertylist* applist)
{

	cpl_ensure_code(frame_name, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(frm_mdark, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(applist, CPL_ERROR_NULL_INPUT);

	cpl_msg_info(cpl_func,"Image case");
	int nsat = 0;
	long int nsat_neg = 0;
	long int nthreshold = 0;
	int plane_nsat_max = 0;
	int plane_nthresh_max = 0;

	const char*  fname = cpl_frame_get_filename(frm_mdark);

	cpl_image* mdark = cpl_image_load(fname, CPL_TYPE_DOUBLE, 0, 1);
	cpl_size sx = cpl_image_get_size_x(mdark);
	cpl_size sy = cpl_image_get_size_y(mdark);
	cpl_size npix = sx * sy;



	cpl_image* img_persistence = cpl_image_load(frame_name, CPL_TYPE_DOUBLE, 0, 0 );

	cpl_image_subtract(img_persistence, mdark);
	cpl_image_save(img_persistence,"image_persistence.fits",CPL_TYPE_DOUBLE,NULL,CPL_IO_DEFAULT);

	nsat = eris_persistence_get_threshpix(img_persistence, saturation,
			CPL_TRUE);
	cpl_msg_info(cpl_func,"Frame %lld saturation pixels above positive threshold: %ld", i, nsat_neg);

	nthreshold = eris_persistence_get_threshpix(img_persistence, threshold,
			CPL_TRUE);

	cpl_msg_info(cpl_func,"Frame %lld Threshold pixels: %ld", i, nthreshold);
	nsat_neg = eris_persistence_get_threshpix(img_persistence,
			saturation_negative, CPL_FALSE);
	cpl_msg_info(cpl_func,"Frame %lld Saturation pixels below negative threshold: %ld", i, nsat_neg);

	nsat += nsat_neg; // combine two sources of saturation pixels.
	cpl_image_delete(img_persistence);
	cpl_image_delete(mdark);

	/* Add a QC parameter */
	char* keyname;
	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NSAT",i);
	cpl_propertylist_append_int(applist, keyname, nsat);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NSAT MAX",i);
	cpl_propertylist_append_int(applist, keyname, nsat);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NSAT TOT",i);
	cpl_propertylist_append_int(applist, keyname, nsat);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 BPSAT FRAC",i);
	cpl_propertylist_append_double(applist, keyname, (double)nsat / npix);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NSAT AVG",i);
	cpl_propertylist_append_int(applist, keyname, nsat);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLICE NSAT MAX",i);
	cpl_propertylist_append_int(applist, keyname, plane_nsat_max);
	cpl_free(keyname);



	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NTHRESH",i);
	cpl_propertylist_append_int(applist, keyname, nthreshold);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NTHRESH MAX",i);
	cpl_propertylist_append_int(applist, keyname, nthreshold);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NTHRESH TOT",i);
	cpl_propertylist_append_int(applist, keyname, nthreshold);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 BPTHRESH FRAC",i);
	cpl_propertylist_append_double(applist, keyname, (double)nthreshold / npix);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NTHRESH AVG",i);
	cpl_propertylist_append_int(applist, keyname, nthreshold);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 SLICE NTHRESH MAX",i);
	cpl_propertylist_append_int(applist, keyname, plane_nthresh_max);
	cpl_free(keyname);


	return cpl_error_get_code();
}

cpl_error_code
eris_get_sat_qc_for_cube(const char* frame_name, cpl_frame* frm_mdark,
		const double saturation, const double saturation_negative,
		const double threshold, const long long i, cpl_propertylist* applist)
{

	cpl_ensure_code(frame_name, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(frm_mdark, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(applist, CPL_ERROR_NULL_INPUT);


	cpl_msg_info(cpl_func,"Cube case");
	const char*  fname = cpl_frame_get_filename(frm_mdark);

	cpl_image* mdark = cpl_image_load(fname, CPL_TYPE_DOUBLE, 0, 1);
	cpl_size sx = cpl_image_get_size_x(mdark);
	cpl_size sy = cpl_image_get_size_y(mdark);
	cpl_size npix = sx * sy;


	cpl_imagelist* iml_persistence = cpl_imagelist_load(frame_name, CPL_TYPE_DOUBLE, 0 );
	cpl_size sz = cpl_imagelist_get_size(iml_persistence);
	char* keyname;
	int nsat = 0;
	int nsat_neg = 0;
	long nsat_max = 0;
	long nthreshold_max = 0;
	int nsat_tot = 0;
	int nthreshold_tot = 0;
	double nsat_avg = 0;
	double nthreshold_avg = 0;
	int nthreshold = 0;
	int plane_nsat_max = 0;
	int plane_nthresh_max = 0;
	double bpsat_frac_tot = 0;
	double bpthresh_frac_tot = 0;
	cpl_image* img;
	for(cpl_size k = 0; k < sz; k++) {
		cpl_imagelist_subtract_image(iml_persistence, mdark);
		img = cpl_imagelist_get(iml_persistence, k);
		nsat = eris_persistence_get_threshpix(img, saturation, CPL_TRUE);
		nthreshold = eris_persistence_get_threshpix(img, threshold, CPL_TRUE);
		nsat_neg = eris_persistence_get_threshpix(img,
				saturation_negative, CPL_FALSE);
		nsat += nsat_neg; // combine both sources of saturation pixels
		/* Add a QC parameter  */

		keyname = cpl_sprintf("ESO QC FRM%lld SLIC%lld NSAT", i, k);
		cpl_propertylist_append_int(applist, keyname, nsat);
		cpl_free(keyname);

		keyname = cpl_sprintf("ESO QC FRM%lld SLIC%lld BPSAT FRAC",i, k);
		cpl_propertylist_append_double(applist, keyname, (double)nsat / npix);
		cpl_free(keyname);

		keyname = cpl_sprintf("ESO QC FRM%lld SLIC%lld NTHRESH",i, k);
		cpl_propertylist_append_int(applist, keyname, nthreshold);
		cpl_free(keyname);

		keyname = cpl_sprintf("ESO QC FRM%lld SLIC%lld BPTHRESH FRAC", i, k);
		cpl_propertylist_append_double(applist, keyname, (double)nthreshold / npix);
		cpl_free(keyname);

		if(nsat > nsat_max) {
			nsat_max = nsat;
			plane_nsat_max = k;
		}

		if(nthreshold > nthreshold_max) {
			nthreshold_max = nthreshold;
			plane_nthresh_max = k;
		}
		nsat_tot += nsat;
		bpsat_frac_tot += (double)nsat / npix;

		nthreshold_tot += nthreshold;
		bpthresh_frac_tot += (double)nthreshold / npix;


	}
	cpl_imagelist_delete(iml_persistence);
    cpl_image_delete(mdark);

	nsat_avg = (double) nsat_tot / sz;

	nthreshold_avg = (double) nthreshold_tot / sz;

	keyname = cpl_sprintf("ESO QC FRM%lld NSAT MAX", i);
	cpl_propertylist_append_int(applist, keyname, nsat_max);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld NSAT TOT", i);
	cpl_propertylist_append_int(applist, keyname, nsat_tot);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld BPSAT FRAC", i);
	cpl_propertylist_append_double(applist, keyname, bpsat_frac_tot / sz);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld NSAT AVG", i);
	cpl_propertylist_append_double(applist, keyname, nsat_avg);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLICE NSAT MAX", i);
	cpl_propertylist_append_int(applist, keyname, plane_nsat_max);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld NTHRESH MAX", i);
	cpl_propertylist_append_int(applist, keyname, nthreshold_max);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld NTHRESH TOT", i);
	cpl_propertylist_append_int(applist, keyname, nthreshold_tot);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld BPTHRESH FRAC", i);
	cpl_propertylist_append_double(applist, keyname, bpthresh_frac_tot / sz);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld NTHRESH AVG", i);
	cpl_propertylist_append_double(applist, keyname, nthreshold_avg);
	cpl_free(keyname);

	keyname = cpl_sprintf("ESO QC FRM%lld SLICE NTHRESH MAX", i);
	cpl_propertylist_append_int(applist, keyname, plane_nthresh_max);
	cpl_free(keyname);

	return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 @brief   get double parameter value if changed by the user
 @param   parlist list of input recipe parameters
 @param   pname   recipe parameter name
 @param   pvalue  recipe parameter value
 @return  cpl_error_code
 */
/*----------------------------------------------------------------------------*/

/* set propertylist (double) value */
cpl_error_code
eris_parameters_get_double(const cpl_parameterlist* parlist,
                    const char* pname, double *pvalue)
{
  cpl_ensure_code(parlist != NULL, CPL_ERROR_NULL_INPUT);
  const cpl_parameter* p = cpl_parameterlist_find_const(parlist, pname);
  cpl_boolean p_has_changed = eris_param_has_changed(p);

  if  ( cpl_parameter_get_default_flag(p) && p_has_changed != CPL_FALSE) {
      *pvalue = cpl_parameter_get_double(p);
  } else {
      *pvalue = cpl_parameter_get_default_double(p);
  }

  return cpl_error_get_code();
}

/* set propertylist (double) value */
cpl_error_code
eris_parameters_get_int(const cpl_parameterlist* parlist,
                    const char* pname, int *pvalue)
{
  cpl_ensure_code(parlist != NULL, CPL_ERROR_NULL_INPUT);
  const cpl_parameter* p = cpl_parameterlist_find_const(parlist, pname);
  cpl_boolean p_has_changed = eris_param_has_changed(p);

  if  ( cpl_parameter_get_default_flag(p) && p_has_changed != CPL_FALSE) {
      *pvalue = cpl_parameter_get_int(p);
  } else {
      *pvalue = cpl_parameter_get_default_int(p);
  }

  return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief   verify if a parameter value has been changed (from command line or
          or rc file by a user)
 @param    p    parameter
 @return   CPL_TRUE if parameter has changes, else CPL_FALSE
 */
/*----------------------------------------------------------------------------*/

cpl_boolean
eris_param_has_changed(const cpl_parameter* p)
{
  cpl_ensure_code(p != NULL,CPL_FALSE);
  cpl_type type = cpl_parameter_get_type(p);
  cpl_boolean has_changed = CPL_FALSE;
  /* handle different data types */
  switch (type) {

    case CPL_TYPE_INT: {
      int val = cpl_parameter_get_int(p);
      int def = cpl_parameter_get_default_int(p);
      has_changed = (val != def)  ? CPL_TRUE : CPL_FALSE;
    }
    break;

    case CPL_TYPE_FLOAT: {
      float val = cpl_parameter_get_double(p);
      float def = cpl_parameter_get_default_double(p);
      has_changed = (val != def)  ? CPL_TRUE : CPL_FALSE;
    }
    break;

    case CPL_TYPE_DOUBLE: {
      double val = cpl_parameter_get_double(p);
      double def = cpl_parameter_get_default_double(p);
      has_changed = (val != def)  ? CPL_TRUE : CPL_FALSE;
    }
    break;

    case CPL_TYPE_STRING:{
      const char* val = cpl_parameter_get_string(p);
      const char* def = cpl_parameter_get_default_string(p);
      if ( strcmp(val,def) != 0 ) {
      has_changed = CPL_TRUE;
      }
    }
    break;

    default:
      cpl_msg_error(cpl_func,"case not found! %d string type: %d",type,
            CPL_TYPE_STRING);
      break;

  }
  return has_changed;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief
 *   Check if all SOF files exist
 *
 * @param frameset The input set-of-frames
 *
 * @return 1 if not all files exist, 0 if they all exist.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
eris_files_dont_exist(cpl_frameset *frameset)
{
    const char *func = "dfs_files_dont_exist";
    cpl_ensure_code(frameset != NULL, CPL_ERROR_NULL_INPUT);

    if (frameset == NULL ) {
        cpl_error_set(func, CPL_ERROR_NULL_INPUT);
        return CPL_ERROR_NULL_INPUT;
    }

    if (cpl_frameset_is_empty(frameset)) {
    	cpl_error_set(func, CPL_ERROR_FILE_NOT_FOUND);
        return CPL_ERROR_FILE_NOT_FOUND;
    }

    cpl_frameset_iterator* it = cpl_frameset_iterator_new(frameset);
    cpl_frame *frame = cpl_frameset_iterator_get(it);
    const char*  fname = NULL;
    cpl_propertylist* plist = NULL;
    while (frame) {
        fname = cpl_frame_get_filename(frame);
    	cpl_ensure_code(strcmp(fname, "/dev/null"), CPL_ERROR_NULL_INPUT);

    	/* test if file exist */
    	if (access(cpl_frame_get_filename(frame), F_OK)) {
    	            cpl_msg_error(func, "File %s (%s) was not found",
    	                            cpl_frame_get_filename(frame),
    	                            cpl_frame_get_tag(frame));
    	            cpl_error_set(func, CPL_ERROR_FILE_NOT_FOUND);
    	            cpl_frameset_iterator_delete(it);
    	            cpl_ensure_code(CPL_FALSE, CPL_ERROR_FILE_NOT_FOUND);
    	}
    	/* if file exist test that at least INSTRUME is defined */
    	plist = cpl_propertylist_load(fname, 0);
    	if(!cpl_propertylist_has(plist, "INSTRUME")) {
    		cpl_propertylist_delete(plist);
    		cpl_frameset_iterator_delete(it);
    		cpl_ensure_code(CPL_FALSE, CPL_ERROR_DATA_NOT_FOUND);
    	}
    	cpl_propertylist_delete(plist);

        cpl_frameset_iterator_advance(it, 1);
        frame = cpl_frameset_iterator_get(it);

    }

    cpl_frameset_iterator_delete(it);

    if (cpl_error_get_code())
        return cpl_error_get_code();

    return CPL_ERROR_NONE;
}

cpl_error_code
eris_image_flag_nan(cpl_image** ima)
{
	cpl_ensure_code(ima != NULL, CPL_ERROR_NULL_INPUT);

	cpl_mask* bpm = cpl_image_get_bpm(*ima);
	cpl_binary* pbpm = cpl_mask_get_data(bpm);
	cpl_size sx = cpl_image_get_size_x(*ima);
	cpl_size sy = cpl_image_get_size_y(*ima);
	double* pdata = cpl_image_get_data(*ima);
	for(cpl_size j = 0; j < sy; j++ ) {
		for(cpl_size i = 0; i < sx; i++ ) {
			if(isnan(pdata[i+j*sx])){

				pbpm[i+j*sx] = CPL_BINARY_1;

			}
		}
	}


	return cpl_error_get_code();
}

/**@}*/
