/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <float.h>
#include <math.h>
#include <string.h>
#include "eris_nix_test_defs.h"
#include "eris_nix_casu_utils.h"
#include "eris_nix_utils.h"
#include <hdrl.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_casu_utils_test  Unit test of eris_nix_casu_utils
 *
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                   Defines
 -----------------------------------------------------------------------------*/

#define SMALL_TEST_SIZE_X 64
#define SMALL_TEST_SIZE_Y 64

/*-----------------------------------------------------------------------------
                                Helper Functions
 -----------------------------------------------------------------------------*/

/**
 * @brief Create a simple test hdrl_image filled with a constant value
 *        The images will have BPMs (bad pixel masks) initialized
 */
static hdrl_image * create_test_hdrl_image(int nx, int ny, double value) {
    hdrl_image * himage = hdrl_image_new(nx, ny);
    cpl_image * img = hdrl_image_get_image(himage);
    cpl_image * err = hdrl_image_get_error(himage);
    
    cpl_image_fill_window(img, 1, 1, nx, ny, value);
    cpl_image_fill_window(err, 1, 1, nx, ny, sqrt(value));
    
    /* Initialize BPMs for both data and error images - 
       the source code expects these to exist */
    cpl_mask * mask_data = cpl_mask_new(nx, ny);
    cpl_mask * mask_err = cpl_mask_new(nx, ny);
    cpl_image_set_bpm(img, mask_data);
    cpl_image_set_bpm(err, mask_err);
    
    return himage;
}

/**
 * @brief Create a simple test confidence image filled with a constant value
 */
static cpl_image * create_test_confidence(int nx, int ny, double value) {
    cpl_image * confidence = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(confidence, 1, 1, nx, ny, value);
    return confidence;
}

/**
 * @brief Create a test located_imagelist with specified number of images
 */
static located_imagelist * create_test_limlist(int nimages, int nx, int ny) {
    located_imagelist * limlist = cpl_malloc(sizeof(located_imagelist));
    limlist->size = nimages;
    limlist->limages = cpl_malloc(nimages * sizeof(located_image *));
    
    for (int i = 0; i < nimages; i++) {
        hdrl_image * himage = create_test_hdrl_image(nx, ny, 10.0 * (i + 1));
        cpl_image * confidence = create_test_confidence(nx, ny, 100.0);
        
        cpl_propertylist * plist = cpl_propertylist_new();
        cpl_propertylist_update_string(plist, "INSTRUME", "ERIS");
        cpl_propertylist_update_double(plist, "MJD-OBS", 60000.0 + i);
        
        cpl_frame * frame = cpl_frame_new();
        char filename[256];
        snprintf(filename, sizeof(filename), "test_image_%d.fits", i);
        cpl_frame_set_filename(frame, filename);
        cpl_frame_set_tag(frame, "TEST");
        cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
        cpl_frame_set_level(frame, CPL_FRAME_LEVEL_NONE);
        cpl_frame_set_type(frame, CPL_FRAME_TYPE_IMAGE);
        
        limlist->limages[i] = enu_located_image_new(himage,
                                                    NULL,
                                                    confidence,
                                                    NULL,
                                                    NULL,
                                                    plist,
                                                    NULL, NULL, NULL, NULL,
                                                    frame);
    }
    
    return limlist;
}

/**
 * @brief Delete a test located_imagelist
 */
static void delete_test_limlist(located_imagelist * limlist) {
    if (limlist == NULL) return;
    
    for (cpl_size i = 0; i < limlist->size; i++) {
        enu_located_image_delete(limlist->limages[i]);
    }
    cpl_free(limlist->limages);
    cpl_free(limlist);
}

/*-----------------------------------------------------------------------------
                                Test Functions
 -----------------------------------------------------------------------------*/

/**
 * @brief Test encu_limlist_to_casu_fits with NULL input
 */
static void test_limlist_to_casu_fits_null_input(void) {
    casu_fits ** indata = NULL;
    casu_fits ** inconf = NULL;
    casu_fits ** invar = NULL;
    
    /* Test with NULL limlist - should return CPL_ERROR_NONE */
    cpl_error_code error = encu_limlist_to_casu_fits(NULL, &indata, &inconf, &invar);
    cpl_test_eq(error, CPL_ERROR_NONE);
    cpl_test_null(indata);
    cpl_test_null(inconf);
    cpl_test_null(invar);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test encu_limlist_to_casu_fits with pre-existing error
 */
static void test_limlist_to_casu_fits_preexisting_error(void) {
    casu_fits ** indata = NULL;
    casu_fits ** inconf = NULL;
    casu_fits ** invar = NULL;
    
    /* Set a pre-existing error */
    cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
    
    /* Test that function returns immediately with pre-existing error */
    cpl_error_code error = encu_limlist_to_casu_fits(NULL, &indata, &inconf, &invar);
    cpl_test_eq(error, CPL_ERROR_ILLEGAL_INPUT);
    
    /* Reset error state */
    cpl_error_reset();
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test encu_limlist_to_casu_fits with valid input
 */
static void test_limlist_to_casu_fits_valid_input(void) {
    const int nimages = 3;
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    
    casu_fits ** indata = NULL;
    casu_fits ** inconf = NULL;
    casu_fits ** invar = NULL;
    
    /* Create test located_imagelist */
    located_imagelist * limlist = create_test_limlist(nimages, nx, ny);
    cpl_test_nonnull(limlist);
    cpl_test_eq(limlist->size, nimages);
    
    /* Test conversion */
    cpl_error_code error = encu_limlist_to_casu_fits(limlist, &indata, &inconf, &invar);
    cpl_test_eq(error, CPL_ERROR_NONE);
    
    /* Verify outputs are allocated */
    cpl_test_nonnull(indata);
    cpl_test_nonnull(inconf);
    cpl_test_nonnull(invar);
    
    /* Verify each casu_fits structure */
    for (int i = 0; i < nimages; i++) {
        cpl_test_nonnull(indata[i]);
        cpl_test_nonnull(inconf[i]);
        cpl_test_nonnull(invar[i]);
        
        /* Verify extension numbers */
        cpl_test_eq(indata[i]->nexten, 1);
        cpl_test_eq(inconf[i]->nexten, 4);
        cpl_test_eq(invar[i]->nexten, 2);
        
        /* Verify filenames are set */
        cpl_test_nonnull(indata[i]->fname);
        cpl_test_nonnull(inconf[i]->fname);
        cpl_test_nonnull(invar[i]->fname);
        
        /* Verify images are wrapped (not NULL) */
        cpl_test_nonnull(casu_fits_get_image(indata[i]));
        cpl_test_nonnull(casu_fits_get_image(inconf[i]));
        cpl_test_nonnull(casu_fits_get_image(invar[i]));
    }
    
    /* Cleanup */
    casu_fits_delete_list(indata, nimages);
    casu_fits_delete_list(inconf, nimages);
    casu_fits_delete_list(invar, nimages);
    delete_test_limlist(limlist);
    
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test encu_limlist_to_casu_fits with NULL himage in located_image
 */
static void test_limlist_to_casu_fits_null_himage(void) {
    casu_fits ** indata = NULL;
    casu_fits ** inconf = NULL;
    casu_fits ** invar = NULL;
    
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    
    /* Create a located_imagelist with NULL himage */
    located_imagelist * limlist = cpl_malloc(sizeof(located_imagelist));
    limlist->size = 1;
    limlist->limages = cpl_malloc(sizeof(located_image *));
    
    cpl_image * confidence = create_test_confidence(nx, ny, 100.0);
    cpl_propertylist * plist = cpl_propertylist_new();
    cpl_frame * frame = cpl_frame_new();
    cpl_frame_set_filename(frame, "test_null_himage.fits");
    cpl_frame_set_tag(frame, "TEST");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
    cpl_frame_set_level(frame, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(frame, CPL_FRAME_TYPE_IMAGE);
    
    /* Create located_image with NULL himage */
    limlist->limages[0] = enu_located_image_new(NULL,  /* NULL himage */
                                                NULL,
                                                confidence,
                                                NULL,
                                                NULL,
                                                plist,
                                                NULL, NULL, NULL, NULL,
                                                frame);
    
    /* Test conversion - should fail with NULL_INPUT error */
    cpl_error_code error = encu_limlist_to_casu_fits(limlist, &indata, &inconf, &invar);
    cpl_test_eq(error, CPL_ERROR_NULL_INPUT);
    
    /* Cleanup - the function allocates the arrays before failing,
       so we need to free them (but not call casu_fits_delete_list 
       since no elements were set) */
    if (indata) cpl_free(indata);
    if (inconf) cpl_free(inconf);
    if (invar) cpl_free(invar);
    delete_test_limlist(limlist);
    cpl_error_reset();
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test encu_limlist_to_casu_fits handles non-finite pixel values
 */
static void test_limlist_to_casu_fits_nonfinite_values(void) {
    const int nimages = 1;
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    
    casu_fits ** indata = NULL;
    casu_fits ** inconf = NULL;
    casu_fits ** invar = NULL;
    
    /* Create a located_imagelist */
    located_imagelist * limlist = cpl_malloc(sizeof(located_imagelist));
    limlist->size = nimages;
    limlist->limages = cpl_malloc(nimages * sizeof(located_image *));
    
    /* Create hdrl_image with BPMs already initialized */
    hdrl_image * himage = create_test_hdrl_image(nx, ny, 10.0);
    cpl_image * confidence = create_test_confidence(nx, ny, 100.0);
    
    /* Insert a NaN value in the image data at position [10, 0] (linear index 10) */
    double * data = cpl_image_get_data_double(hdrl_image_get_image(himage));
    data[10] = NAN;  /* Set one pixel to NaN */
    
    cpl_propertylist * plist = cpl_propertylist_new();
    cpl_propertylist_update_string(plist, "INSTRUME", "ERIS");
    
    cpl_frame * frame = cpl_frame_new();
    cpl_frame_set_filename(frame, "test_nan.fits");
    cpl_frame_set_tag(frame, "TEST");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
    cpl_frame_set_level(frame, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(frame, CPL_FRAME_TYPE_IMAGE);
    
    limlist->limages[0] = enu_located_image_new(himage,
                                                NULL,
                                                confidence,
                                                NULL,
                                                NULL,
                                                plist,
                                                NULL, NULL, NULL, NULL,
                                                frame);
    
    /* Test conversion - should handle NaN by flagging in BPM */
    cpl_error_code error = encu_limlist_to_casu_fits(limlist, &indata, &inconf, &invar);
    cpl_test_eq(error, CPL_ERROR_NONE);
    
    /* Verify outputs are allocated */
    cpl_test_nonnull(indata);
    cpl_test_nonnull(inconf);
    cpl_test_nonnull(invar);
    
    /* Verify confidence was set to 0 for the NaN pixel */
    double * conf_data = cpl_image_get_data_double(limlist->limages[0]->confidence);
    cpl_test_abs(conf_data[10], 0.0, DBL_EPSILON);
    
    /* Verify BPM was set for the NaN pixel */
    cpl_mask * bpm = cpl_image_get_bpm(hdrl_image_get_image(limlist->limages[0]->himage));
    cpl_test_nonnull(bpm);
    cpl_binary * bpm_data = cpl_mask_get_data(bpm);
    cpl_test_eq(bpm_data[10], CPL_BINARY_1);
    
    /* Cleanup */
    casu_fits_delete_list(indata, nimages);
    casu_fits_delete_list(inconf, nimages);
    casu_fits_delete_list(invar, nimages);
    delete_test_limlist(limlist);
    
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test encu_limlist_to_casu_fits with single image
 */
static void test_limlist_to_casu_fits_single_image(void) {
    const int nimages = 1;
    const int nx = SMALL_TEST_SIZE_X;
    const int ny = SMALL_TEST_SIZE_Y;
    
    casu_fits ** indata = NULL;
    casu_fits ** inconf = NULL;
    casu_fits ** invar = NULL;
    
    /* Create test located_imagelist with single image */
    located_imagelist * limlist = create_test_limlist(nimages, nx, ny);
    cpl_test_nonnull(limlist);
    cpl_test_eq(limlist->size, nimages);
    
    /* Test conversion */
    cpl_error_code error = encu_limlist_to_casu_fits(limlist, &indata, &inconf, &invar);
    cpl_test_eq(error, CPL_ERROR_NONE);
    
    /* Verify outputs */
    cpl_test_nonnull(indata);
    cpl_test_nonnull(indata[0]);
    cpl_test_nonnull(inconf);
    cpl_test_nonnull(inconf[0]);
    cpl_test_nonnull(invar);
    cpl_test_nonnull(invar[0]);
    
    /* Verify image dimensions after conversion (should be float type) */
    cpl_image * img = casu_fits_get_image(indata[0]);
    cpl_test_eq(cpl_image_get_size_x(img), nx);
    cpl_test_eq(cpl_image_get_size_y(img), ny);
    cpl_test_eq(cpl_image_get_type(img), CPL_TYPE_FLOAT);
    
    /* Cleanup */
    casu_fits_delete_list(indata, nimages);
    casu_fits_delete_list(inconf, nimages);
    casu_fits_delete_list(invar, nimages);
    delete_test_limlist(limlist);
    
    cpl_test_error(CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_nix_casu_utils module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    /* Test NULL input handling */
    test_limlist_to_casu_fits_null_input();
    
    /* Test pre-existing error handling */
    test_limlist_to_casu_fits_preexisting_error();
    
    /* Test valid input with multiple images */
    test_limlist_to_casu_fits_valid_input();
    
    /* Test with NULL himage in located_image */
    test_limlist_to_casu_fits_null_himage();
    
    /* Test handling of non-finite values */
    test_limlist_to_casu_fits_nonfinite_values();
    
    /* Test with single image */
    test_limlist_to_casu_fits_single_image();

    return cpl_test_end(0);
}

/**@}*/

