/* $Id: eris_ifu_flat.c,v 0.1 2018-07-22 06:06:06 agudo Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*
 * $Author: agudo $
 * $Date: 2018-07-22 06:06:06  $
 * $Revision: 0.1 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>          /* allows the program compilation */
#endif

/*-----------------------------------------------------------------------------
                                Includes
 ----------------------------------------------------------------------------*/
#include <string.h>

#include <cpl.h>
#include <hdrl.h>

#include "eris_utils.h"
#include "eris_pfits.h"
#include "eris_ifu_dfs.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_functions.h"
#include "eris_ifu_error.h"
#include "eris_ifu_debug.h"
#include "eris_ifu_flat_static.h"

/*-----------------------------------------------------------------------------
                            Static variables
 ----------------------------------------------------------------------------*/
static const char eris_ifu_flat_description[] = "\
This recipe performs ERIS/SPIFFIER flat field frames data reduction.\n\
\n\
Optionally one may have in input also several bad pixel maps to be coadded.\n\
\n\
-----------------------------------------------------------------------------\n\
  Input files:\n\
   DO CATG           Explanation                             Required #Frames\n\
   -------           -----------                             -------- -------\n\
   FLAT_LAMP         Flatlamp exposures                         Y       2-n  \n\
                     (at least 3 frames recommended)                         \n\
   BPM_DARK          Bad pixel mask from eris_ifu_dark          Y        1   \n\
   BPM_DETLIN        Bad pixel mask from eris_ifu_detlin        N      [0,1] \n\
   BPM_DIST          Bad pixel mask from eris_ifu_distortion    N      [0,1] \n\
\n\
  Output files:\n\
   DO CATG           Explanation\n\
   -------           -----------\n\
   MASTER_FLAT       Normalised flat field\n\
   BPM_FLAT          Updated bad pixel mask\n\
-----------------------------------------------------------------------------\n\
  Parameters:\n\
  --mode=fast    Simple subtraction of on- and off-frames and collapsing\n\
                 --collapse* parameters apply as well\n\
  --mode=hdrl    The HDRL-way to create a masterflat (not well suited to\n\
                 spectroscopic raw data)\n\
                 --collapse* + --flat* parameters apply as well\n\
  --mode=segment The raw data is segmented vertically and inspected individually\n\
\n\
Information on relevant parameters may be found with\n\
  esorex --params "REC_NAME_FLAT"\n\
  esorex --help   "REC_NAME_FLAT"\n";

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/
cpl_recipe_define(eris_ifu_flat, ERIS_BINARY_VERSION, "Alex Agudo Berbel",
                  PACKAGE_BUGREPORT, "2018",
                  "This recipe performs flat field data reduction",
                  eris_ifu_flat_description);

/*-----------------------------------------------------------------------------
                                Functions code
 ----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    pl  the non-NULL parameterlist to fill
  @return   CPL_ERROR_NONE if everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code eris_ifu_flat_fill_parameterlist(cpl_parameterlist *pl)
{
    cpl_error_code      err         = CPL_ERROR_NONE;
//    cpl_parameter       *p          = NULL;

    cpl_ensure_code(pl, CPL_ERROR_NULL_INPUT);

    TRY
    {
        /* --instrument & --product_depth*/
        BRK_IF_ERROR(
            eris_parlist_config_add_all_recipes(pl, REC_NAME_FLAT));

        /* add common bpm-parameters */
        BRK_IF_ERROR(
            eris_parlist_config_add_bpm(pl, REC_NAME_FLAT));

        /* add common flat-parameters */
        BRK_IF_ERROR(
            eris_parlist_config_add_flat(pl, REC_NAME_FLAT));
    }
    CATCH
    {
        CATCH_MSGS();
        err = cpl_error_get_code();
    }

    return err;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int eris_ifu_flat(cpl_frameset *frameset,
                         const cpl_parameterlist *parlist)
{
    cpl_propertylist    *qc_list                = NULL;
    hdrl_image          *masterFlatHdrlImg_lo   = NULL;
    cpl_image           *qualityImage           = NULL,
                        *masterFlatImage        = NULL,
                        *errorImage             = NULL,
                        *bp_img_flat            = NULL;
    const cpl_mask      *bp_mask_flat           = NULL;
    int                 productDepth            = 0,
                        nrfr                    = 0;
    const char          *instrument             = NULL;

    cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist,  CPL_ERROR_NULL_INPUT);


    /* check for invalid input */
    if(eris_files_dont_exist(frameset) != CPL_ERROR_NONE) {
    	return CPL_ERROR_BAD_FILE_FORMAT;
    }
    TRY
    {
        /* --instrument */
        BRK_IF_NULL(
            instrument = cpl_parameter_get_string(
                                cpl_parameterlist_find_const(parlist,
                                            "eris.eris_ifu_flat.instrument")));
        cpl_msg_info(cpl_func, "Processing %s frames", instrument);

        /* Identify the RAW and CALIB frames in the input frameset */
        BRK_IF_ERROR(
            eris_ifu_dfs_set_groups(frameset));

        /* check number of frames*/
        nrfr = cpl_frameset_count_tags(frameset, ERIS_IFU_PRO_DARK_BPM);
        ASSURE(nrfr == 1,
               CPL_ERROR_ILLEGAL_INPUT,
               "Exactly 1 "ERIS_IFU_PRO_DARK_BPM" frame must be provided.");

        nrfr = cpl_frameset_count_tags(frameset, ERIS_IFU_PRO_DIST_BPM);
        ASSURE((nrfr == 0) || (nrfr == 1),
               CPL_ERROR_ILLEGAL_INPUT,
               "Either 0 or 1 "ERIS_IFU_PRO_DIST_BPM" frame must be provided.");

        nrfr = cpl_frameset_count_tags(frameset, ERIS_IFU_PRO_DETLIN_BPM);
        ASSURE((nrfr == 0) || (nrfr == 1),
               CPL_ERROR_ILLEGAL_INPUT,
               "Either 0 or 1 "ERIS_IFU_PRO_DETLIN_BPM" frame must be provided.");

        /* save additional products if requested via productDepth argument */
        productDepth = cpl_parameter_get_int(
                                cpl_parameterlist_find_const(parlist,
                                        "eris.eris_ifu_flat.product_depth"));

        // get flat-mode
        const char * tmp = cpl_parameter_get_string(cpl_parameterlist_find_const(parlist,
                                         "eris.eris_ifu_flat.mode"));
        flatMode mode = FLAT_MODE_NONE;
        if (!strcmp(tmp, flatModes[0])) {
            mode = FLAT_MODE_SEGMENT;
        } else if (!strcmp(tmp, flatModes[1])) {
            mode = FLAT_MODE_HDRL;
        } else if (!strcmp(tmp, flatModes[2])) {
            mode = FLAT_MODE_FAST;
        } else {
            BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT, "Wrong mode provided.");
        }
        CHECK_ERROR_STATE();

        /* Performing flat calculation */
        BRK_IF_ERROR(
            eris_ifu_flat_static(frameset,
                                 parlist,
                                 mode,
                                 productDepth,
                                 ERIS_IFU_PRO_FLAT,
                                 REC_NAME_FLAT,
                                 "FLAT",
                                 instrument,
                                 &qc_list,
                                 &masterFlatHdrlImg_lo,
                                 &qualityImage));

        /*  fetch master flat image */
        BRK_IF_NULL(
            masterFlatImage = hdrl_image_get_image(masterFlatHdrlImg_lo));
        /* fetch noise image */
        BRK_IF_NULL(
            errorImage = hdrl_image_get_error(masterFlatHdrlImg_lo));

        /* save masterflat image */
        BRK_IF_ERROR(
            cpl_propertylist_update_string(qc_list, CPL_DFS_PRO_CATG, ERIS_IFU_PRO_FLAT));
        //eris_ifu_get_badpix_qc_from_ima(hdrl_image_get_image(masterFlatHdrlImg_lo), qc_list, "FLAT");
        BRK_IF_ERROR(
            eris_ifu_save_deq_image(frameset, NULL, parlist, frameset, NULL,
                                    REC_NAME_FLAT,
                                    qc_list, NULL,
                                    ERIS_IFU_PRO_FLAT_FN,
                                    masterFlatImage, errorImage, rmse,
                                    qualityImage, flag16bit, UNITLESS));

        /* save bpm image */
        BRK_IF_NULL(
            bp_mask_flat = hdrl_image_get_mask_const(masterFlatHdrlImg_lo));
        BRK_IF_NULL(
            bp_img_flat = cpl_image_new_from_mask(bp_mask_flat));
        BRK_IF_ERROR(
            eris_ifu_save_image(frameset, qc_list, parlist, REC_NAME_FLAT,
                                ERIS_IFU_PRO_FLAT_BPM,
                                ERIS_IFU_PRO_FLAT_BPM_FN,
                                CPL_TYPE_USHORT,
                                bp_img_flat));
    }
    CATCH
    {
        CATCH_MSGS();
    }

    eris_ifu_free_propertylist(&qc_list);
    eris_ifu_free_hdrl_image(&masterFlatHdrlImg_lo);
    eris_ifu_free_image(&qualityImage);
    eris_ifu_free_image(&bp_img_flat);

    if (cpl_memory_is_empty() == 0) {
        cpl_memory_dump();
    }

    return (int) cpl_error_get_code();
}
