#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>

#include <cpl.h>

#include "eris_ifu_debug.h"
#include "eris_ifu_error.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_ifu_debug     IFU Debugging and Diagnostic Functions
 *
 * This module provides debugging and diagnostic functions for printing and
 * plotting various CPL data structures. These functions are primarily intended
 * for development and troubleshooting purposes.
 *
 * @par Synopsis:
 * @code
 *   #include "eris_ifu_debug.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief    Print all properties in a propertylist for debugging
 * @param header   The propertylist to print
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * All properties contained in the propertylist are printed as pairs of keyword
 * and data for debugging purposes. The output is only generated when the message
 * level is set to debug.
 *
 * Possible cpl_error_code set in this function:
 * @li CPL_ERROR_NULL_INPUT if header is NULL
 *
 * @note This function is for debugging purposes only and may generate large
 *       amounts of output for property lists with many entries.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_debug_header(const cpl_propertylist *header)
{
    int                 i       = 0;
    const cpl_property  *p      = NULL;
    cpl_type            t;
    cpl_error_code      ret_error   = CPL_ERROR_NONE;

    TRY
    {
        cpl_msg_debug("", "===== START HEADER =====");
        if (header == NULL) {
            cpl_msg_warning("", "Empty header!");
        } else {
            for (i = 0; i < cpl_propertylist_get_size(header); i++) {
                BRK_IF_NULL(
                    p = cpl_propertylist_get_const(header, i));

                t = cpl_property_get_type (p);
                CHECK_ERROR_STATE();
                switch (t) {
                    case CPL_TYPE_BOOL:
                        cpl_msg_debug("", "%s: %d", cpl_property_get_name(p),
                                        cpl_property_get_bool(p));
                        break;
                    case CPL_TYPE_INT:
                        cpl_msg_debug("", "%s: %d", cpl_property_get_name(p),
                                        cpl_property_get_int(p));
                        break;
                    case CPL_TYPE_DOUBLE:
                        cpl_msg_debug("", "%s: %12.16g", cpl_property_get_name(p),
                                        cpl_property_get_double(p));
                        break;
                    case CPL_TYPE_FLOAT:
                        cpl_msg_debug("", "%s: %12.16f", cpl_property_get_name(p),
                                        cpl_property_get_float(p));
                        break;
                    case CPL_TYPE_STRING:
                        cpl_msg_debug("", "%s: %s", cpl_property_get_name(p),
                                        cpl_property_get_string(p));
                        break;
                    default:
                        break;
                }
            }
        }
        cpl_msg_debug("", "====== END HEADER ======");
        CHECK_ERROR_STATE();
    }
    CATCH
    {
        CATCH_MSG();
        ret_error = cpl_error_get_code();
    }

    return ret_error;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Print all frames in a frameset for debugging
 * @param frameset   The frameset to print
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * All frames contained in the frameset are printed for debugging purposes,
 * including their filenames, tags, types, groups, and levels.
 *
 * Possible cpl_error_code set in this function:
 * @li CPL_ERROR_NULL_INPUT if frameset is NULL
 *
 * @note This function calls eris_ifu_debug_frame() for each frame in the set.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_debug_frameset(const cpl_frameset *frameset)
{
    const cpl_frame *frame      = NULL;   /* must not be deleted at the end */
    cpl_error_code  ret_error   = CPL_ERROR_NONE;

    TRY
    {
        cpl_msg_debug("", "====== START FRAMESET ======");
        if (frameset == NULL) {
            cpl_msg_warning("", "Empty frameset!");
        } else {
            frame = cpl_frameset_find_const(frameset, NULL);

            if (GET_NEW_ERROR(void) != CPL_ERROR_NONE) {
                RECOVER();

                cpl_msg_debug("", "====== END FRAMESET ======");

                return CPL_ERROR_NONE;
            }

            while (frame != NULL) {
                eris_ifu_debug_frame(frame);
                BRK_IF_NULL(
                    frame = cpl_frameset_find_const(frameset, NULL));
            }
        }
        cpl_msg_debug("", "====== END FRAMESET ======");
        CHECK_ERROR_STATE();
    }
    CATCH
    {
        CATCH_MSG();
        ret_error = cpl_error_get_code();
    }

    return ret_error;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Print all parameters in a parameterlist for debugging
 * @param plist   The parameterlist to print
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * All parameters contained in the parameterlist are printed for debugging
 * purposes, showing both current values and default values in the format
 * "Parameter name: value [default]".
 *
 * Possible cpl_error_code set in this function:
 * @li CPL_ERROR_NULL_INPUT if plist is NULL
 * @li CPL_ERROR_INVALID_TYPE if a parameter has an unsupported type
 *
 * @note Output is sent to stdout, not through the CPL messaging system.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_debug_parameterlist(const cpl_parameterlist *plist)
{
    cpl_error_code  ret_error   = CPL_ERROR_NONE;
    const cpl_parameter *par = NULL;
    const char* name = NULL;
    int intVal;
    int intDef;
    double dblVal;
    double dblDef;
    const char *strVal;
    const char *strDef;

    TRY
    {
        cpl_msg_debug("", "====== START PARAMETERLIST ======");
        if (plist == NULL) {
            cpl_msg_warning("", "Empty parameterlist!");
        } else {
            printf("Number of elements in parameter list: %lld\n",
                cpl_parameterlist_get_size(plist));
            par = cpl_parameterlist_get_first_const(plist);
            while (par != NULL) {
                name = cpl_parameter_get_name(par);
                switch(cpl_parameter_get_type(par)) {
                    case CPL_TYPE_BOOL:
                        intVal = cpl_parameter_get_bool(par);
                        intDef = cpl_parameter_get_default_bool(par);
                        printf("Parameter %s: %d [%d]\n", name, intVal, intDef);
                        break;
                    case CPL_TYPE_INT:
                        intVal = cpl_parameter_get_int(par);
                        intDef = cpl_parameter_get_default_int(par);
                        printf("Parameter %s: %d [%d]\n", name, intVal, intDef);
                        break;
                    case CPL_TYPE_DOUBLE:
                        dblVal = cpl_parameter_get_double(par);
                        dblDef = cpl_parameter_get_default_double(par);
                        printf("Parameter %s: %f [%f]\n", name, dblVal, dblDef);
                        break;
                    case CPL_TYPE_STRING:
                        strVal = cpl_parameter_get_string(par);
                        strDef = cpl_parameter_get_default_string(par);
                        printf("Parameter %s: %s [%s]\n", name, strVal, strDef);
                        break;
                    default:
                        BRK_WITH_ERROR(CPL_ERROR_INVALID_TYPE);
                        break;
                }
                par = cpl_parameterlist_get_next_const(plist);
            }
        }
        cpl_msg_debug("", "====== END PARAMETERLIST ======");
        CHECK_ERROR_STATE();
    }
    CATCH
    {
        CATCH_MSG();
        ret_error = cpl_error_get_code();
    }

    return ret_error;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Print all properties in a propertylist for debugging
 * @param plist   The propertylist to print
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * All properties contained in the propertylist are printed for debugging
 * purposes using cpl_propertylist_dump().
 *
 * Possible cpl_error_code set in this function:
 * @li CPL_ERROR_NULL_INPUT if plist is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_debug_propertylist(const cpl_propertylist *plist)
{
    cpl_error_code  ret_error   = CPL_ERROR_NONE;

    TRY
    {
        cpl_msg_debug("", "====== START PROPERTYLIST ======");
        if (plist == NULL) {
            cpl_msg_warning("", "Empty propertylist!");
        } else {

            cpl_propertylist_dump(plist, NULL);
        }
        cpl_msg_debug("", "====== END PROPERTYLIST ======");
        CHECK_ERROR_STATE();
    }
    CATCH
    {
        CATCH_MSG();
        ret_error = cpl_error_get_code();
    }

    return ret_error;

}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Print frame information for debugging
 * @param frame   The frame to print
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * The CPL_FRAME_TYPE, CPL_FRAME_LEVEL, and CPL_FRAME_GROUP of the frame are
 * printed for debugging purposes, along with the filename and tag.
 *
 * Possible cpl_error_code set in this function:
 * @li CPL_ERROR_NULL_INPUT if frame is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_debug_frame(const cpl_frame *frame)
{
    const char      *tmp        = NULL;
    int             i           = 0;
    cpl_error_code  ret_error   = CPL_ERROR_NONE;

    TRY
    {
        cpl_msg_debug("", "     ====== START FRAME ======");
        if (frame == NULL) {
            cpl_msg_warning("", "Empty frame!");
        } else {
            tmp = cpl_frame_get_filename(frame);

            if (GET_NEW_ERROR(void) != CPL_ERROR_NONE) {

                RECOVER();
                cpl_msg_debug("", "     ====== END FRAME ======");
                return CPL_ERROR_NONE;
            }

            cpl_msg_debug("", "filename: %s", tmp);
            cpl_msg_debug("", "tag:      %s", cpl_frame_get_tag(frame));

            i = cpl_frame_get_type(frame);
            CHECK_ERROR_STATE();
            switch(i) {
                case CPL_FRAME_TYPE_NONE:
                    cpl_msg_debug("", "type:     CPL_FRAME_TYPE_NONE (%d)", i);
                    break;
                case CPL_FRAME_TYPE_IMAGE:
                    cpl_msg_debug("", "type:     CPL_FRAME_TYPE_IMAGE (%d)", i);
                    break;
                case CPL_FRAME_TYPE_MATRIX:
                    cpl_msg_debug("", "type:     CPL_FRAME_TYPE_MATRIX (%d)", i);
                    break;
                case CPL_FRAME_TYPE_TABLE:
                    cpl_msg_debug("", "type:     CPL_FRAME_TYPE_TABLE (%d)", i);
                    break;
                case CPL_FRAME_TYPE_PAF:
                    cpl_msg_debug("", "type:     CPL_FRAME_TYPE_PAF (%d)", i);
                    break;
                case CPL_FRAME_TYPE_ANY:
                    cpl_msg_debug("", "type:     CPL_FRAME_TYPE_ANY (%d)", i);
                    break;
                default:
                    cpl_msg_debug("", "type:     other ERROR (%d)", i);
            }

            i = cpl_frame_get_group(frame);
            CHECK_ERROR_STATE();
            switch(i) {
                case CPL_FRAME_GROUP_NONE:
                    cpl_msg_debug("", "group:    CPL_FRAME_GROUP_NONE (%d)", i);
                    break;
                case CPL_FRAME_GROUP_RAW:
                    cpl_msg_debug("", "group:    CPL_FRAME_GROUP_RAW (%d)", i);
                    break;
                case CPL_FRAME_GROUP_CALIB:
                    cpl_msg_debug("", "group:    CPL_FRAME_GROUP_CALIB (%d)", i);
                    break;
                case CPL_FRAME_GROUP_PRODUCT:
                    cpl_msg_debug("", "group:    CPL_FRAME_GROUP_PRODUCT (%d)", i);
                    break;
                default:
                    cpl_msg_debug("", "group:    other ERROR (%d)", i);
            }

            i = cpl_frame_get_level(frame);
            CHECK_ERROR_STATE();
            switch(i) {
                case CPL_FRAME_GROUP_NONE:
                    cpl_msg_debug("", "level:    CPL_FRAME_LEVEL_NONE (%d)", i);
                    break;
                case CPL_FRAME_GROUP_RAW:
                    cpl_msg_debug("", "level:    CPL_FRAME_LEVEL_TEMPORARY (%d)", i);
                    break;
                case CPL_FRAME_GROUP_CALIB:
                    cpl_msg_debug("", "level:    CPL_FRAME_LEVEL_INTERMEDIATE (%d)", i);
                    break;
                case CPL_FRAME_GROUP_PRODUCT:
                    cpl_msg_debug("", "level:    CPL_FRAME_LEVEL_FINAL (%d)", i);
                    break;
                default:
                    cpl_msg_debug("", "level:    other ERROR (%d)", i);
            }
        }
        cpl_msg_debug("", "     ====== END FRAME ======");
        CHECK_ERROR_STATE();
    }
    CATCH
    {
        CATCH_MSG();
        ret_error = cpl_error_get_code();
    }

    return ret_error;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Print all images in an imagelist for debugging
 * @param imglist   The imagelist to print
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * All images contained in the imagelist are printed for debugging purposes
 * by calling eris_ifu_debug_image() for each image.
 *
 * Possible cpl_error_code set in this function:
 * @li CPL_ERROR_NULL_INPUT if imglist is NULL
 *
 * @note This function may generate very large amounts of output for large
 *       image lists or large images. Use with caution.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_debug_cube(const cpl_imagelist *imglist)
{
    int             i           = 0;
    cpl_error_code  ret_error   = CPL_ERROR_NONE;

    TRY
    {
        cpl_msg_debug("", "====== START IMAGELIST ======");
        if (imglist == NULL) {
            cpl_msg_warning("", "Empty cube!");
        } else {
            for (i = 0; i < cpl_imagelist_get_size(imglist); i++) {
                BRK_IF_ERROR(
                    eris_ifu_debug_image(cpl_imagelist_get_const(imglist, i)));
            }
        }
        cpl_msg_debug("", "====== END IMAGELIST ======");
        CHECK_ERROR_STATE();
    }
    CATCH
    {
        CATCH_MSG();
        ret_error = cpl_error_get_code();
    }

    return ret_error;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Print all pixel values in an image for debugging
 * @param img   The image to print
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * All pixel values contained in the image are printed for debugging purposes.
 * Values are printed row by row.
 *
 * Possible cpl_error_code set in this function:
 * @li CPL_ERROR_NULL_INPUT if img is NULL
 *
 * @note This function may generate extremely large amounts of output for
 *       large images. Use only with small test images.
 * @note Uses a fixed buffer size which may overflow for very wide images.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_debug_image(const cpl_image *img)
{
    int             x           = 0,
                    y           = 0;
    int             gaga        = 0;
    cpl_error_code  ret_error   = CPL_ERROR_NONE;
    char            line[200000],
                    tmp[2048];

    TRY
    {
        cpl_msg_debug("", "     ====== START IMAGE ======");
        if (img == NULL) {
            cpl_msg_warning("", "Empty image!");
        } else {
            strcpy(line, " ");
            for (y = 1; y <= cpl_image_get_size_y(img); y++) {
                for (x = 1; x <= cpl_image_get_size_x(img); x++) {
                    sprintf(tmp, "%f   ", cpl_image_get(img, x, y, &gaga));
                    strcat(line, tmp);
                    CHECK_ERROR_STATE();
                }
                strcat(line, "");
                cpl_msg_debug("", "%s", line);
            }
        }
        cpl_msg_debug("", "     ====== END IMAGE ======");
        CHECK_ERROR_STATE();
    }
    CATCH
    {
        CATCH_MSG();
        ret_error = cpl_error_get_code();
    }

    return ret_error;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Print all values in a vector for debugging
 * @param vec   The vector to print
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * All values contained in the vector are printed for debugging purposes,
 * one value per line.
 *
 * Possible cpl_error_code set in this function:
 * @li CPL_ERROR_NULL_INPUT if vec is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_debug_vector(const cpl_vector *vec)
{
    int             x           = 0;
    cpl_size        size        = 0;
    cpl_error_code  ret_error   = CPL_ERROR_NONE;
    const double    *pvec       = NULL;

    TRY
    {
        cpl_msg_debug("", "     ====== START VECTOR ======");
        if (vec == NULL) {
            cpl_msg_warning("", "Empty vector!");
        } else {
            BRK_IF_NULL(
                pvec = cpl_vector_get_data_const(vec));

            size = cpl_vector_get_size(vec);
            for (x = 0; x < size; x++) {
                cpl_msg_debug("", "%12.16g   ", pvec[x]);
            }
        }
        cpl_msg_debug("", "     ====== END VECTOR ======");
        CHECK_ERROR_STATE();
    }
    CATCH
    {
        CATCH_MSG();
        ret_error = cpl_error_get_code();
    }

    return ret_error;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief    Print all values in an array for debugging
 * @param arr   The array to print
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * All values contained in the array are printed for debugging purposes.
 * Supports arrays of type CPL_TYPE_INT and CPL_TYPE_DOUBLE.
 *
 * Possible cpl_error_code set in this function:
 * @li CPL_ERROR_NULL_INPUT if arr is NULL
 *
 * @note Unsupported array types will generate a warning message.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_debug_array(const cpl_array *arr)
{
    int             x           = 0;
    cpl_size        size        = 0;
    const int       *pint       = NULL;
    const double    *pdouble    = NULL;
    cpl_error_code  ret_error   = CPL_ERROR_NONE;
    cpl_type        type;

    TRY
    {
        if (arr != NULL) {
            type = cpl_array_get_type(arr);
            size = cpl_array_get_size(arr);

            switch (type) {
                case CPL_TYPE_INT:
                    pint = cpl_array_get_data_int_const(arr);
                    cpl_msg_debug("", "     ====== START ARRAY ======");
                    if (arr== NULL) {
                        cpl_msg_warning("", "Empty array!");
                    } else {
                        for (x = 0; x < size; x++) {
                            cpl_msg_debug("", "%i", pint[x]);
                        }
                    }
                    cpl_msg_debug("", "     ====== END ARRAY ======");
                    break;
                case CPL_TYPE_DOUBLE:
                    pdouble = cpl_array_get_data_double_const(arr);
                    cpl_msg_debug("", "     ====== START ARRAY ======");
                    if (arr== NULL) {
                        cpl_msg_warning("", "Empty array!");
                    } else {
                        for (x = 0; x < size; x++) {
                            cpl_msg_debug("", "%12.16g", pdouble[x]);
                        }
                    }
                    cpl_msg_debug("", "     ====== END ARRAY ======");
                    break;
                default:
                    cpl_msg_debug("", ">>> cpl_type (%d) not supported!", type);
                    break;
            }
            CHECK_ERROR_STATE();
        } else {
            cpl_msg_debug("", "     ====== START ARRAY ======");
            cpl_msg_debug("", "Empty array!");
            cpl_msg_debug("", "     ====== END ARRAY ======");
        }
    }
    CATCH
    {
        CATCH_MSG();
        ret_error = cpl_error_get_code();
    }

    return ret_error;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Print all values in a table for debugging
 * @param table   The table to print
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * All values contained in the table are printed for debugging purposes using
 * cpl_table_dump().
 *
 * Possible cpl_error_code set in this function:
 * @li CPL_ERROR_NULL_INPUT if table is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_debug_table(const cpl_table *table)
{
    cpl_error_code  ret_error   = CPL_ERROR_NONE;

    TRY
    {
        cpl_msg_debug("", "     ====== START TABLE ======");
        if (table == NULL) {
            cpl_msg_warning("", "Empty table pointer!");
        } else {
            cpl_table_dump(table, 0, cpl_table_get_nrow(table), NULL);
        }
        cpl_msg_debug("", "     ====== END TABLE ======");
        CHECK_ERROR_STATE();
    }
    CATCH
    {
        CATCH_MSG();
        ret_error = cpl_error_get_code();
    }

    return ret_error;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Plot a vector for debugging
 * @param pre      Optional string with pre-plot commands (gnuplot syntax)
 * @param opt      Optional string with plotting options (gnuplot syntax)
 * @param vector   The vector to plot
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Plots a vector only if the message level is CPL_MSG_DEBUG. Uses gnuplot
 * for visualization. On Darwin/macOS systems, automatically sets the terminal
 * to x11.
 *
 * Possible cpl_error_code set in this function:
 * - Any error code possibly set by cpl_plot_vector()
 *
 * @note Plotting only occurs when CPL_MSG_DEBUG level is active and vector is not NULL.
 * @note Requires gnuplot to be available on the system.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_plot_vector(const char *pre, const char *opt,
                               const cpl_vector *vector)
{
    cpl_error_code  ret_error   = CPL_ERROR_NONE;

    char            *ostype     = NULL,
                    pre_final[1024];

    TRY
    {
        if ((vector != NULL) &&
            (cpl_msg_get_level() == CPL_MSG_DEBUG))
        {
            strcpy(pre_final, "");
            if (pre != NULL) {
                strcat(pre_final, pre);
            }
            ostype = getenv("OSTYPE");
            if(strcmp(ostype, "darwin") == 0) {
                strcat(pre_final, "set term x11;");
            }

            BRK_IF_ERROR(
                cpl_plot_vector(pre_final, opt, NULL, vector));
        }
    }
    CATCH
    {
        CATCH_MSG();
        ret_error = cpl_error_get_code();
    }

    return ret_error;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Plot vector y against vector x for debugging
 * @param pre   Optional string with pre-plot commands (gnuplot syntax)
 * @param opt   Optional string with plotting options (gnuplot syntax)
 * @param x     The x-axis vector to plot
 * @param y     The y-axis vector to plot
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Plots vector y against vector x as an x-y plot only if the message level
 * is CPL_MSG_DEBUG. Uses gnuplot for visualization.
 *
 * Possible cpl_error_code set in this function:
 * - Any error code possibly set by cpl_plot_bivector()
 *
 * @note Plotting only occurs when CPL_MSG_DEBUG level is active and both vectors are not NULL.
 * @note Requires gnuplot to be available on the system.
 * @note The vectors are temporarily wrapped in a bivector for plotting but are not modified.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_plot_vectors_xy(const char *pre, const char *opt,
                                   const cpl_vector *x, const cpl_vector *y)
{
    cpl_error_code  ret_error   = CPL_ERROR_NONE;

    cpl_bivector    *bi         = NULL;

    char            *ostype     = NULL,
                    pre_final[1024];

    TRY
    {
        if ((x != NULL) && (y != NULL) &&
            (cpl_msg_get_level() == CPL_MSG_DEBUG))
        {
            strcpy(pre_final, "");
            if (pre != NULL) {
                strcat(pre_final, pre);
            }
            ostype = getenv("OSTYPE");
            if(strcmp(ostype, "darwin") == 0) {
                strcat(pre_final, "set term x11;");
            }
// The function cpl_bivector_wrap_vectors will discard the const qualifier
// However as it is know that cpl_plot_bivector will not modifiy its
// bivector argument it is save to turn off the compiler warning

#ifdef __GNUC__
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wcast-qual"
#endif
            BRK_IF_NULL(
                bi = cpl_bivector_wrap_vectors((cpl_vector*)x,
                                               (cpl_vector*)y));
#ifdef __GNUC__
#pragma GCC diagnostic pop
#endif
            BRK_IF_ERROR(
                cpl_plot_bivector(pre_final, opt, NULL, bi));

            cpl_bivector_unwrap_vectors(bi);
        }
    }
    CATCH
    {
        CATCH_MSG();
        ret_error = cpl_error_get_code();
    }

    return ret_error;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Plot two y-vectors against an x-vector for debugging
 * @param pre   Optional string with pre-plot commands (gnuplot syntax)
 * @param opt   Optional array of strings with plotting options for each plot
 * @param x     The x-axis vector
 * @param y1    The first y-axis vector to plot
 * @param y2    The second y-axis vector to plot
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Plots two y-vectors (y1 and y2) against a common x-vector in the same plot,
 * only if the message level is CPL_MSG_DEBUG. Uses gnuplot for visualization.
 *
 * Possible cpl_error_code set in this function:
 * - Any error code possibly set by cpl_plot_bivectors()
 *
 * @note Plotting only occurs when CPL_MSG_DEBUG level is active and all vectors are not NULL.
 * @note Requires gnuplot to be available on the system.
 * @note The vectors are temporarily wrapped in bivectors for plotting but are not modified.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code     eris_ifu_plot_vectors2(const char *pre, const char **opt,
                                     const cpl_vector *x,
                                     const cpl_vector *y1,
                                     const cpl_vector *y2)
{
    cpl_error_code  ret_error   = CPL_ERROR_NONE;

    int             nr_plots            = 2;
    cpl_bivector    *plots[nr_plots];
    char            *ostype             = NULL,
                    pre_final[1024];

    TRY
    {
        if ((x != NULL) && (y1 != NULL) && (y2 != NULL) &&
            (cpl_msg_get_level() == CPL_MSG_DEBUG))
        {
            strcpy(pre_final, "");
            if (pre != NULL) {
                strcat(pre_final, pre);
            }
            ostype = getenv("OSTYPE");
            if(strcmp(ostype, "darwin") == 0) {
                strcat(pre_final, "set term x11;");
            }

            // The function cpl_bivector_wrap_vectors will discard the const qualifier
            // However as it is know that cpl_plot_bivector will not modifiy its
            // bivector argument it is save to turn off the compiler warning

#ifdef __GNUC__
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wcast-qual"
#endif
            BRK_IF_NULL(
                plots[0] = cpl_bivector_wrap_vectors((cpl_vector*)x,
                                                     (cpl_vector*)y1));
            BRK_IF_NULL(
                plots[1] = cpl_bivector_wrap_vectors((cpl_vector*)x,
                                                     (cpl_vector*)y2));

            BRK_IF_ERROR(
                cpl_plot_bivectors(pre_final,
                               (const char**)opt,
                               "",
                               (const cpl_bivector **) plots,
                               nr_plots));
#ifdef __GNUC__
#pragma GCC diagnostic pop
#endif

            int k = 0;
            for (k = 0; k < nr_plots; k++) {
                cpl_bivector_unwrap_vectors(plots[k]);
            }
            CHECK_ERROR_STATE();
        }
    }
    CATCH
    {
        CATCH_MSG();
        ret_error = cpl_error_get_code();
    }

    return ret_error;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Plot an image for debugging
 * @param pre   Optional string with pre-plot commands (gnuplot syntax)
 * @param opt   Optional string with plotting options (gnuplot syntax)
 * @param img   The image to plot
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Plots an image only if the message level is CPL_MSG_DEBUG. Uses gnuplot
 * for visualization with appropriate 2D plotting settings.
 *
 * Possible cpl_error_code set in this function:
 * - Any error code possibly set by cpl_plot_image()
 *
 * @note Plotting only occurs when CPL_MSG_DEBUG level is active and img is not NULL.
 * @note Requires gnuplot to be available on the system.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_plot_image(const char *pre, const char *opt,
                              const cpl_image *img)
{
    cpl_error_code  ret_error   = CPL_ERROR_NONE;

    char            *ostype     = NULL,
                    pre_final[1024];

    TRY
    {
        if ((img != NULL) &&
            (cpl_msg_get_level() == CPL_MSG_DEBUG))
        {
            strcpy(pre_final, "");
            if (pre != NULL) {
                strcat(pre_final, pre);
            }
            ostype = getenv("OSTYPE");
            if(strcmp(ostype, "darwin") == 0) {
                strcat(pre_final, "set term x11;");
            }

            BRK_IF_ERROR(
                cpl_plot_image(pre_final, opt, NULL, img));
        }
    }
    CATCH
    {
        CATCH_MSG();
        ret_error = cpl_error_get_code();
    }

    return ret_error;
}

/**@}*/
