/* 
 * This file is part of the KMOS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 *                              Includes
 *----------------------------------------------------------------------------*/

#include <math.h>
#include <stdio.h>

#include <cpl.h>

#include "eris_ifu_lambda_corr.h"
#include "eris_ifu_jitter_static.h"
#include "eris_ifu_error.h"

FILE *IDL = NULL;
/*----------------------------------------------------------------------------*/
/**
   @defgroup eris_ifu_lambda_corr     Lambda Correction for IFU Spectra
   
   This module provides functions for wavelength (lambda) correction of IFU spectra
   using OH sky line emission features. The correction compensates for flexure-induced
   wavelength shifts by cross-correlating observed OH lines with reference positions.
   
   The typical workflow is:
   1. Extract an integrated spectrum from the IFU cube
   2. Detect OH emission line peaks in the extracted spectrum
   3. Cross-correlate observed peaks with reference OH line positions
   4. Fit a polynomial correction that maps observed wavelengths to true wavelengths
   5. Apply the correction to the entire wavelength solution
   
   The correction polynomial can be of varying degree (typically 0-3), with higher
   degrees allowing for non-linear wavelength distortions but requiring more detected lines.
   
   @{
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
   @brief    Create wavelength correction polynomial from IFU cube using OH lines
   
   This is the main entry point for lambda correction. It extracts an integrated
   spectrum from the input cube, detects OH emission line peaks, and cross-correlates
   them with reference line positions to derive a wavelength correction polynomial.
   
   @param cube         Input data cube (CPL imagelist format), set to NULL if hdrlCube is provided
   @param hdrlCube     Input data cube (HDRL imagelist format), set to NULL if cube is provided
   @param header       FITS header containing WCS and instrument information
   @param peaks        Vector containing reference OH line wavelengths (in microns)
   @param pfit_order   Order of polynomial fit (0=constant shift, 1=linear, 2=quadratic, etc.)
   
   @return Polynomial representing the wavelength correction, or NULL on error
   
   @note Exactly one of cube or hdrlCube must be non-NULL
   @note The returned polynomial must be deallocated with cpl_polynomial_delete()
   @note The correction polynomial maps observed wavelengths to corrected wavelengths:
         lambda_corrected = lambda_observed + polynomial(lambda_observed)
   @note A polynomial order of 0 is recommended for K-band to avoid extrapolation errors
   @note Higher orders (1-3) may improve accuracy in H and J bands with many detected lines
*/
/*----------------------------------------------------------------------------*/
cpl_polynomial *eris_ifu_lcorr_get(cpl_imagelist *cube, hdrl_imagelist *hdrlCube,
                                   const cpl_propertylist *header,
                                   cpl_vector *peaks,
								   const int pfit_order)
{
    cpl_polynomial      *lcorr_coeffs   = NULL;
    cpl_imagelist       *tmpCube        = NULL;
    cpl_image           *tmpImg         = NULL;
    cpl_bivector        *obj_spectrum   = NULL,
                        *ref_spectrum   = NULL;
    int                 ic              = 0;
    const int           format_width    = 14,
                        max_coeffs      = 6;
    char                *coeff_string   = NULL,
                        coeff_dump[format_width * max_coeffs + 1];
    cpl_size            pows[1];
    ifsBand             bandId          = UNDEFINED_BAND;
    //const char          *ref_filename   = NULL;
    //TODO: why we need filter_id, and later that is set, but never used?
//    const char *filter_id;

    TRY
    {
        ASSURE(cube != NULL || hdrlCube != NULL,
            CPL_ERROR_ILLEGAL_INPUT, "One of cube or hdrlCube must be not NULL");
        ASSURE(!(cube != NULL && hdrlCube != NULL),
            CPL_ERROR_ILLEGAL_INPUT, "One of cube or hdrlCube must be NULL");
        ASSURE((peaks != NULL),
                CPL_ERROR_NULL_INPUT, "Not all input data is provided!");

        bandId = eris_ifu_get_band(header);
//        switch (bandId) {
//            case J_LOW:
//            case J_SHORT: case J_MIDDLE: case J_LONG:
//            case J_SPIFFI:
//                filter_id = "J";
//                break;
//            case H_LOW:
//            case H_SHORT: case H_MIDDLE: case H_LONG:
//            case H_SPIFFI:
//                filter_id = "H";
//                break;
//            case K_LOW:
//            case K_SHORT: case K_MIDDLE: case K_LONG:
//            case K_SPIFFI:
//                filter_id = "K";
//                break;
//            case HK_SPIFFI:
//                filter_id = "HK";
//                break;
//            default:
//                filter_id = "X";
//                break;
//        }

        if (cube != NULL) {
            tmpCube = cube;
        } else {
            tmpCube = cpl_imagelist_new();
            for (cpl_size sx = 0; sx < hdrl_imagelist_get_size(hdrlCube); sx++) {
                tmpImg = hdrl_image_get_image(hdrl_imagelist_get(hdrlCube, sx));
                cpl_imagelist_set(tmpCube, cpl_image_duplicate(tmpImg), sx);
            }
            CHECK_ERROR_STATE();
        }

        BRK_IF_ERROR(
            eris_ifu_mask_nans_in_cube(tmpCube));


        BRK_IF_NULL(
            obj_spectrum = eris_ifu_lcorr_extract_spectrum(tmpCube, header, 0.8, NULL));
        //clean memory from temporary object. TODO: generate seg fault. Why??
        /*
        if (cube == NULL) {
            for (cpl_size sx = 0; sx < cpl_imagelist_get_size(tmpCube); sx++) {
        	cpl_imagelist_unset(tmpCube,sx);
            }
        }
        */
        cpl_imagelist_delete(tmpCube);
        BRK_IF_NULL(
            lcorr_coeffs = eris_ifu_lcorr_crosscorrelate_spectra(obj_spectrum,
                                                            peaks, bandId, pfit_order));

        // debug stuff
        coeff_dump[0] = 0;
        for (ic = 0; ic <= cpl_polynomial_get_degree(lcorr_coeffs) && ic < max_coeffs; ic++) {
            pows[0] = ic;
            coeff_string = cpl_sprintf(" %*g,", format_width-2, cpl_polynomial_get_coeff(lcorr_coeffs,pows));
            strncat(coeff_dump, coeff_string, format_width);
            cpl_free(coeff_string); coeff_string = NULL;
        }
        cpl_msg_debug(cpl_func,"Lambda correction coeffs %s", coeff_dump);
    }
    CATCH
    {
        cpl_polynomial_delete(lcorr_coeffs); lcorr_coeffs = NULL;
    }

    cpl_bivector_delete(ref_spectrum); ref_spectrum = NULL;
    cpl_bivector_delete(obj_spectrum); obj_spectrum = NULL;
    cpl_free(coeff_string); coeff_string = NULL;

    return lcorr_coeffs;
}

/*----------------------------------------------------------------------------*/
/**
   @brief    Extract integrated spectrum from IFU cube for brightest spaxels
   
   Creates an integrated spectrum by selecting the brightest spaxels in the cube
   (based on median flux) and summing their spectral contributions. This produces
   a high signal-to-noise spectrum suitable for OH line detection and wavelength
   calibration.
   
   @param cube      Input data cube (imagelist with spectral axis)
   @param header    FITS header containing WCS keywords (NAXIS3, CRVAL3, CDELT3, CRPIX3)
   @param min_frac  Fraction of brightest spaxels to include (e.g., 0.8 = brightest 80%)
   @param range     Optional wavelength ranges to consider for spaxel selection (NULL for all)
                    Format: [min1, max1, min2, max2, ...] in microns
   
   @return Bivector with X=wavelengths (microns) and Y=integrated flux, or NULL on error
   
   @note The returned bivector must be deallocated with cpl_bivector_delete()
   @note Spaxel selection is based on median flux across the specified wavelength range(s)
   @note The wavelength vector is computed from WCS keywords in the header
*/
/*----------------------------------------------------------------------------*/
cpl_bivector *eris_ifu_lcorr_extract_spectrum (const cpl_imagelist *cube,
        const cpl_propertylist *header,
        const double min_frac,
        const cpl_vector *range) {

    cpl_bivector *result = NULL;
    cpl_vector *spectrum = NULL,
               *lambda   = NULL;
    cpl_image *obj_mask = NULL;
    TRY
    {
        ASSURE(cube != NULL && header != NULL,
                CPL_ERROR_NULL_INPUT,
                "Not all input data is provided!");

        BRK_IF_NULL(
                lambda = eris_ifu_lcorr_create_lambda_vector(header));

        if (range != NULL) {
            BRK_IF_NULL(
                    obj_mask = eris_ifu_lcorr_create_object_mask(cube, min_frac, lambda, range));
        } else {
            BRK_IF_NULL(
                    obj_mask = eris_ifu_lcorr_create_object_mask(cube, min_frac, NULL, NULL));
        }

        BRK_IF_ERROR(
                eris_ifu_extract_spec(cube, NULL, obj_mask, &spectrum, NULL));

        BRK_IF_NULL(
                result = cpl_bivector_wrap_vectors(lambda, spectrum));
    }
    CATCH
    {
        CATCH_MSG();
    }

    if (obj_mask != NULL) {cpl_image_delete(obj_mask);}
    if (result == NULL) {
        if (lambda != NULL) {cpl_vector_delete(lambda);}
        if (spectrum != NULL) {cpl_vector_delete(spectrum);}
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
   @brief    Create spaxel selection mask for spectrum extraction
   
   For every spaxel, calculates median flux (optionally restricted to specific
   wavelength ranges) and creates a binary mask selecting the brightest fraction
   of spaxels. This mask is used to extract an integrated spectrum from the
   brightest regions of the field.
   
   @param cube      Input data cube (imagelist)
   @param min_frac  Fraction of brightest spaxels to select (0.0-1.0, e.g., 0.8 for top 80%)
   @param lambda    Wavelength vector corresponding to cube's spectral axis, or NULL
   @param range     Wavelength ranges for flux calculation (NULL for all wavelengths)
                    Format: [min1, max1, min2, max2, ...] pairs in microns
   
   @return Binary mask image (1.0=selected, 0.0=rejected), or NULL on error
   
   @note Either both lambda and range must be provided, or both must be NULL
   @note The range vector must have an even number of elements (lower/upper pairs)
   @note Each lambda entry corresponds to one plane in the cube's z-axis
   @note The returned image must be deallocated with cpl_image_delete()
   @note When range is specified, only wavelengths within the specified range(s) contribute
         to the median flux calculation for spaxel selection
*/
/*----------------------------------------------------------------------------*/
cpl_image *eris_ifu_lcorr_create_object_mask (const cpl_imagelist *cube,
        double min_frac,
        const cpl_vector *lambda,
        const cpl_vector *range) {

    cpl_image *mask      = NULL;
    cpl_imagelist *rcube = NULL;
    const cpl_image *slice = NULL;
    cpl_imagelist *icube = NULL;
    cpl_image *tmp_img   = NULL,
                    *m_img = NULL;
    cpl_vector *sum      = NULL,
               *m_vector = NULL,
               *sorted_sum = NULL;
    cpl_mask         *empty_mask     = NULL;

    int x, y, z,  r;
    int nx, ny, nz, nr, px, cnz= 0;
    int check4range = FALSE;
    const double  *lambda_data = NULL;
    double l_low, l_up = 0.0;
    double max_value = 0.0;
    int *cnts = NULL;
    double *m_data    = NULL;
    float  *mask_data = NULL;

    TRY
    {
        ASSURE(cube != NULL,
                CPL_ERROR_NULL_INPUT,
                "Not all input data is provided!");

        ASSURE(((lambda != NULL) && (range != NULL)) ||
                       ((lambda == NULL) && (range == NULL)) ,
                CPL_ERROR_NULL_INPUT,
                "Not all input data is provided!");
        if (range != NULL) {
            check4range = TRUE;
            ASSURE(cpl_vector_get_size(range)%2 == 0,
                    CPL_ERROR_ILLEGAL_INPUT,
                    "Range vector size must be a multiple of 2");

            nr = cpl_vector_get_size(range) / 2;
        }

        if (lambda != NULL) {
            ASSURE(cpl_vector_get_size(lambda) == cpl_imagelist_get_size(cube),
                    CPL_ERROR_ILLEGAL_INPUT,
                    "Size of lambda vector must be the number of images in the input cube");
            BRK_IF_NULL(
                    lambda_data = cpl_vector_get_data_const(lambda));
        }

        BRK_IF_NULL(
                slice = cpl_imagelist_get_const(cube, 0));
        nx = cpl_image_get_size_x(slice);
        ny = cpl_image_get_size_y(slice);
        nz = cpl_imagelist_get_size(cube);

        // if wavelength range are specified create a new imagelist holding only those lambdas
        // otherwise take original input cube
        if (check4range) {
            cnz = 0;
            BRK_IF_NULL(
                    rcube = cpl_imagelist_new());
            for (z=0; z<nz; z++) {
                int checkOK = FALSE;
                for (r=0; r<nr; r++) {
                    l_low = cpl_vector_get(range, r*2);
                    l_up  = cpl_vector_get(range, r*2+1);
                    if ((lambda_data[z] >= l_low) && (lambda_data[z] <= l_up)) {
                        checkOK = TRUE;
                        break;
                    }
                }
                if (checkOK) {
                    BRK_IF_NULL(
                            slice = cpl_imagelist_get_const(cube, z));
                    BRK_IF_ERROR(
                            cpl_imagelist_set(rcube, cpl_image_duplicate(slice), cnz));
                    cnz++;
                }
            }
            icube =  rcube;
        } else {
            icube = (cpl_imagelist*) cube;
        }
        BRK_IF_ERROR(
                eris_ifu_mask_nans_in_cube(icube));
        BRK_IF_NULL(
                tmp_img =  cpl_imagelist_collapse_median_create(icube));
        BRK_IF_NULL(
                m_img =  cpl_image_cast(tmp_img, CPL_TYPE_DOUBLE));
       BRK_IF_NULL(
                m_data = cpl_image_get_data_double(m_img));
        BRK_IF_NULL(
                m_vector = cpl_vector_wrap(nx * ny, m_data));
        BRK_IF_NULL(
                sorted_sum = cpl_vector_duplicate(m_vector));
        cpl_vector_sort(sorted_sum, CPL_SORT_ASCENDING);
        max_value = cpl_vector_get(sorted_sum, nx*ny*min_frac);

        mask = cpl_image_new(nx,ny,CPL_TYPE_FLOAT);
        BRK_IF_NULL(
                mask_data = cpl_image_get_data_float(mask));
        for (x=0; x<nx; x++) {
            for (y=0; y<ny; y++) {
                px = x+y*nx;
                if (m_data[px] <= max_value) {
                    mask_data[px] = 1.0;
                } else {
                    mask_data[px] = 0.0;
                }
            }
        }

    }
    CATCH
    {
        CATCH_MSG();
        if (mask != NULL) {cpl_image_delete(mask);}
        mask = NULL;
    }
    if (rcube != NULL) {cpl_imagelist_delete(rcube);}
    if (tmp_img != NULL) {cpl_image_delete(tmp_img);}
    if (m_img != NULL) {cpl_image_delete(m_img);}
    if (m_vector != NULL) {cpl_vector_unwrap(m_vector);}
    if (sum != NULL) {cpl_vector_delete(sum);}
    if (cnts != NULL) {cpl_free(cnts);}
    if (sorted_sum != NULL) {cpl_vector_delete(sorted_sum);}
    if (empty_mask != NULL) {cpl_mask_delete(empty_mask);}

    return mask;
}

/*----------------------------------------------------------------------------*/
/**
   @brief    Create wavelength vector from FITS header WCS keywords
   
   Constructs a wavelength vector using the WCS (World Coordinate System) properties
   of the spectral axis (axis 3) from a data cube's FITS header. Uses the standard
   FITS keywords NAXIS3, CRVAL3, CDELT3, and CRPIX3.
   
   @param header  Property list (FITS header) containing WCS keywords
   
   @return Vector of wavelengths in microns, or NULL on error
   
   @note The returned vector must be deallocated with cpl_vector_delete()
   @note Required FITS keywords: NAXIS3, CRVAL3, CDELT3, CRPIX3
   @note Wavelength for plane i is: lambda[i] = (i+1 - CRPIX3) * CDELT3 + CRVAL3
*/
/*----------------------------------------------------------------------------*/

cpl_vector *eris_ifu_lcorr_create_lambda_vector(const cpl_propertylist *header) {

    cpl_vector *lambda_vector = NULL;
    int naxis3, z = 0;
    double  crval3, cdelt3, crpix3 = 0.0;
    double *lambda = NULL;

    TRY {
        ASSURE(header != NULL,
                CPL_ERROR_NULL_INPUT,
                "Not all input data is provided!");
        ASSURE(cpl_propertylist_has(header,NAXIS3) &&
                cpl_propertylist_has(header,CRVAL3) &&
                cpl_propertylist_has(header,CDELT3) &&
                cpl_propertylist_has(header,CRPIX3),
                CPL_ERROR_ILLEGAL_INPUT,
                "missing WCS keywords in header");

        naxis3 = cpl_propertylist_get_int(header, NAXIS3);
        crval3 = cpl_propertylist_get_double(header, CRVAL3);
        cdelt3 = cpl_propertylist_get_double(header, CDELT3);
        crpix3 = cpl_propertylist_get_double(header, CRPIX3);

        BRK_IF_NULL(
                lambda_vector = cpl_vector_new(naxis3));
        BRK_IF_NULL(
                lambda = cpl_vector_get_data(lambda_vector));

        for (z=0; z<naxis3; z++) {
            lambda[z] = (z+1 - crpix3) * cdelt3 + crval3;
        }
     }
    CATCH
    {
        CATCH_MSG();
        if (lambda_vector != NULL) {
            cpl_vector_delete(lambda_vector);
            lambda_vector = NULL;
        }
    }

    return lambda_vector;
}

/*----------------------------------------------------------------------------*/
/**
   @brief    Read OH reference spectrum from FITS file
   
   Loads a reference OH sky emission spectrum from a FITS file and constructs
   a bivector with wavelength and flux values. The wavelength scale is derived
   from WCS keywords in the file's header.
   
   @param filename     Path to FITS file containing the reference spectrum
   @param ext_number   FITS extension number to read (0 for primary HDU)
   
   @return Bivector with X=wavelengths (microns) and Y=flux, or NULL on error
   
   @note The returned bivector must be deallocated with cpl_bivector_delete()
   @note The file must contain FITS keywords: NAXIS1, CRVAL1, CDELT1, CRPIX1
   @note CRPIX1 can be integer, float, or double type
   @note Wavelength for pixel i is: lambda[i] = CRVAL1 + (i + 1 - CRPIX1) * CDELT1
*/
/*----------------------------------------------------------------------------*/
cpl_bivector *eris_ifu_lcorr_read_OH_reference_spectrum(const char *filename,
        cpl_size ext_number) {

    cpl_bivector *result = NULL;
    cpl_vector *lambda  = NULL,
               *spec    = NULL;
    cpl_propertylist *header = NULL;
    double *lambda_data = NULL;
    double crpix, crval, cdelt = 0.0;
    int naxis1 = 0;

    TRY {
        ASSURE(filename != NULL,
                CPL_ERROR_NULL_INPUT,
                "Not all input data is provided!");

        cpl_msg_info(__func__,
                "Using file %s as OH reference spectrum for lambda correction",
                filename);

        spec = cpl_vector_load(filename, ext_number);
        if (cpl_error_get_code() != CPL_ERROR_NONE) {
            if (cpl_error_get_code() == CPL_ERROR_FILE_IO) {
                cpl_msg_error("", "File not found: %s", filename);
            } else {
                cpl_msg_error("", "Problem loading file '%s' (%s --> Code %d)",
                              filename, cpl_error_get_message(),
                              cpl_error_get_code());
            }
        }

        BRK_IF_NULL(
                header = cpl_propertylist_load(filename, ext_number));

        naxis1 = cpl_propertylist_get_int(header, NAXIS1);
        CHECK_ERROR_STATE();
        crval = cpl_propertylist_get_double(header, CRVAL1);
        CHECK_ERROR_STATE();
        cdelt = cpl_propertylist_get_double(header, CDELT1);
        CHECK_ERROR_STATE();
        switch (cpl_propertylist_get_type(header, CRPIX1)) {
        case CPL_TYPE_INT:
            crpix = cpl_propertylist_get_int(header, CRPIX1);
            break;
        case CPL_TYPE_DOUBLE:
            crpix = cpl_propertylist_get_double(header, CRPIX1);
            break;
        case CPL_TYPE_FLOAT:
            crpix = cpl_propertylist_get_float(header, CRPIX1);
            break;
        default:
            BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
                           "CRPIX1 is of wrong type!");
        }
        CHECK_ERROR_STATE();

        BRK_IF_NULL(
                lambda = cpl_vector_new(naxis1));
        BRK_IF_NULL(
                lambda_data = cpl_vector_get_data(lambda));
        int i = 0;
        for (i = 0; i < naxis1; i++) {
            lambda_data[i] = crval + (i + 1-crpix)*cdelt;
        }

        BRK_IF_NULL(
                result = cpl_bivector_wrap_vectors(lambda, spec));
    }
    CATCH
    {
        CATCH_MSG();
    }

    if (header != NULL) {cpl_propertylist_delete(header);}

    return result;

}

/*----------------------------------------------------------------------------*/
/**
   @brief    Detect emission line peak positions in a spectrum
   
   Identifies local maxima (peaks) in the input spectrum that exceed a minimum
   flux threshold. Peaks can be restricted to specified wavelength ranges.
   The algorithm finds peaks by detecting sign changes in the first derivative
   (positive to negative transition).
   
   @param spectrum   Input spectrum as bivector (X=wavelength, Y=flux)
   @param min_frac   Minimum flux threshold as fraction of maximum flux (0.0-1.0)
   @param range      Optional wavelength ranges to search (NULL for all wavelengths)
                     Format: [min1, max1, min2, max2, ...] pairs in microns
   
   @return Array of peak indices (positions in the spectrum vector), or NULL on error
   
   @note The returned array must be deallocated with cpl_array_delete()
   @note Only flux values above (max_flux * min_frac) are considered
   @note A peak is identified where derivative changes from positive to negative
   @note If range is specified, peaks outside all ranges are ignored
   @note The range vector must have an even number of elements
*/
/*----------------------------------------------------------------------------*/

cpl_array *eris_ifu_lcorr_get_peak_positions(const cpl_bivector *spectrum,
        double min_frac,
        cpl_vector *range) {

    cpl_array *positions = NULL;

    cpl_vector *copied_spectrum = NULL;

    int             spec_size       = 0,
                    peak_cnt        = 0,
                    nr              = 0,
                    i               = 0,
                    r               = 0;
    double          min             = 0.0,
                    l_low           = 0.0,
                    l_up            = 0.0;
    double          *spec_data      = NULL,
                    *diff_data      = NULL;
    const double    *lambda_data    = NULL;

    TRY {
        ASSURE(spectrum != NULL && cpl_bivector_get_y_const(spectrum) != NULL,
                CPL_ERROR_NULL_INPUT,
                "Not all input data is provided!");

       if (range != NULL) {
           ASSURE(cpl_bivector_get_x_const(spectrum) != NULL,
                   CPL_ERROR_NULL_INPUT,
                   "Not all input data is provided!");
            ASSURE(cpl_vector_get_size(range)%2 == 0,
                    CPL_ERROR_ILLEGAL_INPUT,
                    "Range vector size must be a multiple of 2");
            BRK_IF_NULL(
                    lambda_data = cpl_vector_get_data_const(cpl_bivector_get_x_const(spectrum)));
             nr = cpl_vector_get_size(range) / 2;
        }

        BRK_IF_NULL(
                copied_spectrum = cpl_vector_duplicate(cpl_bivector_get_y_const(spectrum)));

        BRK_IF_NULL(
                spec_data = cpl_vector_get_data(copied_spectrum));

        spec_size = cpl_vector_get_size(copied_spectrum);

        if (range != NULL) {
            for (i = 0; i < (spec_size-1); i++) {
                int checkOK = FALSE;
                for (r = 0; r < nr; r++) {
                    l_low = cpl_vector_get(range, r*2);
                    l_up  = cpl_vector_get(range, r*2+1);
                    if ((lambda_data[i] >= l_low) && (lambda_data[i] <= l_up)) {
                        checkOK = TRUE;
                        break;
                    }
                }
                if (! checkOK) {
                    spec_data[i] = 0.;
                }
            }
        }

        BRK_IF_NULL(
                diff_data = cpl_malloc((spec_size-1) * sizeof(double)));

        BRK_IF_NULL(
                positions = cpl_array_new(spec_size, CPL_TYPE_INT));

        min = cpl_vector_get_max(copied_spectrum) * min_frac;

        if (spec_data[0] < min) {
            spec_data[0] = 0.;
        }
        for (i = 0; i < (spec_size-1); i++) {
            if (spec_data[i+1] < min) {
                spec_data[i+1] = 0.;
            }
            diff_data[i] = spec_data[i+1] - spec_data[i];
        }

        peak_cnt = 0;
        for (i = 0; i < (spec_size-2); i++) {
            if ((diff_data[i] > 0) && (diff_data[i+1] < 0)) {
                BRK_IF_ERROR (
                        cpl_array_set_int(positions, peak_cnt, i+1));
                peak_cnt++;
            }
        }
        BRK_IF_ERROR (
                cpl_array_set_size(positions, peak_cnt));
    }
    CATCH
    {
        CATCH_MSG();
        if (positions != NULL) {cpl_array_delete(positions);}
        positions = NULL;
    }

    if (diff_data != NULL) {cpl_free(diff_data);}
    if (copied_spectrum != NULL) {cpl_vector_delete(copied_spectrum);}

    return positions;
}

/*----------------------------------------------------------------------------*/
/**
   @brief    Detect and refine emission line peak wavelengths using Gaussian fitting
   
   Identifies emission line peaks in the input spectrum and refines their positions
   by fitting Gaussian profiles. Outlier peaks are rejected based on statistical
   analysis of fitted wavelength shifts and line widths (sigma). This produces a
   clean set of well-measured line wavelengths suitable for wavelength calibration.
   
   @param spectrum   Input spectrum as bivector (X=wavelength in microns, Y=flux)
   @param min_frac   Minimum flux threshold as fraction of maximum flux (0.0-1.0)
   @param range      Optional wavelength ranges to search (NULL for all wavelengths)
                     Format: [min1, max1, min2, max2, ...] pairs in microns
   
   @return Vector of refined peak wavelengths in microns, or NULL on error
   
   @note The returned vector must be deallocated with cpl_vector_delete()
   @note Peaks too close together (< 2 * spectral resolution) are rejected
   @note Outliers are identified using 3-sigma clipping on wavelength shift and line width
   @note Gaussian fitting uses a window of ±5 pixels around each initial peak position
   @note Failed Gaussian fits result in peak rejection
*/
/*----------------------------------------------------------------------------*/
cpl_vector *eris_ifu_lcorr_get_peak_lambdas (const cpl_bivector *spectrum,
        double min_frac,
        cpl_vector *range) {

    cpl_vector *peaks = NULL,
               *final_peaks = NULL;
    cpl_array *positions = NULL;
    cpl_size no_peaks = 0,
        spec_size = 0;
    const int *pos_data = NULL;
    const double *spec_data =  NULL;
    double *lambda_data =  NULL;

    cpl_vector *ref_fit = NULL,
               *sigma_fit = NULL,
               *lambda_ref = NULL,
               *lambda_tmp = NULL,
               *sigma_tmp = NULL;
    int         *invalid_index = NULL;
    cpl_size    peak_idx = 0;
    double      delta_lambda = 0.0,
                lambda_in = 0.0,
                lambda_out = 0.0,
                sigma = 0.0,
                l_median = 0.0,
                l_stdev = 0.0,
                s_median = 0.0,
                s_stdev = 0.0,
                l_limit = 0.0,
                s_limit = 0.0;
    int         validPeaksCnt = 0;
    const char  *stype;
    const int   halfwidth = 5;

    TRY {
        ASSURE(spectrum != NULL &&
                cpl_bivector_get_x_const(spectrum) != NULL &&
                cpl_bivector_get_y_const(spectrum) != NULL,
                CPL_ERROR_NULL_INPUT,
                "Not all input data is provided!");

        BRK_IF_NULL(
                positions = eris_ifu_lcorr_get_peak_positions(spectrum, min_frac, range));
        no_peaks = cpl_array_get_size(positions);

        BRK_IF_NULL(
                peaks = cpl_vector_new(no_peaks));

        BRK_IF_NULL(
                pos_data = cpl_array_get_data_int_const(positions));
        BRK_IF_NULL(
                spec_data = cpl_vector_get_data_const(cpl_bivector_get_x_const(spectrum)));
        BRK_IF_NULL(
                lambda_data = cpl_vector_get_data(peaks));

        spec_size = cpl_bivector_get_size(spectrum);

        cpl_size i = 0;
        for (i = 0; i < no_peaks; i++) {
            if (pos_data[i] >= spec_size) {
                BRK_WITH_ERROR(CPL_ERROR_ACCESS_OUT_OF_RANGE);
            }
            lambda_data[i] = spec_data[pos_data[i]];
        }


        BRK_IF_NULL(
                invalid_index = cpl_calloc(no_peaks, sizeof(int)));
        // scan for peaks which are too close

        delta_lambda = 1.93 / 11200.;

        for (i = 0; i < no_peaks-1; i++) {
            if ((cpl_vector_get(peaks,i+1) - cpl_vector_get(peaks,i)) < 2. * delta_lambda) {
                invalid_index[i] = 1;
                invalid_index[i+1] = 1;
                cpl_msg_debug(cpl_func,
                        "Pair of peaks are rejected because they are too close: %lld - %lld, %f - %f",
                        i, i+1, cpl_vector_get(peaks,i), cpl_vector_get(peaks,i+1));
            }
        }

        // find exact peak location by fitting a gaussian
        // detect and withdraw outliers
        BRK_IF_NULL(
                ref_fit = cpl_vector_new(no_peaks));
        BRK_IF_NULL(
                sigma_fit = cpl_vector_new(no_peaks));
        BRK_IF_NULL(
                lambda_ref = cpl_vector_new(no_peaks));
        BRK_IF_NULL(
                lambda_tmp = cpl_vector_new(no_peaks));
        BRK_IF_NULL(
                sigma_tmp = cpl_vector_new(no_peaks));

        peak_idx = 0;
        stype="reference";
        if (IDL !=NULL ) fprintf(IDL, "%20s: %s\n", "xcorr spectrum type", stype);
        for (i = 0 ; i < no_peaks; i++) {
            lambda_in = cpl_vector_get(peaks,i);
            sigma = 0.0;
            lambda_out = fit_peak (spectrum, spec_size, lambda_in, halfwidth, &sigma);
            if (invalid_index[i] == 0) {
                if (lambda_out > 0.) {
                    BRK_IF_ERROR(
                            cpl_vector_set(ref_fit, i, lambda_out));
                    BRK_IF_ERROR(
                            cpl_vector_set(sigma_fit, i, sigma));
                    BRK_IF_ERROR(
                            cpl_vector_set(lambda_ref, peak_idx, lambda_in));
                    BRK_IF_ERROR(
                            cpl_vector_set(lambda_tmp, peak_idx, lambda_out));
                    BRK_IF_ERROR(
                            cpl_vector_set(sigma_tmp, peak_idx, sigma));
                    peak_idx++;
                } else {
                    invalid_index[i] = 2;
                    cpl_msg_debug(cpl_func,
                            "Gaussian fit failed for %s spectrum, wavelenght %f, peak index %lld",
                            stype, lambda_in, i);
                }
            }
        }
        BRK_IF_ERROR(
                cpl_vector_set_size(lambda_ref,peak_idx));
        BRK_IF_ERROR(
                cpl_vector_set_size(lambda_tmp,peak_idx));
        BRK_IF_ERROR(
                cpl_vector_set_size(sigma_tmp,peak_idx));
        if (IDL !=NULL ) {
            fprintf(IDL, "%20s: %lld ", "xcorr lambda_ref", peak_idx); cpl_vector_dump(lambda_ref, IDL);
            fprintf(IDL, "%20s: %lld ", "xcorr lambda_tmp", peak_idx); cpl_vector_dump(lambda_tmp, IDL);
            fprintf(IDL, "%20s: %lld ", "xcorr sigma_tmp",  peak_idx); cpl_vector_dump(sigma_tmp,  IDL);
        }

        BRK_IF_ERROR(
                cpl_vector_subtract(lambda_tmp,lambda_ref));
        l_median = cpl_vector_get_median_const(lambda_tmp);
        l_stdev = cpl_vector_get_stdev(lambda_tmp);
        s_median = cpl_vector_get_median_const(sigma_tmp);
        s_stdev = cpl_vector_get_stdev(sigma_tmp);
        l_limit = 3. * l_stdev;
        s_limit = 3. * s_stdev;
        if (IDL !=NULL ) fprintf(IDL, "%20s: %f %f %f %f %f %f\n", "xcorr limits", l_median, l_stdev, l_limit, s_median, s_stdev, s_limit);

        for (i = 0; i < no_peaks; i++){
           if (invalid_index[i] == 0) {
               double tmp_lambda = cpl_vector_get(ref_fit,i) - cpl_vector_get(peaks,i);
               double tmp_sigma =  cpl_vector_get(sigma_fit,i);
               if (fabs(tmp_lambda - l_median) >  l_limit) {
                   invalid_index[i] = 3;
                   cpl_msg_debug(cpl_func,
                           "Peak rejected due lambda outlier, %s spectrum, wavelength %f, index %lld, limit %f, value %f",
                           stype, cpl_vector_get(peaks,i), i, l_limit, fabs(tmp_lambda - l_median));
               } else if (fabs(tmp_sigma - s_median) >  s_limit) {
                   cpl_msg_debug(cpl_func,
                           "Peak rejected due sigma outlier, %s spectrum, wavelength %f, index %lld, limit %f, value %f",
                           stype, cpl_vector_get(peaks,i), i, s_limit,fabs(tmp_sigma - s_median));
                   invalid_index[i] = 4;
               }
           }
        }

        validPeaksCnt = 0;
        for (i = 0; i < no_peaks; i++) {
           if (invalid_index[i] == 0) {
               validPeaksCnt++;
           }
        }
        final_peaks = cpl_vector_new(validPeaksCnt);
        validPeaksCnt = 0;
        for (i = 0; i < no_peaks; i++) {
           if (invalid_index[i] == 0) {
               cpl_vector_set(final_peaks, validPeaksCnt,
                   cpl_vector_get(ref_fit,i));
               validPeaksCnt++;
           }
        }
        CHECK_ERROR_STATE();
    }
    CATCH
    {
        eris_ifu_free_vector(&final_peaks);
        final_peaks = NULL;
    }

    if (positions != NULL) {cpl_array_delete(positions);}
    eris_ifu_free_int_array(&invalid_index);
    eris_ifu_free_vector(&peaks);
    eris_ifu_free_vector(&ref_fit);
    eris_ifu_free_vector(&sigma_fit);
    eris_ifu_free_vector(&lambda_ref);
    eris_ifu_free_vector(&lambda_tmp);
    eris_ifu_free_vector(&sigma_tmp);
    return final_peaks;

}

/*----------------------------------------------------------------------------*/
/**
    @brief define a Gaussian profile


    @param x    File name of input FITS file holding the reference spectrum
    @param a   minimum fraction
    @param result Gaussian profile

   @return 1 in case of success, 0 otherwise.

*/
/*----------------------------------------------------------------------------*/
int gauss1d_fnc(const double x[], const double a[], double *result)
{
    double peak   = a[0];
    double mean   = a[1];
    double sigma  = a[2];
    double offset = a[3];

    if (sigma == 0.0) {
        return 1;
    }
    double z = (x[0]- mean)/sigma;
    double ez = exp(-z*z/2.);
    *result = offset + peak * ez;
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
    @brief define a Gaussian profile


    @param x    File name of input FITS file holding the reference spectrum
    @param a   minimum fraction
    @param result Gaussian profile

   @return 1 in case of success, 0 otherwise.

*/
/*----------------------------------------------------------------------------*/
int gauss1d_fncd(const double x[], const double a[], double result[])
{
    double peak   = a[0];
    double mean   = a[1];
    double sigma  = a[2];
//    double offset = a[3];

    if (sigma == 0.0) {
        return 1;
    }
    double z = (x[0]- mean)/sigma;
    double ez = exp(-z*z/2.);

   // derivative in peak
    result[0] = ez;

    // derivative in mean
    result[1] = peak * ez * z/sigma;

    // derivative in sigma
    result[2] =  result[1] * z;

    // derivative in offset
    result[3] = 1.;

    return 0;
}

/*----------------------------------------------------------------------------*/
/**
    @brief fit a peak


    @param spectrum  input spectrum
    @param size size of input spectrum
    @param lambda_in input lambda near which peak is found
    @param half_width input half_with of range where fit is done
    @param sigma_par output sigma

    @return position of peak maximum, 0 in case of failure.

*/
/*----------------------------------------------------------------------------*/
double fit_peak (const cpl_bivector *spectrum, const cpl_size size,
		double lambda_in, int half_width, double *sigma_par) {

    double error = -1.0;
    double x0 = 0., sigma = 0., area = 0., offset = 0., mserr = 0.;
    cpl_size pos_found = 0, low_bound = 0, high_bound = 0;
    cpl_vector *vx = NULL,
               *vy = NULL;
    cpl_fit_mode fit_mode = CPL_FIT_ALL;
    cpl_error_code cpl_err;

    const double sqrt2pi = sqrt(2. * 3.14159265358979323846);

    sigma = *sigma_par;
    if (IDL !=NULL ) fprintf(IDL, "%20s:  %f  %d %lld %f\n", "fitpeak input",  lambda_in, half_width, size, sigma);

    if ((lambda_in < cpl_vector_get(cpl_bivector_get_x_const(spectrum),  0)) ||
        (lambda_in > cpl_vector_get(cpl_bivector_get_x_const(spectrum),size-1))) {
        return error;
    }
    pos_found = cpl_vector_find(cpl_bivector_get_x_const(spectrum), lambda_in);
    if (pos_found < 0) {
        return error;
    }
    low_bound = pos_found - half_width;
    high_bound = pos_found + half_width;
    if (low_bound < 0) {
        low_bound = 0;
    }
    if (high_bound >= size) {
        high_bound = size - 1;
    }

    vx = cpl_vector_extract(cpl_bivector_get_x_const(spectrum), low_bound, high_bound, 1);
    vy = cpl_vector_extract(cpl_bivector_get_y_const(spectrum), low_bound, high_bound, 1);
//    printf("low: %5d   high: %5d    lamda_in: %f\n", low_bound, high_bound, lambda_in);
    if (IDL !=NULL ) {
        fprintf(IDL, "%20s: %lld %lld %lld\n", "fitpeak selection",  low_bound, high_bound, cpl_vector_get_size(vx));
        fprintf(IDL, "%20s: %lld",             "fitpeak vx", cpl_vector_get_size(vx)); cpl_vector_dump(vx, IDL);
        fprintf(IDL, "%20s: %lld",             "fitpeak vy", cpl_vector_get_size(vx)); cpl_vector_dump(vy, IDL);
    }
    if (high_bound-low_bound < 4) {
        if (vx != NULL) {cpl_vector_delete(vx);}
        if (vy != NULL) {cpl_vector_delete(vy);}
       return error;
    }

    if (sigma == 0.0) {
        fit_mode = CPL_FIT_ALL;
    } else {
        fit_mode = CPL_FIT_CENTROID | CPL_FIT_AREA | CPL_FIT_OFFSET;
    }

    cpl_err = cpl_vector_fit_gaussian(vx, NULL, vy, NULL,
            fit_mode, &x0, &sigma, &area, &offset, &mserr, NULL, NULL);
//    printf("vector fit               error: %d  offset: %f    x0: %f   sigma: %f    area: %f    offset: %f      mse: %f\n",
//            cpl_err, lambda_in-x0, x0, sigma, area/sigma/sqrt2pi, offset, mse);
    if (cpl_err != CPL_ERROR_NONE) {
        cpl_error_reset();
    }
    if (IDL !=NULL ) fprintf(IDL, "%20s: %d %f  %f %f %f\n", "fitpeak vectorfit",  cpl_err, area/sigma/sqrt2pi, x0, sigma, offset);


    if (IDL !=NULL ) {
        cpl_matrix *mx = cpl_matrix_wrap(cpl_vector_get_size(vx), 1, cpl_vector_get_data(vx));
        cpl_vector *fit_par = cpl_vector_new(4);
        cpl_vector_set(fit_par,0,cpl_vector_get_max(vy));
        cpl_vector_set(fit_par,1,lambda_in);
        cpl_vector_set(fit_par,2,(cpl_vector_get(vx,cpl_vector_get_size(vx)-1)-cpl_vector_get(vx,0))/9.);
        cpl_vector_set(fit_par,3,cpl_vector_get_min(vy));
        cpl_error_code cpl_err_tmp = cpl_fit_lvmq(mx, NULL,
                vy, NULL,
                fit_par, NULL,
                &gauss1d_fnc, &gauss1d_fncd,
                CPL_FIT_LVMQ_TOLERANCE / 10000.,        // 0.01
                CPL_FIT_LVMQ_COUNT,            // 5
                CPL_FIT_LVMQ_MAXITER * 1000,          // 1000
                &mserr,
                NULL,
                NULL);
        if (cpl_err_tmp != CPL_ERROR_NONE) {
            cpl_error_reset();
        }
        //    printf("lvmq fit                 error: %d  offset: %f    x0: %f   sigma: %f    peak: %f    offset: %f      mse: %f\n",
        //            cpl_err,
        //            lambda_in-cpl_vector_get(fit_par,1),
        //            cpl_vector_get(fit_par,1),
        //            cpl_vector_get(fit_par,2),
        //            cpl_vector_get(fit_par,0),
        //            cpl_vector_get(fit_par,3),
        //            mse);
        fprintf(IDL, "%20s: %d %f  %f %f %f\n", "fitpeak LVMQfit",
                cpl_err_tmp, cpl_vector_get(fit_par,0), cpl_vector_get(fit_par,1),
                cpl_vector_get(fit_par,2),cpl_vector_get(fit_par,3));
        if (mx != NULL) cpl_matrix_unwrap(mx);
        if (fit_par != NULL) cpl_vector_delete(fit_par);
    }
//    printf("\n");
    cpl_vector_delete(vx);
    cpl_vector_delete(vy);

    if (cpl_err != CPL_ERROR_NONE) {
        if (cpl_err != CPL_ERROR_CONTINUE) {
            cpl_msg_error("","%s\n",cpl_error_get_message_default(cpl_err));
        }
        return error;
    }
    *sigma_par = sigma;
    return x0;

}

/*----------------------------------------------------------------------------*/
/**
    @brief cross-correlate two input spectra


    @param object  input spectrum
    @param peaks peaks
    @param bandId band setup
    @param pfit_order


   @return polynomial with solution in case of success, NULL otherwise.

*/
/*----------------------------------------------------------------------------*/
cpl_polynomial *eris_ifu_lcorr_crosscorrelate_spectra(
        cpl_bivector *object,
        cpl_vector *peaks,
        ifsBand bandId,
		const int pfit_order) {

    cpl_vector       *lambda_fit = NULL,
                     *ref_fit = NULL,
                     *sigma_fit = NULL,
                     *lambda_ref = NULL,
                     *lambda_tmp = NULL,
                     *sigma_tmp = NULL,
                     *final_obj_in = NULL,
                     *final_obj_fit = NULL,
                     *final_obj_diff = NULL;
    cpl_polynomial *coeffs = NULL;
    cpl_matrix * samppos = NULL;

    int halfwidth_obj = 0,
        i             = 0;
    cpl_size obj_size = 0,
        no_peaks = 0,
        no_valid_peaks = 0,
        peak_idx = 0,
        v_idx    = 0;
    int *invalid_index = NULL;

    double lambda_in = 0.0,
           lambda_out = 0.0,
           sigma = 0.0,
           l_median = 0.0,
           l_stdev = 0.0,
           s_median = 0.0,
           s_stdev = 0.0,
           l_limit = 0.0,
           s_limit = 0.0,
           delta_lambda = 0.0,
           resolution;

    const char *stype = NULL;

    TRY {
        ASSURE(object != NULL &&
                peaks != NULL,
                CPL_ERROR_NULL_INPUT,
                "Not all input data is provided!");

        resolution = eris_ifu_get_band_resolution(bandId);

        obj_size = cpl_bivector_get_size(object);
        no_peaks = cpl_vector_get_size(peaks);
        if (IDL !=NULL ) {
            fprintf(IDL, "%20s: %lld %lld %f\n", "xcorr input", obj_size, no_peaks, resolution);
            fprintf(IDL, "%20s: %lld ", "xcorr object_l",    obj_size); cpl_vector_dump(cpl_bivector_get_x_const(object), IDL);
            fprintf(IDL, "%20s: %lld ", "xcorr object_v",    obj_size); cpl_vector_dump(cpl_bivector_get_y_const(object), IDL);
            fprintf(IDL, "%20s: %lld ", "xcorr peaks",       no_peaks); cpl_vector_dump(peaks, IDL);
        }

        ASSURE(obj_size > 1 && no_peaks > 1 ,
               CPL_ERROR_ILLEGAL_INPUT,
                "Input spectra must have more than one value");

        BRK_IF_NULL(
                invalid_index = cpl_calloc(no_peaks, sizeof(int)));

        delta_lambda = 2. *
                cpl_vector_get(cpl_bivector_get_x_const(object), cpl_bivector_get_size(object) / 2) /
                resolution;


        halfwidth_obj =  delta_lambda / (cpl_vector_get(cpl_bivector_get_x_const(object),1) -
                                     cpl_vector_get(cpl_bivector_get_x_const(object),0));
        //cpl_msg_info(cpl_func,"computed halfwidth_obj: %d",halfwidth_obj);
        halfwidth_obj = 8;  // 8 pixels allows to recover also large line shifts (due to flexures) with poly order = 0
        //cpl_msg_info(cpl_func,"hard-coded halfwidth_obj: %d",halfwidth_obj);
       // second scan object spectrum
       // find exact peak location by fitting a gaussian
       // detect and withdraw outliers
        BRK_IF_NULL(
                lambda_fit = cpl_vector_new(no_peaks));
        BRK_IF_NULL(
                sigma_fit = cpl_vector_new(no_peaks));
        BRK_IF_NULL(
                lambda_ref = cpl_vector_new(no_peaks));
        BRK_IF_NULL(
                lambda_tmp = cpl_vector_new(no_peaks));
        BRK_IF_NULL(
                sigma_tmp = cpl_vector_new(no_peaks));

        peak_idx = 0;
        stype="object";
        if (IDL !=NULL ) fprintf(IDL, "%20s: %s\n", "xcorr spectrum type", stype);
        for (i = 0; i < no_peaks; i++) {
            lambda_in = cpl_vector_get(peaks,i);
            sigma = 0.0;
            lambda_out = fit_peak (object, obj_size, lambda_in, halfwidth_obj, &sigma);
            if (invalid_index[i] == 0) {
                if (lambda_out > 0.) {
                    BRK_IF_ERROR(
                            cpl_vector_set(lambda_fit, i, lambda_out));
                    BRK_IF_ERROR(
                            cpl_vector_set(sigma_fit, i, sigma));
                    BRK_IF_ERROR(
                            cpl_vector_set(lambda_ref, peak_idx, lambda_in));
                    BRK_IF_ERROR(
                            cpl_vector_set(lambda_tmp, peak_idx, lambda_out));
                    BRK_IF_ERROR(
                            cpl_vector_set(sigma_tmp, peak_idx, sigma));
                    peak_idx++;
                } else {
                    invalid_index[i] = 5;
                    cpl_msg_debug(cpl_func,
                            "Gaussian fit failed for %s spectrum, wavelenght %f, peak index %d",
                            stype, lambda_in, i);
                }
            }
        }
        BRK_IF_ERROR(
                cpl_vector_set_size(lambda_ref,peak_idx));
        BRK_IF_ERROR(
                cpl_vector_set_size(lambda_tmp,peak_idx));
        BRK_IF_ERROR(
                cpl_vector_set_size(sigma_tmp,peak_idx));
        if (IDL !=NULL ) {
            fprintf(IDL, "%20s: %lld ", "xcorr lambda_ref", peak_idx); cpl_vector_dump(lambda_ref, IDL);
            fprintf(IDL, "%20s: %lld ", "xcorr lambda_tmp", peak_idx); cpl_vector_dump(lambda_tmp, IDL);
            fprintf(IDL, "%20s: %lld ", "xcorr sigma_tmp",  peak_idx); cpl_vector_dump(sigma_tmp, IDL);
        }

        BRK_IF_ERROR(
                cpl_vector_subtract(lambda_tmp,lambda_ref));
        l_median = cpl_vector_get_median_const(lambda_tmp);
        l_stdev = cpl_vector_get_stdev(lambda_tmp);
        s_median = cpl_vector_get_median_const(sigma_tmp);
        s_stdev = cpl_vector_get_stdev(sigma_tmp);
        l_limit = 2. * l_stdev;
        s_limit = 2. * s_stdev;
        cpl_vector_delete(lambda_ref);
        cpl_vector_delete(lambda_tmp);
        cpl_vector_delete(sigma_tmp);
        if (IDL !=NULL ) fprintf(IDL, "%20s: %f %f %f %f %f %f\n", "xcorr limits", l_median, l_stdev, l_limit, s_median, s_stdev, s_limit);

        for (i = 0; i < no_peaks; i++){
            if (invalid_index[i] == 0) {
                double tmp_lambda = cpl_vector_get(lambda_fit,i) - cpl_vector_get(peaks,i);
                double tmp_sigma =  cpl_vector_get(sigma_fit,i);
                if (fabs(tmp_lambda - l_median) >  l_limit) {
                    invalid_index[i] = 6;
                    cpl_msg_debug(cpl_func,
                            "Peak rejected due lambda outlier, %s spectrum, wavelength %f, index %d, limit %f, value %f",
                            stype, cpl_vector_get(peaks,i), i, l_limit, fabs(tmp_lambda - l_median));
                } else if (fabs(tmp_sigma - s_median) >  s_limit) {
                    cpl_msg_debug(cpl_func,
                            "Peak rejected due sigma outlier, %s spectrum, wavelength %f, index %d, limit %f, value %f",
                            stype, cpl_vector_get(peaks,i), i, s_limit,fabs(tmp_sigma - s_median));
                    invalid_index[i] = 7;
                }
            }
        }
        cpl_vector_delete(sigma_fit);

        no_valid_peaks = 0;
	char *dbg_msg;
        BRK_IF_NULL(
			     dbg_msg = cpl_malloc((no_peaks * 2 + 1) * sizeof(char)));
	dbg_msg[no_peaks*2] = '\0';
        if (IDL !=NULL ) fprintf(IDL, "%20s: %lld", "xcorr invalid peaks", no_peaks);
        for (i = 0; i < no_peaks; i++){
	    sprintf(&dbg_msg[2*i], "%2d", invalid_index[i]);
            if (invalid_index[i] == 0) {
                no_valid_peaks++;
            }
            if (IDL !=NULL ) fprintf(IDL, " %d", invalid_index[i]);
        }
        if (IDL !=NULL ) {
            fprintf(IDL, "\n");
        }
	cpl_msg_debug(__func__,"%d valid ones of %d peaks: %s", (int) no_valid_peaks, (int) no_peaks, dbg_msg);
	cpl_free(dbg_msg);

        if (no_valid_peaks == 0) {
            cpl_msg_error(cpl_func, " |-----------------------------------------------------|");
            cpl_msg_error(cpl_func, " | No OH lines could be fitted. Aborting.              |");
            cpl_msg_error(cpl_func, " | Run again with parameter --skip_oh_align            |");
            cpl_msg_error(cpl_func, " |                                                     |");
            cpl_msg_error(cpl_func, " | WARNING: the spectral calibration of the outputs    |");
            cpl_msg_error(cpl_func, " |          is expected to be off for several pixels   |");
            cpl_msg_error(cpl_func, " |          due to flexure of the instrument           |");
            cpl_msg_error(cpl_func, " |-----------------------------------------------------|");

            cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_OUTPUT);
            CHECK_ERROR_STATE();
        }

        BRK_IF_NULL(
                final_obj_in = cpl_vector_new(no_valid_peaks));
        BRK_IF_NULL(
                final_obj_fit = cpl_vector_new(no_valid_peaks));
        BRK_IF_NULL(
                final_obj_diff = cpl_vector_new(no_valid_peaks));
        v_idx = 0;
        for (i = 0; i < no_peaks; i++){
            if (invalid_index[i] == 0) {
                double in = cpl_vector_get(peaks,i);
                double fit = cpl_vector_get(lambda_fit,i);
                BRK_IF_ERROR(
                        cpl_vector_set(final_obj_in, v_idx, in));
                BRK_IF_ERROR(
                        cpl_vector_set(final_obj_fit, v_idx, fit));
                BRK_IF_ERROR(
                        cpl_vector_set(final_obj_diff, v_idx, fit-in));
                v_idx++;
            }
        }
        if (IDL !=NULL ) {
            fprintf(IDL, "%20s: %lld ", "xcorr final_obj_in",   no_valid_peaks); cpl_vector_dump(final_obj_in,   IDL);
            fprintf(IDL, "%20s: %lld ", "xcorr final_obj_fit",  no_valid_peaks); cpl_vector_dump(final_obj_fit,  IDL);
            fprintf(IDL, "%20s: %lld ", "xcorr final_obj_diff", no_valid_peaks); cpl_vector_dump(final_obj_diff, IDL);
        }
        /* poly fit deg can be changed by user. 
           0 is better in K band to prevent divergence on OH corr due to extrapolation
           higher walues (was 3 in the past) may allow higher accuracy in H and J band
           with many lines. 
        */

        cpl_size degree[1] = {0}; //essentially taking the mean of final_obj_diff
        degree[0] = pfit_order;
        cpl_msg_info(cpl_func,"Use poly deg: %lld to determine OH correction",degree[0]);
        BRK_IF_NULL(
                coeffs = cpl_polynomial_new(1));

        if (cpl_vector_get_size(final_obj_in) > (degree[0] + 3)) {
            BRK_IF_NULL(
                    samppos = cpl_matrix_wrap(1,cpl_vector_get_size(final_obj_in),
                            cpl_vector_get_data(final_obj_in)));
            BRK_IF_ERROR(
                    cpl_polynomial_fit(coeffs, samppos, NULL, final_obj_diff, NULL, CPL_FALSE,
                            degree, degree));
        } else {
            cpl_msg_warning(NULL,"Too few valid peaks for lambda correction using OH lines");
            // create polynomial a0=0
            cpl_size pows[1];
            pows[0] = 0;
            BRK_IF_ERROR(
                    cpl_polynomial_set_coeff (coeffs, pows, 0.));
        }

        if (IDL !=NULL ) {
            cpl_size pows[1];
            fprintf(IDL, "%20s: %lld", "xcorr polyfit", cpl_polynomial_get_degree(coeffs));
            for (i = 0; i <= cpl_polynomial_get_degree(coeffs); i++) {
                pows[0] = i;
                fprintf(IDL, " %f", cpl_polynomial_get_coeff(coeffs,pows));
            }
            fprintf(IDL, "\n");
       }
    }
    CATCH
    {
        CATCH_MSG();
        if (coeffs != NULL) {
            cpl_polynomial_delete(coeffs);
            coeffs = NULL;
        }
    }

    if (ref_fit != NULL) cpl_vector_delete(ref_fit);
    if (lambda_fit != NULL) cpl_vector_delete(lambda_fit);
    if (final_obj_in != NULL) cpl_vector_delete(final_obj_in);
    if (final_obj_fit != NULL) cpl_vector_delete(final_obj_fit);
    if (final_obj_diff != NULL) cpl_vector_delete(final_obj_diff);
    if (invalid_index != NULL) cpl_free(invalid_index);
    if (samppos != NULL) cpl_matrix_unwrap(samppos);

    return coeffs;
}
/*----------------------------------------------------------------------------*/
/**
    @brief open ASCII file to hold debug results

    @param filename input filename

*/
/*----------------------------------------------------------------------------*/
void eris_ifu_lcorr_open_debug_file (const char* filename) {
    IDL = fopen(filename,"w");
}

/*----------------------------------------------------------------------------*/
/**
    @brief close ASCII file that hold debug results

*/
/*----------------------------------------------------------------------------*/
void eris_ifu_lcorr_close_debug_file (void) {
    fclose(IDL);
    IDL = NULL;
}
/**@}*/
