/* $Id$
 *
 * This file is part of the ERIS/NIX Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

 /*
 * $Author$
 * $Date$
 * $Rev$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <string.h>

#include "eris_nix_gain_linearity.h"
#include "eris_nix_utils.h"

/*-----------------------------------------------------------------------------
                              Static function declarations
 -----------------------------------------------------------------------------*/

static
gain_linearity * engl_gain_linearity_create(cpl_imagelist *,
                                            cpl_mask *,
                                            double *,
                                            const hdrl_value,
                                            const double,
                                            cpl_propertylist *)
    CPL_ATTR_ALLOC;

static
double * engl_repack(const cpl_imagelist *)
    CPL_ATTR_ALLOC;

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_gain_linearity     Gain/linearity Utilities
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a 'gain_linearity' struct
  @param    gain_lin   The struct to be deleted
  @return   Void

  The function deletes a gain_linearity struct and its contents. If the
  pointer is NULL nothing is done and no error is set.
 */
/*----------------------------------------------------------------------------*/

void engl_gain_linearity_delete(gain_linearity * gain_lin) {
    
    if (gain_lin) {
        cpl_imagelist_delete(gain_lin->lin_coeffs);
        cpl_free(gain_lin->ordered_lin_coeffs);
        cpl_mask_delete(gain_lin->bpm);
        cpl_propertylist_delete(gain_lin->plist);
        cpl_free(gain_lin);
    }
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Load a 'gain_linearity' struct from FITS files
  @param    frameset     A frame set
  @param    gain_tag     The frame tag of the detmon_ir_lg_gain_table.fits file
  @param    coeffs_tag   The frame tag of the detmon_ir_lg_coeffs_cube.fits file
  @param    bpm_tag      The frame tag of the detmon_ir_lg_bpm.fits file
  @param    required     True if missing gain/linearity info to result in an error 
  @param    used         The list of raw/calibration frames used
  @return   Pointer to the gain_linearity struct

  The function loads a gain_linearity struct from FITS files.
 */
/*----------------------------------------------------------------------------*/

gain_linearity * engl_gain_linearity_load_from_frameset(const cpl_frameset * frameset,
                                                        const char * gain_tag,
                                                        const char * coeffs_tag,
                                                        const char * bpm_tag,
                                                        const int required,
                                                        cpl_frameset * used) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    const cpl_frame        * linbpmframe = NULL;
    const cpl_frame        * lincoeffsframe = NULL;
    const cpl_frame        * gainframe = NULL;
    cpl_imagelist          * lin_coeffs = NULL;
    cpl_mask               * lin_bpm = NULL;
    double                 * ordered_lin_coeffs = NULL;
    cpl_propertylist       * plist = NULL;
    double                   saturation_limit  = -1.0;
    gain_linearity         * result = NULL;

    /* check parameters */

    cpl_ensure(frameset, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(gain_tag, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(coeffs_tag, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(bpm_tag, CPL_ERROR_NULL_INPUT, NULL);
 
    /* Look for gain, coeffs and bpm files in frameset */

    gainframe = cpl_frameset_find_const(frameset, gain_tag);
    lincoeffsframe = cpl_frameset_find_const(frameset, coeffs_tag);
    linbpmframe = cpl_frameset_find_const(frameset, bpm_tag);

    if (required) {
       cpl_ensure(lincoeffsframe && linbpmframe && gainframe,
         CPL_ERROR_DATA_NOT_FOUND, NULL);
    } else if (!(lincoeffsframe && linbpmframe && gainframe)) {
       cpl_msg_warning(cpl_func, "gain/linearity information not available");
    }
   
    if (lincoeffsframe && linbpmframe && gainframe) { 

        /* Read the gain from the gain file property list */

        plist = cpl_propertylist_load(cpl_frame_get_filename(gainframe), 0);
        double gain_data = cpl_propertylist_get_double(plist, "ESO QC GAIN");
        double gain_err = 0.0;
        hdrl_value gain = (hdrl_value){gain_data, gain_err};

        /* look for saturation_limit in DETMON invocation parameters */

        for (int ip = 1; ip < 100; ip++) {
            char * pname = cpl_sprintf("ESO PRO REC1 PARAM%-2d NAME", ip);
            if (cpl_propertylist_has(plist, pname)) {
                if (strstr(cpl_propertylist_get_string(plist, pname),
                    "saturation_limit")) {
                    char * vname = cpl_sprintf("ESO PRO REC1 PARAM%-2d VALUE", ip);
                    const char * sat_limit = cpl_propertylist_get_string(plist, vname);
                    sscanf(sat_limit, "%lf", &saturation_limit);
                    cpl_free(vname);
                } 
            }
            cpl_free(pname);
            if (saturation_limit > 0.0) break;
        }
        if (saturation_limit < 0.0) {
            cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                  "failed to read saturation_limit");
        } else {
            cpl_msg_info(cpl_func, "gain/linearity saturation limit = %6.1f",
                         saturation_limit);
        } 

        /* Read in the linearity coefficients. They are stored by detmon
           using a call to cpl_dfs_save_imagelist with type 
           CPL_BPP_IEEE_FLOAT */

        lin_coeffs = cpl_imagelist_load(
                     cpl_frame_get_filename(lincoeffsframe), CPL_TYPE_FLOAT, 0);

        /* ..and the mask of non-linear pixels */

        lin_bpm = cpl_mask_load(cpl_frame_get_filename(linbpmframe), 0, 0);

        /* ..reorder the linearity coeffs for more efficient access during
             linearizaion */

        ordered_lin_coeffs = engl_repack(lin_coeffs); 

        /* create the structure */

        result = engl_gain_linearity_create(lin_coeffs, lin_bpm, 
                                            ordered_lin_coeffs, gain,
                                            saturation_limit, plist);

        /* Update 'used' frameset */

        cpl_frame * dup_lincoeffsframe = cpl_frame_duplicate(lincoeffsframe);
        cpl_frameset_insert(used, dup_lincoeffsframe);
        cpl_frame * dup_linbpmframe = cpl_frame_duplicate(linbpmframe);
        cpl_frameset_insert(used, dup_linbpmframe);
        cpl_frame * dup_gainframe = cpl_frame_duplicate(gainframe);
        cpl_frameset_insert(used, dup_gainframe);

        if (cpl_error_get_code() != CPL_ERROR_NONE) {
            engl_gain_linearity_delete(result);
            result = NULL;
        }

        /* tidy up */
    }

    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new gain_linearity struct and initialise the contents
  @param    lin_coeffs          List of images with the linearity fit coeffs
  @param    lin_bpm             Mask of non-linear pixels
  @param    ordered_lin_coeffs  Array of lin coeffs in memory-efficient order
  @param    gain                The gain
  @param    saturation_limit    The saturation limit
  @param    plist               Propertylist associated with info
  @return   The new gain_linearity struct

  The function returns a pointer to a gain_linearity structure with
  components initialised to copies of the given parameters. See
  'engl_repack' for a description of what is meant by 'memory-efficient'
  order.

 */
/*----------------------------------------------------------------------------*/
static
gain_linearity * engl_gain_linearity_create(cpl_imagelist * lin_coeffs,
                                            cpl_mask * lin_bpm,
                                            double * ordered_lin_coeffs,
                                            const hdrl_value gain,
                                            const double saturation_limit,
                                            cpl_propertylist * plist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    cpl_ensure(lin_coeffs && lin_bpm && plist, CPL_ERROR_NULL_INPUT, NULL);

    gain_linearity * result = cpl_malloc(sizeof(gain_linearity)); 
    result->lin_coeffs = lin_coeffs;
    result->bpm = lin_bpm;
    result->ordered_lin_coeffs = ordered_lin_coeffs;
    result->gain = gain;
    result->saturation_limit = saturation_limit;
    result->plist = plist;

    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Create a test gain_linearity struct
  @param    nx                The x dim of the array
  @param    ny                The y dim of the array
  @param    coeff0            Value to give coeff0 for all pixels
  @param    coeff1            Value for coeff1
  @param    coeff2            Value for coeff2
  @param    coeff3            Value for coeff3
  @param    gain              The gain
  @param    saturation        The saturation limit
  @return   The new gain_linearity struct

  The function returns a pointer to a dummy gain_linearity structure for use
  in simple tests.

  The object and its contents are deleted by a call to 
  engl_gain_linearity_delete.
 */
/*----------------------------------------------------------------------------*/

gain_linearity * engl_gain_linearity_test(const cpl_size nx,
                                          const cpl_size ny,
                                          const double coeff0,
                                          const double coeff1,
                                          const double coeff2,
                                          const double coeff3,
                                          const hdrl_value gain,
                                          const double saturation) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    cpl_imagelist * coeffs_imagelist = cpl_imagelist_new();
    cpl_image * c0 = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    cpl_image_fill_window(c0, 1, 1, nx, ny, coeff0);
    cpl_imagelist_set(coeffs_imagelist, c0, 0);
    cpl_image * c1 = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    cpl_image_fill_window(c1, 1, 1, nx, ny, coeff1);
    cpl_imagelist_set(coeffs_imagelist, c1, 1);
    cpl_image * c2 = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    cpl_image_fill_window(c2, 1, 1, nx, ny, coeff2);
    cpl_imagelist_set(coeffs_imagelist, c2, 2);
    cpl_image * c3 = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    cpl_image_fill_window(c3, 1, 1, nx, ny, coeff3);
    cpl_imagelist_set(coeffs_imagelist, c3, 3);

    cpl_mask * bpm = cpl_mask_new(nx, ny);
    cpl_propertylist * lin_plist = cpl_propertylist_new();

    double * ordered_coeffs = engl_repack(coeffs_imagelist);

    gain_linearity * result = engl_gain_linearity_create(coeffs_imagelist,
                                                         bpm,
                                                         ordered_coeffs,
                                                         gain,
                                                         saturation,
                                                         lin_plist);

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        engl_gain_linearity_delete(result);
        result = NULL;
    }

    /* tidy up */

    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Create an array with the linerization coeffs in an efficient
            order for later use.
  @param    lin_coeffs      The list of float-images with the coeffs
  @return   The allocated, repacked, double array.
  @note     The constant term is set to zero.

 */
/*----------------------------------------------------------------------------*/
static
double * engl_repack(const cpl_imagelist * lin_coeffs) {

    const cpl_size fit_order = cpl_imagelist_get_size(lin_coeffs);
    const cpl_size nx =
        cpl_image_get_size_x(cpl_imagelist_get_const(lin_coeffs, 0));
    const cpl_size ny =
        cpl_image_get_size_y(cpl_imagelist_get_const(lin_coeffs, 0));
    double * result = NULL;
    int j=0;

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;
    cpl_ensure(cpl_image_get_type(cpl_imagelist_get_const(lin_coeffs, 0)) ==
               CPL_TYPE_FLOAT, CPL_ERROR_UNSUPPORTED_MODE, NULL);

    result = cpl_malloc(fit_order * nx * ny * sizeof(*result));

    /* set coeff[0]=0 to ensure curve goes through origin, otherwise
       get oddities at low flux levels */
    for (cpl_size ipos=0 ; ipos < nx * ny ; ipos++) {
        const cpl_size ordered_pos = ipos * fit_order + j;
        result[ordered_pos] = 0.0;
    }
    j++;

    for (; j < fit_order; j++) {
        const cpl_image * fit_coeff_image = cpl_imagelist_get_const(lin_coeffs,
                                                                    j);
        const float * fit_coeff_data = cpl_image_get_data_float_const(
                                       fit_coeff_image);
        for (cpl_size ipos=0 ; ipos < nx * ny ; ipos++) {
            const cpl_size ordered_pos = ipos * fit_order + j;
            result[ordered_pos] = fit_coeff_data[ipos];
        }
    }

    return result;
}
 
/**@}*/
