/* $Id$
 *
 * This file is part of the ERIS/NIX Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

 /*
 * $Author$
 * $Date$
 * $Rev$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <string.h>

#include "eris_nix_master_flat.h"
#include "eris_nix_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_master_flat      Utilities relating to 
 *                                     MASTER_FLATs
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new master_flat struct and initialise the contents
  @param    flat the image containing the flatfield
  @param    cold_bpm the mask with 'cold' pixels (may be NULL)
  @param    confidence the image containing the confidence array
  @param    filename the name of the file associated with the flat
  @param    plist the propertylist associated with the flat
  @return   The new master_flat struct

  The function returns a pointer to a master_flat structure that is 
  populated with copies of the given items.

  The object and its contents are deleted by a call to 
  en_master_flat_delete.
 */
/*----------------------------------------------------------------------------*/

master_flat * en_master_flat_create(const hdrl_image * flat,
                                    const cpl_mask * cold_bpm,
                                    const cpl_image * confidence,
                                    const char * filename,
                                    const cpl_propertylist * plist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    /* check for NULL inputs */

    cpl_ensure(flat, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(confidence, CPL_ERROR_NULL_INPUT, NULL);

    master_flat * result = cpl_malloc(sizeof(master_flat));
 
    result->flat = hdrl_image_duplicate(flat);
    if (cold_bpm) {
        result->cold_bpm = cpl_mask_duplicate(cold_bpm);
    } else {
        result->cold_bpm = NULL;
    }
    result->confidence = cpl_image_duplicate(confidence);
    if (filename) {
        result->filename = cpl_strdup(filename);
    } else {
        result->filename = NULL;
    }
    if (plist) {
        result->plist = cpl_propertylist_duplicate(plist);
    } else {
        result->plist = NULL;
    }

    return result;
}

/**@}*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a 'master_flat' struct
  @return   Void

  The function deletes a master_flat struct and its contents. If the
  pointer is NULL then no action is taken but no error set.
 */
/*----------------------------------------------------------------------------*/

void en_master_flat_delete(master_flat * target) {
    
    if (target) {
        cpl_free(target->filename);
        hdrl_image_delete(target->flat);
        cpl_mask_delete(target->cold_bpm);
        cpl_image_delete(target->confidence);
        cpl_propertylist_delete(target->plist);
        cpl_free(target);
    }
}

/**@}*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Load a 'master_flat' struct from a frameset.
  @return   Pointer to the master_flat struct

  The function loads a master_flat struct from a frameset.
 */
/*----------------------------------------------------------------------------*/

master_flat * en_master_flat_load_from_frameset(const cpl_frameset * frameset,
                                                const char * tag,
                                                cpl_frameset * used,
                                                const int required) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    cpl_mask           * cold_bpm = NULL;
    cpl_image          * confidence = NULL;
    const char         * filename = NULL;
    hdrl_image         * flat = NULL;
    cpl_mask           * full_cold_bpm = NULL;
    cpl_image          * full_conf = NULL;
    hdrl_image         * full_flat = NULL;
    mef_extension_list * mefs = NULL;
    cpl_propertylist   * plist = NULL;
    master_flat        * result = NULL;

    /* check parameters */

    cpl_ensure(frameset, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(tag, CPL_ERROR_NULL_INPUT, NULL);
 
    /* Look for tagged file in frameset */

    const cpl_frame * target_frame = cpl_frameset_find_const(frameset, tag);
    enu_check(target_frame != NULL, CPL_ERROR_DATA_NOT_FOUND,
      "SoF has no file tagged %s", tag);
    filename = cpl_frame_get_filename(target_frame);

    /* Read the data in as an hdrl image, any mef extensions present */

    enu_himage_load_from_fits(filename, &flat, &mefs, &plist);
    enu_check_error_code("failed to read file %s", filename);

    /* Obtain the cold pixel mask from its mef extension, if present */

    for (int i=0; i < mefs->size; i++) {
        if (!strcmp(mefs->mef[i]->name, "COLD_BPM")) {
            cold_bpm = cpl_mask_duplicate(
                       (const cpl_mask *) (mefs->mef[i]->data));
            break;
        }
    }

    /* Obtain the confidence array from its mef extension, if present */

    for (int i=0; i<mefs->size; i++) {
        if (!strcmp(mefs->mef[i]->name, "CONFIDENCE")) {
            confidence = cpl_image_duplicate(
              (const cpl_image *) (mefs->mef[i]->data));
            break;
        }
    }
    if (!confidence) cpl_msg_warning(cpl_func, "confidence array not found");

    /* If the flat is windowed then create a full-chip version */

    cpl_size nx = 0;
    cpl_size ny = 0;
    int rot = 0;
    cpl_size strx = 0;
    cpl_size stry = 0;
    cpl_size nx_chip = 0;
    cpl_size ny_chip = 0;
    cpl_boolean windowed = 0;
    enu_get_window_info(&nx,
                        &ny,
                        &rot,
                        &strx,
                        &stry,
                        &nx_chip,
                        &ny_chip,
                        &windowed,
                        plist);
    enu_check_error_code("Failed to read flatfield window information");

    if (nx != nx_chip || ny != ny_chip) {
        cpl_msg_info(cpl_func, "windowed flatfield, extending to cover full chip");
        full_flat = hdrl_image_new(nx_chip,
                                   ny_chip);
        hdrl_image_copy(full_flat,
                        flat,
                        strx,
                        stry);

        if (confidence) {
            full_conf = cpl_image_new(nx_chip,
                                      ny_chip,
                                      cpl_image_get_type(confidence));
            cpl_image_copy(full_conf,
                           confidence,
                           strx,
                           stry);

        }

        if (cold_bpm) {
            full_cold_bpm = cpl_mask_new(nx_chip,
                                         ny_chip);
            cpl_mask_copy(full_cold_bpm,
                          cold_bpm,
                          strx,
                          stry);
        }

    } else {
        full_flat = flat; flat = NULL;
        full_conf = confidence; confidence = NULL;
        full_cold_bpm = cold_bpm; cold_bpm = NULL;
    }

    result = en_master_flat_create(full_flat, 
                                   full_cold_bpm,
                                   full_conf,
                                   filename,
                                   plist);

 cleanup: 

    if (cpl_error_get_code() == CPL_ERROR_NONE) {
        cpl_frame * dup_frame = cpl_frame_duplicate(target_frame);
        cpl_frameset_insert(used, dup_frame);
    } else {
        en_master_flat_delete(result);
        result = NULL;

        if (!required) {

            /* reset the error code if something has gone wrong but the 
               master_flat is not definitely required */

            cpl_msg_warning(cpl_func, "no file tagged %s in SoF", tag);
            cpl_error_reset();
        }
    }

    cpl_mask_delete(full_cold_bpm);
    cpl_mask_delete(cold_bpm);
    cpl_image_delete(full_conf);
    cpl_image_delete(confidence);
    hdrl_image_delete(full_flat);
    hdrl_image_delete(flat);
    enu_mef_extension_list_delete(mefs);
    cpl_propertylist_delete(plist);

    return result;
}

/**@}*/
