/* $Id$
 *
 * This file is part of the ERIS/NIX Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

 /*
 * $Author$
 * $Date$
 * $Rev$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <string.h>

#include "eris_nix_master_wave.h"
#include "eris_nix_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_master_wave     Utilities relating to MASTER_WAVEs
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new master_wave struct and initialise the contents
  @return   The new master_wave struct

  The function returns a pointer to a master_wave structure that is 
  populated with copies of the given items.

  The object and its contents are deleted by a call to 
  en_master_wave_delete.
 */
/*----------------------------------------------------------------------------*/

master_wave * en_master_wave_create(
                       const hdrl_image * raw_wave_image,
                       const cpl_mask * raw_wave_bpm,
                       const cpl_image * raw_wave_confidence,
                       const hdrl_image * calibrated_wave_image,
                       const cpl_mask * calibrated_wave_bpm,
                       const cpl_image * calibrated_wave_confidence,
                       const char * filename,
                       const cpl_propertylist * plist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    /* check for NULL inputs */

    cpl_ensure(raw_wave_image, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(raw_wave_bpm, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(raw_wave_confidence, CPL_ERROR_NULL_INPUT, NULL);

    master_wave * result = cpl_malloc(sizeof(master_wave));
 
    result->raw_wave_image = hdrl_image_duplicate(raw_wave_image);
    result->raw_wave_bpm = cpl_mask_duplicate(raw_wave_bpm);
    result->raw_wave_confidence = cpl_image_duplicate(raw_wave_confidence);
    if (calibrated_wave_image) {
        result->calibrated_wave_image = hdrl_image_duplicate(
                                        calibrated_wave_image);
    }
    if (calibrated_wave_bpm) {
        result->calibrated_wave_bpm = cpl_mask_duplicate(calibrated_wave_bpm);
    }
    if (calibrated_wave_confidence) {
        result->calibrated_wave_confidence = cpl_image_duplicate(
                                             calibrated_wave_confidence);
    }
    if (filename) {
        result->filename = cpl_strdup(filename);
    } else {
        result->filename = NULL;
    }
    if (plist) {
        result->plist = cpl_propertylist_duplicate(plist);
    } else {
        result->plist = NULL;
    }

    return result;
}

/**@}*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a 'master_wave' struct
  @return   Void

  The function deletes a master_wave struct and its contents. If the
  pointer is NULL then no action is taken but no error set.
 */
/*----------------------------------------------------------------------------*/

void en_master_wave_delete(master_wave * target) {
    
    if (target) {
        cpl_free(target->filename);
        hdrl_image_delete(target->raw_wave_image);
        cpl_mask_delete(target->raw_wave_bpm);
        cpl_image_delete(target->raw_wave_confidence);
        hdrl_image_delete(target->calibrated_wave_image);
        cpl_mask_delete(target->calibrated_wave_bpm);
        cpl_image_delete(target->calibrated_wave_confidence);
        cpl_propertylist_delete(target->plist);
        cpl_free(target);
    }
}

/**@}*/
