/* $Id: eris_pfits.c,v 1.14 2013-03-25 11:46:49 cgarcia Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2013-03-25 11:46:49 $
 * $Revision: 1.14 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>

#include "eris_pfits.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_pfits     FITS header protected access
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ancestor of a file.
  @param    aHeaders       property list/headers to read from
  @return   the requested value or NULL on error

  Queries FITS header for ESO.PRO.ANCESTOR
 */
/*----------------------------------------------------------------------------*/
//TODO: why we reset the error in function below?
const char *
eris_pfits_get_ancestor(const cpl_propertylist *aHeaders)
{
  cpl_ensure(aHeaders, CPL_ERROR_NULL_INPUT, NULL);
  cpl_errorstate prestate = cpl_errorstate_get();
  const char *value = cpl_propertylist_get_string(aHeaders, "ESO PRO ANCESTOR");
  cpl_errorstate_set(prestate);
  return value;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    find out the arcfile   
  @param    plist       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char * eris_pfits_get_arcfile(const cpl_propertylist * plist)
{
    const char * value = cpl_propertylist_get_string(plist, "ARCFILE");

    cpl_ensure(value != NULL, cpl_error_get_code(), NULL);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the DIT value 
  @param    plist       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double eris_pfits_get_dit(const cpl_propertylist * plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const double value = cpl_propertylist_get_double(plist, "ESO DET SEQ1 DIT");

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the DIT value
  @param    plist       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int eris_pfits_get_ndit(const cpl_propertylist * plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const int value = cpl_propertylist_get_int(plist, "ESO DET NDIT");

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}


/*---------------------------------------------------------------------------*/
/**
  @brief    find out the character string associated to the EXPTIME keyword
  @param    filename    sinfoni FITS file name
  @return   pointer to statically allocated character string
 */
/*---------------------------------------------------------------------------*/
double eris_pfits_get_exptime(const char * filename)
{
    double exptime ;
    cpl_propertylist* plist = NULL;
    plist = cpl_propertylist_load(filename, 0);
    exptime = cpl_propertylist_get_double(plist, "EXPTIME");
    cpl_propertylist_delete(plist);

    return exptime;
}

/*---------------------------------------------------------------------------*/
/**
  @brief    find out the character string associated to the NAXIS1 keyword
  @param    plist    FITS header
  @return   keyword value
 */
/*---------------------------------------------------------------------------*/
int eris_pfits_get_naxis1(const cpl_propertylist * plist)
{
    return cpl_propertylist_get_int(plist, "NAXIS1");
}


/*---------------------------------------------------------------------------*/
/**
  @brief    find out the character string associated to the NAXIS2 keyword
  @param    plist    FITS header
  @return   keyword value
 */
/*---------------------------------------------------------------------------*/
int eris_pfits_get_naxis2(const cpl_propertylist * plist)
{
    return cpl_propertylist_get_int(plist, "NAXIS2");
}

cpl_propertylist*
eris_ifu_plist_extract_wcs2D(const cpl_propertylist* applist)
{
	cpl_ensure(applist != NULL, CPL_ERROR_NULL_INPUT, NULL);
	cpl_propertylist* result = cpl_propertylist_new();
	double double_value;
	const char* string_value;
	if(cpl_propertylist_has(applist,"CRPIX1")) {
		if (cpl_propertylist_get_type(applist, "CRPIX1") == CPL_TYPE_DOUBLE) {
			double_value = cpl_propertylist_get_double(applist,"CRPIX1");
			cpl_propertylist_append_double(result,"CRPIX1", double_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}
	if(cpl_propertylist_has(applist,"CRPIX2")) {
		if (cpl_propertylist_get_type(applist, "CRPIX2") == CPL_TYPE_DOUBLE) {
			double_value = cpl_propertylist_get_double(applist,"CRPIX2");
			cpl_propertylist_append_double(result,"CRPIX2", double_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}

	if(cpl_propertylist_has(applist,"CRVAL1")) {
		if (cpl_propertylist_get_type(applist, "CRVAL1") == CPL_TYPE_DOUBLE) {
			double_value = cpl_propertylist_get_double(applist,"CRVAL1");
			cpl_propertylist_append_double(result,"CRVAL1", double_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}
	if(cpl_propertylist_has(applist,"CRVAL2")) {
		if (cpl_propertylist_get_type(applist, "CRVAL2") == CPL_TYPE_DOUBLE) {
			double_value = cpl_propertylist_get_double(applist,"CRVAL2");
			cpl_propertylist_append_double(result,"CRVAL2", double_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}

	if(cpl_propertylist_has(applist,"CDELT1")) {
		if (cpl_propertylist_get_type(applist, "CDELT1") == CPL_TYPE_DOUBLE) {
			double_value = cpl_propertylist_get_double(applist,"CDELT1");
			cpl_propertylist_append_double(result,"CDELT1", double_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}
	if(cpl_propertylist_has(applist,"CDELT2")) {
		if (cpl_propertylist_get_type(applist, "CDELT2") == CPL_TYPE_DOUBLE) {
			double_value = cpl_propertylist_get_double(applist,"CDELT2");
			cpl_propertylist_append_double(result,"CDELT2", double_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}

	if(cpl_propertylist_has(applist,"CTYPE1")) {
		if (cpl_propertylist_get_type(applist, "CTYPE1") == CPL_TYPE_STRING) {
			string_value = cpl_propertylist_get_string(applist,"CTYPE1");
			cpl_propertylist_append_string(result,"CTYPE1", string_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}
	if(cpl_propertylist_has(applist,"CTYPE2")) {
		if (cpl_propertylist_get_type(applist, "CTYPE2") == CPL_TYPE_STRING) {
			string_value = cpl_propertylist_get_string(applist,"CTYPE2");
			cpl_propertylist_append_string(result,"CTYPE2", string_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}

	if(cpl_propertylist_has(applist,"CUNIT1")) {
		if (cpl_propertylist_get_type(applist, "CUNIT1") == CPL_TYPE_STRING) {
			string_value = cpl_propertylist_get_string(applist,"CUNIT1");
			cpl_propertylist_append_string(result,"CUNIT1", string_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}
	if(cpl_propertylist_has(applist,"CUNIT2")) {
		if (cpl_propertylist_get_type(applist, "CUNIT2") == CPL_TYPE_STRING) {
			string_value = cpl_propertylist_get_string(applist,"CUNIT2");
			cpl_propertylist_append_string(result,"CUNIT2", string_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}


	if(cpl_propertylist_has(applist,"CD1_1")) {
		if (cpl_propertylist_get_type(applist, "CD1_1") == CPL_TYPE_DOUBLE) {
			double_value = cpl_propertylist_get_double(applist,"CD1_1");
			cpl_propertylist_append_double(result,"CD1_1", double_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}
	if(cpl_propertylist_has(applist,"CD1_2")) {
		if (cpl_propertylist_get_type(applist, "CD1_2") == CPL_TYPE_DOUBLE) {
			double_value = cpl_propertylist_get_double(applist,"CD1_2");
			cpl_propertylist_append_double(result,"CD1_2", double_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}

	if(cpl_propertylist_has(applist,"CD2_1")) {
		if (cpl_propertylist_get_type(applist, "CD2_1") == CPL_TYPE_DOUBLE) {
			double_value = cpl_propertylist_get_double(applist,"CD2_1");
			cpl_propertylist_append_double(result,"CD2_1", double_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}

	if(cpl_propertylist_has(applist,"CD2_2")) {
		if (cpl_propertylist_get_type(applist, "CD2_2") == CPL_TYPE_DOUBLE) {
			double_value = cpl_propertylist_get_double(applist,"CD2_2");
			cpl_propertylist_append_double(result,"CD2_2", double_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}


	return result;
}

cpl_propertylist*
eris_ifu_plist_extract_wcs3D(const cpl_propertylist* applist)
{

	cpl_ensure(applist != NULL, CPL_ERROR_NULL_INPUT, NULL);
	cpl_propertylist* result = cpl_propertylist_new();
	double double_value;
	const char* string_value;

	if(cpl_propertylist_has(applist,"CRPIX3")) {
		if (cpl_propertylist_get_type(applist, "CRPIX3") == CPL_TYPE_DOUBLE) {
			double_value = cpl_propertylist_get_double(applist,"CRPIX3");
			cpl_propertylist_append_double(result,"CRPIX3", double_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}

	if(cpl_propertylist_has(applist,"CRVAL3")) {
		if (cpl_propertylist_get_type(applist, "CRVAL3") == CPL_TYPE_DOUBLE) {
			double_value = cpl_propertylist_get_double(applist,"CRVAL3");
			cpl_propertylist_append_double(result,"CRVAL3", double_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}

	if(cpl_propertylist_has(applist,"CDELT3")) {
		if (cpl_propertylist_get_type(applist, "CDELT3") == CPL_TYPE_DOUBLE) {
			double_value = cpl_propertylist_get_double(applist,"CDELT3");
			cpl_propertylist_append_double(result,"CDELT3", double_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}

	if(cpl_propertylist_has(applist,"CTYPE3")) {
		if (cpl_propertylist_get_type(applist, "CTYPE3") == CPL_TYPE_STRING) {
			string_value = cpl_propertylist_get_string(applist,"CTYPE3");
			cpl_propertylist_append_string(result,"CTYPE3", string_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}

	if(cpl_propertylist_has(applist,"CUNIT3")) {
		if (cpl_propertylist_get_type(applist, "CUNIT3") == CPL_TYPE_STRING) {
			string_value = cpl_propertylist_get_string(applist,"CUNIT3");
			cpl_propertylist_append_string(result,"CUNIT3", string_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}


	if(cpl_propertylist_has(applist,"CD1_3")) {
		if (cpl_propertylist_get_type(applist, "CD1_3") == CPL_TYPE_DOUBLE) {
			double_value = cpl_propertylist_get_double(applist,"CD1_3");
			cpl_propertylist_append_double(result,"CD1_3", double_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}

	if(cpl_propertylist_has(applist,"CD3_1")) {
		if (cpl_propertylist_get_type(applist, "CD3_1") == CPL_TYPE_DOUBLE) {
			double_value = cpl_propertylist_get_double(applist,"CD3_1");
			cpl_propertylist_append_double(result,"CD3_1", double_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}

	if(cpl_propertylist_has(applist,"CD2_3")) {
		if (cpl_propertylist_get_type(applist, "CD2_3") == CPL_TYPE_DOUBLE) {
			double_value = cpl_propertylist_get_double(applist,"CD2_3");
			cpl_propertylist_append_double(result,"CD2_3", double_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}
	if(cpl_propertylist_has(applist,"CD3_2")) {
		if (cpl_propertylist_get_type(applist, "CD3_2") == CPL_TYPE_DOUBLE) {
			double_value = cpl_propertylist_get_double(applist,"CD3_2");
			cpl_propertylist_append_double(result,"CD3_2", double_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}
	if(cpl_propertylist_has(applist,"CD3_3")) {
		if (cpl_propertylist_get_type(applist, "CD3_3") == CPL_TYPE_DOUBLE) {
			double_value = cpl_propertylist_get_double(applist,"CD3_3");
			cpl_propertylist_append_double(result,"CD3_3", double_value);
		} else {
			cpl_error_set(cpl_func,CPL_ERROR_TYPE_MISMATCH);
			return result;
		}
	}



	return result;
}
cpl_propertylist*
eris_ifu_plist_extract_wcs(cpl_propertylist* applist)
{

	cpl_propertylist* result = cpl_propertylist_new();

	cpl_propertylist* tmp = eris_ifu_plist_extract_wcs2D(applist);
	cpl_propertylist_append(result, tmp);
	cpl_propertylist_delete(tmp);

	tmp = eris_ifu_plist_extract_wcs3D(applist);
	cpl_propertylist_append(result, tmp);
	cpl_propertylist_delete(tmp);

	return result;
}

cpl_error_code
eris_pfits_erase_wcs2D(cpl_propertylist* applist)
{

	if(cpl_propertylist_has(applist,"CRPIX1")) {
		cpl_propertylist_erase(applist,"CRPIX1");
	}
	if(cpl_propertylist_has(applist,"CRPIX2")) {
		cpl_propertylist_erase(applist,"CRPIX2");
	}

	if(cpl_propertylist_has(applist,"CRVAL1")) {
		cpl_propertylist_erase(applist,"CRVAL1");
	}
	if(cpl_propertylist_has(applist,"CRVAL2")) {
		cpl_propertylist_erase(applist,"CRVAL2");
	}

	if(cpl_propertylist_has(applist,"CDELT1")) {
		cpl_propertylist_erase(applist,"CDELT1");
	}
	if(cpl_propertylist_has(applist,"CDELT2")) {
		cpl_propertylist_erase(applist,"CDELT2");
	}

	if(cpl_propertylist_has(applist,"CTYPE1")) {
		cpl_propertylist_erase(applist,"CTYPE1");
	}
	if(cpl_propertylist_has(applist,"CTYPE2")) {
		cpl_propertylist_erase(applist,"CTYPE2");
	}

	if(cpl_propertylist_has(applist,"CUNIT1")) {
		cpl_propertylist_erase(applist,"CUNIT1");
	}
	if(cpl_propertylist_has(applist,"CUNIT2")) {
		cpl_propertylist_erase(applist,"CUNIT2");
	}


	if(cpl_propertylist_has(applist,"CD1_1")) {
		cpl_propertylist_erase(applist,"CD1_1");
	}
	if(cpl_propertylist_has(applist,"CD1_2")) {
		cpl_propertylist_erase(applist,"CD1_2");
	}

	if(cpl_propertylist_has(applist,"CD2_1")) {
		cpl_propertylist_erase(applist,"CD2_1");
	}

	if(cpl_propertylist_has(applist,"CD2_2")) {
		cpl_propertylist_erase(applist,"CD2_2");
	}

	return cpl_error_get_code();
}

cpl_error_code
eris_ifu_plist_erase_wcs3D(cpl_propertylist* applist)
{


	if(cpl_propertylist_has(applist,"CRPIX3")) {
		cpl_propertylist_erase(applist,"CRPIX3");
	}
	if(cpl_propertylist_has(applist,"CRVAL3")) {
		cpl_propertylist_erase(applist,"CRVAL3");
	}
	if(cpl_propertylist_has(applist,"CDELT3")) {
		cpl_propertylist_erase(applist,"CDELT3");
	}

	if(cpl_propertylist_has(applist,"CTYPE3")) {
		cpl_propertylist_erase(applist,"CTYPE3");
	}
	if(cpl_propertylist_has(applist,"CUNIT3")) {
		cpl_propertylist_erase(applist,"CUNIT3");
	}


	if(cpl_propertylist_has(applist,"CD1_3")) {
		cpl_propertylist_erase(applist,"CD1_3");
	}
	if(cpl_propertylist_has(applist,"CD3_1")) {
		cpl_propertylist_erase(applist,"CD3_1");
	}
	if(cpl_propertylist_has(applist,"CD2_3")) {
		cpl_propertylist_erase(applist,"CD2_3");
	}
	if(cpl_propertylist_has(applist,"CD3_2")) {
		cpl_propertylist_erase(applist,"CD3_2");
	}
	if(cpl_propertylist_has(applist,"CD3_3")) {
		cpl_propertylist_erase(applist,"CD3_3");
	}



	return cpl_error_get_code();
}

cpl_error_code
eris_ifu_plist_erase_expmap_extra_keys(cpl_propertylist* applist)
{


	if(cpl_propertylist_has(applist,"HDUCLASS")) {
		cpl_propertylist_erase(applist,"HDUCLASS");
	}
	if(cpl_propertylist_has(applist,"HDUDOC")) {
		cpl_propertylist_erase(applist,"HDUDOC");
	}
	if(cpl_propertylist_has(applist,"HDUVERS")) {
		cpl_propertylist_erase(applist,"HDUVERS");
	}

	if(cpl_propertylist_has(applist, "HDUCLAS1")) {
		cpl_propertylist_erase(applist,"HDUCLAS1");
	}
	if(cpl_propertylist_has(applist,"HDUCLAS2")) {
		cpl_propertylist_erase(applist,"HDUCLAS2");
	}


	if(cpl_propertylist_has(applist,"ERRDATA")) {
		cpl_propertylist_erase(applist,"ERRDATA");
	}
	if(cpl_propertylist_has(applist,"QUALDATA")) {
		cpl_propertylist_erase(applist,"QUALDATA");
	}


	return cpl_error_get_code();
}


cpl_error_code
eris_ifu_plist_erase_wcs(cpl_propertylist* applist)
{

	eris_pfits_erase_wcs2D(applist);
	eris_ifu_plist_erase_wcs3D(applist);

	return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
  @brief    find out the i-th raw file name.
  @param    aHeaders       property list/headers to read from
  @param    idx            raw file index
  @return   the requested value or NULL on error

  Queries FITS header for ESO.PRO.REC1.RAW<idx>.NAME
 */
/*----------------------------------------------------------------------------*/
// TODO: why following function reset error code?
const char *
eris_pfits_get_raw_filename(const cpl_propertylist *aHeaders, unsigned int idx)
{
  cpl_ensure(aHeaders, CPL_ERROR_NULL_INPUT, NULL);
  char *key = cpl_sprintf("ESO PRO REC1 RAW%-u NAME", idx);
  cpl_errorstate prestate = cpl_errorstate_get();
  const char *value = cpl_propertylist_get_string(aHeaders, key);
  cpl_errorstate_set(prestate);
  cpl_free(key);
  return value;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    find out the image analysis FWHM from a linear fit (in arcsec)
  @param    aHeaders       property list/headers to read from
  @return   the requested value or 0.0 on error

  Queries FITS header ESO TEL IA FWHMLIN
 */
/*----------------------------------------------------------------------------*/
double
eris_pfits_get_ia_fwhmlin(const cpl_propertylist *aHeaders)
{
  cpl_errorstate prestate = cpl_errorstate_get();
  const double value = cpl_propertylist_get_double(aHeaders, "ESO TEL IA FWHMLIN");
  cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);
  return value;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ambient seeing at start of exposure (in arcsec)
  @param    aHeaders       property list/headers to read from
  @return   the requested value or 0.0 on error

  Queries FITS header ESO TEL AMBI FWHM START
 */
/*----------------------------------------------------------------------------*/
//TODO strange error code setting in case of value < 0
double
eris_pfits_get_fwhm_start(const cpl_propertylist *aHeaders)
{
  cpl_errorstate prestate = cpl_errorstate_get();
  const double value = cpl_propertylist_get_double(aHeaders, "ESO TEL AMBI FWHM START");
  cpl_ensure(cpl_errorstate_is_equal(prestate) && value > 0.,
             cpl_error_get_code(), 0.0);
  return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ambient seeing at end of exposure (in arcsec)
  @param    aHeaders       property list/headers to read from
  @return   the requested value or 0.0 on error

  Queries FITS header ESO TEL AMBI FWHM END
 */
/*----------------------------------------------------------------------------*/
//TODO strange error code setting in case of value < 0
double
eris_pfits_get_fwhm_end(const cpl_propertylist *aHeaders)
{
  cpl_errorstate prestate = cpl_errorstate_get();
  const double value = cpl_propertylist_get_double(aHeaders, "ESO TEL AMBI FWHM END");
  cpl_ensure(cpl_errorstate_is_equal(prestate) && value > 0.,
             cpl_error_get_code(), 0.0);
  return value;
}


/*---------------------------------------------------------------------------*/
/**
  @brief    find out the character string associated to the CDELT3 keyword
  @param    plist    FITS header
  @return   keyword value
 */
/*---------------------------------------------------------------------------*/
double eris_pfits_get_cd33(const cpl_propertylist * plist)
{
    return cpl_propertylist_get_double(plist,"CD3_3");
}


/*---------------------------------------------------------------------------*/
/**
  @brief    find out the character string associated to the CVRVAL3 keyword
  @param    plist    FITS header
  @return   keyword value
 */
/*---------------------------------------------------------------------------*/
double eris_pfits_get_crval3(const cpl_propertylist * plist)
{
    return cpl_propertylist_get_double(plist,"CRVAL3");
}
/*---------------------------------------------------------------------------*/
/**
  @brief    find out the character string associated to the CDELT3 keyword
  @param    plist    FITS header
  @return   keyword value
 */
/*---------------------------------------------------------------------------*/
double eris_pfits_get_cdelt3(const cpl_propertylist * plist)
{
    return cpl_propertylist_get_double(plist,"CDELT3");
}


/*---------------------------------------------------------------------------*/
/**
  @brief    find out the character string associated to the CRPIX3 keyword
  @param    plist    FITS header
  @return   keyword value
 */
/*---------------------------------------------------------------------------*/
double eris_pfits_get_crpix3(const cpl_propertylist * plist)
{
    return cpl_propertylist_get_double(plist,"CRPIX3");
}


/*---------------------------------------------------------------------------*/
/**
  @brief    find out the character string associated to the CRPIX2 keyword
  @param    plist    FITS header
  @return   keyword value
 */
/*---------------------------------------------------------------------------*/
double eris_pfits_get_crpix2(const cpl_propertylist * plist)
{
    return cpl_propertylist_get_double(plist,"CRPIX2");
}

/*---------------------------------------------------------------------------*/
/**
  @brief    find out the character string associated to the CRPIX1 keyword
  @param    plist    FITS header
  @return   keyword value
 */
/*---------------------------------------------------------------------------*/
double eris_pfits_get_crpix1(const cpl_propertylist * plist)
{
    return cpl_propertylist_get_double(plist,"CRPIX1");
}



/*----------------------------------------------------------------------------*/
/**
  @brief    find out the airmass at start of exposure
  @param    aHeaders       property list/headers to read from
  @return   the requested value or 0.0 on error

  Queries FITS header ESO TEL AIRM START
 */
/*----------------------------------------------------------------------------*/
double
eris_pfits_get_airmass_start(const cpl_propertylist *aHeaders)
{
  cpl_errorstate prestate = cpl_errorstate_get();
  const double value = cpl_propertylist_get_double(aHeaders, "ESO TEL AIRM START");
  cpl_ensure(cpl_errorstate_is_equal(prestate) && value > 0., cpl_error_get_code(), 0.0);
  return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the airmass at end of exposure
  @param    aHeaders       property list/headers to read from
  @return   the requested value or 0.0 on error

  Queries FITS header ESO TEL AIRM END
 */
/*----------------------------------------------------------------------------*/
double
eris_pfits_get_airmass_end(const cpl_propertylist *aHeaders)
{
  cpl_errorstate prestate = cpl_errorstate_get();
  const double value = cpl_propertylist_get_double(aHeaders, "ESO TEL AIRM END");
  cpl_ensure(cpl_errorstate_is_equal(prestate) && value > 0., cpl_error_get_code(), 0.0);
  return value;
}


double
eris_pfits_get_airmass(const cpl_propertylist *aHeaders)
{

  return 0.5 * (eris_pfits_get_airmass_start(aHeaders) +
		        eris_pfits_get_airmass_end(aHeaders));
}


/*----------------------------------------------------------------------------*/
/**
  @brief    find out the Julian Date of the observation
  @param    aHeaders       property list/headers to read from
  @return   the requested value or 0.0 on error

  Queries FITS header MJD-OBS
 */
/*----------------------------------------------------------------------------*/
double
eris_pfits_get_mjdobs(const cpl_propertylist *aHeaders)
{
  cpl_errorstate prestate = cpl_errorstate_get();
  const double value = cpl_propertylist_get_double(aHeaders, "MJD-OBS");
  cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);
  return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO program identification
  @param    aHeaders       property list/headers to read from
  @return   the requested value or NULL on error

  Queries FITS header for ESO.OBS.PROG.ID
 */
/*----------------------------------------------------------------------------*/
const char *
eris_pfits_get_progid(const cpl_propertylist *aHeaders)
{
  cpl_errorstate prestate = cpl_errorstate_get();
  const char *value = cpl_propertylist_get_string(aHeaders, "ESO OBS PROG ID");
  cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), NULL);
  return value;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    find out the observation block id
  @param    aHeaders       property list/headers to read from
  @return   the requested value or -1 on error

  Queries FITS header for ESO.OBS.ID
 */
/*----------------------------------------------------------------------------*/
long
eris_pfits_get_obsid(const cpl_propertylist *aHeaders)
{
  cpl_errorstate prestate = cpl_errorstate_get();
  const long value = cpl_propertylist_get_long(aHeaders, "ESO OBS ID");
  cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), -1);
  return value;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    find out the PIPEFILE id
  @param    aHeaders       property list/headers to read from
  @return   the requested value or -1 on error

  Queries FITS header for PIPEFILE
 */
/*----------------------------------------------------------------------------*/
const char*
eris_pfits_get_pipefile(const cpl_propertylist *aHeaders)
{
  cpl_errorstate prestate = cpl_errorstate_get();
  const char* value = NULL;
  if(cpl_propertylist_has(aHeaders, "PIPEFILE")) {
     value = cpl_propertylist_get_string(aHeaders, "PIPEFILE");
  }
  cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), value);
  return value;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    find out the PIPEFILE id
  @param    aHeaders       property list/headers to read from
  @return   the requested value or -1 on error

  Queries FITS header for PIPEFILE
 */
/*----------------------------------------------------------------------------*/
const char*
eris_pfits_get_datamd5(const cpl_propertylist *aHeaders)
{
  cpl_errorstate prestate = cpl_errorstate_get();
  const char* value = NULL;
  if(cpl_propertylist_has(aHeaders, "DATAMD5")) {
     value = cpl_propertylist_get_string(aHeaders, "DATAMD5");
  }
  cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), value);
  return value;
}


cpl_error_code
eris_pfits_clean_header(cpl_propertylist *header, const cpl_boolean clean_extra)
{
  cpl_ensure_code(header != NULL, CPL_ERROR_NULL_INPUT);
  cpl_errorstate prestate = cpl_errorstate_get();

  cpl_propertylist_erase(header,"TELESCOP");
  cpl_propertylist_erase(header,"INSTRUME");
  cpl_propertylist_erase(header,"ARCFILE");
  cpl_propertylist_erase(header,"DATAMD5");
  cpl_propertylist_erase(header,"PIPEFILE");
  cpl_propertylist_erase(header,"OBJECT");
  /* TODO: cannot be removed else fluxcal has problems 
     We probably need to put proper RA,DEC MJD-OBS in 
     primary header of each data cube
  */

  cpl_propertylist_erase(header,"EXPTIME");
  cpl_propertylist_erase(header,"EQUINOX");

  cpl_propertylist_erase(header,"DATE-OBS");
  cpl_propertylist_erase(header,"UTC");
  cpl_propertylist_erase(header,"LST");
  cpl_propertylist_erase(header,"PI-COI");
  cpl_propertylist_erase(header,"OBSERVER");
  cpl_propertylist_erase(header,"RADESYS");
  if(cpl_propertylist_has(header,"PRODCATG")) {
      cpl_propertylist_erase(header,"PRODCATG");
  }
  if(cpl_propertylist_has(header,"COMMENT")) {
	  cpl_propertylist_erase(header,"COMMENT");
  }
  if(clean_extra) {
	  cpl_propertylist_erase_regexp(header, "ESO ADA *", 0);
	  cpl_propertylist_erase_regexp(header, "ESO AOS *", 0);
	  cpl_propertylist_erase_regexp(header, "ESO INS* *", 0);
	  cpl_propertylist_erase_regexp(header, "ESO DET *", 0);
	  cpl_propertylist_erase_regexp(header, "ESO DPR *", 0);
	  cpl_propertylist_erase_regexp(header, "ESO LGS* *", 0);
	  cpl_propertylist_erase_regexp(header, "ESO OCS *", 0);
	  cpl_propertylist_erase_regexp(header, "ESO OBS *", 0);
	  cpl_propertylist_erase_regexp(header, "ESO PRO *", 0);
	  cpl_propertylist_erase_regexp(header, "ESO SEQ *", 0);
	  cpl_propertylist_erase_regexp(header, "ESO TEL *", 0);
	  cpl_propertylist_erase_regexp(header, "ESO TPL *", 0);
  }
  cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), -1);
  return cpl_error_get_code();
}
cpl_error_code
eris_pfits_clean_header_ra_dec_mjd_obs(cpl_propertylist *header)
{
	cpl_propertylist_erase(header,"RA");
	cpl_propertylist_erase(header,"DEC");
	cpl_propertylist_erase(header,"MJD-OBS");
	return cpl_error_get_code();
}
cpl_error_code
eris_pfits_clean_comment(cpl_propertylist *header)
{
	cpl_propertylist_erase(header,"COMMENT");

	return cpl_error_get_code();
}

/**@}*/
