/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <cpl.h>
#include <hdrl.h>
#include <string.h>
#include <eris_ifu_detlin_static.h>
#include <eris_ifu_error.h>
#include <eris_ifu_utils.h>
#include <eris_ifu_dfs.h>

/* Test helper functions */
static cpl_frameset* create_test_frameset(void) {
    cpl_frameset* frameset = cpl_frameset_new();
    
    /* Create test FITS files with known content */
    cpl_image* test_img = cpl_image_new(10, 10, CPL_TYPE_DOUBLE);
    cpl_propertylist* plist = cpl_propertylist_new();
    
    /* Set some test data */
    cpl_image_add_scalar(test_img, 100.0);
    cpl_propertylist_append_double(plist, "ESO DET DIT", 1.0);
    cpl_propertylist_append_string(plist, "ESO DPR TYPE", "LINEARITY_LAMP,ON");
    cpl_propertylist_append_string(plist, "INSTRUME", "ERIS");
    cpl_propertylist_append_string(plist, "ESO SEQ ARM", "SPIFFIER");
    
    /* Save test files */
    cpl_image_save(test_img, "test_on_1.fits", CPL_TYPE_DOUBLE, plist, CPL_IO_CREATE);
    cpl_propertylist_set_string(plist, "ESO DPR TYPE", "LINEARITY_LAMP,OFF");
    cpl_image_save(test_img, "test_off_1.fits", CPL_TYPE_DOUBLE, plist, CPL_IO_CREATE);
    
    /* Add frames to frameset */
    cpl_frame* frame_on = cpl_frame_new();
    cpl_frame* frame_off = cpl_frame_new();
    cpl_frame_set_filename(frame_on, "test_on_1.fits");
    cpl_frame_set_filename(frame_off, "test_off_1.fits");
    cpl_frame_set_tag(frame_on, ERIS_IFU_RAW_LIN);
    cpl_frame_set_tag(frame_off, ERIS_IFU_RAW_LIN);
    cpl_frameset_insert(frameset, frame_on);
    cpl_frameset_insert(frameset, frame_off);
    
    /* Cleanup */
    cpl_image_delete(test_img);
    cpl_propertylist_delete(plist);
    
    return frameset;
}

static void cleanup_test_files(void) {
    remove("test_on_1.fits");
    remove("test_off_1.fits");
}

/* Test cases */

static void test_eris_ifu_detlin_load_frames(void) {
    cpl_frameset* frameset = create_test_frameset();
    hdrl_imagelist* hdrl_imglist_on = NULL;
    hdrl_imagelist* hdrl_imglist_off = NULL;
    cpl_vector* vec_dit_on = NULL;
    cpl_vector* vec_dit_off = NULL;
    
    /* Test normal operation */
    cpl_error_code error = eris_ifu_detlin_load_frames(frameset, 0,
                                                      &hdrl_imglist_on,
                                                      &hdrl_imglist_off,
                                                      &vec_dit_on,
                                                      &vec_dit_off);
    
    /* Verify results */
    cpl_test_eq(error, CPL_ERROR_NONE);
    cpl_test_nonnull(hdrl_imglist_on);
    cpl_test_nonnull(hdrl_imglist_off);
    cpl_test_nonnull(vec_dit_on);
    cpl_test_nonnull(vec_dit_off);
    cpl_test_eq(hdrl_imagelist_get_size(hdrl_imglist_on), 1);
    cpl_test_eq(hdrl_imagelist_get_size(hdrl_imglist_off), 1);
    cpl_test_eq(cpl_vector_get_size(vec_dit_on), 1);
    cpl_test_eq(cpl_vector_get_size(vec_dit_off), 1);
    
    /* Test NULL input handling */
    error = eris_ifu_detlin_load_frames(NULL, 0, &hdrl_imglist_on,
                                       &hdrl_imglist_off, &vec_dit_on,
                                       &vec_dit_off);
    cpl_test_eq(error, CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    hdrl_imagelist_delete(hdrl_imglist_on);
    hdrl_imagelist_delete(hdrl_imglist_off);
    cpl_vector_delete(vec_dit_on);
    cpl_vector_delete(vec_dit_off);
    cpl_frameset_delete(frameset);
    cleanup_test_files();
}

static void test_eris_ifu_detlin_compute_linearity(void) {
    cpl_frameset* frameset = create_test_frameset();
    hdrl_imagelist* imglist = NULL;
    cpl_vector* vec_dit = NULL;
    cpl_parameterlist* parlist = cpl_parameterlist_new();
    cpl_propertylist* qclog = cpl_propertylist_new();
    
    /* Setup test parameters */
    cpl_parameter* param = cpl_parameter_new_value("eris_detlin.degree",
                                                 CPL_TYPE_INT, "Polynomial degree",
                                                 "eris_detlin", 2);
    cpl_parameterlist_append(parlist, param);
    
    /* Load test data */
    hdrl_imagelist* hdrl_imglist_on = NULL;
    hdrl_imagelist* hdrl_imglist_off = NULL;
    cpl_vector* vec_dit_on = NULL;
    cpl_vector* vec_dit_off = NULL;
    eris_ifu_detlin_load_frames(frameset, 0, &hdrl_imglist_on, &hdrl_imglist_off,
                               &vec_dit_on, &vec_dit_off);
    
    /* Test normal operation */
    cpl_image* result = eris_ifu_detlin_compute_linearity(parlist, hdrl_imglist_on,
                                                         vec_dit_on, qclog);
    
    /* Verify results */
    cpl_test_nonnull(result);
    
    /* Test NULL input handling */
    cpl_image* null_result = eris_ifu_detlin_compute_linearity(NULL, hdrl_imglist_on,
                                                              vec_dit_on, qclog);
    cpl_test_null(null_result);
    
    /* Cleanup */
    cpl_image_delete(result);
    hdrl_imagelist_delete(hdrl_imglist_on);
    hdrl_imagelist_delete(hdrl_imglist_off);
    cpl_vector_delete(vec_dit_on);
    cpl_vector_delete(vec_dit_off);
    cpl_parameterlist_delete(parlist);
    cpl_propertylist_delete(qclog);
    cpl_frameset_delete(frameset);
    cleanup_test_files();
}

static void test_eris_ifu_detlin_filter_mask(void) {
    /* Create test mask */
    cpl_mask* test_mask = cpl_mask_new(10, 10);
    cpl_mask_set(test_mask, 5, 5, 1);  /* Set a bad pixel */
    
    /* Create test parameters */
    cpl_parameterlist* parlist = cpl_parameterlist_new();
    cpl_parameter* param_x = cpl_parameter_new_value("eris_ifu_detlin.post-filter-x",
                                                   CPL_TYPE_INT, "Filter X size",
                                                   "eris_ifu_detlin", 3);
    cpl_parameter* param_y = cpl_parameter_new_value("eris_ifu_detlin.post-filter-y",
                                                   CPL_TYPE_INT, "Filter Y size",
                                                   "eris_detlin", 3);
    cpl_parameter* param_mode = cpl_parameter_new_value("eris_ifu_detlin.post-filter-mode",
                                                      CPL_TYPE_STRING, "Filter mode",
                                                      "eris_ifu_detlin", "closing");
    cpl_parameterlist_append(parlist, param_x);
    cpl_parameterlist_append(parlist, param_y);
    cpl_parameterlist_append(parlist, param_mode);
    
    /* Test normal operation */
    cpl_mask* filtered = eris_ifu_detlin_filter_mask(test_mask, parlist);
    cpl_test_nonnull(filtered);
    
    /* Test NULL input handling */
    cpl_mask* null_result = eris_ifu_detlin_filter_mask(NULL, parlist);
    cpl_test_null(null_result);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Test invalid filter mode */
    cpl_parameter_set_string(param_mode, "invalid_mode");
    cpl_mask* invalid_result = eris_ifu_detlin_filter_mask(test_mask, parlist);
    cpl_test_null(invalid_result);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    
    /* Cleanup */
    cpl_mask_delete(test_mask);
    cpl_mask_delete(filtered);
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

static void test_eris_compute_gain(void) {
    cpl_frameset* frameset = create_test_frameset();
    
    /* Test normal operation */
    cpl_table* result = eris_compute_gain(frameset);
    cpl_test_nonnull(result);
    
    /* Verify table structure */
    cpl_test_eq(cpl_table_has_column(result, "adu"), 1);
    cpl_test_eq(cpl_table_has_column(result, "gain"), 1);
    
    /* Test NULL input handling */
    cpl_table* null_result = eris_compute_gain(NULL);
    cpl_test_null(null_result);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_table_delete(result);
    cpl_frameset_delete(frameset);
    cleanup_test_files();
}

int main(void) {
	cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    
    /* Run tests */
    //test_eris_ifu_detlin_load_frames();//TODO fix seg fault
    //test_eris_ifu_detlin_compute_linearity(); //TODO fix seg fault
    test_eris_ifu_detlin_filter_mask();
    test_eris_compute_gain();
    
    return cpl_test_end(0);
}
