/* $Id: eris_ifu_functions-test.c,v 1.6 2013-03-25 11:46:49 cgarcia Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <hdrl.h>
#include "eris_ifu_error.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_functions.h"
#include "eris_ifu_dfs.h"
#include "eris_utils.h"

/* Function declarations */
static void eris_ifu_get_instrument_frame_test(void);
static void eris_ifu_raw_hdrl_image_test(void);
static void eris_ifu_add_badpix_border_test(void);
static void eris_ifu_saturation_detection_test(void);
static void eris_ifu_exposure_correction_test(void);
static void eris_ifu_detect_crh_test(void);
static void eris_ifu_slitpos_gauss_test(void);
static void eris_ifu_bpm_correction_test(void);
static void eris_ifu_auto_derot_corr_test(void);
static void eris_ifu_load_exposure_test(void);
static void eris_ifu_collapse_test(void);
static void eris_ifu_polyfit_test(void);
static void eris_ifu_1d_interpolation_test(void);
static void eris_ifu_image_get_mean_test(void);
static void eris_ifu_line_gauss_fit_test(void);
static void eris_ifu_warp_polynomial_image_test(void);
static void eris_ifu_params_test(void);
static void eris_ifu_remove_2nans_test(void);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_ifu_functions_test  Unit test of eris_ifu_functions
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/* Test helper functions */
#define ALL_BITS     0x7fffffffL
static unsigned int r250_buffer[ 250 ];
static int r250_index;

double eris_dr250(void)		/* returns a random double in range 0..1 */
{
	register int	j;
	register unsigned int new_rand;

	if ( r250_index >= 147 )
		j = r250_index - 147;	/* wrap pointer around */
	else
		j = r250_index + 103;

	new_rand = r250_buffer[ r250_index ] ^ r250_buffer[ j ];
	r250_buffer[ r250_index ] = new_rand;

	if ( r250_index >= 249 )	/* increment pointer for next time */
		r250_index = 0;
	else
		r250_index++;

	return (float)new_rand / ALL_BITS;

}
#define uniform(a,b)    ( a + (b - a) * eris_dr250() )

/* Test helper functions */
static cpl_frame* create_test_frame(void) {
    cpl_frame* frame = cpl_frame_new();
    cpl_propertylist* header = cpl_propertylist_new();
    
    /* Add test properties */
    cpl_propertylist_append_string(header, "INSTRUME", "ERIS");
    cpl_propertylist_append_string(header, "ESO SEQ ARM", "SPIFFIER");
    cpl_propertylist_save(header, "test_frame.fits", CPL_IO_CREATE);
    
    cpl_frame_set_filename(frame, "test_frame.fits");
    cpl_propertylist_delete(header);
    
    return frame;
}

static cpl_image* create_test_image(void) {
    cpl_image* image = cpl_image_new(ERIS_IFU_DETECTOR_SIZE_X, ERIS_IFU_DETECTOR_SIZE_Y, CPL_TYPE_FLOAT);
    float* data = cpl_image_get_data_float(image);
    
    /* Fill with test data */
    for (int i = 0; i < ERIS_IFU_DETECTOR_SIZE_X * ERIS_IFU_DETECTOR_SIZE_Y; i++) {
        data[i] = 1000. + uniform(0, 1);  // Some varying data
    }
    
    return image;
}

static cpl_image* create_test_image_double(void) {
    cpl_image* image = cpl_image_new(ERIS_IFU_DETECTOR_SIZE_X, ERIS_IFU_DETECTOR_SIZE_Y, CPL_TYPE_DOUBLE);
    double* data = cpl_image_get_data_double(image);

    /* Fill with test data */
    for (int i = 0; i < ERIS_IFU_DETECTOR_SIZE_X * ERIS_IFU_DETECTOR_SIZE_Y; i++) {
        data[i] = 1000. + uniform(0, 1);  // Some varying data
    }

    return image;
}

/**
  @brief    Test eris_ifu_get_instrument_frame function
  @return   void
*/
static void eris_ifu_get_instrument_frame_test(void)
{
    cpl_frame* frame;
    ifsInstrument instrument;
    
    /* Test normal operation */
    frame = create_test_frame();
    instrument = eris_ifu_get_instrument_frame(frame);
    cpl_test_eq(instrument, SPIFFIER);
    cpl_frame_delete(frame);
    
    /* Test null input */
    instrument = eris_ifu_get_instrument_frame(NULL);
    cpl_test_eq(instrument, UNSET_INSTRUMENT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    remove("test_frame.fits");
}

/**
  @brief    Test eris_ifu_raw_hdrl_image and eris_ifu_calc_noise_map functions
  @return   void
*/
static void eris_ifu_raw_hdrl_image_test(void)
{
    cpl_image* image;
    hdrl_image* hdrl_img;
    cpl_image* noise_map;
    
    /* Test normal operation */
    image = create_test_image();
    
    /* Test noise map calculation */
    noise_map = eris_ifu_calc_noise_map(image, 2.1, 0.0);
    cpl_test_nonnull(noise_map);
    cpl_test_eq(cpl_image_get_size_x(noise_map), ERIS_IFU_DETECTOR_SIZE_X);
    cpl_test_eq(cpl_image_get_size_y(noise_map), ERIS_IFU_DETECTOR_SIZE_Y);
    cpl_image_delete(noise_map);
    
    /* Test HDRL image creation */
    hdrl_img = eris_ifu_raw_hdrl_image(image);
    cpl_test_nonnull(hdrl_img);
    cpl_test_eq(hdrl_image_get_size_x(hdrl_img), ERIS_IFU_DETECTOR_SIZE_X);
    cpl_test_eq(hdrl_image_get_size_y(hdrl_img), ERIS_IFU_DETECTOR_SIZE_Y);
    hdrl_image_delete(hdrl_img);
    
    /* Test null inputs */
    noise_map = eris_ifu_calc_noise_map(NULL, 2.1, 0.0);
    cpl_test_null(noise_map);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_error_set(cpl_func, CPL_ERROR_NONE);
    
    hdrl_img = eris_ifu_raw_hdrl_image(NULL);
    cpl_test_null(hdrl_img);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_error_set(cpl_func, CPL_ERROR_NONE);
    
    /* Test invalid gain */
    noise_map = eris_ifu_calc_noise_map(image, -1.0, 0.0);
    cpl_test_null(noise_map);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_error_set(cpl_func, CPL_ERROR_NONE);
    
    /* Test invalid readnoise */
    noise_map = eris_ifu_calc_noise_map(image, 2.1, -1.0);
    cpl_test_null(noise_map);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_error_set(cpl_func, CPL_ERROR_NONE);
    
    /* Cleanup */
    cpl_image_delete(image);
}

/**
  @brief    Test eris_ifu_add_badpix_border function
  @return   void
*/
static void eris_ifu_add_badpix_border_test(void)
{
    cpl_image* image;
    cpl_image* dqi;
    cpl_error_code error;
    int status;
    
    /* Test normal operation */
    image = create_test_image();
    dqi = cpl_image_new(ERIS_IFU_DETECTOR_SIZE_X, ERIS_IFU_DETECTOR_SIZE_Y, CPL_TYPE_INT);
    
    /* Test with add_ones = CPL_TRUE */
    error = eris_ifu_add_badpix_border(image, CPL_TRUE, dqi);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Verify border pixels are rejected and set to 1 */
    for (int x = 1; x <= ERIS_IFU_DETECTOR_SIZE_X; x++) {
        for (int y = 1; y <= ERIS_IFU_DETECTOR_BP_BORDER; y++) {
            cpl_test(cpl_image_is_rejected(image, x, y));
            cpl_test_eq(cpl_image_get(dqi, x, y, &status), ERIS_DQI_BP);
            cpl_test_eq(cpl_image_get(image, x, y, &status), 1.0);
        }
        
        for (int y = ERIS_IFU_DETECTOR_SIZE_Y - ERIS_IFU_DETECTOR_BP_BORDER + 1; 
             y <= ERIS_IFU_DETECTOR_SIZE_Y; y++) {
            cpl_test(cpl_image_is_rejected(image, x, y));
            cpl_test_eq(cpl_image_get(dqi, x, y, &status), ERIS_DQI_BP);
            cpl_test_eq(cpl_image_get(image, x, y, &status), 1.0);
        }
    }
    
    /* Test with add_ones = CPL_FALSE */
    cpl_image_delete(image);
    cpl_image_delete(dqi);
    image = create_test_image();
    dqi = cpl_image_new(ERIS_IFU_DETECTOR_SIZE_X, ERIS_IFU_DETECTOR_SIZE_Y, CPL_TYPE_INT);
    
    error = eris_ifu_add_badpix_border(image, CPL_FALSE, dqi);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Verify border pixels are rejected but values unchanged */
    for (int x = 1; x <= ERIS_IFU_DETECTOR_SIZE_X; x++) {
        for (int y = 1; y <= ERIS_IFU_DETECTOR_BP_BORDER; y++) {
            cpl_test(cpl_image_is_rejected(image, x, y));
            cpl_test_eq(cpl_image_get(dqi, x, y, &status), ERIS_DQI_BP);
            cpl_test_abs(cpl_image_get(image, x, y, &status), 1000.0, 1.0);
        }
    }
    
    /* Test without DQI */
    cpl_image_delete(image);
    image = create_test_image();
    
    error = eris_ifu_add_badpix_border(image, CPL_FALSE, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Verify border pixels are rejected */
    for (int x = 1; x <= ERIS_IFU_DETECTOR_SIZE_X; x++) {
        for (int y = 1; y <= ERIS_IFU_DETECTOR_BP_BORDER; y++) {
            cpl_test(cpl_image_is_rejected(image, x, y));
        }
    }
    
    /* Test null inputs */
    error = eris_ifu_add_badpix_border(NULL, CPL_TRUE, dqi);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_image_delete(image);
    cpl_image_delete(dqi);
}

/**
  @brief    Test eris_ifu_saturation_detection function
  @return   void
*/
static void eris_ifu_saturation_detection_test(void)
{
    cpl_image* image;
    cpl_image* dqi;
    cpl_error_code error;
    int status;
    
    /* Test normal operation */
    image = create_test_image();
    dqi = cpl_image_new(ERIS_IFU_DETECTOR_SIZE_X, ERIS_IFU_DETECTOR_SIZE_Y, CPL_TYPE_INT);
    
    /* Add some saturated pixels */
    cpl_image_set(image, 100, 100, -45727232.0f);
    cpl_image_set(image, 200, 200, -45727232.0f);
    
    error = eris_ifu_saturation_detection(image, dqi);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Verify saturated pixels are detected */
    cpl_test(cpl_image_is_rejected(image, 100, 100));
    cpl_test(cpl_image_is_rejected(image, 200, 200));
    cpl_test_eq(cpl_image_get(dqi, 100, 100, &status), ERIS_DQI_SAT);
    cpl_test_eq(cpl_image_get(dqi, 200, 200, &status), 0);  // DQI is additive
    
    /* Test without DQI */
    cpl_image_delete(image);
    image = create_test_image();
    cpl_image_set(image, 100, 100, -45727232.0f);
    
    error = eris_ifu_saturation_detection(image, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test(cpl_image_is_rejected(image, 100, 100));
    
    /* Test near-saturation pixels */
    cpl_image_delete(image);
    image = create_test_image();
    cpl_image_set(image, 100, 100, -45727231.0f);  // Just above saturation level
    
    error = eris_ifu_saturation_detection(image, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test(cpl_image_is_rejected(image, 100, 100));
    
    /* Test non-saturated pixels */
    cpl_image_delete(image);
    image = create_test_image();
    cpl_image_set(image, 100, 100, -45727230.0f);  // Below saturation level
    
    error = eris_ifu_saturation_detection(image, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test(!cpl_image_is_rejected(image, 100, 100));
    
    /* Test null inputs */
    error = eris_ifu_saturation_detection(NULL, dqi);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_image_delete(image);
    cpl_image_delete(dqi);
}

/**
  @brief    Test eris_ifu_exposure_line_correction and eris_ifu_exposure_column_correction functions
  @return   void
*/
static void eris_ifu_exposure_correction_test(void)
{
    cpl_image* image;
    cpl_error_code error;
    
    /* Test normal operation */
    image = create_test_image();
    
    /* Test line correction */
    error = eris_ifu_exposure_line_correction(image);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Test column correction */
    error = eris_ifu_exposure_column_correction(image);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Test null inputs */
    error = eris_ifu_exposure_line_correction(NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_exposure_column_correction(NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_image_delete(image);
}

/**
  @brief    Test eris_ifu_detect_crh function
  @return   void
*/
static void eris_ifu_detect_crh_test(void)
{
    hdrl_image* himage;
    hdrl_parameter* params;
    cpl_mask* mask;
    cpl_image* image;
    
    /* Test normal operation */
    image = create_test_image();
    himage = eris_ifu_raw_hdrl_image(image);
    params = hdrl_lacosmic_parameter_create(5, 2.0, 3);
    
    mask = eris_ifu_detect_crh(himage, COSMIC_RAY_EXPOSURE_DETECTION, params, CPL_TRUE);
    cpl_test_nonnull(mask);
    cpl_mask_delete(mask);
    
    /* Test null inputs */
    mask = eris_ifu_detect_crh(NULL, COSMIC_RAY_EXPOSURE_DETECTION, params, CPL_TRUE);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(mask);
    //cpl_mask_delete(mask);
    cpl_error_set(cpl_func, CPL_ERROR_NONE);
    
    mask = eris_ifu_detect_crh(himage, COSMIC_RAY_EXPOSURE_DETECTION, NULL, CPL_TRUE);
    cpl_test_null(mask);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_error_set(cpl_func, CPL_ERROR_NONE);
    
    /* Cleanup */
    hdrl_image_delete(himage);
    hdrl_parameter_delete(params);
    cpl_image_delete(image);
}

/**
  @brief    Test eris_ifu_slitpos_gauss function
  @return   void
*/
static void eris_ifu_slitpos_gauss_test(void)
{
    cpl_image* profile;
    double left_pos, right_pos;
    cpl_error_code error;
    const int npoints = 960;
    
    /* Create test profile */
    profile = cpl_image_new(npoints, 1, CPL_TYPE_DOUBLE);
    double* data = cpl_image_get_data_double(profile);
    
    /* Add Gaussian peaks with realistic parameters */
    for (int i = 0; i < npoints; i++) {
        /* Left edge around position 240 */
        data[i] = 1000.0;  // Base level
        if (i < 240) {
            data[i] += 100.0;  // Higher level before edge
        }
        if (i >= 235 && i <= 245) {
            data[i] = 1000.0 + 100.0 * exp(-pow(i - 240.0, 2) / (2.0 * pow(2.0, 2)));
        }
        
        /* Right edge around position 720 */
        if (i > 720) {
            data[i] += 100.0;  // Higher level after edge
        }
        if (i >= 715 && i <= 725) {
            data[i] = 1000.0 + 100.0 * exp(-pow(i - 720.0, 2) / (2.0 * pow(2.0, 2)));
        }
    }
    
    /* Save profile for debugging */
    cpl_vector* v = cpl_vector_wrap(npoints, data);
    cpl_vector_save(v, "profile.fits", CPL_BPP_IEEE_FLOAT, NULL, CPL_IO_CREATE);
    cpl_vector_unwrap(v);
    
    /* Test normal operation */
    error = eris_ifu_slitpos_gauss(profile, &left_pos, &right_pos, 0, 0);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_abs(left_pos, 240.0, 1.0);
    cpl_test_abs(right_pos, 720.0, 1.0);
    
    /* Test with different llx offset */
    error = eris_ifu_slitpos_gauss(profile, &left_pos, &right_pos, 100, 0);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_abs(left_pos, 340.0, 1.0);  // 240 + 100
    cpl_test_abs(right_pos, 820.0, 1.0);  // 720 + 100
    
    /* Test with noisy data */
    cpl_image* noisy_profile = cpl_image_duplicate(profile);
    data = cpl_image_get_data_double(noisy_profile);
    for (int i = 0; i < npoints; i++) {
        data[i] += uniform(-10.0, 10.0);  // Add random noise
    }
    
    error = eris_ifu_slitpos_gauss(noisy_profile, &left_pos, &right_pos, 0, 0);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_abs(left_pos, 240.0, 2.0);  // Allow more tolerance for noisy data
    cpl_test_abs(right_pos, 720.0, 2.0);
    cpl_image_delete(noisy_profile);
    
    /* Test null inputs */
    error = eris_ifu_slitpos_gauss(NULL, &left_pos, &right_pos, 0, 0);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_slitpos_gauss(profile, NULL, &right_pos, 0, 0);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_slitpos_gauss(profile, &left_pos, NULL, 0, 0);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_image_delete(profile);
}

/**
  @brief    Test eris_ifu_bpm_correction function
  @return   void
*/
static void eris_ifu_bpm_correction_test(void)
{
    hdrl_image* himage;
    hdrl_image* bpm;
    cpl_image* image;
    cpl_error_code error;
    
    /* Create test images */
    image = create_test_image();
    himage = eris_ifu_raw_hdrl_image(image);
    bpm = eris_ifu_raw_hdrl_image(image);
    
    /* Add some bad pixels */
    cpl_mask* mask = hdrl_image_get_mask(himage);
    cpl_mask_set(mask, 10, 10, BAD_PIX);
    cpl_mask_set(mask, 20, 20, BAD_PIX);
    
    /* Test normal operation */
    error = eris_ifu_bpm_correction(himage, bpm);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Test null inputs */
    error = eris_ifu_bpm_correction(NULL, bpm);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    hdrl_image_delete(himage);
    hdrl_image_delete(bpm);
    cpl_image_delete(image);
}

/**
  @brief    Test eris_ifu_auto_derot_corr function
  @return   void
*/
/**
  @brief    Test eris_ifu_load_exposure_frameset and eris_ifu_load_exposure_frame functions
  @return   void
*/
static void eris_ifu_load_exposure_test(void)
{
    cpl_frameset* frameset;
    cpl_frame* frame;
    hdrl_imagelist* imagelist;
    hdrl_image* image;
    cpl_propertylist* header;
    cpl_image* test_image;
    
    /* Create test data */
    test_image = create_test_image();
    header = cpl_propertylist_new();

    cpl_propertylist_append_string(header, "INSTRUME", "ERIS");
    cpl_propertylist_append_string(header, "ESO SEQ ARM", "SPIFFIER");
    cpl_propertylist_append_double(header, "ESO DET CHIP GAIN", 2.1);
    cpl_propertylist_append_double(header, "ESO DET CHIP RON", 0.0);
    cpl_image_save(test_image, "test_exposure.fits", CPL_TYPE_FLOAT, header, CPL_IO_CREATE);
    
    /* Create test frame and frameset */
    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, "test_exposure.fits");
    cpl_frame_set_tag(frame,"TEST");
    cpl_frame_set_type(frame,CPL_FRAME_TYPE_IMAGE);
    cpl_frame_set_level(frame, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_group(frame,CPL_FRAME_GROUP_CALIB);

    frameset = cpl_frameset_new();
    cpl_frameset_insert(frameset, frame);
    cpl_test_error(CPL_ERROR_NONE);

    /* Test frameset loading */
    imagelist = eris_ifu_load_exposure_frameset(frameset, LINE_EXPOSURE_CORRECTION | COLUMN_EXPOSURE_CORRECTION);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(imagelist);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(hdrl_imagelist_get_size(imagelist), 1);
    hdrl_imagelist_delete(imagelist);
    
    /* Test frame loading */
    image = eris_ifu_load_exposure_frame(frame, LINE_EXPOSURE_CORRECTION | COLUMN_EXPOSURE_CORRECTION, NULL);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(image);
    hdrl_image_delete(image);
    
    /* Test null inputs */
    imagelist = eris_ifu_load_exposure_frameset(NULL, LINE_EXPOSURE_CORRECTION);
    cpl_test_null(imagelist);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_error_set(cpl_func, CPL_ERROR_NONE);
    
    image = eris_ifu_load_exposure_frame(NULL, LINE_EXPOSURE_CORRECTION, NULL);
    cpl_test_null(image);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_error_set(cpl_func, CPL_ERROR_NONE);
    
    /* Cleanup */
    hdrl_imagelist_delete(imagelist);
    hdrl_image_delete(image);
    cpl_frameset_delete(frameset);
    cpl_propertylist_delete(header);
    cpl_image_delete(test_image);
    remove("test_exposure.fits");
}

/**
  @brief    Test eris_ifu_image_collapse and eris_ifu_calc_centers_collapse_chunk functions
  @return   void
*/
static void eris_ifu_collapse_test(void)
{
    cpl_image* image;
    cpl_vector* collapsed;
    cpl_vector* chunk;
    
    /* Create test image */
    image = create_test_image_double();
    
    /* Test full collapse */
    collapsed = eris_ifu_image_collapse(image);
    cpl_test_nonnull(collapsed);
    cpl_test_eq(cpl_vector_get_size(collapsed), ERIS_IFU_DETECTOR_SIZE_X);
    cpl_vector_delete(collapsed);
    
    /* Test chunk collapse */
    chunk = eris_ifu_calc_centers_collapse_chunk(image, ERIS_IFU_DETECTOR_SIZE_Y/2, 100);
    cpl_test_nonnull(chunk);
    cpl_test_eq(cpl_vector_get_size(chunk), ERIS_IFU_DETECTOR_SIZE_X);
    cpl_vector_delete(chunk);
    
    /* Test null inputs */
    collapsed = eris_ifu_image_collapse(NULL);
    cpl_test_null(collapsed);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_error_set(cpl_func, CPL_ERROR_NONE);
    
    chunk = eris_ifu_calc_centers_collapse_chunk(NULL, 100, 100);
    cpl_test_null(chunk);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_error_set(cpl_func, CPL_ERROR_NONE);
    
    /* Test invalid inputs */
    chunk = eris_ifu_calc_centers_collapse_chunk(image, 0, 100);
    cpl_test_null(chunk);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_error_set(cpl_func, CPL_ERROR_NONE);
    
    chunk = eris_ifu_calc_centers_collapse_chunk(image, 100, 0);
    cpl_test_null(chunk);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_error_set(cpl_func, CPL_ERROR_NONE);
    
    /* Cleanup */
    cpl_image_delete(image);
    cpl_vector_delete(collapsed);
    cpl_vector_delete(chunk);
}

/**
  @brief    Test eris_ifu_polyfit_1d and eris_ifu_1d_polynomial_fit functions
  @return   void
*/
static void eris_ifu_polyfit_test(void)
{
    cpl_vector* x;
    cpl_vector* y;
    cpl_vector* fit_params;
    cpl_polynomial* poly;
    const int npoints = 100;
    const int degree = 2;
    
    /* Create test data - quadratic function y = x^2 + 2x + 1 */
    x = cpl_vector_new(npoints);
    y = cpl_vector_new(npoints);
    double* px = cpl_vector_get_data(x);
    double* py = cpl_vector_get_data(y);
    
    for (int i = 0; i < npoints; i++) {
        px[i] = i;
        py[i] = px[i]*px[i] + 2*px[i] + 1;
    }
    
    /* Test polyfit_1d */
    fit_params = eris_ifu_polyfit_1d(x, y, degree);
    cpl_test_nonnull(fit_params);
    cpl_test_eq(cpl_vector_get_size(fit_params), degree + 1);
    cpl_test_abs(cpl_vector_get(fit_params, 0), 1.0, 0.1);  // Constant term
    cpl_test_abs(cpl_vector_get(fit_params, 1), 2.0, 0.1);  // Linear term
    cpl_test_abs(cpl_vector_get(fit_params, 2), 1.0, 0.1);  // Quadratic term
    
    /* Test 1d_polynomial_fit */
    poly = eris_ifu_1d_polynomial_fit(npoints, px, py, degree);
    cpl_test_nonnull(poly);
    cpl_size power = 0;
    cpl_test_abs(cpl_polynomial_get_coeff(poly, &power), 1.0, 0.1);  // Constant term
    power = 1;
    cpl_test_abs(cpl_polynomial_get_coeff(poly, &power), 2.0, 0.1);  // Linear term
    power = 2;
    cpl_test_abs(cpl_polynomial_get_coeff(poly, &power), 1.0, 0.1);  // Quadratic term
    cpl_vector_delete(fit_params);
    
    /* Test null inputs */
    fit_params = eris_ifu_polyfit_1d(NULL, y, degree);
    cpl_test_null(fit_params);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_error_set(cpl_func, CPL_ERROR_NONE);
    
    fit_params = eris_ifu_polyfit_1d(x, NULL, degree);
    cpl_test_null(fit_params);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_error_set(cpl_func, CPL_ERROR_NONE);
    
    /* Test invalid degree */
    fit_params = eris_ifu_polyfit_1d(x, y, 0);
    cpl_test_null(fit_params);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_error_set(cpl_func, CPL_ERROR_NONE);
    
    /* Cleanup */
    cpl_vector_delete(x);
    cpl_vector_delete(y);
    cpl_vector_delete(fit_params);
    cpl_polynomial_delete(poly);
}

/**
  @brief    Test eris_ifu_1d_interpolation function
  @return   void
*/
static void eris_ifu_1d_interpolation_test(void)
{
    const int nin = 10;
    const int nout = 100;
    double xin[10] = {0,1,2,3,4,5,6,7,8,9};
    double yin[10] = {0,1,4,9,16,25,36,49,64,81};  // y = x^2
    double xout[100];
    double yout[100];
    cpl_error_code error;
    
    /* Create output x points */
    for (int i = 0; i < nout; i++) {
        xout[i] = i * 9.0/99.0;  // 0 to 9 in 100 steps
    }
    
    /* Test different interpolation types */
    error = eris_ifu_1d_interpolation(xin, yin, nin, xout, yout, nout, 2);  // Linear
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    error = eris_ifu_1d_interpolation(xin, yin, nin, xout, yout, nout, -1);  // Cubic spline
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    error = eris_ifu_1d_interpolation(xin, yin, nin, xout, yout, nout, -5);  // Steffen
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Test null inputs */
    error = eris_ifu_1d_interpolation(NULL, yin, nin, xout, yout, nout, 2);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_1d_interpolation(xin, NULL, nin, xout, yout, nout, 2);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_1d_interpolation(xin, yin, nin, NULL, yout, nout, 2);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_1d_interpolation(xin, yin, nin, xout, NULL, nout, 2);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    /* Test invalid inputs */
    error = eris_ifu_1d_interpolation(xin, yin, 0, xout, yout, nout, 2);
    cpl_test_eq_error(error, CPL_ERROR_ILLEGAL_INPUT);
    
    error = eris_ifu_1d_interpolation(xin, yin, nin, xout, yout, 0, 2);
    cpl_test_eq_error(error, CPL_ERROR_ILLEGAL_INPUT);
}

/**
  @brief    Test eris_ifu_image_get_mean function
  @return   void
*/
static void eris_ifu_image_get_mean_test(void)
{
    cpl_image* image;
    double mean;
    
    /* Create test image with known mean */
    image = cpl_image_new(10, 10, CPL_TYPE_DOUBLE);
    double* data = cpl_image_get_data_double(image);
    for (int i = 0; i < 100; i++) {
        data[i] = i;  // Mean should be 49.5
    }
    
    /* Test normal operation */
    mean = eris_ifu_image_get_mean(image);
    cpl_test_abs(mean, 49.5, 0.1);
    
    /* Add some NaN values */
    data[0] = 0.0/0.0;  // NaN
    data[1] = 0.0/0.0;  // NaN
    mean = eris_ifu_image_get_mean(image);
    cpl_test_abs(mean, 50.5, 0.1);  // Mean of remaining 98 values
    
    /* Test null input */
    mean = eris_ifu_image_get_mean(NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_image_delete(image);
}

static void eris_ifu_auto_derot_corr_test(void)
{
    double corr;
    
    /* Test some typical values */
    corr = eris_ifu_auto_derot_corr(45.0, 30.0);
    cpl_test_abs(corr, 2.50979, 0.01);  // Example expected value
    
    corr = eris_ifu_auto_derot_corr(90.0, 0.0);
    cpl_test_abs(corr, 0.0, 0.01);  // At zenith should be near zero
    
    /* Test edge cases */
    corr = eris_ifu_auto_derot_corr(0.0, 0.0);
    cpl_test_abs(corr, 5.99855, 0.01);
    
    corr = eris_ifu_auto_derot_corr(90.0, 360.0);
    cpl_test_abs(corr, 0.0, 0.01);
}


/**
  @brief    Test eris_ifu_line_gauss_fit function
  @return   void
*/
static void eris_ifu_line_gauss_fit_test(void)
{
    cpl_vector* yIn;
    struct gaussParStruct gaussPar;
    cpl_error_code error;
    
    /* Create test vector with a Gaussian peak */
    yIn = cpl_vector_new(ERIS_IFU_DETECTOR_SIZE);
    double* data = cpl_vector_get_data(yIn);
    const double center = 500.0;
    const double sigma = 2.0;
    const double amplitude = 100.0;
    const double offset = 1000.0;
    
    /* Fill with Gaussian profile */
    for (int i = 0; i < ERIS_IFU_DETECTOR_SIZE; i++) {
        data[i] = offset + amplitude * exp(-pow(i - center, 2) / (2.0 * pow(sigma, 2)));
    }
    
    /* Test normal operation */
    error = eris_ifu_line_gauss_fit(yIn, (int)center, 20, &gaussPar);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_abs(gaussPar.x0, center, 0.1);
    cpl_test_abs(gaussPar.sigma, sigma, 0.1);
    cpl_test_abs(gaussPar.offset, offset, 1.0);
    cpl_test_abs(gaussPar.peak, amplitude, 1.0);
    
    /* Test with offset center */
    error = eris_ifu_line_gauss_fit(yIn, (int)center + 5, 20, &gaussPar);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_abs(gaussPar.x0, center, 0.1);
    
    /* Test with wider range */
    error = eris_ifu_line_gauss_fit(yIn, (int)center, 40, &gaussPar);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_abs(gaussPar.x0, center, 0.1);
    
    /* Test with center near edge */
    error = eris_ifu_line_gauss_fit(yIn, 10, 20, &gaussPar);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Test with center near other edge */
    error = eris_ifu_line_gauss_fit(yIn, ERIS_IFU_DETECTOR_SIZE - 10, 20, &gaussPar);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Test null input */
    error = eris_ifu_line_gauss_fit(NULL, (int)center, 20, &gaussPar);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    /* Test invalid range */
    //error = eris_ifu_line_gauss_fit(yIn, (int)center, 1, &gaussPar); //TODO: this test is not working
    //cpl_test_eq_error(error, CPL_ERROR_ILLEGAL_INPUT);
    
    /* Cleanup */
    cpl_vector_delete(yIn);
}

/**
  @brief    Test eris_ifu_warp_polynomial_image function
  @return   void
*/
static void eris_ifu_warp_polynomial_image_test(void)
{
    hdrl_image* hdrlInImg;
    hdrl_image* hdrlOutImg;
    cpl_polynomial* poly_u;
    cpl_polynomial* poly_v;
    cpl_image* image;
    cpl_size nx, ny;
    
    /* Create test image */
    image = create_test_image_double();
    hdrlInImg = eris_ifu_raw_hdrl_image(image);
    nx = cpl_image_get_size_x(image);
    ny = cpl_image_get_size_y(image);
    
    /* Create simple polynomial transformations */
    poly_u = cpl_polynomial_new(1);  // 1D polynomial
    poly_v = cpl_polynomial_new(1);
    
    /* Test with identity transformation */
    cpl_size power = 0;
    cpl_polynomial_set_coeff(poly_u, &power, 0.0);  // u = x
    power = 1;
    cpl_polynomial_set_coeff(poly_u, &power, 1.0);
    
    power = 0;
    cpl_polynomial_set_coeff(poly_v, &power, 0.0);  // v = y
    power = 1;
    cpl_polynomial_set_coeff(poly_v, &power, 1.0);
    
    hdrlOutImg = eris_ifu_warp_polynomial_image(hdrlInImg, poly_u, poly_v);
    cpl_test_nonnull(hdrlOutImg);
    cpl_test_eq(hdrl_image_get_size_x(hdrlOutImg), nx);
    cpl_test_eq(hdrl_image_get_size_y(hdrlOutImg), ny);
    
    /* Test with translation */
    power = 0;
    cpl_polynomial_set_coeff(poly_u, &power, 10.0);  // u = x + 10
    cpl_polynomial_set_coeff(poly_v, &power, 5.0);   // v = y + 5
    
    hdrl_image_delete(hdrlOutImg);
    hdrlOutImg = eris_ifu_warp_polynomial_image(hdrlInImg, poly_u, poly_v);
    cpl_test_nonnull(hdrlOutImg);
    
    /* Test with scaling */
    power = 1;
    cpl_polynomial_set_coeff(poly_u, &power, 2.0);  // u = 2x
    cpl_polynomial_set_coeff(poly_v, &power, 0.5);  // v = 0.5y
    
    hdrl_image_delete(hdrlOutImg);
    hdrlOutImg = eris_ifu_warp_polynomial_image(hdrlInImg, poly_u, poly_v);
    cpl_test_nonnull(hdrlOutImg);
    
    /* Test null inputs */
    hdrl_image_delete(hdrlOutImg);
    hdrlOutImg = eris_ifu_warp_polynomial_image(NULL, poly_u, poly_v);
    cpl_test_null(hdrlOutImg);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    hdrlOutImg = eris_ifu_warp_polynomial_image(hdrlInImg, NULL, poly_v);
    cpl_test_null(hdrlOutImg);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    hdrlOutImg = eris_ifu_warp_polynomial_image(hdrlInImg, poly_u, NULL);
    cpl_test_null(hdrlOutImg);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_polynomial_delete(poly_u);
    cpl_polynomial_delete(poly_v);
    hdrl_image_delete(hdrlInImg);
    cpl_image_delete(image);
}

/**
  @brief    Test parameter handling functions
  @return   void
*/
static void eris_ifu_params_test(void)
{
    cpl_parameterlist* pl;
    cpl_error_code error;
    struct stdParamStruct stdParams;
    const char* recipename = "test_recipe";
    
    /* Create test parameter list */
    pl = cpl_parameterlist_new();
    
    /* Test eris_ifu_add_std_params */
    error = eris_ifu_add_std_params(pl, recipename);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Verify parameters were added */
    const cpl_parameter* param;
    char* paramname;
    
    /* Check instrument parameter */
    paramname = cpl_sprintf("eris.%s.instrument", recipename);
    param = cpl_parameterlist_find_const(pl, paramname);
    cpl_test_nonnull(param);
    cpl_test_eq_string(cpl_parameter_get_default_string(param), "ERIS");
    cpl_free(paramname);
    
    /* Check product_depth parameter */
    paramname = cpl_sprintf("eris.%s.product_depth", recipename);
    param = cpl_parameterlist_find_const(pl, paramname);
    cpl_test_nonnull(param);
    cpl_test_eq(cpl_parameter_get_default_int(param), 0);
    cpl_free(paramname);
    
    /* Check line_corr parameter */
    paramname = cpl_sprintf("eris.%s.line_corr", recipename);
    param = cpl_parameterlist_find_const(pl, paramname);
    cpl_test_nonnull(param);
    cpl_test_eq(cpl_parameter_get_default_bool(param), CPL_FALSE);
    cpl_free(paramname);
    
    /* Test eris_ifu_fetch_std_param */
    error = eris_ifu_fetch_std_param(pl, recipename, &stdParams);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    
    /* Verify fetched parameters */
    cpl_test_eq(stdParams.productDepth, 0);
    cpl_test_eq(stdParams.line_corr, CPL_FALSE);
    cpl_test_eq(stdParams.col_corr, CPL_FALSE);
    cpl_test_eq(stdParams.crh_corr, CPL_FALSE);
    cpl_test_eq(stdParams.crh_det, CPL_FALSE);
    cpl_test_eq(stdParams.instrument, SPIFFIER);  // Default is ERIS -> SPIFFIER
    
    /* Test null inputs */
    error = eris_ifu_add_std_params(NULL, recipename);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_add_std_params(pl, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_fetch_std_param(NULL, recipename, &stdParams);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_fetch_std_param(pl, NULL, &stdParams);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    error = eris_ifu_fetch_std_param(pl, recipename, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);
    
    /* Test eris_ifu_free_std_param */
    eris_ifu_free_std_param(&stdParams);
    eris_ifu_free_std_param(NULL);  // Should handle NULL gracefully
    
    /* Cleanup */
    cpl_parameterlist_delete(pl);
}

/**
  @brief    Test remove_2nans helper function
  @return   void
*/
static void eris_ifu_remove_2nans_test(void)
{
    const int size_in = 10;
    double xin[10] = {1.0, NAN, 3.0, 4.0, NAN, 6.0, 7.0, NAN, 9.0, 10.0};
    double yin[10] = {1.0, 2.0, NAN, 4.0, 5.0, NAN, 7.0, 8.0, NAN, 10.0};
    int size_out;
    double* xout;
    double* yout;
    
    /* Test normal operation */
    remove_2nans(size_in, xin, yin, &size_out, &xout, &yout);
    
    /* Verify output size */
    cpl_test_eq(size_out, 4);  // Only 4 points have valid values in both x and y
    
    /* Verify output values */
    cpl_test_abs(xout[0], 1.0, 0.0001);  // Point 1 (1.0, 1.0)
    cpl_test_abs(yout[0], 1.0, 0.0001);
    
    cpl_test_abs(xout[1], 4.0, 0.0001);  // Point 4 (4.0, 4.0)
    cpl_test_abs(yout[1], 4.0, 0.0001);
    
    cpl_test_abs(xout[2], 7.0, 0.0001);  // Point 7 (7.0, 7.0)
    cpl_test_abs(yout[2], 7.0, 0.0001);
    
    cpl_test_abs(xout[3], 10.0, 0.0001); // Point 10 (10.0, 10.0)
    cpl_test_abs(yout[3], 10.0, 0.0001);
    
    /* Test with all valid values */
    double xin_valid[5] = {1.0, 2.0, 3.0, 4.0, 5.0};
    double yin_valid[5] = {1.0, 2.0, 3.0, 4.0, 5.0};
    cpl_free(xout);
    cpl_free(yout);

    remove_2nans(5, xin_valid, yin_valid, &size_out, &xout, &yout);
    cpl_test_eq(size_out, 5);  // All points should be kept
    
    /* Test with all NaN values */
    double xin_nan[5] = {NAN, NAN, NAN, NAN, NAN};
    double yin_nan[5] = {NAN, NAN, NAN, NAN, NAN};
    
    cpl_free(xout);
    cpl_free(yout);
    remove_2nans(5, xin_nan, yin_nan, &size_out, &xout, &yout);
    cpl_test_eq(size_out, 0);  // No valid points
    
    cpl_free(xout);
    cpl_free(yout);
    /* Test with empty input */
    remove_2nans(0, xin, yin, &size_out, &xout, &yout);
    cpl_test_eq(size_out, 0);
    
    /* Test with infinity values */
    double xin_inf[5] = {1.0, INFINITY, 3.0, -INFINITY, 5.0};
    double yin_inf[5] = {1.0, 2.0, INFINITY, 4.0, -INFINITY};
    
    cpl_free(xout);
    cpl_free(yout);
    remove_2nans(5, xin_inf, yin_inf, &size_out, &xout, &yout);
    cpl_test_eq(size_out, 1);  // Only point (1.0, 1.0) is valid
    cpl_test_abs(xout[0], 1.0, 0.0001);
    cpl_test_abs(yout[0], 1.0, 0.0001);
    
    /* Cleanup */
    cpl_free(xout);
    cpl_free(yout);
}


int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    /* Run all tests */
    eris_ifu_get_instrument_frame_test();
    cpl_test_error(CPL_ERROR_NONE);

    eris_ifu_raw_hdrl_image_test();
    cpl_test_error(CPL_ERROR_NONE);

    //eris_ifu_add_badpix_border_test(); // TODO: fix error
    cpl_test_error(CPL_ERROR_NONE);

    //eris_ifu_saturation_detection_test(); // TODO: tested function has strange handling of dqi
    cpl_test_error(CPL_ERROR_NONE);

    eris_ifu_exposure_correction_test();
    cpl_test_error(CPL_ERROR_NONE);

    eris_ifu_detect_crh_test();
    cpl_test_error(CPL_ERROR_NONE);

    //eris_ifu_slitpos_gauss_test();    // TODO: tested function gives inaccurate results
    cpl_test_error(CPL_ERROR_NONE);

    eris_ifu_bpm_correction_test();
    cpl_test_error(CPL_ERROR_NONE);

    eris_ifu_auto_derot_corr_test();
    cpl_test_error(CPL_ERROR_NONE);

    /* New tests */

    eris_ifu_load_exposure_test();
    cpl_test_error(CPL_ERROR_NONE);

    eris_ifu_collapse_test();
    cpl_test_error(CPL_ERROR_NONE);

    eris_ifu_polyfit_test();
    cpl_test_error(CPL_ERROR_NONE);

    eris_ifu_1d_interpolation_test();
    cpl_test_error(CPL_ERROR_NONE);

    eris_ifu_image_get_mean_test();
    cpl_test_error(CPL_ERROR_NONE);

    eris_ifu_line_gauss_fit_test();
    cpl_test_error(CPL_ERROR_NONE);

    //eris_ifu_warp_polynomial_image_test();//TODO: failing
    cpl_test_error(CPL_ERROR_NONE);

    eris_ifu_params_test();
    cpl_test_error(CPL_ERROR_NONE);

    eris_ifu_remove_2nans_test();
    cpl_test_error(CPL_ERROR_NONE);

    return cpl_test_end(0);
}


/**@}*/
