/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author$
 * $Date$
 * $Revision$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include <eris_nix_dfs.h>
#include <eris_nix_gain_linearity.h>
#include <eris_nix_utils.h>
#include "eris_nix_test_defs.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_gain_linearity_test  Unit test of eris_nix_gain_linearity
 *
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                       Prototypes
 -----------------------------------------------------------------------------*/

static void eris_nix_gain_linearity_test(void);

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Textual representation of CPL frame group
  @param    group     to convert
  @return   textual representation
 */
/*----------------------------------------------------------------------------*/
static void eris_nix_gain_linearity_test(void)
{
    cpl_mask           * bpm = NULL;
    gain_linearity     * check = NULL;
    cpl_image          * coeff0 = NULL;
    cpl_image          * coeff1 = NULL;
    cpl_image          * coeff2 = NULL;
    cpl_image          * coeff3 = NULL;
    cpl_imagelist      * coeffs_imagelist = NULL;
    cpl_image          * data = NULL;
    cpl_image          * error = NULL;
    cpl_frameset       * frameset = NULL;
    cpl_frame          * frame1 = NULL;
    cpl_frame          * frame2 = NULL;
    cpl_frame          * frame3 = NULL;
    cpl_frame          * frametest = NULL;
    cpl_propertylist   * lin_plist = NULL;
    cpl_propertylist   * plist = NULL;
    cpl_image          * testconf = NULL;
    hdrl_image         * testimg = NULL;
    located_imagelist  * test_limlist = NULL;
    cpl_frameset       * used = NULL;

    /* Create files containing demo data */

    /* Coeffs */

    coeffs_imagelist = cpl_imagelist_new();
    coeff0 = cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(coeff0, 1, 1, TEST_SIZE_X, TEST_SIZE_Y, 3.0);
    cpl_imagelist_set(coeffs_imagelist, coeff0, 0);
    coeff1 = cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(coeff1, 1, 1, TEST_SIZE_X, TEST_SIZE_Y, 1000.0);
    cpl_imagelist_set(coeffs_imagelist, coeff1, 1);
    coeff2 = cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(coeff2, 1, 1, TEST_SIZE_X, TEST_SIZE_Y, -5.0);
    cpl_imagelist_set(coeffs_imagelist, coeff2, 2);
    coeff3 = cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(coeff3, 1, 1, TEST_SIZE_X, TEST_SIZE_Y, -1.0);
    cpl_imagelist_set(coeffs_imagelist, coeff3, 3);
    lin_plist = cpl_propertylist_new();
    cpl_imagelist_save(coeffs_imagelist, "nix_test_lin_coeffs_cube.fits",
                       CPL_TYPE_DOUBLE, lin_plist, CPL_IO_CREATE);

    /* BPM */

    bpm = cpl_mask_new(TEST_SIZE_X, TEST_SIZE_Y);
    cpl_mask_save(bpm, "nix_test_lin_bpm.fits", lin_plist, CPL_IO_CREATE);

    /* Gain file */
    cpl_propertylist_append_double(lin_plist, "ESO QC GAIN", 10.0);
    cpl_propertylist_append_double(lin_plist, "ESO QC GAIN ERR", 1.0);
    cpl_propertylist_append_string(lin_plist, "ESO PRO REC1 PARAM25 NAME",
                                   "saturation_limit");
    cpl_propertylist_append_string(lin_plist, "ESO PRO REC1 PARAM25 VALUE",
                                   "50000");
    cpl_table* gain_tab = cpl_table_new(1);
    cpl_table_save (gain_tab, lin_plist, NULL, "nix_test_gain_table.fits", CPL_IO_CREATE);
    cpl_table_delete(gain_tab);
    /* Read in gain linearity info from demo files */

    frameset  = cpl_frameset_new();
    frame1 = cpl_frame_new();
    cpl_frame_set_filename(frame1, "nix_test_lin_coeffs_cube.fits");
    cpl_frame_set_tag(frame1, "LIN_COEFFS");
    cpl_frame_set_group(frame1, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame1, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(frame1, CPL_FRAME_TYPE_IMAGE);
    cpl_frameset_insert(frameset, frame1);

    used = cpl_frameset_new();

    /* does it behave when some files missing from frameset? */

    check = engl_gain_linearity_load_from_frameset(frameset,
            "GAIN", "LIN_COEFFS", "LIN_BPM", 0, used);
    cpl_test(!check);
    cpl_test_error(CPL_ERROR_NONE);

    check = engl_gain_linearity_load_from_frameset(NULL,
            "GAIN", "LIN_COEFFS", "LIN_BPM", 1, used);
    cpl_test(!check);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    check = engl_gain_linearity_load_from_frameset(frameset,
            "GAIN", "LIN_COEFFS", "LIN_BPM", 1, used);
    cpl_test(!check);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);

    /* try it with all files present */

    frame2 = cpl_frame_new();
    cpl_frame_set_filename(frame2, "nix_test_lin_bpm.fits");
    cpl_frame_set_tag(frame2, "LIN_BPM");
    cpl_frame_set_group(frame2, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame2, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(frame2, CPL_FRAME_TYPE_IMAGE);
    cpl_frameset_insert(frameset, frame2);

    frame3 = cpl_frame_new();
    cpl_frame_set_filename(frame3, "nix_test_gain_table.fits");
    cpl_frame_set_tag(frame3, "GAIN_TABLE");
    cpl_frame_set_group(frame3, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame3, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(frame3, CPL_FRAME_TYPE_IMAGE);
    cpl_frameset_insert(frameset, frame3);

    check = engl_gain_linearity_load_from_frameset(frameset,
            "GAIN_TABLE", "LIN_COEFFS", "LIN_BPM", 1, used);
    cpl_test(check);
    cpl_test_error(CPL_ERROR_NONE);

    /* create a test located_imagelist */

    data = cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(data, 1, 1, TEST_SIZE_X, TEST_SIZE_Y, 2000.0);
    error = cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(error, 1, 1, TEST_SIZE_X, TEST_SIZE_Y, 40.0);

    testimg = hdrl_image_create(data, error);
    testconf = cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(testconf, 1, 1, TEST_SIZE_X, TEST_SIZE_Y, 5.0);
    plist = cpl_propertylist_new();
    cpl_propertylist_prepend_double(plist, "ESO DET NDIT", 9.0);
    cpl_propertylist_prepend_double(plist, "ESO DET DIT", 15.0);
    cpl_propertylist_prepend_string(plist, "ESO DET READ CURNAME",
                                    "SLOW_UP_THE_RAMP");

    test_limlist = enu_located_imagelist_new(TEST_LIST_SIZE);

    /* located_image takes ownership of all input objects */ 

    for (cpl_size i=0; i<TEST_LIST_SIZE; i++) {
        /* create a new located_image each time as the located_imagelist
           object will try to delete them all when the time comes */ 
        hdrl_image * dupimg = hdrl_image_duplicate(testimg);
        cpl_image * dupconf = cpl_image_duplicate(testconf);
        cpl_propertylist * dup_plist = cpl_propertylist_duplicate(plist);
        frametest = cpl_frame_new();
        char * fname = cpl_sprintf("test%d", (int) i); 
        cpl_frame_set_filename(frametest, fname);
        cpl_frame_set_tag(frametest, "JITTER");
        cpl_frame_set_group(frametest, CPL_FRAME_GROUP_RAW);
        cpl_frame_set_level(frametest, CPL_FRAME_LEVEL_NONE);
        cpl_frame_set_type(frametest, CPL_FRAME_TYPE_IMAGE);
        cpl_frameset_insert(frameset, frametest);
        cpl_frame * dupframe = cpl_frame_duplicate(frametest);
        located_image * newlimg = enu_located_image_new(dupimg,
                                                        NULL,
                                                        dupconf,
                                                        NULL, NULL,
                                                        dup_plist,
                                                        NULL, NULL, NULL, NULL,
                                                        dupframe);
        enu_located_imagelist_insert(test_limlist, newlimg, i); 
        cpl_free(fname);
    }

    /* test engl_lin_find */

    double coeffs[4] = {0.0, 1000.0, -5.0, -1.0};
    double t = engl_lin_find(2000.0, 0.9 * 2000.0 / coeffs[1],
                             1.5 * 2000.0 / coeffs[1],
                             4, coeffs);
    double linval = 0.0 + 1000.0 * t - 5.0 * pow(t,2.0) - pow(t,3.0);
    cpl_test_lt(fabs(linval - 2000.0), MAX_LIN_DIFFERENCE);

    /* .. and engl_lin_correct */

    cpl_binary rejected = 0;
    cpl_binary saturated = 0;
    double corrected = engl_lin_correct(2000.0, 4, coeffs, 
                                        CPL_BINARY_0, 50000.0,
                                        &rejected,
                                        &saturated,
                                        CPL_FALSE);
    cpl_test_eq(rejected, 0);
    cpl_test_eq(saturated, 0);
    double correction = (1000.0 * t) / (0.0 + 1000.0 * t - 5.0
                        * pow(t,2.0) - pow(t,3.0));
    cpl_msg_info(cpl_func, "corrected %20.14f", corrected);
    cpl_test_rel(corrected, 2000.0 * correction, 1.0e-10);

    /* tidy up */

    cpl_mask_delete(bpm);
    cpl_image_delete(data);
    engl_gain_linearity_delete(check);
    cpl_imagelist_delete(coeffs_imagelist);
    cpl_propertylist_delete(lin_plist);
    cpl_image_delete(error);
    cpl_frameset_delete(frameset);
    cpl_propertylist_delete(plist);
    cpl_image_delete(testconf);
    hdrl_image_delete(testimg);
    enu_located_imagelist_delete(test_limlist);
    cpl_frameset_delete(used);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_nix_gain_linearity module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    eris_nix_gain_linearity_test();

    return cpl_test_end(0);

}

/**@}*/
