/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author$
 * $Date$
 * $Revision$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include <eris_nix_dfs.h>
#include <eris_nix_master_dark.h>
#include <eris_nix_utils.h>
#include "eris_nix_test_defs.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_master_dark_test  Unit test of eris_nix_master_dark
 *
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                       Prototypes
 -----------------------------------------------------------------------------*/

static void eris_nix_master_dark_test(void);

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Textual representation of CPL frame group
  @param    group     to convert
  @return   textual representation
 */
/*----------------------------------------------------------------------------*/
static void eris_nix_master_dark_test(void)
{
    master_dark *      check_dark = NULL;
    cpl_image *        dark_data = NULL;
    cpl_image *        dark_error = NULL;
    cpl_frame *        frame1 = NULL;
    cpl_frame *        frame2 = NULL;
    cpl_frameset *     frameset = NULL;
    master_dark *      good_dark = NULL;
    mef_extension_list* mef_extensions = NULL;
    cpl_parameterlist* parlist = NULL;
    cpl_propertylist * plist = NULL;
    cpl_image *        rawdark = NULL;
    cpl_propertylist * rawdark_plist = NULL;
    cpl_image *        testconf = NULL;
    hdrl_image *       testdark = NULL;
    cpl_mask *         testhotbpm = NULL;
    cpl_frameset *     used = NULL;

    good_dark = en_master_dark_create(NULL, NULL, NULL, NULL); 
    cpl_test(!good_dark);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* make test master dark */
  
    good_dark = en_master_dark_test(TEST_SIZE_X, TEST_SIZE_Y, 15.0, 3,
                                    "SLOW_UP_THE_RAMP");
    /* save dark to FITS file */

    mef_extensions = enu_mef_extension_list_new(1);
    mef_extensions->mef[0] = enu_mef_new_mask("HOT_BPM", good_dark->hot_bpm,
                                              NULL);

    frameset  = cpl_frameset_new();
    frame1 = cpl_frame_new();
    rawdark =  cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    rawdark_plist = cpl_propertylist_new();
    cpl_image_save(rawdark, "nix_test_raw_dark.fits", CPL_TYPE_DOUBLE,
                   rawdark_plist, CPL_IO_CREATE);
    cpl_frame_set_filename(frame1, "nix_test_raw_dark.fits");
    cpl_frame_set_group(frame1, CPL_FRAME_GROUP_RAW);
    cpl_frame_set_level(frame1, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_tag(frame1, "DARK");
    cpl_frame_set_type(frame1, CPL_FRAME_TYPE_IMAGE);
    cpl_frameset_insert(frameset, frame1);

    parlist = cpl_parameterlist_new();

    /* Save the master dark to a DFS-compliant MEF file */

    cpl_msg_info(cpl_func, "before %s", cpl_error_get_message());

    enu_dfs_save_himage(frameset,
                        parlist,
                        frameset,
                        CPL_TRUE,
                        good_dark->dark,
                        NULL,
                        mef_extensions,
                        "eris_nix_dark",
                        frame1,
                        good_dark->plist, 
                        NULL,
                        PACKAGE "/" PACKAGE_VERSION,
                        "nix_test_master_dark.fits");
    cpl_test_error(CPL_ERROR_NONE);

    /* Read it in again and check that it matches what was written out */

    frame2 = cpl_frame_new();
    cpl_frame_set_filename(frame2, "nix_test_master_dark.fits");
    cpl_frame_set_tag(frame2, "MASTER_DARK");
    cpl_frame_set_group(frame2, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame2, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_type(frame2, CPL_FRAME_TYPE_IMAGE);
    cpl_frameset_insert(frameset, frame2);

    used = cpl_frameset_new();
    check_dark = en_master_dark_load_from_frameset(frameset,
      "MASTER_DARK", used);

    hdrl_value val = hdrl_image_get_pixel(
      check_dark->dark, 10, 10, NULL);
    cpl_test_leq(fabs(val.data - 7.0), DBL_EPSILON);
    cpl_test_leq(fabs(val.error - 3.0), DBL_EPSILON);

    const char * string1 = cpl_propertylist_get_string(
      check_dark->plist, CPL_DFS_PRO_CATG);
    cpl_test_eq_string(string1, ERIS_NIX_MASTER_DARK_IMG_PRO_CATG);

    double dtest = cpl_propertylist_get_double(
      check_dark->plist, "ESO DET DIT");
    cpl_test_abs(dtest, 15.0, DBL_EPSILON);

    const char * string2 = cpl_propertylist_get_string(
      check_dark->plist, "ESO DET READ CURNAME");
    cpl_test_eq_string(string2, "SLOW_UP_THE_RAMP");

    dtest = cpl_propertylist_get_double(
      check_dark->plist, "ESO QC READ NOISE");
    cpl_test_abs(dtest, 6.2, DBL_EPSILON);

    dtest = cpl_propertylist_get_double(
      check_dark->plist, "ESO QC READ NOISE VAR");
    cpl_test_abs(dtest, 6.3, DBL_EPSILON);

    hdrl_value dark_median = hdrl_image_get_median(check_dark->dark);
    dtest = cpl_propertylist_get_double(
      check_dark->plist, "ESO QC DARK MED");
    cpl_test_abs(dtest, (double) dark_median.data, DBL_EPSILON);

    hdrl_value dark_mean = hdrl_image_get_mean(check_dark->dark);
    dtest = cpl_propertylist_get_double(
      check_dark->plist, "ESO QC DARK MEAN");
    cpl_test_abs(dtest, (double) dark_mean.data, DBL_EPSILON);

    double dark_rms = hdrl_image_get_stdev(check_dark->dark);
    dtest = cpl_propertylist_get_double(
      check_dark->plist, "ESO QC DARK RMS");
    cpl_test_abs(dtest, (double) dark_rms, DBL_EPSILON);

    cpl_size nhot = cpl_mask_count(check_dark->hot_bpm); 
    int itest = cpl_propertylist_get_int(
      check_dark->plist, "ESO QC NUMBER HOT PIXEL");
    cpl_test_abs(itest, (int) nhot, 0);

    dtest = cpl_propertylist_get_double(
      check_dark->plist, "ESO QC PARTICLE_RATE"); 
    cpl_test_abs(dtest, -1.0, DBL_EPSILON);

    /* tidy up */

    en_master_dark_delete(check_dark);
    cpl_image_delete(dark_data);
    cpl_image_delete(dark_error);
    cpl_frameset_delete(frameset);
    en_master_dark_delete(good_dark);
    enu_mef_extension_list_delete(mef_extensions);
    cpl_parameterlist_delete(parlist);
    cpl_propertylist_delete(plist);
    cpl_image_delete(rawdark);
    cpl_propertylist_delete(rawdark_plist);
    cpl_image_delete(testconf);
    hdrl_image_delete(testdark);
    cpl_mask_delete(testhotbpm);
    cpl_frameset_delete(used);
    remove("nix_test_raw_dark.fits");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_nix_master_dark module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    eris_nix_master_dark_test();

    return cpl_test_end(0);

}

/**@}*/
