/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author$
 * $Date$
 * $Revision$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

//#include <libgen.h>
//#include <math.h>
/*#include <regex.h>*/
#include <string.h>

//#include "casu_utils.h"
//#include "casu_mods.h"
/*#include "casu_getstds.c"
*/

//#include "eris_utils.h"
//#include "eris_nix_utils.h"
//#include "eris_pfits.h"
//#include "eris_dfs.h"
#include "eris_nix_dfs.h"
//#include "eris_nix_catalogue.h"
//#include "eris_nix_master_bpm.h"
//#include "eris_nix_master_dark.h"
//#include "eris_nix_match.h"
//#include "eris_nix_gain_linearity.h"
//#include <hdrl.h>
#include <eris_nix_scired.h>
#include <cpl.h>

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

static const char eris_nix_img_skysub_description[] =
"This recipe estimates and subtracts the sky background from a set of \n"
ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG" or "
ERIS_NIX_CAL_DET_STD_JITTER_PRO_CATG" frames.\n"
"\n"
"Input files:\n"
"\n"
"  The recipe has 2 modes of operation.\n"
"  If --sky-source='target' then the sky background will be estimated \n"
"  from the target frames themselves.\n"
"\n"
"  DO CATG                   Explanation             Req.  #Frames\n"
"  -------                   -----------             ---   -------\n"
"  "ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG
                       "     target frames with the   Y      1-n\n"
"             or             detector signature\n"
"  "ERIS_NIX_CAL_DET_STD_JITTER_PRO_CATG
                    "        removed.\n"
"\n"
"  If --sky-source='offset' and for "
ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG"\n"
"  data only, the sky background will be estimated from a set of \n"
"  frames offset from the target region.\n"
"\n"
"  DO CATG                   Explanation             Req.  #Frames\n"
"  -------                   -----------             ---   -------\n"
"  "ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG
                       "     target frames with the   Y      1-n\n"
"                            detector signature\n"
"                            removed.\n"
"  "ERIS_NIX_CAL_DET_SKY_JITTER_PRO_CATG
                    "        sky frames with the      Y      1-m\n"
"                            detector signature\n"
"                            removed.\n"
"\n"
"Output files:\n"
"\n"
"  DO CATG                   Explanation \n"
"  -------                   ----------- \n"
"  "ERIS_NIX_SKYSUB_OBJECT_JITTER_PRO_CATG
                      "      target frames with the sky removed.\n"
"             or\n"
"  "ERIS_NIX_SKYSUB_STD_JITTER_PRO_CATG"\n"
"\n"
"  The output will be FITS files named 'skysub.<input filename>',\n"
"  containing extensions:\n"
"  - DATA, with the target data.\n"
"  - ERR, with the target error plane.\n"
"  - DQ, with the target data quality plane.\n"
"  - CONFIDENCE, with the target confidence plane.\n"
"  - BKG_DATA, with the estimated sky background.\n"
"  - BKG_ERR, with the error on the sky background.\n"
"  - BKG_CONF, with the confidence of the sky background.\n"
"\n"
"Notes on the method.\n"
"  Sky frames are taken from the pool of target or offset\n"
"  according to param --sky-source.\n"
"\n"
"  The background for each frame is estimated from sky frames\n"
"  taken within sky-bracket-time/2 seconds of it.\n"
"\n"
"  A first estimate of the background is made.\n"
"    If param --sky-method='collapse-median':\n"
"    - The estimate is a median collapse of the sky frames\n"
"      selected. This is prone to problems if the field is\n"
"      very crowded.\n"
"    If --sky-method='median-median':\n"
"    - The median of each sky frame is obtained, and the\n"
"      background frame set at the median of those values.\n"
"      The background is forced to be flat, which is more\n"
"      robust in crowded fields, or where there is\n"
"      nebulosity.\n"
"\n"
"  The estimated backgrounds are subtracted from the sky\n"
"  frames, and a search made for objects in each image.\n"
"  Object masks are constructed which blank out astronomical\n"
"  flux the sky frames. The --catalogue.xxx params listed\n"
"  below are used in this search for objects.\n"
"\n"
"  The masked sky frames are then used to re-estimate the\n"
"  sky backgrounds following the same sky-method.\n";
 

#define RECIPE_NAME "eris.eris_nix_img_skysub"
#define CONTEXT "eris."RECIPE_NAME
/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

cpl_recipe_define(eris_nix_img_skysub, ERIS_BINARY_VERSION,
                  "John Lightfoot",
                  PACKAGE_BUGREPORT, "2017",
                  "Subtract sky background from frames",
                  eris_nix_img_skysub_description);

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options    
  @param    self  the non-NULL parameterlist to fill
  @return   CPL_ERROR_NONE iff everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code eris_nix_img_skysub_fill_parameterlist(
  cpl_parameterlist * self) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();


    /* generate the general parameter list for the catalogue */
    eris_nix_catalogue_param_set(CONTEXT, self, 20, 3.0, CPL_TRUE,
    		5.0, CPL_TRUE, 64, 2.0, 3.0, ERIS_NIX_SATURATION_DEFAULT, HDRL_CATALOGUE_ALL);



    /* now the recipe-specific parameters */
    eris_nix_skysub_param_set(CONTEXT, self);
    
    /* coords of pixel to be used for diagnostics during reduction */
    eris_nix_pixel_coord_diagnostic_param_set(CONTEXT, self);
    


    return 0;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int eris_nix_img_skysub(cpl_frameset * frameset,
                                   const cpl_parameterlist * parlist) {


	eris_nix_scired_skysub(frameset, parlist, RECIPE_NAME, CONTEXT);

    return (int) cpl_error_get_code();
}
