/* $Id: eris_persistence_monitor.c,v 1.6 2013-03-26 17:17:32 jtaylor Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: jtaylor $
 * $Date: 2013-03-26 17:17:32 $
 * $Revision: 1.6 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "eris_utils.h"
#include "eris_pfits.h"
#include "eris_dfs.h"

#include <cpl.h>

#include <string.h>
#include <eris_ifu_dfs.h>
#include <eris_nix_dfs.h>
/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/
typedef enum {SLOW=0, FAST=1, UP_TO_RAMP=2} read_out_mode;
static char eris_persistence_monitor_description[] =
"This recipe counts the number of pixel above user defined thresholds.    \n"
"Inputs are ERIS SPIFFIER or NIX PERSISTENCE cubes and their corresponding\n"
"The recipe applies the check only on the 1st input raw frame, thus the set\n"
"of frames should contain only two input frames, the raw data and the     \n"
"Master dark (same DIT). The corresponding associated tags are:           \n"
"PERSISTENCE_IMA                                                          \n"
"MASTER_DARK_IFU (or MASTER_DARK_IMA for NIX data)                        \n"
"\n"
"The number of pixels above saturation or threshold values are stored as QC\n"
"In the output product that has PRO.CATG PERSISTENCE_HDR. Optional product \n"
"when the user sets debug to true is the image on which the pixel are computed"
"\n";

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

cpl_recipe_define(eris_persistence_monitor, ERIS_BINARY_VERSION, "Firstname Lastname",
                  PACKAGE_BUGREPORT, "2017",
                  "Monitor PERSISTENCE image sat. & threshold pixels",
                  eris_persistence_monitor_description);

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options    
  @param    self  the nonn-NULL parameterlist to fill
  @return   CPL_ERROR_NONE iff everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/
static
cpl_error_code eris_persistence_monitor_fill_parameterlist(cpl_parameterlist *self)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_parameter * p;
                                                                       
    /* Fill the parameters list */
    /* --saturation */
    p = cpl_parameter_new_value("eris.eris_persistence_monitor.saturation",
            CPL_TYPE_DOUBLE, "saturation level",
            "eris.eris_persistence_monitor", 60000.);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "saturation");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    p = cpl_parameter_new_value("eris.eris_persistence_monitor.saturation_neg",
               CPL_TYPE_DOUBLE, "negative saturation level",
               "eris.eris_persistence_monitor", -4.5e7);
       cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "saturation_neg");
       cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
       cpl_parameterlist_append(self, p);

    p = cpl_parameter_new_value("eris.eris_persistence_monitor.threshold",
    		CPL_TYPE_DOUBLE, "threshold level",
			"eris.eris_persistence_monitor",300.);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "threshold");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    /* --boolopt */
    p = cpl_parameter_new_value("eris.eris_persistence_monitor.debug",
            CPL_TYPE_BOOL, "If set to 'true' extra products are created",
			"eris.eris_persistence_monitor", CPL_FALSE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "debug");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);
 
    return cpl_errorstate_is_equal(prestate) ? CPL_ERROR_NONE
        : cpl_error_set_where(cpl_func);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int eris_persistence_monitor(cpl_frameset            * frameset,
                    const cpl_parameterlist * parameters)
{
    cpl_parameter *   param = NULL;
    cpl_parameterlist * parlist  = (cpl_parameterlist *) parameters;
    cpl_boolean             debug = CPL_FALSE;
    cpl_frameset        *   rawframes = NULL;
    const cpl_frame     *   firstframe = NULL;

    cpl_propertylist    *   plist = NULL;
    cpl_propertylist    *   applist = NULL;

    int                     nraw;

    double saturation = 0;
    double threshold = 0;
    double saturation_negative = - 4.5e7;

    const char* recipe = "eris_persistence_monitor";
    /* Use the errorstate to detect an error in a function that does not
       return an error code. */
    cpl_errorstate          prestate = cpl_errorstate_get();

    /* HOW TO RETRIEVE INPUT PARAMETERS */
    /* --double */
    param = cpl_parameterlist_find(parlist,
                                       "eris.eris_persistence_monitor.saturation");
    int saturation_sw_set = cpl_parameter_get_default_flag(param);
    saturation = cpl_parameter_get_double(param);


    param = cpl_parameterlist_find(parlist,
                                           "eris.eris_persistence_monitor.saturation_neg");
    //int saturation_neg_sw_set = cpl_parameter_get_default_flag(param);
    saturation_negative = cpl_parameter_get_double(param);

    param = cpl_parameterlist_find(parlist,
                                           "eris.eris_persistence_monitor.threshold");
    int threshold_sw_set = cpl_parameter_get_default_flag(param);
    threshold = cpl_parameter_get_double(param);
    /* --boolopt */
    param = cpl_parameterlist_find(parlist,
                                    "eris.eris_persistence_monitor.debug");
    debug = cpl_parameter_get_bool(param);
  
    if (!cpl_errorstate_is_equal(prestate)) {
        return (int)cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                          "Could not retrieve the input "
                                          "parameters");
    }
    
    /* Identify the RAW and CALIB frames in the input frameset */
    /* We use NIX classifications also for IFU data, where we added support for
     * MASTER_DARK_IFU, as NIX does not do too many checks (that input frame is
     * of the same SEQ.ARM) not needed with unique PRO.CATGs
     */


    cpl_ensure_code(eris_nix_dfs_set_groups(frameset) == CPL_ERROR_NONE,
                        cpl_error_get_code());

 
    /* HOW TO ACCESS INPUT DATA */
    /*  - A required file */
    rawframes = cpl_frameset_new();
    nraw = 0;
    /* TODO: here we could use +eris_dfs_extract_raw_frames */
    for (cpl_size j = 0; j<cpl_frameset_get_size(frameset); j++) {
        const cpl_frame * current_frame;
        current_frame = cpl_frameset_get_position_const(frameset, j);
        if(!strcmp(cpl_frame_get_tag(current_frame), ERIS_PERSISTENCE_IMA) ||
           !strcmp(cpl_frame_get_tag(current_frame), ERIS_PERSISTENCE_CUBE)) {
            cpl_frame * new_frame = cpl_frame_duplicate(current_frame);
            cpl_frameset_insert(rawframes, new_frame);
            nraw++;
        }
    }

    if (nraw == 0) {
        cpl_frameset_delete(rawframes);
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s or %s", ERIS_PERSISTENCE_IMA,
										  ERIS_PERSISTENCE_CUBE);
    }

    cpl_propertylist_delete(plist);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure_code(cpl_errorstate_is_equal(prestate), cpl_error_get_code());

    /* NOW PERFORMING THE DATA REDUCTION */
    /* Let's just load an image for the example */
    cpl_size nframes = cpl_frameset_get_size(rawframes);

    firstframe = cpl_frameset_get_position_const(rawframes, 0);
    const char* fname = cpl_frame_get_filename(firstframe);
    plist = cpl_propertylist_load(fname, 0);
    const char* arm = cpl_propertylist_get_string(plist,FHDR_E_ARM);
    int naxis = cpl_propertylist_get_int(plist,"NAXIS");

    cpl_frame* frm_mdark = NULL;
    const char* read_curname = cpl_propertylist_get_string(plist,
    		"ESO DET READ CURNAME");
    read_out_mode rom = SLOW;

    if (read_curname == NULL) {
        cpl_propertylist_delete(plist);
        cpl_frameset_delete(rawframes);
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "No 'ESO DET READ CURNAME' in raw "
                                     "frame 1/%d", nraw);
    }
    if (strcmp(read_curname,"FAST_UNCORR") == 0) {
    	rom = FAST;
    } else if (strcmp(read_curname,"SLOW_GR_UTR") == 0) {
    	rom = SLOW;
    } else if (strcmp(read_curname,"Normal_UpTheRamp") == 0) {
    	rom = UP_TO_RAMP;
    }
    if(strcmp(arm,"SPIFFIER") == 0) {
    	if(saturation_sw_set == 1) {
           // do nothing
    	} else {
    		saturation = 60000;
    	}
    	if(threshold_sw_set == 1) {
    		// do nothing
    	} else {
    		threshold = 300;
    	}
    	frm_mdark = cpl_frameset_find(frameset,ERIS_IFU_PRO_DARK);
    } else {
    	if(saturation_sw_set == 1) {
    		// do nothing
    	} else {
    		if (rom == FAST) {
    			saturation = 35000;
    		} else {
    			saturation = 15000;
    		}
    	}
    	if(threshold_sw_set == 1) {
    		// do nothing
    	} else {
    		threshold = 300;
    	}
    	frm_mdark = cpl_frameset_find(frameset,ERIS_NIX_MASTER_DARK_IMG_PRO_CATG);
    }
    cpl_propertylist_delete(plist);
    /* make sure the parameters are updated to the actual values set to have
     * proper signature in the products */
    param = cpl_parameterlist_find(parlist,
                                         "eris.eris_persistence_monitor.saturation");
    cpl_parameter_set_double(param, saturation);
    param = cpl_parameterlist_find(parlist,
                                         "eris.eris_persistence_monitor.threshold");
    cpl_parameter_set_double(param, threshold);
    cpl_image* img_persistence = NULL;
    cpl_imagelist* iml_persistence = NULL;

    /* Add the product category  */
    applist = cpl_propertylist_new();
    cpl_propertylist_append_string(applist, CPL_DFS_PRO_CATG,
                                   ERIS_PERSISTENCE_HDR);

    nframes = cpl_frameset_get_size(rawframes);


    //double nsat_avg = 0;
    //double nsat_frac = 0;
    int nsat_max = 0;
    //int nsat_tot = 0;
    int frame_nsat_max = 0;

    //double nthresh_avg = 0;
    //double nthresh_frac = 0;
    int nthresh_max = 0;
    //int nthresh_tot = 0;
    int frame_nthresh_max = 0;


    double nsat_avg_tmp = 0;
    double nsat_frac_tmp = 0;
    int nsat_max_tmp = 0;
    int nsat_tot_tmp = 0;
    //int frame_nsat_max_tmp = 0;


    double nthresh_avg_tmp = 0;
    double nthresh_frac_tmp = 0;
    int nthresh_max_tmp = 0;
    int nthresh_tot_tmp = 0;
    //int frame_nthresh_max_tmp = 0;


    char* keyname;
    for(cpl_size i = 0; i < nframes; i++) {

    	const cpl_frame * frame = cpl_frameset_get_position_const(rawframes, i);
    	fname  = cpl_frame_get_filename(frame);
    	plist = cpl_propertylist_load(fname, 0);
    	arm = cpl_propertylist_get_string(plist,FHDR_E_ARM);
    	naxis = cpl_propertylist_get_int(plist,"NAXIS");
    	cpl_propertylist_delete(plist);
    	cpl_msg_info(cpl_func,"process frame: %lld %s", i, fname);

    	if(naxis == 2) {

    		eris_get_sat_qc_for_image(fname, frm_mdark, saturation,
    				saturation_negative, threshold, i, applist);
    		

    		keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NSAT MAX",i);
    		nsat_max_tmp = cpl_propertylist_get_int(applist, keyname);
    		cpl_free(keyname);

    		keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NSAT TOT",i);
    		nsat_tot_tmp += cpl_propertylist_get_int(applist, keyname);
    		cpl_free(keyname);

    		keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NSAT AVG",i);
    		nsat_avg_tmp += cpl_propertylist_get_int(applist, keyname);
    		cpl_free(keyname);

    		keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 BPSAT FRAC", i);
    		nsat_frac_tmp += cpl_propertylist_get_double(applist, keyname);
    		cpl_free(keyname);

    		keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NTHRESH MAX",i);
    		nthresh_max_tmp = cpl_propertylist_get_int(applist, keyname);
    		cpl_free(keyname);

    		keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NTHRESH TOT",i);
    		nthresh_tot_tmp += cpl_propertylist_get_int(applist, keyname);
    		cpl_free(keyname);

    		keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 NTHRESH AVG",i);
    		nthresh_avg_tmp += cpl_propertylist_get_int(applist, keyname);
    		cpl_free(keyname);

    		keyname = cpl_sprintf("ESO QC FRM%lld SLIC0 BPTHRESH FRAC", i);
    		nthresh_frac_tmp += cpl_propertylist_get_double(applist, keyname);
    		cpl_free(keyname);

    		if(nsat_max_tmp > nsat_max) {
    		   nsat_max = nsat_max_tmp;
    		   frame_nsat_max = i;
    		}

    		if(nthresh_max_tmp > nthresh_max) {
    		   nthresh_max = nthresh_max_tmp;
    		   frame_nthresh_max = i;
    		}


    	} else {

    		eris_get_sat_qc_for_cube(fname, frm_mdark, saturation, saturation_negative,
    				threshold, i, applist);

    		keyname = cpl_sprintf("ESO QC FRM%lld NSAT MAX", i);
    		nsat_max_tmp = cpl_propertylist_get_int(applist, keyname);
    		cpl_free(keyname);

    		keyname = cpl_sprintf("ESO QC FRM%lld NSAT TOT", i);
    		nsat_tot_tmp += cpl_propertylist_get_int(applist, keyname);
    		cpl_free(keyname);

    		keyname = cpl_sprintf("ESO QC FRM%lld NSAT AVG", i);
    		nsat_avg_tmp += cpl_propertylist_get_double(applist, keyname);
    		cpl_free(keyname);

    		keyname = cpl_sprintf("ESO QC FRM%lld BPSAT FRAC", i);
    		nsat_frac_tmp += cpl_propertylist_get_double(applist, keyname);
    		cpl_free(keyname);

    		keyname = cpl_sprintf("ESO QC FRM%lld NTHRESH MAX", i);
    		nthresh_max_tmp = cpl_propertylist_get_int(applist, keyname);
    		cpl_free(keyname);

    		keyname = cpl_sprintf("ESO QC FRM%lld NTHRESH TOT", i);
    		nthresh_tot_tmp += cpl_propertylist_get_int(applist, keyname);
    		cpl_free(keyname);

    		keyname = cpl_sprintf("ESO QC FRM%lld NTHRESH AVG", i);
    		nthresh_avg_tmp += cpl_propertylist_get_double(applist, keyname);
    		cpl_free(keyname);

    		keyname = cpl_sprintf("ESO QC FRM%lld BPTHRESH FRAC", i);
    		nthresh_frac_tmp += cpl_propertylist_get_double(applist, keyname);
    		cpl_free(keyname);

    		if(nsat_max_tmp > nsat_max) {
    			nsat_max = nsat_max_tmp;
    			frame_nsat_max = i;
    		}

    		if(nthresh_max_tmp > nthresh_max) {
    			nthresh_max = nthresh_max_tmp;
    			frame_nthresh_max = i;
    		}

    	}
    	

    	/* HOW TO SAVE A DFS-COMPLIANT PRODUCT TO DISK  */
    	if(debug) {
    		cpl_propertylist_update_string(applist, CPL_DFS_PRO_CATG,
    				ERIS_PERSISTENCE_DBG);
    		if(naxis == 2) {
    			if (cpl_dfs_save_image(frameset, NULL, parlist, frameset, NULL,
    					img_persistence, CPL_BPP_IEEE_FLOAT, ERIS_PERSISTENCE_DBG,
    					applist, NULL, PACKAGE "/" PACKAGE_VERSION,
    					ERIS_PERSISTENCE_DBG_FN)) {
    				/* Propagate the error */
    				(void)cpl_error_set_where(cpl_func);
    			}
    		} else {
    			if (cpl_dfs_save_imagelist(frameset, NULL, parlist, frameset, NULL,
    					iml_persistence, CPL_BPP_IEEE_FLOAT, ERIS_PERSISTENCE_DBG,
    					applist, NULL, PACKAGE "/" PACKAGE_VERSION,
    					ERIS_PERSISTENCE_DBG_FN)) {
    				/* Propagate the error */
    				(void)cpl_error_set_where(cpl_func);
    			}
    		}
    	}
    
    	if(naxis == 2) {
    		cpl_image_delete(img_persistence);
    	} else {
    		cpl_imagelist_delete(iml_persistence);
    	}
    	
    }// end loop over frames
    keyname = cpl_sprintf("ESO QC NSAT MAX");
    cpl_propertylist_append_int(applist, keyname, nsat_max);
    cpl_free(keyname);

    keyname = cpl_sprintf("ESO QC NSAT TOT");
    cpl_propertylist_append_int(applist, keyname, nsat_tot_tmp);
    cpl_free(keyname);

    keyname = cpl_sprintf("ESO QC NSAT AVG");
    cpl_propertylist_append_double(applist, keyname, (nsat_avg_tmp / nframes));
    cpl_free(keyname);

    keyname = cpl_sprintf("ESO QC FRM NSAT MAX");
    cpl_propertylist_append_int(applist, keyname, frame_nsat_max);
    cpl_free(keyname);

    keyname = cpl_sprintf("ESO QC BPSAT FRAC");
    cpl_propertylist_append_double(applist, keyname,
    		(double)(nsat_frac_tmp / nframes));
    cpl_free(keyname);

    	keyname = cpl_sprintf("ESO QC NTHRESH MAX");
    cpl_propertylist_append_int(applist, keyname, nthresh_max);
    cpl_free(keyname);

    keyname = cpl_sprintf("ESO QC NTHRESH TOT");
    cpl_propertylist_append_int(applist, keyname, nthresh_tot_tmp);
    cpl_free(keyname);

    keyname = cpl_sprintf("ESO QC NTHRESH AVG");
    cpl_propertylist_append_double(applist, keyname, (nthresh_avg_tmp / nframes));
    cpl_free(keyname);

    keyname = cpl_sprintf("ESO QC FRM NTHRESH MAX");
    cpl_propertylist_append_int(applist, keyname, frame_nthresh_max);
    cpl_free(keyname);

    keyname = cpl_sprintf("ESO QC BPTHRESH FRAC");
    cpl_propertylist_append_double(applist, keyname, (nthresh_frac_tmp / nframes));
    cpl_free(keyname);

	cpl_dfs_save_propertylist(frameset, NULL, parlist, frameset, NULL,
			recipe, applist, NULL,
			PACKAGE "/" PACKAGE_VERSION, ERIS_PERSISTENCE_HDR_FN);
	

    cpl_propertylist_delete(applist);
    cpl_frameset_delete(rawframes);

   
    return (int)cpl_error_get_code();
}
