# import the needed modules
from __future__ import absolute_import
from __future__ import print_function

try:
    #  import sys
    import numpy

    try:
        from astropy.io import fits as pyfits
    except ImportError:
        import pyfits

    #  import wx
    import matplotlib
    import reflex
    import eris_ifu_plot_common
    import eris_ifu_parameters_common

    from reflex import parseSofJson, RecipeParameter
    from reflex_interactive_app import PipelineInteractiveApp
    from pipeline_product import PipelineProduct
    import eris_ifu_plot_widgets
    import reflex_plot_widgets
    from pipeline_display import SpectrumDisplay, ImageDisplay, ScatterDisplay

    #  matplotlib.use('WXAgg')
    import_sucess = 'true'
    import warnings

    warnings.simplefilter('ignore', UserWarning)

    # NOTE for developers:
    # -If you want to modify the current script to cope
    #  with different parameters, this is the function to modify:
    #  setInteractiveParameters()
    # -If you want to modify the current script to read different data from
    #  the input FITS, these are the functions (class DataPlotterManager) to modify:
    #  readFitsData()    to indicate what are the FITS input to plot
    #  plotSpectrum()    to control 1D image product layout
    #  plotImage()       to control 2D image product layout
    #  plotTable()       to control 1D table product layout
    #  oplotTable()      to control 1D table product overplots
    #  setRecId()        to set rec id (used in parameter prefix)
    #  getArmId()        to get arm id FITS header info
    #
    # -If you want to modify the current script to modify the plots (using the same
    #  data),  this is the function to modify:
    #  plotProductsGraphics()          (from class DataPlotterManager)
    # -If you want to modify the text that appears in the "Help" button,
    #  this is the function to modify:
    #  setWindowHelp()
    # -If you want to modify the title of the window, modify this function:
    #  setWindowTitle()

    # This class deals with the specific details of data reading and final plotting.

    class DataPlotterManager:
        def paragraph(self, text, width=None):
            """ wrap text string into paragraph
                text:  text to format, removes leading space and newlines
                width: if not None, wraps text, not recommended for tooltips as
                       they are wrapped by wxWidgets by default
            """

            import textwrap
            if width is None:
                return textwrap.dedent(text).replace('\n', ' ').strip()
            else:
                return textwrap.fill(textwrap.dedent(text), width=width)

                # 2D image display

        def plotImage(self, obj, subplot, title, tooltip):

            obj.readImage(fits_extension=1)
            obj.read2DLinearWCS(fits_extension=1)
            img_obj = ImageDisplay()

            img_obj.setLabels('[arcsec]', '[arcsec]')
            y_size, x_size = obj.image.shape
            print("crval1", obj.crval1, "cdelt1", obj.cdelt1, "crpix1", obj.crpix1)
            print("crval2", obj.crval2, "cdelt2", obj.cdelt2, "crpix2", obj.crpix2)

            # img_obj.setXLinearWCSAxis(obj.crval1,obj.cdelt1,obj.crpix1)
            # img_obj.setYLinearWCSAxis(obj.crval2,obj.cdelt2,obj.crpix2)

            # img_obj.setLimits((1,x_size),(0,y_size))
            subplot.set_title(title, fontsize=7)
            # check_mean = numpy.mean(obj.image)
            # data_for_stats = numpy.compress(obj.image != numpy.nan, obj.image)
            # data_for_stats = numpy.where(obj.image != numpy.nan, obj.image, 0.0)
            # check_mean = numpy.mean(data_for_stats)
            # where_are_NaNs = numpy.isnan(obj.image)
            # obj.image[where_are_NaNs] = 0
            # pippo
            print("mean image", numpy.mean(obj.image))
            # print("<<<<<<<<<<")
            # print("size", len(data_for_stats))
            # if numpy.isnan(check_mean) :
            if numpy.isnan(obj.image.all()):
                pass
                img_obj.display(subplot, title, self.paragraph(tooltip), obj.image)
            else:
                img_obj.display(subplot, title, self.paragraph(tooltip), obj.image)

        # 3D cube slices display
        def plotCubeSlice(self, obj, extid, slice, subplot, title, tooltip):
            # print('slice=',slice)
            obj.readImage(fits_extension=extid)
            obj.read2DLinearWCS(fits_extension=extid)
            img_obj = ImageDisplay()
            image = obj.image[slice, :, :]
            img_obj.setLabels('[arcsec]', '[arcsec]')
            # img_obj.setXLinearWCSAxis(0,64,1)
            # img_obj.setYLinearWCSAxis(0,64,1)
            img_obj.display(subplot, title, self.paragraph(tooltip), image)

        # This function will read all the columns, images and whatever is needed
        # from the products. The variables , self.plot_x, self.plot_y, etc...
        # are used later in function plotProductsGraphics().
        # Add/delete these variables as you need (only that plotProductsGraphics()
        # has to use the same names).
        # You can also create some additional variables (like statistics) after
        # reading the files.
        # If you use a control variable (self.xxx_found), you can modify
        # later on the layout of the plotting window based on the presence of
        # given input files.
        # sof contains all the set of frames
        def readFitsData(self, fitsFiles):
            # Control variable to check if the interesting files where at the input
            prefix = "OBS_"
            self.obj_found = False
            self.std_found = False
            self.psf_found = False
            self.coadd_obj_fluxcal_found = False
            self.coadd_obj_found = False
            self.coadd_std_found = False
            self.coadd_psf_found = False
            self.mean_coadd_obj_found = False
            self.mean_coadd_std_found = False
            self.mean_coadd_psf_found = False
            self.median_coadd__found = False
            self.median_coadd_std_found = False
            self.median_coadd_psf_found = False
            self.object_nodding_stacked_found = False
            self.psf_nodding_stacked_found = False
            self.std_nodding_stacked_found = False
            self.spectrum_found = False
            self.spectrum_fluxcal_found = False
            self.std_response_found = False
            self.std_efficiency_found = False
            self.sky_med_found = False
            self.enc_energy_found = False
            self.ao_performance_found = False
            self.rec_subtitle = ""
            self.obs_targ = ""
            self.nima = 1
            self.has_subplot_obj_spect = False
            self.fits_ext = 1

            # added dummy values:
            self.NAXIS3 = 2218
            self.crval3 = 2.2000000  # K band
            self.crpix3 = 1109.
            self.cdelt3 = 0.000245000002  # K band
            self.wmax = self.crval3 + (self.NAXIS3 - 1) * self.cdelt3
            self.plane_id = 1000
            self.has_cube_mean = False

            self.cube_plane_selector_is_created = 'False'
            self.product_id = 'stacked frame'
            self.pro1D_id = 'SPECTRUM'
            self.nPro1D = 0
            self.labels_prod = list()
            self.labels_pro1d = list()
            # Read all the products
            frames = dict()
            for frame in fitsFiles:
                if frame == '':
                    continue

                # print("frame name:", frame.name)
                header = pyfits.open(frame.name)
                # Make sure to have only products from the same recipe
                # that used this (common) script
                if 'ESO PRO REC1 ID' in header[0].header:
                    rec_id = header[0].header['ESO PRO REC1 ID']
                    print("rec_id:", rec_id, frame.category)
                    if rec_id == self.rec_id:
                        category = frame.category
                        frames[category] = frame
                        print("frame name:", frame.name, category)
                        if category == "CUBE" or \
                                category == "DAR_CORRECTED_CUBE" or \
                                category == "DAR_CORRECTED_CUBE_COADD" or \
                                category == "DAR_CORRECTED_CUBE_MEAN" or \
                                category == "OBJECT_CUBE" or \
                                category == "OBJECT_CUBE_COADD" or \
                                category == "OBJECT_CUBE_MEAN" or \
                                category == "STD_CUBE" or \
                                category == "STD_CUBE_COADD" or \
                                category == "STD_CUBE_COADD_NOFLAT" or \
                                category == "STD_CUBE_MEAN" or \
                                category == "PSF_CUBE" or \
                                category == "PSF_CUBE_COADD" or \
                                category == "PSF_CUBE_MEAN" or \
                                category == "SKY_CUBE":
                            self.band = header[0].header['ESO INS3 SPGW ID']
                            self.scale = header[0].header['ESO INS3 SPXW ID']
                            if rec_id == "eris_ifu_pupil":
                                self.obs_targ = 'pupil'
                            else:
                                self.obs_targ = header[0].header['ESO OBS TARG NAME']

            print(frames)
            prefixes = [k.split('_')[0] for k in frames.keys()]
            if "FMTCHK" in prefixes and self.rec_id == "eris_ifu_jitter":
                prefix = "FMTCHK_"

            # products not used: OBS_OBJ,OBS_STD,OBS_PSF

            # For any arm search a list of input frames
            # print "arm=",arm , "prefix=",prefix
            # print frames
            key = "OBJECT_CUBE"
            if key in frames:
                self.obj_found = True
                hdulist = frames[key]
                self.obj = PipelineProduct(hdulist)
                self.getNAXIS3(hdulist, 1)
                self.mode = 'SCI'
                self.labels_prod.append(key)
                print("found ", key)

            key = "DAR_CORRECTED_CUBE"
            if key in frames:
                self.obj_found = True
                hdulist = frames[key]
                self.obj = PipelineProduct(hdulist)
                self.getNAXIS3(hdulist, 1)
                self.mode = 'SCI'
                self.labels_prod.append(key)
                print("found ", key)

            key = "STD_CUBE"
            if key in frames:
                self.obj_found = True
                hdulist = frames[key]
                self.obj = PipelineProduct(hdulist)
                self.getNAXIS3(hdulist, 1)
                # self.getWCS(hdulist)
                self.mode = 'STD'
                # self.obs_targ = header[0].header['ESO OBS TARG NAME']
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                self.labels_prod.append(key)
                print("found ", key)

            key = "STD_FLUX_CUBE"
            if key in frames:
                self.obj_found = True
                hdulist = frames[key]
                self.obj = PipelineProduct(hdulist)
                self.getNAXIS3(hdulist, 1)
                # self.getWCS(hdulist)
                self.mode = 'STD'
                # self.obs_targ = header[0].header['ESO OBS TARG NAME']
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                self.labels_prod.append(key)
                print("found ", key)

            key = "PSF_CUBE"
            if key in frames:
                self.obj_found = True
                hdulist = frames[key]
                self.obj = PipelineProduct(hdulist)
                self.getNAXIS3(hdulist, 1)
                # self.getWCS(hdulist)
                self.mode = 'PSF'
                # self.obs_targ = header[0].header['ESO OBS TARG NAME']
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                self.labels_prod.append(key)
                print("found ", key)

            key = "SKY_CUBE"
            if key in frames:
                self.sky_found = True
                hdulist = frames[key]
                self.sky = PipelineProduct(hdulist)
                self.getNAXIS3(hdulist, 1)
                # self.getWCS(hdulist)
                self.mode = 'SKY'
                # self.obs_targ = header[0].header['ESO OBS TARG NAME']
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                self.labels_prod.append(key)
                print("found ", key)

            key = "OBJECT_CUBE_COADD"
            if key in frames:
                self.coadd_obj_found = True
                hdulist = frames[key]
                self.coadd_obj = PipelineProduct(hdulist)
                self.getNAXIS3(hdulist, 1)
                self.mode = 'SCI'
                self.product_id = key
                self.labels_prod.append(key)
                print("found ", key)

            key = "DAR_CORRECTED_CUBE_COADD"
            if key in frames:
                self.coadd_obj_found = True
                hdulist = frames[key]
                self.coadd_obj = PipelineProduct(hdulist)
                self.getNAXIS3(hdulist, 1)
                self.mode = 'SCI'
                self.product_id = key
                self.labels_prod.append(key)
                print("found ", key)

            key = "STD_CUBE_COADD"
            if key in frames:
                self.coadd_std_found = True
                hdulist = frames[key]
                self.coadd_std = PipelineProduct(hdulist)
                self.getNAXIS3(hdulist, 1)
                # self.getWCS(hdulist)
                self.mode = 'STD'
                # self.obs_targ = header[0].header['ESO OBS TARG NAME']
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                self.product_id = key
                self.labels_prod.append(key)
                print("found ", key)

            key = "STD_FLUX_CUBE_COADD"
            if key in frames:
                self.coadd_std_found = True
                hdulist = frames[key]
                self.coadd_std = PipelineProduct(hdulist)
                self.getNAXIS3(hdulist, 1)
                # self.getWCS(hdulist)
                self.mode = 'STD'
                # self.obs_targ = header[0].header['ESO OBS TARG NAME']
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                self.product_id = key
                self.labels_prod.append(key)
                print("found ", key)

            key = "STD_CUBE_COADD_NOFLAT"
            if key in frames:
                self.coadd_std_found = True
                hdulist = frames[key]
                self.coadd_std = PipelineProduct(hdulist)
                self.getNAXIS3(hdulist, 1)
                # self.getWCS(hdulist)
                self.mode = 'STD'
                # self.obs_targ = header[0].header['ESO OBS TARG NAME']
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                self.product_id = key
                self.labels_prod.append(key)
                print("found ", key)

            key = "PSF_CUBE_COADD"
            if key in frames:
                self.coadd_psf_found = True
                hdulist = frames[key]
                self.coadd_psf = PipelineProduct(hdulist)
                self.getNAXIS3(hdulist, 1)
                # self.getWCS(hdulist)
                self.mode = 'PSF'
                # self.obs_targ = header[0].header['ESO OBS TARG NAME']
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                self.product_id = key
                self.labels_prod.append(key)
                print("found ", key)

            key = "OBJECT_CUBE_COADD_FLUXCAL"
            if key in frames:
                self.coadd_obj_fluxcal_found = True
                hdulist = frames[key]
                self.coadd_obj_fluxcal = PipelineProduct(hdulist)
                self.getNAXIS3(hdulist, 1)
                self.mode = 'SCI'
                self.product_id = key
                self.labels_prod.append(key)
                print("found ", key)

            key = "CUBE"
            if key in frames:
                self.coadd_obj_found = True
                hdulist = frames["CUBE"]
                self.coadd_obj = PipelineProduct(hdulist)
                self.getNAXIS3(hdulist, 1)
                # self.getWCS(hdulist)
                self.mode = 'SCI'
                # self.obs_targ = header[0].header['ESO OBS TARG NAME']
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                self.labels_prod.append(key)
                print("found ", key)

            key = "DAR_CORRECTED_CUBE_MEAN"
            if key in frames:
                self.mean_coadd_obj_found = True
                hdulist = frames[key]
                self.mean_coadd_obj = PipelineProduct(hdulist)
                self.cube_mean = self.mean_coadd_obj
                self.has_cube_mean = True
                print("found ", key)

            key = "OBJECT_CUBE_MEAN"
            if key in frames:
                self.mean_coadd_obj_found = True
                hdulist = frames[key]
                self.mean_coadd_obj = PipelineProduct(hdulist)
                self.cube_mean = self.mean_coadd_obj
                self.has_cube_mean = True
                print("found ", key)

            key = "STD_CUBE_MEAN"
            if key in frames:
                self.mean_coadd_std_found = True
                hdulist = frames[key]
                self.mean_coadd_std = PipelineProduct(hdulist)
                self.cube_mean = self.mean_coadd_std
                self.has_cube_mean = True
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                print("found ", key)

            key = "PSF_CUBE_MEAN"
            if key in frames:
                self.mean_coadd_psf_found = True
                hdulist = frames[key]
                self.mean_coadd_psf = PipelineProduct(hdulist)
                self.cube_mean = self.mean_coadd_psf
                self.has_cube_mean = True
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                print("found ", key)

            key = "DAR_CORRECTED_CUBE_MEDIAN"
            if key in frames:
                self.median_coadd_obj_found = True
                hdulist = frames[key]
                self.median_coadd_obj = PipelineProduct(hdulist)
                self.cube_median = self.median_coadd_obj
                print("found ", key)

            key = "OBJECT_CUBE_MEDIAN"
            if key in frames:
                self.median_coadd_obj_found = True
                hdulist = frames[key]
                self.median_coadd_obj = PipelineProduct(hdulist)
                self.cube_median = self.median_coadd_obj
                print("found ", key)

            key = "STD_CUBE_MEDIAN"
            if key in frames:
                self.mean_coadd_std_found = True
                hdulist = frames[key]
                self.median_coadd_std = PipelineProduct(hdulist)
                self.cube_median = self.median_coadd_std
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                print("found ", key)

            key = "STD_FLUX_CUBE_MEDIAN"
            if key in frames:
                self.mean_coadd_std_found = True
                hdulist = frames[key]
                self.median_coadd_std = PipelineProduct(hdulist)
                self.cube_median = self.median_coadd_std
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                print("found ", key)

            key = "PSF_CUBE_MEDIAN"
            if key in frames:
                self.mean_coadd_psf_found = True
                hdulist = frames[key]
                self.median_coadd_psf = PipelineProduct(hdulist)
                self.cube_median = self.median_coadd_psf
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                print("found ", key)

            key = "OBJECT_NODDING_STACKED"
            if key in frames:
                self.object_nodding_stacked_found = True
                hdulist = frames["OBJECT_NODDING_STACKED"]
                self.object_stacked = PipelineProduct(hdulist)
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                self.labels_prod.append(key)
                print("found med coadded obj cube")

            key = "STD_NODDING_STACKED"
            if key in frames:
                self.object_nodding_stacked_found = True
                hdulist = frames["STD_NODDING_STACKED"]
                self.object_stacked = PipelineProduct(hdulist)
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                self.labels_prod.append(key)
                print("found ", key)

            key = "PSF_CALIBRATOR_STACKED"
            if key in frames:
                self.object_nodding_stacked_found = True
                hdulist = frames["PSF_CALIBRATOR_STACKED"]
                self.object_stacked = PipelineProduct(hdulist)
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                print("found med coadded psf cube")

            key = "SPECTRUM"
            if key in frames:
                self.spectrum_found = True
                hdulist = frames["SPECTRUM"]
                self.spectra = PipelineProduct(hdulist)
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                self.nPro1D += 1
                self.labels_pro1d.append(key)
                print("found ", key)

            key = "SPECTRUM_FLUXCAL"
            if key in frames:
                self.spectrum_fluxcal_found = True
                hdulist = frames["SPECTRUM_FLUXCAL"]
                self.spectra_fluxcal = PipelineProduct(hdulist)
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                self.nPro1D += 1
                self.labels_pro1d.append(key)
                print("found ", key)

            print, frames
            key = "EFFICIENCY"
            if key in frames:
                self.efficiency_found = True
                hdulist = frames["EFFICIENCY"]
                self.efficiency = PipelineProduct(hdulist)
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                self.nPro1D += 1
                self.labels_pro1d.append(key)
                print("found ", key)

            key = "RESPONSE"
            if key in frames:
                self.std_response_found = True
                hdulist = frames["RESPONSE"]
                self.response = PipelineProduct(hdulist)
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                self.nPro1D += 1
                self.labels_pro1d.append(key)
                print("found ", key)

            key = "SKY_MED"
            if key in frames:
                self.sky_med_found = True
                hdulist = frames["SKY_MED"]
                self.sky_med = PipelineProduct(hdulist)
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                print("found ", key)

            key = "ENC_ENERGY"
            if key in frames:
                self.enc_energy_found = True
                hdulist = frames["ENC_ENERGY"]
                self.enc_energy = PipelineProduct(hdulist)
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                self.labels_prod.append(key)
                print("found ", key)

            key = "AO_PERFORMANCE"
            if key in frames:
                self.ao_performance_found = True
                hdulist = frames["AO_PERFORMANCE"]
                self.ao_performance = PipelineProduct(hdulist)
                # self.resid_tab_2plot = PlotableResidLineTable(hdulist,1)
                # self.nima += 1
                self.labels_prod.append(key)
                print("found ", key)

                # Set rec id (to have proper recipe parameters prefix)

        def setRecId(self, rec_id):
            # Recipe ID variable to properly define params
            # print("Set Recipe Id")
            self.rec_id = rec_id

        # Get NAXIS info (Xshooter and arm specific)
        def getNAXIS3(self, fitsfile, extid=0):
            hdulist = pyfits.open(fitsfile.name)
            self.NAXIS3 = hdulist[extid].header['NAXIS3']

        def getWCS(self, fitsfile):
            hdulist = pyfits.open(fitsfile.name)
            self.NAXIS1 = hdulist[0].header['NAXIS1']
            self.NAXIS2 = hdulist[0].header['NAXIS2']
            self.NAXIS3 = hdulist[0].header['NAXIS3']

            self.crpix1 = hdulist[0].header['CRPIX1']
            self.crval1 = hdulist[0].header['CRVAL1']
            self.cdelt1 = hdulist[0].header['CDELT1']

            self.crpix2 = hdulist[0].header['CRPIX2']
            self.crval2 = hdulist[0].header['CRVAL2']
            self.cdelt2 = hdulist[0].header['CDELT2']

            self.crpix3 = hdulist[0].header['CRPIX3']
            self.crval3 = hdulist[0].header['CRVAL3']
            self.cdelt3 = hdulist[0].header['CDELT3']
            self.wmax = self.crval3 + (self.NAXIS3 - 1) * self.cdelt3

        # Get arm setting
        def getBandId(self, sof):
            # Recipe ID variable to properly define params
            self.filt_id = "K"
            self.scale_id = "0.025"
            nf = 0

            frames = dict()
            files = sof.files
            for f in files:
                frame = f.name
                if frame == '':
                    continue
                else:
                    nf += 1
                    hdulist = pyfits.open(frame)
                    print("frame", frame)
                    rec_id_list = hdulist[0].header['ESO PRO REC1 ID']
                    # filt_list = hdulist[0].header['ESO FILT1 NAME']
                    # scale_list = hdulist[0].header['ESO OPTI1 NAME']
            if nf != 0:
                self.rec_id = rec_id_list[0:]
                # self.filt_id = filt_list[0:]
                # self.scale_id = scale_list[0:]

            if self.rec_id == "eris_ifu_jitter":
                self.rec_subtitle = "Jitter Observation. "
            if self.rec_id == "eris_ifu_stdstar":
                self.rec_subtitle = "Star Observation. "

        def plotWidgets(self):
            widgets = list()
            min = 0
            max = 100
            title_slider = "plane id"
            if self.coadd_obj_found is True or \
                    self.coadd_std_found is True or \
                    self.coadd_psf_found is True:
                max = self.NAXIS3
                val = numpy.floor(self.NAXIS3 / 2 + 0.5)
                print('val=', val)
                # Cube plane selector
                self.slider = eris_ifu_plot_widgets.InteractiveSlider2(self.cube_plane_selector,
                                                                       self.setPlaneSelectCallBack, min, max, val, 100,
                                                                       1, title_slider)
                widgets.append(self.slider)

                labels_prod = self.labels_prod
                self.radiobutton_pro = reflex_plot_widgets.InteractiveRadioButtons(self.cube_selector,
                                                                                   self.setProductSelectCallBack,
                                                                                   labels_prod, 0, title="")
                widgets.append(self.radiobutton_pro)

            if self.nPro1D > 1:
                labels_pro1d = self.labels_pro1d
                self.radiobutton_pro = reflex_plot_widgets.InteractiveRadioButtons(self.spect_selector,
                                                                                   self.setPro1DSelectCallBack,
                                                                                   labels_pro1d, 0, title="")
                widgets.append(self.radiobutton_pro)

            return widgets

        def setProductSelectCallBack(self, product_id):
            # print("Selected product id:", product_id)
            self.product_id = product_id

            self.displayProducts()
            self.plotProductsGraphics()

            print("product_id:", self.product_id)

        def setPro1DSelectCallBack(self, product_id):
            # print("Selected product id:", product_id)
            self.pro1D_id = product_id

            # self.displayProducts()
            self.plotProductsGraphics()

            print("product_id:", self.product_id)

        def setPlaneSelectCallBack(self, plane_id):
            self.plane_id = int(plane_id)
            plane_id = self.plane_id
            self.subplot_cube.cla()
            print('plane_id', plane_id)
            self.setDisplayCubeSlice(plane_id, self.cube)

        def setDisplayCubeSlice(self, plane_id, cube):
            self.wcen = self.crval3 + (plane_id - 1) * self.cdelt3
            plane = int(plane_id)
            print("crval3=", self.crval3, "cdelt3=", self.cdelt3, "wcen=", self.wcen)
            self.title_cube_obj = r'plane id: %#4.4d $\lambda$: %#6.5g [$\mu$m]' % (plane,
                                                                                    self.wcen) + "\n" + self.mode + \
                                  " band:" + self.band + " " + self.scale + "[mas] target: " + self.obs_targ

            print(plane_id)
            self.plotCubeSlice(cube, 1, plane_id, self.subplot_cube, self.title_cube_obj, self.tooltip_cube_obj)

        # This function creates all the subplots. It is responsible for the plotting
        # layouts.
        # There can different layouts, depending on the availability of data
        # Note that subplot(I,J,K) means the Kth plot in a IxJ grid
        # Note also that the last one is actually a box with text, no graphs.
        def addSubplots(self, figure):
            nrows = self.nima
            row = 1
            # print "nrows=",nrows,row
            # print "resid_tab_found",self.resid_tab_found, "frameON_found",self.frameON_found
            # self.figsize(80,100)
            print("------------------------------------------")
            print("self.coadd_obj_found", self.coadd_obj_found)
            print("self.coadd_std_found", self.coadd_std_found)
            print("self.coadd_psf_found", self.coadd_psf_found)
            if self.coadd_obj_found is True or \
                    self.coadd_std_found is True or \
                    self.coadd_psf_found is True:
                print("coadd cube found")
                print("========================================")
                print("self.mean_coadd_obj_found", self.mean_coadd_obj_found)
                print("self.mean_coadd_std_found", self.mean_coadd_std_found)
                print("self.mean_coadd_psf_found", self.mean_coadd_psf_found)
                self.subplot_cube = figure.add_subplot(2, 2, 1)
                if self.mean_coadd_obj_found is True or \
                        self.mean_coadd_std_found is True or \
                        self.mean_coadd_psf_found is True:
                    print("mean coadd cube found")
                    self.subplot_mean_coadd = figure.add_subplot(2, 2, 2)

                if self.spectrum_found is True:
                    self.subplot_obj_spect = figure.add_subplot(2, 2, 3)
                    self.has_subplot_obj_spect = True
                # self.subplot_obj_stack  = figure.add_subplot(2,2,4)
                row += 1

                self.cube_plane_selector_is_created = 'True'
                self.cube_plane_selector = figure.add_axes([0.60, 0.01, 0.3, 0.01])
                self.cube_selector = figure.add_axes([0.50, 0.065, 0.458, 0.20])
                if self.nPro1D > 1:
                    self.spect_selector = figure.add_axes([0.50, 0.270, 0.458, 0.20])
                # Set the colormap and norm to correspond to the data for which
                # the colorbar will be used.

                cmap = matplotlib.cm.afmhot
                norm1 = matplotlib.colors.Normalize(vmin=0, vmax=self.NAXIS3)
                norm2 = matplotlib.colors.Normalize(vmin=self.crval3, vmax=self.wmax)
                figure.subplots_adjust(bottom=0)
                figure.subplots_adjust(top=1)
                figure.subplots_adjust(right=1)
                figure.subplots_adjust(left=0)
                # the following has problems with py2.7.5 (Paranal)
                # figure.tight_layout(pad=0.4, w_pad=0.5, h_pad=1.0)

                # add colorbar
                # cb1 = matplotlib.colorbar.ColorbarBase(self.cube_plane_pix_range,
                # cmap=cmap,norm=norm1,orientation='horizontal')
                # cb2 = matplotlib.colorbar.ColorbarBase(self.cube_plane_wav_range,
                # cmap=cmap,norm=norm2,orientation='horizontal')

                # self.product_selector = figure.add_axes([0.1, 0.17, 0.30, 0.13])

                if self.object_nodding_stacked_found is True:
                    print("object nodding stacked found")
                    # self.subplot_object_stacked  = figure.add_subplot(2,2,3)

                if self.coadd_obj_found is True:
                    print("med coadd obj found")
                    # self.subplot_ima_diff  = figure.add_subplot(2,2,4)

                elif self.enc_energy_found is True:
                    print('')

            else:
                print("found no spectrum data")
                self.dpm = eris_ifu_plot_common.DataPlotterManager()

                self.subtext_nodata = figure.add_subplot(1, 1, 1)
                self.caseNodata()
            # print "nrows=",nrows,rowself.rec_subtitl

        def caseNodata(self):
            # Data not found info
            self.subtext_nodata.set_axis_off()
            self.text_nodata = """Data not found! 
This may be due to a recipe failure. 
Check your input parameters, 
correct possible typos 
and press 'Re-run recipe' button."""
            self.subtext_nodata.text(0.1, 0.6, self.text_nodata, color='#11557c',
                                     fontsize=18, ha='left', va='center', alpha=1.0)
            self.subtext_nodata.tooltip = """\
                                    Merged spectrum not found in the products"""
            # print "found no spectrum data"

        def displayProducts(self):
            self.tooltip_cube_obj = """\
                        The displayed image corresponds to the
        plane id indicated by the slider below. To view a different
        plane of the science reconstructed cube, the user should select
        the desired plane id with the mouse left button, using the
        slider shown below the image.  
                              """
            # print("product id:",self.product_id)
            if self.product_id == 'CUBE':
                self.cube = self.coadd_obj
                self.setDisplayCubeSlice(self.plane_id, self.cube)
            elif self.product_id == 'DAR_CORRECTED_CUBE':
                self.cube = self.coadd_obj
                self.setDisplayCubeSlice(self.plane_id, self.cube)
            elif self.product_id == 'OBJECT_CUBE':
                self.cube = self.obj
                self.setDisplayCubeSlice(self.plane_id, self.cube)
            elif self.product_id == 'DAR_CORRECTED_CUBE_COADD':
                self.cube = self.coadd_obj
                self.setDisplayCubeSlice(self.plane_id, self.cube)
            elif self.product_id == 'OBJECT_CUBE_COADD':
                self.cube = self.coadd_obj
                self.setDisplayCubeSlice(self.plane_id, self.cube)
            elif self.product_id == 'OBJECT_CUBE_MEAN':
                self.cube = self.coadd_obj
                self.cube_mean = self.mean_coadd_obj
                self.plotImage(self.cube_mean, self.subplot_mean_coadd, title, tooltip)
            elif self.product_id == 'STD_CUBE':
                self.cube = self.coadd_std
                self.setDisplayCubeSlice(self.plane_id, self.cube)
            elif self.product_id == 'STD_FLUX_CUBE':
                self.cube = self.coadd_std
                self.setDisplayCubeSlice(self.plane_id, self.cube)
            elif self.product_id == 'STD_CUBE_COADD':
                self.cube = self.coadd_std
                self.setDisplayCubeSlice(self.plane_id, self.cube)
            elif self.product_id == 'OBJECT_CUBE_COADD_FLUXCAL':
                self.cube = self.coadd_obj_fluxcal
                self.setDisplayCubeSlice(self.plane_id, self.cube)
            elif self.product_id == 'STD_FLUX_CUBE_COADD':
                self.cube = self.coadd_std
                self.setDisplayCubeSlice(self.plane_id, self.cube)
            elif self.product_id == 'STD_CUBE_COADD_NOFLAT':
                self.cube = self.coadd_std
                self.setDisplayCubeSlice(self.plane_id, self.cube)
            elif self.product_id == 'STD_CUBE_MEAN':
                self.cube = self.coadd_obj
                self.cube_mean = self.mean_coadd_std
                self.plotImage(self.cube_mean, self.subplot_mean_coadd, title, tooltip)
            elif self.product_id == 'PSF_CUBE':
                self.cube = self.coadd_psf
                self.setDisplayCubeSlice(self.plane_id, self.cube)
            elif self.product_id == 'PSF_CUBE_COADD':
                self.cube = self.coadd_psf
                self.setDisplayCubeSlice(self.plane_id, self.cube)
            elif self.product_id == 'PSF_CUBE_MEAN':
                self.cube = self.coadd_psf
                self.cube_mean = self.median_coadd_psf
                self.plotImage(self.cube_mean, self.subplot_mean_coadd, title, tooltip)
            elif self.product_id == 'SKY_CUBE':
                self.cube = self.sky
                self.setDisplayCubeSlice(self.plane_id, self.cube)

            if (self.pro1D_id == 'SPECTRUM' or self.pro1D_id == 'SPECTRUM_FLUXCAL') and \
                    self.spectrum_found is True and self.has_subplot_obj_spect is True:
                if self.pro1D_id == 'SPECTRUM':
                    self.spectrum = self.spectra
                    title_spectrum = "extracted spectrum"
                    self.yLab = 'Flux [ADU]'
                    self.x = 'WAVE'
                    self.y = 'FLUX'
                    self.e = 'ERR'
                    self.q = None
                    self.error_id = 'yes'
                    self.isPhase3 = True
                else:
                    self.spectrum = self.spectra_fluxcal
                    title_spectrum = """extracted flux calibrated 
                              spectrum"""
                    self.yLab = 'Flux [erg/cm2/s/AA]'
                    self.x = 'wavelength'
                    self.y = 'flux'
                    self.e = 'flux_error'
                    # self.e=None
                    self.q = 'flux_bpm'
                    self.error_id = 'no'
                    self.isPhase3 = False
                tooltip_spectrum = title_spectrum
                self.xLab = 'Wavelength [nm]'
                self.decode_bp = 2147483647
                print('yerr', self.e, 'phase3 format', self.isPhase3)
                wave = None
                self.dpm.plotTableErrQual(self.spectrum, self.x, self.y, self.e, self.q, self.decode_bp, self.error_id,
                                          self.xLab, self.yLab, self.subplot_obj_spect, title_spectrum,
                                          tooltip_spectrum, wave, self.isPhase3)

            elif self.pro1D_id == 'RESPONSE':
                self.spectrum = self.response
                title_spectrum = "response"
                tooltip_spectrum = title_spectrum
                self.xLab = 'Wavelength [nm]'
                self.yLab = 'Response'
                self.error_id = 'no'
                self.fits_ext = 4
                self.decode_bp = 2147483647
                self.x = 'wavelength'
                self.y = 'response_poly'
                self.e = 'response_poly_error'
                self.q = None

                self.dpm.plotTableErrQual(self.spectrum, self.x, self.y, self.e, self.q, self.decode_bp, self.error_id,
                                          self.xLab, self.yLab, self.subplot_obj_spect, title_spectrum,
                                          tooltip_spectrum, fits_ext=self.fits_ext)

            elif self.pro1D_id == 'EFFICIENCY':
                self.spectrum = self.efficiency
                title_spectrum = "efficiency"
                tooltip_spectrum = title_spectrum
                self.xLab = 'Wavelength [nm]'
                self.yLab = 'Efficiency'
                self.x = 'WAVE'
                self.y = 'EFF'
                self.dpm.plotTable(self.spectrum, self.x, self.y, self.xLab, self.yLab, self.subplot_obj_spect,
                                   title_spectrum, tooltip_spectrum)

        # This is the function that makes the plots.
        # Add new plots or delete them using the given scheme.
        # The data has been already stored in self.plot_x, self.plot_xdif, etc ...
        # It is mandatory to add a tooltip variable to each subplot.
        # One might be tempted to merge addSubplots() and plotProductsGraphics().
        # There is a reason not to do it: addSubplots() is called only once at
        # startup, while plotProductsGraphics() is called always there is a resize.
        def plotProductsGraphics(self):
            # print "arc check:",  self.resid_tab_found, self.frameON_found
            # print "mfat check:",  self.edges_tab_found, self.mflat_found
            if self.coadd_obj_found is True:
                # and self.lineguess_found == True
                title_pref = 'Linear-extracted and Merged Spectrum.'

                tooltip_spectrum = """\
        Plot of the linear-extracted and merged spectrum of the object 
        (blue line) as total flux (ADU) versus wavelength (nm). 
        The  +-1 sigma uncertainties are plotted as upper and lower
        red curves. Note that this spectrum is not flux calibrated.
        """
            # print("coadd_obj_found=",self.coadd_obj_found)
            if self.coadd_obj_found is True or \
                    self.coadd_std_found is True or \
                    self.coadd_psf_found is True:

                # print("prepare cube display")
                plane_id = int(numpy.floor(self.NAXIS3 / 2 + 0.5))
                # self.wcen= self.crval3+(plane_id-1)*self.cdelt3

                self.tooltip_cube_obj = """\
                            The displayed image corresponds to the
        plane id indicated by the slider below. To view a different
        plane of the science reconstructed cube, the user should select
        the desired plane id with the mouse left button, using the
        slider shown below the image.  
                                      """
                title_traces_ifu_slices = 'IFU slices traces'
                tooltip_traces_ifu_slices = """\
                          Traces of the IFU object central position
        observed by slicer number 1,2,3 are displayed respectively in
        colors blue, green, red. If the cube has been perfectly 
        reconstructed the three traces should overlap.
                                      """

                # self.setDisplayCubeSlice(plane_id)
                print("display plane: ", plane_id)
                print("self.spectrum_found", self.spectrum_found)
                if self.spectrum_found is True:
                    self.dpm = eris_ifu_plot_common.DataPlotterManager()
                    self.xLab = 'Wavelength [nm]'
                    self.yLab = 'Flux [ADU]'
                    self.error_id = 'no'
                    self.decode_bp = 2147483647
                    self.x = 'WAVE'
                    self.y = 'FLUX'
                    self.e = 'ERR'
                    self.q = None
                    print('display spectrum')

                    tooltip_spectrum = """\
                            The displayed spectrum corresponds to the
        extraction of the coadded cube on an apertured centered on the object.
                                      """
                    title_spectrum = 'extracted spectrum'

                    self.dpm.plotTableErrQual(self.spectra, self.x, self.y, self.e, self.q, self.decode_bp,
                                              self.error_id, self.xLab, self.yLab, self.subplot_obj_spect,
                                              title_spectrum, tooltip_spectrum, isPhase3=True)
                    # self.dpm.plotTableScatter(self.spectra,'wavelength','counts_bkg', 'Wavelength [nm]','Flux [ADU]',
                    # self.subplot_obj_spect,title_traces_ifu_slices,tooltip_traces_ifu_slices)
                    # self.oplotTableScatter(self.traces_ifu_slices,'WAVELENGTH','POS_2',
                    # self.subplot_traces_ifu_slices,color='green')
                    # self.oplotTableScatter(self.traces_ifu_slices,'WAVELENGTH','POS_3',
                    # self.subplot_traces_ifu_slices,color='red')

                self.displayProducts()
            else:
                self.caseNodata()
                # pass

            # print(self.mean_coadd_obj_found)
            # print(self.median_coadd_std_found)
            # print(self.median_coadd_psf_found)
            # print('pippo')
            self.displayProducts()
            if (self.mean_coadd_obj_found is True or
                self.mean_coadd_std_found is True or
                self.mean_coadd_psf_found is True) and self.has_cube_mean is True:
                # print("prepare image display")
                title = 'collapsed mean'

                tooltip = """\
                            The displayed image corresponds to collapsed mean of the coadded cube.
                                      """
                self.plotImage(self.cube_mean, self.subplot_mean_coadd, title, tooltip)

            if self.object_nodding_stacked_found is True and self.product_id is "stacked frame":
                title = 'stacked frame'

                tooltip = """\
                            The displayed image corresponds to the
         plane id indicated by the slider below. To view a different
         plane of the science reconstructed cube, the user should select
         the desired plane id with the mouse left button, using the
         slider shown below the image.  
                                      """
                self.plotImage(self.object_stacked, self.subplot_obj_stack, title, tooltip)

        # This function specifies which are the parameters that should be presented
        # in the window to be edited.
        # Note that the parameter has to be also in the in_sop port (otherwise it
        # won't appear in the window)
        # The descriptions are used to show a tooltip. They should match one to one
        # with the parameter list
        # Note also that parameters have to be prefixed by the 'recipe name:'

        def setJitterGeneralParameters(self, paramList, rec_id):

            group = "general"
            help1 = "Specifies the product output depth (>0 for auxiliary products). [0]"
            paramList.append(RecipeParameter(rec_id, displayName="product_depth", group=group, description=help1))

            help2 = "If TRUE raw exposure image line correction will be applied. [TRUE]"
            paramList.append(RecipeParameter(rec_id, displayName="line_corr", group=group, description=help2))

            help3 = "If TRUE raw exposure image cosmic ray hit correction will be applied. [FALSE]"
            paramList.append(RecipeParameter(rec_id, displayName="crh_corr", group=group, description=help3))

            help4 = "If TRUE raw exposure image cosmic ray hit detection will be applied. [FALSE]"
            paramList.append(RecipeParameter(rec_id, displayName="crh_detection", group=group, description=help4))

            help5 = "Poisson fluctuation threshold to flag cosmics(see van Dokkum, PASP,113,2001,p1420-27). [5.0]"
            paramList.append(RecipeParameter(rec_id, displayName="crh.sigma_lim", group=group, description=help5))

            help6 = "Minimum contrast between the Laplacian image and the fine structure image that a point must " \
                    "have to be flagged as cosmics. [2.0]"
            paramList.append(RecipeParameter(rec_id, displayName="crh.f_lim", group=group, description=help6))

            help7 = "Maximum number of alghoritm iterations. [3]"
            paramList.append(RecipeParameter(rec_id, displayName="crh.max_iter", group=group, description=help7))

            help8 = "Maximum distance between cube centers to build a mosaic. Mosaic creation requires a lot of RAM. " \
                    "Users may trim this value to fit RAM resources."
            paramList.append(
                RecipeParameter(rec_id, displayName="max-cubes-centres-dist", group=group, description=help8))

            help9 = "Derotator correction. To be changed in case of (small) shifts between the observed object and " \
                    "the FIBRE-NS raw frames. [0.0]"
            paramList.append(RecipeParameter(rec_id, displayName="derot_corr", group=group, description=help9))

            help10 = "If True flux calibrate the extracted spectrum and data cube. [FALSE]"
            paramList.append(RecipeParameter(rec_id, displayName="flux-calibrate", group=group, description=help10))

            help11 = "No. of iterations for bad pixel correction. [1]."
            paramList.append(RecipeParameter(rec_id, displayName="bpc_iter", group=group, description=help11))

        def setJitterSkyCorrectionParameters(self, paramList, rec_id):
            group = "sky_corr"
            help8 = "Use modified sky cube for sky subtraction.0: don't apply, 1: Davies' method, 2: " \
                    "Austrian method). [0]"
            paramList.append(RecipeParameter(rec_id, displayName="sky_tweak", group=group, description=help8))

            help9 = "Skip the OH alignment for the SKY. [FALSE]"
            paramList.append(RecipeParameter(rec_id, displayName="skip_sky_oh_align", group=group, description=help9))

            help10 = "Ignore last sub-band in the sky tweaking. [FALSE]"
            paramList.append(RecipeParameter(rec_id, displayName="discard_subband", group=group, description=help10))

            help11 = "Stretch sky in the sky tweaking. [FALSE]"
            paramList.append(RecipeParameter(rec_id, displayName="stretch", group=group, description=help11))

            help12 = "Stretch polynomial degree. [8]"
            paramList.append(RecipeParameter(rec_id, displayName="stretch_degree", group=group, description=help12))

            help13 = "Stretch resampling method (linear/spline). [spline]"
            paramList.append(RecipeParameter(rec_id, displayName="stretch_resampling", group=group, description=help13))

            help14 = "Subtract thermal background from input cube.(TRUE (apply) or FALSE (don't apply). [TRUE]"
            paramList.append(RecipeParameter(rec_id, displayName="tbsub", group=group, description=help14))

            help15 = "Specify velocity offset correction in km/s for lambda scale. [0.0]"
            paramList.append(RecipeParameter(rec_id, displayName="velocity_offset", group=group, description=help15))

            help16 = "Specify a theshold for the interpolated BPM values. [0.5]"
            paramList.append(RecipeParameter(rec_id, displayName="bpm_threshold", group=group, description=help16))

            help17 = "Specifies the slitlet detection: 'DIST' slitlet distances as detected by distortion recipe or " \
                     "'EDGE' slitlet edges as detected be wavecalrecipe. <DIST | EDGE | GRID> [DIST]"
            paramList.append(
                RecipeParameter(rec_id, displayName="cube.slitlet-detection", group=group, description=help17))

            help18 = """Method used to estimate the sky in case of missing sky frames.
                    0: no sky subtraction;
                    1: sky is taken from the next in MJD-OBS object frame;
                    2: sky is taken from the median of all input sky frames;
                    3: sky is taken from the mean of all input sky frames;
                    4: sky is taken from the median of user spec. rect. box;
                    5: sky is taken from the median of 4 regions at FOV edges;
                    6: sky is taken from the median of 8 regions at FOV edges;
                    7: sky is taken from each cube slice based on mask computed on collapsed cube.
                    8: sky is taken from each cube slice based on 25% clip based mask computed on collapsed cube. [1]"""
            paramList.append(RecipeParameter(rec_id, displayName="aj-method", group=group, description=help18))

            help19 = "The centre of the rectangular sky region (pixel). two integer numbers each in the range " \
                     "[1,64]. [32,32]"
            paramList.append(RecipeParameter(rec_id, displayName="sky-box-center", group=group, description=help19))

            help20 = "The rectangular (small) sky region x,y half widths (pixel).Two numbers each in the range " \
                     "[0,10]. [1,1]"
            paramList.append(RecipeParameter(rec_id, displayName="sky-box-width", group=group, description=help20))

            help21 = """The (small)number of x, y pixels taken as marging to the FOV edges so that sky regions 
            (aj-method=5,6) do not include pixels at the edges of the re-sampled image [pixel]. Two numbers each in 
            the range [0,5].The rectangular (small) sky region x,y half widths (pixel).Two numbers each in the range 
            [0,5]. [1,1]"""
            paramList.append(
                RecipeParameter(rec_id, displayName="sky-box-edges-margin", group=group, description=help21))

            help22 = "If the user has no input sky frame and the sky need to be estimated from user defined pixels " \
                     "set by aj-method this parameter allow to use the median (0) or the mean (1) [0]."
            paramList.append(RecipeParameter(rec_id, displayName="sky-est-method", group=group, description=help22))

            help23 = "If the user has no input sky frame and the sky need to be estimated from user defined pixels " \
                     "set by aj-method this parameter allow to use the kappa of the kappa sigma clip iteration [3]."
            paramList.append(RecipeParameter(rec_id, displayName="sky-est-kappa", group=group, description=help23))

            help24 = "If the user has no input sky frame and the sky need to be estimated from user defined pixels " \
                     "set by aj-method this parameter allow to set the number of the kappa sigma clip iterations [5]."
            paramList.append(RecipeParameter(rec_id, displayName="sky-est-niter", group=group, description=help24))

        def setJitterCubeCreationParameters(self, paramList, rec_id):
            group = "cube_creation"

            help18 = "Specifies the first column offset in case the cube.slitlet-detection parameter is set to " \
                     "'DIST'. [1.0]"
            paramList.append(RecipeParameter(rec_id, displayName="cube.first-col", group=group, description=help18))

            help19 = "Specifies the row interpolation mode: 0 -> no interpolation, otherwise see " \
                     "eris_ifu_1d_interpolation. [-1]"
            paramList.append(RecipeParameter(rec_id, displayName="cube.fine-tune", group=group, description=help19))

            name20 = "cube.combine"
            help20 = "With multi-pointing observations combine cubes into a mosaic. [TRUE]"
            paramList.append(RecipeParameter(rec_id, displayName=name20, group=group, description=help20))

        def setJitterCubeDarParameters(self, paramList, rec_id):
            group = "dar_correction"
            help1 = "Correct Differential Atmospheric Refraction (DAR)"
            paramList.append(RecipeParameter(rec_id, displayName="dar-corr", group=group, description=help1))

            help2 = "The method to shift images for DAR correction."
            paramList.append(RecipeParameter(rec_id, displayName="dar-shift-method", group=group, description=help2))

            help3 = "Kernel length for DAR shift image interpolation. [0]"
            paramList.append(RecipeParameter(rec_id, displayName="dar-shift-length", group=group, description=help3))

            help4 = "Kernel radius for DAR shift image interpolation. [0.0]"
            paramList.append(RecipeParameter(rec_id, displayName="dar-shift-radius", group=group, description=help4))

        def setJitterCubeResamplingParameters(self, paramList, rec_id):
            group = "resampling"

            help20 = "Resampling method. <NEAREST | LINEAR | QUADRATIC | RENKA | DRIZZLE | LANCZOS> [LINEAR]"
            paramList.append(RecipeParameter(rec_id, displayName="method", group=group, description=help20))

            help21 = "Loop distance used by all (but NEAREST) methods to control the number of surrounding voxels " \
                     "that are taken into account. [1]"
            paramList.append(
                RecipeParameter(rec_id, displayName="method.loop-distance", group=group, description=help21))

            help22 = "Use additional weights of 1/err^2. [FALSE]"
            paramList.append(
                RecipeParameter(rec_id, displayName="method.use-errorweights", group=group, description=help22))

            help23 = "Critical radius of the Renka method. [1.25]"
            paramList.append(
                RecipeParameter(rec_id, displayName="method.renka.critical-radius", group=group, description=help23))

            help24 = "Kernel size of the Lanczos method. [2]"
            paramList.append(
                RecipeParameter(rec_id, displayName="method.lanczos.kernel-size", group=group, description=help24))

            help25 = "Drizzle down-scaling factor in x direction. [0.8]"
            paramList.append(
                RecipeParameter(rec_id, displayName="method.drizzle.downscale-x", group=group, description=help25))

            help26 = "Drizzle down-scaling factor in y direction. [0.8]"
            paramList.append(
                RecipeParameter(rec_id, displayName="method.drizzle.downscale-y", group=group, description=help26))

            help27 = "Drizzle down-scaling factor in z direction. [0.8]"
            paramList.append(
                RecipeParameter(rec_id, displayName="method.drizzle.downscale-z", group=group, description=help27))

            help28 = "Minimum right ascension of the output image/cube in degree. [-1.0]"
            paramList.append(RecipeParameter(rec_id, displayName="outgrid.ra-min", group=group, description=help28))

            help29 = "Maximum right ascension of the output image/cube in degree. [-1.0]"
            paramList.append(RecipeParameter(rec_id, displayName="outgrid.ra-max", group=group, description=help29))

            help30 = "Minimum declination of the output image/cube in degree. [-1.0]"
            paramList.append(RecipeParameter(rec_id, displayName="outgrid.dec-min", group=group, description=help30))

            help31 = "Maximum declination of the output image/cube in degree. [-1.0]"
            paramList.append(RecipeParameter(rec_id, displayName="outgrid.dec-max", group=group, description=help31))

            help32 = "Minimum wavelength of the output cube in meter. [-1.0]"
            paramList.append(RecipeParameter(rec_id, displayName="outgrid.lambda-min", group=group, description=help32))

            help33 = "Maximum wavelength of the output cube in meter. [-1.0]"
            paramList.append(RecipeParameter(rec_id, displayName="outgrid.lambda-max", group=group, description=help33))

            help34 = "Output step-size in right ascension in degree. [-1.0]"
            paramList.append(RecipeParameter(rec_id, displayName="outgrid.delta-ra", group=group, description=help34))

            help35 = "Output step-size in declination in degree. [-1.0]"
            paramList.append(RecipeParameter(rec_id, displayName="outgrid.delta-dec", group=group, description=help35))

            help36 = "Output step-size in wavelength in meter. [-1.0]"
            paramList.append(
                RecipeParameter(rec_id, displayName="outgrid.delta-lambda", group=group, description=help36))

            help37 = "Save the table before resampling. [FALSE]"
            paramList.append(RecipeParameter(rec_id, displayName="save-table", group=group, description=help37))

            help38 = "Subtract median of the images in 2D only. [TRUE]"
            paramList.append(
                RecipeParameter(rec_id, displayName="subtract-background", group=group, description=help38))

            help39 = "Ad this margin/border (in percent) to the resampled image/cube. [5.0]"
            paramList.append(RecipeParameter(rec_id, displayName="fieldmargin", group=group, description=help39))

            help40 = "Number or pixels to trim for each plane of the input frames. It should be smaller than half " \
                     "image size. [0]"
            paramList.append(RecipeParameter(rec_id, displayName="edge-trim", group=group, description=help40))

        def setInteractiveParameters(self):
            paramList = list()
            rec_id = self.rec_id
            self.par = eris_ifu_parameters_common.Parameters()

            if rec_id == "eris_ifu_wavecal":
                self.par.setCommonParameters(paramList, rec_id, "common")
                self.setWavecalParameters(paramList, rec_id)

            if (rec_id == "eris_ifu_stdstar" or rec_id == "eris_ifu_jitter"):
                # self.par.setJitterParameters(paramList,"eris_ifu_jitter","jitter")
                self.setJitterGeneralParameters(paramList, rec_id)
                self.setJitterSkyCorrectionParameters(paramList, rec_id)
                self.setJitterCubeCreationParameters(paramList, rec_id)
                self.setJitterCubeDarParameters(paramList, rec_id)
                self.setJitterCubeResamplingParameters(paramList, rec_id)

            rec_supported = ['eris_ifu_jitter', 'eris_ifu_stdstar']
            if rec_id not in rec_supported:
                print("recipe", rec_id, "not supported")

            return paramList

        def setWindowHelp(self):
            help_text = """
This is an interactive window which help asses the quality of the execution of a recipe.
"""
            return help_text

        def setWindowTitle(self):
            # title = 'X-shooter Interactive ' + self.rec_subtitle + self.seq_arm + ' Arm. '
            title = 'ERIS-SPIFFIER Jitter Interactive GUI'
            return title

except ImportError:
    import_sucess = 'false'
    print("Error importing modules pyfits, wx, matplotlib, numpy")
    raise
