from __future__ import with_statement
import sys

try:
    import numpy
    import reflex
    from pipeline_product import PipelineProduct
    from reflex_plot_widgets import *
    import pipeline_display

    import_success = True

except ImportError:
    import_success = False
    print("Error importing modules pyfits, wx, matplotlib, numpy")


def paragraph(text, width=None):
    """ wrap text string into paragraph
       text:  text to format, removes leading space and newlines
       width: if not None, wraps text, not recommended for tooltips as
              they are wrapped by wxWidgets by default
    """
    import textwrap

    if width is None:
        return textwrap.dedent(text).replace("\n", " ").strip()
    else:
        return textwrap.fill(textwrap.dedent(text), width=width)


class DataPlotterManager(object):
    """
    This class must be added to the PipelineInteractiveApp with setPlotManager
    It must have following member functions which will be called by the app:
     - setInteractiveParameters(self)
     - readFitsData(self, fitsFiles):
     - addSubplots(self, figure):
     - plotProductsGraphics(self)
    Following members are optional:
     - setWindowHelp(self)
     - setWindowTitle(self)
     - setCurrentParameterHelper(self, helper)
    """

    # static members
    recipe_name = "eris_nix_dark"
    img_cat = "MASTER_DARK_IMG"

    def setInteractiveParameters(self):
        """
        This function specifies which are the parameters that should be 
        presented in the window to be edited.  Note that the parameter has 
        to also be in the in_sop port (otherwise it won't appear in the 
        window). The descriptions are used to show a tooltip. They should 
        match one to one with the parameter list.
        """
        return [
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="collapse.method",
                group="collapse",
                description="Method used for collapsing "
                            "the data.<MEAN | WEIGHTED_MEAN | MEDIAN | "
                            "SIGCLIP | MINMAX>",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="collapse.sigclip.kappa-low",
                group="collapse",
                description="Low kappa factor for kappa-sigma " "clipping algorithm.",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="collapse.sigclip.kappa-high",
                group="collapse",
                description="High kappa factor for kappa-sigma " "clipping algorithm.",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="collapse.sigclip.niter",
                group="collapse",
                description="Maximum number of clipping iterations "
                            "for kappa-sigma clipping.",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="hotpix.method",
                group="hotpix",
                description="Method used. <FILTER | LEGENDRE>",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="hotpix.filter.kappa-low",
                group="hotpix",
                description="Low RMS scaling factor for image " "thresholding.",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="hotpix.filter.kappa-high",
                group="hotpix",
                description="High RMS scaling factor for image " "thresholding.",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="hotpix.filter.maxiter",
                group="hotpix",
                description="Maximum number of algorithm iterations.",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="hotpix.filter.filter",
                group="hotpix",
                description="Filter mode for image smoothing. "
                            "<AVERAGE | AVERAGE_FAST | MEDIAN>",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="hotpix.filter.border",
                group="hotpix",
                description="Border mode to use for the image "
                            "smoothing filter (only for MEDIAN "
                            "filter). <FILTER | CROP | NOP | COPY> ",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="hotpix.filter.smooth-x",
                group="hotpix",
                description="Kernel x size of the smoothing filter.",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="hotpix.filter.smooth-y",
                group="hotpix",
                description="Kernel y size of the smoothing filter.",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="hotpix.legendre.kappa-low",
                group="hotpix",
                description="Low RMS scaling factor for image " "thresholding.",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="hotpix.legendre.kappa-high",
                group="hotpix",
                description="High RMS scaling factor for image " "thresholding.",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="hotpix.legendre.maxiter",
                group="hotpix",
                description="Maximum number of algorithm iterations.",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="hotpix.legendre.steps-x",
                group="hotpix",
                description="Number of image sampling " "points in x-dir for fitting.",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="hotpix.legendre.steps-y",
                group="hotpix",
                description="Number of image sampling " "points in y-dir for fitting.",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="hotpix.legendre.filter-size-x",
                group="hotpix",
                description="X size of the median box " "around sampling points.",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="hotpix.legendre.filter-size-y",
                group="hotpix",
                description="Y size of the median box " "around sampling points.",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="hotpix.legendre.order-x",
                group="hotpix",
                description="Order of x polynomial for " "the fit.",
            ),
            reflex.RecipeParameter(
                recipe=self.recipe_name,
                displayName="hotpix.legendre.order-y",
                group="hotpix",
                description="Order of y polynomial for " "the fit.",
            ),
        ]

    def readFitsData(self, fitsFiles):
        """
        This function should be used to read and organize the raw fits files
        produced by the recipes.
        It receives as input a list of reflex.FitsFiles
        """
        # organize the files into a dictionary, here we assume we only have
        # one file per category if there are more, one must use a
        # dictionary of lists
        self.frames = dict()
        for f in fitsFiles:
            self.frames[f.category] = PipelineProduct(f)

        # we only have two states, we have data or we don't
        # define the plotting functions we want to use for each
        if self.img_cat in self.frames:
            self._add_subplots = self._add_data_subplots
            self._plot = self._data_plot
        else:
            self._add_subplots = self._add_nodata_subplots
            self._plot = self._nodata_plot

    def addSubplots(self, figure):
        """
        This function should be used to setup the subplots of the gui.  The the
        matplotlib documentation for a description of subplots.
        """
        self._add_subplots(figure)

    def plotProductsGraphics(self):
        """
        This function should be used to plot the data onto the subplots.
        """
        self._plot()

    def setWindowHelp(self):
        return "Help for eris_nix_dark interactive window"

    def setWindowTitle(self):
        return "eris_nix_dark interactive window"

    def _add_nodata_subplots(self, figure):
        self.txt_plot = figure.add_subplot(111)

    def _add_data_subplots(self, figure):
        self.img_data_plot = figure.add_subplot(221)
        self.img_err_plot = figure.add_subplot(222)
        self.img_dq_plot = figure.add_subplot(223)
        self.img_hot_bpm_plot = figure.add_subplot(224)

    def _data_plot(self):
        # get the right category file from our dictionary

        p = self.frames[self.img_cat]
        p.readImage("DATA")
        # setup the image display
        imgdisp = pipeline_display.ImageDisplay()
        imgdisp.setLabels("X", "Y")
        imgdisp.setAspect("equal")
        tooltip = paragraph(
            """\
        MASTER_DARK_IMG data.
        """
        )
        imgdisp.display(self.img_data_plot, "MASTER_DARK_IMG data", tooltip, p.image)

        p.readImage("ERR")
        imgdisp = pipeline_display.ImageDisplay()
        imgdisp.setLabels("X", "Y")
        imgdisp.setAspect("equal")
        tooltip = paragraph(
            """\
        MASTER_DARK_IMG error plane.
        """
        )
        imgdisp.setZAutoLimits(p.image, None)
        if imgdisp.z_lim[0] == imgdisp.z_lim[1]:
            imgdisp.z_lim = imgdisp.z_lim[0] - 0.1, imgdisp.z_lim[1] + 0.1
        imgdisp.display(self.img_err_plot, "MASTER_DARK_IMG error", tooltip, p.image)

        p.readImage("DQ")
        imgdisp = pipeline_display.ImageDisplay()
        imgdisp.setLabels("X", "Y")
        imgdisp.setAspect("equal")
        tooltip = paragraph(
            """\
        MASTER_DARK_IMG data quality plane.
        """
        )
        imgdisp.z_lim = 0, 1
        imgdisp.display(self.img_dq_plot, "MASTER_DARK_IMG quality", tooltip, p.image)

        p.readImage("HOT_BPM")
        imgdisp = pipeline_display.ImageDisplay()
        imgdisp.setLabels("X", "Y")
        imgdisp.setAspect("equal")
        tooltip = paragraph(
            """\
        MASTER_DARK_IMG 'hot pixel' bad pixel mask.
        """
        )
        imgdisp.z_lim = 0, 1
        imgdisp.display(
            self.img_hot_bpm_plot, "MASTER_DARK_IMG hot bpm", tooltip, p.image
        )

    def _nodata_plot(self):
        # could be moved to reflex library?
        self.txt_plot.set_axis_off()
        text_nodata = (
                "Data not found. Input files should contain these"
                " types:\n%s" % self.img_cat
        )
        self.txt_plot.text(
            0.1,
            0.6,
            text_nodata,
            color="#11557c",
            fontsize=18,
            ha="left",
            va="center",
            alpha=1.0,
        )
        self.txt_plot.tooltip = "No data found"

    def plotWidgets(self):
        widgets = list()
        #        if hasattr(self, 'spec_plot') {
        # Updating number
        #            self.clickablespec = InteractiveClickableSubplot(
        #                self.spec_plot, self.increaseFloatNumber)
        #            widgets.append(self.clickablespec)
        #        }
        return widgets

    def increaseFloatNumber(self, point):
        floatpoint = self.getCurrentParameterHelper("floatopt") + 1
        new_params = list()
        new_params.append(
            reflex.RecipeParameter(self.recipe_name, "floatopt", value=str(floatpoint))
        )
        return new_params

    def setCurrentParameterHelper(self, helper):
        self.getCurrentParameterHelper = helper


# This is the 'main' function
if __name__ == "__main__":
    from reflex_interactive_app import PipelineInteractiveApp

    # Create interactive application
    interactive_app = PipelineInteractiveApp(enable_init_sop=True)

    # get inputs from the command line
    interactive_app.parse_args()

    # Check if import failed or not
    if not import_success:
        interactive_app.setEnableGUI(False)

    # Open the interactive window if enabled
    if interactive_app.isGUIEnabled():
        # Get the specific functions for this window
        dataPlotManager = DataPlotterManager()

        interactive_app.setPlotManager(dataPlotManager)
        interactive_app.showGUI()
    else:
        interactive_app.set_continue_mode()

    # Print outputs. This is parsed by the Reflex python actor to
    # get the results. Do not remove
    interactive_app.print_outputs()
    sys.exit()
