from edps import (
    task, alternative_associated_inputs, ReportInput,
    qc1calib, idp, science, qc0, calchecker
)

from .eris_datasources import *
from .eris_image_quality_spiffier import image_quality_spiffier
from .eris_standard_star import standard_star
from .eris_task_function import force_linearity, force_dark, set_cube_combination

__title__ = "ERIS SPIFFIER workflow"

# ---------------- PROCESSING TASKS -----------------

# Compute the linearity correction for SPIFFIER observations
linearity_ifu = (task('linearity_ifu')
                 .with_recipe('eris_ifu_detlin')
                 .with_report("eris_rawdisp", ReportInput.RECIPE_INPUTS)
                 .with_report("eris_detmon", ReportInput.RECIPE_INPUTS_OUTPUTS)
                 .with_main_input(raw_linearity_ifu)
                 .with_meta_targets([qc1calib, calchecker])
                 .build())

# Processes raw darks for SPIFFIER observations
dark_ifu = (task('dark_ifu')
            .with_recipe('eris_ifu_dark')
            .with_report("eris_rawdisp", ReportInput.RECIPE_INPUTS)
            .with_report("eris_master_dark", ReportInput.RECIPE_INPUTS_OUTPUTS)
            .with_main_input(raw_dark_ifu)
            .with_meta_targets([qc1calib, calchecker])
            .build())

# Compute distortion correction
distortion_ifu = (task('distortion_ifu')
                  .with_recipe('eris_ifu_distortion')
                  .with_report("eris_rawdisp", ReportInput.RECIPE_INPUTS)
                  .with_report("eris_distortion", ReportInput.RECIPE_INPUTS_OUTPUTS)
                  .with_main_input(raw_distortion)
                  .with_associated_input(raw_distortion_assoc, max_ret=100)
                  .with_associated_input(ref_line_arc)
                  .with_associated_input(first_wave_fit)
                  .with_associated_input(wave_setup)
                  .with_meta_targets([qc1calib])
                  .build())

# Processes lamp flats for SPIFFIER observations
flat_ifu = (task('flat_ifu')
            .with_recipe('eris_ifu_flat')
            .with_report("eris_rawdisp", ReportInput.RECIPE_INPUTS)
            .with_report("eris_ifu_flat", ReportInput.RECIPE_INPUTS_OUTPUTS)
            .with_main_input(raw_lamp_flat_ifu)
            .with_associated_input(dark_ifu, [BPM_DARK_IFU], match_rules=match_dark_for_flat)
            .with_associated_input(linearity_ifu, [BPM_DETLIN_IFU, bpm_detlin_filt_class], min_ret=0)
            .with_associated_input(distortion_ifu, [bpm_dist_class, DISTORTION], min_ret=1)
            .with_input_filter(BPM_DARK_IFU, BPM_DETLIN_IFU, bpm_dist_class)
            .with_meta_targets([qc1calib, calchecker])
            .build())

# Compute wavelength calibration
wave_ifu = (task('wavelength_ifu')
            .with_recipe('eris_ifu_wavecal')
            .with_report("eris_rawdisp", ReportInput.RECIPE_INPUTS)
            .with_report("eris_wavelength", ReportInput.RECIPE_INPUTS_OUTPUTS)
            .with_main_input(raw_wave_ifu)
            .with_associated_input(distortion_ifu, [DISTORTION])
            .with_associated_input(flat_ifu, [MASTER_FLAT_LAMP, BPM_FLAT])
            .with_associated_input(first_wave_fit)
            .with_associated_input(ref_line_arc)
            .with_associated_input(wave_setup)
            .with_input_filter(DISTORTION, MASTER_FLAT_LAMP, BPM_FLAT)
            .with_meta_targets([qc1calib, calchecker])
            .build())

# Process flux and telluric standards
flux_standard_ifu, telluric_standard_ifu = standard_star(raw_flux_std_ifu, raw_telluric_std_ifu, distortion_ifu,
                                                         flat_ifu, wave_ifu, dark_ifu, linearity_ifu)

standard_star_calibrations_spiffier = (alternative_associated_inputs()
                                       .with_associated_input(flux_standard_ifu, [RESPONSE], min_ret=0)
                                       .with_associated_input(telluric_standard_ifu, [RESPONSE], min_ret=0))

persistence_spiffier, psf, strehl, pupil_spiffier = image_quality_spiffier(dark_ifu, distortion_ifu, flat_ifu, wave_ifu)

# --- Process scientific exposures and combine cubes within the same Observing Block.
science_ifu = (task('object_ifu')
               .with_recipe('eris_ifu_jitter')
               .with_report("eris_ifu_science", ReportInput.RECIPE_INPUTS_OUTPUTS)
               .with_main_input(raw_object_spiffier)
               .with_associated_input(raw_sky_spiffier, min_ret=0, max_ret=100)
               .with_associated_input(distortion_ifu, [DISTORTION])
               .with_associated_input(flat_ifu, [MASTER_FLAT_LAMP, BPM_FLAT])
               .with_associated_input(wave_ifu, [WAVE_MAP])
               .with_alternative_associated_inputs(standard_star_calibrations_spiffier)
               .with_associated_input(oh_spectrum)
               .with_associated_input(extcoeff_table, min_ret=0)
               .with_associated_input(response_windows, min_ret=0)
               .with_associated_input(dark_ifu, [MASTER_DARK_IFU, BPM_DARK_IFU], condition=force_dark)
               .with_associated_input(linearity_ifu, [BPM_DETLIN_IFU], condition=force_linearity)
               .with_job_processing(set_cube_combination)
               .with_input_filter(DISTORTION, MASTER_FLAT_LAMP, BPM_FLAT, WAVE_MAP,
                                  RESPONSE, response_windows_class, MASTER_DARK_IFU, BPM_DARK_IFU, BPM_DETLIN_IFU)
               .with_meta_targets([idp, science, qc0, calchecker])
               .build())
