/* $Id: eris_dfs.c,v 1.9 2013-03-26 17:00:44 jtaylor Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: jtaylor $
 * $Date: 2013-03-26 17:00:44 $
 * $Revision: 1.9 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "eris_ifu_dfs.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_functions.h"
#include "eris_ifu_error.h"
#include "eris_ifu_distortion_static.h"
#include "eris_ifu_debug.h"
#include "eris_utils.h"
#include <eris_pfits.h>
#include <string.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_ifu_dfs  IFU Data Flow System (DFS) Functions
 *
 * This module provides Data Flow System (DFS) compliant I/O functions for ERIS
 * IFU pipeline products, including functions for:
 * - Setting frame groups (RAW/CALIB/PRODUCT)
 * - Saving DFS-compliant FITS products (images, cubes, tables)
 * - Loading DEQ (Data/Error/Quality) format products
 * - Managing calibration frames and framesets
 * - Querying instrument configuration and observation parameters
 * - Appending quality control (QC) parameters
 *
 * All save functions create products compliant with ESO's Data Exchange
 * Qualifiers (DEQ) standard (VLT-SPE-ESO-19500-5667), which defines the
 * structure of data products with separate extensions for data, errors,
 * and data quality information.
 *
 * @par Synopsis:
 * @code
 *   #include "eris_ifu_dfs.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief    Set the frame group (RAW, CALIB, or PRODUCT) for all frames in a frameset
 * @param self   The frameset to process
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Iterates through all frames in the frameset and assigns the appropriate group
 * based on the frame's tag:
 * - RAW: Raw observation frames
 * - CALIB: Calibration frames (reference and product calibrations)
 * - PRODUCT: Pipeline product frames (e.g., from jitter recipe)
 *
 * @note The function also verifies that all frames are from the same instrument.
 * @note Frames without tags or with unknown tags will cause an error.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_dfs_set_groups(cpl_frameset * self)
{
    cpl_error_code  retVal          = CPL_ERROR_NONE;
    const cpl_size  n               = cpl_frameset_get_size(self);
    ifsInstrument   instrument      = UNSET_INSTRUMENT,
                    currInstrument  = UNSET_INSTRUMENT;
    cpl_frame       *frame          = NULL; /* Not allocated, just modified */
    const char      *tag            = NULL;
    char            *stag           = NULL;

    cpl_ensure_code(self, CPL_ERROR_NULL_INPUT);

    TRY
    {
        /* Loop on frames */
        for (cpl_size i = 0; i < n; i++) {
                BRK_IF_NULL(
                    frame = cpl_frameset_get_position(self, i));

                tag = cpl_frame_get_tag(frame);
            if (tag == NULL) {
                    BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
                                                    "Frame %d of %d has no tag",
                                1 + (int)i, (int)n);
            }
                cpl_free(stag);
                BRK_IF_NULL(
                    stag = cpl_sprintf(",%s,", tag));

            if (strstr(ERIS_IFU_RAW_TAGS, stag) != NULL) {
                /* SPIFFIER RAW frames */
                    BRK_IF_ERROR(
                        cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW));

                // check if all frames are from the same instrument

                if (instrument == UNSET_INSTRUMENT) {
                    instrument = eris_ifu_get_instrument_frame(frame);
                } else {
                    currInstrument = eris_ifu_get_instrument_frame(frame);
                    if (currInstrument != instrument) {
                            BRK_WITH_ERROR_MSG(
                                                        CPL_ERROR_BAD_FILE_FORMAT,
                                                        "SOF input files are from "
                                                        "different instruments");
                    }
                }

            } else if (strstr(ERIS_IFU_REF_CALIB_TAGS, stag) != NULL) {
                /* SPIFFIER reference CALIB frames */
                    BRK_IF_ERROR(
                        cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB));
            } else if (strstr(ERIS_IFU_REF_UTIL_TAGS, stag) != NULL) {
                /* SPIFFIER utility frames */
                    BRK_IF_ERROR(
                        cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB));
            } else if (strstr(ERIS_IFU_PRO_CALIB_TAGS, stag) != NULL) {
                /* SPIFFIER product CALIB frames */
                    BRK_IF_ERROR(
                        cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB));
            } else if (strstr(ERIS_IFU_PRO_JITTER_TAGS, stag) != NULL) {
                /* SPIFFIER product Jitter recipe frames */
                    BRK_IF_ERROR(
                        cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT));

                // check if all frames are from the same instrument
                if (instrument == UNSET_INSTRUMENT) {
                    instrument = eris_ifu_get_instrument_frame(frame);
                } else {
                    currInstrument = eris_ifu_get_instrument_frame(frame);
                    if (currInstrument != instrument) {
                            BRK_WITH_ERROR_MSG(
                                                        CPL_ERROR_BAD_FILE_FORMAT,
                                                        "SOF input files are from "
                                                        "different instruments");
                    }
                }

            } else {
                    BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
                                      "Unkown tag \"%s\" for frame %d of %d",
                        cpl_frame_get_tag(frame), 1 + (int)i, (int)n);
            }

        }
    }
    CATCH
    {
        retVal = cpl_error_get_code();
    }

    cpl_free(stag);
    eris_check_error_code("eris_ifu_dfs_set_groups");
    return retVal;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief    Extract frames with a specific tag from a frameset
 * @param in    Input non-empty frameset
 * @param tag   The tag of the frames to extract
 * @return   Newly created frameset containing only frames with the specified tag, or NULL on error
 *
 * Creates a new frameset containing duplicates of all frames from the input
 * frameset that match the specified tag. If no frames match, NULL is returned.
 *
 * @note The returned frameset must be deallocated with cpl_frameset_delete().
 * @note The frames in the returned frameset are duplicates, not references.
 */
/*----------------------------------------------------------------------------*/
cpl_frameset *
eris_ifu_extract_frameset(
        const cpl_frameset  *   in,
        const char          *   tag)
{
    cpl_frameset    *   out;
    cpl_frame       *   loc_frame;
    int                 nbframes;
    int                 i ;

    /* Test entries */
    if (in == NULL) return NULL ;
    if (tag == NULL) return NULL ;

    /* Initialise */
    nbframes = cpl_frameset_get_size(in) ;

    /* Count the frames with the tag */
    if (cpl_frameset_count_tags(in, tag) == 0) return NULL ;

    /* Create the output frameset */
    out = cpl_frameset_new() ;

    /* Loop on the requested frames and store them in out */
    for (i=0 ; i<nbframes ; i++) {
        const cpl_frame *cur_frame = cpl_frameset_get_position_const(in, i);
        if (!strcmp(cpl_frame_get_tag(cur_frame), tag)) {
            loc_frame = cpl_frame_duplicate(cur_frame) ;
            cpl_frameset_insert(out, loc_frame) ;
        }
    }
    eris_check_error_code("eris_ifu_extract_frameset");
    return out ;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Add common HDU classification properties to FITS header
 * @param plist        FITS header property list to modify
 * @param deq_hduclas  Value for HDUCLAS1 keyword
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Adds standard ESO Data Exchange Qualifier (DEQ) classification keywords:
 * HDUCLASS, HDUDOC, HDUVERS, and HDUCLAS1 to the property list.
 *
 * @note This function is typically called internally by save functions.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_heades_add_hduclass_common(cpl_propertylist* plist,
		const char* deq_hduclas)
{
    cpl_propertylist_append_string(plist, FHDR_HDUCLASS, DEQ_HDUCLASS);
    cpl_propertylist_append_string(plist, FHDR_HDUDOC, DEQ_HDUDOC);
    cpl_propertylist_append_string(plist, FHDR_HDUVERS, DEQ_HDUVERS);
    cpl_propertylist_append_string(plist, FHDR_HDUCLAS1, deq_hduclas);
    eris_check_error_code("eris_ifu_heades_add_hduclass_common");
    return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 * @brief    Add data extension classification properties to FITS header
 * @param plist   FITS header property list to modify
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Adds DEQ keywords specific to data extensions, including HDUCLAS2,
 * EXTNAME, ERRDATA, and QUALDATA references.
 *
 * @note This function is typically called internally by save functions.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_heades_add_hduclass_data(cpl_propertylist* plist)
{
    cpl_propertylist_append_string(plist, FHDR_HDUCLAS2, DEQ_HDUCLAS2_DATA);
    cpl_propertylist_update_string(plist, FHDR_EXTNAME, DEQ_SCIDATA);
    cpl_propertylist_append_string(plist, FHDR_ERRDATA, DEQ_ERRDATA);
    cpl_propertylist_append_string(plist, FHDR_QUALDATA, DEQ_QUALDATA);
    eris_check_error_code("eris_ifu_heades_add_hduclass_data");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 * @brief    Add error extension classification properties to FITS header
 * @param plist       FITS header property list to modify
 * @param errorType   Error type (MSE, RMSE, INVMSE, or INVRMSE)
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Adds DEQ keywords specific to error extensions, including HDUCLAS2,
 * HDUCLAS3 (error type), EXTNAME, and references to SCIDATA and QUALDATA.
 *
 * @note This function is typically called internally by save functions.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_heades_add_hduclass_errs(cpl_propertylist* plist,
        deqErrorType errorType)
{
    cpl_propertylist_update_string(plist, FHDR_HDUCLAS2, DEQ_HDUCLAS2_ERROR);
    cpl_propertylist_update_string(plist, FHDR_HDUCLAS3,
            deqErrorTypeString[errorType]);
    cpl_propertylist_update_string(plist, FHDR_EXTNAME, DEQ_ERRDATA);
    cpl_propertylist_update_string(plist, FHDR_SCIDATA, DEQ_SCIDATA);
    cpl_propertylist_update_string(plist, FHDR_QUALDATA, DEQ_QUALDATA);
    eris_check_error_code("eris_ifu_heades_add_hduclass_data");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Add quality extension classification properties to FITS header
 * @param plist         FITS header property list to modify
 * @param qualityType   Quality type (MASKZERO, MASKONE, FLAG32BIT, or FLAG16BIT)
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Adds DEQ keywords specific to quality/bad pixel extensions, including
 * HDUCLAS2, HDUCLAS3 (quality type), EXTNAME, and references to SCIDATA
 * and ERRDATA.
 *
 * @note This function is typically called internally by save functions.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_heades_add_hduclass_qual(cpl_propertylist* plist,
        deqQualityType qualityType){
    cpl_propertylist_update_string(plist, FHDR_HDUCLAS2, DEQ_HDUCLAS2_QUAL);
    cpl_propertylist_update_string(plist, FHDR_HDUCLAS3,
            deqQualityTypeString[qualityType]);
    cpl_propertylist_update_string(plist, FHDR_EXTNAME, DEQ_QUALDATA);
    cpl_propertylist_update_string(plist, FHDR_SCIDATA, DEQ_SCIDATA);
    cpl_propertylist_update_string(plist, FHDR_ERRDATA, DEQ_ERRDATA);
    eris_check_error_code("eris_ifu_heades_add_hduclass_qual");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 * @brief    Save a DFS-compliant image product with data, error, and quality extensions
 * @param allframes        List of all input frames for the recipe
 * @param header           NULL or property list with additional properties for product header
 * @param parlist          List of recipe input parameters
 * @param usedframes       List of raw/calibration frames used for this product
 * @param inherit          NULL or frame from which product inherits header keywords
 * @param recipe           Recipe name
 * @param applist          Property list to append to primary header (must contain PRO.CATG)
 * @param remregexp        NULL or regexp of properties not to put in main header
 * @param filename         Output filename for created product
 * @param image            Data image to save (required)
 * @param error            Error image (optional, can be NULL)
 * @param errorType        Error type: MSE, RMSE, INVMSE, or INVRMSE
 * @param dataQualityMask  Data quality/bad pixel image (optional, can be NULL)
 * @param dataQualityType  Quality type: MASKZERO, MASKONE, FLAG32BIT, or FLAG16BIT
 * @param unit             Physical unit string for BUNIT keyword
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Creates a DFS-compliant FITS file according to VLT-SPE-ESO-19500-5667 with:
 * - Primary HDU: Empty with pipeline provenance keywords
 * - Extension 1: Data image with WCS
 * - Extension 2: Error image (if provided)
 * - Extension 3: Quality/bad pixel map (if provided)
 *
 * @note WCS keywords are removed from the primary header and added only to extensions.
 * @note All extensions contain proper DEQ classification keywords.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_save_deq_image(
        cpl_frameset *allframes,
        cpl_propertylist * header,
        const cpl_parameterlist *parlist,
        const cpl_frameset *usedframes,
        const cpl_frame *inherit,
        const char *recipe,
        const cpl_propertylist *applist,
        const char *remregexp,
        const char *filename,
        const cpl_image *image,
        const cpl_image *error,
        deqErrorType errorType,
        const cpl_image *dataQualityMask,
        deqQualityType dataQualityType,
		const char* unit)
{
    cpl_error_code      retVal      = CPL_ERROR_NONE;
    cpl_propertylist    *dataHeader = NULL,
                        *errsHeader  = NULL,
                        *qualHeader = NULL;
    char                *pipe_id    = NULL;

    cpl_ensure_code(allframes, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(usedframes, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(recipe, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(applist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filename, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(image, CPL_ERROR_NULL_INPUT);


    //cpl_ensure_code(error, CPL_ERROR_NULL_INPUT);
    //cpl_ensure_code(dataQualityMask, CPL_ERROR_NULL_INPUT);
    cpl_propertylist* head_wcs_2d = NULL;
    TRY
	{

    	pipe_id = cpl_sprintf("%s%s%s", PACKAGE, "/", PACKAGE_VERSION);
    	dataHeader = cpl_propertylist_new();
    	eris_ifu_heades_add_hduclass_common(dataHeader, DEQ_HDUCLAS1);
    	eris_ifu_heades_add_hduclass_data(dataHeader);
    	head_wcs_2d = eris_ifu_plist_extract_wcs2D(applist);
    	/* remove WCS from primary header as data go into extensions */
        eris_ifu_plist_erase_wcs( (cpl_propertylist *) applist);
        if(cpl_propertylist_has( (cpl_propertylist *) applist,"BUNIT")) {
            cpl_propertylist_erase((cpl_propertylist *) applist, "BUNIT");
        }
    	if(cpl_propertylist_has( (cpl_propertylist *) applist,"EXTNAME")) {
    		cpl_propertylist_erase( (cpl_propertylist *)applist,"EXTNAME");
    	}
    	if(cpl_propertylist_has( (cpl_propertylist *) applist,"HDUCLAS2")) {
    		cpl_propertylist_erase( (cpl_propertylist *)applist,"HDUCLAS2");
    	}
    	if(cpl_propertylist_has( (cpl_propertylist *) applist,"COMMENT")) {
    		cpl_propertylist_erase( (cpl_propertylist *)applist,"COMMENT");
    	}
    	/* Create an empty primary HDU with only the header information*/
    	cpl_dfs_save_propertylist(allframes, header, parlist, usedframes,
    			inherit, recipe, applist, remregexp, pipe_id, filename);
    	cpl_propertylist_append(dataHeader, head_wcs_2d);

    	/* Write the extension units */
    	if(cpl_propertylist_has(dataHeader,"BUNIT")) {
    		cpl_propertylist_set_string(dataHeader,"BUNIT", unit);
    	} else {
    		cpl_propertylist_append_string(dataHeader,"BUNIT", unit);
    	}
    	eris_pfits_clean_header_ra_dec_mjd_obs(dataHeader);
    	cpl_image_save(image, filename, cpl_image_get_type(image), dataHeader,
    			CPL_IO_EXTEND);

    	if(error) {
    		errsHeader = cpl_propertylist_new();
    		eris_ifu_heades_add_hduclass_common(errsHeader, DEQ_HDUCLAS1);
    		eris_ifu_heades_add_hduclass_errs(errsHeader, errorType);

    		if(cpl_propertylist_has(errsHeader,"BUNIT")) {
    			cpl_propertylist_set_string(errsHeader,"BUNIT", unit);
    		} else {
    			cpl_propertylist_append_string(errsHeader,"BUNIT", unit);
    		}
    		cpl_propertylist_append(errsHeader, head_wcs_2d);
    		eris_pfits_clean_header_ra_dec_mjd_obs(errsHeader);
    		cpl_image_save(error, filename, cpl_image_get_type(error), errsHeader,
    				CPL_IO_EXTEND);
    	}

    	if(dataQualityMask) {
    		qualHeader = cpl_propertylist_new();
    		eris_ifu_heades_add_hduclass_common(qualHeader, DEQ_HDUCLAS1);
    		eris_ifu_heades_add_hduclass_qual(qualHeader, dataQualityType);

    		if(cpl_propertylist_has(qualHeader,"BUNIT")) {
    			cpl_propertylist_set_string(qualHeader,"BUNIT", "");
    		} else {
    			cpl_propertylist_append_string(qualHeader,"BUNIT", "");
    		}
    		cpl_propertylist_append(qualHeader, head_wcs_2d);
    		eris_pfits_clean_header_ra_dec_mjd_obs(qualHeader);
    		cpl_image_save(dataQualityMask, filename,
    				cpl_image_get_type(dataQualityMask), qualHeader, CPL_IO_EXTEND);
    	}

	}
    CATCH
	{
    	retVal = cpl_error_get_code();
	}
    eris_ifu_free_propertylist(&dataHeader);
    eris_ifu_free_propertylist(&errsHeader);
    eris_ifu_free_propertylist(&qualHeader);
    eris_ifu_free_propertylist(&head_wcs_2d);
    eris_ifu_free_string(&pipe_id);
    eris_check_error_code("eris_ifu_save_deq_image");
    return retVal;
}



/*----------------------------------------------------------------------------*/
/**
 * @brief    Save a DFS-compliant cube product with data, error, and quality extensions
 * @param allframes        List of all input frames for the recipe
 * @param header           NULL or property list with additional properties for product header
 * @param parlist          List of recipe input parameters
 * @param usedframes       List of raw/calibration frames used for this product
 * @param inherit          NULL or frame from which product inherits header keywords
 * @param recipe           Recipe name
 * @param applist          Property list to append to primary header (must contain PRO.CATG)
 * @param remregexp        NULL or regexp of properties not to put in main header
 * @param filename         Output filename for created product
 * @param data             Data cube (imagelist) to save
 * @param error            Error cube (imagelist)
 * @param errorType        Error type: MSE, RMSE, INVMSE, or INVRMSE
 * @param dataQualityMask  Data quality cube (imagelist)
 * @param qualityType      Quality type: MASKZERO, MASKONE, FLAG32BIT, or FLAG16BIT
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Creates a DFS-compliant 3D datacube FITS file with:
 * - Primary HDU: Empty with pipeline provenance keywords
 * - Extension 1: Data cube with WCS
 * - Extension 2: Error cube
 * - Extension 3: Quality/bad pixel cube
 *
 * @note The PRODCATG keyword is automatically set based on filename (sky vs object).
 * @note Quality extension is saved as CPL_TYPE_INT regardless of input type.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_save_deq_cube(
        cpl_frameset *allframes,
        cpl_propertylist * header,
        const cpl_parameterlist *parlist,
        const cpl_frameset *usedframes,
        const cpl_frame *inherit,
        const char *recipe,
        cpl_propertylist *applist,
        const char *remregexp,
        const char *filename,
        cpl_imagelist *data,
        cpl_imagelist *error,
        deqErrorType errorType,
        const cpl_imagelist *dataQualityMask,
        deqQualityType qualityType/*,
        cpl_type imageType*/)
{
	cpl_error_code      retVal      = CPL_ERROR_NONE;
	cpl_propertylist    *dataHeader = NULL,
			*errsHeader  = NULL,
			*qualHeader = NULL;
	char                *pipe_id    = NULL;

	cpl_ensure_code(allframes, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(usedframes, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(recipe, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(applist, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(filename, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(data, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(error, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(dataQualityMask, CPL_ERROR_NULL_INPUT);

	TRY
	{

		pipe_id = cpl_sprintf("%s%s%s", PACKAGE, "/", PACKAGE_VERSION);

		/* Create FITS headers for the extensions */
		if(cpl_propertylist_has(applist,"CDELT3")) {
			cpl_propertylist_erase(applist, "CDELT3");
		}
		dataHeader = cpl_propertylist_duplicate(applist);
		errsHeader = cpl_propertylist_duplicate(applist);
		qualHeader = cpl_propertylist_duplicate(applist);

		eris_ifu_heades_add_hduclass_common(dataHeader, "IMAGE");
		eris_ifu_heades_add_hduclass_data(dataHeader);
		eris_ifu_heades_add_hduclass_common(errsHeader, "IMAGE");
		eris_ifu_heades_add_hduclass_errs(errsHeader, errorType);
		eris_ifu_heades_add_hduclass_common(qualHeader, "IMAGE");
		eris_ifu_heades_add_hduclass_qual(qualHeader, qualityType);

		if(cpl_propertylist_has(dataHeader,"HDRVER")) {
			cpl_propertylist_erase(dataHeader,"HDRVER");
		}
		if(cpl_propertylist_has(errsHeader,"HDRVER")) {
			cpl_propertylist_erase(errsHeader,"HDRVER");
		}
		if(cpl_propertylist_has(qualHeader,"HDRVER")) {
			cpl_propertylist_erase(qualHeader,"HDRVER");
		}
		if(cpl_propertylist_has(dataHeader,"CTYPE1")) {
			const char* ctype1 = cpl_propertylist_get_string(dataHeader,"CTYPE1");
            if (!cpl_propertylist_has(errsHeader,"CTYPE1"))
			    cpl_propertylist_append_string(errsHeader, "CTYPE1", ctype1);
            if (!cpl_propertylist_has(qualHeader,"CTYPE1"))
			    cpl_propertylist_append_string(qualHeader, "CTYPE1", ctype1);
		}
		if(cpl_propertylist_has(dataHeader,"CTYPE2")) {
			const char* ctype2 = cpl_propertylist_get_string(dataHeader,"CTYPE2");
            if (!cpl_propertylist_has(errsHeader,"CTYPE2"))
    			cpl_propertylist_append_string(errsHeader, "CTYPE2",ctype2);
            if (!cpl_propertylist_has(errsHeader,"CTYPE2"))
    			cpl_propertylist_append_string(qualHeader, "CTYPE2", ctype2);
		}

		/* Create an empty primary HDU with only the header information*/
        eris_ifu_plist_erase_wcs( (cpl_propertylist *)applist);
        eris_pfits_clean_header(dataHeader, CPL_TRUE);
        eris_pfits_clean_header(errsHeader, CPL_TRUE);
        eris_pfits_clean_header(qualHeader, CPL_TRUE);
        if(strstr(filename, "sky") != NULL) {
           cpl_propertylist_append_string(applist, "PRODCATG", PRODCATG_CUBE_SKY);
        } else {
		   cpl_propertylist_append_string(applist, "PRODCATG", PRODCATG_CUBE_OBJ);
        }
		if(cpl_propertylist_has(applist,"BUNIT")) {
			cpl_propertylist_erase(applist,"BUNIT");
		}
		if(cpl_propertylist_has(applist,"COMMENT")) {
			cpl_propertylist_erase(applist,"COMMENT");
		}


		cpl_dfs_save_propertylist(allframes, header, parlist, usedframes,
				inherit, recipe, applist, remregexp,
				pipe_id, filename);

		/* Write the extension units */
		cpl_imagelist_save(data, filename,
				cpl_image_get_type(cpl_imagelist_get(data,0)),
				dataHeader, CPL_IO_EXTEND);

		cpl_imagelist_save(error, filename,
				cpl_image_get_type(cpl_imagelist_get(error,0)),
				errsHeader, CPL_IO_EXTEND);
		if(cpl_propertylist_has(qualHeader,"BUNIT")) {
			cpl_propertylist_set_string(qualHeader,"BUNIT","");
		} else {
			cpl_propertylist_append_string(qualHeader,"BUNIT","");
		}
		cpl_imagelist_save(dataQualityMask, filename,
				CPL_TYPE_INT,
				qualHeader, CPL_IO_EXTEND);
	}
	CATCH
	{
		retVal = cpl_error_get_code();
	}
	cpl_propertylist_delete(dataHeader);
	cpl_propertylist_delete(errsHeader);
	cpl_propertylist_delete(qualHeader);
	cpl_free(pipe_id);
	eris_check_error_code("eris_ifu_save_deq_cube");
	return retVal;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Save a DFS-compliant Phase 3 image product
 * @param allframes   List of all input frames for the recipe
 * @param parlist     List of recipe input parameters
 * @param usedframes  List of raw/calibration frames used for this product
 * @param recipe      Recipe name
 * @param applist     Property list to append to primary header (must contain PRO.CATG)
 * @param remregexp   NULL or regexp of properties not to put in main header
 * @param filename    Output filename for created product
 * @param image       Data image to save (required)
 * @param type        CPL data type for saving the image
 * @param unit        Physical unit string for BUNIT keyword
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Creates a Phase 3 compliant FITS file with the image in the primary HDU
 * (not as an extension). This format is used for certain simple products.
 *
 * @note The image is saved directly in the primary HDU, not as an extension.
 * @note The BUNIT keyword is added to specify the physical unit.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_save_image_phase3(
        cpl_frameset *allframes,
        const cpl_parameterlist *parlist,
        const cpl_frameset *usedframes,
        const char *recipe,
        const cpl_propertylist *applist,
        const char *remregexp,
        const char *filename,
        const cpl_image *image,
		cpl_type type,
		const char* unit)
{
    cpl_error_code      retVal      = CPL_ERROR_NONE;
    cpl_propertylist    *dataHeader = NULL;
    char                *pipe_id    = NULL;

    cpl_ensure_code(allframes, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(usedframes, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(recipe, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(applist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filename, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(image, CPL_ERROR_NULL_INPUT);

    cpl_propertylist* head_wcs_2d = NULL;
    TRY
	{

    	pipe_id = cpl_sprintf("%s%s%s", PACKAGE, "/", PACKAGE_VERSION);
    	dataHeader = cpl_propertylist_new();

    	if(cpl_propertylist_has( (cpl_propertylist *) applist,"EXTNAME")) {
    		cpl_propertylist_erase( (cpl_propertylist *)applist,"EXTNAME");
    	}


    	if(cpl_propertylist_has(applist,"BUNIT")) {
    		cpl_propertylist_set_string((cpl_propertylist *) applist,"BUNIT", unit);
    	} else {
    		cpl_propertylist_append_string((cpl_propertylist *) applist,"BUNIT", unit);
    	}
    	cpl_dfs_save_image(allframes, NULL, parlist, usedframes, NULL, image, type,
    					recipe, applist, remregexp, pipe_id, filename);
	}
    CATCH
	{
    	retVal = cpl_error_get_code();
	}
    eris_ifu_free_propertylist(&dataHeader);
    eris_ifu_free_propertylist(&head_wcs_2d);
    eris_ifu_free_string(&pipe_id);
    eris_check_error_code("eris_ifu_save_image");
    return retVal;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Save a DFS-compliant table product
 * @param allframes   List of all input frames for the recipe
 * @param header      NULL or property list with additional properties for product header
 * @param parlist     List of recipe input parameters
 * @param usedframes  List of raw/calibration frames used for this product
 * @param inherit     NULL or frame from which product inherits header keywords
 * @param recipe      Recipe name
 * @param procatg     Output product category string
 * @param applist     Property list to append to primary header
 * @param remregexp   NULL or regexp of properties not to put in main header
 * @param filename    Output filename for created product
 * @param table       Data table to save
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Creates a DFS-compliant FITS file with:
 * - Primary HDU: Empty with pipeline provenance keywords
 * - Extension 1: Binary table with DEQ classification
 *
 * @note The table is saved in a binary table extension, not ASCII.
 * @note The PRO.CATG keyword is set to the provided procatg value.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_save_table(
        cpl_frameset *allframes,
        cpl_propertylist *header,
        const cpl_parameterlist *parlist,
        const cpl_frameset *usedframes,
        const cpl_frame *inherit,
        const char *recipe,
        const char *procatg,
        cpl_propertylist *applist,
        const char *remregexp,
        const char *filename,
        const cpl_table *table)
{
    cpl_error_code      retVal      = CPL_ERROR_NONE;
    cpl_propertylist    *dataHeader = NULL;
    char                *pipe_id    = NULL;

    cpl_ensure_code(allframes, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(usedframes, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(recipe, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(applist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filename, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(table, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(procatg, CPL_ERROR_NULL_INPUT);

    TRY
    {

    	pipe_id = cpl_sprintf("%s%s%s", PACKAGE, "/", PACKAGE_VERSION);
    	/* Create FITS headers for the extensions */
    	dataHeader = cpl_propertylist_new();
    	eris_ifu_heades_add_hduclass_common(dataHeader, "TABLE");

    	cpl_propertylist_append_string(dataHeader, FHDR_HDUCLAS2, DEQ_HDUCLAS2_DATA);
    	cpl_propertylist_append_string(dataHeader, FHDR_EXTNAME, DEQ_SCIDATA);
    	cpl_propertylist_update_string(applist, CPL_DFS_PRO_CATG, procatg);

    	/* Create an empty primary HDU with only the header information*/
    	cpl_dfs_save_propertylist(allframes, header, parlist, usedframes,
    			inherit, recipe, applist, remregexp, pipe_id, filename);

    	/* Write the extension unit */
    	cpl_table_save(table, dataHeader, applist, filename, CPL_IO_EXTEND);
    }
    CATCH
    {
        CATCH_MSGS();
        retVal = cpl_error_get_code();
    }

    eris_ifu_free_propertylist(&dataHeader);
    eris_ifu_free_string(&pipe_id);
    eris_check_error_code("eris_ifu_save_table");
    return retVal;
}

///** Function to write a DFS compliant pipeline product with a badpixel map
// * according to VLT-SPE-ESO-19500-5667
//  @brief  Write DFS pipeline product with table.
//  @param  allframes  The list of input frames for the recipe
//  @param  header     NULL or filled with properties written to product header
//  @param  parlist    The list of input parameters
//  @param  usedframes The list of raw/calibration frames used for this product
//  @param  inherit    NULL or product frames inherit their header from this frame
//  @param  recipe     The recipe name
//  @param  applist    Propertylist to append to primary header, w. PRO.CATG
//  @param  remregexp  NULL or regexp of properties not to put in main header
//  @param  filename   Filename of created product
//  @param  bpm        The badpixel map to be saved
//  @return   CPL_ERROR_NONE if OK
//*/
//cpl_error_code eris_ifu_save_bpm(
//        cpl_frameset *allframes,
//        cpl_propertylist *header,
//        const cpl_parameterlist *parlist,
//        const cpl_frameset *usedframes,
//        const cpl_frame *inherit,
//        const char *recipe,
//        const char *procatg,
//        cpl_propertylist *applist,
//        const char *remregexp,
//        const char *filename,
//        const cpl_image *bpm)
//{
//    cpl_error_code      retVal      = CPL_ERROR_NONE;
//    cpl_propertylist    *dataHeader = NULL;
//    char                *pipe_id    = NULL;
//    cpl_ensure_code(allframes, CPL_ERROR_NULL_INPUT);
//    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
//    cpl_ensure_code(usedframes, CPL_ERROR_NULL_INPUT);
//    cpl_ensure_code(recipe, CPL_ERROR_NULL_INPUT);
//    cpl_ensure_code(applist, CPL_ERROR_NULL_INPUT);
//    cpl_ensure_code(filename, CPL_ERROR_NULL_INPUT);
//    cpl_ensure_code(bpm, CPL_ERROR_NULL_INPUT);
//    cpl_ensure_code(procatg, CPL_ERROR_NULL_INPUT);
//    TRY
//    {
//        pipe_id = cpl_sprintf("%s%s%s", PACKAGE, "/", PACKAGE_VERSION);
//        /* Create FITS headers for the extensions */
//
//        dataHeader = cpl_propertylist_new();
//        eris_ifu_heades_add_hduclass_common(dataHeader, "TABLE");
//        cpl_propertylist_append_string(dataHeader, FHDR_HDUCLAS2, DEQ_HDUCLAS2_DATA);
//        cpl_propertylist_append_string(dataHeader, FHDR_EXTNAME, DEQ_SCIDATA);
//        cpl_propertylist_update_string(applist, CPL_DFS_PRO_CATG, procatg);
//        /* Create an empty primary HDU with only the header information*/
//        cpl_dfs_save_propertylist(allframes, header, parlist, usedframes,
//                                  inherit, recipe, applist, remregexp,
//                                  pipe_id, filename));
//        /* Write the extension unit */
//        cpl_image_save(bpm, filename, cpl_image_get_type(bpm), dataHeader,
//                           CPL_IO_EXTEND);
//    }
//    CATCH
//    {
//        CATCH_MSGS();
//        retVal = cpl_error_get_code();
//    }
//    eris_ifu_free_propertylist(&dataHeader);
//    eris_ifu_free_string(&pipe_id);
//    return retVal;
//}

/**
 * @brief    Load a DEQ-format image with data, error, and quality extensions
 * @param filename         Input FITS filename
 * @param primaryHeader    Output primary header property list (pointer to pointer, can be NULL)
 * @param dataImage        Output data image (pointer to pointer)
 * @param errImage         Output error image (pointer to pointer, can be NULL)
 * @param qualImage        Output quality image (pointer to pointer, can be NULL)
 * @param errorType        Output error type detected (pointer, can be NULL)
 * @param qualityType      Output quality type detected (pointer, can be NULL)
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Loads a DEQ-format FITS file with separate data, error, and quality extensions.
 * The file must have exactly 3 extensions. The error and quality types are
 * determined from the HDUCLAS3 keyword in each extension.
 *
 * @note The data image is always loaded; error and quality images are optional.
 * @note All returned images must be freed by the caller using cpl_image_delete().
 * @note If primaryHeader is not NULL, it must be freed with cpl_propertylist_delete().
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_load_deq_image(
		const char *filename,
		cpl_propertylist **primaryHeader,
		cpl_image **dataImage,
		cpl_image **errImage,
		cpl_image **qualImage,
		deqErrorType *errorType,
		deqQualityType *qualityType)
{
	cpl_error_code retVal = CPL_ERROR_NONE;
	cpl_size nExt;
	int extension;
	int match;
	cpl_propertylist *pl = NULL;
	const char *property;

	cpl_ensure_code(filename, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(dataImage, CPL_ERROR_NULL_INPUT);

	TRY
	{
		nExt = cpl_fits_count_extensions(filename);
		CHECK_ERROR_STATE();
		if (nExt != 3) {
			BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
					"For a DEQ FITS file 3 extensions are expected, but %d are found",
					(int) nExt);
		}
		*dataImage = cpl_image_load(filename, CPL_TYPE_UNSPECIFIED, 0, 1);

		if (primaryHeader != NULL) {
			*primaryHeader = cpl_propertylist_load(filename, 0);
		}

		if (errImage != NULL) {
			extension = 2;
			*errImage = cpl_image_load(filename, CPL_TYPE_UNSPECIFIED,
					0, extension);
			pl = cpl_propertylist_load(filename, extension);
			if (cpl_propertylist_has(pl, FHDR_HDUCLAS3)) {
				property = cpl_propertylist_get_string(pl, FHDR_HDUCLAS3);
			} else {
				BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
						"keyword %s does not exist",
						FHDR_HDUCLAS3);
			}
			for (match=0; match<4; match++) {
				if (strcmp(property, deqErrorTypeString[match]) == 0) {
					*errorType = match;
					break;
				}
			}
			if (match > 3) {
				BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
						"The HDUCLAS3 keyword for the error extension (%d) has an"
						" unknown value of %s", extension, property);
			}
			eris_ifu_free_propertylist(&pl);
		}

		if (qualImage != NULL) {
			extension = 3;
			*qualImage = cpl_image_load(filename, CPL_TYPE_UNSPECIFIED,
					0, 3);
			pl = cpl_propertylist_load(filename, extension);
			if (cpl_propertylist_has(pl, FHDR_HDUCLAS3)) {
				property = cpl_propertylist_get_string(pl, FHDR_HDUCLAS3);
			} else {
				BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
						"keyword %s does not exist",
						FHDR_HDUCLAS3);
			}
			for (match=0; match<4; match++) {
				if (strcmp(property, deqQualityTypeString[match]) == 0) {
					*qualityType = match;
					break;
				}
			}
			if (match > 3) {
				BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
						"The HDUCLAS3 keyword for the error extension (%d) has an"
						" unknown value of %s", (int) extension, property);
			}
			eris_ifu_free_propertylist(&pl);
		}
	}
	CATCH
	{
		retVal = cpl_error_get_code();
	}
	eris_check_error_code("eris_ifu_load_deq_image");
	return retVal;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Load a DEQ-format imagelist (cube) with data, error, and quality extensions
 * @param filename         Input FITS filename
 * @param primaryHeader    Output primary header property list (pointer to pointer, can be NULL)
 * @param dataImagelist    Output data imagelist (pointer to pointer)
 * @param errImagelist     Output error imagelist (pointer to pointer, can be NULL)
 * @param qualImagelist    Output quality imagelist (pointer to pointer, can be NULL)
 * @param errorType        Output error type detected (pointer, can be NULL)
 * @param qualityType      Output quality type detected (pointer, can be NULL)
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Loads a DEQ-format datacube FITS file with separate data, error, and quality
 * imagelist extensions. The file must have exactly 3 extensions.
 *
 * @note All returned imagelists must be freed by the caller using cpl_imagelist_delete().
 * @note All three imagelists must have the same size.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_load_deq_imagelist(
		const char *filename,
		cpl_propertylist **primaryHeader,
		cpl_imagelist **dataImagelist,
		cpl_imagelist **errImagelist,
		cpl_imagelist **qualImagelist,
		deqErrorType *errorType,
		deqQualityType *qualityType)
{
	cpl_error_code retVal = CPL_ERROR_NONE;
	cpl_size nExt;
	int extension;
	int match;
	cpl_propertylist *pl = NULL;
	const char *property;

	cpl_ensure_code(filename, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(dataImagelist, CPL_ERROR_NULL_INPUT);

	TRY
	{
		nExt = cpl_fits_count_extensions(filename);
		CHECK_ERROR_STATE();
		if (nExt != 3) {
			BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
					"For a DEQ FITS file 3 extensions are expected, but %d are found",
					(int) nExt);
		}
		*dataImagelist = cpl_imagelist_load(filename, CPL_TYPE_UNSPECIFIED, 1);

		if (primaryHeader != NULL) {
			*primaryHeader = cpl_propertylist_load(filename, 0);
		}

		if (errImagelist != NULL) {
			extension = 2;
			*errImagelist = cpl_imagelist_load(filename, CPL_TYPE_UNSPECIFIED,
					extension);
			pl = cpl_propertylist_load(filename, extension);
			if (cpl_propertylist_has(pl, FHDR_HDUCLAS3)) {
				property = cpl_propertylist_get_string(pl, FHDR_HDUCLAS3);
			} else {
				BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
						"keyword %s does not exist",
						FHDR_HDUCLAS3);
			}
			for (match=0; match<4; match++) {
				if (strcmp(property, deqErrorTypeString[match]) == 0) {
					*errorType = match;
					break;
				}
			}
			if (match > 3) {
				BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
						"The HDUCLAS3 keyword for the error extension (%d) has an"
						" unknown value of %s", extension, property);
			}
			eris_ifu_free_propertylist(&pl);
		}

		if (qualImagelist != NULL) {
			extension = 3;
			*qualImagelist = cpl_imagelist_load(filename, CPL_TYPE_UNSPECIFIED,
					extension);
			pl = cpl_propertylist_load(filename, extension);
			if (cpl_propertylist_has(pl, FHDR_HDUCLAS3)) {
				property = cpl_propertylist_get_string(pl, FHDR_HDUCLAS3);
			} else {
				BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
						"keyword %s does not exist",
						FHDR_HDUCLAS3);
			}
			for (match=0; match<4; match++) {
				if (strcmp(property, deqQualityTypeString[match]) == 0) {
					*qualityType = match;
					break;
				}
			}
			if (match > 3) {
				BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
						"The HDUCLAS3 keyword for the error extension (%d) has an"
						" unknown value of %s", (int) extension, property);
			}
			eris_ifu_free_propertylist(&pl);
		}
	}
	CATCH
	{
		retVal = cpl_error_get_code();
	}
	eris_check_error_code("eris_ifu_load_deq_imagelist");
	return retVal;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Load a DEQ-format file into an HDRL image structure
 * @param filename        Input FITS filename
 * @param primaryHeader   Output primary header property list (pointer to pointer, can be NULL)
 * @param qualImage       Output quality image (pointer to pointer, can be NULL)
 * @param qualityType     Output quality type detected (pointer, can be NULL)
 * @return   HDRL image structure on success, or NULL on error
 *
 * Loads a DEQ-format FITS file and creates an HDRL image structure containing
 * the data and error planes with proper bad pixel masking from the quality
 * extension. The error type is automatically converted to RMSE format required
 * by HDRL.
 *
 * @note The returned HDRL image must be freed by the caller using hdrl_image_delete().
 * @note MSE errors are converted to RMSE by taking square root.
 * @note INVMSE and INVRMSE error types are not supported.
 * @note Quality values of 0 indicate good pixels; other values indicate bad pixels.
 */
/*----------------------------------------------------------------------------*/
hdrl_image * eris_ifu_load_deq_hdrl_image(
		const char *filename,
		cpl_propertylist **primaryHeader,
		cpl_image **qualImage,
		deqQualityType *qualityType)
{
	hdrl_image *image = NULL;
	cpl_image *dataImage = NULL;
	cpl_image *errImage = NULL;
	cpl_mask *mask = NULL;

	cpl_ensure(filename, CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{
		deqErrorType errorType;
		eris_ifu_load_deq_image(filename, primaryHeader,
				&dataImage, &errImage, qualImage, &errorType, qualityType);

		switch(errorType) {
		case mse:
			cpl_image_power(errImage, 0.5);
			break;
		case rmse:
			// nothing to do, RMSE is Ok
			break;
		case invmse:
			CPL_ATTR_FALLTRHU; //intended fall through !
		case invrmse:
			BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
					"Cannot transfer error extension type %s to a HDLR error"
					" image", deqErrorTypeString[errorType]);
			break;
		default:
			BRK_WITH_ERROR_MSG(CPL_ERROR_ACCESS_OUT_OF_RANGE,
					"Internal error: unknown errorType enumeration value: %d",
					errorType);
			break;
		}

		if (qualImage != NULL) {
			switch (*qualityType) {
			case maskzero:
				//intended fall through !
				CPL_ATTR_FALLTRHU; 
			case flag16bit:
				//intended fall through !
				CPL_ATTR_FALLTRHU; 
			case flag32bit: // The value "0" indicates a "good" pixel
				mask = cpl_mask_threshold_image_create(*qualImage,
						-1.0, 1.0);
				break;
			case maskone: // The value "1" indicates a "good" pixel
				mask = cpl_mask_threshold_image_create(*qualImage,
						0.0, 2.0);
				break;
			default:
				BRK_WITH_ERROR_MSG(
						CPL_ERROR_ACCESS_OUT_OF_RANGE,
						"Internal error: unknown qualityType enumeration"
						" value: %d",
						*qualityType);
				break;
			}
		}

		cpl_mask_not(mask); //hey it is a BAD pixel mask!
		cpl_image_set_bpm(dataImage, mask);
		mask = NULL;
		image = hdrl_image_create(dataImage, errImage);
		CHECK_ERROR_STATE();
	}
	CATCH
	{
		cpl_mask_delete(mask);
		hdrl_image_delete(image);
		image = NULL;
	}
	cpl_image_delete(dataImage);
	cpl_image_delete(errImage);
	eris_check_error_code("eris_ifu_load_deq_hdrl_image");
	return image;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Load a DEQ-format cube into an HDRL imagelist structure
 * @param filename         Input FITS filename
 * @param primaryHeader    Output primary header property list (pointer to pointer, can be NULL)
 * @param qualImagelist    Output quality imagelist (pointer to pointer, can be NULL)
 * @param qualityType      Output quality type detected (pointer, can be NULL)
 * @return   HDRL imagelist structure on success, or NULL on error
 *
 * Loads a DEQ-format datacube FITS file and creates an HDRL imagelist structure
 * with proper bad pixel masking. Each plane is processed individually to flag
 * bad pixels from both the quality extension and NaN/Inf values in data or error.
 *
 * @note The returned HDRL imagelist must be freed by the caller using hdrl_imagelist_delete().
 * @note NaN and Inf values in data or error images are automatically flagged as bad.
 * @note All three imagelists must have the same size.
 */
/*----------------------------------------------------------------------------*/
hdrl_imagelist * eris_ifu_load_deq_hdrl_imagelist (
		const char *filename,
		cpl_propertylist **primaryHeader,
		cpl_imagelist **qualImagelist,
		deqQualityType *qualityType)
{
	hdrl_imagelist *imagelist = NULL;
	cpl_imagelist *dataImagelist = NULL;
	cpl_imagelist *errImagelist = NULL;
	cpl_image *dataImage = NULL;
	cpl_image *errImage = NULL;
	cpl_image *qualImage = NULL;
	cpl_mask *mask = NULL;
	deqErrorType errorType;

	cpl_ensure(filename, CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{
		eris_ifu_load_deq_imagelist(filename, primaryHeader,
				&dataImagelist, &errImagelist, qualImagelist, &errorType, qualityType);

		if ((cpl_imagelist_get_size(dataImagelist) != cpl_imagelist_get_size(errImagelist))
				||
				(cpl_imagelist_get_size(dataImagelist) != cpl_imagelist_get_size(*qualImagelist)))
		{
			BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
					"All imagelists of an DEQ imagelist file must have the same size");
		}

		for (cpl_size ix=0; ix < cpl_imagelist_get_size(dataImagelist); ix++) {
			dataImage = cpl_imagelist_get(dataImagelist, ix);
			errImage = cpl_imagelist_get(errImagelist, ix);
			qualImage = cpl_imagelist_get(*qualImagelist, ix);

			switch(errorType) {
			case mse:
				BRK_IF_ERROR(
						cpl_image_power(errImage, 0.5));
				break;
			case rmse:
				// nothing to do, RMSE is Ok
				break;
			case invmse:
				//intended fall through !
				CPL_ATTR_FALLTRHU; 
			case invrmse:
				BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
						"Cannot transfer error extension type %s to a HDLR error"
						" image", deqErrorTypeString[errorType]);
				break;
			default:
				BRK_WITH_ERROR_MSG(CPL_ERROR_ACCESS_OUT_OF_RANGE,
						"Internal error: unknown errorType enumeration value: %d",
						errorType);
				break;
			}

			if (qualImage != NULL) {
				switch (*qualityType) {
				case maskzero:
					//intended fall through !
					CPL_ATTR_FALLTRHU;
				case flag16bit:
					//intended fall through !
					CPL_ATTR_FALLTRHU; 
				case flag32bit: // The value "0" indicates a "good" pixel
					mask = cpl_mask_threshold_image_create(qualImage,
							-1.0, 1.0);
					break;
				case maskone: // The value "1" indicates a "good" pixel
					mask = cpl_mask_threshold_image_create(qualImage,
							0.5, 2.0);
					break;
				default:
					BRK_WITH_ERROR_MSG(
							CPL_ERROR_ACCESS_OUT_OF_RANGE,
							"Internal error: unknown qualityType enumeration"
							" value: %d",
							*qualityType);
					break;
				}
			}
			cpl_mask_not(mask); //hey it is a BAD pixel mask!

			int pis_rejected;
			for (cpl_size ny = 1; ny <= cpl_image_get_size_y(dataImage); ny++) {
				for (cpl_size nx = 1; nx <= cpl_image_get_size_x(dataImage); nx++) {
					if (eris_ifu_is_nan_or_inf(cpl_image_get(dataImage, nx, ny, &pis_rejected)) ||
							eris_ifu_is_nan_or_inf(cpl_image_get(errImage, nx, ny, &pis_rejected))
					) {
                        cpl_mask_set(mask, nx, ny, BAD_PIX);
					}
				}
			}

			cpl_image_set_bpm(dataImage, mask);
			cpl_image_set_bpm(errImage, cpl_mask_duplicate(mask));
			CHECK_ERROR_STATE();
			cpl_imagelist_set(dataImagelist, dataImage, ix);
			cpl_imagelist_set(errImagelist, errImage, ix);

		}
		imagelist = hdrl_imagelist_create(
				dataImagelist, errImagelist);
	}
	CATCH
	{
		hdrl_imagelist_delete(imagelist);
		imagelist = NULL;
	}
	cpl_imagelist_delete(dataImagelist);
	cpl_imagelist_delete(errImagelist);
	eris_check_error_code("eris_ifu_load_deq_hdrl_imagelist");
	return imagelist;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Load a calibration image from a frame
 * @param frame        Input frame
 * @param band         Expected instrument band (or UNDEFINED_BAND to skip check)
 * @param scale        Expected pre-optics scale (or UNDEFINED_SCALE to skip check)
 * @param qualImage    Output quality image (pointer to pointer, can be NULL)
 * @param qualType     Output quality type detected (pointer, can be NULL)
 * @return   HDRL image on success, or NULL on error
 *
 * Loads a calibration image frame and verifies that its band and scale match
 * the expected values. Automatically handles both DEQ format (3 extensions)
 * and old format (single image) files.
 *
 * @note For old format files, a simulated error image is created (10% of data).
 * @note PUPIL scale is always accepted regardless of the scale parameter.
 */
/*----------------------------------------------------------------------------*/
hdrl_image *eris_ifu_load_cal_image_frame(
        const cpl_frame *frame,
        ifsBand band,
        ifsPreopticsScale scale,
        cpl_image **qualImage,
        deqQualityType *qualType)
{
    hdrl_image *image = NULL;
    const char *filename;

    cpl_ensure(frame, CPL_ERROR_NULL_INPUT, NULL);

    TRY
	{
    	filename = cpl_frame_get_filename(frame);
    	image = eris_ifu_load_cal_image_file(
    			filename, band, scale, qualImage, qualType);
	}
    CATCH
    {
        hdrl_image_delete(image);
        image = NULL;
    }
    eris_check_error_code("eris_ifu_load_cal_image_frame");
    return image;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Load a calibration image from a filename
 * @param filename    Input FITS filename
 * @param band        Expected instrument band (or UNDEFINED_BAND to skip check)
 * @param scale       Expected pre-optics scale (or UNDEFINED_SCALE to skip check)
 * @param qualImage   Output quality image (pointer to pointer, can be NULL)
 * @param qualType    Output quality type detected (pointer, can be NULL)
 * @return   HDRL image on success, or NULL on error
 *
 * Loads a calibration image file and verifies that its band and scale match
 * the expected values. Automatically handles both DEQ format (3 extensions)
 * and old format (single image) files.
 *
 * @note For old format files, a simulated error image is created (10% of data absolute value).
 * @note PUPIL scale is always accepted regardless of the scale parameter.
 * @note The function checks number of extensions to determine format.
 */
/*----------------------------------------------------------------------------*/
hdrl_image *eris_ifu_load_cal_image_file(
		const char *filename,
		ifsBand band,
		ifsPreopticsScale scale,
		cpl_image **qualImage,
		deqQualityType *qualType)
{
	hdrl_image *image = NULL;
	cpl_propertylist *header = NULL;
	cpl_image *dataImage = NULL;
	cpl_image *errorImage = NULL;
	cpl_size nExt;
	//    ifsBand thisBand;
	//    ifsPreopticsScale thisScale;

	cpl_ensure(filename, CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{
		nExt = cpl_fits_count_extensions(filename);
		if (nExt == 3) { // new format data/error/quality DEQ
			image = eris_ifu_load_deq_hdrl_image(filename, &header,
					qualImage, qualType);
		} else {  // old format, data only,  error will be simulated
			header = cpl_propertylist_load(filename, 0);
			dataImage = cpl_image_load(filename, CPL_TYPE_UNSPECIFIED,
					0 , 0);
			errorImage =  cpl_image_abs_create(dataImage);
			cpl_image_multiply_scalar(errorImage, 0.1);
			image = hdrl_image_create(dataImage, errorImage);
			*qualImage = NULL;
			if (qualType != NULL) {
				*qualType = unspecified;
			}
		}
		if ((band != UNDEFINED_BAND) &&
				(band != eris_ifu_get_band(header))) {
			BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
					"different instrument band setting"
					" in calibration frame %s", filename);
		}
		if ((scale != UNDEFINED_SCALE) &&
				(scale != PUPIL) &&
				(scale != eris_ifu_get_preopticsScale(header))) {
			BRK_WITH_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
					"different instrument pre-optics setting"
					" in calibration frame %s", filename);
		}

	}
	CATCH
	{
		hdrl_image_delete(image);
		image = NULL;
	}
	eris_ifu_free_propertylist(&header);
	eris_ifu_free_image(&dataImage);
	eris_ifu_free_image(&errorImage);
	eris_check_error_code("eris_ifu_load_cal_image_file");
	return image;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Load old-format distortion polynomials from a table
 * @param filename   Input FITS table filename
 * @param poly_u     Output polynomial for u direction (pointer to pointer)
 * @param poly_v     Output polynomial for v direction (pointer to pointer)
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Loads distortion correction polynomials from an old-format table with
 * columns "degx", "degy", and "coeff". The v-direction polynomial is set
 * to a simple identity (coefficient 1.0 for y^1 term).
 *
 * @note This is for backward compatibility with old SINFONI format.
 * @note The returned polynomials must be freed with cpl_polynomial_delete().
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_load_distortion_polynomials_old(
        const char* filename,
        cpl_polynomial ** poly_u,
        cpl_polynomial ** poly_v)
{
	cpl_error_code retVal = CPL_ERROR_NONE;
	cpl_table *distortionTable = NULL;
	cpl_size nTableRows;
	cpl_size pows[2];
	double coeff;

	TRY
	{
		*poly_u = cpl_polynomial_new(2);
		*poly_v = cpl_polynomial_new(2);

		distortionTable = cpl_table_load(filename,1,0);
		nTableRows = cpl_table_get_nrow(distortionTable);

		for (cpl_size i=0; i<nTableRows; i++) {
			pows[0] =  cpl_table_get_int(distortionTable, "degx", i, NULL);
			pows[1] =  cpl_table_get_int(distortionTable, "degy", i, NULL);
			coeff = cpl_table_get_double(distortionTable, "coeff", i, NULL);
			cpl_polynomial_set_coeff(*poly_u, pows, coeff);
		}
		CHECK_ERROR_STATE();

		pows[0] = 0;
		pows[1] = 1;
		cpl_polynomial_set_coeff(*poly_v, pows, 1.0);
		CHECK_ERROR_STATE();
	}
	CATCH
	{
		retVal = cpl_error_get_code();
	}

	eris_ifu_free_table(&distortionTable);
	eris_check_error_code("eris_ifu_load_cal_image_file");
	return retVal;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Load distortion polynomials and slitlet borders from a table
 * @param filename      Input FITS table filename
 * @param polynomials   Output array of polynomials (one per slitlet, pointer to pointer)
 * @param borders       Output table with slitlet border positions (pointer to pointer)
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Loads distortion correction polynomials for all slitlets. The file must
 * contain SLITLET_CNT+1 extensions: one extension per slitlet with polynomial
 * coefficients, plus a final extension with border positions.
 *
 * @note The polynomial array must be freed by calling cpl_polynomial_delete()
 *       on each element, then freeing the array itself.
 * @note The borders table must be freed with cpl_table_delete().
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_load_distortion_polynomials(
		const char* filename,
		cpl_polynomial ***polynomials,
		cpl_table **borders)
{
	cpl_error_code retVal = CPL_ERROR_NONE;
	cpl_table *distortionTable = NULL;

	TRY
	{
		*borders = NULL;

		*polynomials =
				cpl_calloc(SLITLET_CNT, sizeof(cpl_polynomial*));

		for (int sx = 0; sx < SLITLET_CNT; sx++) {
			cpl_size nTableRows;

			eris_ifu_free_table(&distortionTable);
			distortionTable = cpl_table_load(filename, sx+1, 0);
			nTableRows = cpl_table_get_nrow(distortionTable);

			(*polynomials)[sx] = cpl_polynomial_new(2);

			for (cpl_size i=0; i<nTableRows; i++) {
				cpl_size pows[2];
				double coeff;

				pows[0] =  cpl_table_get_int(distortionTable, "degx", i, NULL);
				pows[1] =  cpl_table_get_int(distortionTable, "degy", i, NULL);
				coeff = cpl_table_get_double(distortionTable, "coeff", i, NULL);
				cpl_polynomial_set_coeff((*polynomials)[sx], pows, coeff);
			}
			CHECK_ERROR_STATE();

		}

		*borders = cpl_table_load(filename, SLITLET_CNT+1, 0);
	}
	CATCH
	{
		if (*polynomials != NULL) {
			for (int sx = 0; sx < SLITLET_CNT; sx++) {
				cpl_polynomial_delete((*polynomials)[sx]);
			}
		}
		eris_ifu_free_table(borders);
		retVal = cpl_error_set_where(cpl_func);
	}

	eris_ifu_free_table(&distortionTable);
	eris_check_error_code("eris_ifu_load_distortion_polynomials");
	return retVal;

}

#define ERIS_IFU_DIST_SLIT_DISTANCE "slitlet_distance"

/*----------------------------------------------------------------------------*/
/**
 * @brief    Load slitlet distance vector from a table
 * @param filename   Input FITS table filename
 * @return   Vector with slitlet distances, or NULL on error
 *
 * Loads the "slitlet_distance" column from the first extension of the
 * specified FITS table file. Supports both DOUBLE and FLOAT column types.
 *
 * @note The returned vector must be freed with cpl_vector_delete().
 */
/*----------------------------------------------------------------------------*/
cpl_vector *eris_ifu_load_distances(
		const char* filename)
{
	cpl_vector  *distances  = NULL;
	cpl_table   *table      = NULL;
	cpl_size    nTableRows  = 0;
	double      val         = 0.;

	TRY
	{
		table = cpl_table_load(filename,1,0);
		nTableRows = cpl_table_get_nrow(table);

		distances = cpl_vector_new(nTableRows);
		cpl_type type = cpl_table_get_column_type(table, ERIS_IFU_DIST_SLIT_DISTANCE);
		for (cpl_size i=0; i<nTableRows; i++) {
			if (type == CPL_TYPE_DOUBLE) {
				val = cpl_table_get_double(table, ERIS_IFU_DIST_SLIT_DISTANCE, i, NULL);
			} else if (type == CPL_TYPE_FLOAT) {
				val = cpl_table_get_float(table, ERIS_IFU_DIST_SLIT_DISTANCE, i, NULL);
			} else {
				SET_ERROR(CPL_ERROR_INVALID_TYPE);
			}
			BRK_IF_ERROR(
					cpl_vector_set(distances, i, val));
		}
	}
	CATCH
	{
		cpl_vector_delete(distances);
		distances = NULL;
	}

	eris_ifu_free_table(&table);
	eris_check_error_code("eris_ifu_load_distances");
	return distances;
}

#define ERIS_IFU_DIST_EDGE_L_old    "pos1"
#define ERIS_IFU_DIST_EDGE_R_old    "pos2"

/*----------------------------------------------------------------------------*/
/**
 * @brief    Load slitlet position bivector from a table
 * @param filename   Input FITS table filename
 * @return   Bivector with slitlet positions (x=left edge, y=right edge), or NULL on error
 *
 * Loads slitlet edge positions from a table. Supports both new format
 * (edge_left/edge_right columns) and old SINFONI format (pos1/pos2 columns).
 *
 * @note The returned bivector must be freed with cpl_bivector_delete().
 */
/*----------------------------------------------------------------------------*/
cpl_bivector  * eris_ifu_load_slit_positions(
		const char* filename)
{
	cpl_bivector *slitPos = NULL;
	cpl_vector *pos1 = NULL;
	cpl_vector *pos2 = NULL;
	cpl_table *slitPosTable = NULL;
	cpl_size nTableRows;

	TRY
	{
		slitPosTable = cpl_table_load(filename,1,0);
		nTableRows = cpl_table_get_nrow(slitPosTable);

		slitPos = cpl_bivector_new(nTableRows);
		pos1 = cpl_bivector_get_x(slitPos);
		pos2 = cpl_bivector_get_y(slitPos);
		CHECK_ERROR_STATE();

		for (cpl_size i=0; i<nTableRows; i++) {
			double val = 0.;
			// check if name of table column is "edge_left"
			val = cpl_table_get_double(slitPosTable, ERIS_IFU_DIST_EDGE_L, i, NULL);
			if (cpl_error_get_code()== CPL_ERROR_DATA_NOT_FOUND) {
				// ok, it seems to be old fashioned sinfoni "pos1"
				RECOVER();
				val = cpl_table_get_double(slitPosTable, ERIS_IFU_DIST_EDGE_L_old, i, NULL);
			}
			cpl_vector_set(pos1, i, val);

			// check if name of table column is "edge_right"
			val = cpl_table_get_double(slitPosTable, ERIS_IFU_DIST_EDGE_R, i, NULL);
			if (cpl_error_get_code()== CPL_ERROR_DATA_NOT_FOUND) {
				// ok, it seems to be old fashioned sinfoni "pos2"
				RECOVER();
				val = cpl_table_get_double(slitPosTable, ERIS_IFU_DIST_EDGE_R_old, i, NULL);
			}
			cpl_vector_set(pos2, i, val);
		}
	}
	CATCH
	{
		cpl_bivector_delete(slitPos);
		slitPos = NULL;
	}

	eris_ifu_free_table(&slitPosTable);
	eris_check_error_code("eris_ifu_load_slit_positions");
	return slitPos;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Append a QC parameter of type INT to a property list
 * @param pl        Property list
 * @param name      QC parameter name (without "ESO QC " prefix)
 * @param val       QC parameter integer value
 * @param comment   Comment string to attach
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Adds a quality control parameter with automatic "ESO QC " prefix to the
 * property list with the specified value and comment.
 *
 * @note The "ESO QC " prefix is automatically added to the name.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_append_qc_int(
        cpl_propertylist *pl,
        const char *name,
        int val,
        const char* comment)
{
	cpl_error_code retVal = CPL_ERROR_NONE;
	char *qcName = NULL;

	cpl_ensure_code(pl, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(name, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(comment, CPL_ERROR_NULL_INPUT);

	TRY
	{
		qcName = cpl_sprintf("ESO QC %s", name);
		cpl_propertylist_append_int(pl, qcName, val);
		cpl_propertylist_set_comment(pl , qcName, comment);
	}
	CATCH
	{
		retVal = cpl_error_get_code();
	}

	eris_ifu_free_string(&qcName);
	eris_check_error_code("eris_ifu_append_qc_int");
	return retVal;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Append a QC parameter of type DOUBLE to a property list
 * @param pl        Property list
 * @param name      QC parameter name (without "ESO QC " prefix)
 * @param val       QC parameter double value
 * @param comment   Comment string to attach
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Adds a quality control parameter with automatic "ESO QC " prefix to the
 * property list with the specified value and comment.
 *
 * @note The "ESO QC " prefix is automatically added to the name.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_append_qc_double(
		cpl_propertylist *pl,
		const char *name,
		double val,
		const char* comment)
{
	cpl_error_code retVal = CPL_ERROR_NONE;
	char *qcName = NULL;

	cpl_ensure_code(pl, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(name, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(comment, CPL_ERROR_NULL_INPUT);

	TRY
	{
		qcName = cpl_sprintf("ESO QC %s", name);
		cpl_propertylist_append_double(pl, qcName, val);
		cpl_propertylist_set_comment(pl , qcName, comment);
	}
	CATCH
	{
		retVal = cpl_error_get_code();
	}

	eris_ifu_free_string(&qcName);
	eris_check_error_code("eris_ifu_append_qc_double");
	return retVal;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Append a QC parameter of type FLOAT to a property list
 * @param pl        Property list
 * @param name      QC parameter name (without "ESO QC " prefix)
 * @param val       QC parameter float value
 * @param comment   Comment string to attach
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Adds a quality control parameter with automatic "ESO QC " prefix to the
 * property list with the specified value and comment.
 *
 * @note The "ESO QC " prefix is automatically added to the name.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_append_qc_float(
		cpl_propertylist *pl,
		const char *name,
		float val,
		const char* comment)
{
	cpl_error_code retVal = CPL_ERROR_NONE;
	char *qcName = NULL;

	cpl_ensure_code(pl, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(name, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(comment, CPL_ERROR_NULL_INPUT);

	TRY
	{
		qcName = cpl_sprintf("ESO QC %s", name);
		cpl_propertylist_append_float(pl, qcName, val);
		cpl_propertylist_set_comment(pl , qcName, comment);
	}
	CATCH
	{
		retVal = cpl_error_get_code();
	}

	eris_ifu_free_string(&qcName);
	eris_check_error_code("eris_ifu_append_qc_float");
	return retVal;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Set (or update) a QC parameter of type INT in a property list
 * @param pl        Property list
 * @param name      QC parameter name (without "ESO QC " prefix)
 * @param val       QC parameter integer value
 * @param comment   Comment string to attach
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Sets or updates a quality control parameter with automatic "ESO QC " prefix.
 * If the parameter already exists, its value is updated.
 *
 * @note The "ESO QC " prefix is automatically added to the name.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_set_qc_int(cpl_propertylist *pl,
		const char *name,
		int val,
		const char *comment)
{
	cpl_error_code  retVal  = CPL_ERROR_NONE;
	char            *qcName = NULL;

	cpl_ensure_code(pl,      CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(name,    CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(comment, CPL_ERROR_NULL_INPUT);

	TRY
	{
		qcName = cpl_sprintf("ESO QC %s", name);
		cpl_propertylist_set_int(pl, qcName, val);
		cpl_propertylist_set_comment(pl, qcName, comment);
	}
	CATCH
	{
		retVal = cpl_error_get_code();
	}

	eris_ifu_free_string(&qcName);
	eris_check_error_code("eris_ifu_append_qc_int");
	return retVal;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Determine if a frame has calibration lamps ON or OFF
 * @param fr   Input frame
 * @return   True if frame is lamp-ON or science observation, false otherwise
 *
 * Checks the DPR.TYPE keyword and calibration lamp status keywords to determine
 * if the frame is an "on" frame. Science frames (STD, PSF, SKY, OBJECT) and
 * frames with any calibration lamp turned on are considered "on" frames.
 *
 * @note Checks both SPIFFIER and SPIFFI lamp status keywords.
 * @note Returns false if the file cannot be read or keywords are missing.
 */
/*----------------------------------------------------------------------------*/
bool eris_ifu_frame_is_on(const cpl_frame *fr)
{
    bool                result      = false;
    cpl_propertylist    *pl         = NULL;
    const char          *fn         = NULL;
    char                *dpr_type   = NULL;

    cpl_ensure(fr != NULL, CPL_ERROR_NULL_INPUT, false);

    TRY
	{
    	fn = cpl_frame_get_filename(fr);
    	eris_ifu_file_exists(fn);

    	pl = cpl_propertylist_load(fn, 0);

    	if (cpl_propertylist_has(pl, FHDR_DPR_TYPE)) {
    		dpr_type = cpl_strdup(
    				cpl_propertylist_get_string(pl, FHDR_DPR_TYPE));
    	} else {
    		BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
    				"keyword %s does not exist",
				FHDR_DPR_TYPE);
    	}

    	if ((strstr(dpr_type, "STD") != NULL) ||
    			(strstr(dpr_type, "PSF") != NULL) ||
			(strstr(dpr_type, "SKY") != NULL) ||
			(strstr(dpr_type, "OBJECT") != NULL))
    	{
    		result = true;
    	} else {
    		if (eris_ifu_get_callamp_status(pl) != 0)
    		{
    			result = true;
    		}
    	}
	}
    CATCH
	{
    	result = false;
	}

    eris_ifu_free_propertylist(&pl);
    eris_ifu_free_string(&dpr_type);
    eris_check_error_code("eris_ifu_frame_is_on");
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Determine if a frame is a sky frame
 * @param fr   Input frame
 * @return   True if frame is a sky frame, false otherwise
 *
 * Checks the DPR.TYPE keyword to determine if the frame contains a sky
 * observation (calibration or science).
 *
 * @note Returns false if the file cannot be read or keywords are missing.
 */
/*----------------------------------------------------------------------------*/
bool eris_ifu_frame_is_sky(const cpl_frame *fr)
{
	bool                result      = false;
	cpl_propertylist    *pl         = NULL;
	const char          *fn         = NULL;
	char                *dpr_type   = NULL;

	cpl_ensure(fr, CPL_ERROR_NULL_INPUT, false);

	TRY
	{
		fn = cpl_frame_get_filename(fr);
		eris_ifu_file_exists(fn);
		pl = cpl_propertylist_load(fn, 0);


		if (cpl_propertylist_has(pl, FHDR_DPR_TYPE)) {
			dpr_type = cpl_strdup(
					cpl_propertylist_get_string(pl, FHDR_DPR_TYPE));
		} else {
			BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
					"keyword %s does not exist",
					FHDR_DPR_TYPE);
			// go to catch clause
		}

		if (strstr(dpr_type, ERIS_IFU_RAW_SKY) != NULL) {
			result = true;
		}
	}
	CATCH
	{
		result = false;
	}

	eris_ifu_free_propertylist(&pl);
	eris_ifu_free_string(&dpr_type);
	eris_check_error_code("eris_ifu_frame_is_sky");
	return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Get the detector integration time (DIT) from FITS header
 * @param header   FITS header property list
 * @return   DIT value in seconds, or 0.0 on error
 *
 * Extracts the DET.SEQ1.DIT keyword value from the header. The keyword name
 * differs between SPIFFIER and SPIFFI instruments.
 *
 * @note Returns 0.0 if the header is NULL or the keyword cannot be read.
 */
/*----------------------------------------------------------------------------*/
float eris_ifu_get_dit(cpl_propertylist *header) {
    float dit = 0;
    ifsInstrument instrument = UNSET_INSTRUMENT;

    cpl_ensure(header, CPL_ERROR_NULL_INPUT, 0);

    TRY
    {
        instrument = eris_ifu_get_instrument(header);
        CHECK_ERROR_STATE();
        if (instrument == SPIFFIER) {
            dit = cpl_propertylist_get_float(header, FHDR_E_DIT);

        } else if (instrument == SPIFFI) {
            dit = cpl_propertylist_get_float(header, FHDR_S_DIT);
        }
        CHECK_ERROR_STATE();
    }
    CATCH
    {
        dit = 0;
    }
    eris_check_error_code("eris_ifu_get_dit");
    return dit;
}

/**
  @brief    Return a bit mask indicating which calibration lamps are turned on
  @param    header of the FITS file as property list
  @return   an integer bit mask, for each active lamp the bit is set

  The bit values for the lamps are defined in the AR_LAMP, KR_LAMP, NE_LAMP,
  XE_LAMP and QTH_LAMP macros.

  Possible cpl_error_code set in this function:\n
  - CPL_ERROR_NULL_INPUT The header parameter is a NULL pointer
  - CPL_ERROR_DATA_NOT_FOUND The header ' an ERIS/SPIFFIER nor
   an SINFONI/SPIFFI one.

 */
int eris_ifu_get_callamp_status(cpl_propertylist *header)
{
    ifsInstrument instrument = UNSET_INSTRUMENT;
    int lampMask = 0;
        int lampAr      = 0;
        int lampKr      = 0;
        int lampNe      = 0;
        int lampXe      = 0;
        int lampQth     = 0;

        cpl_ensure_code(header, CPL_ERROR_NULL_INPUT);

    TRY
    {
        instrument = eris_ifu_get_instrument(header);
        CHECK_ERROR_STATE();
        if (instrument == SPIFFIER) {
            cpl_msg_debug("eris_ifu_get_callamp_status", "ERIS/SPIFFIER found");
            if (cpl_propertylist_has(header, FHDR_E_AR_LAMP_ST)) {
                if (cpl_propertylist_get_bool(header, FHDR_E_AR_LAMP_ST)) {
                    lampMask += AR_LAMP;
                    lampAr = 1;
                }
            }
            if (cpl_propertylist_has(header, FHDR_E_KR_LAMP_ST)) {
                if (cpl_propertylist_get_bool(header, FHDR_E_KR_LAMP_ST)) {
                    lampMask += KR_LAMP;
                    lampKr = 1;
                }
            }
            if (cpl_propertylist_has(header, FHDR_E_NE_LAMP_ST)) {
                if (cpl_propertylist_get_bool(header, FHDR_E_NE_LAMP_ST)) {
                    lampMask += NE_LAMP;
                    lampNe = 1;
                }
            }
            if (cpl_propertylist_has(header, FHDR_E_XE_LAMP_ST)) {
                if (cpl_propertylist_get_bool(header, FHDR_E_XE_LAMP_ST)) {
                    lampMask += XE_LAMP;
                    lampXe = 1;
                }
            }
            if (cpl_propertylist_has(header, FHDR_E_QTH_LAMP_ST)) {
                if (cpl_propertylist_get_bool(header,FHDR_E_QTH_LAMP_ST)) {
                    lampMask += QTH_LAMP;
                    lampQth = 1;
                }
            }
        } else if (instrument == SPIFFI) {
            cpl_msg_debug("eris_ifu_get_callamp_status", "SINFONI/SPIFFI found");
            if (cpl_propertylist_get_bool(header,FHDR_S_AR_LAMP_ST)) {
                lampMask += AR_LAMP;
                lampAr = 1;
            }
            if (cpl_propertylist_get_bool(header,FHDR_S_KR_LAMP_ST)) {
                lampMask += KR_LAMP;
                lampKr = 1;
            }
            if (cpl_propertylist_get_bool(header,FHDR_S_NE_LAMP_ST)) {
                lampMask += NE_LAMP;
                lampNe = 1;
            }
            if (cpl_propertylist_get_bool(header,FHDR_S_XE_LAMP_ST)) {
                lampMask += XE_LAMP;
                lampXe = 1;
            }
            if (cpl_propertylist_get_bool(header,FHDR_S_QTH_LAMP_ST)) {
                lampMask += QTH_LAMP;
                lampQth = 1;
            }
        }
    }
    CATCH
    {
        lampMask = 0;
    }
    cpl_msg_debug(cpl_func,
            "Lamps: Ar: %d, Kr: %d, Ne: %d, Xe: %d, Qth: %d, mask: %d",
            lampAr, lampKr, lampNe, lampXe, lampQth, lampMask);
    eris_check_error_code("eris_ifu_get_callamp_status");
    return lampMask;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Determine preoptic band
  @param    header        input FITS header
  @return   INS3 SPGW ID
 */
ifsBand eris_ifu_get_band(const cpl_propertylist *header)
{
    ifsBand         band        = UNDEFINED_BAND;
    ifsInstrument   instrument  = UNSET_INSTRUMENT;
    const char      *bandId     = NULL;

    TRY
    {
        if (header == NULL) {
            BRK_WITH_ERROR(CPL_ERROR_NULL_INPUT);
        }

        instrument = eris_ifu_get_instrument(header);
        CHECK_ERROR_STATE();
        if (instrument == SPIFFIER) {
            bandId = cpl_propertylist_get_string(header, FHDR_E_GRATING_ID);
            CHECK_ERROR_STATE();
            if (strcmp(bandId, E_GRATING_J_LOW) == 0) {
                band = J_LOW;
            } else if (strcmp(bandId, E_GRATING_J_SHORT) == 0) {
                band = J_SHORT;
            } else if (strcmp(bandId, E_GRATING_J_MIDDLE) == 0) {
                band = J_MIDDLE;
            } else if (strcmp(bandId, E_GRATING_J_LONG) == 0) {
                band = J_LONG;
            } else if (strcmp(bandId, E_GRATING_H_LOW) == 0) {
                band = H_LOW;
            } else if (strcmp(bandId, E_GRATING_H_SHORT) == 0) {
                band = H_SHORT;
            } else if (strcmp(bandId, E_GRATING_H_MIDDLE) == 0) {
                band = H_MIDDLE;
            } else if (strcmp(bandId, E_GRATING_H_LONG) == 0) {
                band = H_LONG;
            } else if (strcmp(bandId, E_GRATING_K_LOW) == 0) {
                band = K_LOW;
            } else if (strcmp(bandId, E_GRATING_K_SHORT) == 0) {
                band = K_SHORT;
            } else if (strcmp(bandId, E_GRATING_K_MIDDLE) == 0) {
                band = K_MIDDLE;
            } else if (strcmp(bandId, E_GRATING_K_LONG) == 0) {
                band = K_LONG;
            } else if (strcmp(bandId, E_PREOPTICS_PUPIL) == 0) {
            	cpl_msg_warning(cpl_func,"found pupil data");
                band = B_PUPIL;
            }
        } else if (instrument == SPIFFI) {
            bandId = cpl_propertylist_get_string(header, FHDR_S_GRATING_ID);
            CHECK_ERROR_STATE();
            if (strcmp(bandId, S_GRATING_J_BAND) == 0) {
                band = J_SPIFFI;
            } else if (strcmp(bandId, S_GRATING_H_BAND) == 0) {
                band = H_SPIFFI;
            } else if (strcmp(bandId, S_GRATING_K_BAND) == 0) {
                band = K_SPIFFI;
            } else if (strcmp(bandId, S_GRATING_HK_BAND) == 0) {
                band = HK_SPIFFI;
            } else {
                band = UNDEFINED_BAND;
            }
        }

    }CATCH
    {
        band = UNDEFINED_BAND;
    }
    eris_check_error_code("eris_ifu_get_band");
    return band;
}

/**
  @brief    Return the used instrument of the FITS file header
  @param    header of the FITS file as property list
  @return   an value of the instrument enumeration (SPIFFIER,SPIFFI,NIX,OTHER)

  Possible cpl_error_code set in this function:\n
  - CPL_ERROR_NULL_INPUT The header parameter is a NULL pointer
  - CPL_ERROR_BAD_FILE_FORMAT Unknown VLT instrument in header

 */
ifsInstrument eris_ifu_get_instrument(const cpl_propertylist *header)
{
    ifsInstrument   instrument      = UNSET_INSTRUMENT;
    const char      *instrumentStr  = NULL;
    const  char     *armStr         = NULL;

    TRY
    {
        if (header == NULL) {
            BRK_WITH_ERROR(CPL_ERROR_NULL_INPUT);
        }

        instrumentStr = cpl_propertylist_get_string(header, FHDR_INSTRUMENT);
        CHECK_ERROR_STATE_MSG(
            "Cannot read "FHDR_INSTRUMENT" FITS keyword");
        if (strcmp(instrumentStr, INSTRUMENT_SINFONI) == 0) {
            instrument = SPIFFI;
        } else if (strcmp(instrumentStr, INSTRUMENT_ERIS) == 0) {
            armStr = cpl_propertylist_get_string(header, FHDR_E_ARM);
            CHECK_ERROR_STATE_MSG(
                "Cannot read "FHDR_E_ARM" FITS keyword");
            if (strcmp(armStr, ARM_SPIFFIER) == 0) {
                instrument = SPIFFIER;
            } else if (strcmp(armStr, ARM_NIX) == 0) {
                cpl_msg_error(cpl_func, "ERIS/NIX frame detected");
                SET_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
                    "ERIS/NIX frame detected");
                instrument = NIX;
            } else {
                cpl_msg_error("",
                    "Unkown instrument arm %s, neither SPIFFIER nor NIX",
                    armStr);
                SET_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
                    "Unkown instrument arm, neither SPIFFIER nor NIX");
                instrument = OTHER_INSTRUMENT;
            }
        } else {
            cpl_msg_error("",
                "Unkown instrument %s, neither ERIS/SPIFFIER nor SINFONI",
                instrumentStr);
            SET_ERROR_MSG(CPL_ERROR_BAD_FILE_FORMAT,
                "Unkown instrument, neither ERIS/SPIFFIER nor SINFONI");
            instrument = OTHER_INSTRUMENT;
        }
    }
    CATCH
    {
        instrument = UNSET_INSTRUMENT;
    }
    eris_check_error_code("eris_ifu_get_instrument");
    return instrument;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Determine observing scale on Sky
  @param    scaleId    scale ID
  @return   scale setting
 */
ifsPreopticsScale
eris_ifu_get_spiffier_preoptics_scale(const char *scaleId)
{

    ifsPreopticsScale scale = UNDEFINED_SCALE;
    if (strcmp(scaleId, E_PREOPTICS_250MAS_SCALE) == 0) {
        scale = S250MAS;
    } else if (strcmp(scaleId, E_PREOPTICS_100MAS_SCALE) == 0) {
        scale = S100MAS;
    } else if (strcmp(scaleId, E_PREOPTICS_25MAS_SCALE) == 0) {
        scale = S25MAS;
    } else if (strcmp(scaleId, E_PREOPTICS_PUPIL) == 0) {
        scale = PUPIL;
    } else {
        scale = UNDEFINED_SCALE;
    }
    eris_check_error_code("eris_ifu_get_spiffier_preoptics_scale");
    return scale;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Determine observing scale on Sky
  @param    scaleId    scale ID
  @return   scale setting
 */
ifsPreopticsScale
eris_ifu_get_spiffi_preoptics_scale(const char *scaleId)
{
    ifsPreopticsScale scale = UNDEFINED_SCALE;
    if (strcmp(scaleId, S_PREOPTICS_250MAS_SCALE) == 0) {
        scale = S250MAS;
    } else if (strcmp(scaleId, S_PREOPTICS_100MAS_SCALE) == 0) {
        scale = S100MAS;
    } else if (strcmp(scaleId, S_PREOPTICS_25MAS_SCALE) == 0) {
        scale = S25MAS;
    } else if (strcmp(scaleId, S_PREOPTICS_PUPIL) == 0) {
        scale = PUPIL;
    } else {
        scale = UNDEFINED_SCALE;
    }
    eris_check_error_code("eris_ifu_get_spiffi_preoptics_scale");
    return scale;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Return the the pre-optics scaling
  @param    header of the FITS file as property list
  @return   an value of the preopticsScale enumeration (S250MAS, S100MAS or
                    S25MAS)
  )

  Possible cpl_error_code set in this function:\n
  - CPL_ERROR_NULL_INPUT The header parameter is a NULL pointer
  - CPL_ERROR_BAD_FILE_FORMAT Unknown VLT instrument in header

 */
/*----------------------------------------------------------------------------*/
ifsPreopticsScale eris_ifu_get_preopticsScale(
        cpl_propertylist *header)
{
    ifsPreopticsScale scale = UNDEFINED_SCALE;
    ifsInstrument instrument = UNSET_INSTRUMENT;
    const char *scaleId;

    TRY
    {
        instrument = eris_ifu_get_instrument(header);
        CHECK_ERROR_STATE();
        if (instrument == SPIFFIER) {
            scaleId = cpl_propertylist_get_string(header, FHDR_E_PREOPTICS_ID);
            CHECK_ERROR_STATE();
            scale = eris_ifu_get_spiffier_preoptics_scale(scaleId);
        } else if (instrument == SPIFFI) {
            scaleId = cpl_propertylist_get_string(header, FHDR_S_PREOPTICS_ID);
            CHECK_ERROR_STATE();
            scale = eris_ifu_get_spiffi_preoptics_scale(scaleId);
        }

    }
    CATCH
    {
        scale = UNDEFINED_SCALE;
    }
    eris_check_error_code("eris_ifu_get_preopticsScale");
    return scale;

}

#define KEY_NAME_COMOFFS_RA                   "ESO OCS CUMOFFS RA"
#define KEY_NAME_COMOFFS_DEC                   "ESO OCS CUMOFFS DEC"


/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO OCS CUMOFFS RA of the observation
  @param    frame input frame
  @return   the requested value or 0.0 on error

  Queries FITS header ESO OCS CUMOFFS RA
 */
/*----------------------------------------------------------------------------*/

double eris_get_cumoffs_ra(cpl_frame * frame)
{
	cpl_ensure(frame, CPL_ERROR_NULL_INPUT, -1.0);
    cpl_propertylist* plist=NULL;
    char* file=NULL;

    double result=0.;
    file = cpl_strdup( cpl_frame_get_filename(frame)) ;

    if ((cpl_error_code)((plist = cpl_propertylist_load(file, 0)) == NULL)) {
        cpl_msg_error(cpl_func, "getting header from reference frame %s",file);
        cpl_propertylist_delete(plist) ;
        cpl_free(file);
        return -1 ;
    }

    if (cpl_propertylist_has(plist, KEY_NAME_COMOFFS_RA)) {
        result=cpl_propertylist_get_double(plist, KEY_NAME_COMOFFS_RA);
    } else {
        cpl_msg_error(cpl_func, "keyword %s does not exist",KEY_NAME_COMOFFS_RA);
        cpl_propertylist_delete(plist) ;
        cpl_free(file);
        return -1;
    }
    cpl_propertylist_delete(plist) ;
    cpl_free(file);
    eris_check_error_code("eris_get_cumoffs_ra");
    return result;

}


/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO OCS CUMOFFS DEC of the observation
  @param    frame input frame
  @return   the requested value or 0.0 on error

  Queries FITS header ESO OCS CUMOFFS DEC
 */
/*----------------------------------------------------------------------------*/

double eris_get_cumoffs_dec(cpl_frame * frame)
{
	cpl_ensure(frame, CPL_ERROR_NULL_INPUT, -999.0);
    cpl_propertylist* plist=NULL;
    char* file=NULL;

    double result=0.;
    file = cpl_strdup( cpl_frame_get_filename(frame)) ;

    if ((cpl_error_code)((plist = cpl_propertylist_load(file, 0)) == NULL)) {
        cpl_msg_error(cpl_func, "getting header from reference frame %s",file);
        cpl_propertylist_delete(plist) ;
        cpl_free(file);
        return -1 ;
    }

    if (cpl_propertylist_has(plist, KEY_NAME_COMOFFS_DEC)) {
        result=cpl_propertylist_get_double(plist, KEY_NAME_COMOFFS_DEC);
    } else {
        cpl_msg_error(cpl_func, "keyword %s does not exist",KEY_NAME_COMOFFS_DEC);
        cpl_propertylist_delete(plist) ;
        cpl_free(file);
        return -1;
    }
    cpl_propertylist_delete(plist) ;
    cpl_free(file);
    eris_check_error_code("eris_get_cumoffs_dec");
    return result;

}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the Julian Date of the observation
  @param    frame input frame
  @return   the requested value or 0.0 on error

  Queries FITS header MJD-OBS
 */
/*----------------------------------------------------------------------------*/
double eris_get_mjd_obs(cpl_frame * frame)
{
	cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, -1.0);

    cpl_propertylist* plist=NULL;
    const char* file=NULL;

    double mjd_obs=0.;
    file = cpl_frame_get_filename(frame) ;

    if ((cpl_error_code)((plist = cpl_propertylist_load(file, 0)) == NULL)) {
        cpl_msg_error(cpl_func, "getting header from reference frame %s",file);
        cpl_propertylist_delete(plist) ;
        return -1.0;
    }
    if(cpl_propertylist_has(plist,"MJD-OBS")){
    	mjd_obs = eris_pfits_get_mjdobs(plist);
    } else {
    	cpl_propertylist_delete(plist) ;
    	cpl_error_set(cpl_func,CPL_ERROR_ILLEGAL_INPUT);
    	return -1.0;
    }


    cpl_propertylist_delete(plist) ;
    eris_check_error_code("eris_get_mjd_obs");
    return mjd_obs;

}

/*---------------------------------------------------------------------------*/
/**
	   @brief    check if input frame tag is a static of master calibration
	   @param    tag input tag
	   @return   CPL_TRUE if true, CPL_FALSE otherwise
 */
/*---------------------------------------------------------------------------*/

cpl_boolean eris_ifu_tag_is_cdb(char *tag)
{
    cpl_boolean result = CPL_FALSE;

    if (tag == NULL) {
        return result;
    }

    if ((strstr(ERIS_IFU_PRO_CALIB_TAGS, tag) != NULL) ||
        (strstr(ERIS_IFU_REF_CALIB_TAGS, tag) != NULL))
    {
		result = CPL_TRUE;
	}

	return result;
}

/*---------------------------------------------------------------------------*/
/**
   @brief    check if input frame tag is a static of master calibration
   @param    tag input tag
   @return   CPL_TRUE if true, CPL_FALSE otherwise
 */
/*---------------------------------------------------------------------------*/

cpl_boolean eris_ifu_tag_is_sky(const char *tag)
{
    cpl_boolean result = CPL_FALSE;

    if (tag == NULL) {
        return result;
    }

	char* tag_loc = (char*) tag;
    if ((strcmp(tag_loc, ERIS_IFU_RAW_SKY) == 0)          ||
        (strcmp(tag_loc, ERIS_IFU_RAW_OBJ_SKY) == 0)      ||
        (strcmp(tag_loc, ERIS_IFU_RAW_STD_FLUX_SKY) == 0) ||
        (strcmp(tag_loc, ERIS_IFU_RAW_STD_SKY) == 0)      ||
        (strcmp(tag_loc, ERIS_IFU_RAW_PSF_SKY) == 0)      ||
        (strcmp(tag_loc, ERIS_IFU_RAW_PUPIL_SKY) == 0))
    {
		result = CPL_TRUE;
	}

    return result;
}

/**
   @brief    Add PRO* keywords to a SINFONI FITS header
   @param    tag input tag
   @return   CPL_TRUE if true, CPL_FALSE otherwise
 */
cpl_boolean eris_ifu_tag_is_obj(const char *tag)
{
    cpl_boolean result = CPL_FALSE;

    if (tag == NULL) {
        return result;
    }

    char* tag_loc = (char*) tag;
    if ((strcmp(tag_loc, ERIS_IFU_RAW_OBJ) == 0)      ||
        (strcmp(tag_loc, ERIS_IFU_RAW_STD) == 0)      ||
        (strcmp(tag_loc, ERIS_IFU_RAW_STD_FLUX) == 0) ||
        (strcmp(tag_loc, ERIS_IFU_RAW_PSF) == 0)      ||
        (strcmp(tag_loc, ERIS_IFU_RAW_PUPIL_LAMP) == 0))
    {
		result = CPL_TRUE;
	}

	return result;
}

/**
   @brief    extract object raw frames from frame set and put them in a new one
   @param    sof   input frame set
   @param    obj   output frame set
   @return   CPL error code
 */
cpl_error_code
eris_ifu_extract_obj_frames(const cpl_frameset * sof, cpl_frameset* obj)
{
	cpl_ensure_code(sof, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(obj, CPL_ERROR_NULL_INPUT);

    cpl_size nsof = cpl_frameset_get_size(sof);

    for (cpl_size i = 0 ; i < nsof ; i++) {
        const cpl_frame* frame = cpl_frameset_get_position_const(sof, i);
    	cpl_frame_group group = cpl_frame_get_group(frame);

    	if(group == CPL_FRAME_GROUP_RAW) {

    		//cpl_msg_info(cpl_func,"fname: %s", cpl_frame_get_filename(frame));
    		const char* tag = cpl_frame_get_tag(frame);

    		if(eris_ifu_tag_is_obj(tag)) {

    			cpl_frame* frame_dup = cpl_frame_duplicate(frame);
    			cpl_frameset_insert(obj, frame_dup);

    		}
    	}

    }

    eris_check_error_code("eris_ifu_extract_obj_frames");
    return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
   @brief    extract sky raw frames from frame set and put them in a new one
   @param    sof   input frame set
   @param    sky   output frame set
   @return   CPL error code
 */
/*---------------------------------------------------------------------------*/
cpl_error_code
eris_ifu_extract_sky_frames(const cpl_frameset * sof, cpl_frameset* sky)
{
	cpl_ensure_code(sof, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(sky, CPL_ERROR_NULL_INPUT);

    cpl_size nsof = cpl_frameset_get_size(sof);
    for (cpl_size i = 0 ; i < nsof ; i++) {
        const cpl_frame* frame = cpl_frameset_get_position_const(sof, i);
        cpl_frame_group group = cpl_frame_get_group(frame);

        if(group == CPL_FRAME_GROUP_RAW) {
        	//cpl_msg_info(cpl_func,"fname: %s", cpl_frame_get_filename(frame));
        	const char* tag = cpl_frame_get_tag(frame);
        	if(eris_ifu_tag_is_sky(tag)) {
        		cpl_frame* frame_dup = cpl_frame_duplicate(frame);
        		cpl_frameset_insert(sky, frame_dup);
        	}

        }
    }

    eris_check_error_code("eris_ifu_extract_sky_frames");
    return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
   @brief extract master and ref frames from input set and put them in a new one
   @param    sof   input frame set
   @param    cdb   output frame set
   @return   CPL error code
 */
/*---------------------------------------------------------------------------*/
cpl_error_code
eris_ifu_extract_mst_frames(const cpl_frameset * sof, cpl_frameset* cdb)
{
	cpl_ensure_code(sof, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(cdb, CPL_ERROR_NULL_INPUT);

	cpl_size nsof = cpl_frameset_get_size(sof);
	for (cpl_size i = 0 ; i < nsof ; i++) {
        const cpl_frame* frame = cpl_frameset_get_position_const(sof,i);
		cpl_frame_group group = cpl_frame_get_group(frame);

		if(group == CPL_FRAME_GROUP_CALIB) {
			/* to go on the file must exist */
			if(cpl_frame_get_tag(frame) != NULL) {
				/* If the frame has a tag we process it. Else it is an object */
				char* tag= (char*) cpl_frame_get_tag(frame);
				if(eris_ifu_tag_is_cdb(tag) == 1) {
					cpl_frame* frame_dup = cpl_frame_duplicate(frame);
					cpl_frameset_insert(cdb, frame_dup);
				}
			}
		}
	}

	eris_check_error_code("eris_ifu_extract_mst_frames");
	return cpl_error_get_code();
}


/**@}*/
