/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "eris_ifu_extract_spec_static.h"
#include "eris_ifu_vector.h"
#include "eris_ifu_error.h"
#include "eris_ifu_utils.h"
#include "eris_utils.h"

/**
 * @defgroup eris_ifu_extract_spec_static   IFU Spectrum Extraction
 *
 * This module provides comprehensive functions for extracting 1D spectra from
 * IFU data cubes using various extraction methods.
 *
 * Key functionalities include:
 * - Simple aperture extraction with user-defined or automatic masks
 * - Optimal extraction following Horne (1986) algorithm
 * - Multiple mask generation methods (circular, Gaussian fit, position-based)
 * - Bad pixel handling and quality flag propagation
 * - Variance and error propagation throughout extraction
 * - Adaptive radius estimation for point sources
 * - Weighted extraction with running median smoothing
 * - Cosmic ray rejection and outlier clipping
 *
 * Extraction methods supported:
 * - MASK: Use provided external mask
 * - MAX: Automatic center detection from maximum flux
 * - POSITION: User-specified center position
 * - OPTIMAL: Horne (1986) optimal extraction with PSF modeling
 * - FIT: 2D Gaussian fitting for PSF determination
 *
 * The optimal extraction implements a modified version of Horne (1986):
 * - PSF model constructed from data
 * - Variance taken from input cube (not updated iteratively)
 * - Slice-by-slice outlier clipping
 * - Running weighted average smoothing instead of polynomial fitting
 * - Quality flags incorporated into rejection scheme
 *
 * @note All extraction methods properly propagate errors and handle bad pixels
 * @note Mask convention: 1 = good pixel, 0 = bad pixel (opposite of CPL binary masks)
 */

/*---------------------------------------------------------------------------*/
/**
 * @brief    Collapse cube to median image
 * @param    cube         Input HDRL imagelist (3D cube)
 * @param    contribute   [out] Contribution map indicating number of valid pixels per spatial position
 * @return   Median-collapsed HDRL image, or NULL on error
 *
 * Collapses the input data cube along the wavelength axis using median statistics.
 * This is useful for creating a reference image to detect source positions or
 * estimate PSF parameters.
 *
 * @note The caller is responsible for freeing the returned hdrl_image
 * @note The contribute image shows how many planes contributed to each pixel
 */
/*---------------------------------------------------------------------------*/
hdrl_image* eris_ifu_extract_spec_collapse(hdrl_imagelist *cube,
                                           cpl_image **contribute)
{
    hdrl_image  *hImage     = NULL;

    cpl_ensure(cube != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(contribute != NULL, CPL_ERROR_NULL_INPUT, NULL);

    TRY{
        BRK_IF_ERROR(
            hdrl_imagelist_collapse(cube, HDRL_COLLAPSE_MEDIAN,
                                    &hImage, contribute));
    } CATCH
    {
        eris_ifu_free_hdrl_image(&hImage);
        eris_ifu_free_image(contribute);
    }

    return hImage;
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Free esSofStruct structure and its members
 * @param    self    Pointer to esSofStruct to be freed
 *
 * Properly deallocates all members of the esSofStruct structure including
 * mask, header, quality imagelist, and cube data.
 *
 * @note This function is NULL-safe (can be called with NULL pointer)
 */
/*---------------------------------------------------------------------------*/
void
eris_ifu_extract_free_esSofStruct(struct esSofStruct* self){
    if (self != NULL) {
        eris_ifu_free_image(&self->mask);
        eris_ifu_free_propertylist(&self->header);
        eris_ifu_free_imagelist(&self->qualImagelist);
        eris_ifu_free_hdrl_imagelist(&self->cube);
    }

}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Create extraction mask by fitting 2D Gaussian to image
 * @param    img    Input HDRL image (typically collapsed cube)
 * @return   Normalized Gaussian mask as cpl_image, or NULL on error
 *
 * Fits a 2D Gaussian function to the input image and creates a normalized
 * mask representing the fitted PSF. The mask values range from 0 (background)
 * to 1 (peak), suitable for weighted extraction.
 *
 * Procedure:
 * 1. Fit 2D Gaussian to find center and width parameters
 * 2. Evaluate Gaussian at each pixel position
 * 3. Normalize mask to [0,1] range
 *
 * @note Fitting uses entire image as fitting window
 * @note Initial center guess is image center
 * @note Mask convention: 1 = good (high weight), 0 = bad (zero weight)
 * @note The caller must free the returned cpl_image
 */
/*---------------------------------------------------------------------------*/
cpl_image * eris_ifu_extract_spec_create_fit_mask(const hdrl_image* img)
{
    cpl_image   *mask = NULL;
    const cpl_image   *image = NULL;
    cpl_array   *gauss_params = NULL;
    cpl_size    nx, ny;
    cpl_size    xposcen, yposcen;
    cpl_size    xwinsize, ywinsize;

    cpl_ensure(img, CPL_ERROR_NULL_INPUT, NULL);

    TRY{
        BRK_IF_NULL(image = hdrl_image_get_image_const(img));

        nx = cpl_image_get_size_x(image);
        ny = cpl_image_get_size_y(image);
        xposcen = nx / 2;
        yposcen = ny / 2;
        xwinsize = nx;
        ywinsize = ny;

        BRK_IF_NULL(gauss_params = cpl_array_new(7, CPL_TYPE_DOUBLE));
        BRK_IF_ERROR(cpl_fit_image_gaussian(image, NULL, xposcen, yposcen,
                                            xwinsize, ywinsize, gauss_params,
                                            NULL, NULL, NULL,
                                            NULL, NULL, NULL,
                                            NULL, NULL, NULL));

        mask = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);

        for (cpl_size iy=1; iy <= ny ; iy++) {
            for (cpl_size ix=1; ix <= nx ; ix++) {
                cpl_image_set(mask, ix, iy,
                              cpl_gaussian_eval_2d(gauss_params, (double) ix, (double) iy));
            }
        }
        CHECK_ERROR_STATE();


        /* Normalise mask */
        cpl_image_subtract_scalar(mask, cpl_image_get_min(mask));
        cpl_image_divide_scalar(mask, cpl_image_get_max(mask));
        cpl_array_delete(gauss_params);
        CHECK_ERROR_STATE();

    } CATCH
    {
        mask = NULL;
    }
    return mask;
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Create circular extraction mask
 * @param    center_x    X-coordinate of circle center (1-indexed)
 * @param    center_y    Y-coordinate of circle center (1-indexed)
 * @param    radius      Circle radius in pixels
 * @param    nx          Width of mask to create
 * @param    ny          Height of mask to create
 * @return   Binary mask with circular aperture, or NULL on error
 *
 * Creates a binary mask with value 1.0 inside the specified circular aperture
 * and 0.0 outside. The circle is clipped at image boundaries if necessary.
 *
 * @note Mask convention: 1.0 = inside aperture (good), 0.0 = outside (bad)
 * @note Center coordinates use 1-based indexing (CPL convention)
 * @note The caller must free the returned cpl_image
 */
/*---------------------------------------------------------------------------*/
cpl_image * eris_ifu_extract_spec_create_circle_mask(
                                    cpl_size center_x,
                                    cpl_size center_y,
                                    double radius,
                                    cpl_size nx,
                                    cpl_size ny)
{
    cpl_image   *mask = NULL;
    double      cen_x, cen_y, x_lo, y_lo, x_hi, y_hi, r;
    double      *pmask;

    cpl_ensure(center_x > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(center_y > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(radius > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(nx > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(ny > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    TRY{
        mask = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        pmask = cpl_image_get_data_double(mask);

        cen_x = (double) center_x - 1.0;
        cen_y = (double) center_y - 1.0;
        // draw circle
        x_lo = floor(cen_x - radius);
        if (x_lo < 0) x_lo = 0;
        y_lo = floor(cen_y - radius);
        if (y_lo < 0) y_lo = 0;
        x_hi = ceil(cen_x + radius);
        if (x_hi > nx) x_hi = (int) nx;
        y_hi = ceil(cen_y + radius);
        if (y_hi > ny) y_hi = (int) ny;
        for (int x = (int) x_lo; x < x_hi; x++) {
            for (int y = (int) y_lo; y < y_hi; y++) {
                r = sqrt(pow(x - cen_x,2) + pow(y - cen_y,2));
                if (r <= radius) pmask[x + y * nx] = 1.0;
            }
        }

    } CATCH
    {
        mask = NULL;
    }
    return mask;
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Create extraction mask using specified method
 * @param    params          Parameter structure containing mask method and settings
 * @param    sof             SOF structure containing input data and optional mask
 * @param    collapsedCube   Collapsed cube image for automatic center detection
 * @param    productDepth    Debug output level
 * @return   Extraction mask, or NULL on error
 *
 * Creates an extraction mask using one of several methods:
 * - MASK: Use externally provided mask from SOF
 * - MAX: Find maximum flux position and create circular mask
 * - POSITION: Use user-specified center with circular mask
 * - OPTIMAL: Estimate optimal radius from FWHM and create circular mask
 * - FIT: Fit 2D Gaussian to create weighted mask
 *
 * @note Mask convention: 1 = good pixel (to extract), 0 = bad pixel (to ignore)
 *       This is OPPOSITE to CPL binary mask convention!
 * @note For MASK method, the input mask from SOF is used directly
 * @note The caller must free the returned cpl_image (except for MASK method)
 * @note Mask dimensions must match cube dimensions
 */
/*---------------------------------------------------------------------------*/
cpl_image * eris_ifu_extract_spec_create_mask(
                                    struct esParamStruct params,
                                    struct esSofStruct sof,
                                    const hdrl_image *collapsedCube,
                                    int productDepth)
{
    cpl_image   *mask = NULL;
    cpl_size    nx;
    cpl_size    ny;
    cpl_size    center_x;
    cpl_size    center_y;
    const cpl_image *img;

    TRY{
        switch (params.mask_method) {
        case MASK:
            mask = sof.mask;
            break;
        case MAX:
            cpl_image_get_maxpos(
                hdrl_image_get_image_const(collapsedCube), &center_x, &center_y);
            CHECK_ERROR_STATE();
            BRK_IF_NULL(
                mask = eris_ifu_extract_spec_create_circle_mask(center_x, center_y,
                                                                params.radius,
                                                                sof.nx, sof.ny));
            break;
        case POSITION:
            BRK_IF_NULL(
                    mask = eris_ifu_extract_spec_create_circle_mask(params.center_x,
                                                                params.center_y,
                                                                params.radius,
                                                                sof.nx, sof.ny));
            break;
        case OPTIMAL:
            // get maxpos
            img = hdrl_image_get_image_const(collapsedCube);
            cpl_image_get_maxpos(img, &center_x, &center_y);
            CHECK_ERROR_STATE();

            // estimate radius by FWHM
            double radius = eris_ifu_opt_extr_estimate_radius(img,
                                                              center_x, center_y,
                                                              params.radius,
                                                              productDepth);

            // create mask
            BRK_IF_NULL(
                mask = eris_ifu_extract_spec_create_circle_mask(center_x, center_y,
                                                                radius,
                                                                sof.nx, sof.ny));
            break;
        case FIT:
            BRK_IF_NULL(
                mask = eris_ifu_extract_spec_create_fit_mask(collapsedCube));
            break;
        default:
            BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
                               "Internal error: unknown mask method %d",
                               params.mask_method);
        }

        nx = cpl_image_get_size_x(mask);
        ny = cpl_image_get_size_y(mask);
        if (nx != sof.nx || ny != sof.ny) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
                               "Dimensions of mask don't match with the cube");
        }
    } CATCH
    {
        mask = NULL;
    }
    return mask;
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Extract 1D spectrum from data cube using weighted sum
 * @param    cube            Input HDRL imagelist (data cube with errors)
 * @param    mask            Extraction mask (1=extract, 0=ignore)
 * @param    startLambda     Starting wavelength in micrometers
 * @param    deltaLambda     Wavelength step in micrometers
 * @param    error           [out] Error vector for extracted spectrum
 * @param    totalFlux       [out] Total integrated flux (unweighted) per wavelength
 * @param    quality         [out] Quality flag vector (0=good)
 * @return   Bivector containing wavelength and flux, or NULL on error
 *
 * Performs simple weighted extraction by summing flux in each wavelength plane
 * within the specified mask. For each wavelength slice:
 * 1. Multiply each pixel by its mask weight
 * 2. Sum weighted flux values
 * 3. Divide by sum of weights to get average flux
 * 4. Propagate errors using standard error propagation
 *
 * The algorithm properly handles bad pixels (marked in BPM) and NaN/Inf values,
 * excluding them from the extraction. Wavelength planes with insufficient valid
 * pixels are omitted from the output.
 *
 * @note Mask can be NULL, in which case uniform weighting (=1) is used
 * @note Bad pixels (BPM) and NaN/Inf values are automatically excluded
 * @note Output vectors are resized to contain only valid wavelength points
 * @note Errors are propagated as sqrt(sum of variance) / weights
 * @note The caller must free the returned bivector and output vectors
 */
/*---------------------------------------------------------------------------*/
cpl_bivector * eris_ifu_extract_spectrum(hdrl_imagelist *cube,
                                         cpl_image *mask,
                                         double startLambda,
                                         double deltaLambda,
                                         cpl_vector **error,
                                         cpl_vector **totalFlux,
										 cpl_vector **quality)
{
    cpl_bivector        *spectrum = NULL;
    cpl_imagelist *data_in = NULL;
    cpl_imagelist *error_in = NULL;
    const cpl_image     *tmpDataImg = NULL;
    const cpl_image     *tmpErrorImg = NULL;
    const cpl_mask      *bpm = NULL;
    const double        *pTmpData = NULL;
    const double        *pTmpError = NULL;
    const cpl_binary    *pBpm = NULL;
    const double        *pMask = NULL;
    cpl_size            nx, ny, nz;
    cpl_size            fill_nz;
    cpl_vector          *lambda = NULL;
    cpl_vector          *spec_data = NULL;
    cpl_vector          *spec_error = NULL;
    cpl_vector          *spec_totFlux = NULL;
    cpl_vector          *spec_qual = NULL;
    double              *pLambda = NULL;
    double              *pData = NULL;
    double              *pError = NULL;
    double              *pTotFlux = NULL;
    double              *pqual = NULL;
    double              sumData;
    double              sumVariance;
    double              weights;
    bool                valid;


    cpl_ensure(cube != NULL,CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(mask != NULL,CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(error != NULL,CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(totalFlux != NULL,CPL_ERROR_NULL_INPUT, NULL);

    TRY{
        nx = hdrl_imagelist_get_size_x(cube);
        ny = hdrl_imagelist_get_size_y(cube);
        nz = hdrl_imagelist_get_size(cube);

        ASSURE((nx == cpl_image_get_size_x(mask)) &&
                   (ny == cpl_image_get_size_y(mask)),
               CPL_ERROR_ILLEGAL_INPUT,
               "Data cube and mask don't have same dimensions!");


        BRK_IF_NULL(data_in = cpl_imagelist_new());
        BRK_IF_NULL(error_in = cpl_imagelist_new());
        for (cpl_size sz = 0; sz < nz; sz++) {
            hdrl_image *tmpImg = hdrl_imagelist_get(cube, sz);
            cpl_imagelist_set(data_in, cpl_image_duplicate(hdrl_image_get_image(tmpImg)), sz);
            cpl_imagelist_set(error_in, cpl_image_duplicate(hdrl_image_get_error(tmpImg)), sz);
        }
        CHECK_ERROR_STATE();

        lambda = cpl_vector_new(nz);
        spec_data = cpl_vector_new(nz);
        spec_error = cpl_vector_new(nz);
        spec_totFlux = cpl_vector_new(nz);
        spec_qual = cpl_vector_new(nz);
        pLambda = cpl_vector_get_data(lambda);
        pData = cpl_vector_get_data(spec_data);
        pError = cpl_vector_get_data(spec_error);
        pTotFlux = cpl_vector_get_data(spec_totFlux);
        pqual = cpl_vector_get_data(spec_qual);
        pMask = cpl_image_get_data_double_const(mask);
        CHECK_ERROR_STATE();

        // loop over all lambda slices
        fill_nz = 0;
        for (cpl_size sz = 0; sz < nz; sz++) {
            tmpDataImg = cpl_imagelist_get(data_in, sz);
            tmpErrorImg = cpl_imagelist_get(error_in, sz);
            bpm = cpl_image_get_bpm_const(tmpDataImg);
            pTmpData = cpl_image_get_data_double_const(tmpDataImg);
            pTmpError = cpl_image_get_data_double_const(tmpErrorImg);
            pBpm = cpl_mask_get_data_const(bpm);
            CHECK_ERROR_STATE();

            // extract spectrum for data
            sumData = 0.0;
            sumVariance = 0.0;
            weights = 0.0;
            valid = false;
            for (cpl_size j = 0; j < ny; j++) {
                for (cpl_size i = 0; i < nx; i++) {
                    // sumData weighted pixels in spatial plane
                    cpl_size p = i+j*nx;
                    if ((pBpm[p] == GOOD_PIX) && !eris_ifu_is_nan_or_inf(pTmpData[p])) {
                        valid = true;
                        if (mask != NULL) {
                            sumData += pTmpData[p] * pMask[p];
                            sumVariance += pow(pTmpError[p] * pMask[p],2);
                            weights += pMask[p];
                            /* AMo: temporarily added for debug to make some check on problematic data
                            if(startLambda + (double) sz * deltaLambda > 2.254 &&
                               startLambda + (double) sz * deltaLambda < 2.270	) {
                            cpl_msg_info(cpl_func,"ok1: pMask[p]: %g",pMask[p]);
                            }
                            */
                        } else {
                            sumData += pTmpData[p];
                            sumVariance += pow(pTmpError[p],2);
                            weights += 1.;
                        }
                    }
                }
            }

            if ((valid == true) && (fabs(weights) > DBL_ZERO_TOLERANCE)) {
                pLambda[fill_nz] = startLambda + (double) sz * deltaLambda;
                pData[fill_nz] = sumData / weights;
                pError[fill_nz] = sqrt(sumVariance) / weights;
                pTotFlux[fill_nz] = sumData;
                pqual[fill_nz] = 0;
                fill_nz++;
            }
            CHECK_ERROR_STATE();
        }
        BRK_IF_ERROR(cpl_vector_set_size(lambda, fill_nz));
        BRK_IF_ERROR(cpl_vector_set_size(spec_data, fill_nz));
        BRK_IF_ERROR(cpl_vector_set_size(spec_error, fill_nz));
        BRK_IF_ERROR(cpl_vector_set_size(spec_totFlux, fill_nz));
        BRK_IF_ERROR(cpl_vector_set_size(spec_qual, fill_nz));

        spectrum = cpl_bivector_wrap_vectors(lambda, spec_data);
        *error = spec_error;
        *totalFlux = spec_totFlux;
        *quality = spec_qual;
        cpl_imagelist_delete(data_in);
        cpl_imagelist_delete(error_in);
    } CATCH {
        spectrum = NULL;
    }
    eris_check_error_code("eris_ifu_extract_spectrum");
    return spectrum;
}





//cpl_bivector * eris_ifu_extract_spectrum2(
//        hdrl_imagelist *cube,
//        cpl_image *mask,
//        double startLambda,
//        double deltaLambda,
//        cpl_vector **error,
//        cpl_vector **totalFlux)
//{
//    cpl_bivector        *spectrum = NULL;
//    cpl_imagelist *data_in = NULL;
//    cpl_imagelist *error_in = NULL;
//    const cpl_image     *tmpDataImg = NULL;
//    const cpl_image     *tmpErrorImg = NULL;
//    const cpl_mask      *bpm = NULL;
//    const double        *pTmpData = NULL;
//    const double        *pTmpError = NULL;
//    const cpl_binary    *pBpm = NULL;
//    const double        *pMask = NULL;
//    cpl_size            nx, ny, nz;
//    cpl_size            fill_nz;
//    cpl_vector          *lambda = NULL;
//    cpl_vector          *spec_data = NULL;
//    cpl_vector          *spec_error = NULL;
//    cpl_vector          *spec_totFlux = NULL;
//    double              *pLambda = NULL;;
//    double              *pData = NULL;;
//    double              *pError = NULL;;
//    double              *pTotFlux = NULL;;
//    double              sumData;
//    double              sumVariance;
//    double              weights;
//    bool                valid;

//    hdrl_image*         cube_median = NULL;
//    cpl_image*          cube_median_data = NULL;
//    cpl_image*          cube_median_errs = NULL;
//    cpl_image*          scale_factor_data = NULL;
//    cpl_image*          scale_factor_errs = NULL;
//    const double        *pFactorData = NULL;
////    const double        *pFactorErrs = NULL;
//    const double        *pModelData = NULL;
//    const double        *pModelErrs = NULL;
////    double              model_flux_sum = 0;
////    double              model_errs_sum = 0;
//    double              model_factor_avg = 0;

//    cpl_size            model_ndata = 0;
//    const cpl_size      kernel_sx = 5;
//    const cpl_size      kernel_sy = 5;
//    cpl_image* contrib_map;

//    cpl_ensure(cube != NULL,CPL_ERROR_NULL_INPUT, NULL);
//    cpl_ensure(mask != NULL,CPL_ERROR_NULL_INPUT, NULL);
//    cpl_ensure(error != NULL,CPL_ERROR_NULL_INPUT, NULL);

//    TRY{
//        nx = hdrl_imagelist_get_size_x(cube);
//        ny = hdrl_imagelist_get_size_y(cube);
//        nz = hdrl_imagelist_get_size(cube);

//        ASSURE((nx == cpl_image_get_size_x(mask)) &&
//               (ny == cpl_image_get_size_y(mask)),
//                        CPL_ERROR_ILLEGAL_INPUT,
//                         "Data cube and mask don't have same dimensions!");
//        eris_print_rec_status(0);
//        hdrl_imagelist_collapse_median(cube, &cube_median, &contrib_map);
//        eris_print_rec_status(1);
//        cube_median_data = hdrl_image_get_image(cube_median);
//        cube_median_errs = hdrl_image_get_error(cube_median);
//        cpl_image_save(cube_median_data,"cube_median_data.fits",CPL_TYPE_DOUBLE,NULL,CPL_IO_DEFAULT);
//        eris_print_rec_status(2);
//        data_in = cpl_imagelist_new();
//        error_in = cpl_imagelist_new();
//        for (cpl_size sz = 0; sz < nz; sz++) {
//            hdrl_image *tmpImg = hdrl_imagelist_get(cube, sz);
//            cpl_imagelist_set(data_in, cpl_image_duplicate(hdrl_image_get_image(tmpImg)), sz);
//            cpl_imagelist_set(error_in, cpl_image_duplicate(hdrl_image_get_error(tmpImg)), sz);
//        }
//        CHECK_ERROR_STATE();

//        lambda = cpl_vector_new(nz);
//        spec_data = cpl_vector_new(nz);
//        spec_error = cpl_vector_new(nz);
//        spec_totFlux = cpl_vector_new(nz);
//        pLambda = cpl_vector_get_data(lambda);
//        pData = cpl_vector_get_data(spec_data);
//        pError = cpl_vector_get_data(spec_error);
//        pTotFlux = cpl_vector_get_data(spec_totFlux);
//        pMask = cpl_image_get_data_double_const(mask);
//        CHECK_ERROR_STATE();

//        // loop over all lambda slices
//        fill_nz = 0;
//        for (cpl_size sz = 0; sz < nz; sz++) {
//            tmpDataImg = cpl_imagelist_get(data_in, sz);
//            tmpErrorImg = cpl_imagelist_get(error_in, sz);
//            bpm = cpl_image_get_bpm_const(tmpDataImg);
//            pTmpData = cpl_image_get_data_double_const(tmpDataImg);
//            pTmpError = cpl_image_get_data_double_const(tmpErrorImg);
//            pBpm = cpl_mask_get_data_const(bpm);
//            eris_print_rec_status(0);
//            scale_factor_data = cpl_image_duplicate(tmpDataImg);
//            scale_factor_errs = cpl_image_duplicate(tmpErrorImg);
//            eris_print_rec_status(1);
//            cpl_image_power(scale_factor_errs, 2);
//            cpl_image_power(scale_factor_errs, 2);
//            eris_print_rec_status(2);
//            cpl_image_divide(scale_factor_data, cube_median_data);
//            cpl_image_divide(scale_factor_errs, cube_median_errs);
//            eris_print_rec_status(3);
//            pFactorData = cpl_image_get_data_double_const(scale_factor_data);
////            pFactorErrs = cpl_image_get_data_double_const(scale_factor_errs);
//            eris_print_rec_status(4);
//            pModelData = cpl_image_get_data_double_const(cube_median_data);
//            pModelErrs = cpl_image_get_data_double_const(cube_median_errs);
//            eris_print_rec_status(5);
//            CHECK_ERROR_STATE();

//            // extract spectrum for data
//            sumData = 0.0;
//            sumVariance = 0.0;
//            weights = 0.0;
//            valid = false;
//            for (cpl_size j = 0; j < ny; j++) {
//                for (cpl_size i = 0; i < nx; i++) {
//                    // sumData weighted pixels in spatial plane
//                    cpl_size p = i+j*nx;
////                    model_flux_sum = 0;
////                    model_errs_sum = 0;
//                    model_ndata = 0;
//                    if ((pBpm[p] == GOOD_PIX) && !eris_ifu_is_nan_or_inf(pTmpData[p])) {
//                        valid = true;
//                        if (mask != NULL) {
//                            sumData += pTmpData[p] * pMask[p];
//                            sumVariance += pow(pTmpError[p] * pMask[p],2);
//                            weights += pMask[p];
//                        } else {
//                            sumData += pTmpData[p];
//                            sumVariance += pow(pTmpError[p],2);
//                            weights += 1.;
//                        }
//                    } else {

//                    	/* we are on a bad pixel, we need to interpolate flux
//                    	 * using the model */
//                        /* Determine average factor: plane_data / model */
//                    	for (cpl_size jj = -kernel_sy; jj <= kernel_sy; jj++) {
//                    		for (cpl_size ii = -kernel_sx; ii <= kernel_sx; ii++) {
//                    			if( (i+ii) >= 0 && (i+ii) < nx &&
//                    					(j+jj) >= 0 && (j+jj) < ny) {

//                    				cpl_size pp = (i+ii)+(j+jj)*nx;
//                                    if ((pBpm[pp] == GOOD_PIX) && !eris_ifu_is_nan_or_inf(pTmpData[pp])) {
//                    					model_factor_avg += pFactorData[pp];
//                    					model_ndata ++;
//                    				}
//                    			}
//                    		}
//                    	}
//                    	if(model_ndata>0) {
//                    		model_factor_avg /= model_ndata;
//                    	}

//                    	/* compute contribute from interpolated bad pixels
//                    	 * scaling data from model and add it to flux determined
//                    	 * on good data
//                    	 */
//                    	if (!eris_ifu_is_nan_or_inf(pModelData[p])) {
//                    		valid = true;
//                    		if (mask != NULL) {
//                    			sumData += model_factor_avg * pModelData[p] * pMask[p];
//                    			sumVariance += pow(model_factor_avg * pModelErrs[p] * pMask[p],2);
//                    			weights += pMask[p];
//                    			if(startLambda + (double) sz * deltaLambda > 1.615 &&
//                    					startLambda + (double) sz * deltaLambda < 1.647	) {
//                    				cpl_msg_info(cpl_func,"lambda: %g model: %g",pLambda[fill_nz], pModelData[p]);
//                    			}
//                    		} else {
//                    			sumData += model_factor_avg * pModelData[p];
//                    			sumVariance += pow(model_factor_avg * pModelErrs[p],2);
//                    			weights += 1.;
//                    		}
//                    	}

//                    }
//                }
//            }

//            if ((valid == true) && (fabs(weights) > DBL_ZERO_TOLERANCE)) {
//                pLambda[fill_nz] = startLambda + (double) sz * deltaLambda;
//                pData[fill_nz] = sumData / weights;
//                pError[fill_nz] = sqrt(sumVariance) / weights;
//                pTotFlux[fill_nz] = sumData;
//                if(startLambda + (double) sz * deltaLambda > 1.615 &&
//                               		startLambda + (double) sz * deltaLambda < 1.647	) {
//                               	cpl_msg_info(cpl_func,"lambda: %g flux: %g",pLambda[fill_nz], pData[fill_nz]);
//                               }
//                fill_nz++;

//            }
//            cpl_image_delete(scale_factor_data);
//            cpl_image_delete(scale_factor_errs);
//            CHECK_ERROR_STATE();
//        }
//        BRK_IF_ERROR(cpl_vector_set_size(lambda, fill_nz));
//        BRK_IF_ERROR(cpl_vector_set_size(spec_data, fill_nz));
//        BRK_IF_ERROR(cpl_vector_set_size(spec_error, fill_nz));
//        BRK_IF_ERROR(cpl_vector_set_size(spec_totFlux, fill_nz));

//        spectrum = cpl_bivector_wrap_vectors(lambda, spec_data);
//        *error = spec_error;
//        *totalFlux = spec_totFlux;
//        cpl_imagelist_delete(data_in);
//        cpl_imagelist_delete(error_in);
//        hdrl_image_delete(cube_median);
//        cpl_image_delete(contrib_map);
//    } CATCH
//    {
//        spectrum = NULL;
//    }
//    eris_check_error_code("eris_ifu_extract_spectrum2");
//    return spectrum;
//}



//cpl_bivector * eris_ifu_extract_spectrum3(
//        hdrl_imagelist *cube,
//        cpl_image *mask,
//        double startLambda,
//        double deltaLambda,
//        cpl_vector **error,
//        cpl_vector **totalFlux)
//{
//    cpl_bivector        *spectrum = NULL;
//    cpl_imagelist *data_in = NULL;
//    cpl_imagelist *error_in = NULL;
//    const cpl_image     *tmpDataImg = NULL;
//    const cpl_image     *tmpErrorImg = NULL;
//    const cpl_mask      *bpm = NULL;
//    const double        *pTmpData = NULL;
//    const double        *pTmpError = NULL;
//    const cpl_binary    *pBpm = NULL;
//    const double        *pMask = NULL;
//    cpl_size            nx, ny, nz;
//    cpl_size            fill_nz;
//    cpl_vector          *lambda = NULL;
//    cpl_vector          *spec_data = NULL;
//    cpl_vector          *spec_error = NULL;
//    cpl_vector          *spec_totFlux = NULL;
//    double              *pLambda = NULL;;
//    double              *pData = NULL;;
//    double              *pError = NULL;;
//    double              *pTotFlux = NULL;;
//    double              sumData;
//    double              sumVariance;
//    double              weights;
//    bool                valid;

//    cpl_imagelist*      cube_data_tmp = NULL;
//    cpl_imagelist*      cube_errs_tmp = NULL;
//    hdrl_image*         cube_median = NULL;
//    cpl_image*          cube_median_data = NULL;
//    cpl_image*          cube_median_errs = NULL;
//    cpl_image*          scale_factor_data = NULL;
//    cpl_image*          scale_factor_errs = NULL;
//    const double        *pFactorData = NULL;
////    const double        *pFactorErrs = NULL;
//    const double        *pModelData = NULL;
//    const double        *pModelErrs = NULL;
////    double              model_flux_sum = 0;
////    double              model_errs_sum = 0;
//    double              model_factor_avg = 0;

//    cpl_size            model_def_hsize = 100;
//    cpl_size            model_ndata = 0;
//    const cpl_size      kernel_sx = 5;
//    const cpl_size      kernel_sy = 5;
//    cpl_image* contrib_map;

//    cpl_ensure(cube != NULL,CPL_ERROR_NULL_INPUT, NULL);
//    cpl_ensure(mask != NULL,CPL_ERROR_NULL_INPUT, NULL);
//    cpl_ensure(error != NULL,CPL_ERROR_NULL_INPUT, NULL);

//    TRY {
//    	eris_print_rec_status(0);
//        nx = hdrl_imagelist_get_size_x(cube);
//        ny = hdrl_imagelist_get_size_y(cube);
//        nz = hdrl_imagelist_get_size(cube);
//        eris_print_rec_status(1);
//        ASSURE((nx == cpl_image_get_size_x(mask)) &&
//               (ny == cpl_image_get_size_y(mask)),
//                        CPL_ERROR_ILLEGAL_INPUT,
//                         "Data cube and mask don't have same dimensions!");

//        hdrl_imagelist_collapse_median(cube, &cube_median, &contrib_map);
//        eris_print_rec_status(2);
//        cube_median_data = hdrl_image_get_image(cube_median);
//        cube_median_errs = hdrl_image_get_error(cube_median);
//        cpl_image_save(cube_median_data,"cube_median_data.fits",CPL_TYPE_DOUBLE,NULL,CPL_IO_DEFAULT);
//        eris_print_rec_status(3);
//        data_in = cpl_imagelist_new();
//        error_in = cpl_imagelist_new();
//        for (cpl_size sz = 0; sz < nz; sz++) {
//            hdrl_image *tmpImg = hdrl_imagelist_get(cube, sz);
//            cpl_imagelist_set(data_in, cpl_image_duplicate(hdrl_image_get_image(tmpImg)), sz);
//            cpl_imagelist_set(error_in, cpl_image_duplicate(hdrl_image_get_error(tmpImg)), sz);
//        }
//        eris_print_rec_status(4);
//        CHECK_ERROR_STATE();

//        lambda = cpl_vector_new(nz);
//        spec_data = cpl_vector_new(nz);
//        spec_error = cpl_vector_new(nz);
//        spec_totFlux = cpl_vector_new(nz);
//        pLambda = cpl_vector_get_data(lambda);
//        pData = cpl_vector_get_data(spec_data);
//        pError = cpl_vector_get_data(spec_error);
//        pTotFlux = cpl_vector_get_data(spec_totFlux);
//        pMask = cpl_image_get_data_double_const(mask);
//        eris_print_rec_status(5);
//        CHECK_ERROR_STATE();

//        // loop over all lambda slices
//        fill_nz = 0;
//        for (cpl_size sz = 0; sz < nz; sz++) {
//        	eris_print_rec_status(6);
//            tmpDataImg = cpl_imagelist_get(data_in, sz);
//            tmpErrorImg = cpl_imagelist_get(error_in, sz);
//            bpm = cpl_image_get_bpm_const(tmpDataImg);
//            pTmpData = cpl_image_get_data_double_const(tmpDataImg);
//            pTmpError = cpl_image_get_data_double_const(tmpErrorImg);
//            pBpm = cpl_mask_get_data_const(bpm);
//            eris_print_rec_status(7);

//            /* determine local model on a slice of half size model_def_hsize */
//            if( sz >= model_def_hsize && sz < (nz-model_def_hsize)) {
//            	cube_data_tmp = cpl_imagelist_new();
//            	cube_errs_tmp = cpl_imagelist_new();
//            	for(cpl_size k = 0; k <= 2*model_def_hsize; k++) {
//            		cpl_size kk = sz - model_def_hsize + k;
//            		cpl_imagelist_set(cube_data_tmp, cpl_imagelist_get(data_in, kk), k);
//            		cpl_imagelist_set(cube_errs_tmp, cpl_imagelist_get(error_in, kk), k);
//            	}

//            	cube_median_data = cpl_imagelist_collapse_median_create(cube_data_tmp);
//            	cube_median_errs = cpl_imagelist_collapse_median_create(cube_errs_tmp);

//            }
//            eris_print_rec_status(8);
//            scale_factor_data = cpl_image_duplicate(tmpDataImg);
//            scale_factor_errs = cpl_image_duplicate(tmpErrorImg);

//            cpl_image_power(scale_factor_errs, 2);
//            cpl_image_power(cube_median_errs, 2);
//            eris_print_rec_status(9);
//            if(cube_median_data != NULL) {
//            	cpl_msg_warning(cpl_func,"Median data for sz: %lld", sz);
//            	cpl_image_divide(scale_factor_data, cube_median_data);
//            } else {
//            	cpl_msg_warning(cpl_func,"Median data for sz: %lld", sz);
//            }
//            if(cube_median_data != NULL) {
//            	cpl_image_divide(scale_factor_errs, cube_median_errs);
//            } else {
//            	cpl_msg_warning(cpl_func,"Median error for sz: %lld", sz);
//            }
//            eris_print_rec_status(10);
//            pFactorData = cpl_image_get_data_double_const(scale_factor_data);
////            pFactorErrs = cpl_image_get_data_double_const(scale_factor_errs);

//            pModelData = cpl_image_get_data_double_const(cube_median_data);
//            pModelErrs = cpl_image_get_data_double_const(cube_median_errs);
//            eris_print_rec_status(11);
//            CHECK_ERROR_STATE();

//            // extract spectrum for data
//            sumData = 0.0;
//            sumVariance = 0.0;
//            weights = 0.0;
//            valid = false;
//            eris_print_rec_status(12);
//            for (cpl_size j = 0; j < ny; j++) {
//                for (cpl_size i = 0; i < nx; i++) {
//                    // sumData weighted pixels in spatial plane
//                    cpl_size p = i+j*nx;
////                    model_flux_sum = 0;
////                    model_errs_sum = 0;
//                    model_ndata = 0;
//                    if ((pBpm[p] == GOOD_PIX) && !eris_ifu_is_nan_or_inf(pTmpData[p])) {
//                        valid = true;
//                        if (mask != NULL) {
//                            sumData += pTmpData[p] * pMask[p];
//                            sumVariance += pow(pTmpError[p] * pMask[p],2);
//                            weights += pMask[p];
//                        } else {
//                            sumData += pTmpData[p];
//                            sumVariance += pow(pTmpError[p],2);
//                            weights += 1.;
//                        }
//                    } else {

//                    	/* we are on a bad pixel, we need to interpolate flux
//                    	 * using the model */
//                        /* Determine average factor: plane_data / model */
//                    	for (cpl_size jj = -kernel_sy; jj <= kernel_sy; jj++) {
//                    		for (cpl_size ii = -kernel_sx; ii <= kernel_sx; ii++) {
//                    			if( (i+ii) >= 0 && (i+ii) < nx &&
//                    					(j+jj) >= 0 && (j+jj) < ny) {

//                    				cpl_size pp = (i+ii)+(j+jj)*nx;
//                                    if ((pBpm[pp] == GOOD_PIX) && !eris_ifu_is_nan_or_inf(pTmpData[pp])) {
//                    					model_factor_avg += pFactorData[pp];
//                    					model_ndata ++;
//                    				}
//                    			}
//                    		}
//                    	}
//                    	if(model_ndata>0) {
//                    		model_factor_avg /= model_ndata;
//                    	}

//                    	/* compute contribute from interpolated bad pixels
//                    	 * scaling data from model and add it to flux determined
//                    	 * on good data
//                    	 */
//                    	if (!eris_ifu_is_nan_or_inf(pModelData[p])) {
//                    		valid = true;
//                    		if (mask != NULL) {
//                    			sumData += model_factor_avg * pModelData[p] * pMask[p];
//                    			sumVariance += pow(model_factor_avg * pModelErrs[p] * pMask[p],2);
//                    			weights += pMask[p];
//                    			/*
//                    			if(startLambda + (double) sz * deltaLambda > 1.615 &&
//                    					startLambda + (double) sz * deltaLambda < 1.647	) {
//                    				cpl_msg_info(cpl_func,"lambda: %g model: %g",pLambda[fill_nz], pModelData[p]);
//                    			}
//                    			*/
//                    		} else {
//                    			sumData += model_factor_avg * pModelData[p];
//                    			sumVariance += pow(model_factor_avg * pModelErrs[p],2);
//                    			weights += 1.;
//                    		}
//                    	}

//                    }
//                }
//            }
//            eris_print_rec_status(13);
//            if ((valid == true) && (fabs(weights) > DBL_ZERO_TOLERANCE)) {
//                pLambda[fill_nz] = startLambda + (double) sz * deltaLambda;
//                pData[fill_nz] = sumData / weights;
//                pError[fill_nz] = sqrt(sumVariance) / weights;
//                pTotFlux[fill_nz] = sumData;
//                /*
//                if(startLambda + (double) sz * deltaLambda > 1.615 &&
//                               		startLambda + (double) sz * deltaLambda < 1.647	) {
//                               	cpl_msg_info(cpl_func,"lambda: %g flux: %g",pLambda[fill_nz], pData[fill_nz]);
//                               }
//                               */
//                fill_nz++;

//            }
//            eris_print_rec_status(14);
//            cpl_image_delete(scale_factor_data);
//            cpl_image_delete(scale_factor_errs);
//            if( sz >= model_def_hsize && sz < (nz-model_def_hsize)) {
//            	cpl_image_delete(cube_median_data);
//            	cpl_image_delete(cube_median_errs);
//            	for(cpl_size k = 2*model_def_hsize; k >= 0; k--) {
//            		cpl_imagelist_unset(cube_data_tmp,k);
//            		cpl_imagelist_unset(cube_errs_tmp,k);
//            	}
//            	cpl_imagelist_delete(cube_data_tmp);
//            	cpl_imagelist_delete(cube_errs_tmp);
//            }
//            eris_print_rec_status(15);
//            CHECK_ERROR_STATE();
//        }
//        eris_print_rec_status(16);
//        BRK_IF_ERROR(cpl_vector_set_size(lambda, fill_nz));
//        BRK_IF_ERROR(cpl_vector_set_size(spec_data, fill_nz));
//        BRK_IF_ERROR(cpl_vector_set_size(spec_error, fill_nz));
//        BRK_IF_ERROR(cpl_vector_set_size(spec_totFlux, fill_nz));

//        spectrum = cpl_bivector_wrap_vectors(lambda, spec_data);
//        *error = spec_error;
//        *totalFlux = spec_totFlux;
//        cpl_imagelist_delete(data_in);
//        cpl_imagelist_delete(error_in);
//        hdrl_image_delete(cube_median);
//        cpl_image_delete(contrib_map);
//        eris_print_rec_status(17);
//    } CATCH
//    {
//        spectrum = NULL;
//    }
//    eris_check_error_code("eris_ifu_extract_spectrum3");
//    return spectrum;
//}

/**
 * @brief    Perform optimal extraction on IFU cube
 * @param    cube            Input HDRL imagelist (data cube with errors)
 * @param    cube_dqi        Data quality imagelist (0=good, >0=bad)
 * @param    img_mask        Extraction mask image (1=extract, 0=ignore)
 * @param    startLambda     Starting wavelength in micrometers
 * @param    deltaLambda     Wavelength step in micrometers
 * @param    productDepth    Debug output level (PD_DEBUG for diagnostic files)
 * @param    error_out       [out] Error vector for extracted spectrum
 * @return   Bivector containing wavelength and optimally extracted flux, or NULL on error
 *
 * Implements a modified version of the Horne (1986) optimal extraction algorithm
 * for point sources in IFU data. The method models the PSF from the data itself
 * and uses it to optimally weight pixels during extraction.
 *
 * Algorithm overview (based on Horne 1986, PASP 98, 609):
 * 1. Initial simple extraction to estimate spectrum
 * 2. Construct PSF model by normalizing each slice by the spectrum
 * 3. Smooth PSF model using running weighted average (not polynomial)
 * 4. Detect and reject outliers using slice-by-slice clipping
 * 5. Recompute spectrum using PSF-weighted extraction
 * 6. Iterate once to improve PSF model
 * 7. Final extraction with updated weights
 *
 * Key modifications from standard Horne (1986):
 * - Variance taken from input cube (not updated iteratively)
 * - Running weighted median smoothing instead of polynomial fitting
 * - Slice-by-slice outlier clipping with adaptive threshold
 * - Quality flags (DQI) incorporated into rejection
 * - Two-iteration scheme (bigloop) for PSF refinement
 *
 * @note This method is designed for point sources (stars, QSOs)
 * @note Significant improvement in S/N compared to simple extraction
 * @note Requires good initial estimate of extraction aperture
 * @note img_mask convention: 1 = good (extract), 0 = bad (ignore)
 * @note The caller must free the returned bivector and error_out vector
 */
/*---------------------------------------------------------------------------*/
cpl_bivector * eris_ifu_optimal_extraction(const hdrl_imagelist *cube,
                                           const cpl_imagelist  *cube_dqi,
                                           const cpl_image      *img_mask,
                                           double               startLambda,
                                           double               deltaLambda,
                                           int                  productDepth,
                                           cpl_vector           **error_out)
{
    cpl_bivector    *spectrum   = NULL;
    cpl_mask        *mask       = NULL;
    cpl_vector      *spec       = NULL,
                    *spec_var   = NULL;
    eris_ifu_vector *spec2      = NULL,
    		       *spec_var2   = NULL;
//    hdrl_image      *img        = NULL;
//    cpl_image       *contribMap = NULL;
//    cpl_size        nx          = 0,
//                    ny          = 0;
//    cpl_size        xcen        = 0,
//                    ycen        = 0;
//    double          fwhm        = 0.;

    cpl_ensure(cube != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(cube_dqi != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(img_mask != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(startLambda > 1.0, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(deltaLambda > 0, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(error_out != NULL,CPL_ERROR_NULL_INPUT, NULL);

    TRY {
//        nx = hdrl_imagelist_get_size_x(cube);
//        ny = hdrl_imagelist_get_size_y(cube);
//
//        // find centre & fwhm from a collapsed image (without fitting)
//        img = eris_ifu_extract_spec_collapse(cube, &contribMap);
//        if (productDepth >= PD_DEBUG) {
//            eris_ifu_save_hdrl_image_dbg(img, "eris_dbg_collapsed_cube", CPL_IO_CREATE, NULL);
//        }
//
//        BRK_IF_ERROR(
//            eris_ifu_opt_extr_get_center_fwhm(img, edge_trim, &xcen, &ycen, &fwhm));
//        cpl_msg_debug(cpl_func,"Mask center at (%d/%d), FWHM: %g)", (int)xcen, (int)ycen, fwhm);
//
//        BRK_IF_NULL(
//            mask = eris_ifu_opt_extr_create_mask(nx, ny, xcen, ycen, fwhm));
//        if (productDepth >= PD_DEBUG) {
//            eris_ifu_save_mask_dbg(mask, "eris_dbg_mask.fits", CPL_IO_CREATE, NULL);
//        }

        // convert image to mask
        // img:  1: good, 0:bad  -> mask: 0: good, 1: bad
        mask = eris_ifu_mask_from_image(img_mask);
        if (productDepth >= PD_DEBUG) {
            eris_ifu_save_mask_dbg(mask, "eris_dbg_mask.fits", CPL_IO_CREATE, NULL);
        }

        BRK_IF_ERROR(
            eris_ifu_opt_extr_simple_extraction(cube, mask, &spec, &spec_var));
        if (productDepth >= PD_DEBUG) {
            eris_ifu_save_vector_dbg(spec, "eris_dbg_spec1.fits", CPL_IO_CREATE, NULL);
            eris_ifu_save_vector_dbg(spec_var, "eris_dbg_spec1.fits", CPL_IO_EXTEND, NULL);
        }

        spec2     = eris_ifu_vector_create(spec),
        spec_var2 = eris_ifu_vector_create(spec_var);
        BRK_IF_NULL(
            spectrum = eris_ifu_opt_extr_doit(cube, cube_dqi, mask,
                                              spec2, spec_var2,
                                              startLambda, deltaLambda,
                                              productDepth,
                                              error_out));
        eris_ifu_free_ifu_vector(&spec2);
        eris_ifu_free_ifu_vector(&spec_var2);

        if (productDepth >= PD_DEBUG) {
            // error ist the total error of the masked area
            eris_ifu_save_bivector_dbg(spectrum, "eris_dbg_spectrum_out.fits", 0, CPL_IO_CREATE);
            eris_ifu_save_vector_dbg(*error_out, "eris_dbg_spectrum_out.fits", CPL_IO_EXTEND, NULL);

            eris_ifu_save_vector_dbg(cpl_bivector_get_x_const(spectrum), "eris_dbg_spectrum_in.fits", CPL_IO_CREATE, NULL);
            eris_ifu_save_vector_dbg(spec, "eris_dbg_spectrum_in.fits", CPL_IO_EXTEND, NULL);
            eris_ifu_opt_extr_vector_sqrt(spec_var);
            eris_ifu_save_vector_dbg(spec_var, "eris_dbg_spectrum_in.fits", CPL_IO_EXTEND, NULL);
        }
    } CATCH
    {
        eris_ifu_free_bivector(&spectrum);
        eris_ifu_free_mask(&mask);
        eris_ifu_free_ifu_vector(&spec2);
        eris_ifu_free_ifu_vector(&spec_var2);
    }
//    eris_ifu_free_image(&contribMap);
    eris_ifu_free_vector(&spec_var);
    eris_ifu_free_vector(&spec);
    eris_ifu_free_mask(&mask);

    return spectrum;
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Find center position and FWHM from collapsed image
 * @param    hdrl_img        Input collapsed cube image
 * @param    edge_trim       Number of edge pixels to ignore
 * @param    xcen            [out] X-coordinate of maximum flux
 * @param    ycen            [out] Y-coordinate of maximum flux
 * @param    fwhm            [out] Estimated FWHM in pixels
 * @return   CPL_ERROR_NONE on success, error code otherwise
 *
 * Determines the position and characteristic size of a point source from
 * a collapsed cube image. The algorithm:
 * 1. Finds the position of maximum flux (ignoring edge pixels)
 * 2. Counts pixels above half-maximum
 * 3. Estimates FWHM as sqrt(number_of_pixels)
 *
 * This provides a rough but robust estimate suitable for defining extraction
 * apertures. Bad pixels (BPM) and NaN values are automatically excluded.
 *
 * @note Output coordinates use 0-based indexing
 * @note FWHM estimate assumes roughly circular source
 * @note Edge trim helps avoid edge artifacts affecting center detection
 */
/*---------------------------------------------------------------------------*/
cpl_error_code eris_ifu_opt_extr_get_center_fwhm(const hdrl_image *hdrl_img,
                                                 int        edge_trim,
                                                 cpl_size   *xcen,
                                                 cpl_size   *ycen,
                                                 double     *fwhm)
{
    cpl_error_code      ret_error   = CPL_ERROR_NONE;
    double              maxval      = -9999e10;
    const cpl_image     *img_data   = NULL;
    const double        *pimg_data  = NULL;
    const cpl_mask      *img_mask   = NULL;
    const cpl_binary    *pimg_mask  = NULL;
    cpl_size            nx          = 0;
    int                 npix        = 0;

    cpl_ensure_code(hdrl_img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(edge_trim >= 0, CPL_ERROR_NULL_INPUT);

    TRY {
        BRK_IF_NULL(
            img_data = hdrl_image_get_image_const(hdrl_img));
        nx = cpl_image_get_size_x(img_data);
        BRK_IF_NULL(
            pimg_data = cpl_image_get_data_double_const(img_data));
        BRK_IF_NULL(
            img_mask = hdrl_image_get_mask_const(hdrl_img));
        BRK_IF_NULL(
            pimg_mask = cpl_mask_get_data_const(img_mask));

        // find maxval and its position
        for (cpl_size x = edge_trim; x < hdrl_image_get_size_x(hdrl_img)-edge_trim; x++) {
            for (cpl_size y = edge_trim; y < hdrl_image_get_size_y(hdrl_img)-edge_trim; y++) {
                if ((pimg_mask[x+y*nx] == GOOD_PIX) &&
                    (pimg_data[x+y*nx] > maxval) &&
                    (!isnan(pimg_data[x+y*nx])))
                {
                    maxval = pimg_data[x+y*nx];
                    *xcen = x;
                    *ycen = y;
                }
            }
        }

        cpl_msg_debug(cpl_func,"Mask max. value: %g)", maxval);

        // find number of pixels where value is bigger than half of maxval
        maxval *= 0.5;
        for (cpl_size x = edge_trim; x < hdrl_image_get_size_x(hdrl_img)-edge_trim; x++) {
            for (cpl_size y = edge_trim; y < hdrl_image_get_size_y(hdrl_img)-edge_trim; y++) {
                if ((pimg_mask[x+y*nx] == GOOD_PIX) &&
                    (pimg_data[x+y*nx] > maxval) &&
                    (!isnan(pimg_data[x+y*nx])))
                {
                    npix++;
                }
            }
        }

        // now define fwhm
        *fwhm = sqrt(npix); // a bit larger than the nominal size
        //*fwhm = 1.5 * sqrt(npix); // a bit larger than the nominal size
    } CATCH {
        CATCH_MSG();
        ret_error = cpl_error_get_code();
        *xcen = -1;
        *ycen = -1;
        *fwhm = NAN;
    }

    return ret_error;
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Create circular binary mask for optimal extraction
 * @param    nx      Width of mask to create
 * @param    ny      Height of mask to create
 * @param    xcen    X-coordinate of circle center (0-indexed)
 * @param    ycen    Y-coordinate of circle center (0-indexed)
 * @param    fwhm    Radius of circular aperture in pixels
 * @return   Binary mask (0=good/inside, 1=bad/outside), or NULL on error
 *
 * Creates a binary mask for optimal extraction where pixels inside the circular
 * aperture are marked as GOOD_PIX (0) and pixels outside are marked as BAD_PIX (1).
 *
 * @note Mask convention: GOOD_PIX (0) = inside aperture, BAD_PIX (1) = outside
 * @note This is the standard CPL binary mask convention
 * @note Center coordinates use 0-based indexing
 * @note The caller must free the returned cpl_mask
 */
/*---------------------------------------------------------------------------*/
cpl_mask* eris_ifu_opt_extr_create_mask(int nx,
                                        int ny,
                                        int xcen,
                                        int ycen,
                                        double fwhm)
{
    cpl_mask    *mask   = NULL;
    cpl_binary  *pmask   =NULL;

    cpl_ensure(nx > 0,   CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(ny > 0,   CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(xcen > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(ycen > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(fwhm > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    TRY {
        BRK_IF_NULL(
            mask = cpl_mask_new(nx, ny));
        BRK_IF_NULL(
            pmask = cpl_mask_get_data(mask));

        for (int x = 0; x < nx; x++) {
            for (int y = 0; y < ny; y++) {
                if (fwhm <= sqrt(pow(x-xcen,2)+pow(y-ycen,2))) {
                    // bad: outside
                    pmask[x+y*nx] = BAD_PIX;
                } else {
                    // good:inside
                    pmask[x+y*nx] = GOOD_PIX;
                }
            }
        }
    } CATCH {
        CATCH_MSG();
        eris_ifu_free_mask(&mask);
    }

    return mask;
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Perform simple aperture extraction in masked region
 * @param    cube        Input HDRL imagelist (data cube)
 * @param    mask        Binary mask defining extraction region
 * @param    spec        [out] Extracted spectrum (simple sum)
 * @param    spec_var    [out] Variance of extracted spectrum
 * @return   CPL_ERROR_NONE on success, error code otherwise
 *
 * Implements Horne (1986) Table 1, Step 4: simple extraction by summing
 * all pixels within the mask for each wavelength slice. This provides
 * the initial spectrum estimate for optimal extraction.
 *
 * @note Pixels marked as BAD_PIX in mask are excluded
 * @note The caller must free the output spec and spec_var vectors
 * @note Errors are propagated as sum of variances
 */
/*---------------------------------------------------------------------------*/
cpl_error_code eris_ifu_opt_extr_simple_extraction(const hdrl_imagelist *cube,
                                                   const cpl_mask       *mask,
                                                   cpl_vector           **spec,
                                                   cpl_vector           **spec_var)
{
    cpl_error_code      ret_error   = CPL_ERROR_NONE;
    const hdrl_image    *tmp_img    = NULL;
    hdrl_image          *tmp_img2   = NULL;
    double              *pspec      = NULL,
                        *pspec_var  = NULL;
    cpl_size            nz = 0;
    hdrl_value          hv;

    cpl_ensure_code(cube != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mask != NULL, CPL_ERROR_NULL_INPUT);

    TRY {
        nz = hdrl_imagelist_get_size(cube);

        BRK_IF_NULL(
                *spec = cpl_vector_new(nz));
        BRK_IF_NULL(
                *spec_var = cpl_vector_new(nz));
        BRK_IF_NULL(
            pspec = cpl_vector_get_data(*spec));
        BRK_IF_NULL(
            pspec_var = cpl_vector_get_data(*spec_var));

        for (int z = 0; z < nz; z++) {
            BRK_IF_NULL(
                tmp_img = hdrl_imagelist_get_const(cube, z));
            BRK_IF_NULL(
                tmp_img2 = hdrl_image_duplicate(tmp_img));
            BRK_IF_ERROR(
                hdrl_image_reject_from_mask(tmp_img2, mask));

            hv = hdrl_image_get_sum(tmp_img2);
            pspec[z] = hv.data;
            pspec_var[z] = hv.error*hv.error;
            hdrl_image_delete(tmp_img2);
        }
    } CATCH {
        CATCH_MSG();
        ret_error = cpl_error_get_code();
        eris_ifu_free_vector(spec);
        eris_ifu_free_vector(spec_var);
    }

    return ret_error;
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Get coordinates of pixels within extraction mask
 * @param    mask        Binary mask (0=good, 1=bad)
 * @param    n_usepix    [out] Number of unmasked pixels
 * @return   Bivector containing (x,y) coordinates of unmasked pixels, or NULL on error
 *
 * Creates a list of pixel coordinates that are included in the extraction
 * (i.e., marked as GOOD_PIX in the mask). This is used to efficiently
 * iterate over only the relevant pixels during optimal extraction.
 *
 * @note Returned bivector contains 0-based pixel coordinates
 * @note The caller must free the returned cpl_bivector
 */
/*---------------------------------------------------------------------------*/
cpl_bivector * eris_ifu_opt_extr_helper_usepix(const cpl_mask *mask, int *n_usepix)
{
    cpl_bivector        *usepix = NULL;
    int                 nx      = 0,
                        ny      = 0,
                        index   = 0;
    double              *px     = NULL,
                        *py     = NULL;
    const cpl_binary    *pmask  = NULL;

    cpl_ensure(mask != NULL, CPL_ERROR_NULL_INPUT, NULL);

    TRY {
        nx = cpl_mask_get_size_x(mask);
        ny = cpl_mask_get_size_y(mask);

        // get number of pixels in mask
        *n_usepix = (nx * ny) - cpl_mask_count(mask);
        cpl_msg_debug(cpl_func,"optimal extraction: n_usepix: %d", *n_usepix);

        if (*n_usepix > 0) {
            usepix = cpl_bivector_new(*n_usepix);
            pmask = cpl_mask_get_data_const(mask);

            px = cpl_vector_get_data(cpl_bivector_get_x(usepix));
            py = cpl_vector_get_data(cpl_bivector_get_y(usepix));

            for (int y = 0; y < ny; y++) {
                for (int x = 0; x < nx; x++) {
                    if (pmask[x+y*nx] == GOOD_PIX) {
                        px[index] = x;
                        py[index] = y;
                        index++;
                    }
                }
            }
        }
    } CATCH {
        eris_ifu_free_bivector(&usepix);
    }

    return usepix;
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Fill image columns with vector values
 * @param    img     Target image to fill
 * @param    vec     Source eris_ifu_vector containing data and mask
 * @return   CPL_ERROR_NONE on success, error code otherwise
 *
 * Copies the vector data into every column of the image, creating a 2D
 * array where each column is identical. Pixels are rejected if the vector
 * mask indicates invalid data.
 *
 * @note Vector size must match image height
 * @note Used to create 2D arrays of spectrum for PSF modeling
 */
/*---------------------------------------------------------------------------*/
cpl_error_code eris_ifu_opt_extr_helper_fill_vertical(cpl_image *img, const eris_ifu_vector *vec)
{
    cpl_ensure_code(vec, CPL_ERROR_NULL_INPUT);

    double          *pimg       = cpl_image_get_data_double(img);
    cpl_size        nx          = cpl_image_get_size_x(img),
                    ny          = cpl_image_get_size_y(img);
    double          *pkvmask    = cpl_vector_get_data(vec->data),
                    *pkvdata    = cpl_vector_get_data(vec->mask);

    cpl_ensure_code(eris_ifu_vector_get_size(vec) == ny, CPL_ERROR_ILLEGAL_INPUT);

    for (int y = 0; y < ny; y++) {
        for (int x = 0; x < nx; x++) {
            if (pkvmask[y] > 0.5) {
                pimg[x+y*nx] = pkvdata[y];
            } else {
                cpl_image_reject(img, x+1, y+1);
            }
        }
    }

    return CPL_ERROR_NONE;
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Extract values from wavelength slice and fill image row
 * @param    img         Target image
 * @param    slice       Source wavelength slice (data or error)
 * @param    usepix      Bivector of pixel coordinates to extract
 * @param    row         Row number in target image to fill
 * @param    power       If non-zero, square the values before storing
 *
 * Extracts pixel values from a wavelength slice at positions specified by
 * usepix and stores them in a row of the target image. This efficiently
 * reorganizes 3D cube data into 2D format for PSF modeling.
 *
 * @note Supports both double and int input slice types
 * @note The power flag is used to convert errors to variances
 */
/*---------------------------------------------------------------------------*/
void eris_ifu_opt_extr_helper_fill_horizontal(cpl_image *img, const cpl_image *slice, const cpl_bivector *usepix, int row, int power) {
    double          *pimg       = cpl_image_get_data_double(img);
    const double    *pslice_d   = NULL;
    const int       *pslice_i   = NULL;
    cpl_size        nx          = cpl_image_get_size_x(slice);
    cpl_type        type        = cpl_image_get_type(slice);
    const double    *px         = cpl_vector_get_data_const(cpl_bivector_get_x_const(usepix)),
                    *py         = cpl_vector_get_data_const(cpl_bivector_get_y_const(usepix));
    cpl_size        n_usepix    = cpl_bivector_get_size(usepix);

    if (type == CPL_TYPE_DOUBLE) {
        pslice_d = cpl_image_get_data_double_const(slice);
    } else if (type == CPL_TYPE_INT) {
        pslice_i = cpl_image_get_data_int_const(slice);
    } else {
        return;
    }

    if (type == CPL_TYPE_DOUBLE) {
        for (int i = 0; i < n_usepix; i++) {
            double val = pslice_d[(int)px[i]+(int)py[i]*nx];
            if (power) {
                pimg[i+row*n_usepix] = pow(val, 2);
            } else {
                pimg[i+row*n_usepix] = val;
            }
        }
    } else {
        for (int i = 0; i < n_usepix; i++) {
            int val = pslice_i[(int)px[i]+(int)py[i]*nx];
            if (power) {
                pimg[i+row*n_usepix] = pow(val, 2);
            } else {
                pimg[i+row*n_usepix] = val;
            }
        }
    }
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Set all negative and NaN values to zero
 * @param    img     Image to modify in place
 *
 * Replaces all negative values and NaN/Inf values with zero. This is used
 * to ensure positivity of PSF models during optimal extraction.
 *
 * @note The image is modified in place
 * @note Logs the number of values that were reset
 */
/*---------------------------------------------------------------------------*/
void eris_ifu_opt_extr_helper_set_positive(cpl_image *img) {
    double      *pimg   = cpl_image_get_data_double(img);
    cpl_size    nx      = cpl_image_get_size_x(img),
                ny      = cpl_image_get_size_y(img);
    int         n_neg   = 0;

    for (int x = 0; x < nx; x++) {
        for (int y = 0; y < ny; y++) {
            if ((pimg[x+y*nx] < 0.) || isnan(pimg[x+y*nx])) {
                pimg[x+y*nx] = 0.;
                n_neg++;
            }
        }
    }
    cpl_msg_debug(cpl_func, "optimal extraction: Resetting %d negative values in original PSF", n_neg);
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Extract column from image as vector
 * @param    img     Source image
 * @param    colnr   Column number to extract (0-indexed)
 * @return   Vector containing column data, or NULL on error
 *
 * @note The caller must free the returned cpl_vector
 */
/*---------------------------------------------------------------------------*/
cpl_vector* eris_ifu_opt_extr_get_col(const cpl_image *img, int colnr)
{
    int             nx      = 0,
                    ny      = 0;
    double          *pvec   = NULL;
    const double    *pimg   = NULL;
    cpl_vector      *vec    = NULL;

    cpl_ensure(img != NULL, CPL_ERROR_NULL_INPUT, NULL);

    nx = cpl_image_get_size_x(img);
    ny = cpl_image_get_size_y(img);

    cpl_ensure(colnr <= nx, CPL_ERROR_ILLEGAL_INPUT, NULL);

    vec = cpl_vector_new(ny);
    pvec = cpl_vector_get_data(vec);
    pimg = cpl_image_get_data_double_const(img);

    for (int y = 0; y < ny; y++) {
        pvec[y] = pimg[colnr+y*nx];
    }

    return vec;
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Extract row from image as vector
 * @param    img     Source image
 * @param    rownr   Row number to extract (0-indexed)
 * @return   Vector containing row data, or NULL on error
 *
 * @note The caller must free the returned cpl_vector
 */
/*---------------------------------------------------------------------------*/
cpl_vector* eris_ifu_opt_extr_get_row(const cpl_image *img, int rownr)
{
    int             nx      = 0,
                    ny      = 0;
    double          *pvec   = NULL;
    const double    *pimg   = NULL;
    cpl_vector      *vec    = NULL;

    cpl_ensure(img != NULL, CPL_ERROR_NULL_INPUT, NULL);

    nx = cpl_image_get_size_x(img);
    ny = cpl_image_get_size_y(img);

    cpl_ensure(rownr <= ny, CPL_ERROR_ILLEGAL_INPUT, NULL);

    vec = cpl_vector_new(nx);
    pvec = cpl_vector_get_data(vec);
    pimg = cpl_image_get_data_double_const(img);

    for (int x = 0; x < nx; x++) {
        pvec[x] = pimg[x+rownr*nx];
    }

    return vec;
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Set row of image from vector
 * @param    img     Target image
 * @param    rownr   Row number to set (0-indexed)
 * @param    vec     Source vector
 * @return   CPL_ERROR_NONE on success, error code otherwise
 *
 * @note Vector size must match image width
 */
/*---------------------------------------------------------------------------*/
cpl_error_code eris_ifu_opt_extr_set_row(cpl_image *img,
                                         int rownr,
                                         const cpl_vector *vec)
{
    int             nx      = 0,
                    ny      = 0;
    const double    *pvec   = NULL;
    double          *pimg   = NULL;

    cpl_ensure_code(img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(vec != NULL, CPL_ERROR_NULL_INPUT);

    nx = cpl_image_get_size_x(img);
    ny = cpl_image_get_size_y(img);

    cpl_ensure_code(rownr <= ny, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(nx == cpl_vector_get_size(vec), CPL_ERROR_ILLEGAL_INPUT);

    pvec = cpl_vector_get_data_const(vec);
    pimg = cpl_image_get_data_double(img);

    for (int x = 0; x < nx; x++) {
        pimg[x+rownr*nx] = pvec[x];
    }

    return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Set column of image from vector
 * @param    img     Target image
 * @param    colnr   Column number to set (0-indexed)
 * @param    vec     Source vector
 * @return   CPL_ERROR_NONE on success, error code otherwise
 *
 * @note Vector size must match image height
 */
/*---------------------------------------------------------------------------*/
cpl_error_code eris_ifu_opt_extr_set_col(cpl_image *img,
                                         int colnr,
                                         const cpl_vector *vec)
{
    int             nx      = 0,
                    ny      = 0;
    double          *pimg   = NULL;
    const double    *pvec   = NULL;

    cpl_ensure_code(img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(vec != NULL, CPL_ERROR_NULL_INPUT);

    nx = cpl_image_get_size_x(img);
    ny = cpl_image_get_size_y(img);

    cpl_ensure_code(colnr <= nx, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(ny == cpl_vector_get_size(vec), CPL_ERROR_ILLEGAL_INPUT);

    pvec = cpl_vector_get_data_const(vec);
    pimg = cpl_image_get_data_double(img);

    for (int y = 0; y < ny; y++) {
        pimg[colnr+y*nx] = pvec[y];
    }

    return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Create wavelength vector for spectrum
 * @param    size            Number of wavelength points
 * @param    startLambda     Starting wavelength in micrometers
 * @param    deltaLambda     Wavelength step in micrometers
 * @return   Wavelength vector
 *
 * Creates a uniformly spaced wavelength grid: lambda[i] = startLambda + i * deltaLambda
 *
 * @note The caller must free the returned cpl_vector
 */
/*---------------------------------------------------------------------------*/
cpl_vector* eris_ifu_opt_extr_create_lambda(int size, double startLambda, double deltaLambda)
{
    cpl_vector *lambda  = cpl_vector_new(size);
    double     *pLambda = cpl_vector_get_data(lambda);

    for (int i = 0; i < size; i++) {
        pLambda[i] = startLambda + (double)i * deltaLambda;
    }
    return lambda;
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Compute square root of vector elements
 * @param    vec     Vector to modify in place
 * @return   CPL_ERROR_NONE on success
 *
 * Applies sqrt() to each element of the vector, typically used to convert
 * variance to standard deviation.
 */
/*---------------------------------------------------------------------------*/
cpl_error_code eris_ifu_opt_extr_vector_sqrt(cpl_vector *vec)
{
    double *pvec = cpl_vector_get_data(vec);
    cpl_ensure_code(vec != NULL, CPL_ERROR_NULL_INPUT);

    for (int i = 0; i < cpl_vector_get_size(vec); i++) {
        pvec[i] = sqrt(pvec[i]);
    }
    return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Core optimal extraction algorithm implementation
 * @param    cube            Input data cube
 * @param    cube_dqi        Data quality image list
 * @param    mask            Extraction aperture mask
 * @param    spec            Initial spectrum estimate
 * @param    spec_var        Initial spectrum variance
 * @param    startLambda     Starting wavelength in micrometers
 * @param    deltaLambda     Wavelength step in micrometers
 * @param    productDepth    Debug output level
 * @param    error_out       [out] Output spectrum errors
 * @return   Bivector containing wavelength and optimally extracted flux
 *
 * This is the main workhorse function implementing the modified Horne (1986)
 * optimal extraction algorithm. See eris_ifu_optimal_extraction() for overview.
 *
 * Key parameters:
 * - sm = 49: smoothing half-length for PSF model (adjustable)
 * - clipfrac = 0.9: percentile for outlier rejection
 * - bigloop: 2 iterations for PSF refinement
 * - iloop: 2 iterations for polynomial fitting
 *
 * @note This function performs the heavy computational work
 * @note Debug output (productDepth >= PD_DEBUG) generates many diagnostic files
 * @note The caller must free the returned bivector and error_out vector
 */
/*---------------------------------------------------------------------------*/
cpl_bivector* eris_ifu_opt_extr_doit(const hdrl_imagelist *cube,
                                     const cpl_imagelist  *cube_dqi,
                                     const cpl_mask       *mask,
                                     const eris_ifu_vector     *spec,
                                     const eris_ifu_vector     *spec_var,
                                     double startLambda,
                                     double deltaLambda,
                                     int productDepth,
                                     cpl_vector **error_out)
{
    cpl_bivector    *spectrum       = NULL,
                    *usepix         = NULL;
    cpl_size        nz              = 0;
    cpl_vector      *vec            = NULL,
                    *rejectvec      = NULL,
                    *vec_err        = NULL,
                    *keep0          = NULL,
                    *new_vec        = NULL;
    eris_ifu_vector *speco          = NULL,
                    *spec_var_err   = NULL,
                    *sigo           = NULL;
    cpl_image       *spec1          = NULL,
                    *psfvals        = NULL,
                    *varvals        = NULL,
                    *qualvals       = NULL,
                    *psfmod         = NULL,
                    *psfmodnew      = NULL,
                    *reject         = NULL,
                    *snr2           = NULL,
                    *residual       = NULL;
    int             n_usepix        = 0,
                    jj              = 0,
                    sm              = 49;  // smoothing half-length (use, e.g. 15 for spectra with wiggles, up to 49 for ideal spectra).
                                            // Smoothing is done over a length of +/-sm pixels.
    double          clipfrac        = 0.9, // fraction to set for percentile clipping
                    *pnew_vec       = NULL,
                    *pqualvals      = NULL,
                    *preject        = NULL,
                    *pkeep0         = NULL/*,
                    *pspeco         = NULL,
                    *psigo          = NULL*/;
    const double    *pvec           = NULL,
                    *prejectvec     = NULL;

    cpl_ensure(cube          != NULL,   CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(cube_dqi      != NULL,   CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(mask          != NULL,   CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(spec          != NULL,   CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(spec_var      != NULL,   CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(error_out     != NULL,   CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(hdrl_imagelist_get_size(cube) == cpl_imagelist_get_size(cube_dqi),
               CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(hdrl_imagelist_get_size(cube) == eris_ifu_vector_get_size(spec),
               CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(hdrl_imagelist_get_size(cube) == eris_ifu_vector_get_size(spec_var),
               CPL_ERROR_ILLEGAL_INPUT, NULL);

    TRY {
        nz = hdrl_imagelist_get_size(cube);

        spec_var_err = eris_ifu_vector_duplicate(spec_var);
        eris_ifu_vector_sqrt(spec_var_err);
        eris_ifu_vector_divide(spec_var_err, spec);
        CHECK_ERROR_STATE();

        // get coordinates of used pixels in mask
        usepix = eris_ifu_opt_extr_helper_usepix(mask, &n_usepix);

        // for convenience spectra of all selected pixels are stored in a 2D array
        spec1 = cpl_image_new(n_usepix, nz, CPL_TYPE_DOUBLE);
        psfvals = cpl_image_new(n_usepix, nz, CPL_TYPE_DOUBLE);    // = (D - S)
        //psfmod  = cpl_image_new(n_usepix, nz, CPL_TYPE_DOUBLE);  // = P
        varvals = cpl_image_new(n_usepix, nz, CPL_TYPE_DOUBLE);    // = V
        qualvals = cpl_image_new(n_usepix, nz, CPL_TYPE_DOUBLE);   // for additional clipping regarding DQI
        pqualvals = cpl_image_get_data_double(qualvals);
        CHECK_ERROR_STATE();

        // begin optimal extraction
        //   1st time through bigloop use initial spectrum
        //   2nd time through bigloop use previous output spectrum
        speco  = eris_ifu_vector_duplicate(spec);    // working spectrum data
//        pspeco = cpl_vector_get_data(speco);
        sigo   = eris_ifu_vector_new(nz);            // working spectrum error
//        psigo  = cpl_vector_get_data(sigo);
        CHECK_ERROR_STATE();

        for (int bigloop = 0; bigloop <= 1; bigloop++) {
            // spec0 is vector
            // spec1 is image (duplicates of speco)
            BRK_IF_ERROR(
                eris_ifu_opt_extr_helper_fill_vertical(spec1, speco));

            // fill data/error/dqi values of used (unmasked) pixels to 2D-array
            // for varvals store the squared values
            for (int z = 0; z < nz; z++) {
                const hdrl_image *hdrl_img = hdrl_imagelist_get_const(cube, z);
                const cpl_image  *slice_d  = hdrl_image_get_image_const(hdrl_img);
                const cpl_image  *slice_e  = hdrl_image_get_error_const(hdrl_img);
                const cpl_image  *slice_q  = cpl_imagelist_get_const(cube_dqi, z);

                eris_ifu_opt_extr_helper_fill_horizontal(psfvals,  slice_d, usepix, z, FALSE);
                eris_ifu_opt_extr_helper_fill_horizontal(varvals,  slice_e, usepix, z, TRUE);
                eris_ifu_opt_extr_helper_fill_horizontal(qualvals, slice_q, usepix, z, FALSE);
            }

            eris_ifu_opt_extr_helper_set_positive(psfvals);

            if (productDepth >= PD_DEBUG) {
                eris_ifu_save_image_dbg(psfvals,  "ers_psfvals.fits", CPL_IO_CREATE, NULL);
                eris_ifu_save_image_dbg(varvals,  "ers_varvals.fits", CPL_IO_CREATE, NULL);
                eris_ifu_save_image_dbg(qualvals, "ers_qualvals.fits", CPL_IO_CREATE, NULL);
            }

            // replace fractions with weighted polynomial fits: the PSF model
            // Horne86, page 613 bottom right & Fig 1
            // two iterations; first with all pixels, second with clipped (from reject)
            reject = cpl_image_new(n_usepix, nz, CPL_TYPE_DOUBLE);    // all zeros initially
            preject = cpl_image_get_data_double(reject);
            CHECK_ERROR_STATE();

            new_vec = cpl_vector_new(nz);
            pnew_vec = cpl_vector_get_data(new_vec);

            for (int iloop = 0; iloop <= 1; iloop++) {
                if (psfmod != NULL) {
                    eris_ifu_free_image(&psfmod);
                }
                psfmod = cpl_image_divide_create(psfvals, spec1);   // Horne86 eq 14, P = (D-S)/f
                eris_ifu_opt_extr_helper_set_positive(psfmod);
                CHECK_ERROR_STATE();

                if (productDepth >= PD_DEBUG) {
                    eris_ifu_save_image_dbg(psfmod, "ers_psfmod.fits", CPL_IO_CREATE, NULL);
                }

                // slow loop
                for (int n = 0; n < n_usepix; n++) {
                    vec = eris_ifu_opt_extr_get_col(psfmod, n);
                    pvec = cpl_vector_get_data_const(vec);
                    CHECK_ERROR_STATE();

                    BRK_IF_NULL(
                        vec_err = eris_ifu_vector_get_data(spec_var_err));
                    BRK_IF_ERROR(
                        cpl_vector_multiply(vec_err, vec));

                    rejectvec = eris_ifu_opt_extr_get_col(reject, n);
                    prejectvec = cpl_vector_get_data_const(rejectvec);
                    CHECK_ERROR_STATE();

                    // vector 'keep0' with indices on which values to keep
                    keep0 = cpl_vector_new(nz);
                    cpl_vector_fill(keep0, -1.0);
                    pkeep0 = cpl_vector_get_data(keep0);
                    jj = 0;

                    for (cpl_size iz = 0; iz < nz; iz++) {

                        if ((fabs(prejectvec[iz]) < 1e-5) &&
                            (pvec[iz] > 0) &&
                            !eris_ifu_is_nan_or_inf(pvec[iz]))
                        {

                            pkeep0[jj++]= iz;

                        }
                    } // end: for iz
                    cpl_vector_delete(rejectvec);
                    BRK_IF_ERROR(
                        eris_ifu_cut_endings(&keep0, NULL, NULL, TRUE));
                    if (keep0 != NULL) {
                        pkeep0 = cpl_vector_get_data(keep0);
                        CHECK_ERROR_STATE();

                        if ((productDepth >= PD_DEBUG) && (n==92)) {
                            eris_ifu_save_vector_dbg(keep0, "ers_keep0.fits", CPL_IO_CREATE, NULL);
                        }

                        // replace polynomial fitting by weighted mean smoothing over +/-sm pixels, after rejection
                        for (int z = 0; z < nz; z++) {
                            // reject keep0
                            int s = cpl_vector_get_size(keep0);
                            cpl_vector* ttt = cpl_vector_new(s);
                            double *pttt = cpl_vector_get_data(ttt);
                            cpl_vector_fill(ttt, -1.0);
                            jj = 0;
                            for (int iz = 0; iz < s; iz++) {
                                int v = (int)(pkeep0[iz] +.5);
                                if ((v > z-sm) && (v < z+sm)) {
                                    pttt[jj++] = iz;
                                }
                            }
                            if (productDepth >= PD_DEBUG) {
                                eris_ifu_save_vector_dbg(ttt, "ers_ttt.fits", CPL_IO_CREATE, NULL);
                            }
                            BRK_IF_ERROR(
                                eris_ifu_cut_endings(&ttt, NULL, NULL, TRUE));
                            if (ttt != NULL) {
                                cpl_vector  *iiuse   = eris_ifu_idl_values_at_indices(keep0, ttt),
                                            *els     = eris_ifu_idl_values_at_indices(vec, iiuse),
                                            *els_err = eris_ifu_idl_values_at_indices(vec_err, iiuse),
                                            *ones    = cpl_vector_new(cpl_vector_get_size(els));
                                CHECK_ERROR_STATE();

                                if ((productDepth >= PD_DEBUG) && (n==92) && (z == 1000)) {
                                    eris_ifu_save_vector_dbg(iiuse, "ers_iiuse.fits", CPL_IO_CREATE, NULL);
                                    eris_ifu_save_vector_dbg(els, "ers_els.fits", CPL_IO_CREATE, NULL);
                                    eris_ifu_save_vector_dbg(els_err, "ers_elserr.fits", CPL_IO_CREATE, NULL);
                                }

                                cpl_vector_fill(ones, 1.0);
                                cpl_vector_divide(els, els_err);
                                cpl_vector_divide(ones, els_err);
                                pnew_vec[z] = cpl_vector_get_sum(els) / cpl_vector_get_sum(ones);
                                CHECK_ERROR_STATE();

                                eris_ifu_free_vector(&iiuse);
                                eris_ifu_free_vector(&ones);
                                eris_ifu_free_vector(&els);
                                eris_ifu_free_vector(&els_err);
                            } else {
                                pnew_vec[z] = NAN;
                            }

                            eris_ifu_free_vector(&ttt);
                        } // end: for z
                        CHECK_ERROR_STATE();

                        BRK_IF_ERROR(
                            eris_ifu_opt_extr_set_col(psfmod, n, new_vec));

                        if ((productDepth >= PD_DEBUG) && (n==92)) {
                            eris_ifu_save_vector_dbg(new_vec, "ers_newvec.fits", CPL_IO_CREATE, NULL);
                        }
                    }
                    eris_ifu_free_vector(&vec);
                    eris_ifu_free_vector(&vec_err);
                    eris_ifu_free_vector(&keep0);
                } // end: for n
                CHECK_ERROR_STATE();

                if (productDepth >= PD_DEBUG) {
                    if (iloop==0) {
                        eris_ifu_save_image_dbg(psfmod, "ers_psfmod_0.fits", CPL_IO_CREATE, NULL);
                    } else {
                        eris_ifu_save_image_dbg(psfmod, "ers_psfmod_1.fits", CPL_IO_CREATE, NULL);
                    }
                }

                // second iteration of polynomial fitting uses Horne86, page 614, top left: ((D-S) - fP)^2/V
                // also use qualitfy flag to reject pixels & define clipping value from data, slice by slice
                // snr2 = (psfvals-psfmod*spec1)^2/varvals
                cpl_image *psfmod_tmp = cpl_image_multiply_create(psfmod, spec1);
                snr2 = cpl_image_subtract_create(psfvals, psfmod_tmp);
                cpl_image_power(snr2, 2);
                cpl_image_divide(snr2, varvals);
                eris_ifu_free_image(&psfmod_tmp);
                CHECK_ERROR_STATE();
                if (productDepth >= PD_DEBUG) {
                    if (iloop==0) {
                        eris_ifu_save_image_dbg(snr2, "ers_snr2_0.fits", CPL_IO_CREATE, NULL);
                        eris_ifu_save_image_dbg(reject, "ers_reject0_0.fits", CPL_IO_CREATE, NULL);
                    } else {
                        eris_ifu_save_image_dbg(snr2, "ers_snr2_1.fits", CPL_IO_CREATE, NULL);
                        eris_ifu_save_image_dbg(reject, "ers_reject0_1.fits", CPL_IO_CREATE, NULL);
                    }
                }
                for (int iz = 0; iz < nz; iz++) {
                    vec = eris_ifu_opt_extr_get_row(snr2, iz);
                    int n_fin = eris_ifu_opt_extr_get_not_finite(vec);
                    if (n_fin > n_usepix*0.25) {
                        //cpl_vector_abs
                        cpl_vector_power(vec, 2);
                        eris_ifu_cpl_vector_sqrt(vec);

                        cpl_vector_sort(vec, CPL_SORT_ASCENDING);
                        double clip = cpl_vector_get(vec, n_usepix*clipfrac)*5;
                        cpl_vector *tmp = eris_ifu_idl_where(vec, clip, gt);
                        if ((tmp != NULL) && (cpl_vector_get_size(tmp) > 0)) {
                            for (int ii = 0; ii < cpl_vector_get_size(tmp); ii++) {
                                preject[(int)(cpl_vector_get(tmp, ii)+.5)+iz*n_usepix] = 1;
                            }
                        }
                        eris_ifu_free_vector(&tmp);
                    } else {
                        for (int ii = 0; ii < n_usepix; ii++) {
                            preject[ii+iz*n_usepix] = 1;
                        }
                    }
                    eris_ifu_free_vector(&vec);
                } // end: for iz
                CHECK_ERROR_STATE();
                if (productDepth >= PD_DEBUG) {
                    if (iloop==0) {
                        eris_ifu_save_image_dbg(reject, "ers_reject1_0.fits", CPL_IO_CREATE, NULL);
                    } else {
                        eris_ifu_save_image_dbg(reject, "ers_reject1_1.fits", CPL_IO_CREATE, NULL);
                    }
                }
                for (int ii = 0; ii < n_usepix * nz; ii++) {
                    if (pqualvals[ii] > 0) {
                        preject[ii] = 1;
                    }
                }
                CHECK_ERROR_STATE();
                if (productDepth >= PD_DEBUG) {
                    if (iloop==0) {
                        eris_ifu_save_image_dbg(reject, "ers_reject2_0.fits", CPL_IO_CREATE, NULL);
                    } else {
                        eris_ifu_save_image_dbg(reject, "ers_reject2_1.fits", CPL_IO_CREATE, NULL);
                    }
                }
                eris_ifu_free_image(&snr2);
            } // end: for iloop
            eris_ifu_free_vector(&new_vec);

            if (productDepth >= PD_DEBUG) {
                eris_ifu_save_image_dbg(reject, "ers_reject3.fits", CPL_IO_CREATE, NULL);
            }
            eris_ifu_free_image(&reject);

            //go through PSF model plane by plane
            // reject low quality & noisy pixels * re-extract spectrum
            // Horne86 page 614, left, & Table 1 step 5c, 7, 8
            // steps 6 is skipped;
            // step 7 includes the quality flag

            eris_ifu_free_image(&psfmodnew);
            BRK_IF_NULL(
                psfmodnew = cpl_image_duplicate(psfmod));

            //residual = (psfvals-psfmod*spec1)^2/varvals
            // for step 7; page 614 top right
            eris_ifu_free_image(&residual);
            residual = cpl_image_multiply_create(psfmod, spec1);
            cpl_image_subtract(residual, psfvals);
            cpl_image_multiply_scalar(residual, -1.);
            cpl_image_power(residual, 2);
            cpl_image_divide(residual, varvals);

            if (productDepth >= PD_DEBUG) {
                eris_ifu_save_image_dbg(psfvals, "ers_psfvalsXXX.fits", CPL_IO_CREATE, NULL);
                eris_ifu_save_image_dbg(psfmod, "ers_psfmodXXX.fits", CPL_IO_CREATE, NULL);
                eris_ifu_save_image_dbg(spec1, "ers_spec1XXX.fits", CPL_IO_CREATE, NULL);
                eris_ifu_save_image_dbg(varvals, "ers_varvalsXXX.fits", CPL_IO_CREATE, NULL);
                eris_ifu_save_image_dbg(residual, "ers_residual.fits", CPL_IO_CREATE, NULL);
            }

            // need to set clip value slice by slice
            // fit linear function to first 90% of sorted residual values and clip
            for (int iz = 0; iz < nz; iz++) {
                cpl_vector *resvec0 = eris_ifu_opt_extr_get_row(residual, iz);
                double     *presvec0 = cpl_vector_get_data(resvec0);

                if (productDepth >= PD_DEBUG) {
                    eris_ifu_save_vector_dbg(resvec0, "ers_resvec0.fits", CPL_IO_CREATE, NULL);
                }

                // vector 'keep0' with indices on which values to keep
                keep0 = cpl_vector_new(n_usepix);
                cpl_vector_fill(keep0, -1.0);
                pkeep0 = cpl_vector_get_data(keep0);
                jj = 0;
                for (int ii = 0; ii < n_usepix; ii++) {
                    if (!eris_ifu_is_nan_or_inf(presvec0[ii]) &&
                        presvec0[ii] > 0.)
                    {
                        pkeep0[jj++]= ii;
                    }
                }
                BRK_IF_ERROR(
                    eris_ifu_cut_endings(&keep0, NULL, NULL, TRUE));

                if (productDepth >= PD_DEBUG) {
                    eris_ifu_save_vector_dbg(keep0, "ers_keep.fits", CPL_IO_CREATE, NULL);
                }

                cpl_vector *p = eris_ifu_opt_extr_get_row(psfmod, iz);
                double     *pp = cpl_vector_get_data(p);

                if ((keep0 != NULL) && (cpl_vector_get_size(keep0) > n_usepix*0.25)) {
                    int nkeep = cpl_vector_get_size(keep0);
                    cpl_vector *resvec  = eris_ifu_idl_values_at_indices(resvec0, keep0),
                        *resvecs = cpl_vector_duplicate(resvec);
                    cpl_vector_sort(resvecs, CPL_SORT_ASCENDING);

                    if ((productDepth >= PD_DEBUG) && (iz == 1000)){
                        eris_ifu_save_vector_dbg(resvecs, "ers_resvecs.fits", CPL_IO_CREATE, NULL);
                    }

                    double clip = cpl_vector_get(resvecs, (int)(nkeep*clipfrac+.5)-1) * 5.;
                    if (clip < 25.) {
                        clip = 25.;
                    }

                    cpl_vector *q = eris_ifu_opt_extr_get_row(qualvals, iz);
                    double     *pq = cpl_vector_get_data(q);
                    eris_ifu_free_vector(&keep0);
                    keep0 = cpl_vector_new(n_usepix);
                    cpl_vector_fill(keep0, -1.0);
                    pkeep0 = cpl_vector_get_data(keep0);
                    jj = 0;
                    for (int ii = 0; ii < n_usepix; ii++) {
                        if ((presvec0[ii] > clip) || (pq[ii] == 1)) {
                            pkeep0[jj++]= ii;
                        }
                    }
                    BRK_IF_ERROR(
                        eris_ifu_cut_endings(&keep0, NULL, NULL, TRUE));
                    if (keep0 != NULL) {
                        pkeep0 = cpl_vector_get_data(keep0);
                        for (int ii = 0; ii < cpl_vector_get_size(keep0); ii++) {
                            pp[(int)(pkeep0[ii])] = 0.;
                        }
                    }

                    double sum = cpl_vector_get_sum(p);
                    for (int ii = 0; ii < cpl_vector_get_size(p); ii++) {
                        pp[ii] /= sum;
                    }

                    eris_ifu_free_vector(&q);
                    eris_ifu_free_vector(&resvecs);
                    eris_ifu_free_vector(&resvec);
                } else {
                    BRK_IF_ERROR(
                        cpl_vector_fill(p, 0.));
                }

                BRK_IF_ERROR(
                    eris_ifu_opt_extr_set_row(psfmodnew, iz, p));

                cpl_vector *vals = eris_ifu_opt_extr_get_row(psfvals, iz),
                           *var  = eris_ifu_opt_extr_get_row(varvals, iz);
                CHECK_ERROR_STATE();
                eris_ifu_opt_extr_convert_0_to_NaN_vec(var);

                // pspeco[iz] = total(p*vals/var)/total(p^2./var)
                cpl_vector *tt = cpl_vector_duplicate(p);
                cpl_vector_multiply(tt, vals);
                cpl_vector_divide(tt, var);
                cpl_vector_power(p, 2.0);
                cpl_vector_divide(p, var);
                double sum_data   = cpl_vector_get_sum(tt),
                       sum_weight = cpl_vector_get_sum(p);
                BRK_IF_ERROR(
                    eris_ifu_vector_set(speco, iz, sum_data / sum_weight));

                //BRK_IF_ERROR(
                //    eris_ifu_vector_set(sigo, iz, sqrt(1./total(p^2/var)));
                BRK_IF_ERROR(
                    eris_ifu_vector_set(sigo, iz, sqrt(1./sum_weight)));

                eris_ifu_free_vector(&tt);
                eris_ifu_free_vector(&vals);
                eris_ifu_free_vector(&var);
                eris_ifu_free_vector(&p);
                eris_ifu_free_vector(&keep0);
                eris_ifu_free_vector(&resvec0);
                CHECK_ERROR_STATE();
            } // end if: iz
            CHECK_ERROR_STATE();
        } // endfor: bigloop
        CHECK_ERROR_STATE();

        // median((speco/sigo)/(spec0/sig0))
        eris_ifu_vector *tt = eris_ifu_vector_duplicate(speco);
        eris_ifu_vector_divide(tt, sigo);
        eris_ifu_vector_power(spec_var_err, -1.);
        eris_ifu_vector_divide(tt, spec_var_err);
        CHECK_ERROR_STATE();

        cpl_msg_info(cpl_func, "typical S/N has increased by %g", eris_ifu_vector_get_median(tt, ERIS_IFU_ARITHMETIC));
        eris_ifu_free_ifu_vector(&tt);

        cpl_vector *lambda = eris_ifu_opt_extr_create_lambda(nz, startLambda, deltaLambda);
        cpl_vector *speco2 = eris_ifu_vector_get_data(speco);
        BRK_IF_NULL(
            spectrum = cpl_bivector_wrap_vectors(lambda, speco2));

        cpl_vector *sigo2 = eris_ifu_vector_get_data(sigo);
        *error_out = sigo2;

        // return sigo as vector (pow? sqrt?)
        // return total_flux??
    } CATCH {
        CATCH_MSG();
        eris_ifu_free_bivector(&spectrum);
        eris_ifu_free_vector(error_out);
    }

    eris_ifu_free_ifu_vector(&speco);
    eris_ifu_free_ifu_vector(&sigo);
    eris_ifu_free_ifu_vector(&spec_var_err);
    eris_ifu_free_bivector(&usepix);
    eris_ifu_free_image(&spec1);
    eris_ifu_free_image(&psfvals);
    eris_ifu_free_image(&psfmod);
    eris_ifu_free_image(&psfmodnew);
    eris_ifu_free_image(&varvals);
    eris_ifu_free_image(&qualvals);
    eris_ifu_free_image(&reject);
    eris_ifu_free_image(&residual);

    return spectrum;
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Count number of finite (non-NaN, non-Inf) values in vector
 * @param    vec     Input vector to analyze
 * @return   Number of finite values
 */
/*---------------------------------------------------------------------------*/
int eris_ifu_opt_extr_get_not_finite(cpl_vector *vec) {
    int n_fin = 0;
    double *pvec = NULL;

    cpl_ensure_code(vec != NULL, CPL_ERROR_NULL_INPUT);

    pvec = cpl_vector_get_data(vec);

    for (int i = 0; i < cpl_vector_get_size(vec); i++) {
        if (!eris_ifu_is_nan_or_inf(pvec[i])) {
            n_fin++;
        }
    }

    return n_fin;
}

/*---------------------------------------------------------------------------*/
/**
 * @brief    Convert all zero values to NaN in vector
 * @param    vec     Vector to modify in place
 *
 * Replaces all values close to zero (within DBL_ZERO_TOLERANCE) with NaN.
 * This is useful for avoiding division by zero in subsequent calculations.
 *
 * @note The vector is modified in place
 * @note Also converts existing NaN values (idempotent operation)
 */
/*---------------------------------------------------------------------------*/
void eris_ifu_opt_extr_convert_0_to_NaN_vec(cpl_vector *vec)
{
    cpl_size    x     = 0;
    double      *pvec = NULL;

    if (vec != NULL) {
        pvec = cpl_vector_get_data(vec);

        x = cpl_vector_get_size(vec);

        for (int i = 0 ; i < x; i++) {
            if (fabs(pvec[i]) < DBL_ZERO_TOLERANCE) {
                pvec[i] = NAN;
            }
        }
    }
}

cpl_image* eris_ifu_opt_extr_estimate_radius_helper(const cpl_image *img,
                                                    cpl_size center_x, cpl_size center_y,
                                                    double radius,
                                                    cpl_size *llx, cpl_size *lly,
                                                    cpl_size *urx, cpl_size *ury)
{
    double          half_maxval = 0.,
                    *pmask      = NULL;
    const double    *pimg       = NULL;
    cpl_image       *mask       = NULL;
    cpl_size        nx          = 0,
                    ny          = 0;

    cpl_ensure(img != NULL, CPL_ERROR_NULL_INPUT, NULL);

    TRY {
        pimg = cpl_image_get_data_double_const(img);
        nx = cpl_image_get_size_x(img);
        ny = cpl_image_get_size_y(img);
        half_maxval = pimg[center_x+nx*center_y] / 2;

        mask = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        pmask = cpl_image_get_data_double(mask);

        *llx = center_x - radius;
        *lly = center_y - radius;
        *urx = center_x + radius;
        *ury = center_y + radius;
        if (*llx < 0) {
            *llx = 0;
        }
        if (*lly < 0) {
            *lly = 0;
        }
        if (*urx >= nx) {
            *urx = nx-1;
        }
        if (*ury >= ny) {
            *ury = ny-1;
        }

        for (cpl_size x = *llx; x < *urx; x++) {
            for (cpl_size y = *lly; y < *ury; y++) {
                if (pimg[x+nx*y] >= half_maxval) {
                    pmask[x+nx*y] = 1;
                }
            }
        }

    } CATCH {
        CATCH_MSG();
        eris_ifu_free_image(&mask);
    }

    return mask;
}

double eris_ifu_opt_extr_estimate_radius(const cpl_image *img,
                                         cpl_size center_x, cpl_size center_y,
                                         double initial_radius,
                                         int productDepth)
{
    double          radius          = 0.,
                    inc_radius      = 2.,
                    area_rectangle  = 1.,
                    area_mask       = 4.;
    cpl_image       *mask           = NULL;
    cpl_size        llx             = 0,
                    lly             = 0,
                    urx             = 0,
                    ury             = 0,
                    nx              = 0,
                    ny              = 0;

    cpl_ensure(img != NULL, CPL_ERROR_NULL_INPUT, -1.);

    TRY {
        nx = cpl_image_get_size_x(img);
        ny = cpl_image_get_size_y(img);
        // define a max. radius: half size of image
        double max_radius = nx/2;
        if (ny/2 < max_radius) {
            max_radius = ny/2;
        }
        radius = initial_radius - inc_radius;
        // calculate ratio of good pixels, if > pi/4 make radius bigger

        while ((area_mask / area_rectangle >= CPL_MATH_PI_4) && (radius < max_radius)) {
            radius += inc_radius;
            eris_ifu_free_image(&mask);
            mask =  eris_ifu_opt_extr_estimate_radius_helper(img,
                                                            center_x, center_y,
                                                            radius,
                                                            &llx, &lly, &urx, &ury);
            area_rectangle = (urx-llx)*(ury-lly);
            area_mask = cpl_image_get_flux(mask);
            if (productDepth >= PD_DEBUG) {
                eris_ifu_save_image_dbg(mask, "eris_dbg_mask.fits", CPL_IO_CREATE, NULL);
            }
            cpl_msg_debug(cpl_func, "  intermediate mask center at (%d/%d), radius: %g, area mask: %g, area rect: %g)",
                                    (int)center_x, (int)center_y, radius, area_mask, area_rectangle);

        }
        if (productDepth >= PD_DEBUG) {
            eris_ifu_save_image_dbg(mask, "eris_dbg_mask.fits", CPL_IO_CREATE, NULL);
        }
        cpl_msg_debug(cpl_func, "Final mask center at (%d/%d), radius: %g, area mask: %g, area rect: %g)",
                                (int)center_x, (int)center_y, radius, area_mask, area_rectangle);
    } CATCH {
        CATCH_MSG();
        radius = 0.;
    }

    eris_ifu_free_image(&mask);

    return radius;
}

///**
// * @brief eris_ifu_opt_extr_median_without_NaN
// * @param vec
// */
//double eris_ifu_opt_extr_median_without_NaN(cpl_vector *vec)
//{
//    cpl_size    x       = 0;
//    double      *pvec   = NULL,
//                *ptmp   = NULL,
//                median  = -1.;
//    int         cnt     = 0;
//    cpl_vector  *tmp    = NULL;
//
//    if (vec != NULL) {
//        pvec = cpl_vector_get_data(vec);
//
//        x = cpl_vector_get_size(vec);
//        for (int i = 0 ; i < x; i++) {
//            if (eris_ifu_is_nan_or_inf(pvec[i])) {
//                cnt++;
//            }
//        }
//
//        if ((x-cnt) > 0) {
//            tmp = cpl_vector_new(x-cnt);
//            ptmp = cpl_vector_get_data(tmp);
//            cnt = 0;
//            for (int i = 0 ; i < x; i++) {
//                if (!eris_ifu_is_nan_or_inf(pvec[i])) {
//                    ptmp[cnt++] = pvec[i];
//                }
//            }
//
//            median = cpl_vector_get_median(tmp);
//            eris_ifu_free_vector(&tmp);
//        }
//    }
//    return median;
//}

/**
 * @brief eris_ifu_vector_sqrt
 * @param v
 * @return
 *
 * Fixes a bug in cpl_vector_sqrt(): When a value is -nan, the function returns an ILLEGAL_INPUT error...
 * sqrt() does return just the same nan value, so let's do so...
 */
cpl_error_code eris_ifu_cpl_vector_sqrt(cpl_vector *vec)
{
    cpl_size        n       = 0;
    double          *pvec   = NULL;
    cpl_error_code  err     = CPL_ERROR_NONE;

    cpl_ensure_code(vec, CPL_ERROR_NULL_INPUT);

    TRY
    {
        n = cpl_vector_get_size(vec);
        CHECK_ERROR_STATE();

        BRK_IF_NULL(
            pvec = cpl_vector_get_data(vec));

        /* Compute the sqrt */
        for (int i = 0; i < n; i++) {
            /* pvec[i] can be NaN, since sqrt() handles NaN correctly:
               sqrt(NaN) = NaN */
            if (pvec[i] >= 0) {
                pvec[i] = sqrt(pvec[i]);
            } else {
                pvec[i] = NAN;
            }
        }
    } CATCH {
        err = cpl_error_get_code();
    }

    return err;
}
