/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <hdrl.h>
#include "eris_utils.h"
#include "eris_ifu_jitter_static.h"
#include "eris_ifu_efficiency_response.h"
#include "eris_ifu_resample.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_strehl.h"
#include "eris_ifu_error.h"
#include "eris_ifu_stdstar_static.h"

/*----------------------------------------------------------------------------*/
/**
    @defgroup eris_ifu_stdstar_static   Standard Star Calibration Pipeline
    
    This module provides the main data reduction pipeline for standard star
    observations used in IFU spectroscopy calibration. It orchestrates the
    complete reduction workflow from raw exposures to calibrated products.
    
    The pipeline handles multiple observation types:
    - STD: Telluric/spectrophotometric standard stars for response calibration
    - STD_FLUX: Flux standard stars for system efficiency calibration
    - PSF_CALIBRATOR: Point spread function calibrators for Strehl computation
    - PUPIL_LAMP: Pupil imaging with calibration lamp
    - PUPIL_SKY: Pupil imaging with sky background
    
    The reduction workflow includes:
    1. Parameter fetching and SOF (Set of Frames) processing
    2. Individual exposure reduction and cube generation
    3. Cube combination and co-addition (if multiple exposures)
    4. Spectral extraction from combined cubes
    5. Instrumental response computation (STD with flat field)
    6. System efficiency computation (STD_FLUX without flat field)
    7. Flux calibration of spectra and cubes
    8. Strehl ratio computation (PSF calibrators)
    
    Processing can be performed with or without flat field correction,
    generating different product categories and calibration outputs.
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Main standard star data reduction pipeline orchestration function
  @param    frameset            Input frameset containing raw/calibration frames
  @param    parlist             Recipe parameters from command-line/config file
  @param    apply_flat_field    Flag: CPL_TRUE=apply flat field, CPL_FALSE=skip flat
  @param    context             Recipe context name (e.g., "eris")
  @param    recipe_name         Name of the calling recipe
  @return   CPL_ERROR_NONE on success, error code otherwise
  
  This is the main orchestration function for standard star observations. It
  executes a complete data reduction pipeline from raw exposures to calibrated
  products, with different processing paths depending on observation type.
  
  ## Processing Workflow:
  
  **1. Initialization and Parameter Fetching**
  - Identifies observation type from frameset tags (STD, STD_FLUX, PSF_CALIBRATOR, etc.)
  - Fetches recipe parameters and configures processing options
  - Initializes SOF (Set of Frames) structure with calibration files
  
  **2. Exposure Processing** (`eris_ifu_jitter_process_exposures`)
  - Processes individual raw exposures
  - Applies detector corrections (linearity, dark, bad pixels)
  - Applies flat field if requested
  - Generates intermediate products
  
  **3. Cube Generation** (`eris_ifu_jitter_process_cubes`)
  - Constructs 3D data cubes from processed exposures
  - Applies wavelength calibration
  - Performs geometric distortion correction
  - Generates different cube types based on processing stage:
    * Basic cubes (OBJECT_CUBE, STD_CUBE, STD_FLUX_CUBE, PSF_CUBE, PUPIL_CUBE)
    * Differential atmospheric refraction corrected (DAR_*)
    * Sky-tweaked cubes (TWEAKED_*)
  
  **4. Cube Combination** (if multiple exposures)
  - Combines/co-adds multiple cubes using HDRL functionality
  - Applies pixel-by-pixel combination with outlier rejection
  - Generates co-added cube products (*_COADD)
  - Special handling for single exposure: duplicates tags and collapses to mean
  
  **5. Pupil Shift Computation** (PUPIL_LAMP/PUPIL_SKY only)
  - Computes pupil alignment shifts
  - Generates quality control parameters
  
  **6. Spectral Extraction** (`eris_ifu_jitter_extract`)
  - Extracts 1D spectra from combined cubes
  - Uses optimal extraction or aperture photometry
  - Generates SPECTRUM products
  
  **7. Instrumental Response Computation** (STD with flat field)
  - `eris_response_compute()`: Computes instrumental response curve
  - Requires: STD observation, extracted spectrum, flat field applied
  - Output: RESPONSE_CURVE product
  
  **8. System Efficiency Computation** (STD without flat field)
  - `eris_efficiency_compute()`: Computes telescope+instrument efficiency
  - Requires: STD_FLUX observation, extracted spectrum, no flat field, 
              flux standard catalog, extinction coefficients
  - Output: EFFICIENCY_CURVE product
  
  **9. Flux Calibration**
  - `eris_flux_calibrate_spectra()`: Calibrates extracted spectra
  - `eris_flux_calibrate_cube2()`: Calibrates combined data cubes
  - Requires: RESPONSE_CURVE product, flat field applied
  - Output: FLUX_CALIBRATED_SPECTRUM and FLUX_CALIBRATED_CUBE
  
  **10. Strehl Ratio Computation** (PSF calibrators with flat field)
  - `eris_ifu_stdstar_strehl_compute()`: Computes Strehl ratio from PSF
  - Measures image quality by comparing to theoretical diffraction limit
  - Requires: PSF_CALIBRATOR observation, flat field applied
  - Output: Strehl ratio QC parameters
  
  ## Observation Type Processing Paths:
  
  | Observation Type | With Flat Field                          | Without Flat Field                 |
  |------------------|------------------------------------------|------------------------------------|
  | STD              | Response computation                     | (No calibration output)            |
  | STD_FLUX         | Flux calibration                         | Efficiency computation             |
  | PSF_CALIBRATOR   | Strehl computation, Flux calibration     | (No calibration output)            |
  | PUPIL_LAMP       | Pupil shift computation                  | Pupil shift computation            |
  | PUPIL_SKY        | Pupil shift computation                  | Pupil shift computation            |
  
  ## Product Categories by Cube Type:
  
  The function handles different product category (PRODCATG) assignments based
  on cube processing stage and flat field application:
  
  - **Basic cubes**: JITTER_STD_CUBE, JITTER_STD_FLUX_CUBE, etc.
  - **DAR-corrected**: JITTER_DAR_CUBE
  - **Sky-tweaked**: JITTER_TWK_CUBE
  - **Co-added**: *_COADD or *_COADD_NOFLAT (without flat)
  
  ## Parameter Control:
  
  Processing steps are controlled by boolean parameters in parlist:
  - `<context>.combine`: Enable cube combination
  - `<context>.extractSource`: Enable spectral extraction
  - `<context>.compute-response`: Enable response computation
  - `<context>.compute-efficiency`: Enable efficiency computation
  - `<context>.flux-calibrate`: Enable flux calibration
  - `<context>.compute-strehl`: Enable Strehl computation
  
  @note     The apply_flat_field flag significantly affects processing:
            - With flat: enables response, flux calibration, Strehl
            - Without flat: enables efficiency computation
            - Product names include "_NOFLAT" suffix when flat is not applied
  
  @note     Function uses TRY/CATCH error handling macros
  @note     All intermediate structures (sof, params) are freed on exit
  @note     Products are automatically added to frameset with appropriate tags
  @note     QC parameters are logged throughout processing
  
  @see eris_ifu_jitter_process_exposures
  @see eris_ifu_jitter_process_cubes
  @see eris_ifu_combine_pbp
  @see eris_ifu_jitter_extract
  @see eris_response_compute
  @see eris_efficiency_compute
  @see eris_flux_calibrate_spectra
  @see eris_flux_calibrate_cube2
  @see eris_ifu_stdstar_strehl_compute
  
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if any input pointer is NULL
  - Various error codes from sub-functions (exposure processing, cube generation, etc.)
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
eris_stdstar_reduction_common(cpl_frameset *frameset,
                              const cpl_parameterlist *parlist,
                              cpl_boolean apply_flat_field,
                              const char *context,
                              const char *recipe_name)
{
    struct stdParamStruct   stdParams   = stdParamStructInit;
    struct sofStruct        sof;
    struct paramStruct      params;
    int                     obj_number  = 0;
    char                    *pipefile_prefix = NULL;

	cpl_ensure(frameset, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
    cpl_ensure(parlist,  CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
    cpl_ensure(context,  CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(recipe_name, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

    TRY {
        if (apply_flat_field) {
            cpl_msg_info(cpl_func, "=============================================");
            cpl_msg_info(cpl_func, "Data reduction with flat field");
            cpl_msg_info(cpl_func, "---------------------------------------------");
            pipefile_prefix = cpl_sprintf("%s", recipe_name);
        } else {
            cpl_msg_info(cpl_func, "=============================================");
            cpl_msg_info(cpl_func, "Data reduction without flat field");
            cpl_msg_info(cpl_func, "---------------------------------------------");
            pipefile_prefix = cpl_sprintf("%s_%s", recipe_name, "no_flat");
        }

        cpl_boolean is_std_flux = CPL_FALSE;
        cpl_boolean is_std_norm = CPL_FALSE;
        cpl_boolean is_pupil_lamp = CPL_FALSE;
        cpl_boolean is_pupil_sky = CPL_FALSE;
        cpl_boolean is_psf = CPL_FALSE;
        if (cpl_frameset_find(frameset,"STD") != NULL){
            is_std_norm = CPL_TRUE;
        }
        else if (cpl_frameset_find(frameset,"PUPIL_SKY") != NULL){
            is_pupil_sky = CPL_TRUE;
        }
        else if (cpl_frameset_find(frameset,"PUPIL_LAMP") != NULL) {
            is_pupil_lamp = CPL_TRUE;
        }
        else if (cpl_frameset_find(frameset,"STD_FLUX") != NULL) {
            is_std_flux = CPL_TRUE;
        }
        else if (cpl_frameset_find(frameset,"PSF_CALIBRATOR") != NULL) {
            is_psf = CPL_TRUE;
        }


        if (is_std_norm) {
            obj_number = eris_frameset_count_tag(frameset, "STD");
        } else if (is_std_flux) {
            obj_number = eris_frameset_count_tag(frameset, "STD_FLUX");
        } else if (is_pupil_lamp) {
                obj_number = eris_frameset_count_tag(frameset, "PUPIL_LAMP");
        } else if (is_pupil_sky) {
                obj_number = eris_frameset_count_tag(frameset, "PUPIL_SKY");
        } else if (is_psf) {
            obj_number = eris_frameset_count_tag(frameset, "PSF_CALIBRATOR");
        }

        (void)memset(&sof, 0, sizeof(sof)); /* Rough init so free works on error */

		cpl_msg_info(cpl_func, "Reading recipe parameters");
		BRK_IF_ERROR(
            eris_ifu_jitter_fetch_params(context, recipe_name, parlist,
                                         &stdParams, &params));
		cpl_msg_info(cpl_func,
                    "Instrument is %d, requested product level is %d",
                    stdParams.instrument, stdParams.productDepth);

		BRK_IF_ERROR(
            eris_ifu_jitter_processSof(frameset,
                                       stdParams,
                                       params,
                                       &sof));

		cpl_msg_info(cpl_func,"Instrument %s, band %s, scale %s",
                    eris_ifu_get_instrumentString(sof.instrument),
                    eris_ifu_get_bandString(sof.band),
                    eris_ifu_get_preopticsScaleString(sof.scale));

		BRK_IF_ERROR(
            eris_ifu_jitter_process_exposures(&sof, stdParams, params,
                                              frameset, parlist, pipefile_prefix,
                                              context));

		cubeType obj_type = OBJECT_CUBE;

		BRK_IF_ERROR(
            eris_ifu_jitter_process_cubes(&sof, stdParams, params, frameset,
                                          parlist, pipefile_prefix, &obj_type));

		cubeType resampled_obj_type = STD_CUBE_COADD;
		const char *input_cube_pro_catg = NULL;
		if (cpl_frameset_count_tags(frameset, ERIS_IFU_PRO_JITTER_DAR_CUBE)
				> 0) {
			cpl_msg_info(cpl_func,"case1");
			if(is_std_norm) {
				obj_type = DAR_STD_CUBE;
				resampled_obj_type = DAR_STD_CUBE_COADD;
				input_cube_pro_catg = ERIS_IFU_PRO_JITTER_DAR_CUBE;
			} else if(is_std_flux) {
				obj_type = DAR_STD_FLUX_CUBE;
				if (apply_flat_field) {
					resampled_obj_type = STD_FLUX_CUBE_COADD;
					input_cube_pro_catg = ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD;
				} else {
					resampled_obj_type = STD_FLUX_CUBE_COADD_NOFLAT;
					input_cube_pro_catg = ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_NOFLAT;
				}
			} else if(is_psf) {
				obj_type = DAR_PSF_CUBE;
				resampled_obj_type = DAR_PSF_CUBE_COADD;
				input_cube_pro_catg = ERIS_IFU_PRO_JITTER_DAR_CUBE;
			} else if(is_pupil_lamp || is_pupil_sky) {
				obj_type = DAR_STD_CUBE;
				resampled_obj_type = DAR_PUPIL_CUBE_COADD;
				input_cube_pro_catg = ERIS_IFU_PRO_JITTER_DAR_CUBE;
			}
		} else if (cpl_frameset_count_tags(frameset, ERIS_IFU_PRO_JITTER_TWK_CUBE)
				> 0 ) {
			cpl_msg_info(cpl_func,"case2");
			if(is_std_norm) {
				obj_type = TWEAKED_STD_CUBE;
				resampled_obj_type = STD_CUBE_COADD;
				input_cube_pro_catg = ERIS_IFU_PRO_JITTER_STD_CUBE_COADD;
			} else if(is_std_flux) {
				obj_type = STD_FLUX_CUBE;
				if (apply_flat_field) {
				    resampled_obj_type = STD_FLUX_CUBE_COADD;
				    input_cube_pro_catg = ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD;
				} else {
					resampled_obj_type = STD_FLUX_CUBE_COADD_NOFLAT;
				    input_cube_pro_catg = ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_NOFLAT;
				}
			} else if(is_psf) {
				obj_type = PSF_CUBE;
				resampled_obj_type = PSF_CUBE_COADD;
				input_cube_pro_catg = ERIS_IFU_PRO_JITTER_TWK_CUBE;
			} else if(is_pupil_lamp || is_pupil_sky) {
				obj_type = PUPIL_CUBE;
				resampled_obj_type = PUPIL_CUBE_COADD;
				input_cube_pro_catg = ERIS_IFU_PRO_JITTER_TWK_CUBE;
			}
		} else {
			cpl_msg_info(cpl_func,"case3");
			if(is_std_norm) {
				obj_type = STD_CUBE;
				resampled_obj_type = STD_CUBE_COADD;
				input_cube_pro_catg = ERIS_IFU_PRO_JITTER_STD_CUBE_COADD;
			} else if(is_std_flux) {
				obj_type = STD_FLUX_CUBE;
				if (apply_flat_field) {
				    resampled_obj_type = STD_FLUX_CUBE_COADD;
				    input_cube_pro_catg = ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD;
				} else {
					resampled_obj_type = STD_FLUX_CUBE_COADD_NOFLAT;
					input_cube_pro_catg = ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_NOFLAT;
				}

			} else if(is_psf) {
				obj_type = PSF_CUBE;
				resampled_obj_type = PSF_CUBE_COADD;
				input_cube_pro_catg = ERIS_IFU_PRO_JITTER_PSF_CUBE_COADD;
			} else if(is_pupil_lamp ||is_pupil_sky) {
				obj_type = PUPIL_CUBE;
				resampled_obj_type = PUPIL_CUBE_COADD;
				input_cube_pro_catg = ERIS_IFU_PRO_JITTER_PUPIL_CUBE_COADD;
			}
		}

		cpl_msg_info(cpl_func,"input cube pro catg: %s",input_cube_pro_catg);

		/* combine cubes with HDRL functionality */
		if (params.combine && obj_number > 1) {
			// eris_ifu_combine(obj_type, stdParams, params, frameset, parlist,
			//         recipe_name, pipefile_prefix);
			char *combDoCatg =  NULL;
			char   *filenameSpec = NULL;
			eris_ifu_jitter_get_procatg_and_filename(obj_type, &combDoCatg,
					&filenameSpec);

            cpl_msg_debug(cpl_func,"GENERATE COMBINED DATA CUBE %s",combDoCatg);
			eris_ifu_combine_pbp(frameset, parlist, combDoCatg, filenameSpec,
					NULL, NULL, NULL, recipe_name, pipefile_prefix);

			eris_ifu_free_string(&combDoCatg);
			eris_ifu_free_string(&filenameSpec);
		} else if (obj_number == 1) {
			char *combDoCatg =  NULL;
			char   *filenameSpec = NULL;
		    eris_ifu_jitter_get_procatg_and_filename(obj_type, &combDoCatg,
								&filenameSpec);
			if(is_std_norm) {
				eris_frameset_duplicate_cube_tag(frameset,
						ERIS_IFU_PRO_JITTER_STD_CUBE, apply_flat_field);
				eris_ifu_cube_collapse_mean_and_save(ERIS_IFU_PRO_JITTER_STD_CUBE,
                        frameset, parlist, recipe_name, /*filenameSpec, */apply_flat_field, CPL_FALSE);
				//cpl_frameset_dump(frameset, stdout);
			}
			else if(is_std_flux) {
				cpl_msg_debug(cpl_func,"duplicate FLUX cube tag");
				eris_frameset_duplicate_cube_tag(frameset,
						ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE, apply_flat_field);
				eris_ifu_cube_collapse_mean_and_save(ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE,
                        frameset, parlist, recipe_name, /*filenameSpec,*/ apply_flat_field, CPL_FALSE);

			} else if(is_pupil_lamp || is_pupil_sky) {
				cpl_msg_debug(cpl_func,"duplicate PUPIL cube tag");
				eris_frameset_duplicate_cube_tag(frameset,
						ERIS_IFU_PRO_JITTER_OBJ_CUBE, apply_flat_field);
				eris_ifu_cube_collapse_mean_and_save(ERIS_IFU_PRO_JITTER_OBJ_CUBE,
                        frameset, parlist, recipe_name, /*filenameSpec, */apply_flat_field, CPL_TRUE);
			} else if(is_psf) {
				cpl_msg_debug(cpl_func,"duplicate PSF cube tag");
				eris_frameset_duplicate_cube_tag(frameset,
						ERIS_IFU_PRO_JITTER_PSF_CUBE, apply_flat_field);
				eris_ifu_cube_collapse_mean_and_save(ERIS_IFU_PRO_JITTER_PSF_CUBE,
                        frameset, parlist, recipe_name, /*filenameSpec, */apply_flat_field, CPL_FALSE);
			}
			cpl_free(combDoCatg);
		    cpl_free(filenameSpec);
		}

		if( is_pupil_lamp || is_pupil_sky) {
//			hdrl_imagelist* himl = NULL;
			cpl_table* qclog_tbl = eris_qclog_init();
			eris_get_pupil_shift(sof.exposureTable[0].cube, 0, &qclog_tbl);
			cpl_free(qclog_tbl);
		}

		/* extract spectra and for STD star computes response and efficiency */
        if (apply_flat_field  == CPL_FALSE) {
            if (is_std_norm) {
				obj_type = STD_CUBE_COADD_NOFLAT;
			} else if (is_std_flux) {
				obj_type = STD_FLUX_CUBE_COADD_NOFLAT;
			}
		}
		if (params.extractSource && eris_can_extract(frameset)) {
			eris_print_rec_status(102);
			cpl_msg_info(cpl_func,"input pro catg: %s obj_type: %d",
					input_cube_pro_catg, resampled_obj_type);

			eris_ifu_jitter_extract(frameset, parlist, resampled_obj_type,
					input_cube_pro_catg, stdParams, pipefile_prefix, context);
			eris_print_rec_status(103);
		}
		//cpl_frameset_dump(frameset, stdout);
		char* param_name = NULL;

		param_name = cpl_sprintf("%s.compute-strehl", context);
		cpl_boolean compute_strehl = cpl_parameter_get_bool(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		param_name = cpl_sprintf("%s.flux-calibrate", context);
		cpl_boolean flux_calibrate = cpl_parameter_get_bool(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		eris_ifu_jitter_get_cube_type_string(obj_type);
		if(obj_type == STD_CUBE || obj_type == STD_CUBE_COADD_NOFLAT ||
		   obj_type == STD_FLUX_CUBE || obj_type == STD_FLUX_CUBE_COADD_NOFLAT||
		   obj_type == DAR_STD_FLUX_CUBE || obj_type == DAR_STD_CUBE_COADD) {

			param_name = cpl_sprintf("%s.compute-efficiency", context);
			cpl_boolean compute_efficiency = cpl_parameter_get_bool(
					cpl_parameterlist_find_const(parlist, param_name));
			cpl_free(param_name);

			param_name = cpl_sprintf("%s.compute-response", context);
			cpl_boolean compute_response = cpl_parameter_get_bool(
					cpl_parameterlist_find_const(parlist, param_name));
			cpl_free(param_name);

			cpl_msg_debug(cpl_func,"compute_response: %d eris_can_compute_response: %d apply_flat_field: %d",
					compute_response,eris_can_compute_response(frameset),
					apply_flat_field);
			if (compute_response && eris_can_compute_response(frameset) &&
					apply_flat_field == CPL_TRUE) {
				cpl_msg_info(cpl_func,"Compute response");
				eris_response_compute(recipe_name, parlist, frameset, frameset);
			}

			cpl_msg_debug(cpl_func, "compute_efficiency: %d eris_can_compute_efficiency: %d",
					compute_efficiency,eris_can_compute_efficiency(frameset));

			if (compute_efficiency && apply_flat_field == CPL_FALSE) {
				if (eris_can_compute_efficiency(frameset)) {
					cpl_msg_info(cpl_func,"Compute efficiency");
					cpl_frame * spectrum_frame = NULL;
					cpl_frame * extcoeff_frame = NULL;
					cpl_frame * fluxstdcat_frame = NULL;

					spectrum_frame = cpl_frameset_find(frameset,
							ERIS_IFU_PRO_JITTER_SPECTRUM_NOFLAT);
					extcoeff_frame = cpl_frameset_find(frameset,
							ERIS_IFU_CALIB_EXTCOEFF_TABLE);
					fluxstdcat_frame = cpl_frameset_find(frameset,
							ERIS_IFU_CALIB_FLUX_STD_CATALOG);

					cpl_table* tab_eff = eris_efficiency_compute(spectrum_frame,
							fluxstdcat_frame, extcoeff_frame, frameset, parlist,
							pipefile_prefix);

					cpl_table_delete(tab_eff);
				}
			}
		}

		//cpl_frameset_dump(frameset, stdout);
		cpl_msg_debug(cpl_func,"flux_calibrate: %d eris_can_flux_calibrate: %d, apply_flat_field:%d",
				flux_calibrate,eris_can_flux_calibrate(frameset),apply_flat_field);
        if (flux_calibrate && apply_flat_field == CPL_TRUE) {
            if (eris_can_flux_calibrate(frameset)) {
				cpl_msg_info(cpl_func,"Flux calibrate extracted spectrum and combined data cube");

				eris_flux_calibrate_spectra(pipefile_prefix, recipe_name,
						parlist, frameset, frameset);
				char* cube_pro_catg = NULL;
                if (is_std_norm) {
					cube_pro_catg = cpl_sprintf("%s",ERIS_IFU_PRO_JITTER_STD_CUBE_COADD);
                } else if (is_std_flux) {
					cube_pro_catg = cpl_sprintf("%s",ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD);
				} else {
					cube_pro_catg = cpl_sprintf("%s",ERIS_IFU_PRO_JITTER_PSF_CUBE_COADD);
				}

				eris_flux_calibrate_cube2(cube_pro_catg, pipefile_prefix,
                        recipe_name, parlist, /*frameset, */frameset);
				cpl_free(cube_pro_catg);
			}
		}

		//if(compute_strehl && !is_std_flux) {
        if (compute_strehl && apply_flat_field) {
			/* for PSF standards we can compute the Strehl */
			cpl_msg_info(cpl_func,"===============================================");
			cpl_msg_info(cpl_func,"===========STREHL COMPUTATION==================");
			cpl_msg_info(cpl_func,"===============================================");
            eris_ifu_stdstar_strehl_compute(frameset, parlist, context/*, recipe_name*/);
		}
		//cpl_frameset_dump(frameset, stdout);
    } CATCH {

	}

	cpl_free(pipefile_prefix);
	eris_ifu_jitter_free_sofStruct(&sof);
	eris_ifu_free_std_param(&stdParams);
	eris_check_error_code("eris_stdstar_reduction_common");

	return cpl_error_get_code();
}

/**@}*/
