
/*********************************************************************
 * E.S.O. - VLT project
 *
 * "@(#) $Id: eris_ifu_vector.h,v 1.3 2013-05-17 15:53:08 aagudo Exp $"
 *
 * who       when        what
 * --------  ----------  ----------------------------------------------
 * agudo     2008-06-12  created
 */

#ifndef ERIS_IFU_VECTOR_H
#define ERIS_IFU_VECTOR_H

/**
    @defgroup eris_ifu_vector wrapper to cpl_vector

    eris_ifu_vector provides the functionality of rejected values like cpl_image
    does. It is a struct containing to vectors. The data vector can contain any
    value. If there are any infinite values in the data, these values will be
    masked already in eris_ifu_vector_create() and eris_ifu_vector_create2().

    The mask vector can just contain the values 0 and 1.
    (this is asserted in eris_ifu_new() and eris_ifu_create() and
    eris_ifu_create2()))

    @par Synopsis:
    @code
      #include <eris_ifu_vector.h>
    @endcode

    @{
 */

/*------------------------------------------------------------------------------
 *              Includes
 *----------------------------------------------------------------------------*/
#include <cpl.h>

/*------------------------------------------------------------------------------
 *              Types
 *----------------------------------------------------------------------------*/
typedef struct {
    cpl_vector *data;
    cpl_vector *mask;
} eris_ifu_vector;

/** @brief The type of median behavior:
    For an odd number of samples the behaviour is the same for both methods.
    For an even number of samples:
    - ERIS_IFU_STATISTICAL: <br>
      The returned value is the lower of the two center elements of the sorted
      input vector.
    - ERIS_IFU_ARITHMETIC <br>
      The arithmetic mean of the two center elements of the sorted input vector
      is calculated
*/
enum medianType {
    ERIS_IFU_STATISTICAL,
    ERIS_IFU_ARITHMETIC
};

/*-----------------------------------------------------------------------------
 *              Prototypes
 *----------------------------------------------------------------------------*/

eris_ifu_vector* eris_ifu_vector_new(
                                int n);
eris_ifu_vector* eris_ifu_vector_new_wrap(
                                int n,
                                const double *data);
eris_ifu_vector* eris_ifu_vector_create(
                                const cpl_vector *data);
eris_ifu_vector* eris_ifu_vector_create2(
                                const cpl_vector *data,
                                const cpl_vector *mask);
void            eris_ifu_vector_delete(
                                eris_ifu_vector *kv);
eris_ifu_vector* eris_ifu_vector_duplicate(
                                const eris_ifu_vector *kv);
cpl_error_code  eris_ifu_vector_set(
                                eris_ifu_vector *kv,
                                int pos,
                                double val);
double          eris_ifu_vector_get_rej(
                                const eris_ifu_vector *kv,
                                int pos,
                                int *rej);
double          eris_ifu_vector_get(
                                const eris_ifu_vector *kv,
                                int pos);
cpl_vector*     eris_ifu_vector_get_mask(
                                const eris_ifu_vector *kv);
cpl_vector      *eris_ifu_vector_get_data(
                                const eris_ifu_vector *kv);
cpl_vector*     eris_ifu_vector_get_bpm(
                                eris_ifu_vector *kv);
cpl_error_code  eris_ifu_vector_reject_from_mask(
                                eris_ifu_vector *kv,
                                const cpl_vector *mask,
                                int keep);

int             eris_ifu_vector_count_rejected(
                                const eris_ifu_vector *kv);
int             eris_ifu_vector_count_non_rejected(
                                const eris_ifu_vector *kv);
int             eris_ifu_vector_is_rejected(
                                const eris_ifu_vector *kv,
                                int n);
cpl_error_code  eris_ifu_vector_reject(
                                eris_ifu_vector *kv,
                                int n);
eris_ifu_vector* eris_ifu_vector_extract(
                                const eris_ifu_vector *kv,
                                int istart,
                                int istop);
cpl_vector*     eris_ifu_vector_create_non_rejected(
                                const eris_ifu_vector *kv);

cpl_error_code  eris_ifu_vector_adapt_rejected(
                                eris_ifu_vector *kv1,
                                eris_ifu_vector *kv2);

cpl_error_code  eris_ifu_vector_add(
                                eris_ifu_vector *kv1,
                                const eris_ifu_vector *kv2);
cpl_error_code  eris_ifu_vector_subtract(
                                eris_ifu_vector *kv1,
                                const eris_ifu_vector *kv2);
cpl_error_code  eris_ifu_vector_multiply(
                                eris_ifu_vector *kv1,
                                const eris_ifu_vector *kv2);
cpl_error_code  eris_ifu_vector_divide(
                                eris_ifu_vector *kv1,
                                const eris_ifu_vector *kv2);
cpl_error_code  eris_ifu_vector_add_scalar(
                                eris_ifu_vector *kv,
                                double addend);
cpl_error_code  eris_ifu_vector_subtract_scalar(
                                eris_ifu_vector *kv,
                                double subtrahend);
cpl_error_code  eris_ifu_vector_multiply_scalar(
                                eris_ifu_vector *kv,
                                double factor);
cpl_error_code  eris_ifu_vector_divide_scalar(
                                eris_ifu_vector *kv,
                                double factor);
cpl_error_code  eris_ifu_vector_abs(
                                eris_ifu_vector *kv);

int             eris_ifu_vector_get_size(
                                const eris_ifu_vector *kv);
double          eris_ifu_vector_get_mean(
                                const eris_ifu_vector *kv);
double          eris_ifu_vector_get_median(
                                const eris_ifu_vector *kv,
                                const enum medianType type);
eris_ifu_vector* eris_ifu_vector_cut_percentian(
                                const eris_ifu_vector *kv,
                                double percentage);
double          eris_ifu_vector_get_sum(
                                const eris_ifu_vector *kv);
double          eris_ifu_vector_get_stdev(
                                const eris_ifu_vector *kv);
double          eris_ifu_vector_get_stdev_median(
                                const eris_ifu_vector *kv);
double          eris_ifu_vector_get_max(
                                const eris_ifu_vector *kv,
                                int *pos);
double          eris_ifu_vector_get_min(
                                const eris_ifu_vector *kv,
                                int *pos);

cpl_error_code  eris_ifu_vector_power(
                                eris_ifu_vector *kv,
                                double exponent);

cpl_error_code  eris_ifu_vector_fill(
                                eris_ifu_vector *kv,
                                double val);

cpl_error_code  eris_ifu_vector_flip(
                                eris_ifu_vector* kv);

eris_ifu_vector* eris_ifu_vector_histogram(
                                const eris_ifu_vector *kv,
                                int nbins);

eris_ifu_vector* eris_ifu_vector_load(
                                const char *filename,
                                int position);
cpl_error_code  eris_ifu_vector_save(
                                const eris_ifu_vector *kv,
                                const char *filename,
                                cpl_type_bpp bpp,
                                const cpl_propertylist *pl,
                                unsigned mode,
                                double rej_val);

cpl_error_code  eris_ifu_vector_dump(
                                const eris_ifu_vector *kv);

int eris_ifu_is_nan_or_inf(double A);

cpl_error_code eris_ifu_vector_sqrt(eris_ifu_vector *ev);



/** @} */

#endif
