/* $Id$
 *
 * This file is part of the ERIS/NIX Pipeline
 * Copyright (C) 2025 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include <hdrl.h>
#include <math.h>
#include <string.h>

#include "eris_nix_img_supersky.h"
#include "eris_nix_utils.h"
#include "eris_nix_dfs.h"
#include "eris_pfits.h"
#include "eris_utils.h"
/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/
static hdrl_image * eris_create_sky_flat_first(
    hdrl_imagelist  * image_list,
    cpl_mask        * masks,
    const char      * method,
    const double      kappa,
    const int         niter,
	const cpl_size    plane_id);

static hdrl_image * eris_create_sky_flat_first2(
    located_imagelist     * jitters,
	located_imagelist     ** sky_1st,
    cpl_mask        * mask,
    const char      * method,
    const double      kappa,
    const int         niter,
	const cpl_size    plane_id);

static hdrl_image * eris_create_sky_flat_final(
    hdrl_imagelist  *   image_list,
    cpl_mask        **  masks,
    const char      *   method,
    const double        kappa,
    const int           niter,
	const cpl_size plane_id,
	cpl_image** contrib);

static hdrl_image * eris_create_sky_flat_final2(
	located_imagelist     * jitters,
    located_imagelist     ** sky_2nd,
    cpl_mask        **  masks,
    const char      *   method,
    const double        kappa,
    const int           niter,
	const cpl_size      plane_id);

static cpl_mask * eris_detect_sources_hdrl_catalogue(
    const cpl_image *   image,
	hdrl_parameter* p);

static cpl_mask * eris_dilate_mask(
    const cpl_mask  *   input_mask,
    int                 radius);

static cpl_error_code
eris_image_stats(const hdrl_image *in_image,
                     double l_sig,
                     double u_sig,
                     const cpl_mask *mask,
                     double trim_fraction,
                     double *median,
                     double *mean,
                     double *stdev,
                     double *tmean,
                     double *tstd,
                     double *mad,
                     double *min_val,
                     double *max_val);

static cpl_error_code
eris_first_sky_sub(const hdrl_imagelist* science_images, const hdrl_image* sky_flat_0,
		hdrl_imagelist** skysub_images);

static cpl_error_code
eris_locate_and_mask_sources(hdrl_imagelist* skysub_images, const cpl_image* bpm_image,
		const cpl_parameterlist* parlist,  const char* prefix, cpl_mask**  source_masks);


static cpl_error_code
eris_locate_and_mask_sources2(  located_imagelist     * jitters, const cpl_image* bpm_image,
		const cpl_parameterlist* parlist,  const char* prefix, cpl_mask**  source_masks);

static cpl_error_code
eris_load_data_and_crea_error(const char* fname, const cpl_size kk, const cpl_mask* bad_pixel_map,
		hdrl_imagelist** science_images);

static cpl_error_code
eris_load_data_and_error_simple(const char* fname, const cpl_size kk, const cpl_mask* bad_pixel_map,
		hdrl_imagelist** science_images);

static cpl_error_code
eris_load_and_error_cube(const char* fname, const cpl_mask* bad_pixel_map, cpl_size* kkk,
		hdrl_imagelist** science_images);

static cpl_error_code
eris_load_and_error_cube2(const char* fname, const cpl_mask* bad_pixel_map,
		const char* method, const double kappa, const int niter,
		hdrl_imagelist** science_images);

static cpl_error_code
eris_create_final_skysub_products(const hdrl_imagelist* science_iml, const hdrl_image* sky_flat, cpl_frameset** products);

static hdrl_imagelist*
eris_crea_imagelist_all(cpl_frameset* raw_frames, cpl_mask* bad_pixel_map);

static cpl_boolean
eris_check_format_is_cube_of_same_size(const cpl_frameset * raw_frames);

static cpl_error_code
eris_load_and_error_cube_slice(const char* fname, const cpl_mask* bad_pixel_map, const cpl_size k,
		const cpl_size kkk, hdrl_imagelist** science_images);


static cpl_error_code
eris_load_and_error_cube_robert(const char* fname, const cpl_mask* bad_pixel_map,
		const char* method, const double kappa, const int niter, const cpl_size kkk,
		hdrl_imagelist** science_images);

static cpl_error_code
eris_save_sky_flat_final( const cpl_frameset* raw_frames, const hdrl_image* sky_flat_final, cpl_frameset** product_frames);

static cpl_error_code
eris_handle_simple_format_case(cpl_frameset * frameset,
		const cpl_parameterlist * parlist,
		cpl_frameset * raw_frames,
		const char * recipe_name,
		const char* combine_method,
		const double sigma_clip,
		const int max_iter,
		const cpl_boolean debug_data,
		cpl_mask * bad_pixel_map,
		cpl_frameset ** product_frames);


static cpl_error_code
eris_handle_simple_format_case2(cpl_frameset * frameset,
		const cpl_parameterlist * parlist,
		const char * recipe_name,
		const char* combine_method,
		const double sigma_clip,
		const int max_iter,
		const cpl_boolean debug_data,
		cpl_mask * bad_pixel_map,
		cpl_frameset ** product_frames);

static cpl_error_code
eris_handle_cube_format_case(cpl_frameset * frameset,
				const cpl_parameterlist * parlist,
				cpl_frameset * raw_frames,
				const char * recipe_name,
				const char* combine_method,
				const double sigma_clip,
				const int max_iter,
				const cpl_boolean debug_data,
				cpl_mask * bad_pixel_map,
				cpl_frameset ** product_frames);


static cpl_error_code
eris_handle_cube_format_robert(cpl_frameset * frameset,
				const cpl_parameterlist * parlist,
				cpl_frameset * raw_frames,
				const char * recipe_name,
				const char* combine_method,
				const double sigma_clip,
				const int max_iter,
				const cpl_boolean debug_data,
				cpl_mask * bad_pixel_map,
				cpl_frameset ** product_frames);

static located_imagelist *
eris_create_final_skysub_products2(const hdrl_imagelist* science_iml,  const cpl_frameset* raw_frames,
		const hdrl_image* sky_flat, cpl_image* confidence, cpl_frameset** products);


static located_imagelist *
eris_create_final_skysub_products3(const hdrl_imagelist* science_iml, const cpl_frameset* frameset,
		const cpl_frameset* raw_frames, const hdrl_image* sky_flat, cpl_image* confidence,
		cpl_frameset** products);

static located_imagelist *
eris_create_final_skysub_products_robert(const cpl_frameset* frameset, const cpl_frameset* raw_frames, const hdrl_image* sky_flat,
		cpl_image* confidence, cpl_frameset** products);

static const int debug = 0;
/*-----------------------------------------------------------------------------
                                   Functions
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create and apply super-sky flat to MIR science frames
  @param    frameset        Input frameset
  @param    parlist         Recipe parameters
  @param    product_frames  Output product frameset (pointer to pointer)
  @return   CPL_ERROR_NONE on success

  Main processing function implementing the DPB-28 algorithm:
  1. Load science frames and bad pixel map
  2. Create first-pass super-sky (sigma-clipped combination)
  3. Subtract from each frame
  4. Detect sources and create masks
  5. Create final super-sky with source masks
  6. Subtract final super-sky and save products
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_nix_img_supersky_run(
    cpl_frameset            *   frameset,
    const cpl_parameterlist *   parlist,
	const char* recipe_name,
    cpl_frameset            **  product_frames)
{
    cpl_frameset    *   raw_frames      = NULL;
    const cpl_frame *   bpm_frame       = NULL;
    cpl_mask        *   bad_pixel_map   = NULL;

    char                  * param_name = NULL;
    const char* context = "eris.eris_nix_img_supersky";
    const cpl_parameter* par;
    int                 num_frames      = 0;

    cpl_error_code      error           = CPL_ERROR_NONE;

    cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(product_frames, CPL_ERROR_NULL_INPUT);

    /* Extract parameters */
    param_name = cpl_sprintf("%s.combine_method", context);
    par = cpl_parameterlist_find_const(parlist, param_name);
    const char* combine_method = cpl_parameter_get_string(par);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.sigma_clip", context);
    par = cpl_parameterlist_find_const(parlist, param_name);
    const double sigma_clip = cpl_parameter_get_double(par);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.max_iter", context);
    par = cpl_parameterlist_find_const(parlist, param_name);
    const int max_iter = cpl_parameter_get_int(par);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.save_skysub", context);
    par = cpl_parameterlist_find_const(parlist, param_name);
    const  cpl_boolean save_skysub = cpl_parameter_get_bool(par);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.debug_data", context);
    par = cpl_parameterlist_find_const(parlist, param_name);
    const cpl_boolean debug_data = cpl_parameter_get_bool(par);
    cpl_free(param_name);

    /* Extract science frames */
    raw_frames = eris_dfs_extract_frames_with_tag (frameset, ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG);

    if (!raw_frames || cpl_frameset_get_size(raw_frames) == 0) {
    	raw_frames = eris_dfs_extract_frames_with_tag (frameset, ERIS_NIX_CAL_DET_STD_JITTER_PRO_CATG);
    }
    if (!raw_frames || cpl_frameset_get_size(raw_frames) == 0) {
        cpl_msg_error(cpl_func, "No science frames found with tag %s or %s",
        		ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG, ERIS_NIX_CAL_DET_STD_JITTER_PRO_CATG);
        error = CPL_ERROR_DATA_NOT_FOUND;
        goto cleanup;
    }
    num_frames = cpl_frameset_get_size(raw_frames);
    cpl_msg_info(cpl_func, "Processing %d science frames", num_frames);

    /* Load bad pixel map */
    bpm_frame = cpl_frameset_find_const(frameset, ERIS_NIX_IMG_SUPERSKY_BPM);
    if (!bpm_frame) {
        cpl_msg_warning(cpl_func, "No bad pixel map found with tag %s", ERIS_NIX_IMG_SUPERSKY_BPM);
    }
    cpl_image * bpm_image = NULL;
    //cpl_mask *inp_mask = NULL;
    if(bpm_frame) {
        bpm_image = cpl_image_load(cpl_frame_get_filename(bpm_frame),  CPL_TYPE_INT, 0, 0);

        /* Convert BPM image to mask (0=good, >0=bad) */
        bad_pixel_map = cpl_mask_threshold_image_create(bpm_image, 0.5, DBL_MAX);
        if(debug) {
            cpl_mask_save(bad_pixel_map, "bad_pixel_map.fits", NULL, CPL_IO_DEFAULT);
        }
        //inp_mask = cpl_mask_threshold_image_create(bpm_image, -0.5, 0.5);
    }
    cpl_boolean format_is_cube_of_same_size = CPL_TRUE;
    cpl_boolean test_cubes_slices = CPL_TRUE;
    format_is_cube_of_same_size = eris_check_format_is_cube_of_same_size(raw_frames);

    //char* fname;
    *product_frames = cpl_frameset_new();
    if(!format_is_cube_of_same_size) {
    	num_frames = cpl_frameset_get_size(raw_frames);
    	cpl_msg_info(cpl_func, "Processing1 %d science frames", num_frames);

    	eris_handle_simple_format_case(frameset, parlist, raw_frames, recipe_name, combine_method,
    			                       sigma_clip, max_iter, debug_data, bad_pixel_map, product_frames);

    } else {
    	/*
    	eris_handle_cube_format_case(frameset, parlist, raw_frames, recipe_name, combine_method,
    	    			                     sigma_clip, max_iter, debug_data, bad_pixel_map, product_frames);
    	*/
		eris_handle_cube_format_robert(frameset, parlist, raw_frames, recipe_name, combine_method,
    			                     sigma_clip, max_iter, debug_data, bad_pixel_map, product_frames);

    }

    eris_print_rec_status(700);

    eris_print_rec_status(800);
    cpl_msg_info(cpl_func, "Super-sky recipe completed successfully");

cleanup:
    eris_print_rec_status(600);
    if (raw_frames) cpl_frameset_delete(raw_frames);
    if (bad_pixel_map) cpl_mask_delete(bad_pixel_map);
    //if (object_jitters) enu_located_imagelist_delete(object_jitters);
    //if (sky_1st) enu_located_imagelist_delete(sky_1st);
    //if(inp_mask) cpl_mask_delete(inp_mask);
    //if(bpm_image !=NULL) cpl_image_delete(bpm_image);

    eris_check_error_code(cpl_func);
    return error;
}



static cpl_error_code
eris_handle_simple_format_case(cpl_frameset * frameset,
		const cpl_parameterlist * parlist,
		cpl_frameset * raw_frames,
		const char * recipe_name,
		const char* combine_method,
		const double sigma_clip,
		const int max_iter,
		const cpl_boolean debug_data,
		cpl_mask * bad_pixel_map,
		cpl_frameset ** product_frames){

	hdrl_imagelist  *   science_images  = NULL;
	hdrl_image      *   sky_flat_0      = NULL;
	hdrl_image      *   sky_flat_final  = NULL;
	cpl_mask        **  source_masks    = NULL;
	hdrl_imagelist  *   skysub_images_0 = NULL;
	/* Load science images as HDRL imagelist */
	science_images = eris_crea_imagelist_all(raw_frames, bad_pixel_map);

	/* Step 1: Create first-pass super-sky */
	cpl_msg_info(cpl_func, "Creating first-pass super-sky flat...");
	eris_print_rec_status(100);
	sky_flat_0 = eris_create_sky_flat_first(science_images, bad_pixel_map, combine_method, sigma_clip,
			max_iter, -1);

	eris_print_rec_status(200);

	/* Step 2: Subtract first-pass sky from each frame */
	eris_first_sky_sub(science_images, sky_flat_0, &skysub_images_0);
	if (sky_flat_0) hdrl_image_delete(sky_flat_0);

	eris_print_rec_status(300);
	/* Step 3: Detect sources in sky-subtracted frames */
	cpl_size num_tot_images = hdrl_imagelist_get_size(skysub_images_0);
	cpl_msg_info(cpl_func,"num_tot_images4: %lld",num_tot_images);
	source_masks = cpl_calloc(num_tot_images, sizeof(cpl_mask *));
	cpl_image * bpm_image = NULL;
	char* prefix = cpl_sprintf("%s.catalogue",recipe_name);
	eris_locate_and_mask_sources(skysub_images_0, bpm_image, parlist, prefix, source_masks);
    cpl_free(prefix);
	/* Step 4: Create final super-sky with source masks */
	eris_print_rec_status(400);
	cpl_msg_info(cpl_func, "Creating final super-sky flat with source masking...");
	cpl_image* contrib = NULL;
	sky_flat_final = eris_create_sky_flat_final(science_images, source_masks,
			combine_method, sigma_clip, max_iter, -1, &contrib);

	if (!sky_flat_final) {
		cpl_msg_error(cpl_func, "Failed to create final sky flat");
		return CPL_ERROR_ILLEGAL_OUTPUT;
	}

	if(sky_flat_final) {
		/* Step 5: Save products */
		eris_print_rec_status(500);

		eris_save_sky_flat_final(raw_frames, sky_flat_final, product_frames);
	}

	/* Save sky-subtracted frames if requested */
	//if (debug) {
		//eris_create_final_skysub_products(science_images, sky_flat_final, product_frames);
		located_imagelist * skysub = eris_create_final_skysub_products3(science_images, frameset, raw_frames, sky_flat_final, contrib, product_frames);
		//char * preface = cpl_sprintf("%s_%s", "jitter", "skysub");
		/* save the sky-subtracted jitter images to FITS files */
		for (cpl_size i = 0; i < skysub->size; i++) {

			// add necessary things to output header. RA and DEC are needed as
	        //           would otherwise be overwritten by DFS
			cpl_propertylist * applist = cpl_propertylist_new();
			cpl_propertylist_update_string(applist, CPL_DFS_PRO_CATG, ERIS_NIX_SKYSUB_OBJECT_JITTER_PRO_CATG);
			cpl_propertylist_update_string(applist, "PRODCATG",
					"ANCILLARY.IMAGE");
			cpl_propertylist_copy_property_regexp(applist,
					skysub->limages[i]->plist,
					"RA|DEC",
					CPL_FALSE);

			// set RA, DEC from WCS rather than let dfs copy it from ano†her frame
			{
				cpl_wcs * wcs = cpl_wcs_new_from_propertylist(
						skysub->limages[i]->plist);
				double ra = 0.0;
				double dec = 0.0;
				enu_get_ra_dec(wcs, &ra, &dec);

				cpl_propertylist_update_double(applist, "RA", ra);
				cpl_propertylist_update_double(applist, "DEC", dec);

				cpl_wcs_delete(wcs);
			}

			// Generate output file name and write the file
			char * out_fname = enu_repreface(cpl_frame_get_filename(
					skysub->limages[i]->
					frame),
					"skysub");

			enu_dfs_save_limage(frameset,
					parlist,
					raw_frames,
					CPL_TRUE,
					skysub->limages[i],
					recipe_name,
					skysub->limages[i]->frame,
					applist,
					PACKAGE "/" PACKAGE_VERSION,
					out_fname);

			cpl_free(out_fname);
			cpl_propertylist_delete(applist);
		}


		/*
		char * preface = cpl_sprintf("%s", "");
		cpl_frameset_dump(frameset,stdout);
		enu_debug_limlist_save(debug_data, skysub, preface, recipe_name, frameset, parlist,
				raw_frames);
		cpl_free(preface);
		*/
		if (skysub) enu_located_imagelist_delete(skysub);
	//}


    if(contrib) cpl_image_delete(contrib);
	if (science_images) hdrl_imagelist_delete(science_images);
	if (sky_flat_final) hdrl_image_delete(sky_flat_final);
	if (skysub_images_0) hdrl_imagelist_delete(skysub_images_0);
	if (source_masks != NULL) {
		for (int i = 0; i < num_tot_images; i++) {
			if (source_masks[i]) cpl_mask_delete(source_masks[i]);
		}
		cpl_free(source_masks);
	}

	eris_check_error_code(cpl_func);
	return cpl_error_get_code();
}



static cpl_error_code
eris_handle_simple_format_case2(cpl_frameset * frameset,
		const cpl_parameterlist * parlist,
		const char * recipe_name,
		const char* combine_method,
		const double sigma_clip,
		const int max_iter,
		const cpl_boolean debug_data,
		cpl_mask * bad_pixel_map,
		cpl_frameset ** product_frames){

	hdrl_imagelist  *   science_images  = NULL;
	hdrl_image      *   sky_flat_0      = NULL;
	hdrl_image      *   sky_flat_final  = NULL;
	located_imagelist     * object_jitters = NULL;
	located_imagelist     * sky_1st = NULL;
	cpl_mask        **  source_masks    = NULL;
	hdrl_imagelist  *   skysub_images_0 = NULL;
	cpl_frameset * raw_frames = cpl_frameset_new();
	/* Load science images as HDRL imagelist */


	object_jitters = enu_limlist_load_from_frameset(frameset,
	    			ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG, raw_frames);
	cpl_msg_info(cpl_func, "Processing3 %lld science frames", cpl_frameset_get_size(raw_frames));
	/* Extract science frames */

	if (!object_jitters) {
		object_jitters = enu_limlist_load_from_frameset(frameset,
				ERIS_NIX_CAL_DET_STD_JITTER_PRO_CATG, raw_frames);
	}
	if (!raw_frames || cpl_frameset_get_size(raw_frames) == 0) {
		cpl_msg_error(cpl_func, "No science frames found with tag %s or %s",
				ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG, ERIS_NIX_CAL_DET_STD_JITTER_PRO_CATG);

		return CPL_ERROR_DATA_NOT_FOUND;
	}
	cpl_msg_info(cpl_func,"n0: %lld",cpl_frameset_get_size(raw_frames));
	science_images = eris_crea_imagelist_all(raw_frames, bad_pixel_map);
    cpl_msg_info(cpl_func,"n1: %lld",hdrl_imagelist_get_size(science_images));
	/* Step 1: Create first-pass super-sky */
	cpl_msg_info(cpl_func, "Creating first-pass super-sky flat...");
	/*
	sky_flat_0 = eris_create_sky_flat_first(science_images, bad_pixel_map, combine_method, sigma_clip,
			max_iter, -1);
	 */
	eris_print_rec_status(100);
	/*
	if(debug) {
		char * preface = cpl_sprintf("%s_%s", "jitters", "input");
		enu_debug_limlist_save(debug_data, object_jitters, preface, recipe_name, frameset, parlist,
				raw_frames);
		cpl_free(preface);
	}
	 */


	sky_flat_0 = eris_create_sky_flat_first2(object_jitters, &sky_1st, bad_pixel_map, combine_method, sigma_clip,
			max_iter, -1);
	eris_print_rec_status(150);
	cpl_msg_info(cpl_func,"n2: %lld",object_jitters->size);


	if(debug) {
		char * preface = cpl_sprintf("%s_%s", "jitters", "sky_1st");
		enu_debug_limlist_save(debug_data, sky_1st, preface, recipe_name, frameset, parlist,
				raw_frames);
		cpl_free(preface);
	}


	eris_print_rec_status(170);

	/* Step 2: Subtract first-pass sky from each frame */
	eris_first_sky_sub(science_images, sky_flat_0, &skysub_images_0);
	if (sky_flat_0) hdrl_image_delete(sky_flat_0);
	cpl_msg_info(cpl_func,"n3: %lld",hdrl_imagelist_get_size(science_images));
	cpl_msg_info(cpl_func,"n4: %lld",hdrl_imagelist_get_size(skysub_images_0));

	eris_print_rec_status(300);
	/* Step 3: Detect sources in sky-subtracted frames */
	cpl_size num_tot_images = hdrl_imagelist_get_size(skysub_images_0);
	cpl_msg_info(cpl_func,"num_tot_images1: %lld",num_tot_images);
	source_masks = cpl_calloc(num_tot_images, sizeof(cpl_mask *));
	cpl_image * bpm_image = NULL;
	char* prefix = cpl_sprintf("%s.catalogue",recipe_name);
	//eris_locate_and_mask_sources(skysub_images_0, bpm_image, parlist, recipe_name, source_masks);
	eris_locate_and_mask_sources2(sky_1st, bpm_image, parlist, prefix, source_masks);
    cpl_free(prefix);
	eris_print_rec_status(400);
	/* Step 4: Create final super-sky with source masks */
	cpl_msg_info(cpl_func, "Creating final super-sky flat with source masking...");
	//sky_flat_final = eris_create_sky_flat_final(science_images, source_masks,
	//		combine_method, sigma_clip, max_iter, -1);
	if (sky_1st) enu_located_imagelist_delete(sky_1st);
	located_imagelist* sky_2nd = enu_located_imagelist_duplicate(object_jitters);
	sky_flat_final = eris_create_sky_flat_final2(object_jitters, &sky_2nd, source_masks, combine_method, sigma_clip, max_iter, -1);

	eris_print_rec_status(450);

	eris_print_rec_status(460);

	if (!sky_flat_final) {
		cpl_msg_error(cpl_func, "Failed to create final sky flat");
		return CPL_ERROR_ILLEGAL_OUTPUT;
	}
	/*
	if(debug) {
		char * preface = cpl_sprintf("%s_%s", "jitters", "sky_2nd");
		enu_debug_limlist_save(debug_data, sky_2nd, preface, recipe_name, frameset, parlist,
				raw_frames);
		cpl_free(preface);
	}
	 */

	if(sky_flat_final) {
		/* Step 5: Save products */
		eris_print_rec_status(500);

		eris_save_sky_flat_final(raw_frames, sky_flat_final, product_frames);
	}

	/* Save sky-subtracted frames if requested */
	if (debug) {
		eris_create_final_skysub_products(science_images, sky_flat_final, product_frames);
	}

	if (science_images) hdrl_imagelist_delete(science_images);
	if (sky_flat_final) hdrl_image_delete(sky_flat_final);
	if (skysub_images_0) hdrl_imagelist_delete(skysub_images_0);
	//if (object_jitters) enu_located_imagelist_delete(object_jitters);
	//if (sky_2nd) enu_located_imagelist_delete(sky_2nd);
	if (source_masks != NULL) {
		for (int i = 0; i < num_tot_images; i++) {
			if (source_masks[i]) cpl_mask_delete(source_masks[i]);
		}
		cpl_free(source_masks);
	}
    cpl_frameset_delete(raw_frames);
	eris_check_error_code(cpl_func);
	return cpl_error_get_code();
}


static cpl_error_code
eris_handle_cube_format_case(cpl_frameset * frameset,
				const cpl_parameterlist * parlist,
				cpl_frameset * raw_frames,
				const char * recipe_name,
				const char* combine_method,
				const double sigma_clip,
				const int max_iter,
				const cpl_boolean debug_data,
				cpl_mask * bad_pixel_map,
				cpl_frameset ** product_frames)
{

	hdrl_imagelist  *   science_images  = NULL;
	hdrl_image      *   sky_flat_0      = NULL;
	hdrl_image      *   sky_flat_final  = NULL;
	cpl_mask        **  source_masks    = NULL;
	hdrl_imagelist  *   skysub_images_0 = NULL;
    cpl_msg_warning(cpl_func,"cube format, process slice by slice");

    const cpl_frame * frame = cpl_frameset_get_position_const(raw_frames, 0);
    const char * filename = cpl_frame_get_filename(frame);
    cpl_imagelist * data_iml = cpl_imagelist_load(filename, CPL_TYPE_DOUBLE, 1);
    cpl_size nplanes = cpl_imagelist_get_size(data_iml);
    cpl_size num_frames = cpl_frameset_get_size(raw_frames);
    for(cpl_size k = 0; k < nplanes; k++) {
    	science_images = hdrl_imagelist_new();
    	for(cpl_size kkk = 0; kkk < num_frames; kkk++) {
    		frame = cpl_frameset_get_position_const(raw_frames, kkk);
    		filename = cpl_frame_get_filename(frame);
    	    eris_load_and_error_cube_slice(filename, bad_pixel_map, k, kkk, &science_images);
    	}

    	/* Step 1: Create first-pass super-sky */
    	cpl_msg_info(cpl_func, "Creating first-pass super-sky flat...");
    	sky_flat_0 = eris_create_sky_flat_first(science_images, bad_pixel_map, combine_method, sigma_clip, max_iter, k);

    	/* Step 2: Subtract first-pass sky from each frame */
    	eris_first_sky_sub(science_images, sky_flat_0, &skysub_images_0);

    	eris_print_rec_status(300);
    	/* Step 3: Detect sources in sky-subtracted frames */
    	cpl_size num_tot_images = hdrl_imagelist_get_size(skysub_images_0);
    	cpl_msg_info(cpl_func,"num_tot_images2: %lld",num_tot_images);
    	source_masks = cpl_calloc(num_tot_images, sizeof(cpl_mask *));
    	cpl_image * bpm_image = NULL;
    	char* prefix = cpl_sprintf("%s.catalogue",recipe_name);
        eris_locate_and_mask_sources(skysub_images_0, bpm_image, parlist, prefix, source_masks);
        cpl_free(prefix);
        eris_print_rec_status(400);
        /* Step 4: Create final super-sky with source masks */
        cpl_msg_info(cpl_func, "Creating final super-sky flat with source masking...");
        cpl_image* contrib = NULL;
        sky_flat_final = eris_create_sky_flat_final(science_images, source_masks,
        		combine_method, sigma_clip, max_iter, k, &contrib);

        //if (debug) {
        	eris_create_final_skysub_products(science_images, sky_flat_final, product_frames);
        //}

    	if (sky_flat_0) hdrl_image_delete(sky_flat_0);
    	if (science_images) hdrl_imagelist_delete(science_images);
    	if (sky_flat_final) hdrl_image_delete(sky_flat_final);
    	if (skysub_images_0) hdrl_imagelist_delete(skysub_images_0);
    	if (contrib) cpl_image_delete(contrib);
    	if (source_masks != NULL) {
    		for (int i = 0; i < num_tot_images; i++) {
    			if (source_masks[i]) cpl_mask_delete(source_masks[i]);
    		}
    		cpl_free(source_masks);
    	}
    }
    cpl_imagelist_delete(data_iml);

	eris_check_error_code(cpl_func);
	return cpl_error_get_code();

}

static cpl_error_code
eris_handle_cube_format_robert(cpl_frameset * frameset,
				const cpl_parameterlist * parlist,
				cpl_frameset * raw_frames,
				const char * recipe_name,
				const char* combine_method,
				const double sigma_clip,
				const int max_iter,
				const cpl_boolean debug_data,
				cpl_mask * bad_pixel_map,
				cpl_frameset ** product_frames)
{

	hdrl_imagelist  *   science_images  = NULL;
	hdrl_image      *   sky_flat_0      = NULL;
	hdrl_image      *   sky_flat_final  = NULL;
	cpl_mask        **  source_masks    = NULL;
	hdrl_imagelist  *   skysub_images_0 = NULL;
    cpl_msg_warning(cpl_func,"cube format, collapse cubes and combine images");

    const cpl_frame * frame = cpl_frameset_get_position_const(raw_frames, 0);
    const char * filename = cpl_frame_get_filename(frame);


    cpl_size num_frames = cpl_frameset_get_size(raw_frames);


    science_images = hdrl_imagelist_new();
    for(cpl_size kkk = 0; kkk < num_frames; kkk++) {
    	frame = cpl_frameset_get_position_const(raw_frames, kkk);
    	filename = cpl_frame_get_filename(frame);

    	eris_load_and_error_cube_robert(filename, bad_pixel_map, combine_method, sigma_clip, max_iter, kkk,
    			&science_images);

    }

    /* Step 1: Create first-pass super-sky */
    cpl_msg_info(cpl_func, "Creating first-pass super-sky flat...");
    sky_flat_0 = eris_create_sky_flat_first(science_images, bad_pixel_map, combine_method, sigma_clip, max_iter, 0);

    /* Step 2: Subtract first-pass sky from each frame */
    eris_first_sky_sub(science_images, sky_flat_0, &skysub_images_0);
    if (sky_flat_0) hdrl_image_delete(sky_flat_0);

    eris_print_rec_status(300);
    /* Step 3: Detect sources in sky-subtracted frames */
    cpl_size num_tot_images = hdrl_imagelist_get_size(skysub_images_0);
    cpl_msg_info(cpl_func,"num_tot_images2: %lld",num_tot_images);
    source_masks = cpl_calloc(num_tot_images, sizeof(cpl_mask *));
    cpl_image * bpm_image = NULL;
    char* prefix = cpl_sprintf("%s.catalogue",recipe_name);
    eris_locate_and_mask_sources(skysub_images_0, bpm_image, parlist, prefix, source_masks);
    hdrl_imagelist_delete(skysub_images_0);
    cpl_free(prefix);

    eris_print_rec_status(400);
    /* Step 4: Create final super-sky with source masks */
    cpl_msg_info(cpl_func, "Creating final super-sky flat with source masking...");
    cpl_image* contrib = NULL;
    sky_flat_final = eris_create_sky_flat_final(science_images, source_masks,
    		combine_method, sigma_clip, max_iter, 0, &contrib);

    if (science_images) hdrl_imagelist_delete(science_images);
    if (source_masks != NULL) {
    	for (int i = 0; i < num_tot_images; i++) {
    		if (source_masks[i]) cpl_mask_delete(source_masks[i]);
    	}
    	cpl_free(source_masks);
    }

    eris_create_final_skysub_products_robert(frameset, raw_frames, sky_flat_final, contrib, product_frames);
    //eris_create_final_skysub_products(science_images, sky_flat_final, product_frames);
    //cpl_msg_info(cpl_func,"about to exit");

    if (sky_flat_final) hdrl_image_delete(sky_flat_final);
    if (contrib) cpl_image_delete(contrib);

	eris_check_error_code(cpl_func);
	return cpl_error_get_code();

}

static cpl_error_code
eris_handle_cube_format_case2(cpl_frameset * frameset,
				const cpl_parameterlist * parlist,
				cpl_frameset * raw_frames,
				const char * recipe_name,
				const char* combine_method,
				const double sigma_clip,
				const int max_iter,
				const cpl_boolean debug_data,
				cpl_mask * bad_pixel_map,
				cpl_frameset ** product_frames)
{

	hdrl_imagelist  *   science_images  = NULL;
	hdrl_image      *   sky_flat_0      = NULL;
	hdrl_image      *   sky_flat_final  = NULL;
	cpl_mask        **  source_masks    = NULL;
	hdrl_imagelist  *   skysub_images_0 = NULL;
    cpl_msg_warning(cpl_func,"cube format, process slice by slice");

    const cpl_frame * frame = cpl_frameset_get_position_const(raw_frames, 0);
    const char * filename = cpl_frame_get_filename(frame);
    cpl_imagelist * data_iml = cpl_imagelist_load(filename, CPL_TYPE_DOUBLE, 1);
    cpl_size nplanes = cpl_imagelist_get_size(data_iml);
    cpl_size num_frames = cpl_frameset_get_size(raw_frames);
    for(cpl_size k = 0; k < nplanes; k++) {
    	science_images = hdrl_imagelist_new();
    	for(cpl_size kkk = 0; kkk < num_frames; kkk++) {
    		frame = cpl_frameset_get_position_const(raw_frames, kkk);
    		filename = cpl_frame_get_filename(frame);
    	    eris_load_and_error_cube_slice(filename, bad_pixel_map, k, kkk, &science_images);
    	}

    	/* Step 1: Create first-pass super-sky */
    	cpl_msg_info(cpl_func, "Creating first-pass super-sky flat...");
    	sky_flat_0 = eris_create_sky_flat_first(science_images, bad_pixel_map, combine_method, sigma_clip, max_iter, k);

    	/* Step 2: Subtract first-pass sky from each frame */
    	eris_first_sky_sub(science_images, sky_flat_0, &skysub_images_0);

    	eris_print_rec_status(300);
    	/* Step 3: Detect sources in sky-subtracted frames */
    	cpl_size num_tot_images = hdrl_imagelist_get_size(skysub_images_0);
    	cpl_msg_info(cpl_func,"num_tot_images2: %lld",num_tot_images);
    	source_masks = cpl_calloc(num_tot_images, sizeof(cpl_mask *));
    	cpl_image * bpm_image = NULL;
    	char* prefix = cpl_sprintf("%s.catalogue",recipe_name);
        eris_locate_and_mask_sources(skysub_images_0, bpm_image, parlist, prefix, source_masks);
        cpl_free(prefix);
        eris_print_rec_status(400);
        /* Step 4: Create final super-sky with source masks */
        cpl_msg_info(cpl_func, "Creating final super-sky flat with source masking...");
        cpl_image* contrib = NULL;
        sky_flat_final = eris_create_sky_flat_final(science_images, source_masks,
        		combine_method, sigma_clip, max_iter, k, &contrib);

        //if (debug) {
        	eris_create_final_skysub_products(science_images, sky_flat_final, product_frames);
        //}

    	if (sky_flat_0) hdrl_image_delete(sky_flat_0);
    	if (science_images) hdrl_imagelist_delete(science_images);
    	if (sky_flat_final) hdrl_image_delete(sky_flat_final);
    	if (skysub_images_0) hdrl_imagelist_delete(skysub_images_0);
    	if (contrib) cpl_image_delete(contrib);
    	if (source_masks != NULL) {
    		for (int i = 0; i < num_tot_images; i++) {
    			if (source_masks[i]) cpl_mask_delete(source_masks[i]);
    		}
    		cpl_free(source_masks);
    	}
    }
    cpl_imagelist_delete(data_iml);

	eris_check_error_code(cpl_func);
	return cpl_error_get_code();

}



static cpl_error_code
eris_save_sky_flat_final( const cpl_frameset* raw_frames, const hdrl_image* sky_flat_final, cpl_frameset** product_frames)
{

	/* Step 5: Save products */
	cpl_size num_frames = cpl_frameset_get_size(raw_frames);
	cpl_msg_info(cpl_func,"num_frames:%lld",num_frames);
	eris_print_rec_status(500);
	//*product_frames = cpl_frameset_new();
	hdrl_value med_sky_final = hdrl_image_get_median(sky_flat_final);

	/* Save master sky flat */
	cpl_propertylist* plist = cpl_propertylist_load(cpl_frame_get_filename(
			cpl_frameset_get_position_const(raw_frames, 0)), 0);
	cpl_propertylist_update_string(plist, CPL_DFS_PRO_CATG, ERIS_NIX_IMG_SUPERSKY_SKYFLAT);
	cpl_propertylist_update_double(plist, "ESO QC SKY MEDIAN", med_sky_final.data);
	cpl_propertylist_update_int(plist, "ESO QC NFRAMES", num_frames);
	eris_print_rec_status(600);
	char* fname = cpl_sprintf("sky_flat_2nd.fits");

	cpl_image_save(hdrl_image_get_image_const(sky_flat_final), fname,CPL_TYPE_DOUBLE, plist, CPL_IO_CREATE);
	cpl_image_save(hdrl_image_get_error_const(sky_flat_final), fname,CPL_TYPE_DOUBLE, plist, CPL_IO_EXTEND);
	cpl_mask_save(hdrl_image_get_mask_const(sky_flat_final), fname, plist, CPL_IO_EXTEND);

	cpl_frame * product = cpl_frame_new();
	cpl_frame_set_filename(product, fname);
	cpl_frame_set_tag(product, ERIS_NIX_IMG_SUPERSKY_SKYFLAT);
	cpl_frame_set_type(product, CPL_FRAME_TYPE_IMAGE);
	cpl_frame_set_group(product, CPL_FRAME_GROUP_PRODUCT);
	cpl_frameset_insert(*product_frames, product);

	cpl_free(fname);
	cpl_propertylist_delete(plist);

	eris_check_error_code(cpl_func);
	return cpl_error_get_code();

}

static cpl_error_code
eris_save_sky_flat_final2( const cpl_frameset* raw_frames, const hdrl_image* sky_flat_final, cpl_frameset** product_frames)
{

	/* Step 5: Save products */
	cpl_size num_frames = cpl_frameset_get_size(raw_frames);
	eris_print_rec_status(500);
	//*product_frames = cpl_frameset_new();
	hdrl_value med_sky_final = hdrl_image_get_median(sky_flat_final);

	/* Save master sky flat */
	cpl_propertylist* plist = cpl_propertylist_load(cpl_frame_get_filename(
			cpl_frameset_get_position_const(raw_frames, 0)), 0);
	cpl_propertylist_update_string(plist, CPL_DFS_PRO_CATG, ERIS_NIX_IMG_SUPERSKY_SKYFLAT);
	cpl_propertylist_update_double(plist, "ESO QC SKY MEDIAN", med_sky_final.data);
	cpl_propertylist_update_int(plist, "ESO QC NFRAMES", num_frames);
	eris_print_rec_status(600);
	char* fname = cpl_sprintf("sky_flat_2nd.fits");

	cpl_image_save(hdrl_image_get_image_const(sky_flat_final), fname,CPL_TYPE_DOUBLE, plist, CPL_IO_CREATE);
	cpl_image_save(hdrl_image_get_error_const(sky_flat_final), fname,CPL_TYPE_DOUBLE, plist, CPL_IO_EXTEND);
	cpl_mask_save(hdrl_image_get_mask_const(sky_flat_final), fname, plist, CPL_IO_EXTEND);

	cpl_frame * product = cpl_frame_new();
	cpl_frame_set_filename(product, fname);
	cpl_frame_set_tag(product, ERIS_NIX_IMG_SUPERSKY_SKYFLAT);
	cpl_frame_set_type(product, CPL_FRAME_TYPE_IMAGE);
	cpl_frame_set_group(product, CPL_FRAME_GROUP_PRODUCT);
	cpl_frameset_insert(*product_frames, product);

	cpl_free(fname);
	cpl_propertylist_delete(plist);

	eris_check_error_code(cpl_func);
	return cpl_error_get_code();

}

static hdrl_imagelist*
eris_crea_imagelist_all(cpl_frameset* raw_frames, cpl_mask* bad_pixel_map)
{

	/* Load science images as HDRL imagelist */
	hdrl_imagelist* science_images = hdrl_imagelist_new();
	cpl_size num_frames = cpl_frameset_get_size(raw_frames);
	cpl_size next = 0;
	const char* format = NULL;
	cpl_propertylist* pheader = NULL;
	eris_print_rec_status(100);
	cpl_size kkk = 0;
	for (cpl_size kk = 0; kk < num_frames; kk++) {
		const cpl_frame * frame = cpl_frameset_get_position_const(raw_frames, kk);
		const char * filename = cpl_frame_get_filename(frame);
		next = cpl_frame_get_nextensions(frame);
		pheader = cpl_propertylist_load(filename,0);
		format = eris_pfits_get_frame_format(pheader);

		if (next  < 4) {
			eris_load_data_and_crea_error(filename, kk, bad_pixel_map, &science_images);
		} else {
			cpl_msg_info(cpl_func,"sci data with extentions %s", filename);
			if(!strcmp(format,"single")) {
				eris_load_data_and_error_simple(filename, kk, bad_pixel_map, &science_images);
			} else {
				eris_load_and_error_cube(filename, bad_pixel_map, &kkk, &science_images);
			}/* end cube format case */
		} /* end case with extensions */
		cpl_propertylist_delete(pheader);
	} /* end loop over input frames */

	cpl_size nx = hdrl_image_get_size_x(hdrl_imagelist_get_const(science_images, 0));
	cpl_size ny = hdrl_image_get_size_y(hdrl_imagelist_get_const(science_images, 0));
	cpl_msg_info(cpl_func, "Image dimensions: %lld x %lld", nx, ny);


	eris_check_error_code(cpl_func);
    return science_images;
}

static cpl_boolean
eris_check_format_is_cube_of_same_size(const cpl_frameset * raw_frames)
{

	cpl_boolean result = CPL_TRUE;
	cpl_size nplanes = 0;
	cpl_size nplanes_tmp = 0;

	cpl_size num_frames = cpl_frameset_get_size(raw_frames);
	cpl_size next = 0;
	for (int kk = 0; kk < num_frames; kk++) {
		const cpl_frame * frame = cpl_frameset_get_position_const(raw_frames, kk);
		next = cpl_frame_get_nextensions(frame);

		if(next > 3) {
			const char * fname = cpl_frame_get_filename(frame);

			cpl_propertylist* xhead = cpl_propertylist_load(fname,1);
			if(cpl_propertylist_has(xhead,"CD3_3")) {
				cpl_imagelist * data_iml = cpl_imagelist_load(fname, CPL_TYPE_DOUBLE, 1);

				nplanes_tmp = cpl_imagelist_get_size(data_iml);
				if(kk == 0) {
					nplanes = nplanes_tmp;
				} else if (nplanes_tmp != nplanes) {
					cpl_msg_info(cpl_func,"cubes not of same size");
					result = CPL_FALSE;
				}
				cpl_imagelist_delete(data_iml);
			} else {
				cpl_msg_info(cpl_func,"Input Data are simple format (images)");
				cpl_propertylist_delete(xhead);
			    return CPL_FALSE;
			}
			cpl_propertylist_delete(xhead);
		} else {
			cpl_msg_info(cpl_func,"Data with less than 4 extensions");
			return CPL_FALSE;
		}
	}
	eris_check_error_code(cpl_func);
	return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    save products
  @param    science_iml HDRL image list
  @param    sky_flat input sky flat
  @param    products frameset to instert products

  @return   error code
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code
eris_create_final_skysub_products(const hdrl_imagelist* science_iml, const hdrl_image* sky_flat, cpl_frameset** products)
{

	cpl_msg_info(cpl_func, "Saving sky-subtracted frames...");
	cpl_size num_tot_images = hdrl_imagelist_get_size(science_iml);
	hdrl_value med_sky = hdrl_image_get_median(sky_flat);

	for (int i = 0; i < num_tot_images; i++) {

		const hdrl_image * himg = hdrl_imagelist_get_const(science_iml, i);
		hdrl_value med_img = hdrl_image_get_median(himg);
		hdrl_value scale = {0, 0};
		scale.data = med_img.data / med_sky.data;

		hdrl_image * skysub = hdrl_image_duplicate(himg);
		hdrl_image * scaled_sky = hdrl_image_duplicate(sky_flat);
		hdrl_image_mul_scalar(scaled_sky, scale);
		hdrl_image_sub_image(skysub, scaled_sky);
		hdrl_value med_skyub = hdrl_image_get_median(skysub);
		/* Create output filename */
		char outfile[256];
		snprintf(outfile, 256, "sci_skysub_2nd_%03d.fits", i);

		/* Load and update header */
		cpl_msg_info(cpl_func,"ESO QC SKY_MED: %g",med_img.data);
		cpl_msg_info(cpl_func,"ESO QC SKYSUB_MED: %g", med_skyub.data);
		/*
        const cpl_frame * frame = cpl_frameset_get_position_const(raw_frames, i);
        plist = cpl_propertylist_load(cpl_frame_get_filename(frame), 0);
        cpl_propertylist_update_string(plist, CPL_DFS_PRO_CATG,
                                       ERIS_NIX_IMG_SUPERSKY_SKYSUB);

        //cpl_propertylist_update_double(plist, "ESO QC SKY_MED", med_img.data);
        //cpl_propertylist_update_double(plist, "ESO QC SKYSUB_MED", med_skyub.data);

		 */

		cpl_image_save(hdrl_image_get_image(skysub), outfile, CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
		cpl_image_save(hdrl_image_get_error(skysub), outfile, CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
		cpl_mask_save(hdrl_image_get_mask(skysub), outfile, NULL, CPL_IO_EXTEND);

		cpl_frame * product = cpl_frame_new();
		cpl_frame_set_filename(product, outfile);
		cpl_frame_set_tag(product, ERIS_NIX_IMG_SUPERSKY_SKYSUB);
		cpl_frame_set_type(product, CPL_FRAME_TYPE_IMAGE);
		cpl_frame_set_group(product, CPL_FRAME_GROUP_PRODUCT);
		cpl_frameset_insert(*products, product);
		//cpl_propertylist_delete(plist);
		hdrl_image_delete(skysub);
		hdrl_image_delete(scaled_sky);
	}

	eris_check_error_code(cpl_func);
	return cpl_error_get_code();
}





static located_imagelist *
eris_create_final_skysub_products2(const hdrl_imagelist* science_iml, const cpl_frameset* raw_frames, const hdrl_image* sky_flat,
		cpl_image* confidence, cpl_frameset** products)
{

	cpl_msg_info(cpl_func, "Saving sky-subtracted frames...");
	cpl_size num_tot_images = hdrl_imagelist_get_size(science_iml);
	hdrl_value med_sky = hdrl_image_get_median(sky_flat);
	located_imagelist     * result = enu_located_imagelist_new(num_tot_images);
	for (int i = 0; i < num_tot_images; i++) {

		const hdrl_image * himg = hdrl_imagelist_get_const(science_iml, i);
		hdrl_value med_img = hdrl_image_get_median(himg);
		hdrl_value scale = {0, 0};
		scale.data = med_img.data / med_sky.data;

		hdrl_image * skysub = hdrl_image_duplicate(himg);
		hdrl_image * scaled_sky = hdrl_image_duplicate(sky_flat);
		hdrl_image_mul_scalar(scaled_sky, scale);
		hdrl_image_sub_image(skysub, scaled_sky);
		hdrl_value med_skyub = hdrl_image_get_median(skysub);
		/* Create output filename */
		char outfile[256];
		snprintf(outfile, 256, "ERIS.2024.sci_skysub_2nd_%03d.fits", i);

		/* Load and update header */
		cpl_msg_info(cpl_func,"ESO QC SKY_MED: %g",med_img.data);
		cpl_msg_info(cpl_func,"ESO QC SKYSUB_MED: %g", med_skyub.data);
		/*
        const cpl_frame * frame = cpl_frameset_get_position_const(raw_frames, i);
        plist = cpl_propertylist_load(cpl_frame_get_filename(frame), 0);
        cpl_propertylist_update_string(plist, CPL_DFS_PRO_CATG,
                                       ERIS_NIX_IMG_SUPERSKY_SKYSUB);

        //cpl_propertylist_update_double(plist, "ESO QC SKY_MED", med_img.data);
        //cpl_propertylist_update_double(plist, "ESO QC SKYSUB_MED", med_skyub.data);

		 */
		cpl_frame * frame = cpl_frameset_get_position(raw_frames, i);
		cpl_propertylist* plist = cpl_propertylist_load(cpl_frame_get_filename(frame), 0);
		cpl_propertylist* dhead = cpl_propertylist_load(cpl_frame_get_filename(frame), 1);
		cpl_propertylist* ehead = cpl_propertylist_load(cpl_frame_get_filename(frame), 2);
		cpl_propertylist* qhead = cpl_propertylist_load(cpl_frame_get_filename(frame), 3);
		cpl_propertylist_save(plist, outfile, CPL_IO_CREATE);
		cpl_image_save(hdrl_image_get_image(skysub), outfile, CPL_TYPE_DOUBLE, dhead, CPL_IO_EXTEND);
		cpl_image_save(hdrl_image_get_error(skysub), outfile, CPL_TYPE_DOUBLE, ehead, CPL_IO_EXTEND);
		cpl_mask_save(hdrl_image_get_mask(skysub), outfile, qhead, CPL_IO_EXTEND);
		cpl_propertylist_delete(dhead);
		cpl_propertylist_delete(ehead);
		cpl_propertylist_delete(qhead);




		cpl_frame * product = cpl_frame_new();
		cpl_frame_set_filename(product, outfile);
		cpl_frame_set_tag(product, ERIS_NIX_IMG_SUPERSKY_SKYSUB);
		cpl_frame_set_type(product, CPL_FRAME_TYPE_IMAGE);
		cpl_frame_set_group(product, CPL_FRAME_GROUP_CALIB);
		cpl_frameset_insert(*products, product);
		located_image * limage = enu_load_limage_from_frame(product, &confidence, CPL_FALSE);
		limage->bkg = hdrl_image_duplicate(scaled_sky);
		limage->frame = product;
		//limage->confidence = cpl_image_duplicate(confidence);
		//limage->plist = cpl_propertylist_duplicate(plist);
		enu_located_imagelist_insert(result,limage, i);

		cpl_propertylist_delete(plist);
		hdrl_image_delete(skysub);
		hdrl_image_delete(scaled_sky);
	}

	eris_check_error_code(cpl_func);
	return result;
}

static located_imagelist *
eris_create_final_skysub_products3(const hdrl_imagelist* science_iml, const cpl_frameset* frameset,
		const cpl_frameset* raw_frames, const hdrl_image* sky_flat,
		cpl_image* confidence, cpl_frameset** products)
{

	cpl_msg_info(cpl_func, "Saving sky-subtracted frames...");
	cpl_size num_tot_images = hdrl_imagelist_get_size(science_iml);
	hdrl_value med_sky = hdrl_image_get_median(sky_flat);
	located_imagelist* object_jitters = NULL;
	//cpl_frameset_dump(frameset,stdout);
	cpl_size sx = hdrl_image_get_size_x(sky_flat);
	cpl_size sy = hdrl_image_get_size_y(sky_flat);
	if(cpl_frameset_count_tags(frameset, ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG) > 0) {
		object_jitters = enu_limlist_load_from_frameset(frameset,
				ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG, raw_frames);
	} else if(cpl_frameset_count_tags(frameset, ERIS_NIX_CAL_DET_STD_JITTER_PRO_CATG) > 0) {
		object_jitters = enu_limlist_load_from_frameset(frameset,
				ERIS_NIX_CAL_DET_STD_JITTER_PRO_CATG, raw_frames);
	}
    //cpl_msg_info(cpl_func,"object_jitters: %p",object_jitters);
	//cpl_size num_tot_images = object_jitters->size;
	cpl_size edge_llx = 50;
	cpl_size edge_lly = 50;
	cpl_size edge_urx = 50;
	cpl_size edge_ury = 50;
	cpl_mask* omask = cpl_mask_new(sx, sy);
	cpl_binary* pmask =  cpl_mask_get_data(omask);

	/* flag left side Y edge  */
	for(cpl_size j = 0; j < edge_lly; j++) {
		for(cpl_size i = 0; i < sx; i++) {
			pmask[i+j*sx] = CPL_BINARY_1;
		}
	}
	/* flag right side Y edge  */
	for(cpl_size j = sy - edge_ury; j < sy; j++) {
		for(cpl_size i = 0; i < sx; i++) {
			pmask[i+j*sx] = CPL_BINARY_1;
		}
	}

	/* flag bottom side X edge  */
	for(cpl_size j = 0; j < sy; j++) {
		for(cpl_size i = 0; i < edge_llx; i++) {
			pmask[i+j*sx] = CPL_BINARY_1;
		}
	}
	/* flag upper side X edge  */
	for(cpl_size j = 0; j < sy; j++) {
		for(cpl_size i = sx - edge_urx; i < sx; i++) {
			pmask[i+j*sx] = CPL_BINARY_1;
		}
	}

	for (cpl_size k = 0; k < num_tot_images; k++) {

		const hdrl_image * himg = hdrl_imagelist_get_const(science_iml, k);
		hdrl_value med_img = hdrl_image_get_median(himg);
		hdrl_value scale = {0, 0};
		scale.data = med_img.data / med_sky.data;
		cpl_msg_warning(cpl_func,"Final frame: %lld med_img.data: %g med_sky_0.data: %g scale.data: %g",
		                		k, med_img.data, med_sky.data, scale.data);
		hdrl_image * skysub = hdrl_image_duplicate(himg);
		hdrl_image * scaled_sky = hdrl_image_duplicate(sky_flat);
		hdrl_image_mul_scalar(scaled_sky, scale);
		hdrl_image_sub_image(skysub, scaled_sky);
		hdrl_value med_skyub = hdrl_image_get_median(skysub);
		/* Create output filename */
		char outfile[256];
		snprintf(outfile, 256, "sci_skysub_2nd_%03lld.fits", k);

		/* Load and update header */
		cpl_msg_info(cpl_func,"ESO QC SKY_MED: %g",med_img.data);
		cpl_msg_info(cpl_func,"ESO QC SKYSUB_MED: %g", med_skyub.data);

        /*

		cpl_frame * frame = cpl_frameset_get_position(raw_frames, i);
		cpl_propertylist* plist = cpl_propertylist_load(cpl_frame_get_filename(frame), 0);
		cpl_propertylist* dhead = cpl_propertylist_load(cpl_frame_get_filename(frame), 1);
		cpl_propertylist* ehead = cpl_propertylist_load(cpl_frame_get_filename(frame), 2);
		cpl_propertylist* qhead = cpl_propertylist_load(cpl_frame_get_filename(frame), 3);
		cpl_propertylist_save(plist, outfile, CPL_IO_CREATE);
		cpl_image_save(hdrl_image_get_image(skysub), outfile, CPL_TYPE_DOUBLE, dhead, CPL_IO_EXTEND);
		cpl_image_save(hdrl_image_get_error(skysub), outfile, CPL_TYPE_DOUBLE, ehead, CPL_IO_EXTEND);
		cpl_mask_save(hdrl_image_get_mask(skysub), outfile, qhead, CPL_IO_EXTEND);
		cpl_propertylist_delete(dhead);
		cpl_propertylist_delete(ehead);
		cpl_propertylist_delete(qhead);
		*/

        if(object_jitters->limages[k]->himage) {
        	hdrl_image_delete(object_jitters->limages[k]->himage);
        	object_jitters->limages[k]->himage=hdrl_image_duplicate(skysub);
        }

		if(object_jitters->limages[k]->bkg) {
			hdrl_image_delete(object_jitters->limages[k]->bkg);
			object_jitters->limages[k]->bkg=hdrl_image_duplicate(scaled_sky);
		}

		if(object_jitters->limages[k]->confidence) {
			cpl_image_delete(object_jitters->limages[k]->confidence);
			object_jitters->limages[k]->confidence=cpl_image_duplicate(confidence);
		}
		int* pconf = cpl_image_get_data_int(object_jitters->limages[k]->confidence);
		object_jitters->limages[k]->object_mask = cpl_mask_duplicate(omask);

		/* flag left side Y edge  */
		for(cpl_size j = 0; j < edge_lly; j++) {
			for(cpl_size i = 0; i < sx; i++) {
				pconf[i+j*sx] = 0;
			}
		}
		/* flag right side Y edge  */
		for(cpl_size j = sy - edge_ury; j < sy; j++) {
			for(cpl_size i = 0; i < sx; i++) {
				pconf[i+j*sx] = 0;
			}
		}

		/* flag bottom side X edge  */
		for(cpl_size j = 0; j < sy; j++) {
			for(cpl_size i = 0; i < edge_llx; i++) {
				pconf[i+j*sx] = 0;
			}
		}
		/* flag upper side X edge  */
		for(cpl_size j = 0; j < sy; j++) {
			for(cpl_size i = sx - edge_urx; i < sx; i++) {
				pconf[i+j*sx] = 0;
			}
		}

		hdrl_image_delete(skysub);
		hdrl_image_delete(scaled_sky);
	}
	cpl_mask_delete(omask);
	eris_check_error_code(cpl_func);
	return object_jitters;
}



static located_imagelist *
eris_create_final_skysub_products_robert(const cpl_frameset* frameset, const cpl_frameset* raw_frames, const hdrl_image* sky_flat,
		cpl_image* confidence, cpl_frameset** products)
{

	cpl_msg_info(cpl_func, "Saving sky-subtracted frames...");
	cpl_size num_frames = cpl_frameset_get_size(raw_frames);

	hdrl_value med_sky = hdrl_image_get_median(sky_flat);
	located_imagelist* object_jitters = NULL;

	cpl_size sx = hdrl_image_get_size_x(sky_flat);
	cpl_size sy = hdrl_image_get_size_y(sky_flat);


	cpl_mask* omask = cpl_mask_new(sx, sy);
	cpl_binary* pmask =  cpl_mask_get_data(omask);

	located_image         * limage = NULL;
	cpl_image             * copyconf = NULL;
	cpl_size collapse_cube = 0;
	cpl_frameset_iterator * frameset_iter =  cpl_frameset_iterator_new(raw_frames);
	cpl_size k = 0;
	const char* fname;
	cpl_propertylist* plist = NULL;
	cpl_propertylist* hdata = NULL;
	cpl_propertylist* herrs = NULL;
	cpl_propertylist* hqual = NULL;
	cpl_size ext_prim = 0;
	cpl_size ext_data = 1;
	cpl_size ext_errs = 2;
	cpl_size ext_qual = 3;
	cpl_image* data = NULL;
	cpl_image* errs = NULL;
	cpl_image* qual = NULL;
	hdrl_image* hima = NULL;

	/*
	if(cpl_frameset_count_tags(frameset, ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG) > 0) {
		object_jitters = enu_limlist_load_from_frameset(frameset,
				ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG, raw_frames);
	} else if(cpl_frameset_count_tags(frameset, ERIS_NIX_CAL_DET_STD_JITTER_PRO_CATG) > 0) {
		object_jitters = enu_limlist_load_from_frameset(frameset,
				ERIS_NIX_CAL_DET_STD_JITTER_PRO_CATG, raw_frames);
	}
	*/



	for (cpl_size iframe = 0; iframe < num_frames; iframe++) {
		cpl_frame * frame = cpl_frameset_iterator_get(frameset_iter);
		fname = cpl_frame_get_filename(frame);
		plist = cpl_propertylist_load(fname, ext_prim);
		hdata = cpl_propertylist_load(fname, ext_data);
		herrs = cpl_propertylist_load(fname, ext_errs);
		hqual = cpl_propertylist_load(fname, ext_qual);
		cpl_imagelist* data_iml = cpl_imagelist_load(fname, CPL_TYPE_DOUBLE, ext_data);
		cpl_imagelist* errs_iml = cpl_imagelist_load(fname, CPL_TYPE_DOUBLE, ext_errs);
		cpl_mask* qual_mask = cpl_mask_load(fname, 0, ext_qual);

		cpl_msg_info(cpl_func, "processing %s", cpl_frame_get_filename(frame));
        hdrl_imagelist* hlist = hdrl_imagelist_new();
		cpl_imagelist* data_list = cpl_imagelist_new();
		cpl_imagelist* errs_list = cpl_imagelist_new();
		cpl_imagelist* qual_list = cpl_imagelist_new();
		cpl_size sz = cpl_imagelist_get_size(data_iml);
		cpl_msg_info(cpl_func, "number of planes %lld", sz);
		char* outfile = NULL;
		for(k = 0; k < sz; k++) {
			data = cpl_imagelist_get(data_iml, k);
			errs = cpl_imagelist_get(errs_iml, k);
			cpl_image_reject_from_mask(data, qual_mask);
			cpl_image_reject_from_mask(errs, qual_mask);
			hima = hdrl_image_create(data, errs);
			hdrl_value med_img = hdrl_image_get_median(hima);
			hdrl_value scale = {0, 0};
			scale.data = med_img.data / med_sky.data;
			cpl_msg_warning(cpl_func,"Final frame: %lld plane %lld med_img.data: %g med_sky_0.data: %g scale.data: %g",
					iframe, k, med_img.data, med_sky.data, scale.data);
			hdrl_image * skysub = hdrl_image_duplicate(hima);
			hdrl_image * scaled_sky = hdrl_image_duplicate(sky_flat);
			hdrl_image_mul_scalar(scaled_sky, scale);
			hdrl_image_sub_image(skysub, scaled_sky);

			hdrl_value med_skyub = hdrl_image_get_median(skysub);

            /*
			if(object_jitters->limages[k]->himage) {
				hdrl_image_delete(object_jitters->limages[k]->himage);
				object_jitters->limages[k]->himage=hdrl_image_duplicate(skysub);
			}

			if(object_jitters->limages[k]->bkg) {
				hdrl_image_delete(object_jitters->limages[k]->bkg);
				object_jitters->limages[k]->bkg=hdrl_image_duplicate(scaled_sky);
			}

			if(object_jitters->limages[k]->confidence) {
				cpl_image_delete(object_jitters->limages[k]->confidence);
				object_jitters->limages[k]->confidence=cpl_image_duplicate(confidence);
			}
			int* pconf = cpl_image_get_data_int(object_jitters->limages[k]->confidence);
			object_jitters->limages[k]->object_mask = cpl_mask_duplicate(omask);
			*/

			/* Create output filename */



			/* Load and update header */
			cpl_msg_info(cpl_func,"ESO QC SKY_MED: %g",med_img.data);
			cpl_msg_info(cpl_func,"ESO QC SKYSUB_MED: %g", med_skyub.data);
			if(debug) {
				if(k == 0) {
					outfile = cpl_sprintf("sci_skysub_2nd_%03lld.fits", iframe);
					cpl_propertylist_save(plist, outfile, CPL_IO_CREATE);
				} else {
					cpl_image_save(hdrl_image_get_image(skysub), outfile, CPL_TYPE_DOUBLE, hdata, CPL_IO_EXTEND);
					cpl_image_save(hdrl_image_get_error(skysub), outfile, CPL_TYPE_DOUBLE, herrs, CPL_IO_EXTEND);
					//cpl_mask_save(hdrl_image_get_mask(skysub), outfile, hqual, CPL_IO_EXTEND);
				}
			}
			hdrl_imagelist_set(hlist, skysub, k);

			hdrl_image_delete(scaled_sky);
			hdrl_image_delete(hima);

		}

		cpl_free(outfile);

		outfile = cpl_sprintf("sci_pro_skysub_2nd_%03lld.fits", iframe);

		for (cpl_size i = 0; i < sz; i++) {
			cpl_imagelist_set(data_list, hdrl_image_get_image(hdrl_imagelist_get(hlist, i)), i);
			cpl_imagelist_set(errs_list, hdrl_image_get_error(hdrl_imagelist_get(hlist, i)), i);
			cpl_image * bpm = cpl_image_new_from_mask(hdrl_image_get_mask(hdrl_imagelist_get(hlist, i)));
			cpl_imagelist_set(qual_list, bpm, i);

		}

		cpl_propertylist_save(plist, outfile, CPL_IO_CREATE);
		cpl_imagelist_save(data_list, outfile, CPL_TYPE_DOUBLE, hdata, CPL_IO_EXTEND);
		cpl_imagelist_save(errs_list, outfile, CPL_TYPE_DOUBLE, herrs, CPL_IO_EXTEND);
		cpl_imagelist_save(qual_list, outfile, CPL_TYPE_INT, herrs, CPL_IO_EXTEND);
		cpl_mask_save(qual_mask, outfile, hqual, CPL_IO_EXTEND);
		cpl_image_save(hdrl_image_get_image(sky_flat), outfile, CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
		cpl_image_save(confidence, outfile, CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);

		cpl_frame * product = cpl_frame_new();
		cpl_frame_set_filename(product, outfile);
		cpl_frame_set_tag(product, ERIS_NIX_IMG_SUPERSKY_SKYFLAT);
		cpl_frame_set_type(product, CPL_FRAME_TYPE_IMAGE);
		cpl_frame_set_group(product, CPL_FRAME_GROUP_PRODUCT);
		 cpl_frame_set_level(product, CPL_FRAME_LEVEL_FINAL);
		cpl_frameset_insert(*products, product);


		cpl_propertylist_delete(plist);
		cpl_propertylist_delete(hdata);
		cpl_propertylist_delete(herrs);
		cpl_propertylist_delete(hqual);

		cpl_imagelist_delete(data_iml);
		cpl_imagelist_delete(errs_iml);
		cpl_mask_delete(qual_mask);
		hdrl_imagelist_delete(hlist);


		for (cpl_size i = sz-1; i >= 0; i--) {
			cpl_imagelist_unset(data_list, i);
			cpl_imagelist_unset(errs_list, i);
			//cpl_imagelist_unset(qual_list, i);
		}

		cpl_imagelist_delete(data_list);
		cpl_imagelist_delete(errs_list);
		cpl_imagelist_delete(qual_list);
        //cpl_msg_info(cpl_func,"about to exit");
		//exit(0);

		cpl_free(outfile);


	}

	cpl_frameset_iterator_delete(frameset_iter);
	cpl_mask_delete(omask);
	eris_check_error_code(cpl_func);

	return object_jitters;
}



/*----------------------------------------------------------------------------*/
/**
  @brief    Create an HDRL imagelist containing data and associated errors
  @param    fname input frame filename
  @param    bad_pixel_map input bad pixel mask
  @param    counter kkk
  @param    science_images output HDRL image list

  @return   error code
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
eris_load_and_error_cube(const char* fname, const cpl_mask* bad_pixel_map, cpl_size* kkk,
		hdrl_imagelist** science_images){

	eris_print_rec_status(0);
	cpl_imagelist * data_iml = cpl_imagelist_load(fname, CPL_TYPE_DOUBLE, 1);
	cpl_imagelist * error_iml = cpl_imagelist_load(fname, CPL_TYPE_DOUBLE, 2);
	cpl_mask * dqual_msk = cpl_mask_load(fname, 0, 3);
	cpl_imagelist * confm_iml = cpl_imagelist_load(fname, CPL_TYPE_DOUBLE, 4);

	double* pconf_img = cpl_image_get_data_double(cpl_imagelist_get(confm_iml,0));

	cpl_binary* pbpm = cpl_mask_get_data(dqual_msk);
	cpl_size sx = cpl_mask_get_size_x(dqual_msk);
	cpl_size sy = cpl_mask_get_size_y(dqual_msk);
	cpl_size sz = cpl_imagelist_get_size(data_iml);

	cpl_mask* mask_tot = cpl_mask_duplicate(dqual_msk);
	if(bad_pixel_map) {
	    cpl_mask_or(mask_tot, bad_pixel_map);
	}
	double* pdata = NULL;
	cpl_image * data_img  = NULL;
	cpl_image * error_img = NULL;
	cpl_size pixel = 0;
	cpl_size j_sx = 0;
	for (cpl_size k = 0; k < sz; k++) {
		data_img = cpl_imagelist_get(data_iml,k);
		error_img = cpl_imagelist_get(error_iml,k);
		pdata = cpl_image_get_data(cpl_imagelist_get(data_iml,k));
		for (cpl_size j = 0; j < sy; j++) {

			j_sx = j * sx;
			for (cpl_size i = 0; i < sx; i++) {

				pixel = i + j_sx;
				if((pconf_img[pixel] == 0) || !isfinite(pdata[pixel])) {
					pbpm[pixel] = CPL_BINARY_1;
				}
			}

		}

		cpl_image_reject_from_mask(data_img, mask_tot);
		cpl_image_reject_from_mask(error_img, mask_tot);
		hdrl_image * himg = hdrl_image_create(data_img, error_img);

		hdrl_imagelist_set(*science_images, himg, *kkk);
		(*kkk)++;
	}

	cpl_mask_delete(mask_tot);
	cpl_mask_delete(dqual_msk);
	cpl_imagelist_delete(data_iml);
	cpl_imagelist_delete(error_iml);
	cpl_imagelist_delete(confm_iml);

	eris_check_error_code(cpl_func);
	return cpl_error_get_code();

}



/*----------------------------------------------------------------------------*/
/**
  @brief    Create an HDRL imagelist containing data and associated errors
  @param    fname input frame filename
  @param    bad_pixel_map input bad pixel mask
  @param    counter kkk
  @param    science_images output HDRL image list

  @return   error code
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
eris_load_and_error_cube_slice(const char* fname, const cpl_mask* bad_pixel_map, const cpl_size k,
		const cpl_size kkk, hdrl_imagelist** science_images){


	cpl_imagelist * data_iml = cpl_imagelist_load(fname, CPL_TYPE_DOUBLE, 1);
	cpl_imagelist * error_iml = cpl_imagelist_load(fname, CPL_TYPE_DOUBLE, 2);
	cpl_mask * dqual_msk = cpl_mask_load(fname, 0, 3);
	cpl_imagelist * confm_iml = cpl_imagelist_load(fname, CPL_TYPE_DOUBLE, 4);

	double* pconf_img = cpl_image_get_data_double(cpl_imagelist_get(confm_iml,0));

	cpl_binary* pbpm = cpl_mask_get_data(dqual_msk);
	cpl_size sx = cpl_mask_get_size_x(dqual_msk);
	cpl_size sy = cpl_mask_get_size_y(dqual_msk);

	cpl_mask* mask_tot = cpl_mask_duplicate(dqual_msk);
    if(bad_pixel_map) {
	    cpl_mask_or(mask_tot, bad_pixel_map);
    }
	double* pdata = NULL;
	cpl_image * data_img  = NULL;
	cpl_image * error_img = NULL;
	cpl_size pixel = 0;
	cpl_size j_sx = 0;

	data_img = cpl_imagelist_get(data_iml,k);
	error_img = cpl_imagelist_get(error_iml,k);
	pdata = cpl_image_get_data(cpl_imagelist_get(data_iml,k));

	for (cpl_size j = 0; j < sy; j++) {

		j_sx = j * sx;
		for (cpl_size i = 0; i < sx; i++) {

			pixel = i + j_sx;
			if((pconf_img[pixel] == 0) || !isfinite(pdata[pixel])) {
				pbpm[pixel] = CPL_BINARY_1;
			}
		}

	}

	cpl_image_reject_from_mask(data_img, mask_tot);
	cpl_image_reject_from_mask(error_img, mask_tot);
	hdrl_image * himg = hdrl_image_create(data_img, error_img);
	hdrl_imagelist_set(*science_images, himg, kkk);

	cpl_mask_delete(mask_tot);
	cpl_mask_delete(dqual_msk);
	cpl_imagelist_delete(data_iml);
	cpl_imagelist_delete(error_iml);
	cpl_imagelist_delete(confm_iml);

	eris_check_error_code(cpl_func);
	return cpl_error_get_code();

}





static cpl_error_code
eris_load_and_error_cube_robert(const char* fname, const cpl_mask* bad_pixel_map,
		const char* method, const double kappa, const int niter, const cpl_size kkk,
		hdrl_imagelist** science_images){


	cpl_imagelist * data_iml = cpl_imagelist_load(fname, CPL_TYPE_DOUBLE, 1);
	cpl_imagelist * error_iml = cpl_imagelist_load(fname, CPL_TYPE_DOUBLE, 2);
	cpl_mask * dqual_msk = cpl_mask_load(fname, 0, 3);
	cpl_imagelist * confm_iml = cpl_imagelist_load(fname, CPL_TYPE_DOUBLE, 4);

	double* pconf_img = cpl_image_get_data_double(cpl_imagelist_get(confm_iml,0));

	cpl_binary* pbpm = cpl_mask_get_data(dqual_msk);

	cpl_size sx = cpl_mask_get_size_x(dqual_msk);
	cpl_size sy = cpl_mask_get_size_y(dqual_msk);
	cpl_size sz = cpl_imagelist_get_size(data_iml);

	cpl_mask* mask_tot = cpl_mask_duplicate(dqual_msk);
    if(bad_pixel_map) {
	    cpl_mask_or(mask_tot, bad_pixel_map);
    }
	double* pdata = NULL;
	cpl_image * data_img  = NULL;
	cpl_image * error_img = NULL;
	cpl_size pixel = 0;
	cpl_size j_sx = 0;
	eris_print_rec_status(10);
	hdrl_imagelist* storage = hdrl_imagelist_new();
	for (cpl_size k = 0; k < sz; k++) {

		eris_print_rec_status(11);
		data_img = cpl_imagelist_get(data_iml,k);
		eris_print_rec_status(12);
		error_img = cpl_imagelist_get(error_iml,k);
		eris_print_rec_status(13);
		pdata = cpl_image_get_data(cpl_imagelist_get(data_iml,k));
		eris_print_rec_status(14);

		for (cpl_size j = 0; j < sy; j++) {

			j_sx = j * sx;
			for (cpl_size i = 0; i < sx; i++) {

				pixel = i + j_sx;
				if((pconf_img[pixel] == 0) || !isfinite(pdata[pixel])) {
					pbpm[pixel] = CPL_BINARY_1;
				}
			}

		}

		cpl_image_reject_from_mask(data_img, mask_tot);
		cpl_image_reject_from_mask(error_img, mask_tot);
		hdrl_image * himg = hdrl_image_create(data_img, error_img);
		hdrl_imagelist_set(storage, himg, k);

	}
	cpl_mask_delete(mask_tot);
	cpl_mask_delete(dqual_msk);
	const hdrl_parameter * collapse_par;

	if (strcmp(method, "median") == 0) {
		collapse_par = hdrl_collapse_median_parameter_create();
	} else {
		cpl_msg_info(cpl_func,"kappa: %g niter: %d",kappa, niter);
		collapse_par = hdrl_collapse_sigclip_parameter_create(kappa, kappa, niter);
	}
	hdrl_image* image_mean;
	cpl_image* conf_map_mean;
	hdrl_imagelist_collapse_mean(storage,&image_mean,&conf_map_mean);
	hdrl_image* collapsed = NULL;
	cpl_image* contribution_map;
	hdrl_imagelist_collapse(storage,collapse_par, &collapsed, &contribution_map);

	hdrl_imagelist_set(*science_images, image_mean, kkk);
	const char* filename = cpl_sprintf("sci_mean_%lld.fits", kkk);
	cpl_image_save(hdrl_image_get_image(image_mean), filename, CPL_TYPE_DOUBLE, NULL, CPL_IO_DEFAULT);
	free(filename);

	filename = cpl_sprintf("sci_mean_clip_%lld.fits", kkk);
	cpl_image_save(hdrl_image_get_image(collapsed), filename, CPL_TYPE_DOUBLE, NULL, CPL_IO_DEFAULT);
	free(filename);

    hdrl_parameter_delete(collapse_par);
	cpl_image_delete(conf_map_mean);
	cpl_image_delete(contribution_map);
    hdrl_image_delete(collapsed);
	cpl_imagelist_delete(data_iml);
	cpl_imagelist_delete(error_iml);
	cpl_imagelist_delete(confm_iml);

    hdrl_imagelist_delete(storage);
	eris_check_error_code(cpl_func);
	return cpl_error_get_code();

}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create an HDRL imagelist containing data and associated errors
  @param    fname input frame filename
  @param    index id
  @param    bad_pixel_map input bad pixel mask
  @param    science_images output HDRL image list

  @return   error code
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
eris_load_data_and_error_simple(const char* fname, const cpl_size kk, const cpl_mask* bad_pixel_map,
		hdrl_imagelist** science_images)
{

	cpl_ensure_code(fname, CPL_ERROR_NULL_INPUT);
	cpl_image * data_img  = cpl_image_load(fname, CPL_TYPE_DOUBLE, 0, 1);
	cpl_image * error_img = cpl_image_load(fname, CPL_TYPE_DOUBLE, 0, 2);
	cpl_mask* dqual_msk = cpl_mask_load(fname, 0, 3);
	cpl_image* confm_img = cpl_image_load(fname, CPL_TYPE_DOUBLE, 0, 4);
	double* pdata = cpl_image_get_data(data_img);
	double* pconf_img = cpl_image_get_data_double(confm_img);
	cpl_binary* pbpm = cpl_mask_get_data(dqual_msk);
	cpl_size sx = cpl_image_get_size_x(confm_img);
	cpl_size sy = cpl_image_get_size_y(confm_img);
    cpl_size j_sx = 0;
    cpl_size pixel = 0;
	for (cpl_size j = 0; j < sy; j++) {

		j_sx = j * sx;
		for (cpl_size i = 0; i < sx; i++) {

			pixel = i + j_sx;
			if((pconf_img[pixel] == 0) || !isfinite(pdata[pixel])) {
				pbpm[pixel] = CPL_BINARY_1;
			}
		}

	}

	cpl_mask* mask_tot = cpl_mask_duplicate(dqual_msk);
	if(bad_pixel_map) {
	    cpl_mask_or(mask_tot, bad_pixel_map);
	}
	cpl_image_reject_from_mask(data_img, mask_tot);
	cpl_image_reject_from_mask(error_img, mask_tot);

	//if(cpl_error_get_code != CPL_ERROR_NONE) {
	//	exit(0);
	//}
	hdrl_image * himg = hdrl_image_create(data_img, error_img);
	hdrl_imagelist_set(*science_images, himg, kk);

	cpl_image_delete(data_img);
	cpl_image_delete(error_img);
	cpl_mask_delete(mask_tot);
    cpl_mask_delete(dqual_msk);
    cpl_image_delete(confm_img);

	// determine statistics of the science images
	cpl_mask* mask = cpl_image_get_bpm(hdrl_image_get_image(himg));
	double median=0, mean=0, stdev=0,tmean=0, tstd=0, mad=0, min_val=0, max_val=0;
	eris_image_stats(
			himg,
			2.0,      // l_sig: lower sigma clip
			2.0,      // u_sig: upper sigma clip
			mask,     // optional mask
			0.1,      // 10% trim fraction
			&median, &mean, &stdev, &tmean, &tstd, &mad, &min_val, &max_val
	);

	cpl_msg_info(cpl_func,"process file: %s", fname);
	cpl_msg_info(cpl_func, "Median=%.3f, Mean=%.3f, StdDev=%.3f "
			"Trimmed: Mean=%.3f, StdDev=%.3f, MAD=%.3f "
			"Range: [%.3f, %.3f]", median, mean, stdev, tmean, tstd, mad, min_val, max_val);

	eris_check_error_code(cpl_func);
	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create an HDRL imagelist containing data and associated errors
  @param    fname input frame filename
  @param    index id
  @param    bad_pixel_map input bad pixel mask
  @param    science_images output HDRL image list

  @return   error code
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
eris_load_data_and_crea_error(const char* fname, const cpl_size kk, const cpl_mask* bad_pixel_map,
		hdrl_imagelist** science_images)
{
	cpl_ensure_code(fname, CPL_ERROR_NULL_INPUT);
    cpl_image * data_img  = NULL;
    cpl_image * error_img = NULL;
	cpl_msg_info(cpl_func,"sci data with less than 4 extensions");
	data_img = cpl_image_load(fname, CPL_TYPE_DOUBLE, 0, 0);
	if (!data_img) {
		cpl_msg_error(cpl_func, "Failed to load frame %s", fname);
		eris_check_error_code(cpl_func);
		return cpl_error_get_code();
	}
	double* pdata = NULL;
	cpl_mask * dqual_msk = NULL;
	pdata = cpl_image_get_data(data_img);
	dqual_msk = cpl_image_get_bpm(data_img);
	cpl_size sx = cpl_image_get_size_x(data_img);
	cpl_size sy = cpl_image_get_size_y(data_img);
	cpl_binary* pbpm = cpl_mask_get_data(dqual_msk);
	cpl_size j_sx = 0;
	cpl_size pixel = 0;
	for (cpl_size j = 0; j < sy; j++) {
		j_sx = j * sx;
		for (cpl_size i = 0; i < sx; i++) {
			pixel = i + j_sx;
			if(!isfinite(pdata[pixel])) {
				pbpm[pixel] = CPL_BINARY_1;
			}
		}

	}
	cpl_mask* mask_tot = cpl_mask_duplicate(dqual_msk);
	if(bad_pixel_map) {
	    cpl_mask_or(mask_tot, bad_pixel_map);
	}
	// Apply bad pixel mask
	cpl_image_reject_from_mask(data_img, mask_tot);

	// Create error image (simple Poisson model for now)
	error_img = cpl_image_duplicate(data_img);
	cpl_image_abs(error_img);
	cpl_image_add_scalar(error_img, 1.0);  // Add readnoise^2
	cpl_image_power(error_img, 0.5);


	hdrl_image * himg = hdrl_image_create(data_img, error_img);
	hdrl_imagelist_set(*science_images, himg, kk);

	cpl_image_delete(data_img);
	cpl_image_delete(error_img);
	cpl_mask_delete(mask_tot);

	eris_check_error_code(cpl_func);
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create array of cpl_masks locating the position of astronomical sources
  @param    skysub_images HDRL images corrected list
  @param    bpm_image   image corresponding to bad pixels
  @param    parlist     input recipe parameters
  @param    recipe_name input recipe name
  @param    source_masks array of cpl_masks with pixel location of objects

  @return   error code
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
eris_locate_and_mask_sources(hdrl_imagelist* skysub_images, const cpl_image* bpm_image,
		const cpl_parameterlist* parlist,  const char* prefix, cpl_mask**  source_masks)
{
	cpl_msg_info(cpl_func, "Detecting sources...");
	cpl_size num_tot_images = hdrl_imagelist_get_size(skysub_images);
	//cpl_msg_info(cpl_func,"num_tot_images: %lld",num_tot_images);
	//source_masks = cpl_calloc(num_tot_images, sizeof(cpl_mask *));

	hdrl_parameter *par_cat = hdrl_catalogue_parameter_parse_parlist(parlist, prefix);
	const hdrl_catalogue_options opt = HDRL_CATALOGUE_SEGMAP;
	hdrl_catalogue_parameter_set_option(par_cat, opt);

	if(bpm_image) {
		if(debug) {
			for (int i = 0; i < num_tot_images; i++) {
				hdrl_image * himg = hdrl_imagelist_get(skysub_images, i);
				cpl_image * data = hdrl_image_get_image(himg);

				cpl_mask *bpm = cpl_mask_threshold_image_create(bpm_image, 0, INT_MAX);
				cpl_image_reject_from_mask(data, bpm);
				cpl_mask_delete(bpm);

				source_masks[i] = eris_detect_sources_hdrl_catalogue(data, par_cat);

				//char* fname = cpl_sprintf("msk_%d.fits",i);
				//cpl_mask_save(source_masks[i], fname, NULL, CPL_IO_DEFAULT);
				//cpl_free(fname);

				/* Convert BPM image to mask (0=good, >0=bad) */
				cpl_mask* bad_pixel_map = cpl_mask_threshold_image_create(bpm_image, 0.5, DBL_MAX);

				int nsources = cpl_mask_count(source_masks[i]) - cpl_mask_count(bad_pixel_map);
				cpl_msg_info(cpl_func, "  Frame %d: detected %d source pixels", i+1, nsources);

				cpl_mask_delete(bad_pixel_map);
			}
		} else {
			for (int i = 0; i < num_tot_images; i++) {
				hdrl_image * himg = hdrl_imagelist_get(skysub_images, i);
				cpl_image * data = hdrl_image_get_image(himg);

				cpl_mask *bpm = cpl_mask_threshold_image_create(bpm_image, 0, INT_MAX);
				cpl_image_reject_from_mask(data, bpm);
				cpl_mask_delete(bpm);

				source_masks[i] = eris_detect_sources_hdrl_catalogue(data, par_cat);

				/* Convert BPM image to mask (0=good, >0=bad) */
				cpl_mask* bad_pixel_map = cpl_mask_threshold_image_create(bpm_image, 0.5, DBL_MAX);

				int nsources = cpl_mask_count(source_masks[i]) - cpl_mask_count(bad_pixel_map);
				cpl_msg_info(cpl_func, "  Frame %d: detected %d source pixels", i+1, nsources);

				cpl_mask_delete(bad_pixel_map);
			}
		}
	} else {
		if(debug) {
			for (int i = 0; i < num_tot_images; i++) {
				hdrl_image * himg = hdrl_imagelist_get(skysub_images, i);
				cpl_image * data = hdrl_image_get_image(himg);
				char* fname = cpl_sprintf("check_data_%d.fits",i);
				cpl_image_save(data, fname, CPL_TYPE_DOUBLE, NULL, CPL_IO_DEFAULT);
				cpl_image_save(hdrl_image_get_error(himg), fname, CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
				cpl_mask_save(hdrl_image_get_mask(himg), fname, NULL, CPL_IO_EXTEND);
				cpl_free(fname);
				source_masks[i] = eris_detect_sources_hdrl_catalogue(data, par_cat);
				//char* fname = cpl_sprintf("msk_%d.fits",i);
				//cpl_mask_save(source_masks[i], fname, NULL, CPL_IO_DEFAULT);
				//cpl_free(fname);

				int nsources = cpl_mask_count(source_masks[i]);
				cpl_msg_info(cpl_func, "  Frame %d: detected %d source pixels", i+1, nsources);
			}
		} else {
			for (int i = 0; i < num_tot_images; i++) {
				hdrl_image * himg = hdrl_imagelist_get(skysub_images, i);
				cpl_image * data = hdrl_image_get_image(himg);

				source_masks[i] = eris_detect_sources_hdrl_catalogue(data, par_cat);

				int nsources = cpl_mask_count(source_masks[i]);
				cpl_msg_info(cpl_func, "  Frame %d: detected %d source pixels", i+1, nsources);
			}
		}
	}


	if(par_cat) hdrl_parameter_delete(par_cat);
	eris_check_error_code(cpl_func);
	return cpl_error_get_code();
}





/*----------------------------------------------------------------------------*/
/**
  @brief    Create array of cpl_masks locating the position of astronomical sources
  @param    skysub_images HDRL images corrected list
  @param    bpm_image   image corresponding to bad pixels
  @param    parlist     input recipe parameters
  @param    recipe_name input recipe name
  @param    source_masks array of cpl_masks with pixel location of objects

  @return   error code
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
eris_locate_and_mask_sources2(  located_imagelist     * jitters, const cpl_image* bpm_image,
		const cpl_parameterlist* parlist,  const char* prefix, cpl_mask**  source_masks)
{
	cpl_msg_info(cpl_func, "Detecting sources...");
	cpl_size num_tot_images = jitters->size;
	cpl_msg_info(cpl_func,"num_tot_images3: %lld",num_tot_images);
	//source_masks = cpl_calloc(num_tot_images, sizeof(cpl_mask *));

	hdrl_parameter *par_cat = hdrl_catalogue_parameter_parse_parlist(parlist, prefix);
	const hdrl_catalogue_options opt = HDRL_CATALOGUE_SEGMAP;
	hdrl_catalogue_parameter_set_option(par_cat, opt);

	if(bpm_image) {
		if(debug) {
			for (int i = 0; i < num_tot_images; i++) {

				cpl_image * data = hdrl_image_get_image(jitters->limages[i]->himage);

				cpl_mask *bpm = cpl_mask_threshold_image_create(bpm_image, 0, INT_MAX);
				cpl_image_reject_from_mask(data, bpm);
				cpl_mask_delete(bpm);

				source_masks[i] = eris_detect_sources_hdrl_catalogue(data, par_cat);

				//char* fname = cpl_sprintf("msk_%d.fits",i);
				//cpl_mask_save(source_masks[i], fname, NULL, CPL_IO_DEFAULT);
				//cpl_free(fname);

				/* Convert BPM image to mask (0=good, >0=bad) */
				cpl_mask* bad_pixel_map = cpl_mask_threshold_image_create(bpm_image, 0.5, DBL_MAX);

				int nsources = cpl_mask_count(source_masks[i]) - cpl_mask_count(bad_pixel_map);
				cpl_msg_info(cpl_func, "  Frame %d: detected %d source pixels", i+1, nsources);

				cpl_mask_delete(bad_pixel_map);
			}
		} else {
			for (int i = 0; i < num_tot_images; i++) {

				cpl_image * data = hdrl_image_get_image(jitters->limages[i]->himage);

				cpl_mask *bpm = cpl_mask_threshold_image_create(bpm_image, 0, INT_MAX);
				cpl_image_reject_from_mask(data, bpm);
				cpl_mask_delete(bpm);

				source_masks[i] = eris_detect_sources_hdrl_catalogue(data, par_cat);

				/* Convert BPM image to mask (0=good, >0=bad) */
				cpl_mask* bad_pixel_map = cpl_mask_threshold_image_create(bpm_image, 0.5, DBL_MAX);

				int nsources = cpl_mask_count(source_masks[i]) - cpl_mask_count(bad_pixel_map);
				cpl_msg_info(cpl_func, "  Frame %d: detected %d source pixels", i+1, nsources);

				cpl_mask_delete(bad_pixel_map);
			}
		}
	} else {
		if(debug) {
			for (int i = 0; i < num_tot_images; i++) {

				cpl_image * data = hdrl_image_get_image(jitters->limages[i]->himage);

				source_masks[i] = eris_detect_sources_hdrl_catalogue(data, par_cat);
				//char* fname = cpl_sprintf("msk_%d.fits",i);
				//cpl_mask_save(source_masks[i], fname, NULL, CPL_IO_DEFAULT);
				//cpl_free(fname);

				int nsources = cpl_mask_count(source_masks[i]);
				cpl_msg_info(cpl_func, "  Frame %d: detected %d source pixels", i+1, nsources);
			}
		} else {
			for (int i = 0; i < num_tot_images; i++) {

				cpl_image * data = hdrl_image_get_image(jitters->limages[i]->himage);

				source_masks[i] = eris_detect_sources_hdrl_catalogue(data, par_cat);

				int nsources = cpl_mask_count(source_masks[i]);
				cpl_msg_info(cpl_func, "  Frame %d: detected %d source pixels", i+1, nsources);
			}
		}
	}


	if(par_cat) hdrl_parameter_delete(par_cat);
	eris_check_error_code(cpl_func);
	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Create a first sky flat corrected imagelist from image list with skyflat
  @param    image_list  HDRL image list to be corrected
  @param    sky_flat    HDRL image to be subtracted
  @param    skysub_images HDRL images correctedc list
  @return   error code
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
eris_first_sky_sub(const hdrl_imagelist* science_images, const hdrl_image* sky_flat_0,
		hdrl_imagelist** skysub_images)
{

    cpl_msg_info(cpl_func, "Subtracting first-pass sky flat...");
    *skysub_images = hdrl_imagelist_new();
    hdrl_value med_sky_0 = hdrl_image_get_median(sky_flat_0);
    char* fname = NULL;
    hdrl_value scale = {0, 0};
    cpl_size num_tot_images = hdrl_imagelist_get_size(science_images);
    for (cpl_size i = 0; i < num_tot_images; i++) {
        const hdrl_image * himg = hdrl_imagelist_get_const(science_images, i);
        cpl_image_reject_value(hdrl_image_get_image(himg), CPL_VALUE_NOTFINITE);
        cpl_image_reject_value(hdrl_image_get_error(himg), CPL_VALUE_NOTFINITE);
        //cpl_image_save(hdrl_image_get_image_const(himg),"test.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_DEFAULT);
        //cpl_image_save(hdrl_image_get_error_const(himg),"test.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
        //cpl_mask_save(hdrl_image_get_mask_const(himg),"test.fits", NULL, CPL_IO_EXTEND);
        hdrl_value med_img = hdrl_image_get_median(himg);;

        scale.data = med_img.data / med_sky_0.data;
        cpl_msg_warning(cpl_func,"1st frame: %lld med_img.data: %g med_sky_0.data: %g scale.data: %g",
                		i, med_img.data, med_sky_0.data, scale.data);
        hdrl_image * skysub = hdrl_image_duplicate(himg);
        hdrl_image * scaled_sky = hdrl_image_duplicate(sky_flat_0);
        hdrl_image_mul_scalar(scaled_sky, scale);

        if(debug) {
        	fname = cpl_sprintf("sci_%lld.fits",i);
        	cpl_image_save(hdrl_image_get_image_const(himg),fname, CPL_TYPE_DOUBLE, NULL, CPL_IO_DEFAULT);
        	cpl_image_save(hdrl_image_get_error_const(himg),fname, CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
        	cpl_mask_save(hdrl_image_get_mask_const(himg),fname, NULL, CPL_IO_EXTEND);
        	cpl_free(fname);
        	fname = cpl_sprintf("scaled_sky_0_%lld.fits",i);
        	cpl_image_save(hdrl_image_get_image(scaled_sky),fname, CPL_TYPE_DOUBLE, NULL, CPL_IO_DEFAULT);
        	cpl_image_save(hdrl_image_get_error(scaled_sky),fname, CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
        	cpl_mask_save(hdrl_image_get_mask(scaled_sky),fname, NULL, CPL_IO_EXTEND);
        	cpl_free(fname);
        }

        hdrl_image_sub_image(skysub, scaled_sky);

        if(debug) {
        	fname = cpl_sprintf("sci_skysub_1st_%03lld.fits",i);
        	cpl_image_save(hdrl_image_get_image(skysub), fname, CPL_TYPE_DOUBLE, NULL, CPL_IO_DEFAULT);
        	cpl_image_save(hdrl_image_get_error(skysub), fname, CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
        	cpl_mask_save(hdrl_image_get_mask(skysub), fname, NULL, CPL_IO_EXTEND);
        	cpl_free(fname);
        }
        hdrl_imagelist_set(*skysub_images, skysub, i);
        hdrl_image_delete(scaled_sky);
        //exit(0);
    }

    eris_check_error_code(cpl_func);
	return cpl_error_get_code();
}

#define HDRL_USE_PRIVATE YES
/*----------------------------------------------------------------------------*/
/**
  @brief    Create sky flat from image list with optional masking
  @param    image_list  HDRL image list
  @param    masks       Array of masks (NULL for first pass)
  @param    method      Combination method ("median" or "sigmaclip")
  @param    kappa       Sigma clipping threshold
  @param    niter       Maximum iterations
  @return   Combined sky flat as HDRL image
 */
/*----------------------------------------------------------------------------*/
static hdrl_image * eris_create_sky_flat_first(
    hdrl_imagelist  * image_list,
    cpl_mask        * mask,
    const char      * method,
    const double      kappa,
    const int         niter,
	const cpl_size    plane_id)
{
    hdrl_image * result = NULL;

    hdrl_imagelist * masked_list = NULL;
    hdrl_parameter * collapse_par = NULL;
    cpl_image* contrib = NULL;

    cpl_ensure(image_list, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(method, CPL_ERROR_NULL_INPUT, NULL);

    int num_images = hdrl_imagelist_get_size(image_list);
    cpl_ensure(num_images > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_msg_warning(cpl_func,"kappa: %g niter: %d", kappa, niter);
    /* Apply masks if provided */
    if (mask) {
        masked_list = hdrl_imagelist_new();

        for (int i = 0; i < num_images; i++) {

            hdrl_image * himg = hdrl_image_duplicate(
                hdrl_imagelist_get_const(image_list, i));

            cpl_image * data = hdrl_image_get_image(himg);
            cpl_image * errs = hdrl_image_get_error(himg);
            cpl_mask * bpm = hdrl_image_get_mask(himg);
            cpl_mask_or(mask,bpm);
            cpl_image_reject_from_mask(data, mask);
            //cpl_image_reject_from_mask(errs, mask);
            hdrl_imagelist_set(masked_list, himg, i);

        }
    } else {
        masked_list = image_list;
    }

    /* Create collapse parameters */
    if (strcmp(method, "median") == 0) {
        collapse_par = hdrl_collapse_median_parameter_create();
    } else {
    	cpl_msg_info(cpl_func,"kappa: %g niter: %d",kappa, niter);
        collapse_par = hdrl_collapse_sigclip_parameter_create(kappa, kappa, niter);
    }

    /* Collapse image list */
    hdrl_imagelist_collapse(image_list, collapse_par, &result, &contrib);
    if(debug) {
    	char* fname;
    	if(plane_id < 0) {
    	    fname = cpl_sprintf("sky_flat_1st.fits");
    	} else {
    		fname = cpl_sprintf("sky_flat_1st_plane_%3.3lld.fits",plane_id);
    	}
    	cpl_image_save(hdrl_image_get_image(result), fname,CPL_TYPE_DOUBLE,NULL,CPL_IO_DEFAULT);
    	cpl_image_save(hdrl_image_get_error(result), fname,CPL_TYPE_DOUBLE,NULL,CPL_IO_EXTEND);
    	cpl_mask_save(hdrl_image_get_mask(result), fname,NULL,CPL_IO_EXTEND);
    	cpl_free(fname);
    }

    /* Cleanup */
    cpl_image_delete(contrib);
    hdrl_parameter_delete(collapse_par);
    if (mask && masked_list) hdrl_imagelist_delete(masked_list);
    eris_check_error_code(cpl_func);
    return result;
}



static hdrl_image * eris_create_sky_flat_first2(
    located_imagelist     * jitters,
	located_imagelist     ** sky_1st,
    cpl_mask        * mask,
    const char      * method,
    const double      kappa,
    const int         niter,
	const cpl_size    plane_id)
{
    hdrl_image * result = NULL;

    //located_imagelist* masked_list = NULL;
    hdrl_imagelist * masked_list = NULL;
    hdrl_parameter * collapse_par = NULL;
    cpl_image* contrib;
    cpl_ensure(jitters, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(method, CPL_ERROR_NULL_INPUT, NULL);
    eris_print_rec_status(0);
    int num_images = jitters->size;
    cpl_ensure(num_images > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_msg_warning(cpl_func,"kappa: %g niter: %d", kappa, niter);
    /* Apply masks if provided */
    eris_print_rec_status(1);
    masked_list = hdrl_imagelist_new();
    *sky_1st = enu_located_imagelist_duplicate(jitters);
    if (mask) {
    	eris_print_rec_status(2);
    	for (int i = 0; i < num_images; i++) {
    		hdrl_image * himg = hdrl_image_duplicate(jitters->limages[i]->himage);
    		cpl_image * data = hdrl_image_get_image(himg);
    		cpl_image * errs = hdrl_image_get_error(himg);
    		cpl_mask * bpm = hdrl_image_get_mask(himg);
    		cpl_mask_or(mask,bpm);
    		cpl_image_reject_from_mask(data, mask);
    		cpl_image_reject_from_mask(errs, mask);
    		hdrl_imagelist_set(masked_list, himg, i);
    	}
    	eris_print_rec_status(3);
    } else {
    	eris_print_rec_status(4);
    	for (int i = 0; i < num_images; i++) {
    		//hdrl_image * himg = hdrl_image_duplicate(jitters->limages[i]->himage);
    		//cpl_image * data = hdrl_image_get_image(himg);
    		hdrl_imagelist_set(masked_list, jitters->limages[i]->himage, i);
    		//cpl_image_save(hdrl_image_get_image(himg), "test.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_DEFAULT);
    		//cpl_image_save(hdrl_image_get_error(himg),  "test.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    		//cpl_mask_save(hdrl_image_get_mask(himg),  "test.fits", NULL, CPL_IO_EXTEND);
    		//exit(0);
    	}
    	eris_print_rec_status(5);
    }

    /* Create collapse parameters */
    if (strcmp(method, "median") == 0) {
        collapse_par = hdrl_collapse_median_parameter_create();
    } else {
    	cpl_msg_info(cpl_func,"kappa: %g niter: %d",kappa, niter);
        collapse_par = hdrl_collapse_sigclip_parameter_create(kappa, kappa, niter);
    }
    eris_print_rec_status(6);
    /* Collapse image list */
    hdrl_imagelist_collapse(masked_list, collapse_par, &result, &contrib);
	for (int i = 0; i < num_images; i++) {
		hdrl_image_sub_image((*sky_1st)->limages[i]->himage,result);
		(*sky_1st)->limages[i]->bkg = hdrl_image_duplicate(result);
		(*sky_1st)->limages[i]->confidence = cpl_image_duplicate(contrib);
	}
    eris_print_rec_status(7);
    if(debug) {
    	char* fname;
    	if(plane_id < 0) {
    	    fname = cpl_sprintf("sky_flat_1st.fits");
    	} else {
    		fname = cpl_sprintf("sky_flat_1st_plane_%3.3lld.fits",plane_id);
    	}
    	cpl_image_save(hdrl_image_get_image(result), fname,CPL_TYPE_DOUBLE,NULL,CPL_IO_DEFAULT);
    	cpl_image_save(hdrl_image_get_error(result), fname,CPL_TYPE_DOUBLE,NULL,CPL_IO_EXTEND);
    	cpl_mask_save(hdrl_image_get_mask(result), fname,NULL,CPL_IO_EXTEND);
    	cpl_free(fname);
    }
    eris_print_rec_status(8);
    /* Cleanup */
    cpl_image_delete(contrib);
    hdrl_parameter_delete(collapse_par);
    if (masked_list) {
    	for (int i = num_images-1; i >= 0; i--) {
    	hdrl_imagelist_unset(masked_list, i);
    	}
    	hdrl_imagelist_delete(masked_list);
    }
    eris_check_error_code(cpl_func);
    return result;
}


/**
  @brief    Create sky flat from image list with optional masking
  @param    image_list  HDRL image list
  @param    masks       Array of masks (NULL for first pass)
  @param    method      Combination method ("median" or "sigmaclip")
  @param    kappa       Sigma clipping threshold
  @param    niter       Maximum iterations
  @return   Combined sky flat as HDRL image
 */
/*----------------------------------------------------------------------------*/
static hdrl_image * eris_create_sky_flat_final(
    hdrl_imagelist  *   image_list,
    cpl_mask        **  masks,
    const char      *   method,
    const double        kappa,
    const int           niter,
	const cpl_size      plane_id,
	cpl_image** contrib)
{
    hdrl_image * result = NULL;
    hdrl_imagelist * masked_list = NULL;
    hdrl_parameter * collapse_par = NULL;


    cpl_ensure(image_list, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(method, CPL_ERROR_NULL_INPUT, NULL);

    int num_images = hdrl_imagelist_get_size(image_list);
    cpl_ensure(num_images > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    /* Apply masks if provided */
    if (masks) {
        masked_list = hdrl_imagelist_new();

        for (int i = 0; i < num_images; i++) {

            hdrl_image * himg = hdrl_image_duplicate(
                hdrl_imagelist_get_const(image_list, i));

            cpl_image * data = hdrl_image_get_image(himg);
            cpl_image * errs = hdrl_image_get_image(himg);
            cpl_mask * msk = hdrl_image_get_mask(himg);

            //AMO: TODO THE FOLLOWING LINE IS SOURCE OF TROUBLES: was
            //cpl_image_reject_from_mask(data, masks[i]);
            //Now IS (to be sure flag also existing bad pixels)

            cpl_mask_or(msk,masks[i]);
            cpl_image_reject_from_mask(data, msk);
            cpl_image_reject_from_mask(errs, msk);
            if(debug) {
            	char* fname = NULL;
            	if(plane_id < 0) {
            	    fname = cpl_sprintf("mask_%d.fits",i);
            	} else {
            		fname = cpl_sprintf("mask_%d_%3.3lld.fits",i,plane_id);
            	}
            	//cpl_msg_warning(cpl_func,"mask[%d]: %p",i, masks[i]);
            	cpl_mask_save(masks[i], fname, NULL, CPL_IO_DEFAULT);
            	cpl_free(fname);
            }

            hdrl_imagelist_set(masked_list, himg, i);

        }
    } else {
        masked_list = image_list;
    }

    /* Create collapse parameters */
    if (strcmp(method, "median") == 0) {
        collapse_par = hdrl_collapse_median_parameter_create();
    } else {
        collapse_par = hdrl_collapse_sigclip_parameter_create(kappa, kappa, niter);
    }


    /* Collapse image list */
    hdrl_imagelist_collapse(masked_list, collapse_par,&result, contrib);
    if(debug) {
    	char* fname;
    	if(plane_id < 0) {
    		fname = cpl_sprintf("sky_flat_2nd.fits");
    	} else {
    		fname = cpl_sprintf("sky_flat_2nd_plane_%3.3lld.fits",plane_id);
    	}
    	cpl_image_save(hdrl_image_get_image(result), fname,CPL_TYPE_DOUBLE,NULL,CPL_IO_DEFAULT);
    	cpl_image_save(hdrl_image_get_error(result), fname,CPL_TYPE_DOUBLE,NULL,CPL_IO_EXTEND);
    	cpl_mask_save(hdrl_image_get_mask(result), fname,NULL,CPL_IO_EXTEND);
    	cpl_free(fname);
    }

    /* Cleanup */
    hdrl_parameter_delete(collapse_par);

    if (masks && masked_list) {
    	hdrl_imagelist_delete(masked_list);
    }

    eris_check_error_code(cpl_func);
    return result;
}


static hdrl_image * eris_create_sky_flat_final2(
	located_imagelist     * jitters,
    located_imagelist     ** sky_2nd,
    cpl_mask        **  masks,
    const char      *   method,
    const double        kappa,
    const int           niter,
	const cpl_size      plane_id)
{
    hdrl_image * result = NULL;
    hdrl_imagelist * masked_list = NULL;
    hdrl_parameter * collapse_par = NULL;
    cpl_image* contrib = NULL;

    cpl_ensure(jitters, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(method, CPL_ERROR_NULL_INPUT, NULL);

    cpl_size num_images = jitters->size;
    cpl_ensure(num_images > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_msg_info(cpl_func,"num_images: %lld",num_images);
    /* Apply masks if provided */
    cpl_msg_warning(cpl_func,"step1");
    if (masks) {
        masked_list = hdrl_imagelist_new();

        for (int i = 0; i < num_images; i++) {

            cpl_image * data = hdrl_image_get_image(jitters->limages[i]->himage);

            //AMO: TODO THE FOLLOWING LINE IS SOURCE OF TROUBLES
            cpl_image_reject_from_mask(data, masks[i]);
            if(debug) {
            	char* fname = NULL;
            	if(plane_id < 0) {
            	    fname = cpl_sprintf("mask_%d.fits",i);
            	} else {
            		fname = cpl_sprintf("mask_%d_%3.3lld.fits",i,plane_id);
            	}
            	//cpl_msg_warning(cpl_func,"mask[%d]: %p",i, masks[i]);
            	cpl_mask_save(masks[i], fname, NULL, CPL_IO_DEFAULT);
            	cpl_free(fname);
            }

            hdrl_imagelist_set(masked_list, jitters->limages[i]->himage, i);

        }
    } else {
        for (int i = 0; i < num_images; i++) {
        	hdrl_imagelist_set(masked_list, jitters->limages[i]->himage, i);
        }
    }
    cpl_msg_warning(cpl_func,"step2");

    /* Create collapse parameters */
    if (strcmp(method, "median") == 0) {
        collapse_par = hdrl_collapse_median_parameter_create();
    } else {
        collapse_par = hdrl_collapse_sigclip_parameter_create(kappa, kappa, niter);
    }
    cpl_msg_warning(cpl_func,"step3");

    /* Collapse image list */
    hdrl_imagelist_collapse(masked_list, collapse_par,&result, &contrib);
    if(debug) {
    	char* fname;
    	if(plane_id < 0) {
    		fname = cpl_sprintf("sky_flat_2nd.fits");
    	} else {
    		fname = cpl_sprintf("sky_flat_2nd_plane_%3.3lld.fits",plane_id);
    	}
    	cpl_image_save(hdrl_image_get_image(result), fname,CPL_TYPE_DOUBLE,NULL,CPL_IO_DEFAULT);
    	cpl_image_save(hdrl_image_get_error(result), fname,CPL_TYPE_DOUBLE,NULL,CPL_IO_EXTEND);
    	cpl_mask_save(hdrl_image_get_mask(result), fname,NULL,CPL_IO_EXTEND);
    	cpl_free(fname);
    }
    cpl_msg_warning(cpl_func,"step4");

    for (int i = 0; i < num_images; i++) {
    	hdrl_image_sub_image((*sky_2nd)->limages[i]->himage,result);
    	(*sky_2nd)->limages[i]->confidence = contrib;
    	(*sky_2nd)->limages[i]->bkg = result;
    }
    cpl_msg_warning(cpl_func,"step5");

    /* Cleanup */
    if (contrib != NULL) {
    	cpl_image_delete(contrib);
    }

    hdrl_parameter_delete(collapse_par);

    if (masks && masked_list) {
    	for (int i = num_images-1; i >= 0; i--) {
    		hdrl_imagelist_unset(masked_list,i);
    	}
    	hdrl_imagelist_delete(masked_list);
    }

    eris_check_error_code(cpl_func);
    return result;
}

static cpl_mask * eris_detect_sources_hdrl_catalogue(
    const cpl_image *   image,
	hdrl_parameter* p)
{

	cpl_mask * mask_res = NULL;

	hdrl_catalogue_result *res = hdrl_catalogue_compute(image, NULL, NULL, p);
	/*
	if(debug) {
	    char* fname = cpl_sprintf("cat_%d.fits",0);
	    cpl_table_save (res->catalogue, NULL, NULL, out_fname, CPL_IO_CREATE);
	    cpl_free(fname);
		char* fname = cpl_sprintf("smap_%d.fits",0);
		cpl_image_save(res->segmentation_map, fname, CPL_TYPE_DOUBLE, NULL, CPL_IO_DEFAULT);
		cpl_free(fname);
		char* fname = cpl_sprintf("bkg_%d.fits",0);
		cpl_image_save(res->background, fname, CPL_TYPE_DOUBLE, NULL, CPL_IO_DEFAULT);
		cpl_free(fname);
	}
	*/
	double max = cpl_image_get_max(res->segmentation_map);
	mask_res = cpl_mask_threshold_image_create(res->segmentation_map, 0.9, max + 0.1);
	hdrl_catalogue_result_delete(res);
	eris_check_error_code(cpl_func);
	return mask_res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Dilate a mask by given radius
  @param    input_mask  Input mask
  @param    radius      Dilation radius
  @return   Dilated mask
 */
/*----------------------------------------------------------------------------*/
static cpl_mask * eris_dilate_mask(
    const cpl_mask  *   input_mask,
    int                 radius)
{
    cpl_mask * result = NULL;
    cpl_mask * temp = NULL;
    cpl_size nx, ny;

    cpl_ensure(input_mask, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(radius > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    nx = cpl_mask_get_size_x(input_mask);
    ny = cpl_mask_get_size_y(input_mask);

    result = cpl_mask_duplicate(input_mask);

    /* Simple box dilation */
    for (int iter = 0; iter < radius; iter++) {
        temp = cpl_mask_duplicate(result);

        for (cpl_size j = 2; j < ny - 1; j++) {
            for (cpl_size i = 2; i < nx - 1; i++) {
                if (cpl_mask_get(temp, i, j) ||
                    cpl_mask_get(temp, i-1, j) ||
                    cpl_mask_get(temp, i+1, j) ||
                    cpl_mask_get(temp, i, j-1) ||
                    cpl_mask_get(temp, i, j+1)) {
                    cpl_mask_set(result, i, j, CPL_BINARY_1);
                }
            }
        }
        cpl_mask_delete(temp);
    }
    eris_check_error_code(cpl_func);
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Compute comprehensive statistics on an image with sigma clipping
 * @param    in_image        Input HDRL image
 * @param    l_sig           Lower sigma clipping factor
 * @param    u_sig           Upper sigma clipping factor
 * @param    mask            Optional bad pixel mask (NULL if not used)
 * @param    trim_fraction   Fraction to trim from both ends (e.g., 0.1 for 10%)
 * @param    median          Output: median value
 * @param    mean            Output: mean value
 * @param    stdev           Output: standard deviation
 * @param    tmean           Output: trimmed mean
 * @param    tstd            Output: trimmed standard deviation
 * @param    mad             Output: median absolute deviation
 * @param    min_val         Output: minimum value
 * @param    max_val         Output: maximum value
 * @return   CPL_ERROR_NONE on success, error code otherwise
 *
 * This function computes robust statistics on an image using sigma clipping
 * and trimmed statistics. The algorithm:
 * 1. Computes initial median, mean, and std
 * 2. Computes trimmed mean (10% by default) for robust center estimate
 * 3. Defines clipping range: trimmed_mean ± l_sig*std to trimmed_mean + u_sig*std
 * 4. Computes final trimmed statistics within clipping range
 * 5. Computes MAD (median absolute deviation) for robust dispersion
 *
 * @note All NaN values are automatically excluded from calculations
 * @note The mask parameter can be NULL; if provided, only masked pixels are used
 * @note MAD is computed as median(|data - median(data)|), a robust alternative to std
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
eris_image_stats(const hdrl_image *in_image,
                 double l_sig,
                 double u_sig,
                 const cpl_mask *mask,
                 double trim_fraction,
                 double *median,
                 double *mean,
                 double *stdev,
                 double *tmean,
                 double *tstd,
                 double *mad,
                 double *min_val,
                 double *max_val)
{
    cpl_ensure_code(in_image, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(median, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mean, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(stdev, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(tmean, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(tstd, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mad, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(min_val, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(max_val, CPL_ERROR_NULL_INPUT);

    const cpl_image *img = hdrl_image_get_image_const(in_image);
    cpl_image *work_img = NULL;
    cpl_vector *data_vec = NULL;

    /* Create working copy of image */
    work_img = cpl_image_duplicate(img);
    if (!work_img) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_OUTPUT,
                                    "Failed to duplicate image");
    }

    /* Apply mask if provided */
    if (mask != NULL) {
        cpl_image_reject_from_mask(work_img, mask);
    }

    /* Step 1: Compute initial statistics (NaN-aware) */
    hdrl_value median0 = hdrl_image_get_median(in_image);
    hdrl_value mean0 = hdrl_image_get_mean(in_image);
    double stdev0 = hdrl_image_get_stdev(in_image);

    *median = median0.data;
    *mean = mean0.data;
    *stdev = stdev0;

    /* Step 2: Convert image to vector for trimmed statistics */
    cpl_size nx = cpl_image_get_size_x(work_img);
    cpl_size ny = cpl_image_get_size_y(work_img);
    cpl_size npix = nx * ny;

    /* Count valid (non-rejected) pixels */
    const cpl_mask *rej_mask = cpl_image_get_bpm_const(work_img);
    cpl_size nvalid = npix;
    if (rej_mask) {
        nvalid = npix - cpl_mask_count(rej_mask);
    }

    if (nvalid == 0) {
        cpl_image_delete(work_img);
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                    "No valid pixels in image");
    }

    /* Extract valid pixel values to vector */
    data_vec = cpl_vector_new(nvalid);
    cpl_size idx = 0;
    for (cpl_size j = 1; j <= ny; j++) {
        for (cpl_size i = 1; i <= nx; i++) {
            int is_rejected = 0;
            double val = cpl_image_get(work_img, i, j, &is_rejected);
            if (!is_rejected && isfinite(val)) {
                cpl_vector_set(data_vec, idx, val);
                idx++;
            }
        }
    }

    /* Adjust vector size if some NaNs were found */
    if (idx < nvalid) {
        cpl_vector *tmp = cpl_vector_extract(data_vec, 0, idx - 1, 1);
        cpl_vector_delete(data_vec);
        data_vec = tmp;
        nvalid = idx;
    }

    /* Step 3: Compute trimmed mean (trim_fraction from each end) */
    cpl_vector_sort(data_vec, CPL_SORT_ASCENDING);
    cpl_size trim_count = (cpl_size)(trim_fraction * nvalid);
    if (trim_count * 2 >= nvalid) {
        trim_count = 0; /* Not enough data for trimming */
    }

    double tmean0_10 = 0.0;
    if (trim_count > 0) {
        /* Compute mean of middle (1 - 2*trim_fraction) of data */
        cpl_vector *trimmed = cpl_vector_extract(data_vec, trim_count,
                                                 nvalid - trim_count - 1, 1);
        tmean0_10 = cpl_vector_get_mean(trimmed);
        cpl_vector_delete(trimmed);
    } else {
        tmean0_10 = cpl_vector_get_mean(data_vec);
    }

    /* Step 4: Define clipping range based on trimmed mean */
    double rmin = tmean0_10 - l_sig * (*stdev);
    double rmax = tmean0_10 + u_sig * (*stdev);

    /* Step 5: Compute trimmed statistics within clipping range */
    cpl_size nclipped = 0;
    double sum_clipped = 0.0;
    double sum_sq_clipped = 0.0;

    for (cpl_size i = 0; i < nvalid; i++) {
        double val = cpl_vector_get(data_vec, i);
        if (val >= rmin && val <= rmax) {
            sum_clipped += val;
            sum_sq_clipped += val * val;
            nclipped++;
        }
    }

    if (nclipped > 0) {
        *tmean = sum_clipped / nclipped;

        if (nclipped > 1) {
            double variance = (sum_sq_clipped - sum_clipped * sum_clipped / nclipped)
                             / (nclipped - 1);
            *tstd = (variance > 0.0) ? sqrt(variance) : 0.0;
        } else {
            *tstd = 0.0;
        }
    } else {
        /* No data within clipping range - use unclipped statistics */
        *tmean = *mean;
        *tstd = *stdev;
    }

    /* Step 6: Compute MAD (Median Absolute Deviation) */
    /* MAD = median(|data - median|) */
    cpl_vector *abs_dev = cpl_vector_duplicate(data_vec);
    cpl_vector_subtract_scalar(abs_dev, *median);
    for (cpl_size i = 0; i < nvalid; i++) {
        double val = cpl_vector_get(abs_dev, i);
        cpl_vector_set(abs_dev, i, fabs(val));
    }
    cpl_vector_sort(abs_dev, CPL_SORT_ASCENDING);

    /*AMO Check MAD computation versus HDRL
    double sigma;
    cpl_image* ima = cpl_image_wrap_double(nvalid, 1, cpl_vector_get_data(abs_dev));

    *mad=cpl_image_get_mad_window(ima, 1, 1, nvalid, 1, &sigma);

    cpl_msg_warning(cpl_func,"HDRL mad: %g", *mad);
    */

    *mad = cpl_vector_get_median(abs_dev);
    //cpl_msg_warning(cpl_func,"OTHER mad: %g", *mad);
    cpl_vector_delete(abs_dev);

    /* Step 7: Get min and max */
    *min_val = cpl_vector_get_min(data_vec);
    *max_val = cpl_vector_get_max(data_vec);

    /* Cleanup */
    cpl_vector_delete(data_vec);
    cpl_image_delete(work_img);
    eris_check_error_code(cpl_func);
    return CPL_ERROR_NONE;
}
