#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>

#include <cpl.h>
#include <cpl_test.h>
//#include <cpl_tools.h>
#include <cpl_memory.h>

#include "eris_ifu_debug.h"
#include "eris_ifu_error.h"

/* Test function prototypes */
static void test_debug_header(void);
static void test_debug_frame(void);
static void test_debug_frameset(void);
static void test_debug_parameterlist(void);
static void test_debug_propertylist(void);
static void test_debug_image(void);
static void test_debug_cube(void);
static void test_debug_vector(void);
static void test_debug_array(void);
static void test_debug_table(void);
static void test_plot_functions(void);

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    /* Run all test functions */
    test_debug_header();
    test_debug_frame();
    test_debug_frameset();
    test_debug_parameterlist();
    test_debug_propertylist();
    test_debug_image();
    //test_debug_cube(); TODO: fix leaks
    test_debug_vector();
    test_debug_array();
    test_debug_table();
    test_plot_functions();

    return cpl_test_end(0);
}

/* Test function implementations */

static void test_debug_header(void)
{
    cpl_propertylist *header = NULL;
    cpl_error_code error;

    /* Test with NULL input */
    error = eris_ifu_debug_header(NULL);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Create a test property list */
    header = cpl_propertylist_new();
    cpl_test_nonnull(header);

    /* Test empty property list */
    error = eris_ifu_debug_header(header);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Add various types of properties */
    cpl_propertylist_append_bool(header, "BOOL_PROP_TRUE", 1);
    cpl_propertylist_append_bool(header, "BOOL_PROP_FALSE", 0);
    cpl_propertylist_append_int(header, "INT_PROP_POS", 42);
    cpl_propertylist_append_int(header, "INT_PROP_NEG", -42);
    cpl_propertylist_append_int(header, "INT_PROP_ZERO", 0);
    cpl_propertylist_append_double(header, "DBL_PROP_PI", 3.14159);
    cpl_propertylist_append_double(header, "DBL_PROP_NEG", -2.718);
    cpl_propertylist_append_double(header, "DBL_PROP_ZERO", 0.0);
    cpl_propertylist_append_float(header, "FLT_PROP_E", 2.71828f);
    cpl_propertylist_append_float(header, "FLT_PROP_NEG", -3.14f);
    cpl_propertylist_append_float(header, "FLT_PROP_ZERO", 0.0f);
    cpl_propertylist_append_string(header, "STR_PROP_NORMAL", "test string");
    cpl_propertylist_append_string(header, "STR_PROP_EMPTY", "");
    cpl_propertylist_append_string(header, "STR_PROP_SPECIAL", "!@#$%^&*()");

    /* Test debug output with all types */
    error = eris_ifu_debug_header(header);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Test error handling by corrupting the property list (if possible in your implementation) */
    /* Note: This might not be possible depending on the CPL implementation */
    
    /* Cleanup */
    cpl_propertylist_delete(header);
}

static void test_debug_frame(void)
{
    cpl_frame *frame = NULL;
    cpl_error_code error;

    /* Test with NULL input */
    error = eris_ifu_debug_frame(NULL);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Create a test frame */
    frame = cpl_frame_new();
    cpl_test_nonnull(frame);

    /* Test empty frame */
    error = eris_ifu_debug_frame(frame);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Test with minimal properties */
    cpl_frame_set_filename(frame, "test.fits");
    error = eris_ifu_debug_frame(frame);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Test all frame types */
    const int frame_types[] = {
        CPL_FRAME_TYPE_NONE,
        CPL_FRAME_TYPE_IMAGE,
        CPL_FRAME_TYPE_MATRIX,
        CPL_FRAME_TYPE_TABLE,
        CPL_FRAME_TYPE_PAF,
        CPL_FRAME_TYPE_ANY
    };
    const int n_frame_types = sizeof(frame_types) / sizeof(frame_types[0]);

    /* Test all frame groups */
    const int frame_groups[] = {
        CPL_FRAME_GROUP_NONE,
        CPL_FRAME_GROUP_RAW,
        CPL_FRAME_GROUP_CALIB,
        CPL_FRAME_GROUP_PRODUCT
    };
    const int n_frame_groups = sizeof(frame_groups) / sizeof(frame_groups[0]);

    /* Test all frame levels */
    const int frame_levels[] = {
        CPL_FRAME_LEVEL_NONE,
        CPL_FRAME_LEVEL_TEMPORARY,
        CPL_FRAME_LEVEL_INTERMEDIATE,
        CPL_FRAME_LEVEL_FINAL
    };
    const int n_frame_levels = sizeof(frame_levels) / sizeof(frame_levels[0]);

    /* Test combinations of types, groups, and levels */
    for (int i = 0; i < n_frame_types; i++) {
        for (int j = 0; j < n_frame_groups; j++) {
            for (int k = 0; k < n_frame_levels; k++) {
                char filename[256];
                char tag[256];
                
                snprintf(filename, sizeof(filename), "test_type%d_group%d_level%d.fits", 
                        frame_types[i], frame_groups[j], frame_levels[k]);
                snprintf(tag, sizeof(tag), "TEST_T%d_G%d_L%d", 
                        frame_types[i], frame_groups[j], frame_levels[k]);

                cpl_frame_set_filename(frame, filename);
                cpl_frame_set_tag(frame, tag);
                cpl_frame_set_type(frame, frame_types[i]);
                cpl_frame_set_group(frame, frame_groups[j]);
                cpl_frame_set_level(frame, frame_levels[k]);

                error = eris_ifu_debug_frame(frame);
                cpl_test_eq(error, CPL_ERROR_NONE);
            }
        }
    }

    /* Test with special characters in filename and tag */
    cpl_frame_set_filename(frame, "test!@#$%^&*().fits");
    cpl_frame_set_tag(frame, "TEST!@#$%^&*()");
    error = eris_ifu_debug_frame(frame);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Test with empty strings */
    cpl_frame_set_filename(frame, "");
    cpl_frame_set_tag(frame, "");
    error = eris_ifu_debug_frame(frame);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Cleanup */
    cpl_frame_delete(frame);
}

static void test_debug_frameset(void)
{
    cpl_frameset *frameset = NULL;
    cpl_frame *frame1 = NULL, *frame2 = NULL;
    cpl_error_code error;

    /* Test with NULL input */
    error = eris_ifu_debug_frameset(NULL);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Create a test frameset with multiple frames */
    frameset = cpl_frameset_new();
    cpl_test_nonnull(frameset);

    frame1 = cpl_frame_new();
    cpl_test_nonnull(frame1);
    cpl_frame_set_filename(frame1, "test1.fits");
    cpl_frame_set_type(frame1, CPL_FRAME_TYPE_IMAGE);
    cpl_frame_set_tag(frame1,"TEST1");
    cpl_frameset_insert(frameset, frame1);
    cpl_test_eq(error, CPL_ERROR_NONE);

    frame2 = cpl_frame_new();
    cpl_test_nonnull(frame2);
    cpl_frame_set_filename(frame2, "test2.fits");
    cpl_frame_set_type(frame2, CPL_FRAME_TYPE_TABLE);
    cpl_frame_set_tag(frame2,"TEST2");
    cpl_frameset_insert(frameset, frame2);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Test debug output */
    error = eris_ifu_debug_frameset(frameset);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Cleanup */
    cpl_frameset_delete(frameset);
    cpl_test_eq(error, CPL_ERROR_NONE);
}

static void test_debug_parameterlist(void)
{
    cpl_parameterlist *plist = NULL;
    cpl_parameter *param = NULL;
    cpl_error_code error;

    /* Test with NULL input */
    error = eris_ifu_debug_parameterlist(NULL);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Create a test parameter list */
    plist = cpl_parameterlist_new();
    cpl_test_nonnull(plist);

    /* Test empty parameter list */
    error = eris_ifu_debug_parameterlist(plist);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Test boolean parameters */
    param = cpl_parameter_new_value("BOOL_PARAM_TRUE", CPL_TYPE_BOOL,
                                  "Boolean parameter (true)", "bool", 1);
    cpl_parameterlist_append(plist, param);

    param = cpl_parameter_new_value("BOOL_PARAM_FALSE", CPL_TYPE_BOOL,
                                  "Boolean parameter (false)", "bool", 0);
    cpl_parameterlist_append(plist, param);

    /* Test integer parameters with different values */
    param = cpl_parameter_new_value("INT_PARAM_POS", CPL_TYPE_INT,
                                  "Integer parameter (positive)", "int", 42);
    cpl_parameterlist_append(plist, param);

    param = cpl_parameter_new_value("INT_PARAM_NEG", CPL_TYPE_INT,
                                  "Integer parameter (negative)", "int", -42);
    cpl_parameterlist_append(plist, param);

    param = cpl_parameter_new_value("INT_PARAM_ZERO", CPL_TYPE_INT,
                                  "Integer parameter (zero)", "int", 0);
    cpl_parameterlist_append(plist, param);

    param = cpl_parameter_new_value("INT_PARAM_MAX", CPL_TYPE_INT,
                                  "Integer parameter (large)", "int", 2147483647);
    cpl_parameterlist_append(plist, param);

    /* Test double parameters with different values */
    param = cpl_parameter_new_value("DBL_PARAM_PI", CPL_TYPE_DOUBLE,
                                  "Double parameter (pi)", "double", 3.14159);
    cpl_parameterlist_append(plist, param);

    param = cpl_parameter_new_value("DBL_PARAM_NEG", CPL_TYPE_DOUBLE,
                                  "Double parameter (negative)", "double", -2.718);
    cpl_parameterlist_append(plist, param);

    param = cpl_parameter_new_value("DBL_PARAM_ZERO", CPL_TYPE_DOUBLE,
                                  "Double parameter (zero)", "double", 0.0);
    cpl_parameterlist_append(plist, param);

    param = cpl_parameter_new_value("DBL_PARAM_LARGE", CPL_TYPE_DOUBLE,
                                  "Double parameter (large)", "double", 1e308);
    cpl_parameterlist_append(plist, param);

    param = cpl_parameter_new_value("DBL_PARAM_SMALL", CPL_TYPE_DOUBLE,
                                  "Double parameter (small)", "double", 1e-308);
    cpl_parameterlist_append(plist, param);

    /* Test string parameters with different values */
    param = cpl_parameter_new_value("STR_PARAM_NORMAL", CPL_TYPE_STRING,
                                  "String parameter (normal)", "string", "test string");
    cpl_parameterlist_append(plist, param);

    param = cpl_parameter_new_value("STR_PARAM_EMPTY", CPL_TYPE_STRING,
                                  "String parameter (empty)", "string", "");
    cpl_parameterlist_append(plist, param);

    param = cpl_parameter_new_value("STR_PARAM_SPECIAL", CPL_TYPE_STRING,
                                  "String parameter (special chars)", "string", "!@#$%^&*()");
    cpl_parameterlist_append(plist, param);

    param = cpl_parameter_new_value("STR_PARAM_LONG", CPL_TYPE_STRING,
                                  "String parameter (long)", "string", 
                                  "This is a very long string parameter that tests the handling of long strings "
                                  "in the debug output to ensure proper formatting and line wrapping if needed");
    cpl_parameterlist_append(plist, param);

    /* Test debug output with all parameters */
    error = eris_ifu_debug_parameterlist(plist);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Test error handling by attempting to add invalid parameters */
    param = cpl_parameter_new_value("INVALID_TYPE", CPL_TYPE_STRING,
                                  NULL, "string", "test");  /* Invalid context */
    if (param != NULL) {
        cpl_parameterlist_append(plist, param);
        error = eris_ifu_debug_parameterlist(plist);
        cpl_test_eq(error, CPL_ERROR_NONE);
    }

    /* Cleanup */
    cpl_parameterlist_delete(plist);
}

static void test_debug_propertylist(void)
{
    cpl_propertylist *plist = NULL;
    cpl_error_code error;

    /* Test with NULL input */
    error = eris_ifu_debug_propertylist(NULL);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Create a test property list */
    plist = cpl_propertylist_new();
    cpl_test_nonnull(plist);

    /* Add properties of different types */
    cpl_propertylist_append_bool(plist, "BOOL_PROP", 1);
    cpl_propertylist_append_int(plist, "INT_PROP", 42);
    cpl_propertylist_append_double(plist, "DBL_PROP", 3.14159);
    cpl_propertylist_append_string(plist, "STR_PROP", "test string");

    /* Test debug output */
    error = eris_ifu_debug_propertylist(plist);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Cleanup */
    cpl_propertylist_delete(plist);
}

static void test_debug_image(void)
{
    cpl_image *img = NULL;
    cpl_error_code error;
    int gaga;  /* Bad pixel flag */

    /* Test with NULL input */
    error = eris_ifu_debug_image(NULL);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Test with 1x1 image */
    img = cpl_image_new(1, 1, CPL_TYPE_DOUBLE);
    cpl_test_nonnull(img);
    cpl_image_set(img, 1, 1, 1.0);
    error = eris_ifu_debug_image(img);
    cpl_test_eq(error, CPL_ERROR_NONE);
    cpl_image_delete(img);

    /* Test with different image sizes */
    const int test_sizes[][2] = {
        {2, 2},   /* Small square */
        {3, 2},   /* Small rectangle */
        {2, 3},   /* Small rectangle (transposed) */
        {5, 5},   /* Medium square */
        {10, 1},  /* Wide strip */
        {1, 10}   /* Tall strip */
    };

    for (size_t i = 0; i < sizeof(test_sizes)/sizeof(test_sizes[0]); i++) {
        img = cpl_image_new(test_sizes[i][0], test_sizes[i][1], CPL_TYPE_DOUBLE);
        cpl_test_nonnull(img);

        /* Fill with test pattern */
        for (int y = 1; y <= test_sizes[i][1]; y++) {
            for (int x = 1; x <= test_sizes[i][0]; x++) {
                double val = (x + y) * 1.5;
                cpl_image_set(img, x, y, val);
            }
        }

        error = eris_ifu_debug_image(img);
        cpl_test_eq(error, CPL_ERROR_NONE);
        cpl_image_delete(img);
    }

    /* Test with different data types */
    const cpl_type test_types[] = {
        CPL_TYPE_INT,
        CPL_TYPE_FLOAT,
        CPL_TYPE_DOUBLE
    };

    for (size_t i = 0; i < sizeof(test_types)/sizeof(test_types[0]); i++) {
        img = cpl_image_new(3, 3, test_types[i]);
        cpl_test_nonnull(img);

        /* Fill with test values including zero, positive, negative, and special values */
        cpl_image_set(img, 1, 1, 0.0);      /* Zero */
        cpl_image_set(img, 1, 2, 1.5);      /* Positive non-integer */
        cpl_image_set(img, 1, 3, -2.5);     /* Negative non-integer */
        cpl_image_set(img, 2, 1, 1000.0);   /* Large positive */
        cpl_image_set(img, 2, 2, -1000.0);  /* Large negative */
        cpl_image_set(img, 2, 3, 3.14159);  /* Pi */
        cpl_image_set(img, 3, 1, 2.71828);  /* e */
        cpl_image_set(img, 3, 2, 1e-10);    /* Very small */
        cpl_image_set(img, 3, 3, 1e10);     /* Very large */

        error = eris_ifu_debug_image(img);
        cpl_test_eq(error, CPL_ERROR_NONE);
        cpl_image_delete(img);
    }

    /* Test with bad pixels */
    img = cpl_image_new(3, 3, CPL_TYPE_DOUBLE);
    cpl_test_nonnull(img);

    /* Set some normal values */
    cpl_image_set(img, 1, 1, 1.0);
    cpl_image_set(img, 1, 2, 2.0);
    cpl_image_set(img, 1, 3, 3.0);

    /* Create and test bad pixel map */
    cpl_mask *bpm = cpl_mask_new(3, 3);
    cpl_test_nonnull(bpm);
    cpl_mask_set(bpm, 2, 2, 1);  /* Mark center pixel as bad */
    cpl_image_set_bpm(img, bpm);

    /* Test getting values with bad pixel flag */
    double val = cpl_image_get(img, 2, 2, &gaga);
    cpl_test_eq(gaga, 1);  /* Should be marked as bad */

    error = eris_ifu_debug_image(img);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Cleanup */
    cpl_image_delete(img);
    //cpl_mask_delete(bpm);
}

static void test_debug_cube(void)
{
    cpl_imagelist *cube = NULL;
    cpl_image *img = NULL;
    cpl_error_code error;

    /* Test with NULL input */
    error = eris_ifu_debug_cube(NULL);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Create a test image cube */
    cube = cpl_imagelist_new();
    cpl_test_nonnull(cube);

    /* Test empty cube */
    error = eris_ifu_debug_cube(cube);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Test cube with single image */
    img = cpl_image_new(2, 2, CPL_TYPE_DOUBLE);
    cpl_test_nonnull(img);
    cpl_image_set(img, 1, 1, 1.0);
    cpl_image_set(img, 1, 2, 2.0);
    cpl_image_set(img, 2, 1, 3.0);
    cpl_image_set(img, 2, 2, 4.0);
    cpl_imagelist_set(cube, img, 0);

    error = eris_ifu_debug_cube(cube);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Test cube with multiple images of different types */
    const cpl_type test_types[] = {
        CPL_TYPE_INT,
        CPL_TYPE_FLOAT,
        CPL_TYPE_DOUBLE
    };

    for (size_t i = 0; i < sizeof(test_types)/sizeof(test_types[0]); i++) {
        img = cpl_image_new(2, 2, test_types[i]);
        cpl_test_nonnull(img);

        /* Fill with test values */
        cpl_image_set(img, 1, 1, i * 1.5);
        cpl_image_set(img, 1, 2, i * 2.5);
        cpl_image_set(img, 2, 1, i * 3.5);
        cpl_image_set(img, 2, 2, i * 4.5);

        cpl_imagelist_set(cube, img, i + 1);
    }

    error = eris_ifu_debug_cube(cube);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Test cube with images of different sizes */
    const int test_sizes[][2] = {
        {1, 1},   /* Single pixel */
        {3, 2},   /* Rectangle */
        {2, 3},   /* Rectangle (transposed) */
        {4, 4}    /* Larger square */
    };

    for (size_t i = 0; i < sizeof(test_sizes)/sizeof(test_sizes[0]); i++) {
        img = cpl_image_new(test_sizes[i][0], test_sizes[i][1], CPL_TYPE_DOUBLE);
        cpl_test_nonnull(img);

        /* Fill with test pattern */
        for (int y = 1; y <= test_sizes[i][1]; y++) {
            for (int x = 1; x <= test_sizes[i][0]; x++) {
                cpl_image_set(img, x, y, (x + y) * 1.5);
            }
        }

        cpl_imagelist_set(cube, img, i + 4);  /* Continue after previous tests */
    }

    error = eris_ifu_debug_cube(cube);
    cpl_test_eq(error, CPL_ERROR_NONE);
    cpl_image_delete(img);
    /* Test cube with bad pixels */
    img = cpl_image_new(3, 3, CPL_TYPE_DOUBLE);
    cpl_test_nonnull(img);

    /* Set some values */
    for (int y = 1; y <= 3; y++) {
        for (int x = 1; x <= 3; x++) {
            cpl_image_set(img, x, y, x * y * 1.0);
        }
    }

    /* Create and add bad pixel map */
    cpl_mask *bpm = cpl_mask_new(3, 3);
    cpl_test_nonnull(bpm);
    cpl_mask_set(bpm, 2, 2, 1);  /* Mark center pixel as bad */
    //cpl_image_set_bpm(img, bpm);

    cpl_imagelist_set(cube, cpl_image_duplicate(img), cpl_imagelist_get_size(cube));
    cpl_mask_delete(bpm);

    error = eris_ifu_debug_cube(cube);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Test error handling for invalid operations */
    /* Note: Add specific error cases if the CPL implementation allows */

    /* Cleanup */
    cpl_imagelist_delete(cube);
    cpl_image_delete(img);
}

static void test_debug_vector(void)
{
    cpl_vector *vec = NULL;
    cpl_error_code error;

    /* Test with NULL input */
    error = eris_ifu_debug_vector(NULL);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Test empty vector (size 0) */

    vec = cpl_vector_new(0);
    cpl_error_reset();
    cpl_test_null(vec);

    error = eris_ifu_debug_vector(vec);
    cpl_test_eq(error, CPL_ERROR_NONE);

    cpl_vector_delete(vec);

    /* Test single element vector */
    vec = cpl_vector_new(1);
    cpl_test_nonnull(vec);
    cpl_vector_set(vec, 0, 3.14159);
    error = eris_ifu_debug_vector(vec);
    cpl_test_eq(error, CPL_ERROR_NONE);
    cpl_vector_delete(vec);

    /* Test vector with various sizes */
    const int test_sizes[] = {2, 3, 5, 10, 100};
    for (size_t i = 0; i < sizeof(test_sizes)/sizeof(test_sizes[0]); i++) {
        vec = cpl_vector_new(test_sizes[i]);
        cpl_test_nonnull(vec);

        /* Fill with test pattern */
        for (int j = 0; j < test_sizes[i]; j++) {
            cpl_vector_set(vec, j, j * 1.5);
        }

        error = eris_ifu_debug_vector(vec);
        cpl_test_eq(error, CPL_ERROR_NONE);
        cpl_vector_delete(vec);
    }

    /* Test vector with special values */
    vec = cpl_vector_new(10);
    cpl_test_nonnull(vec);

    /* Set various test values */
    cpl_vector_set(vec, 0, 0.0);          /* Zero */
    cpl_vector_set(vec, 1, 1.0);          /* One */
    cpl_vector_set(vec, 2, -1.0);         /* Negative one */
    cpl_vector_set(vec, 3, 3.14159);      /* Pi */
    cpl_vector_set(vec, 4, 2.71828);      /* e */
    cpl_vector_set(vec, 5, 1e10);         /* Large positive */
    cpl_vector_set(vec, 6, -1e10);        /* Large negative */
    cpl_vector_set(vec, 7, 1e-10);        /* Small positive */
    cpl_vector_set(vec, 8, -1e-10);       /* Small negative */
    cpl_vector_set(vec, 9, 1.23456789);   /* Multiple decimals */

    error = eris_ifu_debug_vector(vec);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Test vector fill operations */
    cpl_vector_fill(vec, 42.0);  /* Fill with constant */
    error = eris_ifu_debug_vector(vec);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Test vector copy */
    cpl_vector *vec_copy = cpl_vector_duplicate(vec);
    cpl_test_nonnull(vec_copy);
    error = eris_ifu_debug_vector(vec_copy);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Test error handling for invalid operations */
    /* Note: Add specific error cases if the CPL implementation allows */

    /* Cleanup */
    cpl_vector_delete(vec);
    cpl_vector_delete(vec_copy);
}

static void test_debug_array(void)
{
    cpl_array *arr_int = NULL, *arr_double = NULL;
    cpl_error_code error;

    /* Test with NULL input */
    error = eris_ifu_debug_array(NULL);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Test with integer array */
    arr_int = cpl_array_new(3, CPL_TYPE_INT);
    cpl_test_nonnull(arr_int);
    cpl_array_set_int(arr_int, 0, 1);
    cpl_array_set_int(arr_int, 1, 2);
    cpl_array_set_int(arr_int, 2, 3);

    error = eris_ifu_debug_array(arr_int);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Test with double array */
    arr_double = cpl_array_new(3, CPL_TYPE_DOUBLE);
    cpl_test_nonnull(arr_double);
    cpl_array_set_double(arr_double, 0, 1.1);
    cpl_array_set_double(arr_double, 1, 2.2);
    cpl_array_set_double(arr_double, 2, 3.3);

    error = eris_ifu_debug_array(arr_double);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Cleanup */
    cpl_array_delete(arr_int);
    cpl_array_delete(arr_double);
}

static void test_debug_table(void)
{
    cpl_table *table = NULL;
    cpl_error_code error;

    /* Test with NULL input */
    error = eris_ifu_debug_table(NULL);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Create a test table */
    table = cpl_table_new(2);
    cpl_test_nonnull(table);

    /* Add columns of different types */
    cpl_table_new_column(table, "INT_COL", CPL_TYPE_INT);
    cpl_table_new_column(table, "DBL_COL", CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, "STR_COL", CPL_TYPE_STRING);

    /* Set some test values */
    cpl_table_set_int(table, "INT_COL", 0, 1);
    cpl_table_set_int(table, "INT_COL", 1, 2);
    cpl_table_set_double(table, "DBL_COL", 0, 1.1);
    cpl_table_set_double(table, "DBL_COL", 1, 2.2);
    cpl_table_set_string(table, "STR_COL", 0, "test1");
    cpl_table_set_string(table, "STR_COL", 1, "test2");

    /* Test debug output */
    error = eris_ifu_debug_table(table);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Cleanup */
    cpl_table_delete(table);
}

static void test_plot_functions(void)
{
    cpl_vector *vec = NULL, *x = NULL, *y1 = NULL, *y2 = NULL;
    cpl_image *img = NULL;
    cpl_error_code error;
    const char *plot_opts[] = {"w l t 'data1'", "w l t 'data2'"};

    /* Create test vectors */
    vec = cpl_vector_new(3);
    cpl_test_nonnull(vec);
    cpl_vector_set(vec, 0, 1.0);
    cpl_vector_set(vec, 1, 2.0);
    cpl_vector_set(vec, 2, 3.0);

    x = cpl_vector_new(3);
    cpl_test_nonnull(x);
    cpl_vector_set(x, 0, 0.0);
    cpl_vector_set(x, 1, 1.0);
    cpl_vector_set(x, 2, 2.0);

    y1 = cpl_vector_new(3);
    cpl_test_nonnull(y1);
    cpl_vector_set(y1, 0, 1.0);
    cpl_vector_set(y1, 1, 2.0);
    cpl_vector_set(y1, 2, 3.0);

    y2 = cpl_vector_new(3);
    cpl_test_nonnull(y2);
    cpl_vector_set(y2, 0, 2.0);
    cpl_vector_set(y2, 1, 4.0);
    cpl_vector_set(y2, 2, 6.0);

    /* Create test image */
    img = cpl_image_new(2, 2, CPL_TYPE_DOUBLE);
    cpl_test_nonnull(img);
    cpl_image_set(img, 1, 1, 1.0);
    cpl_image_set(img, 1, 2, 2.0);
    cpl_image_set(img, 2, 1, 3.0);
    cpl_image_set(img, 2, 2, 4.0);

    /* Test plot functions */
    error = eris_ifu_plot_vector("set title 'Vector Test';", "w l;", vec);
    cpl_test_eq(error, CPL_ERROR_NONE);

    error = eris_ifu_plot_vectors_xy("set title 'XY Plot Test';", "w l;", x, y1);
    cpl_test_eq(error, CPL_ERROR_NONE);

    error = eris_ifu_plot_vectors2("set title 'Multiple Vectors Test';", plot_opts, x, y1, y2);
    cpl_test_eq(error, CPL_ERROR_NONE);

    error = eris_ifu_plot_image("set title 'Image Test';", "", img);
    cpl_test_eq(error, CPL_ERROR_NONE);

    /* Cleanup */
    cpl_vector_delete(vec);
    cpl_vector_delete(x);
    cpl_vector_delete(y1);
    cpl_vector_delete(y2);
    cpl_image_delete(img);
}
