/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <float.h>
#include <math.h>
#include <string.h>
#include "eris_nix_test_defs.h"
#include "eris_ifu_sdp.h"
#include <hdrl.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_ifu_sdp_test  Unit test of eris_ifu_sdp
 *
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                        Forward declarations for internal functions
 -----------------------------------------------------------------------------*/

/* These functions are defined in eris_ifu_sdp.c but not exposed in the header */
cpl_error_code eris_ifu_cplarray_sort(cpl_array *aArray, cpl_boolean aOrder);
const char * eris_ifu_pfits_get_origfile(const cpl_propertylist *aHeaders);

/*-----------------------------------------------------------------------------
                                   Defines
 -----------------------------------------------------------------------------*/

#define TEST_NCOMBINE 3

/*-----------------------------------------------------------------------------
                                Helper Functions
 -----------------------------------------------------------------------------*/

/**
 * @brief Create a test SDP properties structure with sample values
 */
static eris_ifu_sdp_properties * create_test_properties(void) {
    eris_ifu_sdp_properties * props = eris_ifu_sdp_properties_new();
    
    props->ncombine = TEST_NCOMBINE;
    props->exptime = 600.0;
    props->texptime = 1800.0;
    props->mjd_end = 60000.5;
    props->fovcenter[0] = 180.0;  /* RA */
    props->fovcenter[1] = -30.0;  /* DEC */
    props->wlenrange[0] = 1.5;    /* WAVELMIN in microns */
    props->wlenrange[1] = 1.8;    /* WAVELMAX in microns */
    props->wlerror = 0.026;
    props->specres = 3500.0;
    props->skyres = 0.8;
    props->skyrerr = 0.1;
    props->pixnoise = 1.5e-17;
    props->abmaglimit = 24.5;
    props->fluxcal = CPL_TRUE;
    props->nobj = 1;
    
    /* OBIDs */
    props->obid = cpl_array_new(TEST_NCOMBINE, CPL_TYPE_LONG);
    cpl_array_set_long(props->obid, 0, 123456);
    cpl_array_set_long(props->obid, 1, 123457);
    cpl_array_set_long(props->obid, 2, 123458);
    
    /* Program IDs */
    props->progid = cpl_array_new(TEST_NCOMBINE, CPL_TYPE_STRING);
    cpl_array_set_string(props->progid, 0, "0110.A-1234(A)");
    cpl_array_set_string(props->progid, 1, "0110.A-1234(A)");
    cpl_array_set_string(props->progid, 2, "0110.A-1234(A)");
    
    /* Provenance */
    props->prov = cpl_propertylist_new();
    cpl_propertylist_append_string(props->prov, "PROV1", "ERIS.2024-01-01T00:00:00.000.fits");
    cpl_propertylist_append_string(props->prov, "PROV2", "ERIS.2024-01-01T00:01:00.000.fits");
    cpl_propertylist_append_string(props->prov, "PROV3", "ERIS.2024-01-01T00:02:00.000.fits");
    
    /* Associated files */
    props->asson = cpl_array_new(2, CPL_TYPE_STRING);
    cpl_array_set_string(props->asson, 0, "skip_this.fits");  /* Skipped when nobj==1 */
    cpl_array_set_string(props->asson, 1, "white_light.fits");
    
    /* String properties */
    props->prodcatg = cpl_strdup("SCIENCE.CUBE.IFS");
    props->procsoft = "eris/1.9.0";  /* Not freed - points to header data */
    props->obstech = cpl_strdup("IFU");
    props->specsys = cpl_strdup("TOPOCENT");
    props->referenc = NULL;
    
    return props;
}

/**
 * @brief Create a test header with MJD-OBS keyword
 */
static cpl_propertylist * create_test_header(void) {
    cpl_propertylist * header = cpl_propertylist_new();
    
    cpl_propertylist_update_double(header, "MJD-OBS", 60000.0);
    cpl_propertylist_update_string(header, "INSTRUME", "ERIS");
    cpl_propertylist_update_double(header, "EXPTIME", 600.0);
    
    return header;
}

/*-----------------------------------------------------------------------------
                                Test Functions
 -----------------------------------------------------------------------------*/

/**
 * @brief Test eris_ifu_sdp_properties_new creates valid structure
 */
static void test_properties_new(void) {
    eris_ifu_sdp_properties * props = eris_ifu_sdp_properties_new();
    
    cpl_test_nonnull(props);
    
    /* Verify all fields are zero-initialized */
    cpl_test_eq(props->ncombine, 0);
    cpl_test_abs(props->exptime, 0.0, DBL_EPSILON);
    cpl_test_abs(props->texptime, 0.0, DBL_EPSILON);
    cpl_test_abs(props->mjd_end, 0.0, DBL_EPSILON);
    cpl_test_null(props->obid);
    cpl_test_null(props->progid);
    cpl_test_null(props->prov);
    cpl_test_null(props->asson);
    cpl_test_null(props->prodcatg);
    cpl_test_null(props->procsoft);
    cpl_test_null(props->obstech);
    cpl_test_null(props->specsys);
    cpl_test_null(props->referenc);
    
    eris_ifu_sdp_properties_delete(props);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_ifu_sdp_properties_delete with NULL pointer
 */
static void test_properties_delete_null(void) {
    /* Should not crash */
    eris_ifu_sdp_properties_delete(NULL);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_ifu_sdp_properties_delete with valid pointer
 */
static void test_properties_delete_valid(void) {
    eris_ifu_sdp_properties * props = create_test_properties();
    cpl_test_nonnull(props);
    
    /* Should free all memory without error */
    eris_ifu_sdp_properties_delete(props);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_ifu_sdp_properties_delete with partial initialization
 */
static void test_properties_delete_partial(void) {
    eris_ifu_sdp_properties * props = eris_ifu_sdp_properties_new();
    
    /* Only set some fields */
    props->obid = cpl_array_new(2, CPL_TYPE_LONG);
    props->prodcatg = cpl_strdup("TEST");
    
    /* Should handle partial initialization */
    eris_ifu_sdp_properties_delete(props);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_ifu_cplarray_sort with double array ascending
 */
static void test_cplarray_sort_double_asc(void) {
    cpl_array * arr = cpl_array_new(5, CPL_TYPE_DOUBLE);
    cpl_array_set_double(arr, 0, 3.0);
    cpl_array_set_double(arr, 1, 1.0);
    cpl_array_set_double(arr, 2, 4.0);
    cpl_array_set_double(arr, 3, 1.5);
    cpl_array_set_double(arr, 4, 2.0);
    
    cpl_error_code code = eris_ifu_cplarray_sort(arr, CPL_TRUE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Verify sorted order */
    cpl_test_abs(cpl_array_get_double(arr, 0, NULL), 1.0, DBL_EPSILON);
    cpl_test_abs(cpl_array_get_double(arr, 1, NULL), 1.5, DBL_EPSILON);
    cpl_test_abs(cpl_array_get_double(arr, 2, NULL), 2.0, DBL_EPSILON);
    cpl_test_abs(cpl_array_get_double(arr, 3, NULL), 3.0, DBL_EPSILON);
    cpl_test_abs(cpl_array_get_double(arr, 4, NULL), 4.0, DBL_EPSILON);
    
    cpl_array_delete(arr);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_ifu_cplarray_sort with double array descending
 */
static void test_cplarray_sort_double_desc(void) {
    cpl_array * arr = cpl_array_new(5, CPL_TYPE_DOUBLE);
    cpl_array_set_double(arr, 0, 3.0);
    cpl_array_set_double(arr, 1, 1.0);
    cpl_array_set_double(arr, 2, 4.0);
    cpl_array_set_double(arr, 3, 1.5);
    cpl_array_set_double(arr, 4, 2.0);
    
    cpl_error_code code = eris_ifu_cplarray_sort(arr, CPL_FALSE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Verify descending order */
    cpl_test_abs(cpl_array_get_double(arr, 0, NULL), 4.0, DBL_EPSILON);
    cpl_test_abs(cpl_array_get_double(arr, 1, NULL), 3.0, DBL_EPSILON);
    cpl_test_abs(cpl_array_get_double(arr, 2, NULL), 2.0, DBL_EPSILON);
    cpl_test_abs(cpl_array_get_double(arr, 3, NULL), 1.5, DBL_EPSILON);
    cpl_test_abs(cpl_array_get_double(arr, 4, NULL), 1.0, DBL_EPSILON);
    
    cpl_array_delete(arr);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_ifu_cplarray_sort with int array
 */
static void test_cplarray_sort_int(void) {
    cpl_array * arr = cpl_array_new(4, CPL_TYPE_INT);
    cpl_array_set_int(arr, 0, 5);
    cpl_array_set_int(arr, 1, 2);
    cpl_array_set_int(arr, 2, 8);
    cpl_array_set_int(arr, 3, 1);
    
    cpl_error_code code = eris_ifu_cplarray_sort(arr, CPL_TRUE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    cpl_test_eq(cpl_array_get_int(arr, 0, NULL), 1);
    cpl_test_eq(cpl_array_get_int(arr, 1, NULL), 2);
    cpl_test_eq(cpl_array_get_int(arr, 2, NULL), 5);
    cpl_test_eq(cpl_array_get_int(arr, 3, NULL), 8);
    
    cpl_array_delete(arr);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_ifu_cplarray_sort with long array
 */
static void test_cplarray_sort_long(void) {
    cpl_array * arr = cpl_array_new(4, CPL_TYPE_LONG);
    cpl_array_set_long(arr, 0, 123458);
    cpl_array_set_long(arr, 1, 123456);
    cpl_array_set_long(arr, 2, 123459);
    cpl_array_set_long(arr, 3, 123457);
    
    cpl_error_code code = eris_ifu_cplarray_sort(arr, CPL_TRUE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    cpl_test_eq(cpl_array_get_long(arr, 0, NULL), 123456);
    cpl_test_eq(cpl_array_get_long(arr, 1, NULL), 123457);
    cpl_test_eq(cpl_array_get_long(arr, 2, NULL), 123458);
    cpl_test_eq(cpl_array_get_long(arr, 3, NULL), 123459);
    
    cpl_array_delete(arr);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_ifu_cplarray_sort with string array
 */
static void test_cplarray_sort_string(void) {
    cpl_array * arr = cpl_array_new(4, CPL_TYPE_STRING);
    cpl_array_set_string(arr, 0, "delta");
    cpl_array_set_string(arr, 1, "alpha");
    cpl_array_set_string(arr, 2, "gamma");
    cpl_array_set_string(arr, 3, "beta");
    
    cpl_error_code code = eris_ifu_cplarray_sort(arr, CPL_TRUE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    cpl_test_eq_string(cpl_array_get_string(arr, 0), "alpha");
    cpl_test_eq_string(cpl_array_get_string(arr, 1), "beta");
    cpl_test_eq_string(cpl_array_get_string(arr, 2), "delta");
    cpl_test_eq_string(cpl_array_get_string(arr, 3), "gamma");
    
    cpl_array_delete(arr);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_ifu_cplarray_sort with float array
 */
static void test_cplarray_sort_float(void) {
    cpl_array * arr = cpl_array_new(3, CPL_TYPE_FLOAT);
    cpl_array_set_float(arr, 0, 2.5f);
    cpl_array_set_float(arr, 1, 1.5f);
    cpl_array_set_float(arr, 2, 3.5f);
    
    cpl_error_code code = eris_ifu_cplarray_sort(arr, CPL_TRUE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    cpl_test_abs(cpl_array_get_float(arr, 0, NULL), 1.5f, 1e-6);
    cpl_test_abs(cpl_array_get_float(arr, 1, NULL), 2.5f, 1e-6);
    cpl_test_abs(cpl_array_get_float(arr, 2, NULL), 3.5f, 1e-6);
    
    cpl_array_delete(arr);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_ifu_cplarray_sort with NULL array
 */
static void test_cplarray_sort_null(void) {
    cpl_error_code code = eris_ifu_cplarray_sort(NULL, CPL_TRUE);
    cpl_test_eq(code, CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_ifu_pfits_get_origfile with valid input
 */
static void test_pfits_get_origfile_valid(void) {
    cpl_propertylist * plist = cpl_propertylist_new();
    cpl_propertylist_update_string(plist, "ORIGFILE", "ERIS.2024-01-01T00:00:00.000.fits");
    
    const char * origfile = eris_ifu_pfits_get_origfile(plist);
    cpl_test_nonnull(origfile);
    cpl_test_eq_string(origfile, "ERIS.2024-01-01T00:00:00.000.fits");
    
    cpl_propertylist_delete(plist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_ifu_pfits_get_origfile with missing keyword
 */
static void test_pfits_get_origfile_missing(void) {
    cpl_propertylist * plist = cpl_propertylist_new();
    
    const char * origfile = eris_ifu_pfits_get_origfile(plist);
    cpl_test_null(origfile);
    
    cpl_error_reset();
    cpl_propertylist_delete(plist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_ifu_sdp_properties_update with NULL inputs
 */
static void test_properties_update_null_inputs(void) {
    cpl_propertylist * header = create_test_header();
    eris_ifu_sdp_properties * props = create_test_properties();
    
    /* NULL header */
    cpl_error_code code = eris_ifu_sdp_properties_update(NULL, props);
    cpl_test_eq(code, CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    
    /* NULL properties */
    code = eris_ifu_sdp_properties_update(header, NULL);
    cpl_test_eq(code, CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    
    eris_ifu_sdp_properties_delete(props);
    cpl_propertylist_delete(header);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_ifu_sdp_properties_update with valid inputs
 */
static void test_properties_update_valid(void) {
    cpl_propertylist * header = create_test_header();
    eris_ifu_sdp_properties * props = create_test_properties();
    
    cpl_error_code code = eris_ifu_sdp_properties_update(header, props);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Verify key SDP keywords were added */
    cpl_test(cpl_propertylist_has(header, "RA"));
    cpl_test(cpl_propertylist_has(header, "DEC"));
    cpl_test(cpl_propertylist_has(header, "EXPTIME"));
    cpl_test(cpl_propertylist_has(header, "TEXPTIME"));
    cpl_test(cpl_propertylist_has(header, "NCOMBINE"));
    cpl_test(cpl_propertylist_has(header, "MJD-END"));
    cpl_test(cpl_propertylist_has(header, "OBID1"));
    cpl_test(cpl_propertylist_has(header, "PROG_ID"));
    cpl_test(cpl_propertylist_has(header, "PRODCATG"));
    cpl_test(cpl_propertylist_has(header, "PROCSOFT"));
    cpl_test(cpl_propertylist_has(header, "OBSTECH"));
    cpl_test(cpl_propertylist_has(header, "FLUXCAL"));
    cpl_test(cpl_propertylist_has(header, "WAVELMIN"));
    cpl_test(cpl_propertylist_has(header, "WAVELMAX"));
    cpl_test(cpl_propertylist_has(header, "SPEC_RES"));
    cpl_test(cpl_propertylist_has(header, "SPECSYS"));
    cpl_test(cpl_propertylist_has(header, "SKY_RES"));
    cpl_test(cpl_propertylist_has(header, "PIXNOISE"));
    cpl_test(cpl_propertylist_has(header, "ABMAGLIM"));
    cpl_test(cpl_propertylist_has(header, "REFERENC"));
    
    /* Verify some values */
    cpl_test_abs(cpl_propertylist_get_double(header, "RA"), 180.0, DBL_EPSILON);
    cpl_test_abs(cpl_propertylist_get_double(header, "DEC"), -30.0, DBL_EPSILON);
    cpl_test_abs(cpl_propertylist_get_double(header, "TEXPTIME"), 1800.0, DBL_EPSILON);
    cpl_test_eq(cpl_propertylist_get_int(header, "NCOMBINE"), TEST_NCOMBINE);
    
    /* WAVELMIN/MAX should be in nm (microns * 1000) */
    cpl_test_abs(cpl_propertylist_get_double(header, "WAVELMIN"), 1500.0, 0.1);
    cpl_test_abs(cpl_propertylist_get_double(header, "WAVELMAX"), 1800.0, 0.1);
    
    cpl_test_eq_string(cpl_propertylist_get_string(header, "FLUXCAL"), "ABSOLUTE");
    cpl_test_eq_string(cpl_propertylist_get_string(header, "OBSTECH"), "IFU");
    cpl_test_eq_string(cpl_propertylist_get_string(header, "SPECSYS"), "TOPOCENT");
    
    eris_ifu_sdp_properties_delete(props);
    cpl_propertylist_delete(header);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_ifu_sdp_properties_update with FLUXCAL=FALSE
 */
static void test_properties_update_uncalibrated(void) {
    cpl_propertylist * header = create_test_header();
    eris_ifu_sdp_properties * props = create_test_properties();
    
    props->fluxcal = CPL_FALSE;
    
    cpl_error_code code = eris_ifu_sdp_properties_update(header, props);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    cpl_test_eq_string(cpl_propertylist_get_string(header, "FLUXCAL"), "UNCALIBRATED");
    
    eris_ifu_sdp_properties_delete(props);
    cpl_propertylist_delete(header);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_ifu_sdp_properties_update with multiple unique OBIDs
 */
static void test_properties_update_multiple_obids(void) {
    cpl_propertylist * header = create_test_header();
    eris_ifu_sdp_properties * props = create_test_properties();
    
    /* Set different OBIDs */
    cpl_array_set_long(props->obid, 0, 100);
    cpl_array_set_long(props->obid, 1, 200);
    cpl_array_set_long(props->obid, 2, 300);
    
    cpl_error_code code = eris_ifu_sdp_properties_update(header, props);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Should have OBID1, OBID2, OBID3 */
    cpl_test(cpl_propertylist_has(header, "OBID1"));
    cpl_test(cpl_propertylist_has(header, "OBID2"));
    cpl_test(cpl_propertylist_has(header, "OBID3"));
    
    eris_ifu_sdp_properties_delete(props);
    cpl_propertylist_delete(header);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_ifu_sdp_properties_update with duplicate OBIDs
 */
static void test_properties_update_duplicate_obids(void) {
    cpl_propertylist * header = create_test_header();
    eris_ifu_sdp_properties * props = create_test_properties();
    
    /* Set same OBID for all */
    cpl_array_set_long(props->obid, 0, 123456);
    cpl_array_set_long(props->obid, 1, 123456);
    cpl_array_set_long(props->obid, 2, 123456);
    
    cpl_error_code code = eris_ifu_sdp_properties_update(header, props);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Should only have OBID1 (duplicates removed) */
    cpl_test(cpl_propertylist_has(header, "OBID1"));
    cpl_test(!cpl_propertylist_has(header, "OBID2"));
    
    eris_ifu_sdp_properties_delete(props);
    cpl_propertylist_delete(header);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_ifu_sdp_properties_update with negative skyres (default value)
 */
static void test_properties_update_default_skyres(void) {
    cpl_propertylist * header = create_test_header();
    eris_ifu_sdp_properties * props = create_test_properties();
    
    /* Negative skyres indicates default value */
    props->skyres = -0.5;
    
    cpl_error_code code = eris_ifu_sdp_properties_update(header, props);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* SKY_RES should be absolute value */
    cpl_test_abs(cpl_propertylist_get_double(header, "SKY_RES"), 0.5, DBL_EPSILON);
    
    /* Comment should indicate "default" */
    const char * comment = cpl_propertylist_get_comment(header, "SKY_RES");
    cpl_test_nonnull(comment);
    cpl_test(strstr(comment, "default") != NULL);
    
    eris_ifu_sdp_properties_delete(props);
    cpl_propertylist_delete(header);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_ifu_sdp_properties_update with measured skyres
 */
static void test_properties_update_measured_skyres(void) {
    cpl_propertylist * header = create_test_header();
    eris_ifu_sdp_properties * props = create_test_properties();
    
    /* Positive skyres indicates measured value */
    props->skyres = 0.8;
    
    cpl_error_code code = eris_ifu_sdp_properties_update(header, props);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    cpl_test_abs(cpl_propertylist_get_double(header, "SKY_RES"), 0.8, DBL_EPSILON);
    
    /* Comment should indicate "measured" */
    const char * comment = cpl_propertylist_get_comment(header, "SKY_RES");
    cpl_test_nonnull(comment);
    cpl_test(strstr(comment, "measured") != NULL);
    
    eris_ifu_sdp_properties_delete(props);
    cpl_propertylist_delete(header);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test multiple create/delete cycles for memory leaks
 */
static void test_properties_memory_cycles(void) {
    const int ncycles = 10;
    
    for (int i = 0; i < ncycles; i++) {
        eris_ifu_sdp_properties * props = create_test_properties();
        cpl_test_nonnull(props);
        eris_ifu_sdp_properties_delete(props);
    }
    
    cpl_test_error(CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_ifu_sdp module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    /* Test eris_ifu_sdp_properties_new */
    test_properties_new();
    
    /* Test eris_ifu_sdp_properties_delete */
    test_properties_delete_null();
    test_properties_delete_valid();
    test_properties_delete_partial();
    
    /* Test eris_ifu_cplarray_sort */
    test_cplarray_sort_double_asc();
    test_cplarray_sort_double_desc();
    test_cplarray_sort_int();
    test_cplarray_sort_long();
    test_cplarray_sort_string();
    test_cplarray_sort_float();
    test_cplarray_sort_null();
    
    /* Test eris_ifu_pfits_get_origfile */
    test_pfits_get_origfile_valid();
    test_pfits_get_origfile_missing();
    
    /* Test eris_ifu_sdp_properties_update */
    test_properties_update_null_inputs();
    test_properties_update_valid();
    test_properties_update_uncalibrated();
    test_properties_update_multiple_obids();
    test_properties_update_duplicate_obids();
    test_properties_update_default_skyres();
    test_properties_update_measured_skyres();
    
    /* Test memory management */
    test_properties_memory_cycles();

    return cpl_test_end(0);
}

/**@}*/

