/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <float.h>
#include <math.h>
#include <string.h>
#include "eris_nix_test_defs.h"
#include "eris_nix_catalogue.h"
#include "eris_nix_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_catalogue_test  Unit test of eris_nix_catalogue
 *
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                   Defines
 -----------------------------------------------------------------------------*/

#define TEST_CAT_FILENAME "nix_test_catalogue.fits"
#define TEST_CAT_LOWERCASE_FILENAME "nix_test_catalogue_lowercase.fits"
#define NUM_TEST_SOURCES 20

/*-----------------------------------------------------------------------------
                                Helper Functions
 -----------------------------------------------------------------------------*/

/**
 * @brief Create a test catalogue table with standard column names (RA, DEC)
 */
static cpl_table * create_test_catalogue(int nsources, double ra_center, 
                                         double dec_center, double radius) {
    cpl_table * cat = cpl_table_new(nsources);
    
    /* Create standard columns */
    cpl_table_new_column(cat, "RA", CPL_TYPE_DOUBLE);
    cpl_table_new_column(cat, "DEC", CPL_TYPE_DOUBLE);
    cpl_table_new_column(cat, "Jmag", CPL_TYPE_FLOAT);
    cpl_table_new_column(cat, "Kmag", CPL_TYPE_FLOAT);
    
    /* Fill with test data spread around center */
    for (int i = 0; i < nsources; i++) {
        double angle = 2.0 * CPL_MATH_PI * i / nsources;
        double r = radius * (0.5 + 0.5 * (double)i / nsources);
        double ra = ra_center + r * cos(angle) / cos(dec_center * CPL_MATH_PI / 180.0);
        double dec = dec_center + r * sin(angle);
        
        cpl_table_set_double(cat, "RA", i, ra);
        cpl_table_set_double(cat, "DEC", i, dec);
        cpl_table_set_float(cat, "Jmag", i, 12.0f + i * 0.1f);
        cpl_table_set_float(cat, "Kmag", i, 11.5f + i * 0.1f);
    }
    
    return cat;
}

/**
 * @brief Create a test catalogue with lowercase column names (ra, dec)
 */
static cpl_table * create_test_catalogue_lowercase(int nsources) {
    cpl_table * cat = cpl_table_new(nsources);
    
    /* Create lowercase columns (non-standard naming) */
    cpl_table_new_column(cat, "ra", CPL_TYPE_DOUBLE);
    cpl_table_new_column(cat, "dec", CPL_TYPE_DOUBLE);
    cpl_table_new_column(cat, "mag", CPL_TYPE_FLOAT);
    
    for (int i = 0; i < nsources; i++) {
        cpl_table_set_double(cat, "ra", i, 120.0 + i * 0.01);
        cpl_table_set_double(cat, "dec", i, -30.0 + i * 0.01);
        cpl_table_set_float(cat, "mag", i, 12.0f + i * 0.1f);
    }
    
    return cat;
}

/**
 * @brief Create a test catalogue with Dec column (CASU style)
 */
static cpl_table * create_test_catalogue_dec_style(int nsources) {
    cpl_table * cat = cpl_table_new(nsources);
    
    /* Create CASU-style columns (RA and Dec, not DEC) */
    cpl_table_new_column(cat, "RA", CPL_TYPE_DOUBLE);
    cpl_table_new_column(cat, "Dec", CPL_TYPE_DOUBLE);  /* Note: "Dec" not "DEC" */
    cpl_table_new_column(cat, "mag", CPL_TYPE_FLOAT);
    
    for (int i = 0; i < nsources; i++) {
        cpl_table_set_double(cat, "RA", i, 120.0 + i * 0.01);
        cpl_table_set_double(cat, "Dec", i, -30.0 + i * 0.01);
        cpl_table_set_float(cat, "mag", i, 12.0f + i * 0.1f);
    }
    
    return cat;
}

/**
 * @brief Save a catalogue table to a FITS file
 */
static void save_test_catalogue(cpl_table * cat, const char * filename) {
    cpl_propertylist * pheader = cpl_propertylist_new();
    cpl_propertylist * header = cpl_propertylist_new();
    cpl_table_save(cat, pheader, header, filename, CPL_IO_CREATE);
    cpl_propertylist_delete(pheader);
    cpl_propertylist_delete(header);
}

/*-----------------------------------------------------------------------------
                                Test Functions
 -----------------------------------------------------------------------------*/

/**
 * @brief Test en_catalogue_name_conformance with NULL input
 */
static void test_name_conformance_null_input(void) {
    cpl_error_code error = en_catalogue_name_conformance(NULL);
    cpl_test_eq(error, CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_catalogue_name_conformance with pre-existing error
 */
static void test_name_conformance_preexisting_error(void) {
    cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
    
    cpl_table * cat = create_test_catalogue_lowercase(5);
    cpl_error_code error = en_catalogue_name_conformance(cat);
    cpl_test_eq(error, CPL_ERROR_ILLEGAL_INPUT);
    
    cpl_table_delete(cat);
    cpl_error_reset();
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_catalogue_name_conformance with already-standard names
 */
static void test_name_conformance_standard_names(void) {
    cpl_table * cat = create_test_catalogue(10, 120.0, -30.0, 0.1);
    
    /* Should do nothing since columns are already named RA and DEC */
    cpl_error_code error = en_catalogue_name_conformance(cat);
    cpl_test_eq(error, CPL_ERROR_NONE);
    
    /* Verify columns still exist */
    cpl_test(cpl_table_has_column(cat, "RA"));
    cpl_test(cpl_table_has_column(cat, "DEC"));
    
    cpl_table_delete(cat);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_catalogue_name_conformance with lowercase names
 */
static void test_name_conformance_lowercase_names(void) {
    cpl_table * cat = create_test_catalogue_lowercase(10);
    
    /* Verify original lowercase columns */
    cpl_test(cpl_table_has_column(cat, "ra"));
    cpl_test(cpl_table_has_column(cat, "dec"));
    cpl_test(!cpl_table_has_column(cat, "RA"));
    cpl_test(!cpl_table_has_column(cat, "DEC"));
    
    /* Apply conformance */
    cpl_error_code error = en_catalogue_name_conformance(cat);
    cpl_test_eq(error, CPL_ERROR_NONE);
    
    /* Verify columns are renamed */
    cpl_test(cpl_table_has_column(cat, "RA"));
    cpl_test(cpl_table_has_column(cat, "DEC"));
    
    cpl_table_delete(cat);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_catalogue_name_conformance with CASU-style Dec column
 */
static void test_name_conformance_dec_style(void) {
    cpl_table * cat = create_test_catalogue_dec_style(10);
    
    /* Verify original column names */
    cpl_test(cpl_table_has_column(cat, "RA"));
    cpl_test(cpl_table_has_column(cat, "Dec"));
    cpl_test(!cpl_table_has_column(cat, "DEC"));
    
    /* Apply conformance */
    cpl_error_code error = en_catalogue_name_conformance(cat);
    cpl_test_eq(error, CPL_ERROR_NONE);
    
    /* Verify DEC column is created (Dec is duplicated to DEC) */
    cpl_test(cpl_table_has_column(cat, "RA"));
    cpl_test(cpl_table_has_column(cat, "DEC"));
    
    cpl_table_delete(cat);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_catalogue_load_from_file with NULL filename
 */
static void test_load_from_file_null_input(void) {
    cpl_table * result = en_catalogue_load_from_file(NULL, 0.0, 1.0, -1.0, 1.0);
    cpl_test_null(result);
    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_catalogue_load_from_file with pre-existing error
 */
static void test_load_from_file_preexisting_error(void) {
    cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
    
    cpl_table * result = en_catalogue_load_from_file("test.fits", 0.0, 1.0, -1.0, 1.0);
    cpl_test_null(result);
    
    cpl_error_reset();
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_catalogue_load_from_file with valid file and selection
 */
static void test_load_from_file_valid(void) {
    /* Create and save test catalogue */
    double ra_center = 120.0;
    double dec_center = -30.0;
    double radius = 0.5;  /* degrees */
    
    cpl_table * cat = create_test_catalogue(NUM_TEST_SOURCES, ra_center, 
                                            dec_center, radius);
    save_test_catalogue(cat, TEST_CAT_FILENAME);
    cpl_table_delete(cat);
    
    /* Load with selection window covering all sources */
    double ramin = ra_center - radius * 2.0;
    double ramax = ra_center + radius * 2.0;
    double decmin = dec_center - radius * 2.0;
    double decmax = dec_center + radius * 2.0;
    
    cpl_table * result = en_catalogue_load_from_file(TEST_CAT_FILENAME,
                                                     ramin, ramax, 
                                                     decmin, decmax);
    cpl_test_nonnull(result);
    
    /* Should find most/all sources */
    cpl_size nrows = cpl_table_get_nrow(result);
    cpl_msg_info(cpl_func, "Load from file: found %lld sources", (long long)nrows);
    cpl_test(nrows > 0);
    
    /* Verify standard column names exist */
    cpl_test(cpl_table_has_column(result, "RA"));
    cpl_test(cpl_table_has_column(result, "DEC"));
    
    /* Cleanup */
    cpl_table_delete(result);
    remove(TEST_CAT_FILENAME);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_catalogue_load_from_file with restrictive selection
 */
static void test_load_from_file_restrictive_selection(void) {
    /* Create and save test catalogue */
    double ra_center = 120.0;
    double dec_center = -30.0;
    double radius = 1.0;  /* degrees */
    
    cpl_table * cat = create_test_catalogue(NUM_TEST_SOURCES, ra_center,
                                            dec_center, radius);
    save_test_catalogue(cat, TEST_CAT_FILENAME);
    cpl_table_delete(cat);
    
    /* Load with very restrictive selection window */
    double ramin = ra_center - 0.01;
    double ramax = ra_center + 0.01;
    double decmin = dec_center - 0.01;
    double decmax = dec_center + 0.01;
    
    cpl_table * result = en_catalogue_load_from_file(TEST_CAT_FILENAME,
                                                     ramin, ramax,
                                                     decmin, decmax);
    cpl_test_nonnull(result);
    
    /* Should find fewer sources with restrictive window */
    cpl_size nrows = cpl_table_get_nrow(result);
    cpl_msg_info(cpl_func, "Restrictive selection: found %lld sources", (long long)nrows);
    cpl_test(nrows < NUM_TEST_SOURCES);
    
    /* Cleanup */
    cpl_table_delete(result);
    remove(TEST_CAT_FILENAME);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_catalogue_load_from_file with lowercase column names
 */
static void test_load_from_file_lowercase_columns(void) {
    /* Create and save test catalogue with lowercase columns */
    cpl_table * cat = create_test_catalogue_lowercase(NUM_TEST_SOURCES);
    save_test_catalogue(cat, TEST_CAT_LOWERCASE_FILENAME);
    cpl_table_delete(cat);
    
    /* Load - the function should fix column names */
    cpl_table * result = en_catalogue_load_from_file(TEST_CAT_LOWERCASE_FILENAME,
                                                     119.0, 121.0,
                                                     -31.0, -29.0);
    cpl_test_nonnull(result);
    
    /* Verify standard column names exist (were renamed) */
    cpl_test(cpl_table_has_column(result, "RA"));
    cpl_test(cpl_table_has_column(result, "DEC"));
    
    /* Cleanup */
    cpl_table_delete(result);
    remove(TEST_CAT_LOWERCASE_FILENAME);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_catalogue_load_from_frameset with NULL inputs
 */
static void test_load_from_frameset_null_inputs(void) {
    cpl_frameset * used = cpl_frameset_new();
    
    /* NULL frameset */
    cpl_table * result = en_catalogue_load_from_frameset(NULL, "TEST_TAG",
                                                         0.0, 1.0, -1.0, 1.0,
                                                         used, 0);
    cpl_test_null(result);
    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    
    /* NULL tag */
    cpl_frameset * frameset = cpl_frameset_new();
    result = en_catalogue_load_from_frameset(frameset, NULL,
                                             0.0, 1.0, -1.0, 1.0,
                                             used, 0);
    cpl_test_null(result);
    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    
    cpl_frameset_delete(frameset);
    cpl_frameset_delete(used);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_catalogue_load_from_frameset with missing tag (not required)
 */
static void test_load_from_frameset_missing_tag_optional(void) {
    cpl_frameset * frameset = cpl_frameset_new();
    cpl_frameset * used = cpl_frameset_new();
    
    /* Request non-existent tag, but not required */
    cpl_table * result = en_catalogue_load_from_frameset(frameset, "MISSING_TAG",
                                                         0.0, 1.0, -1.0, 1.0,
                                                         used, 0);  /* not required */
    cpl_test_null(result);
    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NONE);  /* No error since optional */
    
    cpl_frameset_delete(frameset);
    cpl_frameset_delete(used);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_catalogue_load_from_frameset with missing tag (required)
 */
static void test_load_from_frameset_missing_tag_required(void) {
    cpl_frameset * frameset = cpl_frameset_new();
    cpl_frameset * used = cpl_frameset_new();
    
    /* Request non-existent tag, required */
    cpl_table * result = en_catalogue_load_from_frameset(frameset, "MISSING_TAG",
                                                         0.0, 1.0, -1.0, 1.0,
                                                         used, 1);  /* required */
    cpl_test_null(result);
    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_DATA_NOT_FOUND);
    cpl_error_reset();
    
    cpl_frameset_delete(frameset);
    cpl_frameset_delete(used);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_catalogue_load_from_frameset with valid frameset
 */
static void test_load_from_frameset_valid(void) {
    /* Create and save test catalogue */
    cpl_table * cat = create_test_catalogue(NUM_TEST_SOURCES, 120.0, -30.0, 0.5);
    save_test_catalogue(cat, TEST_CAT_FILENAME);
    cpl_table_delete(cat);
    
    /* Create frameset with the catalogue */
    cpl_frameset * frameset = cpl_frameset_new();
    cpl_frame * frame = cpl_frame_new();
    cpl_frame_set_filename(frame, TEST_CAT_FILENAME);
    cpl_frame_set_tag(frame, "PHOTOM_CATALOGUE");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_type(frame, CPL_FRAME_TYPE_TABLE);
    cpl_frameset_insert(frameset, frame);
    
    cpl_frameset * used = cpl_frameset_new();
    
    /* Load from frameset */
    cpl_table * result = en_catalogue_load_from_frameset(frameset, "PHOTOM_CATALOGUE",
                                                         118.0, 122.0, -32.0, -28.0,
                                                         used, 1);
    cpl_test_nonnull(result);
    
    cpl_size nrows = cpl_table_get_nrow(result);
    cpl_msg_info(cpl_func, "Load from frameset: found %lld sources", (long long)nrows);
    cpl_test(nrows > 0);
    
    /* Verify used frameset is populated */
    cpl_test_eq(cpl_frameset_get_size(used), 1);
    
    /* Cleanup */
    cpl_table_delete(result);
    cpl_frameset_delete(frameset);
    cpl_frameset_delete(used);
    remove(TEST_CAT_FILENAME);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_catalogue_conform_naming with NULL inputs
 */
static void test_conform_naming_null_inputs(void) {
    /* NULL frameset */
    cpl_error_code error = en_catalogue_conform_naming(NULL, "TAG");
    cpl_test_eq(error, CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    
    /* NULL tag */
    cpl_frameset * frameset = cpl_frameset_new();
    error = en_catalogue_conform_naming(frameset, NULL);
    cpl_test_eq(error, CPL_ERROR_NULL_INPUT);
    cpl_error_reset();
    
    cpl_frameset_delete(frameset);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_catalogue_conform_naming with missing tag
 */
static void test_conform_naming_missing_tag(void) {
    cpl_frameset * frameset = cpl_frameset_new();
    
    /* Should just warn, not error */
    cpl_error_code error = en_catalogue_conform_naming(frameset, "MISSING_TAG");
    cpl_test_eq(error, CPL_ERROR_NONE);
    
    cpl_frameset_delete(frameset);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test en_catalogue_conform_naming with valid frameset
 */
static void test_conform_naming_valid(void) {
    /* Create and save catalogue with lowercase columns */
    cpl_table * cat = create_test_catalogue_lowercase(10);
    save_test_catalogue(cat, TEST_CAT_LOWERCASE_FILENAME);
    cpl_table_delete(cat);
    
    /* Create frameset */
    cpl_frameset * frameset = cpl_frameset_new();
    cpl_frame * frame = cpl_frame_new();
    cpl_frame_set_filename(frame, TEST_CAT_LOWERCASE_FILENAME);
    cpl_frame_set_tag(frame, "CATALOGUE");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_type(frame, CPL_FRAME_TYPE_TABLE);
    cpl_frameset_insert(frameset, frame);
    
    /* Apply conformance */
    cpl_error_code error = en_catalogue_conform_naming(frameset, "CATALOGUE");
    cpl_test_eq(error, CPL_ERROR_NONE);
    
    /* Reload and verify columns were renamed */
    cpl_table * reloaded = cpl_table_load(TEST_CAT_LOWERCASE_FILENAME, 1, CPL_FALSE);
    cpl_test_nonnull(reloaded);
    cpl_test(cpl_table_has_column(reloaded, "RA"));
    cpl_test(cpl_table_has_column(reloaded, "DEC"));
    
    /* Cleanup */
    cpl_table_delete(reloaded);
    cpl_frameset_delete(frameset);
    remove(TEST_CAT_LOWERCASE_FILENAME);
    cpl_test_error(CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_nix_catalogue module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    /* Test en_catalogue_name_conformance */
    test_name_conformance_null_input();
    test_name_conformance_preexisting_error();
    test_name_conformance_standard_names();
    test_name_conformance_lowercase_names();
    test_name_conformance_dec_style();
    
    /* Test en_catalogue_load_from_file */
    test_load_from_file_null_input();
    test_load_from_file_preexisting_error();
    test_load_from_file_valid();
    test_load_from_file_restrictive_selection();
    test_load_from_file_lowercase_columns();
    
    /* Test en_catalogue_load_from_frameset */
    test_load_from_frameset_null_inputs();
    test_load_from_frameset_missing_tag_optional();
    test_load_from_frameset_missing_tag_required();
    test_load_from_frameset_valid();
    
    /* Test en_catalogue_conform_naming */
    test_conform_naming_null_inputs();
    test_conform_naming_missing_tag();
    test_conform_naming_valid();

    return cpl_test_end(0);
}

/**@}*/


