/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author$
 * $Date$
 * $Revision$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include <eris_nix_detector.h>
#include <eris_nix_dfs.h>
#include <eris_nix_gain_linearity.h>
#include <eris_nix_utils.h>
#include "eris_nix_test_defs.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_detector_test  Unit test of eris_nix_detector
 *
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                       Prototypes
 -----------------------------------------------------------------------------*/

static void eris_nix_detector_test(void);

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Textual representation of CPL frame group
  @param    group     to convert
  @return   textual representation
 */
/*----------------------------------------------------------------------------*/
static void eris_nix_detector_test(void)
{
    cpl_mask           * bpm = NULL;
    cpl_image          * data = NULL;
    cpl_image          * f = NULL;
    gain_linearity     * gain_linearity_val = NULL;
    cpl_image          * imdata = NULL;
    cpl_image          * imerror = NULL;
    located_image      * limage = NULL;
    cpl_image          * linearized = NULL;
    cpl_propertylist   * lin_plist = NULL;
    master_dark        * master_dark_ima = NULL;
    cpl_image          * nonlinear = NULL;
    int                  reject = 0;
    eris_nix_samples   * samples = NULL;
    cpl_image          * test = NULL;

    cpl_image          * error = NULL;
    cpl_propertylist   * plist = NULL;
    cpl_image          * testconf = NULL;
    hdrl_image         * testimg = NULL;
    located_imagelist  * test_limlist = NULL;

    /* Create files containing demo data */

    data = cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(data, 1, 1, TEST_SIZE_X, TEST_SIZE_Y, 50000.0);
    const cpl_size nr = 11;
    const double dit = 1.0;
    const int ndit = 10;
    const double saturation = 60000.0;
    /* f is 0, so all detectors read out at the end of sample i */
    f = cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);

    /* test end_calculate_samples */ 

    samples = end_calculate_samples(NULL, data, nr, dit, f);
    cpl_test(!samples);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    samples = end_calculate_samples(data, NULL, nr, dit, f);
    cpl_test(!samples);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    samples = end_calculate_samples(data, data, 0, dit, f);
    cpl_test(!samples);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    samples = end_calculate_samples(data, data, nr, 0.0, f);
    cpl_test(!samples);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    samples = end_calculate_samples(data, data, nr, dit, NULL);
    cpl_test(!samples);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    samples = end_calculate_samples(data, data, nr, dit, f);

    cpl_test_eq(nr, samples->nr);
    for (cpl_size i=0; i<nr; i++) { 
/*
        cpl_msg_info(cpl_func, "initial s %d %6.2f", (int)i, 
                     cpl_image_get(samples->samples[i], 1, 1, &reject));
*/
        cpl_test_rel((double)(i+1) * 50000.0 * 0.1, 
                     cpl_image_get(samples->samples[i], 1, 1, &reject),
                     DBL_EPSILON);
    }

    /* do we recover the input intensity? */

    test = end_uptheramp_reduce(NULL, dit);
    cpl_test(!test);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    test = end_uptheramp_reduce(samples, 0.0);
    cpl_test(!test);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    test = end_uptheramp_reduce(samples, dit);
    cpl_test_rel(50000.0, cpl_image_get(test, 1, 1, &reject), DBL_EPSILON);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_image_delete(test);

    /* Construct a dummy gain_linearity object */

    gain_linearity_val = engl_gain_linearity_test(TEST_SIZE_X, TEST_SIZE_Y,
                                              10.0, 500.0, -0.5, -0.001,
                                              (hdrl_value) {10.0, 0.5},
                                              saturation);

    /* make the test samples non-linear, as described by the gain_linearity
       object */

    for (cpl_size i=0; i<nr; i++) {
        double t = cpl_image_get(samples->samples[i], 1, 1, &reject);
        t = t / 500.0;
        double nonlinearity = 10.0 + 500.0*t - 0.5*t*t - 0.001*t*t*t;
        cpl_image_fill_window(samples->samples[i], 1, 1, TEST_SIZE_X,
                              TEST_SIZE_Y, nonlinearity);
    }

    /* and get a non-linear result */

    nonlinear = end_uptheramp_reduce(samples, dit);
    cpl_msg_info(cpl_func, "nonlinear result %15.10f",
                 cpl_image_get(nonlinear, 1, 1, &reject));

    /* does linearization work? */

    cpl_size rot = 0;
    cpl_size strx = 1;
    cpl_size stry = 1;
    linearized = end_vacca_linearize_ramp(NULL,
                                          gain_linearity_val,
                                          dit,
                                          nr,
                                          f,
                                          rot, strx, stry,
                                          0, 0);
    cpl_test(!linearized);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    double coeffs[4] = {10.0, 500.0, -0.5, -0.001};
    cpl_binary rejected = 0;
    cpl_binary saturated = 0;
    double pdata = engl_lin_correct(cpl_image_get(nonlinear, 1, 1, &reject),
                                    4, coeffs, 0, saturation, &rejected,
                                    &saturated, CPL_FALSE);
    cpl_msg_info(cpl_func, "lin correct %10.5f %10.5f",
                 cpl_image_get(nonlinear, 1, 1, &reject), pdata);

    linearized = end_vacca_linearize_ramp(nonlinear,
                                          gain_linearity_val,
                                          dit,
                                          nr,
                                          f,
                                          rot, strx, stry,
                                          0, 0);
    cpl_msg_info(cpl_func, "linearized %10.5f",
                 cpl_image_get(linearized, 1, 1, &reject));
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_rel(49846.87612, cpl_image_get(linearized, 1, 1, &reject),
                 1.0e-6);

    /* test the linearization and Vacca error calculation */

    master_dark_ima = en_master_dark_test(TEST_SIZE_X, TEST_SIZE_Y, 15.0, 3,
                                      "SLOW_UP_THE_RAMP");

    /* create a test located_imagelist */

    imdata = cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(imdata, 1, 1, TEST_SIZE_X, TEST_SIZE_Y, 2000.0);
    imerror = cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(imerror, 1, 1, TEST_SIZE_X, TEST_SIZE_Y, 40.0);
    testimg = hdrl_image_create(imdata, imerror);
    testconf = cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(testconf, 1, 1, TEST_SIZE_X, TEST_SIZE_Y, 5.0);
    plist = cpl_propertylist_new();
    cpl_propertylist_prepend_int(plist, "ESO DET NDIT", ndit);
    cpl_propertylist_prepend_double(plist, "ESO DET DIT", dit);
    cpl_propertylist_prepend_int(plist, "ESO DET NDSAMPLES", nr);
    cpl_propertylist_prepend_string(plist, "ESO DET READ CURNAME",
                                    "SLOW_UP_THE_RAMP");
    /* no window */
    cpl_propertylist_prepend_int(plist, "ESO DET SEQ1 WIN ROT", 0);
    cpl_propertylist_prepend_int(plist, "ESO DET SEQ1 WIN STRX", 1);
    cpl_propertylist_prepend_int(plist, "ESO DET SEQ1 WIN STRY", 1);

    /* located_image takes ownership of all input objects */ 

    limage = enu_located_image_new(testimg,
                                   NULL,
                                   testconf,
                                   NULL, NULL,
                                   plist,
                                   NULL, NULL, NULL, NULL, NULL);
    test_limlist = enu_located_imagelist_new(2);
    for (cpl_size i=0; i < 2; i++) {

        /* create a new located_image each time as the located_imagelist
           object will try to delete them all when the time comes */ 

        located_image * newlimg = enu_located_image_duplicate(limage);
        enu_located_imagelist_insert(test_limlist, newlimg, i); 
    }

    /* test call with bad parameters */

    end_linearize_and_variance(NULL,
                               master_dark_ima,
                               test_limlist, -1, -1);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    end_linearize_and_variance(gain_linearity_val,
                               NULL,
                               test_limlist, -1, -1);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    end_linearize_and_variance(gain_linearity_val,
                               master_dark_ima,
                               NULL, -1, -1);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* OK, test operation of routine */

    cpl_msg_info(cpl_func, "before %s", cpl_error_get_message());
    cpl_msg_info(cpl_func, "before %p", (const void*)test_limlist);
    end_linearize_and_variance(gain_linearity_val,
                               master_dark_ima,
                               test_limlist, -1, -1);
    cpl_msg_info(cpl_func, "after %s", cpl_error_get_message());
    cpl_test_error(CPL_ERROR_NONE);

    hdrl_value htest = hdrl_image_get_pixel(test_limlist->limages[0]->
                                            himage, 1, 1, &reject);
    cpl_test_eq(0, reject);
    cpl_test_rel(htest.data, 2008.12, 1.0e-5);
    const double photnoise = htest.data * 1.2 * (nr * nr + 1) /
                             (10.0 * nr * dit * (nr + 1));
    const double readnoise = 12.0 * 6.2 * 6.2 * (nr - 1) /
                             (10.0 * 10.0 * nr * dit * dit * (nr + 1));
    cpl_test_rel(htest.error, sqrt((photnoise + readnoise) / ndit),
                 DBL_EPSILON);

    /* tidy up */

    cpl_mask_delete(bpm);
    cpl_image_delete(data);
    cpl_image_delete(error);
    cpl_image_delete(f);
    engl_gain_linearity_delete(gain_linearity_val);
    cpl_image_delete(imdata);
    cpl_image_delete(imerror);
    cpl_image_delete(linearized);
    enu_located_image_delete(limage);
    cpl_propertylist_delete(lin_plist);
    en_master_dark_delete(master_dark_ima);
    cpl_image_delete(nonlinear);
    end_samples_delete(samples);
    enu_located_imagelist_delete(test_limlist);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_nix_detector module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    eris_nix_detector_test();

    return cpl_test_end(0);

}

/**@}*/
