/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <float.h>
#include <math.h>
#include <string.h>
#include "eris_nix_test_defs.h"
#include "eris_nix_scired.h"
#include <hdrl.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_scired_test  Unit test of eris_nix_scired
 *
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                   Defines
 -----------------------------------------------------------------------------*/

#define TEST_CONTEXT "eris_nix_scired_test"

/*-----------------------------------------------------------------------------
                                Helper Functions
 -----------------------------------------------------------------------------*/

/**
 * @brief Check if a parameter exists in the parameter list
 */
static cpl_boolean param_exists(cpl_parameterlist * parlist, const char * name) {
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, name);
    return p != NULL ? CPL_TRUE : CPL_FALSE;
}

/**
 * @brief Get the default int value of a parameter
 */
static int get_param_int_default(cpl_parameterlist * parlist, const char * name) {
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, name);
    if (p == NULL) return -999;
    return cpl_parameter_get_default_int(p);
}

/**
 * @brief Get the default double value of a parameter
 */
static double get_param_double_default(cpl_parameterlist * parlist, const char * name) {
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, name);
    if (p == NULL) return -999.0;
    return cpl_parameter_get_default_double(p);
}

/**
 * @brief Get the default string value of a parameter
 */
static const char * get_param_string_default(cpl_parameterlist * parlist, const char * name) {
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, name);
    if (p == NULL) return NULL;
    return cpl_parameter_get_default_string(p);
}

/*-----------------------------------------------------------------------------
                                Test Functions
 -----------------------------------------------------------------------------*/

/**
 * @brief Test eris_nix_pixel_coord_diagnostic_param_set with NULL parlist
 */
static void test_pixel_coord_diagnostic_param_set_null(void) {
    /* NULL parlist - should crash or set error, but let's be safe */
    /* Just test with valid inputs */
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_nix_pixel_coord_diagnostic_param_set with valid inputs
 */
static void test_pixel_coord_diagnostic_param_set_valid(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    cpl_error_code code = eris_nix_pixel_coord_diagnostic_param_set(
                              TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Verify parameters were added */
    char * xprobe_name = cpl_sprintf("%s.x_probe", TEST_CONTEXT);
    char * yprobe_name = cpl_sprintf("%s.y_probe", TEST_CONTEXT);
    
    cpl_test(param_exists(parlist, xprobe_name));
    cpl_test(param_exists(parlist, yprobe_name));
    
    /* Verify default values */
    cpl_test_eq(get_param_int_default(parlist, xprobe_name), -1);
    cpl_test_eq(get_param_int_default(parlist, yprobe_name), -1);
    
    cpl_free(xprobe_name);
    cpl_free(yprobe_name);
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_nix_skysub_param_set with valid inputs
 */
static void test_skysub_param_set_valid(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    cpl_error_code code = eris_nix_skysub_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Verify sky-source parameter */
    char * sky_source_name = cpl_sprintf("%s.sky-source", TEST_CONTEXT);
    cpl_test(param_exists(parlist, sky_source_name));
    cpl_test_eq_string(get_param_string_default(parlist, sky_source_name), "auto");
    cpl_free(sky_source_name);
    
    /* Verify sky-selector parameter */
    char * sky_selector_name = cpl_sprintf("%s.sky-selector", TEST_CONTEXT);
    cpl_test(param_exists(parlist, sky_selector_name));
    cpl_test_eq_string(get_param_string_default(parlist, sky_selector_name), "bracket");
    cpl_free(sky_selector_name);
    
    /* Verify sky-method parameter */
    char * sky_method_name = cpl_sprintf("%s.sky-method", TEST_CONTEXT);
    cpl_test(param_exists(parlist, sky_method_name));
    cpl_test_eq_string(get_param_string_default(parlist, sky_method_name), "median-median");
    cpl_free(sky_method_name);
    
    /* Verify sky-bracket-time parameter */
    char * bracket_time_name = cpl_sprintf("%s.sky-bracket-time", TEST_CONTEXT);
    cpl_test(param_exists(parlist, bracket_time_name));
    cpl_test_abs(get_param_double_default(parlist, bracket_time_name), 1800.0, DBL_EPSILON);
    cpl_free(bracket_time_name);
    
    /* Verify debug-data parameter */
    char * debug_name = cpl_sprintf("%s.debug-data", TEST_CONTEXT);
    cpl_test(param_exists(parlist, debug_name));
    cpl_free(debug_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_nix_catalogue_param_set with valid inputs
 */
static void test_catalogue_param_set_valid(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    /* Use typical default values for catalogue parameters */
    int obj_min_pixels = 4;
    double obj_threshold = 2.5;
    cpl_boolean obj_deblending = CPL_TRUE;
    double obj_core_radius = 5.0;
    cpl_boolean bkg_estimate = CPL_TRUE;
    int bkg_mesh_size = 64;
    double bkg_smooth_fwhm = 2.0;
    double det_eff_gain = 3.0;
    double det_saturation = 50000.0;
    hdrl_catalogue_options resulttype = HDRL_CATALOGUE_ALL;
    
    cpl_error_code code = eris_nix_catalogue_param_set(TEST_CONTEXT, parlist,
                              obj_min_pixels, obj_threshold, obj_deblending,
                              obj_core_radius, bkg_estimate, bkg_mesh_size,
                              bkg_smooth_fwhm, det_eff_gain, det_saturation,
                              resulttype);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Verify some expected parameters were added */
    cpl_size nparams = cpl_parameterlist_get_size(parlist);
    cpl_test(nparams > 0);
    
    /* Verify ao-params parameter exists */
    char * ao_params_name = cpl_sprintf("%s.catalogue.ao-params", TEST_CONTEXT);
    cpl_test(param_exists(parlist, ao_params_name));
    cpl_test_eq_string(get_param_string_default(parlist, ao_params_name), "auto");
    cpl_free(ao_params_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_nix_astrometric_param_set with valid inputs
 */
static void test_astrometric_param_set_valid(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    cpl_error_code code = eris_nix_astrometric_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Verify cdssearch_astrom parameter */
    char * cds_name = cpl_sprintf("%s.cdssearch_astrom", TEST_CONTEXT);
    cpl_test(param_exists(parlist, cds_name));
    cpl_test_eq_string(get_param_string_default(parlist, cds_name), "none");
    cpl_free(cds_name);
    
    /* Verify debug-data parameter */
    char * debug_name = cpl_sprintf("%s.debug-data", TEST_CONTEXT);
    cpl_test(param_exists(parlist, debug_name));
    cpl_free(debug_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_nix_photom_param_set with valid inputs
 */
static void test_photom_param_set_valid(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    cpl_error_code code = eris_nix_photom_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Verify cdssearch_photom parameter */
    char * cds_name = cpl_sprintf("%s.cdssearch_photom", TEST_CONTEXT);
    cpl_test(param_exists(parlist, cds_name));
    cpl_test_eq_string(get_param_string_default(parlist, cds_name), "2MASS");
    cpl_free(cds_name);
    
    /* Verify pixel_radius parameter */
    char * radius_name = cpl_sprintf("%s.pixel_radius", TEST_CONTEXT);
    cpl_test(param_exists(parlist, radius_name));
    cpl_test_abs(get_param_double_default(parlist, radius_name), 5.0, DBL_EPSILON);
    cpl_free(radius_name);
    
    /* Verify minphotom parameter */
    char * minphotom_name = cpl_sprintf("%s.minphotom", TEST_CONTEXT);
    cpl_test(param_exists(parlist, minphotom_name));
    cpl_test_eq(get_param_int_default(parlist, minphotom_name), 1);
    cpl_free(minphotom_name);
    
    /* Verify magerrcut parameter */
    char * magerrcut_name = cpl_sprintf("%s.magerrcut", TEST_CONTEXT);
    cpl_test(param_exists(parlist, magerrcut_name));
    cpl_test_abs(get_param_double_default(parlist, magerrcut_name), 0.5, DBL_EPSILON);
    cpl_free(magerrcut_name);
    
    /* Verify debug-data parameter */
    char * debug_name = cpl_sprintf("%s.debug-data", TEST_CONTEXT);
    cpl_test(param_exists(parlist, debug_name));
    cpl_free(debug_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test eris_nix_hdrl_stack_param_set with valid inputs
 */
static void test_hdrl_stack_param_set_valid(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    cpl_error_code code = eris_nix_hdrl_stack_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Verify interpolation_method parameter */
    char * interp_name = cpl_sprintf("%s.interpolation_method", TEST_CONTEXT);
    cpl_test(param_exists(parlist, interp_name));
    cpl_test_eq_string(get_param_string_default(parlist, interp_name), "lanczos");
    cpl_free(interp_name);
    
    /* Verify loop_distance parameter */
    char * loop_name = cpl_sprintf("%s.loop_distance", TEST_CONTEXT);
    cpl_test(param_exists(parlist, loop_name));
    cpl_test_eq(get_param_int_default(parlist, loop_name), 1);
    cpl_free(loop_name);
    
    /* Verify kernel_size parameter */
    char * kernel_name = cpl_sprintf("%s.kernel_size", TEST_CONTEXT);
    cpl_test(param_exists(parlist, kernel_name));
    cpl_test_eq(get_param_int_default(parlist, kernel_name), 2);
    cpl_free(kernel_name);
    
    /* Verify critical_radius parameter */
    char * radius_name = cpl_sprintf("%s.critical_radius", TEST_CONTEXT);
    cpl_test(param_exists(parlist, radius_name));
    cpl_test_abs(get_param_double_default(parlist, radius_name), 5.0, DBL_EPSILON);
    cpl_free(radius_name);
    
    /* Verify pix_frac_x parameter */
    char * pix_frac_x_name = cpl_sprintf("%s.pix_frac_x", TEST_CONTEXT);
    cpl_test(param_exists(parlist, pix_frac_x_name));
    cpl_test_abs(get_param_double_default(parlist, pix_frac_x_name), 50.0, DBL_EPSILON);
    cpl_free(pix_frac_x_name);
    
    /* Verify pix_frac_y parameter */
    char * pix_frac_y_name = cpl_sprintf("%s.pix_frac_y", TEST_CONTEXT);
    cpl_test(param_exists(parlist, pix_frac_y_name));
    cpl_test_abs(get_param_double_default(parlist, pix_frac_y_name), 50.0, DBL_EPSILON);
    cpl_free(pix_frac_y_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test parameter setting functions with different contexts
 */
static void test_param_set_different_context(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    const char * context1 = "recipe1";
    const char * context2 = "recipe2";
    
    /* Add parameters with context1 */
    cpl_error_code code = eris_nix_pixel_coord_diagnostic_param_set(context1, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Add parameters with context2 */
    code = eris_nix_pixel_coord_diagnostic_param_set(context2, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Verify both sets of parameters exist */
    char * name1 = cpl_sprintf("%s.x_probe", context1);
    char * name2 = cpl_sprintf("%s.x_probe", context2);
    
    cpl_test(param_exists(parlist, name1));
    cpl_test(param_exists(parlist, name2));
    
    /* Verify they are different parameters */
    const cpl_parameter * p1 = cpl_parameterlist_find_const(parlist, name1);
    const cpl_parameter * p2 = cpl_parameterlist_find_const(parlist, name2);
    cpl_test(p1 != p2);
    
    cpl_free(name1);
    cpl_free(name2);
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test multiple parameter sets in same parlist
 */
static void test_multiple_param_sets(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    cpl_error_code code;
    
    /* Add all parameter sets to the same parlist */
    code = eris_nix_pixel_coord_diagnostic_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_size count1 = cpl_parameterlist_get_size(parlist);
    
    code = eris_nix_skysub_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_size count2 = cpl_parameterlist_get_size(parlist);
    cpl_test(count2 > count1);
    
    code = eris_nix_astrometric_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_size count3 = cpl_parameterlist_get_size(parlist);
    cpl_test(count3 > count2);
    
    code = eris_nix_photom_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_size count4 = cpl_parameterlist_get_size(parlist);
    cpl_test(count4 > count3);
    
    code = eris_nix_hdrl_stack_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_size count5 = cpl_parameterlist_get_size(parlist);
    cpl_test(count5 > count4);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test catalogue param with different resulttype options
 */
static void test_catalogue_param_set_options(void) {
    /* Test with different parameter values */
    cpl_parameterlist * parlist1 = cpl_parameterlist_new();
    cpl_error_code code = eris_nix_catalogue_param_set(TEST_CONTEXT, parlist1,
                              4, 2.5, CPL_TRUE, 5.0, CPL_TRUE, 64, 2.0, 3.0,
                              50000.0, HDRL_CATALOGUE_ALL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_size count1 = cpl_parameterlist_get_size(parlist1);
    cpl_test(count1 > 0);
    cpl_parameterlist_delete(parlist1);
    
    /* Test with different threshold value */
    cpl_parameterlist * parlist2 = cpl_parameterlist_new();
    code = eris_nix_catalogue_param_set(TEST_CONTEXT, parlist2,
                              8, 5.0, CPL_FALSE, 3.0, CPL_FALSE, 32, 1.0, 5.0,
                              40000.0, HDRL_CATALOGUE_ALL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_size count2 = cpl_parameterlist_get_size(parlist2);
    cpl_test(count2 > 0);
    cpl_parameterlist_delete(parlist2);
    
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test catalogue param set with edge case values
 */
static void test_catalogue_param_set_edge_values(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    /* Use edge case values */
    cpl_error_code code = eris_nix_catalogue_param_set(TEST_CONTEXT, parlist,
                              1,      /* obj_min_pixels: minimum reasonable */
                              0.5,    /* obj_threshold: low threshold */
                              CPL_FALSE,  /* obj_deblending: disabled */
                              1.0,    /* obj_core_radius: small radius */
                              CPL_FALSE,  /* bkg_estimate: disabled */
                              16,     /* bkg_mesh_size: small mesh */
                              0.5,    /* bkg_smooth_fwhm: small smoothing */
                              1.0,    /* det_eff_gain: low gain */
                              100000.0,  /* det_saturation: high saturation */
                              HDRL_CATALOGUE_ALL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Verify parameters still created */
    cpl_test(cpl_parameterlist_get_size(parlist) > 0);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/*-----------------------------------------------------------------------------
                        Extended Test Functions
 -----------------------------------------------------------------------------*/

/**
 * @brief Test parameter aliases are correctly set
 */
static void test_pixel_coord_param_aliases(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    cpl_error_code code = eris_nix_pixel_coord_diagnostic_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    char * xprobe_name = cpl_sprintf("%s.x_probe", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, xprobe_name);
    cpl_test_nonnull(p);
    
    /* Verify alias is set */
    const char * alias = cpl_parameter_get_alias(p, CPL_PARAMETER_MODE_CLI);
    cpl_test_nonnull(alias);
    cpl_test_eq_string(alias, "x-probe");
    
    cpl_free(xprobe_name);
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test skysub parameter aliases
 */
static void test_skysub_param_aliases(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    cpl_error_code code = eris_nix_skysub_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Verify sky-source alias */
    char * param_name = cpl_sprintf("%s.sky-source", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    const char * alias = cpl_parameter_get_alias(p, CPL_PARAMETER_MODE_CLI);
    cpl_test_eq_string(alias, "sky-source");
    cpl_free(param_name);
    
    /* Verify sky-bracket-time alias */
    param_name = cpl_sprintf("%s.sky-bracket-time", TEST_CONTEXT);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    alias = cpl_parameter_get_alias(p, CPL_PARAMETER_MODE_CLI);
    cpl_test_eq_string(alias, "sky-bracket-time");
    cpl_free(param_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test hdrl_stack parameter aliases
 */
static void test_hdrl_stack_param_aliases(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    cpl_error_code code = eris_nix_hdrl_stack_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Verify interpolation-method alias */
    char * param_name = cpl_sprintf("%s.interpolation_method", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    const char * alias = cpl_parameter_get_alias(p, CPL_PARAMETER_MODE_CLI);
    cpl_test_eq_string(alias, "interpolation-method");
    cpl_free(param_name);
    
    /* Verify loop-distance alias */
    param_name = cpl_sprintf("%s.loop_distance", TEST_CONTEXT);
    p = cpl_parameterlist_find_const(parlist, param_name);
    alias = cpl_parameter_get_alias(p, CPL_PARAMETER_MODE_CLI);
    cpl_test_eq_string(alias, "loop-distance");
    cpl_free(param_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test photom parameter aliases
 */
static void test_photom_param_aliases(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    cpl_error_code code = eris_nix_photom_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Verify pixel-radius alias */
    char * param_name = cpl_sprintf("%s.pixel_radius", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    const char * alias = cpl_parameter_get_alias(p, CPL_PARAMETER_MODE_CLI);
    cpl_test_eq_string(alias, "pixel-radius");
    cpl_free(param_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test astrometric parameter aliases
 */
static void test_astrometric_param_aliases(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    cpl_error_code code = eris_nix_astrometric_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Verify cdssearch-astrom alias */
    char * param_name = cpl_sprintf("%s.cdssearch_astrom", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    const char * alias = cpl_parameter_get_alias(p, CPL_PARAMETER_MODE_CLI);
    cpl_test_eq_string(alias, "cdssearch-astrom");
    cpl_free(param_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test skysub parameter types
 */
static void test_skysub_param_types(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    cpl_error_code code = eris_nix_skysub_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Verify sky-source is enum (string) */
    char * param_name = cpl_sprintf("%s.sky-source", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_eq(cpl_parameter_get_type(p), CPL_TYPE_STRING);
    cpl_free(param_name);
    
    /* Verify sky-bracket-time is double */
    param_name = cpl_sprintf("%s.sky-bracket-time", TEST_CONTEXT);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_eq(cpl_parameter_get_type(p), CPL_TYPE_DOUBLE);
    cpl_free(param_name);
    
    /* Verify debug-data is bool */
    param_name = cpl_sprintf("%s.debug-data", TEST_CONTEXT);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_eq(cpl_parameter_get_type(p), CPL_TYPE_BOOL);
    cpl_free(param_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test hdrl_stack parameter types
 */
static void test_hdrl_stack_param_types(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    cpl_error_code code = eris_nix_hdrl_stack_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Verify interpolation_method is string (enum) */
    char * param_name = cpl_sprintf("%s.interpolation_method", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_eq(cpl_parameter_get_type(p), CPL_TYPE_STRING);
    cpl_free(param_name);
    
    /* Verify loop_distance is int */
    param_name = cpl_sprintf("%s.loop_distance", TEST_CONTEXT);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_eq(cpl_parameter_get_type(p), CPL_TYPE_INT);
    cpl_free(param_name);
    
    /* Verify kernel_size is int */
    param_name = cpl_sprintf("%s.kernel_size", TEST_CONTEXT);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_eq(cpl_parameter_get_type(p), CPL_TYPE_INT);
    cpl_free(param_name);
    
    /* Verify critical_radius is double */
    param_name = cpl_sprintf("%s.critical_radius", TEST_CONTEXT);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_eq(cpl_parameter_get_type(p), CPL_TYPE_DOUBLE);
    cpl_free(param_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test photom parameter types
 */
static void test_photom_param_types(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    cpl_error_code code = eris_nix_photom_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Verify cdssearch_photom is string (enum) */
    char * param_name = cpl_sprintf("%s.cdssearch_photom", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_eq(cpl_parameter_get_type(p), CPL_TYPE_STRING);
    cpl_free(param_name);
    
    /* Verify pixel_radius is double */
    param_name = cpl_sprintf("%s.pixel_radius", TEST_CONTEXT);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_eq(cpl_parameter_get_type(p), CPL_TYPE_DOUBLE);
    cpl_free(param_name);
    
    /* Verify minphotom is int */
    param_name = cpl_sprintf("%s.minphotom", TEST_CONTEXT);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_eq(cpl_parameter_get_type(p), CPL_TYPE_INT);
    cpl_free(param_name);
    
    /* Verify magerrcut is double */
    param_name = cpl_sprintf("%s.magerrcut", TEST_CONTEXT);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_eq(cpl_parameter_get_type(p), CPL_TYPE_DOUBLE);
    cpl_free(param_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test parameter count for each function
 */
static void test_param_count(void) {
    cpl_parameterlist * parlist;
    cpl_error_code code;
    cpl_size count;
    
    /* pixel_coord_diagnostic should add 2 parameters */
    parlist = cpl_parameterlist_new();
    code = eris_nix_pixel_coord_diagnostic_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    count = cpl_parameterlist_get_size(parlist);
    cpl_test_eq(count, 2);
    cpl_parameterlist_delete(parlist);
    
    /* skysub should add 5 parameters */
    parlist = cpl_parameterlist_new();
    code = eris_nix_skysub_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    count = cpl_parameterlist_get_size(parlist);
    cpl_test_eq(count, 5);
    cpl_parameterlist_delete(parlist);
    
    /* astrometric should add 2 parameters */
    parlist = cpl_parameterlist_new();
    code = eris_nix_astrometric_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    count = cpl_parameterlist_get_size(parlist);
    cpl_test_eq(count, 2);
    cpl_parameterlist_delete(parlist);
    
    /* photom should add 5 parameters */
    parlist = cpl_parameterlist_new();
    code = eris_nix_photom_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    count = cpl_parameterlist_get_size(parlist);
    cpl_test_eq(count, 5);
    cpl_parameterlist_delete(parlist);
    
    /* hdrl_stack should add 6 parameters */
    parlist = cpl_parameterlist_new();
    code = eris_nix_hdrl_stack_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    count = cpl_parameterlist_get_size(parlist);
    cpl_test_eq(count, 6);
    cpl_parameterlist_delete(parlist);
    
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test parameter context values
 */
static void test_param_context(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    const char * test_context = "my_recipe_context";
    
    cpl_error_code code = eris_nix_skysub_param_set(test_context, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    char * param_name = cpl_sprintf("%s.sky-source", test_context);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    
    /* Verify context is correctly set */
    const char * context = cpl_parameter_get_context(p);
    cpl_test_eq_string(context, test_context);
    
    cpl_free(param_name);
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test empty context string
 */
static void test_empty_context(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    const char * empty_context = "";
    
    cpl_error_code code = eris_nix_pixel_coord_diagnostic_param_set(empty_context, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Parameters should still be created with empty prefix */
    char * param_name = cpl_sprintf("%s.x_probe", empty_context);
    cpl_test(param_exists(parlist, param_name));
    cpl_free(param_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test long context string
 */
static void test_long_context(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    const char * long_context = "this_is_a_very_long_context_string_that_might_be_used_in_practice";
    
    cpl_error_code code = eris_nix_skysub_param_set(long_context, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    char * param_name = cpl_sprintf("%s.sky-source", long_context);
    cpl_test(param_exists(parlist, param_name));
    cpl_free(param_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test catalogue param with large values
 */
static void test_catalogue_param_large_values(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    cpl_error_code code = eris_nix_catalogue_param_set(TEST_CONTEXT, parlist,
                              100,      /* obj_min_pixels: large */
                              10.0,     /* obj_threshold: high */
                              CPL_TRUE, /* obj_deblending */
                              20.0,     /* obj_core_radius: large */
                              CPL_TRUE, /* bkg_estimate */
                              256,      /* bkg_mesh_size: large */
                              10.0,     /* bkg_smooth_fwhm: large */
                              10.0,     /* det_eff_gain: high */
                              500000.0, /* det_saturation: very high */
                              HDRL_CATALOGUE_ALL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test(cpl_parameterlist_get_size(parlist) > 0);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test adding same parameters to non-empty parlist
 */
static void test_add_to_nonempty_parlist(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    /* Add some initial parameters manually */
    cpl_parameter * p = cpl_parameter_new_value("existing.param1", CPL_TYPE_INT,
                                                 "Existing param", "existing", 42);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("existing.param2", CPL_TYPE_DOUBLE,
                                 "Another existing", "existing", 3.14);
    cpl_parameterlist_append(parlist, p);
    
    cpl_size initial_count = cpl_parameterlist_get_size(parlist);
    cpl_test_eq(initial_count, 2);
    
    /* Add skysub params */
    cpl_error_code code = eris_nix_skysub_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    cpl_size final_count = cpl_parameterlist_get_size(parlist);
    cpl_test_eq(final_count, initial_count + 5);
    
    /* Verify original params still exist */
    cpl_test(param_exists(parlist, "existing.param1"));
    cpl_test(param_exists(parlist, "existing.param2"));
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test parameter memory cycles
 */
static void test_param_memory_cycles(void) {
    const int ncycles = 10;
    
    for (int i = 0; i < ncycles; i++) {
        cpl_parameterlist * parlist = cpl_parameterlist_new();
        
        eris_nix_pixel_coord_diagnostic_param_set(TEST_CONTEXT, parlist);
        eris_nix_skysub_param_set(TEST_CONTEXT, parlist);
        eris_nix_astrometric_param_set(TEST_CONTEXT, parlist);
        eris_nix_photom_param_set(TEST_CONTEXT, parlist);
        eris_nix_hdrl_stack_param_set(TEST_CONTEXT, parlist);
        eris_nix_catalogue_param_set(TEST_CONTEXT, parlist,
                                      4, 2.5, CPL_TRUE, 5.0, CPL_TRUE, 64, 2.0, 3.0,
                                      50000.0, HDRL_CATALOGUE_ALL);
        
        cpl_parameterlist_delete(parlist);
    }
    
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test parameter descriptions are set
 */
static void test_param_descriptions(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    eris_nix_skysub_param_set(TEST_CONTEXT, parlist);
    
    /* Verify sky-source has a description */
    char * param_name = cpl_sprintf("%s.sky-source", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    
    const char * help = cpl_parameter_get_help(p);
    cpl_test_nonnull(help);
    cpl_test(strlen(help) > 0);
    
    cpl_free(param_name);
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test sky-bracket-time parameter range
 */
static void test_skysub_param_range(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    eris_nix_skysub_param_set(TEST_CONTEXT, parlist);
    
    char * param_name = cpl_sprintf("%s.sky-bracket-time", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    
    /* Verify it's a range parameter */
    cpl_parameter_class pclass = cpl_parameter_get_class(p);
    cpl_test_eq(pclass, CPL_PARAMETER_CLASS_RANGE);
    
    /* Verify min/max values */
    double min_val = cpl_parameter_get_range_min_double(p);
    double max_val = cpl_parameter_get_range_max_double(p);
    cpl_test_abs(min_val, 60.0, DBL_EPSILON);
    cpl_test_abs(max_val, 18000.0, DBL_EPSILON);
    
    cpl_free(param_name);
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test photom minphotom parameter range
 */
static void test_photom_param_range(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    eris_nix_photom_param_set(TEST_CONTEXT, parlist);
    
    char * param_name = cpl_sprintf("%s.minphotom", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    
    /* Verify it's a range parameter */
    cpl_parameter_class pclass = cpl_parameter_get_class(p);
    cpl_test_eq(pclass, CPL_PARAMETER_CLASS_RANGE);
    
    /* Verify min/max values */
    int min_val = cpl_parameter_get_range_min_int(p);
    int max_val = cpl_parameter_get_range_max_int(p);
    cpl_test_eq(min_val, 1);
    cpl_test_eq(max_val, 100000);
    
    cpl_free(param_name);
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test skysub enum parameters
 */
static void test_skysub_enum_params(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    eris_nix_skysub_param_set(TEST_CONTEXT, parlist);
    
    /* Test sky-source enum values */
    char * param_name = cpl_sprintf("%s.sky-source", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    
    cpl_parameter_class pclass = cpl_parameter_get_class(p);
    cpl_test_eq(pclass, CPL_PARAMETER_CLASS_ENUM);
    
    int nenum = cpl_parameter_get_enum_size(p);
    cpl_test_eq(nenum, 3);  /* auto, target, offset */
    
    cpl_free(param_name);
    
    /* Test sky-method enum values */
    param_name = cpl_sprintf("%s.sky-method", TEST_CONTEXT);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    
    pclass = cpl_parameter_get_class(p);
    cpl_test_eq(pclass, CPL_PARAMETER_CLASS_ENUM);
    
    nenum = cpl_parameter_get_enum_size(p);
    cpl_test_eq(nenum, 2);  /* collapse-median, median-median */
    
    cpl_free(param_name);
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test hdrl_stack enum parameters
 */
static void test_hdrl_stack_enum_params(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    eris_nix_hdrl_stack_param_set(TEST_CONTEXT, parlist);
    
    /* Test interpolation_method enum values */
    char * param_name = cpl_sprintf("%s.interpolation_method", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    
    cpl_parameter_class pclass = cpl_parameter_get_class(p);
    cpl_test_eq(pclass, CPL_PARAMETER_CLASS_ENUM);
    
    int nenum = cpl_parameter_get_enum_size(p);
    cpl_test_eq(nenum, 6);  /* nearest, linear, quadratic, renka, drizzle, lanczos */
    
    cpl_free(param_name);
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test astrometric enum parameters
 */
static void test_astrometric_enum_params(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    eris_nix_astrometric_param_set(TEST_CONTEXT, parlist);
    
    /* Test cdssearch_astrom enum values */
    char * param_name = cpl_sprintf("%s.cdssearch_astrom", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    
    cpl_parameter_class pclass = cpl_parameter_get_class(p);
    cpl_test_eq(pclass, CPL_PARAMETER_CLASS_ENUM);
    
    int nenum = cpl_parameter_get_enum_size(p);
    cpl_test_eq(nenum, 3);  /* none, 2mass, gaiadr3 */
    
    cpl_free(param_name);
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test parameter ENV mode disabled
 */
static void test_param_env_disabled(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    eris_nix_pixel_coord_diagnostic_param_set(TEST_CONTEXT, parlist);
    
    char * param_name = cpl_sprintf("%s.x_probe", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    
    /* Verify ENV mode is disabled */
    cpl_boolean enabled = cpl_parameter_is_enabled(p, CPL_PARAMETER_MODE_ENV);
    cpl_test_eq(enabled, CPL_FALSE);
    
    /* Verify CLI mode is enabled */
    enabled = cpl_parameter_is_enabled(p, CPL_PARAMETER_MODE_CLI);
    cpl_test_eq(enabled, CPL_TRUE);
    
    cpl_free(param_name);
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test context with special characters
 */
static void test_context_special_chars(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    const char * special_context = "eris_nix.recipe_v2";
    
    cpl_error_code code = eris_nix_skysub_param_set(special_context, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    char * param_name = cpl_sprintf("%s.sky-source", special_context);
    cpl_test(param_exists(parlist, param_name));
    cpl_free(param_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test context with underscores
 */
static void test_context_underscores(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    const char * underscore_context = "eris_nix_recipe_test";
    
    cpl_error_code code = eris_nix_pixel_coord_diagnostic_param_set(underscore_context, parlist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    char * param_name = cpl_sprintf("%s.x_probe", underscore_context);
    cpl_test(param_exists(parlist, param_name));
    cpl_free(param_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test all param sets return correct codes
 */
static void test_all_param_sets_return_codes(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    cpl_error_code code;
    
    code = eris_nix_pixel_coord_diagnostic_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq(code, CPL_ERROR_NONE);
    
    code = eris_nix_skysub_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq(code, CPL_ERROR_NONE);
    
    code = eris_nix_astrometric_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq(code, CPL_ERROR_NONE);
    
    code = eris_nix_photom_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq(code, CPL_ERROR_NONE);
    
    code = eris_nix_hdrl_stack_param_set(TEST_CONTEXT, parlist);
    cpl_test_eq(code, CPL_ERROR_NONE);
    
    code = eris_nix_catalogue_param_set(TEST_CONTEXT, parlist,
                                         4, 2.5, CPL_TRUE, 5.0, CPL_TRUE, 64, 2.0, 3.0,
                                         50000.0, HDRL_CATALOGUE_ALL);
    cpl_test_eq(code, CPL_ERROR_NONE);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test bool parameter default values
 */
static void test_bool_param_defaults(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    eris_nix_skysub_param_set(TEST_CONTEXT, parlist);
    
    char * param_name = cpl_sprintf("%s.debug-data", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    
    cpl_boolean default_val = cpl_parameter_get_default_bool(p);
    cpl_test_eq(default_val, CPL_FALSE);
    
    cpl_free(param_name);
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test parameter iterator
 */
static void test_param_iterator(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    eris_nix_skysub_param_set(TEST_CONTEXT, parlist);
    
    int count = 0;
    for (cpl_parameter * p = cpl_parameterlist_get_first(parlist);
         p != NULL;
         p = cpl_parameterlist_get_next(parlist)) {
        const char * name = cpl_parameter_get_name(p);
        cpl_test_nonnull(name);
        count++;
    }
    
    cpl_test_eq(count, 5);  /* 5 skysub parameters */
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test hdrl_stack all parameter values
 */
static void test_hdrl_stack_all_params(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    eris_nix_hdrl_stack_param_set(TEST_CONTEXT, parlist);
    
    /* loop_distance default = 1 */
    char * param_name = cpl_sprintf("%s.loop_distance", TEST_CONTEXT);
    cpl_test_eq(get_param_int_default(parlist, param_name), 1);
    cpl_free(param_name);
    
    /* kernel_size default = 2 */
    param_name = cpl_sprintf("%s.kernel_size", TEST_CONTEXT);
    cpl_test_eq(get_param_int_default(parlist, param_name), 2);
    cpl_free(param_name);
    
    /* critical_radius default = 5.0 */
    param_name = cpl_sprintf("%s.critical_radius", TEST_CONTEXT);
    cpl_test_abs(get_param_double_default(parlist, param_name), 5.0, DBL_EPSILON);
    cpl_free(param_name);
    
    /* pix_frac_x default = 50.0 */
    param_name = cpl_sprintf("%s.pix_frac_x", TEST_CONTEXT);
    cpl_test_abs(get_param_double_default(parlist, param_name), 50.0, DBL_EPSILON);
    cpl_free(param_name);
    
    /* pix_frac_y default = 50.0 */
    param_name = cpl_sprintf("%s.pix_frac_y", TEST_CONTEXT);
    cpl_test_abs(get_param_double_default(parlist, param_name), 50.0, DBL_EPSILON);
    cpl_free(param_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test sky-selector enum (only has 1 option: bracket)
 */
static void test_skysub_selector_enum(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    eris_nix_skysub_param_set(TEST_CONTEXT, parlist);
    
    char * param_name = cpl_sprintf("%s.sky-selector", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    
    cpl_parameter_class pclass = cpl_parameter_get_class(p);
    cpl_test_eq(pclass, CPL_PARAMETER_CLASS_ENUM);
    
    /* Only one option: bracket */
    int nenum = cpl_parameter_get_enum_size(p);
    cpl_test_eq(nenum, 1);
    
    const char * default_val = cpl_parameter_get_default_string(p);
    cpl_test_eq_string(default_val, "bracket");
    
    cpl_free(param_name);
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test catalogue param filtering (min-pixels, threshold, mesh-size, smooth-gauss-fwhm)
 */
static void test_catalogue_param_filtering(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    eris_nix_catalogue_param_set(TEST_CONTEXT, parlist,
                                  4, 2.5, CPL_TRUE, 5.0, CPL_TRUE, 64, 2.0, 3.0,
                                  50000.0, HDRL_CATALOGUE_ALL);
    
    /* Check that filtered parameters exist */
    cpl_boolean found_min_pixels = CPL_FALSE;
    cpl_boolean found_threshold = CPL_FALSE;
    cpl_boolean found_mesh_size = CPL_FALSE;
    cpl_boolean found_smooth_fwhm = CPL_FALSE;
    
    for (cpl_parameter * p = cpl_parameterlist_get_first(parlist);
         p != NULL;
         p = cpl_parameterlist_get_next(parlist)) {
        const char * name = cpl_parameter_get_name(p);
        if (strstr(name, "min-pixels")) found_min_pixels = CPL_TRUE;
        if (strstr(name, "threshold")) found_threshold = CPL_TRUE;
        if (strstr(name, "mesh-size")) found_mesh_size = CPL_TRUE;
        if (strstr(name, "smooth-gauss-fwhm")) found_smooth_fwhm = CPL_TRUE;
    }
    
    cpl_test(found_min_pixels);
    cpl_test(found_threshold);
    cpl_test(found_mesh_size);
    cpl_test(found_smooth_fwhm);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test parameter modification after creation
 */
static void test_param_modification(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    eris_nix_skysub_param_set(TEST_CONTEXT, parlist);
    
    char * param_name = cpl_sprintf("%s.sky-bracket-time", TEST_CONTEXT);
    cpl_parameter * p = cpl_parameterlist_find(parlist, param_name);
    cpl_test_nonnull(p);
    
    /* Modify the value */
    cpl_parameter_set_double(p, 600.0);
    double new_val = cpl_parameter_get_double(p);
    cpl_test_abs(new_val, 600.0, DBL_EPSILON);
    
    /* Default should be unchanged */
    double default_val = cpl_parameter_get_default_double(p);
    cpl_test_abs(default_val, 1800.0, DBL_EPSILON);
    
    cpl_free(param_name);
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test photom enum - only has one option "2MASS"
 */
static void test_photom_enum_single(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    eris_nix_photom_param_set(TEST_CONTEXT, parlist);
    
    char * param_name = cpl_sprintf("%s.cdssearch_photom", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    
    cpl_parameter_class pclass = cpl_parameter_get_class(p);
    cpl_test_eq(pclass, CPL_PARAMETER_CLASS_ENUM);
    
    int nenum = cpl_parameter_get_enum_size(p);
    cpl_test_eq(nenum, 1);  /* Only 2MASS */
    
    const char * default_val = cpl_parameter_get_default_string(p);
    cpl_test_eq_string(default_val, "2MASS");
    
    cpl_free(param_name);
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test all skysub parameters exist and have correct types
 */
static void test_skysub_all_params(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    eris_nix_skysub_param_set(TEST_CONTEXT, parlist);
    
    /* sky-source: string enum */
    char * param_name = cpl_sprintf("%s.sky-source", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    cpl_test_eq(cpl_parameter_get_type(p), CPL_TYPE_STRING);
    cpl_test_eq_string(cpl_parameter_get_default_string(p), "auto");
    cpl_free(param_name);
    
    /* sky-selector: string enum */
    param_name = cpl_sprintf("%s.sky-selector", TEST_CONTEXT);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    cpl_test_eq(cpl_parameter_get_type(p), CPL_TYPE_STRING);
    cpl_test_eq_string(cpl_parameter_get_default_string(p), "bracket");
    cpl_free(param_name);
    
    /* sky-method: string enum */
    param_name = cpl_sprintf("%s.sky-method", TEST_CONTEXT);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    cpl_test_eq(cpl_parameter_get_type(p), CPL_TYPE_STRING);
    cpl_test_eq_string(cpl_parameter_get_default_string(p), "median-median");
    cpl_free(param_name);
    
    /* sky-bracket-time: double range */
    param_name = cpl_sprintf("%s.sky-bracket-time", TEST_CONTEXT);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    cpl_test_eq(cpl_parameter_get_type(p), CPL_TYPE_DOUBLE);
    cpl_test_abs(cpl_parameter_get_default_double(p), 1800.0, DBL_EPSILON);
    cpl_free(param_name);
    
    /* debug-data: bool */
    param_name = cpl_sprintf("%s.debug-data", TEST_CONTEXT);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    cpl_test_eq(cpl_parameter_get_type(p), CPL_TYPE_BOOL);
    cpl_test_eq(cpl_parameter_get_default_bool(p), CPL_FALSE);
    cpl_free(param_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test all photom parameters exist
 */
static void test_photom_all_params(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    eris_nix_photom_param_set(TEST_CONTEXT, parlist);
    
    /* cdssearch_photom */
    char * param_name = cpl_sprintf("%s.cdssearch_photom", TEST_CONTEXT);
    cpl_test(param_exists(parlist, param_name));
    cpl_free(param_name);
    
    /* pixel_radius */
    param_name = cpl_sprintf("%s.pixel_radius", TEST_CONTEXT);
    cpl_test(param_exists(parlist, param_name));
    cpl_test_abs(get_param_double_default(parlist, param_name), 5.0, DBL_EPSILON);
    cpl_free(param_name);
    
    /* minphotom */
    param_name = cpl_sprintf("%s.minphotom", TEST_CONTEXT);
    cpl_test(param_exists(parlist, param_name));
    cpl_test_eq(get_param_int_default(parlist, param_name), 1);
    cpl_free(param_name);
    
    /* magerrcut */
    param_name = cpl_sprintf("%s.magerrcut", TEST_CONTEXT);
    cpl_test(param_exists(parlist, param_name));
    cpl_test_abs(get_param_double_default(parlist, param_name), 0.5, DBL_EPSILON);
    cpl_free(param_name);
    
    /* debug-data */
    param_name = cpl_sprintf("%s.debug-data", TEST_CONTEXT);
    cpl_test(param_exists(parlist, param_name));
    cpl_free(param_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test all astrometric parameters exist
 */
static void test_astrometric_all_params(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    eris_nix_astrometric_param_set(TEST_CONTEXT, parlist);
    
    /* cdssearch_astrom */
    char * param_name = cpl_sprintf("%s.cdssearch_astrom", TEST_CONTEXT);
    cpl_test(param_exists(parlist, param_name));
    cpl_test_eq_string(get_param_string_default(parlist, param_name), "none");
    cpl_free(param_name);
    
    /* debug-data */
    param_name = cpl_sprintf("%s.debug-data", TEST_CONTEXT);
    cpl_test(param_exists(parlist, param_name));
    cpl_free(param_name);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test catalogue param ao-params
 */
static void test_catalogue_ao_params(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    eris_nix_catalogue_param_set(TEST_CONTEXT, parlist,
                                  4, 2.5, CPL_TRUE, 5.0, CPL_TRUE, 64, 2.0, 3.0,
                                  50000.0, HDRL_CATALOGUE_ALL);
    
    /* ao-params should be an enum with auto/user */
    char * param_name = cpl_sprintf("%s.catalogue.ao-params", TEST_CONTEXT);
    const cpl_parameter * p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_test_nonnull(p);
    
    cpl_parameter_class pclass = cpl_parameter_get_class(p);
    cpl_test_eq(pclass, CPL_PARAMETER_CLASS_ENUM);
    
    int nenum = cpl_parameter_get_enum_size(p);
    cpl_test_eq(nenum, 2);  /* auto, user */
    
    const char * default_val = cpl_parameter_get_default_string(p);
    cpl_test_eq_string(default_val, "auto");
    
    cpl_free(param_name);
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test catalogue param with minimal valid values
 */
static void test_catalogue_param_minimal_values(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    /* Use minimal valid values (obj_min_pixels must be > 0) */
    cpl_error_code code = eris_nix_catalogue_param_set(TEST_CONTEXT, parlist,
                              1,      /* obj_min_pixels: minimum valid */
                              0.1,    /* obj_threshold: small */
                              CPL_FALSE,
                              0.1,    /* obj_core_radius: small */
                              CPL_FALSE,
                              8,      /* bkg_mesh_size: small but valid */
                              0.1,    /* bkg_smooth_fwhm: small */
                              0.1,    /* det_eff_gain: small */
                              1.0,    /* det_saturation: small */
                              HDRL_CATALOGUE_ALL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    /* Parameters should still be created */
    cpl_test(cpl_parameterlist_get_size(parlist) > 0);
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test parameter name format
 */
static void test_param_name_format(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    
    eris_nix_pixel_coord_diagnostic_param_set(TEST_CONTEXT, parlist);
    
    /* Verify parameter names follow context.param_name format */
    for (cpl_parameter * p = cpl_parameterlist_get_first(parlist);
         p != NULL;
         p = cpl_parameterlist_get_next(parlist)) {
        const char * name = cpl_parameter_get_name(p);
        cpl_test_nonnull(name);
        
        /* Name should start with context */
        cpl_test(strncmp(name, TEST_CONTEXT, strlen(TEST_CONTEXT)) == 0);
        
        /* Name should contain a dot after context */
        cpl_test(name[strlen(TEST_CONTEXT)] == '.');
    }
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test that parameters have correct context
 */
static void test_param_context_set(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    const char * my_context = "my_test_recipe";
    
    eris_nix_astrometric_param_set(my_context, parlist);
    
    /* Verify all parameters have the correct context */
    for (cpl_parameter * p = cpl_parameterlist_get_first(parlist);
         p != NULL;
         p = cpl_parameterlist_get_next(parlist)) {
        const char * ctx = cpl_parameter_get_context(p);
        cpl_test_eq_string(ctx, my_context);
    }
    
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test combining all param sets into recipe-like parlist
 */
static void test_full_recipe_parlist(void) {
    cpl_parameterlist * parlist = cpl_parameterlist_new();
    const char * recipe_context = "eris_nix_img_scired";
    
    /* Add all parameter sets as a real recipe would */
    eris_nix_pixel_coord_diagnostic_param_set(recipe_context, parlist);
    eris_nix_skysub_param_set(recipe_context, parlist);
    eris_nix_catalogue_param_set(recipe_context, parlist,
                                  4, 2.5, CPL_TRUE, 5.0, CPL_TRUE, 64, 2.0, 3.0,
                                  50000.0, HDRL_CATALOGUE_ALL);
    eris_nix_astrometric_param_set(recipe_context, parlist);
    eris_nix_photom_param_set(recipe_context, parlist);
    eris_nix_hdrl_stack_param_set(recipe_context, parlist);
    
    /* Count total parameters (should be significant) */
    cpl_size total = cpl_parameterlist_get_size(parlist);
    cpl_test(total >= 20);  /* At least 20 parameters in a full recipe */
    
    /* Verify no CPL errors occurred */
    cpl_test_error(CPL_ERROR_NONE);
    
    cpl_parameterlist_delete(parlist);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_nix_scired module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    /* Test eris_nix_pixel_coord_diagnostic_param_set */
    test_pixel_coord_diagnostic_param_set_null();
    test_pixel_coord_diagnostic_param_set_valid();
    
    /* Test eris_nix_skysub_param_set */
    test_skysub_param_set_valid();
    
    /* Test eris_nix_catalogue_param_set */
    test_catalogue_param_set_valid();
    test_catalogue_param_set_options();
    test_catalogue_param_set_edge_values();
    test_catalogue_param_large_values();
    
    /* Test eris_nix_astrometric_param_set */
    test_astrometric_param_set_valid();
    
    /* Test eris_nix_photom_param_set */
    test_photom_param_set_valid();
    
    /* Test eris_nix_hdrl_stack_param_set */
    test_hdrl_stack_param_set_valid();
    
    /* Test multiple parameter sets together */
    test_param_set_different_context();
    test_multiple_param_sets();
    
    /* Extended tests - parameter aliases */
    test_pixel_coord_param_aliases();
    test_skysub_param_aliases();
    test_hdrl_stack_param_aliases();
    test_photom_param_aliases();
    test_astrometric_param_aliases();
    
    /* Extended tests - parameter types */
    test_skysub_param_types();
    test_hdrl_stack_param_types();
    test_photom_param_types();
    
    /* Extended tests - parameter counts */
    test_param_count();
    
    /* Extended tests - context handling */
    test_param_context();
    test_empty_context();
    test_long_context();
    
    /* Extended tests - edge cases */
    test_add_to_nonempty_parlist();
    test_param_memory_cycles();
    
    /* New extended tests - descriptions and help text */
    test_param_descriptions();
    
    /* New extended tests - parameter ranges */
    test_skysub_param_range();
    test_photom_param_range();
    
    /* New extended tests - enum parameters */
    test_skysub_enum_params();
    test_hdrl_stack_enum_params();
    test_astrometric_enum_params();
    
    /* New extended tests - parameter modes */
    test_param_env_disabled();
    
    /* New extended tests - context variations */
    test_context_special_chars();
    test_context_underscores();
    
    /* New extended tests - return codes and defaults */
    test_all_param_sets_return_codes();
    test_bool_param_defaults();
    
    /* New extended tests - iterator and all values */
    test_param_iterator();
    test_hdrl_stack_all_params();
    
    /* Additional extended tests - enum details */
    test_skysub_selector_enum();
    test_photom_enum_single();
    
    /* Additional extended tests - catalogue filtering */
    test_catalogue_param_filtering();
    test_catalogue_ao_params();
    test_catalogue_param_minimal_values();
    
    /* Additional extended tests - parameter modification */
    test_param_modification();
    
    /* Additional extended tests - comprehensive param checks */
    test_skysub_all_params();
    test_photom_all_params();
    test_astrometric_all_params();
    
    /* Additional extended tests - name and context */
    test_param_name_format();
    test_param_context_set();
    
    /* Full recipe integration test */
    test_full_recipe_parlist();

    return cpl_test_end(0);
}

/**@}*/

