/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author$
 * $Date$
 * $Revision$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <eris_utils.h>
/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include <string.h>
#include <cpl.h>

#include "eris_nix_defs.h"
#include "eris_nix_utils.h"
#include "eris_pfits.h"
#include "eris_nix_dfs.h"
#include <hdrl.h>

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

static const char eris_nix_dark_description[] =
"This recipe reduces a set of DARK frames to produce a MASTER_DARK image,\n"
"a bad pixel mask of `hot' pixels, and a measurement of the read-out noise.\n"
"\n"
"Input files:\n"
"\n"
"  DO CATG                         Explanation             Req.  #Frames\n"
"  -------                         -----------             ---   -------\n"
"  "ERIS_NIX_RAW_DARK_DO_CATG
      "                            The raw DARK frames.     Y      2-n\n"
"  "ERIS_NIX_GAIN_PRO_CATG
           "                       Gain information in      Y        1\n"
"                                  file \n"
"                                  detmon_ir_lg_gain_table.fits\n"
"                                  output by recipe\n"
"                                  detmon_ir_lg.\n"
"\n"
"Output files:\n"
"\n"
"  DO CATG                         Explanation \n"
"  -------                         ----------- \n"
"  "ERIS_NIX_MASTER_DARK_IMG_PRO_CATG
             "                     Master DARK result.\n"
"\n"
"  The output will be a FITS file named 'master_dark.fits',\n"
"  containing extensions:\n"
"  - DATA                Dark data\n"
"  - ERR                 Error on the data\n"
"  - DQ                  Data quality\n"
"  - HOT_BPM             hot pixels\n"
"  Relevant FITS keywords are:\n"
"  - QC.READ.NOISE       the measured readout noise\n"
"                        of the array [electrons].\n"
"  - QC.READ.NOISE.VAR   the variance on the readout\n"
"                        noise [electrons^2].\n"
"  - QC.DARK.MED         the median value of DARK \n"
"                        good pixels [ADU].\n"
"  - QC.DARK.MEAN        the mean value of DARK good\n"
"                        pixels [ADU].\n"
"  - QC.DARK.RMS         the r.m.s. of the DARK \n"
"                        good pixels [ADU].\n"
"  - QC.NUMBER.HOT.PIXEL the number of hot pixels\n"
"                        detected.\n"
"  - QC.PARTICLE.RATE    the number of cosmic rays\n"
"                        detected divided by the total\n"
"                        integration time.\n"
"\n";

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/
#define RECIPE_NAME "eris_nix_dark"
#define CONTEXT "eris."RECIPE_NAME
#define ERIS_NIX_DETECTOR_SIZE              2048
#define ERIS_NIX_DETECTOR_SIZE_X            2048
#define ERIS_NIX_DETECTOR_SIZE_Y            2048
cpl_recipe_define(eris_nix_dark, ERIS_BINARY_VERSION, "John Lightfoot",
                  PACKAGE_BUGREPORT, "2017",
                  "Calculate MASTER_DARK and 'hot-pixel' BPM",
                  eris_nix_dark_description);

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options    
  @param    self  the non-NULL parameterlist to fill
  @return   CPL_ERROR_NONE iff everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code eris_nix_dark_fill_parameterlist(cpl_parameterlist
    *self)
{
    cpl_parameterlist * bpm_parlist = NULL;
    cpl_parameterlist * collapse_parlist = NULL;
    hdrl_parameter * filter_defaults = NULL;
    hdrl_parameter * legendre_defaults = NULL;
    hdrl_parameter * minmax_defaults = NULL;
    hdrl_parameter * sigclip_defaults = NULL;

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    /* generate the general parameter list for the collapse */
    hdrl_parameter * mode_def =
        hdrl_collapse_mode_parameter_create(10., 1., 0., HDRL_MODE_MEDIAN, 0);
    sigclip_defaults = hdrl_collapse_sigclip_parameter_create(10.0, 10.0, 3);
    minmax_defaults = hdrl_collapse_minmax_parameter_create(50, 50);
    collapse_parlist = hdrl_collapse_parameter_create_parlist(
      "eris.eris_nix_dark", "collapse", "MEDIAN", sigclip_defaults,
      minmax_defaults, mode_def);

    /* add the subset of parameters to be used, leaving out 'minmax' */

    for (const cpl_parameter * p = 
         cpl_parameterlist_get_first_const(collapse_parlist);
         p != NULL;
         p = cpl_parameterlist_get_next_const(collapse_parlist)) {
        const char * pname = cpl_parameter_get_name(p);
        if (strstr(pname, "minmax") == NULL) {
            cpl_parameter * duplicate = cpl_parameter_duplicate(p);
            cpl_parameterlist_append(self, duplicate);
        }
    }

    /* get the general parameter list for the hot-pixel bpm */

    /* hot (and cold) pixels flagged at 10.0 sigma - see HDRL docs */
    filter_defaults = hdrl_bpm_2d_parameter_create_filtersmooth(10.0, 10.0, 3,
      CPL_FILTER_MEDIAN, CPL_BORDER_FILTER, 21, 21);
    legendre_defaults = hdrl_bpm_2d_parameter_create_legendresmooth(4, 5, 6,
      20, 21, 11, 12, 2, 10);
    bpm_parlist = hdrl_bpm_2d_parameter_create_parlist("eris.eris_nix_dark",
      "hotpix", "FILTER", filter_defaults, legendre_defaults);

    /* add the parameters to be used */

    for (const cpl_parameter * p =
         cpl_parameterlist_get_first(bpm_parlist);
         p != NULL; p = cpl_parameterlist_get_next_const(bpm_parlist)) {
        cpl_parameter * duplicate = cpl_parameter_duplicate(p);
        cpl_parameterlist_append(self, duplicate);
    }    



    /* FPN parameters */
    cpl_parameter * p;
    p = cpl_parameter_new_range("eris.eris_nix_dark.qc_fpn_xmin",
    		CPL_TYPE_INT,
			"qc_fpn_xmin",
			"eris.eris_nix_dark",
			7,1,ERIS_NIX_DETECTOR_SIZE);

    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,"qc_fpn_xmin");

    cpl_parameterlist_append(self, p);


    p = cpl_parameter_new_range("eris.eris_nix_dark.qc_fpn_xmax",
    		CPL_TYPE_INT,
			"qc_fpn_xmax",
			"eris.eris_nix_dark",
			2042,1,ERIS_NIX_DETECTOR_SIZE);

    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,"qc_fpn_xmax");

    cpl_parameterlist_append(self, p);

    p = cpl_parameter_new_range("eris.eris_nix_dark.qc_fpn_ymin",
    		CPL_TYPE_INT,
			"qc_fpn_ymin",
			"eris.eris_nix_dark",
			7,1,ERIS_NIX_DETECTOR_SIZE);

    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,"qc_fpn_ymin");

    cpl_parameterlist_append(self, p);


    p = cpl_parameter_new_range("eris.eris_nix_dark.qc_fpn_ymax",
    		CPL_TYPE_INT,
			"qc_fpn_ymax",
			"eris.eris_nix_dark",
			2042,1,ERIS_NIX_DETECTOR_SIZE);

    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,"qc_fpn_ymax");

    cpl_parameterlist_append(self, p);


    p = cpl_parameter_new_value("eris.eris_nix_dark.qc_fpn_hsize",
    		CPL_TYPE_INT,
			"qc_fpn_hsize",
			"eris.eris_nix_dark",
			10);

    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,"qc_fpn_hsize");

    cpl_parameterlist_append(self, p);


    p = cpl_parameter_new_value("eris.eris_nix_dark.qc_fpn_nsamp",
    		CPL_TYPE_INT,
			"qc_fpn_nsamp",
			"eris.eris_nix_dark",
			1000);

    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,"qc_fpn_nsamp");

    cpl_parameterlist_append(self, p);




    /* cleanup */

    hdrl_parameter_delete(sigclip_defaults);
    hdrl_parameter_delete(minmax_defaults);
    cpl_parameterlist_delete(collapse_parlist);
    hdrl_parameter_delete(mode_def);
    hdrl_parameter_delete(filter_defaults);
    hdrl_parameter_delete(legendre_defaults);
    cpl_parameterlist_delete(bpm_parlist);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int eris_nix_dark(cpl_frameset            * frameset,
                         const cpl_parameterlist * parlist) {

    cpl_propertylist      * applist = NULL;
    hdrl_parameter        * bpm_params =  NULL;
    hdrl_image            * mdark = NULL;
    cpl_image             * contribution = NULL;
    hdrl_parameter        * collapse_params = NULL; 
    cpl_frameset          * dark_frameset = NULL;
    hdrl_image            * diff = NULL;
    const char            * filename = NULL;
    cpl_frameset_iterator * frame_iter = NULL;
    cpl_propertylist      * gain_propertylist = NULL;
    cpl_mask              * hot_bpm = NULL;
    cpl_frame             * inherit = NULL;
    mef_extension_list    * mefs = NULL;
    const cpl_frame       * rawframe = NULL;
    hdrl_imagelist        * dark_himlist = NULL;
    cpl_propertylist      * plist0 = NULL;

    /* Return immediately if an error code is already set */               

    enu_check_error_code("%s():%d: An error is already set: %s",    
      cpl_func, __LINE__, cpl_error_get_where());          

    /* Check input parameters */

    cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);

    /* Set the msg verbosity level from environment variable CPL_MSG_LEVEL */
    /* --can't get this to work */

    cpl_msg_set_level_from_env();
    cpl_msg_severity severity = cpl_msg_get_level();
    cpl_msg_info(cpl_func, "level %d", (int) severity);

    /* check required input tags are present */
    const int ntags = 2;
    const char* required_tags[2] = {ERIS_NIX_RAW_DARK_DO_CATG,
    		ERIS_NIX_GAIN_PRO_CATG,
    };

    cpl_ensure_code(CPL_ERROR_NONE ==
    		eris_dfs_check_input_tags(frameset, required_tags, ntags, 1),
			CPL_ERROR_ILLEGAL_INPUT);


    /* Retrieve input parameters for collapse */

    collapse_params = hdrl_collapse_parameter_parse_parlist(parlist,
                      "eris.eris_nix_dark.collapse");

    /* Retrieve input parameters for 'hot-pixel' bpm */

    bpm_params = hdrl_bpm_2d_parameter_parse_parlist(parlist, 
                 "eris.eris_nix_dark.hotpix");

    enu_check(cpl_error_get_code() == CPL_ERROR_NONE, cpl_error_get_code(),
              "Could not retrieve input parameters");

    /* Identify the RAW and CALIB frames in the input frameset */

    eris_nix_dfs_set_groups(frameset);
    enu_check(cpl_error_get_code() == CPL_ERROR_NONE, cpl_error_get_code(),
              "Could not identify RAW and CALIB frames");

    /* Check at least one DARK frame is present in SoF */

    rawframe = cpl_frameset_find_const(frameset, ERIS_NIX_RAW_DARK_DO_CATG);
    enu_check(rawframe != NULL, CPL_ERROR_DATA_NOT_FOUND, 
              "SOF does not have any file tagged with %s",
              ERIS_NIX_RAW_DARK_DO_CATG);

    /* Check GAIN file is present in SoF */

    const cpl_frame * gainframe = cpl_frameset_find_const(frameset,
                                  ERIS_NIX_GAIN_PRO_CATG);
    enu_check(gainframe != NULL, CPL_ERROR_DATA_NOT_FOUND, 
              "SOF does not have any file tagged with %s",
              ERIS_NIX_GAIN_PRO_CATG);

    /* Construct a frameset with just the DARK frames */

    dark_frameset = cpl_frameset_new();
    frame_iter = cpl_frameset_iterator_new(frameset);

    while (cpl_frameset_iterator_get(frame_iter) != NULL) {
        cpl_frame * frame = cpl_frameset_iterator_get(frame_iter); 
        const char * tag = cpl_frame_get_tag(frame);
        filename = cpl_frame_get_filename(frame);
        cpl_msg_debug(cpl_func, "filename %s tag %s", filename, tag);
        if (!strcmp(tag, ERIS_NIX_RAW_DARK_DO_CATG)) {
            cpl_frameset_insert(dark_frameset, cpl_frame_duplicate(frame));
        }
        cpl_frameset_iterator_advance(frame_iter, 1);
    }

    /* How many DARKS do we have */

    const int ndarks = cpl_frameset_get_size(dark_frameset);
    cpl_msg_info(cpl_func, "Number of DARK frames: %d", ndarks);

    /* Construct an imagelist containing the DARK images */
    dark_himlist = hdrl_imagelist_new();

    for (int idark=0; idark < ndarks; idark++) {

        /* Check that the DARK frames all match */

        const cpl_frame * frame = cpl_frameset_get_position_const(
                                  dark_frameset, idark);
        filename = cpl_frame_get_filename(frame);

        if (idark == 0) {
            plist0 = cpl_propertylist_load(filename, 0);
            inherit = cpl_frame_duplicate(frame);
        } else {
            cpl_propertylist * plist = cpl_propertylist_load(filename, 0);

            /* following should work for both ERIS and NACO-style data */
            enu_check_conformance(plist0, plist, 
                                  "^ESO DET SEQ1 DIT$|"
                                  "^ESO DET READ CURNAME$|"
                                  "^ESO DET DIT$|"
                                  "^ESO DET NCORRS NAME$");
            cpl_propertylist_delete(plist);
        }
        enu_check(cpl_error_get_code()==CPL_ERROR_NONE, cpl_error_get_code(),
                  "error checking frame conformance");
        cpl_image * image = cpl_image_load(filename, HDRL_TYPE_DATA,
                                                 0, 0);
        hdrl_image * himage = hdrl_image_create(image, NULL);
        hdrl_imagelist_set(dark_himlist, himage, idark);

        cpl_image_delete(image);
    }
    enu_check(cpl_error_get_code() == CPL_ERROR_NONE, cpl_error_get_code(),
              "Could not construct DARK imagelist");
    int isize = hdrl_imagelist_get_size(dark_himlist);
    cpl_msg_debug(cpl_func, "Size of hdrl imagelist: %d", isize);
    enu_check(isize > 1, CPL_ERROR_ILLEGAL_INPUT, "need at least 2 DARKs");

    const char * instrume = cpl_propertylist_get_string(plist0, "INSTRUME");
    if (!strcmp(instrume, "ERIS")) {
        cpl_msg_info(cpl_func, "removing read_offsets");
       
        for (cpl_size i = 0; i < hdrl_imagelist_get_size(dark_himlist); i++) {
            enu_remove_read_offsets(hdrl_imagelist_get(dark_himlist, i),
                                    plist0,
                                    NULL,
                                    CPL_FALSE);
        }
    }

    /* Read the detector gain */

    const char * gainfile = cpl_frame_get_filename(gainframe);
    gain_propertylist = cpl_propertylist_load(gainfile, 0);
    double gain = cpl_propertylist_get_double(gain_propertylist,
                                              "ESO QC GAIN");
    enu_check_error_code("Error reading QC.GAIN from %s", gainfile);
    cpl_msg_info(cpl_func, "detector gain %5.2f", gain);

    /* Now calculate the readout-noise using the standard CPL routine
       - though shouldn't the result be divided by sqrt(2) as it is
       - being derived from a difference image */

    const hdrl_image * dark_0 = hdrl_imagelist_get_const(dark_himlist, 0);     
    const hdrl_image * dark_1 = hdrl_imagelist_get_const(dark_himlist, 1);
    diff = hdrl_image_sub_image_create(dark_0, dark_1);
    double read_noise = -1.0;
    double read_noise_std = -1.0;
    cpl_flux_get_noise_window(hdrl_image_get_image_const(diff), NULL,
                              -1, -1, &read_noise, &read_noise_std); 
    read_noise /= sqrt(2.0);
    read_noise_std /= sqrt(2.0);
    enu_check_error_code("Failed to calculate read-out noise");

    /* Convert ron from ADU to electrons */

    double read_noise_adu = read_noise;
    double read_noise_std_adu = read_noise_std;
    read_noise *= gain;
    read_noise_std *= gain;
    cpl_msg_info(cpl_func, "read-out noise = %5.2e +- %5.2e", read_noise,
      read_noise_std);

    /* Estimate the error plane of each dark */

    const char * det_mode = enu_get_det_mode(plist0);
    enu_check_error_code("Failed to read required detector mode");

    /* Calculate the error, assuming no photons, read-noise contribution
       only */

    double variance = 0.0;
    if (!strcmp(det_mode, "Double_RdRstRd")) { 

        /* CDS. Correlated Double Sampling.
           In this mode the intensity is calculated from the difference
           between sets of measurements at the start and end of the
           integration. Variance from Vacca eqn 44. with I = 0 */

        const double dit = enu_get_dit(plist0);
        const double ndit = (double) cpl_propertylist_get_int(plist0,
                                                              "ESO DET NDIT");
        const int nr = 2; /* is this right? */

        variance = 2.0 * read_noise * read_noise / 
                   (gain * gain * nr * ndit * dit * dit);

        cpl_msg_info(cpl_func,
                     "variance=%6.2e read-noise=%6.2f gain=%6.2f ndit=%d "
                     "dit=%6.2f",
                     variance, read_noise, gain, (int) ndit, dit);

    } else if (!strcmp(det_mode, "SLOW_GR_UTR")) {

        /* Continuous sampling technique - line fitting - 'up the ramp'.
           Variance from Vacca eqn 55 with I = 0. */

        double dit = enu_get_dit(plist0);
        const double ndit = (double) cpl_propertylist_get_int(plist0,
                                                              "ESO DET NDIT");
        const int nr = cpl_propertylist_get_int(plist0, "ESO DET NDSAMPLES");

        variance = 12.0 * read_noise * read_noise * (nr - 1.0) / 
                   (gain * gain * nr * ndit * dit * dit * (nr + 1.0));
        cpl_msg_info(cpl_func,
                     "variance=%6.2e read-noise=%6.2f gain=%6.2f ndit=%d "
                     "dit=%6.2f",
                     variance, read_noise, gain, (int) ndit, dit);

    } else if (!strcmp(det_mode, "FAST_UNCORR")) {

        /* FAST_UNCORR is not covered in Vacca et al. We could come up
           with a similar formula but a real uncertainty is the 
           repeatability of the array state after reset at the start of the
           integration.

           Most robust to calculate the MAD directly from the data sequence */

        cpl_imagelist * mad_working = cpl_imagelist_new();

        for (cpl_size i=0; i < hdrl_imagelist_get_size(dark_himlist); i++) {
            cpl_imagelist_set(mad_working, 
                              cpl_image_duplicate(
                              hdrl_image_get_image(
                              hdrl_imagelist_get(dark_himlist, i))), i);
        }

        /* get median of dark frames */

        cpl_image * dark_med =
            cpl_imagelist_collapse_median_create(mad_working);
        cpl_msg_info(cpl_func, "%e", cpl_image_get_median(dark_med));

        /* fill mad_working with abs of diffs between frames and median */

        for (cpl_size i=0; i < hdrl_imagelist_get_size(dark_himlist); i++) {
            cpl_image * diff2 = cpl_image_subtract_create(dark_med,
                                       hdrl_image_get_image_const(
                                       hdrl_imagelist_get(dark_himlist, i)));
            cpl_image_abs(diff2);
            cpl_msg_info(cpl_func, "%d %e", (int)i,
                         cpl_image_get_median(diff2));
            cpl_imagelist_set(mad_working, diff2, i);
        }
        cpl_image_delete(dark_med);
        /* get the median of the diffs and get the median of that */

        cpl_image * mad = cpl_imagelist_collapse_median_create(mad_working);
        double median_mad = cpl_image_get_median(mad);
        cpl_image_delete(mad);
        cpl_msg_info(cpl_func, "median mad %e", median_mad);
        variance = pow(1.4826 * median_mad, 2); 

        cpl_msg_info(cpl_func, "variance=%6.2e", variance);

        cpl_imagelist_delete(mad_working);

    } else {
        cpl_error_set_message(cpl_func, CPL_ERROR_UNSUPPORTED_MODE,
          "unsupported detector mode: %s", det_mode); 
    }
    enu_check_error_code("failed to calculate master_dark error plane");

    for (int idark = 0; idark < isize; idark++) {
        cpl_image * error = hdrl_image_get_error(hdrl_imagelist_get(
                                                 dark_himlist, idark));
        cpl_size nx = cpl_image_get_size_x(error);
        cpl_size ny = cpl_image_get_size_y(error);
        cpl_image_fill_window(error, 1, 1, nx, ny, sqrt(variance)); 
    }

    /* Use hdrl to calculate the MASTER_DARK data */

    hdrl_imagelist_collapse(dark_himlist, collapse_params, &mdark,
                            &contribution);
    enu_check_error_code("Failed to collapse imagelist");

    /* Now compute the 'hot-pixel' BPM, the output will not include 
       pixels already set bad in the input image. */

    hot_bpm = hdrl_bpm_2d_compute(mdark, bpm_params);
    double kappa_high = hdrl_bpm_2d_parameter_get_kappa_high(bpm_params);
    cpl_msg_info(cpl_func, "kappa-high %6.2e", kappa_high);
    enu_check_error_code("failed to calculate hot-pixel BPM");

    /* Set 'hot' pixels in the mask of the dark */

    cpl_mask * old_mask = cpl_image_set_bpm(hdrl_image_get_image(mdark),
                                            hot_bpm);
    if (old_mask) {
        cpl_mask_or(old_mask, hot_bpm);
        cpl_image_reject_from_mask(hdrl_image_get_image(mdark),
                                   old_mask); 
        cpl_mask_delete(old_mask); 
    }

    /* Need to calculate CR rate. Could assume each pixel value that
       was sigma-clipped was a CR, but no direct way to get the number of
       pixels clipped from HDRL */
    /* Could reconstruct the clipping using the hi-lo limits that can
       be returned by HDRL. Or could apply the hot-pixel bpm and 
       use hdrl_lacosmic - concern there is that the dark images might
       possess valid sharp edges */
    /* Not very important, so leave for now */

    cpl_msg_warning(cpl_func, "TBD: calculate CR rate");

    /* construct the propertylist to be saved with the MASTER_DARK */

    applist = cpl_propertylist_new();

    /* Add the product category  */

    cpl_propertylist_append_string(applist, CPL_DFS_PRO_CATG,
                                   ERIS_NIX_MASTER_DARK_IMG_PRO_CATG);
    cpl_propertylist_update_string(applist, "PRODCATG", "ANCILLARY.IMAGE");

    /* Info on the mode used */

    cpl_propertylist_copy_property_regexp(applist, plist0, 
                                          "^ESO DET DIT$|"
                                          "^ESO DET SEQ1 DIT$|"
                                          "^ESO DET READ CURNAME$|"
                                          "^ESO DET NCORRS NAME$", 0);

    /* Add QC parameters  */

    cpl_propertylist_append_double(applist, "ESO QC READ NOISE", read_noise);
    cpl_propertylist_set_comment(applist,"ESO QC READ NOISE",
				"[e-] Read Out Noise");
    cpl_propertylist_append_double(applist, "ESO QC READ NOISE VAR", 
                                   pow(read_noise_std, 2.0));
    cpl_propertylist_set_comment(applist,"ESO QC READ NOISE VAR",
				"[e-] Read Out Noise Variance");

    cpl_propertylist_append_double(applist, "ESO QC READ NOISE ADU", read_noise_adu);
    cpl_propertylist_set_comment(applist,"ESO QC READ NOISE ADU",
				"[ADU] Read Out Noise");
    cpl_propertylist_append_double(applist, "ESO QC READ NOISE VAR ADU", 
                                   pow(read_noise_std_adu, 2.0));
    cpl_propertylist_set_comment(applist,"ESO QC READ NOISE VAR ADU",
				"[ADU] Read Out Noise Variance");

    
    hdrl_value dark_median = hdrl_image_get_median(mdark);
    cpl_propertylist_append_double(applist, "ESO QC DARK MED",
                                   (double) dark_median.data);
    cpl_propertylist_set_comment(applist,"ESO QC DARK MED",
				"[ADU] Median Master Dark");
    hdrl_value dark_mean = hdrl_image_get_mean(mdark);
    cpl_propertylist_append_double(applist, "ESO QC DARK MEAN",
                                   (double) dark_mean.data);
    cpl_propertylist_set_comment(applist,"ESO QC DARK MEAN",
				"[ADU] Mean Master Dark");
    double dark_rms = hdrl_image_get_stdev(mdark);
    cpl_propertylist_append_double(applist, "ESO QC DARK RMS", dark_rms);
    cpl_propertylist_set_comment(applist,"ESO QC DARK RMS",
				"[ADU] RMS Master Dark");
    cpl_size nhot = cpl_mask_count(hot_bpm);
    cpl_size sx = cpl_mask_get_size_x(hot_bpm);
    cpl_size sy = cpl_mask_get_size_y(hot_bpm);
    double fraction = (double) nhot / (sx * sy);
    cpl_propertylist_append_int(applist, "ESO QC NUMBER HOT PIXEL", (int)nhot);
    cpl_propertylist_set_comment(applist,"ESO QC NUMBER HOT PIXEL",
				"Number of hot pixels");
    cpl_propertylist_append_double(applist, "ESO QC HOT PIXEL FRACTION", fraction);
    cpl_propertylist_set_comment(applist,"ESO QC HOT PIXEL FRACTION",
				"Fraction of hot pixels to total");
    double particle_rate = -1.0;
    cpl_propertylist_append_double(applist, "ESO QC PARTICLE_RATE", 
                                   particle_rate);

    /* copied from gain info file */

    cpl_propertylist_append_double(applist, "ESO QC GAIN", gain);
    cpl_propertylist_set_comment(applist,"ESO QC GAIN",
				"[e-/ADU] Detector Gain");

    /* FPN computation */

    double qc_fpn_val = 0, qc_fpn_err = 0;
    cpl_size zone_def[4];
    cpl_size hsize, nsamp;

    zone_def[0] = cpl_parameter_get_int(cpl_parameterlist_find_const(parlist,
    				"eris.eris_nix_dark.qc_fpn_xmin"));

    zone_def[1] = cpl_parameter_get_int(cpl_parameterlist_find_const(parlist,
    				"eris.eris_nix_dark.qc_fpn_xmax"));
    zone_def[2] = cpl_parameter_get_int(cpl_parameterlist_find_const(parlist,
    				"eris.eris_nix_dark.qc_fpn_ymin"));
    zone_def[3] = cpl_parameter_get_int(cpl_parameterlist_find_const(parlist,
    				"eris.eris_nix_dark.qc_fpn_ymax"));
    hsize  = cpl_parameter_get_int(cpl_parameterlist_find_const(parlist,
    				"eris.eris_nix_dark.qc_fpn_hsize"));
    nsamp = cpl_parameter_get_int(cpl_parameterlist_find_const(parlist,
    				"eris.eris_nix_dark.qc_fpn_nsamp"));

    /* window information */

    cpl_size nx = 0;
    cpl_size ny = 0;
    int rot = 0;
    cpl_size strx = 0;
    cpl_size stry = 0;
    cpl_size nx_chip = 0;
    cpl_size ny_chip = 0;
    cpl_boolean windowed = 0;
    enu_get_window_info(&nx,
            &ny,
            &rot,
            &strx,
            &stry,
            &nx_chip,
            &ny_chip,
            &windowed,
            plist0);
    /*
    cpl_msg_info(cpl_func,
    		"nx=:%lld ny: %lld strx: %lld stry: %lld nx_chip: %lld ny_chip: %lld",
			nx, ny, strx, stry, nx_chip, ny_chip);
*/

    zone_def[1] = (zone_def[1] > nx) ? nx: zone_def[1];
    zone_def[3] = (zone_def[3] > ny) ? ny: zone_def[3];

    cpl_msg_info(cpl_func,"Zone: [%lld,%lld,%lld,%lld]",
    		zone_def[0],zone_def[2],zone_def[1],zone_def[3]);
    cpl_image* master_dark_ima = hdrl_image_get_image(mdark);
    if(CPL_ERROR_NONE != cpl_flux_get_noise_window(master_dark_ima, zone_def, hsize,
                                          nsamp, &qc_fpn_val, &qc_fpn_err)) {
    	/* rarely this may fail. If this occurs we set dummy values (to prevent seg
    	 * fault later) and go on */
    	cpl_msg_error(cpl_func,"FPN computation failed, set dummy values");
    	qc_fpn_val=9999;
    	qc_fpn_err=99;
    	cpl_error_reset();
    }

    cpl_propertylist_append_double(applist,"ESO QC DARKFPN", qc_fpn_val);
    cpl_propertylist_set_comment(applist,"ESO QC DARKFPN",
        		                    "Fixed Pattern Noise of combined frames");

    mefs = enu_mef_extension_list_new(1);
    mefs->mef[0] = enu_mef_new_mask(ERIS_NIX_HOT_BPM_EXTENSION, hot_bpm, NULL);

    /* Save the master dark to a DFS-compliant MEF file */

    enu_dfs_save_himage(frameset,
                        parlist,
                        frameset,
                        CPL_TRUE,
                        mdark,
                        NULL,
                        mefs,
                        "eris_nix_dark",
                        inherit,
                        applist, 
                        NULL,
                        PACKAGE "/" PACKAGE_VERSION,
                        "master_dark.fits");
    enu_check_error_code("Failed to save MASTER_DARK");

cleanup:
    cpl_frameset_delete(dark_frameset);
    cpl_propertylist_delete(plist0);
    hdrl_imagelist_delete(dark_himlist);
    if(bpm_params != NULL) hdrl_parameter_delete(bpm_params);
    if(contribution != NULL) cpl_image_delete(contribution);
    if(mdark != NULL) hdrl_image_delete(mdark);

    cpl_propertylist_delete(applist);
    hdrl_image_delete(diff);
    cpl_frameset_iterator_delete(frame_iter);
    cpl_propertylist_delete(gain_propertylist);
    cpl_frame_delete(inherit);
    enu_mef_extension_list_delete(mefs);

    return (int) cpl_error_get_code();
}
