/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author$
 * $Date$
 * $Revision$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

//#include <libgen.h>
//#include <math.h>
#include <string.h>

//#include "eris_nix_utils.h"
#include "eris_nix_dfs.h"
#include "eris_nix_scired.h"
#include <hdrl.h>
#include <cpl.h>

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

static const char eris_nix_img_hdrl_stack_description[] =
"This recipe uses HDRL functions to rebin and stack a set of calibrated \n"
"ERIS/NIX jitter frames.\n"
"\n"
"Input files:\n"
"\n"
"  DO CATG                         Explanation             Req.  #Frames\n"
"  -------                         -----------             ---   -------\n"
"  "ERIS_NIX_CAL_PHOT_OBJECT_JITTER_PRO_CATG
                        "          calibrated frames        Y      1-n\n"
"                                  to be stacked.\n"
"             or\n"
"  "ERIS_NIX_CAL_PHOT_STD_JITTER_PRO_CATG
                     "             calibrated standard      Y      1-n\n"
"                                  frames to be stacked.\n"
"\n"
"Output files:\n"
"\n"
"  DO CATG                         Explanation \n"
"  -------                         ----------- \n"
"  "ERIS_NIX_IMG_OBS_COMBINED_PRO_CATG
                  "                the stacked image of the target.\n"
"             or\n"
"  "ERIS_NIX_IMG_STD_COMBINED_PRO_CATG
                  "                the stacked image of the standard.\n"
"  "ERIS_NIX_IMG_CATALOGUE_PRO_CATG
                   "               catalogue of sources in the\n"
"                                  stacked image.\n"
"\n"
"  The output image will be in a FITS file named \n"
"  'stack.<first input filename>', with extensions:\n"
"  - DATA, with the stacked target or standard data.\n"
"  - ERR, with the error plane.\n"
"  - DQ, with the stacked image quality\n"
"  - CONFIDENCE, with the target confidence plane.\n"
"  A catalogue of sources in the image will be written to FITS file \n"
"  'cat.stack.<first input filename>'.\n"

"\n"
"Notes on the method.\n"
" The stacking uses hdrl_resample_compute to combine all jitter\n"
" images in a single cloud of points. When creating the resampled \n"
" output image all information is used in a single interpolation \n"
" step, i.e. no image-by-image interpolation is done.\n"
"\n";

#define RECIPE_NAME "eris_nix_img_hdrl_stack"
#define CONTEXT "eris."RECIPE_NAME
/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

cpl_recipe_define(eris_nix_img_hdrl_stack, ERIS_BINARY_VERSION,
                  "John Lightfoot",
                  PACKAGE_BUGREPORT,
                  "2021",
                  "Stack calibrated ERIS/NIX jitter frames using HDRL",
                  eris_nix_img_hdrl_stack_description);

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options    
  @param    self  the non-NULL parameterlist to fill
  @return   CPL_ERROR_NONE iff everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code eris_nix_img_hdrl_stack_fill_parameterlist(
  cpl_parameterlist * self) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    hdrl_parameter    * catalogue_defaults = NULL;
    cpl_parameterlist * catalogue_parlist = NULL;
    cpl_parameter     * p = NULL;

    /* The lower rejection threshold to be used during stacking */
/*
    p = cpl_parameter_new_range(CONTEXT".stk_lthr", CPL_TYPE_DOUBLE,
                                "Low rejection threshold", CONTEXT,
                                5.0, 1.0e-6, 1.0e10);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "stk-lthr");
    cpl_parameterlist_append(self, p);
*/
    /* The upper rejection threshold to be used during stacking */
/*
    p = cpl_parameter_new_range(CONTEXT".stk_hthr", CPL_TYPE_DOUBLE,
                                "Upper rejection threshold", CONTEXT,
                                5.0, 1.0e-6, 1.0e10);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "stk-hthr");
    cpl_parameterlist_append(self, p);
*/
    /* Interpolation method */

    eris_nix_hdrl_stack_param_set(CONTEXT, self);
    

    /* generate the general parameter list for the catalogue */

    catalogue_defaults = hdrl_catalogue_parameter_create(4, 2.5, CPL_TRUE,
                         10.0, CPL_TRUE, 64, 2.0, 3.0,
                         ERIS_NIX_SATURATION_DEFAULT, HDRL_CATALOGUE_ALL);
    catalogue_parlist = hdrl_catalogue_parameter_create_parlist(CONTEXT,
                        "catalogue", catalogue_defaults);

    /* subset the parameters to be used */

    for (p = cpl_parameterlist_get_first(catalogue_parlist);
         p != NULL;
         p = cpl_parameterlist_get_next(catalogue_parlist)) {

        const char * pname = cpl_parameter_get_name(p);
        if (strstr(pname, "minmax") == NULL) {
            cpl_parameter * duplicate = cpl_parameter_duplicate(p);
            cpl_parameterlist_append(self, duplicate);
        }
    }

    /* ..add ao-params which specifies how to set params depending on AO */

    p = cpl_parameter_new_enum(CONTEXT".catalogue.ao-params",
                               CPL_TYPE_STRING,
                                "Default catalogue.core-radius and "
                                "catalogue.mesh-size depending on "
                                "AOMODE, or not",
                                CONTEXT,
                                "auto", 2, "auto", "user");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "catalogue.ao-params");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    p = cpl_parameter_new_value(CONTEXT".debug-data",
                                CPL_TYPE_BOOL, "true to save interim results",
                                CONTEXT, CPL_FALSE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "debug-data");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);



    p = cpl_parameter_new_value(CONTEXT".edges-trim",
                                CPL_TYPE_STRING,
								"Trim image edges in the image stacking: "
								"[txmin,tymin,txmax,tymax]. t means trim "
								"the user can trim differently from the min and "
								"max image edge values along X and Y axes. "
								"Each number can be be between 2 and the image half size -1 pixel",
                                CONTEXT, "0,0,0,0");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "edges-trim");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);


    /* tidy up */

    hdrl_parameter_delete(catalogue_defaults);
    cpl_parameterlist_delete(catalogue_parlist);

    return 0;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int eris_nix_img_hdrl_stack(cpl_frameset * frameset,
                                   const cpl_parameterlist * parlist) {


	eris_nix_scired_hdrl_stack(frameset, parlist, RECIPE_NAME, CONTEXT);

    return (int)cpl_error_get_code();
}
