/* $Id$
 *
 * This file is part of the ERIS/NIX Pipeline  
 * Copyright (C) 2025 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "eris_utils.h"
#include "eris_pfits.h"
#include "eris_dfs.h"
#include "eris_nix_dfs.h"
#include "eris_nix_utils.h"
#include "eris_nix_img_supersky.h"

#include <cpl.h>
#include <hdrl.h>
#include <math.h>

/*-----------------------------------------------------------------------------
                                   Static variables
 -----------------------------------------------------------------------------*/
#define RECIPE_NAME "eris.eris_nix_img_supersky"

static const char eris_nix_img_supersky_description[] =
"Creates and applies super-sky flat for mid-infrared (M-IR) imaging data\n"
"to correct for highly variable sky emission in Short-Lp, Lp, and Mp filters.\n"
"\n"
"Input files:\n"
"  NIX_SCIENCE_REDUCED: Detector-corrected science exposures (REQUIRED, multiple)\n"
"  MASTER_BPM: Master bad pixel map (REQUIRED, single)\n"
"\n"
"Output files:\n"
"  MASTER_SKYFLAT: Final super-sky flat\n"
"  SKYSUB_OBJECT_JITTER: Sky-subtracted science frames (if save-skysub=TRUE)\n"
"\n"
"Algorithm:\n"
"1. Create first-pass super-sky by sigma-clipped combination of all frames\n"
"2. Subtract scaled super-sky from each frame\n"
"3. Detect and mask sources in corrected frames using local background\n"
"4. Create final super-sky with source masks applied\n"
"5. Subtract scaled final super-sky from original frames\n"
"\n"
"The recipe handles both point-source (default) and extended-source observations.\n"
"For extended sources, use OFF-SOURCE frames as input.\n";

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

cpl_recipe_define(eris_nix_img_supersky, ERIS_BINARY_VERSION, "ESO Pipeline Team",
                  PACKAGE_BUGREPORT, "2025",
                  "Create and apply super-sky flat for M-IR imaging",
                  eris_nix_img_supersky_description);

/*-----------------------------------------------------------------------------
                                   Functions
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe parameters
  @param    self  the parameter list to fill
  @return   CPL_ERROR_NONE if ok
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code eris_nix_img_supersky_fill_parameterlist(
    cpl_parameterlist *self)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_parameter* p;

    /* Combination method */
    p = cpl_parameter_new_enum("eris.eris_nix_img_supersky.combine_method",
                               CPL_TYPE_STRING,
                               "Method to combine frames",
                               "eris.eris_nix_img_supersky",
                               "sigmaclip", 2,
                               "median", "sigmaclip");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "combine-method");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    /* Sigma clipping threshold */
    p = cpl_parameter_new_value("eris.eris_nix_img_supersky.sigma_clip",
                                CPL_TYPE_DOUBLE,
                                "Sigma threshold for clipping",
                                "eris.eris_nix_img_supersky", 2.0); //2.5
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sigma-clip");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    /* Maximum iterations for sigma clipping */
    p = cpl_parameter_new_value("eris.eris_nix_img_supersky.max_iter",
                                CPL_TYPE_INT,
                                "Maximum iterations for sigma clipping",
                                "eris.eris_nix_img_supersky", 5);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "max-iter");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    /* Source detection threshold */
    p = cpl_parameter_new_value("eris.eris_nix_img_supersky.detect_threshold",
                                CPL_TYPE_DOUBLE,
                                "Detection threshold in sigma above background",
                                "eris.eris_nix_img_supersky", 3.0);//5.0
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "detect-threshold");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    /* Source FWHM */
    p = cpl_parameter_new_value("eris.eris_nix_img_supersky.source_fwhm",
                                CPL_TYPE_DOUBLE,
                                "Stellar FWHM in pixels",
                                "eris.eris_nix_img_supersky", 5.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "source-fwhm");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    /* Minimum source area */
    p = cpl_parameter_new_value("eris.eris_nix_img_supersky.min_area",
                                CPL_TYPE_INT,
                                "Minimum connected pixels for source detection",
                                "eris.eris_nix_img_supersky", 500); //20
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "min-area");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    /* Mask dilation */
    p = cpl_parameter_new_value("eris.eris_nix_img_supersky.mask_dilate",
                                CPL_TYPE_INT,
                                "Source mask dilation radius in pixels",
                                "eris.eris_nix_img_supersky", 10);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "mask-dilate");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    /* Save sky-subtracted frames */
    p = cpl_parameter_new_value("eris.eris_nix_img_supersky.save_skysub",
                                CPL_TYPE_BOOL,
                                "Save individual sky-subtracted frames",
                                "eris.eris_nix_img_supersky", TRUE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "save-skysub");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    /* Save sky-subtracted frames */
    p = cpl_parameter_new_value("eris.eris_nix_img_supersky.debug_data",
    		CPL_TYPE_BOOL,
			"Save individual sky-subtracted frames",
			"eris.eris_nix_img_supersky", TRUE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "debug-data");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);




    /* parameters to mask objects */



    /* Get default parameters in the recipe */
    /* Default values of the hdrl_parameter in the hdrl_catalogue */
    int                    obj_min_pixels  = 300; //4
    double                 obj_threshold   = 3.0; //2.5
    cpl_boolean            obj_deblending  = CPL_TRUE;
    double                 obj_core_radius = 30.;//5.
    cpl_boolean            bkg_estimate    = CPL_TRUE;
    int                    bkg_mesh_size   = 64;
    double                 bkg_smooth_fwhm = 2.;
    double                 det_eff_gain    = 3.;
    double                 det_saturation  = HDRL_SATURATION_INIT;
    hdrl_catalogue_options resulttype      = HDRL_CATALOGUE_BKG;



    /* Create a parameter list */
    hdrl_parameter *c_def = hdrl_catalogue_parameter_create(
    		obj_min_pixels, obj_threshold, obj_deblending, obj_core_radius,
			bkg_estimate, bkg_mesh_size, bkg_smooth_fwhm,
			det_eff_gain, det_saturation, resulttype);


    cpl_parameterlist *s_param = hdrl_catalogue_parameter_create_parlist(RECIPE_NAME, "catalogue", c_def);
    hdrl_parameter_delete(c_def) ;


    /* Duplicate in the input/output parameterlist the parameters in the catalogue */
    for (p = cpl_parameterlist_get_first(s_param); p != NULL;
    		p = cpl_parameterlist_get_next(s_param) ) {
    	cpl_parameterlist_append(self, cpl_parameter_duplicate(p));
    }
    cpl_parameterlist_delete(s_param);

    cpl_parameter *par;

    /* --hdrldemo_bpm_2d.post-filter-x */
    par = cpl_parameter_new_value(RECIPE_NAME".post-filter-x", CPL_TYPE_INT,
    		"X Size of the post filtering kernel.", RECIPE_NAME, 21);//3
    cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "pfx");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* --hdrldemo_bpm_2d.post-filter-y */
    par = cpl_parameter_new_value(RECIPE_NAME".post-filter-y", CPL_TYPE_INT,
    		"Y Size of the post filtering kernel.", RECIPE_NAME, 21);//3
    cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "pfy");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* --hdrldemo_bpm_2d.post-filter-mode */
    par = cpl_parameter_new_enum(RECIPE_NAME".post-filter-mode",
    		CPL_TYPE_STRING, "Post filtering mode.", RECIPE_NAME,
			"dilation", 2, "closing", "dilation");//closing
    cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "pfm");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);




    return cpl_errorstate_is_equal(prestate) ? CPL_ERROR_NONE
        : cpl_error_set_where(cpl_func);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Execute the recipe
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int eris_nix_img_supersky(cpl_frameset            * frameset,
                                  const cpl_parameterlist * parlist)
{
    const cpl_parameter *   param;
    const char          *   combine_method;
    double                  sigma_clip;
    int                     max_iter;
    double                  detect_threshold;
    double                  source_fwhm;
    int                     min_area;
    int                     mask_dilate;
    cpl_boolean             save_skysub;
    cpl_frameset        *   product_frames = NULL;
    cpl_error_code          error;

    /* Retrieve input parameters */
    param = cpl_parameterlist_find_const(parlist,
                                         "eris.eris_nix_img_supersky.combine_method");
    combine_method = cpl_parameter_get_string(param);

    param = cpl_parameterlist_find_const(parlist,
                                         "eris.eris_nix_img_supersky.sigma_clip");
    sigma_clip = cpl_parameter_get_double(param);

    param = cpl_parameterlist_find_const(parlist,
                                         "eris.eris_nix_img_supersky.max_iter");
    max_iter = cpl_parameter_get_int(param);


    param = cpl_parameterlist_find_const(parlist,
                                         "eris.eris_nix_img_supersky.source_fwhm");
    source_fwhm = cpl_parameter_get_double(param);

    param = cpl_parameterlist_find_const(parlist,
                                         "eris.eris_nix_img_supersky.catalogue.obj.min-pixels");
    min_area = cpl_parameter_get_int(param);

    param = cpl_parameterlist_find_const(parlist,
                                            "eris.eris_nix_img_supersky.catalogue.obj.threshold");
    detect_threshold = cpl_parameter_get_double(param);

    param = cpl_parameterlist_find_const(parlist,
                                         "eris.eris_nix_img_supersky.mask_dilate");
    mask_dilate = cpl_parameter_get_int(param);

    param = cpl_parameterlist_find_const(parlist,
                                         "eris.eris_nix_img_supersky.save_skysub");
    save_skysub = cpl_parameter_get_bool(param);

    cpl_msg_info(cpl_func, "Recipe parameters:");
    cpl_msg_info(cpl_func, "  combine_method    = %s", combine_method);
    cpl_msg_info(cpl_func, "  sigma_clip        = %.2f", sigma_clip);
    cpl_msg_info(cpl_func, "  max_iter          = %d", max_iter);
    cpl_msg_info(cpl_func, "  detect_threshold  = %.2f", detect_threshold);
    cpl_msg_info(cpl_func, "  source_fwhm       = %.2f", source_fwhm);
    cpl_msg_info(cpl_func, "  min_area          = %d", min_area);
    cpl_msg_info(cpl_func, "  mask_dilate       = %d", mask_dilate);
    cpl_msg_info(cpl_func, "  save_skysub       = %s", save_skysub ? "TRUE" : "FALSE");

    /* Identify the RAW and CALIB frames in the input frameset */
    cpl_ensure_code(eris_nix_dfs_set_groups(frameset) == CPL_ERROR_NONE,
                    cpl_error_get_code());

    /* Call the main processing function */
    error = eris_nix_img_supersky_run(frameset, parlist, RECIPE_NAME, &product_frames);

    if (error != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "Super-sky creation failed");
        if (product_frames) cpl_frameset_delete(product_frames);
        return (int)cpl_error_set_where(cpl_func);
    }

    cpl_msg_info(cpl_func, "Recipe completed successfully. Generated %lld products",
                 product_frames ? cpl_frameset_get_size(product_frames) : 0);

    if (product_frames) cpl_frameset_delete(product_frames);

    return (int)cpl_error_get_code();
}
