/*
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <complex.h>
#include <string.h>
#include <assert.h>
#include <stdlib.h>
#include <limits.h>
#include <sys/types.h>
#include <regex.h>
#include <ctype.h>
#include "espdr_sdp_spectrum.h"
#include "espdr_msg.h"
#include "espdr_dfs.h"
#ifdef ESPDR_USE_FITS_UPDATE_CHECKSUM
#  include <fitsio2.h>
#endif

/*-----------------------------------------------------------------------------
                                   Defines
 -----------------------------------------------------------------------------*/

/* Manually setup the following while this code remains outside CPL proper: */
#ifndef cpl_error_set_regex

cpl_error_code
cpl_error_set_regex_macro(const char *, cpl_error_code, int,
                          const regex_t *, const char *,
                          unsigned, const char *, ...) CPL_ATTR_PRINTF(7,8);

#define cpl_error_set_regex(code, regcode, preg, ...)           \
    cpl_error_set_regex_macro(cpl_func, code, regcode, preg,    \
                              __FILE__, __LINE__, __VA_ARGS__)

#endif /* cpl_error_set_regex */

#define ESPDR_PRO_TECH     "ESO PRO TECH"
#define ESPDR_NOESODAT  "NOESODAT"
#define ESPDR_OBS_ID    "ESO OBS ID"
#define ESPDR_OBS_PROG_ID  "ESO OBS PROG ID"
#define ESPDR_ARCFILE "ARCFILE"
#define ESPDR_ORIGFILE "ORIGFILE"
#define ESPDR_SDP_KEYWORD_PRODCATG            "PRODCATG"
#define ESPDR_SCIENCE_SPECTRUM                "SCIENCE.SPECTRUM"
#define ESPDR_OBS_TARG_NAME    "ESO OBS TARG NAME"
#define ESPDR_SDP_KEYWORD_OBJECT              "OBJECT"
#define ESPDR_SDP_KEYWORD_TITLE               "TITLE"
#define ESPDR_RA "RA"
#define ESPDR_DEC "DEC"
#define ESPDR_EXPTIME "EXPTIME"
#define ESPDR_INSTRUME "INSTRUME"
#define ESPDR_INSTRUME_VALUE                  "ESPRESSO"
#define ESPDR_MJDOBS "MJD-OBS"
#define ESPDR_MJDEND "MJD-END"
#define ESPDR_QC_FLUX_SN "ESO QC FLUX SN"
#define ESPDR_OBS_TARG_NAME    "ESO OBS TARG NAME"
#define ESPDR_CRPIX1 "CRPIX1"
//#define ESPDR_CRPIX2 "CRPIX2"
//#define ESPDR_CRPIX3 "CRPIX3"

#define nm2AA 10.
#define AA2nm 0.1
#define ESPDR_CDELT1 "CDELT1"
//#define ESPDR_CDELT2 "CDELT2"
//#define ESPDR_CDELT3 "CDELT3"

#define ESPDR_CRVAL1 "CRVAL1"
//#define ESPDR_CRVAL2 "CRVAL2"
//#define ESPDR_CRVAL3 "CRVAL3"

#define ESPDR_NAXIS1 "NAXIS1"
//#define ESPDR_NAXIS2 "NAXIS2"
//#define ESPDR_NAXIS3 "NAXIS3"


#define ESPDR_SDP_KEYWORD_LAMNLIN             "LAMNLIN"
#define ESPDR_SDP_KEYWORD_LAMRMS              "LAMRMS"
#define ESPDR_SDP_KEYWORD_SPEC_RES            "SPEC_RES"
#define ESPDR_SDP_KEYWORD_SPEC_ERR            "SPEC_ERR"
#define ESPDR_SDP_KEYWORD_SPEC_SYE            "SPEC_SYE"

#define ESPDR_CRDER1 "CRDER1"
#define ESPDR_CRDER1_C "Wavelength uncertainty [CUNIT1]"
#define ESPDR_CSYER1 "CSYER1"
#define ESPDR_CSYER1_C "Typical systematic wavelength error [CUNIT1]"
#define ESPDR_CONAD "ESO DET OUT1 CONAD"
#define ESPDR_RON  "ESO DET OUT1 RON"
#define ESPDR_SDP_KEYWORD_GAIN                "GAIN"
#define ESPDR_SDP_KEYWORD_DETRON              "DETRON"
#define ESPDR_SDP_KEYWORD_EFFRON              "EFFRON"
#define ESPDR_SDP_KEYWORD_TOT_FLUX            "TOT_FLUX"
#define ESPDR_SDP_KEYWORD_APERTURE            "APERTURE"
#define ESPDR_SDP_KEYWORD_EXTNAME_VALUE       "SPECTRUM"
#define ESPDR_SDP_KEYWORD_INHERIT_VALUE       CPL_TRUE

#define ESPDR_SDP_KEYWORD_TELAPSE             "TELAPSE"
#define ESPDR_SDP_KEYWORD_TMID                "TMID"

#define ESPDR_SDP_KEYWORD_PRODLVL             "PRODLVL"
#define ESPDR_SDP_KEYWORD_PRODLVL_VALUE       2

#define ESPDR_SDP_KEYWORD_ORIGIN              "ORIGIN"
#define ESPDR_SDP_KEYWORD_ORIGIN_VALUE        "ESO"

#define ESPDR_SDP_KEYWORD_SPECSYS_VALUE       "BARYCENT"

#define ESPDR_PRO_REC1_PIPE_ID "ESO PRO REC1 PIPE ID"

#define ESPDR_SDP_KEYWORD_SPECSYS             "SPECSYS"
#define ESPDR_SDP_KEYWORD_FLUXERR_VALUE       -2.0
#define ESPDR_SDP_KEYWORD_UNCALIBRATED_FLUXERR_VALUE       -1.0
#define ESPDR_SDP_KEYWORD_REFERENC_VALUE      " "
#define ESPDR_SDP_KEYWORD_VOCLASS_VALUE       "SPECTRUM V2.0"
#define ESPDR_SDP_KEYWORD_VOPUB_VALUE         "ESO/SAF"
//#define ESPDR_SDP_EL_UNIT                     "e-"
#define ESPDR_SDP_EL_UNIT                     ""
#define ESPDR_SDP_FLUX_UNIT                   "erg.cm**(-2).s**(-1).angstrom**(-1)"

#define ESPDR_BUNIT "BUNIT"
#define ESPDR_BUNIT_FLUX_ABS_C "erg/s/cm2/Angstrom"
#define ESPDR_BUNIT_FLUX_REL_C "ADU"
#define ESPDR_BUNIT_NONE_C ""


#define ESPDR_SDP_COLUMN_WAVE                 "WAVE"
#define ESPDR_SDP_COLUMN_WAVE_AIR             "WAVE_AIR"

#define ESPDR_SDP_COLUMN_WAVE_TYPE            "spec:Data.SpectralAxis.Value"
#define ESPDR_SDP_COLUMN_WAVE_AIR_TYPE        "eso:Data.SpectralAxis.Value"

#define ESPDR_SDP_COLUMN_WAVE_UCD             "em.wl;meta.main"
#define ESPDR_SDP_COLUMN_WAVE_AIR_UCD         "em.wl;obs.atmos"

#define ESPDR_SDP_COLUMN_WAVE_UNIT            "angstrom"

#define ESPDR_SDP_COLUMN_WAVE_FORMAT          "1E"

#define ESPDR_SDP_COLUMN_FLUX_FINAL           "FLUX"
#define ESPDR_SDP_COLUMN_FLUX_EL              "FLUX_EL"
#define ESPDR_SDP_COLUMN_FLUX_TELL_EL         "FLUX_TELL_EL"
#define ESPDR_SDP_COLUMN_FLUX_EL_SKYSUB       "FLUX_EL_SKYSUB"
#define ESPDR_SDP_COLUMN_FLUX_TELL_EL_SKYSUB  "FLUX_TELL_EL_SKYSUB"

#define ESPDR_SDP_COLUMN_FLUX_CAL             "FLUX_CAL"
#define ESPDR_SDP_COLUMN_FLUX_TELL_CAL        "FLUX_TELL_CAL"

#define ESPDR_SDP_COLUMN_FLUX_CAL_SKYSUB      "FLUX_CAL_SKYSUB"
#define ESPDR_SDP_COLUMN_FLUX_TELL_CAL_SKYSUB "FLUX_TELL_CAL_SKYSUB"

#define ESPDR_SDP_COLUMN_FLUX_TELL_EL         "FLUX_TELL_EL"
#define ESPDR_SDP_COLUMN_ERR_TELL_CAL         "ERR_TELL_CAL"
#define ESPDR_SDP_COLUMN_QUAL_TELL_CAL        "QUAL_TELL_CAL"
#define ESPDR_SDP_COLUMN_ATM_TRANSM           "ATM_TRANSM"

#define ESPDR_SDP_COLUMN_FLUX_EL_UNIT         ESPDR_SDP_EL_UNIT
#define ESPDR_SDP_COLUMN_FLUX_EL_SKYSUB_UNIT  ESPDR_SDP_EL_UNIT
#define ESPDR_SDP_COLUMN_FLUX_CAL_UNIT        ESPDR_SDP_FLUX_UNIT
#define ESPDR_SDP_COLUMN_FLUX_CAL_SKYSUB_UNIT ESPDR_SDP_FLUX_UNIT

#define ESPDR_SDP_COLUMN_FLUX_FORMAT          "1E"


#define ESPDR_SDP_COLUMN_FLUX_FINAL_TYPE      "spec:Data.FluxAxis.Value"
#define ESPDR_SDP_COLUMN_FLUX_EL_TYPE         "eso:Data.FluxAxis.Value"
#define ESPDR_SDP_COLUMN_FLUX_EL_SKYSUB_TYPE  "eso:Data.FluxAxis.Value"
#define ESPDR_SDP_COLUMN_FLUX_CAL_TYPE        "eso:Data.FluxAxis.Value"
#define ESPDR_SDP_COLUMN_FLUX_CAL_SKYSUB_TYPE "eso:Data.FluxAxis.Value"
#define ESPDR_SDP_COLUMN_FLUX_TELL_CAL_SKYSUB_TYPE "eso:Data.FluxAxis.Value"
#define ESPDR_SDP_COLUMN_FLUX_TELL_CAL_TYPE   "eso:Data.FluxAxis.Value"
#define ESPDR_SDP_COLUMN_FLUX_TELL_EL_TYPE       "eso:Data.FluxAxis.Value"

#define ESPDR_SDP_COLUMN_FLUX_SKYSUB_FINAL_UCD       "phot.flux.density;em.wl;src.net;meta.main"
#define ESPDR_SDP_COLUMN_FLUX_FINAL_UCD       "phot.flux.density;em.wl;meta.main"
#define ESPDR_SDP_COLUMN_FLUX_FINAL_SKYSUB_UCD  "phot.flux.density;em.wl;src.net;meta.main"


#define ESPDR_SDP_COLUMN_FLUX_EL_UCD          ""
#define ESPDR_SDP_COLUMN_FLUX_EL_SKYSUB_UCD   ""
#define ESPDR_SDP_COLUMN_ERR_EL_SKYSUB_UCD    ""
//#define ESPDR_SDP_COLUMN_FLUX_EL_SKYSUB_UCD      "meta.number;phot.flux.density;"
#define ESPDR_SDP_COLUMN_FLUX_EL_SKYSUB_UCD      ""
#define ESPDR_SDP_COLUMN_FLUX_CAL_UCD         "phot.flux.density;em.wl"
//#define ESPDR_SDP_COLUMN_FLUX_EL_SKYSUB_UCD   "meta.number;phot.flux.density;"
#define ESPDR_SDP_COLUMN_FLUX_CAL_UCD         "phot.flux.density;em.wl"
#define ESPDR_SDP_COLUMN_FLUX_CAL_SKYSUB_UCD  "phot.flux.density;em.wl;src.net"
#define ESPDR_SDP_COLUMN_FLUX_TELL_CAL_SKYSUB_UCD  "phot.flux.density;em.wl;src.net"
#define ESPDR_SDP_COLUMN_FLUX_TELL_CAL_UCD    "phot.flux.density;em.wl"
//#define ESPDR_SDP_COLUMN_FLUX_TELL_EL_UCD     "phot.flux.density;em.wl;stat.uncalib"
#define ESPDR_SDP_COLUMN_FLUX_TELL_EL_UCD     ""

#define ESPDR_SDP_COLUMN_ERR_FINAL            "ERR"
#define ESPDR_SDP_COLUMN_ERR_EL               "ERR_EL"
#define ESPDR_SDP_COLUMN_ERR_TELL_EL          "ERR_TELL_EL"
#define ESPDR_SDP_COLUMN_ERR_EL_SKYSUB        "ERR_EL_SKYSUB"
#define ESPDR_SDP_COLUMN_ERR_TELL_EL_SKYSUB   "ERR_TELL_EL_SKYSUB"
#define ESPDR_SDP_COLUMN_ERR_CAL              "ERR_CAL"
#define ESPDR_SDP_COLUMN_ERR_CAL_SKYSUB       "ERR_CAL_SKYSUB"
#define ESPDR_SDP_COLUMN_ERR_TELL_CAL_SKYSUB  "ERR_TELL_CAL_SKYSUB"


#define ESPDR_SDP_COLUMN_ERR_EL_UNIT          ESPDR_SDP_EL_UNIT
#define ESPDR_SDP_COLUMN_ERR_EL_SKYSUB_UNIT   ESPDR_SDP_EL_UNIT
#define ESPDR_SDP_COLUMN_ERR_CAL_UNIT         ESPDR_SDP_FLUX_UNIT
#define ESPDR_SDP_COLUMN_ERR_CAL_SKYSUB_UNIT  ESPDR_SDP_FLUX_UNIT
#define ESPDR_SDP_COLUMN_ERR_TELL_CAL_SKYSUB_UNIT  ESPDR_SDP_FLUX_UNIT

#define ESPDR_SDP_COLUMN_ERR_FORMAT           "1E"

#define ESPDR_SDP_COLUMN_ERR_FINAL_TYPE       "spec:Data.FluxAxis.Accuracy.StatError"
#define ESPDR_SDP_COLUMN_ERR_EL_TYPE          "eso:Data.FluxAxis.Accuracy.StatError"
#define ESPDR_SDP_COLUMN_ERR_EL_SKYSUB_TYPE      "eso:Data.FluxAxis.Accuracy.StatError"
#define ESPDR_SDP_COLUMN_ERR_CAL_TYPE         "eso:Data.FluxAxis.Accuracy.StatError"
#define ESPDR_SDP_COLUMN_ERR_CAL_SKYSUB_TYPE  "eso:Data.FluxAxis.Accuracy.StatError"
#define ESPDR_SDP_COLUMN_ERR_TELL_CAL_TYPE    "eso:Data.FluxAxis.Accuracy.StatError"
#define ESPDR_SDP_COLUMN_ERR_TELL_EL_TYPE       "eso:Data.FluxAxis.Accuracy.StatError"

#define ESPDR_SDP_COLUMN_ERR_FINAL_UCD        "stat.error;phot.flux.density;meta.main"
#define ESPDR_SDP_COLUMN_ERR_FINAL_SKYSUB_UCD        "stat.error;phot.flux.density;meta.main;src.net"
#define ESPDR_SDP_COLUMN_ERR_EL_UCD           ""
//#define ESPDR_SDP_COLUMN_ERR_EL_SKYSUB_UCD    "meta.number;phot.flux.density"
#define ESPDR_SDP_COLUMN_ERR_EL_SKYSUB_UCD       ""
#define ESPDR_SDP_COLUMN_ERR_CAL_UCD          "stat.error;phot.flux.density"
#define ESPDR_SDP_COLUMN_ERR_CAL_SKYSUB_UCD   "stat.error;phot.flux.density;src.net"
//#define ESPDR_SDP_COLUMN_ERR_TELL_EL_UCD      "stat.error;phot.flux.density;stat.uncalib"
#define ESPDR_SDP_COLUMN_ERR_TELL_EL_UCD      ""
#define ESPDR_SDP_COLUMN_ERR_TELL_CAL_UCD     "stat.error;phot.flux.density"
#define ESPDR_SDP_COLUMN_ERR_TELL_CAL_SKYSUB_UCD     "stat.error;phot.flux.density;src.net"

#define ESPDR_SDP_COLUMN_QUAL_FINAL           "QUAL"
#define ESPDR_SDP_COLUMN_QUAL_EL              "QUAL_EL"
#define ESPDR_SDP_COLUMN_QUAL_TELL_EL         "QUAL_TELL_EL"
#define ESPDR_SDP_COLUMN_QUAL_EL_SKYSUB       "QUAL_EL_SKYSUB"
#define ESPDR_SDP_COLUMN_QUAL_TELL_EL_SKYSUB  "QUAL_TELL_EL_SKYSUB"
#define ESPDR_SDP_COLUMN_QUAL_CAL             "QUAL_CAL"
#define ESPDR_SDP_COLUMN_QUAL_CAL_SKYSUB      "QUAL_CAL_SKYSUB"
#define ESPDR_SDP_COLUMN_QUAL_TELL_CAL_SKYSUB "QUAL_TELL_CAL_SKYSUB"

#define ESPDR_SDP_COLUMN_QUAL_UNIT            " "

#define ESPDR_SDP_COLUMN_QUAL_FORMAT          "1J"

#define ESPDR_SDP_COLUMN_QUAL_FINAL_TYPE      "spec:Data.FluxAxis.Accuracy.QualityStatus"
#define ESPDR_SDP_COLUMN_QUAL_EL_TYPE         "eso:Data.FluxAxis.Accuracy.QualityStatus"
#define ESPDR_SDP_COLUMN_QUAL_EL_SKYSUB_TYPE     "eso:Data.FluxAxis.Accuracy.QualityStatus"
#define ESPDR_SDP_COLUMN_QUAL_CAL_TYPE        "eso:Data.FluxAxis.Accuracy.QualityStatus"
#define ESPDR_SDP_COLUMN_QUAL_CAL_SKYSUB_TYPE "eso:Data.FluxAxis.Accuracy.QualityStatus"
#define ESPDR_SDP_COLUMN_QUAL_TELL_CAL_TYPE   "eso:Data.FluxAxis.Accuracy.QualityStatus"
#define ESPDR_SDP_COLUMN_QUAL_TELL_EL_TYPE   "eso:Data.FluxAxis.Accuracy.QualityStatus"
#define ESPDR_SDP_COLUMN_ATM_TRANSM_TYPE       ""

#define ESPDR_SDP_COLUMN_QUAL_FINAL_UCD       "meta.code.qual;meta.main"
#define ESPDR_SDP_COLUMN_QUAL_FINAL_SKYSUB_UCD "meta.code.qual;meta.main;src.net"
#define ESPDR_SDP_COLUMN_QUAL_EL_UCD          "meta.code.qual"
#define ESPDR_SDP_COLUMN_QUAL_EL_SKYSUB_UCD   "meta.code.qual"
#define ESPDR_SDP_COLUMN_QUAL_CAL_UCD         "meta.code.qual"
#define ESPDR_SDP_COLUMN_QUAL_CAL_SKYSUB_UCD  "meta.code.qual;src.net"
#define ESPDR_SDP_COLUMN_QUAL_TELL_CAL_UCD    "meta.code.qual"
#define ESPDR_SDP_COLUMN_QUAL_TELL_CAL_SKYSUB_UCD    "meta.code.qual;src.net"
#define ESPDR_SDP_COLUMN_QUAL_TELL_CAL_EL_UCD    "meta.code.qual"
#define ESPDR_SDP_COLUMN_QUAL_ATM_TRANSM_UCD    "phys.transmission"
#define ESPDR_SDP_COLUMN_QUAL_ATM_TRANSM_UNIT       ""

#define ESPDR_SDP_COLUMN_SNR                  "SNR"
#define ESPDR_SDP_COLUMN_SNR_UNIT             " "
#define ESPDR_SDP_COLUMN_SNR_FORMAT           "1E"
#define ESPDR_SDP_COLUMN_SNR_TYPE             "eso:Data.FluxAxis.Accuracy.SNR"
#define ESPDR_SDP_COLUMN_SNR_UCD              "stat.snr"

#define KEY_TUTYP               "TUTYP"
#define KEY_TUTYP_COMMENT       "IVOA data model element for field "
#define KEY_TUCD                "TUCD"
#define KEY_TCOMM               "TCOMM"
#define KEY_TCOMM_COMMENT       "Description for field "


#define KEY_ARCFILE             "ARCFILE"
#define KEY_ORIGFILE            "ORIGFILE"
#define KEY_RA                  "RA"
#define KEY_RA_COMMENT          "[deg] Spectroscopic target position (J2000)"
#define KEY_DEC                 "DEC"
#define KEY_DEC_COMMENT         "[deg] Spectroscopic target position (J2000)"
#define KEY_EXPTIME             "EXPTIME"
#define KEY_EXPTIME_COMMENT     "[s] Total integration time per pixel"
#define KEY_TEXPTIME            "TEXPTIME"
#define KEY_TEXPTIME_COMMENT    "[s] Total integration time of all exposures"
#define KEY_TIMESYS             "TIMESYS"
#define KEY_TIMESYS_COMMENT     "Time system used"
#define KEY_MJDOBS              "MJD-OBS"
#define KEY_MJDOBS_COMMENT      "[d] Start of observations (days)"
#define KEY_MJDEND              "MJD-END"
#define KEY_MJDEND_COMMENT      "[d] End of observations (days)"
#define KEY_PRODLVL             "PRODLVL"
#define KEY_PRODLVL_VALUE       2
#define KEY_PRODLVL_COMMENT     "Phase 3 product level: 1-raw, 2-science grade, 3-advanced"
#define KEY_PROCSOFT            "PROCSOFT"
#define KEY_PROCSOFT_COMMENT    "ESO pipeline version"
#define KEY_PRODCATG            "PRODCATG"
#define KEY_PRODCATG_COMMENT    "Data product category"
#define KEY_ORIGIN              "ORIGIN"
#define KEY_ORIGIN_VALUE        "ESO"
#define KEY_ORIGIN_COMMENT      "European Southern Observatory"
#define KEY_EXT_OBJ             "EXT_OBJ"
#define KEY_EXT_OBJ_COMMENT     "TRUE if extended"
#define KEY_DISPELEM            "DISPELEM"
#define KEY_DISPELEM_COMMENT    "Dispersive element name"
#define KEY_SPECSYS             "SPECSYS"
#define KEY_SPECSYS_VALUE       "BARYCENT"
#define KEY_SPECSYS_COMMENT     "Reference frame for spectral coordinates"
#define KEY_PROG_ID             "PROG_ID"
#define KEY_PROG_ID_COMMENT     "ESO programme identification"
#define KEY_OBID                "OBID"
#define KEY_OBID_COMMENT        "Observation block ID"
#define KEY_M_EPOCH             "M_EPOCH"
#define KEY_M_EPOCH_COMMENT     "TRUE if resulting from multiple epochs"
#define KEY_OBSTECH             "OBSTECH"
#define KEY_OBSTECH_COMMENT     "Technique for observation"
#define KEY_FLUXCAL             "FLUXCAL"
#define KEY_FLUXCAL_COMMENT     "Type of flux calibration (ABSOLUTE or UNCALIBRATED)"
#define KEY_CONTNORM            "CONTNORM"
#define KEY_CONTNORM_COMMENT    "TRUE if normalised to the continuum"
#define KEY_WAVELMIN            "WAVELMIN"
#define KEY_WAVELMIN_COMMENT    "[nm] Minimum wavelength"
#define KEY_WAVELMAX            "WAVELMAX"
#define KEY_WAVELMAX_COMMENT    "[nm] Maximum wavelength"
#define KEY_SPEC_BIN            "SPEC_BIN"
#define KEY_SPEC_BIN_COMMENT    "[nm] Wavelength bin size"
#define KEY_TOT_FLUX            "TOT_FLUX"
#define KEY_TOT_FLUX_COMMENT    "TRUE if photometric conditions and all source flux is captured"
#define KEY_FLUXERR             "FLUXERR"
#define KEY_FLUXERR_VALUE       -1.0
#define KEY_FLUXERR_COMMENT     "Uncertainty in flux scale (%)"
#define KEY_REFERENC            "REFERENC"
#define KEY_REFERENC_VALUE      " "
#define KEY_REFERENC_COMMENT    "Reference publication"
#define KEY_SPEC_RES            "SPEC_RES"
#define KEY_SPEC_RES_COMMENT    "Reference spectral resolving power"
#define KEY_SPEC_ERR            "SPEC_ERR"
#define KEY_SPEC_ERR_COMMENT    "[nm] Statistical error in spectral coordinate"
#define KEY_SPEC_SYE            "SPEC_SYE"
#define KEY_SPEC_SYE_COMMENT    "[nm] Systematic error in spectral coordinate"
#define KEY_LAMNLIN             "LAMNLIN"
#define KEY_LAMNLIN_COMMENT     "Number of arc lines used for the wavel. solution"
#define KEY_LAMRMS              "LAMRMS"
#define KEY_LAMRMS_COMMENT      "[nm] RMS of the residuals of the wavel. solution"
#define KEY_GAIN                "GAIN"
#define KEY_GAIN_COMMENT        "Conversion factor (e-/ADU) electrons per data unit"
#define KEY_DETRON              "DETRON"
#define KEY_DETRON_COMMENT      "Readout noise per output (e-)"
#define KEY_EFFRON              "EFFRON"
#define KEY_EFFRON_COMMENT      "Median effective readout noise (e-)"
#define KEY_SNR                 "SNR"
#define KEY_SNR_COMMENT         "Median signal to noise ratio per order"
#define KEY_SNR_AVG             "ESO QC SNR AVG"
#define KEY_SNR_AVG_COMMENT     "Mean signal to noise ratio values > 0"
#define KEY_SPEC_AVG            "ESO QC SPEC AVG"
#define KEY_SPEC_COMMENT        "Mean FLUX_EL_SKYSUB or FLUX_EL values > 0"
#define KEY_NCOMBINE            "NCOMBINE"
#define KEY_NCOMBINE_COMMENT    "No. of combined raw science data files"
#define KEY_PROV                "PROV"
#define KEY_PROV_COMMENT        "Originating raw science file"
#define KEY_ASSON               "ASSON"
#define KEY_ASSON_COMMENT       "Associated file name"
#define KEY_ASSOC               "ASSOC"
#define KEY_ASSOC_COMMENT       "Associated file category"
#define KEY_ASSOM               "ASSOM"
#define KEY_ASSOM_COMMENT       "Associated file md5sum"
#define KEY_VOCLASS             "VOCLASS"
#define KEY_VOCLASS_VALUE       "SPECTRUM V2.0"
#define KEY_VOCLASS_COMMENT     "VO Data Model"
#define KEY_VOPUB               "VOPUB"
#define KEY_VOPUB_VALUE         "ESO/SAF"
#define KEY_VOPUB_COMMENT       "VO Publishing Authority"
#define KEY_TITLE               "TITLE"
#define KEY_TITLE_COMMENT       "Dataset title"
#define KEY_OBJECT              "OBJECT"
#define KEY_OBJECT_COMMENT      "Target designation"
#define KEY_OBJECT_PHDU_COMMENT "Original target."
#define KEY_APERTURE            "APERTURE"
#define KEY_APERTURE_COMMENT    "[deg] Aperture diameter"
#define KEY_TELAPSE             "TELAPSE"
#define KEY_TELAPSE_COMMENT     "[s] Total elapsed time"
#define KEY_TMID                "TMID"
#define KEY_TMID_COMMENT        "[d] MJD mid exposure"
#define KEY_SPEC_VAL            "SPEC_VAL"
#define KEY_SPEC_VAL_COMMENT    "[nm] Mean wavelength"
#define KEY_SPEC_BW             "SPEC_BW"
#define KEY_SPEC_BW_COMMENT     "[nm] Bandpass width = Wmax - Wmin"
#define KEY_TDMIN(n)            "TDMIN"#n
#define KEY_TDMIN1_COMMENT      "Start in spectral coordinate"
#define KEY_TDMAX(n)            "TDMAX"#n
#define KEY_TDMAX1_COMMENT      "Stop in spectral coordinate"
#define KEY_TUTYP               "TUTYP"
#define KEY_TUTYP_COMMENT       "IVOA data model element for field "
#define KEY_TUCD                "TUCD"
#define KEY_TUCD_COMMENT        "UCD for field "
#define KEY_TUCD_EL_COMMENT     "UCD for field, electron "
#define KEY_TCOMM               "TCOMM"
#define KEY_TCOMM_COMMENT       "Description for field "
#define KEY_NELEM               "NELEM"
#define KEY_NELEM_COMMENT       "Length of the data arrays"
#define KEY_NOESODAT            "NOESODAT"
#define KEY_NOESODAT_COMMENT    "Data from no ESO instrument"
#define KEY_EXTNAME             "EXTNAME"
#define KEY_EXTNAME_VALUE       "SPECTRUM"
#define KEY_EXTNAME_COMMENT     "Extension name"
#define KEY_INHERIT             "INHERIT"
#define KEY_INHERIT_VALUE       CPL_TRUE
#define KEY_INHERIT_COMMENT     "Primary header keywords are inherited"

/* A regular expression to select all keywords relevant to a spectrum class. */
#define ALL_KEYS_REGEXP \
  "^(" KEY_RA "|" \
  KEY_DEC "|" \
  KEY_EXPTIME "|" \
  KEY_TEXPTIME "|" \
  KEY_TIMESYS "|" \
  KEY_MJDOBS "|" \
  KEY_MJDEND "|" \
  KEY_PRODLVL "|" \
  KEY_PROCSOFT "|" \
  KEY_PRODCATG "|" \
  KEY_ORIGIN "|" \
  KEY_DISPELEM "|" \
  KEY_SPECSYS "|" \
  KEY_PROG_ID "|" \
  KEY_OBID "[0-9]+|" \
  KEY_M_EPOCH "|" \
  KEY_OBSTECH "|" \
  KEY_FLUXCAL "|" \
  KEY_CONTNORM "|" \
  KEY_WAVELMIN "|" \
  KEY_WAVELMAX "|" \
  KEY_SPEC_BIN "|" \
  KEY_TOT_FLUX "|" \
  KEY_FLUXERR "|" \
  KEY_REFERENC "|" \
  KEY_SPEC_RES "|" \
  KEY_SPEC_ERR "|" \
  KEY_SPEC_SYE "|" \
  KEY_LAMNLIN "|" \
  KEY_LAMRMS "|" \
  KEY_GAIN "|" \
  KEY_DETRON "|" \
  KEY_EFFRON "|" \
  KEY_SNR "|" \
  KEY_NCOMBINE "|" \
  KEY_PROV "[0-9]+|" \
  KEY_ASSON "[0-9]+|" \
  KEY_ASSOC "[0-9]+|" \
  KEY_ASSOM "[0-9]+|" \
  KEY_VOCLASS "|" \
  KEY_VOPUB "|" \
  KEY_TITLE "|" \
  KEY_OBJECT "|" \
  KEY_APERTURE "|" \
  KEY_TELAPSE "|" \
  KEY_TMID "|" \
  KEY_SPEC_VAL "|" \
  KEY_SPEC_BW "|" \
  KEY_TDMIN(1) "|" \
  KEY_TDMAX(1) "|" \
  KEY_TUTYP "[0-9]+|" \
  KEY_TUCD "[0-9]+|" \
  KEY_TCOMM "[0-9]+|" \
  KEY_NELEM "|" \
  KEY_EXTNAME "|" \
  KEY_INHERIT ")$"

/* A regular expression to select keywords from all explicit SDP spectrum
 * keywords that should land up in the primary HDU. */
#define PRIMARY_HDU_KEYS_REGEXP \
  "^(" KEY_RA "|" \
  KEY_DEC "|" \
  KEY_EXPTIME "|" \
  KEY_TEXPTIME "|" \
  KEY_TIMESYS "|" \
  KEY_MJDOBS "|" \
  KEY_MJDEND "|" \
  KEY_PRODLVL "|" \
  KEY_PROCSOFT "|" \
  KEY_PRODCATG "|" \
  KEY_ORIGIN "|" \
  KEY_EXT_OBJ "|" \
  KEY_DISPELEM "|" \
  KEY_SPECSYS "|" \
  KEY_PROG_ID "|" \
  KEY_OBID "[0-9]+|" \
  KEY_M_EPOCH "|" \
  KEY_OBSTECH "|" \
  KEY_FLUXCAL "|" \
  KEY_CONTNORM "|" \
  KEY_WAVELMIN "|" \
  KEY_WAVELMAX "|" \
  KEY_SPEC_BIN "|" \
  KEY_TOT_FLUX "|" \
  KEY_FLUXERR "|" \
  KEY_REFERENC "|" \
  KEY_SPEC_RES "|" \
  KEY_SPEC_ERR "|" \
  KEY_SPEC_SYE "|" \
  KEY_LAMNLIN "|" \
  KEY_LAMRMS "|" \
  KEY_GAIN "|" \
  KEY_DETRON "|" \
  KEY_EFFRON "|" \
  KEY_SNR "|" \
  KEY_NCOMBINE "|" \
  KEY_NOESODAT "|" \
  KEY_EXT_OBJ "|" \
  KEY_PROV "[0-9]+|" \
  KEY_ASSON "[0-9]+|" \
  KEY_ASSOC "[0-9]+|" \
  KEY_ASSOM "[0-9]+|" \
  KEY_OBJECT ")$"

/* A regular expression to select keywords from all explicit SDP spectrum
 * keywords that should land up in the extension HDU. */
#define EXTENSION_HDU_KEYS_REGEXP \
  "^(" KEY_RA "|" \
  KEY_DEC "|" \
  KEY_VOCLASS "|" \
  KEY_VOPUB "|" \
  KEY_TITLE "|" \
  KEY_OBJECT "|" \
  KEY_APERTURE "|" \
  KEY_TELAPSE "|" \
  KEY_TMID "|" \
  KEY_SPEC_VAL "|" \
  KEY_SPEC_BW "|" \
  KEY_TDMIN(1) "|" \
  KEY_TDMAX(1) "|" \
  KEY_TUTYP "[0-9]+|" \
  KEY_TUCD "[0-9]+|" \
  KEY_TCOMM "[0-9]+|" \
  KEY_NELEM "|" \
  KEY_EXTNAME "|" \
  KEY_INHERIT ")$"


#define GET_SET_METHODS(param, keyname, type, rettype, defaultval, comment) \
  rettype espdr_sdp_spectrum_get_##param(const espdr_sdp_spectrum *self) \
  { \
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, defaultval); \
    assert(self->proplist != NULL); \
    if (cpl_propertylist_has(self->proplist, keyname)) { \
      return cpl_propertylist_get_##type(self->proplist, keyname); \
    } else { \
      return defaultval; \
    } \
  } \
  cpl_error_code espdr_sdp_spectrum_reset_##param(espdr_sdp_spectrum *self) \
  { \
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT); \
    assert(self->proplist != NULL); \
    (void) cpl_propertylist_erase(self->proplist, keyname); \
    return CPL_ERROR_NONE; \
  } \
  cpl_error_code espdr_sdp_spectrum_set_##param(espdr_sdp_spectrum *self, \
                                                 rettype value) \
  { \
    cpl_error_code error; \
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT); \
    assert(self->proplist != NULL); \
    if (cpl_propertylist_has(self->proplist, keyname)) { \
      error = cpl_propertylist_set_##type(self->proplist, keyname, value); \
    } else { \
      error = cpl_propertylist_append_##type(self->proplist, keyname, value); \
      if (! error) { \
        error = cpl_propertylist_set_comment(self->proplist, keyname, comment);\
        if (error) { \
          /* Delete entry if we could not set the comment to maintain a */ \
          /* consistent state. */ \
          cpl_errorstate prestate = cpl_errorstate_get(); \
          (void) cpl_propertylist_erase(self->proplist, keyname); \
          cpl_errorstate_set(prestate); \
        } \
      } \
    } \
    return error; \
  } \
  cpl_error_code espdr_sdp_spectrum_copy_##param(espdr_sdp_spectrum *self, \
                              const cpl_propertylist *plist, const char *name) \
  { \
    /* Note: check for plist or name equal NULL is done in the CPL calls. */ \
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT); \
    assert(self->proplist != NULL); \
    if (cpl_propertylist_has(plist, name)) { \
      cpl_errorstate prestate = cpl_errorstate_get(); \
      rettype value = cpl_propertylist_get_##type(plist, name); \
      if (cpl_errorstate_is_equal(prestate)) { \
        return espdr_sdp_spectrum_set_##param(self, value); \
      } else { \
        return cpl_error_set_message(cpl_func, cpl_error_get_code(), \
                  "Could not set '%s'. Likely the source '%s' keyword has a" \
                  " different format or type.", keyname, name); \
      } \
    } else { \
      return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND, \
                  "Could not set '%s' since the '%s' keyword was not found.", \
                  keyname, name); \
    } \
  }

#define GET_SET_ARRAY_METHODS(param, keyname, type, rettype, defaultval, \
                              comment) \
  rettype espdr_sdp_spectrum_get_##param(const espdr_sdp_spectrum *self, \
                                          cpl_size index) \
  { \
    char *name; \
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, defaultval); \
    assert(self->proplist != NULL); \
    name = cpl_sprintf("%s%"CPL_SIZE_FORMAT, keyname, index); \
    rettype result = defaultval; \
    if (cpl_propertylist_has(self->proplist, name)) { \
      result = cpl_propertylist_get_##type(self->proplist, name); \
    } \
    cpl_free(name); \
    return result; \
  } \
  cpl_error_code espdr_sdp_spectrum_reset_##param(espdr_sdp_spectrum *self, \
                                                   cpl_size index) \
  { \
    char *name; \
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT); \
    assert(self->proplist != NULL); \
    name = cpl_sprintf("%s%"CPL_SIZE_FORMAT, keyname, index); \
    (void) cpl_propertylist_erase(self->proplist, name); \
    cpl_free(name); \
    return CPL_ERROR_NONE; \
  } \
  cpl_error_code espdr_sdp_spectrum_set_##param(espdr_sdp_spectrum *self, \
                                                 cpl_size index, rettype value)\
  { \
    cpl_error_code error; \
    char *name; \
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT); \
    assert(self->proplist != NULL); \
    name = cpl_sprintf("%s%"CPL_SIZE_FORMAT, keyname, index); \
    if (cpl_propertylist_has(self->proplist, name)) { \
      error = cpl_propertylist_set_##type(self->proplist, name, value); \
    } else { \
      error = cpl_propertylist_append_##type(self->proplist, name, value); \
      if (! error) { \
        error = cpl_propertylist_set_comment(self->proplist, name, comment);\
        if (error) { \
          /* Delete entry if we could not set the comment to maintain a */ \
          /* consistent state. */ \
          cpl_errorstate prestate = cpl_errorstate_get(); \
          (void) cpl_propertylist_erase(self->proplist, name); \
          cpl_errorstate_set(prestate); \
        } \
      } \
    } \
    cpl_free(name); \
    return error; \
  } \
  cpl_error_code espdr_sdp_spectrum_copy_##param(\
                              espdr_sdp_spectrum *self, cpl_size index, \
                              const cpl_propertylist *plist, const char *name) \
  { \
    /* Note: check for plist or name equal NULL is done in the CPL calls. */ \
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT); \
    assert(self->proplist != NULL); \
    if (cpl_propertylist_has(plist, name)) { \
      cpl_errorstate prestate = cpl_errorstate_get(); \
      rettype value = cpl_propertylist_get_##type(plist, name); \
      if (cpl_errorstate_is_equal(prestate)) { \
        return espdr_sdp_spectrum_set_##param(self, index, value); \
      } else { \
        return cpl_error_set_message(cpl_func, cpl_error_get_code(), \
                  "Could not set '%s%"CPL_SIZE_FORMAT"'. Likely the source" \
                  " '%s' keyword has a different format or type.", \
                  keyname, index, name); \
      } \
    } else { \
      return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND, \
                  "Could not set '%s%"CPL_SIZE_FORMAT"' since the '%s'" \
                  " keyword was not found.", keyname, index, name); \
    } \
  }


#define GET_SET_METHODS_TYPE_BOOL(param, keyname, comment) \
  GET_SET_METHODS(param, keyname, bool, cpl_boolean, CPL_FALSE, comment)

#define GET_SET_METHODS_TYPE_DOUBLE(param, keyname, comment) \
  GET_SET_METHODS(param, keyname, double, double, NAN, comment)

#define GET_SET_METHODS_TYPE_INT(param, keyname, comment) \
  GET_SET_METHODS(param, keyname, int, int, -1, comment)

#define GET_SET_METHODS_TYPE_STRING(param, keyname, comment) \
  GET_SET_METHODS(param, keyname, string, const char *, NULL, comment)

#define GET_SET_ARRAY_METHODS_TYPE_INT(param, keyname, comment) \
  GET_SET_ARRAY_METHODS(param, keyname, int, int, -1, comment)

#define GET_SET_ARRAY_METHODS_TYPE_STRING(param, keyname, comment) \
  GET_SET_ARRAY_METHODS(param, keyname, string, const char *, NULL, comment)


#define ESPDR_TYPE_NELEM  CPL_TYPE_LONG_LONG | CPL_TYPE_UNSPECIFIED

/*----------------------------------------------------------------------------*/
/**
 * @defgroup espdr_sdp_spectrum    SDP 1D spectrum
 *
 * This module implements a Science Data Product (SDP) 1D spectrum object.
 * Various functions are provided to manipulate this kind of spectrum object.
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                                   Type definition
 -----------------------------------------------------------------------------*/
struct espdr_sdp_spectrum {
  /* Indicates the number of data points of the spectrum. */
  cpl_size nelem;

  /* Stores all the SDP keywords for the primary header and table extension. */
  cpl_propertylist *proplist;

  /* The table for the spectrum data points. */
  cpl_table *table;
};

/**
 * @internal
 * @brief Definition of SDP 1D spectrum structure.
 */
struct _espdr_sdp_spectrum_ {
  /* Indicates the number of data points of the spectrum. */
  cpl_size nelem;

  /* Stores all the SDP keywords for the primary header and table extension. */
  cpl_propertylist *proplist;

  /* The table for the spectrum data points. */
  cpl_table *table;
};


/**
 * @internal
 * @brief A record structure containing information about a keyword.
 */
typedef struct _espdr_keyword_record_ {
  /* The name of the keyword. */
  const char *name;

  /* The keyword's default comment. */
  const char *comment;

  /* The keyword's type code. */
  cpl_type type;

  /* Is the keyword an array keyword or not (e.g. PROVi). */
  cpl_boolean is_array_key;

} espdr_keyword_record;


/*
 * This is a row record structure used in the lookup tables implemented in the
 * espdr_sdp_spectrum_create function.
 */
struct _espdr_column_info {
    const char* name;
    cpl_type    type;
    const char* unit;
    const char* format;
    const char* tutyp;
    const char* tucd;
};

/*-----------------------------------------------------------------------------
                                   Internal function prototypes
 -----------------------------------------------------------------------------*/

static cpl_boolean
_espdr_property_equal(const cpl_property *a, const cpl_property *b);

static cpl_boolean
_espdr_array_equal(const cpl_array *a, const cpl_array *b, cpl_size n);

static cpl_boolean
_espdr_table_column_equal(const cpl_table *a, const cpl_table *b,
                           const char *name, cpl_boolean only_intersect);

static cpl_error_code
_espdr_sdp_spectrum_copy_column(espdr_sdp_spectrum *self, const char *to_name,
                                 const cpl_table* table, const char *from_name);

static cpl_error_code
_espdr_sdp_spectrum_divide_columns(espdr_sdp_spectrum *self, const char *dividend_name,
                                   const char *divisor_name);

static cpl_error_code
_espdr_sdp_spectrum_duplicate_column(espdr_sdp_spectrum *self, const char *to,
                                   const char *from);

static cpl_size
_espdr_sdp_spectrum_count_keywords(const espdr_sdp_spectrum *self,
                                    const char *regexp);

static cpl_size
_espdr_sdp_spectrum_get_column_index(const espdr_sdp_spectrum *self,
                                      const char *name);

static const char *
_espdr_sdp_spectrum_get_column_keyword(const espdr_sdp_spectrum *self,
                                        const char *name, const char *keyword);

static cpl_error_code
_espdr_sdp_spectrum_set_column_keyword(espdr_sdp_spectrum *self,
                                        const char *name,
                                        const char *value,
                                        const char *keyword,
                                        const char *comment);

static void
_espdr_sdp_spectrum_erase_column_keywords(espdr_sdp_spectrum *self,
                                           const char *name);

static char * _espdr_make_regexp(const cpl_propertylist *plist,
                                  const char *extra);

#ifndef NDEBUG
static cpl_boolean _espdr_keyword_table_is_sorted(
                            const espdr_keyword_record *table, size_t entries);
#endif

static const espdr_keyword_record *
_espdr_sdp_spectrum_get_keyword_record(const char *name);

/*-----------------------------------------------------------------------------
                                   Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief Creates a new spectrum object.
 */
/*----------------------------------------------------------------------------*/
espdr_sdp_spectrum * espdr_sdp_spectrum_new(void)
{
  espdr_sdp_spectrum * obj = cpl_malloc(sizeof(espdr_sdp_spectrum));
  obj->nelem = 0;
  obj->proplist = cpl_propertylist_new();
  obj->table = cpl_table_new(1);
  espdr_check_error_code(cpl_func);
  return obj;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Duplicates an existing spectrum object.
 */
/*----------------------------------------------------------------------------*/
espdr_sdp_spectrum *
espdr_sdp_spectrum_duplicate(const espdr_sdp_spectrum *other)
{
  espdr_sdp_spectrum * obj;

  cpl_ensure(other != NULL, CPL_ERROR_NULL_INPUT, NULL);

  assert(other->proplist != NULL);
  assert(other->table != NULL);

  obj = cpl_malloc(sizeof(espdr_sdp_spectrum));
  obj->nelem = other->nelem;
  obj->proplist = cpl_propertylist_duplicate(other->proplist);
  obj->table = cpl_table_duplicate(other->table);
  espdr_check_error_code(cpl_func);
  return obj;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Destroys an existing spectrum object.
 */
/*----------------------------------------------------------------------------*/
void espdr_sdp_spectrum_delete(espdr_sdp_spectrum *self)
{
  if (self != NULL) {
    assert(self->proplist != NULL);
    assert(self->table != NULL);
    cpl_propertylist_delete(self->proplist);
    cpl_table_delete(self->table);
    cpl_free(self);
    espdr_check_error_code(cpl_func);
  }
}

/*----------------------------------------------------------------------------*/
/**
 * @internal
 * @brief Check if the types and values of two properties match.
 */
/*----------------------------------------------------------------------------*/
static cpl_boolean
_espdr_property_equal(const cpl_property *a, const cpl_property *b)
{
  int value_not_equal;
  cpl_type type;
  const char *sa, *sb;

  assert(a != NULL);
  assert(b != NULL);

  /* Check the types are the same. */
  type = cpl_property_get_type(a);
  if (cpl_property_get_type(b) != type) return CPL_FALSE;

  /* Check that the values are the same. */
  switch (type) {
  case CPL_TYPE_CHAR:
    value_not_equal = cpl_property_get_char(a) != cpl_property_get_char(b);
    break;
  case CPL_TYPE_BOOL:
    value_not_equal = cpl_property_get_bool(a) != cpl_property_get_bool(b);
    break;
  case CPL_TYPE_INT:
    value_not_equal = cpl_property_get_int(a) != cpl_property_get_int(b);
    break;
  case CPL_TYPE_LONG:
    value_not_equal = cpl_property_get_long(a) != cpl_property_get_long(b);
    break;
  case CPL_TYPE_LONG_LONG:
    value_not_equal =
      cpl_property_get_long_long(a) != cpl_property_get_long_long(b);
    break;
  case CPL_TYPE_FLOAT:
    value_not_equal = cpl_property_get_float(a) != cpl_property_get_float(b);
    break;
  case CPL_TYPE_DOUBLE:
    value_not_equal = cpl_property_get_double(a) != cpl_property_get_double(b);
    break;
  case CPL_TYPE_STRING:
    sa = cpl_property_get_string(a);
    sb = cpl_property_get_string(b);
    if (sa == NULL && sb == NULL) {
      value_not_equal = 0;
    } else if (sa != NULL && sb != NULL) {
      value_not_equal = strcmp(sa, sb) != 0;
    } else {
      return CPL_FALSE;
    }
    break;
#ifdef _Complex_I
  case CPL_TYPE_FLOAT_COMPLEX:
    value_not_equal =
      cpl_property_get_float_complex(a) != cpl_property_get_float_complex(b);
    break;
  case CPL_TYPE_DOUBLE_COMPLEX:
    value_not_equal =
      cpl_property_get_double_complex(a) != cpl_property_get_double_complex(b);
    break;
#endif
  default:
    cpl_error_set_message(cpl_func, CPL_ERROR_INVALID_TYPE,
                          "Unsupported data type found in property '%s'.",
                          cpl_property_get_name(a));
    espdr_check_error_code(cpl_func);
    return CPL_FALSE;
  }
  if (value_not_equal) {
	  espdr_check_error_code(cpl_func);
	  return CPL_FALSE;
  }

  /* If we got here then the type and value must be equal so return true. */
  espdr_check_error_code(cpl_func);
  return CPL_TRUE;
}


/*----------------------------------------------------------------------------*/
/**
 * @internal
 * @brief Check if two arrays have the same elements.
 * @param a The first array to check.
 * @param b The second array to check.
 * @param n The number of elements to check, from [0..n-1].
 * @return CPL_TRUE if the first n elements are identical and CPL_FALSE
 *      otherwise. If an error occurs then an error code is set. This can be
 *      checked with @c cpl_error_get_code.
 */
/*----------------------------------------------------------------------------*/
static cpl_boolean
_espdr_array_equal(const cpl_array *a, const cpl_array *b, cpl_size n)
{
  cpl_type type;

  assert(a != NULL);
  assert(b != NULL);
  assert(n <= cpl_array_get_size(a));
  assert(n <= cpl_array_get_size(b));

  type = cpl_array_get_type(a);
  if (type != cpl_array_get_type(b)) return CPL_FALSE;

  if (type == CPL_TYPE_STRING) {
    /* Handle strings: */
    cpl_size i;
    const char **stra = cpl_array_get_data_string_const(a);
    const char **strb = cpl_array_get_data_string_const(b);
    cpl_error_ensure(stra != NULL && strb != NULL, cpl_error_get_code(),
                     return CPL_FALSE, "Failed to get %s data for array.",
                     cpl_type_get_name(type));
    for (i = 0; i < n; ++i) {
      if (stra[i] == NULL && strb[i] == NULL) continue;
      if (stra[i] == NULL || strb[i] == NULL) return CPL_FALSE;
      if (strcmp(stra[i], strb[i]) != 0) return CPL_FALSE;
    }

  } else {
    /* Handle fundamental types: */
    cpl_size size, i;
    const void *va, *vb;

    switch (type) {
    case CPL_TYPE_INT:
      size = sizeof(int);
      va = cpl_array_get_data_int_const(a);
      vb = cpl_array_get_data_int_const(b);
      break;
    case CPL_TYPE_LONG_LONG:
      size = sizeof(long long);
      va = cpl_array_get_data_long_long_const(a);
      vb = cpl_array_get_data_long_long_const(b);
      break;
    case CPL_TYPE_FLOAT:
      size = sizeof(float);
      va = cpl_array_get_data_float_const(a);
      vb = cpl_array_get_data_float_const(b);
      break;
    case CPL_TYPE_DOUBLE:
      size = sizeof(double);
      va = cpl_array_get_data_double_const(a);
      vb = cpl_array_get_data_double_const(b);
      break;
#ifdef _Complex_I
    case CPL_TYPE_FLOAT_COMPLEX:
      size = sizeof(_Complex float);
      va = cpl_array_get_data_float_complex_const(a);
      vb = cpl_array_get_data_float_complex_const(b);
      break;
    case CPL_TYPE_DOUBLE_COMPLEX:
      size = sizeof(_Complex double);
      va = cpl_array_get_data_double_complex_const(a);
      vb = cpl_array_get_data_double_complex_const(b);
      break;
#endif
    default:
      cpl_error_set_message(cpl_func, CPL_ERROR_INVALID_TYPE,
                            "Unsupported data type.");
      return CPL_FALSE;
    }
    cpl_error_ensure(va != NULL && vb != NULL, cpl_error_get_code(),
                     return CPL_FALSE, "Failed to get %s data for array.",
                     cpl_type_get_name(type));
    for (i = 0; i < n; ++i) {
      int valid_a = cpl_array_is_valid(a, i);
      int valid_b = cpl_array_is_valid(b, i);
      if (! valid_a && ! valid_b) continue;
      if (! valid_a || ! valid_b) return CPL_FALSE;
      const void *vai = (const char *)va + (size * i);
      const void *vbi = (const char *)vb + (size * i);
      if (memcmp(vai, vbi, size) != 0) return CPL_FALSE;
    }
  }

  /* If we get here then the first n elements of the arrays are equal. */
  espdr_check_error_code(cpl_func);
  return TRUE;
}

/*----------------------------------------------------------------------------*/
/**
 * @internal
 * @brief Check if the named table column is identical in both tables.
 * @note If @c only_intersect is CPL_TRUE then only the overlapping part of data
 *      arrays is checked. In other words, assume n1 is the column depth (i.e.
 *      length of an array table item) for table @c a and n2 is the depth for
 *      the item in table @c b at the same location. Then only the first N data
 *      elements are checked for equality in the arrays, where N = min(n1, n2).
 * @note The column format strings are ignored since these only affect printing
 *      when using @c printf. The format information is anyway lost between
 *      saves and loads of table information.
 */
/*----------------------------------------------------------------------------*/
static cpl_boolean
_espdr_table_column_equal(const cpl_table *a, const cpl_table *b,
                           const char *name, cpl_boolean only_intersect)
{
  cpl_type type;
  cpl_size nrows, na, nb, i;
  const char *sa, *sb;

  assert(a != NULL);
  assert(b != NULL);

  nrows = cpl_table_get_nrow(a);
  if (only_intersect) {
    cpl_size nrows2 = cpl_table_get_nrow(b);
    if (nrows2 < nrows) nrows = nrows2;
  } else {
    if (cpl_table_get_nrow(b) != nrows) return CPL_FALSE;
  }

  /* Column types must be the same. */
  type = cpl_table_get_column_type(a, name);
  if (cpl_table_get_column_type(b, name) != type) return CPL_FALSE;

  /* Column dimensions must be the same. */
  na = cpl_table_get_column_dimensions(a, name);
  nb = cpl_table_get_column_dimensions(b, name);
  if (na != nb) return CPL_FALSE;

  /* Check that the column unit is the same. */
  sa = cpl_table_get_column_unit(a, name);
  sb = cpl_table_get_column_unit(b, name);
  cpl_error_ensure(sa != NULL && sb != NULL, cpl_error_get_code(),
                   return CPL_FALSE,
                   "Failed to get unit strings for column '%s'.", name);
  if (strcmp(sa, sb) != 0) return CPL_FALSE;

  /* Check that the values are the same. For arrays we check that the parts of
   * the arrays that overlap are at least the same. */
  if (type & CPL_TYPE_POINTER) {
    cpl_errorstate prestate;
    /* Handle array cells: */
    const cpl_array **va = cpl_table_get_data_array_const(a, name);
    const cpl_array **vb = cpl_table_get_data_array_const(b, name);
    cpl_error_ensure(va != NULL && vb != NULL,
                     cpl_error_get_code(), return CPL_FALSE,
                     "Failed to get %s data for column '%s'.",
                     cpl_type_get_name(type), name);
    if (only_intersect) {
      for (i = 0; i < nrows; ++i) {
        /* If both arrays are NULL then they are equal,
         * but not if only one is NULL. */
        if (va[i] == NULL && vb[i] == NULL) continue;
        if (va[i] == NULL || vb[i] == NULL) return CPL_FALSE;
        prestate = cpl_errorstate_get();
        cpl_size n1 = cpl_array_get_size(va[i]);
        cpl_size n2 = cpl_array_get_size(vb[i]);
        cpl_size n = n1 < n2 ? n1 : n2;
        if (! _espdr_array_equal(va[i], vb[i], n)) return CPL_FALSE;
        cpl_error_ensure(cpl_errorstate_is_equal(prestate),
                         cpl_error_get_code(), return CPL_FALSE,
                         "Failed when trying to match %s data for column '%s'.",
                         cpl_type_get_name(type), name);
      }
    } else {
      for (i = 0; i < nrows; ++i) {
        /* If both arrays are NULL then they are equal,
         * but not if only one is NULL. */
        if (va[i] == NULL && vb[i] == NULL) continue;
        if (va[i] == NULL || vb[i] == NULL) return CPL_FALSE;
        prestate = cpl_errorstate_get();
        cpl_size n = cpl_array_get_size(va[i]);
        if (n != cpl_array_get_size(vb[i])) return CPL_FALSE;
        if (! _espdr_array_equal(va[i], vb[i], n)) return CPL_FALSE;
        cpl_error_ensure(cpl_errorstate_is_equal(prestate),
                         cpl_error_get_code(), return CPL_FALSE,
                         "Failed when trying to match %s data for column '%s'.",
                         cpl_type_get_name(type), name);
      }
    }

  } else if (type == CPL_TYPE_STRING) {
    /* Handle strings: */
    const char **va = cpl_table_get_data_string_const(a, name);
    const char **vb = cpl_table_get_data_string_const(b, name);
    cpl_error_ensure(va != NULL && vb != NULL,
                     cpl_error_get_code(), return CPL_FALSE,
                     "Failed to get %s data for column '%s'.",
                     cpl_type_get_name(type), name);
    if (only_intersect) {
      for (i = 0; i < nrows; ++i) {
        if (va[i] == NULL && vb[i] == NULL) continue;
        if (va[i] == NULL || vb[i] == NULL) return CPL_FALSE;
        size_t n1 = strlen(va[i]);
        size_t n2 = strlen(vb[i]);
        size_t n = n1 < n2 ? n1 : n2;
        if (strncmp(va[i], vb[i], (cpl_size)n) != 0) return CPL_FALSE;
      }
    } else {
      for (i = 0; i < nrows; ++i) {
        if (va[i] == NULL && vb[i] == NULL) continue;
        if (va[i] == NULL || vb[i] == NULL) return CPL_FALSE;
        if (strcmp(va[i], vb[i]) != 0) return CPL_FALSE;
      }
    }

  } else {
    /* Handle fundamental types and strings: */
    cpl_size size;
    const void *va, *vb;

    switch (type) {
    case CPL_TYPE_INT:
      size = sizeof(int);
      va = cpl_table_get_data_int_const(a, name);
      vb = cpl_table_get_data_int_const(b, name);
      break;
    case CPL_TYPE_LONG_LONG:
      size = sizeof(long long);
      va = cpl_table_get_data_long_long_const(a, name);
      vb = cpl_table_get_data_long_long_const(b, name);
      break;
    case CPL_TYPE_FLOAT:
      size = sizeof(float);
      va = cpl_table_get_data_float_const(a, name);
      vb = cpl_table_get_data_float_const(b, name);
      break;
    case CPL_TYPE_DOUBLE:
      size = sizeof(double);
      va = cpl_table_get_data_double_const(a, name);
      vb = cpl_table_get_data_double_const(b, name);
      break;
#ifdef _Complex_I
    case CPL_TYPE_FLOAT_COMPLEX:
      size = sizeof(_Complex float);
      va = cpl_table_get_data_float_complex_const(a, name);
      vb = cpl_table_get_data_float_complex_const(b, name);
      break;
    case CPL_TYPE_DOUBLE_COMPLEX:
      size = sizeof(_Complex double);
      va = cpl_table_get_data_double_complex_const(a, name);
      vb = cpl_table_get_data_double_complex_const(b, name);
      break;
#endif
    default:
      cpl_error_set_message(cpl_func, CPL_ERROR_INVALID_TYPE,
                          "Unsupported data type found in column '%s'.", name);
      return CPL_FALSE;
    }
    cpl_error_ensure(va != NULL && vb != NULL,
                     cpl_error_get_code(), return CPL_FALSE,
                     "Failed to get %s data for column '%s'.",
                     cpl_type_get_name(type), name);
    for (i = 0; i < nrows; ++i) {
      int valid_a = cpl_table_is_valid(a, name, i);
      int valid_b = cpl_table_is_valid(b, name, i);
      if (! valid_a && ! valid_b) continue;
      if (! valid_a || ! valid_b) return CPL_FALSE;
      const void *vai = (const char *)va + (size * i);
      const void *vbi = (const char *)vb + (size * i);
      if (memcmp(vai, vbi, size) != 0) return CPL_FALSE;
    }
  }

  /* If we got here then the columns must be equal so return true. */
  espdr_check_error_code(cpl_func);
  return CPL_TRUE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Compares two spectra if they are equal.
 * @param a The first spectrum to compare.
 * @param a The second spectrum to compare.
 * @param only_intersect Boolean flag indicating if only the intersection
 *      between two spectra should be checked.
 *
 * If @c only_intersect is set to CPL_FALSE then @c a and @c b are checked for
 * an exact match. i.e. number of keywords or table columns must be the same and
 * all values identical.
 * If set to CPL_TRUE, the comparison is more forgiving. In this case @c a is
 * allowed to have keywords or table columns that @c b does not have and visa
 * versa. However, any keywords/columns that exist in both spectrum objects must
 * have identical values.
 */
/*----------------------------------------------------------------------------*/
cpl_boolean espdr_sdp_spectrum_equal(const espdr_sdp_spectrum *a,
                                      const espdr_sdp_spectrum *b,
                                      cpl_boolean only_intersect)
{
  cpl_errorstate prestate;
  cpl_size na, i;
  cpl_boolean no_match = CPL_FALSE;
  cpl_array *names;
  const char *name;
  const cpl_property *pa, *pb;

  cpl_ensure(a != NULL && b != NULL, CPL_ERROR_NULL_INPUT, CPL_FALSE);

  assert(a->proplist != NULL);
  assert(a->table != NULL);
  assert(b->proplist != NULL);
  assert(b->table != NULL);

  na = cpl_propertylist_get_size(a->proplist);

  if (only_intersect) {
    /* Check that the values are the same if the keywords are in both property
     * lists. (Ignore comments) */
    for (i = 0; i < na; ++i) {
      pa = cpl_propertylist_get_const(a->proplist, i);
      cpl_error_ensure(pa != NULL, cpl_error_get_code(), return CPL_FALSE,
                  "Failed to get property structure %"CPL_SIZE_FORMAT".", i);
      name = cpl_property_get_name(pa);
      cpl_error_ensure(name != NULL, cpl_error_get_code(), return CPL_FALSE,
                  "Failed to get the name for property %"CPL_SIZE_FORMAT".", i);
      pb = cpl_propertylist_get_property_const(b->proplist, name);
      if (pb != NULL) {
        prestate = cpl_errorstate_get();
        if (! _espdr_property_equal(pa, pb)) return CPL_FALSE;
        if (! cpl_errorstate_is_equal(prestate)) return CPL_FALSE;
      }
    }

    /* Check that the columns with the same names in both tables are identical
     * for the parts of the data arrays that overlap. */
    prestate = cpl_errorstate_get();
    na = cpl_table_get_ncol(a->table);
    names = cpl_table_get_column_names(a->table);
    for (i = 0; i < na; ++i) {
      name = cpl_array_get_string(names, i);
      cpl_error_ensure(name != NULL, cpl_error_get_code(), break,
                  "Failed to get the name for column %"CPL_SIZE_FORMAT".", i);
      if (cpl_table_has_column(b->table, name)) {
        if (! _espdr_table_column_equal(a->table, b->table, name, CPL_TRUE)) {
          no_match = CPL_TRUE;
          break;
        }
      }
    }
    cpl_array_delete(names);
    /* Check that no errors occurred and all columns were processed. */
    if (no_match || ! cpl_errorstate_is_equal(prestate)) return CPL_FALSE;

  } else { /* not only_intersect */
    cpl_size nb;
    if (a->nelem != b->nelem) return CPL_FALSE;

    /* Check that the property lists are identical. (Ignore comments) */
    nb = cpl_propertylist_get_size(b->proplist);
    if (na != nb) return CPL_FALSE;
    for (i = 0; i < na; ++i) {
      pa = cpl_propertylist_get_const(a->proplist, i);
      cpl_error_ensure(pa != NULL, cpl_error_get_code(), return CPL_FALSE,
                  "Failed to get property structure %"CPL_SIZE_FORMAT".", i);
      name = cpl_property_get_name(pa);
      cpl_error_ensure(name != NULL, cpl_error_get_code(), return CPL_FALSE,
                  "Failed to get the name for property %"CPL_SIZE_FORMAT".", i);
      pb = cpl_propertylist_get_property_const(b->proplist, name);
      if (pb == NULL) return CPL_FALSE;
      prestate = cpl_errorstate_get();
      if (! _espdr_property_equal(pa, pb)) return CPL_FALSE;
      if (! cpl_errorstate_is_equal(prestate)) return CPL_FALSE;
    }

    /* Check that the tables are identical. */
    prestate = cpl_errorstate_get();
    na = cpl_table_get_ncol(a->table);
    nb = cpl_table_get_ncol(b->table);
    if (na != nb) return CPL_FALSE;
    names = cpl_table_get_column_names(a->table);
    for (i = 0; i < na; ++i) {
      name = cpl_array_get_string(names, i);
      cpl_error_ensure(name != NULL, cpl_error_get_code(), break,
                  "Failed to get the name for column %"CPL_SIZE_FORMAT".", i);
      if (! cpl_table_has_column(b->table, name)
          || ! _espdr_table_column_equal(a->table, b->table, name, CPL_FALSE))
      {
        no_match = CPL_TRUE;
        break;
      }
    }
    cpl_array_delete(names);
    /* Check that no errors occurred and all columns were processed. */
    if (no_match || ! cpl_errorstate_is_equal(prestate)) return CPL_FALSE;
  }

  /* If we got to this point then all checks passed, so return true (a == b). */
  espdr_check_error_code(cpl_func);
  return CPL_TRUE;
}

/*----------------------------------------------------------------------------*/
/**
 * @internal
 * @brief Counts the number of keywords matching a given regular expression.
 */
/*----------------------------------------------------------------------------*/
static cpl_size
_espdr_sdp_spectrum_count_keywords(const espdr_sdp_spectrum *self,
                                    const char *regexp)
{
  cpl_error_code error;
  cpl_size result = 0;
  cpl_propertylist *list = cpl_propertylist_new();

  assert(self != NULL);
  assert(self->proplist != NULL);

  error = cpl_propertylist_copy_property_regexp(list, self->proplist, regexp,
                                                CPL_FALSE);
  if (! error) {
    result = cpl_propertylist_get_size(list);
  }
  cpl_propertylist_delete(list);
  espdr_check_error_code(cpl_func);
  return result;
}


cpl_size espdr_sdp_spectrum_count_obid(const espdr_sdp_spectrum *self)
{
  cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, 0);
  return _espdr_sdp_spectrum_count_keywords(self, "^OBID[0-9]+$");
}

cpl_size espdr_sdp_spectrum_count_prov(const espdr_sdp_spectrum *self)
{
  cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, 0);
  return _espdr_sdp_spectrum_count_keywords(self, "^PROV[0-9]+$");
}

cpl_size espdr_sdp_spectrum_count_asson(const espdr_sdp_spectrum *self)
{
  cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, 0);
  return _espdr_sdp_spectrum_count_keywords(self, "^ASSON[0-9]+$");
}

cpl_size espdr_sdp_spectrum_count_assoc(const espdr_sdp_spectrum *self)
{
  cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, 0);
  return _espdr_sdp_spectrum_count_keywords(self, "^ASSOC[0-9]+$");
}

cpl_size espdr_sdp_spectrum_count_assom(const espdr_sdp_spectrum *self)
{
  cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, 0);
  return _espdr_sdp_spectrum_count_keywords(self, "^ASSOM[0-9]+$");
}


#ifndef NDEBUG

/**
 * @internal
 * @brief Check if keyword table is sorted and has unique entries.
 */
static cpl_boolean _espdr_keyword_table_is_sorted(
                            const espdr_keyword_record *table, size_t entries)
{
  size_t i;
  if (entries < 2) return CPL_TRUE;
  for (i = 0; i < entries-1; ++i) {
    if (strcmp(table[i].name, table[i+1].name) >= 0) {
      return CPL_FALSE;
    }
  }
  return CPL_TRUE;
}

#endif /* NDEBUG */


static const espdr_keyword_record *
_espdr_sdp_spectrum_get_keyword_record(const char *name)
{
  /* The following table should contain all valid SDP spectrum keywords being
   * handled. NOTE: this table must be kept sorted since we perform a binary
   * search on the first column (i.e. the keyword name). */
  static const espdr_keyword_record keyword_table[] = {
  /*    Name              Comment              Type         Is an array key */
    {KEY_APERTURE,  KEY_APERTURE_COMMENT, CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_ASSOC,     KEY_ASSOC_COMMENT,    CPL_TYPE_STRING,   CPL_TRUE},
    {KEY_ASSOM,     KEY_ASSOM_COMMENT,    CPL_TYPE_STRING,   CPL_TRUE},
    {KEY_ASSON,     KEY_ASSON_COMMENT,    CPL_TYPE_STRING,   CPL_TRUE},
    {KEY_CONTNORM,  KEY_CONTNORM_COMMENT, CPL_TYPE_BOOL,     CPL_FALSE},
    {KEY_DEC,       KEY_DEC_COMMENT,      CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_DETRON,    KEY_DETRON_COMMENT,   CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_DISPELEM,  KEY_DISPELEM_COMMENT, CPL_TYPE_STRING,   CPL_FALSE},
    {KEY_EFFRON,    KEY_EFFRON_COMMENT,   CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_EXPTIME,   KEY_EXPTIME_COMMENT,  CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_EXTNAME,   KEY_EXTNAME_COMMENT,  CPL_TYPE_STRING,   CPL_FALSE},
    {KEY_EXT_OBJ,   KEY_EXT_OBJ_COMMENT,  CPL_TYPE_BOOL,     CPL_FALSE},
    {KEY_FLUXCAL,   KEY_FLUXCAL_COMMENT,  CPL_TYPE_STRING,   CPL_FALSE},
    {KEY_FLUXERR,   KEY_FLUXERR_COMMENT,  CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_GAIN,      KEY_GAIN_COMMENT,     CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_INHERIT,   KEY_INHERIT_COMMENT,  CPL_TYPE_BOOL,     CPL_FALSE},
    {KEY_LAMNLIN,   KEY_LAMNLIN_COMMENT,  CPL_TYPE_INT,      CPL_FALSE},
    {KEY_LAMRMS,    KEY_LAMRMS_COMMENT,   CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_MJDEND,    KEY_MJDEND_COMMENT,   CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_MJDOBS,    KEY_MJDOBS_COMMENT,   CPL_TYPE_DOUBLE,   CPL_FALSE},
    //{KEY_M_EPOCH,   KEY_M_EPOCH_COMMENT,  CPL_TYPE_BOOL,     CPL_FALSE},
    {KEY_NCOMBINE,  KEY_NCOMBINE_COMMENT, CPL_TYPE_INT,      CPL_FALSE},
    {KEY_NELEM,     KEY_NELEM_COMMENT,    ESPDR_TYPE_NELEM, CPL_FALSE},
    {KEY_NOESODAT,  KEY_NOESODAT_COMMENT, CPL_TYPE_BOOL, CPL_TRUE},
    {KEY_OBID,      KEY_OBID_COMMENT,     CPL_TYPE_INT,      CPL_TRUE},
    {KEY_OBJECT,    KEY_OBJECT_COMMENT,   CPL_TYPE_STRING,   CPL_FALSE},
    {KEY_OBSTECH,   KEY_OBSTECH_COMMENT,  CPL_TYPE_STRING,   CPL_FALSE},
    {KEY_ORIGIN,    KEY_ORIGIN_COMMENT,   CPL_TYPE_STRING,   CPL_FALSE},
    {KEY_PROCSOFT,  KEY_PROCSOFT_COMMENT, CPL_TYPE_STRING,   CPL_FALSE},
    {KEY_PRODCATG,  KEY_PRODCATG_COMMENT, CPL_TYPE_STRING,   CPL_FALSE},
    {KEY_PRODLVL,   KEY_PRODLVL_COMMENT,  CPL_TYPE_INT,      CPL_FALSE},
    {KEY_PROG_ID,   KEY_PROG_ID_COMMENT,  CPL_TYPE_STRING,   CPL_FALSE},
    {KEY_PROV,      KEY_PROV_COMMENT,     CPL_TYPE_STRING,   CPL_TRUE},
    {KEY_RA,        KEY_RA_COMMENT,       CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_REFERENC,  KEY_REFERENC_COMMENT, CPL_TYPE_STRING,   CPL_FALSE},
    {KEY_SNR,       KEY_SNR_COMMENT,      CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_SPECSYS,   KEY_SPECSYS_COMMENT,  CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_SPEC_BIN,  KEY_SPEC_BIN_COMMENT, CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_SPEC_BW,   KEY_SPEC_BW_COMMENT,  CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_SPEC_ERR,  KEY_SPEC_ERR_COMMENT, CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_SPEC_RES,  KEY_SPEC_RES_COMMENT, CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_SPEC_SYE,  KEY_SPEC_SYE_COMMENT, CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_SPEC_VAL,  KEY_SPEC_VAL_COMMENT, CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_TCOMM,     KEY_TCOMM_COMMENT,    CPL_TYPE_STRING,   CPL_TRUE},
    {KEY_TDMAX(1),  KEY_TDMAX1_COMMENT,   CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_TDMIN(1),  KEY_TDMIN1_COMMENT,   CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_TELAPSE,   KEY_TELAPSE_COMMENT,  CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_TEXPTIME,  KEY_TEXPTIME_COMMENT, CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_TIMESYS,   KEY_TIMESYS_COMMENT,  CPL_TYPE_STRING,   CPL_FALSE},
    {KEY_TITLE,     KEY_TITLE_COMMENT,    CPL_TYPE_STRING,   CPL_FALSE},
    {KEY_TMID,      KEY_TMID_COMMENT,     CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_TOT_FLUX,  KEY_TOT_FLUX_COMMENT, CPL_TYPE_BOOL,     CPL_FALSE},
    {KEY_TUCD,      KEY_TUCD_COMMENT,     CPL_TYPE_STRING,   CPL_TRUE},
    {KEY_TUTYP,     KEY_TUTYP_COMMENT,    CPL_TYPE_STRING,   CPL_TRUE},
    {KEY_VOCLASS,   KEY_VOCLASS_COMMENT,  CPL_TYPE_STRING,   CPL_FALSE},
    {KEY_VOPUB,     KEY_VOPUB_COMMENT,    CPL_TYPE_STRING,   CPL_FALSE},
    {KEY_WAVELMAX,  KEY_WAVELMAX_COMMENT, CPL_TYPE_DOUBLE,   CPL_FALSE},
    {KEY_WAVELMIN,  KEY_WAVELMIN_COMMENT, CPL_TYPE_DOUBLE,   CPL_FALSE}
  };

  static const size_t tablesize =
                        sizeof(keyword_table) / sizeof(espdr_keyword_record);
  size_t low = 0;             /* Low end of search region. */
  size_t high = tablesize-1;  /* High end of search region. */
  const espdr_keyword_record *record = NULL;

  assert(_espdr_keyword_table_is_sorted(keyword_table, tablesize));
  assert(name != NULL);

  /* Binary search for the keyword record who's name forms the prefix of the
   * 'name' string passed to this function or is equal to that string. We cannot
   * just check if they equal since the OBIDi, PROVi, ASSONi, ASSOCi, ASSOMi,
   * TUTYPi and TUCDi keywords all have a number suffix that needs to be dealt
   * with. */
  do {
    size_t mid = (low + high) >> 1;   /* Find mid point between low and high. */
    size_t keylen = strlen(keyword_table[mid].name);
    int result = strncmp(name, keyword_table[mid].name, keylen);
    if (result == 0) {
      record = &keyword_table[mid];
      break;
    } else if (result < 0) {
      if (mid >= 1) {
        high = mid - 1;
      } else {
        return NULL;
      }
    } else {
      low = mid + 1;
      if (low > high) return NULL;
    }
  } while (1);

  assert(record != NULL);

  if (strlen(record->name) != strlen(name)) {
    if (! record->is_array_key) return NULL;
    /* Have to check if the keyword format is correct. Should only have digits
     * following the name prefix. */
    const char *c = name + strlen(record->name);
    while (*c != '\0') {
      if (! isdigit(*c)) return NULL;
      ++c;
    }
  }
  espdr_check_error_code(cpl_func);
  return record;
}


cpl_error_code espdr_sdp_spectrum_copy_keyword(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name)
{
  const espdr_keyword_record *key;
  cpl_errorstate prestate = cpl_errorstate_get();
  cpl_boolean spectrum_has_keyword;

  cpl_ensure_code(self != NULL && plist != NULL && name != NULL,
                  CPL_ERROR_NULL_INPUT);

  assert(self->proplist != NULL);

  if (! cpl_propertylist_has(plist, name)) {
    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                    "Could not set '%s' since the keyword was not found in the"
                    " source list.", name);
  }

  key = _espdr_sdp_spectrum_get_keyword_record(name);
  if (key == NULL) {
    return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                    "The keyword name '%s' is not valid for an SPD spectrum.",
                    name);
  }

  spectrum_has_keyword = cpl_propertylist_has(self->proplist, name);

  switch ((int) key->type) {
  case CPL_TYPE_BOOL:
    {
      /* Note: we update with the following functions rather than using the
       * cpl_propertylist_copy_property function since this way we get basic
       * typecasting functionality. e.g. floats get converted to doubles. */
      cpl_boolean value = cpl_propertylist_get_bool(plist, name);
      cpl_propertylist_update_bool(self->proplist, name, value);
    }
    break;
  case CPL_TYPE_INT:
    {
      int value = cpl_propertylist_get_int(plist, name);
      cpl_propertylist_update_int(self->proplist, name, value);
    }
    break;
  case CPL_TYPE_DOUBLE:
    {
      double value = cpl_propertylist_get_double(plist, name);
      cpl_propertylist_update_double(self->proplist, name, value);
    }
    break;
  case CPL_TYPE_STRING:
    {
      const char *value = cpl_propertylist_get_string(plist, name);
      cpl_propertylist_update_string(self->proplist, name, value);
    }
    break;
  case ESPDR_TYPE_NELEM:
    {
      /* Special case where we update the nelem field. */
      spectrum_has_keyword = CPL_TRUE;  /* Skip trying to set comment. */
      cpl_size value = (cpl_size) cpl_propertylist_get_long_long(plist, name);
      if (cpl_errorstate_is_equal(prestate)) {
        espdr_sdp_spectrum_set_nelem(self, value);
      }
    }
    break;
  default:
    return cpl_error_set_message(cpl_func, CPL_ERROR_INVALID_TYPE,
                      "Cannot handle type '%s'.", cpl_type_get_name(key->type));
  }

  if (! spectrum_has_keyword) {
    cpl_propertylist_set_comment(self->proplist, name, key->comment);
  }

  if (! cpl_errorstate_is_equal(prestate)) {
    if (! spectrum_has_keyword) {
      /* Make sure the keyword is removed if we have an error and it was not
       * there to begin with. */
      prestate = cpl_errorstate_get();
      (void) cpl_propertylist_erase(self->proplist, name);
      cpl_errorstate_set(prestate);
    }
    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                  "Could not set '%s'. Likely the keyword from the source list"
                  " has a different format or type.", name);
  }
  espdr_check_error_code(cpl_func);
  return CPL_ERROR_NONE;
}


cpl_error_code espdr_sdp_spectrum_copy_property(espdr_sdp_spectrum *self,
                                                 const cpl_property *prop)
{
  const char *name;
  const espdr_keyword_record *key;
  cpl_errorstate prestate = cpl_errorstate_get();
  cpl_boolean spectrum_has_keyword;

  cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

  assert(self->proplist != NULL);

  name = cpl_property_get_name(prop);
  if (name == NULL) return cpl_error_get_code();

  key = _espdr_sdp_spectrum_get_keyword_record(name);
  if (key == NULL) {
    return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                    "The keyword name '%s' is not valid for an SPD spectrum.",
                    name);
  }

  spectrum_has_keyword = cpl_propertylist_has(self->proplist, name);

  switch ((int) key->type) {
  case CPL_TYPE_BOOL:
    {
      cpl_boolean value = cpl_property_get_bool(prop);
      cpl_propertylist_update_bool(self->proplist, name, value);
    }
    break;
  case CPL_TYPE_INT:
    {
      int value = cpl_property_get_int(prop);
      cpl_propertylist_update_int(self->proplist, name, value);
    }
    break;
  case CPL_TYPE_DOUBLE:
    {
      double value = cpl_property_get_double(prop);
      cpl_propertylist_update_double(self->proplist, name, value);
    }
    break;
  case CPL_TYPE_STRING:
    {
      const char *value = cpl_property_get_string(prop);
      cpl_propertylist_update_string(self->proplist, name, value);
    }
    break;
  case ESPDR_TYPE_NELEM:
    {
      /* Special case where we update the nelem field. */
      spectrum_has_keyword = CPL_TRUE;  /* Skip trying to set comment. */
      cpl_size value = (cpl_size) cpl_property_get_long_long(prop);
      if (cpl_errorstate_is_equal(prestate)) {
        espdr_sdp_spectrum_set_nelem(self, value);
      }
    }
    break;
  default:
    return cpl_error_set_message(cpl_func, CPL_ERROR_INVALID_TYPE,
                      "Cannot handle type '%s'.", cpl_type_get_name(key->type));
  }

  if (! spectrum_has_keyword) {
    cpl_propertylist_set_comment(self->proplist, name, key->comment);
  }

  if (! cpl_errorstate_is_equal(prestate)) {
    if (! spectrum_has_keyword) {
      /* Make sure the keyword is removed if we have an error and it was not
       * there to begin with. */
      prestate = cpl_errorstate_get();
      (void) cpl_propertylist_erase(self->proplist, name);
      cpl_errorstate_set(prestate);
    }
    espdr_check_error_code(cpl_func);
    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                      "Could not set '%s'. Likely the source property has a"
                      " different format or type.", name);
  }
  espdr_check_error_code(cpl_func);
  return CPL_ERROR_NONE;
}


cpl_error_code espdr_sdp_spectrum_copy_property_regexp(
                                                espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *regexp,
                                                int invert)
{
  cpl_propertylist *sublist = NULL;
  cpl_propertylist *origlist = NULL;
  cpl_errorstate prestate = cpl_errorstate_get();
  cpl_size i;

  cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

  assert(self->proplist != NULL);

  sublist = cpl_propertylist_new();
  origlist = cpl_propertylist_new();
  cpl_propertylist_copy_property_regexp(origlist, self->proplist, regexp,
                                        invert);
  cpl_propertylist_copy_property_regexp(sublist, plist, regexp, invert);
  if (cpl_propertylist_has(sublist, KEY_NELEM)) {
    /* Move the NELEM key to the end of the list so that rollback on error is
     * easier. */
    cpl_propertylist_erase(sublist, KEY_NELEM);
    cpl_propertylist_copy_property(sublist, plist, KEY_NELEM);
  }
  if (! cpl_errorstate_is_equal(prestate)) goto cleanup;

  for (i = 0; i < cpl_propertylist_get_size(sublist); ++i) {
    const cpl_property *p = cpl_propertylist_get_const(sublist, i);
    const char *name = cpl_property_get_name(p);
    espdr_sdp_spectrum_copy_keyword(self, sublist, name);
    if (! cpl_errorstate_is_equal(prestate)) goto cleanup;
  }

  cpl_propertylist_delete(sublist);
  cpl_propertylist_delete(origlist);
  espdr_check_error_code(cpl_func);
  return CPL_ERROR_NONE;

cleanup:
  /* Cleanup here if an error occurred, by restoring the keywords to the
   * original values. */
  prestate = cpl_errorstate_get();
  (void) cpl_propertylist_copy_property_regexp(self->proplist, origlist,
                                               ".*", 0);
  cpl_errorstate_set(prestate);
  cpl_propertylist_delete(sublist);
  cpl_propertylist_delete(origlist);
  espdr_check_error_code(cpl_func);
  return cpl_error_get_code();
}


GET_SET_METHODS_TYPE_DOUBLE(ra, KEY_RA, KEY_RA_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(dec, KEY_DEC, KEY_DEC_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(exptime, KEY_EXPTIME, KEY_EXPTIME_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(texptime, KEY_TEXPTIME, KEY_TEXPTIME_COMMENT)
GET_SET_METHODS_TYPE_STRING(timesys, KEY_TIMESYS, KEY_TIMESYS_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(mjdobs, KEY_MJDOBS, KEY_MJDOBS_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(mjdend, KEY_MJDEND, KEY_MJDEND_COMMENT)
GET_SET_METHODS_TYPE_INT(prodlvl, KEY_PRODLVL, KEY_PRODLVL_COMMENT)
GET_SET_METHODS_TYPE_STRING(procsoft, KEY_PROCSOFT, KEY_PROCSOFT_COMMENT)
GET_SET_METHODS_TYPE_STRING(prodcatg, KEY_PRODCATG, KEY_PRODCATG_COMMENT)
GET_SET_METHODS_TYPE_STRING(origin, KEY_ORIGIN, KEY_ORIGIN_COMMENT)
GET_SET_METHODS_TYPE_BOOL(extobj, KEY_EXT_OBJ, KEY_EXT_OBJ_COMMENT)
GET_SET_METHODS_TYPE_STRING(dispelem, KEY_DISPELEM, KEY_DISPELEM_COMMENT)
GET_SET_METHODS_TYPE_STRING(specsys, KEY_SPECSYS, KEY_SPECSYS_COMMENT)
GET_SET_METHODS_TYPE_STRING(progid, KEY_PROG_ID, KEY_PROG_ID_COMMENT)
GET_SET_ARRAY_METHODS_TYPE_INT(obid, KEY_OBID, KEY_OBID_COMMENT)
//GET_SET_METHODS_TYPE_BOOL(mepoch, KEY_M_EPOCH, KEY_M_EPOCH_COMMENT)
GET_SET_METHODS_TYPE_STRING(obstech, KEY_OBSTECH, KEY_OBSTECH_COMMENT)
GET_SET_METHODS_TYPE_STRING(fluxcal, KEY_FLUXCAL, KEY_FLUXCAL_COMMENT)
GET_SET_METHODS_TYPE_BOOL(contnorm, KEY_CONTNORM, KEY_CONTNORM_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(wavelmin, KEY_WAVELMIN, KEY_WAVELMIN_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(wavelmax, KEY_WAVELMAX, KEY_WAVELMAX_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(specbin, KEY_SPEC_BIN, KEY_SPEC_BIN_COMMENT)
GET_SET_METHODS_TYPE_BOOL(totflux, KEY_TOT_FLUX, KEY_TOT_FLUX_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(fluxerr, KEY_FLUXERR, KEY_FLUXERR_COMMENT)
GET_SET_METHODS_TYPE_STRING(referenc, KEY_REFERENC, KEY_REFERENC_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(specres, KEY_SPEC_RES, KEY_SPEC_RES_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(specerr, KEY_SPEC_ERR, KEY_SPEC_ERR_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(specsye, KEY_SPEC_SYE, KEY_SPEC_SYE_COMMENT)
GET_SET_METHODS_TYPE_INT(lamnlin, KEY_LAMNLIN, KEY_LAMNLIN_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(lamrms, KEY_LAMRMS, KEY_LAMRMS_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(gain, KEY_GAIN, KEY_GAIN_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(detron, KEY_DETRON, KEY_DETRON_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(effron, KEY_EFFRON, KEY_EFFRON_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(snr, KEY_SNR, KEY_SNR_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(snr_avg, KEY_SNR_AVG, KEY_SNR_AVG_COMMENT)
GET_SET_METHODS_TYPE_INT(ncombine, KEY_NCOMBINE, KEY_NCOMBINE_COMMENT)
GET_SET_ARRAY_METHODS_TYPE_STRING(prov, KEY_PROV, KEY_PROV_COMMENT)
GET_SET_ARRAY_METHODS_TYPE_STRING(asson, KEY_ASSON, KEY_ASSON_COMMENT)
GET_SET_ARRAY_METHODS_TYPE_STRING(assoc, KEY_ASSOC, KEY_ASSOC_COMMENT)
GET_SET_ARRAY_METHODS_TYPE_STRING(assom, KEY_ASSOM, KEY_ASSOM_COMMENT)
GET_SET_METHODS_TYPE_STRING(voclass, KEY_VOCLASS, KEY_VOCLASS_COMMENT)
GET_SET_METHODS_TYPE_STRING(vopub, KEY_VOPUB, KEY_VOPUB_COMMENT)
GET_SET_METHODS_TYPE_STRING(title, KEY_TITLE, KEY_TITLE_COMMENT)
GET_SET_METHODS_TYPE_STRING(object, KEY_OBJECT, KEY_OBJECT_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(aperture, KEY_APERTURE, KEY_APERTURE_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(telapse, KEY_TELAPSE, KEY_TELAPSE_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(tmid, KEY_TMID, KEY_TMID_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(specval, KEY_SPEC_VAL, KEY_SPEC_VAL_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(specbw, KEY_SPEC_BW, KEY_SPEC_BW_COMMENT)
GET_SET_METHODS_TYPE_STRING(extname, KEY_EXTNAME, KEY_EXTNAME_COMMENT)
GET_SET_METHODS_TYPE_BOOL(inherit, KEY_INHERIT, KEY_INHERIT_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(tdmin, KEY_TDMIN(1), KEY_TDMIN1_COMMENT)
GET_SET_METHODS_TYPE_DOUBLE(tdmax, KEY_TDMAX(1), KEY_TDMAX1_COMMENT)


#if 0
/*----------------------------------------------------------------------------*/
/**
 * @brief Replace a keyword comment by a custom string.
 * @param self    The spectrum object to update.
 * @param keyword The name of the keyword to update.
 * @param comment The string to use as keyword comment.
 * @return @c CPL_ERROR_NONE on success or an appropriate error code otherwise.
 *
 * This function will replace the comment of the keyword @c keyword with
 * the string @c comment. The string @c keyword is the full keyword name.
 * Both, the keyword @c keyword and the new comment @c comment must not
 * be @a NULL.
 * If any error occurs then a error code is set and returned, otherwise
 * @c CPL_ERROR_NONE is returned on success.
 */
/*----------------------------------------------------------------------------*/

cpl_error_code
espdr_sdp_spectrum_replace_comment(espdr_sdp_spectrum *self,
                                    const char *keyword,
                                    const char *comment)
{
    cpl_ensure_code((self != NULL) && (keyword != NULL) && (comment != NULL),
                    CPL_ERROR_NULL_INPUT);
    cpl_ensure_code((self->proplist != NULL), CPL_ERROR_ILLEGAL_INPUT);

    if (!cpl_propertylist_has(self->proplist, keyword)) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                    "Could not find '%s' keyword.", keyword);
    }

    cpl_propertylist_set_comment(self->proplist, keyword, comment);
    return CPL_ERROR_NONE;
}
#endif


cpl_error_code espdr_sdp_spectrum_append_prov(espdr_sdp_spectrum *self,
                                               cpl_size firstindex,
                                               const cpl_frameset *frames)
{
  cpl_frameset_iterator* iter = NULL;
  cpl_propertylist* keywords = NULL;
  const cpl_frame* frame;
  cpl_size index = firstindex;

  /* Note: check for NULL already done in espdr_sdp_spectrum_set_prov below. */
  assert(self != NULL);
  assert(self->proplist != NULL);

  iter = cpl_frameset_iterator_new(frames);
  frame = cpl_frameset_iterator_get_const(iter);
  while (frame != NULL) {
    cpl_error_code error;
    const char* value = NULL;

    /* Load the keywords from the raw frame. */
    const char* filename = cpl_frame_get_filename(frame);
    cpl_error_ensure(filename != NULL, cpl_error_get_code(), goto cleanup,
                     "%s", cpl_error_get_message());
    keywords = cpl_propertylist_load(filename, 0);
    cpl_error_ensure(filename != NULL, cpl_error_get_code(), goto cleanup,
                     "Could not load keywords from primary HDU in '%s'.",
                     filename);

    /* Try set the value to ARCFILE or ORIGFILE or just the filename, whichever
     * is found first in that order. */
    if (cpl_propertylist_has(keywords, KEY_ARCFILE)) {
      value = cpl_propertylist_get_string(keywords, KEY_ARCFILE);
      cpl_error_ensure(value != NULL, cpl_error_get_code(), goto cleanup,
                       "Could not extract the '%s' keyword value from '%s'.",
                       KEY_ARCFILE, filename);
    } else if (cpl_propertylist_has(keywords, KEY_ORIGFILE)) {
      value = cpl_propertylist_get_string(keywords, KEY_ORIGFILE);
      cpl_error_ensure(value != NULL, cpl_error_get_code(), goto cleanup,
                       "Could not extract the '%s' keyword value from '%s'.",
                       KEY_ORIGFILE, filename);
    } else {
      value = filename;
    }

    /* Add the next PROVi keyword. */
    error = espdr_sdp_spectrum_set_prov(self, index, value);
    cpl_error_ensure(! error, error, goto cleanup,
                     "%s", cpl_error_get_message());
    cpl_propertylist_delete(keywords);
    keywords = NULL;

    /* Increment the iterator to the next frame. */
    cpl_errorstate status = cpl_errorstate_get();
    cpl_frameset_iterator_advance(iter, 1);
    if (cpl_error_get_code() == CPL_ERROR_ACCESS_OUT_OF_RANGE) {
        cpl_errorstate_set(status);
    }
    frame = cpl_frameset_iterator_get_const(iter);
    ++index;
  }

  cpl_frameset_iterator_delete(iter);
  espdr_check_error_code(cpl_func);
  return CPL_ERROR_NONE;

cleanup:
  /* Cleanup if an error occurs. Note: delete methods already check for NULL. */
  cpl_frameset_iterator_delete(iter);
  cpl_propertylist_delete(keywords);
  espdr_check_error_code(cpl_func);
  return cpl_error_get_code();
}


cpl_size espdr_sdp_spectrum_get_nelem(const espdr_sdp_spectrum *self)
{
  cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, 0);
  return self->nelem;
}


cpl_error_code espdr_sdp_spectrum_reset_nelem(espdr_sdp_spectrum *self)
{
  return espdr_sdp_spectrum_set_nelem(self, 0);
}


cpl_error_code espdr_sdp_spectrum_set_nelem(espdr_sdp_spectrum *self,
                                             cpl_size value)
{
  cpl_size ncol;
  cpl_error_code error = CPL_ERROR_NONE;

  cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, 0);

  assert(self->table != NULL);

  ncol = cpl_table_get_ncol(self->table);
  if (ncol > 0) {
    /* Update all column depths. */
    cpl_size i;
    cpl_array *names = cpl_table_get_column_names(self->table);
    for (i = 0; i < ncol; ++i) {
      const char *name = cpl_array_get_string(names, i);
      error = cpl_table_set_column_depth(self->table, name, value);
      if (error) {
        /* If an error occurs then set the columns that were changed back to
         * the previous value. */
        cpl_size j;
        cpl_errorstate prestate = cpl_errorstate_get();
        for (j = 0; j < i; ++j) {
          (void) cpl_table_set_column_depth(self->table, name, self->nelem);
        }
        cpl_errorstate_set(prestate);
        break;
      }
    }
    cpl_array_delete(names);
  }
  if (! error) {
    self->nelem = value;
  }
  espdr_check_error_code(cpl_func);
  return error;
}


cpl_error_code espdr_sdp_spectrum_copy_nelem(espdr_sdp_spectrum *self,
                                              const cpl_propertylist *plist,
                                              const char *name)
{
  /* Note: check for plist or name == NULL is done in cpl_propertylist calls. */
  cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

  assert(self->proplist != NULL);

  if (cpl_propertylist_has(plist, name)) {
    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_size value = (cpl_size) cpl_propertylist_get_long_long(plist, name);
    if (cpl_errorstate_is_equal(prestate)) {
      return espdr_sdp_spectrum_set_nelem(self, value);
    } else {
      return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                "Could not set '%s'. Likely the source '%s' keyword has a"
                " different format or type.", KEY_NELEM, name);
    }
  } else {
	  espdr_check_error_code(cpl_func);
    return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                "Could not set '%s' since the '%s' keyword was not found.",
                KEY_NELEM, name);
  }
  espdr_check_error_code(cpl_func);
}


cpl_size espdr_sdp_spectrum_get_ncol(const espdr_sdp_spectrum *self)
{
  cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, 0);
  assert(self->table != NULL);
  return cpl_table_get_ncol(self->table);
}


cpl_boolean espdr_sdp_spectrum_has_column(const espdr_sdp_spectrum *self,
                                           const char* name)
{
  cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, 0);
  assert(self->table != NULL);
  return cpl_table_has_column(self->table, name);
}


cpl_array *
espdr_sdp_spectrum_get_column_names(const espdr_sdp_spectrum *self)
{
  cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
  assert(self->table != NULL);
  return cpl_table_get_column_names(self->table);
}


cpl_error_code
espdr_sdp_spectrum_new_column(espdr_sdp_spectrum *self, const char *name,
                               cpl_type type)
{
  cpl_error_code error;
  cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
  assert(self->table != NULL);
  error = cpl_table_new_column_array(self->table, name, type, self->nelem);
  if (error) {
    cpl_error_set_message(cpl_func, cpl_error_get_code(),
                          "Failed to create a new column called '%s'.", name);
  }
  espdr_check_error_code(cpl_func);
  return error;
}


cpl_error_code
espdr_sdp_spectrum_add_column(espdr_sdp_spectrum *self, const char *name,
                               cpl_type type, const char *unit,
                               const char *format, const char *tutyp,
                               const char *tucd, const cpl_array *data)
{
  cpl_error_code error;

  /* Note: check for name equals NULL should already be done in table calls. */
  cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

  assert(self->table != NULL);

  /* Setup a new array cell column and fill its properties (possibly with
   * defaults). */
  error = cpl_table_new_column_array(self->table, name, type, self->nelem);
  if (unit != NULL && *unit != '\0') {
    error |= cpl_table_set_column_unit(self->table, name, unit);
  } else {
    error |= cpl_table_set_column_unit(self->table, name, " ");
  }
  if (format != NULL) {
    error |= cpl_table_set_column_format(self->table, name, format);
  }
  if (tutyp != NULL) {
    error |= espdr_sdp_spectrum_set_column_tutyp(self, name, tutyp);
  } else {
    error |= espdr_sdp_spectrum_set_column_tutyp(self, name, "");
  }
  if (tucd != NULL) {
    error |= espdr_sdp_spectrum_set_column_tucd(self, name, tucd);
  } else {
    error |= espdr_sdp_spectrum_set_column_tucd(self, name, "");
  }

  /* Fill the table cell with the data array if available, else add an empty
   * array. */
  if (! error) {
    if (data != NULL) {
      error = cpl_table_set_array(self->table, name, 0, data);
    } else {
      cpl_array *array = cpl_array_new(self->nelem, type);
      if (array != NULL) {
        error = cpl_table_set_array(self->table, name, 0, array);
        cpl_array_delete(array);
      } else {
        error = cpl_error_get_code();
      }
    }
  }

  if (error) {
    /* Remove the column just added if there was an error. We initially save and
     * finally restore the error state since we might generate secondary errors
     * when trying to remove the partially created column. But these secondary
     * errors are expected and irrelevant. */
    cpl_errorstate prestate = cpl_errorstate_get();
    _espdr_sdp_spectrum_erase_column_keywords(self, name);
    (void) cpl_table_erase_column(self->table, name);
    cpl_errorstate_set(prestate);
    error = cpl_error_set_message(cpl_func, cpl_error_get_code(),
                            "Failed to create a new column called '%s'.", name);
  }
  espdr_check_error_code(cpl_func);
  return error;
}


cpl_error_code
espdr_sdp_spectrum_delete_column(espdr_sdp_spectrum *self, const char *name)
{
  cpl_errorstate prestate = cpl_errorstate_get();
  cpl_error_code error = CPL_ERROR_NONE;

  cpl_ensure_code(self != NULL && name != NULL, CPL_ERROR_NULL_INPUT);

  assert(self->table != NULL);

  _espdr_sdp_spectrum_erase_column_keywords(self, name);
  if (! cpl_errorstate_is_equal(prestate)) {
    error |= cpl_error_get_code();
  }
  error |= cpl_table_erase_column(self->table, name);
  if (error) {
    return cpl_error_get_code();
  } else {
    return CPL_ERROR_NONE;
  }
}


static cpl_error_code
_espdr_sdp_spectrum_copy_column(espdr_sdp_spectrum *self, const char *to_name,
                                 const cpl_table* table, const char *from_name)
{
  cpl_error_code error;

  assert(self != NULL);
  assert(self->table != NULL);

  error = cpl_table_duplicate_column(self->table, to_name, table, from_name);
  if (error) return error;
  error |= espdr_sdp_spectrum_set_column_tutyp(self, to_name, "");
  error |= espdr_sdp_spectrum_set_column_tucd(self, to_name, "");
  if (error) {
    /* Rollback changes if an error occurred. */
    cpl_errorstate prestate = cpl_errorstate_get();
    _espdr_sdp_spectrum_erase_column_keywords(self, to_name);
    (void) cpl_table_erase_column(self->table, to_name);
    cpl_errorstate_set(prestate);
    return cpl_error_get_code();
  }
  return CPL_ERROR_NONE;
}

static cpl_error_code
_espdr_sdp_spectrum_divide_columns(espdr_sdp_spectrum *self, const char *dividend_name,
                                   const char *divisor_name)
{
  cpl_error_code error;

  assert(self != NULL);
  espdr_print_rec_status(101);
  //cpl_table_dump_structure(self->table,stdout);
  cpl_msg_info(cpl_func,"dividend: %s",dividend_name);
  cpl_msg_info(cpl_func,"divisor: %s",divisor_name);
  cpl_msg_info(cpl_func, "dividend type: %u",
          cpl_table_get_column_type(self->table, dividend_name));
  cpl_msg_info(cpl_func, "divisor type: %u",
           cpl_table_get_column_type(self->table, divisor_name));
  cpl_msg_info(cpl_func, "type short %d",CPL_TYPE_SHORT);
  error = cpl_table_divide_columns(self->table, dividend_name, divisor_name);
  espdr_print_rec_status(10002);
  if (error) return error;
  error |= espdr_sdp_spectrum_set_column_tutyp(self, dividend_name, "");
  espdr_print_rec_status(10003);
  error |= espdr_sdp_spectrum_set_column_tucd(self, dividend_name, "");
  espdr_print_rec_status(10004);
  if (error) {
    /* Rollback changes if an error occurred. */
    cpl_errorstate prestate = cpl_errorstate_get();
    _espdr_sdp_spectrum_erase_column_keywords(self, dividend_name);
    (void) cpl_table_erase_column(self->table, dividend_name);
    cpl_errorstate_set(prestate);
    espdr_print_rec_status(10005);
    return cpl_error_get_code();
  }
  espdr_print_rec_status(10006);
  espdr_check_error_code(cpl_func);
  return CPL_ERROR_NONE;
}

static cpl_error_code
_espdr_sdp_spectrum_duplicate_column(espdr_sdp_spectrum *self, const char *to,
                                   const char *from)
{
  cpl_error_code error;

  assert(self != NULL);

  //cpl_table_dump_structure(self->table,stdout);
  error = cpl_table_duplicate_column(self->table, to, self->table, from);

  if (error) return error;
  error |= espdr_sdp_spectrum_set_column_tutyp(self, to, "");
  error |= espdr_sdp_spectrum_set_column_tucd(self, to, "");

  if (error) {
    /* Rollback changes if an error occurred. */
    cpl_errorstate prestate = cpl_errorstate_get();
    _espdr_sdp_spectrum_erase_column_keywords(self, to);
    (void) cpl_table_erase_column(self->table, to);
    cpl_errorstate_set(prestate);

    return cpl_error_get_code();
  }
  return CPL_ERROR_NONE;
}

cpl_error_code
espdr_sdp_spectrum_copy_column(espdr_sdp_spectrum *self,
                                const cpl_table* table, const char *name)
{
  /* Note: check for table == NULL || name == NULL should already be done in the
   * cpl_table calls. */
  cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
  return _espdr_sdp_spectrum_copy_column(self, name, table, name);
}


cpl_error_code
espdr_sdp_spectrum_divide_columns(espdr_sdp_spectrum *self,const char *dividend_name,
                                const char *divisor_name)
{
  /* Note: check for table == NULL || name == NULL should already be done in the
   * cpl_table calls. */
  cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
  return _espdr_sdp_spectrum_divide_columns(self, dividend_name, divisor_name);
}

cpl_error_code
espdr_sdp_spectrum_duplicate_column(espdr_sdp_spectrum *self,const char *to,
                                const char *from)
{
  /* Note: check for table == NULL || name == NULL should already be done in the
   * cpl_table calls. */
  cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
  return _espdr_sdp_spectrum_duplicate_column(self, to, from);
}

cpl_error_code
espdr_sdp_spectrum_copy_column_regexp(espdr_sdp_spectrum *self,
                                       const cpl_table* table,
                                       const char *regexp, int invert)
{
  regex_t re;
  cpl_array *names = NULL;
  cpl_size n, i;
  int reg_error_code;

  /* Note: table == NULL is checked in the cpl_table calls. */
  cpl_ensure_code(self != NULL && regexp != NULL, CPL_ERROR_NULL_INPUT);

  assert(self->table != NULL);

  reg_error_code = regcomp(&re, regexp, REG_EXTENDED | REG_NOSUB);
  if (reg_error_code != 0) {
    return cpl_error_set_regex(CPL_ERROR_ILLEGAL_INPUT, reg_error_code, &re,
                               "regexp='%s', invert=%d", regexp, invert);
  }

  /* Go through all column names in the table we are copying from and mark the
   * names the regular expression filters out as invalid. */
  names = cpl_table_get_column_names(table);
  n = cpl_array_get_size(names);
  for (i = 0; i < n; ++i) {
    int match;
    const char *namei = cpl_array_get_string(names, i);
    cpl_error_ensure(! cpl_table_has_column(self->table, namei),
                     CPL_ERROR_ILLEGAL_OUTPUT, goto cleanup,
                     "The column '%s' already exists in the spectrum.", namei);
    match = (regexec(&re, namei, 0, NULL, 0) == 0);
    if ((! match && ! invert) || (match && invert)) {
      cpl_array_set_invalid(names, i);
    }
  }
  /* Now copy only the valid columns. */
  for (i = 0; i < n; ++i) {
    if (cpl_array_is_valid(names, i)) {
      const char *namei = cpl_array_get_string(names, i);
      cpl_error_code error = _espdr_sdp_spectrum_copy_column(self, namei,
                                                              table, namei);
      if (error) {
        cpl_errorstate prestate;
        cpl_size j;
        cpl_error_set_message(cpl_func, error, "Could not copy column '%s'.",
                              namei);
        /* Remove any columns already added if we got an error copying any
         * column. */
        prestate = cpl_errorstate_get();
        for (j = 0; j < i; ++j) {
          namei = cpl_array_get_string(names, i);
          _espdr_sdp_spectrum_erase_column_keywords(self, namei);
          (void) cpl_table_erase_column(self->table, namei);
        }
        cpl_errorstate_set(prestate);
        goto cleanup;
      }
    }
  }
  cpl_array_delete(names);
  regfree(&re);
  return CPL_ERROR_NONE;

cleanup:
  /* This is a cleanup section to delete objects when an error occurs. */
  cpl_array_delete(names);
  regfree(&re);
  return cpl_error_get_code();
}


cpl_error_code
espdr_sdp_spectrum_update_column(espdr_sdp_spectrum *self, const char *name,
                                  const cpl_table* table, const char *colname,
                                  int flags)
{
  char *orig_unit = NULL;
  char *orig_format = NULL;
  cpl_errorstate prestate = cpl_errorstate_get();

  /* Note: check for name, colname equals NULL should already be done in the
   * cpl_table calls. */
  cpl_ensure_code(self != NULL && table != NULL, CPL_ERROR_NULL_INPUT);

  assert(self->table != NULL);

  if (! cpl_table_has_column(self->table, name)) {
    /* The column does not exist in the spectrum so just copy it. */
    return _espdr_sdp_spectrum_copy_column(self, name, table, colname);
  }

  /* Make sure the source column exists. */
  if (! cpl_table_has_column(table, colname)) {
    return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                 "Column '%s' not found in table.", colname);
  }

  /* Update the unit and format values if requested. Note, we copy the original
   * value to be able to restore it if an error occurs. */
  if (flags & ESPDR_COLUMN_UNIT) {
    const char* unit = cpl_table_get_column_unit(table, colname);
    /* Prevent completely empty strings else cfitsio silently deletes the
     * keyword. */
    if (unit != NULL && *unit == '\0') {
      unit = " ";
    }
    orig_unit = cpl_strdup(cpl_table_get_column_unit(self->table, name));
    cpl_table_set_column_unit(self->table, name, unit);
    if (! cpl_errorstate_is_equal(prestate)) goto cleanup;
  }
  if (flags & ESPDR_COLUMN_FORMAT) {
    orig_format = cpl_strdup(cpl_table_get_column_format(self->table, name));
    cpl_table_set_column_format(self->table, name,
                                cpl_table_get_column_format(table, colname));
    if (! cpl_errorstate_is_equal(prestate)) goto cleanup;
  }

  /* Update the data array. Leave this to the last task since it is normally
   * cheaper to rollback changes to the unit and format strings if an error
   * occurs. */
  if (flags & ESPDR_COLUMN_DATA) {
    if (cpl_table_get_column_type(self->table, name) !=
        cpl_table_get_column_type(table, colname)) {
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                    "The table column '%s' and spectrum column '%s' do not"
                    " have the same types.", colname, name);
      goto cleanup;
    }
    if (cpl_table_get_column_depth(self->table, name) !=
        cpl_table_get_column_depth(table, colname)) {
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                    "The table column '%s' and spectrum column '%s' do not"
                    " have the same dimensions.", colname, name);
      goto cleanup;
    }
    const cpl_array* data = cpl_table_get_array(table, colname, 0);
    if (data == NULL) goto cleanup;
    cpl_table_set_array(self->table, name, 0, data);
    if (! cpl_errorstate_is_equal(prestate)) goto cleanup;
  }

  cpl_free(orig_unit);
  cpl_free(orig_format);
  return CPL_ERROR_NONE;

cleanup:
  /* Cleanup if error occurred by rolling back modifications. */
  prestate = cpl_errorstate_get();
  if (orig_unit != NULL) {
    (void) cpl_table_set_column_unit(self->table, name, orig_unit);
    cpl_free(orig_unit);
  }
  if (orig_format != NULL) {
    (void) cpl_table_set_column_format(self->table, name, orig_format);
    cpl_free(orig_format);
  }
  cpl_errorstate_set(prestate);
  return cpl_error_get_code();
}


cpl_type espdr_sdp_spectrum_get_column_type(const espdr_sdp_spectrum *self,
                                             const char *name)
{
  cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, CPL_TYPE_INVALID);
  assert(self->table != NULL);
  return cpl_table_get_column_type(self->table, name);
}


const char *
espdr_sdp_spectrum_get_column_unit(const espdr_sdp_spectrum *self,
                                    const char *name)
{
  cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
  assert(self->table != NULL);
  return cpl_table_get_column_unit(self->table, name);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set the physical units for a column.
 * @param self The spectrum object to update.
 * @param name The name of the column to update.
 * @param unit The string indicating the physical units for the column.
 * @return @c CPL_ERROR_NONE on success or an appropriate error code otherwise.
 *
 * This function will set the string indicating the physical units for the
 * column named  by @c name. Valid values for @c unit is @a NULL or a string
 * with at least one character. Empty strings will be implicitly converted to a
 * string with a single space character since CFITSIO does not allow empty
 * strings for the @a TUNIT keywords. If any error occurs then a error code is
 * set and returned, otherwise @c CPL_ERROR_NONE is returned on success.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
espdr_sdp_spectrum_set_column_unit(espdr_sdp_spectrum *self,
                                    const char *name, const char *unit)
{
  cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
  assert(self->table != NULL);
  /* Prevent completely empty strings else cfitsio silently deletes the
   * keyword. */
  if (unit != NULL && *unit == '\0') {
    unit = " ";
  }
  return cpl_table_set_column_unit(self->table, name, unit);
}


cpl_error_code
espdr_sdp_spectrum_copy_column_unit(espdr_sdp_spectrum *self,
                                     const char *name,
                                     const cpl_propertylist *plist,
                                     const char *key)
{
  cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

  assert(self->table != NULL);

  if (cpl_propertylist_has(plist, key)) {
    cpl_errorstate prestate = cpl_errorstate_get();
    const char *value = cpl_propertylist_get_string(plist, key);
    if (cpl_errorstate_is_equal(prestate)) {
      /* Prevent completely empty strings else cfitsio silently deletes the
       * keyword. */
      if (value != NULL && *value == '\0') {
        value = " ";
      }
      return cpl_table_set_column_unit(self->table, name, value);
    } else {
      return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                "Could not set the unit for column '%s'. Likely the source '%s'"
                " keyword is not a string.", name, key);
    }
  } else {
    return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                "Could not set the unit for column '%s' since the '%s' keyword"
                " was not found.", name, key);
  }
}


const char *
espdr_sdp_spectrum_get_column_format(const espdr_sdp_spectrum *self,
                                      const char *name)
{
  cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
  assert(self->table != NULL);
  return cpl_table_get_column_format(self->table, name);
}


cpl_error_code
espdr_sdp_spectrum_set_column_format(espdr_sdp_spectrum *self,
                                      const char *name, const char *format)
{
  cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
  assert(self->table != NULL);
  return cpl_table_set_column_format(self->table, name, format);
}


static cpl_size
_espdr_sdp_spectrum_get_column_index(const espdr_sdp_spectrum *self,
                                      const char *name)
{
  cpl_size i, n;
  cpl_array *names;

  assert(self != NULL);
  assert(self->table != NULL);
  assert(name != NULL);

  /* Try find the index number of the column. */
  names = cpl_table_get_column_names(self->table);
  n = cpl_array_get_size(names);
  for (i = 0; i < n; ++i) {
    const char *namei = cpl_array_get_string(names, i);
    if (strcmp(namei, name) == 0) {
      cpl_array_delete(names);
      return i;
    }
  }
  cpl_array_delete(names);
  return (cpl_size)-1;
}


static const char *
_espdr_sdp_spectrum_get_column_keyword(const espdr_sdp_spectrum *self,
                                        const char *name, const char *keyword)
{
  cpl_size index;
  const char *result = NULL;

  assert(self != NULL);
  assert(self->proplist != NULL);
  assert(name != NULL);
  assert(keyword != NULL);

  index = _espdr_sdp_spectrum_get_column_index(self, name);
  if (index != (cpl_size)-1) {
    /* If the index number was found then try return the property value. */
    char *propname = cpl_sprintf("%s%"CPL_SIZE_FORMAT, keyword, index+1);
    if (cpl_propertylist_has(self->proplist, propname)) {
      result = cpl_propertylist_get_string(self->proplist, propname);
    }
    cpl_free(propname);
  } else {
    cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                "Could not find '%s' keyword for column '%s'.", keyword, name);
  }
  return result;
}


static cpl_error_code
_espdr_sdp_spectrum_set_column_keyword(espdr_sdp_spectrum *self,
                                        const char *name,
                                        const char *value,
                                        const char *keyword,
                                        const char *comment)
{
  cpl_size index;
  char *propname, *pcomment;

  assert(self != NULL);
  assert(self->proplist != NULL);
  assert(name != NULL);
  assert(keyword != NULL);
  assert(comment != NULL);

  index = _espdr_sdp_spectrum_get_column_index(self, name);
  /* If the index was not found then return an error message. */
  if (index == (cpl_size)-1) {
    return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                "Could not find '%s' keyword for column '%s'.", keyword, name);
  }
  /* Since the index number was found then try set or add the property value. */
  cpl_error_code error = CPL_ERROR_NONE;
  propname = cpl_sprintf("%s%"CPL_SIZE_FORMAT, keyword, index+1);
  pcomment = cpl_sprintf("%s%"CPL_SIZE_FORMAT, comment, index+1);
  if (cpl_propertylist_has(self->proplist, propname)) {
    if (value != NULL) {
      error = cpl_propertylist_set_string(self->proplist, propname, value);
    } else {
      (void) cpl_propertylist_erase(self->proplist, propname);
    }
  } else if (value != NULL) {
    error = cpl_propertylist_append_string(self->proplist, propname, value);
    if (! error) {
      error = cpl_propertylist_set_comment(self->proplist, propname,
                                           pcomment);
      if (error) {
        /* Delete entry if we could not set the comment to maintain a
         * consistent state */
        cpl_errorstate prestate = cpl_errorstate_get();
        (void) cpl_propertylist_erase(self->proplist, propname);
        cpl_errorstate_set(prestate);
      }
    }
  }
  cpl_free(propname);
  cpl_free(pcomment);
  return error;
}


static void
_espdr_sdp_spectrum_erase_column_keywords(espdr_sdp_spectrum *self,
                                           const char *name)
{
  cpl_size index;

  assert(self != NULL);
  assert(self->proplist != NULL);
  assert(name != NULL);

  index = _espdr_sdp_spectrum_get_column_index(self, name);
  if (index != (cpl_size)-1) {
    char *propname = cpl_sprintf("%s%"CPL_SIZE_FORMAT, KEY_TUTYP, index+1);
    cpl_propertylist_erase(self->proplist, propname);
    cpl_free(propname);
    propname = cpl_sprintf("%s%"CPL_SIZE_FORMAT, KEY_TUCD, index+1);
    cpl_propertylist_erase(self->proplist, propname);
    cpl_free(propname);
    propname = cpl_sprintf("%s%"CPL_SIZE_FORMAT, KEY_TCOMM, index+1);
    cpl_propertylist_erase(self->proplist, propname);
    cpl_free(propname);
  }
}

/**
 * @internal
 * @brief Calculates a value for MJD-END from the list of used frames.
 *
 * The MJD_END value is the taken as the highest MJD-OBS value found in the
 * product frame of list of used frames, plus the exposure time for the last
 * frame.
 *
 * @param product_frame  The product frame to check.
 * @param used_frames  The list of used frames to check.
 * @param arm  The instrument arm being used.
 * @return A value to use for MJD-END or NAN if an error occurred. If an error
 *      did occur then an error code will be set that can be checked with
 *      @c cpl_error_get_code.
 */
static double
espdr_calculate_mjd_end(const cpl_frame *product_frame,
                                    const cpl_frameset *used_frames)
{
  cpl_propertylist* keywords = NULL;
  cpl_frameset_iterator* iter = NULL;
  double mjdend = NAN;

  /* Calculate MJD-END by finding the largest MJD-OBS value from the raw input
   * files. */
  const char* filename = cpl_frame_get_filename(product_frame);
  keywords = cpl_propertylist_load(filename, 0);
  mjdend = cpl_propertylist_get_double(keywords, ESPDR_MJDOBS);

  iter = cpl_frameset_iterator_new(used_frames);
  const cpl_frame* frame = cpl_frameset_iterator_get_const(iter);
  while (frame != NULL) {
    if (cpl_frame_get_group(frame) == CPL_FRAME_GROUP_RAW) {
      /* Load the primary header keywords from the raw frame. */
      filename = cpl_frame_get_filename(frame);
      cpl_propertylist_delete(keywords);
      keywords = cpl_propertylist_load(filename, 0);
      /* Extract the MJD-OBS and EXPTIME keywords and update the mjdend
       * candidate. */
      double obsval=0, expval=0, total=0;
      double factor=1./(86400.);
      obsval = cpl_propertylist_get_double(keywords, ESPDR_MJDOBS);

      expval = cpl_propertylist_get_double(keywords, ESPDR_EXPTIME);

      total = obsval + expval * factor;
      if (total > mjdend) mjdend = total;

    }
    /* Increment the iterator to the next frame. */
    cpl_errorstate status = cpl_errorstate_get();
    cpl_frameset_iterator_advance(iter, 1);
    if (cpl_error_get_code() == CPL_ERROR_ACCESS_OUT_OF_RANGE) {
        cpl_errorstate_set(status);
    }
    frame = cpl_frameset_iterator_get_const(iter);
  }



  cpl_frameset_iterator_delete(iter);
  cpl_propertylist_delete(keywords);
  return mjdend;
}

const char *
espdr_sdp_spectrum_get_column_tutyp(const espdr_sdp_spectrum *self,
                                     const char *name)
{
  cpl_errorstate prestate = cpl_errorstate_get();
  const char *result;
  cpl_ensure(self != NULL && name != NULL, CPL_ERROR_NULL_INPUT, NULL);
  result = _espdr_sdp_spectrum_get_column_keyword(self, name, KEY_TUTYP);
  if (! cpl_errorstate_is_equal(prestate)) {
    cpl_error_set_where(cpl_func);
  }
  return result;
}


cpl_error_code
espdr_sdp_spectrum_set_column_tutyp(espdr_sdp_spectrum *self,
                                     const char *name, const char *tutyp)
{
  cpl_error_code error;
  cpl_ensure_code(self != NULL && name != NULL, CPL_ERROR_NULL_INPUT);
  error = _espdr_sdp_spectrum_set_column_keyword(self, name, tutyp,
                                                  KEY_TUTYP, KEY_TUTYP_COMMENT);
  if (error) {
    cpl_error_set_where(cpl_func);
  }
  return error;
}


cpl_error_code
espdr_sdp_spectrum_copy_column_tutyp(espdr_sdp_spectrum *self,
                                      const char *name,
                                      const cpl_propertylist *plist,
                                      const char *key)
{
  cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

  assert(self->table != NULL);

  if (cpl_propertylist_has(plist, key)) {
    cpl_errorstate prestate = cpl_errorstate_get();
    const char *value = cpl_propertylist_get_string(plist, key);
    if (cpl_errorstate_is_equal(prestate)) {
      return espdr_sdp_spectrum_set_column_tutyp(self, name, value);
    } else {
      cpl_size index = _espdr_sdp_spectrum_get_column_index(self, name) + 1;
      return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                "Could not set '%s%"CPL_SIZE_FORMAT"' for column '%s'. Likely"
                " the source '%s' keyword is not a string.",
                KEY_TUTYP, index, name, key);
    }
  } else {
    cpl_size index = _espdr_sdp_spectrum_get_column_index(self, name) + 1;
    return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                "Could not set '%s%"CPL_SIZE_FORMAT"' for column '%s' since the"
                " '%s' keyword was not found.", KEY_TUTYP, index, name, key);
  }
}


const char *
espdr_sdp_spectrum_get_column_tucd(const espdr_sdp_spectrum *self,
                                    const char *name)
{
  cpl_errorstate prestate = cpl_errorstate_get();
  const char *result;
  cpl_ensure(self != NULL && name != NULL, CPL_ERROR_NULL_INPUT, NULL);
  result = _espdr_sdp_spectrum_get_column_keyword(self, name, KEY_TUCD);
  if (! cpl_errorstate_is_equal(prestate)) {
    cpl_error_set_where(cpl_func);
  }
  return result;
}


cpl_error_code
espdr_sdp_spectrum_set_column_tucd(espdr_sdp_spectrum *self,
                                    const char *name, const char *tucd)
{
  cpl_error_code error;
  cpl_ensure_code(self != NULL && name != NULL, CPL_ERROR_NULL_INPUT);
  error = _espdr_sdp_spectrum_set_column_keyword(self, name, tucd,
                                                  KEY_TUCD, KEY_TUCD_COMMENT);
  if (error) {
    cpl_error_set_where(cpl_func);
  }
  return error;
}

cpl_error_code
espdr_sdp_spectrum_set_column_tucd_el(espdr_sdp_spectrum *self,
                                    const char *name, const char *tucd)
{
  cpl_error_code error;
  cpl_ensure_code(self != NULL && name != NULL, CPL_ERROR_NULL_INPUT);
  error = _espdr_sdp_spectrum_set_column_keyword(self, name, tucd,
                                                  KEY_TUCD, KEY_TUCD_EL_COMMENT);
  if (error) {
    cpl_error_set_where(cpl_func);
  }
  return error;
}


cpl_error_code
espdr_sdp_spectrum_copy_column_tucd(espdr_sdp_spectrum *self,
                                     const char *name,
                                     const cpl_propertylist *plist,
                                     const char *key)
{
  cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

  assert(self->table != NULL);

  if (cpl_propertylist_has(plist, key)) {
    cpl_errorstate prestate = cpl_errorstate_get();
    const char *value = cpl_propertylist_get_string(plist, key);
    if (cpl_errorstate_is_equal(prestate)) {
      return espdr_sdp_spectrum_set_column_tucd(self, name, value);
    } else {
      cpl_size index = _espdr_sdp_spectrum_get_column_index(self, name) + 1;
      return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                "Could not set '%s%"CPL_SIZE_FORMAT"' for column '%s'. Likely"
                " the source '%s' keyword is not a string.",
                KEY_TUCD, index, name, key);
    }
  } else {
    cpl_size index = _espdr_sdp_spectrum_get_column_index(self, name) + 1;
    return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                "Could not set '%s%"CPL_SIZE_FORMAT"' for column '%s' since the"
                " '%s' keyword was not found.", KEY_TUCD, index, name, key);
  }
}


const char *
espdr_sdp_spectrum_get_column_tcomm(const espdr_sdp_spectrum *self,
                                     const char *name)
{
  cpl_errorstate prestate = cpl_errorstate_get();
  const char *result;
  cpl_ensure(self != NULL && name != NULL, CPL_ERROR_NULL_INPUT, NULL);
  result = _espdr_sdp_spectrum_get_column_keyword(self, name, KEY_TCOMM);
  if (! cpl_errorstate_is_equal(prestate)) {
    cpl_error_set_where(cpl_func);
  }
  return result;
}


cpl_error_code
espdr_sdp_spectrum_set_column_tcomm(espdr_sdp_spectrum *self,
                                     const char *name, const char *tcomm)
{
  cpl_error_code error;
  cpl_ensure_code(self != NULL && name != NULL, CPL_ERROR_NULL_INPUT);
  error = _espdr_sdp_spectrum_set_column_keyword(self, name, tcomm,
                                                  KEY_TCOMM, KEY_TCOMM_COMMENT);
  if (error) {
    cpl_error_set_where(cpl_func);
  }
  return error;
}


cpl_error_code
espdr_sdp_spectrum_copy_column_tcomm(espdr_sdp_spectrum *self,
                                      const char *name,
                                      const cpl_propertylist *plist,
                                      const char *key)
{
  cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

  assert(self->table != NULL);

  if (cpl_propertylist_has(plist, key)) {
    cpl_errorstate prestate = cpl_errorstate_get();
    const char *value = cpl_propertylist_get_string(plist, key);
    if (cpl_errorstate_is_equal(prestate)) {
      return espdr_sdp_spectrum_set_column_tcomm(self, name, value);
    } else {
      cpl_size index = _espdr_sdp_spectrum_get_column_index(self, name) + 1;
      return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                "Could not set '%s%"CPL_SIZE_FORMAT"' for column '%s'. Likely"
                " the source '%s' keyword is not a string.",
                KEY_TCOMM, index, name, key);
    }
  } else {
    cpl_size index = _espdr_sdp_spectrum_get_column_index(self, name) + 1;
    return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                "Could not set '%s%"CPL_SIZE_FORMAT"' for column '%s' since the"
                " '%s' keyword was not found.", KEY_TCOMM, index, name, key);
  }
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Replace the comment of a column description keyword.
 * @param self    The spectrum object to update.
 * @param name    The name of the column to update.
 * @param keyword The name of the keyword to update.
 * @param comment The string to use as keyword comment.
 * @return @c CPL_ERROR_NONE on success or an appropriate error code otherwise.
 *
 * This function will replace the comment of the keyword @c keyword of
 * the column @c name with the string @c comment. The string @c keyword
 * is the keyword name without the column index appended. The latter is
 * deduced from the column name @name. The new comment @c comment must not
 * be @a NULL.
 * If any error occurs then a error code is set and returned, otherwise
 * @c CPL_ERROR_NONE is returned on success.
 */
/*----------------------------------------------------------------------------*/

cpl_error_code
espdr_sdp_spectrum_replace_column_comment(espdr_sdp_spectrum *self,
                                           const char *name,
                                           const char *keyword,
                                           const char *comment)
{
    cpl_ensure_code((self != NULL), CPL_ERROR_NULL_INPUT);
    cpl_ensure_code((self->proplist != NULL), CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code((name != NULL) && (keyword != NULL) && (comment != NULL),
                    CPL_ERROR_NULL_INPUT);

    cpl_size index = _espdr_sdp_spectrum_get_column_index(self, name);
    if (index == (cpl_size)-1) {
      return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                  "Could not find column '%s'.", name);
    }

    char *propname = cpl_sprintf("%s%"CPL_SIZE_FORMAT, keyword, index + 1);
    if (!cpl_propertylist_has(self->proplist, propname)) {
        cpl_free(propname);
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                  "Could not find '%s' keyword for column '%s'.", keyword,
                  name);
    }

    cpl_propertylist_set_comment(self->proplist, propname, comment);
    cpl_free(propname);

    return CPL_ERROR_NONE;
}


const cpl_array *
espdr_sdp_spectrum_get_column_data(const espdr_sdp_spectrum *self,
                                    const char *name)
{
  cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
  assert(self->table != NULL);
  return cpl_table_get_array(self->table, name, 0);
}


cpl_error_code
espdr_sdp_spectrum_set_column_data(espdr_sdp_spectrum *self,
                                    const char *name, const cpl_array *array)
{
  cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
  assert(self->table != NULL);
  return cpl_table_set_array(self->table, name, 0, array);
}


static char * _espdr_make_regexp(const cpl_propertylist *plist,
                                  const char *extra)
{
  /* Minimum number of characters required for possible "^(", "|" or ")$"
   * fragments and a null character to end the string. */
  static const cpl_size min_chars_required = 6;

  /* Start, end and join fragments for the regular expression, to form a string
   * of the following form: "^(KEY1|KEY2 .. |KEYN)$" */
  static const char *start_fragment = "^(";
  static const char *end_fragment = ")$";
  static const char *join_fragment = "|";

  cpl_size extra_length = (extra != NULL ? (cpl_size) strlen(extra) : 0);
  cpl_size regexp_size, bytesleft, nkeys, i;
  char *writepos;
  char *regexp = NULL;

  assert(plist != NULL);

  nkeys = cpl_propertylist_get_size(plist);
  if (nkeys == 0) {
    /* Handle special case where plist is empty. */
    if (extra != NULL) {
      return cpl_sprintf("%s%s%s", start_fragment, extra, end_fragment);
    } else {
      return cpl_strdup("");
    }
  }

  /* Allocate enough space to store the regexp. 80 = FITS card width. */
  regexp_size = nkeys * 80 + min_chars_required + extra_length;
  regexp = cpl_malloc(regexp_size);

  bytesleft = regexp_size;
  writepos = regexp;
  for (i = 0; i < nkeys; ++i) {
    cpl_size name_length, fragment_length;
    const char *name, *fragment;

    /* Fetch the property name string. */
    const cpl_property *p = cpl_propertylist_get_const(plist, i);
    cpl_error_ensure(p != NULL, cpl_error_get_code(), goto cleanup,
      "Unexpected error accessing property structure %"CPL_SIZE_FORMAT".", i);
    name = cpl_property_get_name(p);
    cpl_error_ensure(name != NULL, cpl_error_get_code(), goto cleanup,
      "Unexpected error accessing the name of property %"CPL_SIZE_FORMAT".", i);
    name_length = (cpl_size) strlen(name);

    /* Figure out the regexp start/join string fragment to use. */
    fragment = (i == 0) ? start_fragment : join_fragment;
    fragment_length = (cpl_size) strlen(fragment);

    while (bytesleft <
           fragment_length + name_length + extra_length + min_chars_required)
    {
      /* Allocate more space if we still run out of space for the regexp.
       * Note: the realloc either succeeds or aborts on failure. */
      bytesleft += regexp_size;
      regexp_size += regexp_size;
      regexp = cpl_realloc(regexp, regexp_size);
      writepos = regexp + (regexp_size - bytesleft);
    }

    /* Write the start/join fragment and then the key name strings. */
    strncpy(writepos, fragment, bytesleft);
    bytesleft -= fragment_length;
    writepos += fragment_length;
    strncpy(writepos, name, bytesleft);
    bytesleft -= name_length;
    writepos += name_length;
  }

  /* Write the extra string and end fragment string to complete the regexp. */
  if (extra != NULL) {
    strncpy(writepos, join_fragment, bytesleft);
    bytesleft -= (cpl_size) strlen(join_fragment);
    writepos += (cpl_size) strlen(join_fragment);
    strncpy(writepos, extra, bytesleft);
    bytesleft -= extra_length;
    writepos += extra_length;
  }
  strncpy(writepos, end_fragment, bytesleft);
  /* Null terminate the string buffer for safety. */
  regexp[regexp_size-1] = '\0';

  return regexp;

cleanup:
  /* Cleanup in case of error: */
  cpl_free(regexp);
  return NULL;
}

/*
static cpl_error_code
espdr_sdp_spectrum_set_column_data_double(espdr_sdp_spectrum *self,
		double* data, const cpl_size size, const char *name)
{

	cpl_array* array = cpl_array_wrap_double(data, size);
	cpl_table_new_column_array(self->table, name, CPL_TYPE_DOUBLE,size);
	espdr_sdp_spectrum_set_column_data(self, name, array);
	cpl_array_unwrap(array);

	return cpl_error_get_code();
}
*/

static cpl_error_code
espdr_sdp_spectrum_set_column_data_int(espdr_sdp_spectrum *self,
		int* data, const cpl_size size, const char *name)
{

	cpl_array* array = cpl_array_wrap_int(data, size);
	cpl_table_new_column_array(self->table, name, CPL_TYPE_INT,size);
	espdr_sdp_spectrum_set_column_data(self, name, array);
	cpl_array_unwrap(array);

	return cpl_error_get_code();
}
/**
 * @internal
 * @brief Fills SDP spectrum object with PROVi keywords.
 *
 * Fills the spectrum with PROVi keywords corresponding to the raw science data
 * files used for a final SDP product. The PROVi keyword is filled with the
 * content of ARCFILE if this keyword is found in the raw file primary header.
 * Otherwise ORIGFILE is used if ARCFILE is missing. If none of these keywords
 * is found then just the disk file name is used.
 *
 * @param[out] spectrum  The SDP spectrum object to fill with the new keywords.
 * @param[in] used_frames  The list of used frames to add as PROV keywords.
 *
 * If an error occurs then an appropriate error code is set. The caller can
 * check for this with a call to @c cpl_error_get_code().
 */
static void
espdr_fill_provenance_keywords(espdr_sdp_spectrum* spectrum,
                                         const cpl_frameset* used_frames)
{
  cpl_propertylist* keywords = NULL;
  cpl_frameset_iterator* iter = NULL;
  int ncombine = 0;

  iter = cpl_frameset_iterator_new(used_frames);
  const cpl_frame* frame = cpl_frameset_iterator_get_const(iter);
  while (frame != NULL) {
    if (cpl_frame_get_group(frame) == CPL_FRAME_GROUP_RAW) {
      /* Load the keywords from the raw frame. */
      const char* filename;
      filename = cpl_frame_get_filename(frame);
      keywords = cpl_propertylist_load(filename, 0);
      /* Try set the prov_value to ARCFILE or ORIGFILE or just the filename,
       * whichever is found first in that order. */
      const char* prov_value = NULL;
      if (cpl_propertylist_has(keywords, ESPDR_ARCFILE)) {
        prov_value = cpl_propertylist_get_string(keywords, ESPDR_ARCFILE);
      } else if (cpl_propertylist_has(keywords, ESPDR_ORIGFILE)) {
        prov_value = cpl_propertylist_get_string(keywords, ESPDR_ORIGFILE);
      } else {
        prov_value = filename;
      }
      /* Add the next PROVi keyword. */
      ++ncombine;
      espdr_sdp_spectrum_set_prov(spectrum, ncombine, prov_value);
      cpl_propertylist_delete(keywords);
      keywords = NULL;
    }
    /* Increment the iterator to the next frame. */
    cpl_errorstate status = cpl_errorstate_get();
    cpl_frameset_iterator_advance(iter, 1);
    if (cpl_error_get_code() == CPL_ERROR_ACCESS_OUT_OF_RANGE) {
        cpl_errorstate_set(status);
    }
    frame = cpl_frameset_iterator_get_const(iter);
  }

  /* Finally add the NCOMBINE keyword. */
  espdr_sdp_spectrum_set_ncombine(spectrum, ncombine);


  cpl_frameset_iterator_delete(iter);
  cpl_propertylist_delete(keywords);
}

/**
 * @internal
 * @brief Calculates a value for EXPTIME from the list of used frames.
 *
 * The EXPTIME value is simply the sum of EXPTIME values from individual frames.
 *
 * @param used_frames  The list of used frames to check.
 * @param arm  The instrument arm being used.
 * @return A value to use for EXPTIME or NAN it could not be computed. If an
 *      error occurred in addition then an error code will be set that can be
 *      checked with @c cpl_error_get_code.
 */
static double espdr_calculate_exptime(const cpl_frameset *used_frames)
{
  cpl_propertylist* keywords = NULL;
  cpl_frameset_iterator* iter = NULL;
  double total = 0;

  iter = cpl_frameset_iterator_new(used_frames);
  const cpl_frame* frame = cpl_frameset_iterator_get_const(iter);
  while (frame != NULL) {
    if (cpl_frame_get_group(frame) == CPL_FRAME_GROUP_RAW) {
      /* Load the primary header keywords from the raw frame. */
      const char* filename;
      filename = cpl_frame_get_filename(frame);
      keywords = cpl_propertylist_load(filename, 0);
      /* Extract the EXPTIME keyword and add it to the sum. */
      double exptime = cpl_propertylist_get_double(keywords, ESPDR_EXPTIME);

      cpl_propertylist_delete(keywords);
      keywords = NULL;
      total += exptime;
    }
    /* Increment the iterator to the next frame. */
    cpl_errorstate status = cpl_errorstate_get();
    cpl_frameset_iterator_advance(iter, 1);
    if (cpl_error_get_code() == CPL_ERROR_ACCESS_OUT_OF_RANGE) {
        cpl_errorstate_set(status);
    }
    frame = cpl_frameset_iterator_get_const(iter);
  }

  cpl_frameset_iterator_delete(iter);
  cpl_propertylist_delete(keywords);
  return total;


  /* Enter this cleanup section if an error occurs. */
  cpl_frameset_iterator_delete(iter);
  cpl_propertylist_delete(keywords);
  return NAN;
}


espdr_sdp_spectrum * espdr_sdp_spectrum_load(const char *filename)
{
  cpl_error_code error;
  espdr_sdp_spectrum *obj;
  cpl_propertylist *plist = NULL;
  cpl_propertylist *tmpplist = NULL;
  cpl_table *table = NULL;
  cpl_array *names = NULL;
  cpl_array *emptyarray = NULL;
  cpl_size nelem, ext, i;
  char *regexp = NULL;

  cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);

  /* Load the property list from file, making sure the properties from the
   * primary HDU take precedence over those from the extension if any keywords
   * are duplicated. Note, we only load keywords known to the spectrum class. */
  plist = cpl_propertylist_load_regexp(filename, 0, ALL_KEYS_REGEXP, 0);
  cpl_error_ensure(plist != NULL, cpl_error_get_code(), goto cleanup,
      "Could not load property list from primary HDU when loading file '%s'.",
      filename);

  /* We have to create a regexp to filter out keywords already loaded from the
   * primary HDU. */
  regexp = _espdr_make_regexp(plist, NULL);
  cpl_error_ensure(regexp != NULL, cpl_error_get_code(), goto cleanup,
                   "Could not create regular expression to filter keywords.");

  /* Try find the spectrum extension from which to load the table. If the
   * extension name cannot be found then just use the first extension. */
  ext = cpl_fits_find_extension(filename, KEY_EXTNAME_VALUE);
  cpl_error_ensure(ext != (cpl_size)-1, cpl_error_get_code(), goto cleanup,
                   "Failed to get the extension '%s' from file '%s'.",
                   KEY_EXTNAME_VALUE, filename);
  if (ext == 0) ext = 1;

  /* Load only the SDP keywords from the extension. */
  tmpplist = cpl_propertylist_load_regexp(filename, ext, ALL_KEYS_REGEXP, 0);
  cpl_error_ensure(tmpplist != NULL, cpl_error_get_code(), goto cleanup,
                   "Could not load property list from extension %"
                   CPL_SIZE_FORMAT" when loading file '%s'.", ext, filename);

  /* Append keywords to plist that are not already in plist. */
  error = cpl_propertylist_copy_property_regexp(plist, tmpplist, regexp, 1);
  cpl_error_ensure(! error, error, goto cleanup,
                   "Failed to append keywords from file '%s' extension %"
                   CPL_SIZE_FORMAT".", filename, ext);

  /* Delete temporary objects that are no longer needed. */
  cpl_propertylist_delete(tmpplist);
  tmpplist = NULL;
  cpl_free(regexp);
  regexp = NULL;

  table = cpl_table_load(filename, (int)ext, CPL_TRUE);
  cpl_error_ensure(table != NULL, cpl_error_get_code(), goto cleanup,
                   "Could not load the spectrum table from extension %"
                   CPL_SIZE_FORMAT" when loading file '%s'.", ext, filename);

  /* Set the nelem value from the NELEM keyword if found, else work it out. */
  if (cpl_propertylist_has(plist, KEY_NELEM)) {
    cpl_errorstate prestate = cpl_errorstate_get();
    nelem = (cpl_size) cpl_propertylist_get_long_long(plist, KEY_NELEM);
    /* Remove NELEM since the value is instead stored in the nelem variable. */
    cpl_propertylist_erase(plist, KEY_NELEM);
    cpl_error_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(),
                  goto cleanup, "Could not process the temporary '%s' keyword.",
                  KEY_NELEM);
  } else {
    cpl_msg_warning(cpl_func,
                    "Keyword '%s' not found in file '%s'. Possibly corrupted."
                    " Will try find correct value from the table and continue.",
                    KEY_NELEM, filename);
    nelem = 0;
    if (cpl_table_get_nrow(table) > 0) {
      names = cpl_table_get_column_names(table);
      if (names != NULL) {
        if (cpl_array_get_size(names) > 0) {
          const char *name = cpl_array_get_string(names, 0);
          nelem = cpl_table_get_column_depth(table, name);
        }
        cpl_array_delete(names);
        names = NULL;
      }
    }
  }

  names = cpl_table_get_column_names(table);
  cpl_error_ensure(names != NULL, cpl_error_get_code(), goto cleanup,
          "Could not get table column names when loading file '%s'.", filename);
  for (i = 0; i < cpl_array_get_size(names); ++i) {
    int j;
    const char *name = cpl_array_get_string(names, 0);
    cpl_type type = cpl_table_get_column_type(table, name);
    if ((type & CPL_TYPE_POINTER) == 0) continue;  /* Only handle array columns.*/
    for (j = 0; j < cpl_table_get_nrow(table); ++j) {
      if (cpl_table_get_array(table, name, j) != NULL) continue;
      emptyarray = cpl_array_new(nelem, type & (~CPL_TYPE_POINTER));
      cpl_error_ensure(emptyarray != NULL, cpl_error_get_code(), goto cleanup,
            "Could not create empty array when spectrum table from file '%s'.",
            filename);
      error = cpl_table_set_array(table, name, j, emptyarray);
      cpl_array_delete(emptyarray);
      emptyarray = NULL;
    }
  }
  cpl_array_delete(names);

  /* Create new spectrum instance and return it. */
  obj = cpl_malloc(sizeof(espdr_sdp_spectrum));
  obj->nelem = nelem;
  obj->proplist = plist;
  obj->table = table;
  return obj;

cleanup:
  /* Perform memory cleanup if an error occurred. The cpl_error_ensure macros
   * will send the control flow to this point when an error is detected.
   * Note: cpl_*_delete functions already check for NULL pointers. */
  cpl_propertylist_delete(plist);
  cpl_propertylist_delete(tmpplist);
  cpl_table_delete(table);
  cpl_array_delete(names);
  cpl_array_delete(emptyarray);
  cpl_free(regexp);
  return NULL;
}


cpl_error_code espdr_sdp_spectrum_save(const espdr_sdp_spectrum *self,
                                        const char *filename,
                                        const cpl_propertylist *extra_pheader,
                                        const cpl_propertylist *extra_theader)
{
  cpl_error_code error;
  cpl_propertylist *primarykeys = NULL;
  cpl_propertylist *tablekeys = NULL;
  char *regexp = NULL;

  cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

  assert(self->proplist != NULL);
  assert(self->table != NULL);

  /* Make a regular expression to filter out all keywords found in the spectrum
   * object's proplist and NELEM from the extra header keywords. */
  regexp = _espdr_make_regexp(self->proplist, KEY_NELEM);
  cpl_error_ensure(regexp != NULL, cpl_error_get_code(), goto cleanup,
                   "Could not create regular expression to filter keywords.");

  /* Copy out keywords that should be in the primary HDU header from the full
   * list of keywords in proplist. */
  primarykeys = cpl_propertylist_new();
  error = cpl_propertylist_copy_property_regexp(primarykeys, self->proplist,
                                                PRIMARY_HDU_KEYS_REGEXP, 0);
  cpl_error_ensure(! error, error, goto cleanup,
                   "Failed to extract keywords for primary HDU.");

  /* Use a different comment name for OBJECT in the primary HDU to more closely
   * follow standard document. */
  if (cpl_propertylist_has(primarykeys, KEY_OBJECT)) {
    error = cpl_propertylist_set_comment(primarykeys, KEY_OBJECT,
                                         KEY_OBJECT_PHDU_COMMENT);
    cpl_error_ensure(! error, error, goto cleanup,
              "Could not update comment for '%s' in primary HDU.", KEY_OBJECT);
  }

  /* Copy any extra keywords that are not already in the primary HDU header. */
  if (extra_pheader != NULL) {
    error = cpl_propertylist_copy_property_regexp(primarykeys, extra_pheader,
                                                  regexp, 1);
    cpl_error_ensure(! error, error, goto cleanup,
                     "Could not add extra keywords for primary HDU.");
  }

  /* Copy out keywords for the table header from all in proplist. */
  tablekeys = cpl_propertylist_new();
  error = cpl_propertylist_copy_property_regexp(tablekeys, self->proplist,
                                                EXTENSION_HDU_KEYS_REGEXP, 0);
  cpl_error_ensure(! error, error, goto cleanup,
                   "Failed to extract keywords for extension HDU.");

  /* Add the NELEM keyword from the nelem variable. */
  cpl_error_ensure(self->nelem <= INT_MAX, CPL_ERROR_INCOMPATIBLE_INPUT,
                   goto cleanup,
                   "The value for the keyword '%s' is too big (> %d).",
                   KEY_NELEM, INT_MAX);
  error = cpl_propertylist_append_int(tablekeys, KEY_NELEM,
                                      (int) self->nelem);
  error |= cpl_propertylist_set_comment(tablekeys, KEY_NELEM,
                                        KEY_NELEM_COMMENT);
  cpl_error_ensure(! error, error, goto cleanup,
              "Could not add keyword '%s' to primary HDU or set the comment.",
              KEY_NELEM);

  /* Copy extra keywords that are not already in the extension HDU header. */
  if (extra_theader != NULL) {
    error = cpl_propertylist_copy_property_regexp(tablekeys, extra_theader,
                                                  regexp, 1);
    cpl_error_ensure(! error, error, goto cleanup,
                     "Could not add extra keywords for extension HDU.");
  }

  cpl_free(regexp);
  regexp = NULL;

  /* Add some mandatory keywords with default values that are still not found
   * in the primary or extension property lists, since they were not set in the
   * spectrum or in the extra header lists. */
  error = CPL_ERROR_NONE;
  if (! cpl_propertylist_has(primarykeys, KEY_ORIGIN)) {
    error |= cpl_propertylist_append_string(primarykeys, KEY_ORIGIN,
                                            KEY_ORIGIN_VALUE);
    error |= cpl_propertylist_set_comment(primarykeys, KEY_ORIGIN,
                                          KEY_ORIGIN_COMMENT);
  }
  if (! cpl_propertylist_has(primarykeys, KEY_PRODLVL)) {
    error |= cpl_propertylist_append_int(primarykeys, KEY_PRODLVL,
                                         KEY_PRODLVL_VALUE);
    error |= cpl_propertylist_set_comment(primarykeys, KEY_PRODLVL,
                                          KEY_PRODLVL_COMMENT);
  }
  if (! cpl_propertylist_has(primarykeys, KEY_SPECSYS)) {
    error |= cpl_propertylist_append_string(primarykeys, KEY_SPECSYS,
                                            KEY_SPECSYS_VALUE);
    error |= cpl_propertylist_set_comment(primarykeys, KEY_SPECSYS,
                                          KEY_SPECSYS_COMMENT);
  }
  if (! cpl_propertylist_has(primarykeys, KEY_FLUXERR)) {
    error |= cpl_propertylist_append_double(primarykeys, KEY_FLUXERR,
                                         KEY_FLUXERR_VALUE);
    error |= cpl_propertylist_set_comment(primarykeys, KEY_FLUXERR,
                                          KEY_FLUXERR_COMMENT);
  }
  if (! cpl_propertylist_has(tablekeys, KEY_VOCLASS)) {
    error |= cpl_propertylist_append_string(tablekeys, KEY_VOCLASS,
                                            KEY_VOCLASS_VALUE);
    error |= cpl_propertylist_set_comment(tablekeys, KEY_VOCLASS,
                                          KEY_VOCLASS_COMMENT);
  }
  if (! cpl_propertylist_has(tablekeys, KEY_VOPUB)) {
    error |= cpl_propertylist_append_string(tablekeys, KEY_VOPUB,
                                            KEY_VOPUB_VALUE);
    error |= cpl_propertylist_set_comment(tablekeys, KEY_VOPUB,
                                          KEY_VOPUB_COMMENT);
  }
  if (! cpl_propertylist_has(tablekeys, KEY_EXTNAME)) {
    error |= cpl_propertylist_append_string(tablekeys, KEY_EXTNAME,
                                            KEY_EXTNAME_VALUE);
    error |= cpl_propertylist_set_comment(tablekeys, KEY_EXTNAME,
                                          KEY_EXTNAME_COMMENT);
  }
  if (! cpl_propertylist_has(tablekeys, KEY_INHERIT)) {
    error |= cpl_propertylist_append_bool(tablekeys, KEY_INHERIT,
                                          KEY_INHERIT_VALUE);
    error |= cpl_propertylist_set_comment(tablekeys, KEY_INHERIT,
                                          KEY_INHERIT_COMMENT);
  }
  cpl_error_ensure(! error, cpl_error_get_code(), goto cleanup,
                   "Could not set default header keywords for file '%s'.",
                   filename);

  error = cpl_table_save(self->table, primarykeys, tablekeys, filename,
                         CPL_IO_CREATE);
  cpl_error_ensure(! error, error, goto cleanup,
                   "Could not save the spectrum table to file '%s'.", filename);

  cpl_propertylist_delete(primarykeys);
  cpl_propertylist_delete(tablekeys);

  return CPL_ERROR_NONE;

cleanup:
  /* Cleanup memory if an error occurred. Note: cpl_*_delete functions already
   * check for NULL pointers. */
  cpl_propertylist_delete(primarykeys);
  cpl_propertylist_delete(tablekeys);
  cpl_free(regexp);
  return cpl_error_get_code();
}


static cpl_error_code
espdr_plist_purge_list(cpl_propertylist* purge, const cpl_propertylist* pref)
{
	 const cpl_size sz1 = cpl_propertylist_get_size(purge);
	 const cpl_size sz2 = cpl_propertylist_get_size(pref);
	 const char* pname = NULL;
	 for (cpl_size j = 0; j < sz2; j++) {
		 const cpl_property *p1 = cpl_propertylist_get_const(pref, j);
		 pname = cpl_property_get_name(p1);
		 if(cpl_propertylist_has(purge,pname)) {
		    cpl_propertylist_erase(purge, pname);
		 }
	 }
     //cpl_propertylist_dump(purge,stdout);
	 return cpl_error_get_code();
}


static cpl_error_code
espdr_plist_overwrite_list(cpl_propertylist* copy, const cpl_propertylist* pref)
{
	 const cpl_size sz1 = cpl_propertylist_get_size(copy);
	 const cpl_size sz2 = cpl_propertylist_get_size(pref);
	 const char* pname = NULL;
	 const cpl_property *p2 = NULL;

	 for (cpl_size j = 0; j < sz2; j++) {
		 p2 = cpl_propertylist_get_const(pref, j);
		 pname = cpl_property_get_name(p2);
		 if(cpl_propertylist_has(copy,pname)) {
			 cpl_propertylist_copy_property(copy,  pref, pname);
		 } else {
			 cpl_propertylist_append_property(copy,  p2);
		 }
	 }
     //cpl_propertylist_dump(pref,stdout);
	 return cpl_error_get_code();
}





cpl_error_code espdr_dfs_save_spectrum(cpl_frameset              * allframes,
                                        cpl_propertylist          * header,
                                        const cpl_parameterlist   * parlist,
                                        const cpl_frameset        * used_frames,
                                        const cpl_frame           * inherit,
                                        const espdr_sdp_spectrum * spectrum,
                                        const char                * recipe,
                                        const cpl_propertylist    * applist,
                                        const cpl_propertylist    * tablelist,
                                        const char                * remregexp,
                                        const char                * pipe_id,
                                        const char                * dict_id,
                                        const char                * filename,
										const cpl_boolean trim_keys)
{
  const char       * procat;
  cpl_propertylist * plist = NULL;
  cpl_frame        * product_frame = NULL;
  cpl_error_code     error;

  cpl_ensure_code(allframes  != NULL, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(parlist    != NULL, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(used_frames != NULL, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(spectrum   != NULL, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(recipe     != NULL, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(applist    != NULL, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(pipe_id    != NULL, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(dict_id    != NULL, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(filename   != NULL, CPL_ERROR_NULL_INPUT);

  procat = cpl_propertylist_get_string(applist, CPL_DFS_PRO_CATG);
  cpl_error_ensure(procat != NULL, cpl_error_get_code(), goto cleanup,
              "Could not find keyword '%s' in 'applist'.", CPL_DFS_PRO_CATG);

  /* Create product frame */
  product_frame = cpl_frame_new();
  error = cpl_frame_set_filename(product_frame, filename);
  error |= cpl_frame_set_tag(product_frame, procat);
  error |= cpl_frame_set_type(product_frame, CPL_FRAME_TYPE_TABLE);
  error |= cpl_frame_set_group(product_frame, CPL_FRAME_GROUP_PRODUCT);
  error |= cpl_frame_set_level(product_frame, CPL_FRAME_LEVEL_FINAL);
  cpl_error_ensure(! error, cpl_error_get_code(), goto cleanup,
                   "Failed to setup the product frame.");

  /* Check if we should return the header information actually filled or just
   * create a temporary local list. */
  if (header != NULL) {
    cpl_propertylist_empty(header);
    plist = header;
  } else {
    plist = cpl_propertylist_new();
  }

  /* Add any QC parameters here. */
  error = cpl_propertylist_append(plist, applist);
  cpl_error_ensure(! error, error, goto cleanup,
          "Could not append extra keywords when writing file '%s'.", filename);

  /* Add DataFlow keywords. */
  error = cpl_dfs_setup_product_header(plist, product_frame, used_frames,
                                       parlist, recipe, pipe_id, dict_id,
                                       inherit);
  cpl_error_ensure(! error, error, goto cleanup,
          "Failed to setup DFS keywords when writing file '%s'.", filename);

  if(trim_keys) {
          /* for NIRPS IDPs */
	  cpl_frame* hframe = cpl_frameset_find(allframes, "HEADER");
	  const char* fname = cpl_frame_get_filename(hframe);
	  cpl_propertylist* plist_ref = cpl_propertylist_load(fname, 0);
	  
	  espdr_plist_overwrite_list(plist, plist_ref);
	  //cpl_propertylist_dump(plist,stdout);

	  //cpl_frameset* tmp_set = cpl_frameset_duplicate(allframes);
	  cpl_frameset_erase(allframes, "HEADER");


		  //cpl_frameset_dump(tmp_set,stdout);
		  //cpl_frame* pro_frm = cpl_frameset_find(tmp_set,output_procatg);
		  //cpl_frameset_insert(frameset,cpl_frame_duplicate(pro_frm));
		  //cpl_frameset_delete(tmp_set);

	  /*
	  cpl_propertylist_erase(plist, "ESO TEL TARG ALPHA");
	  cpl_propertylist_erase(plist, "ESO TEL TARG DELTA");
	  cpl_propertylist_erase(plist, "ESO TEL TARG PMA");
	  cpl_propertylist_erase(plist, "ESO TEL TARG PMD");
	  cpl_propertylist_erase(plist, "ESO OCS TARG SPTYPE");
	  cpl_propertylist_erase(plist, "ESO TEL TARG RADVEL");
	  cpl_propertylist_erase(plist, "ESO TEL AIRM START");
	  cpl_propertylist_erase(plist, "ESO TEL AIRM END");
	  cpl_propertylist_erase(plist, "ESO TEL ALT");
	  cpl_propertylist_erase(plist, "ESO TEL AMBI RHUM");
	  cpl_propertylist_erase(plist, "ESO TEL AMBI TEMP");
	  cpl_propertylist_erase(plist, "ESO TEL AMBI PRES START");
	  cpl_propertylist_erase(plist, "ESO TEL AMBI PRES END");
	  cpl_propertylist_erase(plist, "PIPELINE MASK TIMESTAMP");
	  cpl_propertylist_erase(plist, "UTC");
	  */
  }


  /* We have to update the extra keywords again for the primary HDU to make
   * sure we have the ability to override what cpl_dfs_setup_product_header
   * sets. The reason for still having the cpl_propertylist_append above is to
   * make sure we use comments as given by the applist and not as found in the
   * raw file we inherit from. The SDP format prefers standardised comments, not
   * necessarily used by the raw files. */
  error = cpl_propertylist_copy_property_regexp(plist, applist, ".*", 0);
  cpl_error_ensure(! error, error, goto cleanup,
          "Could not update extra keywords when writing file '%s'.", filename);

  if (remregexp != NULL) {
    cpl_errorstate prestate = cpl_errorstate_get();
    (void) cpl_propertylist_erase_regexp(plist, remregexp, 0);
    cpl_error_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(),
                     goto cleanup,
                     "Failed to filter keywords when writing file '%s'.",
                     filename);
  }

  error = espdr_sdp_spectrum_save(spectrum, filename, plist, tablelist);
  cpl_error_ensure(! error, error, goto cleanup,
                   "Failed to save SPD spectrum to file '%s'.", filename);

  /* Optionally return the SDP keywords that were written to the output. */
  if (header != NULL) {
    error = cpl_propertylist_copy_property_regexp(header, spectrum->proplist,
                                                  ".*", 0);
    cpl_error_ensure(! error, error, goto cleanup,
                     "Could not return SDP keywords in header output.");
  }

  /* Insert the frame of the saved file in the input frameset. */
  error = cpl_frameset_insert(allframes, product_frame);
  cpl_error_ensure(! error, error, goto cleanup,
        "Failed to insert new product frame when writing file '%s'.", filename);

  /* Delete output property list if it was only a temporary local object. */
  if (plist != header) cpl_propertylist_delete(plist);

  return CPL_ERROR_NONE;

cleanup:
  /* If an error occurred we come here to cleanup memory. Note that the delete
   * functions already check for NULL pointers. */
  if (header != NULL) {
    cpl_errorstate prestate = cpl_errorstate_get();
    (void) cpl_propertylist_empty(header);
    cpl_errorstate_set(prestate);
  } else {
    cpl_propertylist_delete(plist);
  }
  cpl_frame_delete(product_frame);
  return cpl_error_get_code();
}


void espdr_sdp_spectrum_dump(const espdr_sdp_spectrum *self, FILE *stream)
{
  if (stream == NULL) {
    stream = stdout;
  }
  if (self == NULL) {
    fprintf(stream, "NULL SDP spectrum\n\n");
    return;
  }

  assert(self->proplist != NULL);
  assert(self->table != NULL);

  fprintf(stream, "SDP spectrum at address %p\n", (void*)self);
  fprintf(stream, "NELEM = %"CPL_SIZE_FORMAT"\n", self->nelem);
  cpl_propertylist_dump(self->proplist, stream);
  cpl_table_dump_structure(self->table, stream);
  cpl_table_dump(self->table, 0, cpl_table_get_nrow(self->table), stream);
}

/**
 * @internal
 * @brief Finds the earliest value for MJD_OBS in the set of inputs.
 *
 * The earliest value found for MJD_OBS in the list of input frames is returned.
 *
 * @param usedframes  The list of used frames to check.
 * @return the earliest MJD_OBS or NAN if an error occurred. If an error did
 *      occur then an error code will be set that can be checked with
 *      @c cpl_error_get_code.
 */
static double espdr_find_first_mjd_obs(const cpl_frameset *usedframes)
{
  cpl_propertylist* keywords = NULL;
  cpl_frameset_iterator* iter = NULL;
  double mjdobs = NAN;

  iter = cpl_frameset_iterator_new(usedframes);
  const cpl_frame* frame = cpl_frameset_iterator_get_const(iter);
  while (frame != NULL) {
    if (cpl_frame_get_group(frame) == CPL_FRAME_GROUP_RAW) {
      /* Load the primary header keywords from the raw frame. */
      const char* filename;
      filename = cpl_frame_get_filename(frame);
      keywords = cpl_propertylist_load(filename, 0);
      /* Extract the MJD-OBS keyword and check if its the earliest one. */
      double val;
      val = cpl_propertylist_get_double(keywords, ESPDR_MJDOBS);
      cpl_propertylist_delete(keywords);
      keywords = NULL;
      if (isnan(mjdobs)) {
        mjdobs = val;
      } else if (val < mjdobs) {
        mjdobs = val;
      }
    }
    /* Increment the iterator to the next frame. */
    cpl_errorstate status = cpl_errorstate_get();
    cpl_frameset_iterator_advance(iter, 1);
    if (cpl_error_get_code() == CPL_ERROR_ACCESS_OUT_OF_RANGE) {
        cpl_errorstate_set(status);
    }
    frame = cpl_frameset_iterator_get_const(iter);
  }


  cpl_frameset_iterator_delete(iter);
  cpl_propertylist_delete(keywords);
  return mjdobs;
}
static cpl_error_code
espdr_sdp_spectrum_set_colum_data_int(cpl_size nelem, cpl_error_code error,
		cpl_array* array, int* values, const char* cname,
		espdr_sdp_spectrum* sdp_spectrum) {
	array = cpl_array_wrap_int(values, nelem);
	error |= espdr_sdp_spectrum_set_column_data(sdp_spectrum, cname, array);
	cpl_array_unwrap(array);
	return error;
}

static cpl_error_code
espdr_sdp_spectrum_set_colum_data_int_const(cpl_size nelem, cpl_error_code error,
		cpl_array* array, const int* values, const char* cname,
		espdr_sdp_spectrum* sdp_spectrum) {
	array = cpl_array_wrap_int((int*)values, nelem);
	error |= espdr_sdp_spectrum_set_column_data(sdp_spectrum, cname, array);
	cpl_array_unwrap(array);
	return error;
}

static cpl_error_code
espdr_sdp_spectrum_set_colum_data_double(cpl_size nelem, cpl_error_code error,
		cpl_array* array, double* specerrs, const char* cname,
		espdr_sdp_spectrum* sdp_spectrum) {
	array = cpl_array_wrap_double(specerrs, nelem);
	error |= espdr_sdp_spectrum_set_column_data(sdp_spectrum, cname, array);
	cpl_array_unwrap(array);
	return error;
}

static cpl_error_code
espdr_sdp_spectrum_set_colum_data_double_const(cpl_size nelem, cpl_error_code error,
		cpl_array* array, const double* specerrs, const char* cname,
		espdr_sdp_spectrum* sdp_spectrum) {
	array = cpl_array_wrap_double((double*) specerrs, nelem);
	error |= espdr_sdp_spectrum_set_column_data(sdp_spectrum, cname, array);
	cpl_array_unwrap(array);
	return error;
}

static void replace_spaces_with_commas(char *str) { 
    for (int i = 0; str[i] != '\0'; i++) { 
        if (str[i] == ' ') { 
            str[i] = '_'; // Replace space with a comma 
        } 
    } 
}

static cpl_matrix*
espdr_sdp_spectrum_fill_specres(const char* instrume)
{

  cpl_matrix* specres_lut;
  cpl_size rows;
  cpl_size cols;
  if(strcmp(instrume,"ESPRESSO") == 0) {
      rows = 5;
      cols = 2;
  } else {
      rows = 2;
      cols = 1;
  }
  specres_lut = cpl_matrix_new(rows, cols);
  if(strcmp(instrume,"ESPRESSO") == 0) {

      /* BLUE camera */
      //cpl_matrix_set(specres_lut, raw, col, value);
      cpl_matrix_set(specres_lut, 0, 0, 140000); //SINGLEHR11
      cpl_matrix_set(specres_lut, 1, 0, 140000); //SINGLEHR21
      cpl_matrix_set(specres_lut, 2, 0, 190000); //SINGLE-UHR
      cpl_matrix_set(specres_lut, 3, 0,  70000); //MULTIMR42
      cpl_matrix_set(specres_lut, 4, 0,  70000); //MULTIMR84

      /* RED camera */
      cpl_matrix_set(specres_lut, 0, 1, 140000); //SINGLEHR11
      cpl_matrix_set(specres_lut, 1, 1, 140000); //SINGLEHR21
      cpl_matrix_set(specres_lut, 2, 1, 190000); //SINGLE-UHR
      cpl_matrix_set(specres_lut, 3, 1,  70000); //MULTIMR42
      cpl_matrix_set(specres_lut, 4, 1,  70000); //MULTIMR84

  } else if(strcmp(instrume,"NIRPS") == 0) {

      cpl_matrix_set(specres_lut, 0, 0, 80000); //NIRPS-HA
      cpl_matrix_set(specres_lut, 1, 0, 70000); //NIRPS-HE

  } else if(strcmp(instrume,"HARPS") == 0) {

      cpl_msg_warning(cpl_func,"add specres for: %s ",instrume);
      cpl_matrix_set(specres_lut, 0, 0, 115000); //HARPS-HAM
      cpl_matrix_set(specres_lut, 1, 0,  80000); //HARPS-EGGS
  } else if(strcmp(instrume,"CORALIE") == 0) {
      cpl_msg_warning(cpl_func,"add specres for: %s ",instrume);
      cpl_matrix_set(specres_lut, 0, 0, 60000); //CORALIE
      cpl_matrix_set(specres_lut, 1, 0, 60000); //CORALIE
    
  } else {
      cpl_msg_warning(cpl_func,"INSTRUMENT: %s not supported",instrume);
  }
  espdr_check_error_code(cpl_func);
  return specres_lut;
}

static cpl_error_code
espdr_sdp_spectrum_get_ins_mode_id_camera_id_and_aperture(
    const char* ins_mode, const char* instrume, const char* det_name,
    int* ins_mode_id, double* aperture, int* camera_id)
{


  //espdr_msg("ins_mode %s",ins_mode);
  if (strstr(ins_mode,"SINGLEUHR")) {
      *ins_mode_id = 2;
      *aperture = 0.5; // [deg] later we scale to arcsec
  } else if (strstr(ins_mode,"SINGLEHR")) {
      *ins_mode_id = 1;
      *aperture = 1.0;
  } else if (strstr(ins_mode,"MULTIMR")) {
      *ins_mode_id = 4;
      *aperture = 4.0;
  } else if (strcmp(ins_mode,"HA") == 0) { /* NIRPS */
      *ins_mode_id = 0;
      *aperture = 0.4;
  } else if (strcmp(ins_mode,"HE") == 0) {
      *ins_mode_id = 1;
      *aperture = 0.9;
  } else if (strstr(ins_mode,"HAM")) {/* HARPS */
      *ins_mode_id = 0;
      *aperture = 1.0;
  } else if (strstr(ins_mode,"EGGS")) {
      *ins_mode_id = 1;
      *aperture = 2.0;
  } else if (strstr(ins_mode,"HARPS")) {
      *ins_mode_id = 0;
      *aperture = 1.0;
  } else if (strstr(ins_mode,"CORALIE")) {
      *ins_mode_id = 0;
      *aperture = 1.0;
  } else {
      *ins_mode_id = -1;
  }

  if ( ((strcmp(instrume,"ESPRESSO") == 0) &&
      (strstr(det_name,"BLUE") != NULL)) ||
      (strcmp(instrume,"NIRPS") == 0) ||
      (strcmp(instrume,"CORALIE") == 0) ||
      (strcmp(instrume,"HARPS") == 0) ) {
      *camera_id = 0;
  } else {
      *camera_id = 1;
  }
  /*
   espdr_msg("ins_mode_id %d aperture %g camera_id %d", ins_mode_id, aperture,
   camera_id);
   */
  espdr_check_error_code(cpl_func);
  return cpl_error_get_code();

}
static cpl_error_code
espdr_sdp_spectrum_prepare_header(const cpl_table* s1d_table,
				  espdr_sdp_spectrum    * sdp_spectrum,
				  const cpl_propertylist * sourcekeys,
				  const cpl_frameset     * used_frames,
				  cpl_boolean is_fluxcal)
{
  cpl_errorstate prestate = cpl_errorstate_get();
  cpl_error_code error = CPL_ERROR_NONE;
  cpl_ensure_code(s1d_table != NULL, CPL_ERROR_NULL_INPUT);

  double wavemin = cpl_table_get_column_min(s1d_table,"wavelength");
  double wavemax = cpl_table_get_column_max(s1d_table,"wavelength");
  int nrow = cpl_table_get_nrow(s1d_table);

  if (! cpl_errorstate_is_equal(prestate)) {
      return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
				   "Failed to find required wavelength information.");
  }

  int ut_id=0;
  const char* telescope = cpl_propertylist_get_string(sourcekeys,"TELESCOP");
  if (strcmp(telescope,"ESO-VLT-U1")  == 0 ){
      ut_id = 1;
  } else if (strcmp(telescope,"ESO-VLT-U2") == 0 ){
      ut_id = 2;
  } else if (strcmp(telescope,"ESO-VLT-U3")  == 0 ){
      ut_id = 3;
  } else if (strcmp(telescope,"ESO-VLT-U4")  == 0 ){
      ut_id = 4;
  }
  //espdr_msg("ut id: %d",ut_id);

  /* Set default values for various SDP keywords. */
  error |= espdr_sdp_spectrum_set_prodcatg(sdp_spectrum, ESPDR_SCIENCE_SPECTRUM);
  //error |= espdr_sdp_spectrum_set_mepoch(sdp_spectrum, CPL_FALSE);
  error |= espdr_sdp_spectrum_set_contnorm(sdp_spectrum, CPL_FALSE);

  const char* instrume = cpl_propertylist_get_string(sourcekeys,"INSTRUME");
  espdr_msg("INSTRUME: %s",instrume);
  if ( (strcmp(instrume,"ESPRESSO") == 0) ||
       (strcmp(instrume,"CORALIE") == 0) ||
       (strcmp(instrume,"NIRPS") == 0) ) {
      error |= espdr_sdp_spectrum_set_fluxerr(sdp_spectrum,
					      ESPDR_SDP_KEYWORD_FLUXERR_VALUE);
  } else {
      error |= espdr_sdp_spectrum_set_fluxerr(sdp_spectrum,
					      ESPDR_SDP_KEYWORD_UNCALIBRATED_FLUXERR_VALUE);
  }
  error |= espdr_sdp_spectrum_set_referenc(sdp_spectrum,
					   ESPDR_SDP_KEYWORD_REFERENC_VALUE);
  error |= espdr_sdp_spectrum_set_voclass(sdp_spectrum,
					  ESPDR_SDP_KEYWORD_VOCLASS_VALUE);
  error |= espdr_sdp_spectrum_set_vopub(sdp_spectrum,
					ESPDR_SDP_KEYWORD_VOPUB_VALUE);


  /* Take the PRODLVL, ORIGIN and SPECSYS keywords from the source list or
   * hardcode their values. */
  if (cpl_propertylist_has(sourcekeys, ESPDR_SDP_KEYWORD_PRODLVL)) {
      error |= espdr_sdp_spectrum_copy_prodlvl(sdp_spectrum, sourcekeys,
					       ESPDR_SDP_KEYWORD_PRODLVL);
  } else {
      error |= espdr_sdp_spectrum_set_prodlvl(sdp_spectrum,
					      ESPDR_SDP_KEYWORD_PRODLVL_VALUE);
  }

  if (cpl_propertylist_has(sourcekeys, ESPDR_SDP_KEYWORD_ORIGIN)) {
      error |= espdr_sdp_spectrum_copy_origin(sdp_spectrum, sourcekeys,
					      ESPDR_SDP_KEYWORD_ORIGIN);
  } else {
      error |= espdr_sdp_spectrum_set_origin(sdp_spectrum,
					     ESPDR_SDP_KEYWORD_ORIGIN_VALUE);
  }

  if (cpl_propertylist_has(sourcekeys, ESPDR_SDP_KEYWORD_SPECSYS)) {
      error |= espdr_sdp_spectrum_copy_specsys(sdp_spectrum, sourcekeys,
					       ESPDR_SDP_KEYWORD_SPECSYS);
  } else {
      error |= espdr_sdp_spectrum_set_specsys(sdp_spectrum,
					      ESPDR_SDP_KEYWORD_SPECSYS_VALUE);
  }

  /* Copy ESO.PRO.REC1.PIPE.ID to PROCSOFT if it is available. */
  if (cpl_propertylist_has(sourcekeys, ESPDR_PRO_REC1_PIPE_ID)) {
      error |= espdr_sdp_spectrum_copy_procsoft(sdp_spectrum, sourcekeys,
						ESPDR_PRO_REC1_PIPE_ID);
  }

  error |= espdr_sdp_spectrum_set_dispelem(sdp_spectrum, "VIS");
  /* Set PROG_ID from ESO.OBS.PROG.ID */
  if  (cpl_propertylist_has(sourcekeys, ESPDR_OBS_PROG_ID)) {
	  if  (strcmp(instrume,"CORALIE") != 0) {
		  error |= espdr_sdp_spectrum_copy_progid(sdp_spectrum, sourcekeys,
				  ESPDR_OBS_PROG_ID);
	  }
  } else {
    cpl_propertylist_append_bool(sdp_spectrum->proplist,ESPDR_NOESODAT,CPL_TRUE);
  }
  /* Set OBID1 from ESO.OBS.ID */
  if  (cpl_propertylist_has(sourcekeys, ESPDR_OBS_ID)) {
	  if  (strcmp(instrume,"CORALIE") != 0) {
		  error |= espdr_sdp_spectrum_copy_obid(sdp_spectrum, 1,
				  sourcekeys, ESPDR_OBS_ID);
	  }
  } else {
	  if  (!cpl_propertylist_has(sourcekeys, ESPDR_OBS_ID)) {
		  cpl_propertylist_append_bool(sdp_spectrum->proplist,ESPDR_NOESODAT,CPL_TRUE);
	  }
  }
  
  if (strcmp(instrume,"CORALIE") == 0 &&
      !cpl_propertylist_has(sourcekeys, ESPDR_NOESODAT)) {
    cpl_propertylist_append_bool(sdp_spectrum->proplist,ESPDR_NOESODAT,CPL_TRUE);
    cpl_propertylist_append_bool(sdp_spectrum->proplist,KEY_EXT_OBJ,CPL_FALSE);
  }
  if(cpl_propertylist_has(sdp_spectrum->proplist,KEY_EXT_OBJ)) {
	  cpl_boolean key_value = cpl_propertylist_get_bool(sdp_spectrum->proplist,KEY_EXT_OBJ);
  }
  /* Set OBSTECH from ESO.PRO.TECH */
  error |= espdr_sdp_spectrum_copy_obstech(sdp_spectrum, sourcekeys,
					   ESPDR_PRO_TECH);

  /* Set the FLUXCAL keyword based on value of  is_fluxcal */
  /*
  const char* procatg = cpl_propertylist_get_string(sourcekeys,
						    CPL_DFS_PRO_CATG);
						    */
  const char* fluxcal;
  if(!is_fluxcal) {
      fluxcal = "UNCALIBRATED";

  } else {
      fluxcal = "ABSOLUTE";
  }

  error |= espdr_sdp_spectrum_set_fluxcal(sdp_spectrum, fluxcal);

  /* Setup the WAVELMIN, WAVELMAX and SPEC_BIN keywords from WCS information. */
  prestate = cpl_errorstate_get();

  if (cpl_errorstate_is_equal(prestate)) {

      /* keywords in nm units */
      double wavelmin = cpl_table_get_column_min(s1d_table,"wavelength") * AA2nm;
      double wavelmax = cpl_table_get_column_max(s1d_table,"wavelength") * AA2nm;
      int nrow = cpl_table_get_nrow(s1d_table);

      double specbin = (wavelmax - wavelmin) / (nrow-1.) ;

      error |= espdr_sdp_spectrum_set_wavelmin(sdp_spectrum, wavelmin);
      error |= espdr_sdp_spectrum_set_wavelmax(sdp_spectrum, wavelmax);
      error |= espdr_sdp_spectrum_set_specbin(sdp_spectrum, specbin);
  } else {
      error = cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
				    "Could not find WCS information.");
  }

  /* Reassign the OBJECT keyword to use the ESO.OBS.TARG.NAME value.
   * Also get the TITLE from the same OBJECT keyword value. */
  const char* object = cpl_propertylist_get_string(sourcekeys,
						   ESPDR_OBS_TARG_NAME);
  if (object != NULL) {
      error |= espdr_sdp_spectrum_set_object(sdp_spectrum, object);
      char* objstr = cpl_strdup(object);
      char* title;
      
      if (cpl_propertylist_has(sourcekeys, KEY_OBJECT) &&
          cpl_propertylist_has(sourcekeys, ESPDR_OBS_ID) &&
          cpl_propertylist_has(sourcekeys, "DATE-OBS")
      ) {

         title = cpl_sprintf("%s_%d_%s",
			     cpl_propertylist_get_string(sourcekeys, KEY_OBJECT),
	 	             cpl_propertylist_get_int(sourcekeys, ESPDR_OBS_ID),
		             cpl_propertylist_get_string(sourcekeys, "DATE-OBS"));
         replace_spaces_with_commas(title); 
         error |= espdr_sdp_spectrum_set_title(sdp_spectrum, title);

      } else {
      
         /* Find second occurrence of ',' in objstr and set it to NULL to trim off
          * anything after that comma. Then generate the title and limit it to 68
          * characters. */
         char* c = strchr(objstr, ',');
         if (c != NULL) {
	     c = strchr(c+1, ',');
	     if (c != NULL) *c = '\0';
         }
	 title = cpl_sprintf("%.68s", objstr);
         replace_spaces_with_commas(title); 
         error |= espdr_sdp_spectrum_set_title(sdp_spectrum, title);
	 }
      cpl_free(title);
      cpl_free(objstr);
  } else {
      error = cpl_error_set_message(cpl_func, cpl_error_get_code(),
				    "Could not assign '%s' or '%s' since the '%s' keyword could"
				    " not be found.", ESPDR_SDP_KEYWORD_OBJECT,
				    ESPDR_SDP_KEYWORD_TITLE, ESPDR_OBS_TARG_NAME);
  }

  if(cpl_propertylist_has(sourcekeys, ESPDR_RA)) {
      error |= espdr_sdp_spectrum_copy_ra(sdp_spectrum, sourcekeys, ESPDR_RA);
  }

  if(cpl_propertylist_has(sourcekeys, ESPDR_DEC)) {
      error |= espdr_sdp_spectrum_copy_dec(sdp_spectrum, sourcekeys, ESPDR_DEC);
  }

  /* Calculate and set the EXPTIME and TEXPTIME (must be identical). */
  prestate = cpl_errorstate_get();
  double exptime = espdr_calculate_exptime(used_frames);
  if (! isnan(exptime)) {
      error |= espdr_sdp_spectrum_set_exptime(sdp_spectrum, exptime);
      error |= espdr_sdp_spectrum_set_texptime(sdp_spectrum, exptime);
  } else if (! cpl_errorstate_is_equal(prestate)) {
      error = cpl_error_get_code();
  }

  /* Copy the MJD-OBS and MJD-END keywords and store their values, which will be
   * used later. */
  prestate = cpl_errorstate_get();
  double mjdobs = cpl_propertylist_get_double(sourcekeys, ESPDR_MJDOBS);
  double mjdend = cpl_propertylist_get_double(sourcekeys, ESPDR_MJDEND);

  if (cpl_errorstate_is_equal(prestate)) {
      error |= espdr_sdp_spectrum_set_mjdobs(sdp_spectrum, mjdobs);
      error |= espdr_sdp_spectrum_set_mjdend(sdp_spectrum, mjdend);
  } else {
      mjdobs = mjdend = NAN;
  }

  /* Copy the SNR keyword from the ESO.QC.FLUX.SN keyword. */
  if (cpl_propertylist_has(sourcekeys, ESPDR_QC_FLUX_SN)) {
      error |= espdr_sdp_spectrum_copy_snr(sdp_spectrum, sourcekeys,
					   ESPDR_QC_FLUX_SN);
  }
  if (cpl_propertylist_has(sourcekeys, KEY_SNR)) {
      error |= espdr_sdp_spectrum_copy_snr(sdp_spectrum, sourcekeys,
					   KEY_SNR);
  }

  /* Get nominal values for LAMNLIN, LAMRMS and SPEC_RES from lookup tables */
  const char* ins_mode = NULL;
  if (cpl_propertylist_has(sourcekeys, "ESO INS MODE")) {
      ins_mode = cpl_propertylist_get_string(sourcekeys,"ESO INS MODE");
  } else {
      ins_mode = instrume;
  }

  int binx = -1;
  if (cpl_propertylist_has(sourcekeys, "ESO DET BINX")) {
      binx = cpl_propertylist_get_int(sourcekeys, "ESO DET BINX");
  } else {
      if (cpl_propertylist_has(sourcekeys, "ESO DET WIN1 BINX")) {
	  binx = cpl_propertylist_get_int(sourcekeys, "ESO DET WIN1 BINX");
      } else {
	  binx = 1;
      }
  }

  const char* det_name = NULL;
  if (cpl_propertylist_has(sourcekeys, "ESO DET NAME")) {
      det_name = cpl_propertylist_get_string(sourcekeys,"ESO DET NAME");
  } else {
      det_name = instrume;
  }

  int ins_mode_id = -1;
  int camera_id = -1;
  double aperture = 0;

  espdr_sdp_spectrum_get_ins_mode_id_camera_id_and_aperture(ins_mode, instrume,
							    det_name,
							    &ins_mode_id,
							    &aperture,
							    &camera_id);

  double lamnlin = -1;
  double lamrms = -1;
  double specres = -1;
  /* BLUE, RED */
  double lamnlin_lut[2] = {380, 523};
  double lamrms_lut[2][2] = {
      /*    Before           After
       * 1st Jan 2012    1st Jan 2012    CAMERA */
      {     0.013,          0.003},   /* BLUE */
      {     0.013,          0.002},   /* RED */
  };


  cpl_matrix* specres_lut = espdr_sdp_spectrum_fill_specres(instrume);

  /* Figure out the date epoch of the raw data. We check if the date range
   * overlaps the epoch threshold. If the data does then we dont try figure out
   * which epoch to choose LAMRMS for. */
  double date_threshold = 55927.0; /* Julian Date for 1/1/2012 */
  int softepoch = -1; /* epoch of the software used. 0 = before 1/1/2012 and
	                                                     1 = after 1/1/2012. */
  prestate = cpl_errorstate_get();
  double first_mjdobs = espdr_find_first_mjd_obs(used_frames);
  if (cpl_errorstate_is_equal(prestate)) {
      if (! (first_mjdobs < date_threshold && date_threshold < mjdend)) {
	  if (first_mjdobs < date_threshold) {
	      softepoch = 0;
	  } else {
	      softepoch = 1;
	  }
      }
  } else {
      error = cpl_error_get_code();
  }

  /* Identify the lookup entry for the slit that was used. */


  /* Copy the LAMNLIN, LAMRMS and SPEC_RES keywords from the source keywords
   * if they already exist, otherwise set them from the lookup table. */
  if (cpl_propertylist_has(sourcekeys, ESPDR_SDP_KEYWORD_LAMNLIN)) {
      prestate = cpl_errorstate_get();
      int val = cpl_propertylist_get_int(sourcekeys, ESPDR_SDP_KEYWORD_LAMNLIN);
      if (cpl_errorstate_is_equal(prestate)) {
	  lamnlin = val;
      } else {
	  error = cpl_error_get_code();
      }
  } else {
      if (camera_id != -1) {
	  lamnlin = lamnlin_lut[camera_id];
      }
  }
  if (cpl_propertylist_has(sourcekeys, ESPDR_SDP_KEYWORD_LAMRMS)) {
      prestate = cpl_errorstate_get();
      double val = cpl_propertylist_get_double(sourcekeys,
					       ESPDR_SDP_KEYWORD_LAMRMS);
      if (cpl_errorstate_is_equal(prestate)) {
	  lamrms = val;
      } else {
	  error = cpl_error_get_code();
      }
  } else {
      if (camera_id != -1 && softepoch != -1) {
	  lamrms = lamrms_lut[camera_id][softepoch];
      }
  }
  if (cpl_propertylist_has(sourcekeys, ESPDR_SDP_KEYWORD_SPEC_RES)) {
      prestate = cpl_errorstate_get();
      double val = cpl_propertylist_get_double(sourcekeys,
					       ESPDR_SDP_KEYWORD_SPEC_RES);
      if (cpl_errorstate_is_equal(prestate)) {
	  specres = val;
      } else {
	  error = cpl_error_get_code();
      }
  } else {
      if (ins_mode_id != -1 && camera_id != -1) {
	  //espdr_msg("ins_mode_id %d camera_id %d", ins_mode_id, camera_id);
	  specres = cpl_matrix_get(specres_lut, ins_mode_id, camera_id);
      } else {
	  espdr_msg("Case ins_mode_id %d camera_id %d not supported",
		    ins_mode_id,camera_id);
      }
  }
  cpl_matrix_delete(specres_lut);
  /* Set the LAMNLIN, LAMRMS and SPEC_RES keywords if they were actually
   * calculated. */
  if (lamnlin != -1) {
      error |= espdr_sdp_spectrum_set_lamnlin(sdp_spectrum, lamnlin);
  }
  if (lamrms != -1) {
      error |= espdr_sdp_spectrum_set_lamrms(sdp_spectrum, lamrms);
  }
  if (specres != -1) {
      error |= espdr_sdp_spectrum_set_specres(sdp_spectrum, specres);
  }

  /* Calculate the SPEC_ERR keyword if it is not already in the source keywords.
   * Otherwise we just copy it to the primary header. If we could not calculate
   * the keyword then do not set it. */
  if (cpl_propertylist_has(sourcekeys, ESPDR_SDP_KEYWORD_SPEC_ERR)) {
      prestate = cpl_errorstate_get();
      double val = cpl_propertylist_get_double(sourcekeys,
					       ESPDR_SDP_KEYWORD_SPEC_ERR);
      if (cpl_errorstate_is_equal(prestate)) {
	  error |= espdr_sdp_spectrum_set_specerr(sdp_spectrum, val);
      } else {
	  error = cpl_error_get_code();
      }
  } else if (lamnlin > 0) {
      double specerr = -1;
      if (cpl_propertylist_has(sourcekeys, ESPDR_CRDER1)) {
	  prestate = cpl_errorstate_get();
	  double crder1 = cpl_propertylist_get_double(sourcekeys, ESPDR_CRDER1);
	  if (cpl_errorstate_is_equal(prestate) && crder1 != 0) {
	      specerr = crder1 / sqrt(lamnlin);
	  } else {
	      error = cpl_error_get_code();
	  }
      }
      if (specerr == -1 && lamrms != -1) {
	  specerr = lamrms / sqrt(lamnlin);
      }
      if (specerr != -1) {
	  error |= espdr_sdp_spectrum_set_specerr(sdp_spectrum, specerr);
      }
  }

  /* Calculate the SPEC_SYE keyword if it is not already in the source keywords.
   * Otherwise we just copy it to the primary header. If we could not calculate
   * the keyword then do not set it. */
  if (cpl_propertylist_has(sourcekeys, ESPDR_SDP_KEYWORD_SPEC_SYE)) {
      error |= espdr_sdp_spectrum_copy_specsye(sdp_spectrum, sourcekeys,
					       ESPDR_SDP_KEYWORD_SPEC_SYE);
  } else if (lamnlin > 0) {
      double specsye = NAN;
      if (cpl_propertylist_has(sourcekeys, ESPDR_CSYER1)) {
	  prestate = cpl_errorstate_get();
	  double csyer1 = cpl_propertylist_get_double(sourcekeys, ESPDR_CSYER1);
	  if (cpl_errorstate_is_equal(prestate)) {
	      specsye = csyer1 / sqrt(lamnlin);
	  } else {
	      error = cpl_error_get_code();
	  }
      }
      if (! isnan(specsye)) {
	  error |= espdr_sdp_spectrum_set_specsye(sdp_spectrum, specsye);
      }
  }

  /* Set the TOT_FLUX keyword to true if the ESO.INS.OPTIi.NAME keyword
   * contains "5.0x<Length>" where <Length> can be any number.
   * Also setup the APERTURE keyword from the ESO.INS.OPTIi.NAME keyword. */


  cpl_boolean totflux = CPL_FALSE;
  //TODO Check again this setting
  if (aperture > 1.0) {
      totflux = CPL_TRUE;
  }

  error |= espdr_sdp_spectrum_set_totflux(sdp_spectrum, totflux);
  double deg2arcsec = 3600.;
  aperture /= deg2arcsec;
  error |= espdr_sdp_spectrum_set_aperture(sdp_spectrum, aperture);


  /* Calculate the TELAPSE and TMID keywords from MJD-OBS and MJD-END. */
  if (! isnan(mjdobs) && ! isnan(mjdend)) {
      /* this is a special case for which to prevent rounding errors due to
       * the very small difference MJDEND - MJDSTART, and using the fact that
       * MJDEND - MJDSTART= EXPTIME (or DIT*NDIT) we compute TELAPSE as
       * TELAPSE=EXPTIME
       */
      double telapse = cpl_propertylist_get_double(sourcekeys, ESPDR_EXPTIME);

      error |= espdr_sdp_spectrum_set_telapse(sdp_spectrum,telapse);

      error |= espdr_sdp_spectrum_set_tmid(sdp_spectrum,
					   (mjdend + mjdobs) * 0.5);
  } else {
      error = cpl_error_set_message(cpl_func, cpl_error_get_code(),
				    "Could not assign '%s' nor '%s' since one could not read"
				    " either the '%s' or '%s' keyword.", ESPDR_SDP_KEYWORD_TELAPSE,
				    ESPDR_SDP_KEYWORD_TMID, ESPDR_MJDOBS, ESPDR_MJDEND);
  }

  /* Setup the TDMIN and TDMAX keywords from wavelength minimum and maximum. */
  error |= espdr_sdp_spectrum_set_tdmin(sdp_spectrum, wavemin);
  error |= espdr_sdp_spectrum_set_tdmax(sdp_spectrum, wavemax);

  /* Setup the SPEC_VAL and SPEC_BW keywords from the spectrum wavelength
   * information. */
  error |= espdr_sdp_spectrum_set_specval(sdp_spectrum,
					  (wavemax + wavemin) * 0.5 * AA2nm);
  error |= espdr_sdp_spectrum_set_specbw(sdp_spectrum, (wavemax - wavemin) * AA2nm);

  error |= espdr_sdp_spectrum_set_nelem(sdp_spectrum, nrow);
  error |= espdr_sdp_spectrum_set_extname(sdp_spectrum,
					  ESPDR_SDP_KEYWORD_EXTNAME_VALUE);
  error |= espdr_sdp_spectrum_set_inherit(sdp_spectrum,
					  ESPDR_SDP_KEYWORD_INHERIT_VALUE);

  /* Check the error state and stop at this point (cleaning up memory) if there
   * have been any errors. */
  if (error) {
      return cpl_error_set_message(cpl_func, cpl_error_get_code(),
				   "Failed to initialise keyword list for the SDP spectrum.");
  }

  /* Figure out the units to use for FLUX and ERR table columns. REMOVED
   */

  /* Finally copy the table data. REMOVED */
  /* Copy the uncalibrated data if available. REMOVED */


  espdr_check_error_code(cpl_func);
  return cpl_error_get_code();
}

static cpl_error_code
espdr_sdp_spectrum_compute_snr(double* specflux, double* specerrs, int* specqual,
		const int nelem,espdr_sdp_spectrum    * sdp_spectrum,
		cpl_error_code *error)
{
	double* data = cpl_malloc(nelem * sizeof(double));
	cpl_array* array = cpl_array_wrap_double(data, nelem);
	//TODO: is this computed onnly on flux values > 0?
	if (data != NULL) {
		cpl_size i;
		for (i = 0; i < nelem; ++i) {
			double denom = specerrs[i];

			data[i] = (denom != 0) ? specflux[i] / denom : 0;
			data[i] = (data[i] != CPL_VALUE_NOTFINITE) ? data[i] : 0;
			//if(data[i] == CPL_VALUE_NOTFINITE || data[i] < 0  ) {
			if(data[i] == CPL_VALUE_NOTFINITE || specqual[i] != 0) {
			    data[i] = 0;
			}
		}
	} else {
		*error |= cpl_error_get_code();
	}

	cpl_size sz = 0;
	*error |= espdr_sdp_spectrum_set_column_data(sdp_spectrum,
			ESPDR_SDP_COLUMN_SNR, array);
        cpl_table* t = cpl_table_new(nelem);
        cpl_table_wrap_double(t,data,"SNR");
        sz = cpl_table_and_selected_double(t,"SNR", CPL_GREATER_THAN, 0);
        
        double snr_median = 0;
        double snr_mean = 0;
        cpl_table* x = cpl_table_extract_selected(t);
        if(sz > 1) {
            snr_median = cpl_table_get_column_median(x, "SNR");
            snr_mean = cpl_table_get_column_mean(x, "SNR");
        } else if (sz == 1) {
            snr_median = cpl_table_get_double(x,"SNR", 0, NULL);
            snr_mean = snr_median;
        }
        cpl_table_delete(x);
        cpl_table_unwrap(t,"SNR");
        cpl_table_delete(t);
        /*
	double snr_median = cpl_array_get_median(array);
	double snr_mean = cpl_array_get_mean(array);
	*/
    if(isnan(snr_median)) {
    	espdr_msg_warning("SNR is NAN! Reset SDP SNR to -999 to allow product saving");
    	snr_median = -999.;
    }
	espdr_sdp_spectrum_set_snr(sdp_spectrum,snr_median);
	espdr_sdp_spectrum_set_snr_avg(sdp_spectrum,snr_mean);
	cpl_array_unwrap(array);
	cpl_free(data);
	espdr_check_error_code(cpl_func);
	return cpl_error_get_code();
}


/**
 * @brief Converts data into Science Data Product 1D spectrum format.
 *
 * @param[out] sdp_spectrum The SDP output spectrum object the will be filled.
 * @param[in] used_frames List of frames used to generate the output spectrum.
 * @param[in] sourcekeys The list of FITS keywords from the source product file
 *  being converted to SDP format.
 * @param[in] spectrum The loaded flux calibrated spectrum in X-Shooter format.
 * @param[in] uncal_spectrum The loaded uncalibrated spectrum in X-Shooter
 *  format.
 *
 * @note If the @a table and @a props objects are not empty, then the spectrum
 *  data and keywords will be appended to the existing values in those objects
 *  instead.
 */

static cpl_error_code
espdr_sdp_spectrum_create2(
		espdr_sdp_spectrum    * sdp_spectrum,
		const cpl_frameset     * all_frames,
		const cpl_frameset     * used_frames,
		const cpl_propertylist * sourcekeys,
		cpl_table              * s1d_table,
		cpl_boolean is_fluxcal)
{

	cpl_errorstate prestate;
	cpl_error_code error = CPL_ERROR_NONE;

	cpl_ensure_code(sdp_spectrum != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(all_frames != NULL,   CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(used_frames != NULL,   CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(sourcekeys != NULL,   CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(s1d_table != NULL,   CPL_ERROR_NULL_INPUT);

	/* From here we start filling the spectrum header  */
	prestate = cpl_errorstate_get();

	/* prepare header */
	error |= espdr_sdp_spectrum_prepare_header(s1d_table, sdp_spectrum,
			sourcekeys, used_frames, is_fluxcal);

	/* prepare the data product */
	const char* column_wave_air_unit = NULL;
	const char* column_wave_unit = NULL;
	const char* column_flux_unit = NULL;
	const char* column_errs_unit = NULL;
	const char* column_qual_unit = NULL;
	const char* column_flux_cal_unit = NULL;
	const char* column_errs_cal_unit = NULL;

	//TODO: Note that the column names should be set by a common define
	column_wave_unit = cpl_table_get_column_unit(s1d_table,"wavelength");
	if(!is_fluxcal) {
		column_wave_air_unit = cpl_table_get_column_unit(s1d_table,"wavelength_air");
		column_flux_unit = cpl_table_get_column_unit(s1d_table,"flux");
		column_errs_unit = cpl_table_get_column_unit(s1d_table,"error");
		column_qual_unit = cpl_table_get_column_unit(s1d_table,"quality");
	} else {
		column_flux_cal_unit = cpl_table_get_column_unit(s1d_table,"flux_cal");
		column_errs_cal_unit = cpl_table_get_column_unit(s1d_table,"error_cal");
	}

	if (
			(column_flux_unit == NULL && column_errs_unit == NULL) &&
			(column_flux_cal_unit == NULL && column_errs_cal_unit == NULL)
	) {
		return cpl_error_get_code();
	}



	/* The following constructs a list of columns that we should setup in the
	 * spectrum table. */
	struct _espdr_column_info columns[] = {
			{ESPDR_SDP_COLUMN_WAVE, CPL_TYPE_DOUBLE, column_wave_unit,
					ESPDR_SDP_COLUMN_WAVE_FORMAT, ESPDR_SDP_COLUMN_WAVE_TYPE,
					ESPDR_SDP_COLUMN_WAVE_UCD},
			{ESPDR_SDP_COLUMN_FLUX_EL, CPL_TYPE_DOUBLE, ESPDR_SDP_EL_UNIT,
					ESPDR_SDP_COLUMN_FLUX_FORMAT, ESPDR_SDP_COLUMN_FLUX_EL_TYPE,
					ESPDR_SDP_COLUMN_FLUX_EL_UCD},
			{ESPDR_SDP_COLUMN_ERR_EL, CPL_TYPE_DOUBLE, ESPDR_SDP_EL_UNIT,
					ESPDR_SDP_COLUMN_ERR_FORMAT, ESPDR_SDP_COLUMN_ERR_EL_TYPE,
					ESPDR_SDP_COLUMN_ERR_EL_UCD},
			{ESPDR_SDP_COLUMN_QUAL_EL, CPL_TYPE_INT, ESPDR_SDP_COLUMN_QUAL_UNIT,
					ESPDR_SDP_COLUMN_QUAL_FORMAT, ESPDR_SDP_COLUMN_QUAL_EL_TYPE,
					ESPDR_SDP_COLUMN_QUAL_EL_UCD},
			{ESPDR_SDP_COLUMN_SNR, CPL_TYPE_DOUBLE, ESPDR_SDP_COLUMN_SNR_UNIT,
					ESPDR_SDP_COLUMN_SNR_FORMAT, ESPDR_SDP_COLUMN_SNR_TYPE,
				    ESPDR_SDP_COLUMN_SNR_UCD},
			{ESPDR_SDP_COLUMN_WAVE_AIR, CPL_TYPE_DOUBLE, column_wave_air_unit,
					ESPDR_SDP_COLUMN_WAVE_FORMAT, ESPDR_SDP_COLUMN_WAVE_AIR_TYPE,
					ESPDR_SDP_COLUMN_WAVE_AIR_UCD},
		    {NULL, CPL_TYPE_INVALID, NULL, NULL, NULL, NULL}
	};

	struct _espdr_column_info fluxcal_columns[] = {
			{ESPDR_SDP_COLUMN_FLUX_CAL, CPL_TYPE_DOUBLE, ESPDR_SDP_COLUMN_FLUX_CAL_UNIT,
					ESPDR_SDP_COLUMN_FLUX_FORMAT, ESPDR_SDP_COLUMN_FLUX_CAL_TYPE,
					ESPDR_SDP_COLUMN_FLUX_CAL_UCD},
			{ESPDR_SDP_COLUMN_ERR_CAL, CPL_TYPE_DOUBLE, ESPDR_SDP_COLUMN_ERR_CAL_UNIT,
					ESPDR_SDP_COLUMN_ERR_FORMAT, ESPDR_SDP_COLUMN_ERR_CAL_TYPE,
					ESPDR_SDP_COLUMN_ERR_CAL_UCD},
			{NULL, CPL_TYPE_INVALID, NULL, NULL, NULL, NULL}
	};

	struct _espdr_column_info* column;

	/* Add the spectrum columns to the table. */
	for (column = columns; column->name != NULL; ++column) {
		error |= espdr_sdp_spectrum_add_column(sdp_spectrum, column->name,
				column->type, column->unit, column->format,
				column->tutyp, column->tucd, NULL);
		if (error) {
			return cpl_error_set_message(cpl_func, cpl_error_get_code(),
					"Failed to setup the '%s' column.", column->name);
		}
	}

	if(is_fluxcal) {
		/* Add the spectrum columns to the table. */
		for (column = fluxcal_columns; column->name != NULL; ++column) {

			error |= espdr_sdp_spectrum_add_column(sdp_spectrum, column->name,
					column->type, column->unit, column->format,
					column->tutyp, column->tucd, NULL);
			if (error) {
				return cpl_error_set_message(cpl_func, cpl_error_get_code(),
						"Failed to setup the '%s' column.", column->name);
			}
		}
	}

	/* Finally copy the table data. */
	cpl_size nelem = (cpl_size) cpl_table_get_nrow(s1d_table);
	double* specwave = cpl_table_get_data_double(s1d_table,"wavelength");
	double* specwave_air = NULL;
	double* specflux = NULL;
	double* specerrs = NULL;
	double* specflux_cal = NULL;
	double* specerrs_cal = NULL;
	int* specqual = NULL;

	if(!is_fluxcal) {
		specwave_air = cpl_table_get_data_double(s1d_table,"wavelength_air");
		specflux = cpl_table_get_data_double(s1d_table,"flux");
		specerrs = cpl_table_get_data_double(s1d_table,"error");
		specqual = cpl_table_get_data_int(s1d_table,"quality");
	} else {
		specflux_cal = cpl_table_get_data_double(s1d_table,"flux_cal");
		specerrs_cal = cpl_table_get_data_double(s1d_table,"error_cal");
	}

	cpl_array* array = NULL;
	{ /* common case */
		/* Prepare the wavelength array and set it. */
		error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
				specwave, ESPDR_SDP_COLUMN_WAVE, sdp_spectrum);
	}

	if (specflux != NULL && specerrs != NULL && specqual != NULL) {

		/* Prepare the other data arrays and assign them. */
		error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
				specflux, ESPDR_SDP_COLUMN_FLUX_EL, sdp_spectrum);
		error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
				specerrs, ESPDR_SDP_COLUMN_ERR_EL, sdp_spectrum);
		error |= espdr_sdp_spectrum_set_colum_data_int(nelem, error, array,
				specqual, ESPDR_SDP_COLUMN_QUAL_EL, sdp_spectrum);

		/* Prepare the signal to noise column data. */
		espdr_sdp_spectrum_compute_snr(specflux, specerrs, specqual,
					       nelem, sdp_spectrum, &error);

	}

	if (specwave_air != NULL) {

		error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
				specwave_air, ESPDR_SDP_COLUMN_WAVE_AIR, sdp_spectrum);

	}

	if (specflux_cal != NULL && specerrs_cal != NULL) {
		error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
				specflux_cal, ESPDR_SDP_COLUMN_FLUX_CAL, sdp_spectrum);

		error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
				specerrs_cal, ESPDR_SDP_COLUMN_ERR_CAL, sdp_spectrum);
	}

	espdr_check_error_code(cpl_func);
	espdr_check_error_code(cpl_func);
	return cpl_error_get_code();
}

/** adjust final column to most processed information
 *
 *
 */
static cpl_error_code
espdr_set_final_cols_for_tell_case(espdr_sdp_spectrum * sdp_spectrum,
		const int nelem, const cpl_boolean is_tell_fluxcal,
		const cpl_boolean is_tell_skycorr )
{
	cpl_error_code error = CPL_ERROR_NONE;
	const cpl_array* array_flux_tell = NULL;
	const cpl_array* array_errs_tell = NULL;
	const cpl_array* array_qual_tell = NULL;
	cpl_array* array = NULL;
	const double* flux_tell = NULL;
	const double* errs_tell = NULL;
	const int* qual_tell = NULL;

	if(is_tell_fluxcal) {
		if(is_tell_skycorr) {
			array_flux_tell = cpl_table_get_array(sdp_spectrum->table,
					ESPDR_SDP_COLUMN_FLUX_TELL_CAL_SKYSUB, 0);
			flux_tell  = cpl_array_get_data_double_const(array_flux_tell);
			espdr_sdp_spectrum_set_colum_data_double_const(nelem, error, array, flux_tell,
					ESPDR_SDP_COLUMN_FLUX_FINAL, sdp_spectrum);
			espdr_sdp_spectrum_set_column_tucd(sdp_spectrum,
					ESPDR_SDP_COLUMN_FLUX_FINAL,
					ESPDR_SDP_COLUMN_FLUX_FINAL_SKYSUB_UCD);

			array_errs_tell = cpl_table_get_array(sdp_spectrum->table,
					ESPDR_SDP_COLUMN_ERR_TELL_CAL_SKYSUB, 0);
			errs_tell  = cpl_array_get_data_double_const(array_errs_tell);
			espdr_sdp_spectrum_set_colum_data_double_const(nelem, error, array, errs_tell,
					ESPDR_SDP_COLUMN_ERR_FINAL, sdp_spectrum);
			espdr_sdp_spectrum_set_column_tucd(sdp_spectrum,
											ESPDR_SDP_COLUMN_ERR_FINAL,
											ESPDR_SDP_COLUMN_ERR_FINAL_SKYSUB_UCD);

			array_qual_tell = cpl_table_get_array(sdp_spectrum->table,
					ESPDR_SDP_COLUMN_QUAL_TELL_CAL_SKYSUB, 0);


			qual_tell  = cpl_array_get_data_int_const(array_qual_tell);
			espdr_sdp_spectrum_set_colum_data_int_const(nelem, error, array, qual_tell,
					ESPDR_SDP_COLUMN_QUAL_FINAL, sdp_spectrum);
			espdr_sdp_spectrum_set_column_tucd(sdp_spectrum,
					ESPDR_SDP_COLUMN_QUAL_FINAL,
					ESPDR_SDP_COLUMN_QUAL_FINAL_SKYSUB_UCD);

		} else {

			array_flux_tell = cpl_table_get_array(sdp_spectrum->table,
					ESPDR_SDP_COLUMN_FLUX_TELL_CAL, 0);
			flux_tell  = cpl_array_get_data_double_const(array_flux_tell);
			espdr_sdp_spectrum_set_colum_data_double_const(nelem, error, array, flux_tell,
					ESPDR_SDP_COLUMN_FLUX_FINAL, sdp_spectrum);
			 espdr_sdp_spectrum_set_column_tucd(sdp_spectrum,
					 ESPDR_SDP_COLUMN_FLUX_FINAL,
					 ESPDR_SDP_COLUMN_FLUX_FINAL_UCD);

			array_errs_tell = cpl_table_get_array(sdp_spectrum->table,
					ESPDR_SDP_COLUMN_ERR_TELL_CAL, 0);
			errs_tell  = cpl_array_get_data_double_const(array_errs_tell);
			espdr_sdp_spectrum_set_colum_data_double_const(nelem, error, array, errs_tell,
					ESPDR_SDP_COLUMN_ERR_FINAL, sdp_spectrum);
			espdr_sdp_spectrum_set_column_tucd(sdp_spectrum,
																	ESPDR_SDP_COLUMN_ERR_FINAL,
																	ESPDR_SDP_COLUMN_ERR_FINAL_UCD);

			array_qual_tell = cpl_table_get_array(sdp_spectrum->table,
					ESPDR_SDP_COLUMN_QUAL_TELL_CAL, 0);
			qual_tell  = cpl_array_get_data_int_const(array_qual_tell);
			espdr_sdp_spectrum_set_colum_data_int_const(nelem, error, array, qual_tell,
					ESPDR_SDP_COLUMN_QUAL_FINAL, sdp_spectrum);
			espdr_sdp_spectrum_set_column_tucd(sdp_spectrum,
								ESPDR_SDP_COLUMN_QUAL_FINAL,
								ESPDR_SDP_COLUMN_QUAL_FINAL_UCD);

		}
	} else {
		if(is_tell_skycorr) {

			//cpl_table_dump_structure(sdp_spectrum->table, stdout);
			array_flux_tell = cpl_table_get_array(sdp_spectrum->table,
					ESPDR_SDP_COLUMN_FLUX_TELL_EL_SKYSUB, 0);

			flux_tell  = cpl_array_get_data_double_const(array_flux_tell);
			espdr_sdp_spectrum_set_colum_data_double_const(nelem, error, array, flux_tell,
					ESPDR_SDP_COLUMN_FLUX_FINAL, sdp_spectrum);

			array_errs_tell = cpl_table_get_array(sdp_spectrum->table,
					ESPDR_SDP_COLUMN_ERR_TELL_EL_SKYSUB, 0);
			errs_tell  = cpl_array_get_data_double_const(array_errs_tell);
			espdr_sdp_spectrum_set_colum_data_double_const(nelem, error, array, errs_tell,
					ESPDR_SDP_COLUMN_ERR_FINAL, sdp_spectrum);

			array_qual_tell = cpl_table_get_array(sdp_spectrum->table,
					ESPDR_SDP_COLUMN_QUAL_TELL_EL_SKYSUB, 0);
			qual_tell  = cpl_array_get_data_int_const(array_qual_tell);
			espdr_sdp_spectrum_set_colum_data_int_const(nelem, error, array, qual_tell,
					ESPDR_SDP_COLUMN_QUAL_FINAL, sdp_spectrum);

		} else {

			array_flux_tell = cpl_table_get_array(sdp_spectrum->table,
					ESPDR_SDP_COLUMN_FLUX_TELL_EL, 0);
			flux_tell  = cpl_array_get_data_double_const(array_flux_tell);
			espdr_sdp_spectrum_set_colum_data_double_const(nelem, error, array, flux_tell,
					ESPDR_SDP_COLUMN_FLUX_FINAL, sdp_spectrum);
			espdr_sdp_spectrum_set_column_tucd(sdp_spectrum,
								 ESPDR_SDP_COLUMN_FLUX_FINAL,
								 ESPDR_SDP_COLUMN_FLUX_FINAL_UCD);

			array_errs_tell = cpl_table_get_array(sdp_spectrum->table,
					ESPDR_SDP_COLUMN_ERR_TELL_EL, 0);
			errs_tell  = cpl_array_get_data_double_const(array_errs_tell);
			espdr_sdp_spectrum_set_colum_data_double_const(nelem, error, array, errs_tell,
					ESPDR_SDP_COLUMN_ERR_FINAL, sdp_spectrum);
			espdr_sdp_spectrum_set_column_tucd(sdp_spectrum,
											   ESPDR_SDP_COLUMN_ERR_FINAL,
											   ESPDR_SDP_COLUMN_ERR_FINAL_UCD);

			array_qual_tell = cpl_table_get_array(sdp_spectrum->table,
					ESPDR_SDP_COLUMN_QUAL_TELL_EL, 0);
			qual_tell  = cpl_array_get_data_int_const(array_qual_tell);
			espdr_sdp_spectrum_set_colum_data_int_const(nelem, error, array, qual_tell,
					ESPDR_SDP_COLUMN_QUAL_FINAL, sdp_spectrum);
			espdr_sdp_spectrum_set_column_tucd(sdp_spectrum,
														   ESPDR_SDP_COLUMN_QUAL_FINAL,
														   ESPDR_SDP_COLUMN_QUAL_FINAL_UCD);

		}
	}
	espdr_check_error_code(cpl_func);
    return cpl_error_get_code();
}

static cpl_error_code
eris_ifu_handle_tellurics(espdr_sdp_spectrum * sdp_spectrum,
		const cpl_frameset * all_frames,
		const cpl_table        * s1d_el_skysub,
		const cpl_table        * s1d_flux_skysub,
		const char* column_flux_cal_unit)
{

	cpl_error_code error = CPL_ERROR_NONE;
	/* handle tellurics */
	const char* fname = NULL;
	cpl_table* tell_tab = NULL;
	cpl_table* tell_fluxcal_tab = NULL;
	double* tell_corr_f = NULL;
	double* tell_corr_e = NULL;
	double* atm_trasm = NULL;

	const double* flux_cal = NULL;
	const double* errs_cal = NULL;
    const int* qual_cal = NULL;

	const double* flux_el = NULL;
	const double* errs_el = NULL;
	const int* qual_el = NULL;

	double* flux_cal_tell = NULL;
	double* errs_cal_tell = NULL;
	int* qual_cal_tell = NULL;

	double* flux_el_tell = NULL;
	double* errs_el_tell = NULL;
	int* qual_el_tell = NULL;

	int* qual_ref = NULL;


	const cpl_frame* frame = NULL;

	const cpl_array* array_atm_trasm = NULL;

	const cpl_array* array_flux_cal = NULL;
	const cpl_array* array_errs_cal = NULL;
	const cpl_array* array_qual_cal = NULL;

	const cpl_array* array_flux_el = NULL;
	const cpl_array* array_errs_el = NULL;
	const cpl_array* array_qual_el = NULL;

	const cpl_array* array_flux_cal_tell = NULL;
	const cpl_array* array_errs_cal_tell = NULL;
	const cpl_array* array_qual_cal_tell = NULL;

	const cpl_array* array_flux_el_tell = NULL;
	const cpl_array* array_errs_el_tell = NULL;
	const cpl_array* array_qual_el_tell = NULL;

	cpl_size nelem;
	cpl_array* array = NULL;

	struct _espdr_column_info* column;
	cpl_boolean is_tell_fluxcal = CPL_FALSE;
	cpl_boolean is_tell_skycorr = CPL_FALSE;
	if(espdr_frameset_has_frame(all_frames, "S1D_TELL_CORR_A") == CPL_TRUE) {
		cpl_msg_info(cpl_func,"add S1D_TELL_CORR_A info");
		/*
		 * 1) compute atm_trasm as atm_trasm[i] =  flux[i] / flux_tell[i];
		 *   where flux is the flux observed from the ground (not corrected by
		 *   tell) and flux_tell is the flux from sky (telluric corrected)
		 *
		 *   1a) in case data are not flux calibrated the fluxes are in electron
		 *
		 *   1c) in case data are flux calibrated the flux is taken from flux
		 *       calibrated data.
		 *       1b) in previois cases if data are sky subtracted one take the
		 *       flux from another frame, the S1D_SKYSUB_A
		 *
		 * 2) Then compute extra columns as current columns multiplied by atm_trans
		 *
		 * 3) change meaning of main FLUX column
		 *
		 * 4) SNR column
		 */

		/* in any case one has S1D_TELL_CORR_A product. Thus one can create the
		 * corresponding columns. We also create the ATM_TRASM column that is
		 * the first to be created
		 */
		espdr_print_rec_status(100);
		cpl_msg_info(cpl_func,"create FLUX_TEL_EL columns");
		struct _espdr_column_info fluxcal_columns[] = {
				{ESPDR_SDP_COLUMN_FLUX_TELL_EL, CPL_TYPE_DOUBLE, ESPDR_SDP_EL_UNIT,
						ESPDR_SDP_COLUMN_FLUX_FORMAT, ESPDR_SDP_COLUMN_FLUX_TELL_EL_TYPE,
						ESPDR_SDP_COLUMN_FLUX_TELL_EL_UCD},
				{ESPDR_SDP_COLUMN_ERR_TELL_EL, CPL_TYPE_DOUBLE, ESPDR_SDP_EL_UNIT,
						ESPDR_SDP_COLUMN_ERR_FORMAT, ESPDR_SDP_COLUMN_ERR_TELL_CAL_TYPE,
						ESPDR_SDP_COLUMN_ERR_TELL_EL_UCD},
				{ESPDR_SDP_COLUMN_QUAL_TELL_EL, CPL_TYPE_INT, ESPDR_SDP_COLUMN_QUAL_UNIT,
						ESPDR_SDP_COLUMN_QUAL_FORMAT, ESPDR_SDP_COLUMN_QUAL_TELL_CAL_TYPE,
						ESPDR_SDP_COLUMN_QUAL_TELL_CAL_EL_UCD},
				{ESPDR_SDP_COLUMN_ATM_TRANSM, CPL_TYPE_DOUBLE, ESPDR_SDP_COLUMN_QUAL_ATM_TRANSM_UNIT,
					    ESPDR_SDP_COLUMN_ERR_FORMAT, ESPDR_SDP_COLUMN_ATM_TRANSM_TYPE,
						ESPDR_SDP_COLUMN_QUAL_ATM_TRANSM_UCD},

				{NULL, CPL_TYPE_INVALID, NULL, NULL, NULL, NULL}
		};
		/* Add the fluxcal spectrum columns to the table. */
		for (column = fluxcal_columns; column->name != NULL; ++column) {

			error = espdr_sdp_spectrum_add_column(sdp_spectrum, column->name,
					column->type, column->unit, column->format,
					column->tutyp, column->tucd, NULL);
			if (error) {
				return cpl_error_set_message(cpl_func, cpl_error_get_code(),
						"Failed to setup the '%s' column.", column->name);
			}
		}

		/* then we fill the ATM_TRASM column. Here we have to distinguish cases:
		 * 1) No fluxcal: 1a) no skysub 1b) with skysub
		 * 2) flucal      2a) no fluxal 2b) with flucal
		 */
		if(espdr_frameset_has_frame(all_frames, "S1D_TELL_CORR_FLUXCAL_A") == CPL_FALSE) {
			/* No fluxal */
			cpl_msg_info(cpl_func,"case NO fluxcal");
			is_tell_fluxcal = CPL_FALSE;
			frame = cpl_frameset_find_const(all_frames, "S1D_TELL_CORR_A");
			fname = cpl_frame_get_filename(frame);
			tell_tab = cpl_table_load(fname, 1, 0);

			tell_corr_f = (double*) cpl_table_get_data_double_const(tell_tab,"flux");
			tell_corr_e = (double*) cpl_table_get_data_double_const(tell_tab,"error");
			qual_ref = (int*) cpl_table_get_data_int_const(tell_tab,"quality");
			nelem = cpl_table_get_nrow(tell_tab);

			cpl_msg_info(cpl_func,"fill FLUX_TEL_EL columns");
			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					tell_corr_f, ESPDR_SDP_COLUMN_FLUX_TELL_EL, sdp_spectrum);

			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					tell_corr_e, ESPDR_SDP_COLUMN_ERR_TELL_EL, sdp_spectrum);

			error |= espdr_sdp_spectrum_set_colum_data_int(nelem, error, array,
					qual_ref, ESPDR_SDP_COLUMN_QUAL_TELL_EL, sdp_spectrum);


			/* then fill the ATM_TRASM column */
			array_atm_trasm = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_ATM_TRANSM, 0);
			atm_trasm  = (double*) cpl_array_get_data_double_const(array_atm_trasm);
			array_flux_el_tell = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_FLUX_TELL_EL, 0);
			flux_el_tell  = (double*) cpl_array_get_data_double_const(array_flux_el_tell);
			array_errs_el_tell = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_ERR_TELL_EL, 0);
			errs_el_tell  = (double*) cpl_array_get_data_double_const(array_errs_el_tell);
			array_qual_el_tell = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_QUAL_TELL_EL, 0);
			qual_el_tell  = (int*) cpl_array_get_data_int_const(array_qual_el_tell);
			/* two cases: */
			if(s1d_el_skysub == NULL) {
				cpl_msg_info(cpl_func,"case NO skysub");
				is_tell_skycorr = CPL_FALSE;
				/* 1) data are not sky subtracted */
				array_flux_el = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_FLUX_EL, 0);
				array_errs_el = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_ERR_EL, 0);
				array_qual_el = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_QUAL_EL, 0);

				flux_el  = cpl_array_get_data_double_const(array_flux_el);
				errs_el  = cpl_array_get_data_double_const(array_errs_el);
				qual_el  = cpl_array_get_data_int_const(array_qual_el);

			} else {
				cpl_msg_info(cpl_func,"case IS skysub");
				is_tell_skycorr = CPL_TRUE;
				/* 2) data are sky subtracted: in this case better to use a
				 *    differebnt product in the definition of the atm_trasm  */
				array_flux_el = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_FLUX_EL_SKYSUB, 0);
				array_errs_el = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_ERR_EL_SKYSUB, 0);
				array_qual_el = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_QUAL_EL_SKYSUB, 0);

				flux_el  = cpl_array_get_data_double_const(array_flux_el);
				errs_el  = cpl_array_get_data_double_const(array_errs_el);
				qual_el  = cpl_array_get_data_int_const(array_qual_el);

			}

			/* this sets ol=nly the atm_trasm values */
			for (cpl_size i = 0; i < nelem; i++) {
				//cpl_msg_info(cpl_func,"i=%lld", i);
			  		    if(isinf(flux_el[i] / flux_el_tell[i])) {
					      atm_trasm[i] = 0.0 / 0.0;
					    } else {
					      atm_trasm[i] =  flux_el[i] / flux_el_tell[i];
					    }
				//qual_el_tell[i] = qual_ref[i];
			}

			/* finally fill the ATM_TRANSM column */
			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					atm_trasm, ESPDR_SDP_COLUMN_ATM_TRANSM, sdp_spectrum);

			/* take again the pointers to columns to get the proper columns */
			array_flux_el = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_FLUX_EL, 0);
			array_errs_el = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_ERR_EL, 0);
			array_qual_el = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_QUAL_EL, 0);

			flux_el  = cpl_array_get_data_double_const(array_flux_el);
			errs_el  = cpl_array_get_data_double_const(array_errs_el);
			qual_el  = cpl_array_get_data_int_const(array_qual_el);

			/* correct for tellurics */
			for (cpl_size i = 0; i < nelem; i++) {
				//cpl_msg_info(cpl_func,"i=%lld", i);
			  if( atm_trasm[i] == 0.0 ) {
				flux_el_tell[i] = 0.0;
				errs_el_tell[i] = 0.0;
			  } else {
				flux_el_tell[i] = flux_el[i] / atm_trasm[i];
				errs_el_tell[i] = errs_el[i] / atm_trasm[i];
			  }
				//qual_el_tell[i] = qual_ref[i];
			}

			cpl_msg_info(cpl_func,"fill FLUX_TEL_EL columns");
			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					flux_el_tell, ESPDR_SDP_COLUMN_FLUX_TELL_EL,
					sdp_spectrum);
			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					errs_el_tell, ESPDR_SDP_COLUMN_ERR_TELL_EL,
					sdp_spectrum);
			// quality extension already filled before, from reference

			/* Point 2) we create the extra columns, only in case of skysub data as
			 * for FP case the extra columns have been already created */
			if(s1d_el_skysub != NULL) {
				cpl_msg_info(cpl_func,"create FLUX_TEL_EL_SKYSUB columns");
				struct _espdr_column_info fluxcal_columns[] = {
						{ESPDR_SDP_COLUMN_FLUX_TELL_EL_SKYSUB, CPL_TYPE_DOUBLE, ESPDR_SDP_EL_UNIT,
								ESPDR_SDP_COLUMN_FLUX_FORMAT, ESPDR_SDP_COLUMN_FLUX_TELL_EL_TYPE,
								ESPDR_SDP_COLUMN_FLUX_TELL_EL_UCD},
						{ESPDR_SDP_COLUMN_ERR_TELL_EL_SKYSUB, CPL_TYPE_DOUBLE, ESPDR_SDP_EL_UNIT,
								ESPDR_SDP_COLUMN_ERR_FORMAT, ESPDR_SDP_COLUMN_ERR_TELL_CAL_TYPE,
								ESPDR_SDP_COLUMN_ERR_TELL_EL_UCD},
						{ESPDR_SDP_COLUMN_QUAL_TELL_EL_SKYSUB, CPL_TYPE_INT, ESPDR_SDP_COLUMN_QUAL_UNIT,
								ESPDR_SDP_COLUMN_QUAL_FORMAT, ESPDR_SDP_COLUMN_QUAL_TELL_CAL_TYPE,
								ESPDR_SDP_COLUMN_QUAL_TELL_CAL_EL_UCD},

						{NULL, CPL_TYPE_INVALID, NULL, NULL, NULL, NULL}
				};
				/* Add the fluxcal spectrum columns to the table. */
				for (column = fluxcal_columns; column->name != NULL; ++column) {

					error = espdr_sdp_spectrum_add_column(sdp_spectrum, column->name,
							column->type, column->unit, column->format,
							column->tutyp, column->tucd, NULL);
					if (error) {
						return cpl_error_set_message(cpl_func, cpl_error_get_code(),
								"Failed to setup the '%s' column.", column->name);
					}
				}


				array_atm_trasm = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_ATM_TRANSM, 0);
				atm_trasm  = (double*) cpl_array_get_data_double_const(array_atm_trasm);

				array_flux_el_tell = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_FLUX_TELL_EL_SKYSUB, 0);
				array_errs_el_tell = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_ERR_TELL_EL_SKYSUB, 0);
				array_qual_el_tell = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_QUAL_TELL_EL_SKYSUB, 0);

				flux_el_tell  = (double*) cpl_array_get_data_double_const(array_flux_el_tell);
				errs_el_tell  = (double*) cpl_array_get_data_double_const(array_errs_el_tell);
				qual_el_tell  = (int*) cpl_array_get_data_int_const(array_qual_el_tell);

				array_flux_el = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_FLUX_EL_SKYSUB, 0);
				array_errs_el = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_ERR_EL_SKYSUB, 0);
				array_qual_el = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_QUAL_EL_SKYSUB, 0);

				flux_el  = cpl_array_get_data_double_const(array_flux_el);
				errs_el  = cpl_array_get_data_double_const(array_errs_el);
				qual_el  = cpl_array_get_data_int_const(array_qual_el);

				for (cpl_size i = 0; i < nelem; i++) {
					//cpl_msg_info(cpl_func,"i=%lld", i);
				    if( atm_trasm[i] == 0.0 ) {
				        flux_el_tell[i] = 0.0;
				        errs_el_tell[i] = 0.0;
					qual_el_tell[i] = qual_ref[i];
			            } else {
					flux_el_tell[i] = flux_el[i] / atm_trasm[i];
					errs_el_tell[i] = errs_el[i] / atm_trasm[i];
					qual_el_tell[i] = qual_ref[i];
				    }
				}
				cpl_msg_info(cpl_func,"fill FLUX_TEL_EL_SKYSUB columns");
				error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
						flux_el_tell, ESPDR_SDP_COLUMN_FLUX_TELL_EL_SKYSUB,
						sdp_spectrum);
				error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
						errs_el_tell, ESPDR_SDP_COLUMN_ERR_TELL_EL_SKYSUB,
						sdp_spectrum);
				error |= espdr_sdp_spectrum_set_colum_data_int(nelem, error, array,
						qual_ref, ESPDR_SDP_COLUMN_QUAL_TELL_EL_SKYSUB,
						sdp_spectrum);
			}
			cpl_table_delete(tell_tab);

		} else {
			/* fluxcal */
			cpl_msg_info(cpl_func,"case IS fluxcal");
			is_tell_fluxcal = CPL_TRUE;

			/* first fill the TELL_EL columns */
			frame = cpl_frameset_find_const(all_frames, "S1D_TELL_CORR_A");
			fname = cpl_frame_get_filename(frame);
			tell_tab = cpl_table_load(fname, 1, 0);

			tell_corr_f = (double*) cpl_table_get_data_double_const(tell_tab,"flux");
			tell_corr_e = (double*) cpl_table_get_data_double_const(tell_tab,"error");
			qual_ref = (int*) cpl_table_get_data_int_const(tell_tab,"quality");
			nelem = cpl_table_get_nrow(tell_tab);

			cpl_msg_info(cpl_func,"fluxcal fill FLUX_TEL_EL columns");
			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					tell_corr_f, ESPDR_SDP_COLUMN_FLUX_TELL_EL, sdp_spectrum);

			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					tell_corr_e, ESPDR_SDP_COLUMN_ERR_TELL_EL, sdp_spectrum);

			error |= espdr_sdp_spectrum_set_colum_data_int(nelem, error, array,
					qual_ref, ESPDR_SDP_COLUMN_QUAL_TELL_EL, sdp_spectrum);

            /* use different table extention to keep open the other one */
			frame = cpl_frameset_find_const(all_frames, "S1D_TELL_CORR_FLUXCAL_A");
			fname = cpl_frame_get_filename(frame);
			tell_fluxcal_tab = cpl_table_load(fname, 1, 0);

			tell_corr_f = (double*) cpl_table_get_data_double_const(tell_fluxcal_tab,"flux_cal");
			tell_corr_e = (double*) cpl_table_get_data_double_const(tell_fluxcal_tab,"error_cal");

			nelem = cpl_table_get_nrow(tell_fluxcal_tab);

			/* we need to create the extra columns. ATM_TRASM is already there */
			cpl_msg_info(cpl_func,"create FLUX_TEL_CAL columns");
			struct _espdr_column_info fluxcal_columns[] = {
					{ESPDR_SDP_COLUMN_FLUX_TELL_CAL, CPL_TYPE_DOUBLE, column_flux_cal_unit,
							ESPDR_SDP_COLUMN_FLUX_FORMAT, ESPDR_SDP_COLUMN_FLUX_TELL_CAL_TYPE,
							ESPDR_SDP_COLUMN_FLUX_TELL_CAL_UCD},
					{ESPDR_SDP_COLUMN_ERR_TELL_CAL, CPL_TYPE_DOUBLE, column_flux_cal_unit,
							ESPDR_SDP_COLUMN_ERR_FORMAT, ESPDR_SDP_COLUMN_ERR_TELL_CAL_TYPE,
			    			ESPDR_SDP_COLUMN_ERR_TELL_CAL_UCD},
					{ESPDR_SDP_COLUMN_QUAL_TELL_CAL, CPL_TYPE_INT, ESPDR_SDP_COLUMN_QUAL_UNIT,
							ESPDR_SDP_COLUMN_QUAL_FORMAT, ESPDR_SDP_COLUMN_QUAL_TELL_CAL_TYPE,
							ESPDR_SDP_COLUMN_QUAL_TELL_CAL_UCD},

					{NULL, CPL_TYPE_INVALID, NULL, NULL, NULL, NULL}
			};
			/* Add the fluxcal spectrum columns to the table. */
			for (column = fluxcal_columns; column->name != NULL; ++column) {

				error = espdr_sdp_spectrum_add_column(sdp_spectrum, column->name,
						column->type, column->unit, column->format,
						column->tutyp, column->tucd, NULL);
				if (error) {
					return cpl_error_set_message(cpl_func, cpl_error_get_code(),
							"Failed to setup the '%s' column.", column->name);
				}
			}

			cpl_msg_info(cpl_func,"fill FLUX_TEL_CAL columns");
			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					tell_corr_f, ESPDR_SDP_COLUMN_FLUX_TELL_CAL, sdp_spectrum);

			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					tell_corr_e, ESPDR_SDP_COLUMN_ERR_TELL_CAL, sdp_spectrum);

			// quality extension already filled before from reference
			espdr_print_rec_status(520);
			cpl_table_delete(tell_fluxcal_tab);


			error |= espdr_sdp_spectrum_set_colum_data_int(nelem, error, array,
								qual_ref, ESPDR_SDP_COLUMN_QUAL_TELL_CAL, sdp_spectrum);


			/* then fill the ATM_TRASM column */
			array_atm_trasm = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_ATM_TRANSM, 0);
			atm_trasm  = (double*) cpl_array_get_data_double_const(array_atm_trasm);


			array_flux_el = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_FLUX_EL, 0);
			array_errs_el = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_ERR_EL, 0);
			array_qual_el = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_QUAL_EL, 0);

			flux_el  = cpl_array_get_data_double_const(array_flux_el);
			errs_el  = cpl_array_get_data_double_const(array_errs_el);
			qual_el  = cpl_array_get_data_int_const(array_qual_el);

			array_flux_el_tell = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_FLUX_TELL_EL, 0);
			array_errs_el_tell = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_ERR_TELL_EL, 0);
			array_qual_el_tell = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_QUAL_TELL_EL, 0);

			flux_el_tell  = (double*) cpl_array_get_data_double_const(array_flux_el_tell);
			errs_el_tell  = (double*) cpl_array_get_data_double_const(array_errs_el_tell);
			qual_el_tell  = (int*) cpl_array_get_data_int_const(array_qual_el_tell);

			array_flux_cal_tell = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_FLUX_TELL_CAL, 0);
			array_errs_cal_tell = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_ERR_TELL_CAL, 0);
			array_qual_cal_tell = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_QUAL_TELL_CAL, 0);

			flux_cal_tell  = (double*) cpl_array_get_data_double_const(array_flux_cal_tell);
			errs_cal_tell  = (double*) cpl_array_get_data_double_const(array_errs_cal_tell);
			qual_cal_tell  = (int*) cpl_array_get_data_int_const(array_qual_cal_tell);



			espdr_print_rec_status(530);
			/* two cases: */
			if(s1d_flux_skysub == NULL) {
				/* 1) data are not sky subtracted */
				cpl_msg_info(cpl_func,"case NO skysub");
				is_tell_skycorr = CPL_FALSE;
				array_flux_cal = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_FLUX_CAL, 0);
				array_errs_cal = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_ERR_CAL, 0);
				array_qual_cal = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_QUAL_CAL, 0);

				flux_cal  = cpl_array_get_data_double_const(array_flux_cal);
				errs_cal  = cpl_array_get_data_double_const(array_errs_cal);
				qual_cal  = cpl_array_get_data_int_const(array_qual_cal);

			} else {
				/* 2) data are sky subtracted: in this case better to use a
				 *    different product in the definition of the atm_trasm  */
				cpl_msg_info(cpl_func,"case IS skysub");
				is_tell_skycorr = CPL_TRUE;
				array_flux_cal = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_FLUX_CAL_SKYSUB, 0);
				array_errs_cal = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_ERR_CAL_SKYSUB, 0);
				array_qual_cal = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_QUAL_CAL_SKYSUB, 0);

				flux_cal  = cpl_array_get_data_double_const(array_flux_cal);
				errs_cal  = cpl_array_get_data_double_const(array_errs_cal);
				qual_cal  = cpl_array_get_data_int_const(array_qual_cal);

			}

			espdr_print_rec_status(540);
			for (cpl_size i = 0; i < nelem; i++) {
			    if(isinf(flux_cal[i] / flux_cal_tell[i])) {
			         atm_trasm[i] = 0. / 0.;
			      }  else {
			         atm_trasm[i] =  flux_cal[i] / flux_cal_tell[i];
			      }
			      if( atm_trasm[i] == 0.0 ) {
				flux_el_tell[i] = 0.0;
				errs_el_tell[i] = 0.0;
				qual_el_tell[i] = qual_ref[i];
			      } else {
				flux_el_tell[i] = flux_el[i] / atm_trasm[i];
				errs_el_tell[i] = errs_el[i] / atm_trasm[i];
				qual_el_tell[i] = qual_ref[i];
			      }
			}


			/* create/update relevant columns: special case for skycor/not */
			array_flux_cal = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_FLUX_CAL, 0);
			array_errs_cal = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_ERR_CAL, 0);
			array_qual_cal = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_QUAL_CAL, 0);

			flux_cal  = cpl_array_get_data_double_const(array_flux_cal);
			errs_cal  = cpl_array_get_data_double_const(array_errs_cal);
			qual_cal  = cpl_array_get_data_int_const(array_qual_cal);

			for (cpl_size i = 0; i < nelem; i++) {
			    if( atm_trasm[i] == 0.0 ) {
				flux_cal_tell[i] = 0.0;
				errs_cal_tell[i] = 0.0;
				qual_cal_tell[i] = qual_ref[i];
			    } else {
				flux_cal_tell[i] = flux_cal[i] / atm_trasm[i];
				errs_cal_tell[i] = errs_cal[i] / atm_trasm[i];
				qual_cal_tell[i] = qual_ref[i];
			    }
			}
			espdr_print_rec_status(550);
			/* finally fill the ATM_TRANSM column */
			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					atm_trasm, ESPDR_SDP_COLUMN_ATM_TRANSM, sdp_spectrum);
			espdr_print_rec_status(560);

			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					flux_el_tell, ESPDR_SDP_COLUMN_FLUX_TELL_EL, sdp_spectrum);
			espdr_print_rec_status(570);
			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					errs_el_tell, ESPDR_SDP_COLUMN_ERR_TELL_EL, sdp_spectrum);
			espdr_print_rec_status(580);


			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					flux_cal_tell, ESPDR_SDP_COLUMN_FLUX_TELL_CAL, sdp_spectrum);
			espdr_print_rec_status(670);
			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					errs_cal_tell, ESPDR_SDP_COLUMN_ERR_TELL_CAL, sdp_spectrum);
			espdr_print_rec_status(680);

			error |= espdr_sdp_spectrum_set_colum_data_int(nelem, error, array,
					qual_ref, ESPDR_SDP_COLUMN_QUAL_TELL_CAL, sdp_spectrum);

			espdr_print_rec_status(690);

		}
		espdr_print_rec_status(700);
		/* Point 2) we create the extra columns, only in case of skysub data as
		 * for FP case the extra columns have been already created */
		if(s1d_flux_skysub != NULL) {
			cpl_msg_info(cpl_func,"creates FLUX_TEL_EL_SKYSUB columns");
			struct _espdr_column_info fluxel_skysub_columns[] = {
								{ESPDR_SDP_COLUMN_FLUX_TELL_EL_SKYSUB, CPL_TYPE_DOUBLE, ESPDR_SDP_EL_UNIT,
										ESPDR_SDP_COLUMN_FLUX_FORMAT, ESPDR_SDP_COLUMN_FLUX_TELL_EL_TYPE,
										ESPDR_SDP_COLUMN_FLUX_TELL_EL_UCD},
								{ESPDR_SDP_COLUMN_ERR_TELL_EL_SKYSUB, CPL_TYPE_DOUBLE, ESPDR_SDP_EL_UNIT,
										ESPDR_SDP_COLUMN_ERR_FORMAT, ESPDR_SDP_COLUMN_ERR_TELL_EL_TYPE,
										ESPDR_SDP_COLUMN_ERR_TELL_EL_UCD},
								{ESPDR_SDP_COLUMN_QUAL_TELL_EL_SKYSUB, CPL_TYPE_INT, ESPDR_SDP_COLUMN_QUAL_UNIT,
										ESPDR_SDP_COLUMN_QUAL_FORMAT, ESPDR_SDP_COLUMN_QUAL_TELL_EL_TYPE,
										ESPDR_SDP_COLUMN_QUAL_TELL_CAL_UCD},

								{NULL, CPL_TYPE_INVALID, NULL, NULL, NULL, NULL}
						};
			/* Add the fluxcal spectrum columns to the table. */
			for (column = fluxel_skysub_columns; column->name != NULL; ++column) {

				error = espdr_sdp_spectrum_add_column(sdp_spectrum, column->name,
						column->type, column->unit, column->format,
						column->tutyp, column->tucd, NULL);
				if (error) {
					return cpl_error_set_message(cpl_func, cpl_error_get_code(),
							"Failed to setup the '%s' column.", column->name);
				}
			}

			cpl_msg_info(cpl_func,"creates FLUX_TEL_CAL_SKYSUB columns");
			struct _espdr_column_info fluxcal_skysub_columns[] = {
					{ESPDR_SDP_COLUMN_FLUX_TELL_CAL_SKYSUB, CPL_TYPE_DOUBLE, column_flux_cal_unit,
							ESPDR_SDP_COLUMN_FLUX_FORMAT, ESPDR_SDP_COLUMN_FLUX_TELL_CAL_TYPE,
							ESPDR_SDP_COLUMN_FLUX_TELL_CAL_SKYSUB_UCD},
					{ESPDR_SDP_COLUMN_ERR_TELL_CAL_SKYSUB, CPL_TYPE_DOUBLE, column_flux_cal_unit,
							ESPDR_SDP_COLUMN_ERR_FORMAT, ESPDR_SDP_COLUMN_ERR_TELL_CAL_TYPE,
							ESPDR_SDP_COLUMN_ERR_TELL_CAL_SKYSUB_UCD},
					{ESPDR_SDP_COLUMN_QUAL_TELL_CAL_SKYSUB, CPL_TYPE_INT, ESPDR_SDP_COLUMN_QUAL_UNIT,
							ESPDR_SDP_COLUMN_QUAL_FORMAT, ESPDR_SDP_COLUMN_QUAL_TELL_CAL_TYPE,
							ESPDR_SDP_COLUMN_QUAL_TELL_CAL_SKYSUB_UCD},

					{NULL, CPL_TYPE_INVALID, NULL, NULL, NULL, NULL}
			};
			/* Add the fluxcal spectrum columns to the table. */
			for (column = fluxcal_skysub_columns; column->name != NULL; ++column) {

				error = espdr_sdp_spectrum_add_column(sdp_spectrum, column->name,
						column->type, column->unit, column->format,
						column->tutyp, column->tucd, NULL);
				if (error) {
					return cpl_error_set_message(cpl_func, cpl_error_get_code(),
							"Failed to setup the '%s' column.", column->name);
				}
			}


			array_atm_trasm = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_ATM_TRANSM, 0);
			atm_trasm  = (double*) cpl_array_get_data_double_const(array_atm_trasm);


			array_flux_cal_tell = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_FLUX_TELL_CAL_SKYSUB, 0);
			array_errs_cal_tell = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_ERR_TELL_CAL_SKYSUB, 0);
			array_qual_cal_tell = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_QUAL_TELL_CAL_SKYSUB, 0);

			flux_cal_tell  = (double*) cpl_array_get_data_double_const(array_flux_cal_tell);
			errs_cal_tell  = (double*) cpl_array_get_data_double_const(array_errs_cal_tell);
			qual_cal_tell  = (int*) cpl_array_get_data_int_const(array_qual_cal_tell);


			array_flux_el_tell = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_FLUX_TELL_EL_SKYSUB, 0);
			array_errs_el_tell = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_ERR_TELL_EL_SKYSUB, 0);
			array_qual_el_tell = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_QUAL_TELL_EL_SKYSUB, 0);

			flux_el_tell  = (double*) cpl_array_get_data_double_const(array_flux_el_tell);
			errs_el_tell  = (double*) cpl_array_get_data_double_const(array_errs_el_tell);
			qual_el_tell  = (int*) cpl_array_get_data_int_const(array_qual_el_tell);

			array_flux_cal = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_FLUX_CAL_SKYSUB, 0);
			array_errs_cal = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_ERR_CAL_SKYSUB, 0);
			array_qual_cal = cpl_table_get_array(sdp_spectrum->table, ESPDR_SDP_COLUMN_QUAL_CAL_SKYSUB, 0);

			flux_cal  = cpl_array_get_data_double_const(array_flux_cal);
			errs_cal  = cpl_array_get_data_double_const(array_errs_cal);
			qual_cal  = cpl_array_get_data_int_const(array_qual_cal);

			for (cpl_size i = 0; i < nelem; i++) {
				//cpl_msg_info(cpl_func,"i=%lld", i);
			    if( atm_trasm[i] == 0.0 ) {
				flux_cal_tell[i] = 0.0;
				errs_cal_tell[i] = 0.0;
				qual_cal_tell[i] = qual_ref[i];

				flux_el_tell[i] = 0.0;
				errs_el_tell[i] = 0.0;
				qual_el_tell[i] = qual_ref[i];
			    } else {

				flux_cal_tell[i] = flux_cal[i] / atm_trasm[i];
				errs_cal_tell[i] = errs_cal[i] / atm_trasm[i];
				qual_cal_tell[i] = qual_ref[i];

				flux_el_tell[i] = flux_el[i] / atm_trasm[i];
				errs_el_tell[i] = errs_el[i] / atm_trasm[i];
				qual_el_tell[i] = qual_ref[i];
			    }
				//cpl_msg_info(cpl_func,"i=%lld qual: %d", i,qual_cal_tell[i]);
			}
			cpl_msg_info(cpl_func,"fill FLUX_TEL_EL_SKYSUB columns");
			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					flux_el_tell, ESPDR_SDP_COLUMN_FLUX_TELL_EL_SKYSUB,
					sdp_spectrum);
			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					errs_el_tell, ESPDR_SDP_COLUMN_ERR_TELL_EL_SKYSUB,
					sdp_spectrum);

			error |= espdr_sdp_spectrum_set_colum_data_int(nelem, error, array,
					qual_ref, ESPDR_SDP_COLUMN_QUAL_TELL_EL_SKYSUB,
					sdp_spectrum);



			cpl_msg_info(cpl_func,"fill FLUX_TEL_CAL_SKYSUB columns");
			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					flux_cal_tell, ESPDR_SDP_COLUMN_FLUX_TELL_CAL_SKYSUB,
					sdp_spectrum);
			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					errs_cal_tell, ESPDR_SDP_COLUMN_ERR_TELL_CAL_SKYSUB,
					sdp_spectrum);


			error |= espdr_sdp_spectrum_set_colum_data_int(nelem, error, array,
					qual_ref, ESPDR_SDP_COLUMN_QUAL_TELL_CAL_SKYSUB,
					sdp_spectrum);

			cpl_table_delete(tell_tab);

		}
		espdr_print_rec_status(800);
	} /* end tell cor handling */
	espdr_print_rec_status(900);
	/* adjust final column to most processed information */

    espdr_set_final_cols_for_tell_case(sdp_spectrum, nelem, is_tell_fluxcal,
    		is_tell_skycorr);

    espdr_check_error_code(cpl_func);
	return cpl_error_get_code();
}

/**
 * @brief Converts data into Science Data Product 1D spectrum format.
 *
 * @param[out] sdp_spectrum The SDP output spectrum object the will be filled.
 * @param[in] used_frames List of frames used to generate the output spectrum.
 * @param[in] sourcekeys The list of FITS keywords from the source product file
 *  being converted to SDP format.
 * @param[in] spectrum The loaded flux calibrated spectrum in X-Shooter format.
 * @param[in] uncal_spectrum The loaded uncalibrated spectrum in X-Shooter
 *  format.
 *
 * @note If the @a table and @a props objects are not empty, then the spectrum
 *  data and keywords will be appended to the existing values in those objects
 *  instead.
 */

static cpl_error_code
espdr_sdp_spectrum_merged_create2(
		espdr_sdp_spectrum    * sdp_spectrum,
		const cpl_frameset     * all_frames,
		const cpl_frameset     * used_frames,
		const cpl_propertylist * sourcekeys,
		const cpl_table        * s1d_el_skysub,
		const cpl_table        * s1d_flux_skysub,
		const cpl_table        * s1d_el,
		const cpl_table        * s1d_flux
		)
{

	cpl_errorstate prestate = cpl_errorstate_get();
	cpl_error_code error = CPL_ERROR_NONE;

	cpl_ensure_code(sdp_spectrum != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(all_frames != NULL,   CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(used_frames != NULL,   CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(sourcekeys != NULL,   CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(s1d_el != NULL,   CPL_ERROR_NULL_INPUT);

	//cpl_ensure_code(s1d_flux != NULL,   CPL_ERROR_NULL_INPUT);

	if (! cpl_errorstate_is_equal(prestate)) {
		return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT,
				"Missing required input.");
	}

	/* From here we start filling the spectrum header  */
	if(s1d_flux != NULL ) {
		if(s1d_flux_skysub != NULL ) {
			error |= espdr_sdp_spectrum_prepare_header(s1d_flux_skysub,
					sdp_spectrum, sourcekeys, used_frames, CPL_TRUE);
		} else {
			error |= espdr_sdp_spectrum_prepare_header(s1d_flux, sdp_spectrum,
					sourcekeys, used_frames, CPL_TRUE);
		}
	} else {
		if(s1d_el_skysub != NULL ) {
			error |= espdr_sdp_spectrum_prepare_header(s1d_el_skysub, sdp_spectrum,
					sourcekeys, used_frames, CPL_FALSE);
		} else {
			error |= espdr_sdp_spectrum_prepare_header(s1d_el, sdp_spectrum,
					sourcekeys, used_frames, CPL_FALSE);
		}
	}

	const char* column_wave_air_unit = NULL;
	const char* column_wave_final_unit = NULL;

	const char* column_flux_final_unit = NULL;
	const char* column_errs_final_unit = NULL;


	const char* column_flux_unit = NULL;
	const char* column_errs_unit = NULL;
	const char* column_qual_unit = NULL;

	const char* column_flux_cal_unit = NULL;
	const char* column_errs_cal_unit = NULL;
	const char* column_sky_unit = NULL;

	const cpl_table* s1d_final = NULL;

	/* assign proper columns according to possible cases */
	if( s1d_flux != NULL ) {
		if(s1d_flux_skysub != NULL ) {
			s1d_final = s1d_flux_skysub;
			espdr_msg("Flux: skysub flux calibrated");
		} else {
			s1d_final = s1d_flux;
			espdr_msg("Flux: flux calibrated");
		}
		column_flux_final_unit = ESPDR_SDP_FLUX_UNIT;
		column_errs_final_unit = ESPDR_SDP_FLUX_UNIT;
	} else {
		if(s1d_el_skysub != NULL ) {
			espdr_msg("Flux skysub in e-");
		    s1d_final = s1d_el_skysub;
		} else {
			espdr_msg("Flux in e-");
			s1d_final = s1d_el;
		}
		column_flux_final_unit = ESPDR_SDP_EL_UNIT;
		column_errs_final_unit = ESPDR_SDP_EL_UNIT;
	}

	//TODO: Note that the column names should be set by a common define
	column_wave_final_unit = cpl_table_get_column_unit(s1d_final,"wavelength");

	//wavelength_air is only in s1d_el which is always present
	column_wave_air_unit = cpl_table_get_column_unit(s1d_el,"wavelength_air");


	column_flux_unit = cpl_table_get_column_unit(s1d_el,"flux");
	column_errs_unit = cpl_table_get_column_unit(s1d_el,"error");
	column_qual_unit = cpl_table_get_column_unit(s1d_el,"quality");

	if(s1d_flux != NULL) {
		column_flux_cal_unit = cpl_table_get_column_unit(s1d_flux,"flux_cal");
		column_errs_cal_unit = cpl_table_get_column_unit(s1d_flux,"error_cal");
	}

	if(s1d_el_skysub != NULL) {
		column_sky_unit = cpl_table_get_column_unit(s1d_el_skysub,"flux");
	}

	if (
		(column_flux_final_unit == NULL && column_errs_final_unit == NULL) &&
		(column_flux_unit == NULL && column_errs_unit == NULL) &&
		(column_flux_cal_unit == NULL && column_errs_cal_unit == NULL)
	) {
		return cpl_error_get_code();
	}

	/* Build a list of columns that we should setup in the spectrum table. */
	struct _espdr_column_info columns[] = {
			{ESPDR_SDP_COLUMN_WAVE, CPL_TYPE_DOUBLE, column_wave_final_unit,
					ESPDR_SDP_COLUMN_WAVE_FORMAT, ESPDR_SDP_COLUMN_WAVE_TYPE,
					ESPDR_SDP_COLUMN_WAVE_UCD},

			{ESPDR_SDP_COLUMN_FLUX_FINAL, CPL_TYPE_DOUBLE, column_flux_final_unit,
					ESPDR_SDP_COLUMN_FLUX_FORMAT, ESPDR_SDP_COLUMN_FLUX_FINAL_TYPE,
					ESPDR_SDP_COLUMN_FLUX_SKYSUB_FINAL_UCD},
			{ESPDR_SDP_COLUMN_ERR_FINAL, CPL_TYPE_DOUBLE, column_errs_final_unit,
					ESPDR_SDP_COLUMN_ERR_FORMAT, ESPDR_SDP_COLUMN_ERR_FINAL_TYPE,
					ESPDR_SDP_COLUMN_ERR_FINAL_SKYSUB_UCD},
			{ESPDR_SDP_COLUMN_QUAL_FINAL, CPL_TYPE_INT, ESPDR_SDP_COLUMN_QUAL_UNIT,
					ESPDR_SDP_COLUMN_QUAL_FORMAT, ESPDR_SDP_COLUMN_QUAL_FINAL_TYPE,
					ESPDR_SDP_COLUMN_QUAL_FINAL_SKYSUB_UCD},

			{ESPDR_SDP_COLUMN_SNR, CPL_TYPE_DOUBLE, ESPDR_SDP_COLUMN_SNR_UNIT,
					ESPDR_SDP_COLUMN_SNR_FORMAT, ESPDR_SDP_COLUMN_SNR_TYPE,
					ESPDR_SDP_COLUMN_SNR_UCD},
			{ESPDR_SDP_COLUMN_WAVE_AIR, CPL_TYPE_DOUBLE, column_wave_air_unit,
					ESPDR_SDP_COLUMN_WAVE_FORMAT, ESPDR_SDP_COLUMN_WAVE_AIR_TYPE,
					ESPDR_SDP_COLUMN_WAVE_AIR_UCD},

			{ESPDR_SDP_COLUMN_FLUX_EL, CPL_TYPE_DOUBLE, ESPDR_SDP_EL_UNIT,
					ESPDR_SDP_COLUMN_FLUX_FORMAT, ESPDR_SDP_COLUMN_FLUX_EL_TYPE,
					ESPDR_SDP_COLUMN_FLUX_EL_UCD},
				//PIPPO
			{ESPDR_SDP_COLUMN_ERR_EL, CPL_TYPE_DOUBLE, ESPDR_SDP_EL_UNIT,
					ESPDR_SDP_COLUMN_ERR_FORMAT, ESPDR_SDP_COLUMN_ERR_EL_TYPE,
					ESPDR_SDP_COLUMN_ERR_EL_UCD},
			{ESPDR_SDP_COLUMN_QUAL_EL, CPL_TYPE_INT, ESPDR_SDP_COLUMN_QUAL_UNIT,
					ESPDR_SDP_COLUMN_QUAL_FORMAT, ESPDR_SDP_COLUMN_QUAL_EL_TYPE,
					ESPDR_SDP_COLUMN_QUAL_EL_UCD},

			{NULL, CPL_TYPE_INVALID, NULL, NULL, NULL, NULL}
	};

	struct _espdr_column_info* column;
	/* Add the spectrum columns to the table. */
	for (column = columns; column->name != NULL; ++column) {
		error = espdr_sdp_spectrum_add_column(sdp_spectrum, column->name,
				column->type, column->unit, column->format,
				column->tutyp, column->tucd, NULL);
		if (error) {
			return cpl_error_set_message(cpl_func, cpl_error_get_code(),
					"Failed to setup the '%s' column.", column->name);
		}
	}

	if(s1d_flux != NULL) {
		struct _espdr_column_info fluxcal_columns[] = {
				{ESPDR_SDP_COLUMN_FLUX_CAL, CPL_TYPE_DOUBLE, column_flux_cal_unit,
						ESPDR_SDP_COLUMN_FLUX_FORMAT, ESPDR_SDP_COLUMN_FLUX_CAL_TYPE,
						ESPDR_SDP_COLUMN_FLUX_CAL_UCD},
				{ESPDR_SDP_COLUMN_ERR_CAL, CPL_TYPE_DOUBLE, column_errs_cal_unit,
						ESPDR_SDP_COLUMN_ERR_FORMAT, ESPDR_SDP_COLUMN_ERR_CAL_TYPE,
						ESPDR_SDP_COLUMN_ERR_CAL_UCD},
				{ESPDR_SDP_COLUMN_QUAL_CAL, CPL_TYPE_INT, ESPDR_SDP_COLUMN_QUAL_UNIT,
						ESPDR_SDP_COLUMN_QUAL_FORMAT, ESPDR_SDP_COLUMN_QUAL_CAL_TYPE,
						ESPDR_SDP_COLUMN_QUAL_CAL_UCD},
				{NULL, CPL_TYPE_INVALID, NULL, NULL, NULL, NULL}
		};

		/* Add the fluxcal spectrum columns to the table. */
		for (column = fluxcal_columns; column->name != NULL; ++column) {

			error = espdr_sdp_spectrum_add_column(sdp_spectrum, column->name,
					column->type, column->unit, column->format,
					column->tutyp, column->tucd, NULL);
			if (error) {
				return cpl_error_set_message(cpl_func, cpl_error_get_code(),
						"Failed to setup the '%s' column.", column->name);
			}
		}
	}

	/* FLUXCAL SKYSUB */
	if(s1d_flux_skysub != NULL) {
		struct _espdr_column_info fluxcal_columns[] = {
				{ESPDR_SDP_COLUMN_FLUX_CAL_SKYSUB, CPL_TYPE_DOUBLE, column_flux_cal_unit,
						ESPDR_SDP_COLUMN_FLUX_FORMAT, ESPDR_SDP_COLUMN_FLUX_CAL_SKYSUB_TYPE,
						ESPDR_SDP_COLUMN_FLUX_CAL_SKYSUB_UCD},
				{ESPDR_SDP_COLUMN_ERR_CAL_SKYSUB, CPL_TYPE_DOUBLE, column_errs_cal_unit,
						ESPDR_SDP_COLUMN_ERR_FORMAT, ESPDR_SDP_COLUMN_ERR_CAL_SKYSUB_TYPE,
						ESPDR_SDP_COLUMN_ERR_CAL_SKYSUB_UCD},
				{ESPDR_SDP_COLUMN_QUAL_CAL_SKYSUB, CPL_TYPE_INT, ESPDR_SDP_COLUMN_QUAL_UNIT,
						ESPDR_SDP_COLUMN_QUAL_FORMAT, ESPDR_SDP_COLUMN_QUAL_CAL_SKYSUB_TYPE,
						ESPDR_SDP_COLUMN_QUAL_CAL_SKYSUB_UCD},
				{NULL, CPL_TYPE_INVALID, NULL, NULL, NULL, NULL}
		};

		/* Add the fluxcal spectrum columns to the table. */
		for (column = fluxcal_columns; column->name != NULL; ++column) {

			error = espdr_sdp_spectrum_add_column(sdp_spectrum, column->name,
					column->type, column->unit, column->format,
					column->tutyp, column->tucd, NULL);
			if (error) {
				return cpl_error_set_message(cpl_func, cpl_error_get_code(),
						"Failed to setup the '%s' column.", column->name);
			}
		}

	}

	if(s1d_el_skysub != NULL) {
        //PIPPO
		struct _espdr_column_info sky_columns[] = {
				{ESPDR_SDP_COLUMN_FLUX_EL_SKYSUB, CPL_TYPE_DOUBLE, ESPDR_SDP_EL_UNIT,
						ESPDR_SDP_COLUMN_FLUX_FORMAT, ESPDR_SDP_COLUMN_FLUX_CAL_TYPE,
						ESPDR_SDP_COLUMN_FLUX_EL_SKYSUB_UCD},
				{ESPDR_SDP_COLUMN_ERR_EL_SKYSUB, CPL_TYPE_DOUBLE, ESPDR_SDP_EL_UNIT,
						ESPDR_SDP_COLUMN_ERR_FORMAT, ESPDR_SDP_COLUMN_ERR_EL_SKYSUB_TYPE,
						ESPDR_SDP_COLUMN_ERR_EL_SKYSUB_UCD},
				{ESPDR_SDP_COLUMN_QUAL_EL_SKYSUB, CPL_TYPE_INT, ESPDR_SDP_COLUMN_QUAL_UNIT,
						ESPDR_SDP_COLUMN_QUAL_FORMAT, ESPDR_SDP_COLUMN_QUAL_EL_SKYSUB_TYPE,
						ESPDR_SDP_COLUMN_QUAL_EL_SKYSUB_UCD},
				{NULL, CPL_TYPE_INVALID, NULL, NULL, NULL, NULL}
		};

		/* Add the sky spectrum columns to the table. */
		for (column = sky_columns; column->name != NULL; ++column) {

			error = espdr_sdp_spectrum_add_column(sdp_spectrum, column->name,
					column->type, column->unit, column->format,
					column->tutyp, column->tucd, NULL);

			if (error) {
				return cpl_error_set_message(cpl_func, cpl_error_get_code(),
						"Failed to setup the '%s' column.", column->name);
			}
		}

	}

	/* Finally copy the table data. */
	cpl_size nelem = (cpl_size) cpl_table_get_nrow(s1d_el);
	double* specwave = NULL;
	double* specwave_air = NULL;

	double* specflux = NULL;
	double* specerrs = NULL;
	int* specqual = NULL;

	double* specflux_cal = NULL;
	double* specerrs_cal = NULL;

	double* specflux_el = NULL;
	double* specerrs_el = NULL;
	int* specqual_el = NULL;

	double* skyflux = NULL;
	double* skyerrs = NULL;
	int* skyqual = NULL;

	double* skyflux_cal = NULL;
	double* skyerrs_cal = NULL;
	int* skyqual_cal = NULL;

    specwave = (double*) cpl_table_get_data_double_const(s1d_final,"wavelength");
    specwave_air = (double*) cpl_table_get_data_double_const(s1d_el,"wavelength_air");

	if( s1d_flux != NULL ) {
        specflux = (double*) cpl_table_get_data_double_const(s1d_final,"flux_cal");
        specerrs = (double*) cpl_table_get_data_double_const(s1d_final,"error_cal");
	} else {
        specflux = (double*) cpl_table_get_data_double_const(s1d_final,"flux");
        specerrs = (double*) cpl_table_get_data_double_const(s1d_final,"error");
	}
	/* quality info is always only in s1d_el */
    specqual = (int*) cpl_table_get_data_int_const(s1d_el,"quality");

    specflux_el = (double*) cpl_table_get_data_double_const(s1d_el,"flux");
    specerrs_el = (double*) cpl_table_get_data_double_const(s1d_el,"error");
    specqual_el = (int*) cpl_table_get_data_int_const(s1d_el,"quality");
	if(s1d_flux != NULL) {
        specflux_cal = (double*) cpl_table_get_data_double_const(s1d_flux,"flux_cal");
        specerrs_cal = (double*) cpl_table_get_data_double_const(s1d_flux,"error_cal");
	}

	if(s1d_el_skysub != NULL) {
        skyflux = (double*) cpl_table_get_data_double_const(s1d_el_skysub,"flux");
        skyerrs = (double*) cpl_table_get_data_double_const(s1d_el_skysub,"error");
        skyqual = (int*) cpl_table_get_data_int_const(s1d_el_skysub,"quality");
	}

	if(s1d_flux_skysub != NULL) {
        skyflux_cal = (double*) cpl_table_get_data_double_const(s1d_flux_skysub,"flux_cal");
        skyerrs_cal = (double*) cpl_table_get_data_double_const(s1d_flux_skysub,"error_cal");
        skyqual_cal = (int*) cpl_table_get_data_int_const(s1d_el,"quality");
	}

	cpl_array* array = NULL;
	/* Prepare the wavelength array and set it. */
	error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
			specwave, ESPDR_SDP_COLUMN_WAVE, sdp_spectrum);

	if (specflux != NULL && specerrs != NULL && specqual != NULL) {

		/* Prepare the other data arrays and assign them. */
		error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
				specflux, ESPDR_SDP_COLUMN_FLUX_FINAL, sdp_spectrum);
		if(s1d_flux_skysub == NULL) {
			espdr_sdp_spectrum_set_column_tucd(sdp_spectrum,
					ESPDR_SDP_COLUMN_FLUX_FINAL,
					ESPDR_SDP_COLUMN_FLUX_FINAL_UCD);
		}

		error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
				specerrs, ESPDR_SDP_COLUMN_ERR_FINAL, sdp_spectrum);

		if(s1d_flux_skysub == NULL) {
			espdr_sdp_spectrum_set_column_tucd(sdp_spectrum,
					ESPDR_SDP_COLUMN_ERR_FINAL,
					ESPDR_SDP_COLUMN_ERR_FINAL_UCD);
		}

		error |= espdr_sdp_spectrum_set_colum_data_int(nelem, error, array,
				specqual, ESPDR_SDP_COLUMN_QUAL_FINAL, sdp_spectrum);

		if(s1d_flux_skysub == NULL) {
			espdr_sdp_spectrum_set_column_tucd(sdp_spectrum,
					ESPDR_SDP_COLUMN_QUAL_FINAL,
					ESPDR_SDP_COLUMN_QUAL_FINAL_UCD);
		}

		error |= espdr_sdp_spectrum_compute_snr(specflux, specerrs,
							specqual, nelem,
							sdp_spectrum, &error);

	}

	if (specflux_el != NULL && specerrs_el != NULL && specqual_el != NULL) {
		error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
				specflux_el, ESPDR_SDP_COLUMN_FLUX_EL, sdp_spectrum);

		error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
				specerrs_el, ESPDR_SDP_COLUMN_ERR_EL, sdp_spectrum);

		error |= espdr_sdp_spectrum_set_colum_data_int(nelem, error, array,
				specqual_el, ESPDR_SDP_COLUMN_QUAL_EL, sdp_spectrum);
	}

	if (specwave_air != NULL) {
		error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
				specwave_air, ESPDR_SDP_COLUMN_WAVE_AIR, sdp_spectrum);
	}

	if (specflux_cal != NULL && specerrs_cal != NULL) {
		error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
				specflux_cal, ESPDR_SDP_COLUMN_FLUX_CAL, sdp_spectrum);
		error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
				specerrs_cal, ESPDR_SDP_COLUMN_ERR_CAL, sdp_spectrum);
		error |= espdr_sdp_spectrum_set_colum_data_int(nelem, error, array,
						specqual_el, ESPDR_SDP_COLUMN_QUAL_CAL, sdp_spectrum);
	}

	if(s1d_el_skysub != NULL) {

		if (skyflux != NULL) {
			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					skyflux, ESPDR_SDP_COLUMN_FLUX_EL_SKYSUB, sdp_spectrum);
			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					skyerrs, ESPDR_SDP_COLUMN_ERR_EL_SKYSUB, sdp_spectrum);
			error |= espdr_sdp_spectrum_set_colum_data_int(nelem, error, array,
					skyqual, ESPDR_SDP_COLUMN_QUAL_EL_SKYSUB, sdp_spectrum);
		}

	}

	if(s1d_flux_skysub != NULL) {

		if (skyflux_cal != NULL) {
			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					skyflux_cal, ESPDR_SDP_COLUMN_FLUX_CAL_SKYSUB, sdp_spectrum);
			error |= espdr_sdp_spectrum_set_colum_data_double(nelem, error, array,
					skyerrs_cal, ESPDR_SDP_COLUMN_ERR_CAL_SKYSUB, sdp_spectrum);
			error |= espdr_sdp_spectrum_set_colum_data_int(nelem, error, array,
					skyqual_cal, ESPDR_SDP_COLUMN_QUAL_CAL_SKYSUB, sdp_spectrum);
		}

	}

	if(cpl_frameset_find_const(all_frames, "S1D_TELL_CORR_A") != NULL){

	    eris_ifu_handle_tellurics(sdp_spectrum, all_frames, s1d_el_skysub,
		    	      s1d_flux_skysub, column_flux_cal_unit);
	}

	espdr_check_error_code(cpl_func);
	return cpl_error_get_code();
}


/**
 * @brief Creates a 1D spectrum product in the Science Data Product format.
 *
 * This function converts a 1D spectrum written to a fits files in the X-Shooter
 * native format to the Science Data Product (SDP) format. The spectrum data is
 * loaded from the @a flux_cal_frame and @c uncal_frame, converted to the SDP
 * format and written to a new file with the same name as the original product,
 * but with the "SDP_" prefix added. Either the flux calibrated frame or the
 * uncalibrated frame must be provided or both.
 * The caller should check the value of @c cpl_error_get_code() to see if an
 * error occurred during the conversion.
 *
 * @param flux_cal_frame  The flux calibrated 1D spectrum frame that we convert
 *      to SDP format. Can be NULL if none is available, but only if the
 *      @c uncal_frame is not NULL.
 * @param uncal_frame  The uncalibrated 1D spectrum that we also use in the
 *      conversion. Can be NULL if none is available, but only if the
 *      @c flux_cal_frame is not NULL.
 * @param frameset  The list of all frames to which we add the newly generated
 *      frame.
 * @param used_frames  A frame set containing a list of all frames used to
 *      product the spectrum in the flux_cal_frame.
 * @param parameters  The parameters passed to the recipe by esorex.
 * @param recipe_id  The ID string of the recipe.
 * @param instrument  The X-Shooter instrument structure containing
 *      configuration information about the instrument.
 */
cpl_error_code
espdr_add_sdp_product_spectrum2(const cpl_frame *product_frame,
                                  cpl_frameset *frameset,
                                  const cpl_frameset *used_frames,
                                  const cpl_parameterlist *parameters,
                                  const char *recipe_id,
								  cpl_boolean is_fluxcal)
{

  cpl_propertylist* tablekeys = cpl_propertylist_new();
  cpl_propertylist* primarykeys = cpl_propertylist_new();
  cpl_propertylist* sourcekeys = NULL;
  char* output_procatg = NULL;
  char filename[80];
  cpl_table              * s1d_table =NULL;

  cpl_ensure_code(frameset != NULL, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(used_frames != NULL,   CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(parameters != NULL,   CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(recipe_id != NULL,   CPL_ERROR_NULL_INPUT);

  if (product_frame == NULL) {
    cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT,"Provide input frame");
    goto cleanup;
  }

  const char* fname = cpl_frame_get_filename(product_frame);
  const char* pcatg = cpl_frame_get_tag(product_frame);
  cpl_boolean has_header = CPL_FALSE;
  cpl_frame* hframe = NULL;
   if(NULL != cpl_frameset_find(frameset, "HEADER")) {
 	  has_header=CPL_TRUE;

 	  cpl_frameset_erase(used_frames, "HEADER");

   }

  output_procatg = cpl_sprintf("S1D_FINAL_B");
  s1d_table = cpl_table_load(fname,1,0);
  sourcekeys = cpl_propertylist_load(fname, 0);
  const char* instrume = cpl_propertylist_get_string(sourcekeys,"INSTRUME");
  sprintf(filename,"%s_%s.fits",instrume,output_procatg);
  cpl_free(output_procatg);
  /* Copy all existing QC keywords from the product we are converting. */
  cpl_propertylist_copy_property_regexp(primarykeys, sourcekeys, "^ESO QC ",
		  CPL_FALSE);

  /* Need to copy the ESO.QC.FLUX.SN keyword from uncal_frame.
  if (uncal_frame != NULL) {
    if (! cpl_propertylist_has(sourcekeys, ESPDR_QC_FLUX_SN)) {
      cpl_propertylist* tmpkeys;
      tmpkeys = cpl_propertylist_load(cpl_frame_get_filename(uncal_frame), 0);
      assure(tmpkeys != NULL, cpl_error_get_code(),
             "%s", cpl_error_get_message());
      if (cpl_propertylist_has(tmpkeys, ESPDR_QC_FLUX_SN)) {
        cpl_propertylist_copy_property(sourcekeys, tmpkeys, ESPDR_QC_FLUX_SN);
      }
      // Again copy all QC keywords from the non-flux calibrated product.
      cpl_propertylist_copy_property_regexp(primarykeys, tmpkeys,
                                            "^ESO QC ", CPL_FALSE);
      cpl_propertylist_delete(tmpkeys);
      assure(cpl_error_get_code() == CPL_ERROR_NONE, cpl_error_get_code(),
             "%s", cpl_error_get_message());
    }
  }
  */

  /* NOT NEEDED FOR ESPRESSO
   * Force the INSTRUME keyword to be "ESPRESSO" since old data can have a
   * different value for this keyword.
  cpl_propertylist_update_string(primarykeys, ESPDR_INSTRUME,
                                       ESPDR_INSTRUME_VALUE);
  const char* inst_comment = cpl_propertylist_get_comment(primarykeys,
                                                          ESPDR_INSTRUME);
  if (inst_comment == NULL || strcmp(inst_comment, "") == 0) {
    check(cpl_propertylist_set_comment(primarykeys, ESPDR_INSTRUME,
                                       "Instrument used."));
  }
  */

  if (pcatg != NULL) {
    output_procatg = cpl_sprintf("S1D_FINAL_B");
    cpl_propertylist_update_string(primarykeys, CPL_DFS_PRO_CATG,
                                         output_procatg);
  } else {
    cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                          "Could not identify the correct ARM value to be used"
                          " for '%s'.", CPL_DFS_PRO_CATG);
    espdr_msg_error("%s", cpl_error_get_message());
    goto cleanup;
  }

  /* Calculate MJD-END by finding the largest MJD-OBS value from the raw input
   * files. */
  double mjdend;
  mjdend = espdr_calculate_mjd_end(product_frame, used_frames);
  cpl_propertylist_update_double(sourcekeys, ESPDR_MJDEND, mjdend);

  /* Load the raw spectra and fill the SDP spectrum object.
  if (flux_cal_frame != NULL) {
    check(spectrum = espdr_spectrum_load((cpl_frame*)flux_cal_frame));
  }
  if (uncal_frame != NULL) {
    check(uncal_spectrum = espdr_spectrum_load((cpl_frame*)uncal_frame));
  }
  */
  cpl_errorstate prestate = cpl_errorstate_get();
  espdr_sdp_spectrum    * sdp_spectrum = espdr_sdp_spectrum_new();
  espdr_sdp_spectrum_create2(sdp_spectrum, frameset, used_frames, sourcekeys,
                             s1d_table, is_fluxcal);
  espdr_sdp_spectrum_set_prodcatg(sdp_spectrum, ESPDR_ANCILLARY_SPECTRUM);
    cpl_table_delete(s1d_table);

  if (! cpl_errorstate_is_equal(prestate)) {
    espdr_msg_error("%s", cpl_error_get_message());
    goto cleanup;
  }

  espdr_fill_provenance_keywords(sdp_spectrum, used_frames);

  /* Create dummy association keywords. */
  //int assoc_count;
  //const cpl_parameter* p;
  //char paramname[256];
  /* TODO: parameter to be added to recipe
  sprintf(paramname,"espdr.%s.%s",recipe_id,"dummy-association-keys");
  p = cpl_parameterlist_find_const(parameters,paramname);
  assoc_count = cpl_parameter_get_int(p);
  */
  /*
  assoc_count=1;

  int assoi;
  for (assoi = 1; assoi <= assoc_count; ++assoi) {
    espdr_sdp_spectrum_set_asson(sdp_spectrum, assoi, " ");
    espdr_sdp_spectrum_set_assoc(sdp_spectrum, assoi, " ");
    espdr_sdp_spectrum_set_assom(sdp_spectrum, assoi, " ");
  }
  */
  /* for FITS files writes only asson
  espdr_sdp_spectrum_set_asson(sdp_spectrum, 1, "S2D_A.fits");
  espdr_sdp_spectrum_set_asson(sdp_spectrum, 2, "S2D_BLAZE_A.fits");
  espdr_sdp_spectrum_set_asson(sdp_spectrum, 3, "DRIF_MATRIX_B.fits");
  espdr_sdp_spectrum_set_asson(sdp_spectrum, 4, "CCF_A.fits");
  espdr_sdp_spectrum_set_asson(sdp_spectrum, 5, "S1D_FINAL_B.fits");
  espdr_sdp_spectrum_set_asson(sdp_spectrum, 6, "S2D_A_SKYSUB .fits");
  */
  cpl_msg_info(cpl_func, "Writing SDP spectrum as FITS table product(%s): %s",
               output_procatg, filename);

  /* Save the filled SDP spectrum to file, but strip the checksum keywords. */
  const char* remregexp = "^(CHECKSUM|DATASUM)$";

  if(has_header) {

	  //cpl_frameset* tmp_set = cpl_frameset_duplicate(frameset);
	  //cpl_frameset_erase(tmp_set, "HEADER");
	  cpl_frameset_erase(used_frames, "HEADER");
	  espdr_dfs_save_spectrum(frameset, NULL, parameters, used_frames, product_frame,
			  sdp_spectrum, recipe_id, primarykeys, NULL, remregexp,
			  PACKAGE "/" PACKAGE_VERSION, "PRO-1.15", filename, has_header);

	  //cpl_frame* pro_frm = cpl_frameset_find(tmp_set,output_procatg);
	  //cpl_frameset_insert(frameset,cpl_frame_duplicate(pro_frm));

	  //cpl_frameset_delete(tmp_set);
  } else {

	  espdr_dfs_save_spectrum(frameset, NULL, parameters, used_frames, product_frame,
			  sdp_spectrum, recipe_id, primarykeys, NULL, remregexp,
			  PACKAGE "/" PACKAGE_VERSION, "PRO-1.15", filename, has_header);


  }




  /* AMO: is this really needed?
  espdr_fits_update_checksums(filename);
  */



cleanup:
  espdr_sdp_spectrum_delete(sdp_spectrum);

  cpl_free(output_procatg);
  cpl_propertylist_delete(sourcekeys);
  cpl_propertylist_delete(primarykeys);
  cpl_propertylist_delete(tablekeys);
  espdr_check_error_code(cpl_func);
  return cpl_error_get_code();
}


/**
 * @brief Creates a 1D spectrum product in the Science Data Product format.
 *
 * This function converts a 1D spectrum written to a fits files in the X-Shooter
 * native format to the Science Data Product (SDP) format. The spectrum data is
 * loaded from the @a flux_cal_frame and @c uncal_frame, converted to the SDP
 * format and written to a new file with the same name as the original product,
 * but with the "SDP_" prefix added. Either the flux calibrated frame or the
 * uncalibrated frame must be provided or both.
 * The caller should check the value of @c cpl_error_get_code() to see if an
 * error occurred during the conversion.
 *
 * @param flux_cal_frame  The flux calibrated 1D spectrum frame that we convert
 *      to SDP format. Can be NULL if none is available, but only if the
 *      @c uncal_frame is not NULL.
 * @param uncal_frame  The uncalibrated 1D spectrum that we also use in the
 *      conversion. Can be NULL if none is available, but only if the
 *      @c flux_cal_frame is not NULL.
 * @param frameset  The list of all frames to which we add the newly generated
 *      frame.
 * @param used_frames  A frame set containing a list of all frames used to
 *      product the spectrum in the flux_cal_frame.
 * @param parameters  The parameters passed to the recipe by esorex.
 * @param recipe_id  The ID string of the recipe.
 * @param instrument  The X-Shooter instrument structure containing
 *      configuration information about the instrument.
 */
cpl_error_code
espdr_add_sdp_product_merged_spectrum2(const cpl_frame *product_frame,
                                       cpl_frameset *frameset,
                                       const cpl_frameset *used_frames,
                                       const cpl_parameterlist *parameters,
                                       const char *recipe_id,
                                       const cpl_table* s1d_fluxcal,
                                       const cpl_table* s1d_fluxcal_skysub,
                                       const cpl_table* s1d_el_skysub,
                                       const cpl_boolean has_ccf)
{

  cpl_propertylist* tablekeys = cpl_propertylist_new();
  cpl_propertylist* primarykeys = cpl_propertylist_new();
  cpl_propertylist* sourcekeys = NULL;
  char* output_procatg = NULL;
  char filename[80];
  cpl_table              * s1d_el =NULL;

  cpl_ensure_code(frameset != NULL, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(used_frames != NULL,   CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(parameters != NULL,   CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(recipe_id != NULL,   CPL_ERROR_NULL_INPUT);

  if (product_frame == NULL) {
    cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT,"Provide input frame");
    goto cleanup;
  }
  const char* fname = NULL;
  const char* pcatg = NULL;

  cpl_boolean has_header = CPL_FALSE;
  cpl_frame* hframe = NULL;
  cpl_frame* rframe = NULL;
  cpl_propertylist* plist_add = NULL;
  cpl_propertylist* plist_raw = NULL;
  if(NULL != cpl_frameset_find(frameset, "HEADER")) {
	  hframe = cpl_frameset_find(frameset, "HEADER");
	  fname = cpl_frame_get_filename(hframe);
	  plist_add = cpl_propertylist_load(fname, 0);
	  if(NULL != cpl_frameset_find(frameset, "OBJ_FP")) {
		  rframe = cpl_frameset_find(frameset, "OBJ_FP");
		  fname = cpl_frame_get_filename(rframe);
		  plist_raw = cpl_propertylist_load(fname, 0);
	  } else if(NULL != cpl_frameset_find(frameset, "OBJ_SKY")) {
		  rframe = cpl_frameset_find(frameset, "OBJ_SKY");
		  fname = cpl_frame_get_filename(rframe);
		  plist_raw = cpl_propertylist_load(fname, 0);
	  } else if(NULL != cpl_frameset_find(frameset, "OBJ_DARK")) {
		  rframe = cpl_frameset_find(frameset, "OBJ_DARK");
		  fname = cpl_frame_get_filename(rframe);
		  plist_raw = cpl_propertylist_load(fname, 0);
	  } else if(NULL != cpl_frameset_find(frameset, "OBJ_THAR")) {
		  rframe = cpl_frameset_find(frameset, "OBJ_THAR");
		  fname = cpl_frame_get_filename(rframe);
		  plist_raw = cpl_propertylist_load(fname, 0);
          } else if(NULL != cpl_frameset_find(frameset, "OBJ_LFC")) {
		  rframe = cpl_frameset_find(frameset, "OBJ_LFC");
		  fname = cpl_frame_get_filename(rframe);
		  plist_raw = cpl_propertylist_load(fname, 0);
	  }
	  has_header=CPL_TRUE;
  } 
  if(NULL != cpl_frameset_find(used_frames, "HEADER")) {
	  has_header=CPL_TRUE;
	  cpl_frameset_erase(used_frames, "HEADER");
  }
  fname = cpl_frame_get_filename(product_frame);
  pcatg = cpl_frame_get_tag(product_frame);

  output_procatg = cpl_sprintf("S1D_FINAL_A");
  s1d_el = cpl_table_load(fname,1,0);
  sourcekeys = cpl_propertylist_load(fname, 0);
  const char* instrume = cpl_propertylist_get_string(sourcekeys,"INSTRUME");
  sprintf(filename,"%s_%s.fits",instrume,output_procatg);
  cpl_free(output_procatg);
  if(has_header) {
	  //espdr_plist_purge_list(plist_add, plist_raw);
	  //espdr_plist_overwrite_list(plist_add, plist_raw);
  }
  cpl_propertylist_delete(plist_raw);

  /* Copy all existing QC keywords from the product we are converting. */
  cpl_propertylist_copy_property_regexp(primarykeys, sourcekeys, "^ESO QC ",
		  CPL_FALSE);

  /* Need to copy the ESO.QC.FLUX.SN keyword from uncal_frame.
  if (uncal_frame != NULL) {
    if (! cpl_propertylist_has(sourcekeys, ESPDR_QC_FLUX_SN)) {
      cpl_propertylist* tmpkeys;
      tmpkeys = cpl_propertylist_load(cpl_frame_get_filename(uncal_frame), 0);
      assure(tmpkeys != NULL, cpl_error_get_code(),
             "%s", cpl_error_get_message());
      if (cpl_propertylist_has(tmpkeys, ESPDR_QC_FLUX_SN)) {
        cpl_propertylist_copy_property(sourcekeys, tmpkeys, ESPDR_QC_FLUX_SN);
      }
      // Again copy all QC keywords from the non-flux calibrated product.
      cpl_propertylist_copy_property_regexp(primarykeys, tmpkeys,
                                            "^ESO QC ", CPL_FALSE);
      cpl_propertylist_delete(tmpkeys);
      assure(cpl_error_get_code() == CPL_ERROR_NONE, cpl_error_get_code(),
             "%s", cpl_error_get_message());
    }
  }
  */

  /* NOT NEEDED FOR ESPRESSO
   * Force the INSTRUME keyword to be "ESPRESSO" since old data can have a
   * different value for this keyword.
  cpl_propertylist_update_string(primarykeys, ESPDR_INSTRUME,
                                       ESPDR_INSTRUME_VALUE);
  const char* inst_comment = cpl_propertylist_get_comment(primarykeys,
                                                          ESPDR_INSTRUME);
  if (inst_comment == NULL || strcmp(inst_comment, "") == 0) {
    check(cpl_propertylist_set_comment(primarykeys, ESPDR_INSTRUME,
                                       "Instrument used."));
  }
  */

  if (pcatg != NULL) {
    output_procatg = cpl_sprintf("S1D_FINAL_A");
    cpl_propertylist_update_string(primarykeys, CPL_DFS_PRO_CATG,
                                         output_procatg);
  } else {
    cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                          "Could not identify the correct ARM value to be used"
                          " for '%s'.", CPL_DFS_PRO_CATG);
    espdr_msg_error("%s", cpl_error_get_message());
    goto cleanup;
  }

  /* Calculate MJD-END by finding the largest MJD-OBS value from the raw input
   * files. */
  double mjdend;
  mjdend = espdr_calculate_mjd_end(product_frame, used_frames);
  cpl_propertylist_update_double(sourcekeys, ESPDR_MJDEND, mjdend);

  /* Load the raw spectra and fill the SDP spectrum object.
  if (flux_cal_frame != NULL) {
    check(spectrum = espdr_spectrum_load((cpl_frame*)flux_cal_frame));
  }
  if (uncal_frame != NULL) {
    check(uncal_spectrum = espdr_spectrum_load((cpl_frame*)uncal_frame));
  }
  */
  cpl_errorstate prestate = cpl_errorstate_get();
  espdr_sdp_spectrum    * sdp_spectrum = espdr_sdp_spectrum_new();

  espdr_sdp_spectrum_merged_create2(sdp_spectrum, frameset, used_frames,
		sourcekeys, s1d_el_skysub, s1d_fluxcal_skysub,  s1d_el, s1d_fluxcal);

  if (! cpl_errorstate_is_equal(prestate)) {
    espdr_msg_error("%s", cpl_error_get_message());
    goto cleanup;
  }

  espdr_fill_provenance_keywords(sdp_spectrum, used_frames);

  /* Create dummy association keywords. */
  //int assoc_count;
  //const cpl_parameter* p;
  //char paramname[256];
  /* TODO: parameter to be added to recipe
  sprintf(paramname,"espdr.%s.%s",recipe_id,"dummy-association-keys");
  p = cpl_parameterlist_find_const(parameters,paramname);
  assoc_count = cpl_parameter_get_int(p);
  */
    /*
  assoc_count=1;

  int assoi;
  for (assoi = 1; assoi <= assoc_count; ++assoi) {
    espdr_sdp_spectrum_set_asson(sdp_spectrum, assoi, " ");
    espdr_sdp_spectrum_set_assoc(sdp_spectrum, assoi, " ");
    espdr_sdp_spectrum_set_assom(sdp_spectrum, assoi, " ");
  }
  */
  int counter = 0;
  char asson_value[40];
  sprintf(asson_value,"%s_S2D_A.fits", instrume);
  espdr_sdp_spectrum_set_asson(sdp_spectrum, ++counter, asson_value);
  sprintf(asson_value,"%s_S2D_BLAZE_A.fits", instrume);
  espdr_sdp_spectrum_set_asson(sdp_spectrum,++counter, asson_value);
  sprintf(asson_value,"%s_DRIFT_MATRIX_B.fits", instrume);
  espdr_sdp_spectrum_set_asson(sdp_spectrum, ++counter, asson_value);
  sprintf(asson_value,"%s_S1D_FINAL_B.fits", instrume);
  espdr_sdp_spectrum_set_asson(sdp_spectrum, ++counter, asson_value);
  if(has_ccf) {
    sprintf(asson_value,"%s_CCF_A.fits", instrume);
    espdr_sdp_spectrum_set_asson(sdp_spectrum, ++counter, asson_value);
  }
  
  if(s1d_fluxcal_skysub) {
      sprintf(asson_value,"%s_S2D_A_SKYSUB.fits", instrume);
    espdr_sdp_spectrum_set_asson(sdp_spectrum, ++counter, asson_value);
  }
  if(espdr_frameset_has_frame(frameset, "S1D_TELL_CORR_FLUXCAL_A") == CPL_TRUE) {
 	   sprintf(asson_value,"%s_S1D_TELL_CORR_FLUXCAL_A.fits", instrume);
 	    espdr_sdp_spectrum_set_asson(sdp_spectrum, ++counter, asson_value);
  }

  if(espdr_frameset_has_frame(frameset, "S1D_TELL_CORR_A") == CPL_TRUE) {
	  sprintf(asson_value,"%s_S1D_TELL_CORR_A.fits", instrume);
	  espdr_sdp_spectrum_set_asson(sdp_spectrum, ++counter, asson_value);
	  /* update CCF QC keywords if required */
	  cpl_frame* frame = cpl_frameset_find(frameset,"S1D_TELL_CORR_A");
	  const char* fname = cpl_frame_get_filename(frame);
	  cpl_propertylist* plist = cpl_propertylist_load(fname,0);
	  int qc_tell_check = cpl_propertylist_get_int(plist,"ESO QC TELL CHECK");
	  if(qc_tell_check==1) {
		  cpl_msg_info(cpl_func,"update CCF QC keywprds");
		  double ccf_rv = cpl_propertylist_get_double(plist, "ESO QC CCF RV");
		  double ccf_rv_err = cpl_propertylist_get_double(plist, "ESO QC CCF RV ERROR");
		  double ccf_fwhm = cpl_propertylist_get_double(plist, "ESO QC CCF FWHM");
		  double ccf_fwhm_err = cpl_propertylist_get_double(plist, "ESO QC CCF FWHM ERROR");
		  double ccf_contrast = cpl_propertylist_get_double(plist, "ESO QC CCF CONTRAST");
		  double ccf_contrast_err = cpl_propertylist_get_double(plist, "ESO QC CCF CONTRAST ERROR");
		  double ccf_continuum = cpl_propertylist_get_double(plist, "ESO QC CCF CONTINUUM");
		  
		  double ccf_flux_asymmetry = cpl_propertylist_get_double(plist, "ESO QC CCF FLUX ASYMMETRY");
		  double ccf_flux_asymmetry_err = cpl_propertylist_get_double(plist, "ESO QC CCF FLUX ASYMMETRY ERROR");
		 
		  double ccf_bis_span = cpl_propertylist_get_double(plist, "ESO QC CCF BIS SPAN");
		  double ccf_bis_span_err = cpl_propertylist_get_double(plist, "ESO QC CCF BIS SPAN ERROR");

		  cpl_propertylist_update_double(primarykeys, "ESO QC CCF RV", ccf_rv);
		  cpl_propertylist_update_double(primarykeys, "ESO QC CCF RV ERROR", ccf_rv_err);
		  cpl_propertylist_update_double(primarykeys, "ESO QC CCF FWHM", ccf_fwhm);
		  cpl_propertylist_update_double(primarykeys, "ESO QC CCF FWHM ERROR", ccf_fwhm_err);
		  cpl_propertylist_update_double(primarykeys, "ESO QC CCF CONTRAST", ccf_contrast);
	      cpl_propertylist_update_double(primarykeys, "ESO QC CCF CONTRAST ERROR", ccf_contrast_err);
		  cpl_propertylist_update_double(primarykeys, "ESO QC CCF CONTINUUM", ccf_continuum);
		  cpl_propertylist_update_double(primarykeys, "ESO QC CCF FLUX ASYMMETRY", ccf_flux_asymmetry);
		  cpl_propertylist_update_double(primarykeys, "ESO QC CCF FLUX ASYMMETRY ERROR", ccf_flux_asymmetry_err);
		  cpl_propertylist_update_double(primarykeys, "ESO QC CCF BIS SPAN", ccf_bis_span);
		  cpl_propertylist_update_double(primarykeys, "ESO QC CCF BIS SPAN ERROR", ccf_bis_span_err);

	  }
	  cpl_propertylist_delete(plist);



  }

  if(espdr_frameset_has_frame(frameset, "S2D_TELL_SPECTRUM_A") == CPL_TRUE) {
 	   sprintf(asson_value,"%s_S2D_TELL_SPECRUM_A.fits", instrume);
 	    espdr_sdp_spectrum_set_asson(sdp_spectrum, ++counter, asson_value);
   }

  cpl_msg_info(cpl_func, "Writing SDP spectrum as FITS table product(%s): %s",
               output_procatg, filename);

  /* Save the filled SDP spectrum to file, but strip the checksum keywords. */
  const char* remregexp = "^(CHECKSUM|DATASUM)$";

  if(has_header) {

	  //cpl_frameset* tmp_set = cpl_frameset_duplicate(frameset);
	  //cpl_frameset_erase(tmp_set, "HEADER");

	  espdr_dfs_save_spectrum(frameset, NULL, parameters, used_frames, product_frame,
			  sdp_spectrum, recipe_id, primarykeys, NULL, remregexp,
			  PACKAGE "/" PACKAGE_VERSION, "PRO-1.15", filename, has_header);
	  //cpl_frameset_dump(tmp_set,stdout);
	  //cpl_frame* pro_frm = cpl_frameset_find(tmp_set,output_procatg);
	  //cpl_frameset_insert(frameset,cpl_frame_duplicate(pro_frm));
	  //cpl_frameset_delete(tmp_set);
  } else {

	  espdr_dfs_save_spectrum(frameset, NULL, parameters, used_frames, product_frame,
			  sdp_spectrum, recipe_id, primarykeys, NULL, remregexp,
			  PACKAGE "/" PACKAGE_VERSION, "PRO-1.15", filename, has_header);

  }


  /* AMO: is this really needed?
  espdr_fits_update_checksums(filename);
  */



cleanup:
  cpl_table_delete(s1d_el);
  espdr_sdp_spectrum_delete(sdp_spectrum);

  cpl_free(output_procatg);
  cpl_propertylist_delete(sourcekeys);
  cpl_propertylist_delete(primarykeys);
  cpl_propertylist_delete(tablekeys);
  espdr_check_error_code(cpl_func);
  return cpl_error_get_code();
}



#ifdef ESPDR_USE_FITS_UPDATE_CHECKSUM

/**
 * @brief Updates the FITS standard CHECKSUM and DATASUM keywords.
 * @param filename  The name of the FITS file to update.
 * @return a CPL error code indicating success or failure.
 *
 * @note To use this function one needs to declare the preprocessor macro
 *   @c ESPDR_USE_FITS_UPDATE_CHECKSUM when compiling the @c libirplib.a
 *   library or including the header files. Also, additional linker flags to
 *   the cfitsio library need to be added when any binaries are linked with
 *   @c libirplib.a.
 */
cpl_error_code espdr_fits_update_checksums(const char* filename)
{
  fitsfile* filehandle;
  int error = 0; /* must be initialised to zero before call to cfitsio. */

  if (fits_open_diskfile(&filehandle, filename, READWRITE, &error)) {
    return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_IO,
                "Could not open file '%s' to update CHECKSUM keywords"
                " (error = %d).", filename, error);
  }


  int i = 0;
  while (! fits_movabs_hdu(filehandle, ++i, NULL, &error)) {
    if (fits_write_chksum(filehandle, &error)) {
      return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_IO,
                  "Could not update the CHECKSUM keywords in '%s' HDU %d"
                  " (error = %d).", filename, i, error);
    }
  }
  /* Reset after normal error */
  if (error == END_OF_FILE) error = 0;

  if (fits_close_file(filehandle, &error)) {
    return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_IO,
                "There was a problem trying to close the file '%s'"
                " (error = %d).", filename, error);
    espdr_check_error_code(cpl_func);
  }
  espdr_check_error_code(cpl_func);
  return CPL_ERROR_NONE;
}

#endif /* ESPDR_USE_FITS_UPDATE_CHECKSUM */
