/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: asegovia $
 * $Date: 2014-10-28 15:13:41 $
 * $Revision: $
 * $Name: not supported by cvs2svn $
 */


#include <espdr_blaze.h>
#include <espdr_overscan.h>
#include <espdr_fit.h>

#include <gsl/gsl_errno.h>
#include <gsl/gsl_spline.h>
#include <gsl/gsl_histogram.h>
#include <gsl/gsl_multifit.h>
#include <gsl/gsl_sort.h>

/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/
/**
 @brief measures the blaze in one spectral order by applying a sliding average.
 @param spectrum_order  order to be treated.
 @param weight		weights values to be applied. 
 @param size_slidingbox sliding box size.
 @param blaze_RE	Calculated blaze function.
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_measure_blaze(cpl_image *spectrum_order, 
                                   double *weight,
                                   int size_slidingbox,
                                   cpl_image **blaze_order_RE) {

	int i, j, imin, imax, subsize, nx;
	double flux, w, blaze = 1.0;
	double *blaze_order, *spectrum_order_data;	

	cpl_image *spectrum_order_copy;
	spectrum_order_copy = cpl_image_duplicate(spectrum_order); 	

	nx = cpl_image_get_size_x(spectrum_order_copy);

	blaze_order = cpl_image_get_data_double(*blaze_order_RE); 

	spectrum_order_data = cpl_image_get_data_double(spectrum_order_copy);
	
	imin = 0;
    while (imin < nx && spectrum_order_data[imin] == 0.) imin++;

	imax = nx-1;
    while ( imax >= 0 && spectrum_order_data[imax] == 0.) imax--;

    for (i = imin; i < imin+size_slidingbox && i < nx; i++) {
		subsize = i-imin+1;
		flux = 0.;
		w = 0.;
		for (j = imin; j < i+subsize; j++) {
			flux += spectrum_order_data[j]*weight[j];
			w += weight[j];
		}
		blaze = (w > 0.) ? (flux/w) : blaze;
		blaze_order[i] = blaze;
	}
	
    for (i = imin+size_slidingbox; i <= imax-size_slidingbox && i < nx; i++) {
		flux = 0.;
		w = 0.;
		for (j = i-size_slidingbox; j <= i+size_slidingbox; j++) {
			flux += spectrum_order_data[j]*weight[j];
			w += weight[j];
		}
		blaze = (w > 0.) ? (flux/w) : blaze;
		blaze_order[i] = blaze;
	}
	
    for (i = imax-size_slidingbox+1; i <= imax && i < nx && i >= 0; i++) {
		subsize = imax-i+1;
		flux = 0.;
		w = 0.;
		for (j = i-subsize+1; j <= imax; j++) {
			flux += spectrum_order_data[j]*weight[j];
			w += weight[j];
		}
		blaze = (w > 0.) ? (flux/w) : blaze;
		blaze_order[i] = blaze;
	}

	cpl_image_delete(spectrum_order_copy);
		
	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Calculates blaze and flat according to a prescribed method.
 @param blaze_computation_method Desired method for the calcuation "slidind_avg"
				 by default, "polynom" as a choice.
 @param blaze_pol_deg		 In case of polynom method, polynom degree is
				 needed.
 @param size_sliding_box         Size of the sliding box.
 @param blaze_clip		 blaze clip factor to discard faulty pixels.
 @param flat_RE			 Calculated flat.
 @param blaze_RE		 Calculated blaze.
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_blaze_and_flat_calculation(cpl_image *extracted_e2ds, 
					const char *blaze_computation_method,
					int blaze_pol_deg,
					int size_sliding_box,
					double blaze_clip,
					cpl_image *spectrum_error_table,
					cpl_image **flat_RE,
					cpl_image **blaze_RE) {
    
	cpl_error_code my_error = CPL_ERROR_NONE;

    if (strcmp(blaze_computation_method, "polynome") == 0) {
        espdr_msg("POLYNOME METHOD ...");
        
        my_error = espdr_blaze_and_flat_fit_order(extracted_e2ds,
                                                  blaze_pol_deg,
                                                  spectrum_error_table,
                                                  flat_RE,
                                                  blaze_RE);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_blaze_and_flat_fit_order failed: %s",
                     cpl_error_get_message_default(my_error));
        
    } else {

        espdr_msg("SLIDING AVERAGE METHOD ...");
        espdr_msg("clip = %.2f slid_box = %d",
                  blaze_clip, size_sliding_box);
        
        int nx, ny, i, q, order, e2ds_order_x_size;
        double *e2ds_order_data, *blaze_order_measure_data, rms_flat;
        
        double *flat_RE_data, *blaze_RE_data;
        
        flat_RE_data = cpl_image_get_data_double(*flat_RE);
        blaze_RE_data = cpl_image_get_data_double(*blaze_RE);
        
        cpl_image *e2ds_order;
        cpl_vector *rms_flat_order_vector;
        
        nx = cpl_image_get_size_x(extracted_e2ds);
        ny = cpl_image_get_size_y(extracted_e2ds);
        
        cpl_image *blaze_order_measure = cpl_image_new(nx,1,CPL_TYPE_DOUBLE);
        
        double *blaze_weight = (double*)cpl_calloc(nx, sizeof(double));
        double *residuals = (double*)cpl_calloc(nx, sizeof(double));
        double *flat_rms_stdev = (double*)cpl_calloc(nx, sizeof(double));
        
        espdr_msg("espdr_blaze_and_flat_calculation started ...");
        espdr_msg("extracted_e2ds size x=%d size y=%d",nx,ny);
        
        for (order = 0; order < ny; order++) {
            
            e2ds_order = cpl_image_extract(extracted_e2ds,
                                           1, order+1, nx, order+1);
            
            e2ds_order_x_size = cpl_image_get_size_x(e2ds_order);
            
            e2ds_order_data = cpl_image_get_data_double(e2ds_order);
            
            for (i = 0; i < e2ds_order_x_size; i++) {
                blaze_weight[i] = 1.0;
                if(e2ds_order_data[i] == 0.) {
                    blaze_weight[i] = 0.;
                }
            }
            
            my_error = espdr_measure_blaze(e2ds_order,
                                           blaze_weight,
                                           size_sliding_box,
                                           &blaze_order_measure);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_measure_blaze failed: %s",
                         cpl_error_get_message_default(my_error));
            
            blaze_order_measure_data =
                        cpl_image_get_data_double(blaze_order_measure);
            
            for (i = 0; i < e2ds_order_x_size; i++) {
                q = e2ds_order_x_size * order + i;
                blaze_RE_data[q] = blaze_order_measure_data[i];
            }
            
            for (i = 0; i < e2ds_order_x_size; i++) {
                q = e2ds_order_x_size * order + i;
                if(e2ds_order_data[i] > 0.){
                    flat_RE_data[q] = e2ds_order_data[i] /
                    blaze_RE_data[q];
                } else {
                    flat_RE_data[q] = 1.0;
                }
            }
            
            for (i = 0; i < e2ds_order_x_size; i++) {
                q = e2ds_order_x_size * order + i;
                flat_rms_stdev[i] = flat_RE_data[q];
            }
            
            rms_flat_order_vector =
                    cpl_vector_wrap(e2ds_order_x_size, flat_rms_stdev);
            rms_flat = cpl_vector_get_stdev(rms_flat_order_vector);
            cpl_vector_unwrap(rms_flat_order_vector);
            
            for (i = 0; i < e2ds_order_x_size; i++) {
                q = e2ds_order_x_size * order + i;
                residuals[i] = fabs(blaze_RE_data[q]
                                    - e2ds_order_data[i]) / blaze_RE_data[q];
                
                if (residuals[i] > blaze_clip * rms_flat)
                    blaze_weight[i] = 0.;
            }
            
            my_error = espdr_measure_blaze(e2ds_order,
                                           blaze_weight,
                                           size_sliding_box,
                                           &blaze_order_measure);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_measure_blaze failed: %s",
                         cpl_error_get_message_default(my_error));
            
            blaze_order_measure_data =
                        cpl_image_get_data_double(blaze_order_measure);
            
            for (i = 0; i < e2ds_order_x_size; i++) {
                q = e2ds_order_x_size * order + i;
                blaze_RE_data[q] = blaze_order_measure_data[i];
            }
            
            for (i = 0; i < e2ds_order_x_size; i++) {
                q = e2ds_order_x_size * order + i;
                if (e2ds_order_data[i] > 0.) {
                    flat_RE_data[q] = e2ds_order_data[i] / blaze_RE_data[q];
                    // Added to eliminate the very low flat values behind the dust
                    if (flat_RE_data[q] < 0.05) {
                        flat_RE_data[q] = 0.05;
                    }
                } else {
                    flat_RE_data[q] = 1.0;
                }
            }
            
            for (i = 0; i < e2ds_order_x_size; i++) {
                q = e2ds_order_x_size*order+i;
                flat_rms_stdev[i] = flat_RE_data[q];
            }
            
            cpl_image_delete(e2ds_order);
        }
        
        if (blaze_weight != NULL) {
            cpl_free(blaze_weight);
        }

        if (residuals != NULL) {
            cpl_free(residuals);
        }

        if (flat_rms_stdev != NULL) {
            cpl_free(flat_rms_stdev);
        }

        if (blaze_order_measure!= NULL) {
            cpl_image_delete(blaze_order_measure);
        }
    }
    
	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Normalize the S2D
 @param S2D
 @param S2D error
 @param normalization window
 @param physical orders IDs
 @param returned noramlized S2D
 @param returned normalized S2D error
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_s2d_normalisation(cpl_image *flux,
                                       cpl_image *error,
                                       int window,
                                       int *physical_orders_id,
                                       cpl_image **flux_RE,
                                       cpl_image **error_RE) {

    int flux_size_x = cpl_image_get_size_x(flux);
    int flux_size_y = cpl_image_get_size_y(flux);
    int ccd_center = flux_size_x/2;
    int i,k;
    double *flux_data = NULL;
    double *error_data = NULL;
    double *peak_flux = NULL;
    double flux_sum;
    double peak_flux_order,peak_flux_sum;
    int range_ini =  -1*(int)window;
    int range_end = (int)window;
    int q,order,count;
    int phy_order = 0;
    
	double *flux_RE_data;
	flux_RE_data = cpl_image_get_data_double(*flux_RE);	

	double *error_RE_data;
	error_RE_data = cpl_image_get_data_double(*error_RE);	

	cpl_image *flux_order;
	cpl_image *error_order;

	peak_flux = (double *) cpl_calloc(flux_size_y,sizeof(double));

	/* init peak_flux */	
	for (int order = 1; order < flux_size_y; order++) peak_flux[order] = 0.0;

	order = 0;
	peak_flux_sum = 0.;
	count = 0;

	while ( order < flux_size_y ) {
		phy_order = physical_orders_id[order];
		flux_sum = 0.0;
		flux_order = cpl_image_extract(flux,
				   	1, order+1, flux_size_x, order+1);
		flux_data = cpl_image_get_data_double(flux_order);
        
		for (k=range_ini; k < range_end; k++) {
			flux_sum = flux_sum + flux_data[ccd_center+k];
		}
        
		peak_flux_order = flux_sum / 2. / window;
        
		if ((order == 0) || ((order > 0) &&
                             (phy_order == physical_orders_id[order-1]))) {
			peak_flux_sum += peak_flux_order;
			count++; 
		} else {
			for (int i = order-1; i > order-1-count; i--) peak_flux[i] = peak_flux_sum;
			peak_flux_sum = peak_flux_order;
			count = 1;
		}

		if (order == flux_size_y-1) {
			for (int i = order; i > order-count; i--) peak_flux[i] = peak_flux_sum;
		}
        
		cpl_image_delete(flux_order);
        
		order++;
	}

	for (order = 0; order < flux_size_y; order++) {
		flux_order = cpl_image_extract(flux, 1, order+1,
                                       flux_size_x, order+1);
		flux_data = cpl_image_get_data_double(flux_order);
        
		error_order = cpl_image_extract(error, 1, order+1,
                                        flux_size_x, order+1);
		error_data = cpl_image_get_data_double(error_order);
        
		for (i = 0; i < flux_size_x; i++) {
			q = flux_size_x * order + i;
			flux_RE_data[q] = flux_data[i] / peak_flux[order];
			error_RE_data[q] = error_data[i] / peak_flux[order];
		}
        cpl_image_delete(flux_order);
        cpl_image_delete(error_order);
	}
    
	cpl_free(peak_flux);
    
	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Correct blaze function for spectral energy distribution of the source
	and instrumental efficiency by fitting flux distribution at blaze peak 
	(or order center, where the flux is known to be 1).

 @param blaze                 blaze image spectrum to be corrected.
 @param wave                  wave image to get the wavelengths for further
			      calculation.
 @param avgblaze_window_calc  Average window blaze calculation.
 @param flux_interpolated_RE  Calculated flux interpolation.
 @param blaze_RE              Corrected blaze.
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_correct_s2d_instrumental(cpl_image *flux,
                                              cpl_image *error,
                                              cpl_image *wave,
                                              cpl_image *dll,
                                              int window,
                                              int deg,
                                              int *physical_orders_id,
                                              char *blaze_peaks_fit_method,
                                              cpl_image **flux_RE,
                                              cpl_image **error_RE) {

	cpl_error_code my_error = CPL_ERROR_NONE;

    int i, k, q, order, wave_size_x,
    flux_size_x, flux_size_y, phy_order_nb;
    int range_ini = -1*(int)window;
    int range_end = (int)window;
    double chisq = 0.0;
	double *wave_data = NULL;
    double *dll_data = NULL;
	double *flux_data = NULL;
	double *error_data = NULL;
	double *peak_flux = NULL;
	double *peak_wave = NULL;
    double *llc;
	double *flux_RE_data = cpl_image_get_data_double(*flux_RE);
	double *error_RE_data = cpl_image_get_data_double(*error_RE);
	cpl_image *flux_order;
	cpl_image *error_order;
	cpl_image *wave_order;
    cpl_image *dll_order;
    
	wave_size_x = cpl_image_get_size_x(wave);
	flux_size_x = cpl_image_get_size_x(flux);
	flux_size_y = cpl_image_get_size_y(flux);
    
	int ccd_center = wave_size_x/2;
    
    // Not working when physical orders are missing in the S2D
    //phy_order_nb = physical_orders_id[0] - physical_orders_id[flux_size_y-1]+1;
    
    phy_order_nb = 1;
    //np.unique physical order
    for (i = 0; i < flux_size_y-1; i++) {
        if (physical_orders_id[i] != physical_orders_id[i+1]) {
            phy_order_nb++;
        }
    }
    
	peak_flux = (double *) cpl_calloc(phy_order_nb, sizeof(double));
	peak_wave = (double *) cpl_calloc(phy_order_nb, sizeof(double));
	llc = (double *) cpl_calloc(flux_size_y, sizeof(double));
	//dll = (double *) cpl_calloc(flux_size_x, sizeof(double));

	//lcc calculation, lcc is the wavelength values at the middle of X
    //direction of a wave order.
	for (order = 0; order < flux_size_y; order++) {
		wave_order = cpl_image_extract(wave, 1, order+1,
                                       flux_size_x, order+1);
		wave_data = cpl_image_get_data_double(wave_order);
		llc[order] = wave_data[ccd_center];
		cpl_image_delete(wave_order);
	}

	//double sum_center_flux = 0.0;
    double wave_diff = 0.0;
    double peak_flux_sum, flux_sum, peak_flux_order;
    int phy_order, phy_index;
    
    order = 0;
    peak_flux_sum = 0.;
    phy_index = 0;
    while ( order < flux_size_y ) {
        phy_order = physical_orders_id[order];
        flux_sum = 0.0;
        wave_order = cpl_image_extract(wave, 1, order+1,
                                       flux_size_x, order+1);
        wave_data = cpl_image_get_data_double(wave_order);
        
        flux_order = cpl_image_extract(flux, 1, order+1,
                                       flux_size_x, order+1);
        flux_data = cpl_image_get_data_double(flux_order);
        
        for (k = range_ini; k < range_end; k++) {
            flux_sum = flux_sum + flux_data[ccd_center+k];
        }
        
        peak_flux_order = flux_sum;
        
        if((order == 0) || ((order > 0) &&
                            (phy_order == physical_orders_id[order-1]))){
            peak_flux_sum += peak_flux_order;
            wave_diff = wave_data[ccd_center+window]-
            wave_data[ccd_center-window];
            peak_wave[phy_index] = wave_data[ccd_center];
        } else {
            peak_flux[phy_index] = peak_flux_sum/wave_diff;
            phy_index++;
            wave_diff = wave_data[ccd_center+window]-
            wave_data[ccd_center-window];
            peak_wave[phy_index] = wave_data[ccd_center];
            peak_flux_sum = peak_flux_order;
        }
        
        if(order == flux_size_y-1) {
            peak_flux[phy_order_nb-1] = peak_flux_sum/wave_diff;
        }
        
        cpl_image_delete(flux_order);
        cpl_image_delete(wave_order);
        
        order++;
    }
    
	espdr_poly_data *p_data = (espdr_poly_data *)cpl_malloc
					(sizeof(espdr_poly_data));
	p_data->x = (double *) cpl_calloc (phy_order_nb , sizeof (double));
	p_data->y = (double *) cpl_calloc (phy_order_nb , sizeof (double));
	p_data->err = (double *) cpl_calloc (phy_order_nb , sizeof (double));
	p_data->n = phy_order_nb;

	double *fit = NULL;
	double *coeffs = NULL;
	double *coeffs_err = NULL;
	double y2;

	fit = (double *) cpl_calloc (phy_order_nb , sizeof(double));
	coeffs = (double *) cpl_calloc (deg , sizeof(double));
	coeffs_err = (double *) cpl_calloc (deg , sizeof(double));
	
	for (order = 0; order < flux_size_y; order++) {
		flux_order = cpl_image_extract(flux, 1, order+1,
                                       flux_size_x, order+1);
		flux_data = cpl_image_get_data_double(flux_order);

		error_order = cpl_image_extract(error, 1, order+1,
                                        flux_size_x, order+1);
		error_data = cpl_image_get_data_double(error_order);

		wave_order = cpl_image_extract(wave, 1, order+1,
                                       flux_size_x, order+1);
		wave_data = cpl_image_get_data_double(wave_order);

        //dll contains all the pixels size in delta wavelenght
        //units for the current wave order.
        dll_order = cpl_image_extract(dll, 1, order+1,
                                      flux_size_x, order+1);
        dll_data = cpl_image_get_data_double(dll_order);
		
		//espdr_lambda_pixel_size(wave_data, flux_size_x,dll);

		for (i = 0; i < phy_order_nb; i++) {
			p_data->x[i] = peak_wave[i];
			p_data->y[i] = peak_flux[i];
			p_data->err[i] = 1.;	
		}
		
        if (strcmp(blaze_peaks_fit_method, "polynomial") == 0) {
            my_error = espdr_fit_poly(p_data,deg,fit,
                                      coeffs,coeffs_err,&chisq,0);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_fit_poly failed: %s",
                         cpl_error_get_message_default(my_error));
            
            for (int j = 0; j < flux_size_x; j++) {
                q = flux_size_x*order+j;
                y2 = 0.;
                for (int c = deg-1; c >= 0; c--) {
                    y2 = y2 * wave_data[j] + coeffs[c];
                }
                flux_RE_data[q] = flux_data[j]/(y2*dll_data[j]);
                error_RE_data[q] = error_data[j]/(y2*dll_data[j]);
            }
            
        } else {
            if (strcmp(blaze_peaks_fit_method, "spline") == 0) {
                double *y3 = NULL;
                y3 = (double *) cpl_calloc (flux_size_x , sizeof(double));
                
                my_error = espdr_spline_bounded_interpolation(p_data->x,
                                                              p_data->y,
                                                              phy_order_nb,
                                                              wave_data,
                                                              y3,
                                                              flux_size_x,
                                                              wave_data[0],
                                                              wave_data[flux_size_x-1]);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_spline_bounded_interpolation failed: %s",
                             cpl_error_get_message_default(my_error));
                
                for (int j = 0; j < flux_size_x; j++) {
                    q = flux_size_x*order+j;
                    flux_RE_data[q] = flux_data[j]/(y3[j]*dll_data[j]);
                    error_RE_data[q] = error_data[j]/(y3[j]*dll_data[j]);
                }
                
                cpl_free(y3);
            } else {
                espdr_msg_error("Unknown blaze peaks fit method: %s",
                                blaze_peaks_fit_method);
                return(CPL_ERROR_INCOMPATIBLE_INPUT);
            }
        }
        
		cpl_image_delete(flux_order);
		cpl_image_delete(error_order);
		cpl_image_delete(wave_order);
        cpl_image_delete(dll_order);
	}

    cpl_free(p_data->x);
    cpl_free(p_data->y);
    cpl_free(p_data->err);
    cpl_free(p_data);
    
    cpl_free(fit);
    cpl_free(coeffs);
    cpl_free(coeffs_err);
    
    //cpl_free(dll);
    cpl_free(llc);
    cpl_free(peak_flux);
    cpl_free(peak_wave);
    
	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Calculates blaze and flat according to a prescribed method.
 @param blaze_computation_method Desired method for the calcuation "slidind_avg"
				 by default, "polynom" as a choice.
 @param blaze_pol_deg		 In case of polynom method, polynom degree is
				 needed.
 @param size_sliding_box         Size of the sliding box.
 @param blaze_clip		 blaze clip factor to discard faulty pixels.
 @param flat_RE			 Calculated flat.
 @param blaze_RE		 Calculated blaze.
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_blaze_and_flat_fit_order(cpl_image *extracted_e2ds,
                                              int blaze_pol_deg,
                                              cpl_image *spectrum_error_table,
                                              cpl_image **flat_RE,
                                              cpl_image **blaze_RE) {

	cpl_error_code my_error = CPL_ERROR_NONE;

	int i, q, nx, ny, order;
	double *e2ds_order_data, *spectrum_error_data;
    double chisq = 0.0;
	double *flat_RE_data, *blaze_RE_data;

	flat_RE_data = cpl_image_get_data_double(*flat_RE);
	blaze_RE_data = cpl_image_get_data_double(*blaze_RE);

	nx = cpl_image_get_size_x(extracted_e2ds);
	ny = cpl_image_get_size_y(extracted_e2ds);
	
	cpl_image *e2ds_order;
	cpl_image *spectrum_error;

	espdr_poly_data *p_data = (espdr_poly_data *)cpl_malloc
					(sizeof(espdr_poly_data));
	p_data->x = (double *) cpl_calloc (nx , sizeof (double));
	p_data->y = (double *) cpl_calloc (nx , sizeof (double));
	p_data->err = (double *) cpl_calloc (nx , sizeof (double));
	p_data->n = nx;

	double *fit = NULL;
	double *coeffs = NULL;
	double *coeffs_err = NULL;

	fit = (double *) cpl_calloc (nx , sizeof(double));
	coeffs = (double *) cpl_calloc (blaze_pol_deg , sizeof(double));
	coeffs_err = (double *) cpl_calloc (blaze_pol_deg , sizeof(double));

	for (order = 0; order < ny; order++) {
		e2ds_order = cpl_image_extract(extracted_e2ds,
                                       1, order+1, nx, order+1);
		e2ds_order_data = cpl_image_get_data_double(e2ds_order);

		spectrum_error = cpl_image_extract(spectrum_error_table,
                                           1, order+1, nx, order+1);
		spectrum_error_data = cpl_image_get_data_double(spectrum_error);

		for (i = 0; i < nx; i++) {
			p_data->x[i] = (double)i;
			p_data->y[i] = e2ds_order_data[i];
			p_data->err[i] = spectrum_error_data[i];	
		}
		
		my_error = espdr_fit_poly(p_data, blaze_pol_deg,
                                  fit, coeffs, coeffs_err, &chisq, 0);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_fit_poly failed: %s",
                     cpl_error_get_message_default(my_error));

		for (i = 0; i < nx; i++) {
			q = nx * order + i;
			blaze_RE_data[q] = fit[i];	
		}
        
		for (i = 0; i < nx; i++) {
			q = nx * order + i;
			flat_RE_data[q] = e2ds_order_data[i] / blaze_RE_data[q];
		}
        
        cpl_image_delete(e2ds_order);
        cpl_image_delete(spectrum_error);
	}

	espdr_msg("espdr_blaze_and_flat_calculation blaze and flat products delivered.");

    cpl_free(p_data->x);
    cpl_free(p_data->y);
    cpl_free(p_data->err);
    
    cpl_free(fit);
    cpl_free(coeffs);
    cpl_free(coeffs_err);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Remove the blaze function from an spectrum
 @param extracted_spectrum_table spectrum to be blaze corrected
 @param spectrum_error_table spectrum error table to be corrected
 @param spectrum_quality_table the quality table is untouch, is added in order
to return the whole estructure, spectrum, error and quality tables
 @param blaze blaze function image
 @param extracted_spectrum_table_corrected_RE spectrum corrected
 @param spectrum_error_table_corrected_RE error table corrected
 @param spectrum_quality_table_RE quality map returned
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_correct_blaze(cpl_image *extracted_spectrum_table,
                                   cpl_image *spectrum_error_table,
                                   cpl_image *spectrum_quality_table,
                                   cpl_image *blaze,
                                   cpl_image **extracted_spectrum_table_corrected_RE,
                                   cpl_image **spectrum_error_table_corrected_RE,
                                   cpl_image **spectrum_quality_table_RE) {

	cpl_image *extracted_spectrum_table_corrected;
	cpl_image *spectrum_error_table_corrected;
	cpl_image *spectrum_quality_table_corrected;

	extracted_spectrum_table_corrected =
		cpl_image_divide_create(extracted_spectrum_table,blaze);

	spectrum_error_table_corrected =
		cpl_image_divide_create(spectrum_error_table,blaze);

    spectrum_quality_table_corrected =
        cpl_image_duplicate(spectrum_quality_table);
    
	*extracted_spectrum_table_corrected_RE =
		extracted_spectrum_table_corrected;

	*spectrum_error_table_corrected_RE =
		spectrum_error_table_corrected;

	*spectrum_quality_table_RE =
		spectrum_quality_table_corrected;

	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Remove the flat from an spectrum and also from it's error relative table
 @param extracted_spectrum_table spectrum to be flat corrected
 @param spectrum_error_table spectrum error relative table to be corrected
 @param spectrum_quality_table the quality table is untouch, is added in order
to return the whole estructure, spectrum, error and quality tables
 @param flat_spectrum_table flat calculated spectrum
 @param flat_spectrum_error_table flat relative error table
 @param extracted_spectrum_table_corrected_RE spectrum corrected
 @param spectrum_error_table_corrected_RE error table corrected
 @param spectrum_quality_table_RE quality map returned
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_correct_flat(cpl_image *extracted_spectrum_table,
                                  cpl_image *spectrum_error_table,
                                  cpl_image *spectrum_quality_table,
                                  cpl_image *flat_spectrum_table,
                                  cpl_image *flat_spectrum_error_table,
                                  cpl_image *flat_spectrum_quality,
                                  cpl_image **extracted_spectrum_table_corrected_RE,
                                  cpl_image **spectrum_error_table_corrected_RE,
                                  cpl_image **spectrum_quality_table_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    espdr_ensure(extracted_spectrum_table == NULL, CPL_ERROR_NULL_INPUT,
                 "extracted_spectrum_table is NULL");
    espdr_ensure(spectrum_error_table == NULL, CPL_ERROR_NULL_INPUT,
                 "spectrum_error_table is NULL");
    espdr_ensure(spectrum_quality_table == NULL, CPL_ERROR_NULL_INPUT,
                 "spectrum_quality_table is NULL");
    espdr_ensure(flat_spectrum_table == NULL, CPL_ERROR_NULL_INPUT,
                 "flat_spectrum_table is NULL");
    espdr_ensure(flat_spectrum_error_table == NULL, CPL_ERROR_NULL_INPUT,
                 "flat_spectrum_error_table is NULL");
    espdr_ensure(flat_spectrum_quality == NULL, CPL_ERROR_NULL_INPUT,
                 "flat_spectrum_quality is NULL");
    
    cpl_image *extracted_spectrum_table_corrected;
    cpl_image *spectrum_quality_table_corrected;
    
    extracted_spectrum_table_corrected =
        cpl_image_divide_create(extracted_spectrum_table, flat_spectrum_table);
    
	/* ASE: the mathematics behind the following CPL operations intended to
 	*  calculates error propagation, is as follows: 
	*	sqrt ( ( s2d_error/flat )**2 + ( flat_error *
	*	s2d/flat**2 )**2 )
	*/
    
	/* ( s2d_error/flat )**2 */
	cpl_image *div_op = 
        cpl_image_divide_create(spectrum_error_table, flat_spectrum_table);
	cpl_image *pow_op = cpl_image_power_create(div_op, 2.0);
    
	/* ( flat_error*s2d/flat**2 )**2 */
	cpl_image *pow_op_2 = cpl_image_power_create(flat_spectrum_table,2.0);
	cpl_image *div_op_2 = cpl_image_divide_create(extracted_spectrum_table,pow_op_2);
	cpl_image *mult_op = cpl_image_multiply_create(flat_spectrum_error_table,div_op_2);
	cpl_image *pow_op_3 = cpl_image_power_create(mult_op,2.0);
    
	/* ( s2d_error/flat )**2 + ( flat_error*s2d/flat**2 )**2 */
	cpl_image *sum_op = cpl_image_add_create(pow_op,pow_op_3);
    
	/* sqrt(( s2d_error/flat )**2 + ( flat_error*s2d/flat**2 )**2) */
	cpl_image *sqr_op =	 cpl_image_power_create(sum_op,0.5);
    
	int nx = cpl_image_get_size_x(sum_op);
	int ny = cpl_image_get_size_y(sum_op);
    
	spectrum_quality_table_corrected = cpl_image_new(nx, ny, CPL_TYPE_INT);
    
    my_error = cpl_image_or(spectrum_quality_table_corrected,
                            spectrum_quality_table,
                            flat_spectrum_quality);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_image_or failed : %s",
                 cpl_error_get_message_default(my_error));
    
	*extracted_spectrum_table_corrected_RE = extracted_spectrum_table_corrected;
    
	*spectrum_error_table_corrected_RE = sqr_op;
    
	*spectrum_quality_table_RE = spectrum_quality_table_corrected;
    
    cpl_image_delete(div_op);
    cpl_image_delete(pow_op);
    cpl_image_delete(pow_op_2);
    cpl_image_delete(div_op_2);
    cpl_image_delete(mult_op);
    cpl_image_delete(pow_op_3);
    cpl_image_delete(sum_op);
    
    return cpl_error_get_code();
}

