/*                                                                            *
 *   This file is part of the ESO X-shooter Pipeline                          *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2013-08-05 17:38:05 $
 * $Revision: 1.4 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------
 Includes
 ----------------------------------------------------------------------------*/
#include <assert.h>

#include <fcntl.h>    /* used for fileutils_copy/move */
#include <sys/stat.h> /* used for fileutils_copy/move */

#include <espdr_dfs.h>
#include <espdr_msg.h>
#include <espdr_parameters.h>
#include <espdr_instrument.h>

#include <cpl.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup espdr_dfs  DFS related functions
 */
/*----------------------------------------------------------------------------*/

/**@{*/

cpl_frame* espdr_get_mbias_from_set(const cpl_frameset* set,
                                    cpl_frameset* used_frames) {
    
    cpl_ensure(set, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(used_frames, CPL_ERROR_NULL_INPUT, NULL);

    /* Extract master BIAS frame */
    cpl_frame* mbias_frame = espdr_frame_find(set,ESPDR_PRO_CATG_MBIAS_RES);
    cpl_error_ensure(mbias_frame != NULL, CPL_ERROR_NULL_INPUT, NULL,
                     "MASTER BIAS RES frame extraction failed");
    
    cpl_frameset_insert(used_frames, cpl_frame_duplicate(mbias_frame));
    espdr_print_rec_status(0);
    return mbias_frame;
}


cpl_frame* espdr_get_hpixmap_from_set(const cpl_frameset* set,
                                      cpl_frameset* used_frames) {
    
    cpl_ensure(set, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(used_frames, CPL_ERROR_NULL_INPUT, NULL);

    /* Extract hot pixels mask frame */
    cpl_frame* hpix_frame = espdr_frame_find(set, ESPDR_PRO_CATG_HOT_PIXELS);
    cpl_error_ensure(hpix_frame != NULL, CPL_ERROR_NULL_INPUT, NULL,
                     "HOT PIXELS frame extraction failed");
    
    cpl_frameset_insert(used_frames, cpl_frame_duplicate(hpix_frame));
    
    espdr_print_rec_status(0);
    return hpix_frame;
}


cpl_frame* espdr_get_mdark_from_set(const cpl_frameset* set,
                                    cpl_frameset* used_frames) {
    
    cpl_ensure(set, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(used_frames, CPL_ERROR_NULL_INPUT, NULL);
    
    /* Extract master dark frame */
    cpl_frame* mdark_frame = espdr_frame_find(set, ESPDR_PRO_CATG_MDARK);
    
    if (mdark_frame != NULL) {
        cpl_frameset_insert(used_frames, cpl_frame_duplicate(mdark_frame));
    }
    
    espdr_print_rec_status(0);
    return mdark_frame;
}


cpl_frame* espdr_get_bpixmap_from_set(const cpl_frameset* set,
                                      cpl_frameset* used_frames) {
    
    cpl_ensure(set, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(used_frames, CPL_ERROR_NULL_INPUT, NULL);

    /* Extract bad pixels mask frame */
    cpl_frame* bpix_frame = espdr_frame_find(set, ESPDR_PRO_CATG_BAD_PIXELS);
    cpl_error_ensure(bpix_frame != NULL, CPL_ERROR_NULL_INPUT, NULL,
                     "BAD PIXELS frame extraction failed");
    
    cpl_frameset_insert(used_frames, cpl_frame_duplicate(bpix_frame));
    
    espdr_print_rec_status(0);
    return bpix_frame;
}


cpl_frame* espdr_get_orders_mask_from_set(const cpl_frameset* set,
                                          cpl_frameset* used_frames) {
    
    cpl_ensure(set, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(used_frames, CPL_ERROR_NULL_INPUT, NULL);
    
    /* Extract master dark frame */
    cpl_frame* orders_mask_frame = espdr_frame_find(set, ESPDR_ORDERS_MASK);
    
    if (orders_mask_frame != NULL) {
        cpl_frameset_insert(used_frames, cpl_frame_duplicate(orders_mask_frame));
    }
    
    espdr_print_rec_status(0);
    return orders_mask_frame;
}


cpl_error_code espdr_get_orders_coeff_from_set(const cpl_frameset* set,
                                               cpl_frameset* used_frames,
                                               const int fibres_nb,
                                               const int ext_nb,
                                               cpl_table **** orders_coeffs) {
    
    espdr_ensure(set == NULL, CPL_ERROR_NULL_INPUT,
                 "Input set of frames is NULL");
    espdr_ensure(used_frames == NULL, CPL_ERROR_NULL_INPUT,
                 "used_frames set of frames is NULL");

    /* Extract orders maps frame */
    char pro_catg_tag[64];
    const char *filename = NULL;
    *orders_coeffs = (cpl_table ***)cpl_malloc(fibres_nb * sizeof(cpl_table**));
    
    for (int i = 0; i < fibres_nb; i++) {
        
        (*orders_coeffs)[i] = (cpl_table **)cpl_malloc(ext_nb * sizeof(cpl_table *));
        
        sprintf(pro_catg_tag, "%s_%c", ESPDR_PRO_CATG_ORDERS, fibre_name[i]);
        
        cpl_frame* order_map_frame = espdr_frame_find(set,pro_catg_tag);
        espdr_ensure(order_map_frame == NULL, CPL_ERROR_NULL_INPUT,
                     "ORDERS map frame extraction failed");
        
        cpl_frameset_insert(used_frames, cpl_frame_duplicate(order_map_frame));
        
        filename = cpl_frame_get_filename(order_map_frame);
        
        for (int j = 0; j < ext_nb; j++) {
            (*orders_coeffs)[i][j] = cpl_table_load(filename, ext_nb+1+j, 0);
        }
        
    }
    espdr_print_rec_status(0);
    
    return cpl_error_get_code();
}


cpl_error_code espdr_get_order_profile_from_set(const cpl_frameset* set,
                                                cpl_frameset* used_frames,
                                                const int fibres_nb,
                                                const int ext_nb,
                                                cpl_frame *** order_profile_fibre_frame) {
    
    espdr_ensure(set == NULL, CPL_ERROR_NULL_INPUT,
                 "Input set of frames is NULL");
    espdr_ensure(used_frames == NULL, CPL_ERROR_NULL_INPUT,
                 "used_frames set of frames is NULL");

    /* Extract order profile frame per fibre */
    char pro_catg_tag[64];
    
    *order_profile_fibre_frame = (cpl_frame **)cpl_malloc(fibres_nb * sizeof(cpl_frameset *));
    
    for (int i = 0; i < fibres_nb; i++) {
        
        sprintf(pro_catg_tag, "%s_%c",
                ESPDR_PRO_CATG_MASTER_ORDER_PROFILE, fibre_name[i]);
        
        if (espdr_frameset_has_frame(set, pro_catg_tag)) {
            (*order_profile_fibre_frame)[i] = espdr_frame_find(set, pro_catg_tag);
        } else {
            sprintf(pro_catg_tag, "%s_%c",
                    ESPDR_PRO_CATG_ORDER_PROFILE, fibre_name[i]);
            (*order_profile_fibre_frame)[i] = espdr_frame_find(set, pro_catg_tag);
            
            if((*order_profile_fibre_frame)[i] == NULL) {
                espdr_msg("ORDER_PROFILE for fibre %c frame extraction failed",
                          fibre_name[i]);
                espdr_msg("No ORDER_PROFILE frame for fibre %c, exiting.",
                          fibre_name[i]);
                return CPL_ERROR_NULL_INPUT;
            }
        }
        
        cpl_frameset_insert(used_frames, cpl_frame_duplicate((*order_profile_fibre_frame)[i]));
        
    }
    espdr_print_rec_status(0);
    
    return cpl_error_get_code();
}


cpl_error_code espdr_get_flat_fibre_from_set(const cpl_frameset* set,
                                             cpl_frameset* used_frames,
                                             const int fibres_nb,
                                             const int ext_nb,
                                             cpl_frame *** flat_fibre_frame) {
    
    espdr_ensure(set == NULL, CPL_ERROR_NULL_INPUT,
                 "Input set of frames is NULL");
    espdr_ensure(used_frames == NULL, CPL_ERROR_NULL_INPUT,
                 "used_frames set of frames is NULL");
    
    /* Extract flat frame per fibre */
    char pro_catg_tag[64];
    *flat_fibre_frame = (cpl_frame **)cpl_malloc(fibres_nb * sizeof(cpl_frameset *));
    
    for (int i = 0; i < fibres_nb; i++) {
        
        sprintf(pro_catg_tag, "%s_%c",
                ESPDR_PRO_CATG_MASTER_FLAT, fibre_name[i]);
        if(espdr_frameset_has_frame(set, pro_catg_tag)) {
            (*flat_fibre_frame)[i] = espdr_frame_find(set, pro_catg_tag);
        } else {
            sprintf(pro_catg_tag, "%s_%c",ESPDR_PRO_CATG_FLAT, fibre_name[i]);
            (*flat_fibre_frame)[i] = espdr_frame_find(set, pro_catg_tag);
            
            if((*flat_fibre_frame)[i] == NULL) {
                espdr_msg("FLAT for fibre %c frame extraction failed",
                          fibre_name[i]);
                espdr_msg("No FLAT frame for fibre %c, exiting.",
                          fibre_name[i]);
                return CPL_ERROR_NULL_INPUT;
            }
            
        }
        
        cpl_frameset_insert(used_frames, cpl_frame_duplicate((*flat_fibre_frame)[i]));
    }
    espdr_print_rec_status(0);
    
    return cpl_error_get_code();
}


cpl_error_code espdr_get_blaze_fibre_from_set(const cpl_frameset* set,
                                              cpl_frameset* used_frames,
                                              const int fibres_nb,
                                              const int ext_nb,
                                              cpl_frame *** blaze_fibre_frame) {
    
    espdr_ensure(set == NULL, CPL_ERROR_NULL_INPUT,
                 "Input set of frames is NULL");
    espdr_ensure(used_frames == NULL, CPL_ERROR_NULL_INPUT,
                 "used_frames set of frames is NULL");
    
    /* Extract BLAZE frame per fibre */
    char pro_catg_tag[64];
    *blaze_fibre_frame = (cpl_frame **)cpl_malloc(fibres_nb * sizeof(cpl_frameset *));
    
    for (int i = 0; i < fibres_nb; i++) {
        
        sprintf(pro_catg_tag, "%s_%c",
                ESPDR_PRO_CATG_MASTER_BLAZE, fibre_name[i]);
        if(espdr_frameset_has_frame(set, pro_catg_tag)) {
            (*blaze_fibre_frame)[i] = espdr_frame_find(set, pro_catg_tag);
        } else {
            sprintf(pro_catg_tag, "%s_%c",
                    ESPDR_PRO_CATG_BLAZE, fibre_name[i]);
            (*blaze_fibre_frame)[i] = espdr_frame_find(set, pro_catg_tag);
            
            if((*blaze_fibre_frame)[i] == NULL) {
                espdr_msg("BLAZE for fibre %c frame extraction failed",
                          fibre_name[i]);
                espdr_msg("No BLAZE frame for fibre %c, exiting.",
                          fibre_name[i]);
                return CPL_ERROR_NULL_INPUT;
            }
        }
        
        cpl_frameset_insert(used_frames, cpl_frame_duplicate((*blaze_fibre_frame)[i]));
    }
    espdr_print_rec_status(0);
    
    return cpl_error_get_code();
}


cpl_error_code espdr_get_pixel_geom_fibre_from_set(const cpl_frameset* set,
                                                   cpl_frameset* used_frames,
                                                   const int fibres_nb,
                                                   const int ext_nb,
                                                   int** pixel_geom_nb_check,
                                                   cpl_frame *** pixel_geom_fibre_frame) {
    espdr_ensure(set == NULL, CPL_ERROR_NULL_INPUT,
                 "Input set of frames is NULL");
    espdr_ensure(used_frames == NULL, CPL_ERROR_NULL_INPUT,
                 "used_frames set of frames is NULL");
    
    char pro_catg_tag[64];
    *pixel_geom_fibre_frame = (cpl_frame **)cpl_malloc(fibres_nb * sizeof(cpl_frame *));
    
    *pixel_geom_nb_check = (int *)cpl_calloc(fibres_nb, sizeof(int));
    for (int i = 0; i < fibres_nb; i++) {
        
        sprintf(pro_catg_tag, "%s_%c",
                ESPDR_PRO_CATG_PIXEL_GEOM, fibre_name[i]);
        (*pixel_geom_fibre_frame)[i] = espdr_frame_find(set, pro_catg_tag);
        
        if ((*pixel_geom_fibre_frame)[i] != NULL) {
            cpl_frameset_insert(used_frames, cpl_frame_duplicate((*pixel_geom_fibre_frame)[i]));
            (*pixel_geom_nb_check)[i]++;
        } else {
            espdr_msg("No PIXEL_GEOM frame for fibre %c", fibre_name[i]);
        }
    }
    espdr_print_rec_status(0);
    
    return cpl_error_get_code();
    
}

cpl_error_code espdr_get_pixel_size_fibre_from_set(const cpl_frameset* set,
                                                   cpl_frameset* used_frames,
                                                   const int fibres_nb,
                                                   const int ext_nb,
                                                   int** pixel_size_nb_check,
                                                   cpl_frame *** pixel_size_fibre_frame) {
    espdr_ensure(set == NULL, CPL_ERROR_NULL_INPUT,
                 "Input set of frames is NULL");
    espdr_ensure(used_frames == NULL, CPL_ERROR_NULL_INPUT,
                 "used_frames set of frames is NULL");
    
    char pro_catg_tag[64];
    *pixel_size_fibre_frame = (cpl_frame **)cpl_malloc(fibres_nb * sizeof(cpl_frame *));
    
    *pixel_size_nb_check = (int *)cpl_calloc(fibres_nb, sizeof(int));
    for (int i = 0; i < fibres_nb; i++) {
        
        sprintf(pro_catg_tag, "%s_%c",
                ESPDR_PRO_CATG_PIXEL_SIZE, fibre_name[i]);
        (*pixel_size_fibre_frame)[i] = espdr_frame_find(set, pro_catg_tag);
        
        if ((*pixel_size_fibre_frame)[i] != NULL) {
            cpl_frameset_insert(used_frames, cpl_frame_duplicate((*pixel_size_fibre_frame)[i]));
            (*pixel_size_nb_check)[i]++;
        } else {
            espdr_msg("No PIXEL_SIZE frame for fibre %c", fibre_name[i]);
        }
    }
    espdr_print_rec_status(0);
    
    return cpl_error_get_code();
    
}

cpl_error_code espdr_get_static_wave_matrix_from_set(const cpl_frameset* set,
                                                     cpl_frameset* used_frames,
                                                     const int fibres_nb,
                                                     cpl_image ***wave_matrix) {
    
    *wave_matrix = (cpl_image **)cpl_malloc(fibres_nb*sizeof(cpl_image *));
    cpl_frame *wave_frame[fibres_nb];
    const char *wave_name;
    char wave_catg_tag[64];
    
    for (int i = 0; i < fibres_nb; i++) {
        
        sprintf(wave_catg_tag,"%s_%c",
                ESPDR_PRO_CATG_STATIC_WAVE_MATRIX, fibre_name[i]);
        
        if (espdr_frameset_has_frame(set, wave_catg_tag)) {
            wave_frame[i] = espdr_frame_find(set, wave_catg_tag);
            espdr_ensure(wave_frame[i] == NULL, CPL_ERROR_NULL_INPUT,
                         "STATIC WAVE MATRIX %c frame extraction failed",
                         fibre_name[i]);
            
            cpl_frameset_insert(used_frames, cpl_frame_duplicate(wave_frame[i]));
            
            wave_name = cpl_frame_get_filename(wave_frame[i]);
            (*wave_matrix)[i] = cpl_image_load(wave_name,CPL_TYPE_DOUBLE,0,1);
        } else {
            (*wave_matrix)[i] = NULL;
        }

    }
    
    return cpl_error_get_code();
    
}


cpl_error_code espdr_get_static_dll_matrix_from_set(const cpl_frameset* set,
                                                     cpl_frameset* used_frames,
                                                     const int fibres_nb,
                                                     cpl_image ***dll_matrix) {
    
    *dll_matrix = (cpl_image **)cpl_malloc(fibres_nb*sizeof(cpl_image *));
    cpl_frame *dll_frame[fibres_nb];
    const char *dll_name;
    char dll_catg_tag[64];
    
    for (int i = 0; i < fibres_nb; i++) {
        
        sprintf(dll_catg_tag,"%s_%c",
                ESPDR_PRO_CATG_STATIC_DLL_MATRIX, fibre_name[i]);
        
        if (espdr_frameset_has_frame(set, dll_catg_tag)) {
            dll_frame[i] = espdr_frame_find(set, dll_catg_tag);
            espdr_ensure(dll_frame[i] == NULL, CPL_ERROR_NULL_INPUT,
                         "STATIC DLL MATRIX %c frame extraction failed",
                         fibre_name[i]);
            
            cpl_frameset_insert(used_frames, cpl_frame_duplicate(dll_frame[i]));
            
            dll_name = cpl_frame_get_filename(dll_frame[i]);
            (*dll_matrix)[i] = cpl_image_load(dll_name,CPL_TYPE_DOUBLE,0,1);
            
        } else {
            (*dll_matrix)[i] = NULL;
        }
        
    }
    
    return cpl_error_get_code();
}


cpl_error_code espdr_get_raw_flat_frames_from_set(const cpl_frameset* set,
                                                  const int fibres_nb,
                                                  const int raw_flat_limit_nb,
                                                  cpl_frameset **flat_frames,
                                                  cpl_frameset *used_frames) {
    
    char frame_tag[15];
    int flat_frameset_size = 0;
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    for (int i = 0; i < fibres_nb; i++) {
        flat_frames[i] = cpl_frameset_new();
        my_error = espdr_frame_extract_by_tag(set, ESPDR_FLAT_RAW,
                                              flat_frames[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Raw FLAT for fibre %c frame extraction failed",
                     fibre_name[i]);
        flat_frameset_size = cpl_frameset_get_size(flat_frames[i]);
        
        if (flat_frameset_size == 0) {
            sprintf(frame_tag, "%s_%c", ESPDR_FLAT_RAW, fibre_name[i]);
            my_error = espdr_frame_extract_by_tag(set, frame_tag,
                                                  flat_frames[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "Raw FLAT for fibre %c frame extraction failed",
                         fibre_name[i]);
            
            flat_frameset_size = cpl_frameset_get_size(flat_frames[i]);
        }
            
        if (flat_frameset_size < raw_flat_limit_nb) {
            espdr_msg_warning("The number of input raw FLAT frames for fibre %c is not enough (%d), should be minimum %d, exiting.",
                              fibre_name[i], flat_frameset_size, raw_flat_limit_nb);
            for (int j = 0; j <= i; j++) {
                cpl_frameset_delete(flat_frames[j]);
            }
            return (CPL_ERROR_ILLEGAL_INPUT);
        }
        
        for (int f = 0; f < flat_frameset_size; f++) {
            my_error = cpl_frameset_insert(used_frames,
                                           cpl_frame_duplicate(cpl_frameset_get_position(flat_frames[i], f)));
        }
    }
    
    return cpl_error_get_code();
}


cpl_frame* espdr_get_wave_matrix_frame_from_set(cpl_frameset* set,
                                                cpl_frameset* used_frames,
                                                const char* suffix1,
                                                const char* suffix2,
                                                const int fid) {
    
    char wave_catg_tag[32];
    cpl_frame* wave_frame;
    
    sprintf(wave_catg_tag, "%s_%s",
            ESPDR_PRO_CATG_WAVE_MATRIX, suffix1);
    if (espdr_frameset_has_frame(set, wave_catg_tag)) {
        wave_frame = espdr_frame_find(set, wave_catg_tag);
    } else {
        sprintf(wave_catg_tag,"%s_%s",
                ESPDR_PRO_CATG_WAVE_MATRIX, suffix2);
        if (espdr_frameset_has_frame(set, wave_catg_tag)) {
            wave_frame = espdr_frame_find(set, wave_catg_tag);
        } else {
            sprintf(wave_catg_tag, "%s_%s",
                    ESPDR_PRO_CATG_WAVE_MATRIX_DRIFT, suffix1);
            if (espdr_frameset_has_frame(set, wave_catg_tag)) {
                wave_frame = espdr_frame_find(set, wave_catg_tag);
            } else {
                sprintf(wave_catg_tag, "%s_%s",
                        ESPDR_PRO_CATG_WAVE_MATRIX_DRIFT, suffix2);
                if (espdr_frameset_has_frame(set, wave_catg_tag)) {
                    wave_frame = espdr_frame_find(set, wave_catg_tag);
                } else {
                    espdr_msg_error("No WAVE_MATRIX frame for fibre %c, exiting",
                                    fibre_name[fid]);
                    return (NULL);
                }
            }
        }
    }
    
    cpl_frameset_insert(used_frames, cpl_frame_duplicate(wave_frame));
    
    return wave_frame;
}


cpl_frame* espdr_get_dll_matrix_frame_from_set(cpl_frameset* set,
                                               cpl_frameset* used_frames,
                                               const char* suffix1,
                                               const char* suffix2,
                                               const int fid) {
    
    char wave_catg_tag[32];
    cpl_frame* wave_frame;
    
    sprintf(wave_catg_tag, "%s_%s",
            ESPDR_PRO_CATG_DLL_MATRIX, suffix1);
    if (espdr_frameset_has_frame(set, wave_catg_tag)) {
        wave_frame = espdr_frame_find(set, wave_catg_tag);
    } else {
        sprintf(wave_catg_tag,"%s_%s",
                ESPDR_PRO_CATG_DLL_MATRIX, suffix2);
        if (espdr_frameset_has_frame(set, wave_catg_tag)) {
            wave_frame = espdr_frame_find(set, wave_catg_tag);
        } else {
            sprintf(wave_catg_tag, "%s_%s",
                    ESPDR_PRO_CATG_DLL_MATRIX_DRIFT, suffix1);
            if (espdr_frameset_has_frame(set, wave_catg_tag)) {
                wave_frame = espdr_frame_find(set, wave_catg_tag);
            } else {
                sprintf(wave_catg_tag, "%s_%s",
                        ESPDR_PRO_CATG_DLL_MATRIX_DRIFT, suffix2);
                if (espdr_frameset_has_frame(set, wave_catg_tag)) {
                    wave_frame = espdr_frame_find(set, wave_catg_tag);
                } else {
                    espdr_msg_error("No DLL_MATRIX frame for fibre %c, exiting",
                                    fibre_name[fid]);
                    return (NULL);
                }
            }
        }
    }
    
    cpl_frameset_insert(used_frames, cpl_frame_duplicate(wave_frame));
    
    return wave_frame;
}


cpl_error_code espdr_get_static_res_map_from_set(const cpl_frameset* set,
                                                 cpl_frameset* used_frames,
                                                 cpl_image **res_map) {
    
    cpl_frame *res_map_frame;
    const char *res_map_name;
        
    if (espdr_frameset_has_frame(set, ESPDR_RES_MAP_A)) {
        res_map_frame = espdr_frame_find(set, ESPDR_RES_MAP_A);
        espdr_ensure(res_map_frame == NULL, CPL_ERROR_NULL_INPUT,
                     "STATIC RESOLUTION MAP frame extraction failed");
        
        cpl_frameset_insert(used_frames, cpl_frame_duplicate(res_map_frame));
        
        res_map_name = cpl_frame_get_filename(res_map_frame);
        *res_map = cpl_image_load(res_map_name,CPL_TYPE_DOUBLE,0,1);
        
    } else {
        *res_map = NULL;
    }

    return cpl_error_get_code();
}


cpl_error_code espdr_get_static_model_frames_from_set(const cpl_frameset* set,
                                                      cpl_frameset *used_frames,
                                                      cpl_frameset *hitran_qt_frames,
                                                      cpl_frameset *hitran_strongest_frames,
                                                      cpl_frameset *hitran_ccf_masks) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    my_error = espdr_frame_extract_by_tag(set, ESPDR_HITRAN_LINES_QT, hitran_qt_frames);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error, "HITRAN_LINES_QT frames extraction failed");
    my_error = espdr_frame_extract_by_tag(set, ESPDR_HITRAN_LINES_STRONGEST, hitran_strongest_frames);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error, "HITRAN_LINES_STRONGEST frames extraction failed");
    my_error = espdr_frame_extract_by_tag(set, ESPDR_HITRAN_CCF_MASK, hitran_ccf_masks);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error, "HITRAN_CCF_MASK frames extraction failed");
    
    cpl_size hitran_qt_frameset_size = cpl_frameset_get_size(hitran_qt_frames);
    espdr_ensure(hitran_qt_frameset_size == 0, CPL_ERROR_NULL_INPUT,
                 "No HITRAN LINES QT, the telluric correction can't be done");
    cpl_size hitran_strongest_frameset_size = cpl_frameset_get_size(hitran_strongest_frames);
    espdr_ensure(hitran_strongest_frameset_size == 0, CPL_ERROR_NULL_INPUT,
                 "No HITRAN LINES STRONGEST, the telluric correction can't be done");
    cpl_size hitran_ccf_frameset_size = cpl_frameset_get_size(hitran_ccf_masks);
    espdr_ensure(hitran_ccf_frameset_size == 0, CPL_ERROR_NULL_INPUT,
                 "No HITRAN CCF MASKs, the telluric correction can't be done");
    
    for (int f = 0; f < hitran_qt_frameset_size; f++) {
        my_error = cpl_frameset_insert(used_frames,
                        cpl_frame_duplicate(cpl_frameset_get_position(hitran_qt_frames, f)));
    }

    for (int f = 0; f < hitran_strongest_frameset_size; f++) {
        my_error = cpl_frameset_insert(used_frames,
                        cpl_frame_duplicate(cpl_frameset_get_position(hitran_strongest_frames, f)));
    }
    
    for (int f = 0; f < hitran_ccf_frameset_size; f++) {
        //espdr_msg("Hitran CCF mask [%d] is %s", f,
        //          cpl_frame_get_filename(cpl_frameset_get_position(hitran_ccf_masks, f)));
        my_error = cpl_frameset_insert(used_frames,
                        cpl_frame_duplicate(cpl_frameset_get_position(hitran_ccf_masks, f)));
    }

    return cpl_error_get_code();
}


cpl_error_code espdr_get_OH_static_frames_from_set(const cpl_frameset* set,
                                                   cpl_frameset *used_frames,
                                                   cpl_image **master_spectrum_A,
                                                   cpl_image **master_error_A,
                                                   cpl_image **dll_A,
                                                   cpl_image **master_spectrum_B,
                                                   cpl_image **master_error_B,
                                                   cpl_image **dll_B,
                                                   cpl_frame **lines_regions_frame) {

    cpl_frame *master_spectrum_A_fr = espdr_frame_find(set, ESPDR_OH_SKY_MASTER_A);
    cpl_frame *master_spectrum_B_fr = espdr_frame_find(set, ESPDR_OH_SKY_MASTER_B);
    *lines_regions_frame = espdr_frame_find(set, ESPDR_OH_LINES_REGIONS);
    espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE, cpl_error_get_code(),
                 "OH correction frames extraction failed");
    
    *master_spectrum_A = cpl_image_load(cpl_frame_get_filename(master_spectrum_A_fr), CPL_TYPE_DOUBLE, 0, 1);
    *master_error_A = cpl_image_load(cpl_frame_get_filename(master_spectrum_A_fr), CPL_TYPE_DOUBLE, 0, 2);
    *dll_A = cpl_image_load(cpl_frame_get_filename(master_spectrum_A_fr), CPL_TYPE_DOUBLE, 0, 3);
    *master_spectrum_B = cpl_image_load(cpl_frame_get_filename(master_spectrum_B_fr), CPL_TYPE_DOUBLE, 0, 1);
    *master_error_B = cpl_image_load(cpl_frame_get_filename(master_spectrum_B_fr), CPL_TYPE_DOUBLE, 0, 2);
    *dll_B = cpl_image_load(cpl_frame_get_filename(master_spectrum_B_fr), CPL_TYPE_DOUBLE, 0, 3);
    //*lines_regions = cpl_table_load(cpl_frame_get_filename(lines_regions_fr), 1, 0);
    espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE, cpl_error_get_code(),
                 "OH static calibrations loading failed");

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Set the group as RAW or CALIB in a frameset
 @param    set: the input frameset
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/
cpl_error_code espdr_dfs_set_groups(cpl_frameset *set) {

	cpl_errorstate prestate = cpl_errorstate_get();
	cpl_frame * frame = NULL;
	int         i = 0;

	/* Loop on frames */
	int nset = cpl_frameset_get_size(set);
	cpl_frameset_iterator* iter = cpl_frameset_iterator_new(set);
	frame = cpl_frameset_iterator_get(iter);

	for ( int iter_frame = 0; iter_frame < nset; iter_frame++ ) {

		const char * tag = cpl_frame_get_tag(frame);

		espdr_msg_debug("Frame tag is %s", tag);

		if (tag == NULL) {
			cpl_msg_warning(cpl_func, "Frame %d has no tag", i);
		} else if (ESPDR_IS_RAW(tag)) {
			/* RAW frames */
            //espdr_msg("Is RAW");
			cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
		} else if (ESPDR_IS_CALIB(tag)) {
			/* CALIB frames */
            //espdr_msg("Is CALIB");
			cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
		} else if (ESPDR_IS_CONFIG(tag)) {
			/* CONFIG frames, considered as CALIB frames */
			cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
            //espdr_msg("Is CONFIG");
		} else {
			/* UNRECOGNIZED frames */
			espdr_msg_error("Unrecongnized TAG: %s", tag);
			cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
		}
        
		cpl_frameset_iterator_advance(iter, 1);
		frame = cpl_frameset_iterator_get(iter);
	}
	cpl_frameset_iterator_delete(iter);

	if (!cpl_errorstate_is_equal(prestate)) {
		return cpl_error_set_message(cpl_func, cpl_error_get_code(),
				"Could not identify RAW and CALIB "
				"frames");
	}

	return CPL_ERROR_NONE;
}

/*---------------------------------------------------------------------------*/
/**
 @brief		defines a new frame
 @param		frame       frame to be created
 @param		filename    frame filename
 @param		group       frame group
 @param		level       frame level
 @param		type        frame type
 @param		tag         frame tag
 @return	CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_frame_new(cpl_frame **frame, 
		const char *filename,
		cpl_frame_group group,
		cpl_frame_level level,
		cpl_frame_type type,
		const char *tag) {

	*frame = cpl_frame_new();

	cpl_frame_set_filename(*frame, filename);
	espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE, cpl_error_get_code(), 
			"Error setting filename: %s", filename);

	cpl_frame_set_group(*frame, group);
	espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE, cpl_error_get_code(), 
			"Error setting group: %d", group);

	cpl_frame_set_level(*frame, level);
	espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE, cpl_error_get_code(), 
			"Error setting level: %d", level);

	cpl_frame_set_type(*frame, type);
	espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE, cpl_error_get_code(), 
			"Error setting type: %d", type);

	cpl_frame_set_tag(*frame, tag);
	espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE, cpl_error_get_code(), 
			"Error setting tag: %s", tag);

	return(cpl_error_get_code());
}



/*---------------------------------------------------------------------------*/
/**
 @brief        insert a frameset with a TAG into another frameset
 @param        set    frameset to be inserted
 @param        tag    frames tag
 @param        out    frameset tagged with the tag
 @param        use    frameset to be inserted into
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_insert_frame(const cpl_frameset* set,
                                  const char* tag,
                                  cpl_frameset* out,
                                  cpl_frameset* use){
    
    cpl_ensure_code(set != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(use != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(tag != NULL, CPL_ERROR_NULL_INPUT);
    
    const cpl_frame* f = cpl_frameset_find_const(set, tag);
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        espdr_msg_error("Frame %s not found, exiting.", tag);
        return CPL_ERROR_NULL_INPUT;
    }
    
    cpl_frameset_insert(use, cpl_frame_duplicate(f));
    cpl_frameset_insert(out, cpl_frame_duplicate(f));
    
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 *  @brief  Cleanup the frameset structure
 *  @param  set frameset structure to be cleaned
 *  @param  nb  number of framesets in the structure
 *  @return CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_frameset_cleanup(cpl_frameset** set,
                                      const int nb) {
    
    const char* fname = NULL;
    cpl_frame* frame = NULL;
    int size = 0;
    char cmd[COMMENT_LENGTH];
    
    for (int it = 0; it < nb; it++) {
        size = cpl_frameset_get_size(set[it]);
        for (int i = 0; i < size; i++) {
            frame = cpl_frameset_get_position(set[it], i);
            fname = cpl_frame_get_filename(frame);
            espdr_msg("file to be cleanup : %s",fname);
            sprintf(cmd, "rm  %s", fname);
            system(cmd);
        }
    }
    
    return (cpl_error_get_code());
}

/*---------------------------------------------------------------------------*/
/**
 @brief		    split input set if raw and master calibrations sub sets
 @param         frameset    set of frames to be split
 @param[out]    raw_RE      returned set of raw frames
 @param[out]    calib_RE    returned set of calib frames
 @return	    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_frame_split_raw_calib(cpl_frameset *set,
                                           cpl_frameset *raw_RE,
                                           cpl_frameset *calib_RE) {

    espdr_ensure(set == NULL, CPL_ERROR_NULL_INPUT,
                 "Input set of frames is NULL");
    
	int i = 0;
	cpl_frame* frame = NULL;
	raw_RE = cpl_frameset_new();
	calib_RE = cpl_frameset_new();

	int nset = cpl_frameset_get_size(set);
	cpl_frameset_iterator* iter = cpl_frameset_iterator_new(set);
	frame = cpl_frameset_iterator_get(iter);

	for ( int iter_frame = 0; iter_frame < nset; iter_frame++ ) {

		const char* tag = cpl_frame_get_tag(frame);

		if (tag == NULL) {
			cpl_msg_warning(cpl_func, "Frame %d has no tag", i);
		} else if (ESPDR_IS_RAW(tag)) {
			/* RAW frames */
			cpl_frameset_insert(raw_RE, cpl_frame_duplicate(frame));
		} else if (ESPDR_IS_CALIB(tag)) {
			cpl_frameset_insert(calib_RE, cpl_frame_duplicate(frame));
		}
		cpl_frameset_iterator_advance(iter, 1);
		frame = cpl_frameset_iterator_get(iter);
	}
	cpl_frameset_iterator_delete(iter);

	if (calib_RE == NULL) {
		espdr_msg_warning("No calibrations frames in the input set");
	}
	if (raw_RE == NULL) {
		espdr_msg_warning("No raw frames in the input set");
	}

	return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 @brief	extract from input set master calibrations sub sets
 @param         frameset    set of frames to be split
 @param[out]    calib_RE    returned set of calib frames
 @return	    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_frameset_extract_calib(cpl_frameset *set,
                                           cpl_frameset **calib_RE) {

    espdr_ensure(set == NULL, CPL_ERROR_NULL_INPUT,
                 "Input set of frames is NULL");

	int i = 0;
	cpl_frame* frame = NULL;

	int nset = cpl_frameset_get_size(set);
	cpl_frameset_iterator* iter = cpl_frameset_iterator_new(set);
	frame = cpl_frameset_iterator_get(iter);
	*calib_RE = cpl_frameset_new();
	for ( int iter_frame = 0; iter_frame < nset; iter_frame++ ) {

		const char* tag = cpl_frame_get_tag(frame);

		if (tag == NULL) {
			cpl_msg_warning(cpl_func, "Frame %d has no tag", i);
		} else if (ESPDR_IS_CALIB(tag)) {
			cpl_frameset_insert(*calib_RE, cpl_frame_duplicate(frame));
		}
		cpl_frameset_iterator_advance(iter, 1);
		frame = cpl_frameset_iterator_get(iter);
	}
	cpl_frameset_iterator_delete(iter);

	if (*calib_RE == NULL) {
		espdr_msg_warning("No calibrations frames in the input set");
	}

	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief		    extract from a frameset the subset of frames with a given tag
 @param         set             set of frames to be splitted
 @param         tag             tag to extract frames according to
 @param[out]    extracted_RE    returned set of extracted frames
 @return	    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_frame_extract_by_tag(const cpl_frameset *set,
                                          const char* tag,
                                          cpl_frameset* extracted_RE) {

    espdr_ensure(set == NULL, CPL_ERROR_NULL_INPUT,
                 "Input set of frames is NULL");

    cpl_frame* frame = NULL;
    
	int nset = cpl_frameset_get_size(set);
	cpl_frameset_iterator* iter = cpl_frameset_iterator_new(set);
	frame = cpl_frameset_iterator_get(iter);

	for ( int iter_frame = 0; iter_frame < nset; iter_frame++ ) {

		const char* curr_tag = cpl_frame_get_tag(frame);

		if (!strcmp(curr_tag, tag)) {
			cpl_frameset_insert(extracted_RE, cpl_frame_duplicate(frame));
		}
		cpl_frameset_iterator_advance(iter, 1);
		frame = cpl_frameset_iterator_get(iter);
	}
	cpl_frameset_iterator_delete(iter);

	return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 * AMO: added
 @brief     check if the frameset contains a frame with a given tag
 @param     set     set of frames to be checked
 @param     tag     tag to extract frames according to
 @return    TRUE if contains, FALSE if not
 */
/*---------------------------------------------------------------------------*/

cpl_boolean espdr_frameset_has_frame(const cpl_frameset *set,
                                     const char* tag) {
    
    cpl_ensure(set != NULL, CPL_ERROR_NULL_INPUT, CPL_FALSE);

    const cpl_frame* frame = NULL;
    frame = cpl_frameset_find_const(set, tag);
    
    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
        espdr_msg_error("Frame %s not found",tag);
        return CPL_FALSE;
    }
    
    if (frame != NULL) {
        return CPL_TRUE;
    } else {
        return CPL_FALSE;
    }
    
    return CPL_FALSE;
}


/*---------------------------------------------------------------------------*/
/**
 * AMO: added
 @brief     extract from a frameset a frame that has a given tag
 @param     set     set of frames to be split
 @param     tag     tag to extract frames according to
 @return    found frame iff OK else NULL;
 */
/*---------------------------------------------------------------------------*/

cpl_frame* espdr_frame_find(const cpl_frameset *set,
                            const char* tag) {

    cpl_ensure(set != NULL, CPL_ERROR_NULL_INPUT, NULL);

	const cpl_frame* frame = NULL;
	frame = cpl_frameset_find_const(set, tag);

	if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
		espdr_msg_error("Frame %s not found",tag);
		return NULL;
	}

	return (cpl_frame *) frame;
}



/*---------------------------------------------------------------------------*/
/**
 @brief		    Creates and fills an image list with all images extensions from
                an image frame
 @param         input_frame     frame to extract from
 @param         frame_type      input image frame type
 @param         ext_nr          number of extenstions in FITS frame
 @param[out]    extracted_RE    imagelist filled with images from frame extensions
 @return	    CPL_ERROR_NONE iff OK

 TODO: remove ext_nr input parameter. The number of extensions of a frame can be
 obtained with the function cpl_frame_get_nextensions()
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_extract_extensions(cpl_frame *input_frame,
		cpl_type frame_type,
		const int ext_nr,
		cpl_imagelist **extracted_RE) {

	int i;
	cpl_image *curr_image = NULL;
	cpl_error_code my_error = CPL_ERROR_NONE;

	for (i = 0; i < ext_nr; i++) {
        espdr_msg("Loading %d image from frame %s",
                  i, cpl_frame_get_filename(input_frame));
		curr_image = cpl_image_load(cpl_frame_get_filename(input_frame),
				frame_type, 0, i+1);
        my_error = cpl_error_get_code();
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_load failed: %s",
                     cpl_error_get_message_default(my_error));
		my_error = cpl_imagelist_set(*extracted_RE,
				cpl_image_duplicate(curr_image), i);
		espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
				"cpl_imagelist_set failed: %s",
				cpl_error_get_message_default(my_error));
		cpl_image_delete(curr_image);
	}

	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief		    Extract from all the frames the defined raw outputs and put them
                in the returned imagelist in the order of extraction
 @param         input_frameset  set of frames to extract from
 @param         frame_type      type of the frames
 @param         CCD_geom        CCD geometry
 @param[out]    all_outputs_RE  returned imagelist of extracted outputs
 @return	    CPL_ERROR_NONE iff OK

 TODO:
 1) it would be useful to have a function that loads only a given detector
 read-out port. This is a small change of this.
 The advantage is that such function would require much less RAM
 2) The loop over frames can be done without using deprecated functions
 3) This function is not trivial and need to be better documented along the code
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_extract_raw_outputs(cpl_frameset *input_frameset,
		cpl_type frame_type,
		espdr_CCD_geometry *CCD_geom,
		cpl_imagelist **all_outputs_RE) {

	int i, j, k, index;
	int llx, lly, urx, ury;
	int x, y;
	cpl_frame *curr_frame = NULL;
	cpl_image *curr_image = NULL, *curr_output = NULL;
	cpl_error_code my_error = CPL_ERROR_NONE;

	//espdr_msg("Starting espdr_extract_raw_outputs");

	espdr_ensure(input_frameset == NULL, CPL_ERROR_NULL_INPUT,
			"Input frameset is NULL");

	espdr_ensure(CCD_geom == NULL, CPL_ERROR_NULL_INPUT,
			"CCD geometry structure is NULL");

	index = 0;
	int nset = cpl_frameset_get_size(input_frameset);
	cpl_frameset_iterator* iter = cpl_frameset_iterator_new(input_frameset);
	curr_frame = cpl_frameset_iterator_get(iter);

	for ( int iter_frame = 0; iter_frame < nset; iter_frame++ ) {
		for (i = 0; i < CCD_geom->ext_nb; i++) {
			curr_image = cpl_image_load(cpl_frame_get_filename(curr_frame),
					frame_type, 0, i+1);
			my_error = cpl_error_get_code();
			espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
					"cpl_image_load failed: %s",
					cpl_error_get_message_default(my_error));

			int size_x = cpl_image_get_size_x(curr_image);
			int size_y = cpl_image_get_size_y(curr_image);
			//espdr_msg("curr_image size x: %d, size y : %d",
            //          size_x, size_y);

			espdr_ensure(size_x != CCD_geom->exts[i].image_nx,
					CPL_ERROR_ILLEGAL_INPUT,
					"The image size x should be %d and is %d",
					CCD_geom->exts[i].image_nx, size_x);
			espdr_ensure(size_y != CCD_geom->exts[i].image_ny,
					CPL_ERROR_ILLEGAL_INPUT,
					"The image size y should be %d and is %d",
					CCD_geom->exts[i].image_ny, size_y);

			x = 0;
			for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
				llx = x+1;
				urx = x+CCD_geom->exts[i].outputs[j][0].raw_nx;
				y = 0;
				for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
					//espdr_msg("ext: %d, out_x: %d, out_y:%d", i, j, k);
					lly = y+1+CCD_geom->exts[i].outputs[j][k].ppscan_ny;
					ury = y+CCD_geom->exts[i].outputs[j][k].raw_ny
							-CCD_geom->exts[i].outputs[j][k].poscan_ny;
					curr_output = cpl_image_extract(curr_image,
							llx, lly, urx, ury);
					//espdr_msg("llx = %d, lly = %d, urx = %d, ury = %d",
                    //          llx, lly, urx, ury);
					my_error = cpl_error_get_code();
					espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
							"cpl_image_extract failed: %s",
							cpl_error_get_message_default(my_error));
					my_error = cpl_imagelist_set(*all_outputs_RE,
							cpl_image_duplicate(curr_output),
							index);
					espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
							"cpl_imagelist_set failed: %s",
							cpl_error_get_message_default(my_error));
					cpl_image_delete(curr_output);
					index++;
					y = ury+CCD_geom->exts[i].outputs[j][k].poscan_ny;
				}
				x = urx;
			}
			cpl_image_delete(curr_image);
		}
		cpl_frameset_iterator_advance(iter, 1);
		curr_frame = cpl_frameset_iterator_get(iter);
	}
	cpl_frameset_iterator_delete(iter);

	cpl_frame_delete(curr_frame);
    
	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief            Extract from all the frames the defined raw outputs and put them
 in the returned imagelist in the order of extraction
 @param         input_frameset  set of frames to extract from
 @param         frame_type      type of the frames
 @param         CCD_geom        CCD geometry
 @param[out]    all_outputs_RE  returned imagelist of extracted outputs
 @return        CPL_ERROR_NONE iff OK
 
 TODO:
 1) it would be useful to have a function that loads only a given detector
 read-out port. This is a small change of this.
 The advantage is that such function would require much less RAM
 2) The loop over frames can be done without using deprecated functions
 3) This function is not trivial and need to be better documented along the code
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_extract_raw_outputs_one_frame(cpl_frame *input_frame,
                                         cpl_type frame_type,
                                         espdr_CCD_geometry *CCD_geom,
                                         cpl_imagelist **all_outputs_RE) {
    
    int i, j, k, index;
    int llx, lly, urx, ury;
    int x, y;
    cpl_image *curr_image = NULL, *curr_output = NULL;
    cpl_error_code my_error = CPL_ERROR_NONE;
    const char* fname = cpl_frame_get_filename(input_frame);

    //espdr_msg("Starting espdr_extract_raw_outputs_one_frame");
    
    espdr_ensure(input_frame == NULL, CPL_ERROR_NULL_INPUT,
                 "Input frame is NULL");
    
    espdr_ensure(CCD_geom == NULL, CPL_ERROR_NULL_INPUT,
                 "CCD geometry structure is NULL");
    
    index = 0;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        curr_image = cpl_image_load(fname, frame_type, 0, i+1);
        my_error = cpl_error_get_code();
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_load failed: %s",
                     cpl_error_get_message_default(my_error));
        
        int size_x = cpl_image_get_size_x(curr_image);
        int size_y = cpl_image_get_size_y(curr_image);
        //espdr_msg("curr_image size x: %d, size y : %d",
        //          size_x, size_y);
        
        espdr_ensure(size_x != CCD_geom->exts[i].image_nx,
                     CPL_ERROR_ILLEGAL_INPUT,
                     "The image size x should be %d and is %d",
                     CCD_geom->exts[i].image_nx, size_x);
        espdr_ensure(size_y != CCD_geom->exts[i].image_ny,
                     CPL_ERROR_ILLEGAL_INPUT,
                     "The image size y should be %d and is %d",
                     CCD_geom->exts[i].image_ny, size_y);
        
        x = 0;
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            llx = x+1;
            urx = x+CCD_geom->exts[i].outputs[j][0].raw_nx;
            y = 0;
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                //espdr_msg("ext: %d, out_x: %d, out_y:%d", i, j, k);
                lly = y+1+CCD_geom->exts[i].outputs[j][k].ppscan_ny;
                ury = y+CCD_geom->exts[i].outputs[j][k].raw_ny
                        -CCD_geom->exts[i].outputs[j][k].poscan_ny;
                curr_output = cpl_image_extract(curr_image,
                                                llx, lly, urx, ury);
                //espdr_msg("llx = %d, lly = %d, urx = %d, ury = %d",
                //          llx, lly, urx, ury);
                my_error = cpl_error_get_code();
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "cpl_image_extract failed: %s",
                             cpl_error_get_message_default(my_error));
                my_error = cpl_imagelist_set(*all_outputs_RE,
                                             cpl_image_duplicate(curr_output),
                                             index);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "cpl_imagelist_set failed: %s",
                             cpl_error_get_message_default(my_error));
                cpl_image_delete(curr_output);
                index++;
                y = ury+CCD_geom->exts[i].outputs[j][k].poscan_ny;
            }
            x = urx;
        }
        cpl_image_delete(curr_image);
    }
    
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief		    Extract from all the frames the defined raw outputs and put them
                in the returned imagelist in the order of extraction
 @param         input_frameset  set of frames to extract from
 @param         frame_type      type of the frames
 @param         CCD_geom        CCD geometry
 @param[out]    all_outputs_RE  returned imagelist of extracted outputs
 @return	    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_extract_real_outputs(cpl_frame *curr_frame,
                                          cpl_type frame_type,
                                          espdr_CCD_geometry *CCD_geom,
                                          int start,
                                          cpl_imagelist **all_outputs_RE) {

	int i, j, k, index;
	int llx, lly, urx, ury;
	int x, y;
	cpl_image *curr_image = NULL, *curr_output = NULL;
	cpl_error_code my_error = CPL_ERROR_NONE;
	bool take_off_ppscan = false;

	//espdr_msg("Starting espdr_extract_raw_outputs");
    
	index = 0;
    if (cpl_frame_get_nextensions(curr_frame) > CCD_geom->ext_nb) {
        
    }
    
    //for (i = 0; i < CCD_geom->ext_nb; i++) {
    // This change is needed, because we added pxl_nb extensions of type CPL_TYPE_INT to the master dark
    for (i = start; i < start+CCD_geom->ext_nb; i++) {
        curr_image = cpl_image_load(cpl_frame_get_filename(curr_frame), frame_type, 0, i+1);
        my_error = cpl_error_get_code();
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                "cpl_image_load failed: %s", cpl_error_get_message_default(my_error));

        //int size_x = cpl_image_get_size_x(curr_image);
        int size_y = cpl_image_get_size_y(curr_image);
        //espdr_msg("curr_image size x: %d, size y : %d", size_x, size_y);
        //espdr_msg("Image size: %d x %d", CCD_geom->exts[i].image_nx, CCD_geom->exts[i].image_ny);
        //espdr_msg("CCD size: %d x %d", CCD_geom->exts[i].CCD_nx, CCD_geom->exts[i].CCD_ny);

        //espdr_ensure(size_x != CCD_geom->exts[i].CCD_nx, CPL_ERROR_ILLEGAL_INPUT,
        //             "The image size x should be %d and is %d", CCD_geom->exts[i].image_nx, size_x);
        //espdr_ensure(size_y != CCD_geom->exts[i].CCD_ny, CPL_ERROR_ILLEGAL_INPUT,
        //             "The image size y should be %d and is %d", CCD_geom->exts[i].image_ny, size_y);
        
        
        int ext_index = i;
        if (i >= CCD_geom->ext_nb) {
            ext_index = i - CCD_geom->ext_nb;
        }
        if (size_y == CCD_geom->exts[ext_index].image_ny) {
            take_off_ppscan = true;
        } else {
            if (size_y == CCD_geom->exts[ext_index].CCD_ny) {
                take_off_ppscan = false;
            } else {
                espdr_msg_error("Wrong image size y, is %d, should be %d or %d",
                        size_y, CCD_geom->exts[ext_index].CCD_ny, CCD_geom->exts[ext_index].image_ny);
                return(CPL_ERROR_ILLEGAL_INPUT);
            }
        }
        //espdr_msg("Do we take off the ppscan? %s", take_off_ppscan ? "true" : "false");

        x = 0;
        for (j = 0; j < CCD_geom->exts[ext_index].out_nb_x; j++) {
            llx = x+1;
            urx = x+CCD_geom->exts[ext_index].outputs[j][0].real_nx;
            y = 0;
            for (k = 0; k < CCD_geom->exts[ext_index].out_nb_y; k++) {
                if (take_off_ppscan) {
                    lly = y+1+CCD_geom->exts[ext_index].outputs[j][k].ppscan_ny;
                } else {
                    lly = y+1;
                }
                ury = lly-1+CCD_geom->exts[ext_index].outputs[j][k].real_ny;
                curr_output = cpl_image_extract(curr_image,
                        llx, lly, urx, ury);
                //espdr_msg("real_ny = %d", CCD_geom->exts[i].outputs[j][k].real_ny);
                //espdr_msg("ppscan_ny = %d, poscan_ny = %d",
                //          CCD_geom->exts[i].outputs[j][k].ppscan_ny,
                //          CCD_geom->exts[i].outputs[j][k].poscan_ny);
                //espdr_msg("llx = %d, lly = %d, urx = %d, ury = %d", llx, lly, urx, ury);
                my_error = cpl_error_get_code();
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                        "cpl_image_extract failed: %s", cpl_error_get_message_default(my_error));
                my_error = cpl_imagelist_set(*all_outputs_RE,
                                             cpl_image_duplicate(curr_output), index);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                        "cpl_imagelist_set failed: %s", cpl_error_get_message_default(my_error));
                cpl_image_delete(curr_output);
                index++;
                if (take_off_ppscan) {
                    y = ury + CCD_geom->exts[ext_index].outputs[j][k].poscan_ny;
                } else {
                    y = ury;
                }
            }
            x = urx;
        }
        cpl_image_delete(curr_image);
    }

	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief		    Extract from all the frames the defined raw outputs and put them
                in the returned imagelist in the order of extraction
 @param         input_frameset  set of frames to extract from
 @param         frame_type      type of the frames
 @param         CCD_geom        CCD geometry
 @param[out]    all_outputs_RE  returned imagelist of extracted outputs
 @return	    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_extract_real_outputs_from_raw(cpl_frameset *input_frameset,
		cpl_type frame_type,
		const espdr_CCD_geometry *CCD_geom,
		cpl_imagelist **all_outputs_RE) {

	int i, j, k, index;
	int llx, lly, urx, ury;
	int x, y;
	int total_oscan_nx;
	cpl_frame *curr_frame = NULL;
	cpl_image *curr_image = NULL, *curr_output = NULL;
	cpl_error_code my_error = CPL_ERROR_NONE;

	//espdr_msg("Starting espdr_extract_real_outputs_from_raw");

	index = 0;
	int nset = cpl_frameset_get_size(input_frameset);
	cpl_frameset_iterator* iter = cpl_frameset_iterator_new(input_frameset);
	curr_frame = cpl_frameset_iterator_get(iter);
	//espdr_msg("Treating %d frames", nset);

	for ( int iter_frame = 0; iter_frame < nset; iter_frame++ ) {

		for (i = 0; i < CCD_geom->ext_nb; i++) {
			curr_image = cpl_image_load(cpl_frame_get_filename(curr_frame),
					frame_type, 0, i+1);
			my_error = cpl_error_get_code();
			espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
					"cpl_image_load failed: %s",
					cpl_error_get_message_default(my_error));

			//int size_x = cpl_image_get_size_x(curr_image);
			//int size_y = cpl_image_get_size_y(curr_image);
			//espdr_msg("curr_image size x: %d, size y : %d", size_x, size_y);
			//espdr_msg("Image size: %d x %d",
			//          CCD_geom->exts[i].image_nx, CCD_geom->exts[i].image_ny);
			//espdr_msg("CCD size: %d x %d",
			//          CCD_geom->exts[i].CCD_nx, CCD_geom->exts[i].CCD_ny);

			x = 0;
			for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
				llx = x+CCD_geom->exts[i].outputs[j][0].pscan_urx+1;
				urx = llx-1+CCD_geom->exts[i].outputs[j][0].real_nx;
				y = 0;
				for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
					lly = y+1+CCD_geom->exts[i].outputs[j][k].ppscan_ny;
					ury = lly-1+CCD_geom->exts[i].outputs[j][k].real_ny;
					curr_output = cpl_image_extract(curr_image,
							llx, lly, urx, ury);
					//espdr_msg("real_ny = %d",
							//          CCD_geom->exts[i].outputs[j][k].real_ny);
					//espdr_msg("ppscan_ny = %d, poscan_ny = %d",
					//          CCD_geom->exts[i].outputs[j][k].ppscan_ny,
					//          CCD_geom->exts[i].outputs[j][k].poscan_ny);
					//espdr_msg("llx = %d, lly = %d, urx = %d, ury = %d",
					//          llx, lly, urx, ury);
					//espdr_msg("extracted img sixe: %lld x %lld",
					//          cpl_image_get_size_x(curr_output),
					//          cpl_image_get_size_y(curr_output));
					my_error = cpl_error_get_code();
					espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
							"cpl_image_extract failed: %s",
							cpl_error_get_message_default(my_error));
					my_error = cpl_imagelist_set(*all_outputs_RE,
							cpl_image_duplicate(curr_output),
							index);
					espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
							"cpl_imagelist_set failed: %s",
							cpl_error_get_message_default(my_error));
					cpl_image_delete(curr_output);
					index++;
					y = ury + CCD_geom->exts[i].outputs[j][k].poscan_ny;
				}
				//x = urx+CCD_geom->exts[i].outputs[j][0].oscan_urx;
				total_oscan_nx = CCD_geom->exts[i].outputs[j][0].raw_nx -
						CCD_geom->exts[i].outputs[j][0].real_nx -
						CCD_geom->exts[i].outputs[j][0].pscan_urx;
				x = urx+total_oscan_nx;
			}
			cpl_image_delete(curr_image);
		}

		cpl_frameset_iterator_advance(iter, 1);
		curr_frame = cpl_frameset_iterator_get(iter);
	}
	cpl_frameset_iterator_delete(iter);

	cpl_frame_delete(curr_frame);

	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief            Extract from all the frames the defined raw outputs and put them
 in the returned imagelist in the order of extraction
 @param         input_frameset  set of frames to extract from
 @param         frame_type      type of the frames
 @param         CCD_geom        CCD geometry
 @param[out]    all_outputs_RE  returned imagelist of extracted outputs
 @return        CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_extract_real_outputs_one_frame(cpl_frame *input_frame,
                                                   cpl_type frame_type,
                                                   const espdr_CCD_geometry *CCD_geom,
                                                   cpl_imagelist **all_outputs_RE) {
    
    int i, j, k, index;
    int llx, lly, urx, ury;
    int x, y;
    int total_oscan_nx;
    cpl_image *curr_image = NULL, *curr_output = NULL;
    cpl_error_code my_error = CPL_ERROR_NONE;
    const char* fname = cpl_frame_get_filename(input_frame);
    
    //espdr_msg("Starting espdr_extract_real_outputs_one_frame");
    
    index = 0;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        curr_image = cpl_image_load(fname, frame_type, 0, i+1);
        my_error = cpl_error_get_code();
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_load failed: %s",
                     cpl_error_get_message_default(my_error));
        
        //int size_x = cpl_image_get_size_x(curr_image);
        //int size_y = cpl_image_get_size_y(curr_image);
        //espdr_msg("curr_image size x: %d, size y : %d", size_x, size_y);
        //espdr_msg("Image size: %d x %d",
        //          CCD_geom->exts[i].image_nx, CCD_geom->exts[i].image_ny);
        //espdr_msg("CCD size: %d x %d",
        //          CCD_geom->exts[i].CCD_nx, CCD_geom->exts[i].CCD_ny);
        
        x = 0;
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            llx = x+CCD_geom->exts[i].outputs[j][0].pscan_urx+1;
            urx = llx-1+CCD_geom->exts[i].outputs[j][0].real_nx;
            y = 0;
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                lly = y+1+CCD_geom->exts[i].outputs[j][k].ppscan_ny;
                ury = lly-1+CCD_geom->exts[i].outputs[j][k].real_ny;
                curr_output = cpl_image_extract(curr_image,
                                                llx, lly, urx, ury);
                //espdr_msg("real_ny = %d",
                //          CCD_geom->exts[i].outputs[j][k].real_ny);
                //espdr_msg("ppscan_ny = %d, poscan_ny = %d",
                //          CCD_geom->exts[i].outputs[j][k].ppscan_ny,
                //          CCD_geom->exts[i].outputs[j][k].poscan_ny);
                //espdr_msg("llx = %d, lly = %d, urx = %d, ury = %d",
                //          llx, lly, urx, ury);
                //espdr_msg("extracted img sixe: %lld x %lld",
                //          cpl_image_get_size_x(curr_output),
                //          cpl_image_get_size_y(curr_output));
                my_error = cpl_error_get_code();
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "cpl_image_extract failed: %s",
                             cpl_error_get_message_default(my_error));
                my_error = cpl_imagelist_set(*all_outputs_RE,
                                             cpl_image_duplicate(curr_output),
                                             index);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "cpl_imagelist_set failed: %s",
                             cpl_error_get_message_default(my_error));
                cpl_image_delete(curr_output);
                index++;
                y = ury + CCD_geom->exts[i].outputs[j][k].poscan_ny;
            }
            //x = urx+CCD_geom->exts[i].outputs[j][0].oscan_urx;
            total_oscan_nx = CCD_geom->exts[i].outputs[j][0].raw_nx -
                                CCD_geom->exts[i].outputs[j][0].real_nx -
                                CCD_geom->exts[i].outputs[j][0].pscan_urx;
            x = urx+total_oscan_nx;
        }
        cpl_image_delete(curr_image);
    }
    
    return cpl_error_get_code();
}




/*---------------------------------------------------------------------------*/
/**
 @brief		Save image with the complete FITS header and all the extensions
 @param     all_frames      recipe input frames
 @param		parameters      recipe parameters
 @param     used_frames     frames used for this product
 @param     recipe          recipe name
 @param     keywords        main header keywords
 @param     keywords_ext    extension keywords
 @param     filename        product filename
 @param     images_to_save  list of images to be saved
 @param     image_type      product image type
 @param     CCD_geom        CCD geometry
 @return	CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_dfs_image_save(cpl_frameset *all_frames,
		const cpl_parameterlist *parameters,
		const cpl_frameset *used_frames,
		const char *recipe,
		cpl_propertylist *keywords,
		cpl_propertylist **keywords_ext,
		const char *filename,
		cpl_imagelist *images_to_save,
		cpl_type image_type,
		espdr_CCD_geometry *CCD_geom) {

	int i;
	cpl_error_code my_error = CPL_ERROR_NONE;

	my_error = cpl_dfs_save_propertylist(all_frames, NULL, parameters,
			used_frames, NULL, recipe,
			keywords, NULL,
			PACKAGE "/" PACKAGE_VERSION,
			filename);

	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Error saving primary header: %s", cpl_error_get_message());

	for (i = 0; i < CCD_geom->ext_nb; i++) {
		my_error = cpl_image_save(cpl_imagelist_get(images_to_save, i),
				filename, image_type,
				keywords_ext[i], CPL_IO_EXTEND);
		espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
				"Error saving extension %d: %s",
				i, cpl_error_get_message());
	}

	/* Verify the saved frame */
	my_error = espdr_verify_fits_image(filename);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Error in verifying image %s", filename);

	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief        Save image with the complete FITS header and all the extensions
 @param     all_frames      recipe input frames
 @param        parameters      recipe parameters
 @param     used_frames     frames used for this product
 @param     recipe          recipe name
 @param     keywords        main header keywords
 @param     keywords_ext    extension keywords
 @param     filename        product filename
 @param     images_to_save  list of images to be saved
 @param     image_type      product image type
 @param     CCD_geom        CCD geometry
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_dfs_image_save_with_pxl_nb(cpl_frameset *all_frames,
        const cpl_parameterlist *parameters,
        const cpl_frameset *used_frames,
        const char *recipe,
        cpl_propertylist *keywords,
        cpl_propertylist **keywords_ext,
        const char *filename,
        cpl_imagelist *images_to_save,
        cpl_imagelist *images_pxl_nb_to_save,
        cpl_type image_type,
        espdr_CCD_geometry *CCD_geom) {

    int i;
    cpl_error_code my_error = CPL_ERROR_NONE;

    my_error = cpl_dfs_save_propertylist(all_frames, NULL, parameters,
            used_frames, NULL, recipe,
            keywords, NULL,
            PACKAGE "/" PACKAGE_VERSION,
            filename);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
            "Error saving primary header: %s", cpl_error_get_message());

    for (i = 0; i < CCD_geom->ext_nb; i++) {
        my_error = cpl_image_save(cpl_imagelist_get(images_to_save, i),
                filename, image_type,
                keywords_ext[i], CPL_IO_EXTEND);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                "Error saving extension %d: %s",
                i, cpl_error_get_message());
    }
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        my_error = cpl_image_save(cpl_imagelist_get(images_pxl_nb_to_save, i),
                filename, CPL_TYPE_INT,
                keywords_ext[i], CPL_IO_EXTEND);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                "Error saving extension %d: %s",
                i, cpl_error_get_message());
    }
    
    /* Verify the saved frame */
    my_error = espdr_verify_fits_image(filename);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
            "Error in verifying image %s", filename);

    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief		Save image with the complete FITS header and all the extensions
 @param     all_frames      recipe input frames
 @param		parameters      recipe parameters
 @param     used_frames     frames used for this product
 @param     recipe          recipe name
 @param     keywords        main header keywords
 @param     filename        product filename
 @param     image_to_save   image to be saved
 @param     image_type      product image type
 @return	CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_dfs_image_save_one_ext(cpl_frameset *all_frames,
		const cpl_parameterlist *parameters,
		const cpl_frameset *used_frames,
		const char *recipe,
		cpl_propertylist *keywords,
		const char *filename,
		cpl_image *image_to_save,
		cpl_type image_type) {

	cpl_error_code my_error = CPL_ERROR_NONE;

	my_error = cpl_dfs_save_propertylist(all_frames, NULL, parameters,
			used_frames, NULL, recipe,
			keywords, "RADECSYS",
			PACKAGE "/" PACKAGE_VERSION,
			filename);

	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Error saving primary header: %s", cpl_error_get_message());

	my_error = cpl_image_save(image_to_save,
			filename, image_type,
			NULL, CPL_IO_EXTEND);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Error saving image: %s",
			cpl_error_get_message());

	/* Verify the saved image */
	my_error = espdr_verify_fits_image(filename);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Error in verifying image %s", filename);

	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief		Save image with the complete FITS header and all the extensions
 @param     all_frames      recipe input frames
 @param		parameters      recipe parameters
 @param     used_frames     frames used for this product
 @param     recipe          recipe name
 @param     keywords        main header keywords
 @param     filename        product filename
 @param     images_to_save  list of images to be saved
 @return	CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_dfs_save_data_err_qual(cpl_frameset *all_frames,
		const cpl_parameterlist *parameters,
		const cpl_frameset *used_frames,
		const char *recipe,
		cpl_propertylist *keywords,
		const char *filename,
		cpl_image **images_to_save) {

	int i;
	cpl_error_code my_error = CPL_ERROR_NONE;
	cpl_type image_type = CPL_TYPE_FLOAT;
	cpl_propertylist *keywords_ext = NULL;

	my_error = cpl_dfs_save_propertylist(all_frames, NULL, parameters,
			used_frames, NULL, recipe,
			keywords, "RADECSYS",
			PACKAGE "/" PACKAGE_VERSION,
			filename);

	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Error saving primary header: %s",
			cpl_error_get_message());

	keywords_ext = cpl_propertylist_new();
	my_error = cpl_propertylist_append_string(keywords_ext,
			"EXTNAME", "SCIDATA");
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Error appending EXTNAME to extension header: %s",
			cpl_error_get_message());

	for (i = 0; i < 3; i++) {
		if (i > 1) {
			image_type = CPL_TYPE_INT;
		}

		if (i == 1) {
			my_error = cpl_propertylist_update_string(keywords_ext,
					"EXTNAME", "ERRDATA");
			espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
					"Error updating EXTNAME in extension header: %s",
					cpl_error_get_message());
		}

		if (i == 2) {
			my_error = cpl_propertylist_update_string(keywords_ext,
					"EXTNAME", "QUALDATA");
			espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
					"Error updating EXTNAME in extension header: %s",
					cpl_error_get_message());
		}

		my_error = cpl_image_save(images_to_save[i],
				filename, image_type,
				keywords_ext, CPL_IO_EXTEND);
		espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
				"Error saving extension %d: %s",
				i, cpl_error_get_message());
	}


	/* Verify the saved frame */
	my_error = espdr_verify_fits_image(filename);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Error in verifying image %s", filename);
    cpl_propertylist_delete(keywords_ext);

	return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 @brief		Save image with the complete FITS header and all the extensions
 @param     all_frames      recipe input frames
 @param		parameters      recipe parameters
 @param     used_frames     frames used for this product
 @param     recipe          recipe name
 @param     keywords        main header keywords
 @param     filename        product filename
 @param     images_to_save  list of images to be saved
 @return	CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_dfs_save_data_err_qual_wave(cpl_frameset *all_frames,
		const cpl_parameterlist *parameters,
		const cpl_frameset *used_frames,
		const char *recipe,
		cpl_propertylist *keywords,
		const char *filename,
		cpl_image **images_to_save) {

	int i;
	cpl_error_code my_error = CPL_ERROR_NONE;
	cpl_type image_type = CPL_TYPE_DOUBLE;
	cpl_propertylist *keywords_ext = NULL;

	my_error = cpl_dfs_save_propertylist(all_frames, NULL, parameters,
			used_frames, NULL, recipe,
			keywords, "RADECSYS",
			PACKAGE "/" PACKAGE_VERSION,
			filename);

	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Error saving primary header: %s", cpl_error_get_message());

	keywords_ext = cpl_propertylist_new();
	my_error = cpl_propertylist_append_string(keywords_ext,
			"EXTNAME", "SCIDATA");
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Error appending EXTNAME to extension header: %s",
			cpl_error_get_message());

	for (i = 0; i < 7; i++) {
        
        if (images_to_save[i] != NULL) {

            if (i < 2) {
                image_type = CPL_TYPE_FLOAT;
            }
            
            if (i == 2) {
                image_type = CPL_TYPE_USHORT;
            }
            
            if (i > 2) {
                image_type = CPL_TYPE_DOUBLE;
            }
            
            if (i == 1) {
                my_error = cpl_propertylist_update_string(keywords_ext,
                                                          "EXTNAME",
                                                          "ERRDATA");
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Error updating EXTNAME in extension header: %s",
                             cpl_error_get_message());
            }
            
            if (i == 2) {
                my_error = cpl_propertylist_update_string(keywords_ext,
                                                          "EXTNAME",
                                                          "QUALDATA");
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Error updating EXTNAME in extension header: %s",
                             cpl_error_get_message());
            }
            
            if (i == 3) {
                my_error = cpl_propertylist_update_string(keywords_ext,
                                                          "EXTNAME",
                                                          "WAVEDATA_VAC_BARY");
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Error updating EXTNAME in extension header: %s",
                             cpl_error_get_message());
            }
            
            if (i == 4) {
                my_error = cpl_propertylist_update_string(keywords_ext,
                                                          "EXTNAME",
                                                          "WAVEDATA_AIR_BARY");
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Error updating EXTNAME in extension header: %s",
                             cpl_error_get_message());
            }
            
            if (i == 5) {
                my_error = cpl_propertylist_update_string(keywords_ext,
                                                          "EXTNAME",
                                                          "DLLDATA_VAC_BARY");
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Error updating EXTNAME in extension header: %s",
                             cpl_error_get_message());
            }
            
            if (i == 6) {
                my_error = cpl_propertylist_update_string(keywords_ext,
                                                          "EXTNAME",
                                                          "DLLDATA_AIR_BARY");
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Error updating EXTNAME in extension header: %s",
                             cpl_error_get_message());
            }
            
            my_error = cpl_image_save(images_to_save[i],
                                      filename, image_type,
                                      keywords_ext, CPL_IO_EXTEND);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "Error saving extension %d: %s",
                         i, cpl_error_get_message());
        }
	}


	/* Verify the saved frame */
	my_error = espdr_verify_fits_image(filename);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Error in verifying image %s", filename);

	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Save image with the complete FITS header and all the extensions
 @param     all_frames      recipe input frames
 @param     parameters      recipe parameters
 @param     used_frames     frames used for this product
 @param     recipe          recipe name
 @param     keywords        main header keywords
 @param     filename        product filename
 @param     images_to_save  list of images to be saved
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_dfs_save_S2D(cpl_image *spectrum_img,
                                  cpl_image *error_img,
                                  cpl_image *quality_img,
                                  cpl_frameset* frameset,
                                  cpl_frameset* used_frames,
                                  cpl_parameterlist* parameters,
                                  cpl_propertylist* keywords,
                                  const char *filename,
                                  const char *pro_catg,
                                  const char *recipe) {
    
    cpl_image **images_to_save = (cpl_image**)cpl_malloc(3*sizeof(cpl_image*));
    cpl_propertylist *keywords_S2D = cpl_propertylist_duplicate(keywords);
    cpl_error_code my_error;
    
    images_to_save[0] = cpl_image_duplicate(spectrum_img);
    images_to_save[1] = cpl_image_duplicate(error_img);
    images_to_save[2] = cpl_image_duplicate(quality_img);
    
    my_error = cpl_propertylist_update_string(keywords_S2D,
                                              PRO_CATG_KW,
                                              pro_catg);
    
    my_error = espdr_dfs_save_data_err_qual(frameset, parameters,
                                            used_frames, recipe,
                                            keywords_S2D,
                                            filename,
                                            images_to_save);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_save_data_err_qual failed: %s for file %s",
                 cpl_error_get_message_default(my_error),
                 filename);
    
    for (int i = 0; i < 3; i++) {
        cpl_image_delete(images_to_save[i]);
    }
    
    cpl_free(images_to_save);
    cpl_propertylist_delete(keywords_S2D);

    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Save image with the complete FITS header and all the extensions
 @param     all_frames      recipe input frames
 @param     parameters      recipe parameters
 @param     used_frames     frames used for this product
 @param     recipe          recipe name
 @param     keywords        main header keywords
 @param     keywords_ext    extension keywords
 @param     filename        product filename
 @param     images_to_save  list of masks to be saved
 @param     CCD_geom        CCD geometry
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_mask_save(cpl_frameset *all_frames,
                               const cpl_parameterlist *parameters,
                               const cpl_frameset *used_frames,
                               const char *recipe,
                               cpl_propertylist *keywords,
                               cpl_propertylist **keywords_ext,
                               const char *filename,
                               cpl_mask **masks_to_save,
                               espdr_CCD_geometry *CCD_geom) {

    int i;
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    my_error = cpl_dfs_save_propertylist(all_frames, NULL, parameters,
                                         used_frames, NULL, recipe,
                                         keywords, NULL,
                                         PACKAGE "/" PACKAGE_VERSION,
                                         filename);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Error saving primary header: %s", cpl_error_get_message());
    
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        my_error = cpl_mask_save(masks_to_save[i], filename,
                                 keywords_ext[i], CPL_IO_EXTEND);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Error saving extension %d: %s",
                     i, cpl_error_get_message());
    }
    
    return cpl_error_get_code();
}




/*---------------------------------------------------------------------------*/
/**
 @brief     Save FITS table in ASCII format
 @param     input_imagelist     input image, outputs in the imagelist
 @return    CPL_ERROR_NONE iff OK
 
 Verify if the image is not NULL, has the right number of extensions,
 is of the right size and computes the mean.
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_save_table_ASCII(cpl_table *FITS_table,
                                      const char *filename) {
    
    
    FILE *fp;
    int rows_nb = cpl_table_get_nrow(FITS_table);
    
    if ((fp = fopen(filename, "w")) == NULL) {
        espdr_msg_error("The file %s can't be opened for writing", filename);
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    
    cpl_table_dump(FITS_table, 0, rows_nb, fp);
    
    fclose(fp);
    
    return cpl_error_get_code();
}



/*---------------------------------------------------------------------------*/
/**
 @brief         Add int keyword to the propertlylist
 @param         keyword_name    keyword name
 @param         keyword_value   keyword value
 @param         comment         keyword comment
 @param[out]    keywords_RE     propertylist returned
 @return	    CPL_ERROR_NONE iff OK

 TODO: there is no need to pass keywords_RE with a double pointer
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_keyword_add_int(const char *keyword_name,
		int keyword_value,
		const char *comment,
		cpl_propertylist **keywords_RE) {

	cpl_error_code my_error = CPL_ERROR_NONE;

	espdr_ensure(*keywords_RE == NULL, CPL_ERROR_NULL_INPUT,
			"propertylist *keywords_RE is NULL");

	double keyword_value_d = keyword_value;
	if (isnan(keyword_value_d) || isinf(keyword_value_d)) {
		keyword_value = -9999;
	}

	my_error = cpl_propertylist_update_int(*keywords_RE,
			keyword_name,
			keyword_value);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Error inserting %s keyword is %s", keyword_name,
			cpl_error_get_message_default(my_error));

	my_error = cpl_propertylist_set_comment(*keywords_RE,
			keyword_name,
			comment);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Error inserting %s keyword comment is %s", keyword_name,
			cpl_error_get_message_default(my_error));

	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief         Add double keyword to the propertlylist
 @param         keyword_name    keyword name
 @param         keyword_value   keyword value
 @param         comment         keyword comment
 @param[out]    keywords_RE     propertylist returned
 @return        CPL_ERROR_NONE iff OK

 TODO: there is no need to pass keywords_RE with a double pointer
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_keyword_add_double(const char *keyword_name,
		double keyword_value,
		const char *comment,
		cpl_propertylist **keywords_RE) {

	cpl_error_code my_error = CPL_ERROR_NONE;

	espdr_ensure(*keywords_RE == NULL, CPL_ERROR_NULL_INPUT,
			"propertylist *keywords_RE is NULL");

	if (isnan(keyword_value) || isinf(keyword_value)) {
		keyword_value = -9999.9;
	}

	my_error = cpl_propertylist_update_double(*keywords_RE,
			keyword_name,
			keyword_value);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Error inserting %s keyword is %s", keyword_name,
			cpl_error_get_message_default(my_error));

	my_error = cpl_propertylist_set_comment(*keywords_RE,
			keyword_name,
			comment);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Error inserting %s keyword comment is %s", keyword_name,
			cpl_error_get_message_default(my_error));

	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Add string keyword to the propertlylist
 @param         keyword_name    keyword name
 @param         keyword_value   keyword value
 @param         comment         keyword comment
 @param[out]    keywords_RE     propertylist returned
 @return	CPL_ERROR_NONE iff OK

 TODO: there is no need to pass keywords_RE with a double pointer
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_keyword_add_string(const char *keyword_name,
		const char *keyword_value,
		const char *comment,
		cpl_propertylist **keywords_RE) {

	cpl_error_code my_error = CPL_ERROR_NONE;

	my_error = cpl_propertylist_update_string(*keywords_RE,
			keyword_name,
			keyword_value);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Error inserting %s keyword is %s", keyword_name,
			cpl_error_get_message_default(my_error));

	my_error = cpl_propertylist_set_comment(*keywords_RE,
			keyword_name,
			comment);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"Error inserting %s keyword comment is %s", keyword_name,
			cpl_error_get_message_default(my_error));

	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Add ARCFILE keyword to the FITS header
 @param     frameset to add the keyword
 @param     property property to add the keyword
 @return    error code
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_add_arcfile_to_header(cpl_frameset *frameset,
                                           cpl_propertylist **keywords) {
    
    cpl_propertylist *keywords_aux = NULL;
    char qc_key_prefix[256];
    char *arc_filename;
    //char *pro_catg_val;
    //char *recipe;
    char *tag;
    cpl_frame *curr_frame = NULL;
    int input_frameset_size = cpl_frameset_get_size(frameset);
    int i,file_num;
    
    file_num = 0;
    for (i = 0; i < input_frameset_size; i++) {
        curr_frame = cpl_frameset_get_position(frameset, i);
        tag =  (char *)cpl_frame_get_tag(curr_frame);
        
        keywords_aux = cpl_propertylist_load(
                                             cpl_frame_get_filename(curr_frame), 0);
        espdr_ensure(keywords_aux == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                     "keywords_aux are NULL");
        
        if(cpl_propertylist_has(keywords_aux,ARCFILE)) {
            arc_filename = (char *)
            cpl_propertylist_get_string(keywords_aux,ARCFILE);
        } else {
            cpl_propertylist_delete(keywords_aux);
            continue;
        }
        
        if(cpl_propertylist_has(keywords_aux,PRO_CATG)) {
            //pro_catg_val = (char *)
            cpl_propertylist_get_string(keywords_aux,PRO_CATG);
        } else {
            cpl_propertylist_delete(keywords_aux);
            continue;
        }
        
        if(cpl_propertylist_has(keywords_aux,RECIPE)) {
            //recipe = (char *)
            cpl_propertylist_get_string(keywords_aux,RECIPE);
        } else {
            cpl_propertylist_delete(keywords_aux);
            continue;
        }
        
        file_num++;
        sprintf(qc_key_prefix,"%s_%d",
                REF_ARC,
                file_num);
        //pro_catg_val);
        cpl_propertylist_append_string(
                                       *keywords,qc_key_prefix,
                                       arc_filename);
        cpl_propertylist_set_comment(*keywords,qc_key_prefix,
                                     tag);
        //pro_catg_val);
        //recipe);
        strcpy(qc_key_prefix,"");
        
        cpl_propertylist_delete(keywords_aux);
    }
    
    return (cpl_error_get_code());
}

/*---------------------------------------------------------------------------*/
/**
 @brief     Dump property
 @param     property property to be dumped
 @return    void
 */
/*---------------------------------------------------------------------------*/
void espdr_property_dump(cpl_property *property) {
    
    const char *name = cpl_property_get_name(property);
    const char *comment = cpl_property_get_comment(property);
    
    char c;
    
    long size = cpl_property_get_size(property);
    
    cpl_type type = cpl_property_get_type(property);
    
    
    fprintf(stderr, "Property at address %p\n", property);
    fprintf(stderr, "\tname   : %p '%s'\n", name, name);
    fprintf(stderr, "\tcomment: %p '%s'\n", comment, comment);
    fprintf(stderr, "\ttype   : %#09x\n", type);
    fprintf(stderr, "\tsize   : %ld\n", size);
    fprintf(stderr, "\tvalue  : ");
    
    
    switch (type) {
        case CPL_TYPE_CHAR:
            c = cpl_property_get_char(property);
            if (!c)
                fprintf(stderr, "''");
            else
                fprintf(stderr, "'%c'", c);
            break;
            
        case CPL_TYPE_BOOL:
            fprintf(stderr, "%d", cpl_property_get_bool(property));
            break;
            
        case CPL_TYPE_INT:
            fprintf(stderr, "%d", cpl_property_get_int(property));
            break;
            
        case CPL_TYPE_LONG:
            fprintf(stderr, "%ld", cpl_property_get_long(property));
            break;
            
        case CPL_TYPE_FLOAT:
            fprintf(stderr, "%.7g", cpl_property_get_float(property));
            break;
            
        case CPL_TYPE_DOUBLE:
            fprintf(stderr, "%.15g", cpl_property_get_double(property));
            break;
            
        case CPL_TYPE_STRING:
            fprintf(stderr, "'%s'", cpl_property_get_string(property));
            break;
            
        default:
            fprintf(stderr, "unknown.");
            break;
            
    }
    
    fprintf(stderr, "\n");
    
    return;
    
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Concatenate the keyword name with a number: "KEYWORD_PREFIX OUT_x_y_z <keyword>"
 @param     keyword     keyword to add numbers
 @param     prefix      keyword prefix
 @param     extension   extension number to add
 @param     output_x    x direction output number to add
 @param     output_y    y direction output number to add
 @return concatenated keyword
 
 The function returns a string that is the conactenation of the given keyword
 name and a given number.
 */
/*---------------------------------------------------------------------------*/

char *espdr_add_ext_output_index_to_keyword(const char *keyword,
                                            const char *prefix,
                                            const int extension,
                                            const int output_x,
                                            const int output_y) {
        
    char* new_string = cpl_sprintf("%s QC EXT%d ROX%d ROY%d%s",
                                   prefix, extension, output_x, output_y, keyword);
    
    return (new_string);
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Concatenate the keyword name with a number: "KEYWORD_PREFIX OUT_x_y_z <keyword>"
 @param     keyword     keyword to add numbers
 @param     prefix      keyword prefix
 @param     extension   extension number to add
 @param     output_x    x direction output number to add
 @param     output_y    y direction output number to add
 @return concatenated keyword
 
 The function returns a string that is the conactenation of the given keyword
 name and a given number.
 */
/*---------------------------------------------------------------------------*/

char *espdr_add_ext_detector_index_to_keyword(const char *prefix,
                                              const char *keyword,
                                              const int extension,
                                              const int output_x,
                                              const int output_y) {
        
    char* new_string = cpl_sprintf("%s%d PART%d%d %s",
                                   prefix, extension, output_x, output_y, keyword);
    
    return (new_string);
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Concatenate the keyword name with a number: "<prefix> OUTx <postfix>"
 @param     prefix      keyword prefix
 @param     postfix     keyword postfix
 @param     output_nr   number of theoutput to be included in the keyword
 @return    concatenated keyword
 
 The function returns a string that is the conactenation of the given keyword
 name and a given number.
 */
/*---------------------------------------------------------------------------*/

char *espdr_add_output_index_to_keyword(const char *prefix,
                                        const char *postfix,
                                        const int output_nr) {
    
    char* new_string = cpl_sprintf("%s%d%s", prefix, output_nr, postfix);
    
    return (new_string);
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Concatenate the keyword name with 2 numbers: "<prefix> EXTx nr <postfix>"
 @param     prefix      keyword prefix
 @param     postfix     keyword postfix
 @param     output_nr   number of theoutput to be included in the keyword
 @return    concatenated keyword
 
 The function returns a string that is the conactenation of the given keyword
 name and a given number.
 */
/*---------------------------------------------------------------------------*/

char *espdr_add_ext_nr_index_to_keyword(const char *prefix,
                                        const char *postfix,
                                        const int ext_nr,
                                        const int nr) {
        
    char* new_string = cpl_sprintf("%s%d EXT%d%s", prefix, nr, ext_nr, postfix);
    
    return (new_string);
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Concatenate the keyword name with 2 numbers: "<prefix> EXTx nr <postfix>"
 @param     prefix      keyword prefix
 @param     postfix     keyword postfix
 @param     output_nr   number of theoutput to be included in the keyword
 @return    concatenated keyword
 
 The function returns a string that is the conactenation of the given keyword
 name and a given number.
 */
/*---------------------------------------------------------------------------*/

char *espdr_add_index_to_keyword(const char *prefix,
                                 const char *postfix,
                                 const int nr) {
        
    char* new_string = cpl_sprintf("%s%d %s", prefix, nr, postfix);
    
    return (new_string);
}



/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @return   MJD-OBS of the frame
 */
/*----------------------------------------------------------------------------*/

static double espdr_get_mjd_obs(cpl_frameset *frameset, char *tag) {
    
    double mjd_calib = 0.0;
    cpl_frame *calib_frame = espdr_frame_find(frameset, tag);
    if (calib_frame != NULL) {
        const char *filename = cpl_frame_get_filename(calib_frame);
        cpl_propertylist *keywords_tmp = cpl_propertylist_load(filename, 0);
        if (cpl_propertylist_has(keywords_tmp, "MJD-OBS")) {
            mjd_calib = cpl_propertylist_get_double(keywords_tmp, "MJD-OBS");
        }
        cpl_propertylist_delete(keywords_tmp);
    }
    
    return(mjd_calib);
}

/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @return   MJD-OBS of the frame
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code espdr_save_calibration_interval(cpl_frameset *frameset,
                                                      espdr_inst_config *inst_config,
                                                      cpl_propertylist *keywords,
                                                      char *frame_tag,
                                                      char *KW_tag,
                                                      char *recipe_name,
                                                      double mjd_science) {
    double mjd_calib = 0.0;
    char KW_char[32];
    char comment[128];
    
    mjd_calib = espdr_get_mjd_obs(frameset, frame_tag);
    if (mjd_calib != 0.0) {
        mjd_calib = round((mjd_calib - mjd_science) * 10000) / 10000.0;
        sprintf(KW_char, "%s QC %s DELTA_T", inst_config->prefix, KW_tag);
        cpl_propertylist_append_double(keywords, KW_char, mjd_calib);
        sprintf(comment, "%s - %s delta time [days]", KW_tag, recipe_name);
        cpl_propertylist_set_comment(keywords, KW_char, comment);
    }
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_compute_calibrations_intervals(cpl_frameset *frameset,
                                                    espdr_inst_config *inst_config,
                                                    cpl_propertylist *keywords,
                                                    char *wave_cal_source,
                                                    char *fibre_b,
                                                    double *mjd_obs_delta_time_wave,
                                                    char *recipe_name,
                                                    int *recipes) {

    double mjd_science = 0.0;
    double mjd_calib = 0.0;
    char KW_char[32];
    char frame_tag[32];
    char comment[128];
    
    if (cpl_propertylist_has(keywords, "MJD-OBS")) {
        mjd_science = cpl_propertylist_get_double(keywords, "MJD-OBS");
    }
    //espdr_msg("INTERVAL:::::: MJD science: %f", mjd_science);
    
    if (recipes[0] == 1) {
        espdr_save_calibration_interval(frameset, inst_config, keywords,
                                        ESPDR_PRO_CATG_MBIAS_RES, "BIAS",
                                        recipe_name, mjd_science);
    }
    if (recipes[1] == 1){
        espdr_save_calibration_interval(frameset, inst_config, keywords,
                                        ESPDR_PRO_CATG_HOT_PIXELS, "DARK",
                                        recipe_name, mjd_science);
    }
    if (recipes[2] == 1) {
        espdr_save_calibration_interval(frameset, inst_config, keywords,
                                        ESPDR_PRO_CATG_BAD_PIXELS, "LED_FF",
                                        recipe_name, mjd_science);
    }
    if (recipes[3] == 1) {
        espdr_save_calibration_interval(frameset, inst_config, keywords,
                                        ESPDR_PRO_CATG_ORDERS_A, "ORDERDEF_A",
                                        recipe_name, mjd_science);
        espdr_save_calibration_interval(frameset, inst_config, keywords,
                                        ESPDR_PRO_CATG_ORDERS_B, "ORDERDEF_B",
                                        recipe_name, mjd_science);
    }
    if (recipes[4] == 1) {
        espdr_save_calibration_interval(frameset, inst_config, keywords,
                                        ESPDR_PRO_CATG_FLAT_A, "FLAT_A",
                                        recipe_name, mjd_science);
        espdr_save_calibration_interval(frameset, inst_config, keywords,
                                        ESPDR_PRO_CATG_FLAT_B, "FLAT_B",
                                        recipe_name, mjd_science);
    }
    
    if (recipes[5] == 1) {
        espdr_save_calibration_interval(frameset, inst_config, keywords,
                                        ESPDR_PRO_CATG_FP_SEARCHED_LINE_TABLE_FP_FP_A,
                                        "FP_TABLE_A", recipe_name, mjd_science);
        espdr_save_calibration_interval(frameset, inst_config, keywords,
                                        ESPDR_PRO_CATG_FP_SEARCHED_LINE_TABLE_FP_FP_B,
                                        "FP_TABLE_B", recipe_name, mjd_science);
    }
    
    
    if (recipes[6] == 1) {
        for (int fibre = 0; fibre < inst_config->fibres_nb; fibre++) {
            sprintf(KW_char, "%s QC WAVE_%c DELTA_T", inst_config->prefix, fibre_name[fibre]);
            mjd_obs_delta_time_wave[fibre] = round(mjd_obs_delta_time_wave[fibre] * 10000) / 10000.0;
            cpl_propertylist_append_double(keywords, KW_char, mjd_obs_delta_time_wave[fibre]);
            sprintf(comment, "WAVE_%c - %s delta time [days]", fibre_name[fibre], recipe_name);
            cpl_propertylist_set_comment(keywords, KW_char, comment);
        }
    }
    
    if (recipes[7] == 1) {
        if ((strcmp(fibre_b, "SKY") != 0) && (strcmp(fibre_b, "DARK") != 0)) {
            sprintf(frame_tag, "%s_%s", ESPDR_PRO_CATG_CONTAM_SRC, fibre_b);
            espdr_save_calibration_interval(frameset, inst_config, keywords,
                                                       frame_tag, "CONTAM", recipe_name, mjd_science);
        }
    }
    
    if (recipes[8] == 1) {
        espdr_save_calibration_interval(frameset, inst_config, keywords,
                                                   ESPDR_PRO_CATG_REL_EFF, "EFF_AB",
                                                   recipe_name, mjd_science);
    }
    if (recipes[9] == 1) {
        espdr_save_calibration_interval(frameset, inst_config, keywords,
                                                   ESPDR_PRO_CATG_ABS_EFF, "CAL_FLUX",
                                                   recipe_name, mjd_science);
    }
        
    return cpl_error_get_code();
}




/**@}*/
