/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2019-11-28 15:13:41 $
 * $Revision: $
 * $Name: not supported by cvs2svn $
 */


#include <espdr_wave_TH_drift_cal.h>
#include <espdr_wave_THAR_cal.h>
#define KSIG_EXTRACTION_MIN -2.0
#define KSIG_EXTRACTION_MAX 20.0
/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    parameters     the parameters list
 @param    frameset   the frames list

 In case of failure the cpl_error_code is set.
 */
/*---------------------------------------------------------------------------*/

int espdr_wave_TH_drift(cpl_parameterlist *parameters,
                               cpl_frameset *frameset,
							   const char* recipe_id) {

    cpl_frameset *wave_frames_to_reduce = NULL;
    cpl_frame *wave_frame = NULL;
    cpl_frameset *used_frames = NULL;

    int *TH_saturated_lines_nb;
    int *AR_saturated_lines_nb;
    double *TH_flux_ratio_median, *TH_rv_diff_median;
    double *AR_flux_ratio_median, *AR_rv_diff_median;

    espdr_line_param **STATIC_lines_table = NULL;
    espdr_line_param **TH_lines_table = NULL;
    espdr_line_param **AR_lines_table = NULL;
    double **REF_AR_peak_pxl = NULL;
    double **REF_AR_peak_pxl_err = NULL;
    double **REF_AR_ll = NULL;
    espdr_raw_line_param **raw_lines_table = NULL;
    cpl_table **TH_lines_table_FITS = NULL;
    cpl_table **AR_lines_table_FITS = NULL;
    cpl_table **raw_lines_table_FITS = NULL;

    int *size_x, *size_y;
    int **lines_nb_per_order = NULL;
    int **AR_lines_nb_per_order = NULL;
    double **rms_per_order = NULL;
    double **chisq_per_order = NULL;
    double **resolution = NULL;
    double *resol_median = NULL;
    double *lamp_offset_ar = NULL;
    double *lamp_offset_ar1 = NULL;
    double *lamp_offset_ar2 = NULL;

	cpl_error_code my_error = CPL_ERROR_NONE;

    const int rec_ntags = 20;
    const char* rec_tags[20] = {
        ESPDR_PRO_CATG_HOT_PIXELS, ESPDR_PRO_CATG_BAD_PIXELS, ESPDR_CCD_GEOM,
        ESPDR_PRO_CATG_ORDERS_A, ESPDR_PRO_CATG_ORDERS_B,
        ESPDR_PRO_CATG_REF_LINE_TABLE_A, ESPDR_PRO_CATG_REF_LINE_TABLE_B,
        ESPDR_PRO_CATG_STATIC_LINE_TABLE_A, ESPDR_PRO_CATG_STATIC_LINE_TABLE_B,
        ESPDR_PRO_CATG_AR_LINE_TABLE_A, ESPDR_PRO_CATG_AR_LINE_TABLE_B,
        ESPDR_PRO_CATG_STATIC_WAVE_MATRIX_A, ESPDR_PRO_CATG_STATIC_WAVE_MATRIX_B,
        ESPDR_PRO_CATG_STATIC_DLL_MATRIX_A, ESPDR_PRO_CATG_STATIC_DLL_MATRIX_B,
        ESPDR_PRO_CATG_PIXEL_GEOM_A, ESPDR_PRO_CATG_PIXEL_GEOM_B,
        ESPDR_WAVE_THAR_THAR_RAW, ESPDR_WAVE_THAR_FP_RAW, ESPDR_WAVE_FP_THAR_RAW
    };
    int is_required[20] = {1, 1, 1, 1, 1,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

	cpl_msg_set_level(CPL_MSG_INFO);

	espdr_msg("Starting wave_TH_drift");

    espdr_ensure(espdr_check_input_tags(frameset,
                                        rec_tags, is_required,
                                        rec_ntags) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_inst_config(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_flat(frameset, 1) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong flat input tag!");

    /* Identify the RAW and CALIB frames in the input frameset */
    espdr_ensure(espdr_dfs_set_groups(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(),
                 "DFS setting groups failed. Expected inputs are:\n"
                 "raw wave THAR_THAR image, tagged as THAR_THAR\n"
                 "CCD geometry table, tagged as CCD_GEOM\n"
                 "instrument configuration table, tagged as MASTER_INST_CONFIG or INST_CONFIG\n"
                 "ThAr lines reference tables, tagged as REF_LINE_TABLE_A and REF_LINE_TABLE_B\n"
                 "ThAr lines static tables, tagged as STATIC_LINE_TABLE_A and STATIC_LINE_TABLE_B\n"
                 "hot pixel mask image, tagged as HOT_PIXEL_MASK\n"
                 "bad pixel mask image, tagged as BAD_PIXEL_MASK\n"
                 "orders definition images for fibres A and B, tagged as ORDER_TABLE_A & ORDER_TABLE_B\n"
                 "orders profile images for fibres A and B, tagged as ORDER_PROFILE_A & ORDER_PROFILE_B\n"
                 "flat images for fibres A and B, tagged as FLAT_A & FLAT_B\n"
                 "blaze images for fibres A and B, tagged as BLAZE_A & BLAZE_B\n"
                 "reference wave matrix for fibres A and B, tagged as STATIC_WAVE_MATRIX_A/B\n"
                 "reference dll matrix for fibres A and B, tagged as STATIC_DLL_MATRIX_A/B\n");

    espdr_CCD_geometry *CCD_geom                    = NULL;
    espdr_inst_config *inst_config                  = NULL;
    espdr_qc_keywords *qc_kws                       = NULL;
    espdr_OVSC_param *OVSC_param                    = NULL;
    espdr_WAVE_TH_drift_param *WAVE_TH_drift_param  = NULL;
    cpl_frame* CCD_geom_frame                       = NULL;
    cpl_frame* inst_config_frame                    = NULL;

    CCD_geom_frame        = espdr_frame_find(frameset, ESPDR_CCD_GEOM);
    espdr_ensure(CCD_geom_frame == NULL,cpl_error_get_code(),
                 "CCD geometry frame not found!");
    CCD_geom              = espdr_CCD_geom_init(parameters, CCD_geom_frame);
    inst_config_frame     = espdr_get_inst_config(frameset);
    espdr_ensure(inst_config_frame == NULL,cpl_error_get_code(),
                 "Instrument config frame not found!");
    inst_config           = espdr_inst_config_init(parameters,
                                               CCD_geom->ext_nb,
                                               inst_config_frame);
    OVSC_param            = espdr_parameters_OVSC_init(recipe_id, parameters);
    WAVE_TH_drift_param   = espdr_WAVE_TH_drift_param_init(recipe_id, parameters);

    used_frames = cpl_frameset_new();
    my_error = cpl_frameset_insert(used_frames, cpl_frame_duplicate(CCD_geom_frame));
    my_error = cpl_frameset_insert(used_frames, cpl_frame_duplicate(inst_config_frame));

    /* Filling up the DRS QC KWs structure */
    qc_kws = (espdr_qc_keywords *)cpl_malloc(sizeof(espdr_qc_keywords));
    my_error = espdr_fill_qc_keywords(inst_config, qc_kws);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Filling QC KWs failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_ensure(qc_kws == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "QC KWs structure is NULL");

    /* Checking the TAG of raw frame */
    wave_frame = espdr_frame_find(frameset, ESPDR_WAVE_THAR_THAR_RAW);
    if (wave_frame == NULL) {
        wave_frame = espdr_frame_find(frameset, ESPDR_WAVE_THAR_FP_RAW);
        if (wave_frame == NULL) {
            wave_frame = espdr_frame_find(frameset, ESPDR_WAVE_FP_THAR_RAW);
            if (wave_frame == NULL) {
                espdr_msg_error("No raw frame with any of the TAGs: THAR_THAR, THAR_FP, FP_THAR, exiting");
                return(CPL_ERROR_NULL_INPUT);
            }
        }
    }

    const char *WAVE_tag = cpl_frame_get_tag(wave_frame);
    my_error = cpl_frameset_insert(used_frames, cpl_frame_duplicate(wave_frame));
    wave_frames_to_reduce = cpl_frameset_new();
    cpl_frameset_insert(wave_frames_to_reduce, cpl_frame_duplicate(wave_frame));
    espdr_msg("WAVE_tag: %s", WAVE_tag);

    espdr_src_type fibre_source[inst_config->fibres_nb];
    char **fibre_WAVE = (char **)cpl_malloc(inst_config->fibres_nb * sizeof(char *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        fibre_WAVE[i] = (char *)cpl_malloc(TAG_LENGTH*sizeof(char));
    }
    my_error = espdr_get_fibre_source(WAVE_tag, fibre_source, fibre_WAVE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_get_fibre_source failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_msg("fibre_WAVE: %s & %s", fibre_WAVE[0], fibre_WAVE[1]);

    int *orders_nb_per_fibre = (int *)cpl_calloc(inst_config->fibres_nb, sizeof(int));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        orders_nb_per_fibre[i] = 0;
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            orders_nb_per_fibre[i] += inst_config->orders_nb[i*CCD_geom->ext_nb+j];
        }
    }

    cpl_image **pixel_geom_image = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **pixel_size_image = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_corr_spectrum = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_corr_error = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_corr_qual = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_blaze_corr_spectrum = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_blaze_corr_error = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_blaze_corr_qual = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_propertylist **keywords_fibre = (cpl_propertylist **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_propertylist *));
    cpl_imagelist *CCD_corrected_image = cpl_imagelist_new();
    cpl_table ***orders_coeffs = NULL;
    double *RON = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
    my_error = espdr_process_wave_till_extraction(frameset, parameters, recipe_id,
                                                  wave_frames_to_reduce,
                                                  wave_frame,
                                                  used_frames,
                                                  inst_config,
                                                  CCD_geom,
                                                  qc_kws,
                                                  orders_nb_per_fibre,
                                                  fibre_source,
                                                  fibre_WAVE,
                                                  keywords_fibre,
                                                  CCD_corrected_image,
                                                  &orders_coeffs,
                                                  RON,
                                                  flat_corr_spectrum,
                                                  flat_corr_error,
                                                  flat_corr_qual,
                                                  flat_blaze_corr_spectrum,
                                                  flat_blaze_corr_error,
                                                  flat_blaze_corr_qual,
                                                  pixel_geom_image,
                                                  pixel_size_image);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_process_wave_till_extraction failed: %s",
                 cpl_error_get_message_default(my_error));

    char pro_catg_tag[TAG_LENGTH];
    cpl_frame **REF_line_table_frame = (cpl_frame **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_frame *));
    cpl_frame **STATIC_line_table_frame = (cpl_frame **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_frame *));
    cpl_frame **AR_line_table_frame = (cpl_frame **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_frame *));
    cpl_image **ref_wave_matrix_img = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    cpl_image **ref_dll_matrix_img = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));

    my_error = espdr_extract_lines_tables(frameset, recipe_id,
                                          used_frames,
                                          inst_config,
                                          fibre_source,
                                          fibre_WAVE,
                                          0, // compute_drift
                                          ESPDR_PRO_CATG_REF_LINE_TABLE,
                                          ESPDR_PRO_CATG_STATIC_LINE_TABLE,
                                          REF_line_table_frame,
                                          STATIC_line_table_frame,
                                          ref_wave_matrix_img,
                                          ref_dll_matrix_img,
                                          NULL);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_extract_lines_tables failed: %s",
                 cpl_error_get_message_default(my_error));
    
    // Reading AR table if different then REF table
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        sprintf(pro_catg_tag, "%s_%c", ESPDR_PRO_CATG_AR_LINE_TABLE, fibre_name[i]);
        AR_line_table_frame[i] = espdr_frame_find(frameset, pro_catg_tag);
        if (AR_line_table_frame[i] == NULL) {
            AR_line_table_frame[i] = cpl_frame_duplicate(REF_line_table_frame[i]);
        }
    }
    
    /* LINES TABLE reading */
    espdr_msg("Lines tables memory allocation & initialization");

    STATIC_lines_table = (espdr_line_param **)cpl_malloc(inst_config->fibres_nb *sizeof(espdr_line_param *));
    TH_lines_table = (espdr_line_param **)cpl_malloc(inst_config->fibres_nb * sizeof(espdr_line_param *));
    AR_lines_table = (espdr_line_param **)cpl_malloc(inst_config->fibres_nb * sizeof(espdr_line_param *));
    REF_AR_peak_pxl = (double **)cpl_malloc(inst_config->fibres_nb * sizeof(double *));
    REF_AR_peak_pxl_err = (double **)cpl_malloc(inst_config->fibres_nb * sizeof(double *));
    REF_AR_ll = (double **)cpl_malloc(inst_config->fibres_nb * sizeof(double *));
    raw_lines_table = (espdr_raw_line_param **)cpl_malloc(inst_config->fibres_nb * sizeof(espdr_raw_line_param *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        if (fibre_source[i] == THAR) {
            STATIC_lines_table[i] = (espdr_line_param *)cpl_malloc(MAX_NB_SPECTRAL_LINES * sizeof(espdr_line_param));
            TH_lines_table[i] = (espdr_line_param *)cpl_malloc(MAX_NB_SPECTRAL_LINES * sizeof(espdr_line_param));
            AR_lines_table[i] = (espdr_line_param *)cpl_malloc(MAX_NB_SPECTRAL_LINES * sizeof(espdr_line_param));
            REF_AR_peak_pxl[i] = (double *)cpl_calloc(MAX_NB_SPECTRAL_LINES, sizeof(double));
            REF_AR_peak_pxl_err[i] = (double *)cpl_calloc(MAX_NB_SPECTRAL_LINES, sizeof(double));
            REF_AR_ll[i] = (double *)cpl_calloc(MAX_NB_SPECTRAL_LINES, sizeof(double));

            my_error = espdr_lines_table_init(STATIC_lines_table[i], MAX_NB_SPECTRAL_LINES);
            my_error = espdr_lines_table_init(TH_lines_table[i], MAX_NB_SPECTRAL_LINES);
            my_error = espdr_lines_table_init(AR_lines_table[i], MAX_NB_SPECTRAL_LINES);
            raw_lines_table[i] = (espdr_raw_line_param *)cpl_malloc(MAX_NB_SPECTRAL_LINES * sizeof(espdr_raw_line_param));
            my_error = espdr_raw_lines_table_init(raw_lines_table[i], MAX_NB_SPECTRAL_LINES);
        }
    }
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "lines table init for STATIC, TH, AR or raw lines table failed: %s",
                 cpl_error_get_message_default(my_error));

    int STATIC_lines_nb[inst_config->fibres_nb];
    int TH_lines_nb[inst_config->fibres_nb];
    int AR_lines_nb[inst_config->fibres_nb];
    int raw_lines_nb[inst_config->fibres_nb];
    lines_nb_per_order = (int **)cpl_malloc(inst_config->fibres_nb*sizeof(int *));
    AR_lines_nb_per_order = (int **)cpl_malloc(inst_config->fibres_nb*sizeof(int *));
    rms_per_order = (double **)cpl_malloc(inst_config->fibres_nb*sizeof(double *));
    chisq_per_order = (double **)cpl_malloc(inst_config->fibres_nb*sizeof(double *));
    size_x = (int *)cpl_calloc(inst_config->fibres_nb, sizeof(int *));
    size_y = (int *)cpl_calloc(inst_config->fibres_nb, sizeof(int *));

    for (int i = 0; i < inst_config->fibres_nb; i++) {
        STATIC_lines_nb[i] = 0;
        TH_lines_nb[i] = 0;
        AR_lines_nb[i] = 0;
        raw_lines_nb[i] = 0;
        size_x[i] = cpl_image_get_size_x(flat_blaze_corr_spectrum[i]);
        size_y[i] = cpl_image_get_size_y(flat_blaze_corr_spectrum[i]);
        lines_nb_per_order[i] = (int *)cpl_calloc(orders_nb_per_fibre[i], sizeof(int));
        AR_lines_nb_per_order[i] = (int *)cpl_calloc(orders_nb_per_fibre[i], sizeof(int));
        rms_per_order[i] = (double *)cpl_calloc(orders_nb_per_fibre[i], sizeof(double));
        chisq_per_order[i] = (double *)cpl_calloc(orders_nb_per_fibre[i], sizeof(double));
    }

    TH_saturated_lines_nb = (int *)cpl_calloc(inst_config->fibres_nb, sizeof(int));
    TH_flux_ratio_median = (double *)cpl_calloc(inst_config->fibres_nb, sizeof(double));
    TH_rv_diff_median = (double *)cpl_calloc(inst_config->fibres_nb, sizeof(double));
    AR_saturated_lines_nb = (int *)cpl_calloc(inst_config->fibres_nb, sizeof(int));
    AR_flux_ratio_median = (double *)cpl_calloc(inst_config->fibres_nb, sizeof(double));
    AR_rv_diff_median = (double *)cpl_calloc(inst_config->fibres_nb, sizeof(double));
    resolution = (double **)cpl_calloc(inst_config->fibres_nb, sizeof(double *));
    resol_median = (double *)cpl_calloc(inst_config->fibres_nb, sizeof(double));
    lamp_offset_ar = (double *)cpl_calloc(inst_config->fibres_nb, sizeof(double));
    lamp_offset_ar1 = (double *)cpl_calloc(inst_config->fibres_nb, sizeof(double));
    lamp_offset_ar2 = (double *)cpl_calloc(inst_config->fibres_nb, sizeof(double));

    for (int i = 0; i < inst_config->fibres_nb; i++) {
        if (fibre_source[i] == THAR) {
            espdr_msg("Reading STATIC line table");
            // Read the STATIC lines table, which was fitted on the reference S2D
            //my_error = espdr_read_REF_DRIFT_LINES_TABLE_ASCII(
            my_error = espdr_read_REF_DRIFT_LINES_TABLE(
                                    STATIC_line_table_frame[i],
                                    STATIC_lines_table[i],
                                    &STATIC_lines_nb[i],
                                    'T');
            espdr_msg("STATIC lines table for fibre %c has %d lines",
                      fibre_name[i], STATIC_lines_nb[i]);

#if SAVE_DEBUG_PRODUCT_WAVE_TH_DRIFT
            my_error = espdr_save_debug_table_RDB_short(STATIC_lines_table[i], STATIC_lines_nb[i],
                                                        "STATIC_READ_LINE_TABLE",
                                                        i, inst_config);
#endif
            espdr_msg("Reading TH line table");
            //my_error = espdr_read_REF_DRIFT_LINES_TABLE_ASCII(
            //my_error = espdr_read_REF_DRIFT_LINES_TABLE_CSV(
            my_error = espdr_read_REF_DRIFT_LINES_TABLE(
                                    REF_line_table_frame[i],
                                    TH_lines_table[i],
                                    &TH_lines_nb[i],
                                    'T');
            espdr_msg("TH lines table for fibre %c has %d lines",
                      fibre_name[i], TH_lines_nb[i]);

#if SAVE_DEBUG_PRODUCT_WAVE_TH_DRIFT
            my_error = espdr_save_debug_table_RDB_short(TH_lines_table[i], TH_lines_nb[i],
                                                        "TH_REF_READ_LINE_TABLE",
                                                        i, inst_config);
#endif

            // Check of REF lines tables
            int j_limit = ESPDR_MAX(TH_lines_nb[i], STATIC_lines_nb[i]);
            for (int j = 0; j < j_limit; j++) {
                if ((STATIC_lines_table[i][j].order != TH_lines_table[i][j].order) ||
                    (fabs(STATIC_lines_table[i][j].wavelength - TH_lines_table[i][j].wavelength) > 0.0001)) {
                    espdr_msg("STATIC: %d %f %f\tTH: %d %f %f, pxl_diff: %f",
                              STATIC_lines_table[i][j].order, STATIC_lines_table[i][j].peak_pxl,
                              STATIC_lines_table[i][j].wavelength, TH_lines_table[i][j].order,
                              TH_lines_table[i][j].peak_pxl, TH_lines_table[i][j].wavelength,
                              fabs(STATIC_lines_table[i][j].peak_pxl - TH_lines_table[i][j].peak_pxl));
                    espdr_msg_error("The two line tables: STATIC & REF are not aligned, exiting");
                    return (CPL_ERROR_INCOMPATIBLE_INPUT);
                }
            }

            // Fit THAR lines
            espdr_ensure((flat_blaze_corr_spectrum[i] == NULL) ||
                         (flat_blaze_corr_error[i] == NULL) ||
                         (flat_blaze_corr_qual[i] == NULL),
                         CPL_ERROR_NULL_INPUT,
                         "FLAT & BLAZE corrected spectrum for fibre %c is NULL", fibre_name[i]);

            espdr_msg("Fitting THAR lines");
            my_error = espdr_fit_REF_lines(TH_lines_table[i],
                                           TH_lines_nb[i],
                                           flat_blaze_corr_spectrum[i],
                                           flat_blaze_corr_error[i],
                                           flat_blaze_corr_qual[i],
                                           pixel_geom_image[i],
                                           inst_config->thar_peak_flux_threshold,
                                           inst_config->thar_thar_fit_window_size,
                                           inst_config->thar_fit_tolerance_window_size,
                                           lines_nb_per_order[i],
                                           &TH_saturated_lines_nb[i],
                                           &TH_flux_ratio_median[i],
                                           &TH_rv_diff_median[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_fit_REF_lines failed: %s",
                         cpl_error_get_message_default(my_error));
            espdr_msg("espdr_fit_lines done");
        
            my_error = espdr_THAR_lines_qc(TH_lines_table[i],
                                           TH_lines_nb[i],
                                           inst_config,
                                           lines_nb_per_order[i]);
            int valid_TH_lines_nb = 0;
            for (int j = 0; j < TH_lines_nb[i]; j++) {
                if (TH_lines_table[i][j].qc_flag == 1) {
                    valid_TH_lines_nb++;
                }
            }
            espdr_msg("Number of valid TH lines after the fit: %d", valid_TH_lines_nb);

#if SAVE_DEBUG_PRODUCT_WAVE_TH_DRIFT
            my_error = espdr_save_debug_table_RDB(TH_lines_table[i], TH_lines_nb[i],
                                                  "TH_REF_FITTED_LINE_TABLE",
                                                  i, inst_config);
#endif

            // Compute the resolution of Thorium lines
            resolution[i] = (double *)cpl_calloc(TH_lines_nb[i], sizeof(double));;
            my_error = espdr_compute_THAR_resolution(TH_lines_table[i],
                                                     resolution[i],
                                                     TH_lines_nb[i],
                                                     &resol_median[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_compute_THAR_resolution failed: %s",
                         cpl_error_get_message_default(my_error));

#if PROCESS_RAW_LINES_IN_DRIFT
            /* Fitting raw lines */
            espdr_msg("Fitting raw lines for fibre %c", fibre_name[i]);
            my_error = espdr_fit_THAR_raw_lines(TH_lines_table[i],
                                                TH_lines_nb[i],
                                                CCD_corrected_image,
                                                i,
                                                orders_coeffs[i],
                                                pixel_geom_image[i],
                                                inst_config->thar_thar_fit_window_size,
                                                inst_config,
                                                CCD_geom,
                                                RON,
                                                &raw_lines_nb[i],
                                                raw_lines_table[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_fit_THAR_raw_lines failed for fibre %c: %s",
                         fibre_name[i], cpl_error_get_message_default(my_error));

            raw_lines_table_FITS[i] = cpl_table_new(raw_lines_nb[i]);
            my_error = espdr_FITS_raw_lines_table_create(raw_lines_table_FITS[i]);
            my_error = espdr_FITS_raw_lines_table_fill(raw_lines_table_FITS[i],
                                                       raw_lines_table[i],
                                                       raw_lines_nb[i]);
            espdr_msg("Raw lines table has %lld rows",
                      cpl_table_get_nrow(raw_lines_table_FITS[i]));
#endif

            // AR
            // Read the static lines table and put it into lines_table
            espdr_msg("Reading AR line table");
            //my_error = espdr_read_REF_DRIFT_LINES_TABLE_ASCII(
            //my_error = espdr_read_REF_DRIFT_LINES_TABLE_CSV(
            my_error = espdr_read_REF_DRIFT_LINES_TABLE(
                                AR_line_table_frame[i],
                                AR_lines_table[i],
                                &AR_lines_nb[i],
                                'A');
            espdr_msg("AR lines nb for fibre %c: %d",
                      fibre_name[i], AR_lines_nb[i]);

            for (int j = 0; j < AR_lines_nb[i]; j++) {
                int pxl;
                REF_AR_peak_pxl[i][j] = AR_lines_table[i][j].peak_pxl;
                REF_AR_peak_pxl_err[i][j] = AR_lines_table[i][j].peak_pxl_err;
                double part_i, part_f;
                part_f = modf(AR_lines_table[i][j].peak_pxl, &part_i);
                REF_AR_ll[i][j] = cpl_image_get(ref_wave_matrix_img[i], (int)part_i,
                                                AR_lines_table[i][j].order, &pxl) +
                                    cpl_image_get(ref_dll_matrix_img[i], (int)part_i,
                                                  AR_lines_table[i][j].order, &pxl) * part_f;
                //espdr_msg("order: %d, pxl: %f, modf: %f and %f, ll_cat = %f, ll = %f, diff_ll = %f",
                //          AR_lines_table[i][j].order, AR_lines_table[i][j].peak_pxl,
                //          part_i, part_f, AR_lines_table[i][j].wavelength, REF_AR_ll[i][j],
                //          AR_lines_table[i][j].wavelength - REF_AR_ll[i][j]);
            }

            espdr_msg("Fitting AR lines");
            my_error = espdr_fit_REF_lines(AR_lines_table[i],
                                           AR_lines_nb[i],
                                           flat_blaze_corr_spectrum[i],
                                           flat_blaze_corr_error[i],
                                           flat_blaze_corr_qual[i],
                                           pixel_geom_image[i],
                                           inst_config->thar_peak_flux_threshold,
                                           inst_config->thar_thar_fit_window_size,
                                           inst_config->thar_fit_tolerance_window_size,
                                           AR_lines_nb_per_order[i],
                                           &AR_saturated_lines_nb[i],
                                           &AR_flux_ratio_median[i],
                                           &AR_rv_diff_median[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_fit_REF_lines failed: %s",
                         cpl_error_get_message_default(my_error));
            espdr_msg("espdr_fit_lines done");

            my_error = espdr_THAR_lines_qc(AR_lines_table[i],
                                           AR_lines_nb[i],
                                           inst_config,
                                           AR_lines_nb_per_order[i]);
            
            
#if SAVE_DEBUG_PRODUCT_WAVE_TH_DRIFT
            my_error = espdr_save_debug_table_RDB(AR_lines_table[i], AR_lines_nb[i],
                                                  "AR_REF_FITTED_LINE_TABLE",
                                                  i, inst_config);
#endif

        }
    }

    espdr_msg("Starting the drift calculation");

    espdr_line_param **diff_lines_table = (espdr_line_param **)cpl_malloc
                                    (inst_config->fibres_nb * sizeof(espdr_line_param *));
    int diff_lines_nb[inst_config->fibres_nb];
    int invalid_diff_lines_nb = 0;
    int valid_lines_nb[inst_config->fibres_nb][CCD_geom->ext_nb];
    int rejected_lines_nb[inst_config->fibres_nb][CCD_geom->ext_nb];
    double fraction_rejected[inst_config->fibres_nb][CCD_geom->ext_nb];
    double MAD[inst_config->fibres_nb][CCD_geom->ext_nb];
    double MAD2[inst_config->fibres_nb][CCD_geom->ext_nb];
    double RMS[inst_config->fibres_nb][CCD_geom->ext_nb];
    double RMS2[inst_config->fibres_nb][CCD_geom->ext_nb];
    double CHI2[inst_config->fibres_nb][CCD_geom->ext_nb];
    int *rejected_lines_nb9[inst_config->fibres_nb][CCD_geom->ext_nb];
    double *MAD9[inst_config->fibres_nb][CCD_geom->ext_nb];
    double *RMS9[inst_config->fibres_nb][CCD_geom->ext_nb];
    int *rejected_lines_nb99[inst_config->fibres_nb][CCD_geom->ext_nb];
    double *MAD99[inst_config->fibres_nb][CCD_geom->ext_nb];
    double *RMS99[inst_config->fibres_nb][CCD_geom->ext_nb];

    for (int i = 0; i < inst_config->fibres_nb; i++) {
        if (fibre_source[i] == THAR) {
            diff_lines_nb[i] = ESPDR_MAX(TH_lines_nb[i], STATIC_lines_nb[i]);
            diff_lines_table[i] = (espdr_line_param *)cpl_malloc
                                    (diff_lines_nb[i] * sizeof(espdr_line_param));
            for (int ext = 0; ext < CCD_geom->ext_nb; ext++) {
                valid_lines_nb[i][ext] = 0;
            }

            my_error = espdr_compute_diff_lines_table(STATIC_lines_table[i],
                                                      STATIC_lines_nb[i],
                                                      TH_lines_table[i],
                                                      TH_lines_nb[i],
                                                      diff_lines_nb[i],
                                                      i, inst_config, CCD_geom,
                                                      diff_lines_table[i],
                                                      valid_lines_nb[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_compute_diff_lines_table failed: %s for fibre %c",
                         cpl_error_get_message_default(my_error),
                         fibre_name[i]);
        }
    }

    cpl_image **drift_matrix = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **new_wave_matrix = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    cpl_image **new_dll_matrix = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    cpl_image **new_air_wave_matrix = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    cpl_image **new_air_dll_matrix = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));

    for (int i = 0; i < inst_config->fibres_nb; i++) {
        if (fibre_source[i] == THAR) {
            double **coeffs = (double **)cpl_malloc(CCD_geom->ext_nb * sizeof(double *));
            for (int ext = 0; ext < CCD_geom->ext_nb; ext++) {
                coeffs[ext] = (double *) cpl_calloc((inst_config->th_drift_fit_deg_x+1)*
                                                    (inst_config->th_drift_fit_deg_y+1), sizeof(double));
                rejected_lines_nb[i][ext] = 0;
                MAD[i][ext] = 0.0;
                CHI2[i][ext] = 0.0;
                rejected_lines_nb9[i][ext] = (int *)cpl_calloc(9, sizeof(int));
                MAD9[i][ext] = (double *)cpl_calloc(9, sizeof(double));
                RMS9[i][ext] = (double *)cpl_calloc(9, sizeof(double));
                rejected_lines_nb99[i][ext] = (int *)cpl_calloc(9, sizeof(int));
                MAD99[i][ext] = (double *)cpl_calloc(9, sizeof(double));
                RMS99[i][ext] = (double *)cpl_calloc(9, sizeof(double));
                for (int j = 0; j < 9; j++) {
                    rejected_lines_nb9[i][ext][j] = 0;
                    MAD9[i][ext][j] = 0.0;
                    RMS9[i][ext][j] = 0.0;
                    rejected_lines_nb99[i][ext][j] = 0;
                    MAD99[i][ext][j] = 0.0;
                    RMS99[i][ext][j] = 0.0;
                }
            }

            for (int j = 0; j < diff_lines_nb[i]; j++) {
                if (diff_lines_table[i][j].order == -1) {
                    espdr_msg("diff_line %d, qc %d, order %d - pxl(ref): %f, pxl(thar): %f  \t%f\t%f\t%f\t%f\t%f\t%f\t%d\t%f\t%f\t%s",
                              j, diff_lines_table[i][j].qc_flag, diff_lines_table[i][j].order, STATIC_lines_table[i][j].peak_pxl,
                              TH_lines_table[i][j].peak_pxl,
                              diff_lines_table[i][j].peak_pxl, diff_lines_table[i][j].peak_pxl_err,
                              diff_lines_table[i][j].fwhm, diff_lines_table[i][j].fwhm_err,
                              diff_lines_table[i][j].peak_flux, diff_lines_table[i][j].peak_flux_err,
                              diff_lines_table[i][j].qc_flag, diff_lines_table[i][j].wavelength,
                              diff_lines_table[i][j].dispersion, diff_lines_table[i][j].element_name);
                }
            }

            my_error = espdr_fit_drift_matrix(STATIC_lines_table[i], TH_lines_table[i],
                                              diff_lines_table[i], diff_lines_nb[i],
                                              valid_lines_nb[i],
                                              ref_wave_matrix_img[i],
                                              ref_dll_matrix_img[i],
                                              inst_config, CCD_geom,
                                              size_x[i], size_y[i],
                                              orders_nb_per_fibre[i], i,
                                              rejected_lines_nb[i],
                                              MAD[i], MAD2[i],
                                              RMS[i], RMS2[i],
                                              coeffs,
                                              CHI2[i],
                                              MAD9[i],
                                              RMS9[i],
                                              rejected_lines_nb9[i],
                                              MAD99[i],
                                              RMS99[i],
                                              rejected_lines_nb99[i],
                                              drift_matrix[i],
                                              &new_wave_matrix[i],
                                              &new_dll_matrix[i],
                                              &new_air_wave_matrix[i],
                                              &new_air_dll_matrix[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_fit_drift_matrix failed: %s for fibre %c",
                         cpl_error_get_message_default(my_error),
                         fibre_name[i]);

#if SAVE_DEBUG_PRODUCT_WAVE_TH_DRIFT
            my_error = espdr_save_debug_table_RDB(TH_lines_table[i], TH_lines_nb[i],
                                                  "TH_REF_QC_CHECKED_LINE_TABLE",
                                                  i, inst_config);
#endif

            for (int ext = 0; ext < CCD_geom->ext_nb; ext++) {
                fraction_rejected[i][ext] = (double)rejected_lines_nb[i][ext] /
                                            (double)valid_lines_nb[i][ext];
                espdr_msg("Ext %d: Rejected lines nb: %d vs valid lines nb: %d, rejected fraction: %f",
                          ext, rejected_lines_nb[i][ext], valid_lines_nb[i][ext],
                          fraction_rejected[i][ext]);
                espdr_msg("MAD of residuals: %f", MAD[i][ext]);
            }
            
            my_error = espdr_compute_THAR_lamp_offset(AR_lines_table[i],
                                                      AR_lines_nb[i],
                                                      REF_AR_peak_pxl_err[i],
                                                      REF_AR_ll[i],
                                                      coeffs,
                                                      new_wave_matrix[i],
                                                      new_dll_matrix[i],
                                                      CCD_geom,
                                                      inst_config,
                                                      i,
                                                      &lamp_offset_ar[i],
                                                      &lamp_offset_ar1[i],
                                                      &lamp_offset_ar2[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_compute_THAR_lamp_offset failed: %s for fibre %c",
                         cpl_error_get_message_default(my_error),
                         fibre_name[i]);
            espdr_msg("Lamp offset for fibre %c is %f (AR), %f (AR1), %f (AR2)",
                      fibre_name[i], lamp_offset_ar[i], lamp_offset_ar1[i], lamp_offset_ar2[i]);

            for (int ext = 0; ext < CCD_geom->ext_nb; ext++) {
                cpl_free(coeffs[ext]);
            }
            cpl_free(coeffs);
            
            double AR_offset = 0.0;
            if (strcmp(inst_config->instrument, "HARPS") == 0) {
                espdr_msg("Lamp offset AR1 is applied");
                AR_offset = lamp_offset_ar1[i];
            }
            else if (strcmp(inst_config->instrument, "HARPN") == 0) {
                espdr_msg("Lamp offset AR1 is applied");
                AR_offset = lamp_offset_ar1[i];
            }
            else {
                espdr_msg("Lamp offset AR is applied");
                AR_offset = lamp_offset_ar[i];
            }
            my_error = cpl_image_multiply_scalar(new_wave_matrix[i],
                                                 1.0 + AR_offset/LIGHT_SPEED);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_image_multiply_scalar failed: %s for fibre %c",
                         cpl_error_get_message_default(my_error),
                         fibre_name[i]);
            my_error = cpl_image_multiply_scalar(new_air_wave_matrix[i],
                                                 1.0 + AR_offset/LIGHT_SPEED);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_image_multiply_scalar failed: %s for fibre %c",
                         cpl_error_get_message_default(my_error),
                         fibre_name[i]);
        }
    }

    for (int i = 0; i < inst_config->fibres_nb; i++) {
        espdr_msg("Adding QC KWs");
        if (fibre_source[i] == THAR) {
            int valid_THAR_lines_nb = 0, nonvalid_THAR_lines_nb = 0;
            for (int j = 0; j < TH_lines_nb[i]; j++) {
                if (TH_lines_table[i][j].qc_flag == 1) {
                    valid_THAR_lines_nb++;
                } else {
                    nonvalid_THAR_lines_nb++;
                }
            }
            espdr_msg("Valid THAR lines (fibre %c) nb: %d, nonvalid: %d",
                      fibre_name[i], valid_THAR_lines_nb, nonvalid_THAR_lines_nb);

            my_error = espdr_wave_TH_drift_QC(lines_nb_per_order[i],
                                              resol_median[i],
                                              lamp_offset_ar[i],
                                              lamp_offset_ar1[i],
                                              lamp_offset_ar2[i],
                                              orders_nb_per_fibre[i],
                                              TH_saturated_lines_nb[i],
                                              valid_THAR_lines_nb,
                                              TH_flux_ratio_median[i],
                                              TH_rv_diff_median[i],
                                              fraction_rejected[i],
                                              MAD[i], MAD2[i],
                                              RMS[i], RMS2[i],
                                              CHI2[i],
                                              MAD9[i],
                                              RMS9[i],
                                              rejected_lines_nb9[i],
                                              MAD99[i],
                                              RMS99[i],
                                              rejected_lines_nb99[i],
                                              fibre_source[i],
                                              i,
                                              new_wave_matrix[i],
                                              flat_corr_qual[i],
                                              inst_config, CCD_geom, qc_kws,
                                              keywords_fibre[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_wave_QC failed: %s for fibre %c",
                         cpl_error_get_message_default(my_error),
                         fibre_name[i]);
        }
    }

    TH_lines_table_FITS = (cpl_table **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_table *));
    AR_lines_table_FITS = (cpl_table **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_table *));
    raw_lines_table_FITS = (cpl_table **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_table *));

    for (int i = 0; i < inst_config->fibres_nb; i++) {

        my_error = espdr_save_wave_S2D_products(frameset, parameters, recipe_id,
                                                used_frames, WAVE_tag, i,
                                                keywords_fibre[i],
                                                inst_config,
                                                flat_corr_spectrum[i],
                                                flat_corr_error[i],
                                                flat_corr_qual[i],
                                                flat_blaze_corr_spectrum[i],
                                                flat_blaze_corr_error[i],
                                                flat_blaze_corr_qual[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_save_wave_S2D_products failed: %s for fibre %c",
                     cpl_error_get_message_default(my_error), fibre_name[i]);

        if (fibre_source[i] == THAR) {
            TH_lines_table_FITS[i] = cpl_table_new(TH_lines_nb[i]);
            int save_table = 0;
#if SAVE_TH_REF_TABLES_IN_DRIFT
            save_table = 1;
#endif
            my_error = espdr_FITS_lines_table_resol_create(TH_lines_table_FITS[i]);
            my_error = espdr_FITS_lines_table_resol_fill(TH_lines_table_FITS[i],
                                                         TH_lines_table[i],
                                                         resolution[i],
                                                         TH_lines_nb[i],
                                                         save_table);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "FITS TH table creation failed: %s",
                         cpl_error_get_message_default(my_error));

            AR_lines_table_FITS[i] = cpl_table_new(AR_lines_nb[i]);
            my_error = espdr_FITS_lines_table_resol_create(AR_lines_table_FITS[i]);
            my_error = espdr_FITS_lines_table_resol_fill(AR_lines_table_FITS[i],
                                                         AR_lines_table[i],
                                                         resolution[i],
                                                         AR_lines_nb[i],
                                                         0);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "FITS AR table creation failed: %s",
                         cpl_error_get_message_default(my_error));

            for (int j = 0; j < diff_lines_nb[i]; j++) {
                if (diff_lines_table[i][j].order == -1) {
                    espdr_msg("diff_line ord %d (%.4f-%.4f) = %.4f",
                              diff_lines_table[i][j].order,
                              TH_lines_table[i][j].peak_pxl,
                              STATIC_lines_table[i][j].peak_pxl,
                              diff_lines_table[i][j].peak_pxl);
                }
            }

            // Saving products
            my_error = espdr_save_wave_TH_drift_products(frameset, parameters, recipe_id,
                                                         used_frames, WAVE_tag, i,
                                                         keywords_fibre[i],
                                                         inst_config,
                                                         new_wave_matrix[i],
                                                         new_air_wave_matrix[i],
                                                         new_dll_matrix[i],
                                                         new_air_dll_matrix[i],
                                                         TH_lines_table_FITS[i],
                                                         AR_lines_table_FITS[i],
                                                         raw_lines_table_FITS[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_save_wave_THAR_products failed: %s for fibre %c",
                         cpl_error_get_message_default(my_error),
                         fibre_name[i]);

#if PROCESS_RAW_LINES_IN_DRIFT && SAVE_DEBUG_PRODUCT_WAVE_TH_DRIFT
            my_error = espdr_save_debug_raw_table_RDB(raw_lines_table[i],
                                                      raw_lines_nb[i],
                                                      "LINE_TABLE_RAW",
                                                      i, inst_config);
#endif

        }
    } // for loop on fibres

    espdr_msg("Cleaning the memory");
    espdr_msg("Cleaning keywords lists");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        if (fibre_source[i] == THAR) {
            cpl_propertylist_delete(keywords_fibre[i]);
        }
    }

	cpl_free(qc_kws);

    espdr_msg("Cleaning calibration products");
	cpl_imagelist_delete(CCD_corrected_image);

    for (int i = 0; i < inst_config->fibres_nb; i++) {
        if (fibre_source[i] == THAR) {
            cpl_image_delete(ref_wave_matrix_img[i]);
            cpl_image_delete(ref_dll_matrix_img[i]);
        }
    }
    cpl_free(ref_wave_matrix_img);
    cpl_free(ref_dll_matrix_img);

    cpl_frameset_delete(used_frames);

    espdr_msg("Cleaning orders coeffs");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            cpl_table_delete(orders_coeffs[i][j]);
        }
        cpl_free(orders_coeffs[i]);
    }
    cpl_free(orders_coeffs);

    espdr_msg("Cleaning products");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_image_delete(flat_corr_spectrum[i]);
        cpl_image_delete(flat_corr_error[i]);
        cpl_image_delete(flat_corr_qual[i]);
    }
    cpl_free(flat_corr_spectrum);
    cpl_free(flat_corr_error);
    cpl_free(flat_corr_qual);

    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_image_delete(flat_blaze_corr_spectrum[i]);
        cpl_image_delete(flat_blaze_corr_error[i]);
        cpl_image_delete(flat_blaze_corr_qual[i]);
    }
    cpl_free(flat_blaze_corr_spectrum);
    cpl_free(flat_blaze_corr_error);
    cpl_free(flat_blaze_corr_qual);

    espdr_msg("Cleaning pixel geometry pointers");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_image_delete(pixel_geom_image[i]);
        cpl_image_delete(pixel_size_image[i]);
    }
    cpl_free(pixel_geom_image);
    cpl_free(pixel_size_image);

    espdr_msg("Cleaning products images");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        if (fibre_source[i] == THAR) {
            cpl_image_delete(new_wave_matrix[i]);
            cpl_image_unwrap(new_dll_matrix[i]);
            cpl_image_delete(new_air_wave_matrix[i]);
            cpl_image_delete(new_air_dll_matrix[i]);
        }
    }
    cpl_free(new_wave_matrix);
    cpl_free(new_dll_matrix);
    cpl_free(new_air_wave_matrix);
    cpl_free(new_air_dll_matrix);

    espdr_msg("Cleaning global variables");
    cpl_free(RON);

    espdr_msg("Cleaning global QC structures");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_free(lines_nb_per_order[i]);
        cpl_free(AR_lines_nb_per_order[i]);
        cpl_free(rms_per_order[i]);
        cpl_free(chisq_per_order[i]);
    }

    cpl_free(lines_nb_per_order);
    cpl_free(AR_lines_nb_per_order);
    cpl_free(rms_per_order);
    cpl_free(chisq_per_order);
    cpl_free(size_x);
    cpl_free(size_y);

    cpl_free(orders_nb_per_fibre);

    espdr_msg("Cleaning additional data tables");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_free(resolution[i]);
    }
    cpl_free(resolution);
    cpl_free(resol_median);
    cpl_free(lamp_offset_ar);
    cpl_free(lamp_offset_ar1);
    cpl_free(lamp_offset_ar2);

    cpl_free(TH_saturated_lines_nb);
    cpl_free(TH_flux_ratio_median);
    cpl_free(TH_rv_diff_median);
    cpl_free(AR_saturated_lines_nb);
    cpl_free(AR_flux_ratio_median);
    cpl_free(AR_rv_diff_median);

    espdr_msg("Cleaning lines tables");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        if (fibre_source[i] == THAR) {
            cpl_free(STATIC_lines_table[i]);
            cpl_free(TH_lines_table[i]);
            cpl_free(AR_lines_table[i]);
            cpl_free(REF_AR_peak_pxl[i]);
            cpl_free(REF_AR_peak_pxl_err[i]);
            cpl_free(REF_AR_ll[i]);
            cpl_free(diff_lines_table[i]);
            cpl_free(raw_lines_table[i]);
        }
    }
    cpl_free(STATIC_lines_table);
    cpl_free(TH_lines_table);
    cpl_free(AR_lines_table);
    cpl_free(REF_AR_peak_pxl);
    cpl_free(REF_AR_peak_pxl_err);
    cpl_free(REF_AR_ll);
    cpl_free(diff_lines_table);
    cpl_free(raw_lines_table);

    espdr_msg("Cleaning FITS tables");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        if (fibre_source[i] == THAR) {
            cpl_table_delete(TH_lines_table_FITS[i]);
            cpl_table_delete(AR_lines_table_FITS[i]);
#if PROCESS_RAW_LINES_IN_DRIFT
            cpl_table_delete(raw_lines_table_FITS[i]);
#endif
        }
    }
    cpl_free(TH_lines_table_FITS);
    cpl_free(AR_lines_table_FITS);
#if PROCESS_RAW_LINES_IN_DRIFT
    cpl_free(raw_lines_table_FITS);
#endif

    espdr_msg("Cleaning parameters");
    espdr_parameters_OVSC_delete(OVSC_param);
    espdr_parameters_WAVE_TH_drift_delete(WAVE_TH_drift_param);
    espdr_parameters_CCD_geometry_delete(CCD_geom);
    espdr_parameters_inst_config_delete(inst_config);

    espdr_msg("End of the recipe espdr_wave_TH_DRIFT");

	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create the structure for WAVE input parameters
 @param    recipe_id	recipe identifier
 @param    list			list of input parameters
 @param    p			input WAVE parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_TH_drift_create(const char* recipe_id,
                                                     cpl_parameterlist *list,
                                                     espdr_WAVE_TH_drift_param *p) {
    
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    cpl_error_code my_error = cpl_error_get_code();
	
	/* Fill the parameter list */
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "The parameters list NULL");
	double ksig_extraction_min = KSIG_EXTRACTION_MIN;
	double ksig_extraction_max = KSIG_EXTRACTION_MAX;
    
	/* Fill the parameter list */
    cpl_error_code error_got = CPL_ERROR_NONE;
    char comment[COMMENT_LENGTH];
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "thar_extraction_method",
                                            p->extraction_method,
                                            "Method used to extract orders");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter extraction_method to the list");
    
    sprintf(comment, "ksigma for extraction, must be between: %.2lf and %.2lf",
            ksig_extraction_min, ksig_extraction_max);
    error_got = espdr_parameters_new_range_double(recipe_id, list,
                                                  "thar_extraction_ksigma",
                                                  p->ksig_extraction,
                                                  KSIG_EXTRACTION_MIN,
                                                  KSIG_EXTRACTION_MAX,
                                                  comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter ksig_extraction to the list");
    
    return (CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the structure for ORDERDEF input parameters
 @param    p	input ORDERDEF parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_TH_drift_delete(espdr_WAVE_TH_drift_param* p) {
	
    cpl_free((void *)p->extraction_method);
    cpl_free(p);
    p = NULL;
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Get the WAVE recipe parameters
 @param     recipe_id   recipe ID
 @param     param_list  parameters list
 @param     WAVE_param  WAVE parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_TH_drift_get(const char* recipe_id,
                                                  cpl_parameterlist* param_list,
                                                  espdr_WAVE_TH_drift_param *WAVE_TH_drift_param) {
	
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(param_list == NULL, CPL_ERROR_NULL_INPUT,
				 "Parameters list is NULL");
	
	/* Fill the structure */
    WAVE_TH_drift_param->extraction_method = espdr_parameters_get_string(recipe_id,
                                                                         param_list,
                                                                         "thar_extraction_method");
    
    WAVE_TH_drift_param->ksig_extraction = espdr_parameters_get_double(recipe_id,
                                                                       param_list,
                                                                       "thar_extraction_ksigma");
    
	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     print the WAVE parameters
 @param     WAVE_param  WAVE parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_TH_drift_print(espdr_WAVE_TH_drift_param *WAVE_TH_drift_param) {
	
	espdr_msg("\tWAVE TH drift parameters:");
    espdr_msg("\t\tWAVE_TH_drift extraction method = %s",
              WAVE_TH_drift_param->extraction_method);
    espdr_msg("\t\tWAVE_TH_drift extraction ksigma = %.2f",
              WAVE_TH_drift_param->ksig_extraction);

	return (cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Init WAVE_param structure
 
 @param    recipe_id    recipe name
 @param    parameters   recipe input parameters
 
 @return   allocated structure iff OK
 */
/*----------------------------------------------------------------------------*/

espdr_WAVE_TH_drift_param *espdr_WAVE_TH_drift_param_init(const char *recipe_id,
                                                          cpl_parameterlist *parameters) {
    
    cpl_error_code my_error;
    
    espdr_WAVE_TH_drift_param *WAVE_TH_drift_param =
                (espdr_WAVE_TH_drift_param *)cpl_malloc(sizeof(espdr_WAVE_TH_drift_param));
    
    /* Read the wave parameters */
    my_error = espdr_parameters_WAVE_TH_drift_get(recipe_id, parameters, WAVE_TH_drift_param);
    my_error = espdr_parameters_WAVE_TH_drift_print(WAVE_TH_drift_param);
    if(my_error != CPL_ERROR_NONE) {
        return NULL;
    } else {
        return WAVE_TH_drift_param;
    }
}



/*----------------------------------------------------------------------------*/
/**
 @brief Read WAVE_LINES_TABLE_ASCII from RDB table
 @param         wave_line_table frame containing lines table in ASCII format
 @param[out]    lines_table     read lines table
 @param[out]    lines_nb_RE     numebr of read lines
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_read_REF_DRIFT_LINES_TABLE_ASCII(cpl_frame *ref_line_table,
                                                      espdr_line_param *lines_table,
                                                      int *lines_nb_RE,
                                                      char source) {

    const char *filename = NULL;
    FILE *waveptr = NULL;
    char line[COMMENT_LENGTH];
    int lines_nb = 0;
    espdr_line_param read_line;
    double read_residuals, read_resolution, read_d, read_sig_d, read_sig_d_nolim;
    
    espdr_ensure(ref_line_table == NULL, CPL_ERROR_NULL_INPUT,
                 "Input REF LINES TABLE frame is NULL");
    espdr_ensure(lines_table == NULL, CPL_ERROR_NULL_INPUT,
                 "REF LINES TABLE structure is NULL");
    
    filename = cpl_frame_get_filename(ref_line_table);
    
    espdr_msg("Reading ASCII REF lines from %s", filename);
    
    waveptr = fopen(filename, "r");
    espdr_ensure(waveptr == NULL, CPL_ERROR_FILE_IO,
                 "The file %s can't be read", filename);
    
    // Get rid of the header
    fgets(line, sizeof(line), waveptr);
    fgets(line, sizeof(line), waveptr);
    // Read the file content
    /*
    while (fscanf(waveptr,
                  "%d\t%lf\t%lf\t%lf\t%lf\t%d",
                  &lines_table[lines_nb].order,
                  &lines_table[lines_nb].peak_pxl,
                  &lines_table[lines_nb].wavelength,
                  &ll_errors[lines_nb],
                  &lines_table[lines_nb].peak_flux,
                  &lines_table[lines_nb].qc_flag) != EOF) {
    */
    while (fscanf(waveptr,
                  "%d\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%d\t%lf\t%lf\t%lf\t%d\t%s\t%lf\t%lf\t%lf\t%lf\t%lf\n",
                  &read_line.order,
                  &read_line.peak_pxl,
                  &read_line.peak_pxl_err,
                  &read_line.fwhm,
                  &read_line.fwhm_err,
                  &read_line.peak_flux,
                  &read_line.peak_flux_err,
                  &read_line.qc_flag,
                  &read_line.wavelength,
                  &read_line.wavelength_err,
                  &read_line.dispersion,
                  &read_line.grouping,
                  &read_line.element_name,
                  &read_d,
                  &read_sig_d,
                  &read_sig_d_nolim,
                  &read_residuals,
                  &read_resolution) != EOF) {
        /*
        printf("%d\t%lf\t%lf\t%lf\t%lf\t%d\n",
               read_line.order,
               read_line.peak_pxl,
               read_line.wavelength,
               read_line.wavelength_err,
               read_line.peak_flux,
               read_line.qc_flag);
        */
        
        if ((read_line.qc_flag == 1) &&
            (read_line.element_name[0] == source)) {
            //((strstr(read_line.element_name, "TH") != NULL) ||
            // (strstr(read_line.element_name, "Th") != NULL) ||
            //((strstr(read_line.element_name, source) != NULL) ||
            // (strstr(read_line.element_name, "NONE") != NULL))) {
                
            lines_table[lines_nb].order = read_line.order;
            lines_table[lines_nb].peak_pxl = read_line.peak_pxl;
            lines_table[lines_nb].peak_pxl_err = read_line.peak_pxl_err;
            // reading MULTIMR84 with MULTIMR42
            //lines_table[lines_nb].peak_pxl = lines_table[lines_nb].peak_pxl/2;
            // reading MULTIMR42 with SINGLEHR lines tables
            //lines_table[lines_nb].peak_pxl = read_line.peak_pxl/2.0 + 33.0 - 5.0;
            // reading SINGLEUHR with SINGLEHR lines tables
            //lines_table[lines_nb].peak_pxl = read_line.peak_pxl + 33.0;
            lines_table[lines_nb].wavelength = read_line.wavelength;
            lines_table[lines_nb].wavelength_err = read_line.wavelength_err;
            lines_table[lines_nb].peak_flux = read_line.peak_flux;
            lines_table[lines_nb].peak_flux_err = read_line.peak_flux_err;
            lines_table[lines_nb].fwhm = read_line.fwhm;
            lines_table[lines_nb].fwhm_err = read_line.fwhm_err;
            lines_table[lines_nb].qc_flag = read_line.qc_flag;
            lines_table[lines_nb].dispersion = read_line.dispersion;
            lines_table[lines_nb].grouping = read_line.grouping;
            strcpy(lines_table[lines_nb].element_name, read_line.element_name);
            /*
            printf("%d\t%lf\t%lf\t%lf\t%lf\t%d\n",
                   lines_table[lines_nb].order,
                   lines_table[lines_nb].peak_pxl,
                   lines_table[lines_nb].wavelength,
                   lines_table[lines_nb].wavelength_err,
                   lines_table[lines_nb].peak_flux,
                   lines_table[lines_nb].qc_flag);
            */
            lines_nb++;
        }
    }

    fclose(waveptr);
    
    *lines_nb_RE = lines_nb;
    
    return(cpl_error_get_code());
}

/*----------------------------------------------------------------------------*/
/**
 @brief Read WAVE_LINES_TABLE_ASCII from RDB table
 @param         wave_line_table frame containing lines table in ASCII format
 @param[out]    lines_table     read lines table
 @param[out]    lines_nb_RE     numebr of read lines
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_read_REF_DRIFT_LINES_TABLE_CSV(cpl_frame *ref_line_table,
                                                    espdr_line_param *lines_table,
                                                    int *lines_nb_RE,
                                                    char source) {

    const char *filename = NULL;
    FILE *waveptr = NULL;
    char line[COMMENT_LENGTH];
    int lines_nb = 0;
    espdr_line_param read_line;
    double read_residuals, read_resolution, read_d, read_sig_d, read_sig_d_nolim;
    
    espdr_ensure(ref_line_table == NULL, CPL_ERROR_NULL_INPUT,
                 "Input REF LINES TABLE frame is NULL");
    espdr_ensure(lines_table == NULL, CPL_ERROR_NULL_INPUT,
                 "REF LINES TABLE structure is NULL");
    
    filename = cpl_frame_get_filename(ref_line_table);
    
    espdr_msg("Reading ASCII REF lines from %s", filename);
    
    waveptr = fopen(filename, "r");
    espdr_ensure(waveptr == NULL, CPL_ERROR_FILE_IO,
                 "The file %s can't be read", filename);
    
    // Get rid of the header
    fgets(line, sizeof(line), waveptr);
    //fgets(line, sizeof(line), waveptr);
    // Read the file content
    /*
    while (fscanf(waveptr,
                  "%d\t%lf\t%lf\t%lf\t%lf\t%d",
                  &lines_table[lines_nb].order,
                  &lines_table[lines_nb].peak_pxl,
                  &lines_table[lines_nb].wavelength,
                  &ll_errors[lines_nb],
                  &lines_table[lines_nb].peak_flux,
                  &lines_table[lines_nb].qc_flag) != EOF) {
    */
    
    // Adaptation CORALIE
    /*
    double peak_pxl_rms, fwhm_rms, peak_flux_rms;
    while (fscanf(waveptr,
                  "%d\t%lf\t%lf\t%lf\t%lf\t%lf\t%d\t%lf\t%lf\t%s\n",
                  &read_line.order,
                  &read_line.peak_pxl,
                  &read_line.peak_pxl_err,
                  &read_line.fwhm,
                  &read_line.fwhm_err,
                  &read_line.peak_flux,
                  &read_line.qc_flag,
                  &read_line.dispersion,
                  &read_line.wavelength,
                  &read_line.element_name) != EOF) {
    */
    
    int ID;
    while (fscanf(waveptr,
                  "%d,%lf,%lf,%lf,%lf,%lf,%lf,%d,%lf,%lf,%lf,%d,%s\n",
                  //&ID,
                  &read_line.order,
                  &read_line.peak_pxl,
                  &read_line.peak_pxl_err,
                  &read_line.fwhm,
                  &read_line.fwhm_err,
                  &read_line.peak_flux,
                  &read_line.peak_flux_err,
                  &read_line.qc_flag,
                  &read_line.wavelength,
                  &read_line.wavelength_err,
                  &read_line.dispersion,
                  &read_line.grouping,
                  &read_line.element_name) != EOF) {
     

        
        if (lines_nb < -1) {
            printf("%d\t%lf\t%lf\t%lf\t%lf\t%d\n",
                   read_line.order,
                   read_line.peak_pxl,
                   read_line.wavelength,
                   read_line.wavelength_err,
                   read_line.peak_flux,
                   read_line.qc_flag);
        }
        
        
        //if (read_line.qc_flag == 1) {
            //((strstr(read_line.element_name, "TH") != NULL) ||
            // (strstr(read_line.element_name, "Th") != NULL) ||
            //((strstr(read_line.element_name, source) != NULL) ||
            // (strstr(read_line.element_name, "NONE") != NULL))) {
        //if ((strstr(read_line.element_name, "TH") != NULL) ||
        //    (strstr(read_line.element_name, "Th") != NULL)) {
            //lines_table[lines_nb].order = read_line.order+1;
            lines_table[lines_nb].order = read_line.order;
            //lines_table[lines_nb].peak_pxl = read_line.peak_pxl-3;
            lines_table[lines_nb].peak_pxl = read_line.peak_pxl;
            lines_table[lines_nb].peak_pxl_err = read_line.peak_pxl_err;
            lines_table[lines_nb].wavelength = read_line.wavelength;
            lines_table[lines_nb].wavelength_err = read_line.wavelength_err;
            lines_table[lines_nb].peak_flux = read_line.peak_flux;
            lines_table[lines_nb].peak_flux_err = read_line.peak_flux_err;
            lines_table[lines_nb].fwhm = read_line.fwhm;
            lines_table[lines_nb].fwhm_err = read_line.fwhm_err;
            lines_table[lines_nb].qc_flag = read_line.qc_flag;
            lines_table[lines_nb].dispersion = read_line.dispersion;
            lines_table[lines_nb].grouping = read_line.grouping;
            strcpy(lines_table[lines_nb].element_name, read_line.element_name);
            /*
            printf("%d\t%lf\t%lf\t%lf\t%lf\t%d\n",
                   lines_table[lines_nb].order,
                   lines_table[lines_nb].peak_pxl,
                   lines_table[lines_nb].wavelength,
                   lines_table[lines_nb].wavelength_err,
                   lines_table[lines_nb].peak_flux,
                   lines_table[lines_nb].qc_flag);
            */
            lines_nb++;
        //}
    }

    fclose(waveptr);
    
    *lines_nb_RE = lines_nb;
    
    return(cpl_error_get_code());
}

/*----------------------------------------------------------------------------*/
/**
 @brief Read WAVE_LINES_TABLE from fits table
 @param         wave_line_table frame containing lines table in FITS format
 @param[out]    lines_table     read lines table
 @param[out]    lines_nb_RE     numebr of read lines
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_read_REF_DRIFT_LINES_TABLE(cpl_frame *ref_line_table,
                                                espdr_line_param *lines_table,
                                                int *lines_nb_RE,
                                                char source) {
    
    const char *filename = NULL;
    int lines_nb = 0;
    int i;
    cpl_table *line_t = NULL;
    
    espdr_ensure(ref_line_table == NULL, CPL_ERROR_NULL_INPUT,
                 "Input LINE TABLE frame is NULL");
    espdr_ensure(lines_table == NULL, CPL_ERROR_NULL_INPUT,
                 "Lines table structure is NULL");
    
    filename = cpl_frame_get_filename(ref_line_table);
    espdr_msg("Reading lines from %s", filename);
    
    line_t = cpl_table_load(filename,1,0);
    
    int table_length = cpl_table_get_nrow(line_t);
    int *col_order = cpl_table_get_data_int(line_t,
                                            COL_NAME_ORDER);
    double *col_x0 = cpl_table_get_data_double(line_t,
                                               COL_NAME_PEAK_PXL);
    double *col_x0_err = cpl_table_get_data_double(line_t,
                                                   COL_NAME_PEAK_PXL_ERR);
    double *col_fwhm = cpl_table_get_data_double(line_t,
                                                 COL_NAME_FWHM);
    double *col_fwhm_err = cpl_table_get_data_double(line_t,
                                                     COL_NAME_FWHM_ERR);
    double *col_k = cpl_table_get_data_double(line_t,
                                              COL_NAME_PEAK_FLUX);
    double *col_k_err = cpl_table_get_data_double(line_t,
                                                  COL_NAME_PEAK_FLUX_ERR);
    double *col_ll = cpl_table_get_data_double(line_t,
                                               COL_NAME_WAVELENGTH);
    double *col_ll_err = cpl_table_get_data_double(line_t,
                                                   COL_NAME_WAVELENGTH_ERR);
    double *col_disp = cpl_table_get_data_double(line_t,
                                               COL_NAME_DISPERSION);
    int *col_grouping = cpl_table_get_data_int(line_t,
                                               COL_NAME_GROUPING);
    int *col_qc = cpl_table_get_data_int(line_t,
                                         COL_NAME_QC_FLAG);
    char **col_elem = cpl_table_get_data_string(line_t, COL_NAME_ELEMENT_NAME);
    
    for(i = 0; i < table_length; i++){
        //printf("%d\t%lf\t%lf\t%lf\t%d\t%s\t%c\t%c\n",
        //       col_order[i], col_ll[i], col_x0[i], col_k[i],
        //       col_qc[i], col_elem[i], col_elem[i][0], source);
        if ((col_qc[i] == 1) &&
            (col_elem[i][0] == source)) {
            //((strstr(col_elem[i], "TH") != NULL) ||
            // (strstr(col_elem[i], "Th") != NULL) ||
            //((strstr(col_elem[i], source) != NULL) ||
            // (strstr(col_elem[i], "NONE") != NULL))) {
            lines_table[lines_nb].order = col_order[i];
            lines_table[lines_nb].peak_pxl = col_x0[i];
            lines_table[lines_nb].peak_pxl_err = col_x0_err[i];
            lines_table[lines_nb].fwhm = col_fwhm[i];
            lines_table[lines_nb].fwhm_err = col_fwhm_err[i];
            lines_table[lines_nb].peak_flux = col_k[i];
            lines_table[lines_nb].peak_flux_err = col_k_err[i];
            lines_table[lines_nb].wavelength = col_ll[i];
            lines_table[lines_nb].wavelength_err = col_ll_err[i];
            lines_table[lines_nb].dispersion = col_disp[i];
            lines_table[lines_nb].grouping = col_grouping[i];
            lines_table[lines_nb].qc_flag = col_qc[i];
            strcpy(lines_table[lines_nb].element_name, col_elem[i]);
            if (lines_table[lines_nb].order == -1) {
                printf("%d\t%lf\t%lf\t%lf\t%d\t%s\n",
                       lines_table[lines_nb].order,
                       lines_table[lines_nb].wavelength,
                       lines_table[lines_nb].peak_pxl,
                       lines_table[lines_nb].peak_flux,
                       lines_table[lines_nb].qc_flag,
                       lines_table[lines_nb].element_name);
            }
            lines_nb++;
        }
    }
    
    espdr_msg("Read %d lines", lines_nb);
    
    cpl_table_delete(line_t);
    
    *lines_nb_RE = lines_nb;
    
    return(cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief         Compute the difference in pixels between lines tables REF & TH
 @param         AR lines table
 @param         AR lines table length
 @param[out]    lines_table     read lines table
 @param[out]    lines_nb_RE     numebr of read lines
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_compute_diff_lines_table(espdr_line_param *REF_lines_table,
                                              int REF_lines_nb,
                                              espdr_line_param *TH_lines_table,
                                              int TH_lines_nb,
                                              int diff_lines_nb,
                                              int fibre,
                                              espdr_inst_config *inst_config,
                                              espdr_CCD_geometry *CCD_geom,
                                              espdr_line_param *diff_lines_table_RE,
                                              int *valid_lines_nb_RE) {
    
    int j;
    
    //for (j = 0; j < diff_lines_nb; j++){
    //    if (j%100 == 0){
    //        espdr_msg("index %d: ref_line_pxl = %.6f, compa_line_pxl = %.6f",j,REF_lines_table[j].peak_pxl,TH_lines_table[j].peak_pxl);
    //    }
    //}
    
    // REF - THAR = delta_pixel for all Th lines
    int invalid_diff_lines_nb = 0;
    int valid_lines_nb[CCD_geom->ext_nb];
    for (int ext = 0; ext < CCD_geom->ext_nb; ext++) {
        valid_lines_nb[ext] = 0;
    }
    
    int REF_valid_lines_nb = 0;
    for (j = 0; j < REF_lines_nb; j++) {
        if (REF_lines_table[j].qc_flag == 1) {
            REF_valid_lines_nb++;
        }
    }
    
    int TH_valid_lines_nb = 0;
    for (j = 0; j < TH_lines_nb; j++) {
        if (TH_lines_table[j].qc_flag == 1) {
            TH_valid_lines_nb++;
        }
    }
    
    espdr_msg("Number of valid lines: REF = %d, TH = %d",
              REF_valid_lines_nb, TH_valid_lines_nb);
    
    for (j = 0; j < diff_lines_nb; j++) {
        diff_lines_table_RE[j].order = REF_lines_table[j].order;
        diff_lines_table_RE[j].peak_pxl = TH_lines_table[j].peak_pxl - REF_lines_table[j].peak_pxl;
        // we decide here to take only the error on the REF file while normaly we should do sqrt(TH_lines_table[j].peak_pxl^2+REF_lines_table[j].peak_pxl^2)
        // However, at each warmup, the blaze change significantly, by up to 20% on the edges of the orders. This will weight differently the TH line used
        // and might create some instability as a function of time.
        diff_lines_table_RE[j].peak_pxl_err = sqrt((REF_lines_table[j].peak_pxl_err *
                                                 REF_lines_table[j].peak_pxl_err) +
                                                (REF_lines_table[j].peak_pxl_err *
                                                 REF_lines_table[j].peak_pxl_err));
        diff_lines_table_RE[j].fwhm = TH_lines_table[j].fwhm;
        diff_lines_table_RE[j].fwhm_err = TH_lines_table[j].fwhm_err;
        diff_lines_table_RE[j].peak_flux = TH_lines_table[j].peak_flux;
        diff_lines_table_RE[j].peak_flux_err = TH_lines_table[j].peak_flux_err;
        diff_lines_table_RE[j].qc_flag = REF_lines_table[j].qc_flag && TH_lines_table[j].qc_flag;
        diff_lines_table_RE[j].wavelength = REF_lines_table[j].wavelength;
        diff_lines_table_RE[j].wavelength_err = REF_lines_table[j].wavelength_err;
        diff_lines_table_RE[j].lambda_peak = REF_lines_table[j].lambda_peak;
        diff_lines_table_RE[j].Nreal = REF_lines_table[j].Nreal;
        diff_lines_table_RE[j].N = REF_lines_table[j].N;
        diff_lines_table_RE[j].dispersion = REF_lines_table[j].dispersion;
        diff_lines_table_RE[j].grouping = REF_lines_table[j].grouping;
        strcpy(diff_lines_table_RE[j].element_name, REF_lines_table[j].element_name);
        
        if (REF_lines_table[j].order != TH_lines_table[j].order) {
            espdr_msg_warning("Mismatched order: REF - %d\tTH - %d",
                      REF_lines_table[j].order, TH_lines_table[j].order);
            diff_lines_table_RE[j].qc_flag = 0;
        }
        if (fabs(REF_lines_table[j].wavelength - TH_lines_table[j].wavelength) > 0.0001) {
            espdr_msg_warning("Mismatched wavelength: REF - %f\tTH - %f",
                      REF_lines_table[j].wavelength, TH_lines_table[j].wavelength);
            diff_lines_table_RE[j].qc_flag = 0;
        }
        if (strcmp(REF_lines_table[j].element_name, TH_lines_table[j].element_name) != 0) {
            espdr_msg_warning("Mismatched elem name: REF - %s\tTH - %s",
                      REF_lines_table[j].element_name, TH_lines_table[j].element_name);
            diff_lines_table_RE[j].qc_flag = 0;
        }
        if (diff_lines_table_RE[j].qc_flag == 0) {
            invalid_diff_lines_nb++;
        }
        
        for (int ext = 0; ext < CCD_geom->ext_nb; ext++) {
            if (diff_lines_table_RE[j].order == -1) {
                espdr_msg("diff_line ord %d (%.4f-%.4f) = %.4f, QC: %d",
                          diff_lines_table_RE[j].order,
                          TH_lines_table[j].peak_pxl,
                          REF_lines_table[j].peak_pxl,
                          diff_lines_table_RE[j].peak_pxl,
                          diff_lines_table_RE[j].qc_flag);
            }
            if ((diff_lines_table_RE[j].qc_flag == 1) &&
                (diff_lines_table_RE[j].order >= espdr_get_first_order(fibre, ext, inst_config, CCD_geom)) &&
                (diff_lines_table_RE[j].order <= espdr_get_last_order(fibre, ext, inst_config, CCD_geom))) {
                valid_lines_nb[ext]++;
            }
        }
        
        if (diff_lines_table_RE[j].order == -1) {
            espdr_msg("diff_line %d, order %d - pxl(ref): %f, pxl(thar): %f  \t%f\t%f\t%f\t%f\t%f\t%f\t%d\t%f\t%s",
                      j, diff_lines_table_RE[j].order, REF_lines_table[j].peak_pxl,
                      TH_lines_table[j].peak_pxl,
                      diff_lines_table_RE[j].peak_pxl, diff_lines_table_RE[j].peak_pxl_err,
                      diff_lines_table_RE[j].fwhm, diff_lines_table_RE[j].fwhm_err,
                      diff_lines_table_RE[j].peak_flux, diff_lines_table_RE[j].peak_flux_err,
                      diff_lines_table_RE[j].qc_flag, diff_lines_table_RE[j].wavelength,
                      diff_lines_table_RE[j].element_name);
        }
        if (diff_lines_table_RE[j].peak_pxl_err == 0.0) {
            espdr_msg("diff_line %d: order; %d - pxl(ref): %f, pxl_err(ref): %f, pxl(thar): %f, pxl_err(thar): %f",
                      j, diff_lines_table_RE[j].order, REF_lines_table[j].peak_pxl, REF_lines_table[j].peak_pxl_err,
                      TH_lines_table[j].peak_pxl, TH_lines_table[j].peak_pxl_err);
        }
    }
    espdr_msg("There are %d over %d invalid lines in the diff table",
              invalid_diff_lines_nb, diff_lines_nb);
    
    for (int ext = 0; ext < CCD_geom->ext_nb; ext++) {
        espdr_msg("Valid lines on DET%d: %d", ext, valid_lines_nb[ext]);
        valid_lines_nb_RE[ext] = valid_lines_nb[ext];
    }
    
    return(cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief         Fit the drift matrix
 @param         AR lines table
 @param         AR lines table length
 @param[out]    lines_table     read lines table
 @param[out]    lines_nb_RE     numebr of read lines
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_fit_drift_matrix(espdr_line_param *STATIC_lines_table,
                                      espdr_line_param *TH_lines_table,
                                      espdr_line_param *diff_lines_table,
                                      int diff_lines_nb,
                                      int *valid_lines_nb,
                                      cpl_image *ref_wave_matrix_img,
                                      cpl_image *ref_dll_matrix_img,
                                      espdr_inst_config *inst_config,
                                      espdr_CCD_geometry *CCD_geom,
                                      int size_x,
                                      int size_y,
                                      int orders_nb_per_fibre,
                                      int fibre,
                                      int *rejected_lines_nb_RE,
                                      double *MAD_RE,
                                      double *MAD2_RE,
                                      double *RMS_RE,
                                      double *RMS2_RE,
                                      double **ll_coeffs_RE,
                                      double *CHI2_RE,
                                      double **MAD9_RE,
                                      double **RMS9_RE,
                                      int **rejected_lines_nb9_RE,
                                      double **MAD99_RE,
                                      double **RMS99_RE,
                                      int **rejected_lines_nb99_RE,
                                      cpl_image *drift_matrix_RE,
                                      cpl_image **wave_matrix_RE,
                                      cpl_image **dll_matrix_RE,
                                      cpl_image **air_wave_matrix_RE,
                                      cpl_image **air_dll_matrix_RE) {
    
    int j;
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    espdr_msg("Starting espdr_fit_drift_matrix");
    
    // Do 2D_poly_fit to fill up all the S2D
    espdr_poly_2D_data *p_data = (espdr_poly_2D_data *)cpl_malloc(sizeof(espdr_poly_2D_data));
    espdr_poly_2D_data *p2_data = (espdr_poly_2D_data *)cpl_malloc(sizeof(espdr_poly_2D_data));
    double chisq;
    double *fit = NULL;
    double **coeffs = NULL;
    double **coeffs_err = NULL;
    double *fit2 = NULL;
    double **coeffs2 = NULL;
    double **coeffs2_err = NULL;
    double *residuals = NULL;
    double *residuals2 = NULL;
    double *res_for_MAD = NULL;
    double *res_for_MAD2 = NULL;
    double *ll_lines = NULL;
    double res_MAD = 0.0, res_median = 0.0;
    cpl_vector *res_vector = NULL;
    cpl_vector *res_vector2 = NULL;
    int fit_size = 0, index_fit = 0;
    int *line_index_for_p = NULL;
    int *line_index_for_p2 = NULL;

    
    coeffs = (double **)cpl_malloc(CCD_geom->ext_nb * sizeof(double *));
    coeffs_err = (double **)cpl_malloc(CCD_geom->ext_nb * sizeof(double *));
    coeffs2 = (double **)cpl_malloc(CCD_geom->ext_nb * sizeof(double *));
    coeffs2_err = (double **)cpl_malloc(CCD_geom->ext_nb * sizeof(double *));
    for (int ext = 0; ext < CCD_geom->ext_nb; ext++) {
        fit_size = valid_lines_nb[ext];
        line_index_for_p = (int *)calloc(fit_size, sizeof(int));
        p_data->x = (double *) cpl_calloc (fit_size, sizeof (double));
        p_data->y = (double *) cpl_calloc (fit_size, sizeof (double));
        p_data->z = (double *) cpl_calloc (fit_size, sizeof (double));
        p_data->err = (double *) cpl_calloc (fit_size, sizeof (double));
        p_data->n = fit_size;
        p_data->m = (inst_config->th_drift_fit_deg_x+1) * (inst_config->th_drift_fit_deg_y+1);
        fit = (double *) cpl_calloc (fit_size, sizeof(double));
        coeffs[ext] = (double *) cpl_calloc (p_data->m, sizeof(double));
        coeffs_err[ext] = (double *) cpl_calloc (p_data->m, sizeof(double));
        ll_lines = (double *) cpl_calloc (fit_size, sizeof (double));
        
        index_fit = 0;
        for (j = 0; j < diff_lines_nb; j++) {
            if ((diff_lines_table[j].qc_flag == 1) &&
                (diff_lines_table[j].order >= espdr_get_first_order(fibre, ext, inst_config, CCD_geom)) &&
                (diff_lines_table[j].order <= espdr_get_last_order(fibre, ext, inst_config, CCD_geom))) {
                p_data->x[index_fit] = STATIC_lines_table[j].peak_pxl;
                p_data->y[index_fit] = (double)(diff_lines_table[j].order);
                //p_data->z[index_fit] = diff_lines_table[j].peak_pxl;
                //p_data->err[index_fit] = diff_lines_table[j].peak_pxl_err;
                // We go into the velocity space, because shifts are in velocity
                // and if we stay in the pixel domain, we see at first the dispersion
                // which is non-linear and therefore cannot be corrected properly
                // with a linear combination (polynomial in our case)
                // Note that diff_lines_table.dispersion and
                // diff_lines_table.wavelength are from the REF file (see creation of
                // diff_lines_table above)
                p_data->z[index_fit] = LIGHT_SPEED * diff_lines_table[j].peak_pxl *
                        diff_lines_table[j].dispersion / diff_lines_table[j].wavelength;
                p_data->err[index_fit] = LIGHT_SPEED * diff_lines_table[j].peak_pxl_err *
                        diff_lines_table[j].dispersion / diff_lines_table[j].wavelength;
                ll_lines[index_fit] = diff_lines_table[j].wavelength;
                if (j == -1) {
                    espdr_msg("Fit input[%d]: x: %.8f, y: %.8f, x_diff: %.8f rv_diff: %.8f, err: %.8f",
                              index_fit, p_data->x[index_fit],
                              p_data->y[index_fit],
                              diff_lines_table[j].peak_pxl,
                              p_data->z[index_fit],
                              p_data->err[index_fit]);
                }
                if (p_data->err[index_fit] == 0.0) {
                    espdr_msg("j: %d, err: %f, diff_err: %f, diff_disp: %f, dif ll: %f",
                              j, p_data->err[index_fit], diff_lines_table[j].peak_pxl_err,
                              diff_lines_table[j].dispersion, diff_lines_table[j].wavelength);
                }
                line_index_for_p[index_fit] = j;
                index_fit++;
            }
        }
        
        espdr_msg("Polynomial degree [ext = %d] dispersion = %d, cross-dispersion = %d", ext, inst_config->th_drift_fit_deg_x, inst_config->th_drift_fit_deg_y);
        espdr_msg("First fit with %d lines (ext %d)", fit_size, ext);
        
        my_error = espdr_fit_poly_2D(p_data, fit, inst_config,
                                     coeffs[ext], coeffs_err[ext], &chisq, 0);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_fit_poly failed: %s",
                     cpl_error_get_message_default(my_error));

        espdr_msg("CHI2 for first drift matrix fit = %f", chisq);
        //int index_poly = 0;
        //for (int pow_x = 0; pow_x <= inst_config->th_drift_fit_deg_x; pow_x++){
        //    for (int pow_y = 0; pow_y <= inst_config->th_drift_fit_deg_y; pow_y++){
        //        espdr_msg("COEFFS %d: %e", index_poly, coeffs[ext][index_poly]);
        //        index_poly++;
        //    }
        //} 
        
        residuals = (double *) cpl_calloc (fit_size, sizeof (double));
        res_for_MAD = (double *) cpl_calloc (fit_size, sizeof (double));
        for (j = 0; j < fit_size; j++) {
            residuals[j] = p_data->z[j] - fit[j];
            res_for_MAD[j] = p_data->z[j] - fit[j];
        }
        
        res_vector = cpl_vector_wrap(fit_size, res_for_MAD);
        res_median = cpl_vector_get_median(res_vector);
        cpl_vector_unwrap(res_vector);
        espdr_msg("Median[ext = %d] = %.10f", ext, res_median);
        
        for (j = 0; j < fit_size; j++) {
            res_for_MAD[j] = res_for_MAD[j] - res_median;
            if (res_for_MAD[j] < 0.0) {
                res_for_MAD[j] = - res_for_MAD[j];
            }
        }
        
        espdr_msg("Removing lines with general MAD");
        res_vector = cpl_vector_wrap(fit_size, res_for_MAD);
        res_MAD = cpl_vector_get_median(res_vector)/0.6744897501960817;
        RMS_RE[ext] = cpl_vector_get_stdev(res_vector);
        cpl_vector_unwrap(res_vector);
        espdr_msg("MAD[ext = %d] = %.10f", ext, res_MAD);
        MAD_RE[ext] = res_MAD;
        
        int kept_lines_nb = 0;
        for (j = 0; j < fit_size; j++) {
            if (fabs(residuals[j] - res_median) > inst_config->th_drift_fit_ksigma * res_MAD) {
                espdr_msg("---> removing line at pxl %f, order %.0f, ll: %f, residual too big: %f",
                          p_data->x[j], p_data->y[j], ll_lines[j], fabs(residuals[j] - res_median));
            } else {
                if (p_data->y[j] == -1) {
                    espdr_msg("keeping line at pxl %f, order %.0f, ll: %f, residual OK: %f",
                              p_data->x[j], p_data->y[j], ll_lines[j], fabs(residuals[j] - res_median));
                }
                kept_lines_nb++;
            }
        }
        
        espdr_msg("Nb of lines rejected on the whole CCD: %d", fit_size-kept_lines_nb);
        
        // Adding the computation of MAD & RMS per CCD region, CCD is divided into 3x3=9 regions
        kept_lines_nb = 0;
        int rejected_lines_total = 0;
        int lines_nb_box[3][3];
        int left[3], right[3], top[3], bottom[3];
        double **res_for_MAD9 = (double **)cpl_calloc(9, sizeof(double *));
        double res_median9[9];
        int index9 = 0;
        espdr_msg("Removing lines within parts of the CCD with corresponding MADs");
        for (int ccd_x = 0; ccd_x < 3; ccd_x++) {
            left[ccd_x] = ccd_x * size_x/3;
            right[ccd_x] = (ccd_x+1) * size_x/3;
            for (int ccd_y = 0; ccd_y < 3; ccd_y++) {
                bottom[ccd_y] = ccd_y * size_y/3;
                top[ccd_y] = (ccd_y+1) * size_y/3;
                espdr_msg("left[%d] = %d, right[%d] = %d, bottom[%d] = %d, top[%d] = %d",
                          ccd_x, left[ccd_x], ccd_x, right[ccd_x], ccd_y, bottom[ccd_y], ccd_y, top[ccd_y]);
                lines_nb_box[ccd_x][ccd_y] = 0;
                for (j = 0; j < fit_size; j++) {
                    //espdr_msg("Checking line: x = %f, y = %f", p_data->x[j], p_data->y[j]);
                    if ((p_data->x[j] > left[ccd_x]) && (p_data->x[j] <= right[ccd_x]) &&
                        (p_data->y[j] > bottom[ccd_y]) && (p_data->y[j] <= top[ccd_y])) {
                        lines_nb_box[ccd_x][ccd_y]++;
                        //espdr_msg("Line added");
                    } else {
                        //espdr_msg("Line discarded");
                    }
                }
                espdr_msg("lines_nb in [%d,%d] = %d", ccd_x, ccd_y, lines_nb_box[ccd_x][ccd_y]);
                res_for_MAD9[index9] = (double *)cpl_calloc(lines_nb_box[ccd_x][ccd_y], sizeof(double));
                int index_line = 0;
                for (j = 0; j < fit_size; j++) {
                    if ((p_data->x[j] > left[ccd_x]) && (p_data->x[j] <= right[ccd_x]) &&
                        (p_data->y[j] > bottom[ccd_y]) && (p_data->y[j] <= top[ccd_y])) {
                        res_for_MAD9[index9][index_line] = p_data->z[j] - fit[j];
                        //espdr_msg("res_for_MAD9[%d][%d] = %f", index9, index_line, res_for_MAD9[index9][index_line]);
                        index_line++;
                    }
                }
                res_vector = cpl_vector_wrap(lines_nb_box[ccd_x][ccd_y], res_for_MAD9[index9]);
                RMS9_RE[ext][index9] = cpl_vector_get_stdev(res_vector);
                res_median9[index9] = cpl_vector_get_median(res_vector);
                cpl_vector_unwrap(res_vector);
                for (j = 0; j < lines_nb_box[ccd_x][ccd_y]; j++) {
                    res_for_MAD9[index9][j] = res_for_MAD9[index9][j] - res_median9[index9];
                    if (res_for_MAD9[index9][j] < 0.0) {
                        res_for_MAD9[index9][j] = - res_for_MAD9[index9][j];
                    }
                }
                res_vector = cpl_vector_wrap(lines_nb_box[ccd_x][ccd_y], res_for_MAD9[index9]);
                MAD9_RE[ext][index9] = cpl_vector_get_median(res_vector)/0.6744897501960817;
                espdr_msg("median9[%d][%d] = %f\tMAD9[%d][%d] = %.10f\tRMS9[%d][%d] = %.10f",
                          ext, index9, res_median9[index9], ext, index9, MAD9_RE[ext][index9], ext, index9, RMS9_RE[ext][index9]);
                cpl_vector_unwrap(res_vector);
                
                for (j = 0; j < fit_size; j++) {
                    if ((p_data->x[j] > left[ccd_x]) && (p_data->x[j] <= right[ccd_x]) &&
                        (p_data->y[j] > bottom[ccd_y]) && (p_data->y[j] <= top[ccd_y])) {
                        if (fabs(residuals[j] - res_median9[index9]) > inst_config->th_drift_fit_ksigma * MAD9_RE[ext][index9]) {
                            espdr_msg("---> removing line %d at pxl %f, order %.0f, ll: %f, residual too big: %f [pxl], %f [km/s]",
                                      j, p_data->x[j], p_data->y[j], ll_lines[j], fabs(residuals[j] - res_median9[index9]),
                                      fabs(residuals[j] - res_median9[index9])*inst_config->mask_width);
                            rejected_lines_nb9_RE[ext][index9]++;
                            rejected_lines_total++;
                            TH_lines_table[line_index_for_p[j]].qc_flag = 0;
                        } else {
                            if (p_data->y[j] == -1) {
                                espdr_msg("keeping line at pxl %f, order %.0f, ll: %f, residual OK: %f",
                                          p_data->x[j], p_data->y[j], ll_lines[j], fabs(residuals[j] - res_median9[index9]));
                            }
                            kept_lines_nb++;
                        }
                    }
                }
                espdr_msg("rejected lines [%d] = %d", index9, rejected_lines_nb9_RE[ext][index9]);
                index9++;
            }
        }
        
        if (kept_lines_nb < fit_size/2.0) {
            espdr_msg_error("Too many lines eliminated: %d, exiting",
                            fit_size - kept_lines_nb);
            return(CPL_ERROR_INCOMPATIBLE_INPUT);
        }
        
        espdr_msg("Second fit with %d lines (ext %d)", kept_lines_nb, ext);
        int fit_size2 = kept_lines_nb;
        line_index_for_p2 = (int *)calloc(fit_size2, sizeof(int));

        p2_data->x = (double *) cpl_calloc (fit_size2, sizeof (double));
        p2_data->y = (double *) cpl_calloc (fit_size2, sizeof (double));
        p2_data->z = (double *) cpl_calloc (fit_size2, sizeof (double));
        p2_data->err = (double *) cpl_calloc (fit_size2, sizeof (double));
        p2_data->n = fit_size2;
        p2_data->m = (inst_config->th_drift_fit_deg_x+1) * (inst_config->th_drift_fit_deg_y+1);
        fit2 = (double *) cpl_calloc (fit_size2, sizeof(double));
        coeffs2[ext] = (double *) cpl_calloc (p2_data->m, sizeof(double));
        coeffs2_err[ext] = (double *) cpl_calloc (p2_data->m, sizeof(double));
        
        index_fit = 0;
        index9 = 0;
        for (int ccd_x = 0; ccd_x < 3; ccd_x++) {
            left[ccd_x] = ccd_x * size_x/3;
            right[ccd_x] = (ccd_x+1) * size_x/3;
            for (int ccd_y = 0; ccd_y < 3; ccd_y++) {
                bottom[ccd_y] = ccd_y * size_y/3;
                top[ccd_y] = (ccd_y+1) * size_y/3;
                for (j = 0; j < fit_size; j++) {
                    if ((p_data->x[j] > left[ccd_x]) && (p_data->x[j] <= right[ccd_x]) &&
                        (p_data->y[j] > bottom[ccd_y]) && (p_data->y[j] <= top[ccd_y])) {
                        if (fabs(residuals[j] - res_median9[index9]) <= inst_config->th_drift_fit_ksigma * MAD9_RE[ext][index9]) {
                            p2_data->x[index_fit] = p_data->x[j];
                            p2_data->y[index_fit] = p_data->y[j];
                            p2_data->z[index_fit] = p_data->z[j];
                            p2_data->err[index_fit] = p_data->err[j];
                            //espdr_msg("x: %f\ty: %f\tz: %f\terr: %f",
                            //          p2_data->x[index_fit], p2_data->y[index_fit],
                            //          p2_data->z[index_fit], p2_data->err[index_fit]);
                            line_index_for_p2[index_fit] = line_index_for_p[j];
                            index_fit++;
                        } else {
                            espdr_msg("---> removing line %d at pxl %f, order %.0f, ll: %f, resid: %f, median: %f, MAD: %f, residual too big: %f [pxl], %f [km/s]",
                                      j, p_data->x[j], p_data->y[j], ll_lines[j], residuals[j], res_median9[index9], MAD9_RE[ext][index9],
                                      fabs(residuals[j] - res_median), fabs(residuals[j] - res_median9[index9])*inst_config->mask_width);
                        }
                    }
                }
                index9++;
            }
        }
        
        my_error = espdr_fit_poly_2D(p2_data, fit2, inst_config,
                                     coeffs2[ext], coeffs2_err[ext], &chisq, 0);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_fit_poly failed: %s",
                     cpl_error_get_message_default(my_error));
        
        espdr_msg("CHI2 for second drift matrix fit = %f", chisq);
        int index_poly = 0;
        for (int pow_x = 0; pow_x <= inst_config->th_drift_fit_deg_x; pow_x++){
            for (int pow_y = 0; pow_y <= inst_config->th_drift_fit_deg_y; pow_y++){
                espdr_msg("COEFFS %d: %e", index_poly, coeffs2[ext][index_poly]);
                index_poly++;
            }
        }
        
        residuals2 = (double *) cpl_calloc (fit_size2, sizeof (double));
        res_for_MAD2 = (double *) cpl_calloc (fit_size2, sizeof (double));
        for (j = 0; j < fit_size2; j++) {
            residuals2[j] = p2_data->z[j] - fit2[j];
            res_for_MAD2[j] = p2_data->z[j] - fit2[j];
        }
        
        res_vector2 = cpl_vector_wrap(fit_size2, res_for_MAD2);
        res_median = cpl_vector_get_median(res_vector2);
        RMS2_RE[ext] = cpl_vector_get_stdev(res_vector2);
        cpl_vector_unwrap(res_vector2);
        espdr_msg("Median[ext = %d] = %.10f", ext, res_median);
        
        for (j = 0; j < fit_size2; j++) {
            res_for_MAD2[j] = res_for_MAD2[j] - res_median;
            if (res_for_MAD2[j] < 0.0) {
                res_for_MAD2[j] = - res_for_MAD2[j];
            }
        }
        
        espdr_msg("Removing lines with general MAD");
        res_vector2 = cpl_vector_wrap(fit_size2, res_for_MAD2);
        res_MAD = cpl_vector_get_median(res_vector2)/0.6744897501960817;
        cpl_vector_unwrap(res_vector2);
        espdr_msg("MAD[ext = %d] = %.10f", ext, res_MAD);
        MAD2_RE[ext] = res_MAD;
        
        // Adding the computation of MAD & RMS per CCD region after 2nd fit, CCD is divided into 3x3=9 regions
        int kept_lines_nb2 = 0;
        int rejected_lines_total2 = 0;
        //int lines_nb_box[3][3];
        //int left[3], right[3], top[3], bottom[3];
        double **res_for_MAD99 = (double **)cpl_calloc(9, sizeof(double *));
        double res_median99[9];
        index9 = 0;
        for (int ccd_x = 0; ccd_x < 3; ccd_x++) {
            //left[ccd_x] = ccd_x * size_x/3;
            //right[ccd_x] = (ccd_x+1) * size_x/3;
            for (int ccd_y = 0; ccd_y < 3; ccd_y++) {
                //bottom[ccd_y] = ccd_y * size_y/3;
                //top[ccd_y] = (ccd_y+1) * size_y/3;
                espdr_msg("left[%d] = %d, right[%d] = %d, bottom[%d] = %d, top[%d] = %d",
                          ccd_x, left[ccd_x], ccd_x, right[ccd_x], ccd_y, bottom[ccd_y], ccd_y, top[ccd_y]);
                lines_nb_box[ccd_x][ccd_y] = 0;
                for (j = 0; j < fit_size2; j++) {
                    //espdr_msg("Checking line: x = %f, y = %f", p_data->x[j], p_data->y[j]);
                    if ((p2_data->x[j] > left[ccd_x]) && (p2_data->x[j] <= right[ccd_x]) &&
                        (p2_data->y[j] > bottom[ccd_y]) && (p2_data->y[j] <= top[ccd_y])) {
                        lines_nb_box[ccd_x][ccd_y]++;
                        //espdr_msg("Line added");
                    } else {
                        //espdr_msg("Line discarded");
                    }
                }
                espdr_msg("lines_nb in [%d,%d] = %d", ccd_x, ccd_y, lines_nb_box[ccd_x][ccd_y]);
                res_for_MAD99[index9] = (double *)cpl_calloc(lines_nb_box[ccd_x][ccd_y], sizeof(double));
                int index_line = 0;
                for (j = 0; j < fit_size2; j++) {
                    if ((p2_data->x[j] > left[ccd_x]) && (p2_data->x[j] <= right[ccd_x]) &&
                        (p2_data->y[j] > bottom[ccd_y]) && (p2_data->y[j] <= top[ccd_y])) {
                        res_for_MAD99[index9][index_line] = p2_data->z[j] - fit2[j];
                        //espdr_msg("2nd fit: res_for_MAD99[%d][%d] = %f", index9, index_line, res_for_MAD99[index9][index_line]);
                        index_line++;
                    }
                }
                res_vector = cpl_vector_wrap(lines_nb_box[ccd_x][ccd_y], res_for_MAD99[index9]);
                RMS99_RE[ext][index9] = cpl_vector_get_stdev(res_vector);
                res_median99[index9] = cpl_vector_get_median(res_vector);
                cpl_vector_unwrap(res_vector);
                for (j = 0; j < lines_nb_box[ccd_x][ccd_y]; j++) {
                    res_for_MAD99[index9][j] = res_for_MAD99[index9][j] - res_median99[index9];
                    if (res_for_MAD99[index9][j] < 0.0) {
                        res_for_MAD99[index9][j] = - res_for_MAD99[index9][j];
                    }
                }
                res_vector = cpl_vector_wrap(lines_nb_box[ccd_x][ccd_y], res_for_MAD99[index9]);
                MAD99_RE[ext][index9] = cpl_vector_get_median(res_vector)/0.6744897501960817;
                espdr_msg("2nd fit: median99[%d][%d] = %f\tMAD99[%d][%d] = %.10f\tRMS99[%d][%d] = %.10f",
                          ext, index9, res_median99[index9], ext, index9, MAD99_RE[ext][index9], ext, index9, RMS99_RE[ext][index9]);
                cpl_vector_unwrap(res_vector);
                
                for (j = 0; j < fit_size2; j++) {
                    if ((p2_data->x[j] > left[ccd_x]) && (p2_data->x[j] <= right[ccd_x]) &&
                        (p2_data->y[j] > bottom[ccd_y]) && (p2_data->y[j] <= top[ccd_y])) {
                        if (fabs(residuals2[j] - res_median99[index9]) > inst_config->th_drift_fit_ksigma * MAD99_RE[ext][index9]) {
                            espdr_msg("---> removing line at pxl %f, order %.0f, ll: %f, residual too big: %f [pxl], %f [km/s]",
                                      p_data->x[j], p_data->y[j], ll_lines[j], fabs(residuals2[j] - res_median99[index9]),
                                      fabs(residuals2[j] - res_median99[index9])*inst_config->mask_width);
                            rejected_lines_nb99_RE[ext][index9]++;
                            rejected_lines_total2++;
                            TH_lines_table[line_index_for_p2[j]].qc_flag = 0;
                        } else {
                            if (p2_data->y[j] == -1) {
                                espdr_msg("keeping line at pxl %f, order %.0f, ll: %f, residual OK: %f",
                                          p_data->x[j], p_data->y[j], ll_lines[j], fabs(residuals2[j] - res_median99[index9]));
                            }
                            kept_lines_nb2++;
                        }
                    }
                }
                espdr_msg("rejected lines [%d] = %d", index9, rejected_lines_nb99_RE[ext][index9]);
                index9++;
            }
        }
        
        
        for (j = 0; j < p2_data->m; j++) {
            ll_coeffs_RE[ext][j] = coeffs2[ext][j];
        }
        CHI2_RE[ext] = chisq;
        
        rejected_lines_nb_RE[ext] = valid_lines_nb[ext] - kept_lines_nb;
        
        espdr_msg("Number of lines rejected after the 1st fit on DET[%d]: %d / %d",
                  ext, rejected_lines_nb_RE[ext], valid_lines_nb[ext]);
        espdr_msg("Number of lines that should be rejected after the 2nd fit on DET[%d]: %d / %d",
                  ext, kept_lines_nb - kept_lines_nb2, kept_lines_nb);

        cpl_free(p_data->x);
        cpl_free(p_data->y);
        cpl_free(p_data->z);
        cpl_free(p_data->err);
        cpl_free(fit);
        cpl_free(ll_lines);
        
        cpl_free(p2_data->x);
        cpl_free(p2_data->y);
        cpl_free(p2_data->z);
        cpl_free(p2_data->err);
        cpl_free(fit2);
        cpl_free(residuals);
        cpl_free(res_for_MAD);
        
        cpl_free(line_index_for_p);
        cpl_free(line_index_for_p2);
    }
    
    cpl_free(p_data);
    cpl_free(p2_data);

    double *drift_matrix = (double *)cpl_calloc(size_x*orders_nb_per_fibre, sizeof(double));
    int index_drift = 0;
    // Construct the drift_matrix
    //espdr_msg("Size: %d x %d", size_x, orders_nb_per_fibre);
    for (int order = 1; order <= orders_nb_per_fibre; order++) {
        for (int pxl = 1; pxl <= size_x; pxl++) {
            for (int ext = 0; ext < CCD_geom->ext_nb; ext++) {
                if ((order >= espdr_get_first_order(fibre, ext, inst_config, CCD_geom)) &&
                    (order <= espdr_get_last_order(fibre, ext, inst_config, CCD_geom))) {
                    
                    drift_matrix[index_drift] = 0;
                    int index_poly=0;
                    for (int pow_x = 0; pow_x <= inst_config->th_drift_fit_deg_x; pow_x++){
                        for (int pow_y = 0; pow_y <= inst_config->th_drift_fit_deg_y; pow_y++){
                            drift_matrix[index_drift] += coeffs2[ext][index_poly]*gsl_pow_int(pxl,pow_x)*gsl_pow_int(order,pow_y);
                            index_poly++;
                        }
                    }
                    
                }
            }
            index_drift++;
        }
    }
    
    for (int ext = 0; ext < CCD_geom->ext_nb; ext++) {
        cpl_free(coeffs[ext]);
        cpl_free(coeffs_err[ext]);
        cpl_free(coeffs2[ext]);
        cpl_free(coeffs2_err[ext]);
    }
    cpl_free(coeffs);
    cpl_free(coeffs_err);
    cpl_free(coeffs2);
    cpl_free(coeffs2_err);

    cpl_image *drift_matrix_img = cpl_image_wrap_double(size_x, orders_nb_per_fibre,
                                                        drift_matrix);
    
    //Going back in pixel shift space as we did the fit in velocity space
    //The new drift_matric is obtained by doing
    //drift_matrix * ref_wave_matrix_img / ref_dll_matrix_img / C
    my_error = cpl_image_multiply(drift_matrix_img, ref_wave_matrix_img);
    my_error = cpl_image_divide(drift_matrix_img, ref_dll_matrix_img);
    my_error = cpl_image_divide_scalar(drift_matrix_img, LIGHT_SPEED);
    
#if SAVE_DEBUG_PRODUCT_WAVE_TH_DRIFT
    
    char drift_filename[64];
    sprintf(drift_filename, "%s_DRIFT_MATRIX_%c.fits",
            inst_config->instrument, fibre_name[fibre]);
    my_error = cpl_image_save(drift_matrix_img, drift_filename, CPL_TYPE_DOUBLE,
                              NULL, CPL_IO_CREATE);
#endif
    drift_matrix_RE = cpl_image_duplicate(drift_matrix_img);
    
    cpl_image *drift_matrix_ll = cpl_image_multiply_create(drift_matrix_img,
                                                           ref_dll_matrix_img);
    cpl_image *new_wave_matrix = cpl_image_subtract_create(ref_wave_matrix_img,
                                                           drift_matrix_ll);
    
    cpl_image_unwrap(drift_matrix_img);
    cpl_free(drift_matrix);
    cpl_image_delete(drift_matrix_ll);
    
    double *new_wave_matrix_data = (double *)cpl_image_get_data_double(new_wave_matrix);
    double *new_dll_matrix_data = (double *)cpl_malloc(size_x*size_y*sizeof(double));
    
    int index_dll = 0;
    cpl_image *new_wave_matrix_order = NULL;
    double *new_wave_matrix_order_data = NULL;
    for (int order = 0; order < size_y; order++) {
        new_wave_matrix_order = cpl_image_extract(new_wave_matrix, 1, order+1,
                                                  size_x, order+1);
        new_wave_matrix_order_data = cpl_image_get_data_double(new_wave_matrix_order);
        new_dll_matrix_data[index_dll] = new_wave_matrix_order_data[1] -
                                            new_wave_matrix_order_data[0];
        index_dll++;
        for (int pxl = 1; pxl < size_x; pxl++) {
            new_dll_matrix_data[index_dll] = new_wave_matrix_order_data[pxl] -
                                                new_wave_matrix_order_data[pxl-1];
            index_dll++;
            //if (pxl % 1000 == 0) {
            //espdr_msg("ORDER %d: PIXEL %d, WAVE value = %.6f", order, pxl, new_wave_matrix_order_data[pxl]);
            //}
        }
        cpl_image_delete(new_wave_matrix_order);
    }
    cpl_image *new_dll_matrix = cpl_image_wrap_double(size_x, size_y, new_dll_matrix_data);
    
    cpl_image *new_air_wave_matrix = cpl_image_new(size_x, size_y, CPL_TYPE_DOUBLE);
    cpl_image *new_air_dll_matrix = cpl_image_new(size_x, size_y, CPL_TYPE_DOUBLE);
    
    // compute the air wave matrix
    for (int order = 1; order <= size_y; order++){
        for (int pix = 1; pix <= size_x; pix++){
            int pir = 0;
            double ll = cpl_image_get(new_wave_matrix, pix, order, &pir);
            double ll_air = espdr_compute_n_air(ll, N_AIR_COEFF_T, N_AIR_COEFF_P);
            double dll = cpl_image_get(new_dll_matrix, pix, order, &pir);
            double dll_air = espdr_compute_n_air(dll, N_AIR_COEFF_T, N_AIR_COEFF_P);
            
            cpl_image_set(new_air_wave_matrix, pix, order, ll/ll_air);
            cpl_image_set(new_air_dll_matrix, pix, order, dll/dll_air);
            
            //espdr_msg("ASE DEBUG: ---> ord=%d, pix=%d, ll=%.12f, n_air=%.12f, ll/n_air=%.12f, wave=%.12f, air_wave=%.12f",
            //order, pix, ll, ll_air, ll/ll_air, cpl_image_get(new_wave_matrix,pix,order,&pir), cpl_image_get(new_air_wave_matrix,pix,order,&pir));
        }
    }
    
    *wave_matrix_RE = cpl_image_duplicate(new_wave_matrix);
    cpl_image_delete(new_wave_matrix);
    
    *dll_matrix_RE = cpl_image_duplicate(new_dll_matrix);
    cpl_image_unwrap(new_dll_matrix);
    cpl_free(new_dll_matrix_data);
    
    *air_wave_matrix_RE = cpl_image_duplicate(new_air_wave_matrix);
    cpl_image_delete(new_air_wave_matrix);
    
    *air_dll_matrix_RE = cpl_image_duplicate(new_air_dll_matrix);
    cpl_image_delete(new_air_dll_matrix);
    
    return(cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief         Computa THAR lamp offset
 @param         AR lines table
 @param         AR lines table length
 @param[out]    lines_table     read lines table
 @param[out]    lines_nb_RE     numebr of read lines
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_compute_THAR_lamp_offset(espdr_line_param *AR_lines_table,
                                              int AR_lines_nb,
                                              double *REF_AR_peak_pxl_err,
                                              double *REF_AR_ll,
                                              double **ll_coeffs,
                                              cpl_image *wave_matrix,
                                              cpl_image *dll_matrix,
                                              espdr_CCD_geometry *CCD_geom,
                                              espdr_inst_config *inst_config,
                                              int fibre,
                                              double *lamp_offset_ar,
                                              double *lamp_offset_ar1,
                                              double *lamp_offset_ar2) {
    
    int i, index_AR, index_AR1, index_AR2, AR_len, AR1_len, AR2_len, ext, order, px;
    double pxl, ll, sum_line_sig_line = 0.0, sum_sig_line = 0.0;
    double *ll_AR, *sig_ll_AR, *line_offset, *sig_line_offset, *line_offset_AR1, *line_offset_AR2;
    double med_v = 0.0, med_v_AR1 = 0.0, med_v_AR2 = 0.0;
    double part_i, part_f;
    cpl_vector *line_offset_vector = NULL, *line_offset_AR1_vector = NULL, *line_offset_AR2_vector = NULL;
    
    espdr_msg("Starting compute THAR lamp offset for fibre %c", fibre_name[fibre]);
    
    AR_len = 0;
    AR1_len = 0;
    AR2_len = 0;
    for (i = 0; i < AR_lines_nb; i++) {
        if (AR_lines_table[i].qc_flag == 1) {
            AR_len++;
            if ((strcmp(AR_lines_table[i].element_name, "AR1") == 0) ||
                (strcmp(AR_lines_table[i].element_name, "ArI") == 0)) {
                AR1_len++;
            }
            if ((strcmp(AR_lines_table[i].element_name, "AR2") == 0) ||
                (strcmp(AR_lines_table[i].element_name, "ArII") == 0)) {
                AR2_len++;
            }
        }
    }
    
    espdr_msg("AR_len: %d, AR1_len: %d, AR2_len: %d", AR_len, AR1_len, AR2_len);
    
    ll_AR = (double *)cpl_calloc(AR_len, sizeof(double));
    sig_ll_AR = (double *)cpl_calloc(AR_len, sizeof(double));
    line_offset = (double *)cpl_calloc(AR_len, sizeof(double));
    line_offset_AR1 = (double *)cpl_calloc(AR1_len, sizeof(double));
    line_offset_AR2 = (double *)cpl_calloc(AR2_len, sizeof(double));
    sig_line_offset = (double *)cpl_calloc(AR_len, sizeof(double));
    
    
    index_AR = 0;
    index_AR1 = 0;
    index_AR2 = 0;
    for (i = 0; i < AR_lines_nb; i++) {
        if (AR_lines_table[i].qc_flag == 1) {
            order = AR_lines_table[i].order;
            pxl = AR_lines_table[i].peak_pxl;
            ll = AR_lines_table[i].wavelength;
            //ll = REF_AR_ll[i];
            
            /*
            ext = espdr_get_ext_index_for_order(order, fibre,
                                                inst_config, CCD_geom);
            ll_AR[index_AR] = ll_coeffs[ext][0] +
                                ll_coeffs[ext][1]*pxl +
                                ll_coeffs[ext][2]*order +
                                ll_coeffs[ext][3]*pxl*pxl +
                                ll_coeffs[ext][4]*pxl*order +
                                ll_coeffs[ext][5]*order*order +
                                ll_coeffs[ext][6]*pxl*pxl*pxl +
                                ll_coeffs[ext][7]*pxl*pxl*order +
                                ll_coeffs[ext][8]*pxl*order*order +
                                ll_coeffs[ext][9]*order*order*order;
            ll_AR[index_AR] = ll_AR[index_AR] * cpl_image_get(dll_matrix, round(pxl), order, &px);
            ll_AR[index_AR] = ll - ll_AR[index_AR];
            */
            
            part_f = modf(pxl, &part_i);
            ll_AR[index_AR] = cpl_image_get(wave_matrix, (int)part_i, order, &px) +
                                cpl_image_get(dll_matrix, (int)part_i, order, &px) * part_f;
            
            sig_ll_AR[index_AR] = REF_AR_peak_pxl_err[i] * cpl_image_get(dll_matrix, round(pxl), order, &px);
            
            line_offset[index_AR] = (ll_AR[index_AR] - ll) / ll * LIGHT_SPEED;
            //line_offset[index_AR] = (ll_AR[index_AR] - AR_lines_table[i].wavelength) /
            //                            AR_lines_table[i].wavelength * LIGHT_SPEED;
            sig_line_offset[index_AR] = sig_ll_AR[index_AR] / ll * LIGHT_SPEED;
            
            if (strcmp(AR_lines_table[i].element_name, "AR1") == 0) {
                line_offset_AR1[index_AR1] = line_offset[index_AR];
                index_AR1++;
            }
            if (strcmp(AR_lines_table[i].element_name, "AR2") == 0) {
                line_offset_AR2[index_AR2] = line_offset[index_AR];
                index_AR2++;
            }
            
            //espdr_msg("%d--> order: %d, pxl: %f (=%d + %f), ll: %f, qc: %d, ext: %d, ll_AR: %f, rv: %f",
            //          i, order, pxl, (int)part_i, ll, AR_lines_table[i].qc_flag, ext,
            //          ll_AR[index_AR], line_offset[index_AR]);
            
            sum_line_sig_line += line_offset[index_AR] * sig_line_offset[index_AR];
            sum_sig_line += sig_line_offset[index_AR];
            index_AR++;
        }
    }
    
    med_v = sum_line_sig_line / sum_sig_line;
    
    line_offset_vector = cpl_vector_wrap(AR_len, line_offset);
    med_v = cpl_vector_get_median(line_offset_vector);
    cpl_vector_unwrap(line_offset_vector);
    
    line_offset_AR1_vector = cpl_vector_wrap(AR1_len, line_offset_AR1);
    med_v_AR1 = cpl_vector_get_median(line_offset_AR1_vector);
    cpl_vector_unwrap(line_offset_AR1_vector);
    
    line_offset_AR2_vector = cpl_vector_wrap(AR2_len, line_offset_AR2);
    med_v_AR2 = cpl_vector_get_median(line_offset_AR2_vector);
    cpl_vector_unwrap(line_offset_AR2_vector);
    
    *lamp_offset_ar = med_v / inst_config->th_lamp_offset_ar;
    *lamp_offset_ar1 = med_v_AR1 / inst_config->th_lamp_offset_ar1;
    *lamp_offset_ar2 = med_v_AR2 / inst_config->th_lamp_offset_ar2;
    
    //espdr_msg("---------> OLD LAMP OFFSETS: %f, %f, %f",
    //          *lamp_offset_ar, med_v/inst_config->th_lamp_offset_ar1, med_v/inst_config->th_lamp_offset_ar2);
    //espdr_msg("---------> NEW LAMP OFFSETS: %f, %f, %f",
    //          *lamp_offset_ar, *lamp_offset_ar1, *lamp_offset_ar2);
    
    cpl_free(ll_AR);
    cpl_free(sig_ll_AR);
    cpl_free(line_offset);
    cpl_free(sig_line_offset);
    
    return (cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief Save the QC KWs
 @param         lines_nb_per_order  number of valid lines for each order
 @param         thar_rejected_lines number of rejected lines
 @param         flux ratio          flux ratio
 @param         global_shift        global shift
 @param         chisq_per_order     reduced CHI2 of ll solution per order
 @param         rms_per_order       dispersion of residuals around fit per order
 @param         res_per_order       spectral resolution per order
 @param         lamp_offset         ThAr lamp offset
 @param         orders_nb           number of orders
 @param         fibre_source        calibration source type
 @param         inst_config         instrument config
 @param         qc_kws              QC KWs names
 @param[out]    keywords_RE         saved QC keywords
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_wave_TH_drift_QC(int *lines_nb_per_order,
                                      double resol_median,
                                      double lamp_offset_ar,
                                      double lamp_offset_ar1,
                                      double lamp_offset_ar2,
                                      int orders_nb,
                                      int saturated_lines_nb,
                                      int valid_lines_nb,
                                      double flux_ratio_median,
                                      double rv_diff_median,
                                      double *fraction_rejected,
                                      double *MAD,
                                      double *MAD2,
                                      double *RMS,
                                      double *RMS2,
                                      double *CHI2,
                                      double **MAD9,
                                      double **RMS9,
                                      int **rejected_lines_nb9,
                                      double **MAD99,
                                      double **RMS99,
                                      int **rejected_lines_nb99,
                                      espdr_src_type fibre_source,
                                      int fibre_nr,
                                      cpl_image *wave_matrix,
                                      cpl_image *s2d_qual,
                                      espdr_inst_config *inst_config,
                                      espdr_CCD_geometry *CCD_geom,
                                      espdr_qc_keywords *qc_kws,
                                      cpl_propertylist *keywords_RE) {

    int i, j, total_lines_nb = 0;
    char *new_keyword = NULL;
    char *drift_kw = NULL;
    char comment[COMMENT_LENGTH];
    int resol_check = 1;
    int mad_check = 1;
    int rms_check = 1;
    int fraction_reject_check = 1;
    int check_flag = 1, total_check_flag = 1;
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    int min_lines_total = inst_config->th_drift_min_lines_total[fibre_nr];
    int max_satur_lines = inst_config->th_drift_max_satur_lines;
    double wave_min_resol = inst_config->wave_min_resol;
    double thar_min_flux_ratio = inst_config->thar_min_flux_ratio[fibre_nr];
    double thar_max_flux_ratio = inst_config->thar_max_flux_ratio[fibre_nr];
    double MAD_limit = inst_config->th_drift_fit_max_mad;
    double RMS_limit = inst_config->th_drift_fit_max_rms;
    double fraction_rejected_limit = inst_config->th_drift_max_reject_lines_fraction;
    
    /* Added keywords checks + change to -1.0 */
    if (isnan(resol_median) || isinf(resol_median)) {
        resol_median = -1.0;
    }
    
    /* QC KWs per order */
    for (i = 0; i < orders_nb; i++) {
        total_lines_nb += lines_nb_per_order[i];
    }
    
    
    /* Global QC KWs */
    
    sprintf(comment, "Thar lines resolution median");
    my_error = espdr_keyword_add_double(qc_kws->qc_wave_order_resol_median_kw,
                                        resol_median,
                                        comment,
                                        &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_ORDERS_RESOL_MEDIAN_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    sprintf(comment, "Thar saturated lines nb");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_thar_satur_lines_nb_kw,
                                     saturated_lines_nb,
                                     comment,
                                     &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_THAR_SATUR_LINES_NB_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    sprintf(comment, "Thar valid lines nb");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_thar_valid_lines_nb_kw,
                                     valid_lines_nb,
                                     comment,
                                     &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_THAR_LINES_NB_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    int tmp_flux_ratio = flux_ratio_median * 10000.0;
    double formatted_flux_ratio = tmp_flux_ratio / 10000.0;
    sprintf(comment, "Thar lines flux ratio median");
    my_error = espdr_keyword_add_double(qc_kws->qc_wave_thar_flux_ratio_median_kw,
                                        formatted_flux_ratio,
                                        comment,
                                        &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_THAR_FLUX_RATIO_MEDIAN_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    int tmp_rv_diff = rv_diff_median * 10000.0;
    double formatted_rv_diff = tmp_rv_diff / 10000.0;
    sprintf(comment, "[m/s] Thar lines RV difference median");
    my_error = espdr_keyword_add_double(qc_kws->qc_wave_thar_rv_diff_median_kw,
                                        formatted_rv_diff,
                                        comment,
                                        &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_THAR_RV_DIFF_MEDIAN_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    sprintf(comment, "TH lamp offset AR");
    my_error = espdr_keyword_add_double(qc_kws->qc_wave_thar_lamp_offset_ar_kw,
                                        lamp_offset_ar,
                                        comment,
                                        &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_THAR_LAMP_OFFSET_AR_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    sprintf(comment, "TH lamp offset AR1");
    my_error = espdr_keyword_add_double(qc_kws->qc_wave_thar_lamp_offset_ar1_kw,
                                        lamp_offset_ar1,
                                        comment,
                                        &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_THAR_LAMP_OFFSET_AR1_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    sprintf(comment, "TH lamp offset AR2");
    my_error = espdr_keyword_add_double(qc_kws->qc_wave_thar_lamp_offset_ar2_kw,
                                        lamp_offset_ar2,
                                        comment,
                                        &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_THAR_LAMP_OFFSET_AR2_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    /* QC KWs per extension */
    
    // MAD, RMS and rejected lines afte the first fit
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        int index9 = 0;
        for (int ccd_x = 0; ccd_x < 3; ccd_x++) {
            for (int ccd_y = 0; ccd_y < 3; ccd_y++) {
                
                int mad_tmp = MAD9[i][index9] * 10000.0;
                MAD9[i][index9] = (double) mad_tmp / 10000.0;
                new_keyword = espdr_add_ext_detector_index_to_keyword(qc_kws->qc_wave_res_mad_kw_first,
                                                                      qc_kws->qc_wave_res_mad_kw_last,
                                                                      i, ccd_x, ccd_y);
                my_error = espdr_keyword_add_double(new_keyword,
                                                    MAD9[i][index9],
                                                    "MAD per detector part",
                                                    &keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword QC_THAR_EXT%d_DET%d_DET%d_RESIDUAL_MAD to the propertylist failed: %s",
                             i, ccd_x, ccd_y, cpl_error_get_message_default(my_error));
                
                int rms_tmp = RMS9[i][index9] * 10000.0;
                RMS9[i][index9] = (double) rms_tmp / 10000.0;
                new_keyword = espdr_add_ext_detector_index_to_keyword(qc_kws->qc_wave_res_rms_kw_first,
                                                                      qc_kws->qc_wave_res_rms_kw_last,
                                                                      i, ccd_x, ccd_y);
                my_error = espdr_keyword_add_double(new_keyword,
                                                    RMS9[i][index9],
                                                    "RMS per detector part",
                                                    &keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword QC_THAR_EXT%d_DET%d_DET%d_RESIDUAL_RMS to the propertylist failed: %s",
                             i, ccd_x, ccd_y, cpl_error_get_message_default(my_error));
                
                new_keyword = espdr_add_ext_detector_index_to_keyword(qc_kws->qc_wave_reject_lines_nb_kw_first,
                                                                      qc_kws->qc_wave_reject_lines_nb_kw_last,
                                                                      i, ccd_x, ccd_y);
                my_error = espdr_keyword_add_int(new_keyword,
                                                 rejected_lines_nb9[i][index9],
                                                 "Drift fit residuals rejected lines nb per detector part",
                                                 &keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword QC_THAR_EXT%d_DET%d_DET%d_REJECT_LINES_NB to the propertylist failed: %s",
                             i, ccd_x, ccd_y, cpl_error_get_message_default(my_error));
                
                index9++;
            }
        }
    }
    
    // MAD, RMS and rejected lines after the second fit
    char *suffix = (char *)cpl_malloc(64 * sizeof(char));
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        int index9 = 0;
        for (int ccd_x = 0; ccd_x < 3; ccd_x++) {
            for (int ccd_y = 0; ccd_y < 3; ccd_y++) {
                
                //if (MAD99[i][index9] > MAD_limit) {
                //    mad_check = 0;
                //    total_check_flag = 0;
                //}
                
                int mad_tmp = MAD99[i][index9] * 10000.0;
                MAD99[i][index9] = (double) mad_tmp / 10000.0;
                sprintf(suffix, "%s2", qc_kws->qc_wave_res_mad_kw_last);
                new_keyword = espdr_add_ext_detector_index_to_keyword(qc_kws->qc_wave_res_mad_kw_first,
                                                                      suffix, i, ccd_x, ccd_y);
                my_error = espdr_keyword_add_double(new_keyword,
                                                    MAD99[i][index9],
                                                    "MAD per detector part",
                                                    &keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword QC_THAR_EXT%d_DET%d_DET%d_RESIDUAL_MAD to the propertylist failed: %s",
                             i, ccd_x, ccd_y, cpl_error_get_message_default(my_error));
                
                //if (RMS99[i][index9] > RMS_limit) {
                //    rms_check = 0;
                //    total_check_flag = 0;
                //}
                
                int rms_tmp = RMS99[i][index9] * 10000.0;
                RMS99[i][index9] = (double) rms_tmp / 10000.0;
                sprintf(suffix, "%s2", qc_kws->qc_wave_res_rms_kw_last);
                new_keyword = espdr_add_ext_detector_index_to_keyword(qc_kws->qc_wave_res_rms_kw_first,
                                                                      suffix, i, ccd_x, ccd_y);
                my_error = espdr_keyword_add_double(new_keyword,
                                                    RMS99[i][index9],
                                                    "RMS per detector part",
                                                    &keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword QC_THAR_EXT%d_DET%d_DET%d_RESIDUAL_RMS to the propertylist failed: %s",
                             i, ccd_x, ccd_y, cpl_error_get_message_default(my_error));
                
                sprintf(suffix, "%s2", qc_kws->qc_wave_reject_lines_nb_kw_last);
                new_keyword = espdr_add_ext_detector_index_to_keyword(qc_kws->qc_wave_reject_lines_nb_kw_first,
                                                                      suffix, i, ccd_x, ccd_y);
                my_error = espdr_keyword_add_int(new_keyword,
                                                 rejected_lines_nb99[i][index9],
                                                 "Drift fit residuals rejected lines nb per detector part",
                                                 &keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword QC_THAR_EXT%d_DET%d_DET%d_REJECT_LINES_NB to the propertylist failed: %s",
                             i, ccd_x, ccd_y, cpl_error_get_message_default(my_error));
                
                index9++;
            }
        }
    }
    cpl_free(suffix);

    for (i = 0; i < CCD_geom->ext_nb; i++) {
        if (MAD2[i] > MAD_limit) {
            mad_check = 0;
            total_check_flag = 0;
        }
        
        int mad_tmp = MAD[i] * 10000.0;
        MAD[i] = (double) mad_tmp / 10000.0;
        new_keyword = espdr_add_index_to_keyword(qc_kws->qc_wave_res_mad_kw_first,
                                                 qc_kws->qc_wave_res_mad_kw_last,
                                                 i);
        my_error = espdr_keyword_add_double(new_keyword,
                                            MAD[i],
                                            "Drift fit residuals MAD per detector",
                                            &keywords_RE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_THAR_DET%d_RESIDUAL_MAD to the propertylist failed: %s",
                     i, cpl_error_get_message_default(my_error));
        
        cpl_free(new_keyword);
        
        mad_tmp = MAD2[i] * 10000.0;
        MAD2[i] = (double) mad_tmp / 10000.0;
        new_keyword = espdr_add_index_to_keyword(qc_kws->qc_wave_res_mad2_kw_first,
                                                 qc_kws->qc_wave_res_mad2_kw_last,
                                                 i);
        my_error = espdr_keyword_add_double(new_keyword,
                                            MAD2[i],
                                            "Drift 2nd fit residuals MAD per detector",
                                            &keywords_RE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_THAR_DET%d_RESIDUAL_MAD2 to the propertylist failed: %s",
                     i, cpl_error_get_message_default(my_error));
        
        cpl_free(new_keyword);
        
        if (RMS2[i] > RMS_limit) {
            rms_check = 0;
            total_check_flag = 0;
        }
        
        int rms_tmp = RMS[i] * 10000.0;
        RMS[i] = (double) rms_tmp / 10000.0;
        new_keyword = espdr_add_index_to_keyword(qc_kws->qc_wave_res_rms_kw_first,
                                                 qc_kws->qc_wave_res_rms_kw_last,
                                                 i);
        my_error = espdr_keyword_add_double(new_keyword,
                                            RMS[i],
                                            "Drift fit residuals RMS per detector",
                                            &keywords_RE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_THAR_DET%d_RESIDUAL_RMS to the propertylist failed: %s",
                     i, cpl_error_get_message_default(my_error));
        
        cpl_free(new_keyword);
        
        rms_tmp = RMS2[i] * 10000.0;
        RMS2[i] = (double) rms_tmp / 10000.0;
        new_keyword = espdr_add_index_to_keyword(qc_kws->qc_wave_res_rms2_kw_first,
                                                 qc_kws->qc_wave_res_rms2_kw_last,
                                                 i);
        my_error = espdr_keyword_add_double(new_keyword,
                                            RMS2[i],
                                            "Drift 2nd fit residuals RMS per detector",
                                            &keywords_RE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_THAR_DET%d_RESIDUAL_RMS2 to the propertylist failed: %s",
                     i, cpl_error_get_message_default(my_error));
        
        cpl_free(new_keyword);
        
        int chi2_tmp = CHI2[i] * 10000.0;
        CHI2[i] = (double) chi2_tmp / 10000.0;
        new_keyword = espdr_add_index_to_keyword(qc_kws->qc_wave_fit_chi2_kw_first,
                                                 qc_kws->qc_wave_fit_chi2_kw_last,
                                                 i);
        my_error = espdr_keyword_add_double(new_keyword,
                                            CHI2[i],
                                            "Drift fit CHI2 per detector",
                                            &keywords_RE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_THAR_DET%d_FIT_CHI2 to the propertylist failed: %s",
                     i, cpl_error_get_message_default(my_error));
        
        cpl_free(new_keyword);
        
        if (fraction_rejected[i] > fraction_rejected_limit) {
            fraction_reject_check = 0;
            total_check_flag = 0;
        }
        
        int frac_tmp = fraction_rejected[i] * 10000.0;
        fraction_rejected[i] = (double) frac_tmp / 10000.0;
        new_keyword = espdr_add_index_to_keyword(qc_kws->qc_wave_reject_lines_fraction_kw_first,
                                                 qc_kws->qc_wave_reject_lines_fraction_kw_last,
                                                 i);
        
        my_error = espdr_keyword_add_double(new_keyword,
                                            fraction_rejected[i],
                                            "Drift fit rejected lines fraction per detector",
                                            &keywords_RE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_THAR_DET%d_REJECT_LINES_FRACTION to the propertylist failed: %s",
                     i, cpl_error_get_message_default(my_error));
        
        cpl_free(new_keyword);
    }
    
    /* QC CHECK KWs */
    
    if ((valid_lines_nb >= min_lines_total) || (fibre_source != THAR)) {
        check_flag = 1;
    } else {
        check_flag = 0;
        total_check_flag = 0;
    }
    
    sprintf(comment, "QC on total number of detected lines");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_lines_tot_check_kw,
                                     check_flag,
                                     comment,
                                     &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_LINES_TOT_CHECK to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    if ((saturated_lines_nb <= max_satur_lines) || (fibre_source != THAR)) {
        check_flag = 1;
    } else {
        check_flag = 0;
        total_check_flag = 0;
    }
    
    sprintf(comment, "QC on number of saturated lines");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_satur_lines_check_kw,
                                     check_flag,
                                     comment,
                                     &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_SATUR_LINES_CHECK to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    if ((resol_median > 0.0) && (resol_median < wave_min_resol)) {
        resol_check = 0;
    }
    
    if ((resol_check == 1) || (fibre_source != THAR)) {
        check_flag = 1;
    } else {
        check_flag = 0;
        total_check_flag = 0;
    }
    
    sprintf(comment, "QC on spectral resolution");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_resol_check_kw,
                                     check_flag,
                                     comment,
                                     &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_WAVE_RESOL_CHECK to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    //espdr_msg("Flux ratio limits of ThAr lines: %.2f <-> %.2f",
    //          thar_min_flux_ratio, thar_max_flux_ratio);
    if ((flux_ratio_median > thar_min_flux_ratio) && (flux_ratio_median < thar_max_flux_ratio)) {
        check_flag = 1;
    } else {
        espdr_msg_warning("The Th lines flux changed significantly, ratio = %.4f",
                          flux_ratio_median);
        check_flag = 0;
        total_check_flag = 0;
    }
    
    sprintf(comment, "QC on TH lines flux ratio");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_thar_flux_ratio_check_kw,
                                     check_flag,
                                     comment,
                                     &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_THAR_FLUX_RATIO_CHECK to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    sprintf(comment, "QC on drift fit residuals MAD");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_res_mad_check_kw,
                                     mad_check,
                                     comment,
                                     &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_THAR_RESIDUALS_MAD_CHECK to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    sprintf(comment, "QC on drift fit residuals RMS");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_res_rms_check_kw,
                                     rms_check,
                                     comment,
                                     &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_THAR_RESIDUALS_RMS_CHECK to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    sprintf(comment, "QC on rejected lines fraction");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_reject_lines_fraction_check_kw,
                                     fraction_reject_check,
                                     comment,
                                     &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_THAR_REJECT_LINES_FRACTION_CHECK to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    // Check on the increasing wavelength though the order
    if (fibre_source == THAR) {
        cpl_image *wave_order = NULL;
        cpl_image *qual_order = NULL;
        double *wave_order_data = NULL;
        int *qual_order_data = NULL;
        int size_x = cpl_image_get_size_x(wave_matrix);
        int size_y = cpl_image_get_size_y(wave_matrix);
        int decreasing = 0;
        check_flag = 1;
        for (i = 0; i < size_y; i++) {
            wave_order = cpl_image_extract(wave_matrix, 1, i+1, size_x, i+1);
            wave_order_data = cpl_image_get_data_double(wave_order);
            qual_order = cpl_image_extract(s2d_qual, 1, i+1, size_x, i+1);
            qual_order_data = cpl_image_get_data_int(qual_order);
            decreasing = 0;
            for (j = 1; j < size_x; j++) {
                if ((qual_order_data[j] != OTHER_BAD_PIXEL) &&
                    (wave_order_data[j] < wave_order_data[j-1])) {
                    check_flag = 0;
                    total_check_flag = 0;
                    decreasing = 1;
                }
            }
            if (decreasing == 1) {
                espdr_msg_warning("Wavelength decreasing in order %d", i+1);
            }
        }
        
        sprintf(comment, "QC on wave sol quality");
        my_error = espdr_keyword_add_int(qc_kws->qc_wave_sol_qual_check_kw,
                                         check_flag,
                                         comment,
                                         &keywords_RE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_WAVE_SOL_QUAL_CHECK to the propertylist failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    
    sprintf(comment, "Overall WAVE QC");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_check_kw,
                                     total_check_flag,
                                     comment,
                                     &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_WAVE_CHECK to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    
    return(cpl_error_get_code());
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Save wave_TH_drift products
 @param
 @return    CPL_ERROR_NONE iff OK
 
 */
/*---------------------------------------------------------------------------*/


cpl_error_code espdr_save_wave_TH_drift_products(cpl_frameset *frameset,
                                                 cpl_parameterlist *parameters,
                                                 const char *recipe_id,
                                                 cpl_frameset *used_frames,
                                                 const char *WAVE_tag,
                                                 int fibre_nr,
                                                 cpl_propertylist *keywords,
                                                 espdr_inst_config *inst_config,
                                                 cpl_image *wave_matrix,
                                                 cpl_image *air_wave_matrix,
                                                 cpl_image *dll_matrix,
                                                 cpl_image *air_dll_matrix,
                                                 cpl_table *TH_table,
                                                 cpl_table *AR_table,
                                                 cpl_table *raw_table) {
    cpl_error_code my_error = CPL_ERROR_NONE;
    char filename[FILENAME_LENGTH];
    char *new_keyword =(char *)cpl_malloc(KEYWORD_LENGTH*sizeof(char));
    
    // Saving vacuum and air WAVE_MATRIX
    /* Save the PRO.CATG */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_WAVE_MATRIX_DRIFT, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);
    
    sprintf(filename, "%s_%c.fits",
            inst_config->wave_matrix_drift_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving WAVE_MATRIX_DRIFT for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);
    
    my_error = espdr_dfs_image_save_one_ext(frameset, parameters,
                                            used_frames, recipe_id,
                                            keywords, filename,
                                            wave_matrix, CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save_one_ext failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
    espdr_msg("WAVE_MATRIX saved under: %s", filename);
    
    /* Save the PRO.CATG */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_AIR_WAVE_MATRIX_DRIFT, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);
    
    sprintf(filename, "%s_%c.fits",
            inst_config->air_wave_matrix_drift_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving AIR_WAVE_MATRIX_DRIFT for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);
    
    my_error = espdr_dfs_image_save_one_ext(frameset, parameters,
                                            used_frames, recipe_id,
                                            keywords, filename,
                                            air_wave_matrix, CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save_one_ext failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
    espdr_msg("AIR_WAVE_MATRIX_DRIFT saved under: %s", filename);
    
    // Saving vacuum and air dll matrices
    /* Save the PRO.CATG */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_DLL_MATRIX_DRIFT, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);
    
    sprintf(filename, "%s_%c.fits",
            inst_config->dll_matrix_drift_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving DLL_MATRIX_DRIFT for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);
    
    my_error = espdr_dfs_image_save_one_ext(frameset, parameters,
                                            used_frames, recipe_id,
                                            keywords, filename,
                                            dll_matrix, CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save_one_ext failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
    
    /* Save the PRO.CATG */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_AIR_DLL_MATRIX_DRIFT, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);
    
    sprintf(filename, "%s_%c.fits",
            inst_config->air_dll_matrix_drift_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving AIR_DLL_MATRIX_DRIFT for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);
    
    my_error = espdr_dfs_image_save_one_ext(frameset, parameters,
                                            used_frames, recipe_id,
                                            keywords, filename,
                                            air_dll_matrix, CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save_one_ext failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
        
    // Saving TH & AR lines table
    /* Save the PRO.CATG */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_TH_LINE_TABLE, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);
    
    sprintf(filename, "%s_%c.fits",
            inst_config->th_line_table_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving TH_LINE_TABLE for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);
    
    my_error = cpl_dfs_save_table(frameset, NULL, parameters, used_frames,
                                  NULL, TH_table, NULL,
								  recipe_id, keywords, NULL,
                                  PACKAGE "/" PACKAGE_VERSION,
                                  filename);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
    
    /* Save the PRO.CATG */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_AR_LINE_TABLE, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);
    
    sprintf(filename, "%s_%c.fits",
            inst_config->ar_line_table_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving AR_LINE_TABLE for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);
    
    my_error = cpl_dfs_save_table(frameset, NULL, parameters, used_frames,
                                  NULL, AR_table, NULL,
								  recipe_id, keywords, NULL,
                                  PACKAGE "/" PACKAGE_VERSION,
                                  filename);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
    
#if SAVE_TH_REF_TABLES_IN_DRIFT
    /* Save the TH_REF_TABLES FITS */
    char ins_mode_bin[32];
    //char valid_date[16] = "2017-09-01";
    //char valid_date[16] = "2018-10-14";
    //char valid_date[16] = "2019-06-21";
    //char valid_date[16] = "2014-11-10"; // CORALIE after November 2014
    char valid_date[16] = "2011-01-01"; // CORALIE after June 2007
    my_error = espdr_save_TH_REF_TABLE(TH_table,
                                       frameset,
                                       used_frames,
                                       parameters,
                                       inst_config,
                                       keywords,
                                       fibre_nr,
                                       "STATIC",
                                       //57997.0,
                                       //58405.0,
                                       //58655.0,
                                       //56971.0, // CORALIE after 10th November 2014
                                       54252.0, // CORALIE after 1st June 2007
                                       valid_date,
                                       ins_mode_bin);
    espdr_msg("INS BIN MODE: %s, valid date: %s",
              ins_mode_bin, valid_date);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_save_TH_REF_TABLE failed: %s",
                 cpl_error_get_message_default(my_error));
#endif

#if SAVE_AR_REF_TABLES_IN_DRIFT
    char AR_ins_mode_bin[32];
    //char valid_date_AR[16] = "2014-11-10"; // CORALIE after November 2014
    char valid_date_AR[16] = "2007-06-01"; // CORALIE after June 2007
    espdr_msg("valid date: %s", valid_date_AR);
    my_error = espdr_save_TH_REF_TABLE(AR_table,
                                       frameset,
                                       used_frames,
                                       parameters,
                                       inst_config,
                                       keywords,
                                       fibre_nr,
                                       "AR",
                                       //56971, // CORALIE after 10th November 2014
                                       54252.0, // CORALIE after 1st June 2007
                                       valid_date_AR,
                                       AR_ins_mode_bin);
    espdr_msg("INS BIN MODE: %s, valid date: %s",
              AR_ins_mode_bin, valid_date_AR);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_save_AR_TABLE failed: %s",
                 cpl_error_get_message_default(my_error));

#endif
    
#if PROCESS_RAW_LINES_IN_DRIFT
    /* Save the raw lines fits table */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_LINE_TABLE_RAW, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);
    
    sprintf(filename, "%s_%c.fits",
            inst_config->line_table_raw_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving LINE_TABLE_RAW for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);
    
    my_error = cpl_dfs_save_table(frameset, NULL, parameters, used_frames,
                                  NULL, raw_table, NULL,
                                  RECIPE_ID, keywords, NULL,
                                  PACKAGE "/" PACKAGE_VERSION,
                                  filename);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
#endif
    
#if SAVE_REF_MATRICES_IN_DRIFT
    espdr_msg("Saving STATIC WAVE/DLL MATRIXes");
    static char *date = "2007-06-01";
    double static_mjd = 54252.0;
    
    sprintf(new_keyword, "STATIC_WAVE_MATRIX_%c", fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);
    my_error = cpl_propertylist_update_double(keywords, "MJD-OBS", static_mjd);

    sprintf(filename, "%s_STATIC_WAVE_MATRIX_%c_%s.fits",
            inst_config->instrument, fibre_name[fibre_nr], date);
    espdr_msg("Saving STATIC_WAVE_MATRIX for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);
    
    my_error = espdr_dfs_image_save_one_ext(frameset, parameters,
                                            used_frames, recipe_id,
                                            keywords, filename,
                                            wave_matrix, CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save_one_ext failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
    espdr_msg("STATIC_WAVE_MATRIX saved under: %s", filename);
    
    sprintf(new_keyword, "STATIC_DLL_MATRIX_%c", fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);
    my_error = cpl_propertylist_update_double(keywords, "MJD-OBS", static_mjd);

    sprintf(filename, "%s_STATIC_DLL_MATRIX_%c_%s.fits",
            inst_config->instrument, fibre_name[fibre_nr], date);
    espdr_msg("Saving STATIC_DLL_MATRIX for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);
    
    my_error = espdr_dfs_image_save_one_ext(frameset, parameters,
                                            used_frames, recipe_id,
                                            keywords, filename,
                                            dll_matrix, CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save_one_ext failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
    espdr_msg("STATIC_DLL_MATRIX saved under: %s", filename);
    
#endif

    return (cpl_error_get_code());
}
