/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: asegovia $
 * $Date: 2015-05-01 14:00:00 $
 * $Revision:     $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------
 Includes
 ----------------------------------------------------------------------------*/

/* Utility fonctions */
#include <espdr_utils.h>
#include <espdr_background.h>
#include <espdr_msg.h>
#include <espdr_science.h>
#include <espdr_bepop.h>
#include <espdr_sdp_spectrum.h>

/* DFS functions */
#include <espdr_dfs.h>
#include <espdr_instrument.h>
#include <espdr_keywords.h>

/* DRL functions */
#include <espdr_bias.h>
#include <espdr_flat.h>
#include <espdr_blaze.h>
#include <espdr_parameters.h>
#include <espdr_CCD.h>
#include <espdr_orders.h>
#include <espdr_led_flat.h>
#include <espdr_ccf.h>
#include <espdr_flux.h>
#include <espdr_drift.h>
#include <espdr_utils.h>
#include <espdr_pfits.h>
/* Library */
#include <cpl.h>
#include <assert.h>
#include <gsl/gsl_statistics_double.h>	

/*----------------------------------------------------------------------------
 Defines
 ----------------------------------------------------------------------------*/

#define RECIPE_ID "espdr_sci_red"
#define RECIPE_AUTHOR "Ch. Lovis, A. Segovia"
#define RECIPE_CONTACT "https://support.eso.org"

struct espdr_sdp_spectrum {
    /* Indicates the number of data points of the spectrum. */
    cpl_size nelem;
    
    /* Stores all the SDP keywords for the primary header and table extension. */
    cpl_propertylist *proplist;
    
    /* The table for the spectrum data points. */
    cpl_table *table;
};

/*-----------------------------------------------------------------------------
 Plugin registration
 -----------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist * list);

/*----------------------------------------------------------------------------
 Private functions prototypes
 ----------------------------------------------------------------------------*/

/*
 *   Plugin initalization, execute and cleanup handlers
 */

static int espdr_sci_red_create(cpl_plugin *);
static int espdr_sci_red_exec(cpl_plugin *);
static int espdr_sci_red_destroy(cpl_plugin *);

/*----------------------------------------------------------------------------
 Static variables
 ----------------------------------------------------------------------------*/

//static int fibres_nb_sr = inst_config->fibres_nb;

static char espdr_sci_red_description_short[] = "Performs science reduction";

static char espdr_sci_red_description[] =
"This is the main recipe of the ESPRESSO DRL, used to process all science \
frames. It makes use of all calibration products previously generated and \
creates both S2D and S1D science spectra with associated error and quality \
maps. For stellar spectra, the recipe also computes the radial velocity via \
the cross-correlation method.\
Input Frames : \n\
OBJ_SKY or OBJ_FP or OBJ_DARK or OBJ_THAR or OBJ_LFC\n\
- A static table indicating the detector geometry (Tag = CCD_GEOM)\n\
- A static table to set instrument config. parameters (Tag = INST_CONFIG)\n\
- The atmospheric exctinction table (Tag = EXT_TABLE)\n\
- A reference flux STD star catalog (Tag= STD_TABLE)\n\
- A master bias residuals frame (optional) (Format=PRE, Tag = MASTER_BIAS_RES)\n\
- The hot pixel map frame (Format=PRE, Tag = HOT_PIXEL_MASK)\n\
- The bad pixel map frame (Format=PRE, Tag = BAD_PIXEL_MASK)\n\
- An order table for each fiber (Format=PRE, Tag = ORDER_TABLE_A/B)\n\
- order profile frames (Format=PRE, Tag = ORDER_PROFILE_A,B)\n\
- master flat frames (Format=S2D, Tag = FSPECTRUM_A,B)\n\
- blaze frames (Format=S2D, Tag = BLAZE_A,B)\n\
- A wave map (vacuum) for fibre A (Format=PRE, Tag = WAVE_MATRIX_THAR_FP_A)\n\
- A wave map (vacuum) for fibre B (Format=PRE, Tag = WAVE_MATRIX_FP_THAR_B)\n\
- A DLL map (vacuum) for fibre A (Format=PRE, Tag = DLL_MATRIX_THAR_FP_A)\n\
- A DLL map (vacuum) for fibre B (Format=PRE, Tag = DLL_MATRIX_FP_THAR_B)\n\
- The S2D extracted blaze spectrum for fibre A/B (Format=PRE, Tag = S2D_BLAZE_THAR_FP_A/FP_THAR_B)\n\
- the contamination of fibre B on fibre A	(Tag = 'CONTAM_THAR', 'CONTAM_LFC' or 'CONTAM_FP') \n\
- fibre B relative to fibre A 	(Format=PRE, Tag = 'REL_EFF_B') \n\
- absolute efficiency for fibre A (Format=S2D, Tag=ABS_EFF_A)\n\
- A static MASK_TABLE (Tag=MASK_TABLE)\n\
- A static MASK_LUT  (Tag=MASK_LUT)\n\
- A static FLUX_TEMPLATE (Tag=FLUX_TEMPLATE)\n\
Products : \n\
- the residuals from CCF computation (PRO.CATG=CCF_RESIDUALS_A)\n\
- the drift matrix relative to fibre B (PRO.CATG=DRIFT_MATRIX_B)\n\
- the S2D spectrum blaze fibre A (PRO.CATG=S2D_BLAZE_A)\n\
- the S2D spectrum for fibre A,B (PRO.CATG=S2D_A/B)\n\
- the S1D spectrum for fibre A (PRO.CATG=S1D_SKYSUB_A)\n\
- the S1D flux calibrated spectrum for fibre A (PRO.CATG=S1D_FLUXCAL_A)\n\
- the CCF for fibre A(PRO.CATG=CCF_A)\n\
Optional products: (created if extra_products_sw=TRUE and cosmic_detection_sw=1)\n\
- the Cosmics map (PRO.CATG=CRH_MAP)\n\
- the frame used to determine the Cosmic ray map (PRO.CATG=CCD_CORR_SCIENCE)\n\
- the post-filtered cosmics map (if lacosmic.post-filter-x/y are both >0) \n\
\n";

/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/**
 @brief    Build the list of available plugins, for this module.
 @param    list    the plugin list
 @return   0 if everything is ok, -1 otherwise

 Create the recipe instance and make it available to the application using
 the interface. This function is exported.
 */
/*---------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist * list)
{
	cpl_recipe *recipe = NULL;
	cpl_plugin *plugin = NULL;

	recipe = cpl_calloc(1, sizeof(*recipe));

	if (recipe == NULL) {
		return 1;
	}

	plugin = &recipe->interface;
	if (cpl_plugin_init(plugin,
		CPL_PLUGIN_API,	/* Plugin API */
		ESPDR_BINARY_VERSION,	/* Plugin version */
		CPL_PLUGIN_TYPE_RECIPE,	/* Plugin type */
		RECIPE_ID,	/* Plugin name */
		espdr_sci_red_description_short,	/* Short help */
		espdr_sci_red_description,	/* Detailed help */
		RECIPE_AUTHOR,	/* Author name */
		RECIPE_CONTACT,	/* Contact address as in xsh,
			           PACKAGE_BUGREPORT in the tpl */
		espdr_get_license(),	/* Copyright */
		espdr_sci_red_create,
		espdr_sci_red_exec,
		espdr_sci_red_destroy)) {
		cpl_msg_error(cpl_func, "Plugin initialiazation failed");
		(void) cpl_error_set_where(cpl_func);
		return 1;
	}

	if (cpl_pluginlist_append(list, plugin)) {
		cpl_msg_error(cpl_func, "Error adding plugin to list");
		(void) cpl_error_set_where(cpl_func);
		return 1;
	}


    /*return (cpl_error_get_code() != CPL_ERROR_NONE); as in xsh*/
	return 0;
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Setup the recipe options
 @param    plugin  the plugin
 @return   0 if everything is ok

 Create the recipe instance and make it available to the application using
 the interface.

 */
/*--------------------------------------------------------------------------*/

static int espdr_sci_red_create(cpl_plugin * plugin)
{
	cpl_recipe *recipe = NULL;

	cpl_error_code error_got = CPL_ERROR_NONE;

	/* Init sci_red parameters */
	const char* init_ovsc_method = "mean";
	const char* init_bias_correction_method = "overscan";
	double init_ovsc_ksigma = 4.0;
	int init_ovsc_max_iter = 10;


    const char* init_background_sw = "on";
    int init_bkgr_grid_size_x = 577;
    int init_bkgr_grid_size_y = 256;
	const char* init_wave_cal_source = "THAR";
	const char* init_mask_table_id = "XX";
	double init_rv_center = -9999.;
	double init_rv_range = 20.0;
	double init_rv_step = 0.5;
    const char* init_extraction_method = "horne";
	double init_ksigma_cosmic = 3.5;
	const char* init_bias_res_removal_sw = "on";
    const char* init_flux_correction_type = "AUTO";
	const char* init_drift_correction_sw = "on";
    const char* init_drift_method = "flux_global_drift_global_sequential_fit";
    const char* init_drift_space = "pixel";
    double init_drift_ksigma = 50;
    const char* init_sky_sub_method = "pixel-by-pixel";
    int init_sky_sub_sliding_box_size = 50;
    double init_slit_loss = -1.0;
    int init_crh_activation_sw = 0;
    int init_post_filter_x = 0;
    int init_post_filter_y = 0;
    const char* init_post_filter_mode = "dilation";
    double init_lacosmic_sigma_lim = 4.0;
    double init_lacosmic_f_lim = 4.0;
    int init_lacosmic_max_iter = 5;
    
	/* Do not create the recipe if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

	espdr_ensure(plugin == NULL, CPL_ERROR_NULL_INPUT, "Null plugin");

	/* Verify plugin type */
    espdr_ensure(cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE,
                 CPL_ERROR_TYPE_MISMATCH, "Plugin is not a recipe");

	/* Get the recipe */
	recipe = (cpl_recipe *)plugin;

	/* Create the parameters list in the cpl_recipe object */
	recipe->parameters = cpl_parameterlist_new();
    espdr_ensure(recipe->parameters == NULL,
                 CPL_ERROR_ILLEGAL_OUTPUT,
                 "Parameter list allocation failed");

	/* Set sci_red parameters */
    espdr_OVSC_param OVSC_param = {init_ovsc_method, init_ovsc_ksigma,
        init_ovsc_max_iter, init_bias_correction_method};


    espdr_SCI_RED_param SCI_RED_param = {
        init_background_sw,
        init_bkgr_grid_size_x,
        init_bkgr_grid_size_y,
        init_wave_cal_source,
        init_mask_table_id,
        init_rv_center,
        init_rv_range,
        init_rv_step,
        init_extraction_method,
        init_ksigma_cosmic,
        init_bias_res_removal_sw,
        init_flux_correction_type,
        init_drift_correction_sw,
        init_drift_method,
        init_drift_space,
        init_drift_ksigma,
        init_sky_sub_method,
        init_sky_sub_sliding_box_size,
        init_slit_loss,
        init_crh_activation_sw,
        init_post_filter_x,
        init_post_filter_y,
        init_post_filter_mode,
        init_lacosmic_sigma_lim,
        init_lacosmic_f_lim,
        init_lacosmic_max_iter
    };

	error_got = espdr_parameters_OVSC_create(RECIPE_ID,
						recipe->parameters,
						&OVSC_param);

	espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
			"Error creating the OVSC parameters data structure");

	error_got = espdr_parameters_SCI_RED_create(RECIPE_ID,
						recipe->parameters,
						&SCI_RED_param);

	espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
			"Error creating the SCI_RED parameters data structure");

	return (error_got);
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Execute the plugin instance given by the interface
 @param    plugin  the plugin
 @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int espdr_sci_red_exec(cpl_plugin * plugin)
{
	cpl_recipe *recipe = NULL;
	int recipe_status;
	cpl_errorstate initial_errorstate = cpl_errorstate_get();

	/* Return immediately if an error code is already set */
	if (cpl_error_get_code() != CPL_ERROR_NONE) {
		cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
		cpl_func, __LINE__, cpl_error_get_where());
		return (int)cpl_error_get_code();
	}

	espdr_ensure(plugin == NULL, CPL_ERROR_NULL_INPUT, "Null plugin");

	/* Verify plugin type */
	espdr_ensure(cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE,
                 CPL_ERROR_TYPE_MISMATCH, "Plugin is not a recipe");

	/* Get the recipe */
	recipe = (cpl_recipe *) plugin;

	/* Verify parameter and frame lists */
	espdr_ensure(recipe->parameters == NULL,
                 CPL_ERROR_NULL_INPUT,
                 "Recipe invoked with NULL parameter list");

	espdr_ensure(recipe->frames == NULL, CPL_ERROR_NULL_INPUT,
                 "Recipe invoked with NULL frame set");

    /* Set the time display for logs */
    cpl_msg_set_time_on();
    
	/* Invoke the recipe */
	recipe_status = espdr_sci_red(recipe->parameters, recipe->frames, RECIPE_ID);

	/* Ensure DFS-compliance of the products */
	if (cpl_dfs_update_product_header(recipe->frames)) {
		if (!recipe_status) recipe_status = (int)cpl_error_get_code();
	}

	if (!cpl_errorstate_is_equal(initial_errorstate)) {
		/* Dump the error history since recipe execution start.
		At this point the recipe cannot recover from the error */
		cpl_errorstate_dump(initial_errorstate, CPL_FALSE, NULL);
	}

	/*printf("recipe status: %d\n", recipe_status);*/
    
    /* Cancel the time display for logs */
    cpl_msg_set_time_off();
    
	return recipe_status;
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Destroy what has been created by the 'create' function
 @param    plugin  the plugin
 @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int espdr_sci_red_destroy(cpl_plugin * plugin)
{
	cpl_recipe *recipe = NULL;

	cpl_error_reset();

	/* Verify the plugin existance */
	espdr_ensure(plugin == NULL, CPL_ERROR_NULL_INPUT, "Null plugin");

	/* Verify plugin type */
	espdr_ensure(cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE,
                 CPL_ERROR_TYPE_MISMATCH, "Plugin is not a recipe");

	/* Get the recipe */
	recipe = (cpl_recipe *) plugin;

	cpl_parameterlist_delete(recipe->parameters);

	return (cpl_error_get_code());
}


