/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2024-01-24 16:52:15 $
 * $Revision: 1.6 $
 * $Name: not supported by cvs2svn $
 */

#include <espdr_OH_correction.h>

/*----------------------------------------------------------------------------
 Functions code, functions to correct OH lines
 ----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/
/**
 @brief     Compute and apply the OH correction
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_correct_OH_lines(const cpl_frameset* set,
                                      cpl_frameset* used_frames,
                                      espdr_inst_config *inst_config,
                                      espdr_qc_keywords *qc_kws,
                                      double Texp,
                                      double airmass_mean,
                                      cpl_image *s2d_flux_A,
                                      cpl_image *s2d_error_A,
                                      cpl_image *s2d_qual_A,
                                      cpl_image *s2d_flux_B,
                                      cpl_image *s2d_error_B,
                                      cpl_image *s2d_qual_B,
                                      cpl_image *wave_matrix_A,
                                      cpl_image *wave_matrix_B,
                                      cpl_image **OH_corr_flux_RE,
                                      cpl_image **OH_corr_error_RE,
                                      cpl_image **OH_corr_qual_RE,
                                      cpl_propertylist **OH_keywords_RE) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    
    cpl_image *master_spectrum_A, *master_spectrum_B, *master_error_A;
    cpl_image *master_error_B, *wavelength_A, *wavelength_B;
    cpl_image *order_master_A_img, *order_master_err_A_img, *order_wavelength_A_img;
    cpl_image *order_master_B_img, *order_wavelength_B_img;
    cpl_image *flux_A_OH_corrected, *error_A_OH_corrected;
    cpl_image *order_OH_flux_A_img, *order_OH_error_A_img;
    cpl_frame *OH_lines_regions_frame;
    espdr_OH_line *OH_lines_table = NULL;
    int OH_lines_nb;
    
    my_error = espdr_get_OH_static_frames_from_set(set, used_frames,
                                                   &master_spectrum_A,
                                                   &master_error_A,
                                                   &wavelength_A,
                                                   &master_spectrum_B,
                                                   &master_error_B,
                                                   &wavelength_B,
                                                   &OH_lines_regions_frame);
    
    cpl_image *order_wave_A_img, *order_wave_B_img;
    cpl_image *order_flux_A_img, *order_error_A_img, *order_flux_B_img;
    cpl_image *order_flux_base_A_img, *order_flux_base_B_img, *order_flux_mad_sub_B_img;
    cpl_image *order_flux_median_A_img, *order_flux_median_B_img, *order_flux_mad_B_img;
    cpl_image *flux_base_A_img, *flux_base_B_img, *flux_mad_sub_B_img;
    cpl_image *flux_median_A_img, *flux_median_B_img, *flux_mad_B_img;
    double *order_wave_A, *order_vel_A, *order_wave_B, *order_vel_B;
    double *runn_median_flux_A, *runn_median_flux_B, *runn_mad_flux_B;
    int nx_s2d = cpl_image_get_size_x(s2d_flux_A);
    int ny_s2d = cpl_image_get_size_y(s2d_flux_A);
    int nx_wave_matrix = cpl_image_get_size_x(wave_matrix_B);
    int nx_master = cpl_image_get_size_x(master_spectrum_B);
    double mid_ll;
    
    espdr_msg("S2D_A flux size: %lld x %lld",
              cpl_image_get_size_x(s2d_flux_A),
              cpl_image_get_size_y(s2d_flux_A));
    espdr_msg("Master spectrum A size: %lld x %lld",
              cpl_image_get_size_x(master_spectrum_A),
              cpl_image_get_size_y(master_spectrum_A));
    espdr_msg("Master error A size: %lld x %lld",
              cpl_image_get_size_x(master_error_A),
              cpl_image_get_size_y(master_error_A));
    espdr_msg("wavelength A size: %lld x %lld",
              cpl_image_get_size_x(wavelength_A),
              cpl_image_get_size_y(wavelength_A));
    espdr_msg("S2D_B flux size: %lld x %lld",
              cpl_image_get_size_x(s2d_flux_B),
              cpl_image_get_size_y(s2d_flux_B));
    espdr_msg("Master spectrum B size: %lld x %lld",
              cpl_image_get_size_x(master_spectrum_B),
              cpl_image_get_size_y(master_spectrum_B));
    espdr_msg("Master error B size: %lld x %lld",
              cpl_image_get_size_x(master_error_B),
              cpl_image_get_size_y(master_error_B));
    espdr_msg("wavelength B size: %lld x %lld",
              cpl_image_get_size_x(wavelength_B),
              cpl_image_get_size_y(wavelength_B));
    
    OH_lines_table = (espdr_OH_line *)cpl_malloc(MAX_NUMBER_OH_LINES * sizeof(espdr_OH_line));
    my_error = espdr_OH_lines_table_init(OH_lines_table, MAX_NUMBER_OH_LINES);
    
    my_error = espdr_read_OH_LINES_TABLE(OH_lines_regions_frame, OH_lines_table, &OH_lines_nb);
    //my_error = espdr_read_OH_LINES_TABLE_CSV(OH_lines_regions_frame, OH_lines_table, &OH_lines_nb);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, cpl_error_get_code(),
                 "espdr_read_OH_LINES_TABLE_CSV failed: %s",
                 cpl_error_get_message_default(my_error));
    
    espdr_msg("nx_s2d: %d, ny_s2d: %d", nx_s2d, ny_s2d);
    
    flux_base_A_img = cpl_image_new(nx_s2d, ny_s2d, CPL_TYPE_DOUBLE);
    flux_base_B_img = cpl_image_new(nx_s2d, ny_s2d, CPL_TYPE_DOUBLE);
    flux_mad_sub_B_img = cpl_image_new(nx_s2d, ny_s2d, CPL_TYPE_DOUBLE);
    flux_median_A_img = cpl_image_new(nx_s2d, ny_s2d, CPL_TYPE_DOUBLE);
    flux_median_B_img = cpl_image_new(nx_s2d, ny_s2d, CPL_TYPE_DOUBLE);
    flux_mad_B_img = cpl_image_new(nx_s2d, ny_s2d, CPL_TYPE_DOUBLE);

    for (int order = 0; order < cpl_image_get_size_y(s2d_flux_B); order++) {
        //espdr_msg("order %d", order);
        order_vel_A = (double *)cpl_calloc(nx_master, sizeof(double));
        order_wave_A_img = cpl_image_extract(wavelength_A, 1, order+1, nx_master, order+1);
        order_wave_A = cpl_image_get_data_double(order_wave_A_img);
        mid_ll = order_wave_A[(int)(nx_master/2)];
        for (int i = 0; i < nx_master; i++) {
            order_vel_A[i] = (order_wave_A[i] - mid_ll) / mid_ll * LIGHT_SPEED / 1000.0;
        }
        order_flux_A_img = cpl_image_extract(master_spectrum_A, 1, order+1, nx_master, order+1);
        
        order_vel_B = (double *)cpl_calloc(nx_wave_matrix, sizeof(double));
        order_wave_B_img = cpl_image_extract(wave_matrix_B, 1, order+1, nx_wave_matrix, order+1);
        order_wave_B = cpl_image_get_data_double(order_wave_B_img);
        mid_ll = order_wave_B[(int)(nx_wave_matrix/2)];
        for (int i = 0; i < nx_wave_matrix; i++) {
            order_vel_B[i] = (order_wave_B[i] - mid_ll) / mid_ll * LIGHT_SPEED / 1000.0;
        }
        order_flux_B_img = cpl_image_extract(s2d_flux_B, 1, order+1, nx_s2d, order+1);
        
        // Running median on A
        //espdr_msg("Computing running median A");
        runn_median_flux_A = (double *)cpl_calloc(nx_s2d, sizeof(double));
        my_error = espdr_compute_running_median_mad(order_flux_A_img, order_vel_A, nx_master,
                                                    RUNNING_MEDIAN_HALF_WINDOW, 0, runn_median_flux_A);
        order_flux_median_A_img = cpl_image_wrap_double(nx_master, 1, runn_median_flux_A);
        
        // Subtract median for A
        
        order_flux_base_A_img = cpl_image_subtract_create(order_flux_A_img, order_flux_median_A_img);
        
        // Running median (+/- 50 km/s) on B
        
        //espdr_msg("Computing running median B");
        runn_median_flux_B = (double *)cpl_calloc(nx_s2d, sizeof(double));
        my_error = espdr_compute_running_median_mad(order_flux_B_img, order_vel_B, nx_s2d,
                                                    RUNNING_MEDIAN_HALF_WINDOW, 0, runn_median_flux_B);
        order_flux_median_B_img = cpl_image_wrap_double(nx_s2d, 1, runn_median_flux_B);
        
        // Subtract median for B
        
        order_flux_base_B_img = cpl_image_subtract_create(order_flux_B_img, order_flux_median_B_img);
        
        // Running MAD
        
        //espdr_msg("Computing running mad B");
        runn_mad_flux_B = (double *)cpl_calloc(nx_s2d, sizeof(double));
        my_error = espdr_compute_running_median_mad(order_flux_B_img, order_vel_B, nx_s2d,
                                                    RUNNING_MEDIAN_HALF_WINDOW, 1, runn_mad_flux_B);
        order_flux_mad_B_img = cpl_image_wrap_double(nx_s2d, 1, runn_mad_flux_B);
        
        // Subtract MAD for B
        
        my_error = cpl_image_multiply_scalar(order_flux_mad_B_img, MAD_FACTOR);
        order_flux_mad_sub_B_img = cpl_image_subtract_create(order_flux_base_B_img, order_flux_mad_B_img);
        
        // Insert into full images
        
        my_error = cpl_image_copy(flux_base_A_img, order_flux_base_A_img, 1, order+1);
        my_error = cpl_image_copy(flux_base_B_img, order_flux_base_B_img, 1, order+1);
        my_error = cpl_image_copy(flux_mad_sub_B_img, order_flux_mad_sub_B_img, 1, order+1);
        my_error = cpl_image_copy(flux_median_A_img, order_flux_median_A_img, 1, order+1);
        my_error = cpl_image_copy(flux_median_B_img, order_flux_median_B_img, 1, order+1);
        my_error = cpl_image_copy(flux_mad_B_img, order_flux_mad_B_img, 1, order+1);
        
        cpl_image_delete(order_flux_base_A_img);
        cpl_image_delete(order_flux_base_B_img);
        cpl_image_delete(order_flux_mad_sub_B_img);
        cpl_image_unwrap(order_flux_median_A_img);
        cpl_free(runn_median_flux_A);
        cpl_image_unwrap(order_flux_median_B_img);
        cpl_free(runn_median_flux_B);
        cpl_image_unwrap(order_flux_mad_B_img);
        cpl_free(runn_mad_flux_B);
        
    }
    
    //my_error = cpl_image_save(s2d_flux_A, "NIRPS_S2D_flux_A.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    //my_error = cpl_image_save(s2d_flux_B, "NIRPS_S2D_flux_B.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    //my_error = cpl_image_save(flux_base_A_img, "NIRPS_flux_base_A.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    //my_error = cpl_image_save(flux_base_B_img, "NIRPS_flux_base_B.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    //my_error = cpl_image_save(flux_mad_sub_B_img, "NIRPS_flux_mad_sub_B.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    //my_error = cpl_image_save(flux_median_A_img, "NIRPS_flux_median_A.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    //my_error = cpl_image_save(flux_median_B_img, "NIRPS_flux_median_B.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    //my_error = cpl_image_save(flux_mad_B_img, "NIRPS_flux_mad_B.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    
    // Compute the integrated flux
    double int_flux_s2d_B[OH_lines_nb];
    double int_flux_master_B[OH_lines_nb];
    int line_present_B[OH_lines_nb];
    double OH_ratio[OH_lines_nb];
    int OH_index = 0, wave_index = 0, wave_index_start, wave_index_end;
    int master_wave_index = 0, master_wave_index_start, master_wave_index_end;
    for (int order = 0; order < ny_s2d; order++) {
        //espdr_msg("Computing OH lines from order %d", order);
        
        order_flux_B_img = cpl_image_extract(flux_base_B_img, 1, order+1, nx_s2d, order+1);
        order_flux_mad_sub_B_img = cpl_image_extract(flux_mad_sub_B_img, 1, order+1, nx_s2d, order+1);
        order_wave_B_img = cpl_image_extract(wave_matrix_B, 1, order+1, nx_wave_matrix, order+1);
        double *order_flux_B_data = cpl_image_get_data_double(order_flux_B_img);
        double *order_flux_mad_sub_B_data = cpl_image_get_data_double(order_flux_mad_sub_B_img);
        double *order_wave_B_data = cpl_image_get_data_double(order_wave_B_img);
        
        order_master_B_img = cpl_image_extract(master_spectrum_B, 1, order+1, nx_master, order+1);
        order_wavelength_B_img = cpl_image_extract(wavelength_B, 1, order+1, nx_master, order+1);
        double *order_master_B_data = cpl_image_get_data_double(order_master_B_img);
        double *order_wavelength_B_data = cpl_image_get_data_double(order_wavelength_B_img);

        while (OH_lines_table[OH_index].order == order) {
            
            //espdr_msg("Computing the integrated flux for line %d", OH_index);
            // if line > 6 * MAD, compute the integrated flux
            wave_index = 0;
            while ((wave_index < nx_wave_matrix) && (order_wave_B_data[wave_index] < OH_lines_table[OH_index].wave_start_B)) {
                wave_index++;
            }
            if (fabs(OH_lines_table[OH_index].wave_start_B - order_wave_B_data[wave_index]) < fabs(OH_lines_table[OH_index].wave_start_B - order_wave_B_data[wave_index+1])) {
                wave_index_start = wave_index;
            } else {
                wave_index_start = wave_index+1;
            }
            double wavelength_end = OH_lines_table[OH_index].wave_start_B+OH_lines_table[OH_index].region_size_B;
            while ((wave_index < nx_wave_matrix) && (order_wave_B_data[wave_index] < wavelength_end)) {
                wave_index++;
            }
            if (fabs(wavelength_end - order_wave_B_data[wave_index]) <
                fabs(wavelength_end - order_wave_B_data[wave_index+1])) {
                wave_index_end = wave_index;
            } else {
                wave_index_end = wave_index+1;
            }
            wave_index_end = wave_index_end + 1; // to be aligned with Avidaans' code
            
            master_wave_index = 0;
            while ((master_wave_index < nx_master) && (order_wavelength_B_data[master_wave_index] < OH_lines_table[OH_index].wave_start_B)) {
                master_wave_index++;
            }
            if (fabs(OH_lines_table[OH_index].wave_start_B - order_wave_B_data[master_wave_index]) < fabs(OH_lines_table[OH_index].wave_start_B - order_wave_B_data[master_wave_index+1])) {
                master_wave_index_start = master_wave_index;
            } else {
                master_wave_index_start = master_wave_index+1;
            }
            while ((master_wave_index < nx_master) && (order_wavelength_B_data[master_wave_index] < wavelength_end)) {
                master_wave_index++;
            }
            if (fabs(wavelength_end - order_wave_B_data[master_wave_index]) < fabs(wavelength_end - order_wave_B_data[master_wave_index+1])) {
                master_wave_index_end = master_wave_index;
            } else {
                master_wave_index_end = master_wave_index+1;
            }
            master_wave_index_end = master_wave_index_end + 1; // to be aligned with Avidaans' code
            
            line_present_B[OH_index] = 0;
            int_flux_s2d_B[OH_index] = 0.0;
            int_flux_master_B[OH_index] = 0.0;
            int positive_points_nb = 0;
            
            if ((master_wave_index_start > 1) && (master_wave_index_end < nx_master-2)) {
                for (int i = master_wave_index_start; i < master_wave_index_end; i++) {
                    int_flux_master_B[OH_index] = int_flux_master_B[OH_index] + order_master_B_data[i];
                }
            }
            
            if ((wave_index_start > 1) && (wave_index_end < nx_s2d-2)) {
                for (int i = wave_index_start; i < wave_index_end; i++) {
                    if (order_flux_mad_sub_B_data[i] > 0.0) {
                        positive_points_nb++;
                    }
                }
                if (positive_points_nb > 1) {
                    line_present_B[OH_index] = 1;
                }
                if (line_present_B[OH_index] == 1) {
                    for (int i = wave_index_start; i < wave_index_end; i++) {
                            int_flux_s2d_B[OH_index] = int_flux_s2d_B[OH_index] + order_flux_B_data[i];
                    }
                }
            }
            
            //if ((wave_index_start < 2) || (wave_index_end > (nx_s2d-2))) {
            //    int_flux_s2d_B[OH_index] = 0.0;
            //}

            // Compute the integrated flux ratio
            OH_ratio[OH_index] = 0.0;
            if ((OH_lines_table[OH_index].both_fibres == 1) && (line_present_B[OH_index] == 1) &&
                OH_lines_table[OH_index].integrated_flux != 0.0) {
                OH_ratio[OH_index] = fabs(int_flux_s2d_B[OH_index]) / OH_lines_table[OH_index].integrated_flux;
                if (OH_ratio[OH_index] < 1e-4) {
                    espdr_msg("Line %d: very small ratio: %e --> 0.0", OH_index, OH_ratio[OH_index]);
                    OH_ratio[OH_index] = 0.0;
                }
            }
            
            if ((OH_index == -1) || (OH_index == -1)) {
                espdr_msg("line %d: line present B: %d (both_fibres: %d), ll_start: %f, region: %f, int_flux_s2d_B: %f, int_flux_master_B: %f, int_flux_master_B_computed: %f, ratio: %f, index start: %d, index end: %d",
                          OH_index, line_present_B[OH_index], OH_lines_table[OH_index].both_fibres,
                          OH_lines_table[OH_index].wave_start_B, OH_lines_table[OH_index].region_size_B,
                          int_flux_s2d_B[OH_index], OH_lines_table[OH_index].integrated_flux,
                          int_flux_master_B[OH_index], OH_ratio[OH_index], wave_index_start, wave_index_end);
            }
            OH_index++;
        }
        
        cpl_image_delete(order_flux_B_img);
        cpl_image_delete(order_flux_mad_sub_B_img);
        cpl_image_delete(order_wave_B_img);
        cpl_image_delete(order_master_B_img);
        cpl_image_delete(order_wavelength_B_img);
    }
    
    // Compute the ratio
    
    // Compute the median ratio
    int ratio_nb = 0;
    for (int i = 0; i < OH_lines_nb; i++) {
        if ((OH_lines_table[i].both_fibres == 1) && (line_present_B[i] == 1) && (OH_ratio[i] != 0.0)) {
            ratio_nb++;
            //espdr_msg("line %d: ll = %f, ratio = %f", i, OH_lines_table[i].wave_start_B, OH_ratio[i]);
            //printf("%d,%f,%f\n", i, OH_lines_table[i].wave_start_B, OH_ratio[i]);
        }
    }
    double median_ratio = 0.0;
    double flux_ratio_RMS = 0.0;
    double flux_ratio_MAD = 0.0;
    int ratio_index = 0;
    if (ratio_nb > 0) {
        cpl_image *median_ratio_img = cpl_image_new(ratio_nb, 1, CPL_TYPE_DOUBLE);
        for (int i = 0; i < OH_lines_nb; i++) {
            if ((OH_lines_table[i].both_fibres == 1) && (line_present_B[i] == 1) && (OH_ratio[i] != 0.0)) {
                my_error = cpl_image_set(median_ratio_img, ratio_index+1, 1, OH_ratio[i]);
                ratio_index++;
            }
            //if (OH_ratio[i] != 0.0) {
            //    espdr_msg("order %d line %d ratio %f", OH_lines_table[i].order, i, OH_ratio[i]);
            //}
        }
        
        median_ratio = cpl_image_get_mad(median_ratio_img, &flux_ratio_MAD);
        cpl_image *median_ratio_img_norm = cpl_image_subtract_scalar_create(median_ratio_img, median_ratio);
        flux_ratio_RMS = sqrt(cpl_image_get_sqflux(median_ratio_img_norm) / ratio_nb);
        cpl_image_delete(median_ratio_img);
        cpl_image_delete(median_ratio_img_norm);
    } else {
        espdr_msg_warning("No correct ratio to compute the median ratio, median ratio is set to 0.0");
    }
    
    espdr_msg("Median ratio: %f (ratio_index: %d), RMS: %f, MAD: %f",
              median_ratio, ratio_index, flux_ratio_RMS, flux_ratio_MAD);
    
    for (int i = 0; i < OH_lines_nb; i++) {
        if (OH_ratio[i] == 0.0) {
            OH_ratio[i] = median_ratio;
        }
    }
    
    for (int i = 0; i < OH_lines_nb; i++) {
        //if ((OH_ratio[i] < 0.25 * median_ratio) || (OH_ratio[i] > 1.75 * median_ratio)) {
        if ((OH_ratio[i] < 0.33 * median_ratio) || (OH_ratio[i] > 1.67 * median_ratio)) {
            espdr_msg("Line %d (order %d), orig ratio: %f, lower limit: %f, upper limit; %f, final ratio: %f",
                      i, OH_lines_table[i].order, OH_ratio[i], 0.33 * median_ratio, 1.67 * median_ratio, median_ratio);
            OH_ratio[i] = median_ratio;
        }
        //espdr_msg("Line %d (order %d) - ratio: %f", i, OH_lines_table[i].order, OH_ratio[i]);
    }
    
    flux_A_OH_corrected = cpl_image_new(nx_s2d, ny_s2d, CPL_TYPE_DOUBLE);
    error_A_OH_corrected = cpl_image_new(nx_s2d, ny_s2d, CPL_TYPE_DOUBLE);
    OH_index = 0;
    for (int order = 0; order < ny_s2d; order++) {
        //espdr_msg("Correcting OH lines from order %d", order);
        
        order_flux_A_img = cpl_image_extract(s2d_flux_A, 1, order+1, nx_s2d, order+1);
        order_error_A_img = cpl_image_extract(s2d_error_A, 1, order+1, nx_s2d, order+1);
        order_wave_A_img = cpl_image_extract(wave_matrix_A, 1, order+1, nx_wave_matrix, order+1);
        double *order_flux_A_data = cpl_image_get_data_double(order_flux_A_img);
        double *order_error_A_data = cpl_image_get_data_double(order_error_A_img);
        double *order_wave_A_data = cpl_image_get_data_double(order_wave_A_img);
        order_OH_flux_A_img = cpl_image_duplicate(order_flux_A_img);
        double *order_OH_flux_A_data = cpl_image_get_data_double(order_OH_flux_A_img);
        order_OH_error_A_img = cpl_image_duplicate(order_error_A_img);
        double *order_OH_error_A_data = cpl_image_get_data_double(order_OH_error_A_img);
        
        order_master_A_img = cpl_image_extract(flux_base_A_img, 1, order+1, nx_master, order+1);
        order_master_err_A_img = cpl_image_extract(master_error_A, 1, order+1, nx_master, order+1);
        order_wavelength_A_img = cpl_image_extract(wavelength_A, 1, order+1, nx_master, order+1);
        double *order_master_A_data = cpl_image_get_data_double(order_master_A_img);
        double *order_master_err_A_data = cpl_image_get_data_double(order_master_err_A_img);
        double *order_wavelength_A_data = cpl_image_get_data_double(order_wavelength_A_img);

        while (OH_lines_table[OH_index].order == order) {
            //espdr_msg("Correcting OH line %d", OH_index);
            wave_index = 0;
            while ((wave_index < nx_wave_matrix) && (order_wave_A_data[wave_index] < OH_lines_table[OH_index].wave_start_A)) {
                wave_index++;
            }
            if (fabs(OH_lines_table[OH_index].wave_start_A - order_wave_A_data[wave_index]) <
                fabs(OH_lines_table[OH_index].wave_start_A - order_wave_A_data[wave_index+1])) {
                wave_index_start = wave_index;
            } else {
                wave_index_start = wave_index+1;
            }
            
            double wavelength_end = OH_lines_table[OH_index].wave_start_A+OH_lines_table[OH_index].region_size_A;
            while ((wave_index < nx_wave_matrix) && (order_wave_A_data[wave_index] < wavelength_end)) {
                wave_index++;
            }
            if (fabs(wavelength_end - order_wave_A_data[wave_index]) <
                fabs(wavelength_end - order_wave_A_data[wave_index+1])) {
                wave_index_end = wave_index;
            } else {
                wave_index_end = wave_index+1;
            }
            wave_index_end = wave_index_end + 1; // to be aligned with Avidaans' code

            master_wave_index = 0;
            while ((master_wave_index < nx_master) && (order_wavelength_A_data[master_wave_index] < OH_lines_table[OH_index].wave_start_A)) {
                master_wave_index++;
            }
            if (fabs(OH_lines_table[OH_index].wave_start_A - order_wavelength_A_data[master_wave_index]) <
                fabs(OH_lines_table[OH_index].wave_start_A - order_wavelength_A_data[master_wave_index+1])) {
                master_wave_index_start = master_wave_index;
            } else {
                master_wave_index_start = master_wave_index+1;
            }
            while ((master_wave_index < nx_master) && (order_wavelength_A_data[master_wave_index] < wavelength_end)) {
                master_wave_index++;
            }
            if (fabs(wavelength_end - order_wavelength_A_data[master_wave_index]) <
                fabs(wavelength_end - order_wavelength_A_data[master_wave_index+1])) {
                master_wave_index_end = master_wave_index;
            } else {
                master_wave_index_end = master_wave_index+1;
            }
            master_wave_index_end = master_wave_index_end + 1; // to be aligned with Avidaans' code
            
            // Interpolation between target_A and master_A
            
            double *x1 = (double *)cpl_calloc(master_wave_index_end - master_wave_index_start + 4, sizeof(double));
            double *x2 = (double *)cpl_calloc(wave_index_end - wave_index_start + 4, sizeof(double));
            double *y1 = (double *)cpl_calloc(master_wave_index_end - master_wave_index_start + 4, sizeof(double));
            double *y2 = (double *)cpl_calloc(wave_index_end - wave_index_start + 4, sizeof(double));
            double *err1 = (double *)cpl_calloc(master_wave_index_end - master_wave_index_start + 4, sizeof(double));
            double *err2 = (double *)cpl_calloc(wave_index_end - wave_index_start + 4, sizeof(double));

            if ((master_wave_index_start > 2) && (master_wave_index_end < nx_master-2) && (wave_index_start > 2) && (wave_index_end < nx_s2d-2)) {
                //espdr_msg("Interpolation for line %d", OH_index);
                
                for (int i = 0; i < master_wave_index_end - master_wave_index_start + 4; i++) {
                    x1[i] = order_wavelength_A_data[master_wave_index_start-2+i];
                    y1[i] = order_master_A_data[master_wave_index_start-2+i];
                    err1[i] = order_master_err_A_data[master_wave_index_start-2+i];
                }
                for (int i = 0; i < wave_index_end - wave_index_start + 4; i++) {
                    x2[i] = order_wave_A_data[wave_index_start-2+i];
                }
                
                double x_min = ESPDR_MIN(x1[0], x2[0]);
                double x_max = ESPDR_MAX(x1[master_wave_index_end-master_wave_index_start+3], x2[wave_index_end-wave_index_start+3]);
                
                my_error = espdr_spline_bounded_interpolation(x1, y1, master_wave_index_end-master_wave_index_start+4,
                                                              x2, y2, wave_index_end-wave_index_start+4,
                                                              x_min, x_max);
                
                my_error = espdr_spline_bounded_interpolation(x1, err1, master_wave_index_end-master_wave_index_start+4,
                                                              x2, err2, wave_index_end-wave_index_start+4,
                                                              x_min, x_max);
            } else {
                espdr_msg("Interpolation for line %d can't be done, line is too close to the border", OH_index);
            }
            
            for (int i = wave_index_start; i < wave_index_end; i++) {
                order_OH_flux_A_data[i] = order_flux_A_data[i] - y2[i+2-wave_index_start]*OH_ratio[OH_index];
                order_OH_error_A_data[i] = sqrt(order_error_A_data[i]*order_error_A_data[i] + OH_ratio[OH_index]*OH_ratio[OH_index]*err2[i+2-wave_index_start]*err2[i+2-wave_index_start]);
            }
            
            //if ((OH_index == 181) || (OH_index == 335) || (OH_index == 378) || (OH_index == 412) || (OH_index == 431) || (OH_index == 451) || (OH_index == 463)) {
            //    for (int i = wave_index_start; i < wave_index_end; i++) {
            //        espdr_msg("line %d (order %d): OH_corr[%d] = %f, flux_A[%d] = %f, y2[%d] = %f, ratio[%d] = %f, y2*ratio = %f",
            //                  OH_index, order, i, order_OH_flux_A_data[i], i, order_flux_A_data[i],
            //                  i+2-wave_index_start, y2[i+2-wave_index_start], OH_index, OH_ratio[OH_index],
            //                  y2[i+2-wave_index_start] * OH_ratio[OH_index]);
            //    }
            //}
            
            cpl_free(x1);
            cpl_free(y1);
            cpl_free(x2);
            cpl_free(y2);
            
            OH_index++;
        }
        
        my_error = cpl_image_copy(flux_A_OH_corrected, order_OH_flux_A_img, 1, order+1);
        my_error = cpl_image_copy(error_A_OH_corrected, order_OH_error_A_img, 1, order+1);
        
        cpl_image_delete(order_flux_A_img);
        cpl_image_delete(order_wave_A_img);
        cpl_image_delete(order_OH_flux_A_img);
        cpl_image_delete(order_OH_error_A_img);
    }

    //my_error = cpl_image_save(flux_A_OH_corrected, "NIRPS_flux_A_OH_corrected.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    //my_error = cpl_image_save(error_A_OH_corrected, "NIRPS_error_A_OH_corrected.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    
    *OH_corr_flux_RE = cpl_image_duplicate(flux_A_OH_corrected);
    *OH_corr_error_RE = cpl_image_duplicate(error_A_OH_corrected);
    *OH_corr_qual_RE = cpl_image_duplicate(s2d_qual_A);
    
    cpl_image_delete(flux_A_OH_corrected);
    cpl_image_delete(error_A_OH_corrected);
    
    espdr_msg("Computing the QC keywords for OH correction");
    my_error = espdr_OH_corr_QC(inst_config, qc_kws, Texp, airmass_mean, ratio_nb,
                                median_ratio, flux_ratio_RMS, flux_ratio_MAD, OH_keywords_RE);
    espdr_msg("OH QC KWs done");
    
    return (cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Compute the running median or MAD
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_compute_running_median_mad(cpl_image *input_img,
                                                double *input_idx,
                                                int input_size,
                                                int half_window_size,
                                                int mad_flag, // 0 = median, 1 = MAD
                                                double *output) {

    int left_idx, right_idx;
    double mad_median;
    
    for (int i = 0; i < input_size; i++) {
//#if 0
        // Running median taking +/- half_window_size of km/s, indices in input_idx
        if (fabs(input_idx[i] - input_idx[0]) < half_window_size) {
            left_idx = 0;
            right_idx = i+1;
            while ((right_idx < input_size) && (fabs(input_idx[right_idx] - input_idx[i]) < half_window_size)) {
                right_idx++;
            }
        } else {
            if (fabs(input_idx[i] - input_idx[input_size-1]) < half_window_size) {
                left_idx = i-1;
                while ((left_idx >= 0) && (fabs(input_idx[i] - input_idx[left_idx]) < half_window_size)) {
                    left_idx--;
                }
                right_idx = input_size-1;
            } else {
                left_idx = i-1;
                while ((left_idx >= 0) && (fabs(input_idx[i] - input_idx[left_idx]) < half_window_size)) {
                    left_idx--;
                }
                right_idx = i+1;
                while ((right_idx < input_size) && (fabs(input_idx[right_idx] - input_idx[i]) < half_window_size)) {
                    right_idx++;
                }
            }
        }
//#endif
#if 0
        // Running median taking +/- half_window_size of pixels, indices in input_img
        if (i < half_window_size) {
            left_idx = 0;
        } else {
            left_idx = i - half_window_size;
        }
        if (i >= (input_size - half_window_size)) {
            right_idx = input_size-1;
        } else {
            right_idx = i + half_window_size;
        }
#endif
        
        //espdr_msg("left idx: %d, right idx: %d", left_idx, right_idx);
        
        int pxl_rjct, good_pxl_nb = 0;
        double pxl_value;
        for (int j = left_idx+1; j < right_idx+1; j++) {
            pxl_value = cpl_image_get(input_img, j, 1, &pxl_rjct);
            if (pxl_rjct == 0) {
                good_pxl_nb++;
            }
        }
        
        if (good_pxl_nb == 0) {
            output[i] = 0.0;
        } else {
            output[i] = cpl_image_get_mad_window(input_img, left_idx+1, 1, right_idx+1, 1, &mad_median);
            if (mad_flag == 1) {
                output[i] = mad_median;
            }
        }
        espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE, cpl_error_get_code(),
                     "median/mad computation failed for pxl %d: %s", i,
                     cpl_error_get_message_default(cpl_error_get_code()));
        
    }
    
    return (cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief Init the OH_lines_table structure
 @param lines_table table to be initialized
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_OH_lines_table_init(espdr_OH_line *lines_table,
                                         int lines_nb) {

    for (int i = 0; i < lines_nb; i++) {
        lines_table[i].order = 0;
        lines_table[i].wave_start_A = 0.0;
        lines_table[i].region_size_A = 0.0;
        lines_table[i].wave_start_B = 0.0;
        lines_table[i].region_size_B = 0.0;
        lines_table[i].both_fibres = 1;
        lines_table[i].integrated_flux = 0.0;
        strcpy(lines_table[i].comment, "NONE");
    }
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Read OH_LINES_TABLE from fits table
 @param         OH_line_table FITS table with OH lines regions
 @param[out]    OH_lines_table     read OH lines table
 @param[out]    lines_nb_RE     numebr of read lines
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_read_OH_LINES_TABLE(cpl_frame *OH_lines_regions_frame,
                                         espdr_OH_line *OH_lines_table,
                                         int *lines_nb_RE) {
    
    espdr_msg("Reading OH lines regions");
    
    cpl_table *OH_lines_regions = cpl_table_load(cpl_frame_get_filename(OH_lines_regions_frame), 1, 0);
    espdr_ensure(OH_lines_regions == NULL, CPL_ERROR_NULL_INPUT,
                 "OH lines table is NULL, exiting");
    
    int table_length = cpl_table_get_nrow(OH_lines_regions);
    //cpl_table_dump(OH_lines_regions, 0, table_length, NULL);
    
    int lines_nb = 0;
    int *col_order = cpl_table_get_data_int(OH_lines_regions, COL_NAME_OH_ORDER);
    double *col_wave_start_A = cpl_table_get_data_double(OH_lines_regions, COL_NAME_WAVE_START_A);
    double *col_region_size_A = cpl_table_get_data_double(OH_lines_regions, COL_NAME_REGION_SIZE_A);
    double *col_wave_start_B = cpl_table_get_data_double(OH_lines_regions, COL_NAME_WAVE_START_B);
    double *col_region_size_B = cpl_table_get_data_double(OH_lines_regions, COL_NAME_REGION_SIZE_B);
    int *col_both_fibres = cpl_table_get_data_int(OH_lines_regions, COL_NAME_BOTH_FIBRES);
    double *col_integrated_flux = cpl_table_get_data_double(OH_lines_regions, COL_NAME_INTEGRATED_FLUX);
    
    //espdr_msg("col_order type: %s", cpl_type_get_name(cpl_table_get_column_type(OH_lines_regions, COL_NAME_OH_ORDER)));
    //espdr_msg("col_wave_start_A type: %s", cpl_type_get_name(cpl_table_get_column_type(OH_lines_regions, COL_NAME_WAVE_START_A)));
    //espdr_msg("col_region_size_A type: %s", cpl_type_get_name(cpl_table_get_column_type(OH_lines_regions, COL_NAME_REGION_SIZE_A)));
    //espdr_msg("col_wave_start_B type: %s", cpl_type_get_name(cpl_table_get_column_type(OH_lines_regions, COL_NAME_WAVE_START_B)));
    //espdr_msg("col_region_size_B type: %s", cpl_type_get_name(cpl_table_get_column_type(OH_lines_regions, COL_NAME_REGION_SIZE_B)));
    //espdr_msg("col_both_fibres type: %s", cpl_type_get_name(cpl_table_get_column_type(OH_lines_regions, COL_NAME_BOTH_FIBRES)));
    //espdr_msg("col_integrated_flux type: %s", cpl_type_get_name(cpl_table_get_column_type(OH_lines_regions, COL_NAME_INTEGRATED_FLUX)));

    
    //espdr_msg("OH lines table:");
    //printf("no\torder\twave_start\tregion_size\tboth_fibres\tintegrated_flux\tcomment\n");
    for (int i = 0; i < table_length; i++) {
        OH_lines_table[i].order = col_order[i];
        OH_lines_table[i].wave_start_A = col_wave_start_A[i];
        OH_lines_table[i].region_size_A = col_region_size_A[i];
        OH_lines_table[i].wave_start_B = col_wave_start_B[i];
        OH_lines_table[i].region_size_B = col_region_size_B[i];
        OH_lines_table[i].both_fibres = col_both_fibres[i];
        OH_lines_table[i].integrated_flux = col_integrated_flux[i];
        strcpy(OH_lines_table[i].comment, "NONE");

        //printf("%d\t%d\t%f\t%f\t%f\t%f\t%d\t%f\t%s\n", i, OH_lines_table[i].order,
        //       OH_lines_table[i].wave_start_A, OH_lines_table[i].region_size_A,
        //       OH_lines_table[i].wave_start_B, OH_lines_table[i].region_size_B,
        //       OH_lines_table[i].both_fibres, OH_lines_table[i].integrated_flux,
        //       OH_lines_table[i].comment);
        
        lines_nb++;
    }
    
    *lines_nb_RE = lines_nb;
    
    return(cpl_error_get_code());
}

    
/*----------------------------------------------------------------------------*/
/**
 @brief Read OH_LINES_TABLE from fits table
 @param         OH_line_table FITS table with OH lines regions
 @param[out]    OH_lines_table     read OH lines table
 @param[out]    lines_nb_RE     numebr of read lines
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_read_OH_LINES_TABLE_CSV(cpl_frame *OH_lines_regions_frame,
                                             espdr_OH_line *OH_lines_table,
                                             int *lines_nb_RE) {

    const char *filename = NULL;
    FILE *OHptr = NULL;
    char line[COMMENT_LENGTH];
    int lines_nb = 0, col0;
    espdr_OH_line read_line;
    
    espdr_ensure(OH_lines_regions_frame == NULL, CPL_ERROR_NULL_INPUT,
                 "Input OH LINES TABLE frame is NULL");
    espdr_ensure(OH_lines_table == NULL, CPL_ERROR_NULL_INPUT,
                 "OH LINES TABLE structure is NULL");
    
    filename = cpl_frame_get_filename(OH_lines_regions_frame);
    
    espdr_msg("Reading CSV OH lines from %s", filename);
    
    OHptr = fopen(filename, "r");
    espdr_ensure(OHptr == NULL, CPL_ERROR_FILE_IO,
                 "The file %s can't be read", filename);
    
    // Get rid of the header
    fgets(line, sizeof(line), OHptr);
    
    while (fscanf(OHptr,
                  "%d,%d,%lf,%lf,%lf,%lf,%d,%lf",
                  &col0,
                  &read_line.order,
                  &read_line.wave_start_A,
                  &read_line.region_size_A,
                  &read_line.wave_start_B,
                  &read_line.region_size_B,
                  &read_line.both_fibres,
                  &read_line.integrated_flux) != EOF) {
        
        OH_lines_table[lines_nb].order = read_line.order;
        OH_lines_table[lines_nb].wave_start_A = read_line.wave_start_A;
        OH_lines_table[lines_nb].region_size_A = read_line.region_size_A;
        OH_lines_table[lines_nb].wave_start_B = read_line.wave_start_B;
        OH_lines_table[lines_nb].region_size_B = read_line.region_size_B;
        OH_lines_table[lines_nb].both_fibres = read_line.both_fibres;
        OH_lines_table[lines_nb].integrated_flux = read_line.integrated_flux;
        //strcpy(OH_lines_table[lines_nb].comment, read_line.comment);
        
        //printf("%d: %d, %f, %f, %f, %f, %d, %f, %s\n",
        //       lines_nb, OH_lines_table[lines_nb].order,
        //       OH_lines_table[lines_nb].wave_start_A,
        //       OH_lines_table[lines_nb].region_size_A,
        //       OH_lines_table[lines_nb].wave_start_B,
        //       OH_lines_table[lines_nb].region_size_B,
        //       OH_lines_table[lines_nb].both_fibres,
        //       OH_lines_table[lines_nb].integrated_flux,
        //       OH_lines_table[lines_nb].comment);
        
        lines_nb++;
    }
    
    espdr_msg("%d lines read from %s", lines_nb, filename);
    
    *lines_nb_RE = lines_nb;
    
    return(cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief Save OH_LINES_TABLE in fits table format
 @param         OH_line_table FITS table with OH lines regions
 @param[out]    OH_lines_table     read OH lines table
 @param[out]    lines_nb_RE     numebr of read lines
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_save_OH_lines_table(espdr_OH_line *OH_lines_table,
                                         int OH_lines_nb) {

    
    /* Save the OH_LINES_TABLE FITS */
    cpl_propertylist *OH_KWs = cpl_propertylist_new();
    cpl_table *OH_lines_table_FITS = cpl_table_new(OH_lines_nb);

    cpl_table_new_column(OH_lines_table_FITS, COL_NAME_OH_ORDER, CPL_TYPE_INT);
    cpl_table_new_column(OH_lines_table_FITS, COL_NAME_WAVE_START_A, CPL_TYPE_DOUBLE);
    cpl_table_new_column(OH_lines_table_FITS, COL_NAME_REGION_SIZE_A, CPL_TYPE_DOUBLE);
    cpl_table_new_column(OH_lines_table_FITS, COL_NAME_WAVE_START_B, CPL_TYPE_DOUBLE);
    cpl_table_new_column(OH_lines_table_FITS, COL_NAME_REGION_SIZE_B, CPL_TYPE_DOUBLE);
    cpl_table_new_column(OH_lines_table_FITS, COL_NAME_BOTH_FIBRES, CPL_TYPE_INT);
    cpl_table_new_column(OH_lines_table_FITS, COL_NAME_INTEGRATED_FLUX, CPL_TYPE_DOUBLE);

    for (int i = 0; i < OH_lines_nb; i++) {
    	cpl_table_set_int(OH_lines_table_FITS, COL_NAME_OH_ORDER, i, OH_lines_table[i].order);
    	cpl_table_set_double(OH_lines_table_FITS, COL_NAME_WAVE_START_A, i, OH_lines_table[i].wave_start_A);
    	cpl_table_set_double(OH_lines_table_FITS, COL_NAME_REGION_SIZE_A, i, OH_lines_table[i].region_size_A);
    	cpl_table_set_double(OH_lines_table_FITS, COL_NAME_WAVE_START_B, i, OH_lines_table[i].wave_start_B);
    	cpl_table_set_double(OH_lines_table_FITS, COL_NAME_REGION_SIZE_B, i, OH_lines_table[i].region_size_B);
    	cpl_table_set_int(OH_lines_table_FITS, COL_NAME_BOTH_FIBRES, i, OH_lines_table[i].both_fibres);
    	cpl_table_set_double(OH_lines_table_FITS, COL_NAME_INTEGRATED_FLUX, i, OH_lines_table[i].integrated_flux);
    }

    /* Save the PRO.CATG */
    cpl_propertylist_append_string(OH_KWs, PRO_CATG_KW, ESPDR_OH_LINES_REGIONS);
    cpl_propertylist_append_string(OH_KWs, "ESO PRO TYPE", "STATIC");

    cpl_propertylist_append_string(OH_KWs, "INSTRUME", "NIRPS");
    cpl_propertylist_append_string(OH_KWs, "ESO INS MODE", "HA");

    cpl_table_save(OH_lines_table_FITS, OH_KWs, NULL,
			"NIRPS_HA_OH_LINES_TABLE.fits", CPL_IO_CREATE);
    
    return(cpl_error_get_code());
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Check the OH correction QC
 @param    CCD_geom         CCD geometry
 @param    exposure_time    exposure time
 @param    inst_config      instrument configuration structure
 @param    qc_kws           structure holdin quality control, keys
 @param    keywords_RE      fits header to hold QC
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_OH_corr_QC(espdr_inst_config *inst_config,
                                espdr_qc_keywords *qc_kws,
                                double Texp,
                                double airmass_mean,
                                int OH_lines_nb_used,
                                double median_ratio,
                                double ratio_RMS,
                                double ratio_MAD,
                                cpl_propertylist **OH_keywords_RE) {
    
    int lines_nb_check = 1;
    int median_ratio_check = 1;
    int ratio_mad_check = 1;
    double norm_median_ratio = median_ratio * (SKY_MASTER_TEXP / Texp) / airmass_mean;
    double norm_ratio_RMS = ratio_RMS * (SKY_MASTER_TEXP / Texp) / airmass_mean;
    double norm_ratio_MAD = ratio_MAD * (SKY_MASTER_TEXP / Texp) / airmass_mean;
    double norm_lines_nb = (double)(OH_lines_nb_used) * (SKY_MASTER_TEXP / Texp) / airmass_mean;
    int oh_global_check = 1;
    
    if (OH_lines_nb_used < inst_config->oh_corr_lines_nb_limit) {
        lines_nb_check = 0;
        oh_global_check = 0;
    }
    if ((norm_median_ratio < inst_config->oh_corr_ratio_lower_limit) ||
        (norm_median_ratio > inst_config->oh_corr_ratio_upper_limit)) {
        median_ratio_check = 0;
        oh_global_check = 0;
    }
    if ((norm_ratio_MAD > inst_config->oh_corr_ratio_mad_limit) || (ratio_MAD == 0.0)) {
        ratio_mad_check = 0;
        oh_global_check = 0;
    }
    
    espdr_keyword_add_int(qc_kws->qc_sci_red_oh_corr_oh_lines_nb_kw,
                                     OH_lines_nb_used, "OH lines NB for ratio computation",
                                     OH_keywords_RE);
    espdr_keyword_add_int(qc_kws->qc_sci_red_oh_corr_oh_lines_nb_norm_kw,
                                     norm_lines_nb, "OH lines normalized NB for ratio computation",
                                     OH_keywords_RE);
    espdr_keyword_add_int(qc_kws->qc_sci_red_oh_corr_oh_lines_check_kw,
                                     lines_nb_check, "OH lines NB check",
                                     OH_keywords_RE);
    espdr_keyword_add_double(qc_kws->qc_sci_red_oh_corr_median_ratio_kw,
                                        median_ratio, "Median ratio",
                                        OH_keywords_RE);
    espdr_keyword_add_double(qc_kws->qc_sci_red_oh_corr_median_ratio_norm_kw,
                                        norm_median_ratio, "Median ratio normalized",
                                        OH_keywords_RE);
    espdr_keyword_add_int(qc_kws->qc_sci_red_oh_corr_median_ratio_check_kw,
                                     median_ratio_check, "Median ratio check",
                                     OH_keywords_RE);
    espdr_keyword_add_double(qc_kws->qc_sci_red_oh_corr_ratio_rms_kw,
                                        ratio_RMS, "Flux ratio RMS",
                                        OH_keywords_RE);
    espdr_keyword_add_double(qc_kws->qc_sci_red_oh_corr_ratio_rms_norm_kw,
                                        norm_ratio_RMS, "Flux ratio RMS normalized",
                                        OH_keywords_RE);
    espdr_keyword_add_double(qc_kws->qc_sci_red_oh_corr_ratio_mad_kw,
                                        ratio_MAD, "Flux ratio MAD",
                                        OH_keywords_RE);
    espdr_keyword_add_double(qc_kws->qc_sci_red_oh_corr_ratio_mad_norm_kw,
                                        norm_ratio_MAD, "Flux ratio MAD normalized",
                                        OH_keywords_RE);
    espdr_keyword_add_int(qc_kws->qc_sci_red_oh_corr_ratio_mad_check_kw,
                                     ratio_mad_check, "Flux ratio MAD check",
                                     OH_keywords_RE);
    espdr_keyword_add_int(qc_kws->qc_sci_red_oh_corr_check_kw,
                                     oh_global_check, "OH correction global check",
                                     OH_keywords_RE);
    
    return(cpl_error_get_code());
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Propagate the OH correction QCs into the global QC
 @param    inst_config      instrument configuration structure
 @param    qc_kws           structure holdin quality control, keys
 @param    OH_keywords      keywords with OH correction QC KWs
 @param    keywords_RE      fits header to hold QC
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_propagate_OH_QC(espdr_qc_keywords *qc_kws,
                                     cpl_propertylist *OH_QC_keywords,
                                     cpl_propertylist **keywords_RE) {
    

    int global_OH_QC = cpl_propertylist_get_int(OH_QC_keywords,
                                                qc_kws->qc_sci_red_oh_corr_check_kw);
    int global_QC = cpl_propertylist_get_int(*keywords_RE,
                                             qc_kws->qc_sci_red_check_kw);
    
    if ((global_QC == 1) && (global_OH_QC == 0)) {
        cpl_propertylist_update_int(*keywords_RE, qc_kws->qc_sci_red_check_kw, global_OH_QC);
    }
    
    return(cpl_error_get_code());
}
