/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: asegovia $
 * $Date: 2014-10-28 15:13:41 $
 * $Revision:     $
 * $Name: not supported by cvs2svn $
 */


#include <espdr_background.h>
#include <espdr_overscan.h>

#include <gsl/gsl_errno.h>
#include <gsl/gsl_spline.h>
#include <gsl/gsl_histogram.h>
#include <gsl/gsl_multifit.h>
#include <gsl/gsl_sort.h>

/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/
/**
 @brief  computes the background ilumination for science frames
 @param  input_image image under treatment
 @param  ccdgain CCD gain value
 @param  ccdsigdet CCD RON value
 @param  image_background_measured_RE resultant background image
 @return CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_measure_background_science(
		const cpl_image *input_image,
		const cpl_image *orderdef,
		const double ccdgain,
		const double ccdsigdet,
		const int bkgr_grid_size_x,
		const int bkgr_grid_size_y,
		cpl_image **image_background_measured_RE,
		cpl_image **image_background_substracted_RE) {

	cpl_error_code my_error = CPL_ERROR_NONE;

	int i, j, ii, jj, size_x, size_y, n1, n2, nbin,
	index, hwhm, dim[2], *xc, *yc;

	double min, max, value, pvalue, chisq, mode, *xcd,
	*ycd, *ctr, *limit, *bkgr, *pflux, *aa_vector, *x2, *y2;

	cpl_image *grid_input_image;
	double *grid_input_image_data;

	cpl_image *grid_back_image;
	double *grid_back_image_data;

	gsl_histogram *h;
	gsl_matrix *X, *covar;
	gsl_vector *range, *bin, *p;
	gsl_multifit_linear_workspace *work;

	n1 = cpl_image_get_size_x(input_image);
	n2 = cpl_image_get_size_y(input_image);
	min = cpl_image_get_min(input_image);
	max = cpl_image_get_max(input_image);

	size_x = bkgr_grid_size_x;
	size_y = bkgr_grid_size_y;

	dim[0] = n1/2/size_x;
	dim[1] = n2/2/size_y;

	xc = (int *) cpl_calloc((dim[0]),sizeof(int));
	yc = (int *) cpl_calloc((dim[1]),sizeof(int));
	xcd = (double *) cpl_calloc(dim[0],sizeof(double));
	ycd = (double *) cpl_calloc(dim[1],sizeof(double));

	for (i = 0; i < dim[0]; i++) {
        xc[i] = size_x + 2*i*size_x;
        xcd[i] = size_x + 2*i*size_x;
    }

	for (j = 0; j < dim[1]; j++) {
        yc[j] = size_y + 2*j*size_y;
        ycd[j] = size_y + 2*j*size_y;
    }

	aa_vector = (double *) cpl_calloc((n1*n2),sizeof(double));

	double **aa = (double **) cpl_calloc(n1,sizeof(double *));
	for (i = 0; i < n1; i++) {
		aa[i] = (double *)cpl_calloc((n2),sizeof(double));
	}

	double **dd = (double **) cpl_calloc((dim[0]),sizeof(double *));
	for (i = 0; i < dim[0]; i++) {
		dd[i] = (double *)cpl_calloc((dim[1]),sizeof(double));
	}
    //cpl_msg_info(cpl_func,"ccdgain: %g max: %g min: %g", ccdgain, max, min);
    double pos_ccdgain = ccdgain;
    if (ccdgain <=0 ) {
    	pos_ccdgain = abs(ccdgain);
    } 
    

	nbin = (int) round((max-min)/pos_ccdgain);
	//cpl_msg_info(cpl_func,"nbin: %d", nbin);

	ctr = (double *) cpl_calloc((nbin+1),sizeof(double));
	limit = (double *) cpl_calloc((nbin+1),sizeof(double));

	for (i = 0; i <= nbin; i++) {
		ctr[i] = (round(min/ccdgain)-1.0+(double)i) * ccdgain;
		limit[i] = ctr[i] - 0.5*ccdgain;
	}
	h = gsl_histogram_alloc(nbin);
	gsl_histogram_set_ranges(h, limit, nbin+1);
	pflux = (double *) cpl_calloc(4*size_x*size_y,sizeof(double));

	p = gsl_vector_alloc(3);
	covar = gsl_matrix_alloc(3,3);
	int xx = 0;
	int yy = 0;
	int llx, lly, urx, ury;
	for (int lin = 0; lin < dim[0]; lin++) {
		llx = xx+1;
		urx = xx+(2*size_x);

		yy = 0;
		for (int col = 0; col < dim[1]; col++) {
			lly = yy+1;
			ury = yy+(2*size_y);

			grid_back_image = cpl_image_extract(orderdef,
                                                llx, lly, urx, ury);
			grid_back_image_data = cpl_image_get_data_double(grid_back_image);

			grid_input_image = cpl_image_extract(input_image,
                                                 llx, lly, urx, ury);
			grid_input_image_data = cpl_image_get_data_double(grid_input_image);

			for (int l = 0; l < 4*size_x*size_y; l++) {

				pflux[l] = grid_back_image_data[l];

			}

			gsl_sort(pflux, 1, 4*size_x*size_y);

			for (int l = 0; l < 4*size_x*size_y; l++) {
				pvalue = grid_back_image_data[l];
				if (pvalue < fmax(0.01*pflux[4*size_x*size_y-2*size_x],
                                  4.0*ccdsigdet)) {
                    
					value = grid_input_image_data[l];
					gsl_histogram_increment(h, value);
				}
			}

			index = gsl_histogram_max_bin(h);
			hwhm = (int) round(sqrt(fmax(ctr[index],0.)+
					ccdsigdet*ccdsigdet)/ccdgain*2.3548/2.);
			range = gsl_vector_alloc (2*hwhm);
			bin = gsl_vector_alloc (2*hwhm);
			X = gsl_matrix_alloc (2*hwhm, 3);
			work = gsl_multifit_linear_alloc (2*hwhm, 3);
            for (ii = 0; ii < (2*hwhm) && (index-hwhm+ii)>= 0; ii++) {
				gsl_vector_set(range, ii, ctr[index-hwhm+ii]);
				gsl_vector_set(bin, ii, h->bin[index-hwhm+ii]);
				for (jj = 0; jj < 3; jj++)
					gsl_matrix_set(X,ii,jj,gsl_pow_int(gsl_vector_get(range,
                                                                      ii),jj));
			}

			gsl_multifit_linear(X, bin, p, covar, &chisq, work);
			mode = -gsl_vector_get(p,1)/2./gsl_vector_get(p,2);

            if (((index-hwhm) >= 0) && ((mode < ctr[index-hwhm]) || (mode > ctr[index+hwhm-1]))) {
				dd[lin][col] = ctr[index];
			} else {
				dd[lin][col] = mode;
			}

			cpl_image_delete(grid_back_image);
			cpl_image_delete(grid_input_image);

			gsl_vector_free(range);
			gsl_vector_free(bin);
			gsl_matrix_free(X);
			gsl_multifit_linear_free(work);
			gsl_histogram_reset(h);

			//espdr_msg("ASE DEBUG : llx=%d lly=%d urx=%d ury=%d dd[%d][%d]=%f"
					//,llx,lly,urx,ury,lin,col,dd[lin][col]);

			yy = ury;
		}
		xx = urx;
	}
	gsl_vector_free(p);
	gsl_matrix_free(covar);
	gsl_histogram_free(h);
	cpl_free(ctr); cpl_free(limit); cpl_free(pflux);

	bkgr = (double *) cpl_calloc((dim[0]),sizeof(double));
	y2 = (double *) cpl_calloc(n1,sizeof(double));
	x2 = (double *) cpl_calloc(n1,sizeof(double));
	for (i = 0; i < n1; i++) x2[i] = (double)i;
	for (j = 0; j < dim[1]; j++) {

		for (i = 0; i < dim[0]; i++) {
			bkgr[i] = dd[i][j];
		}

		my_error = espdr_spline_bounded_interpolation(xcd, bkgr, dim[0],
                                                      x2, y2, n1, 0, n1);

        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_spline_bounded_interpolation failed: %s",
                     cpl_error_get_message_default(my_error));

		for (i = 0; i < n1; i++) {
			aa[i][yc[j]] = y2[i];
		}
	}
    
	cpl_free(bkgr);
	cpl_free(y2);
	cpl_free(x2);
    
	bkgr = (double *) cpl_calloc((dim[1]),sizeof(double));
	y2 = (double *) cpl_calloc(n2,sizeof(double));
	x2 = (double *) cpl_calloc(n2,sizeof(double));
    
    for (j = 0; j < n2; j++) {
        x2[j] = (double)j;
    }
	for (i = 0; i < n1; i++) {
        
		for (j = 0; j < dim[1]; j++) {
			bkgr[j] = aa[i][yc[j]];
		}
        
		my_error = espdr_spline_bounded_interpolation(ycd, bkgr, dim[1],
                                                      x2, y2, n2, 0, n2);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_spline_bounded_interpolation failed: %s",
                     cpl_error_get_message_default(my_error));
        
		for (j = 0; j < n2; j++) {
			aa[i][j] =
					y2[j];
		}
	}

	cpl_free(bkgr);
	cpl_free(y2);
	cpl_free(x2);
	int ind = 0;
	for (j = 0; j < n2; j++) {
		for (i = 0; i < n1; i++) {
			aa_vector[ind] = aa[i][j];
			ind++;
		}
	}
	cpl_image* ima_tmp = cpl_image_wrap_double(n1,n2,aa_vector);
	*image_background_measured_RE = cpl_image_duplicate(ima_tmp);
	cpl_image_unwrap(ima_tmp);
    
	*image_background_substracted_RE = cpl_image_duplicate(input_image);
	cpl_image_subtract(*image_background_substracted_RE,
			*image_background_measured_RE);
    
	for (i = 0; i < n1; i++) {
		cpl_free(aa[i]);
	}
	cpl_free(aa);

	for (i = 0; i < dim[0]; i++) {
		cpl_free(dd[i]);
	}
	cpl_free(dd);
	cpl_free(xc);
    cpl_free(yc);
    cpl_free(xcd);
	cpl_free(ycd);
    cpl_free(aa_vector);
	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief  computes the background ilumination for flat field frames by measuring
         grid sectors over the image
 @param  input_image image under treatment
 @param  bkgr_grid_size_x grid size in X	
 @param  bkgr_grid_size_y grid size in Y
 @param  minlevel_factor min factor to be apply in order to chose the min level
	     flux inside the grid.
 @param  image_background_measured_RE resultant background image
 @param  image_background_substracted_RE resultant background substracted image
 @return CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_measure_background_ff(
		const cpl_image *input_image,
		const int bkgr_grid_size_x,
		const int bkgr_grid_size_y,
		const int minlevel_factor,
		cpl_image **image_background_measured_RE,
		cpl_image **image_background_substracted_RE) {

	int n1 = cpl_image_get_size_x(input_image);
	int n2 = cpl_image_get_size_y(input_image);

	int size_x = bkgr_grid_size_x;
	int size_y = bkgr_grid_size_y;

    int dim[2];
	dim[0] = n1/2/size_x;
	dim[1] = n2/2/size_y;
	int *xc = (int *) cpl_calloc(dim[0],sizeof(int));
	int *yc = (int *) cpl_calloc(dim[1],sizeof(int));
	double *xcd = (double *) cpl_calloc(dim[0],sizeof(double));
	double *ycd = (double *) cpl_calloc(dim[1],sizeof(double));

	for (int i = 0; i < dim[0]; i++) {
        xc[i] = size_x + 2*i*size_x;
        xcd[i] = size_x + 2*i*size_x;
    }
	for (int j = 0; j < dim[1]; j++) {
        yc[j] = size_y + 2*j*size_y;
        ycd[j] = size_y + 2*j*size_y;
    }
    
    double *aa_vector = (double *) cpl_calloc((n1*n2),sizeof(double));
	double **aa = (double **) cpl_calloc(n1,sizeof(double *));
    
	for (int i = 0; i < n1; i++) {
		aa[i] = (double *)cpl_calloc((n2),sizeof(double));
	}
    
    double **minlevel = (double **) cpl_calloc(dim[1],sizeof(double *));
    
	int xx = 0;
	int yy = 0;
	int llx, lly, urx, ury;
    double *grid_data;
    cpl_image *grid = NULL;
    for (int col = 0; col < dim[1]; col++) {
        
        minlevel[col] = (double *)cpl_calloc(dim[0],sizeof(double));
        lly = yy+1;
        ury = yy+(2*size_y);
        xx = 0;
        
        for (int lin = 0; lin < dim[0]; lin++) {
            
            llx = xx+1;
            urx = xx+(2*size_x);
            
			grid = cpl_image_extract(input_image, llx, lly, urx, ury);
            
			grid_data = cpl_image_get_data_double(grid);
			gsl_sort(grid_data, 1, 4*size_x*size_y);
            minlevel[col][lin] = grid_data[minlevel_factor*size_x];
			cpl_image_delete(grid);
            
            xx = urx;
		}
        yy = ury;
	}

	double *bkgr = (double *) cpl_calloc(dim[0],sizeof(double));
	double *y2 = (double *) cpl_calloc(n1,sizeof(double));
	double *x2 = (double *) cpl_calloc(n1,sizeof(double));
    
#pragma omp parallel for default(none) shared(n1,x2)
    for (int i = 0; i < n1; i++) {
        x2[i] = (double)i;
    }
    
	for (int j = 0; j < dim[1]; j++) {
        
        cpl_error_code my_error = CPL_ERROR_NONE;
		my_error = espdr_spline_bounded_interpolation(xcd,
                                                      minlevel[j],
                                                      dim[0],
                                                      x2, y2, n1, 0, n1);
        
		espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
				"espdr_spline_bounded_interpolation failed: %s",
				cpl_error_get_message_default(my_error));
        
#pragma omp parallel for default(none) shared(n1,y2,yc,aa,j)
		for (int i = 0; i < n1; i++) {
			aa[i][yc[j]] = y2[i];
		}
	}
    
	cpl_free(bkgr);
	cpl_free(y2);
	cpl_free(x2);

	bkgr = (double *) cpl_calloc(dim[1],sizeof(double));
	y2 = (double *) cpl_calloc(n2,sizeof(double));
	x2 = (double *) cpl_calloc(n2,sizeof(double));
    
#pragma omp parallel for default(none) shared(n2,x2)
    for (int j = 0; j < n2; j++) {
        x2[j] = (double)j;
    }
    
	for (int i = 0; i < n1; i++) {
		for (int j = 0; j < dim[1]; j++) {
			bkgr[j] = aa[i][yc[j]];
		}

        cpl_error_code my_error = CPL_ERROR_NONE;
		my_error = espdr_spline_bounded_interpolation(ycd, bkgr, dim[1],
                                                      x2, y2, n2, 0, n2);

		espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
				"espdr_spline_bounded_interpolation failed: %s",
				cpl_error_get_message_default(my_error));

#pragma omp parallel for default(none) shared(n2,y2,yc,aa,i)
		for (int j = 0; j < n2; j++) {
			aa[i][j] = y2[j];
		}
	}

	cpl_free(bkgr);
	cpl_free(y2);
	cpl_free(x2);

	int ind = 0;
	for (int j = 0; j < n2; j++) {
		for (int i = 0; i < n1; i++) {
			aa_vector[ind] = aa[i][j];
			ind++;
		}
	}
	cpl_image* ima_tmp = cpl_image_wrap_double(n1, n2, aa_vector);
	*image_background_measured_RE = cpl_image_duplicate(ima_tmp);
	cpl_image_unwrap(ima_tmp);
    
	*image_background_substracted_RE = cpl_image_duplicate(input_image);
	cpl_image_subtract(*image_background_substracted_RE,
                       *image_background_measured_RE);

    for (int j = 0; j < dim[1]; j++) {
		cpl_free(minlevel[j]);
	}
	cpl_free(minlevel);
    
	for (int i = 0; i < n1; i++) {
		cpl_free(aa[i]);
	}
	cpl_free(aa);
	cpl_free(xc);
    cpl_free(yc);
    cpl_free(xcd);
	cpl_free(ycd);
    cpl_free(aa_vector);
    
	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief  sequencer function to calculate the background of science frames by
	     extension
 @param  input_image_list image under treatment by extension
 @param  ccdgain CCD gain value
 @param  ccdsigdet CCD RON value
 @param  imagelist_background_measured_RE resultant background image by
	     extension
 @return CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_measure_background_science_by_ext(
		const cpl_imagelist *input_image_list,
		const cpl_imagelist *orderdef_image_list,
		const double *ccdgain,
		const double *ccdsigdet,
		const int bkgr_grid_size_x,
		const int bkgr_grid_size_y,
		const espdr_CCD_geometry *CCD_geom,
		cpl_imagelist *imagelist_background_measured_RE,
		cpl_imagelist *imagelist_background_substracted_RE){

	int index = 0;
	cpl_image *master_science_bkgr_measure = NULL;
	cpl_image *master_science_bkgr_substracted = NULL;

	const cpl_image *curr_input_image_ext;
	const cpl_image *curr_order_image_ext;

	cpl_error_code my_error;

	for (int j = 0; j < CCD_geom->ext_nb; j++) {

		espdr_msg("Background science start for extesion %d",j);

		curr_input_image_ext = cpl_imagelist_get_const(input_image_list,j);
		curr_order_image_ext = cpl_imagelist_get_const(orderdef_image_list,j);

		my_error = espdr_measure_background_science(
				curr_input_image_ext,
				curr_order_image_ext,
				ccdgain[j],
				ccdsigdet[j],
				bkgr_grid_size_x,
				bkgr_grid_size_y,
				&master_science_bkgr_measure,
				&master_science_bkgr_substracted);

		espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
				"espdr_measure_background_science_by_ext failed");

		cpl_imagelist_set(imagelist_background_measured_RE,
				master_science_bkgr_measure,
				index);

		cpl_imagelist_set(imagelist_background_substracted_RE,
				master_science_bkgr_substracted,
				index);

		index++;
		espdr_msg("Background science finished for extension %d",j);
	}

	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief  sequencer function to calculate the background of flat field frames by
         extension
 @param  input_image_list image under treatment by extension
 @param  CCD_geom CCD geometry structure
 @param  bkgr_grid_size_x grid size in X	
 @param  bkgr_grid_size_y grid size in Y
 @param  minlevel_factor min factor to be apply in order to chose the min level
	     flux inside the grid.
 @param  image_background_measured_RE resultant background image by extension
 @param  image_background_substracted_RE resultant background substracted image
	     by extension
 @return CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_measure_background_ff_by_ext(
		const cpl_imagelist *input_image_list,
		const espdr_CCD_geometry *CCD_geom,
		const int bkgr_grid_size_x,
		const int bkgr_grid_size_y,
		const int minlevel_factor,
		cpl_imagelist *imagelist_background_measured_RE,
		cpl_imagelist *imagelist_background_substracted_RE) {

	int index = 0;
	cpl_image *master_flat_bkgr_measure = NULL;
	cpl_image *master_flat_bkgr_substracted = NULL;
	const cpl_image *curr_image_ext;

	for (int j = 0; j < CCD_geom->ext_nb; j++) {

		espdr_msg("Background FF start at extesion %d",j);

		curr_image_ext = cpl_imagelist_get_const(input_image_list,j);

		espdr_measure_background_ff(
				curr_image_ext,
				bkgr_grid_size_x,
				bkgr_grid_size_y,
				minlevel_factor,
				&master_flat_bkgr_measure,
				&master_flat_bkgr_substracted);

		cpl_imagelist_set(imagelist_background_measured_RE,
				cpl_image_duplicate(master_flat_bkgr_measure),
				index);

		cpl_imagelist_set(imagelist_background_substracted_RE,
				cpl_image_duplicate(master_flat_bkgr_substracted),
				index);
		cpl_image_delete(master_flat_bkgr_measure);
		cpl_image_delete(master_flat_bkgr_substracted);

		index++;
		espdr_msg("Background FF finished for extension %d",j);

	}

	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief  calculates the background substraction between the original image and
	     it's background measured
 @param  input_image image under treatment
 @param  data_bkgr_measurement_image image with background measured
 @param  image_size_x image size in X
 @param  image_size_y image size in Y
 @param  image_background_substracted_RE resultant background substracted image
 @return CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code remove_inter_order_background(
		const double *data_input_image,
		const double *data_bkgr_measurement_image,
		const int image_size_x,
		const int image_size_y,
		cpl_image **image_background_substracted_RE) {

	double *background_substracted = 
			(double *) cpl_calloc((image_size_x*image_size_y),sizeof(double));

	int ind=0;	
	for (int j = 0; j < image_size_y; j++) {
		for (int i = 0; i < image_size_x; i++) {
			background_substracted[ind] = 
					data_input_image[i+image_size_x*j]
									 -data_bkgr_measurement_image[i+image_size_x*j];
			ind++;
		}
	}

	*image_background_substracted_RE = 
			cpl_image_duplicate(cpl_image_wrap_double(
					image_size_x,
					image_size_y,
					background_substracted));

	cpl_free(background_substracted);	

	return cpl_error_get_code();
}
