/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: asegovia $
 * $Date: 2015-05-01 15:13:41 $
 * $Revision:     $
 * $Name: not supported by cvs2svn $
 */


#include <espdr_flux.h>
#include <espdr_overscan.h>
#include <espdr_instrument.h>
#include <espdr_background.h>
#include <espdr_blaze.h>
#include <espdr_wave_cal.h>
#include <espdr_orders.h>
#include <espdr_pixels.h>
#include <espdr_science.h>
#include <stdlib.h>

/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/**
 @brief     Save cal_flux products
 @param
 @return    CPL_ERROR_NONE iff OK

 */
/*---------------------------------------------------------------------------*/


cpl_error_code espdr_save_cal_flux_products(cpl_frameset *frameset,
                                            cpl_parameterlist *parameters,
                                            const char * recipe_id,
                                            cpl_frameset *used_frames,
                                            espdr_inst_config *inst_config,
                                            cpl_propertylist *keywords,
                                            cpl_image *flux_s2d_blaze,
                                            cpl_image *error_s2d_blaze,
                                            cpl_image *qual_s2d_blaze,
                                            cpl_image *flux_s2d,
                                            cpl_image *error_s2d,
                                            cpl_image *qual_s2d,
                                            cpl_image *wave_shifted,
                                            cpl_image *wave_air_shifted,
                                            cpl_image *dll_shifted,
                                            cpl_image *dll_air_shifted,
                                            cpl_table *s1d_table_merged,
                                            cpl_table *s1d_table_energy,
                                            cpl_table *slit_corrected_avg_flux,
                                            cpl_table *efficiency,
                                            cpl_table *s1d_eff) {

    char product_filename[64];
    cpl_image **images_to_save;
    cpl_frame* product_frame;
    cpl_frameset *frameset_orig = cpl_frameset_duplicate(frameset);

    images_to_save = (cpl_image**)cpl_malloc(7*sizeof(cpl_image*));
    images_to_save[0] = cpl_image_duplicate(flux_s2d);
    images_to_save[1] = cpl_image_duplicate(error_s2d);
    images_to_save[2] = cpl_image_duplicate(qual_s2d);
    images_to_save[3] = cpl_image_duplicate(wave_shifted);
    images_to_save[4] = cpl_image_duplicate(wave_air_shifted);
    images_to_save[5] = cpl_image_duplicate(dll_shifted);
    images_to_save[6] = cpl_image_duplicate(dll_air_shifted);

    sprintf(product_filename, "%s_S2D_STD_A.fits", inst_config->instrument);
    cpl_propertylist_update_string(keywords, PRO_CATG_KW, ESPDR_PRO_CATG_S2D_STD_A);
    cpl_propertylist_erase(keywords, "RADECSYS");
    espdr_dfs_save_data_err_qual_wave(frameset_orig, parameters,
                                                 used_frames, recipe_id,
                                                 keywords, product_filename,
                                                 images_to_save);
    espdr_frame_new(&product_frame, product_filename,
                    CPL_FRAME_GROUP_PRODUCT,
                    CPL_FRAME_LEVEL_FINAL,
                    CPL_FRAME_TYPE_IMAGE,
                    ESPDR_PRO_CATG_S2D_STD_A);
    cpl_frameset_insert(frameset, product_frame);

    for (int i = 0; i < 7; i++) {
        cpl_image_delete(images_to_save[i]);
    }

    images_to_save[0] = cpl_image_duplicate(flux_s2d_blaze);
    images_to_save[1] = cpl_image_duplicate(error_s2d_blaze);
    images_to_save[2] = cpl_image_duplicate(qual_s2d_blaze);
    images_to_save[3] = cpl_image_duplicate(wave_shifted);
    images_to_save[4] = cpl_image_duplicate(wave_air_shifted);
    images_to_save[5] = cpl_image_duplicate(dll_shifted);
    images_to_save[6] = cpl_image_duplicate(dll_air_shifted);

    sprintf(product_filename, "%s_S2D_BLAZE_STD_A.fits", inst_config->instrument);
    cpl_propertylist_update_string(keywords, PRO_CATG_KW, ESPDR_PRO_CATG_S2D_BLAZE_STD_A);

    espdr_dfs_save_data_err_qual_wave(frameset_orig, parameters,
                                                 used_frames, recipe_id,
                                                 keywords, product_filename,
                                                 images_to_save);
    espdr_frame_new(&product_frame, product_filename,
                    CPL_FRAME_GROUP_PRODUCT,
                    CPL_FRAME_LEVEL_FINAL,
                    CPL_FRAME_TYPE_IMAGE,
                    ESPDR_PRO_CATG_S2D_BLAZE_STD_A);
    cpl_frameset_insert(frameset, product_frame);

    for (int i = 0; i < 7; i++) {
        cpl_image_delete(images_to_save[i]);
    }
    cpl_free(images_to_save);

    sprintf(product_filename, "%s_S1D_STD_A.fits", inst_config->instrument);
    cpl_propertylist_update_string(keywords, PRO_CATG_KW, ESPDR_PRO_CATG_S1D_STD_A);

    cpl_dfs_save_table(frameset_orig, NULL, parameters, used_frames,
                                  NULL, s1d_table_merged,
                                  NULL, recipe_id, keywords, "RADECSYS",
                                  PACKAGE "/" PACKAGE_VERSION, product_filename);
    espdr_frame_new(&product_frame, product_filename,
                    CPL_FRAME_GROUP_PRODUCT,
                    CPL_FRAME_LEVEL_FINAL,
                    CPL_FRAME_TYPE_IMAGE,
                    ESPDR_PRO_CATG_S1D_STD_A);
    cpl_frameset_insert(frameset, product_frame);

    sprintf(product_filename, "%s_S1D_ENERGY_STD_A.fits", inst_config->instrument);
    cpl_propertylist_update_string(keywords, PRO_CATG_KW, ESPDR_PRO_CATG_S1D_ENERGY_STD_A);

    cpl_dfs_save_table(frameset_orig, NULL, parameters, used_frames,
                                  NULL, s1d_table_energy,
                                  NULL, recipe_id, keywords, "RADECSYS",
                                  PACKAGE "/" PACKAGE_VERSION, product_filename);
    espdr_frame_new(&product_frame, product_filename,
                    CPL_FRAME_GROUP_PRODUCT,
                    CPL_FRAME_LEVEL_FINAL,
                    CPL_FRAME_TYPE_IMAGE,
                    ESPDR_PRO_CATG_S1D_ENERGY_STD_A);
    cpl_frameset_insert(frameset, product_frame);

    sprintf(product_filename, "%s_AVG_FLUX_STD_A.fits", inst_config->instrument);
    cpl_propertylist_update_string(keywords, PRO_CATG_KW, ESPDR_PRO_CATG_AVG_FLUX_STD_A);

    cpl_dfs_save_table(frameset_orig, NULL, parameters, used_frames,
                                  NULL, slit_corrected_avg_flux,
                                  NULL, recipe_id, keywords, "RADECSYS",
                                  PACKAGE "/" PACKAGE_VERSION, product_filename);
    espdr_frame_new(&product_frame, product_filename,
                    CPL_FRAME_GROUP_PRODUCT,
                    CPL_FRAME_LEVEL_FINAL,
                    CPL_FRAME_TYPE_IMAGE,
                    ESPDR_PRO_CATG_AVG_FLUX_STD_A);
    cpl_frameset_insert(frameset, product_frame);

    sprintf(product_filename, "%s_ABS_EFF_RAW_A.fits", inst_config->instrument);
    cpl_propertylist_update_string(keywords, PRO_CATG_KW, ESPDR_PRO_CATG_ABS_EFF_RAW);

    cpl_dfs_save_table(frameset_orig, NULL,
                                  parameters,used_frames,
                                  NULL,efficiency,
                                  NULL,recipe_id,keywords,"RADECSYS",
                                  PACKAGE "/" PACKAGE_VERSION,
                                  product_filename);
    espdr_frame_new(&product_frame, product_filename,
                    CPL_FRAME_GROUP_PRODUCT,
                    CPL_FRAME_LEVEL_FINAL,
                    CPL_FRAME_TYPE_IMAGE,
                    ESPDR_PRO_CATG_ABS_EFF_RAW);
    cpl_frameset_insert(frameset, product_frame);

    sprintf(product_filename, "%s_ABS_EFF_A.fits", inst_config->instrument);
    cpl_propertylist_set_string(keywords, PRO_CATG_KW, ESPDR_PRO_CATG_ABS_EFF);

    cpl_dfs_save_table(frameset_orig, NULL,
                                  parameters,used_frames,
                                  NULL,s1d_eff,
                                  NULL,recipe_id,keywords,"RADECSYS",
                                  PACKAGE "/" PACKAGE_VERSION,
                                  product_filename);
    espdr_frame_new(&product_frame, product_filename,
                    CPL_FRAME_GROUP_PRODUCT,
                    CPL_FRAME_LEVEL_FINAL,
                    CPL_FRAME_TYPE_IMAGE,
                    ESPDR_PRO_CATG_ABS_EFF);
    cpl_frameset_insert(frameset, product_frame);

    cpl_frameset_delete(frameset_orig);

    return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 @brief     Add extra QC keywords
 @param     keywords    property list to add the extra KWs
 @param     ext_nb      number of extensions
 @return    0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static cpl_error_code espdr_add_extra_qc(cpl_propertylist** keywords,
                                         const int ext_nb) {

    const char* ksuff[2] = {" BLUE", " RED"};
    cpl_propertylist_append_int(*keywords,"ESO QC PROC VERSION", 2);
    int* ord_ref = espdr_get_ord_ref(*keywords, "espdr_cal_flux");
    espdr_add_qc_key_stat_pri(keywords, ext_nb,"MAX FLUX", ksuff, CPL_FALSE, CPL_FALSE,
                              CPL_FALSE, CPL_TRUE, CPL_FALSE);

    espdr_add_qc_key_stat_ord_pri(keywords, ext_nb, "SNR", "DUMMY", CPL_TRUE, CPL_TRUE,
                                  CPL_TRUE, CPL_TRUE, CPL_FALSE);


    espdr_add_qc_key_flux_ord_ref(keywords, ext_nb, ord_ref, "SNR",ksuff,
                                  CPL_TRUE, CPL_TRUE, CPL_TRUE, CPL_TRUE, CPL_FALSE);

    espdr_add_qc_airm_info(keywords);
    espdr_add_qc_iwv_info(keywords);
    cpl_free(ord_ref);
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    parameters     the parameters list
 @param    frameset   the frames list

 In case of failure the cpl_error_code is set.
 */
/*---------------------------------------------------------------------------*/
int espdr_cal_flux(cpl_parameterlist *parameters, cpl_frameset *frameset,
		const char* recipe_id) {

	cpl_frameset *used_frames = NULL;

	cpl_error_code my_error = CPL_ERROR_NONE;

    const int rec_ntags = 20;
    const char* rec_tags[20] = {ESPDR_FLUX_RAW,
        ESPDR_PRO_CATG_HOT_PIXELS, ESPDR_PRO_CATG_MDARK,
        ESPDR_PRO_CATG_BAD_PIXELS, ESPDR_CCD_GEOM,
        ESPDR_PRO_CATG_ORDERS_A, ESPDR_PRO_CATG_ORDERS_B,
        ESPDR_STD_STARS_TABLE, ESPDR_EXTINCTION_TABLE,
        ESPDR_PRO_CATG_WAVE_MATRIX_THAR_FP_A,
        ESPDR_PRO_CATG_WAVE_MATRIX_FP_THAR_B,
        ESPDR_PRO_CATG_WAVE_MATRIX_DRIFT_THAR_FP_A,
        ESPDR_PRO_CATG_WAVE_MATRIX_DRIFT_THAR_THAR_A,
        ESPDR_PRO_CATG_WAVE_MATRIX_DRIFT_THAR_THAR_B,
        ESPDR_PRO_CATG_DLL_MATRIX_THAR_FP_A,
        ESPDR_PRO_CATG_DLL_MATRIX_FP_THAR_B,
        ESPDR_PRO_CATG_DLL_MATRIX_DRIFT_THAR_FP_A,
        ESPDR_PRO_CATG_DLL_MATRIX_DRIFT_THAR_THAR_A,
        ESPDR_PRO_CATG_DLL_MATRIX_DRIFT_THAR_THAR_B,
        ESPDR_ORDERS_MASK
    };
    int is_required[20] = {1, 1, 0, 1, 1,
        1, 1, // ORDER_TABLEs
        1, 1, // STD & extinction tables
        0, 0, 0, 0, 0, // WAVE MATRIXes
        0, 0, 0, 0, 0, // DLL MATRIXes
        0 // ORDERS MASK
    };

	cpl_msg_set_level(CPL_MSG_INFO);
	//cpl_msg_set_level(CPL_MSG_DEBUG);

    espdr_ensure(espdr_check_input_tags(frameset,
                                        rec_tags, is_required,
                                        rec_ntags) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_inst_config(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_flat(frameset, 1) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong flat input tag!");

    espdr_CCD_geometry *CCD_geom = NULL;
    espdr_inst_config *inst_config = NULL;
    espdr_qc_keywords *qc_kws = NULL;
    espdr_OVSC_param *OVSC_param = NULL;
    espdr_CAL_FLUX_param *CAL_FLUX_param = NULL;
    cpl_frame* CCD_geom_frame = NULL;
    cpl_frame* inst_config_frame = NULL;

    espdr_msg("Starting cal_flux");

    /* Identify the RAW and CALIB frames in the input frameset */
    espdr_ensure(espdr_dfs_set_groups(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(),
                 "DFS setting groups failed. Expected inputs are:\n"
                 "raw standard star (flux) image, tagged as FLUX\n"
                 "CCD geometry table, tagged as CCD_GEOM\n"
                 "instrument configuration table, tagged as MASTER_INST_CONFIG or INST_CONFIG\n"
                 "table of stabdards stars fluxes, tagged as STD_TABLE\n"
                 "extinction table, tagged as EXT_TABLE\n"
                 "hot pixel mask image, tagged as HOT_PIXEL_MASK\n"
                 "bad pixel mask image, tagged as BAD_PIXEL_MASK\n"
                 "orders definition images for fibres A and B, tagged as ORDER_TABLE_A & ORDER_TABLE_B\n"
                 "orders profile images for fibres A and B, tagged as ORDER_PROFILE_A & ORDER_PROFILE_B\n"
                 "flat images for fibres A and B, tagged as FLAT_A & FLAT_B\n"
                 "blaze images for fibres A and B, tagged as BLAZE_A & BLAZE_B\n"
                 "wave matrix for fibres A or B, tagged as WAVE_MATRIX_THAR_FP_A or WAVE_MATRIX_FP_THAR_B\n"
                 "dll matrix for fibres A or B, tagged as DLL_MATRIX_THAR_FP_A or DLL_MATRIX_FP_THAR_B\n");

    /* AMO Changed way to read CCD geom frame and to init relevant recipe
     * internal parameters structures to simplify this calling function
     */
    OVSC_param        = espdr_parameters_OVSC_init(recipe_id, parameters);
    CCD_geom_frame    = espdr_frame_find(frameset, ESPDR_CCD_GEOM);
    CCD_geom          = espdr_CCD_geom_init(parameters, CCD_geom_frame);
    inst_config_frame = espdr_get_inst_config(frameset);
    inst_config       = espdr_inst_config_init(parameters,
                                               CCD_geom->ext_nb,
                                               inst_config_frame);
    CAL_FLUX_param    = espdr_CAL_FLUX_param_init(recipe_id, parameters);

    used_frames = cpl_frameset_new();
    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(CCD_geom_frame));
    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(inst_config_frame));

    /* Filling up the DRS QC KWs structure */
    qc_kws = (espdr_qc_keywords *)cpl_malloc( sizeof(espdr_qc_keywords));
    my_error = espdr_fill_qc_keywords(inst_config, qc_kws);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Filling QC KWs failed: %s",
                 cpl_error_get_message_default(my_error));

    espdr_ensure(qc_kws == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "QC KWs structure is NULL");

    cpl_frame *flux_frame = espdr_frame_find(frameset, ESPDR_FLUX_RAW);
    espdr_ensure(flux_frame == NULL, CPL_ERROR_NULL_INPUT,
                 "No input raw FLUX frame");

    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(flux_frame));

    /* Load flux keywords from the first image (primary header) */
	const char *input_filename = cpl_frame_get_filename(flux_frame);
    espdr_msg("KEYWORDS for CAL_FLUX input filename: %s", input_filename);
    cpl_propertylist *keywords = cpl_propertylist_load(input_filename, 0);
    espdr_ensure(keywords == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "keywords are NULL");

#if SAVE_DEBUG_PRODUCT_PREPROCESSING
    // Adding artificially the ARCFILE if not present - final filename - to be removed when using ESO DFS
    if (cpl_propertylist_has(keywords, "ARCFILE") == 0) {
        char *arc_filename = strrchr(input_filename, '/')+1;
        espdr_msg("ARCFILE not existing - updating the filename: %s", arc_filename);
        my_error = cpl_propertylist_update_string(keywords, "ARCFILE", arc_filename);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Updating the ARCFILE KW (%s) failed: %s",
                     arc_filename, cpl_error_get_message_default(my_error));
    }
#endif

    espdr_msg("CAL_FLUX processing started ...");

    const char *star_name = cpl_propertylist_get_string(keywords, inst_config->targ_name_kw);
    cpl_table *std_star_table, *extinction_table;
    my_error = espdr_cal_flux_get_static_tables(frameset, used_frames, star_name,
                                                &std_star_table, &extinction_table);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_cal_flux_get_static_tables failed: %s",
                 cpl_error_get_message_default(my_error));

    double exp_time_hour, ra_ft, dec_ft, pm_ra, pm_dec;
    int year, month, day, hour, minutes;
    double seconds, time_to_mid_exposure = 0.5, airmass;

    my_error = espdr_get_berv_params(inst_config, qc_kws, keywords, flux_frame,
                                     &exp_time_hour, &ra_ft, &dec_ft, &pm_ra, &pm_dec,
                                     &year, &month, &day, &hour, &minutes, &seconds,
                                     &time_to_mid_exposure, &airmass);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_get_berv_params failed: %s",
                 cpl_error_get_message_default(my_error));

    /* BERV calculation */
    double BERV;    /* Barycentric Earth radial velocity (km/s) */
    double VTOT;    /* Total velocity of observer (km/s) */
    double BJD;     /* Barycentric time of light arrival (UTC Julian date) */
    double BERVMAX; /* Yearly maximum value of BERV (km/s) */
    double SED;     /* Sun-Earth distance (in km) */
    double BERV_factor;

    my_error = espdr_compute_BERV(ra_ft, dec_ft, pm_ra, pm_dec,
                                  year, month, day, hour, minutes, seconds,
                                  time_to_mid_exposure,
                                  inst_config->tel_geolon,
                                  inst_config->tel_geolat,
                                  inst_config->tel_geoelev,
                                  inst_config->berv_type_init,
                                  &BERV, &VTOT, &BJD, &BERVMAX, &SED);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_compute_BERV failed: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = espdr_compute_BERV_factor(BERV, SED, VTOT, &BERV_factor);
    
    // Background subtraction parameter managemenet
    char background_sw[10];
    sprintf(background_sw,"%s","on");

    cpl_parameter *par_background_sw = cpl_parameterlist_find(parameters,
                                                              "espdr.espdr_cal_flux.background_sw");
    int background_sw_set = cpl_parameter_get_default_flag(par_background_sw);
    if (background_sw_set == 1) {
        sprintf(background_sw, "%s", CAL_FLUX_param->background_sw);
        cpl_parameter_set_string(par_background_sw, CAL_FLUX_param->background_sw);
    }
    espdr_msg("Background SW is %s", background_sw);

    int *orders_nb_per_fibre = NULL;
    orders_nb_per_fibre = (int *)cpl_calloc(inst_config->fibres_nb, sizeof(int));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        orders_nb_per_fibre[i] = 0;
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            orders_nb_per_fibre[i] += inst_config->orders_nb[i*CCD_geom->ext_nb+j];
        }
    }

    int **cosmics_nb = (int **) cpl_calloc(inst_config->fibres_nb,sizeof(int *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cosmics_nb[i] = (int *) cpl_calloc(orders_nb_per_fibre[i],sizeof(int));
    }

    double **snr_per_fibre = (double **) cpl_calloc(inst_config->fibres_nb, sizeof(double *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        snr_per_fibre[i] = (double *) cpl_calloc(orders_nb_per_fibre[i],sizeof(double));
    }

    double mjd_obs_delta_time_wave[inst_config->fibres_nb];
    double *ron = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
    cpl_image **blaze_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **wave_matrix = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **wave_matrix_air = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **dll_matrix = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **dll_matrix_air = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_corr_flux_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    cpl_image **flat_corr_err_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    cpl_image **flat_corr_qual_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    my_error = espdr_process_raw_till_extraction(frameset, parameters, recipe_id,
                                                 flux_frame, used_frames,
                                                 inst_config, CCD_geom, qc_kws,
                                                 keywords,
                                                 0, // remove_bias_res_flag
                                                 "THAR", "SKY", 0,
                                                 background_sw,
                                                 inst_config->extraction_method,
                                                 NULL, NULL, NULL, // AR offset
                                                 ron, cosmics_nb, snr_per_fibre,
                                                 mjd_obs_delta_time_wave,
                                                 blaze_table,
                                                 wave_matrix, wave_matrix_air,
                                                 dll_matrix, dll_matrix_air,
                                                 NULL, // s2d_ref_image
                                                 flat_corr_flux_table,
                                                 flat_corr_err_table,
                                                 flat_corr_qual_table);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_process_raw_till_extraction failed: %s",
                 cpl_error_get_message_default(my_error));

    // Creating the wavelength grids
    cpl_image *wavelenght_grid_pow_10 = NULL;
    cpl_image *wavelenght_air_grid_pow_10 = NULL;
    my_error = espdr_create_wavelength_grid(inst_config->log_lambda_ini,
                                            inst_config->log_lambda_end,
                                            inst_config->delta_log_lambda,
                                            &wavelenght_grid_pow_10,
                                            &wavelenght_air_grid_pow_10);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_create_wavelength_grid failed: %s",
                 cpl_error_get_message_default(my_error));

    cpl_image *flux_s2d, *error_s2d, *qual_s2d;
    cpl_image *wave_air_shifted, *wave_shifted, *dll_air_shifted, *dll_shifted;
    cpl_table *s1d_table_merged, *s1d_table_energy;

    my_error = espdr_shift_wavelength_solution(BERV_factor,
                                               wave_matrix[0],
                                               dll_matrix[0],
                                               &wave_shifted,
                                               &dll_shifted);
    my_error = espdr_shift_wavelength_solution(BERV_factor,
                                               wave_matrix_air[0],
                                               dll_matrix_air[0],
                                               &wave_air_shifted,
                                               &dll_air_shifted);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_shift_wavelength_solution failed for fibre A: %s",
                 cpl_error_get_message_default(my_error));

    my_error = espdr_correct_blaze(flat_corr_flux_table[0],
                                   flat_corr_err_table[0],
                                   flat_corr_qual_table[0],
                                   blaze_table[0],
                                   &flux_s2d,
                                   &error_s2d,
                                   &qual_s2d);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_correct_blaze failed for skysub fibreA: %s",
                 cpl_error_get_message_default(my_error));

    my_error = espdr_rebin_and_merge_orders(wavelenght_grid_pow_10,
                                            wavelenght_air_grid_pow_10,
                                            wave_shifted, /* wave_matrix in vacuum */
                                            dll_shifted,
                                            flux_s2d,
                                            error_s2d,
                                            qual_s2d,
                                            &s1d_table_merged);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_rebin_and_merge_orders failed for skysub fiber A: %s",
                 cpl_error_get_message_default(my_error));
    espdr_ensure(s1d_table_merged == NULL, CPL_ERROR_NULL_INPUT,
                 "s2d_table_merged is NULL, exiting");
    s1d_table_energy = cpl_table_duplicate(s1d_table_merged);

    double exp_time = cpl_propertylist_get_double(keywords, inst_config->Texp_kw);
    my_error = espdr_convert_photons_to_energy(inst_config->tel_surface,
                                               exp_time,
                                               &s1d_table_energy);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_convert_photons_to_energy failed for fiber A: %s",
                 cpl_error_get_message_default(my_error));

	/* Selecting the star from Standard table ... */
	espdr_msg("OBJECT NAME: %s", star_name);
	cpl_table *std_star_selected;
	my_error = espdr_load_standard_stars_table(std_star_table,
                                               star_name,
                                               &std_star_selected);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_load_standard_stars_table failed: %s",
                 cpl_error_get_message_default(my_error));

	/* Measuring the avg flux ... */
	cpl_table *measured_eff_flux;
	my_error = espdr_measure_standard_flux(s1d_table_energy,
                                           std_star_selected,
                                           &measured_eff_flux);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_measure_standard_flux failed: %s",
                 cpl_error_get_message_default(my_error));

	cpl_table *inter_extinction;
	my_error = espdr_pre_extinction_table_interpolation(extinction_table,
                                                        std_star_table,
                                                        &inter_extinction);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_pre_extinction_table_interpolation failed: %s",
                 cpl_error_get_message_default(my_error));

	/* Correcting the avg measured flux from airmass ... */
	cpl_table *airmass_corrected_avg_flux;
    my_error = espdr_correct_extinction(measured_eff_flux,
                                        inter_extinction,
                                        airmass,
                                        &airmass_corrected_avg_flux);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_correct_extinction failed: %s",
                 cpl_error_get_message_default(my_error));

    double seeing = 0.0;
    int seeing_kw_qc = 1;
    my_error = espdr_get_seeing(inst_config, keywords, &seeing, &seeing_kw_qc);

    cpl_table *slit_corrected_avg_flux;
    my_error = espdr_correct_slit(airmass_corrected_avg_flux,
                                  seeing, inst_config,
                                  &slit_corrected_avg_flux);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_correct_slit failed : %s",
                 cpl_error_get_message_default(my_error));

	/* Caltulating the efficiency over the standard wavelength ... */
	cpl_table *efficiency;
	my_error = espdr_compute_efficiency(std_star_selected,
                                        slit_corrected_avg_flux,
                                        &efficiency);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_compute_efficiency failed : %s",
                 cpl_error_get_message_default(my_error));

	cpl_table *s1d_eff;
	if (strcmp(inst_config->flux_eff_interp_method,"spline") == 0) {
		espdr_msg("Efficiency interpolation spline method ...");
		my_error = espdr_interpolated_eff(efficiency,
					      s1d_table_energy, &s1d_eff);
	} else if (strcmp(inst_config->flux_eff_interp_method,"poly") == 0) {
		espdr_msg("Efficiency interpolation polynomial method ...");
		my_error = espdr_polynomial_eff(inst_config->flux_poly_deg,
						efficiency, s1d_table_energy, &s1d_eff);
	} else {
		espdr_msg("Wrong efficiency computation interpolation method.");
        return (CPL_ERROR_TYPE_MISMATCH);
	}
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Efficiency interpolation failed : %s",
                 cpl_error_get_message_default(my_error));

    /* QC function */
	my_error = espdr_cal_flux_QC(CCD_geom, inst_config, qc_kws,
                                 efficiency, s1d_eff,
                                 BERV, BJD, BERVMAX, BERV_factor,
                                 cosmics_nb[0],
                                 snr_per_fibre[0],
                                 orders_nb_per_fibre[0],
                                 seeing_kw_qc,
                                 &keywords);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_cal_flux_QC failed: %s",
                 cpl_error_get_message_default(my_error));

    if ((strcmp(inst_config->instrument, "HARPN") != 0) &&
        (strcmp(inst_config->instrument, "CORALIE") != 0)) {
        my_error = espdr_add_extra_qc(&keywords, CCD_geom->ext_nb);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_add_extra_qc failed: %s ",
                     cpl_error_get_message_default(my_error));
    }

    /* Products saving */
    my_error = espdr_save_cal_flux_products(frameset, parameters, recipe_id,
                                            used_frames, inst_config, keywords,
                                            flat_corr_flux_table[0], flat_corr_err_table[0],
                                            flat_corr_qual_table[0],
                                            flux_s2d, error_s2d, qual_s2d,
                                            wave_shifted, wave_air_shifted,
                                            dll_shifted, dll_air_shifted,
                                            s1d_table_merged, s1d_table_energy,
                                            slit_corrected_avg_flux, efficiency, s1d_eff);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_save_cal_flux_products failed: %s",
                 cpl_error_get_message_default(my_error));

    /* cleaning memory */
    espdr_msg("Cleaning memory");
	for (int i = 0; i < inst_config->fibres_nb; i++) {
		cpl_image_delete(wave_matrix[i]);
		cpl_image_delete(wave_matrix_air[i]);
        cpl_image_delete(dll_matrix[i]);
        cpl_image_delete(dll_matrix_air[i]);
		cpl_image_delete(blaze_table[i]);
        cpl_image_delete(flat_corr_flux_table[i]);
        cpl_image_delete(flat_corr_err_table[i]);
        cpl_image_delete(flat_corr_qual_table[i]);
        cpl_free(cosmics_nb[i]);
        cpl_free(snr_per_fibre[i]);
	}
    cpl_free(wave_matrix);
    cpl_free(wave_matrix_air);
    cpl_free(dll_matrix);
    cpl_free(dll_matrix_air);
    cpl_free(blaze_table);
    cpl_free(flat_corr_flux_table);
    cpl_free(flat_corr_err_table);
    cpl_free(flat_corr_qual_table);

	cpl_image_delete(flux_s2d);
	cpl_image_delete(error_s2d);
	cpl_image_delete(qual_s2d);

    cpl_image_delete(wavelenght_grid_pow_10);
    cpl_image_delete(wavelenght_air_grid_pow_10);

    cpl_free(wave_air_shifted);
    cpl_free(wave_shifted);
    cpl_free(dll_air_shifted);
    cpl_free(dll_shifted);

    cpl_propertylist_delete(keywords);
    cpl_free(qc_kws);

    cpl_frameset_delete(used_frames);

    cpl_free(ron);
    cpl_free(cosmics_nb);
    cpl_free(orders_nb_per_fibre);
    cpl_free(snr_per_fibre);

    espdr_msg("Cleaning tables");
    cpl_table_delete(std_star_table);
    cpl_table_delete(std_star_selected);
    cpl_table_delete(measured_eff_flux);
    cpl_table_delete(extinction_table);
    cpl_table_delete(inter_extinction);
    cpl_table_delete(airmass_corrected_avg_flux);
    cpl_table_delete(slit_corrected_avg_flux);
    cpl_table_delete(efficiency);
    cpl_table_delete(s1d_eff);
    cpl_table_delete(s1d_table_merged);
    cpl_table_delete(s1d_table_energy);

    espdr_parameters_OVSC_delete(OVSC_param);
    espdr_parameters_CAL_FLUX_delete(CAL_FLUX_param);
    espdr_parameters_CCD_geometry_delete(CCD_geom);
    espdr_parameters_inst_config_delete(inst_config);

	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Create the structure for CAL_FLUX input parameters
 @param    recipe_id	recipe identifier
 @param    list			list of input parameters
 @param    p			input CAL_FLUX parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_CAL_FLUX_create(const char* recipe_id,
                                            cpl_parameterlist *list,
                                            espdr_CAL_FLUX_param *p) {
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, 
		     "The parameters list NULL");
    
	cpl_error_code error_got;
    char comment[COMMENT_LENGTH];
    int bkgr_x_min = BKGR_X_MIN;
    int bkgr_x_max = BKGR_X_MAX;
    int bkgr_y_min = BKGR_Y_MIN;
    int bkgr_y_max = BKGR_Y_MAX;
    
	/* Fill the parameter list */
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "background_sw",
                                            p->background_sw,
                                            "Background subtraction activation (on/off)");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter background_sw to the list");
    
    sprintf(comment,
            "Grid size in x used to calculate the background, between: %d and %d",
            bkgr_x_min,bkgr_x_max);
    error_got = espdr_parameters_new_range_int(recipe_id, list,
                                               "flux_bkgr_grid_size_x",
                                               p->bkgr_grid_size_x,
                                               bkgr_x_min, bkgr_x_max,
                                               comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter bkgr_grid_size_x to the list");
    
    sprintf(comment,
            "Grid size in y used to calculate the background, between: %d and %d",
            bkgr_y_min,bkgr_y_max);
    error_got = espdr_parameters_new_range_int(recipe_id, list,
                                               "flux_bkgr_grid_size_y",
                                               p->bkgr_grid_size_y,
                                               bkgr_y_min, bkgr_y_max,
                                               comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter bkgr_grid_size_y to the list");
    
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "flux_extraction_method",
                                            p->extraction_method,
                                            "Method used to extract orders");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter extraction_method to the list");
    
    error_got = espdr_parameters_new_double(recipe_id, list,
                                            "flux_extraction_ksigma",
                                            p->extraction_ksigma,
                                            "ksigma used to extract orders");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter extraction_method to the list");
    
    error_got = espdr_parameters_new_range_int(recipe_id, list,
                                               "flux_poly_deg", p->poly_deg, 1, 89,
                                               "Efficiency computation fit polynomial degree.");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter poly_deg to the list");
    
	return (CPL_ERROR_NONE);
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the structure for CAL_FLUX input parameters
 @param    p	input CAL_FLUX parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_CAL_FLUX_delete(espdr_CAL_FLUX_param* p) {
	
    cpl_free((void *)p->background_sw);
    cpl_free((void *)p->extraction_method);
    cpl_free(p);
    p = NULL;
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Get the cal_flux recipe parameters
 @param     recipe_id   recipe ID
 @param     param_list  parameters list
 @param     CAL_FLUX_param  CAL_FLUX parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_CAL_FLUX_get(const char* recipe_id,
                                         cpl_parameterlist* param_list,
                                         espdr_CAL_FLUX_param *CAL_FLUX_param) {
	
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(param_list == NULL, CPL_ERROR_NULL_INPUT,
				 "Parameters list is NULL");
	
	/* Fill the structure */
    CAL_FLUX_param->background_sw = espdr_parameters_get_string(recipe_id,
                                                                param_list,
                                                                "background_sw");
    
    CAL_FLUX_param->bkgr_grid_size_x = espdr_parameters_get_int(recipe_id,
                                                                param_list,
                                                                "flux_bkgr_grid_size_x");
    
    CAL_FLUX_param->bkgr_grid_size_y = espdr_parameters_get_int(recipe_id,
                                                                param_list,
                                                                "flux_bkgr_grid_size_y");
    
    CAL_FLUX_param->extraction_method = espdr_parameters_get_string(recipe_id,
                                                                    param_list,
                                                                    "flux_extraction_method");
    
    CAL_FLUX_param->extraction_ksigma = espdr_parameters_get_double(recipe_id,
                                                                    param_list,
                                                                    "flux_extraction_ksigma");
    
    CAL_FLUX_param->poly_deg = espdr_parameters_get_int(recipe_id,
                                                        param_list,
                                                        "flux_poly_deg");
    
	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     print the CAL_FLUX parameters
 @param     CAL_FLUX_param  CAL_FLUX parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_CAL_FLUX_print(
				espdr_CAL_FLUX_param *CAL_FLUX_param) {
	
	espdr_msg("\tCAL_FLUX parameters:");
    espdr_msg("\t\tCAL_FLUX background subtraction = %s",
              CAL_FLUX_param->background_sw);
    espdr_msg("\t\tCAL_FLUX background grid size x = %d",
              CAL_FLUX_param->bkgr_grid_size_x);
    espdr_msg("\t\tCAL_FLUX background grid size y = %d",
              CAL_FLUX_param->bkgr_grid_size_y);
    espdr_msg("\t\tCAL_FLUX extraction method = %s",
              CAL_FLUX_param->extraction_method);
    espdr_msg("\t\tCAL_FLUX extraction ksigma = %.2f",
              CAL_FLUX_param->extraction_ksigma);
    espdr_msg("\t\tCAL_FLUX polynomial degree = %d",
              CAL_FLUX_param->poly_deg);

	return (CPL_ERROR_NONE);
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Init CAL_FLUX_param structure
 
 @param    recipe_id    recipe name
 @param    parameters   recipe input parameters
 
 @return   allocated structure iff OK
 */
/*----------------------------------------------------------------------------*/

espdr_CAL_FLUX_param * espdr_CAL_FLUX_param_init(const char *recipe_id, 
					 cpl_parameterlist *parameters) {
    
    espdr_CAL_FLUX_param *CAL_FLUX_param =
    (espdr_CAL_FLUX_param *)cpl_malloc(sizeof(espdr_CAL_FLUX_param));
    
    /* Read the cal_flux parameters */
	espdr_parameters_CAL_FLUX_get(recipe_id, parameters, CAL_FLUX_param);
	espdr_parameters_CAL_FLUX_print(CAL_FLUX_param);
    if(cpl_error_get_code() != CPL_ERROR_NONE) {
        return NULL;
    } else {
        return CAL_FLUX_param;
    }

}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_cal_flux_get_static_tables(cpl_frameset *frameset,
                                                cpl_frameset *used_frames,
                                                const char *star_name,
                                                cpl_table **std_star_table,
                                                cpl_table **extinction_table) {
    
    const char *table_filename = NULL;
    
    /* Extract standard stars table file frame */
    cpl_frame *std_star_table_frame = espdr_frame_find(frameset, ESPDR_STD_STARS_TABLE);
    cpl_frameset_insert(used_frames, cpl_frame_duplicate(std_star_table_frame));
    table_filename = cpl_frame_get_filename(std_star_table_frame);
    espdr_msg("Standard stars table : %s", table_filename);
    *std_star_table = cpl_table_load(table_filename, 1, 0);
    espdr_msg("Standard stars table rows number = %lld", cpl_table_get_nrow(*std_star_table));
    espdr_msg("Standard stars table cols number = %lld", cpl_table_get_ncol(*std_star_table));
    
    /* Check if star is on the standard stars table */
    double *std_star_flux = cpl_table_get_data_double(*std_star_table, star_name);
    
    if (std_star_flux == NULL) {
        espdr_msg_error("Selected star (%s) not found in the STD TABLE: ", star_name);
        cpl_array *star_names = cpl_table_get_column_names(*std_star_table);
        cpl_array_dump(star_names, 0, cpl_array_get_size(star_names), NULL);
        cpl_array_delete(star_names);
        return (CPL_ERROR_ILLEGAL_INPUT);
    }
    espdr_msg("Standard star: %s present in the STD TABLE: %s", star_name, table_filename);
    
    /* Extract extinction table file frame */
    cpl_frame *extinction_table_frame = espdr_frame_find(frameset, ESPDR_EXTINCTION_TABLE);
    cpl_frameset_insert(used_frames, cpl_frame_duplicate(extinction_table_frame));
    table_filename = cpl_frame_get_filename(extinction_table_frame);
    espdr_msg("Extinction table : %s", table_filename);
    *extinction_table = cpl_table_load(table_filename, 1, 0);
    espdr_msg("Extinction table rows number = %lld", cpl_table_get_nrow(*extinction_table));
    espdr_msg("Extinction table cols number = %lld", cpl_table_get_ncol(*extinction_table));
    
    return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/


cpl_error_code espdr_convert_photons_to_energy(double tel_surface,
					       double exp_time,
					       cpl_table **s1d_energy_RE) {

	double energy, energy_per_s_cm2_A, delta_lambda, flux, error, lambda;

	int table_length = cpl_table_get_nrow(*s1d_energy_RE);

	for (int i = 0; i < table_length-1; i++) {

		lambda = cpl_table_get(*s1d_energy_RE,"wavelength",i,NULL);
		
        //energy = H * C / 1e-10 / lambda * 1e7;
        energy = PLANCK_CONSTANT * LIGHT_SPEED / 1e-10 / lambda * 1e7;

		delta_lambda = 
			cpl_table_get(*s1d_energy_RE,"wavelength",i+1,NULL)
		        - cpl_table_get(*s1d_energy_RE,"wavelength",i,NULL);

		energy_per_s_cm2_A = energy / exp_time / (tel_surface * 1e4) / delta_lambda;

		flux = cpl_table_get(*s1d_energy_RE,"flux",i,NULL);
		error = cpl_table_get(*s1d_energy_RE,"error",i,NULL);

		cpl_table_set(*s1d_energy_RE,"flux",i,flux*energy_per_s_cm2_A);

		cpl_table_set(*s1d_energy_RE,"error",i,error*energy_per_s_cm2_A);

	}
    
    /* Last value */
    cpl_table_set(*s1d_energy_RE, "flux",table_length-1,
                  cpl_table_get(*s1d_energy_RE,"flux",table_length-2,NULL));

    cpl_table_set(*s1d_energy_RE, "error",table_length-1,
                  cpl_table_get(*s1d_energy_RE,"error",table_length-2,NULL));

	return cpl_error_get_code();
}	


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_load_standard_stars_table(cpl_table *standard_star_table,
					       const char* star,
					       cpl_table **selected_star_RE) {
    
	double *std_star_flux = cpl_table_get_data_double(standard_star_table,star);
	double *std_wavelength = cpl_table_get_data_double(standard_star_table,"wavelength");
    int std_table_length = cpl_table_get_nrow(standard_star_table);
    *selected_star_RE = cpl_table_new(std_table_length);
    cpl_table_new_column(*selected_star_RE,"wavelength",CPL_TYPE_DOUBLE);
    cpl_table_new_column(*selected_star_RE,"flux",CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(*selected_star_RE, "wavelength", "[angstrom]");
    cpl_table_set_column_unit(*selected_star_RE, "flux", spectrum_flux_unit);

    for (int i = 0; i < std_table_length ; i++) {
        cpl_table_set_double(*selected_star_RE, "wavelength", i, std_wavelength[i]);
        cpl_table_set_double(*selected_star_RE, "flux", i, std_star_flux[i]);
    }
    
	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_measure_standard_flux(cpl_table *s1d_energy,
                                           cpl_table *selected_star,
                                           cpl_table **measured_eff_flux_RE) {

    int s1d_energy_length = cpl_table_get_nrow(s1d_energy);
    int std_star_length = cpl_table_get_nrow(selected_star);
    
	int    index_start, index_end, count_flux, quality_flag;
	double flux_energy, sum_flux_energy, avg_measured_flux, bound_ini,
	       bound_end, std_flux, std_wave;

    *measured_eff_flux_RE = cpl_table_new(std_star_length);
    cpl_table_new_column(*measured_eff_flux_RE,"wavelength",CPL_TYPE_DOUBLE);
    cpl_table_new_column(*measured_eff_flux_RE,"std_flux",CPL_TYPE_DOUBLE);
    cpl_table_new_column(*measured_eff_flux_RE,"flux_erg_avg",CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(*measured_eff_flux_RE,"wavelength","[angstrom]");
    cpl_table_set_column_unit(*measured_eff_flux_RE,"std_flux", spectrum_flux_unit);
    cpl_table_set_column_unit(*measured_eff_flux_RE,"flux_erg_avg", spectrum_flux_unit);
    
	/* Loop Lambda 1, std star table*/
	double central_wave;
    
	/* filling out the first and last elements */
	std_flux  = cpl_table_get(selected_star,"flux",0,NULL);
	std_wave = cpl_table_get(selected_star,"wavelength",0,NULL);
	cpl_table_set_double(*measured_eff_flux_RE,"wavelength",0,std_wave);
	cpl_table_set_double(*measured_eff_flux_RE,"std_flux",0,std_flux);
	std_flux  = cpl_table_get(selected_star,"flux",std_star_length-1,NULL);
	std_wave = cpl_table_get(selected_star,"wavelength",std_star_length-1,NULL);
	cpl_table_set_double(*measured_eff_flux_RE,"wavelength",std_star_length-1,std_wave);
	cpl_table_set_double(*measured_eff_flux_RE,"std_flux",std_star_length-1,std_flux);

 	for (int l1 = 1; l1 < std_star_length-1; l1++) {  /* Discarding first and last elements */
		index_start = 0;
		index_end = s1d_energy_length-1;

		std_flux  = cpl_table_get(selected_star,"flux",l1,NULL);
		bound_ini = cpl_table_get(selected_star,"wavelength",l1-1,NULL);
		bound_end = cpl_table_get(selected_star,"wavelength",l1+1,NULL);
		central_wave = cpl_table_get(selected_star,"wavelength",l1,NULL);

		bound_ini = (central_wave + bound_ini)/2;	
		bound_end = (central_wave + bound_end)/2;

		cpl_table_set_double(*measured_eff_flux_RE,"wavelength",l1,central_wave);
		cpl_table_set_double(*measured_eff_flux_RE,"std_flux",l1,std_flux);

		/* Initial bound at lambda 2, s1d erg converted wavelength */
		while ((index_start < s1d_energy_length) &&
               (cpl_table_get(s1d_energy,"wavelength",index_start,NULL) <= bound_ini))
		index_start++;

		/* End bound at lambda 2 */
		while ((index_end >= 0) &&
               (cpl_table_get(s1d_energy,"wavelength",index_end,NULL) >= bound_end))
		index_end--;
			
		/* Computing the flux erg converted average on the boundary */
		if (index_start < index_end) {
			sum_flux_energy = 0.;
			count_flux = 0;
			for (int x = index_start; x <=index_end; x++) {
                flux_energy = cpl_table_get(s1d_energy,"flux",x,NULL);
                quality_flag = cpl_table_get(s1d_energy,"quality",x,NULL);
                    if (!isnan(flux_energy) && (quality_flag!=OTHER_BAD_PIXEL)) {
                        sum_flux_energy = sum_flux_energy+(flux_energy);
                        count_flux++;
                    }
                }

			if (count_flux > 0) {
				avg_measured_flux = sum_flux_energy / count_flux;
			} else {
				avg_measured_flux = 0.;
			}
            
            cpl_table_set_double(*measured_eff_flux_RE,
                                "flux_erg_avg",l1,avg_measured_flux);
		}
	}		

	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_pre_extinction_table_interpolation(cpl_table *ext_table,
					   cpl_table *std_table, 
					   cpl_table **inter_extinction_RE) {

	cpl_error_code my_error = CPL_ERROR_NONE;

    int std_table_length = cpl_table_get_nrow(std_table);
    int ext_table_length = cpl_table_get_nrow(ext_table);
    
    *inter_extinction_RE = cpl_table_new(std_table_length);
    cpl_table_new_column(*inter_extinction_RE,
                         "wavelength",CPL_TYPE_DOUBLE);
    cpl_table_new_column(*inter_extinction_RE,
                         "extinction_interp",CPL_TYPE_DOUBLE);
    my_error = cpl_table_set_column_unit(*inter_extinction_RE,
                                         "wavelength",
                                         "[angstrom]");
    my_error = cpl_table_set_column_unit(*inter_extinction_RE,
                                         "extinction_interp",
                                         "[mag/airmass]");

	double *y2 = (double *)cpl_calloc(std_table_length,sizeof(double));
	my_error = espdr_spline_bounded_interpolation(
			cpl_table_get_data_double(ext_table,"wavelength"),
			cpl_table_get_data_double(ext_table,"extinction"),
			ext_table_length,
			cpl_table_get_data_double(std_table,"wavelength"),
			y2,
			std_table_length,
			cpl_table_get(std_table,
					  "wavelength",
					   0,
					   NULL),
			cpl_table_get(std_table,
					  "wavelength",
					   std_table_length-1,
					   NULL));

	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
		     "espdr_spline_bounded_interpolation failed: %s",
		     cpl_error_get_message_default(my_error));

    for (int j = 0; j < std_table_length; j++) {
        
        cpl_table_set_double(*inter_extinction_RE,
                             "wavelength",j,
                             cpl_table_get(std_table,
                                           "wavelength",
                                           j,
                                           NULL));
        
        cpl_table_set_double(*inter_extinction_RE,
                             "extinction_interp",j,
                             y2[j]);
    }
    
	cpl_free(y2);

	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_correct_extinction(cpl_table *avg_flux,
                                        cpl_table *ext_table,
                                        double airmass,
                                        cpl_table **corrected_flux_RE) {

    int avg_flux_length = cpl_table_get_nrow(avg_flux);
    double ratio,magnitude;
    
    *corrected_flux_RE = cpl_table_new(avg_flux_length);
    cpl_table_new_column(*corrected_flux_RE,
                         "wavelength", CPL_TYPE_DOUBLE);
    cpl_table_new_column(*corrected_flux_RE,
                         "extcorr_flux", CPL_TYPE_DOUBLE);
    cpl_table_new_column(*corrected_flux_RE,
                         "raw_flux", CPL_TYPE_DOUBLE);
    cpl_table_new_column(*corrected_flux_RE,
                         "std_flux", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(*corrected_flux_RE,
                                         "wavelength",
                                         "[angstrom]");
    cpl_table_set_column_unit(*corrected_flux_RE,
                                         "extcorr_flux", spectrum_flux_unit);
                                         
    cpl_table_set_column_unit(*corrected_flux_RE,
                                         "raw_flux", spectrum_flux_unit);
                                        
    cpl_table_set_column_unit(*corrected_flux_RE,
                                         "std_flux", spectrum_flux_unit);
                                         

    for (int j = 0; j < avg_flux_length; j++) {
        
        magnitude = cpl_table_get(ext_table,"extinction_interp",j,NULL)*
                    (airmass); /* Airmass removed */
        
        ratio = pow(10,magnitude/2.5);
        
        cpl_table_set_double(*corrected_flux_RE,
                             "wavelength", j,
                             cpl_table_get(ext_table,"wavelength",
                                           j,NULL));
        
        cpl_table_set_double(*corrected_flux_RE,
                             "extcorr_flux", j,
                             cpl_table_get(avg_flux,"flux_erg_avg",
                                           //j,NULL)/ratio);
                                           j,NULL)*ratio);
        
        cpl_table_set_double(*corrected_flux_RE,
                             "raw_flux", j,
                             cpl_table_get(avg_flux,"flux_erg_avg",
                                           j,NULL));
        
        cpl_table_set_double(*corrected_flux_RE,
                             "std_flux", j,
                             cpl_table_get(avg_flux,"std_flux",
                                           j,NULL));
        
    }
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_correct_slit(cpl_table *extinction_corrected_flux,
                                  double seeing,
                                  espdr_inst_config *inst_config,
                                  cpl_table **slit_corrected_flux_RE) {
    
    int corr_flux_length = cpl_table_get_nrow(extinction_corrected_flux);
    int i;
    double wavelength, iq, slit, ext_corr_flux, flux;
    double fibre_size, vltsh_ll, vltsh_iq, inst_iq;
    
    *slit_corrected_flux_RE = cpl_table_new(corr_flux_length);
    cpl_table_new_column(*slit_corrected_flux_RE,
                         "wavelength", CPL_TYPE_DOUBLE);
    cpl_table_new_column(*slit_corrected_flux_RE,
                         "slitcorr_flux", CPL_TYPE_DOUBLE);
	cpl_table_new_column(*slit_corrected_flux_RE,
						 "extcorr_flux", CPL_TYPE_DOUBLE);
    cpl_table_new_column(*slit_corrected_flux_RE,
                         "raw_flux", CPL_TYPE_DOUBLE);
    cpl_table_new_column(*slit_corrected_flux_RE,
                         "std_flux", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(*slit_corrected_flux_RE,
                                         "wavelength",
                                         "[angstrom]");
    cpl_table_set_column_unit(*slit_corrected_flux_RE,
                                         "slitcorr_flux", spectrum_flux_unit);
                                         
    cpl_table_set_column_unit(*slit_corrected_flux_RE,
                                         "extcorr_flux", spectrum_flux_unit);
                                        
    cpl_table_set_column_unit(*slit_corrected_flux_RE,
                                         "raw_flux", spectrum_flux_unit);
                                        
    cpl_table_set_column_unit(*slit_corrected_flux_RE,
                                         "std_flux", spectrum_flux_unit);
                                        

    fibre_size = inst_config->fibre_size;
    vltsh_ll = inst_config->vltsh_wavelength;
    vltsh_iq = inst_config->vltsh_iq;
    inst_iq = inst_config->inst_iq;
    
    //espdr_msg("Seeing: %f, vltsh_iq: %f, vltsh_ll: %f, fibre_size: %f, inst_iq: %f",
    //          seeing, vltsh_iq, vltsh_ll, fibre_size, inst_iq);
    
    for (i = 0; i < corr_flux_length; i++) {
        wavelength = cpl_table_get(extinction_corrected_flux,
                                   "wavelength", i, NULL);
        cpl_table_set_double(*slit_corrected_flux_RE,
                             "wavelength", i, wavelength);
        
        cpl_table_set_double(*slit_corrected_flux_RE,
                             "raw_flux", i,
                             cpl_table_get(extinction_corrected_flux,
                                           "raw_flux", i, NULL));
        
        cpl_table_set_double(*slit_corrected_flux_RE,
                             "std_flux", i,
                             cpl_table_get(extinction_corrected_flux,
                                           "std_flux", i, NULL));
        
		cpl_table_set_double(*slit_corrected_flux_RE,
							 "extcorr_flux", i,
							 cpl_table_get(extinction_corrected_flux,
										   "extcorr_flux", i, NULL));
        
        if (strcmp(inst_config->instrument, "NIRPS") == 0) {
            if (inst_config->fibre_size == 0.4) {
                slit = 0.6;
            } else {
                slit = 0.75;
            }
        } else {
            iq = sqrt(seeing*seeing - vltsh_iq*vltsh_iq) * pow((wavelength/vltsh_ll), -0.2);
            slit = 1.0 - exp(-0.53 * ((fibre_size * fibre_size)/(iq * iq + inst_iq * inst_iq)));
        }
        ext_corr_flux = cpl_table_get(extinction_corrected_flux,
                                      "extcorr_flux", i, NULL);
        flux = ext_corr_flux / slit;
        cpl_table_set_double(*slit_corrected_flux_RE,
                             "slitcorr_flux", i, flux);
    }
    
    // new corrected_flux = corrected_flux / slit
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_discard_wavelength_region(double bound_ini,
                                         double bound_end,
                                         cpl_table **discarded_wave_flux_RE) {

    int table_length = cpl_table_get_nrow(*discarded_wave_flux_RE);
    
    int index_start = 0;
    int index_end = table_length-1;
    
    /*Initial bound */
    while ((index_start < table_length) &&
           (cpl_table_get(*discarded_wave_flux_RE,"wavelength",index_start,NULL) <=
            bound_ini))
        index_start++;
    
	/*End bound */
	index_end = table_length-1;
	while ((index_end >= 0) &&
	(cpl_table_get(*discarded_wave_flux_RE,"wavelength",index_end,NULL) >=
	bound_end))
	index_end--;	

	espdr_msg("Initial table lenght %d",table_length);
	espdr_msg("%d rows discarded, for wavelength region between %f -> %f",
	index_end-index_start,bound_ini,bound_end);

	cpl_table_erase_window(*discarded_wave_flux_RE,
				index_start,
				index_end-index_start);

	int table_length_reduced = cpl_table_get_nrow(*discarded_wave_flux_RE);
	espdr_msg("Table length after wavelength region discarded %d",
	table_length_reduced);

	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_compute_efficiency(cpl_table *std_star,
                                        cpl_table *avg_flux_corrected,
                                        cpl_table **efficiency_RE) {

    int std_length = cpl_table_get_nrow(std_star);
    
    *efficiency_RE = cpl_table_new(std_length);
    cpl_table_new_column(*efficiency_RE,"wavelength",CPL_TYPE_DOUBLE);
    cpl_table_new_column(*efficiency_RE,"efficiency",CPL_TYPE_DOUBLE);
	cpl_table_new_column(*efficiency_RE,"extcorr_efficiency",CPL_TYPE_DOUBLE);
    cpl_table_new_column(*efficiency_RE,"raw_efficiency",CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(*efficiency_RE, "wavelength", "[angstrom]");
    cpl_table_set_column_unit(*efficiency_RE, "efficiency",  "");
    cpl_table_set_column_unit(*efficiency_RE, "extcorr_efficiency", "");
    cpl_table_set_column_unit(*efficiency_RE, "raw_efficiency", "");

    for (int j = 0; j < std_length; j++) {
        cpl_table_set_double(*efficiency_RE,
                             "wavelength", j,
                             cpl_table_get(avg_flux_corrected,
                                           "wavelength",j,NULL));
        
        cpl_table_set_double(*efficiency_RE,
                             "efficiency", j,
                             cpl_table_get(avg_flux_corrected,
                                           "slitcorr_flux",j,NULL)/
                             cpl_table_get(std_star,"flux",j,NULL));

		cpl_table_set_double(*efficiency_RE,
							 "extcorr_efficiency", j,
							 cpl_table_get(avg_flux_corrected,
										   "extcorr_flux",j,NULL)/
							 cpl_table_get(std_star,"flux",j,NULL));
		
		cpl_table_set_double(*efficiency_RE,
                             "raw_efficiency", j,
                             cpl_table_get(avg_flux_corrected,
                                           "raw_flux",j,NULL)/
                             cpl_table_get(std_star,"flux",j,NULL));
    }
    
	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_interpolated_eff(cpl_table *eff_table,
				      cpl_table *s1d_table, 
				      cpl_table **s1d_eff_RE) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    
    int s1d_length = cpl_table_get_nrow(s1d_table);
    int eff_length = cpl_table_get_nrow(eff_table);
    
    double ini_eff, end_eff;
    
    ini_eff = cpl_table_get(eff_table,"wavelength",0,NULL);
    end_eff = cpl_table_get(eff_table,"wavelength",eff_length-1,NULL);
    
	/* Discarding wavelength regions across efficiency avg length ...*/
    if((region_1_bound_ini >= ini_eff) && (region_1_bound_end <= end_eff)) {
        my_error = espdr_discard_wavelength_region(region_1_bound_ini,
                                                   region_1_bound_end,
                                                   &eff_table);
    }
    
    if((region_2_bound_ini >= ini_eff) && (region_2_bound_end <= end_eff)) {
        my_error = espdr_discard_wavelength_region(region_2_bound_ini,
                                                   region_2_bound_end,
                                                   &eff_table);
    }
    
    if((region_3_bound_ini >= ini_eff) && (region_3_bound_end <= end_eff)) {
        my_error = espdr_discard_wavelength_region(region_3_bound_ini,
                                                   region_3_bound_end,
                                                   &eff_table);
    }
    
    if((region_4_bound_ini >= ini_eff) && (region_4_bound_end <= end_eff)) {
        my_error = espdr_discard_wavelength_region(region_4_bound_ini,
                                                   region_4_bound_end,
                                                   &eff_table);
    }
    
    if((region_5_bound_ini >= ini_eff) && (region_5_bound_end <= end_eff)) {
        my_error = espdr_discard_wavelength_region(region_5_bound_ini,
                                                   region_5_bound_end,
                                                   &eff_table);
    }
    
    if((region_6_bound_ini >= ini_eff) && (region_6_bound_end <= end_eff)) {
        my_error = espdr_discard_wavelength_region(region_6_bound_ini,
                                                   region_6_bound_end,
                                                   &eff_table);
    }
    
    if((region_7_bound_ini >= ini_eff) && (region_7_bound_end <= end_eff)) {
        my_error = espdr_discard_wavelength_region(region_7_bound_ini,
                                                   region_7_bound_end,
                                                   &eff_table);
    }
    
    if((region_8_bound_ini >= ini_eff) && (region_8_bound_end <= end_eff)) {
        my_error = espdr_discard_wavelength_region(region_8_bound_ini,
                                                   region_8_bound_end,
                                                   &eff_table);
    }
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_discard_wavelength_region failed: %s",
                 cpl_error_get_message_default(my_error));
    
    eff_length = cpl_table_get_nrow(eff_table);
    
	cpl_table *eff_table_no_zeros = cpl_table_duplicate(eff_table);
	cpl_table_unselect_all(eff_table_no_zeros);

	double flux_zero;
	
    for (int z = 0; z < eff_length; z++) {
        
        flux_zero = cpl_table_get(eff_table_no_zeros,
                                  "efficiency",z,NULL);
        
        if(flux_zero == 0.) {
            cpl_table_select_row(eff_table_no_zeros,z);
        }
    }

	cpl_table_erase_selected(eff_table_no_zeros);

    int eff_length_no_zeros = cpl_table_get_nrow(eff_table_no_zeros);
    
    *s1d_eff_RE = cpl_table_new(s1d_length);
    cpl_table_new_column(*s1d_eff_RE, "wavelength",
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(*s1d_eff_RE, "efficiency_interpolated",
                         CPL_TYPE_DOUBLE);
    my_error = cpl_table_set_column_unit(*s1d_eff_RE,
                                         "wavelength",
                                         "[angstrom]");
    my_error = cpl_table_set_column_unit(*s1d_eff_RE,
                                         "efficiency_interpolated","");

	double *y2 = (double *) cpl_calloc(s1d_length,sizeof(double));

	my_error = espdr_spline_bounded_interpolation(
		cpl_table_get_data_double(eff_table_no_zeros,"wavelength"),
		cpl_table_get_data_double(eff_table_no_zeros,"efficiency"),
		eff_length_no_zeros,
		cpl_table_get_data_double(s1d_table,"wavelength"),
		y2,
		s1d_length,
		cpl_table_get(s1d_table,	/*xmin*/
                      "wavelength",0,NULL),
		cpl_table_get(s1d_table,	/*xmax*/
                      "wavelength",s1d_length-1,NULL));

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
             "espdr_spline_interpolation failed: %s",
             cpl_error_get_message_default(my_error));

	/* Interpolating efficiency across S1D length ...*/
    for (int j = 0; j < s1d_length; j++) {
        cpl_table_set_double(*s1d_eff_RE,"wavelength",j,
                             cpl_table_get(s1d_table,
                                           "wavelength",j,NULL));
        cpl_table_set_double(*s1d_eff_RE,
                             "efficiency_interpolated",j,y2[j]);
    }

    cpl_free(y2);
    cpl_table_delete(eff_table_no_zeros);

	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_polynomial_eff(int pol_deg,
                                    cpl_table *eff_table,
                                    cpl_table *s1d_table,
                                    cpl_table **s1d_eff_RE) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    
    int s1d_length = cpl_table_get_nrow(s1d_table);
    int eff_length = cpl_table_get_nrow(eff_table);
    
	int ini_eff, end_eff;

	ini_eff = cpl_table_get(eff_table,"wavelength",0,NULL);
	end_eff = cpl_table_get(eff_table,"wavelength",eff_length-1,NULL);

    /* Discarding wavelength regions across efficiency avg length ...*/

	if((region_1_bound_ini >= ini_eff) && (region_1_bound_end <= end_eff)) {
        my_error = espdr_discard_wavelength_region(region_1_bound_ini,
                                               region_1_bound_end,
                                               &eff_table);
	}

	if((region_2_bound_ini >= ini_eff) && (region_2_bound_end <= end_eff)) {
        my_error = espdr_discard_wavelength_region(region_2_bound_ini,
                                                   region_2_bound_end,
                                                   &eff_table);
	}

	if((region_3_bound_ini >= ini_eff) && (region_3_bound_end <= end_eff)) {
        my_error = espdr_discard_wavelength_region(region_3_bound_ini,
                                                   region_3_bound_end,
                                                   &eff_table);
	}

	if((region_4_bound_ini >= ini_eff) && (region_4_bound_end <= end_eff)) {
        my_error = espdr_discard_wavelength_region(region_4_bound_ini,
                                                   region_4_bound_end,
                                                   &eff_table);
	}

    if((region_5_bound_ini >= ini_eff) && (region_5_bound_end <= end_eff)) {
        my_error = espdr_discard_wavelength_region(region_5_bound_ini,
                                                   region_5_bound_end,
                                                   &eff_table);
    }
    
    if((region_6_bound_ini >= ini_eff) && (region_6_bound_end <= end_eff)) {
        my_error = espdr_discard_wavelength_region(region_6_bound_ini,
                                                   region_6_bound_end,
                                                   &eff_table);
    }
    
    if((region_7_bound_ini >= ini_eff) && (region_7_bound_end <= end_eff)) {
        my_error = espdr_discard_wavelength_region(region_7_bound_ini,
                                                   region_7_bound_end,
                                                   &eff_table);
    }
    
    if((region_8_bound_ini >= ini_eff) && (region_8_bound_end <= end_eff)) {
        my_error = espdr_discard_wavelength_region(region_8_bound_ini,
                                                   region_8_bound_end,
                                                   &eff_table);
    }
    
    eff_length = cpl_table_get_nrow(eff_table);
    
    cpl_table *eff_table_no_zeros = cpl_table_duplicate(eff_table);
    cpl_table_unselect_all(eff_table_no_zeros);
    
    double flux_zero;
    
    for (int z = 0; z < eff_length; z++) {
        flux_zero = cpl_table_get(eff_table_no_zeros,
                                  "efficiency",z,NULL);
        if(flux_zero == 0.) {
            cpl_table_select_row(eff_table_no_zeros,z);
        }
    }
    
    cpl_table_erase_selected(eff_table_no_zeros);
    
	int eff_length_no_zeros = cpl_table_get_nrow(eff_table_no_zeros);

    *s1d_eff_RE = cpl_table_new(s1d_length);
    cpl_table_new_column(*s1d_eff_RE,"wavelength",
                         CPL_TYPE_DOUBLE);
    cpl_table_new_column(*s1d_eff_RE,"efficiency_interpolated",
                         CPL_TYPE_DOUBLE);
    my_error = cpl_table_set_column_unit(*s1d_eff_RE,
                                         "wavelength","[angstrom]");
    my_error = cpl_table_set_column_unit(*s1d_eff_RE,
                                         "efficiency_interpolated","");

    espdr_poly_data *p_data = (espdr_poly_data *)cpl_malloc(sizeof(espdr_poly_data));
    
    p_data->x = (double *) cpl_calloc(eff_length_no_zeros,sizeof(double));
    p_data->y = (double *) cpl_calloc(eff_length_no_zeros,sizeof(double));
	p_data->err = (double *) cpl_calloc(eff_length_no_zeros,sizeof(double));

    double *fit = NULL;
    double *coeffs = NULL;
    double *coeffs_err = NULL;
	double chisq = 0.0;

	for (int i = 0; i < eff_length_no_zeros; i++) {
		p_data->x[i] = cpl_table_get(eff_table_no_zeros,"wavelength",i,NULL);
		p_data->y[i] = cpl_table_get(eff_table_no_zeros,"efficiency",i,NULL);
		p_data->err[i] = 1.;
	}
    
    p_data->n = eff_length_no_zeros;
    
    fit = (double *) cpl_calloc (eff_length_no_zeros , sizeof(double));
    coeffs = (double *) cpl_calloc (pol_deg , sizeof(double));
    coeffs_err = (double *) cpl_calloc (pol_deg , sizeof(double));
    
	my_error = espdr_fit_poly(p_data, pol_deg, fit,
                              coeffs, coeffs_err, &chisq, 1);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
		     "espdr_fit_poly failed: %s",
		     cpl_error_get_message_default(my_error));

	double y2, x2;

    for (int j = 0; j < s1d_length; j++) {
        x2 = cpl_table_get(s1d_table, "wavelength", j, NULL);
        cpl_table_set_double(*s1d_eff_RE, "wavelength", j, x2);
        y2 = 0.;
        for (int c = pol_deg-1; c >= 0; c--) {
            y2 = y2 * x2 + coeffs[c];
        }
        cpl_table_set_double(*s1d_eff_RE,"efficiency_interpolated",j,y2);
    }
    
    cpl_free(fit);
    cpl_free(coeffs);
    cpl_free(coeffs_err);
    cpl_free(p_data);
    
	cpl_table_delete(eff_table_no_zeros);
	return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 @brief    Check the cal_flux QC
 @param    CCD_geom         CCD geometry
 @param    exposure_time    exposure time
 @param    inst_config      instrument configuration structure
 @param    qc_kws           structure holdin quality control, keys
 @param    keywords_RE      fits header to hold QC
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_cal_flux_QC(espdr_CCD_geometry *CCD_geom,
                                 espdr_inst_config *inst_config,
                                 espdr_qc_keywords *qc_kws,
                                 cpl_table *raw_abs_eff,
                                 cpl_table *abs_eff,
                                 double berv,
                                 double bjd,
                                 double bervmax,
                                 double berv_factor,
                                 int *cosmics_nb,
                                 double *snr,
                                 int orders_nr,
                                 int seeing_kw_qc,
                                 cpl_propertylist **keywords_RE) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    
    int i;
    char *new_keyword;
    
    int saturation_QC = 1;
    int snr_QC = 1;
    int global_QC = 1;
    int abs_eff_min_QC = 1;
    int abs_eff_max_QC = 1;
    
    double abs_eff_min = 0.0;
    double abs_eff_max = 0.0;
    double abs_eff_mean = 0.0;
    double raw_abs_eff_min = 0.0;
    double raw_abs_eff_max = 0.0;
    double raw_abs_eff_mean = 0.0;
	double extcorr_abs_eff_min = 0.0;
	double extcorr_abs_eff_max = 0.0;
	double extcorr_abs_eff_mean = 0.0;

    espdr_ensure(abs_eff == NULL, CPL_ERROR_NULL_INPUT,
                 "espdr_cal_flux_QC failed: NULL absolute efficiency table");
    
    saturation_QC = cpl_propertylist_get_int(*keywords_RE, qc_kws->qc_saturation_check_kw);
    if (saturation_QC == 0) {
        global_QC = 0;
    }
    
    /* QC BERV KWs */
    
	my_error = espdr_keyword_add_double(qc_kws->qc_sci_red_berv,
					berv,"Barycentric correction [km/s]",keywords_RE);

	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
		 "Add keyword QC_BERV to the propertylist failed: %s",
		 cpl_error_get_message_default(my_error));

	my_error = espdr_keyword_add_double(qc_kws->qc_sci_red_bjd,
					bjd,"Barycentric Julian date (TDB) [JD]",keywords_RE);

	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
		 "Add keyword QC_BJD to the propertylist failed: %s",
		 cpl_error_get_message_default(my_error));

	my_error = espdr_keyword_add_double(qc_kws->qc_sci_red_bervmax,
					bervmax,"Barycentric max [km/s]",keywords_RE);

	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
		 "Add keyword QC_BERVMAX to the propertylist failed: %s",
		 cpl_error_get_message_default(my_error));
    
    my_error = espdr_keyword_add_double(qc_kws->qc_sci_red_berv_factor,
                    berv_factor,"BERV factor",keywords_RE);

    /* Cosmics KWs */
    
    for (i = 0; i < orders_nr; i++) {
        new_keyword =
        espdr_add_output_index_to_keyword(qc_kws->qc_cal_flux_order_cosmic_nb_kw_first,
                                          qc_kws->qc_cal_flux_order_cosmic_nb_kw_last,
                                          i+1);
        
        my_error = espdr_keyword_add_int(new_keyword,
                                         cosmics_nb[i],
                                         "Cosmics number in the order",
                                         keywords_RE);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_ORDER_COSMIC_NB_KW to the propertylist failed: %s",
                     cpl_error_get_message_default(my_error));
        
        cpl_free(new_keyword);
    }
    
    // espdr_msg("Saving SNR KWs");
    for (i = 1; i <= orders_nr; i++) {
        
        new_keyword = espdr_add_output_index_to_keyword(qc_kws->qc_cal_flux_order_snr_kw_first,
                                                        qc_kws->qc_cal_flux_order_snr_kw_last,
                                                        i);
        my_error = espdr_keyword_add_double(new_keyword, snr[i-1],
                                            "SNR in the order", keywords_RE);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_ORDER_SNR_KW to the propertylist failed: %s",
                     cpl_error_get_message_default(my_error));
        
        cpl_free(new_keyword);
        
    }
    
    cpl_vector *snr_vector = cpl_vector_wrap(orders_nr, snr);
    double snr_median = cpl_vector_get_median(snr_vector);
    espdr_msg("SNR median: %f, SNR limit: %.2f", snr_median, inst_config->flux_snr_limit);
    my_error = espdr_keyword_add_double(qc_kws->qc_cal_flux_snr_median_kw, snr_median,
                                        "median SNR", keywords_RE);
    
    if (snr_median < inst_config->flux_snr_limit) {
        snr_QC = 0;
    }
    
    my_error = espdr_keyword_add_int(qc_kws->qc_cal_flux_snr_check_kw,
                                     snr_QC,"SNR QC",keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_CALFLUX_SNR_CHECK_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    /* RAW and EXTCORR efficiency KWs */
    
    int eff_length = cpl_table_get_nrow(raw_abs_eff);
    
    cpl_table *raw_abs_eff_no_zeros = cpl_table_duplicate(raw_abs_eff);
    cpl_table_unselect_all(raw_abs_eff_no_zeros);
    
    double eff_zero;
    for (int z = 0; z < eff_length; z++) {
        eff_zero = cpl_table_get(raw_abs_eff_no_zeros,
                                  "raw_efficiency",z,NULL);
        if (eff_zero == 0.0) {
            cpl_table_select_row(raw_abs_eff_no_zeros, z);
        }
    }
    
    cpl_table_erase_selected(raw_abs_eff_no_zeros);
    
    raw_abs_eff_min = cpl_table_get_column_min(raw_abs_eff_no_zeros,
                                               "raw_efficiency");
    my_error = espdr_keyword_add_double(qc_kws->qc_cal_flux_raw_abs_eff_min_kw,
                                        raw_abs_eff_min,
                                        "Minimum raw absolute efficiency value",
                                        keywords_RE);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_RAW_ABS_EFF_MIN to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    raw_abs_eff_max = cpl_table_get_column_max(raw_abs_eff_no_zeros,
                                               "raw_efficiency");
    my_error = espdr_keyword_add_double(qc_kws->qc_cal_flux_raw_abs_eff_max_kw,
                                        raw_abs_eff_max,
                                        "Maximum raw absolute efficiency value",
                                        keywords_RE);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_RAW_ABS_EFF_MAX to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    raw_abs_eff_mean = cpl_table_get_column_mean(raw_abs_eff_no_zeros,
                                                "raw_efficiency");
    my_error = espdr_keyword_add_double(qc_kws->qc_cal_flux_raw_abs_eff_mean_kw,
                                        raw_abs_eff_mean,
                                        "Mean raw absolute efficiency value",
                                        keywords_RE);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_RAW_ABS_EFF_MEAN to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));

	extcorr_abs_eff_min = cpl_table_get_column_min(raw_abs_eff_no_zeros,
											   "extcorr_efficiency");
	my_error = espdr_keyword_add_double(qc_kws->qc_cal_flux_extcorr_abs_eff_min_kw,
										extcorr_abs_eff_min,
										"Minimum extinction-corrected absolute efficiency value",
										keywords_RE);
	
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
				 "Add keyword QC_EXTCORR_ABS_EFF_MIN to the propertylist failed: %s",
				 cpl_error_get_message_default(my_error));
	
	extcorr_abs_eff_max = cpl_table_get_column_max(raw_abs_eff_no_zeros,
											   "extcorr_efficiency");
	my_error = espdr_keyword_add_double(qc_kws->qc_cal_flux_extcorr_abs_eff_max_kw,
										extcorr_abs_eff_max,
										"Maximum extinction-corrected absolute efficiency value",
										keywords_RE);
	
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
				 "Add keyword QC_EXTCORR_ABS_EFF_MAX to the propertylist failed: %s",
				 cpl_error_get_message_default(my_error));
	
	extcorr_abs_eff_mean = cpl_table_get_column_mean(raw_abs_eff_no_zeros,
												 "extcorr_efficiency");
	my_error = espdr_keyword_add_double(qc_kws->qc_cal_flux_extcorr_abs_eff_mean_kw,
										extcorr_abs_eff_mean,
										"Mean extinction-corrected absolute efficiency value",
										keywords_RE);
	
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
				 "Add keyword QC_EXTCORR_ABS_EFF_MEAN to the propertylist failed: %s",
				 cpl_error_get_message_default(my_error));

	cpl_table_delete(raw_abs_eff_no_zeros);
    
    /* Efficiency KWs */
    
    abs_eff_min = cpl_table_get_column_min(abs_eff,
                                           "efficiency_interpolated");
    my_error = espdr_keyword_add_double(qc_kws->qc_cal_flux_abs_eff_min_kw,
                                        abs_eff_min,
                                        "Minimum absolute efficiency value",
                                        keywords_RE);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_ABS_EFF_MIN to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    if (abs_eff_min < inst_config->abs_eff_min_limit) {
        abs_eff_min_QC = 0;
        global_QC = 0;
    }
    
    my_error = espdr_keyword_add_int(qc_kws->qc_cal_flux_abs_eff_min_check_kw,
                                     abs_eff_min_QC,
                                     "Min absolute efficiency QC",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_MIN_ABS_EFF_CHECK to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));

    abs_eff_max = cpl_table_get_column_max(abs_eff,
                                           "efficiency_interpolated");
    my_error = espdr_keyword_add_double(qc_kws->qc_cal_flux_abs_eff_max_kw,
                                        abs_eff_max,
                                        "Maximun absolute efficiency value",
                                        keywords_RE);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_ABS_EFF_MAX to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    if (abs_eff_max < inst_config->abs_eff_max_limit) {
        abs_eff_max_QC = 0;
        global_QC = 0;
    }
    
    my_error = espdr_keyword_add_int(qc_kws->qc_cal_flux_abs_eff_max_check_kw,
                                     abs_eff_max_QC,
                                     "Max absolute efficiency QC",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_MAX_ABS_EFF_CHECK to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    abs_eff_mean = cpl_table_get_column_mean(abs_eff,
                                             "efficiency_interpolated");
    my_error = espdr_keyword_add_double(qc_kws->qc_cal_flux_abs_eff_mean_kw,
                                        abs_eff_mean,
                                        "Mean absolute efficiency value",
                                        keywords_RE);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_ABS_EFF_MEAN to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = espdr_keyword_add_int(qc_kws->qc_cal_flux_seeing_kw_check_kw,
                                     seeing_kw_qc,
                                     "CALFLUX VLTSH seeing KW check",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_CALFLUX_SEEING_KW_CHECK_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    if (seeing_kw_qc == 0) {
        global_QC = 0;
    }
    
    my_error = espdr_keyword_add_int(qc_kws->qc_cal_flux_check_kw,
                                     global_QC,
                                     "CALFLUX global QC",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_CALFLUX_CHECK_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    return cpl_error_get_code();
}

