/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2014-12-16 15:13:41 $
 * $Revision: $
 * $Name: not supported by cvs2svn $
 */


#include <espdr_wave_THAR_cal.h>
#include <espdr_pfits.h>
#define KSIG_EXTRACTION_MIN -2.0
#define KSIG_EXTRACTION_MAX 20.0
#define D_FIT_POLY_DEG_MIN 1
#define D_FIT_POLY_DEG_MAX 50
#define D_FIT_SMALL_RES_LIMIT_MIN 1
#define D_FIT_SMALL_RES_LIMIT_MAX 1000
#define WAVE_SOL_POLY_DEG_MIN 1
#define WAVE_SOL_POLY_DEG_MAX 10
/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
/**
 @brief     Save wave_THAR products
 @param
 @return    CPL_ERROR_NONE iff OK

 */
/*---------------------------------------------------------------------------*/

static cpl_error_code espdr_save_wave_THAR_products(cpl_frameset *frameset,
                                             cpl_parameterlist *parameters,
                                             const char *recipe_id,
                                             cpl_frameset *used_frames,
                                             const char *WAVE_tag,
                                             int fibre_nr,
                                             cpl_propertylist *keywords,
                                             espdr_inst_config *inst_config,
                                             cpl_image *wave_matrix,
                                             cpl_image *air_wave_matrix,
                                             cpl_image *dll_matrix,
                                             cpl_image *air_dll_matrix,
                                             cpl_table *coeffs_table,
                                             cpl_table *FP_table,
                                             cpl_table *THAR_table,
                                             cpl_table *raw_table) {
    cpl_error_code my_error = CPL_ERROR_NONE;
    char filename[FILENAME_LENGTH];
    char *new_keyword =(char *)cpl_malloc(KEYWORD_LENGTH*sizeof(char));

    // Saving vacuum WAVE_MATRIX
    /* Save the PRO.CATG */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_WAVE_MATRIX, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);
    sprintf(filename, "%s_%c.fits",
            inst_config->wave_matrix_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving WAVE_MATRIX for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);

    my_error = espdr_dfs_image_save_one_ext(frameset, parameters,
                                            used_frames, recipe_id,
                                            keywords, filename,
                                            wave_matrix, CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save_one_ext failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
    espdr_msg("WAVE_MATRIX saved under: %s", filename);

    // Saving air WAVE_MATRIX
    /* Save the PRO.CATG */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_AIR_WAVE_MATRIX, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);

    sprintf(filename, "%s_%c.fits",
            inst_config->air_wave_matrix_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving AIR_WAVE_MATRIX for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);

    my_error = espdr_dfs_image_save_one_ext(frameset, parameters,
                                            used_frames, recipe_id,
                                            keywords, filename,
                                            air_wave_matrix, CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save_one_ext failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
    espdr_msg("AIR_WAVE_MATRIX saved under: %s", filename);

    // Saving vacuum and air dll matrices
    /* Save the PRO.CATG */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_DLL_MATRIX, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);

    sprintf(filename, "%s_%c.fits",
            inst_config->dll_matrix_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving DLL_MATRIX for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);

    my_error = espdr_dfs_image_save_one_ext(frameset, parameters,
                                            used_frames, recipe_id,
                                            keywords, filename,
                                            dll_matrix, CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save_one_ext failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);

    /* Save the PRO.CATG */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_AIR_DLL_MATRIX, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);

    sprintf(filename, "%s_%c.fits",
            inst_config->air_dll_matrix_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving AIR_DLL_MATRIX for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);

    my_error = espdr_dfs_image_save_one_ext(frameset, parameters,
                                            used_frames, recipe_id,
                                            keywords, filename,
                                            air_dll_matrix, CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save_one_ext failed: %s for file %s",
                 cpl_error_get_message_default(my_error),
                 filename);

    /* Save the PRO.CATG */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_WAVE_TABLE, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);

    sprintf(filename, "%s_%c.fits",
            inst_config->wave_table_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving WAVE_TABLE for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);

    my_error = cpl_dfs_save_table(frameset, NULL, parameters, used_frames,
                                  NULL, coeffs_table, NULL,
								  recipe_id, keywords, NULL,
                                  PACKAGE "/" PACKAGE_VERSION,
                                  filename);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_dfs_save_table failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);

    /* Save the PRO.CATG */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_FP_FITTED_LINE_TABLE, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);

    sprintf(filename, "%s_%c.fits",
            inst_config->fp_fitted_line_table_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving FP FITTED LINES TABLE for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);

    my_error = cpl_dfs_save_table(frameset, NULL, parameters, used_frames,
                                  NULL, FP_table, NULL,
								  recipe_id, keywords, NULL,
                                  PACKAGE "/" PACKAGE_VERSION,
                                  filename);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);

    /* Save the PRO.CATG */
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_THAR_LINE_TABLE, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);

    sprintf(filename, "%s_%c.fits",
            inst_config->thar_line_table_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving THAR_LINE_TABLE for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);

    my_error = cpl_dfs_save_table(frameset, NULL, parameters, used_frames,
                                  NULL, THAR_table, NULL,
								  recipe_id, keywords, NULL,
                                  PACKAGE "/" PACKAGE_VERSION,
                                  filename);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);

#if PROCESS_RAW_LINES
    /* Save the raw lines fits table */
    //char *new_keyword =(char *)cpl_malloc(KEYWORD_LENGTH*sizeof(char));
    sprintf(new_keyword, "%s_%s_%c",
            ESPDR_PRO_CATG_LINE_TABLE_RAW, WAVE_tag, fibre_name[fibre_nr]);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords,PRO_CATG_KW,new_keyword);

    sprintf(filename, "%s_%c.fits",
            inst_config->line_table_raw_filename_no_fits, fibre_name[fibre_nr]);
    espdr_msg("Saving LINE_TABLE_RAW for fibre %c in %s, PRO.CATG = %s",
              fibre_name[fibre_nr], filename, new_keyword);
    my_error = cpl_dfs_save_table(frameset, NULL, parameters, used_frames,
                                  NULL, raw_table, NULL,
								  recipe_id, keywords, NULL,
                                  PACKAGE "/" PACKAGE_VERSION,
                                  filename);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
#endif

#if SAVE_STATIC_MATRICES
    /* Save the STATIC WAVE & DLL MATRICES FITS */
    my_error = espdr_save_static_matrixes(wave_matrix,
                                          dll_matrix,
                                          frameset,
                                          used_frames,
                                          parameters,
                                          inst_config,
                                          keywords,
                                          fibre_nr,
                                          //57997.0, // 2017-09-01
                                          //58405.0, // 2018-10-14
                                          //58655.0, // 2019-06-21
                                          //59260.0, // 2021-02-15
                                          //59335.0, // 2021-05-01
                                          //59670.0, // 2022-04-01
                                          //59884.0, // 2022-11-01
                                          56971.0, // CORALIE after 10th November 2014
                                          //"2017-09-01");
                                          //"2018-10-14");
                                          //"2019-06-21");
                                          //"2021-02-15");
                                          //"2021-05-01");
                                          //"2022-04-01");
                                          //"2022-11-01");
                                          "2014-11-10");
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_save_static_matrixes failed: %s",
                 cpl_error_get_message_default(my_error));
#endif

#if SAVE_TH_REF_TABLES
    char ins_mode_bin[32];
    //char valid_date[16] = "2017-09-01";
    //char valid_date[16] = "2018-10-14";
    //char valid_date[16] = "2019-06-21";
    //char valid_date[16] = "2015-05-25"; //HARPS post fibre
    //char valid_date[16] = "2003-01-01"; //HARPS pre fibre
    //char valid_date[16] = "2021-01-19"; // HA redcut
    //char valid_date[16] = "2021-03-02"; // HE redcut
    //char valid_date[16] = "2021-02-15"; // HA & HE first full spectrum
    //char valid_date[16] = "2021-05-01"; // HE after May 2021
    //char valid_date[16] = "2022-04-01"; // HA & HE after April 2022
    //char valid_date[16] = "2022-11-01"; // HA & HE after November 2022
    char valid_date[16] = "2014-11-10"; // CORALIE after 10th November 2014
    my_error = espdr_save_TH_REF_TABLE(THAR_table,
                                       frameset,
                                       used_frames,
                                       parameters,
                                       inst_config,
                                       keywords,
                                       fibre_nr,
                                       "REF",
                                       //57997.0,
                                       //58405.0,
                                       //58655.0,
                                       //57168.0, // HARPS post fibre
                                       //52641.0, // HARPS pre fibre
                                       //59233.0, // HA redcut
                                       //59275.0, // HE redcut
                                       //59260, // HA & HE full spectrum after 15th Feb 2021
                                       //59335, // HA & HE after 1st May 2021
                                       //59670, // HA & HE after 1st April 2022
                                       //59884, // HA & HE after 1st November 2022
                                       56971, // CORALIE after 10th November 2014
                                       valid_date,
                                       ins_mode_bin);
    espdr_msg("INS BIN MODE: %s, valid date: %s",
              ins_mode_bin, valid_date);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_save_TH_REF_TABLE failed: %s",
                 cpl_error_get_message_default(my_error));
#endif

    return (cpl_error_get_code());
}

/*----------------------------------------------------------------------------*/
/**
 @brief Fit reference ThAr calibration lines
 @param[out]    lines_table             detected lines peaks
 @param         lines_nb                number of lines in the table
 @param         s2d                     input data
 @param         s2d_err                 input data errors
 @param         s2d_qual                input data quality flags
 @param         pixel_geom              pixel geometry image
 @param         flux_threshold          flux treshold for line detection
 @param         min_x0_err              limit for the peak pixel error
 @param         window_size             lines fitting window size
 @param[out]    lines_nb_per_order_RE   computed number of lines for each order
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_fit_REF_lines(espdr_line_param *lines_table,
                                   int lines_nb,
                                   cpl_image *s2d,
                                   cpl_image *s2d_err,
                                   cpl_image *s2d_qual,
                                   cpl_image *pixel_geom,
                                   double flux_threshold,
                                   int window_size,
                                   int tolerance_window,
                                   int *lines_nb_per_order_RE,
                                   int *saturated_lines_nb_RE,
                                   double *flux_ratio_median_RE,
                                   double *rv_diff_median_RE) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    int line_index, fit_index;
    int order, fit_start, fit_end, fit_vector_length, pxl;
    int left_margin, right_margin, high_pxl;
    double max_flux, min_flux1, min_flux2;
    cpl_image *s2d_order = NULL;
    cpl_image *s2d_line = NULL;
    cpl_image *s2d_qual_line = NULL;
    cpl_image *s2d_err_order = NULL;
    cpl_image *pxl_geom_order = NULL;
    double *s2d_order_data = NULL;
    int *s2d_qual_data = NULL;
    //double *s2d_line_data = NULL;
    double *s2d_err_order_data = NULL;
    double *pxl_geom_order_data = NULL;
    int s2d_size_x = 0;
    int s2d_size_y = 0;
    espdr_ngauss_data *g_data = NULL;
    espdr_ngauss_result *g_res = NULL;
    double *x0 = NULL;
    cpl_size px;

    int saturation_OK = 1;
    int saturated_lines_nb = 0;
    double flux_ref[lines_nb], flux_fit[lines_nb], x0_ref[lines_nb], x0_fit[lines_nb];
    int flag_for_QC[lines_nb];

    espdr_ensure(lines_table == NULL, CPL_ERROR_NULL_INPUT,
                 "Lines input table is NULL");
    espdr_ensure(s2d == NULL, CPL_ERROR_NULL_INPUT,
                 "s2d image is NULL");
    espdr_ensure(s2d_err == NULL, CPL_ERROR_NULL_INPUT,
                 "s2d image error is NULL");
    espdr_ensure(s2d_qual == NULL, CPL_ERROR_NULL_INPUT,
                 "s2d image quality is NULL");
    espdr_ensure(pixel_geom == NULL, CPL_ERROR_NULL_INPUT,
                 "pixel geometry image is NULL");

    espdr_msg("Starting lines fit, with tolerance +/- %d", tolerance_window);

    g_data = (espdr_ngauss_data *)cpl_malloc(sizeof(espdr_ngauss_data));
    g_res = (espdr_ngauss_result *)cpl_malloc(sizeof(espdr_ngauss_result));

    s2d_size_x = cpl_image_get_size_x(s2d);
    s2d_size_y = cpl_image_get_size_y(s2d);
    espdr_msg("S2D size: %d x %d", s2d_size_x, s2d_size_y);

    line_index = 0;
    int CCD_margin = window_size + tolerance_window;
    while (line_index < lines_nb) {
        fit_index = 0;
        order = lines_table[line_index].order;
        flag_for_QC[line_index] = 0;
        flux_ref[line_index] = lines_table[line_index].peak_flux;
        flux_fit[line_index] = lines_table[line_index].peak_flux;
        x0_ref[line_index] = lines_table[line_index].peak_pxl;
        x0_fit[line_index] = lines_table[line_index].peak_pxl;

        if ((lines_table[line_index].qc_flag == 1) && (order <= s2d_size_y)) {
            left_margin = (int)(lines_table[line_index].peak_pxl + 0.5) - CCD_margin;
            if (left_margin <= 0) {
                lines_table[line_index].qc_flag = 2;
            } else {
                right_margin = (int)(lines_table[line_index].peak_pxl + 0.5) + CCD_margin;
                if (right_margin >= s2d_size_x) {
                    lines_table[line_index].qc_flag = 2;
                } else {
                    high_pxl = (int)(lines_table[line_index].peak_pxl + 0.5);
                    cpl_image *test_img = cpl_image_extract(s2d,
                                                            high_pxl-tolerance_window, order,
                                                            high_pxl+tolerance_window, order);
                    my_error = cpl_error_get_code();
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "Extract S2D line: order = %d, high_pxl = %d: %s",
                                 order, high_pxl,
                                 cpl_error_get_message_default(my_error));
                    double *test_img_data = cpl_image_get_data_double(test_img);
                    double test_max = 0.0;
                    int test_max_index = 0;
                    for (int i = 0; i < tolerance_window*2+1; i++) {
                        if (line_index == -1) {
                            espdr_msg("test_img[%d] = %lf", i, test_img_data[i]);
                        }
                        if (test_img_data[i] > test_max) {
                            test_max = test_img_data[i];
                            test_max_index = i;
                        }
                    }
                    if (line_index == -1) {
                        espdr_msg("test_max_index: %d", test_max_index);
                    }
                    px = test_max_index+high_pxl-tolerance_window;
                    //cpl_size py = order;
                    //my_error = cpl_image_get_maxpos_window(s2d,
                    //                                       high_pxl-tolerance_window, order,
                    //                                       high_pxl+tolerance_window, order,
                    //                                       &px, &py);
                    //espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                    //             "Get max pos: order = %d, high_pxl = %d: %s",
                    //             order, high_pxl,
                    //             cpl_error_get_message_default(my_error));

                    if (order == -1) {
                        espdr_msg("Line %d pxl %lf, high_pxl: %d, px: %lld",
                                  line_index, lines_table[line_index].peak_pxl,
                                  high_pxl, px);
                    }

                    if (llabs(high_pxl - px) < tolerance_window) {
                        //espdr_msg("Peak position: %lld, %lld, high pxl: %d",
                        //          px, py, high_pxl);

                        high_pxl = px;

                        // Check if the fit window has saturated pixels

                        s2d_qual_line = cpl_image_extract(s2d_qual,
                                                          high_pxl-window_size, order,
                                                          high_pxl+window_size, order);
                        s2d_qual_data = cpl_image_get_data_int(s2d_qual_line);
                        saturation_OK = 1;
                        for (int i = 0; i < 2*window_size+1; i++) {
                            if (s2d_qual_data[i] == AD_CONVERTER_SATURATION) {
                                //espdr_msg("Line at pxl %d, order %d saturated [i = %d]",
                                //          high_pxl, order, i);
                                saturation_OK = 0;
                            }
                        }

                        if (saturation_OK == 1) {
                            s2d_line = cpl_image_extract(s2d,
                                                         high_pxl-window_size, order,
                                                         high_pxl+window_size, order);
                            //espdr_msg("s2d_line size x: %lld, window_size: %d, high pxl: %d",
                            //          cpl_image_get_size_x(s2d_line), window_size, high_pxl);
                            my_error = cpl_error_get_code();
                            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                         "Extract S2D line: order = %d, high_pxl = %d: %s",
                                         order, high_pxl,
                                         cpl_error_get_message_default(my_error));

                            max_flux = cpl_image_get_max(s2d_line);
                            min_flux1 = cpl_image_get_min_window(s2d_line,
                                                                 1, 1,
                                                                 window_size+1, 1);
                            min_flux2 = cpl_image_get_min_window(s2d_line,
                                                                 window_size+1, 1,
                                                                 window_size*2+1, 1);

                            if (order == -1) {
                                espdr_msg("order: %d, pxl: %lf, max_flux: %lf, min_flux: %lf",
                                          order, lines_table[line_index].peak_pxl,
                                          max_flux, (min_flux1+min_flux2)/2.0);
                            }

                            if ((max_flux - ((min_flux1+min_flux2)/2.0)) > flux_threshold) {
                                // Do the fit
                                //espdr_msg("Preparing the fit of line %d at pxl %d in order %d",
                                //          line_index, high_pxl, order);
                                fit_index = 1;
                                fit_start = high_pxl - window_size;
                                fit_end = high_pxl + window_size;
                                espdr_ensure(fit_end > s2d_size_x, CPL_ERROR_ILLEGAL_OUTPUT,
                                             "fit window end outside of the CCD");

                                s2d_order = cpl_image_extract(s2d, fit_start, order, fit_end, order);
                                my_error = cpl_error_get_code();
                                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                             "Extract S2D order %d: %s", order,
                                             cpl_error_get_message_default(my_error));

                                s2d_err_order = cpl_image_extract(s2d_err, fit_start,
                                                                  order, fit_end, order);

                                pxl_geom_order = cpl_image_extract(pixel_geom,
                                                                   fit_start, order, fit_end, order);
                                s2d_order_data = cpl_image_get_data_double(s2d_order);
                                s2d_err_order_data = cpl_image_get_data_double(s2d_err_order);
                                pxl_geom_order_data = cpl_image_get_data_double(pxl_geom_order);


                                fit_vector_length = fit_end - fit_start + 1;
                                g_data->n = fit_vector_length;
                                g_data->m = fit_index;
                                g_data->x = (double *)cpl_calloc(fit_vector_length, sizeof (double));
                                g_data->y = (double *)cpl_calloc(fit_vector_length, sizeof (double));
                                g_data->err = (double *)cpl_calloc(fit_vector_length, sizeof (double));
                                g_data->fit = (double *)cpl_calloc(fit_vector_length, sizeof (double));

                                g_res->k = (double *)cpl_calloc(fit_index, sizeof (double));
                                g_res->sig_k = (double *)cpl_calloc(fit_index, sizeof (double));
                                g_res->x0 = (double *)cpl_calloc(fit_index, sizeof (double));
                                g_res->sig_x0 = (double *)cpl_calloc(fit_index, sizeof (double));
                                g_res->fwhm = (double *)cpl_calloc(fit_index, sizeof (double));
                                g_res->sig_fwhm = (double *)cpl_calloc(fit_index, sizeof (double));

                                x0 = (double *)cpl_calloc(fit_index, sizeof(double));
                                x0[0] = high_pxl - fit_start;

                                for (pxl = 0; pxl <= fit_end - fit_start; pxl++) {
                                    g_data->x[pxl] = pxl_geom_order_data[pxl] - fit_start;
                                    g_data->y[pxl] = s2d_order_data[pxl];
                                    g_data->err[pxl] = s2d_err_order_data[pxl];
                                    if (line_index == -1) {
                                        espdr_msg("BEFORE NGAUSS: pxl: %d, x: %lf, y: %lf, err: %lf",
                                                  pxl, g_data->x[pxl], g_data->y[pxl],
                                                  g_data->err[pxl]);
                                    }
                                }

                                //espdr_msg("Fitting line at pxl %d in order %d", high_pxl, order);

                                my_error = espdr_fit_Ngauss(g_data, x0, 3.0, g_res, 1, 0);
                                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                             "espdr_fit_Ngauss failed: %s",
                                             cpl_error_get_message_default(my_error));

                                lines_table[line_index].peak_pxl = g_res->x0[0] + fit_start;
                                lines_table[line_index].peak_pxl_err = g_res->sig_x0[0];
                                lines_table[line_index].fwhm = g_res->fwhm[0];
                                lines_table[line_index].fwhm_err = g_res->sig_fwhm[0];
                                lines_table[line_index].peak_flux = g_res->k[0];
                                lines_table[line_index].peak_flux_err = g_res->sig_k[0];

                                //if (lines_table[line_index].peak_pxl_err < min_x0_err) {
                                //    lines_table[line_index].peak_pxl_err = min_x0_err;
                                //}

                                if (line_index == -1) {
                                    espdr_msg("AFTER NGAUSS: peak_pxl: %lf, fwhm: %lf, peak_flux: %lf, fit_start: %d",
                                              lines_table[line_index].peak_pxl,
                                              lines_table[line_index].fwhm,
                                              lines_table[line_index].peak_flux,
                                              fit_start);
                                }

                                //espdr_msg("Cleaning fit data structures memory");
                                cpl_free(x0);

                                cpl_free(g_data->x);
                                cpl_free(g_data->y);
                                cpl_free(g_data->fit);
                                cpl_free(g_data->err);

                                cpl_free(g_res->k);
                                cpl_free(g_res->sig_k);
                                cpl_free(g_res->x0);
                                cpl_free(g_res->sig_x0);
                                cpl_free(g_res->fwhm);
                                cpl_free(g_res->sig_fwhm);

                                cpl_image_delete(s2d_order);
                                cpl_image_delete(s2d_err_order);
                                cpl_image_delete(pxl_geom_order);

                                lines_nb_per_order_RE[order-1]++;

                                flag_for_QC[line_index] = 1;
                                flux_fit[line_index] = lines_table[line_index].peak_flux;
                                x0_fit[line_index] = lines_table[line_index].peak_pxl;

                            } else {
                                espdr_msg_warning("Peak height too small at pixel %d (%d) in order %d",
                                                  high_pxl, high_pxl + ORDER_SHIFT, order);
                                lines_table[line_index].qc_flag = 2;
                                //if (order == 10) {
                                //    espdr_msg("order: %d, pxl: %lf rejected, max_flux: %lf, min_flux: %lf",
                                //              order,
                                //              lines_table[line_index].peak_pxl,
                                //              max_flux, (min_flux1+min_flux2)/2.0);
                                //}
                            } // if ((max_flux - ((min_flux1+min_flux2)/2.0)) > flux_threshold)

                            cpl_image_delete(s2d_line);

                        } else {
                            espdr_msg_warning("ThAr line at pixel %d in order %d is saturated",
                                              high_pxl, order);
                            saturated_lines_nb++;
                            lines_table[line_index].qc_flag = 2;
                        }

                        cpl_image_delete(test_img);
                    } else {
                        espdr_msg_warning("ThAr line not found at pixel %d (%d) in order %d",
                                  high_pxl, high_pxl + ORDER_SHIFT, order);
                        lines_table[line_index].qc_flag = 2;
                    } // if (fabs(high_pxl - px) < tolerance_window)
                } // if (right_margin >= s2d_size_x)
            } // (left_margin <= 0)
        } // if ((lines_table[line_index].qc_flag == 1) && (order <= s2d_size_y))
        line_index++;
    } // while (line_index < lines_nb)


    espdr_msg("Nb of detected saturated lines: %d", saturated_lines_nb);
    *saturated_lines_nb_RE = saturated_lines_nb;

    int lines_nb_for_QC = 0;
    double *flux_ratio_for_QC = NULL;
    double *rv_diff_for_QC = NULL;
    cpl_vector *flux_ratio_vector, *rv_diff_vector;
    for (int i = 0; i < lines_nb; i++) {
        if (flag_for_QC[i] == 1) {
            lines_nb_for_QC++;
        }
    }
    flux_ratio_for_QC = (double *)cpl_calloc(lines_nb_for_QC, sizeof(double));
    rv_diff_for_QC = (double *)cpl_calloc(lines_nb_for_QC, sizeof(double));

    int index_for_QC = 0;
    for (int i = 0; i < lines_nb; i++) {
        if ((flag_for_QC[i] == 1) && (flux_ref[i] != -1.0)) {
            flux_ratio_for_QC[index_for_QC] = flux_fit[i] / flux_ref[i] ;
            rv_diff_for_QC[index_for_QC] = (x0_fit[i] - x0_ref[i]) *
                                            lines_table[i].dispersion / lines_table[i].wavelength * LIGHT_SPEED;
            //espdr_msg("line: %d flux: %f\t%f\t%f\t\tx0: %.10f\t%.10f\t%.10f, disp: %.10f",
            //          i, flux_ref[i], flux_fit[i], flux_ratio_for_QC[index_for_QC],
            //          x0_ref[i], x0_fit[i], rv_diff_for_QC[index_for_QC], lines_table[i].dispersion);
            index_for_QC++;
        }
    }
    flux_ratio_vector = cpl_vector_wrap(lines_nb_for_QC, flux_ratio_for_QC);
    rv_diff_vector = cpl_vector_wrap(lines_nb_for_QC, rv_diff_for_QC);

    *flux_ratio_median_RE = cpl_vector_get_median(flux_ratio_vector);
    *rv_diff_median_RE = cpl_vector_get_median(rv_diff_vector);

    cpl_vector_unwrap(flux_ratio_vector);
    cpl_vector_unwrap(rv_diff_vector);

    cpl_free(flux_ratio_for_QC);
    cpl_free(rv_diff_for_QC);

    cpl_free(g_data);
    cpl_free(g_res);

    return(cpl_error_get_code());
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Add extra QC keywords
 @param     keywords    property list to add the extra KWs
 @param     ext_nb      number of extensions
 @param     mjd_obs_delta_time_fp   mysterious Andrea's parameter
 @return    0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static cpl_error_code espdr_add_extra_qc(cpl_propertylist** keywords,
                                         const int ext_nb,
                                         double mjd_obs_delta_time_fp) {

    const char* ksuff[2] = {" BLUE", " RED"};
    cpl_propertylist_append_int(*keywords,"ESO QC PROC VERSION", 2);
    int* ord_ref = espdr_get_ord_ref(*keywords, "espdr_wave_THAR");
    espdr_add_qc_key_wave_THAR_ord_ref(keywords,ext_nb, ord_ref, "FP LINES NB", ksuff,
                                       CPL_TRUE, CPL_TRUE, CPL_FALSE, CPL_FALSE, CPL_FALSE);
    espdr_add_qc_key_stat_ord_pri(keywords,ext_nb, "FP LINES NB", "DUMMY", CPL_TRUE, CPL_TRUE,
                                  CPL_FALSE, CPL_FALSE, CPL_FALSE);
    espdr_add_qc_key_stat_ord_pri(keywords,ext_nb, "CHI2", "DUMMY", CPL_TRUE, CPL_TRUE,
                                  CPL_FALSE, CPL_FALSE, CPL_FALSE);

    espdr_add_qc_key_stat_ord_pri(keywords,ext_nb, "RMS", "DUMMY", CPL_TRUE, CPL_TRUE,
                                  CPL_FALSE, CPL_FALSE, CPL_FALSE);
    if(mjd_obs_delta_time_fp != -999) {
    char key_val[40];
    cpl_property* p;
    sprintf(key_val,"ESO QC DELTA TIME FP");
    cpl_msg_info(cpl_func,"key_val %s",key_val);
    cpl_propertylist_append_double(*keywords, key_val, mjd_obs_delta_time_fp);
    cpl_propertylist_set_comment(*keywords, key_val,
                                 "MJD_OBS difference of recipe inputs files FP_THAR and S2D_BLAZE_FP_FP_B");
    for (cpl_size i = 0; i < ext_nb; i++) {

        sprintf(key_val,"ESO QC DRIFT DET%lld MEAN",i);
        cpl_msg_info(cpl_func,"key_val %s",key_val);
        //cpl_propertylist_dump(*keywords,stdout);
        p = cpl_propertylist_get_property(*keywords,key_val);

        double drift_mean = cpl_property_get_double(p);
        double qc_drift_vel =
        espdr_drift_vel(drift_mean, mjd_obs_delta_time_fp);
        sprintf(key_val,"ESO QC DRIFT DET%lld VEL",i);
        cpl_msg_info(cpl_func,"key_val %s",key_val);
        cpl_propertylist_append_double(*keywords, key_val, qc_drift_vel);
        cpl_propertylist_set_comment(*keywords, key_val,
                                     "[cm/s/h] Instrumental drift of wavelength calibration");
    }
    }

    cpl_free(ord_ref);
    return cpl_error_get_code();
}





/*---------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    parameters     the parameters list
 @param    frameset   the frames list

 In case of failure the cpl_error_code is set.
 */
/*---------------------------------------------------------------------------*/

int espdr_wave_THAR(cpl_parameterlist *parameters, cpl_frameset *frameset,
		const char* recipe_id) {

    cpl_frameset *wave_frames_to_reduce = NULL;
    cpl_frame *wave_frame = NULL;
    cpl_frameset *used_frames = NULL;

    /* wavelength solution related images */
    cpl_image *wave_matrix_sol = NULL;
    cpl_image *dll_matrix_sol = NULL;
    cpl_image *air_wave_matrix_sol = NULL;
    cpl_image *air_dll_matrix_sol = NULL;

    /* drift related images */
    cpl_image *s2d_blaze_FP_spectrum = NULL;
    cpl_image *s2d_blaze_FP_error = NULL;
    cpl_image *s2d_blaze_FP_qual = NULL;
    cpl_image *drift_matrix = NULL;

    int TH_saturated_lines_nb;
    double TH_flux_ratio_median, TH_rv_diff_median;

    /* FITS tables products */
    espdr_line_param **FP_lines_table = NULL;
    espdr_line_param *REF_lines_table = NULL;
    espdr_raw_line_param *raw_lines_table = NULL;
    cpl_table *THAR_lines_table_FITS = NULL;
    cpl_table **FP_lines_table_FITS = NULL;
    cpl_table **coeffs_table_FITS = NULL;
    cpl_table *raw_lines_table_FITS = NULL;
    cpl_table *drift_table = NULL;
    char column_name[KEYWORD_LENGTH];

    /* wavelength solution data */
    int size_x;
    int **lines_nb_per_order = NULL;
    double ***coeffs_per_order = NULL;
    double **rms_per_order = NULL;
    double **chisq_per_order = NULL;
    int **order_fit = NULL;
    double *wave_matrix_sol_data = NULL;
    double *dll_matrix_sol_data = NULL;
    double *air_wave_matrix_sol_data = NULL;
    double *air_dll_matrix_sol_data = NULL;
    double *d_computed = NULL;
    double *d_computed_err = NULL;
    double *d_computed_err_no_min = NULL;
    double *residuals = NULL;
    double *dm = NULL;
    double *resolution = NULL;
    double resol_median = 0.0;
    double *d_static = NULL;
    double *d_static_err = NULL;
    double *d_static_err_no_min = NULL;

	cpl_error_code my_error = CPL_ERROR_NONE;

    const int rec_ntags = 25;
    const char* rec_tags[25] = {
        ESPDR_PRO_CATG_HOT_PIXELS, ESPDR_PRO_CATG_MDARK,
        ESPDR_PRO_CATG_BAD_PIXELS, ESPDR_CCD_GEOM,
        ESPDR_PRO_CATG_ORDERS_A, ESPDR_PRO_CATG_ORDERS_B,
        ESPDR_PRO_CATG_FP_SEARCHED_LINE_TABLE_FP_FP_A,
        ESPDR_PRO_CATG_FP_SEARCHED_LINE_TABLE_FP_FP_B,
        ESPDR_PRO_CATG_REF_LINE_TABLE_A, ESPDR_PRO_CATG_REF_LINE_TABLE_B,
        ESPDR_PRO_CATG_S2D_BLAZE_FP_FP_A, ESPDR_PRO_CATG_S2D_BLAZE_FP_FP_B,
        ESPDR_PRO_CATG_STATIC_WAVE_MATRIX_A, ESPDR_PRO_CATG_STATIC_WAVE_MATRIX_B,
        ESPDR_PRO_CATG_STATIC_DLL_MATRIX_A, ESPDR_PRO_CATG_STATIC_DLL_MATRIX_B,
        ESPDR_PRO_CATG_PIXEL_GEOM_A, ESPDR_PRO_CATG_PIXEL_GEOM_B,
        ESPDR_PRO_CATG_PIXEL_SIZE_A, ESPDR_PRO_CATG_PIXEL_SIZE_B,
        ESPDR_WAVE_THAR_FP_RAW, ESPDR_WAVE_FP_THAR_RAW,
        ESPDR_WAVE_THAR_DARK_RAW, ESPDR_WAVE_DARK_THAR_RAW, ESPDR_ORDERS_MASK
    };
    int is_required[25] = {1, 0, 1, 1, 1, 1,
        0, 0, // FP_SEARCH_LINES_TABLEs, can be A or B
        0, 0, // REF_LINE_TABLEs, can be A or B
        0, 0, 0, 0, 0, 0, // products for drift calculation, can be B or A
        0, 0, 0, 0, // pixel geometry & size maps
        0, 0, 0, 0, // raw frames, can be THAR_FP or FP_THAR or THAR_DARK or DARK_THAR
        0 // ORDERS MASK
    };

	cpl_msg_set_level(CPL_MSG_INFO);

	espdr_msg("Starting wave_THAR");

    espdr_ensure(espdr_check_input_tags(frameset,
                                        rec_tags, is_required,
                                        rec_ntags) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Error: %s, exiting.",
                 cpl_error_get_message_default(cpl_error_get_code()));
    espdr_ensure(espdr_check_input_inst_config(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_flat(frameset, 1) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong flat input tag!");

    /* Identify the RAW and CALIB frames in the input frameset */
    espdr_ensure(espdr_dfs_set_groups(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(),
                 "DFS setting groups failed. Expected inputs are:\n"
                 "raw wave THAR_FP, THAR_DARK or FP_THAR, DARK_THAR image, tagged as THAR_FP/DARK or FP/DARK_THAR\n"
                 "CCD geometry table, tagged as CCD_GEOM\n"
                 "instrument configuration table, tagged as MASTER_INST_CONFIG or INST_CONFIG\n"
                 "ThAr lines reference table, tagged as REF_LINE_TABLEA or REF_LINE_TABLE_B\n"
                 "hot pixel mask image, tagged as HOT_PIXEL_MASK\n"
                 "bad pixel mask image, tagged as BAD_PIXEL_MASK\n"
                 "orders definition images for fibres A and B, tagged as ORDER_TABLE_A & ORDER_TABLE_B\n"
                 "orders profile images for fibres A and B, tagged as ORDER_PROFILE_A & ORDER_PROFILE_B\n"
                 "flat images for fibres A and B, tagged as FLAT_A & FLAT_B\n"
                 "blaze images for fibres A and B, tagged as BLAZE_A & BLAZE_B\n"
                 "FP lines table for fibres A or B, tagged as FP_SEARCHED_LINE_TABLE_FP_FP_A/B\n"
                 "static wave matrix for fibres B or A, tagged as STATIC_WAVE_MATRIX_B/A\n"
                 "static dll matrix for fibres B or A, tagged as STATIC_DLL_MATRIX_B/A\n"
                 "S2D blaze of FP_FP for fibres B or A, tagged as S2D_BLAZE_FP_FP_B/A\n");

    espdr_CCD_geometry *CCD_geom            = NULL;
    espdr_inst_config *inst_config          = NULL;
    espdr_qc_keywords *qc_kws               = NULL;
    espdr_OVSC_param *OVSC_param            = NULL;
    espdr_WAVE_THAR_param *WAVE_THAR_param  = NULL;
    cpl_frame* CCD_geom_frame               = NULL;
    cpl_frame* inst_config_frame            = NULL;

    CCD_geom_frame    = espdr_frame_find(frameset, ESPDR_CCD_GEOM);
    espdr_ensure(CCD_geom_frame == NULL,cpl_error_get_code(),
                 "CCD geometry frame not found!");
    CCD_geom          = espdr_CCD_geom_init(parameters, CCD_geom_frame);
    inst_config_frame = espdr_get_inst_config(frameset);
    espdr_ensure(inst_config_frame == NULL,cpl_error_get_code(),
                 "Instrument config frame not found!");
    inst_config       = espdr_inst_config_init(parameters,
                                               CCD_geom->ext_nb,
                                               inst_config_frame);
    OVSC_param        = espdr_parameters_OVSC_init(recipe_id, parameters);
    WAVE_THAR_param   = espdr_WAVE_THAR_param_init(recipe_id, parameters);

    used_frames = cpl_frameset_new();
    my_error = cpl_frameset_insert(used_frames, cpl_frame_duplicate(CCD_geom_frame));
    my_error = cpl_frameset_insert(used_frames, cpl_frame_duplicate(inst_config_frame));

    /* Filling up the DRS QC KWs structure */
    qc_kws = (espdr_qc_keywords *)cpl_malloc(sizeof(espdr_qc_keywords));
    my_error = espdr_fill_qc_keywords(inst_config, qc_kws);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Filling QC KWs failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_ensure(qc_kws == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "QC KWs structure is NULL");

    int compute_drift = 2;
    wave_frame = espdr_frame_find(frameset, ESPDR_WAVE_THAR_FP_RAW);
    if (wave_frame == NULL) {
        wave_frame = espdr_frame_find(frameset, ESPDR_WAVE_FP_THAR_RAW);
        if (wave_frame == NULL) {
            wave_frame = espdr_frame_find(frameset, ESPDR_WAVE_THAR_DARK_RAW);
            if (wave_frame == NULL) {
                wave_frame = espdr_frame_find(frameset, ESPDR_WAVE_DARK_THAR_RAW);
                if (wave_frame == NULL) {
                    espdr_msg_error("No input raw frame with any of the TAGs: THAR_FP, FP_THAR, THAR_DARK, DARK_THAR, exiting");
                    return(CPL_ERROR_NULL_INPUT);
                } else {
                    compute_drift = 0;
                }
            } else {
                compute_drift = 0;
            }
        }
    }
    
    const char *WAVE_tag = cpl_frame_get_tag(wave_frame);
    my_error = cpl_frameset_insert(used_frames, cpl_frame_duplicate(wave_frame));
    wave_frames_to_reduce = cpl_frameset_new();
    cpl_frameset_insert(wave_frames_to_reduce, cpl_frame_duplicate(wave_frame));
    if (strcmp(WAVE_tag, "THAR_FP") == 0) {
        compute_drift = inst_config->thar_drift_th_fp_sw;
    } else {
        if (strcmp(WAVE_tag, "FP_THAR") == 0) {
            compute_drift = inst_config->thar_drift_fp_th_sw;
        }
    }
    espdr_ensure(compute_drift == 2, CPL_ERROR_INCOMPATIBLE_INPUT,
                 "Wrong TAG: %s", WAVE_tag);
    espdr_msg("WAVE_tag: %s, compute_drift = %d", WAVE_tag, compute_drift);
    
    espdr_src_type fibre_source[inst_config->fibres_nb];
    char **fibre_WAVE = (char **)cpl_malloc(inst_config->fibres_nb * sizeof(char *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        fibre_WAVE[i] = (char *)cpl_malloc(TAG_LENGTH*sizeof(char));
    }
    my_error = espdr_get_fibre_source(WAVE_tag, fibre_source, fibre_WAVE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_get_fibre_source failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_msg("fibre_WAVE: %s & %s", fibre_WAVE[0], fibre_WAVE[1]);

    int *orders_nb_per_fibre = (int *)cpl_calloc(inst_config->fibres_nb, sizeof(int));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        orders_nb_per_fibre[i] = 0;
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            orders_nb_per_fibre[i] += inst_config->orders_nb[i*CCD_geom->ext_nb+j];
        }
    }

    cpl_image **pixel_geom_image = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **pixel_size_image = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_corr_spectrum = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_corr_error = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_corr_qual = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_blaze_corr_spectrum = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_blaze_corr_error = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_blaze_corr_qual = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_propertylist **keywords_fibre = (cpl_propertylist **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_propertylist *));
    cpl_imagelist *CCD_corrected_image = cpl_imagelist_new();
    cpl_table ***orders_coeffs = NULL;
    double *RON = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
    cpl_frame** REF_line_table_frame = (cpl_frame **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_frame *));
    cpl_frame** FP_line_table_frame = (cpl_frame **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_frame *));
    cpl_image **static_wave_matrix_img = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    cpl_image **static_dll_matrix_img = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    cpl_frame *s2d_blaze_FP_frame = NULL;
    my_error = espdr_process_wave_till_extraction(frameset, parameters, recipe_id,
                                                  wave_frames_to_reduce,
                                                  wave_frame,
                                                  used_frames,
                                                  inst_config,
                                                  CCD_geom,
                                                  qc_kws,
                                                  orders_nb_per_fibre,
                                                  fibre_source,
                                                  fibre_WAVE,
                                                  keywords_fibre,
                                                  CCD_corrected_image,
                                                  &orders_coeffs,
                                                  RON,
                                                  flat_corr_spectrum,
                                                  flat_corr_error,
                                                  flat_corr_qual,
                                                  flat_blaze_corr_spectrum,
                                                  flat_blaze_corr_error,
                                                  flat_blaze_corr_qual,
                                                  pixel_geom_image,
                                                  pixel_size_image);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_process_wave_till_extraction failed: %s",
                 cpl_error_get_message_default(my_error));

    char pro_catg_tag[TAG_LENGTH];
    sprintf(pro_catg_tag, "%s_FP_FP", ESPDR_PRO_CATG_FP_SEARCHED_LINE_TABLE);
    my_error = espdr_extract_lines_tables(frameset, recipe_id,
                                          used_frames,
                                          inst_config,
                                          fibre_source,
                                          fibre_WAVE,
                                          compute_drift,
                                          ESPDR_PRO_CATG_REF_LINE_TABLE,
                                          pro_catg_tag,
                                          REF_line_table_frame,
                                          FP_line_table_frame,
                                          static_wave_matrix_img,
                                          static_dll_matrix_img,
                                          &s2d_blaze_FP_frame);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_extract_lines_tables failed: %s",
                 cpl_error_get_message_default(my_error));

    /* determine mjd_obs_delta_time_fp used later for QC */
    double mjd_obs_delta_time_fp = -999;
    if (compute_drift == 1) {
        /* Extract S2D_BLAZE_FP, STATIC WAVE and DLL MATRICES extensions */
        const char *s2d_blaze_FP_filename = cpl_frame_get_filename(s2d_blaze_FP_frame);
        s2d_blaze_FP_spectrum = cpl_image_load(s2d_blaze_FP_filename, CPL_TYPE_DOUBLE, 0, 1);
        s2d_blaze_FP_error = cpl_image_load(s2d_blaze_FP_filename, CPL_TYPE_DOUBLE, 0, 2);
        s2d_blaze_FP_qual = cpl_image_load(s2d_blaze_FP_filename, CPL_TYPE_INT, 0, 3);
        cpl_propertylist* plist = cpl_propertylist_load(cpl_frame_get_filename(s2d_blaze_FP_frame),0);
        double mjd_obs_blaze = 0.0, mjd_obs_raw = 0.0;
        if (cpl_propertylist_has(plist, "MJD-OBS")) {
            mjd_obs_blaze = espdr_pfits_get_mjdobs(plist);
        }
        cpl_propertylist_delete(plist);
        plist = cpl_propertylist_load(cpl_frame_get_filename(wave_frame),0);
        if (cpl_propertylist_has(plist, "MJD-OBS")) {
            mjd_obs_raw = espdr_pfits_get_mjdobs(plist);
        }
        cpl_propertylist_delete(plist);
        mjd_obs_delta_time_fp = mjd_obs_blaze - mjd_obs_raw;
    }

    /* LINES TABLE reading */
    espdr_msg("Lines tables & coeffs tables memory allocation");
    FP_lines_table = (espdr_line_param **)cpl_malloc(inst_config->fibres_nb * sizeof(espdr_line_param *));
    REF_lines_table = (espdr_line_param *)cpl_malloc(MAX_NB_SPECTRAL_LINES * sizeof(espdr_line_param));
    raw_lines_table = (espdr_raw_line_param *)cpl_malloc(MAX_NB_SPECTRAL_LINES * sizeof(espdr_raw_line_param));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        FP_lines_table[i] = (espdr_line_param *)cpl_malloc(MAX_NB_SPECTRAL_LINES * sizeof(espdr_line_param));
        my_error = espdr_lines_table_init(FP_lines_table[i],MAX_NB_SPECTRAL_LINES);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Lines table FP init failed: %s",
                     cpl_error_get_message_default(my_error));
        if (fibre_source[i] == THAR) {
            my_error = espdr_lines_table_init(REF_lines_table, MAX_NB_SPECTRAL_LINES);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "Lines table REF init failed: %s",
                         cpl_error_get_message_default(my_error));
        }
    }
    my_error = espdr_raw_lines_table_init(raw_lines_table, MAX_NB_SPECTRAL_LINES);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Lines table raw init failed: %s",
                 cpl_error_get_message_default(my_error));

    int FP_lines_nb[inst_config->fibres_nb];
    int REF_lines_nb = 0;
    int raw_lines_nb = 0;

    lines_nb_per_order = (int **)cpl_malloc(inst_config->fibres_nb*sizeof(int *));
    coeffs_per_order = (double ***)cpl_malloc(inst_config->fibres_nb*sizeof(double **));
    coeffs_table_FITS = (cpl_table **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_table*));
    rms_per_order = (double **)cpl_malloc(inst_config->fibres_nb*sizeof(double *));
    chisq_per_order = (double **)cpl_malloc(inst_config->fibres_nb*sizeof(double *));
    order_fit = (int **)cpl_malloc(inst_config->fibres_nb*sizeof(double *));

    for (int i = 0; i < inst_config->fibres_nb; i++) {
        size_x = cpl_image_get_size_x(flat_blaze_corr_spectrum[i]);
        lines_nb_per_order[i] = (int *)cpl_calloc(orders_nb_per_fibre[i], sizeof(int));
        coeffs_per_order[i] = (double **)cpl_malloc(orders_nb_per_fibre[i] * sizeof(double*));
        rms_per_order[i] = (double *)cpl_calloc(orders_nb_per_fibre[i], sizeof(double));
        chisq_per_order[i] = (double *)cpl_calloc(orders_nb_per_fibre[i], sizeof(double));
        order_fit[i] = (int *)cpl_calloc(orders_nb_per_fibre[i], sizeof(int));
        FP_lines_nb[i] = 0;
        for (int j = 0; j < orders_nb_per_fibre[i]; j++) {
            coeffs_per_order[i][j] = (double *)cpl_calloc(inst_config->wave_sol_poly_deg, sizeof(double));
            for (int k = 0; k < inst_config->wave_sol_poly_deg; k++) {
                coeffs_per_order[i][j][k] = 0.0;
            }
        }
        coeffs_table_FITS[i] = cpl_table_new(orders_nb_per_fibre[i]);
        my_error = cpl_table_new_column(coeffs_table_FITS[i], COL_NAME_ORDER, CPL_TYPE_INT);
        for (int j = 0; j < inst_config->wave_sol_poly_deg; j++) {
            sprintf(column_name, "COEFF_%d", j);
            my_error = cpl_table_new_column(coeffs_table_FITS[i], column_name, CPL_TYPE_DOUBLE);
        }

        if (fibre_source[i] == THAR) {
            wave_matrix_sol_data = (double *)cpl_calloc(orders_nb_per_fibre[i] * size_x, sizeof(double));
            dll_matrix_sol_data = (double *)cpl_calloc(orders_nb_per_fibre[i] * size_x, sizeof(double));
            air_wave_matrix_sol_data = (double *)cpl_calloc(orders_nb_per_fibre[i] * size_x, sizeof(double));
            air_dll_matrix_sol_data = (double *)cpl_calloc(orders_nb_per_fibre[i] * size_x, sizeof(double));
            for (int j = 0; j < orders_nb_per_fibre[i]*size_x; j++) {
                wave_matrix_sol_data[j] = 0.0;
                dll_matrix_sol_data[j] = 0.0;
                air_wave_matrix_sol_data[j] = 0.0;
                air_dll_matrix_sol_data[j] = 0.0;
            }
        }
    }

    for (int i = 0; i < inst_config->fibres_nb; i++) {
        if (fibre_source[i] == THAR) {
            d_static = (double *)cpl_calloc(MAX_NB_SPECTRAL_LINES, sizeof(double));
            d_static_err = (double *)cpl_calloc(MAX_NB_SPECTRAL_LINES, sizeof(double));
            d_static_err_no_min = (double *)cpl_calloc(MAX_NB_SPECTRAL_LINES, sizeof(double));
            // THAR
            // Read the static lines table and put it into lines_table
            //my_error = espdr_read_REF_LINES_TABLE_ASCII(
            //my_error = espdr_read_REF_LINES_TABLE_CSV(
            my_error = espdr_read_REF_LINES_TABLE(
                                    REF_line_table_frame[i],
                                    REF_lines_table,
                                    d_static,
                                    d_static_err,
                                    d_static_err_no_min,
                                    &REF_lines_nb);

            //for (j = 0; j < REF_lines_nb; j++) {
             //espdr_msg("order = %d, pxl: %f, grouping: %d",
             //          REF_lines_table[j].order, REF_lines_table[j].peak_pxl, REF_lines_table[j].grouping);
             //espdr_msg("x0 = %f",REF_lines_table[j].peak_pxl);
             //espdr_msg("sig_x0 = %f",REF_lines_table[j].peak_pxl_err);
             //espdr_msg("fwhm = %f",REF_lines_table[j].fwhm);
             //espdr_msg("sig_fwhm = %f",REF_lines_table[j].fwhm_err);
             //espdr_msg("k = %f",REF_lines_table[j].peak_flux);
             //espdr_msg("sig_k = %f",REF_lines_table[j].peak_flux_err);
             //espdr_msg("qc = %d",REF_lines_table[j].qc_flag);
             //espdr_msg("ll = %f",REF_lines_table[j].wavelength);
             //espdr_msg("sig_ll = %f",REF_lines_table[j].wavelength_err);
             //espdr_msg("dlldx = %f",REF_lines_table[j].dispersion);
             //espdr_msg("grouping = %d",REF_lines_table[j].grouping);
             //espdr_msg("col_elem = %s",REF_lines_table[j].element_name);
            //}

            espdr_msg("REF lines nb for fibre %c: %d",
                      fibre_name[i], REF_lines_nb);

#if SAVE_DEBUG_PRODUCT_WAVE_THAR
            my_error = espdr_save_debug_table_RDB_short(REF_lines_table, REF_lines_nb,
                                                        "THAR_READ_LINE_TABLE",
                                                        i, inst_config);
#endif

            my_error = espdr_read_FP_LINES_TABLE(
            //my_error = espdr_read_FP_LINES_TABLE_ASCII(
                                FP_line_table_frame[i],
                                FP_lines_table[i], &FP_lines_nb[i]);
            espdr_msg("FP lines nb for fibre %c: %d",
                      fibre_name[i], FP_lines_nb[i]);

            // Fit REF lines
            espdr_ensure((flat_blaze_corr_spectrum[i] == NULL) ||
                         (flat_blaze_corr_error[i] == NULL) ||
                         (flat_blaze_corr_qual[i] == NULL),
                         CPL_ERROR_NULL_INPUT,
                         "FLAT & BLAZE corrected spectrum for fibre %c is NULL", fibre_name[i]);

            espdr_msg("Fitting Th REF lines");
            my_error = espdr_fit_REF_lines(REF_lines_table,
                                           REF_lines_nb,
                                           flat_blaze_corr_spectrum[i],
                                           flat_blaze_corr_error[i],
                                           flat_blaze_corr_qual[i],
                                           pixel_geom_image[i],
                                           inst_config->thar_peak_flux_threshold,
                                           inst_config->thar_thar_fit_window_size,
                                           inst_config->thar_fit_tolerance_window_size,
                                           lines_nb_per_order[i],
                                           &TH_saturated_lines_nb,
                                           &TH_flux_ratio_median,
                                           &TH_rv_diff_median);
            //espdr_msg("Valid lines per order after fit:");
            //for (j = 0; j < orders_nb_per_fibre[i]; j++) {
            //    espdr_msg("Order %d: lines nb: %d",
            //              j+1, lines_nb_per_order[i][j]);
            //}
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_fit_lines failed: %s",
                         cpl_error_get_message_default(my_error));
            espdr_msg("espdr_fit_lines done");

            my_error = espdr_THAR_lines_qc(REF_lines_table,
                                           REF_lines_nb,
                                           inst_config,
                                           lines_nb_per_order[i]);

#if SAVE_DEBUG_PRODUCT_WAVE_THAR
            my_error = espdr_save_debug_table_RDB_long(REF_lines_table,
                                                       d_static,
                                                       d_static_err,
                                                       d_static_err_no_min,
                                                       REF_lines_nb,
                                                       "THAR_FITTED_LINE_TABLE",
                                                       i, inst_config);
#endif

            if (compute_drift == 1) {
                /* DRIFT computation */
                drift_matrix = cpl_image_new(cpl_image_get_size_x(flat_blaze_corr_spectrum[(i+1)%2]),
                                             cpl_image_get_size_x(flat_blaze_corr_spectrum[(i+1)%2]),
                                             CPL_TYPE_DOUBLE);
                int first_order_second_ext;
                drift_table = cpl_table_new(CCD_geom->ext_nb);
                my_error = espdr_drift_table_init(drift_table, CCD_geom, inst_config, (i+1)%2,
                                                  &first_order_second_ext);

                my_error = espdr_measure_drift(flat_corr_spectrum[(i+1)%2],
                                               flat_corr_error[(i+1)%2],
                                               flat_corr_qual[(i+1)%2],
                                               s2d_blaze_FP_spectrum,
                                               s2d_blaze_FP_error,
                                               s2d_blaze_FP_qual,
                                               static_wave_matrix_img[(i+1)%2],
                                               static_dll_matrix_img[(i+1)%2],
                                               inst_config->drift_method_fp,
                                               inst_config->drift_space,
                                               inst_config->drift_ksigma,
                                               inst_config->drift_max_flux_threshold,
                                               inst_config->drift_min_flux_threshold,
                                               inst_config->image_cosmics_part,
                                               &drift_table,
                                               &drift_matrix);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_measure_drift failed for fibre %c: %s",
                             fibre_name[(i+1)%2], cpl_error_get_message_default(my_error));

                espdr_msg("Drift computed, applying");
                /* Apply the computed drift to the FP_lines_table */
                double drift_mean[CCD_geom->ext_nb], drift_slope_o[CCD_geom->ext_nb], drift_slope_x[CCD_geom->ext_nb];
                double drift_slope_o_err[CCD_geom->ext_nb], drift_slope_x_err[CCD_geom->ext_nb];
                int first_drift_order[CCD_geom->ext_nb], last_drift_order[CCD_geom->ext_nb];
                for (int j = 0; j < CCD_geom->ext_nb; j++) {
                    first_drift_order[j] = cpl_table_get_int(drift_table, "first_order", j, NULL);
                    last_drift_order[j] = cpl_table_get_int(drift_table, "last_order", j, NULL);
                    drift_mean[j] = cpl_table_get_double(drift_table, "drift_mean", j, NULL);
                    drift_slope_o[j] = cpl_table_get_double(drift_table, "drift_slope_o", j, NULL);
                    drift_slope_o_err[j] = cpl_table_get_double(drift_table, "drift_slope_o_err", j, NULL);
                    drift_slope_x[j] = cpl_table_get_double(drift_table, "drift_slope_x", j, NULL);
                    drift_slope_x_err[j] = cpl_table_get_double(drift_table, "drift_slope_x_err", j, NULL);
                    espdr_msg("DRIFT DET%d: mean: %.10f, slope_o: %.10f, slope_o_err: %.10f, slope_x: %.10f, slope_x_err: %.10f, orders: %d <-> %d",
                              j, drift_mean[j], drift_slope_o[j], drift_slope_o_err[j],
                              drift_slope_x[j], drift_slope_x_err[j],
                              first_drift_order[j], last_drift_order[j]);
                }
                for (int j = 0; j < FP_lines_nb[i]; j++) {
                    if (FP_lines_table[i][j].order < first_order_second_ext) {
                        FP_lines_table[i][j].peak_pxl = FP_lines_table[i][j].peak_pxl + drift_mean[0] +
                        drift_slope_x[0] * (FP_lines_table[i][j].peak_pxl-1) +
                        drift_slope_o[0] * (FP_lines_table[i][j].order-1);
                    } else {
                        FP_lines_table[i][j].peak_pxl = FP_lines_table[i][j].peak_pxl + drift_mean[1] +
                        drift_slope_x[1] * (FP_lines_table[i][j].peak_pxl-1) +
                        drift_slope_o[1] * (FP_lines_table[i][j].order-1);
                    }
                }

                cpl_image_delete(drift_matrix);
                espdr_msg("Drift done");
            }

#if SAVE_DEBUG_PRODUCT_WAVE_THAR
            my_error = espdr_save_debug_table_RDB(FP_lines_table[i], FP_lines_nb[i],
                                                  "FP_DRIFTED_LINE_TABLE",
                                                  i, inst_config);
#endif

            espdr_msg("Checking FP peaks for fibre %c", fibre_name[i]);
            my_error = espdr_check_delta_FP_peaks(FP_lines_table[i], FP_lines_nb[i],
                                                  inst_config->delta_fp_tolerance);
            espdr_ensure(my_error == CPL_ERROR_INCOMPATIBLE_INPUT, my_error,
                         "Missing peaks in the FP image, aborting");

            espdr_msg("Wavelength solution computation start");

            espdr_msg("Finding the first FP peak ll");
            my_error = espdr_find_first_FP_ll(REF_lines_table, REF_lines_nb,
                                              inst_config->fp_d_angstrom,
                                              FP_lines_table[i], FP_lines_nb[i]);

            int CCD_middle = cpl_image_get_size_x(flat_blaze_corr_spectrum[i])/2;
            my_error = espdr_get_all_FP_ll_per_order(FP_lines_table[i],
                                                     FP_lines_nb[i], CCD_middle,
                                                     inst_config->fp_d_angstrom);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_get_all_FP_ll_per_order failed: %s",
                         cpl_error_get_message_default(my_error));

            espdr_msg("Getting the ll in the whole image");
            my_error = espdr_get_all_FP_ll_whole_image(FP_lines_table[i],
                                                       FP_lines_nb[i],
                                                       inst_config,
                                                       orders_nb_per_fibre[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_get_all_FP_ll_whole_image failed: %s",
                         cpl_error_get_message_default(my_error));

#if SAVE_DEBUG_PRODUCT_WAVE_THAR
            my_error = espdr_save_debug_table_RDB(FP_lines_table[i], FP_lines_nb[i],
                                                  "FP_FITTED_LINE_TABLE_first",
                                                  i, inst_config);
#endif

            // Global FP lambda fit
            d_computed = (double *)cpl_calloc(REF_lines_nb, sizeof(double));
            d_computed_err = (double *)cpl_calloc(REF_lines_nb, sizeof(double));
            d_computed_err_no_min = (double *)cpl_calloc(REF_lines_nb, sizeof(double));
            residuals = (double *)cpl_calloc(REF_lines_nb, sizeof(double));
            dm = (double *)cpl_calloc(FP_lines_nb[i], sizeof(double));
            espdr_msg("REF lines nb before fit lambda = %d", REF_lines_nb);
            espdr_msg("Fitting D(lambda)");
            my_error = espdr_fit_d_lambda(REF_lines_table, REF_lines_nb,
                                          FP_lines_table[i], FP_lines_nb[i],
                                          inst_config,
                                          d_static, d_static_err, d_static_err_no_min,
                                          d_computed, d_computed_err, d_computed_err_no_min,
                                          residuals, dm);
            espdr_msg("REF lines nb after fit lambda = %d", REF_lines_nb);

            int test_nb_of_qc0 = 0;
            for (int j = 0; j < FP_lines_nb[i]; j++) {
                if (FP_lines_table[i][j].qc_flag == 0) {
                    test_nb_of_qc0++;
                }
            }
            espdr_msg("Nb of qc = 0 in FP lines before ll_sol: %d", test_nb_of_qc0);

            espdr_msg("Fitting the wavelength solution for fibre %c", fibre_name[i]);
            my_error = espdr_fit_ll_sol(FP_lines_table[i],
                                        FP_lines_nb[i],
                                        inst_config,
                                        pixel_geom_image[i],
                                        pixel_size_image[i],
                                        fibre_source[i],
                                        i,
                                        lines_nb_per_order[i],
                                        rms_per_order[i],
                                        chisq_per_order[i],
                                        order_fit[i],
                                        coeffs_per_order[i],
                                        wave_matrix_sol_data,
                                        dll_matrix_sol_data,
                                        air_wave_matrix_sol_data,
                                        air_dll_matrix_sol_data);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_fit_ll_sol failed for fibre %c: %s",
                         fibre_name[i], cpl_error_get_message_default(my_error));

            test_nb_of_qc0 = 0;
            for (int j = 0; j < FP_lines_nb[i]; j++) {
                if (FP_lines_table[i][j].qc_flag == 0) {
                    test_nb_of_qc0++;
                }
            }
            espdr_msg("Nb of qc = 0 in FP lines after ll_sol: %d", test_nb_of_qc0);

            espdr_msg("REF lines nb before dlldx = %d", REF_lines_nb);
            espdr_msg("Computing dlldx for THAR lines for fibre %c",
                      fibre_name[i]);
            my_error = espdr_compute_dlldx(REF_lines_table,
                                           REF_lines_nb,
                                           orders_nb_per_fibre[i],
                                           inst_config,
                                           coeffs_per_order[i]);
            espdr_msg("REF lines nb after dlldx = %d", REF_lines_nb);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_compute_dlldx failed: %s",
                         cpl_error_get_message_default(my_error));

            // Compute the resolution of Thorium lines
            espdr_msg("REF lines nb before resolution = %d", REF_lines_nb);
            resolution = (double *)cpl_calloc(REF_lines_nb, sizeof(double));;
            my_error = espdr_compute_THAR_resolution(REF_lines_table,
                                                     resolution,
                                                     REF_lines_nb,
                                                     &resol_median);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_compute_THAR_resolution failed: %s",
                         cpl_error_get_message_default(my_error));
            espdr_msg("REF lines nb after resolution = %d", REF_lines_nb);

            espdr_msg("Computing dlldx for FP lines for fibre %c", fibre_name[i]);
            my_error = espdr_compute_dlldx(FP_lines_table[i],
                                           FP_lines_nb[i],
                                           orders_nb_per_fibre[i],
                                           inst_config,
                                           coeffs_per_order[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_compute_dlldx failed: %s",
                         cpl_error_get_message_default(my_error));

            /* Create vacuum and air WAVE_MATRIX images */
            size_x = cpl_image_get_size_x(flat_blaze_corr_spectrum[i]);
            espdr_msg("size_x = %d", size_x);
            wave_matrix_sol = cpl_image_wrap(size_x, orders_nb_per_fibre[i],
                                             CPL_TYPE_DOUBLE,
                                             wave_matrix_sol_data);
            espdr_msg("WAVE_MATRIX size: %lld x %lld",
                      cpl_image_get_size_x(wave_matrix_sol),
                      cpl_image_get_size_y(wave_matrix_sol));

            air_wave_matrix_sol = cpl_image_wrap(size_x, orders_nb_per_fibre[i],
                                                 CPL_TYPE_DOUBLE,
                                                 air_wave_matrix_sol_data);
            espdr_msg("WAVE_MATRIX size: %lld x %lld",
                      cpl_image_get_size_x(air_wave_matrix_sol),
                      cpl_image_get_size_y(air_wave_matrix_sol));

            /* Create vacuum and air DLL_MATRIX image */
            dll_matrix_sol = cpl_image_wrap(size_x, orders_nb_per_fibre[i],
                                            CPL_TYPE_DOUBLE,
                                            dll_matrix_sol_data);

            air_dll_matrix_sol = cpl_image_wrap(size_x, orders_nb_per_fibre[i],
                                                CPL_TYPE_DOUBLE,
                                                air_dll_matrix_sol_data);

            /* Create WAVE_TABLE fits */
            for (int j = 0; j < orders_nb_per_fibre[i]; j++) {
                my_error = cpl_table_set_int(coeffs_table_FITS[i],
                                             COL_NAME_ORDER, j, j+1);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "cpl_table_set_int failed for order %d: %s",
                             j, cpl_error_get_message_default(my_error));
                for (int k = 0; k < inst_config->wave_sol_poly_deg; k++) {
                    sprintf(column_name, "COEFF_%d", k);
                    my_error = cpl_table_set_double(coeffs_table_FITS[i],
                                                    column_name, j,
                                                    coeffs_per_order[i][j][k]);
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "cpl_table_set_double failed for coeff %d: %s",
                                 k, cpl_error_get_message_default(my_error));
                }
            }

#if SAVE_DEBUG_PRODUCT_WAVE_THAR
            my_error = espdr_save_debug_table_RDB(FP_lines_table[i], FP_lines_nb[i],
                                                  "FP_FITTED_LINE_TABLE",
                                                  i, inst_config);
#endif

#if PROCESS_RAW_LINES
            /* Fitting raw lines */
            espdr_msg("Fitting raw lines for fibre %c", fibre_name[i]);
            my_error = espdr_fit_THAR_raw_lines(REF_lines_table,
                                                REF_lines_nb,
                                                CCD_corrected_image,
                                                i,
                                                orders_coeffs[i],
                                                pixel_geom_image[i],
                                                inst_config->thar_thar_fit_window_size,
                                                inst_config,
                                                CCD_geom,
                                                RON,
                                                &raw_lines_nb,
                                                raw_lines_table);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_fit_THAR_raw_lines failed for fibre %c: %s",
                         fibre_name[i], cpl_error_get_message_default(my_error));
            espdr_msg("raw lines fitted");

            raw_lines_table_FITS = cpl_table_new(raw_lines_nb);
            my_error = espdr_FITS_raw_lines_table_create(raw_lines_table_FITS);
            my_error = espdr_FITS_raw_lines_table_fill(raw_lines_table_FITS,
                                                       raw_lines_table,
                                                       raw_lines_nb);
            espdr_msg("Raw lines table has %lld rows",
                      cpl_table_get_nrow(raw_lines_table_FITS));
#endif

#if SAVE_DEBUG_PRODUCT_WAVE_THAR
            my_error = espdr_save_debug_table_RDB_ll_sol(REF_lines_table,
                                                         d_computed,
                                                         d_computed_err,
                                                         d_computed_err_no_min,
                                                         //d_static,
                                                         //d_static_err,
                                                         //d_static_err_no_min,
                                                         residuals,
                                                         resolution,
                                                         REF_lines_nb,
                                                         "THAR_LL_SOL_LINE_TABLE",
                                                         i, inst_config);
#endif

        } else {
            espdr_msg("Source on fibre %c is %s --> doing nothing",
                      fibre_name[i], fibre_WAVE[i]);

            for (int j = 0; j < orders_nb_per_fibre[i]; j++) {
                lines_nb_per_order[i][j] = -1;
                chisq_per_order[i][j] = -1.0;
                rms_per_order[i][j] = -1.0;
                order_fit[i][j] = -1.0;
            }
        } // end if fibre[i] == THAR

        espdr_msg("Finished with fibre %c", fibre_name[i]);
    } // for loop on fibres

    espdr_msg("Create FITS ll sol table");
    THAR_lines_table_FITS = cpl_table_new(REF_lines_nb);
    my_error = espdr_FITS_ll_sol_table_create(THAR_lines_table_FITS);
    my_error = espdr_FITS_ll_sol_table_fill(THAR_lines_table_FITS,
                                            REF_lines_table,
                                            d_computed,
                                            d_computed_err,
                                            d_computed_err_no_min,
                                            residuals,
                                            resolution,
                                            REF_lines_nb);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "FITS REF table creation failed: %s",
                 cpl_error_get_message_default(my_error));

    FP_lines_table_FITS = (cpl_table **)cpl_malloc
                        (inst_config->fibres_nb * sizeof(cpl_table*));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        if (fibre_source[i] == THAR) {
            FP_lines_table_FITS[i] = cpl_table_new(FP_lines_nb[i]);
            my_error = espdr_FITS_lines_table_create(FP_lines_table_FITS[i]);
            my_error = espdr_FITS_lines_table_fill(FP_lines_table_FITS[i],
                                                   FP_lines_table[i],
                                                   FP_lines_nb[i]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "FITS FP table creation for fibre %c failed: %s",
                         fibre_name[i], cpl_error_get_message_default(my_error));
            espdr_msg("FP line table for fibre %c has %lld rows",
                      fibre_name[i],
                      cpl_table_get_nrow(FP_lines_table_FITS[i]));
        }
    }

    espdr_msg("Adding QC KWs");
    int valid_THAR_lines_nb = 0, nonvalid_THAR_lines_nb = 0;
    for (int i = 0; i < REF_lines_nb; i++) {
        if (REF_lines_table[i].qc_flag == 1) {
            valid_THAR_lines_nb++;
        } else {
            nonvalid_THAR_lines_nb++;
        }
    }
    espdr_msg("Valid THAR lines nb: %d, nonvalid: %d",
              valid_THAR_lines_nb, nonvalid_THAR_lines_nb);

    for (int i = 0; i < inst_config->fibres_nb; i++) {
        my_error = espdr_wave_THAR_QC(lines_nb_per_order[i],
                                      chisq_per_order[i],
                                      rms_per_order[i],
                                      order_fit[i],
                                      resol_median,
                                      orders_nb_per_fibre[i],
                                      TH_saturated_lines_nb,
                                      valid_THAR_lines_nb,
                                      TH_flux_ratio_median,
                                      TH_rv_diff_median,
                                      fibre_source[i],
                                      i,
                                      drift_table,
                                      wave_matrix_sol, flat_corr_qual[i],
                                      inst_config, CCD_geom, qc_kws,
                                      keywords_fibre[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_wave_THAR_QC failed: %s for fibre %c",
                     cpl_error_get_message_default(my_error),
                     fibre_name[i]);

        // always use the FP table from the fibre with THAR source
        int index_ll = 0;
        if (fibre_source[i] == THAR) {
            index_ll = i;
        } else {
            index_ll = (i+1)%2;
        }

        espdr_msg("from wave_matrix_sol_data: first ll: %f, last ll :%f",
                  wave_matrix_sol_data[0],
                  wave_matrix_sol_data[orders_nb_per_fibre[index_ll] * size_x - 1]);
        my_error = espdr_wave_FP_D_QC(REF_lines_table,
                                      REF_lines_nb,
                                      FP_lines_table[index_ll],
                                      FP_lines_nb[index_ll],
                                      d_computed_err,
                                      residuals,
                                      dm,
                                      wave_matrix_sol_data[0],
                                      wave_matrix_sol_data[orders_nb_per_fibre[index_ll]*size_x-1],
                                      inst_config, qc_kws,
                                      keywords_fibre[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_wave_FP_D_QC failed: %s for fibre %c",
                     cpl_error_get_message_default(my_error),
                     fibre_name[i]);
    }

    for (int i = 0; i < inst_config->fibres_nb; i++) {
        // Put the global wave check on the non-THAR fiber as on the THAR fiber
        if (fibre_source[i] != THAR) {
            int wave_check_THAR = cpl_propertylist_get_int(keywords_fibre[(i+1)%2],
                                                           qc_kws->qc_wave_check_kw);
            my_error = cpl_propertylist_update_int(keywords_fibre[i],
                                                   qc_kws->qc_wave_check_kw,
                                                   wave_check_THAR);
        }
    }

    for (int i = 0; i < inst_config->fibres_nb; i++) {

        my_error = espdr_save_wave_S2D_products(frameset, parameters, recipe_id,
                                                used_frames, WAVE_tag, i,
                                                keywords_fibre[i],
                                                inst_config,
                                                flat_corr_spectrum[i],
                                                flat_corr_error[i],
                                                flat_corr_qual[i],
                                                flat_blaze_corr_spectrum[i],
                                                flat_blaze_corr_error[i],
                                                flat_blaze_corr_qual[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_save_wave_S2D_products failed: %s for fibre %c",
                     cpl_error_get_message_default(my_error), fibre_name[i]);

        if (fibre_source[i] == THAR) {
            if ((strcmp(inst_config->instrument, "HARPN") != 0) &&
                (strcmp(inst_config->instrument, "CORALIE") != 0)) {
                my_error = espdr_add_extra_qc(&(keywords_fibre)[i], CCD_geom->ext_nb,
                                              mjd_obs_delta_time_fp);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_add_extra_qc failed: %s ",
                             cpl_error_get_message_default(my_error));
            }

            my_error = espdr_save_wave_THAR_products(frameset, parameters, recipe_id,
                                                     used_frames, WAVE_tag, i,
                                                     keywords_fibre[i],
                                                     inst_config,
                                                     wave_matrix_sol,
                                                     air_wave_matrix_sol,
                                                     dll_matrix_sol,
                                                     air_dll_matrix_sol,
                                                     coeffs_table_FITS[i],
                                                     FP_lines_table_FITS[i],
                                                     THAR_lines_table_FITS,
                                                     raw_lines_table_FITS);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_save_wave_THAR_products failed: %s for fibre %c",
                         cpl_error_get_message_default(my_error),
                         fibre_name[i]);

#if SAVE_TH_REF_TABLES & SAVE_DEBUG_PRODUCT_WAVE_THAR
            my_error = espdr_save_debug_table_RDB_ll_sol(REF_lines_table,
                                                         d_computed,
                                                         d_computed_err,
                                                         d_computed_err_no_min,
                                                         residuals,
                                                         resolution,
                                                         REF_lines_nb,
                                                         "THAR_LL_SOL_LINE_TABLE",
                                                         i, inst_config);
#endif

#if SAVE_DEBUG_PRODUCT_WAVE_THAR
            my_error = espdr_save_debug_wave_table_RDB(coeffs_per_order[i],
                                                       orders_nb_per_fibre[i],
                                                       "WAVE_TABLE",
                                                       i, inst_config);
#endif

#if PROCESS_RAW_LINES && SAVE_DEBUG_PRODUCT_WAVE_THAR
            my_error = espdr_save_debug_raw_table_RDB(raw_lines_table,
                                                      raw_lines_nb,
                                                      "LINE_TABLE_RAW",
                                                      i, inst_config);
#endif

        }
    }

    espdr_msg("Cleaning the memory");
    espdr_msg("Cleaning keywords lists");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_propertylist_delete(keywords_fibre[i]);
    }
	cpl_free(qc_kws);

    espdr_msg("Cleaning calibration products");
	cpl_imagelist_delete(CCD_corrected_image);

    if (compute_drift == 1) {
        for (int i = 0; i < inst_config->fibres_nb; i++) {
            if (fibre_source[i] == THAR) {
                cpl_image_delete(static_wave_matrix_img[(i+1)%2]);
                cpl_image_delete(static_dll_matrix_img[(i+1)%2]);
            }
        }
        cpl_free(static_wave_matrix_img);
        cpl_free(static_dll_matrix_img);
        cpl_image_delete(s2d_blaze_FP_spectrum);
        cpl_image_delete(s2d_blaze_FP_error);
        cpl_image_delete(s2d_blaze_FP_qual);
    }

    cpl_frameset_delete(used_frames);
    cpl_free(REF_line_table_frame);
    cpl_free(FP_line_table_frame);

    espdr_msg("Cleaning orders coeffs");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            cpl_table_delete(orders_coeffs[i][j]);
        }
        cpl_free(orders_coeffs[i]);
    }
    cpl_free(orders_coeffs);

    espdr_msg("Cleaning products");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_image_delete(flat_corr_spectrum[i]);
        cpl_image_delete(flat_corr_error[i]);
        cpl_image_delete(flat_corr_qual[i]);
    }
    cpl_free(flat_corr_spectrum);
    cpl_free(flat_corr_error);
    cpl_free(flat_corr_qual);

    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_image_delete(flat_blaze_corr_spectrum[i]);
        cpl_image_delete(flat_blaze_corr_error[i]);
        cpl_image_delete(flat_blaze_corr_qual[i]);
    }
    cpl_free(flat_blaze_corr_spectrum);
    cpl_free(flat_blaze_corr_error);
    cpl_free(flat_blaze_corr_qual);

    espdr_msg("Cleaning pixel geometry & size pointers");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_image_delete(pixel_geom_image[i]);
        cpl_image_delete(pixel_size_image[i]);
    }
    cpl_free(pixel_geom_image);
    cpl_free(pixel_size_image);

    espdr_msg("Cleaning global variables");
    cpl_free(RON);

    espdr_msg("Cleaning global QC structures");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_free(lines_nb_per_order[i]);
        cpl_free(rms_per_order[i]);
        cpl_free(chisq_per_order[i]);
        cpl_free(order_fit[i]);
        for (int j = 0; j < orders_nb_per_fibre[i]; j++) {
            cpl_free(coeffs_per_order[i][j]);
        }
        cpl_free(coeffs_per_order[i]);
    }
    cpl_free(lines_nb_per_order);
    cpl_free(rms_per_order);
    cpl_free(chisq_per_order);
    cpl_free(order_fit);
    cpl_free(coeffs_per_order);
    cpl_free(orders_nb_per_fibre);

    espdr_msg("Cleaning additional data tables");
    cpl_free(d_computed);
    cpl_free(d_computed_err);
    cpl_free(d_computed_err_no_min);
    cpl_free(residuals);
    cpl_free(resolution);
    cpl_free(d_static);
    cpl_free(d_static_err);
    cpl_free(d_static_err_no_min);

    espdr_msg("Cleaning matrices data structures");
    cpl_image_unwrap(wave_matrix_sol);
    cpl_image_unwrap(dll_matrix_sol);
    cpl_image_unwrap(air_wave_matrix_sol);
    cpl_image_unwrap(air_dll_matrix_sol);
    cpl_free(wave_matrix_sol_data);
    cpl_free(dll_matrix_sol_data);
    cpl_free(air_wave_matrix_sol_data);
    cpl_free(air_dll_matrix_sol_data);

    espdr_msg("Cleaning lines tables");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_free(FP_lines_table[i]);
    }
    cpl_free(REF_lines_table);
    cpl_free(raw_lines_table);
    cpl_free(FP_lines_table);

    espdr_msg("Cleaning FITS tables");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_table_delete(coeffs_table_FITS[i]);
        if (fibre_source[i] == THAR) {
            cpl_table_delete(FP_lines_table_FITS[i]);
        }
    }
#if PROCESS_RAW_LINES
    cpl_table_delete(raw_lines_table_FITS);
#endif
    cpl_table_delete(THAR_lines_table_FITS);
    cpl_free(coeffs_table_FITS);
    cpl_free(FP_lines_table_FITS);
    cpl_table_delete(drift_table);

    espdr_msg("Cleaning parameters");
    espdr_parameters_OVSC_delete(OVSC_param);
    espdr_parameters_WAVE_THAR_delete(WAVE_THAR_param);
    espdr_parameters_CCD_geometry_delete(CCD_geom);
    espdr_parameters_inst_config_delete(inst_config);

    espdr_msg("End of the recipe espdr_wave_THAR");
	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create the structure for WAVE input parameters
 @param    recipe_id	recipe identifier
 @param    list			list of input parameters
 @param    p			input WAVE parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_THAR_create(const char* recipe_id,
                                                 cpl_parameterlist *list,
                                                 espdr_WAVE_THAR_param *p) {
    
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    cpl_error_code my_error = cpl_error_get_code();
	
	/* Fill the parameter list */
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "The parameters list NULL");
	double ksig_extraction_min = KSIG_EXTRACTION_MIN;
	double ksig_extraction_max = KSIG_EXTRACTION_MAX;
    
	/* Fill the parameter list */
    cpl_error_code error_got = CPL_ERROR_NONE;
    char comment[COMMENT_LENGTH];
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "thar_extraction_method",
                                            p->extraction_method,
                                            "Method used to extract orders");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter extraction_method to the list");
    
    sprintf(comment, "ksigma for extraction, must be between: %.2lf and %.2lf",
            ksig_extraction_min, ksig_extraction_max);
    error_got = espdr_parameters_new_range_double(recipe_id, list,
                                                  "thar_extraction_ksigma",
                                                  p->ksig_extraction,
                                                  KSIG_EXTRACTION_MIN,
                                                  KSIG_EXTRACTION_MAX,
                                                  comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter ksig_extraction to the list");
    
    sprintf(comment,
            "d fit poly deg in wave_THAR, must be between: %d and %d",
            D_FIT_POLY_DEG_MIN, D_FIT_POLY_DEG_MAX);
    my_error = espdr_parameters_new_range_int(recipe_id, list,
                                              "d_fit_poly_deg",
                                              p->d_fit_poly_deg,
                                              D_FIT_POLY_DEG_MIN,
                                              D_FIT_POLY_DEG_MAX,
                                              comment);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Error adding parameter d_fit_poly_deg to the list");
    
    sprintf(comment,
            "d fit small res_limit in wave_THAR, must be between: %d and %d",
            D_FIT_SMALL_RES_LIMIT_MAX, D_FIT_SMALL_RES_LIMIT_MAX);
    my_error = espdr_parameters_new_range_double(recipe_id, list,
                                                 "d_fit_small_res_limit",
                                                 p->d_fit_small_res_limit,
                                                 D_FIT_SMALL_RES_LIMIT_MIN,
                                                 D_FIT_SMALL_RES_LIMIT_MAX,
                                                 comment);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Error adding parameter d_fit_poly_deg to the list");
    
    sprintf(comment,
            "wavelength solution poly deg in wave_THAR, must be between: %d and %d",
            WAVE_SOL_POLY_DEG_MIN, WAVE_SOL_POLY_DEG_MIN);
    my_error = espdr_parameters_new_range_int(recipe_id, list,
                                              "wave_sol_poly_deg",
                                              p->wave_sol_poly_deg,
                                              WAVE_SOL_POLY_DEG_MIN,
                                              WAVE_SOL_POLY_DEG_MAX,
                                              comment);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Error adding parameter d_fit_poly_deg to the list");
    
    return (CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the structure for ORDERDEF input parameters
 @param    p	input ORDERDEF parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_THAR_delete(espdr_WAVE_THAR_param* p) {
	
    cpl_free((void *)p->extraction_method);
    cpl_free(p);
    p = NULL;
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Get the WAVE recipe parameters
 @param     recipe_id   recipe ID
 @param     param_list  parameters list
 @param     WAVE_param  WAVE parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_THAR_get(const char* recipe_id,
                                              cpl_parameterlist* param_list,
                                              espdr_WAVE_THAR_param *WAVE_THAR_param) {
	
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(param_list == NULL, CPL_ERROR_NULL_INPUT,
				 "Parameters list is NULL");
	
	/* Fill the structure */
    WAVE_THAR_param->extraction_method = espdr_parameters_get_string(recipe_id,
                                                                     param_list,
                                                                     "thar_extraction_method");
    
    WAVE_THAR_param->ksig_extraction = espdr_parameters_get_double(recipe_id,
                                                                   param_list,
                                                                   "thar_extraction_ksigma");
    
    WAVE_THAR_param->d_fit_poly_deg = espdr_parameters_get_int(recipe_id,
                                                               param_list,
                                                               "d_fit_poly_deg");
    
    WAVE_THAR_param->d_fit_small_res_limit = espdr_parameters_get_double(recipe_id,
                                                                         param_list,
                                                                         "d_fit_small_res_limit");
    
    WAVE_THAR_param->wave_sol_poly_deg = espdr_parameters_get_int(recipe_id,
                                                                  param_list,
                                                                  "wave_sol_poly_deg");
    
	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     print the WAVE parameters
 @param     WAVE_param  WAVE parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_THAR_print(espdr_WAVE_THAR_param *WAVE_THAR_param) {
	
	espdr_msg("\tWAVE THAR parameters:");
    espdr_msg("\t\tWAVE_THAR extraction method = %s",
              WAVE_THAR_param->extraction_method);
    espdr_msg("\t\tWAVE_THAR extraction ksigma = %.2f",
              WAVE_THAR_param->ksig_extraction);
    espdr_msg("\t\tWAVE_THAR D(lambda) fit polynomial degree = %d",
              WAVE_THAR_param->d_fit_poly_deg);
    espdr_msg("\t\tWAVE_THAR D(lambda) fit small resilduals limit = %.2f",
              WAVE_THAR_param->d_fit_small_res_limit);
    espdr_msg("\t\tWAVE_THAR wavelength solution fit polynomial degree = %d",
              WAVE_THAR_param->wave_sol_poly_deg);

	return (cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Init WAVE_param structure
 
 @param    recipe_id    recipe name
 @param    parameters   recipe input parameters
 
 @return   allocated structure iff OK
 */
/*----------------------------------------------------------------------------*/

espdr_WAVE_THAR_param *espdr_WAVE_THAR_param_init(const char *recipe_id,
                                             cpl_parameterlist *parameters) {
    
    cpl_error_code my_error;
    
    espdr_WAVE_THAR_param *WAVE_THAR_param =
                (espdr_WAVE_THAR_param *)cpl_malloc(sizeof(espdr_WAVE_THAR_param));
    
    /* Read the wave parameters */
    my_error = espdr_parameters_WAVE_THAR_get(recipe_id, parameters, WAVE_THAR_param);
    my_error = espdr_parameters_WAVE_THAR_print(WAVE_THAR_param);
    //if(cpl_error_get_code() != CPL_ERROR_NONE) {
    if(my_error != CPL_ERROR_NONE) {
        return NULL;
    } else {
        return WAVE_THAR_param;
    }
}

#if 0
/*----------------------------------------------------------------------------*/
/**
 @brief    Read the wave recipe parameters
 @param         parameters   recipe input parameters
 @param         frameset     recipe input frames
 @param         recipe_id    recipe name
 @param[out]    CCD_geom     CCD geometry structure
 @param[out]    inst_config  instrument config
 @param[out]    OVSC_param   overscan parameter structure
 @param[out]    WAVE_param   WAVE parameter structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_wave_THAR_read_parameters(cpl_frameset *frameset,
                                               espdr_CCD_geometry *CCD_geom,
                                               espdr_inst_config *inst_config) {
	
	cpl_frameset *CCD_geom_frames = cpl_frameset_new();
	cpl_frame *CCD_geom_table_frame = NULL;
	const char *CCD_geom_filename = NULL;
    cpl_frameset *inst_config_frames = cpl_frameset_new();
    cpl_frame *inst_config_table_frame = NULL;
    const char *inst_config_filename = NULL;
	cpl_error_code my_error;
	
	/* Extract and read the CCD geometry FITS table */
	my_error = espdr_frame_extract_by_tag(frameset, ESPDR_CCD_GEOM,
                                          CCD_geom_frames);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error, 
			   "CCD_GEOM parameters frame extraction failed");
	
	espdr_ensure(CCD_geom_frames == NULL, CPL_ERROR_NULL_INPUT, 
			   "No CCD geometry parameters table %s, exiting", 
			   cpl_error_get_message());
	
    /* Extract and read the instrument configuration FITS table */
    my_error = espdr_frame_extract_by_tag(frameset, ESPDR_INST_CONFIG,
                                          inst_config_frames);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Instrument parameters frame extraction failed");
    
    espdr_ensure(inst_config_frames == NULL, CPL_ERROR_NULL_INPUT,
                 "No instrument parameters table %s, exiting",
                 cpl_error_get_message());
    
    /* Read the CCD geometry configuration file provided in SOF */
    CCD_geom_table_frame = cpl_frameset_get_position(CCD_geom_frames, 0);
    CCD_geom_filename = cpl_frame_get_filename(CCD_geom_table_frame);
    my_error = espdr_read_CCD_parameters(CCD_geom_filename, CCD_geom);
    espdr_ensure(my_error != CPL_ERROR_NONE, CPL_ERROR_INCOMPATIBLE_INPUT,
                 "Error getting CCD geometry: %s",
                 cpl_error_get_message_default(my_error));
    
    /* Read the instrument configuration file provided in SOF */
    inst_config_table_frame = cpl_frameset_get_position(inst_config_frames, 0);
    inst_config_filename = cpl_frame_get_filename(inst_config_table_frame);
    my_error = espdr_read_inst_parameters(inst_config_filename,
                                          CCD_geom->ext_nb,
                                          inst_config);
    espdr_ensure(my_error != CPL_ERROR_NONE, CPL_ERROR_INCOMPATIBLE_INPUT,
                 "Error getting instrument configuration: %s",
                 cpl_error_get_message_default(my_error));
    
	cpl_frameset_delete(CCD_geom_frames);
    cpl_frameset_delete(inst_config_frames);
	
	return cpl_error_get_code();
}
#endif



/*----------------------------------------------------------------------------*/
/**
 @brief Read WAVE_LINES_TABLE_ASCII from RDB table
 @param         wave_line_table frame containing lines table in ASCII format
 @param[out]    lines_table     read lines table
 @param[out]    lines_nb_RE     numebr of read lines
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_read_REF_LINES_TABLE_ASCII(cpl_frame *ref_line_table,
                                                espdr_line_param *lines_table,
                                                double *d_static,
                                                double *d_static_err,
                                                double *d_static_err_no_min,
                                                int *lines_nb_RE) {
    
    const char *filename = NULL;
    FILE *waveptr = NULL;
    char line[COMMENT_LENGTH];
    int lines_nb = 0;
    espdr_line_param read_line;
    double read_d_static, read_d_static_err, read_d_static_err_no_min;
    double read_residuals, read_resolution;
    
    espdr_ensure(ref_line_table == NULL, CPL_ERROR_NULL_INPUT,
                 "Input REF LINES TABLE frame is NULL");
    espdr_ensure(lines_table == NULL, CPL_ERROR_NULL_INPUT,
                 "REF LINES TABLE structure is NULL");
    
    filename = cpl_frame_get_filename(ref_line_table);
    
    espdr_msg("Reading ASCII REF lines from %s", filename);
    
    waveptr = fopen(filename, "r");
    espdr_ensure(waveptr == NULL, CPL_ERROR_FILE_IO,
                 "The file %s can't be read", filename);
    
    // Get rid of the header
    fgets(line, sizeof(line), waveptr);
    fgets(line, sizeof(line), waveptr);
    // Read the file content
    /*
    while (fscanf(waveptr,
                  "%d\t%lf\t%lf\t%lf\t%lf\t%d",
                  &lines_table[lines_nb].order,
                  &lines_table[lines_nb].peak_pxl,
                  &lines_table[lines_nb].wavelength,
                  &ll_errors[lines_nb],
                  &lines_table[lines_nb].peak_flux,
                  &lines_table[lines_nb].qc_flag) != EOF) {
    */
    while (fscanf(waveptr,
                  "%d\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%d\t%lf\t%lf\t%lf\t%d\t%s\t%lf\t%lf\t%lf\t%lf\t%lf\n",
                  &read_line.order,
                  &read_line.peak_pxl,
                  &read_line.peak_pxl_err,
                  &read_line.fwhm,
                  &read_line.fwhm_err,
                  &read_line.peak_flux,
                  &read_line.peak_flux_err,
                  &read_line.qc_flag,
                  &read_line.wavelength,
                  &read_line.wavelength_err,
                  &read_line.dispersion,
                  &read_line.grouping,
                  &read_line.element_name,
                  &read_d_static,
                  &read_d_static_err,
                  &read_d_static_err_no_min,
                  &read_residuals,
                  &read_resolution) != EOF) {
        /*
        printf("%d\t%lf\t%lf\t%lf\t%lf\t%d\n",
               read_line.order,
               read_line.peak_pxl,
               read_line.wavelength,
               read_line.wavelength_err,
               read_line.peak_flux,
               read_line.qc_flag);
        */
        
        if ((read_line.qc_flag == 1) &&
            ((strstr(read_line.element_name, "TH") != NULL) ||
             (strstr(read_line.element_name, "Th") != NULL) ||
             (strstr(read_line.element_name, "NONE") != NULL))) {
            
            lines_table[lines_nb].order = read_line.order;
            lines_table[lines_nb].peak_pxl = read_line.peak_pxl - ORDER_SHIFT - ORDER_START_SHIFT;
            lines_table[lines_nb].peak_pxl_err = read_line.peak_pxl_err;
            // reading MULTIMR84 with MULTIMR42
            //lines_table[lines_nb].peak_pxl = lines_table[lines_nb].peak_pxl/2;
            // reading MULTIMR42 with SINGLEHR lines tables
            //lines_table[lines_nb].peak_pxl = read_line.peak_pxl/2.0 + 33.0 - 5.0;
            // reading SINGLEUHR with SINGLEHR lines tables
            //lines_table[lines_nb].peak_pxl = read_line.peak_pxl + 33.0;
            lines_table[lines_nb].wavelength = read_line.wavelength;
            lines_table[lines_nb].wavelength_err = read_line.wavelength_err;
            lines_table[lines_nb].peak_flux = read_line.peak_flux;
            lines_table[lines_nb].peak_flux_err = read_line.peak_flux_err;
            lines_table[lines_nb].fwhm = read_line.fwhm;
            lines_table[lines_nb].fwhm_err = read_line.fwhm_err;
            lines_table[lines_nb].qc_flag = read_line.qc_flag;
            lines_table[lines_nb].dispersion = read_line.dispersion;
            lines_table[lines_nb].grouping = read_line.grouping;
            strcpy(lines_table[lines_nb].element_name, read_line.element_name);
            d_static[lines_nb] = read_d_static;
            d_static_err[lines_nb] = read_d_static_err;
            d_static_err_no_min[lines_nb] = read_d_static_err_no_min;
            /*
            printf("%d\t%lf\t%lf\t%lf\t%lf\t%d\n",
                   lines_table[lines_nb].order,
                   lines_table[lines_nb].peak_pxl,
                   lines_table[lines_nb].wavelength,
                   lines_table[lines_nb].wavelength_err,
                   lines_table[lines_nb].peak_flux,
                   lines_table[lines_nb].qc_flag);
            */
            lines_nb++;
        }
    }

    fclose(waveptr);
    
    *lines_nb_RE = lines_nb;
    
    return(cpl_error_get_code());
}




/*----------------------------------------------------------------------------*/
/**
 @brief Read WAVE_LINES_TABLE_CSV from CSV file
 @param         wave_line_table frame containing lines table in ASCII format
 @param[out]    lines_table     read lines table
 @param[out]    lines_nb_RE     numebr of read lines
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_read_REF_LINES_TABLE_CSV(cpl_frame *ref_line_table,
                                              espdr_line_param *lines_table,
                                              double *d_static,
                                              double *d_static_err,
                                              double *d_static_err_no_min,
                                              int *lines_nb_RE) {
    
    const char *filename = NULL;
    FILE *waveptr = NULL;
    char line[COMMENT_LENGTH];
    int lines_nb = 0;
    espdr_line_param read_line;
    
    espdr_line_param *tmp_lines_table = (espdr_line_param *)cpl_malloc(2000*sizeof(espdr_line_param));
    
    espdr_ensure(ref_line_table == NULL, CPL_ERROR_NULL_INPUT,
                 "Input REF LINES TABLE frame is NULL");
    espdr_ensure(lines_table == NULL, CPL_ERROR_NULL_INPUT,
                 "REF LINES TABLE structure is NULL");
    
    filename = cpl_frame_get_filename(ref_line_table);
    
    espdr_msg("Reading CSV REF lines from %s", filename);
    
    waveptr = fopen(filename, "r");
    espdr_ensure(waveptr == NULL, CPL_ERROR_FILE_IO,
                 "The file %s can't be read", filename);
    
    // Get rid of the header
    fgets(line, sizeof(line), waveptr);

    // Read the file content
    
    while (fscanf(waveptr,
                  "%d,%lf,%lf,%lf,%lf,%lf,%lf,%d,%lf,%lf,%lf,%d,%s",
                  &read_line.order,
                  &read_line.peak_pxl,
                  &read_line.peak_pxl_err,
                  &read_line.fwhm,
                  &read_line.fwhm_err,
                  &read_line.peak_flux,
                  &read_line.peak_flux_err,
                  &read_line.qc_flag,
                  &read_line.wavelength,
                  &read_line.wavelength_err,
                  &read_line.dispersion,
                  &read_line.grouping,
                  &read_line.element_name) != EOF) {


        //int lines_too_close = 0;
        //int lines_printed = 0;
        
    /*
    while (fscanf(waveptr,
                  "%lf,%lf,%d,%lf,%d,%s",
                  &read_line.wavelength,
                  &read_line.peak_pxl,
                  &read_line.order,
                  &read_line.peak_flux,
                  &read_line.qc_flag,
                  &read_line.element_name) != EOF) {
    */
        /*
        printf("%d\t%f\t%f\t%f\t%s\t%d\n",
               read_line.order,
               read_line.peak_pxl,
               read_line.wavelength,
               read_line.peak_flux,
               read_line.element_name,
               read_line.qc_flag);
        */
        
        if ((read_line.order == -1) && (read_line.qc_flag == 1)) {
            printf("%d\t%f\t%f\t%f\t%s\t%d\n",
                   read_line.order,
                   read_line.peak_pxl,
                   read_line.wavelength,
                   read_line.peak_flux,
                   read_line.element_name,
                   read_line.qc_flag);
        }
        //lines_printed++;
        
        
        //printf("%d\t%f\t%f\t%f\t%s\t%d\n",
        //       read_line.order,
        //       read_line.peak_pxl,
        //       read_line.wavelength,
        //       read_line.peak_flux,
        //       read_line.element_name,
        //       read_line.qc_flag);

        
        if ((read_line.qc_flag == 1) && (strcmp(read_line.element_name, "Ne") != 0)) {
            tmp_lines_table[lines_nb].order = read_line.order;
            tmp_lines_table[lines_nb].peak_pxl = read_line.peak_pxl;//- 63; HE_B //- 38; HE_A in La Silla
            // -35 for HE A -60-75 for HE B, additional 34 (between 27 & 41) pxl after May 2021: 69 for A & 94 for B
            tmp_lines_table[lines_nb].peak_pxl_err = read_line.peak_pxl_err;
            tmp_lines_table[lines_nb].wavelength = read_line.wavelength; //*10.0; - Nolan frames are in nm
            tmp_lines_table[lines_nb].wavelength_err = read_line.wavelength_err;
            tmp_lines_table[lines_nb].peak_flux = read_line.peak_flux;
            tmp_lines_table[lines_nb].peak_flux_err = read_line.peak_flux_err;
            tmp_lines_table[lines_nb].fwhm = read_line.fwhm;
            tmp_lines_table[lines_nb].fwhm_err = read_line.fwhm_err;
            tmp_lines_table[lines_nb].qc_flag = read_line.qc_flag;
            tmp_lines_table[lines_nb].dispersion = read_line.dispersion;
            tmp_lines_table[lines_nb].grouping = read_line.grouping;
            strcpy(tmp_lines_table[lines_nb].element_name, read_line.element_name);
            d_static[lines_nb] = 0.0;
            d_static_err[lines_nb] = 0.0;
            d_static_err_no_min[lines_nb] = 0.0;
            /*
            if (lines_nb > 0) {
                if (fabs(tmp_lines_table[lines_nb].peak_pxl - tmp_lines_table[lines_nb-1].peak_pxl) < 20.0) {
                    //printf("%d\t%f (%f)\t%f\t%f (%f)\t%s\t%d\n",
                    //       tmp_lines_table[lines_nb].order,
                    //       tmp_lines_table[lines_nb].peak_pxl,
                    //       tmp_lines_table[lines_nb-1].peak_pxl,
                    //       tmp_lines_table[lines_nb].wavelength,
                    //       tmp_lines_table[lines_nb].peak_flux,
                    //       tmp_lines_table[lines_nb-1].peak_flux,
                    //       tmp_lines_table[lines_nb].element_name,
                    //       tmp_lines_table[lines_nb].qc_flag);
                    espdr_msg_warning("The lines %f & %f in order %d are too close to each other",
                                      tmp_lines_table[lines_nb-1].peak_pxl, tmp_lines_table[lines_nb].peak_pxl,
                                      tmp_lines_table[lines_nb].order);
                    tmp_lines_table[lines_nb].qc_flag = 0;
                    tmp_lines_table[lines_nb-1].qc_flag = 0;
                    lines_too_close++;
                }
            }
            */
            //printf("%d\t%f\t%f\t%f\t%s\t%d\n",
            //       tmp_lines_table[lines_nb].order,
            //       tmp_lines_table[lines_nb].peak_pxl,
            //       tmp_lines_table[lines_nb].wavelength,
            //       tmp_lines_table[lines_nb].peak_flux,
            //       tmp_lines_table[lines_nb].element_name,
            //       tmp_lines_table[lines_nb].qc_flag);
            lines_nb++;
        }
    }

    int index = 0;
    for (int i = 0; i < lines_nb; i++) {
        if (tmp_lines_table[i].qc_flag == 1) {
            lines_table[index].order = tmp_lines_table[i].order;
            lines_table[index].peak_pxl = tmp_lines_table[i].peak_pxl;
            lines_table[index].peak_pxl_err = tmp_lines_table[i].peak_pxl_err;
            lines_table[index].wavelength = tmp_lines_table[i].wavelength;
            lines_table[index].wavelength_err = tmp_lines_table[i].wavelength_err;
            lines_table[index].peak_flux = tmp_lines_table[i].peak_flux;
            lines_table[index].peak_flux_err = tmp_lines_table[i].peak_flux_err;
            lines_table[index].fwhm = tmp_lines_table[i].fwhm;
            lines_table[index].fwhm_err = tmp_lines_table[i].fwhm_err;
            lines_table[index].qc_flag = tmp_lines_table[i].qc_flag;
            lines_table[index].dispersion = tmp_lines_table[i].dispersion;
            lines_table[index].grouping = tmp_lines_table[i].grouping;
            strcpy(lines_table[index].element_name, tmp_lines_table[i].element_name);
            index++;
        }
    }
    
    //espdr_msg("Lines table with good lines:");
    //for (int i = 0; i < index; i++) {
    //    printf("%d\t%f\t%f\t%f\t%s\t%d\n",
    //           lines_table[i].order,
    //           lines_table[i].peak_pxl,
    //           lines_table[i].wavelength,
    //           lines_table[i].peak_flux,
    //           lines_table[i].element_name,
    //           lines_table[i].qc_flag);
    //}
    //espdr_msg("Nb of rejected lines because too close: %d", lines_too_close*2);
    espdr_msg("Nb  of good lines: %d", index);
    
    fclose(waveptr);
    
    *lines_nb_RE = lines_nb;
    *lines_nb_RE = index;

    return(cpl_error_get_code());
}




/*----------------------------------------------------------------------------*/
/**
 @brief Read WAVE_LINES_TABLE from fits table
 @param         wave_line_table frame containing lines table in FITS format
 @param[out]    lines_table     read lines table
 @param[out]    lines_nb_RE     numebr of read lines
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_read_REF_LINES_TABLE(cpl_frame *ref_line_table,
                                          espdr_line_param *lines_table,
                                          double *d_static,
                                          double *d_static_err,
                                          double *d_static_err_no_min,
                                          int *lines_nb_RE) {
    
    const char *filename = NULL;
    int lines_nb = 0;
    int i;
    cpl_table *line_t = NULL;
    
    espdr_ensure(ref_line_table == NULL, CPL_ERROR_NULL_INPUT,
                 "Input LINE TABLE frame is NULL");
    espdr_ensure(lines_table == NULL, CPL_ERROR_NULL_INPUT,
                 "Lines table structure is NULL");
    
    filename = cpl_frame_get_filename(ref_line_table);
    espdr_msg("Reading FITS REF lines from %s", filename);
    
    //line_t = (cpl_table *)cpl_malloc(sizeof(cpl_table *));
    line_t = cpl_table_load(filename,1,0);
    
    int table_length = cpl_table_get_nrow(line_t);
    int *col_order = cpl_table_get_data_int(line_t,
                                            COL_NAME_ORDER);
    double *col_x0 = cpl_table_get_data_double(line_t,
                                               COL_NAME_PEAK_PXL);
    double *col_x0_err = cpl_table_get_data_double(line_t,
                                                   COL_NAME_PEAK_PXL_ERR);
    double *col_fwhm = cpl_table_get_data_double(line_t,
                                                 COL_NAME_FWHM);
    double *col_fwhm_err = cpl_table_get_data_double(line_t,
                                                     COL_NAME_FWHM_ERR);
    double *col_k = cpl_table_get_data_double(line_t,
                                              COL_NAME_PEAK_FLUX);
    double *col_k_err = cpl_table_get_data_double(line_t,
                                                  COL_NAME_PEAK_FLUX_ERR);
    double *col_ll = cpl_table_get_data_double(line_t,
                                               COL_NAME_WAVELENGTH);
    double *col_ll_err = cpl_table_get_data_double(line_t,
                                                   COL_NAME_WAVELENGTH_ERR);
    double *col_disp = cpl_table_get_data_double(line_t,
                                               COL_NAME_DISPERSION);
    int *col_grouping = cpl_table_get_data_int(line_t,
                                               COL_NAME_GROUPING);
    int *col_qc = cpl_table_get_data_int(line_t,
                                         COL_NAME_QC_FLAG);
    double *col_d = cpl_table_get_data_double(line_t, COL_NAME_D);
    double *col_d_err = cpl_table_get_data_double(line_t, COL_NAME_D_ERR);
    double *col_d_err_no_min = cpl_table_get_data_double(line_t, COL_NAME_D_ERR_NO_MIN);
    char **col_elem = cpl_table_get_data_string(line_t, COL_NAME_ELEMENT_NAME);
    
    for(i = 0; i < table_length; i++){
        if ((col_qc[i] == 1) &&
            ((strstr(col_elem[i], "TH") != NULL) ||
             (strstr(col_elem[i], "Th") != NULL) ||
             (strstr(col_elem[i], "NONE") != NULL) ||
             (strstr(col_elem[i], "UI") != NULL) ||
             (strstr(col_elem[i], "Ne") != NULL) ||
             (strstr(col_elem[i], "SP") != NULL) ||
             (strstr(col_elem[i], "?") != NULL) )) {
                lines_table[lines_nb].order = col_order[i];
                lines_table[lines_nb].peak_pxl = col_x0[i];
                lines_table[lines_nb].peak_pxl_err = col_x0_err[i];
                lines_table[lines_nb].fwhm = col_fwhm[i];
                lines_table[lines_nb].fwhm_err = col_fwhm_err[i];
                lines_table[lines_nb].peak_flux = col_k[i];
                lines_table[lines_nb].peak_flux_err = col_k_err[i];
                lines_table[lines_nb].wavelength = col_ll[i];
                lines_table[lines_nb].wavelength_err = col_ll_err[i];
                lines_table[lines_nb].dispersion = col_disp[i];
                lines_table[lines_nb].grouping = col_grouping[i];
                lines_table[lines_nb].qc_flag = col_qc[i];
                d_static[lines_nb] = col_d[i];
                d_static_err[lines_nb] = col_d_err[i];
                d_static_err_no_min[lines_nb] = col_d_err_no_min[i];
                strcpy(lines_table[lines_nb].element_name, col_elem[i]);
                if (lines_table[lines_nb].order == -1) {
                    printf("%d\t%lf\t%lf\t%lf\t%d\t%d\t%f\t%f\t%s\n",
                           lines_table[lines_nb].order,
                           lines_table[lines_nb].peak_pxl,
                           lines_table[lines_nb].wavelength,
                           lines_table[lines_nb].peak_flux,
                           lines_table[lines_nb].qc_flag,
                           lines_table[lines_nb].grouping,
                           d_static[lines_nb],
                           d_static_err[lines_nb],
                           lines_table[lines_nb].element_name);
                }
                lines_nb++;
            }
    }
    
    //espdr_msg("Read %d lines", lines_nb);
    
    cpl_table_delete(line_t);
    
    *lines_nb_RE = lines_nb;
    
    return(cpl_error_get_code());
}

/*----------------------------------------------------------------------------*/
/**
 @brief Read FP_LINES_TABLE from fits table
 @param         wave_line_table frame containing lines table in FITS format
 @param[out]    lines_table     read lines table
 @param[out]    lines_nb_RE     numebr of read lines
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_read_FP_LINES_TABLE(cpl_frame *wave_line_table,
                                           espdr_line_param *lines_table,
                                           int *lines_nb_RE) {
    
    const char *filename = NULL;
    int lines_nb = 0;
    int i;
    cpl_table *line_t = NULL;
    
    espdr_ensure(wave_line_table == NULL, CPL_ERROR_NULL_INPUT,
                 "Input LINE TABLE frame is NULL");
    espdr_ensure(lines_table == NULL, CPL_ERROR_NULL_INPUT,
                 "Lines table structure is NULL");
    
    filename = cpl_frame_get_filename(wave_line_table);
    
    espdr_msg("Reading FP lines from %s", filename);
    
    line_t = cpl_table_load(filename,1,0);
    
    int table_length = cpl_table_get_nrow(line_t);
    
    espdr_msg("Got %d rows from FITS table", table_length);
    
    int *col_order = cpl_table_get_data_int(line_t,
                                            COL_NAME_ORDER);
    double *col_x0 = cpl_table_get_data_double(line_t,
                                               COL_NAME_PEAK_PXL);
    double *col_sig_x0 = cpl_table_get_data_double(line_t,
                                                   COL_NAME_PEAK_PXL_ERR);
    double *col_fwhm = cpl_table_get_data_double(line_t,
                                                 COL_NAME_FWHM);
    double *col_sig_fwhm = cpl_table_get_data_double(line_t,
                                                     COL_NAME_FWHM_ERR);
    double *col_k = cpl_table_get_data_double(line_t,
                                              COL_NAME_PEAK_FLUX);
    double *col_sig_k = cpl_table_get_data_double(line_t,
                                                  COL_NAME_PEAK_FLUX_ERR);
    int *col_qc = cpl_table_get_data_int(line_t,
                                         COL_NAME_QC_FLAG);
    double *col_ll = cpl_table_get_data_double(line_t,
                                               COL_NAME_WAVELENGTH);
    double *col_ll_err = cpl_table_get_data_double(line_t,
                                                   COL_NAME_WAVELENGTH_ERR);
    double *col_dlldx = cpl_table_get_data_double(line_t,
                                                  COL_NAME_DISPERSION);
    int *col_grouping = cpl_table_get_data_int(line_t,
                                               COL_NAME_GROUPING);
    char **col_elem = cpl_table_get_data_string(line_t,
                                                COL_NAME_ELEMENT_NAME);
    
    for(i = 0; i < table_length; i++){
        lines_table[i].order = col_order[i];
        lines_table[i].peak_pxl = col_x0[i] ;
        //lines_table[i].peak_pxl = col_x0[i] + 6;
        lines_table[i].peak_pxl_err = col_sig_x0[i];
        lines_table[i].fwhm = col_fwhm[i];
        lines_table[i].fwhm_err = col_sig_fwhm[i];
        lines_table[i].peak_flux = col_k[i];
        lines_table[i].peak_flux_err = col_sig_k[i];
        lines_table[i].qc_flag = col_qc[i];
        lines_table[i].wavelength = col_ll[i];
        lines_table[i].wavelength = col_ll_err[i];
        lines_table[i].dispersion = col_dlldx[i];
        lines_table[i].grouping = col_grouping[i];
        strcpy(lines_table[i].element_name, col_elem[i]);
        // N is used to mark reference FP peaks
        lines_table[i].N = 0;
        if (lines_table[i].order == -1) {
            printf("%d\t%d\t%.7lf\t%lf\t%lf\t%lf\t%.7lf\t%lf\t%d\t%.8f\t%.8lf\t%lf\t%d\t%s\n",
                   i,
                   lines_table[i].order,
                   lines_table[i].peak_pxl,
                   lines_table[i].peak_pxl_err,
                   lines_table[i].fwhm,
                   lines_table[i].fwhm_err,
                   lines_table[i].peak_flux,
                   lines_table[i].peak_flux_err,
                   lines_table[i].qc_flag,
                   lines_table[i].wavelength,
                   lines_table[i].wavelength_err,
                   lines_table[i].dispersion,
                   lines_table[i].grouping,
                   lines_table[i].element_name);
        }
        lines_nb++;
    }
    
    cpl_table_delete(line_t);
    
    *lines_nb_RE = lines_nb;
    
    return(cpl_error_get_code());
}




/*----------------------------------------------------------------------------*/
/**
 @brief QC on THAR lines
 @param         lines_table             detected lines peaks
 @param         lines_nb                number of lines in the table
 @param         inst_config             instrument config
 @param[out]    lines_nb_per_order_RE   number of valid lines per order
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_THAR_lines_qc(espdr_line_param *lines_table,
                                   int lines_nb,
                                   espdr_inst_config *inst_config,
                                   int *lines_nb_per_order_RE) {
    
    espdr_ensure(lines_table == NULL, CPL_ERROR_NULL_INPUT,
                 "The input table is NULL");
    int i, j, k, group, group_QC;
    int test_order = -1;
    
    
    for (i = 0; i < lines_nb; i++) {
        
        if (lines_table[i].order == test_order) {
            espdr_msg("order %d, pxl %lf, ll: %lf, before check qc = %d",
                      lines_table[i].order, lines_table[i].peak_pxl,
                      lines_table[i].wavelength, lines_table[i].qc_flag);
        }
        
        // Check if the fit produced any nans or infs
        if (isnan(lines_table[i].peak_pxl) || isinf(lines_table[i].peak_pxl) ||
            isnan(lines_table[i].peak_pxl_err) || isinf(lines_table[i].peak_pxl_err) ||
            isnan(lines_table[i].fwhm) || isinf(lines_table[i].fwhm) ||
            isnan(lines_table[i].fwhm_err) || isinf(lines_table[i].fwhm_err) ||
            isnan(lines_table[i].peak_flux) || isinf(lines_table[i].peak_flux) ||
            isnan(lines_table[i].peak_flux_err) || isinf(lines_table[i].peak_flux_err)) {
            if (lines_table[i].order == test_order) {
                espdr_msg("order %d, pxl %lf, ll: %lf, nans or infs",
                          lines_table[i].order, lines_table[i].peak_pxl,
                          lines_table[i].wavelength);
            }
            lines_table[i].qc_flag = 0;
            lines_table[i].fwhm = -1.0;
            lines_table[i].fwhm_err = -1.0;
            lines_table[i].peak_flux = -1.0;
            lines_table[i].peak_flux_err = -1.0;
            lines_table[i].dispersion = -1.0;
        }
        
        //QC_flag_orig = lines_table[i].qc_flag;
        if (lines_table[i].peak_pxl_err > inst_config->line_max_sig_x0) {
            //if (lines_table[i].peak_pxl_err > 5.0) {
            if (lines_table[i].order == test_order) {
                espdr_msg("order %d, pxl %lf, ll: %lf, peak pxl error too big (%lf)",
                          lines_table[i].order, lines_table[i].peak_pxl,
                          lines_table[i].wavelength, lines_table[i].peak_pxl_err);
            }
            lines_table[i].qc_flag = 0;
            lines_table[i].fwhm = -1.0;
            lines_table[i].fwhm_err = -1.0;
            lines_table[i].peak_flux = -1.0;
            lines_table[i].peak_flux_err = -1.0;
            lines_table[i].dispersion = -1.0;
        }
        if (lines_table[i].fwhm > inst_config->line_max_fwhm) {
            //if ((lines_table[i].fwhm > 8) || (lines_table[i].fwhm < 3)) {
            //if (lines_table[i].fwhm > 5.0) {
            if (lines_table[i].order == test_order) {
                espdr_msg("order %d, pxl %lf, ll: %lf, fwhm too big (%lf)",
                          lines_table[i].order, lines_table[i].peak_pxl,
                          lines_table[i].wavelength, lines_table[i].fwhm);
            }
            lines_table[i].qc_flag = 0;
            lines_table[i].fwhm = -1.0;
            lines_table[i].fwhm_err = -1.0;
            lines_table[i].peak_flux = -1.0;
            lines_table[i].peak_flux_err = -1.0;
            lines_table[i].dispersion = -1.0;
        }
        if (lines_table[i].fwhm <= 0.0) {
            if (lines_table[i].order == test_order) {
                espdr_msg("order %d, pxl %lf, ll: %lf, fwhm too big (%lf)",
                          lines_table[i].order, lines_table[i].peak_pxl,
                          lines_table[i].wavelength, lines_table[i].fwhm);
            }
            lines_table[i].qc_flag = 0;
            lines_table[i].fwhm = -1.0;
            lines_table[i].fwhm_err = -1.0;
            lines_table[i].peak_flux = -1.0;
            lines_table[i].peak_flux_err = -1.0;
            lines_table[i].dispersion = -1.0;
        }
        if (lines_table[i].peak_flux < inst_config->line_min_flux_el) {
            if (lines_table[i].order == test_order) {
                espdr_msg("order %d, pxl %lf, ll: %lf, flux too small (%lf)",
                          lines_table[i].order, lines_table[i].peak_pxl,
                          lines_table[i].wavelength, lines_table[i].peak_flux);
            }
            lines_table[i].qc_flag = 0;
            lines_table[i].fwhm = -1.0;
            lines_table[i].fwhm_err = -1.0;
            lines_table[i].peak_flux = -1.0;
            lines_table[i].peak_flux_err = -1.0;
            lines_table[i].dispersion = -1.0;
        }
        // The following test is added,
        // because the fit to compute the resolution was failing
        if (lines_table[i].fwhm_err == 0.0) {
            if (lines_table[i].order == test_order) {
                espdr_msg("order %d, pxl %lf, ll: %lf, fwhm error == 0.0",
                          lines_table[i].order, lines_table[i].peak_pxl,
                          lines_table[i].wavelength);
            }
            lines_table[i].qc_flag = 0;
            lines_table[i].fwhm = -1.0;
            lines_table[i].fwhm_err = -1.0;
            lines_table[i].peak_flux = -1.0;
            lines_table[i].peak_flux_err = -1.0;
            lines_table[i].dispersion = -1.0;
        }
    }
    
    /* Take off the whole group which has the line with qc_flag == 0 */
    i = 0;
    while (i < lines_nb) {
        j = i;
        group = lines_table[j].grouping;
        group_QC = 1;
        while ((j < lines_nb) && (lines_table[j].grouping == group)) {
            if (lines_table[j].qc_flag == 0) {
                group_QC = 0;
                if (lines_table[i].order == test_order) {
                    espdr_msg("order %d, pxl %lf, ll: %lf, one of the group qc = 0",
                              lines_table[i].order, lines_table[i].peak_pxl,
                              lines_table[i].wavelength);
                }
            }
            j++;
        }
        if (group_QC == 0) {
            for (k = i; k < j; k++) {
                if (lines_table[i].order == test_order) {
                    espdr_msg("order %d, pxl %lf, ll: %lf, group = 0",
                              lines_table[i].order, lines_table[i].peak_pxl,
                              lines_table[i].wavelength);
                }
                lines_table[k].qc_flag = 0;
                lines_table[i].fwhm = -1.0;
                lines_table[i].fwhm_err = -1.0;
                lines_table[i].peak_flux = -1.0;
                lines_table[i].peak_flux_err = -1.0;
                lines_table[i].dispersion = -1.0;
                lines_nb_per_order_RE[lines_table[k].order-1]--;
            }
        }
        i = j;
    }
    
    for (i = 0; i < lines_nb; i++) {
        // Lines was not fitted because of close to the border
        // or too small flux difference between max_flux and min_flux
        // or saturated
        // But the rest of the group is OK
        if (lines_table[i].qc_flag == 2) {
            if (lines_table[i].order == test_order) {
                espdr_msg("order %d, pxl %lf, ll: %lf, qc = 2",
                          lines_table[i].order, lines_table[i].peak_pxl,
                          lines_table[i].wavelength);
            }
            lines_table[i].qc_flag = 0;
            lines_table[i].fwhm = -1.0;
            lines_table[i].fwhm_err = -1.0;
            lines_table[i].peak_flux = -1.0;
            lines_table[i].peak_flux_err = -1.0;
            lines_table[i].dispersion = -1.0;
        }
    }
    
    return(cpl_error_get_code());
}



/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param[out]
 @param[out]
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_find_first_FP_ll(espdr_line_param *REF_lines_table,
                                      int REF_lines_nb,
                                      double fp_d_angstrom,
                                      espdr_line_param *FP_lines_table,
                                      int FP_lines_nb) {
    
    espdr_ensure(REF_lines_table == NULL, CPL_ERROR_NULL_INPUT,
                 "REF lines input table is NULL");
    espdr_ensure(FP_lines_table == NULL, CPL_ERROR_NULL_INPUT,
                 "FP lines input table is NULL");
    
    int i, j;
    int ref_order;
    double ref_pxl, fp_pxl, fp_pxl_prev;
    double ref_lambda;
    
    for (i = 0; i < REF_lines_nb; i++) {
        
        if (REF_lines_table[i].qc_flag == 1) {
            ref_order = REF_lines_table[i].order;
            ref_pxl = REF_lines_table[i].peak_pxl;
            ref_lambda = REF_lines_table[i].wavelength;
            
            j = 0;
            // Finding the right order
            while ((j < FP_lines_nb) && (FP_lines_table[j].order < ref_order)) {
                j++;
            }
            // Finding the right pxl
            while ((j < FP_lines_nb) && (FP_lines_table[j].order == ref_order) &&
                   (FP_lines_table[j].peak_pxl < ref_pxl)) {
                j++;
            }
            
            if ((j == 0) || (FP_lines_table[j-1].order != ref_order)) {
                espdr_msg_warning("No FP line before the ThAr line (order %d, pxl: %.2lf)",
                                  ref_order, ref_pxl);
            } else {
                if ((j >= FP_lines_nb) || (FP_lines_table[j].order != ref_order)) {
                    espdr_msg_warning("No FP line after the ThAr line (order %d, pxl: %.2lf)",
                                      ref_order, ref_pxl);
                } else {
                    if ((FP_lines_table[j].qc_flag == 0) || (FP_lines_table[j-1].qc_flag == 0)) {
                        espdr_msg_warning("The FP peak found in order %d has qc flag 0",
                                          ref_order);
                    } else {
                        fp_pxl = FP_lines_table[j].peak_pxl;
                        fp_pxl_prev = FP_lines_table[j-1].peak_pxl;
                        FP_lines_table[j].wavelength = ref_lambda +
                            (ref_lambda*ref_lambda)/(2.0 * fp_d_angstrom) *
                                ((fp_pxl - ref_pxl) / (fp_pxl - fp_pxl_prev));
                        FP_lines_table[j].N = 1;
                        if (ref_order == -1 || ref_order == -1) {
                        espdr_msg("FP lines found (order %d): ref: %lf, ll = %f,  FP-1: %lf, FP+1: %lf",
                                  ref_order,
                                  REF_lines_table[i].peak_pxl,
                                  REF_lines_table[i].wavelength,
                                  FP_lines_table[j-1].peak_pxl,
                                  FP_lines_table[j].peak_pxl);
                        }
                    }
                }
            }
        }
    }
    
    
    return(cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param[out]
 @param[out]
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_get_all_FP_ll_per_order(espdr_line_param *FP_lines_table,
                                             int FP_lines_nb,
                                             int CCD_middle,
                                             double fp_d_angstrom) {
    
    int i, j, k, order_first, order_last, order;
    int FP_ref_index, FP_ref_found;
    double FP_ref_ll;
    double FP_ref_x0;
    int FP_ref_middle_distance;
    int peak_nb;
    
    i = 0;
    order_first = 0;
    order_last = 0;
    while (i < FP_lines_nb) {
        
        // For each order find the reference the closest to the CCD middle
        j = i;
        order = FP_lines_table[j].order;
        order_first = j;
        FP_ref_found = 0;
        FP_ref_index = 0;
        FP_ref_middle_distance = CCD_middle * 2;
        while ((j < FP_lines_nb) && (FP_lines_table[j].order == order)) {
            if ((FP_lines_table[j].wavelength != 0) && (FP_lines_table[j].qc_flag == 1) &&
                (abs(CCD_middle - (int)FP_lines_table[j].peak_pxl) < FP_ref_middle_distance)) {
                if (order == -1) {
                    espdr_msg("order %d, x0: %lf, dist from CCD middle: %d, line qc: %d",
                              order, FP_lines_table[j].peak_pxl,
                              abs(CCD_middle - (int)FP_lines_table[j].peak_pxl),
                              FP_lines_table[j].qc_flag);
                }
                FP_ref_index = j;
                FP_ref_found = 1;
                FP_lines_table[j].grouping = 0;
                FP_ref_ll = FP_lines_table[j].wavelength;
                FP_ref_x0 = FP_lines_table[j].peak_pxl;
                FP_ref_middle_distance = abs(CCD_middle - (int)FP_lines_table[j].peak_pxl);
            }
            j++;
        }
        order_last = j;
        
        //espdr_msg("order: %d, ref_index: %d, order start: %d order end: %d, ll_ref: %.10lf, pxl_ref: %lf",
        //          order, FP_ref_index, order_first, order_last, FP_ref_ll, FP_ref_x0);
        if (FP_ref_found == 1) {
            peak_nb = 1;
            for (k = FP_ref_index - 1; k >= order_first; k--) {
                FP_lines_table[k].grouping = peak_nb;
                FP_lines_table[k].wavelength = 1 / (1/FP_ref_ll + (double)(peak_nb)/2.0/fp_d_angstrom);
                if (order == -1 || order == -1) {
                    espdr_msg("line %d (x0 = %lf) nb = %d, ll = %lf",
                              k, FP_lines_table[k].peak_pxl, peak_nb,
                              FP_lines_table[k].wavelength);
                }
                peak_nb++;
            }
            peak_nb = -1;
            for (k = FP_ref_index + 1; k < order_last; k++) {
                FP_lines_table[k].grouping = peak_nb;
                FP_lines_table[k].wavelength = 1 / (1/FP_ref_ll + (double)(peak_nb)/2.0/fp_d_angstrom);
                if (order == -1 || order == -1) {
                    espdr_msg("line %d (x0 = %lf) nb = %d, ll = %lf",
                              k, FP_lines_table[k].peak_pxl, peak_nb,
                              FP_lines_table[k].wavelength);
                }
                peak_nb--;
            }
        } else {
            for (k = order_first; k < order_last; k++) {
                FP_lines_table[k].grouping = -1;
            }
            espdr_msg_warning("No FP line identified for order %d, exiting", order);
            return (CPL_ERROR_INCOMPATIBLE_INPUT);
        }
        
        i = j;
    }
    
    return(cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param[out]
 @param[out]
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_get_all_FP_ll_whole_image(espdr_line_param *FP_lines_table,
                                               int FP_lines_nb,
                                               espdr_inst_config *inst_config,
                                               int orders_nb) {
    
    int i, j, k, l, last_order;
    int FP_ref_id, mode_nr_int, current_order;
    double mode_nr;
    int mode_ref_nr, mode_ref_id, same_peak_id;
    double mode_ref_ll, ll_diff;
    double start_ll[orders_nb];
    double end_ll[orders_nb];
    int mode_start_id[orders_nb];
    int mode_end_id[orders_nb];
    int reference_peak[orders_nb];
    int index_start, index_end;
    
    start_ll[0] = FP_lines_table[0].wavelength;
    mode_start_id[0] = 0;
    index_start = 1;
    index_end = 0;
    for (i = 1; i < FP_lines_nb-1; i++) {
        if (FP_lines_table[i].order != FP_lines_table[i-1].order) {
            end_ll[index_end] = FP_lines_table[i-1].wavelength;
            mode_end_id[index_end] = i-1;
            index_end++;
            start_ll[index_start] = FP_lines_table[i].wavelength;
            mode_start_id[index_end] = i;
            index_start++;
        }
        if (FP_lines_table[i].grouping == 0) {
            reference_peak[FP_lines_table[i].order-1] = i;
        }
    }
    end_ll[index_end] = FP_lines_table[FP_lines_nb-1].wavelength;
    mode_end_id[index_end] = FP_lines_nb-1;
    
    //for (i = 0; i < orders_nb; i++) {
    //    espdr_msg("order %d: %.10lf - %.10lf", i+1, start_ll[i], end_ll[i]);
    //}
    
    for (i = 0; i < inst_config->slices_nb_per_phys_order; i++) {
        last_order = orders_nb - inst_config->slices_nb_per_phys_order + i + 1;
        //espdr_msg("last_order = %d", last_order);
        j = 0;
        while (j < FP_lines_nb) {
            if ((FP_lines_table[j].order == last_order) && (FP_lines_table[j].qc_flag == 1)) {
                //espdr_msg("Found line in last order (%d)", last_order);
                FP_ref_id = j;
                j = FP_lines_nb;
            }
            j++;
        }
        
        mode_nr = 2.0 * inst_config->fp_d_angstrom / FP_lines_table[FP_ref_id].wavelength;

        //espdr_msg("Reference line for slice %d found in order %d pxl %lf has mode nr: %lf",
        //          i, FP_lines_table[FP_ref_id].order, FP_lines_table[FP_ref_id].peak_pxl,
        //          mode_nr);
        
        
        current_order = last_order;
        mode_ref_nr = (int)(round(mode_nr));
        
        if (fabs(mode_nr - (double)(mode_ref_nr)) > 0.4) {
            espdr_msg_warning(ANSI_COLOR_RED
                              "The mode computed with D0 = %f is far from integer value: %f (order %d)"
                              ANSI_COLOR_RESET,
                              inst_config->fp_d_angstrom, mode_nr, current_order);
        }
        for (k = 0; k < (orders_nb/inst_config->slices_nb_per_phys_order); k++) {
            
            j = FP_ref_id;
            mode_nr_int = mode_ref_nr;
            while ((j < FP_lines_nb) && (FP_lines_table[j].order == current_order)) {
                FP_lines_table[j].grouping = mode_nr_int;
                //if (FP_lines_table[j].qc_flag == 1) {
                    // Taking the last valid FP peak
                    //mode_ref_nr = mode_nr_int;
                    //mode_ref_ll = FP_lines_table[j].wavelength;
                    //mode_ref_id = j;
                //}
                j++;
                mode_nr_int--;
            }
            
            j = FP_ref_id;
            mode_nr_int = mode_ref_nr;
            while ((j >= 0) && (FP_lines_table[j].order == current_order)) {
                FP_lines_table[j].grouping = mode_nr_int;
                j--;
                mode_nr_int++;
            }
            
#if 0
            // Taking the first valid FP peak - works for almost all orders
            mode_ref_nr = mode_nr_int-1;
            mode_ref_ll = FP_lines_table[j+1].wavelength;
            mode_ref_id = j+1;
#endif
            
            if (current_order == -1) {
                espdr_msg("LL limits: for order %d: %f - %f",
                          current_order-inst_config->slices_nb_per_phys_order,
                          start_ll[current_order-1-inst_config->slices_nb_per_phys_order],
                          end_ll[current_order-1-inst_config->slices_nb_per_phys_order]);
            }
            
            int reference_found = 0;
            if (current_order <= inst_config->slices_nb_per_phys_order) {
                reference_found = 1;
            }
            for (l = FP_lines_nb-1; l >= 0; l--) {
                // current_order can't be < 3
                
                if ((current_order > inst_config->slices_nb_per_phys_order) &&
                    (FP_lines_table[l].order == current_order) &&
                    (FP_lines_table[l].qc_flag == 1) &&
                    (FP_lines_table[l].wavelength > start_ll[current_order-1-inst_config->slices_nb_per_phys_order]) &&
                    (FP_lines_table[l].wavelength < end_ll[current_order-1-inst_config->slices_nb_per_phys_order])) {
                    mode_ref_nr = FP_lines_table[l].grouping;
                    mode_ref_ll = FP_lines_table[l].wavelength;
                    mode_ref_id = l;
                    reference_found = 1;
                    if (current_order == -2) {
                        espdr_msg("Ref OK ---> order: %d, pxl: %f, ll: %f",
                                  current_order, FP_lines_table[mode_ref_id].peak_pxl,
                                  FP_lines_table[mode_ref_id].wavelength);
                    }
                }
            }
            
            //espdr_msg("Order: %d reference found: %d",
            //          current_order, reference_found);
            
            if (reference_found) {
                //espdr_msg("Ref order: %d, pxl: %f, ll: %f",
                //          current_order,
                //          FP_lines_table[mode_ref_id].peak_pxl,
                //          FP_lines_table[mode_ref_id].wavelength);
                
                // Linking two orders
                
                current_order = current_order - inst_config->slices_nb_per_phys_order;
                if (current_order == -2) {
                    //espdr_msg("current order: %d, order[j]: %d, pxl[j]: %lf, ll[j]: %.10lf, ll[j++]: %.10lf",
                    //          current_order, FP_lines_table[j].order, FP_lines_table[j].peak_pxl,
                    //          FP_lines_table[j].wavelength, FP_lines_table[j+1].wavelength);
                    //espdr_msg("j = %d--> order[j]: %d %lf %.10lf, order[j+1]: %d, %lf, %.10lf",
                    //          j, FP_lines_table[j].order, FP_lines_table[j].peak_pxl,
                    //          FP_lines_table[j].wavelength, FP_lines_table[j+1].order,
                    //          FP_lines_table[j+1].peak_pxl, FP_lines_table[j+1].wavelength);
                    espdr_msg("last order: %d, current order: %d, current order ll: %.10lf, mode_nr: %d",
                              last_order, current_order, mode_ref_ll, mode_ref_nr);
                    espdr_msg("Ref order: %d, pxl: %f, ll: %f",
                              current_order+inst_config->slices_nb_per_phys_order,
                              FP_lines_table[mode_ref_id].peak_pxl,
                              FP_lines_table[mode_ref_id].wavelength);
                }
                
                if (current_order > 0) {
                    while ((j >= 0) && (FP_lines_table[j].order > current_order)) {
                        j--;
                    }
                    ll_diff = 100.0;
                    while ((j >= 0) && (FP_lines_table[j].order == current_order)) {
                        if (fabs(FP_lines_table[j].wavelength - mode_ref_ll) < ll_diff) {
                            ll_diff = fabs(FP_lines_table[j].wavelength - mode_ref_ll);
                            same_peak_id = j;
                        }
                        j--;
                    }
                    
                    if (current_order == -2) {
                        espdr_msg("Peak found: order: %d, pxl: %lf, ll: %.10lf, ll_diff: %f",
                                  FP_lines_table[same_peak_id].order,
                                  FP_lines_table[same_peak_id].peak_pxl,
                                  FP_lines_table[same_peak_id].wavelength,
                                  ll_diff);
                        
                        espdr_msg("Matching order %d ll: %.10lf, difference: %.10lf",
                                  current_order, FP_lines_table[same_peak_id].wavelength, ll_diff);
                    }
                    
                    if (ll_diff > 0.1 *
                        FP_lines_table[same_peak_id].wavelength *
                        FP_lines_table[same_peak_id].wavelength /
                            (2.0 * inst_config->fp_d_angstrom)) {
                        espdr_msg_warning("Peak in order %d at pxl %lf doesn't match well with peak at pxl %lf in order %d; ll diff: %.10lf",
                                          current_order+inst_config->slices_nb_per_phys_order,
                                          FP_lines_table[mode_ref_id].peak_pxl,
                                          FP_lines_table[same_peak_id].peak_pxl,
                                          current_order, ll_diff);
                    }
                    
                    
                    FP_lines_table[same_peak_id].grouping = mode_ref_nr;
                    FP_ref_id = same_peak_id;
                }
            } else {
                
                if (inst_config->wave_orders_analytical_link_sw == 1) {
                    espdr_msg_warning("Linking two orders: %d and %d analytically, wavelengths are not overlapping",
                                      current_order-inst_config->slices_nb_per_phys_order,
                                      current_order);
                    // The peak with the most accurate ll is the one with grouping == 0
                    int curr_peak_id = reference_peak[current_order-1];
                    double curr_peak_mode = FP_lines_table[curr_peak_id].grouping;
                    double curr_peak_lambda = FP_lines_table[curr_peak_id].wavelength;
                    int next_peak_id = reference_peak[current_order-inst_config->slices_nb_per_phys_order-1];
                    double next_peak_lambda = FP_lines_table[next_peak_id].wavelength;
                    
                    double delta_mode = curr_peak_mode * (curr_peak_lambda/next_peak_lambda - 1.0);
                    
                    //espdr_msg("curr id: %d, curr mode: %f, curr ll: %f, next id: %d, next ll: %f, delta mode: %f",
                    //          curr_peak_id, curr_peak_mode, curr_peak_lambda,
                    //          next_peak_id, next_peak_lambda, delta_mode);
                    
                    if ((delta_mode - round(delta_mode)) > 0.1) {
                        espdr_msg_warning("The analytical linking is not precise enough: %f, should be less then 0.1",
                                          delta_mode - round(delta_mode));
                    }
                    
                    FP_lines_table[next_peak_id].grouping = curr_peak_mode + (int)(round(delta_mode));
                    mode_ref_nr = FP_lines_table[next_peak_id].grouping;
                    FP_ref_id = next_peak_id;
                    current_order = current_order - inst_config->slices_nb_per_phys_order;                    
                } else {
                    espdr_msg("Can't link two orders: %d and %d, wavelengths are not overlapping",
                              current_order-inst_config->slices_nb_per_phys_order,
                              current_order);
                    return (CPL_ERROR_INCOMPATIBLE_INPUT);
                }
            }
        }
    }
    
    /*
    espdr_msg("Mode number check");
    for (j = 0; j < FP_lines_nb; j++) {
        if (FP_lines_table[j].qc_flag == 1) {
            mode_nr = 2.0 * FP_D / FP_lines_table[j].wavelength;
            espdr_msg("Order %d, x0: %lf, ll: %.10lf, mode_nr: %lf",
                      FP_lines_table[j].order, FP_lines_table[j].peak_pxl,
                      FP_lines_table[j].wavelength, mode_nr);
        }
    }
    */
    
    /*
    // Mode calculation from ll
    for (int order = orders_nb-1; order >= 0; order--) {
        double ll_start = start_ll[order];
        double ll_end = end_ll[order];
        double ll_diff = ll_end - ll_start;
        int mode_start = FP_lines_table[mode_start_id[order]].grouping;
        int mode_end = FP_lines_table[mode_end_id[order]].grouping;
        int mode_diff_int = mode_end - mode_start;
        double mode_diff_ll = - 2.0 * FP_D * ll_diff / (ll_start * ll_end);
        espdr_msg("order %d: ll_start = %f\tll_end = %f\tll_diff = %f\tmode_start[%d] = %d\tmode_end[%d] = %d\tmode_diff = %d\tmode_diff_ll = %.2f",
                  order+1, ll_start, ll_end, ll_diff,
                  mode_start_id[order], mode_start,
                  mode_end_id[order], mode_end,
                  mode_diff_int, mode_diff_ll);
    }
    */
    
    return(cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief Compute d, d_err and d_err_no_min
 @param
 @param[out]
 @param[out]
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code espdr_compute_d(espdr_line_param *THAR_lines_table,
                                      int index_THAR,
                                      espdr_line_param *FP_lines_table,
                                      int FP_lines_nb,
                                      int index_FP,
                                      espdr_inst_config *inst_config,
                                      double *d_computed,
                                      double *d_err,
                                      double *d_err_no_min) {

    int i;
    double N_FP_peaks_taken = 0.0;
    double m_FP[N_FP_PEAKS];
    double x_FP[N_FP_PEAKS];
    int flag_FP[N_FP_PEAKS];
    int index_peaks = 0;
    int thar_order = THAR_lines_table[index_THAR].order;
    double x_thar = THAR_lines_table[index_THAR].peak_pxl;
    double disp = 0.0, sum_for_d = 0.0;
    double sum_x_FP = 0.0, sum_x_FP_2 = 0.0, sum_x_FP_over_m_FP = 0.0, sum_1_over_m_FP = 0.0;
    double th_lambda_err = THAR_lines_table[index_THAR].wavelength_err / THAR_lines_table[index_THAR].wavelength;
    double x_m_err = FP_lines_table[index_FP].peak_pxl_err;
    double th_x_err = THAR_lines_table[index_THAR].peak_pxl_err;
    double m = (double)(FP_lines_table[index_FP].grouping);
    double min_x0_err = inst_config->line_min_sig_x0;

    for (i = 0; i < N_FP_PEAKS; i++) {
        x_FP[i] = 0.0;
        m_FP[i] = 0.0;
        flag_FP[i] = 0;
    }
    index_peaks = 0;
    N_FP_peaks_taken = 0;
    for (i = index_FP-3; i < index_FP+3; i++) {
        if ((i >= 0) && (i < FP_lines_nb)) {
            if ((FP_lines_table[i].order == thar_order) && (FP_lines_table[i].qc_flag == 1)) {
                x_FP[index_peaks] = FP_lines_table[i].peak_pxl;
                m_FP[index_peaks] = (double)(FP_lines_table[i].grouping);
                flag_FP[index_peaks] = 1;
                index_peaks++;
                N_FP_peaks_taken = N_FP_peaks_taken + 1.0;;
            }
        }
    }
    
    for (i = 0; i < N_FP_PEAKS; i++) {
        if (flag_FP[i] == 1) {
            sum_x_FP += x_FP[i];
            sum_x_FP_2 += x_FP[i] * x_FP[i];
            sum_1_over_m_FP += 1.0 / m_FP[i];
            sum_x_FP_over_m_FP += x_FP[i] / m_FP[i];
        }
    }
    
    disp = (sum_x_FP_over_m_FP - (sum_x_FP * sum_1_over_m_FP / N_FP_peaks_taken)) /
                    (sum_x_FP_2 - sum_x_FP*sum_x_FP / N_FP_peaks_taken);
    sum_for_d = 0.0;
    for (i = 0; i < N_FP_PEAKS; i++) {
        if (flag_FP[i] == 1) {
            //espdr_msg("Th line %d, peak: %d: m_i: %f, x_i: %f, x_thar: %f, x_i - x_thar: %f, disp: %f, m_i*disp*x_diff: %f",
            //          index_THAR, i, m_FP[i], x_FP[i], x_thar, x_FP[i] - x_thar, disp, m_FP[i] * disp * (x_FP[i] - x_thar));
            sum_for_d += m_FP[i] / (1.0 - m_FP[i] * disp * (x_FP[i] - x_thar));
        }
    }
    
    *d_computed = THAR_lines_table[index_THAR].wavelength / 2.0 / N_FP_peaks_taken * sum_for_d;

    //espdr_msg("Th line %d: N: %f, ll/2/N: %f, sum_for_d: %f, d_computed: %f",
    //          index_THAR, N_FP_peaks_taken,
    //          THAR_lines_table[index_THAR].wavelength / 2.0 / N_FP_peaks_taken,
    //          sum_for_d, *d_computed);
    
    *d_err = inst_config->fp_d_angstrom / sqrt(N_FP_peaks_taken) *
                sqrt(th_lambda_err*th_lambda_err +
                     m*m * disp*disp * (x_m_err*x_m_err +
                                        2*min_x0_err*min_x0_err +
                                        th_x_err*th_x_err));
    
    *d_err_no_min = inst_config->fp_d_angstrom / sqrt(N_FP_peaks_taken) *
                    sqrt(th_lambda_err*th_lambda_err +
                         m*m * disp*disp * (x_m_err*x_m_err +
                                            th_x_err*th_x_err));
    
    //espdr_msg("Th line %d: D/sqrt(N): %f, th_ll_err^2: %.16f, m*m: %d, disp^2: %.16f, x_m_err^2: %f, th_x_err^2: %f, 2*min_x_err^2: %f, sqrt(2nd part): %f ---> err: %f",
    //          index_THAR, inst_config->fp_d_angstrom / sqrt((double)(N_FP_peaks_taken)),
    //          th_lambda_err*th_lambda_err, m*m, disp*disp,
    //          x_m_err*x_m_err, th_x_err*th_x_err, 2*min_x0_err*min_x0_err,
    //          sqrt(th_lambda_err*th_lambda_err + (double)(m)*(double)(m) * disp*disp * (x_m_err*x_m_err + 2*min_x0_err*min_x0_err + th_x_err*th_x_err)),
    //          d_err);
    
    return (cpl_error_get_code());
}

/*----------------------------------------------------------------------------*/
/**
 @brief Compute resolution of the Thorium lines
 @param
 @param[out]
 @param[out]
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code espdr_fit_d(espdr_line_param *THAR_lines_table,
                                  int THAR_lines_nb,
                                  espdr_inst_config *inst_config,
                                  double *residuals,
                                  double *p_data_x,
                                  double *p_data_y,
                                  double *p_data_error,
                                  double *all_p_data_x,
                                  double *all_p_data_y,
                                  double *all_p_data_error,
                                  int fit_size,
                                  int *fit_mask,
                                  double *coeffs,
                                  double *coeffs_err) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    int i, order, index_fit, index_fit_no_outliers, good_data_nb;
    espdr_poly_data *p_data = NULL;
    double *fit, chisq;
    espdr_poly_data *p2_data = NULL;
    double *fit2;
    espdr_poly_data *p3_data = NULL;
    double *fit3;
    int poly_deg = inst_config->d_fit_poly_deg;
    double residuals_big_limit = inst_config->d_fit_big_res_limit;
    double residuals_small_limit = inst_config->d_fit_small_res_limit;
    
    // Fitting the D values from the reference table
    p_data = (espdr_poly_data *)cpl_malloc(sizeof(espdr_poly_data));
    p_data->x = (double *) cpl_calloc (fit_size, sizeof (double));
    p_data->y = (double *) cpl_calloc (fit_size, sizeof (double));
    p_data->err = (double *) cpl_calloc (fit_size, sizeof (double));
    p_data->n = fit_size;
    
    fit = (double *) cpl_calloc (fit_size, sizeof(double));
    
    for (i = 0; i < fit_size; i++) {
        p_data->x[i] = p_data_x[i];
        p_data->y[i] = p_data_y[i];
        p_data->err[i] = p_data_error[i];
        if (i == -1) {
            espdr_msg("Fit input[%d]: x: %lf, y: %lf, err: %lf",
                      i, p_data->x[i], p_data->y[i], p_data->err[i]);
        }
    }
    
    my_error = espdr_fit_poly(p_data, poly_deg,
                              fit, coeffs, coeffs_err, &chisq, 0);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_fit_poly failed: %s",
                 cpl_error_get_message_default(my_error));
    
    // Take off the big outliers
    espdr_msg("Taking off outliers with residuals bigger than %f", residuals_big_limit);
    good_data_nb = 0;
    index_fit_no_outliers = 0;
    index_fit = 0;
    for (i = 0; i < THAR_lines_nb; i++) {
        order = THAR_lines_table[i].order;
        //if ((THAR_lines_table[i].qc_flag == 1) && (index_fit < fit_size)) {
        if ((fit_mask[i] == 1) && (index_fit < fit_size)) {
            residuals[i] = p_data->y[index_fit] - fit[index_fit];
            //espdr_msg("order %d, pxl: %f, Residuals[%d] = %f", order, THAR_lines_table[i].peak_pxl, i, residuals[i]);
            if (fabs(residuals[i]) < residuals_big_limit) {
                good_data_nb++;
            } else {
                THAR_lines_table[i].qc_flag = 0;
                fit_mask[i] = 0;
            }
            index_fit++;
        } else {
            residuals[i] = 0.0;
        }
    }
    
    if (index_fit > fit_size) {
        espdr_msg_warning("Mismatch between nb of lines in the fit and nb of lines with QC = 1!");
    }
    
    //espdr_msg("THAR lines nb: %d, first fit size: %d, good lines nb: %d, index fit: %d, QC1: %d",
    //          THAR_lines_nb, fit_size, good_data_nb, index_fit, QC1_lines_nb);
    
    if (good_data_nb < fit_size) {
        // second fit is necessary
        espdr_msg("Performing second fit after %d outliers elimination", fit_size-good_data_nb);
        p2_data = (espdr_poly_data *)cpl_malloc(sizeof(espdr_poly_data));
        p2_data->x = (double *) cpl_calloc (good_data_nb, sizeof (double));
        p2_data->y = (double *) cpl_calloc (good_data_nb, sizeof (double));
        p2_data->err = (double *) cpl_calloc (good_data_nb, sizeof (double));
        p2_data->n = good_data_nb;
        
        fit2 = (double *) cpl_calloc (good_data_nb, sizeof(double));
        
        for (i = 0; i < THAR_lines_nb; i++) {
            order = THAR_lines_table[i].order;
            //if (THAR_lines_table[i].qc_flag == 1) {
            if (fit_mask[i] == 1) {
                // the qc_flag is set to 0 if res too big in the loop before
                p2_data->x[index_fit_no_outliers] = all_p_data_x[i];
                p2_data->y[index_fit_no_outliers] = all_p_data_y[i];
                p2_data->err[index_fit_no_outliers] = all_p_data_error[i];
                index_fit_no_outliers++;
            }
        }
        
        my_error = espdr_fit_poly(p2_data, poly_deg,
                                  fit2, coeffs, coeffs_err, &chisq, 0);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_fit_poly failed: %s",
                     cpl_error_get_message_default(my_error));
    } else {
        espdr_msg("No second fit, no outliers eliminated");
    }
    
    // Take off the small outliers
    espdr_msg("Taking off outliers with residuals bigger than %f", residuals_small_limit);
    int best_data_nb = 0;
    index_fit_no_outliers = 0;
    index_fit = 0;
    for (i = 0; i < THAR_lines_nb; i++) {
        order = THAR_lines_table[i].order;
        //if ((THAR_lines_table[i].qc_flag == 1) && (index_fit < good_data_nb)) {
        if ((fit_mask[i] == 1) && (index_fit < good_data_nb)) {
            if (good_data_nb < fit_size) {
                residuals[i] = p2_data->y[index_fit] - fit2[index_fit];
            } else {
                residuals[i] = p_data->y[index_fit] - fit[index_fit];
            }
            //espdr_msg("order %d, pxl: %f, Residuals[%d] = %f", order, THAR_lines_table[i].peak_pxl, i, residuals[i]);
            if (fabs(residuals[i]) < residuals_small_limit) {
                best_data_nb++;
            } else {
                THAR_lines_table[i].qc_flag = 0;
                fit_mask[i] = 0;
            }
            index_fit++;
        } else {
            residuals[i] = 0.0;
        }
    }
    
    if (index_fit > good_data_nb) {
        espdr_msg_warning("Mismatch between nb of lines in the fit and nb of lines with QC = 1!");
    }
    
    //espdr_msg("THAR lines nb: %d, first fit size: %d, good lines nb: %d, index fit: %d, QC1: %d",
    //          THAR_lines_nb, fit_size, good_data_nb, index_fit, QC1_lines_nb);
    
    if (best_data_nb < good_data_nb) {
        // third fit is necessary
        espdr_msg("Performing third fit after %d outliers elimination", good_data_nb-best_data_nb);
        p3_data = (espdr_poly_data *)cpl_malloc(sizeof(espdr_poly_data));
        p3_data->x = (double *) cpl_calloc (best_data_nb, sizeof (double));
        p3_data->y = (double *) cpl_calloc (best_data_nb, sizeof (double));
        p3_data->err = (double *) cpl_calloc (best_data_nb, sizeof (double));
        p3_data->n = best_data_nb;
        
        fit3 = (double *) cpl_calloc (best_data_nb, sizeof(double));
        
        for (i = 0; i < THAR_lines_nb; i++) {
            order = THAR_lines_table[i].order;
            //if (THAR_lines_table[i].qc_flag == 1) {
            if (fit_mask[i] == 1) {
                // the qc_flag is set to 0 if res too big in the loop before
                p3_data->x[index_fit_no_outliers] = all_p_data_x[i];
                p3_data->y[index_fit_no_outliers] = all_p_data_y[i];
                p3_data->err[index_fit_no_outliers] = all_p_data_error[i];
                index_fit_no_outliers++;
            }
        }
        
        my_error = espdr_fit_poly(p3_data, poly_deg,
                                  fit3, coeffs, coeffs_err, &chisq, 0);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_fit_poly failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    
    espdr_msg("Polynomial degree: %d, CHI2 = %lf", poly_deg, chisq);
    espdr_msg("COEFFS\tCOEFFS_ERR");
    for (i = 0; i < poly_deg; i++) {
        espdr_msg("%.15E\t%.15E", coeffs[i], coeffs_err[i]);
    }
    
#if SAVE_DEBUG_PRODUCT_WAVE_THAR
    char filename[64];
    sprintf(filename, "%s_D_fit_coeffs_%d.rdb", inst_config->instrument, poly_deg);
    FILE *ltaptr = fopen(filename, "w");
    espdr_ensure(ltaptr == NULL, CPL_ERROR_FILE_IO,
                 "Couldn't open file: %s", filename);
    fprintf(ltaptr, "coeffs\tcoeffs_err\n");
    fprintf(ltaptr, "------\t----------\n");
    for (i = 0; i < poly_deg; i++) {
        fprintf(ltaptr, "%.15E\t%.15E\n",
                coeffs[i], coeffs_err[i]);
    }
#endif
    
    if (best_data_nb < good_data_nb) {
        espdr_msg("Taking the third fit, small outliers rejected");
    } else {
        if (good_data_nb < fit_size) {
            espdr_msg("Taking the second fit, big outliers rejected");
        } else {
            espdr_msg("Taking the first fit, no outlirs rejected");
        }
    }
    
    index_fit = 0;
    for (i = 0; i < THAR_lines_nb; i++) {
        order = THAR_lines_table[i].order;
        if (THAR_lines_table[i].qc_flag == 1) {
            if (best_data_nb < good_data_nb) {
                residuals[i] = p3_data->y[index_fit] - fit3[index_fit];
            } else {
                if (good_data_nb < fit_size) {
                    residuals[i] = p2_data->y[index_fit] - fit2[index_fit];
                } else {
                    residuals[i] = p_data->y[index_fit] - fit[index_fit];
                }
            }
            index_fit++;
        } else {
            residuals[i] = 0.0;
        }
    }
    
    if (best_data_nb < good_data_nb) {
        cpl_free(p3_data->x);
        cpl_free(p3_data->y);
        cpl_free(p3_data->err);
        cpl_free(p3_data);
        cpl_free(fit3);
    }
    
    if (good_data_nb < fit_size) {
        cpl_free(p2_data->x);
        cpl_free(p2_data->y);
        cpl_free(p2_data->err);
        cpl_free(p2_data);
        cpl_free(fit2);
    }
    
    cpl_free(p_data->x);
    cpl_free(p_data->y);
    cpl_free(p_data->err);
    cpl_free(p_data);
    cpl_free(fit);
    
    return (cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief Compute resolution of the Thorium lines
 @param
 @param[out]
 @param[out]
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code espdr_fit_d_kernel(espdr_poly_data *p_data,
                                         double *inv_m,
                                         int kernel_size,
                                         double fp_d_angstrom,
                                         int poly_deg,
                                         double fwhm,
                                         int *THAR_lines_indexes,
                                         double *residuals,
                                         double *dm) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    int fit_size = p_data->n;
    double chi2 = 0.0, sigma = 0.0, kernel = 0.0;
    double fwhm_m = fwhm / 2.0 / fp_d_angstrom;
    
    espdr_msg("Starting the d_fit_kernel with poly_deg = %d and fwhm = %.2f",
              poly_deg, fwhm);
    
    // Fitting the D values from the reference table
    // fit_size = valid_THAR_lines_nb, kernel_size = FP_lines_nb
    
    double *fit = (double *) cpl_calloc (fit_size, sizeof(double));
    double *coeffs = (double *) cpl_calloc (poly_deg, sizeof(double));
    double *coeffs_err = (double *) cpl_calloc (poly_deg, sizeof(double));
    double *err_orig = (double *) cpl_calloc (fit_size, sizeof(double));
    for (int i = 0; i < fit_size; i++) {
        err_orig[i] = p_data->err[i];
    }
    
    sigma = fwhm_m / 2.0 / sqrt(2.0 * log(2.0));
    
    //double minimal_number = DBL_MIN; // = 2.225074e-308
    //espdr_msg("minimal number: %e", minimal_number);
    
    for (int i = 0; i < kernel_size; i++) {
        for (int j = 0; j < fit_size; j++) {
            kernel = exp(-(p_data->x[j] - inv_m[i])*(p_data->x[j] - inv_m[i])/2.0/(sigma*sigma))+1.0e-100;
            p_data->err[j] = err_orig[j]/kernel;
            if (i == -1) {
                espdr_msg("kernel[%d] = %.16f, p_data->err[%d] = %f",
                          j, kernel, j, p_data->err[j]);
            }
        }
        if (i == -1) {
            espdr_msg("Fit input[%d]: x: %lf, y: %lf, err: %lf",
                      i, p_data->x[i], p_data->y[i], p_data->err[i]);
        }
        my_error = espdr_fit_poly(p_data, poly_deg,
                                  fit, coeffs, coeffs_err, &chi2, 0);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_fit_poly failed: %s",
                     cpl_error_get_message_default(my_error));
        
        for (int j = 0; j < fit_size; j++) {
            if (inv_m[i] == p_data->x[j]) {
                residuals[THAR_lines_indexes[j]] = p_data->y[j] - fit[j];
            }
        }
        
        for (int j = 0; j < poly_deg; j++) {
            dm[i] = dm[i] + coeffs[j] * pow(inv_m[i], j);
        }
    }
    
    espdr_msg("D_fit_kernel finished");
    
    cpl_free(err_orig);
    cpl_free(fit);
    cpl_free(coeffs);
    cpl_free(coeffs_err);
    
    return (cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param[out]
 @param[out]
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_fit_d_lambda(espdr_line_param *THAR_lines_table,
                                  int THAR_lines_nb,
                                  espdr_line_param *FP_lines_table,
                                  int FP_lines_nb,
                                  espdr_inst_config *inst_config,
                                  double *d_static,
                                  double *d_static_err,
                                  double *d_static_err_no_min,
                                  double *d_computed,
                                  double *d_computed_err,
                                  double *d_computed_err_no_min,
                                  double *residuals,
                                  double *dm_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    int index_FP;
    int thar_order, m;
    double x_thar;
    int new_d_static = 0;
    
    cpl_vector *d_static_vector = cpl_vector_wrap(THAR_lines_nb, d_static);
    if (cpl_vector_get_sum(d_static_vector) == 0.0) {
        new_d_static = 1;
    }
    cpl_vector *d_static_err_vector = cpl_vector_wrap(THAR_lines_nb, d_static_err);
    if (cpl_vector_get_sum(d_static_err_vector) == 0.0) {
        new_d_static = 1;
    }
#if SAVE_TH_REF_TABLES
    new_d_static = 1;
#endif
    
    espdr_msg("New d_static needed? %d", new_d_static);
    
    if (new_d_static == 1) {
        
        index_FP = 0;
        for (int i = 0; i < THAR_lines_nb; i++) {
            if (THAR_lines_table[i].qc_flag == 1) {
                thar_order = THAR_lines_table[i].order;
                x_thar = THAR_lines_table[i].peak_pxl;
                index_FP = 0;
                // Finding the right order
                while ((index_FP < FP_lines_nb) && (FP_lines_table[index_FP].order < thar_order)) {
                    index_FP++;
                }
                // Finding the right pxl
                while ((index_FP < FP_lines_nb) && (FP_lines_table[index_FP].order == thar_order) &&
                       (FP_lines_table[index_FP].peak_pxl < x_thar)) {
                    index_FP++;
                }
                
                if ((index_FP == 0) || (FP_lines_table[index_FP-1].order != thar_order) ||
                    (FP_lines_table[index_FP-1].qc_flag !=1)) {
                    espdr_msg_warning("No FP line before the ThAr line (order %d, pxl: %.2lf)",
                                      thar_order, x_thar);
                    d_computed[i] = 0.0;
                    d_computed_err[i] = 0.0;
                    d_computed_err_no_min[i] = 0.0;
                    d_static_err_no_min[i] = 0.0;
                    THAR_lines_table[i].qc_flag = 0;
                } else {
                    if ((index_FP >= FP_lines_nb) || (FP_lines_table[index_FP].order != thar_order) ||
                        (FP_lines_table[index_FP].qc_flag != 1)) {
                        espdr_msg_warning("No FP line after the ThAr line (order %d, pxl: %.2lf)",
                                          thar_order, x_thar);
                        d_computed[i] = 0.0;
                        d_computed_err[i] = 0.0;
                        d_computed_err_no_min[i] = 0.0;
                        d_static_err_no_min[i] = 0.0;
                        THAR_lines_table[i].qc_flag = 0;
                    } else {
                        if ((d_static[i] == 0) && (cpl_vector_get_sum(d_static_vector) != 0.0)) {
                            // This line was not fitted in the TH_REF_TABLE and not all the lines have values = 0.0, i.e. not the first use
                            d_computed[i] = 0.0;
                            d_computed_err[i] = 0.0;
                            d_computed_err_no_min[i] = 0.0;
                            d_static_err_no_min[i] = 0.0;
                            THAR_lines_table[i].qc_flag = 0;
                        } else {
                            my_error = espdr_compute_d(THAR_lines_table, i, FP_lines_table, FP_lines_nb, index_FP, inst_config,
                                                       &d_computed[i], &d_computed_err[i], &d_computed_err_no_min[i]);
                            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                         "espdr_compute_d failed: %s",
                                         cpl_error_get_message_default(my_error));
                            
                            m = FP_lines_table[index_FP].grouping;
                            THAR_lines_table[i].grouping = m;
                            
                            d_static_err_no_min[i] = d_computed_err_no_min[i];
                        }
                    }
                }
            } else {
                d_computed[i] = 0.0;
                d_computed_err[i] = 0.0;
                d_computed_err_no_min[i] = 0.0;
                d_static_err_no_min[i] = 0.0;
            }
        }
        
    } else { // new_d_static == 0
        
        index_FP = 0;
        for (int i = 0; i < THAR_lines_nb; i++) {
            
            thar_order = THAR_lines_table[i].order;
            x_thar = THAR_lines_table[i].peak_pxl;
            
            if (d_static[i] == 0) { // This line was not fitted in the TH_REF_TABLE
                d_computed[i] = 0.0;
                d_computed_err[i] = 0.0;
                d_computed_err_no_min[i] = 0.0;
                THAR_lines_table[i].qc_flag = 0;
            } else {
                
                m = THAR_lines_table[i].grouping;
                
                if (THAR_lines_table[i].qc_flag == 1) {
                    index_FP = 0;
                    // Finding the right order
                    while ((index_FP < FP_lines_nb) && (FP_lines_table[index_FP].order < thar_order)) {
                        index_FP++;
                    }
                    // Finding the right pxl
                    //while ((index_FP < FP_lines_nb) && (FP_lines_table[index_FP].order == thar_order) &&
                    //       (FP_lines_table[index_FP].peak_pxl < x_thar)) {
                    while ((index_FP < FP_lines_nb) && (FP_lines_table[index_FP].order == thar_order) &&
                           (FP_lines_table[index_FP].grouping > m)) {
                        index_FP++;
                    }
                    
                    if ((index_FP == 0) || (FP_lines_table[index_FP-1].order != thar_order) ||
                        (FP_lines_table[index_FP-1].qc_flag !=1)) {
                        espdr_msg_warning("No FP line before the ThAr line (order %d, pxl: %.2lf)",
                                          thar_order, x_thar);
                        d_computed[i] = 0.0;
                        d_computed_err[i] = 0.0;
                        d_computed_err_no_min[i] = 0.0;
                        THAR_lines_table[i].qc_flag = 0;
                    } else {
                        if ((index_FP >= FP_lines_nb) || (FP_lines_table[index_FP].order != thar_order) ||
                            (FP_lines_table[index_FP].qc_flag != 1)) {
                            espdr_msg_warning("No FP line after the ThAr line (order %d, pxl: %.2lf)",
                                              thar_order, x_thar);
                            d_computed[i] = 0.0;
                            d_computed_err[i] = 0.0;
                            d_computed_err_no_min[i] = 0.0;
                            THAR_lines_table[i].qc_flag = 0;
                        } else {
                            my_error = espdr_compute_d(THAR_lines_table, i, FP_lines_table, FP_lines_nb, index_FP, inst_config,
                                                       &d_computed[i], &d_computed_err[i], &d_computed_err_no_min[i]);
                            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                         "espdr_compute_d failed: %s",
                                         cpl_error_get_message_default(my_error));
                            
                        }
                    }
                } else { // qc_flag == 0
                    d_computed[i] = 0.0;
                    d_computed_err[i] = 0.0;
                    d_computed_err_no_min[i] = 0.0;
                }
            }
        } // for THAR lines
    } // if new d_static

    espdr_msg("THAR lines nb: %d", THAR_lines_nb);
    
    espdr_msg("Updating the wavelengths of FP peaks, via polynomial fit with gaussian kernel");
    // New method with local polynomial fit using gaussian kernel
    int valid_THAR_lines_nb = 0;
    for (int i = 0; i < THAR_lines_nb; i++) {
        if (THAR_lines_table[i].qc_flag == 1) {
            valid_THAR_lines_nb++;
        }
    }

    // Fitting the D values from the reference table
    espdr_poly_data *p_data_kernel = (espdr_poly_data *)cpl_malloc(sizeof(espdr_poly_data));
    p_data_kernel->x = (double *) cpl_calloc (valid_THAR_lines_nb, sizeof (double));
    p_data_kernel->y = (double *) cpl_calloc (valid_THAR_lines_nb, sizeof (double));
    p_data_kernel->err = (double *) cpl_calloc (valid_THAR_lines_nb, sizeof (double));
    p_data_kernel->n = valid_THAR_lines_nb;
    double *inv_m = (double *) cpl_calloc (FP_lines_nb, sizeof(double));
    double *dm = (double *) cpl_calloc (FP_lines_nb, sizeof(double));

    int index_THAR = 0;
    int *THAR_lines_indexes = (int *) cpl_calloc (valid_THAR_lines_nb, sizeof(int));
    for (int i = 0; i < THAR_lines_nb; i++) {
        if (THAR_lines_table[i].qc_flag == 1) {
            p_data_kernel->x[index_THAR] = 1.0 / THAR_lines_table[i].grouping;
            p_data_kernel->y[index_THAR] = d_computed[i];
            p_data_kernel->err[index_THAR] = d_computed_err[i];
            THAR_lines_indexes[index_THAR] = i;
            index_THAR++;
        }
    }
    
    for (int i = 0; i < FP_lines_nb; i++) {
        inv_m[i] = 1.0 / FP_lines_table[i].grouping;
    }
    
    my_error = espdr_fit_d_kernel(p_data_kernel, inv_m, FP_lines_nb,
                                  inst_config->fp_d_angstrom,
                                  inst_config->d_fit_kernel_poly_deg,
                                  inst_config->d_fit_kernel_fwhm,
                                  THAR_lines_indexes, residuals, dm);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_fit_d_kernel failed: %s",
                 cpl_error_get_message_default(my_error));
    cpl_free(THAR_lines_indexes);
    
    double residuals_big_limit = inst_config->d_fit_big_res_limit;
    int do_sigma_clipping = 0;
    int lines_clipped_nb = 0;
    
    for (int i = 0; i < THAR_lines_nb; i++) {
        if (THAR_lines_table[i].qc_flag == 1) {
            if (fabs(residuals[i]) > residuals_big_limit) {
                THAR_lines_table[i].qc_flag = 0;
                do_sigma_clipping = 1;
                espdr_msg("Sig clipping line at ll = %f order %d",
                          THAR_lines_table[i].wavelength, THAR_lines_table[i].order);
                lines_clipped_nb++;
            }
        }
    }
    
    espdr_msg("Number of lines clipped with the limit %f: %d",
              residuals_big_limit, lines_clipped_nb);
    
    espdr_poly_data *p_data2_kernel = (espdr_poly_data *)cpl_malloc(sizeof(espdr_poly_data));
    
    if (do_sigma_clipping == 1) {
        valid_THAR_lines_nb = 0;
        for (int i = 0; i < THAR_lines_nb; i++) {
            if (THAR_lines_table[i].qc_flag == 1) {
                valid_THAR_lines_nb++;
            }
        }
        
        cpl_free(dm);
        
        // Fitting the D values from the reference table
        p_data2_kernel->x = (double *) cpl_calloc (valid_THAR_lines_nb, sizeof (double));
        p_data2_kernel->y = (double *) cpl_calloc (valid_THAR_lines_nb, sizeof (double));
        p_data2_kernel->err = (double *) cpl_calloc (valid_THAR_lines_nb, sizeof (double));
        p_data2_kernel->n = valid_THAR_lines_nb;
        dm = (double *) cpl_calloc (FP_lines_nb, sizeof(double));

        index_THAR = 0;
        int *THAR_lines_indexes2 = (int *) cpl_calloc (valid_THAR_lines_nb, sizeof(int));
        for (int i = 0; i < THAR_lines_nb; i++) {
            if (THAR_lines_table[i].qc_flag == 1) {
                p_data2_kernel->x[index_THAR] = 1.0 / THAR_lines_table[i].grouping;
                p_data2_kernel->y[index_THAR] = d_computed[i];
                p_data2_kernel->err[index_THAR] = d_computed_err[i];
                THAR_lines_indexes2[index_THAR] = i;
                index_THAR++;
            }
        }
        
        my_error = espdr_fit_d_kernel(p_data2_kernel, inv_m, FP_lines_nb,
                                      inst_config->fp_d_angstrom,
                                      inst_config->d_fit_kernel_poly_deg,
                                      inst_config->d_fit_kernel_fwhm,
                                      THAR_lines_indexes2, residuals, dm);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_fit_d_kernel failed: %s",
                     cpl_error_get_message_default(my_error));
        
        cpl_free(THAR_lines_indexes2);
    } // do_sigma_clipping == 1
    
    // Compute the wavelength
    for (int i = 0; i < FP_lines_nb; i++) {
        FP_lines_table[i].wavelength = (2.0 * (dm[i])) / (double)(FP_lines_table[i].grouping);
        dm_RE[i] = dm[i];
    }
    
    espdr_msg("Freeing mamory");
    
    cpl_free(p_data_kernel->x);
    cpl_free(p_data_kernel->y);
    cpl_free(p_data_kernel->err);
    cpl_free(p_data_kernel);
    if (do_sigma_clipping == 1) {
        cpl_free(p_data2_kernel->x);
        cpl_free(p_data2_kernel->y);
        cpl_free(p_data2_kernel->err);
    }
    cpl_free(p_data2_kernel);
    cpl_free(inv_m);
    cpl_free(dm);
    
    return(cpl_error_get_code());
}




/*----------------------------------------------------------------------------*/
/**
 @brief Compute resolution of the Thorium lines
 @param
 @param[out]
 @param[out]
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_compute_THAR_resolution(espdr_line_param *THAR_lines_table,
                                             double *resolution,
                                             int THAR_lines_nb,
                                             double *resol_median) {
    
    int i;
    cpl_array *resol_array;
    int *idxNULL = NULL;
    int invalid_nb = 0;
    
    for (i = 0; i < THAR_lines_nb; i++) {
        if (THAR_lines_table[i].qc_flag == 1) {
            resolution[i] = THAR_lines_table[i].wavelength /
                    (THAR_lines_table[i].dispersion * THAR_lines_table[i].fwhm);
        } else {
            resolution[i] = 0.0;
        }
    }
    
    resol_array = cpl_array_wrap_double(resolution, THAR_lines_nb);
    
    for (i = 0; i < THAR_lines_nb; i++) {
        if (cpl_array_get(resol_array, i, idxNULL) == 0.0) {
            cpl_array_set_invalid(resol_array, i);
            invalid_nb++;
        }
    }
    
    if (invalid_nb == THAR_lines_nb) {
        *resol_median = 0.0;
    } else {
        *resol_median = cpl_array_get_median(resol_array);
    }
    
    cpl_array_unwrap(resol_array);
    
    return(cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief Fit calibration lines on the raw frame
 @param         lines_table         detected lines peaks
 @param         lines_nb            number of lines in the table
 @param         raw                 input data
 @param         fibre_nr            number of the fibre
 @param         orders_coeffs       coeffs of the orders fit
 @param         pixel_geom          pixel geometry image
 @param         window_size_x       size of the fit window
 @param         inst_config         instrument config
 @param         CCD_geom            CCD geometry
 @param         RON                 Read-Out Noise per output
 @param[out]    raw_lines_nb_RE     computed number of fitted raw lines
 @param[out]    raw_lines_table_RE  table of fitted raw lines
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_fit_THAR_raw_lines(espdr_line_param *lines_table,
                                        int lines_nb,
                                        cpl_imagelist *raw,
                                        int fibre_nr,
                                        cpl_table **orders_coeffs,
                                        cpl_image *pixel_geom,
                                        int window_size_x,
                                        espdr_inst_config *inst_config,
                                        espdr_CCD_geometry *CCD_geom,
                                        double *RON,
                                        int *raw_lines_nb_RE,
                                        espdr_raw_line_param *raw_lines_table_RE) {
    
    if (pixel_geom == NULL) {
        /* ASE : Do nothing Just to document this parameter is not used */
    }
    
    int i, k, pxl, line, pixel, output_index;
    int order, ext_nr, order_in_ext, coeff, coeffs_nb;
    int left_margin, right_margin, lower_margin, upper_margin;
    int line_index, raw_line_index;
    int fit_window_x, fit_window_y, window_size_y;
    int size_x, size_y, order_pos_int, raw_size_x;
    double peak_x, order_pos, flux;
    char column_name[KEYWORD_LENGTH];
    espdr_ngauss_data *g_data_x = NULL;
    espdr_ngauss_data *g_data_y = NULL;
    espdr_ngauss_result *g_res_x = NULL;
    double *x0;
    cpl_image *raw_line = NULL;
    double *raw_data = NULL;
    cpl_error_code my_error = CPL_ERROR_NONE;
    int slices_nb = inst_config->slices_nb;
    int inter_slice_dist = inst_config->inter_slice_dist;
    int slice_width = inst_config->slice_width;
    int background_pxls = inst_config->background_pxls;
    cpl_image **raw_table = NULL;
    int good_lines_nb = 0;
    int smallest_order_start = 0;
    
    //espdr_msg("Starting the raw fit for fibre %c", fibre_name[fibre_nr]);
    
    espdr_ensure(cpl_imagelist_get_size(raw) != CCD_geom->ext_nb,
                 CPL_ERROR_INCOMPATIBLE_INPUT,
                 "The input raw imagelist has wrong length");
    
    double *one_order_coeffs = NULL;
    
    raw_table = (cpl_image **)cpl_malloc(CCD_geom->ext_nb*sizeof(cpl_image *));
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        raw_table[i] = cpl_imagelist_get(raw, i);
    }
    
    fit_window_x = 2 * window_size_x + 1;
    window_size_y = (int)(((slices_nb-1)*inter_slice_dist+slice_width+1)/2) +
    background_pxls;
    fit_window_y = 2 * window_size_y + 1;
    
    g_data_x = (espdr_ngauss_data *)cpl_malloc(sizeof(espdr_ngauss_data));
    g_data_x->n = fit_window_x;
    g_data_x->m = 1;
    g_data_x->delta = 0.0;
    g_data_x->x = (double *) cpl_calloc (fit_window_x, sizeof(double));
    g_data_x->y = (double *) cpl_calloc (fit_window_x, sizeof(double));
    g_data_x->err = (double *) cpl_calloc (fit_window_x, sizeof(double));
    g_data_x->fit = (double *)cpl_calloc (fit_window_x, sizeof(double));
    
    g_res_x = (espdr_ngauss_result *) cpl_malloc (sizeof(espdr_ngauss_result));
    g_res_x->k = (double *) cpl_calloc (1, sizeof(double));
    g_res_x->sig_k = (double *) cpl_calloc (1, sizeof(double));
    g_res_x->x0 = (double *) cpl_calloc (1, sizeof(double));
    g_res_x->sig_x0 = (double *) cpl_calloc (1, sizeof(double));
    g_res_x->fwhm = (double *) cpl_calloc (1, sizeof(double));
    g_res_x->sig_fwhm = (double *) cpl_calloc (1, sizeof(double));
    
    x0 = (double *)cpl_calloc(1, sizeof(double));
    
    g_data_y = (espdr_ngauss_data *)cpl_malloc(sizeof(espdr_ngauss_data));
    g_data_y->n = fit_window_y;
    g_data_y->m = slices_nb;
    g_data_y->delta = 0.0;
    g_data_y->x = (double *) cpl_calloc (fit_window_y, sizeof(double));
    g_data_y->y = (double *) cpl_calloc (fit_window_y, sizeof(double));
    g_data_y->err = (double *) cpl_calloc (fit_window_y, sizeof(double));
    g_data_y->fit = (double *)cpl_calloc (fit_window_y, sizeof(double));
    
    espdr_msg("lines table has %d lines", lines_nb);
    i = 0;
    line_index = 0;
    raw_line_index = 0;
    
    // Get the smallest order start
    smallest_order_start = cpl_image_get_size_x(raw_table[0]);
    for (i = 0; i < CCD_geom->ext_nb * inst_config->fibres_nb; i++) {
        if (inst_config->order_start[i] < smallest_order_start) {
            smallest_order_start = inst_config->order_start[i];
        }
    }
    //espdr_msg("Smallest order start: %d", smallest_order_start);
    
    while (line_index < lines_nb) {
        //espdr_msg("Treating line %d with QC = %d",
        //          line_index, lines_table[line_index].qc_flag);
        if (lines_table[line_index].qc_flag == 1) {
            order = lines_table[line_index].order;
            
            ext_nr = espdr_get_ext_index_for_order(order, fibre_nr,
                                                   inst_config, CCD_geom);
            
            // Add the smallest order start to have to correct peak_x
            //int order_index = CCD_geom->ext_nb * ext_nr + fibre_nr;
            //peak_x = (int)(lines_table[line_index].peak_pxl + 0.5) +
            //               inst_config->order_start[order_index];
            
            peak_x = (int)(lines_table[line_index].peak_pxl + 0.5) + smallest_order_start;
            
            //espdr_msg("RAW_FIT:::::::::: order_index = %d, order_start = %d",
            //          order_index, inst_config->order_start[order_index]);
            //peak_x = (int)(lines_table[line_index].peak_pxl + 0.5);
            left_margin = peak_x - window_size_x;
            right_margin = peak_x + window_size_x;
            
            size_x = cpl_image_get_size_x(raw_table[ext_nr]);
            size_y = cpl_image_get_size_y(raw_table[ext_nr]);
            
            order_in_ext = espdr_get_order_nr_in_ext(order, fibre_nr, ext_nr,
                                                     inst_config, CCD_geom);
            
            coeffs_nb = cpl_table_get_ncol(orders_coeffs[ext_nr]);
            one_order_coeffs = (double *)cpl_calloc(coeffs_nb, sizeof(double));
            
            //espdr_msg("before coeffs, coeffs nb: %d", coeffs_nb);
            
            //espdr_msg("order: %d, ext: %d", order, ext_nr);
            for (coeff = 0; coeff < coeffs_nb; coeff++) {
                sprintf(column_name, "COEFF_%d", coeff);
                one_order_coeffs[coeff] = cpl_table_get(orders_coeffs[ext_nr],
                                                        column_name,
                                                        order_in_ext-1,
                                                        NULL);
            }
            
            //espdr_msg("COEFFS: %.10lf, %.10lf, %.10lf, %.10lf, %.10lf",
            //          one_order_coeffs[0], one_order_coeffs[1],
            //          one_order_coeffs[2], one_order_coeffs[3],
            //          one_order_coeffs[4]);
            order_pos = 0.0;
            for (k = coeffs_nb-1; k > 0; k--) {
                order_pos = (order_pos + one_order_coeffs[k]) * peak_x;
            }
            order_pos = order_pos + one_order_coeffs[0];
            order_pos_int = (int)(order_pos + 0.5);
            
            //espdr_msg("order pos computed, raw line %d, order_pos_int: %d",
            //          raw_line_index, order_pos_int);
            
            upper_margin = order_pos_int + window_size_y;
            lower_margin = order_pos_int - window_size_y;
            
            if ((left_margin > 0) && (right_margin < size_x) &&
                (upper_margin < size_y) && (lower_margin > 0)) {
                
                //espdr_msg("fitting raw line: %d", raw_line_index);
                raw_line = cpl_image_extract(raw_table[ext_nr],
                                             peak_x-window_size_x,
                                             order_pos_int,
                                             peak_x+window_size_x,
                                             order_pos_int);
                raw_data = cpl_image_get_data_double(raw_line);
                
                raw_size_x = cpl_image_get_size_x(raw_table[ext_nr]);
                pixel = (order_pos_int - 1) * raw_size_x + peak_x -1;
                output_index = espdr_get_output_index_for_pixel(ext_nr,
                                                                pixel,
                                                                CCD_geom);
                
                raw_lines_table_RE[raw_line_index].order = order;
                raw_lines_table_RE[raw_line_index].qc_flag = 1;
                raw_lines_table_RE[raw_line_index].wavelength =
                                    lines_table[line_index].wavelength;
                raw_lines_table_RE[raw_line_index].dispersion =
                                    lines_table[line_index].dispersion;
                strcpy(raw_lines_table_RE[raw_line_index].element_name,
                       lines_table[line_index].element_name);
                
                /* Gaussian fit x direction */
                //espdr_msg("Fitting x direction");
                x0[0] = peak_x;
                for (pxl = 0; pxl < 2*window_size_x+1; pxl++) {
                    g_data_x->x[pxl] = pxl + peak_x - window_size_x;
                    g_data_x->y[pxl] = raw_data[pxl];
                    if (g_data_x->y[pxl] < 0.0) {
                        flux = -raw_data[pxl];
                    } else {
                        flux = raw_data[pxl];
                    }
                    g_data_x->err[pxl] = sqrt(flux +
                                              RON[output_index]*RON[output_index]);
                }
                
                my_error = espdr_fit_Ngauss(g_data_x, x0, 3.0, g_res_x, 1, 0);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_fit_Ngauss failed: %s",
                             cpl_error_get_message_default(my_error));
                
                for (line = 0; line < 1; line++) {
                    raw_lines_table_RE[raw_line_index].peak_x = g_res_x->x0[line];
                    raw_lines_table_RE[raw_line_index].peak_x_err = g_res_x->sig_x0[line];
                    raw_lines_table_RE[raw_line_index].fwhm_x = g_res_x->fwhm[line];
                    raw_lines_table_RE[raw_line_index].fwhm_x_err = g_res_x->sig_fwhm[line];
                    raw_lines_table_RE[raw_line_index].flux_x = g_res_x->k[line];
                    raw_lines_table_RE[raw_line_index].flux_x_err = g_res_x->sig_k[line];
                    
                    //if (raw_lines_table_RE[raw_line_index].peak_x_err < min_x0_err) {
                    //    raw_lines_table_RE[raw_line_index].peak_x_err = min_x0_err;
                    //}
                }
                
                /* Quality check */
                
                if (raw_lines_table_RE[raw_line_index].peak_x_err > inst_config->line_max_sig_x0) {
                    //espdr_msg("line: %d peak_x_err too big(%lf > %lf), order %d, pxl %.0lfx%.0lf",
                    //          raw_line_index,
                    //          raw_lines_table_RE[raw_line_index].peak_x_err,
                    //          inst_config->line_max_sig_x0,
                    //          order, peak_x, order_pos);
                    raw_lines_table_RE[raw_line_index].qc_flag = 0;
                }
                if (raw_lines_table_RE[raw_line_index].fwhm_x > inst_config->line_max_fwhm) {
                    //espdr_msg("line: %d fwhm_x too big (%lf > %lf), order %d, pxl %.0lfx%.0lf",
                    //          raw_line_index,
                    //          raw_lines_table_RE[raw_line_index].fwhm_x,
                    //          inst_config->line_max_fwhm,
                    //          order, peak_x, order_pos);
                    raw_lines_table_RE[raw_line_index].qc_flag = 0;
                }
                if (raw_lines_table_RE[raw_line_index].flux_x < inst_config->line_min_flux_el) {
                    //espdr_msg("line: %d flux_x too small (%lf < %lf), order %d, pxl %.0lfx%.0lf",
                    //          raw_line_index,
                    //          raw_lines_table_RE[raw_line_index].flux_x,
                    //          inst_config->line_min_flux_el,
                    //          order, peak_x, order_pos);
                    raw_lines_table_RE[raw_line_index].qc_flag = 0;
                }
                
                cpl_image_delete(raw_line);
                
                /* N-Gaussian fit y direction */
                //espdr_msg("Fitting y direction");
                raw_line = cpl_image_extract(raw_table[ext_nr],
                                             peak_x,
                                             order_pos_int-window_size_y,
                                             peak_x,
                                             order_pos_int+window_size_y);
                raw_data = cpl_image_get_data_double(raw_line);
                
                //espdr_msg("Filling g_data_y");
                g_data_y->m = slices_nb;
                g_data_y->n = 2*window_size_y+1;
                for (pxl = 0; pxl < 2*window_size_y+1; pxl++) {
                    g_data_y->x[pxl] = pxl + order_pos_int - window_size_y;
                    g_data_y->y[pxl] = raw_data[pxl];
                    if (g_data_y->y[pxl] < 0.0) {
                        flux = -raw_data[pxl];
                    } else {
                        flux = raw_data[pxl];
                    }
                    g_data_y->err[pxl] = sqrt(flux +
                                              RON[output_index]*RON[output_index]);
                }
                
                //espdr_msg("fit_Ngauss_equi_dist");
                my_error = espdr_fit_Ngauss_equi_dist(g_data_y,
                                                      inter_slice_dist,
                                                      slice_width,
                                                      &raw_lines_table_RE[raw_line_index].peak_y,
                                                      &raw_lines_table_RE[raw_line_index].peak_y_err,
                                                      &raw_lines_table_RE[raw_line_index].flux_y,
                                                      &raw_lines_table_RE[raw_line_index].flux_y_err,
                                                      &raw_lines_table_RE[raw_line_index].fwhm_y,
                                                      &raw_lines_table_RE[raw_line_index].fwhm_y_err,
                                                      1, 0);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_fit_Ngauss_equi_dist failed: %s",
                             cpl_error_get_message_default(my_error));
                
                //espdr_msg("Quality check");
                /* Quality check */
                if (raw_lines_table_RE[raw_line_index].peak_y_err > inst_config->line_max_sig_x0) {
                    //espdr_msg("line: %d peak_y_err too big(%lf > %lf), order %d, pxl %.0lfx%.0lf",
                    //          raw_line_index,
                    //          raw_lines_table_RE[raw_line_index].peak_y_err,
                    //          inst_config->line_max_sig_x0,
                    //          order, peak_x, order_pos);
                    raw_lines_table_RE[raw_line_index].qc_flag = 0;
                }
                if (raw_lines_table_RE[raw_line_index].fwhm_y > inst_config->line_max_fwhm) {
                    //espdr_msg("line: %d fwhm_y too big (%lf > %lf), order %d, pxl %.0lfx%.0lf",
                    //          raw_line_index,
                    //          raw_lines_table_RE[raw_line_index].fwhm_y,
                    //          inst_config->line_max_fwhm,
                    //          order, peak_x, order_pos);
                    raw_lines_table_RE[raw_line_index].qc_flag = 0;
                }
                if (raw_lines_table_RE[raw_line_index].flux_y < inst_config->line_min_flux_el) {
                    //espdr_msg("line: %d flux_y too small (%lf < %lf), order %d, pxl %.0lfx%.0lf",
                    //          raw_line_index,
                    //          raw_lines_table_RE[raw_line_index].flux_y,
                    //          inst_config->line_min_flux_el,
                    //          order, peak_x, order_pos);
                    raw_lines_table_RE[raw_line_index].qc_flag = 0;
                }
                
                cpl_image_delete(raw_line);
                
                cpl_free(one_order_coeffs);
                
                if (raw_lines_table_RE[raw_line_index].qc_flag == 1) {
                    good_lines_nb++;
                }
                
                raw_line_index++;
            } // end OK for fit
        }
        line_index++;
    }
    
    *raw_lines_nb_RE = raw_line_index;
    
    espdr_msg("Correctly fitted raw lines number: %d", good_lines_nb);
    
    cpl_free(x0);
    
    cpl_free(g_data_x->x);
    cpl_free(g_data_x->y);
    cpl_free(g_data_x->fit);
    cpl_free(g_data_x->err);
    cpl_free(g_data_x);
    
    cpl_free(g_res_x->k);
    cpl_free(g_res_x->sig_k);
    cpl_free(g_res_x->x0);
    cpl_free(g_res_x->sig_x0);
    cpl_free(g_res_x->fwhm);
    cpl_free(g_res_x->sig_fwhm);
    cpl_free(g_res_x);
    
    cpl_free(g_data_y->x);
    cpl_free(g_data_y->y);
    cpl_free(g_data_y->fit);
    cpl_free(g_data_y->err);
    cpl_free(g_data_y);
    
    cpl_free(raw_table);
    
    return(cpl_error_get_code());
}



/*----------------------------------------------------------------------------*/
/**
 @brief Save the QC KWs
 @param         lines_nb_per_order  number of valid lines for each order
 @param         thar_rejected_lines number of rejected lines
 @param         flux ratio          flux ratio
 @param         global_shift        global shift
 @param         chisq_per_order     reduced CHI2 of ll solution per order
 @param         rms_per_order       dispersion of residuals around fit per order
 @param         res_per_order       spectral resolution per order
 @param         lamp_offset         ThAr lamp offset
 @param         orders_nb           number of orders
 @param         fibre_source        calibration source type
 @param         inst_config         instrument config
 @param         qc_kws              QC KWs names
 @param[out]    keywords_RE         saved QC keywords
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_wave_THAR_QC(int *lines_nb_per_order,
                                  double *chisq_per_order,
                                  double *rms_per_order,
                                  int *order_fit,
                                  double resol_median,
                                  int orders_nb,
                                  int saturated_lines_nb,
                                  int valid_lines_nb,
                                  double flux_ratio_median,
                                  double rv_diff_median,
                                  espdr_src_type fibre_source,
                                  int fibre_nr,
                                  cpl_table *drift_results,
                                  cpl_image *wave_matrix,
                                  cpl_image *s2d_qual,
                                  espdr_inst_config *inst_config,
                                  espdr_CCD_geometry *CCD_geom,
                                  espdr_qc_keywords *qc_kws,
                                  cpl_propertylist *keywords_RE) {
    
    int total_lines_nb = 0;
    char *new_keyword = NULL;
    char comment[COMMENT_LENGTH];
    int lines_nb_check = 1;
    int chi2_check = 1, rms_check = 1, resol_check = 1;
    int drift_chi2_check = 1;
    int drift_flux_ratio_check = 1;
    int drift_mean_check = 1;
    int drift_mean_err_check = 1;
    int check_flag = 1, total_check_flag = 1;
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    int min_lines_total = inst_config->fp_min_lines_total;
    int min_lines_per_order = inst_config->wave_sol_min_lines_per_order;
    double thar_sol_max_chi2 = inst_config->thar_sol_max_chi2;
    double thar_sol_max_rms = inst_config->thar_sol_max_rms;
    double wave_min_resol = inst_config->wave_min_resol;
    double thar_min_flux_ratio = inst_config->thar_min_flux_ratio[fibre_nr];
    double thar_max_flux_ratio = inst_config->thar_max_flux_ratio[fibre_nr];
    
    /* Added keywords checks + change to -1.0 */
    for (int i = 0; i < orders_nb; i++) {
        if (isnan(chisq_per_order[i]) || isinf(chisq_per_order[i])) {
            chisq_per_order[i] = -1.0;
        }
        if (isnan(rms_per_order[i]) || isinf(rms_per_order[i])) {
            rms_per_order[i] = -1.0;
        }
    }
    
    if (isnan(resol_median) || isinf(resol_median)) {
        resol_median = -1.0;
    }
    
    /* QC KWs per order */
    for (int i = 0; i < orders_nb; i++) {
        new_keyword = espdr_add_index_to_keyword(qc_kws->qc_wave_order_lines_nb_kw_first,
                                                 qc_kws->qc_wave_order_lines_nb_kw_last,
                                                 i+1);
        sprintf(comment, "Number of valid FP lines in order %d", i+1);
        my_error = espdr_keyword_add_int(new_keyword, lines_nb_per_order[i], comment,
                                         &keywords_RE);
        cpl_free(new_keyword);
        
        if ((lines_nb_per_order[i] >= 0) && (lines_nb_per_order[i] < min_lines_per_order)) {
            lines_nb_check = 0;
        }
        
        new_keyword = espdr_add_index_to_keyword(qc_kws->qc_wave_order_chisq_kw_first,
                                                 qc_kws->qc_wave_order_chisq_kw_last,
                                                 i+1);
        sprintf(comment, "Reduced CHI2 of ll solution for order %d", i+1);
        my_error = espdr_keyword_add_double(new_keyword, chisq_per_order[i], comment,
                                            &keywords_RE);
        cpl_free(new_keyword);
        
        if (chisq_per_order[i] > thar_sol_max_chi2) {
            chi2_check = 0;
        }
        
        new_keyword = espdr_add_index_to_keyword(qc_kws->qc_wave_order_rms_kw_first,
                                                 qc_kws->qc_wave_order_rms_kw_last,
                                                 i+1);
        sprintf(comment, "[m/s] Dispersion of residuals around fit for order %d", i+1);
        my_error = espdr_keyword_add_double(new_keyword, rms_per_order[i], comment,
                                            &keywords_RE);
        cpl_free(new_keyword);
        
        if (rms_per_order[i] > thar_sol_max_rms) {
            rms_check = 0;
        }
        
        total_lines_nb += lines_nb_per_order[i];
    }
    if (total_lines_nb < 0) total_lines_nb = -1;
    
    /* Global QC KWs */
    
    sprintf(comment, "Thar lines resolution median");
    my_error = espdr_keyword_add_double(qc_kws->qc_wave_order_resol_median_kw,
                                        resol_median, comment, &keywords_RE);
    
    sprintf(comment, "Thar saturated lines nb");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_thar_satur_lines_nb_kw,
                                     saturated_lines_nb, comment, &keywords_RE);
    
    sprintf(comment, "Thar valid lines nb");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_thar_valid_lines_nb_kw,
                                     valid_lines_nb, comment, &keywords_RE);
    
    int tmp_flux_ratio = flux_ratio_median * 10000.0;
    double formatted_flux_ratio = tmp_flux_ratio / 10000.0;
    sprintf(comment, "Thar lines flux ratio median");
    my_error = espdr_keyword_add_double(qc_kws->qc_wave_thar_flux_ratio_median_kw,
                                        formatted_flux_ratio, comment, &keywords_RE);
    
    int tmp_rv_diff = rv_diff_median * 10000.0;
    double formatted_rv_diff = tmp_rv_diff / 10000.0;
    sprintf(comment, "[m/s] Thar lines RV difference median");
    my_error = espdr_keyword_add_double(qc_kws->qc_wave_thar_rv_diff_median_kw,
                                        formatted_rv_diff, comment, &keywords_RE);
    

    sprintf(comment, "Total number of valid FP lines");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_lines_nb_kw,
                                     total_lines_nb, comment, &keywords_RE);
    
    espdr_full_drift_QC(drift_results, inst_config, CCD_geom, qc_kws,
                        &drift_chi2_check, &drift_flux_ratio_check,
                        &drift_mean_check, &drift_mean_err_check,
                        keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_full_drift_QC failed: %s",
                 cpl_error_get_message_default(my_error));
    
    /* QC CHECK KWs */
    if ((lines_nb_check == 1) || (fibre_source != THAR)) {
        check_flag = 1;
    } else {
        check_flag = 0;
        total_check_flag = 0;
    }
    
    sprintf(comment, "QC on min number of lines/order");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_order_lines_min_check_kw,
                                     check_flag, comment, &keywords_RE);
    
    espdr_msg("--->>> total lines nb: %d, min nb: %d, fibre_source: %d",
              total_lines_nb, min_lines_total, fibre_source);
    if ((total_lines_nb >= min_lines_total) || (fibre_source != THAR)) {
        check_flag = 1;
    } else {
        check_flag = 0;
        total_check_flag = 0;
    }
    espdr_msg("check flag: %d, total check flag: %d", check_flag, total_check_flag);
    
    sprintf(comment, "QC on total number of detected lines");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_lines_tot_check_kw,
                                     check_flag, comment, &keywords_RE);
    
    if ((chi2_check == 1) || (fibre_source != THAR)) {
        check_flag = 1;
    } else {
        check_flag = 0;
        total_check_flag = 0;
    }
    
    sprintf(comment, "QC on wavelength solution (CHI2)");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_thar_chi2_check_kw,
                                     check_flag, comment, &keywords_RE);
    
    if ((rms_check == 1) || (fibre_source != THAR)) {
        check_flag = 1;
    } else {
        check_flag = 0;
        total_check_flag = 0;
    }
    
    sprintf(comment, "QC on wavelength solution (RMS)");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_thar_rms_check_kw,
                                     check_flag, comment, &keywords_RE);
    
    if ((resol_median > 0.0) && (resol_median < wave_min_resol)) {
        resol_check = 0;
    }
    
    if ((resol_check == 1) || (fibre_source != THAR)) {
        check_flag = 1;
    } else {
        check_flag = 0;
        total_check_flag = 0;
    }
    
    sprintf(comment, "QC on spectral resolution");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_resol_check_kw,
                                     check_flag, comment, &keywords_RE);
    
    //espdr_msg("Flux ratio limits of ThAr lines: %.2f <-> %.2f",
    //          thar_min_flux_ratio, thar_max_flux_ratio);
    if ((flux_ratio_median > thar_min_flux_ratio) && (flux_ratio_median < thar_max_flux_ratio)) {
        check_flag = 1;
    } else {
        espdr_msg_warning("The Th lines flux changed significantly, ratio = %.4f",
                          flux_ratio_median);
        check_flag = 0;
        total_check_flag = 0;
    }
    
    sprintf(comment, "QC on TH lines flux ratio");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_thar_flux_ratio_check_kw,
                                     check_flag, comment, &keywords_RE);
    
    if ((drift_chi2_check == 1) || (fibre_source != THAR)) {
        check_flag = 1;
    } else {
        check_flag = 0;
        total_check_flag = 0;
    }
    
    if (drift_results != NULL) {
        sprintf(comment, "QC on drift (CHI2)");
        my_error = espdr_keyword_add_int(qc_kws->qc_wave_drift_chi2_check_kw,
                                         check_flag, comment, &keywords_RE);
    }
    
    // Check on the increasing wavelength through the order
    if (fibre_source == THAR) {
        cpl_image *wave_order = NULL;
        cpl_image *qual_order = NULL;
        double *wave_order_data = NULL;
        int *qual_order_data = NULL;
        int size_x = cpl_image_get_size_x(wave_matrix);
        int size_y = cpl_image_get_size_y(wave_matrix);
        int decreasing = 0;
        check_flag = 1;
        for (int i = 0; i < size_y; i++) {
            wave_order = cpl_image_extract(wave_matrix, 1, i+1, size_x, i+1);
            wave_order_data = cpl_image_get_data_double(wave_order);
            qual_order = cpl_image_extract(s2d_qual, 1, i+1, size_x, i+1);
            qual_order_data = cpl_image_get_data_int(qual_order);
            decreasing = 0;
            for (int j = 1; j < size_x; j++) {
                if ((qual_order_data[j] != OTHER_BAD_PIXEL) &&
                    (wave_order_data[j] < wave_order_data[j-1])) {
                    check_flag = 0;
                    total_check_flag = 0;
                    decreasing = 1;
                }
            }
            if (decreasing == 1) {
                espdr_msg_warning("Wavelength decreasing in order %d", i+1);
            }
        }
        
        sprintf(comment, "QC on wave sol quality");
        my_error = espdr_keyword_add_int(qc_kws->qc_wave_sol_qual_check_kw,
                                         check_flag, comment, &keywords_RE);
    }
    
    sprintf(comment, "Overall WAVE QC");
    my_error = espdr_keyword_add_int(qc_kws->qc_wave_check_kw,
                                     total_check_flag, comment, &keywords_RE);
    
    return(cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief Save the QC KWs
 @param         THAR_lines_table    table of THAR lines
 @param         THAR_lines_nb           number of THAR lines in the table
 @param         d_computed                  FP D computed for each TH line
 @param         d_computed_err         error of FP D computed for each TH line
 @param         residuals                    residuals of the FP D fit
 @param         inst_config                instrument config
 @param         qc_kws                           QC KWs names
 @param[out]    keywords_RE                saved QC keywords
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_wave_FP_D_QC(espdr_line_param *THAR_lines_table,
                                  int THAR_lines_nb,
                                  espdr_line_param *FP_lines_table,
                                  int FP_lines_nb,
                                  double *d_computed_err,
                                  double *residuals,
                                  double *dm,
                                  double ll_first,
                                  double ll_last,
                                  espdr_inst_config *inst_config,
                                  espdr_qc_keywords *qc_kws,
                                  cpl_propertylist *keywords_RE) {
    
    char comment[COMMENT_LENGTH];
    char *new_keyword = NULL;
    double fp_d_mean = 0.0, fp_d_mean_err = 0.0;
    double fp_d_residuals_rms = 0.0;
    double fp_d_residuals_min = 1000000000.0, fp_d_residuals_max = -1000000000.0;
    double weights = 0.0;
    int valid_lines_nb = 0;
    
   for (int i = 0; i < THAR_lines_nb; i++) {
        if (THAR_lines_table[i].qc_flag == 1) {
            if (d_computed_err[i] != 0.0) {
                weights += 1.0/(d_computed_err[i] * d_computed_err[i]);
                valid_lines_nb++;
            } else {
                espdr_msg_warning("Incoherence: THAR line qc is 1, but d_err is 0.0");
            }
        }
    }
    if (weights > 0.0) {
        fp_d_mean_err = 1.0 / sqrt(weights);
    }
    
    double sum = 0.0;
    for (int i = 0; i < FP_lines_nb; i++) {
        sum += dm[i];
    }
    fp_d_mean = sum / FP_lines_nb;
    
    espdr_msg("fp_d_mean: %f, sum: %f, FP_lines_nb: %d", fp_d_mean, sum, FP_lines_nb);
    
    sprintf(comment, "[A] weighted mean D from FP lines");
    double fp_d_mean_rounded = round(fp_d_mean * 1000.0) / 1000.0;
    espdr_keyword_add_double(qc_kws->qc_wave_fp_d_mean_kw,
                                        fp_d_mean_rounded, comment, &keywords_RE);
    
    sprintf(comment, "[A] global stat error on D");
    double fp_d_mean_err_rounded = round(fp_d_mean_err * 1000.0) / 1000.0;
    espdr_keyword_add_double(qc_kws->qc_wave_fp_d_mean_error_kw,
                                        fp_d_mean_err_rounded, comment, &keywords_RE);
    
    double *valid_residuals = (double *) cpl_calloc (valid_lines_nb, sizeof(double));
    double *valid_residuals_weights = (double *) cpl_calloc (valid_lines_nb, sizeof(double));
    int index = 0;
    for (int i = 0; i < THAR_lines_nb; i++) {
        if (THAR_lines_table[i].qc_flag == 1) {
            if (residuals[i] < fp_d_residuals_min) {
                fp_d_residuals_min = residuals[i];
            }
            if (residuals[i] > fp_d_residuals_max) {
                fp_d_residuals_max = residuals[i];
            }
            valid_residuals[index] = residuals[i];
            valid_residuals_weights[index] = 1.0/(d_computed_err[i] * d_computed_err[i]);
            index++;
        }
    }
    fp_d_residuals_rms = espdr_wrms(valid_residuals, valid_residuals_weights, valid_lines_nb);
    
    sprintf(comment, "[A] RMS of D resid around S-G fit");
    double fp_d_residuals_rms_rounded = round(fp_d_residuals_rms * 1000.0) / 1000.0;
    espdr_keyword_add_double(qc_kws->qc_wave_fp_d_residuals_rms_kw,
                                        fp_d_residuals_rms_rounded, comment, &keywords_RE);
    
    double fp_d_residuals_min_rounded = round(fp_d_residuals_min * 1000.0) / 1000.0;
    sprintf(comment, "[A] MIN of D resid around S-G fit");
    espdr_keyword_add_double(qc_kws->qc_wave_fp_d_residuals_min_kw,
                                        fp_d_residuals_min_rounded, comment, &keywords_RE);
    
    double fp_d_residuals_max_rounded = round(fp_d_residuals_max * 1000.0) / 1000.0;
    sprintf(comment, "[A] MAX of D resid around S-G fit");
    espdr_keyword_add_double(qc_kws->qc_wave_fp_d_residuals_max_kw,
                                        fp_d_residuals_max_rounded, comment, &keywords_RE);
    
    
    double fp_d_zone_mean[4], fp_d_zone_mean_err[4], fp_d_zone_wave[4];
    double ll_span = (ll_last - ll_first) / 4.0;
    //espdr_msg("ll_first: %f, ll_last: %f, ll_span: %f",
    //          ll_first, ll_last, ll_span);
    
    double sum_chrom[4], weights_chrom[4], ll_sum_chrom[4];
    for (int i = 0 ; i < 4; i++) {
        sum_chrom[i] = 0.0;
        weights_chrom[i] = 0.0;
        ll_sum_chrom[i] = 0.0;
    }
    for (int i = 0; i < THAR_lines_nb; i++) {
        if (THAR_lines_table[i].qc_flag == 1) {
            if (d_computed_err[i] != 0.0) {
                if (THAR_lines_table[i].wavelength < ll_first + ll_span) {
                    // ZONE 1
                    weights_chrom[0] += 1.0/(d_computed_err[i] * d_computed_err[i]);
                 } else {
                     if (THAR_lines_table[i].wavelength < ll_first + 2*ll_span) {
                         // ZONE 2
                         weights_chrom[1] += 1.0/(d_computed_err[i] * d_computed_err[i]);
                     } else {
                         if (THAR_lines_table[i].wavelength < ll_first + 3*ll_span) {
                             // ZONE 3
                             weights_chrom[2] += 1.0/(d_computed_err[i] * d_computed_err[i]);
                        } else {
                             // ZONE 4
                             weights_chrom[3] += 1.0/(d_computed_err[i] * d_computed_err[i]);
                         }
                     }
                 }
            } else {
                espdr_msg_warning("Incoherence: THAR line qc is 1, but d_err is 0.0");
            }
        }
    }
    
    double FP_zone_lines_nb[4];
    for (int i = 0; i < 4; i++) {
        fp_d_zone_mean_err[i] = 0.0;
        if (weights_chrom[i] != 0.0) {
            fp_d_zone_mean_err[i] = 1.0 / sqrt(weights_chrom[i]);
        }
        FP_zone_lines_nb[i] = 0;
    }
    
    for (int i = 0; i < FP_lines_nb; i++) {
        if (FP_lines_table[i].wavelength < ll_first + ll_span) {
            // ZONE 1
            sum_chrom[0] += dm[i];
            ll_sum_chrom[0] += FP_lines_table[i].wavelength;
            FP_zone_lines_nb[0]++;
        } else {
            if (FP_lines_table[i].wavelength < ll_first + 2*ll_span) {
                // ZONE 2
                sum_chrom[1] += dm[i];
                ll_sum_chrom[1] += FP_lines_table[i].wavelength;
                FP_zone_lines_nb[1]++;
            } else {
                if (FP_lines_table[i].wavelength < ll_first + 3*ll_span) {
                    // ZONE 3
                    sum_chrom[2] += dm[i];
                    ll_sum_chrom[2] += FP_lines_table[i].wavelength;
                    FP_zone_lines_nb[2]++;
                } else {
                    // ZONE 4
                    sum_chrom[3] += dm[i];
                    ll_sum_chrom[3] += FP_lines_table[i].wavelength;
                    FP_zone_lines_nb[3]++;
                }
            }
        }
    }
    
    for (int i = 0; i < 4; i++) {
        fp_d_zone_mean[i] = sum_chrom[i] / FP_zone_lines_nb[i];
        fp_d_zone_wave[i] = ll_sum_chrom[i] / FP_zone_lines_nb[i];
    }
    
    double rounded_value = 0.0;
    for (int i = 0; i < 4; i++) {
        new_keyword = espdr_add_index_to_keyword(qc_kws->qc_wave_fp_d_zone_kw_first,
                                                 qc_kws->qc_wave_fp_d_zone_mean_kw_last,
                                                 i+1);
        sprintf(comment, "[A] weighted mean D in zone %d", i+1);
        rounded_value = round(fp_d_zone_mean[i] * 1000.0) / 1000.0;
        espdr_keyword_add_double(new_keyword, rounded_value, comment, &keywords_RE);
        cpl_free(new_keyword);
        
        new_keyword = espdr_add_index_to_keyword(qc_kws->qc_wave_fp_d_zone_kw_first,
                                                 qc_kws->qc_wave_fp_d_zone_mean_error_kw_last,
                                                 i+1);
        sprintf(comment, "[A] error on D in zone %d", i+1);
        rounded_value = round(fp_d_zone_mean_err[i] * 1000.0) / 1000.0;
        espdr_keyword_add_double(new_keyword, rounded_value, comment, &keywords_RE);
        cpl_free(new_keyword);
        
        new_keyword = espdr_add_index_to_keyword(qc_kws->qc_wave_fp_d_zone_kw_first,
                                                 qc_kws->qc_wave_fp_d_zone_wave_kw_last,
                                                 i+1);
        sprintf(comment, "[A] central wavelength of zone %d", i+1);
        rounded_value = round(fp_d_zone_wave[i] * 1000.0) / 1000.0;
        espdr_keyword_add_double(new_keyword, rounded_value, comment, &keywords_RE);
        cpl_free(new_keyword);
    }
    
    return(cpl_error_get_code());
}



