/*-----------------------------------------------------------------*/
/*! 
  \file inspector.c 
  \brief display details about the ephemeris file.

  \author  M. Gastineau 
           Astronomie et Systemes Dynamiques, IMCCE, CNRS, Observatoire de Paris. 

   Copyright, 2011, 2012, 2013, 2014, 2015, 2016, CNRS
   email of the author : gastineau@imcce.fr

  History:                                                                
*/
/*-----------------------------------------------------------------*/

/*-----------------------------------------------------------------*/
/* License  of this file :
 This file is "triple-licensed", you have to choose one  of the three licenses 
 below to apply on this file.
 
    CeCILL-C
    	The CeCILL-C license is close to the GNU LGPL.
    	( http://www.cecill.info/licences/Licence_CeCILL-C_V1-en.html )
   
 or CeCILL-B
        The CeCILL-B license is close to the BSD.
        (http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.txt)
  
 or CeCILL v2.0
      The CeCILL license is compatible with the GNU GPL.
      ( http://www.cecill.info/licences/Licence_CeCILL_V2-en.html )
 

This library is governed by the CeCILL-C, CeCILL-B or the CeCILL license under 
French law and abiding by the rules of distribution of free software.  
You can  use, modify and/ or redistribute the software under the terms 
of the CeCILL-C,CeCILL-B or CeCILL license as circulated by CEA, CNRS and INRIA  
at the following URL "http://www.cecill.info". 

As a counterpart to the access to the source code and  rights to copy,
modify and redistribute granted by the license, users are provided only
with a limited warranty  and the software's author,  the holder of the
economic rights,  and the successive licensors  have only  limited
liability. 

In this respect, the user's attention is drawn to the risks associated
with loading,  using,  modifying and/or developing or reproducing the
software by the user in light of its specific status of free software,
that may mean  that it is complicated to manipulate,  and  that  also
therefore means  that it is reserved for developers  and  experienced
professionals having in-depth computer knowledge. Users are therefore
encouraged to load and test the software's suitability as regards their
requirements in conditions enabling the security of their systems and/or 
data to be ensured and,  more generally, to use and operate it in the 
same conditions as regards security. 

The fact that you are presently reading this means that you have had
knowledge of the CeCILL-C,CeCILL-B or CeCILL license and that you accept its terms.
*/
/*-----------------------------------------------------------------*/

#include "calcephconfig.h"
#if HAVE_STDIO_H
#include <stdio.h>
#endif
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#if HAVE_ERRNO_H
#include <errno.h>
#endif
#if HAVE_MATH_H
#include <math.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifndef __cplusplus
#if HAVE_STDBOOL_H
#include <stdbool.h>
#endif
#endif /*__cplusplus */

#include "real.h"
#include "calceph.h"
#include "calcephinternal.h"
#include "util.h"

int main(int argc, char *argv[])
{
    t_calcephbin *peph;
    const char *filename;
    struct calcephbin_inpop *pinpop;
    struct calcephbin_spice *pspice;
    struct SPICEkernel *pkernel;
    struct SPKfile *pspk;
    struct TXTPCKfile *ptxtpck;
    struct TXTPCKconstant *listconstant;
    struct TXTFKfile *ptxtfk;
    struct TXTFKframe *listframe;
    struct SPKSegmentList * listseg;
    char nameconstant[CALCEPH_MAX_CONSTANTNAME];
    double valueconstant;
    int j;

    if (argc != 2)
    {
        printf("Usage : calceph_inspector filename\n");
        printf("                          filename : binary ephemeris file name\n\n");
        printf("e.g.  : calceph_inspector inpop08a_m100_p100_littleendian.dat\n\n");
        return 1;
    }
    printf("calceph_inspector -  CALCEPH version %d.%d.%d\n\n",
           CALCEPH_VERSION_MAJOR, CALCEPH_VERSION_MINOR, CALCEPH_VERSION_PATCH);
    filename = argv[1];
    peph = calceph_open(filename);
    if (peph)
    {

        switch (peph->etype)
        {
            case CALCEPH_espice:
                printf("SPICE file format\n");
                pspice = &peph->data.spkernel;
                /*if (pspice==NULL) printf("peph->data.spkernel is NULL\n"); */
                pkernel = pspice->list;
                while (pkernel != NULL)
                {
                    switch (pkernel->filetype)
                    {
                        case DAF_SPK:
                            printf("SPK file\n");
                            pspk = &pkernel->filedata.spk;
                            printf("Binary file format ID : '%.*s'\n", IFNAME_LEN, pspk->header.bff);
                            printf
                                ("body        center      start time (JD) end time (JD)     frame  data type    data information\n");
                            printf
                                ("------------------------------------------------------------------------------------------------------\n");
                            for (listseg = pspk->list_seg; listseg != NULL;
                                 listseg = listseg->next)
                            {
                                for (j = 0; j < listseg->array_seg.count; j++)
                                {
                                    struct SPKSegmentHeader *seg = listseg->array_seg.array + j;

                                    printf("%-12d %-12d%-16.2f%-16.2f %-10d%-12d", seg->body, seg->center,
                                           seg->T_begin / 86400E0 + 2.451545E+06, seg->T_end / 86400E0 + 2.451545E+06,
                                           seg->refframe, seg->datatype);
                                    switch (seg->datatype)
                                    {
                                        case SPK_SEGTYPE1:
                                            printf("MDA number of records : %6d  TDB\n", seg->seginfo.data1.count_record);
                                            break;

                                        case SPK_SEGTYPE3:
                                        case SPK_SEGTYPE2:
                                            printf("Time span per record: %4.2g (day)\tnumber of records : %6d  TDB\n",
                                                   seg->seginfo.data2.T_len / 86400E0, seg->seginfo.data2.count_record);
                                            break;
                                            
                                        case SPK_SEGTYPE103:
                                        case SPK_SEGTYPE102:
                                            printf("Time span per record: %4.2g (day)\tnumber of records : %6d  TCB\n",
                                                   seg->seginfo.data2.T_len / 86400E0, seg->seginfo.data2.count_record);
                                            break;

                                        case SPK_SEGTYPE12:
                                            printf("Time span per record: %4.2g (day)\tnumber of records : %6d\tHermite Interpolation of degree %d - Equal Time Steps  TDB\n", 
                                                    seg->seginfo.data12.step_size / 86400E0, seg->seginfo.data12.count_record, seg->seginfo.data12.degree);
                                            break;

                                        case SPK_SEGTYPE13:
                                            printf("number of records : %6d\tHermite Interpolation of degree %d - Unequal Time Steps  TDB\n", 
                                                    seg->seginfo.data13.count_record, seg->seginfo.data13.degree);
                                            break;

                                        case SPK_SEGTYPE20:
                                           printf("Time span per record: %4.2g (day)\tnumber of records : %6d  TDB Dscale=%g Tscale=%g\n",
                                                   seg->seginfo.data20.T_len / 86400E0, seg->seginfo.data20.count_record,
                                                   seg->seginfo.data20.dscale,  seg->seginfo.data20.tscale);
                                            break;
                                            
                                        case SPK_SEGTYPE120:
                                           printf("Time span per record: %4.2g (day)\tnumber of records : %6d  TCB Dscale=%g Tscale=%g\n",
                                                   seg->seginfo.data20.T_len / 86400E0, seg->seginfo.data20.count_record,
                                                   seg->seginfo.data20.dscale,  seg->seginfo.data20.tscale);
                                            break;

                                       default:
                                            printf("Unsupported segment (type=%d).", (int) seg->datatype);
                                            break;
                                    }
                                }
                                printf("\n");
                            }
                            break;

                        case TXT_PCK:
                            printf("text PCK file\n");
                            ptxtpck = &pkernel->filedata.txtpck;
                            printf("list of constant of the text PCK file : \n");
                            for (listconstant = ptxtpck->listconstant; listconstant != NULL;
                                 listconstant = listconstant->next)
                            {
                                struct TXTPCKvalue *listvalue;

                                printf("'%s' = ( ", listconstant->name);
                                for (listvalue = listconstant->value; listvalue != NULL; listvalue = listvalue->next)
                                {
                                    printf("%.*s", (int) (listvalue->loclast - listvalue->locfirst),
                                           ptxtpck->buffer + listvalue->locfirst);
                                    if (listvalue->next != NULL)
                                        printf(" , ");
                                }
                                printf(" ) \n");
                            }
                            break;

                        case DAF_PCK:
                            printf("binary PCK file\n");
                            pspk = &pkernel->filedata.spk;
                            printf("Binary file format ID : '%.*s'\n", IFNAME_LEN, pspk->header.bff);
                            printf("body     start time (JD)  end time (JD)    frame  data type   data information\n");
                            printf
                                ("-------------------------------------------------------------------------------------\n");
                            for (listseg = pspk->list_seg; listseg != NULL;
                                 listseg = listseg->next)
                            {
                                for (j = 0; j < listseg->array_seg.count; j++)
                                {
                                    struct SPKSegmentHeader *seg = listseg->array_seg.array + j;

                                    printf("%-8d %-16.2f %-16.2f %-7d%-12d", seg->body,
                                           seg->T_begin / 86400E0 + 2.451545E+06, seg->T_end / 86400E0 + 2.451545E+06,
                                           seg->refframe, seg->datatype);
                                    switch (seg->datatype)
                                    {
                                        case SPK_SEGTYPE3:
                                        case SPK_SEGTYPE2:
                                            printf("Time span per record: %4.2g (day)  number of records : %6d  TDB\n",
                                                   seg->seginfo.data2.T_len / 86400E0, seg->seginfo.data2.count_record);
                                            break;

                                        case SPK_SEGTYPE103:
                                        case SPK_SEGTYPE102:
                                            printf("Time span per record: %4.2g (day)  number of records : %6d  TCB\n",
                                                   seg->seginfo.data2.T_len / 86400E0, seg->seginfo.data2.count_record);
                                            break;

                                        case SPK_SEGTYPE20:
                                           printf("Time span per record: %4.2g (day)\tnumber of records : %6d  TDB Dscale=%g Tscale=%g\n",
                                                   seg->seginfo.data20.T_len /*/ 86400E0*/, seg->seginfo.data20.count_record,
                                                   seg->seginfo.data20.dscale,  seg->seginfo.data20.tscale);
                                            break;

                                        case SPK_SEGTYPE120:
                                           printf("Time span per record: %4.2g (day)\tnumber of records : %6d  TCB Dscale=%g Tscale=%g\n",
                                                   seg->seginfo.data20.T_len /*/ 86400E0*/, seg->seginfo.data20.count_record,
                                                   seg->seginfo.data20.dscale,  seg->seginfo.data20.tscale);
                                            break;

                                        default:
                                            printf("Unsupported segment (type=%d).", (int) seg->datatype);
                                            break;
                                    }
                                }
                                printf("\n");
                            }
                            break;

                        case TXT_FK:
                            printf("text FK file\n");
                            ptxtfk = &pkernel->filedata.txtfk;
                            printf("list of frame of the text FK file : \n");
                            for (listframe = ptxtfk->listframe; listframe != NULL; listframe = listframe->next)
                            {
                                printf("'%s' = ( %d, class=%2d, classid=%8d, center = %8d ",
                                       listframe->name,
                                       listframe->frame_id,
                                       listframe->classtype, listframe->class_id, listframe->center);

                                printf(" ) \n");
                            }
                            printf("list of constant of the text FK file : \n");
                            for (listconstant = ptxtfk->txtpckfile.listconstant; listconstant != NULL;
                                 listconstant = listconstant->next)
                            {
                                struct TXTPCKvalue *listvalue;

                                printf("'%s' = ( ", listconstant->name);
                                for (listvalue = listconstant->value; listvalue != NULL; listvalue = listvalue->next)
                                {
                                    printf("%.*s", (int) (listvalue->loclast - listvalue->locfirst),
                                           ptxtfk->txtpckfile.buffer + listvalue->locfirst);
                                    if (listvalue->next != NULL)
                                        printf(" , ");
                                }
                                printf(" ) \n");
                            }
                            break;

                        default:
                            fatalerror("Unknown SPICE type in %d\n", (int) pkernel->filetype);
                            break;
                    }
                    pkernel = pkernel->next;
                }
                break;

            case CALCEPH_ebinary:
                printf("INPOP or original DE file format\n");
                pinpop = &peph->data.binary;
                printf("Ephemerids number : %d\n", pinpop->H1.DENUM);
                printf("Start time : %23.16e\n", pinpop->H1.timeData[0]);
                printf("End time : %23.16e\n", pinpop->H1.timeData[1]);
                printf("Time Span : %23.16e\n", pinpop->H1.timeData[2]);
                printf("list of constants :\n");
                for (j=1; j<=calceph_getconstantcount(peph); j++)
                {
                    calceph_getconstantindex(peph, j, nameconstant, &valueconstant);
                    printf("'%s'\t= %23.16E\n", nameconstant, valueconstant);
                }

                break;

            default:
                fatalerror("Unknown ephemeris type in calceph_getconstant\n");
                break;
        }

        calceph_close(peph);
    }
    return 0;
}
