/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2024-01-24 16:52:15 $
 * $Revision: 1.6 $
 * $Name: not supported by cvs2svn $
 */

#ifndef ESPDR_OH_CORRECTION_H
#define ESPDR_OH_CORRECTION_H

/*----------------------------------------------------------------------------
 Includes
 ----------------------------------------------------------------------------*/
/* DRL functions*/
#include <espdr_instrument.h>
#include <espdr_msg.h>
#include <espdr_CCD.h>
#include <espdr_dfs.h>
#include <espdr_pixels.h>
#include <espdr_fit.h>
#include <espdr_utils.h>
#include <espdr_science.h>
#include <espdr_ccf.h>

/* Library */
#include <cpl.h>
#include <assert.h>
#include <gsl/gsl_multimin.h>
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_vector.h>
#include <gsl/gsl_blas.h>
#include <gsl/gsl_multifit_nlinear.h>
#include <time.h>

/*----------------------------------------------------------------------------
 Constants
 ----------------------------------------------------------------------------*/

#define RUNNING_MEDIAN_HALF_WINDOW 50.0
#define MAX_NUMBER_OH_LINES 1000
#define MAD_FACTOR 6.0
#define SKY_MASTER_TEXP 601.9164

/*----------------------------------------------------------------------------
 Data structures
 ----------------------------------------------------------------------------*/

typedef struct {
    int order;
    double wave_start_A;
    double region_size_A;
    double wave_start_B;
    double region_size_B;
    int both_fibres;
    double integrated_flux;
    char comment[256];
} espdr_OH_line;

/* Columns of the LINES_TABLE */
static const char COL_NAME_OH_ORDER[] = "Order";
static const char COL_NAME_WAVE_START_A[] = "Wavelength Start A";
static const char COL_NAME_REGION_SIZE_A[] = "Region Size A";
static const char COL_NAME_WAVE_START_B[] = "Wavelength Start B";
static const char COL_NAME_REGION_SIZE_B[] = "Region Size B";
static const char COL_NAME_BOTH_FIBRES[] = "Both Fibers";
static const char COL_NAME_INTEGRATED_FLUX[] = "Integrated Flux B";
static const char COL_NAME_OH_COMMENT[] = "Comments";

/*----------------------------------------------------------------------------
 Functions prototypes
 ----------------------------------------------------------------------------*/

cpl_error_code espdr_correct_OH_lines(const cpl_frameset* set,
                                      cpl_frameset* used_frames,
                                      espdr_inst_config *inst_config,
                                      espdr_qc_keywords *qc_kws,
                                      double Texp,
                                      double airmass_mean,
                                      cpl_image *s2d_flux_A,
                                      cpl_image *s2d_error_A,
                                      cpl_image *s2d_qual_A,
                                      cpl_image *s2d_flux_B,
                                      cpl_image *s2d_error_B,
                                      cpl_image *s2d_qual_B,
                                      cpl_image *wave_matrix_A,
                                      cpl_image *wave_matrix_B,
                                      cpl_image **OH_corr_flux_RE,
                                      cpl_image **OH_corr_error_RE,
                                      cpl_image **OH_corr_qual_RE,
                                      cpl_propertylist **OH_keywords_RE);

cpl_error_code espdr_compute_running_median_mad(cpl_image *input_img,
                                                double *input_idx,
                                                int input_size,
                                                int half_window_size,
                                                int mad_flag, // 0 = median, 1 = MAD
                                                double *output);

cpl_error_code espdr_OH_lines_table_init(espdr_OH_line *lines_table,
                                         int lines_nb);

cpl_error_code espdr_read_OH_LINES_TABLE(cpl_frame *OH_lines_regions_frame,
                                         espdr_OH_line *OH_lines_table,
                                         int *lines_nb_RE);

cpl_error_code espdr_read_OH_LINES_TABLE_CSV(cpl_frame *OH_lines_regions_frame,
                                             espdr_OH_line *OH_lines_table,
                                             int *lines_nb_RE);

cpl_error_code espdr_save_OH_lines_table(espdr_OH_line *OH_lines_table,
                                         int OH_lines_nb);

cpl_error_code espdr_OH_corr_QC(espdr_inst_config *inst_config,
                                espdr_qc_keywords *qc_kws,
                                double Texp,
                                double airmass_mean,
                                int OH_lines_nb_used,
                                double median_ratio,
                                double flux_ratio_RMS,
                                double flux_ratio_MAD,
                                cpl_propertylist **OH_keywords_RE);

cpl_error_code espdr_propagate_OH_QC(espdr_qc_keywords *qc_kws,
                                     cpl_propertylist *OH_QC_keywords,
                                     cpl_propertylist **keywords_RE);

#endif /* ESPDR_OH_CORRECTION_H */
