/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: asegovia $
 * $Date: 2014-10-28 15:13:41 $
 * $Revision:     $
 * $Name: not supported by cvs2svn $
 */


#include <espdr_flat.h>
#include <espdr_overscan.h>
#include <espdr_instrument.h>
#include <espdr_background.h>
#include <espdr_blaze.h>
#include <espdr_wave_cal.h>
#include <espdr_detector_signature.h>

#define KSIG_EXTRACTION_MIN -2.0
#define KSIG_EXTRACTION_MAX 20.0
/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    parameters     the parameters list
 @param    frameset   the frames list

 In case of failure the cpl_error_code is set.
 */
/*---------------------------------------------------------------------------*/

int espdr_mflat(cpl_parameterlist *parameters, cpl_frameset *frameset,
		const char* recipe_id) {

	int i = 0, j = 0;

	cpl_frameset *used_frames = NULL;

	cpl_error_code my_error = CPL_ERROR_NONE;

    const int rec_ntags = 14;
    const char* rec_tags[14] = {ESPDR_FLAT_A_RAW, ESPDR_FLAT_B_RAW, ESPDR_FLAT_RAW,
        ESPDR_PRO_CATG_HOT_PIXELS, ESPDR_PRO_CATG_MDARK,
        ESPDR_PRO_CATG_BAD_PIXELS, ESPDR_CCD_GEOM,
        ESPDR_PRO_CATG_ORDERS_A, ESPDR_PRO_CATG_ORDERS_B,
        ESPDR_PRO_CATG_STATIC_WAVE_MATRIX_A, ESPDR_PRO_CATG_STATIC_WAVE_MATRIX_B,
        ESPDR_PRO_CATG_STATIC_DLL_MATRIX_A, ESPDR_PRO_CATG_STATIC_DLL_MATRIX_B,
        ESPDR_ORDERS_MASK,
    };
    int is_required[14] = {0, 0, 0, 1, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0};

	cpl_msg_set_level(CPL_MSG_INFO);
	//cpl_msg_set_level(CPL_MSG_DEBUG);

	espdr_msg("Starting flat");

    espdr_ensure(espdr_check_input_tags(frameset,
                                        rec_tags, is_required,
                                        rec_ntags) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_inst_config(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");

    /* Identify the RAW and CALIB frames in the input frameset */
    espdr_ensure(espdr_dfs_set_groups(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(),
                 "DFS setting groups failed. Expected inputs are:\n"
                 "raw flat images for fibre A and B, tagged as FLAT_A or FLAT_B\n"
                 "CCD geometry table, tagged as CCD_GEOM\n"
                 "instrument configuration table, tagged as MASTER_INST_CONFIG or INST_CONFIG\n"
                 "hot pixel mask image, tagged as HOT_PIXEL_MASK\n"
                 "bad pixel mask image, tagged as BAD_PIXEL_MASK\n"
                 "orders definition images for fibres A and B, tagged as ORDER_TABLE_A & ORDER_TABLE_B\n"
                 "static wave matrixes for fibres A and B, tagged STATIC_WAVE_MATRIX_A/B");

    espdr_CCD_geometry *CCD_geom = NULL;
    espdr_inst_config *inst_config = NULL;
    espdr_qc_keywords *qc_kws = NULL;
    espdr_OVSC_param *OVSC_param = NULL;
    espdr_FLAT_param *FLAT_param = NULL;
    cpl_frame* CCD_geom_frame = NULL;
    cpl_frame* inst_config_frame = NULL;

    /* AMO Changed way to read CCD geom frame and to init relevant recipe
     * internal parameters structures to simplify this calling function
     */
    CCD_geom_frame    = espdr_frame_find(frameset, ESPDR_CCD_GEOM);
    espdr_ensure(CCD_geom_frame == NULL,cpl_error_get_code(),
                 "CCD geometry frame not found!");
    CCD_geom          = espdr_CCD_geom_init(parameters, CCD_geom_frame);
    inst_config_frame = espdr_get_inst_config(frameset);
    espdr_ensure(inst_config_frame == NULL,cpl_error_get_code(),
                 "Instrument config frame not found!");
    inst_config       = espdr_inst_config_init(parameters,
                                               CCD_geom->ext_nb,
                                               inst_config_frame);
    OVSC_param        = espdr_parameters_OVSC_init(recipe_id, parameters);
    FLAT_param        = espdr_FLAT_param_init(recipe_id, parameters);

    used_frames = cpl_frameset_new();
    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(CCD_geom_frame));
    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(inst_config_frame));

    /* Calculating the total orders number */
    int *orders_nb_per_fibre = NULL;
    orders_nb_per_fibre = (int *)cpl_calloc(inst_config->fibres_nb, sizeof(int));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        orders_nb_per_fibre[i] = 0;
        for (j = 0; j < CCD_geom->ext_nb; j++) {
            orders_nb_per_fibre[i] += inst_config->orders_nb[i*CCD_geom->ext_nb+j];
        }
    }

    /* Getting the physical orders id */
    int **physical_orders_id = (int**) cpl_calloc(inst_config->fibres_nb,sizeof(int *));

    for (int i = 0; i < inst_config->fibres_nb; i++) {
        physical_orders_id[i] = (int *)cpl_calloc(orders_nb_per_fibre[i],sizeof(int));
    }

    for (int fibre = 0; fibre < inst_config->fibres_nb; fibre++) {
        espdr_get_order_phys_numbers(CCD_geom,
                                     inst_config,
                                     fibre,
                                     physical_orders_id[fibre]);
    }

    // Background subtraction parameter managemenet
    char background_sw[10];
    sprintf(background_sw,"%s","on");

    cpl_parameter *par_background_sw =
    cpl_parameterlist_find(parameters,"espdr.espdr_mflat.background_sw");
    int background_sw_set = cpl_parameter_get_default_flag(par_background_sw);
    if (background_sw_set == 1) {
        sprintf(background_sw, "%s", FLAT_param->background_sw);
        cpl_parameter_set_string(par_background_sw, FLAT_param->background_sw);
    }
    espdr_msg("Background SW is %s", background_sw);


	/* Produce the master flat */
	my_error = espdr_process_flat(frameset,
                                  CCD_geom,
                                  inst_config,
                                  physical_orders_id,
                                  used_frames,
                                  0, // flag to remove or not the master bias residuals
                                  background_sw,
                                  parameters);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_process_flat failed: %s",
                 cpl_error_get_message_default(my_error));

    for (i = 0; i < inst_config->fibres_nb; i++) {
        cpl_free(physical_orders_id[i]);
    }
    cpl_free(physical_orders_id);
    cpl_free(orders_nb_per_fibre);

    espdr_parameters_OVSC_delete(OVSC_param);
    espdr_parameters_FLAT_delete(FLAT_param);
    espdr_parameters_CCD_geometry_delete(CCD_geom);
    espdr_parameters_inst_config_delete(inst_config);
    cpl_free(qc_kws);

    cpl_frameset_delete(used_frames);

	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create the structure for FLAT input parameters
 @param    recipe_id	recipe identifier
 @param    list			list of input parameters
 @param    p			input FLAT parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_FLAT_create(const char* recipe_id,
                                            cpl_parameterlist *list,
                                            espdr_FLAT_param *p) {
    
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, 
		     "The parameters list NULL");

	/* Fill the parameter list */
	char comment[COMMENT_LENGTH];
	double ksigma_min = KSIGMA_MIN;
	double ksigma_max = KSIGMA_MAX;
	double ksig_extraction_min = KSIG_EXTRACTION_MIN;
	double ksig_extraction_max = KSIG_EXTRACTION_MAX;
	int bkgr_x_min = BKGR_X_MIN;
	int bkgr_x_max = BKGR_X_MAX;
	int bkgr_y_min = BKGR_Y_MIN;
	int bkgr_y_max = BKGR_Y_MAX;


	/* check parameters */
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT,
			"The parameters list NULL");

	/* Fill the parameter list */
    cpl_error_code error_got;
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "flat_sig_clip_method",
                                            p->flat_sigma_clipping_method,
                                            "method for sigma clipping in master FLAT, can be: mean or median");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter flat_sigma_clipping_method to the list");
    
    sprintf(comment,
            "ksigma for sigma clipping in MFLAT, must be between: %.2lf and %.2lf",
            ksigma_min, ksigma_max);
    error_got = espdr_parameters_new_range_double(recipe_id, list,
                                                  "flat_ksigma", p->flat_ksigma,
                                                  KSIGMA_MIN, KSIGMA_MAX,
                                                  comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter flat_ksigma to the list");
    
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "background_sw",
                                            p->background_sw,
                                            "Background subtraction activation (on/off)");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter background_sw to the list");
    
    sprintf(comment,
            "Grid size in x used to calculate the background, between: %d and %d",
            bkgr_x_min,bkgr_x_max);
    error_got = espdr_parameters_new_range_int(recipe_id, list,
                                               "bkgr_grid_size_x",
                                               p->bkgr_grid_size_x,
                                               bkgr_x_min, bkgr_x_max,
                                               comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter bkgr_grid_size_x to the list");
    
    sprintf(comment,
            "Grid size in y used to calculate the background, between: %d and %d",
            bkgr_y_min,bkgr_y_max);
    error_got = espdr_parameters_new_range_int(recipe_id, list,
                                               "bkgr_grid_size_y",
                                               p->bkgr_grid_size_y,
                                               bkgr_y_min, bkgr_y_max,
                                               comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter bkgr_grid_size_y to the list");
    
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "extraction_method",
                                            p->extraction_method,
                                            "Method used to extract orders");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter extraction_method to the list");
    
    sprintf(comment, "ksigma for extraction, must be between: %.2lf and %.2lf",
            ksig_extraction_min, ksig_extraction_max);
    error_got = espdr_parameters_new_range_double(recipe_id, list,
                                                  "extraction_ksigma",
                                                  p->ksig_extraction,
                                                  KSIG_EXTRACTION_MIN,
                                                  KSIG_EXTRACTION_MAX,
                                                  comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter ksig_extraction to the list");
    
	return (CPL_ERROR_NONE);
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the structure for FLAT input parameters
 @param    p	input FLAT parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_FLAT_delete(espdr_FLAT_param* p) {
	
    cpl_free((void *)p->background_sw);
    cpl_free((void *)p->extraction_method);
    cpl_free((void *)p->flat_sigma_clipping_method);
    cpl_free(p);
    p = NULL;
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Get the mflat recipe parameters
 @param     recipe_id   recipe ID
 @param     param_list  parameters list
 @param     FLAT_param  FLAT parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_FLAT_get(const char* recipe_id,
                                         cpl_parameterlist* param_list,
                                         espdr_FLAT_param *FLAT_param) {
	
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(param_list == NULL, CPL_ERROR_NULL_INPUT,
				 "Parameters list is NULL");
	
	/* Fill the structure */
    FLAT_param->flat_sigma_clipping_method = espdr_parameters_get_string(recipe_id,
                                                                         param_list,
                                                                         "flat_sig_clip_method");
    
    FLAT_param->flat_ksigma = espdr_parameters_get_double(recipe_id,
                                                          param_list,
                                                          "flat_ksigma");
    
    FLAT_param->background_sw = espdr_parameters_get_string(recipe_id,
                                                            param_list,
                                                            "background_sw");
    
    FLAT_param->bkgr_grid_size_x = espdr_parameters_get_int(recipe_id,
                                                            param_list,
                                                            "bkgr_grid_size_x");
    
    FLAT_param->bkgr_grid_size_y = espdr_parameters_get_int(recipe_id,
                                                            param_list,
                                                            "bkgr_grid_size_y");
    
    FLAT_param->extraction_method = espdr_parameters_get_string(recipe_id,
                                                                param_list,
                                                                "extraction_method");
    
    FLAT_param->ksig_extraction = espdr_parameters_get_double(recipe_id,
                                                              param_list,
                                                              "extraction_ksigma");
    
    
	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     print the FLAT parameters
 @param     FLAT_param  FLAT parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_FLAT_print(espdr_FLAT_param *FLAT_param) {
	
	espdr_msg("\tFLAT parameters:");
    espdr_msg("\t\tFLAT stacking sigma clipping method = %s",
              FLAT_param->flat_sigma_clipping_method);
    espdr_msg("\t\tFLAT stacking ksigma = %.2f",
              FLAT_param->flat_ksigma);
    espdr_msg("\t\tFLAT background subtraction = %s",
              FLAT_param->background_sw);
    espdr_msg("\t\tFLAT background grid size x = %d",
              FLAT_param->bkgr_grid_size_x);
    espdr_msg("\t\tFLAT background grid size y = %d",
              FLAT_param->bkgr_grid_size_y);
    espdr_msg("\t\tFLAT extraction method = %s",
              FLAT_param->extraction_method);
    espdr_msg("\t\tFLAT extraction ksigma = %.2f",
              FLAT_param->ksig_extraction);
    
	return (CPL_ERROR_NONE);
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Init FLAT_param structure
 
 @param    recipe_id    recipe name
 @param    parameters   recipe input parameters
 
 @return   allocated structure iff OK
 */
/*----------------------------------------------------------------------------*/

espdr_FLAT_param * espdr_FLAT_param_init(const char *recipe_id, 
					 cpl_parameterlist *parameters) {
    
    espdr_FLAT_param *FLAT_param =
    (espdr_FLAT_param *)cpl_malloc(sizeof(espdr_FLAT_param));
    
    /* Read the flat parameters */
    espdr_parameters_FLAT_get(recipe_id, parameters, FLAT_param);
    espdr_parameters_FLAT_print(FLAT_param);
    if(cpl_error_get_code() != CPL_ERROR_NONE) {
        return NULL;
    } else {
        return FLAT_param;
    }
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Add extra QC keywords
 @param     keywords    property list to add the extra KWs
 @param     ext_nb      number of extensions
 @return    0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static cpl_error_code espdr_add_extra_qc(cpl_propertylist** keywords,
                                         const int ext_nb) {
    
    const char* ksuff[2] = {" BLUE", " RED"};
    cpl_propertylist_append_int(*keywords,"ESO QC PROC VERSION", 2);
    espdr_add_qc_key_stat_pri(keywords, ext_nb, "MAX FLUX", ksuff, CPL_FALSE,
                              CPL_FALSE, CPL_FALSE, CPL_TRUE, CPL_FALSE);
    int* ord_ref = espdr_get_ord_ref(*keywords, "espdr_mflat");
    espdr_add_qc_key_flat_ord_ref(keywords, ext_nb, ord_ref, "RMS", ksuff,
                                  CPL_TRUE, CPL_TRUE, CPL_TRUE, CPL_TRUE, CPL_FALSE);
    espdr_add_qc_key_stat_ord_pri(keywords, ext_nb, "SNR", "FLAT", CPL_TRUE,
                                  CPL_TRUE, CPL_TRUE, CPL_TRUE, CPL_FALSE);
    cpl_free(ord_ref);
    return cpl_error_get_code();
}



/*---------------------------------------------------------------------------*/
/**
 @brief    Process flat
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/


cpl_error_code espdr_process_flat(cpl_frameset *frameset,
                                  espdr_CCD_geometry *CCD_geom,
                                  espdr_inst_config *inst_config,
                                  int **physical_orders_id,
                                  cpl_frameset *used_frames,
                                  int remove_bias_res,
                                  char *bkgr_sw,
                                  cpl_parameterlist *parameters) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    cpl_frameset **flat_frames = NULL;
    double **max_flux;
    cpl_imagelist *geometry_corrected_saturation_mask = NULL;
    cpl_imagelist *master_flat_iml = NULL;
    cpl_imagelist *master_dark_iml = NULL;
    cpl_imagelist *master_dark_pxl_nb_iml = NULL;
    cpl_imagelist *orders_mask_iml = NULL;
    
    cpl_table ***orders_coeffs = NULL;
    cpl_image **wave_matrix;
    cpl_image **dll_matrix;
    
    const char *input_filename = NULL;
    const char *input_filename_ext = NULL;
    const char *input_filename_HP = NULL;
    const char *input_filename_BP = NULL;
    cpl_propertylist *keywords = NULL;
    cpl_propertylist **keywords_ext = NULL;
    cpl_propertylist *keywords_HP = NULL;
    cpl_propertylist *keywords_BP = NULL;

    double RON_FLAT[inst_config->fibres_nb][CCD_geom->total_output_nb];
    double CONAD_FLAT[inst_config->fibres_nb][CCD_geom->total_output_nb];
    double DARK_FLAT[inst_config->fibres_nb][CCD_geom->total_output_nb];
    double DARK_RON_FLAT[inst_config->fibres_nb][CCD_geom->total_output_nb];
    double exp_time_hour = 0.0;
    double exp_time_hour_mdark = 0.0;

    cpl_image *extracted_spectrum_table = NULL;
    cpl_image *spectrum_error_table = NULL;
    cpl_image *spectrum_quality_table = NULL;
    
    cpl_imagelist *final_master_background_measure_flat = NULL;
    cpl_imagelist *final_master_background_substracted_flat = NULL;
    char *flat_bkgr_measured_frame_filename;
    char *flat_bkgr_substracted_frame_filename;
    char s2d_filename[64];
    char pro_catg_tag[64];
    char flat_name[64];
    char blaze_name[64];
    
    flat_frames = (cpl_frameset **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_frameset *));
    max_flux = (double**)cpl_malloc(inst_config->fibres_nb*sizeof(double *));
    
    /* Extract raw flat frames */
    my_error = espdr_get_raw_flat_frames_from_set(frameset,
                                                  inst_config->fibres_nb,
                                                  inst_config->raw_flat_limit_nb,
                                                  flat_frames,
                                                  used_frames);
    
    espdr_ensure(my_error == CPL_ERROR_ILLEGAL_INPUT, my_error,
                 "Not enough raw FLATs, exiting");
    
    /* Extract hot pixels mask frame */
    cpl_frame* hot_pixels_frame = espdr_get_hpixmap_from_set(frameset, used_frames);
    /* Extract all the outputs from the hot pixels frame */
    cpl_imagelist *hot_pixels_list = cpl_imagelist_new();
    my_error = espdr_extract_real_outputs(hot_pixels_frame, CPL_TYPE_INT,
                                          CCD_geom, 0, &hot_pixels_list);
    

    /* Extract bad pixels mask frame */
    cpl_frame* bad_pixels_frame = espdr_get_bpixmap_from_set(frameset, used_frames);
    /* Extract all the outputs from the bad pixels frame */
    cpl_imagelist *bad_pixels_list = cpl_imagelist_new();
    my_error = espdr_extract_real_outputs(bad_pixels_frame, CPL_TYPE_INT,
                                          CCD_geom, 0, &bad_pixels_list);

    // Update bad pixel mask with all the nans from raw frames
    if (inst_config->inst_type == NIR) {
        for (int i = 0; i < inst_config->fibres_nb; i++) {
            my_error = espdr_update_bad_pixel_mask(flat_frames[i], CCD_geom,
                                                   cpl_imagelist_get(bad_pixels_list, 0));
        }
    }
    
    // Create the complete (HOT, BAD and NaNs) pixels mask
    cpl_imagelist *pixels_mask = cpl_imagelist_new();
    my_error = espdr_create_hot_bad_pixels_mask(hot_pixels_list,
                                                bad_pixels_list,
                                                &pixels_mask);
    
    cpl_imagelist *merged_pixels_mask = cpl_imagelist_new();
    my_error = espdr_image_merge_2_dim(pixels_mask, CCD_geom, CPL_TYPE_INT,
                                       &merged_pixels_mask);
    
    cpl_imagelist *geometry_corrected_pixels_mask = cpl_imagelist_new();
    my_error = espdr_correct_geometry(merged_pixels_mask, CCD_geom,
                                      geometry_corrected_pixels_mask);
    cpl_imagelist_delete(merged_pixels_mask);
    
    /* Extract master dark frame */
    cpl_frame* master_dark_frame = espdr_get_mdark_from_set(frameset,
                                                            used_frames);
    /* Extract all the outputs from the master dark frame if provided */
    if (master_dark_frame != NULL) {
        master_dark_iml = cpl_imagelist_new();
        my_error = espdr_extract_real_outputs(master_dark_frame, CPL_TYPE_DOUBLE,
                                              CCD_geom, 0, &master_dark_iml);
        master_dark_pxl_nb_iml = cpl_imagelist_new();
        my_error = espdr_extract_real_outputs(master_dark_frame, CPL_TYPE_INT,
                                              CCD_geom, CCD_geom->ext_nb,
                                              &master_dark_pxl_nb_iml);
    }
    
    /* Extract orders mask frame */
    cpl_frame* orders_mask_frame = espdr_get_orders_mask_from_set(frameset,
                                                                  used_frames);
    /* Extract all the outputs from the orders mask frame if provided */
    if (orders_mask_frame != NULL) {
        orders_mask_iml = cpl_imagelist_new();
        my_error = espdr_extract_extensions(orders_mask_frame,
                                            CPL_TYPE_INT,
                                            CCD_geom->ext_nb,
                                            &orders_mask_iml);
    }
    
    /* Extract orders coeffs from orders maps frame */
    espdr_get_orders_coeff_from_set(frameset, used_frames,
                                    inst_config->fibres_nb,
                                    CCD_geom->ext_nb,
                                    &orders_coeffs);
    
    /* Extract static wave frame */
    espdr_get_static_wave_matrix_from_set(frameset, used_frames,
                                          inst_config->fibres_nb,
                                          &wave_matrix);
    
    /* Extract static dll frame */
    espdr_get_static_dll_matrix_from_set(frameset, used_frames,
                                         inst_config->fibres_nb,
                                         &dll_matrix);
    
    /* Filling up the DRS QC KWs structure */
    espdr_qc_keywords *qc_kws = (espdr_qc_keywords *)cpl_malloc(sizeof(espdr_qc_keywords));
    my_error = espdr_fill_qc_keywords(inst_config, qc_kws);
    
    /* Load primary header keywords from the hot pixels image */
    if (master_dark_frame != NULL) {
        input_filename_HP = cpl_frame_get_filename(master_dark_frame);
        espdr_msg("KEYWORDS for MASTER DARK input filename: %s",
                  input_filename_HP);
    } else {
        input_filename_HP = cpl_frame_get_filename(hot_pixels_frame);
        espdr_msg("KEYWORDS for HOT PIXELS input filename: %s",
                  input_filename_HP);
    }
    keywords_HP = cpl_propertylist_load(input_filename_HP, 0);
    
    /* Load primary header keywords from the bad pixels image */
    input_filename_BP = cpl_frame_get_filename(bad_pixels_frame);
    espdr_msg("KEYWORDS for BAD PIXELS input filename: %s",
              input_filename_BP);
    keywords_BP = cpl_propertylist_load(input_filename_BP, 0);
    
    cpl_propertylist **product_keywords = (cpl_propertylist **)
                cpl_malloc(inst_config->fibres_nb * sizeof(cpl_propertylist *));
    
    espdr_msg("FLAT processing started ...");
    cpl_frameset* calib_set;

    espdr_frameset_extract_calib(used_frames,&calib_set);

    cpl_frameset* flat_set_for_header = NULL;
    char flat_tag[40];
    for (int i = 0; i < inst_config->fibres_nb; i++) {

        if(i == 0) {
            sprintf(flat_tag,"FLAT_A");
        } else {
            sprintf(flat_tag,"FLAT_B");
        }
        flat_set_for_header = cpl_frameset_new();
        espdr_frame_extract_by_tag(frameset, flat_tag, flat_set_for_header);
        if (cpl_frameset_get_size(flat_set_for_header) == 0) {
            espdr_frame_extract_by_tag(frameset, "FLAT", flat_set_for_header);
        }
        cpl_frameset_join(flat_set_for_header,calib_set);

        long long flat_set_size = cpl_frameset_get_size(flat_frames[i]);
        espdr_msg("FLAT_%c frameset contains %lld frames", fibre_name[i], flat_set_size);
        
        max_flux[i] = (double *) cpl_calloc (CCD_geom->total_output_nb,
                                             sizeof(double));
                                          
        /* Load flat keywords from the first image (primary header) */
        input_filename = cpl_frame_get_filename(cpl_frameset_get_position(flat_frames[i], 0));
        espdr_msg("KEYWORDS for FLAT input filename: %s", input_filename);
        keywords = cpl_propertylist_load(input_filename, 0);
        espdr_ensure(keywords == NULL, CPL_ERROR_ILLEGAL_OUTPUT, "keywords are NULL");
        
        /* Saturation check on first frame */
        my_error = espdr_check_flat_saturation(flat_frames[i],
                                               CCD_geom,
                                               inst_config,
                                               keywords,
                                               pixels_mask,
                                               max_flux[i],
                                               &geometry_corrected_saturation_mask);
        
        product_keywords[i] = cpl_propertylist_duplicate(keywords);
        if (cpl_propertylist_has(product_keywords[i], "OBJECT")) {
            espdr_msg("OBJECT for fibre %c: %s",
                      fibre_name[i], cpl_propertylist_get_string(product_keywords[i], "OBJECT"));
        }
        keywords_ext = (cpl_propertylist**)cpl_malloc(
                                            CCD_geom->ext_nb * sizeof(cpl_propertylist*));
        input_filename_ext = cpl_frame_get_filename(cpl_frameset_get_position
                                                    (flat_frames[i], 0));
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            keywords_ext[j] = cpl_propertylist_load(input_filename_ext, j+1);
        }
        
        int recipes[10] = {1, 1, 1, 1, 0, 0, 0, 0, 0, 0};
        my_error = espdr_compute_calibrations_intervals(frameset, inst_config, product_keywords[i],
                                                        NULL, NULL, NULL, "FLAT", recipes);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_compute_calibrations_intervals failed: %s",
                     cpl_error_get_message_default(my_error));

        my_error = espdr_get_master_flat(flat_frames[i],
                                         master_dark_iml,
                                         orders_mask_iml,
                                         CCD_geom,
                                         inst_config,
                                         product_keywords[i],
                                         keywords_HP,
                                         keywords_BP,
                                         qc_kws,
                                         remove_bias_res,
                                         CONAD_FLAT[i],
                                         DARK_FLAT[i],
                                         DARK_RON_FLAT[i],
                                         RON_FLAT[i],
                                         &master_flat_iml);
        
        /* Getting the sum of the FLUX */
        espdr_msg("Getting the sum of the FLUX, flat frames number = %lld",
                  flat_set_size);
        my_error = cpl_imagelist_multiply_scalar(master_flat_iml,
                                                 flat_set_size);
        
        /* Background subtraction process */
        final_master_background_measure_flat = cpl_imagelist_new();
        final_master_background_substracted_flat = cpl_imagelist_new();
        my_error = espdr_measure_background_ff_by_ext(
                                        master_flat_iml,
                                        CCD_geom,
                                        inst_config->bkgr_grid_size_x,
                                        inst_config->bkgr_grid_size_y,
                                        inst_config->minlevel_factor,
                                        final_master_background_measure_flat,
                                        final_master_background_substracted_flat);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_measure_inter_order_background failed");
        
#if SAVE_DEBUG_PRODUCT_FLAT
        flat_bkgr_measured_frame_filename = cpl_sprintf("%s_%s_%c.fits",
                inst_config->instrument, "background_map", fibre_name[i]);

        sprintf(pro_catg_tag,"FF_BACKGROUND_MAP_%c",fibre_name[i]);
        my_error = cpl_propertylist_update_string(product_keywords[i],
                                                  PRO_CATG_KW, pro_catg_tag);
        my_error = espdr_keyword_add_int(qc_kws->qc_flat_check_kw, 1,
                                         "FLAT global QC", &product_keywords[i]);

        /* Save the background measured fits frame */
        my_error = espdr_dfs_image_save(frameset, parameters,
                                        frameset, "espdr_mflat",
                                        product_keywords[i], keywords_ext,
                                        flat_bkgr_measured_frame_filename,
                                        final_master_background_measure_flat,
                                        CPL_TYPE_FLOAT, CCD_geom);
        cpl_free(flat_bkgr_measured_frame_filename);
#endif
        
        /* Extraction */
        
        int total_number_of_orders = 0;
        for (int ext = 0; ext < CCD_geom->ext_nb; ext++)
            total_number_of_orders += inst_config->orders_nb[CCD_geom->ext_nb * i + ext];
        int *cosmics_nb = (int *)cpl_calloc(total_number_of_orders, sizeof(int));
        
        cpl_imagelist *input_iml = NULL;
        if (strcmp(bkgr_sw, "off") == 0) {
            input_iml = master_flat_iml;
        } else {
            input_iml = final_master_background_substracted_flat;
        }
        
        exp_time_hour = cpl_propertylist_get_double(product_keywords[i],
                                                    inst_config->Texp_kw) / 3600.0;
        
        exp_time_hour_mdark = cpl_propertylist_get_double(keywords_HP,
                                                          inst_config->Texp_kw) / 3600.0;
        
        if(strcmp(inst_config->extraction_method, "horne") == 0) {
            
            my_error = espdr_horne_extraction_one_fibre(
                                        input_iml,
                                        geometry_corrected_pixels_mask,
                                        geometry_corrected_saturation_mask,
                                        NULL, // no CRH mask
                                        final_master_background_measure_flat,
                                        final_master_background_substracted_flat, // order profile
                                        NULL, master_dark_iml, master_dark_pxl_nb_iml,
                                        orders_coeffs[i], i,
                                        inst_config, CCD_geom,
                                        RON_FLAT[i], DARK_FLAT[i], DARK_RON_FLAT[i], CONAD_FLAT[i],
                                        exp_time_hour, exp_time_hour_mdark,
                                        inst_config->extraction_window_size[i],
                                        inst_config->tolerance_rejection,
                                        inst_config->extraction_ksigma_tun,
                                        cosmics_nb,
                                        &extracted_spectrum_table,
                                        &spectrum_error_table,
                                        &spectrum_quality_table);
            
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_horne_extraction failed");
            
        } else if (strcmp(inst_config->extraction_method, "simple") == 0) {
            
            espdr_msg(ANSI_COLOR_RED
                      "Warning : Simple extraction method ..."
                      ANSI_COLOR_RESET);
            
            my_error = espdr_simple_extraction_one_fibre(
                                        input_iml,
                                        geometry_corrected_pixels_mask,
                                        geometry_corrected_saturation_mask,
                                        NULL, // no CRH mask
                                        final_master_background_measure_flat,
                                        master_dark_iml, master_dark_pxl_nb_iml,
                                        orders_coeffs[i], i,
                                        inst_config, CCD_geom,
                                        RON_FLAT[i], DARK_FLAT[i], DARK_RON_FLAT[i], CONAD_FLAT[i],
                                        exp_time_hour, exp_time_hour_mdark,
                                        inst_config->extraction_window_size[i],
                                        &extracted_spectrum_table,
                                        &spectrum_error_table,
                                        &spectrum_quality_table);
            
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_simple_extraction failed: %s",
                         cpl_error_get_message_default(my_error));
            
            espdr_msg("Simple extraction done");
            
        } else {
            
            espdr_msg("Error: %s is an unknown extraction method. Allowed only 'horne' and 'simple'",
                      inst_config->extraction_method);
            exit(EXIT_FAILURE);

        }
        
        
        /* Flux normalization & instrumental correction */
        cpl_image *flux_normalized = cpl_image_new(cpl_image_get_size_x(extracted_spectrum_table),
                                                   cpl_image_get_size_y(extracted_spectrum_table),
                                                   CPL_TYPE_DOUBLE);
        
        cpl_image *error_normalized = cpl_image_new(cpl_image_get_size_x(spectrum_error_table),
                                                    cpl_image_get_size_y(spectrum_error_table),
                                                    CPL_TYPE_DOUBLE);
        
        if ((wave_matrix[i] == NULL) || (dll_matrix[i] == NULL)) { /* NO WAVE_MATRIX or DLL_MATRIX */
            
            espdr_msg(ANSI_COLOR_RED
                      "Warning: espdr_correct_s2d_instrumental not executed"
                      ANSI_COLOR_RESET);
            
            my_error = espdr_s2d_normalisation(extracted_spectrum_table,
                                               spectrum_error_table,
                                               inst_config->blaze_flat_calc_slide_box,
                                               physical_orders_id[i],
                                               &flux_normalized,
                                               &error_normalized);
            
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_s2d_normalisation failed: %s",
                         cpl_error_get_message_default(my_error));
            
        } else {
            
            my_error = espdr_correct_s2d_instrumental(extracted_spectrum_table,
                                                      spectrum_error_table,
                                                      wave_matrix[i],
                                                      dll_matrix[i],
                                                      inst_config->blaze_peaks_fit_slide_box,
                                                      inst_config->blaze_peaks_fit_poly_deg, // 10,
                                                      physical_orders_id[i],
                                                      inst_config->blaze_peaks_fit_method,
                                                      &flux_normalized,
                                                      &error_normalized);
            
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_correct_s2d_instrumental failed: %s",
                         cpl_error_get_message_default(my_error));
            
        }
        
        /* Save the extracted spectrum */
        cpl_image **images_to_save = (cpl_image**)cpl_malloc(3*sizeof(cpl_image*));
        images_to_save[0] = cpl_image_duplicate(extracted_spectrum_table);
        images_to_save[1] = cpl_image_duplicate(spectrum_error_table);
        images_to_save[2] = cpl_image_duplicate(spectrum_quality_table);
        
        sprintf(pro_catg_tag, "%s_%c", ESPDR_PRO_CATG_FS2D, fibre_name[i]);
        my_error = cpl_propertylist_update_string(product_keywords[i],
                                                  PRO_CATG_KW, pro_catg_tag);
        my_error = espdr_keyword_add_int(qc_kws->qc_flat_check_kw, 1,
                                         "FLAT global QC", &product_keywords[i]);

        sprintf(s2d_filename, "%s_spectrum_extracted_%c.fits",
                inst_config->instrument, fibre_name[i]);
        
        // spectrum extracted by Horne,
        // saved temporarily just for checks.
        my_error = espdr_dfs_save_data_err_qual(frameset, parameters,
                                                flat_set_for_header, "espdr_mflat",
                                                product_keywords[i], s2d_filename,
                                                images_to_save);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_dfs_image_save failed: %s for file %s",
                     cpl_error_get_message_default(my_error),
                     s2d_filename);
        
        for (int i = 0; i < 3; i++) {
            cpl_image_delete(images_to_save[i]);
        }
        cpl_free(images_to_save);
        
        long long size_x = cpl_image_get_size_x(extracted_spectrum_table);
        long long size_y = cpl_image_get_size_y(extracted_spectrum_table);
        espdr_msg("Extracted spectrum of fibre %c has size %lldx%lld",
                  fibre_name[i], size_x, size_y);
        
        cpl_image *flat_one_fibre = cpl_image_new(size_x, size_y, CPL_TYPE_DOUBLE);
        cpl_image *blaze_one_fibre = cpl_image_new(size_x, size_y, CPL_TYPE_DOUBLE);

        my_error = espdr_blaze_and_flat_calculation(
                                    flux_normalized,
                                    inst_config->blaze_flat_calc_method,
                                    inst_config->blaze_flat_calc_poly_deg,
                                    inst_config->blaze_flat_calc_slide_box,
                                    inst_config->blaze_flat_calc_clip,
                                    error_normalized,
                                    &flat_one_fibre,
                                    &blaze_one_fibre);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_blaze_and_flat_calculation failed: %s",
                     cpl_error_get_message_default(my_error));
        
        cpl_image_delete(flux_normalized);
        cpl_image_delete(error_normalized);
        
        my_error = espdr_flat_QC(CCD_geom,
                                 inst_config,
                                 qc_kws,
                                 flat_set_size,
                                 max_flux[i],
                                 final_master_background_measure_flat,
                                 cosmics_nb,
                                 total_number_of_orders,
                                 extracted_spectrum_table,
                                 spectrum_error_table,
                                 flat_one_fibre,
                                 &product_keywords[i]);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_flat_QC failed: %s",
                     cpl_error_get_message_default(my_error));
        
        // Saving BLAZE product
        sprintf(blaze_name, "%s_BLAZE_%c.fits",
                inst_config->instrument,fibre_name[i]);
        sprintf(pro_catg_tag, "%s_%c", ESPDR_PRO_CATG_BLAZE, fibre_name[i]);
        my_error = cpl_propertylist_update_string(product_keywords[i], PRO_CATG_KW,
                                                  pro_catg_tag);
        
        my_error = espdr_dfs_image_save_one_ext(frameset, parameters,
                                                flat_set_for_header, "espdr_mflat",
                                                product_keywords[i], blaze_name,
                                                blaze_one_fibre,
                                                CPL_TYPE_FLOAT);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_dfs_image_save failed: %s for file %s",
                     cpl_error_get_message_default(my_error),
                     blaze_name);
        
        cpl_image *spectrum_error_relative = cpl_image_new(size_x, size_y, CPL_TYPE_DOUBLE);
        my_error = espdr_compute_flat_rel_error(extracted_spectrum_table,
                                                spectrum_error_table,
                                                spectrum_error_relative);
        
        // Saving FLAT product
        cpl_image **flat_images_to_save = (cpl_image**)cpl_malloc(3*sizeof(cpl_image*));
        flat_images_to_save[0] = cpl_image_duplicate(flat_one_fibre);
        flat_images_to_save[1] = cpl_image_duplicate(spectrum_error_relative);
        flat_images_to_save[2] = cpl_image_duplicate(spectrum_quality_table);
        
        sprintf(flat_name,"%s_FLAT_%c.fits", inst_config->instrument, fibre_name[i]);
        sprintf(pro_catg_tag,"%s_%c", ESPDR_PRO_CATG_FLAT, fibre_name[i]);
        my_error = cpl_propertylist_update_string(product_keywords[i], PRO_CATG_KW, pro_catg_tag);
        
        if (strcmp(inst_config->instrument, "HARPN") != 0 &&
            (strcmp(inst_config->instrument, "CORALIE") != 0)) {
            my_error = espdr_add_extra_qc(&product_keywords[i], CCD_geom->ext_nb);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_add_extra_qc failed: %s ",
                         cpl_error_get_message_default(my_error));
        }
        
        my_error = espdr_dfs_save_data_err_qual(frameset, parameters,
                                                flat_set_for_header, "espdr_mflat",
                                                product_keywords[i], flat_name,
                                                flat_images_to_save);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_dfs_image_save failed: %s for file %s",
                     cpl_error_get_message_default(my_error),
                     flat_name);
        
        cpl_image_delete(spectrum_error_relative);
        for (int i = 0; i < 3; i++) {
            cpl_image_delete(flat_images_to_save[i]);
        }
        cpl_free(flat_images_to_save);
        
        // Saving ORDER_PROFILE (background subtracted) product
        flat_bkgr_substracted_frame_filename = cpl_sprintf("%s_%c.fits",
                inst_config->master_flat_order_profile_filename_no_fits, fibre_name[i]);
        sprintf(pro_catg_tag,"%s_%c", ESPDR_PRO_CATG_ORDER_PROFILE, fibre_name[i]);
        my_error = cpl_propertylist_update_string(product_keywords[i], PRO_CATG_KW, pro_catg_tag);
        
        my_error = espdr_dfs_image_save(frameset, parameters,
                                        flat_set_for_header, "espdr_mflat",
                                        product_keywords[i], keywords_ext,
                                        flat_bkgr_substracted_frame_filename,
                                        final_master_background_substracted_flat,
                                        CPL_TYPE_FLOAT, CCD_geom);
        
        cpl_image_delete(extracted_spectrum_table);
        cpl_image_delete(spectrum_error_table);
        cpl_image_delete(spectrum_quality_table);
        
        cpl_image_delete(flat_one_fibre);
        cpl_image_delete(blaze_one_fibre);
        
        cpl_imagelist_delete(master_flat_iml);
        cpl_imagelist_delete(final_master_background_substracted_flat);
        cpl_imagelist_delete(final_master_background_measure_flat);
        cpl_free(cosmics_nb);
        
        for (int i = 0; i < CCD_geom->ext_nb; i++) {
            cpl_propertylist_delete(keywords_ext[i]);
        }
        cpl_free(flat_bkgr_substracted_frame_filename);
        cpl_free(keywords_ext);
        cpl_propertylist_delete(keywords);
        cpl_frameset_delete(flat_set_for_header);
        cpl_imagelist_delete(geometry_corrected_saturation_mask);
    } // end loop on fibres
    cpl_frameset_delete(calib_set);

    // Cleaning memory
    cpl_propertylist_delete(keywords_HP);
    cpl_propertylist_delete(keywords_BP);
    for (int i = 0 ; i < inst_config->fibres_nb; i++) {
        cpl_propertylist_delete(product_keywords[i]);
        cpl_free(max_flux[i]);
    }
    cpl_free(product_keywords);
    cpl_free(max_flux);
    
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            cpl_table_delete(orders_coeffs[i][j]);
        }
        cpl_free(orders_coeffs[i]);
    }
    cpl_free(orders_coeffs);
    
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_image_delete(wave_matrix[i]);
        cpl_image_delete(dll_matrix[i]);
        cpl_frameset_delete(flat_frames[i]);
    }
    cpl_free(wave_matrix);
    cpl_free(dll_matrix);
    cpl_free(qc_kws);
    cpl_imagelist_delete(bad_pixels_list);
    cpl_imagelist_delete(hot_pixels_list);
    if (master_dark_frame != NULL) {
        cpl_imagelist_delete(master_dark_iml);
        cpl_imagelist_delete(master_dark_pxl_nb_iml);
    }
    cpl_imagelist_delete(geometry_corrected_pixels_mask);
    cpl_imagelist_delete(pixels_mask);
    cpl_free(flat_frames);
    
    return (cpl_error_get_code());
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Check flat saturation
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_check_flat_saturation(cpl_frameset *flat_frames,
                                           espdr_CCD_geometry *CCD_geom,
                                           espdr_inst_config *inst_config,
                                           cpl_propertylist *keywords,
                                           cpl_imagelist *pixels_mask,
                                           double *max_flux,
                                           cpl_imagelist **sat_mask_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    /* Extracting real outputs from the first frame */
    
    cpl_frame *first_flat_frame = cpl_frameset_get_position(flat_frames, 0);
    cpl_frameset *first_flat_frameset = cpl_frameset_new();
    my_error = cpl_frameset_insert(first_flat_frameset, cpl_frame_duplicate(first_flat_frame));
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_frameset_insert failed: %s",
                 cpl_error_get_message_default(my_error));
    
    cpl_imagelist *real_first_flat_iml = cpl_imagelist_new();
    my_error = espdr_extract_real_outputs_from_raw(first_flat_frameset,
                                                   CPL_TYPE_DOUBLE,
                                                   CCD_geom,
                                                   &real_first_flat_iml);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_extract_real_outputs_from_raw failed: %s",
                 cpl_error_get_message_default(my_error));
    
    if (inst_config->inst_type == NIR) {
        char const *input_filename_flat = cpl_frame_get_filename(first_flat_frame);
        espdr_msg("First flat frame input filename: %s", input_filename_flat);
        cpl_propertylist *keywords_flat = cpl_propertylist_load(input_filename_flat, 0);
        double texp = cpl_propertylist_get_double(keywords_flat, inst_config->Texp_kw);
        //espdr_msg("--->>>>>  exposure time: %f", texp);
        my_error = cpl_imagelist_multiply_scalar(real_first_flat_iml, texp);
        cpl_propertylist_delete(keywords_flat);
    }
    
    // Patch for HARPN, between 2019-03-21 (BJD = 58564) - 2019-04-29 (BJD = 58603)
    // and 2019-08-08 (BJD = 58704) - 2020-02-19 (BJD = 58899), where FLATS are slightly saturated
    // There is no other way to get the data reduced, so we accept these slightly saturated FLATs
    double cosmics_part = inst_config->image_cosmics_part;
    
    if (strcmp(inst_config->instrument, "HARPN") == 0) {
        if (cpl_propertylist_has(keywords, "MJD-OBS")) {
            double mjd_obs = cpl_propertylist_get_double(keywords, "MJD-OBS");
            if (((mjd_obs > 58564) && (mjd_obs < 58603)) ||
                ((mjd_obs > 58704) && (mjd_obs < 58899))) {
                cosmics_part = 0.000262;
                espdr_msg_warning(ANSI_COLOR_RED"Frame slightly saturated can be accepted, cosmics part = %f, instead of %f"ANSI_COLOR_RESET,
                                  cosmics_part, inst_config->image_cosmics_part);
            }
        }
    }
    
    my_error = espdr_check_saturation(real_first_flat_iml,
                                      CCD_geom,
                                      inst_config,
                                      pixels_mask,
                                      cosmics_part,
                                      max_flux,
                                      1, // print max flux values
                                      sat_mask_RE);
    
    cpl_imagelist_delete(real_first_flat_iml);
    cpl_frameset_delete(first_flat_frameset);

    return (cpl_error_get_code());
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Get master flat
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_get_master_flat(cpl_frameset *flat_frames,
                                     cpl_imagelist *master_dark_iml,
                                     cpl_imagelist *orders_mask_iml,
                                     espdr_CCD_geometry *CCD_geom,
                                     espdr_inst_config *inst_config,
                                     cpl_propertylist *keywords,
                                     cpl_propertylist *keywords_HP,
                                     cpl_propertylist *keywords_BP,
                                     espdr_qc_keywords *qc_kws,
                                     int remove_bias_res,
                                     double *CONAD_FLAT,
                                     double *DARK_FLAT,
                                     double *DARK_RON_FLAT,
                                     double *RON_FLAT,
                                     cpl_imagelist **master_flat_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    cpl_imagelist *mflat_outputs = cpl_imagelist_new();
    cpl_image *CCD_corrected_flat = NULL;
    cpl_imagelist *corrected_imagelist = NULL;
    cpl_image *master_flat_image = NULL;
    cpl_image *orders_mask_image = NULL;
    cpl_image *output = NULL;
    cpl_frame *frame = NULL;
    const char *fname = NULL;
    int nset = cpl_frameset_get_size(flat_frames);
    int index_current_output = 0;
    int ix, iy;
    int llx, lly, urx, ury;
    int real_llx, real_lly, real_urx, real_ury;
    int used_real_nx, used_real_ny;
    cpl_image *curr_image = NULL;
    cpl_image *curr_mdark = NULL;
    double conad = 0.0;
    double dark = 0.0;
    double dark_ron = 0.0;
    double ron = 0.0;
    double RON_mean[CCD_geom->total_output_nb];
    int total_cosmics[CCD_geom->ext_nb];
    
    /* Processing on total extensions at time */
    int index_dark = 0;
    for (int e = 0; e < CCD_geom->ext_nb; e++) {
        ix = 0;
        /* Processing on total outputs x at time */
        for (int ox = 0; ox < CCD_geom->exts[e].out_nb_x; ox++) {
            llx = ix + 1;
            urx = ix+CCD_geom->exts[e].outputs[ox][0].raw_nx;
            iy = 0;
            /* Processing on total outputs y at time */
            for (int oy = 0; oy < CCD_geom->exts[e].out_nb_y; oy++) {
                int index = 0;
                
                lly = iy+1+CCD_geom->exts[e].outputs[ox][oy].ppscan_ny;
                ury = iy+CCD_geom->exts[e].outputs[ox][oy].raw_ny
                        -CCD_geom->exts[e].outputs[ox][oy].poscan_ny;
                
                corrected_imagelist = cpl_imagelist_new();
                
                if (master_dark_iml != NULL) {
                    curr_mdark = cpl_imagelist_get(master_dark_iml, index_dark);
                    index_dark++;
                } else {
                    curr_mdark = NULL;
                }
                
                /* Processing on total flats at time */
                cpl_frameset_iterator* iter = cpl_frameset_iterator_new(flat_frames);
                frame = cpl_frameset_iterator_get(iter);
                for (int iter_frame = 0; iter_frame < nset; iter_frame++ ) {
                    
                    fname = cpl_frame_get_filename(frame);
                    
                    if (inst_config->inst_type == NIR) {
                        curr_image = cpl_image_load(fname, CPL_TYPE_DOUBLE, 0, e+1);
#if SAVE_DEBUG_PRODUCT_PREPROCESSING
                        keywords = cpl_propertylist_load(fname, 0);
                        if (cpl_propertylist_has(keywords, "ARCFILE") == 0) {
                            char *arc_filename = strrchr(fname, '/')+1;
                            espdr_msg("ARCFILE not existing - updating the filename: %s", arc_filename);
                            my_error = cpl_propertylist_update_string(keywords, "ARCFILE", arc_filename);
                            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                         "Updating the ARCFILE KW (%s) failed: %s",
                                         arc_filename, cpl_error_get_message_default(my_error));
                        }
#endif
                    } else {
                        curr_image = cpl_image_load_window(fname,
                                                           CPL_TYPE_DOUBLE, 0, e+1,
                                                           llx, lly, urx, ury);
                    }
                    
                    espdr_ensure(curr_image == NULL,
                                 CPL_ERROR_ILLEGAL_OUTPUT,
                                 "Raw flat image got from the list is NULL");
                    
                    output = cpl_image_duplicate(curr_image);
                    cpl_image_delete(curr_image);
                    
                    espdr_ensure(my_error != CPL_ERROR_NONE,
                                 my_error,
                                 "cpl_imagelist_set failed: %s",
                                 cpl_error_get_message_default(my_error));
                    
                    espdr_msg("Removing the CCD signature ...");
                    
                    if (orders_mask_iml != NULL) {
                        orders_mask_image = cpl_imagelist_get(orders_mask_iml,
                                                              index_current_output);
                    } else {
                        orders_mask_image = NULL;
                    }
                    
                    my_error = espdr_remove_det_signature_one_output(output,
                                                                     orders_mask_image,
                                                                     NULL,
                                                                     curr_mdark,
                                                                     keywords,
                                                                     keywords_HP,
                                                                     keywords_BP, qc_kws,
                                                                     inst_config,
                                                                     remove_bias_res,
                                                                     e, ox, oy, CCD_geom,
                                                                     &conad,
                                                                     &dark,
                                                                     &dark_ron,
                                                                     &ron,
                                                                     &CCD_corrected_flat);
                    
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "espdr_remove_detector_signature_one_output failed: %s",
                                 cpl_error_get_message_default(my_error));
                    
                    // Image added to corrected_imagelist list for
                    // further stacking per output.
                    
                    my_error = cpl_imagelist_set(corrected_imagelist,
                                                 cpl_image_duplicate(CCD_corrected_flat),
                                                 index);
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "CCD_corrected_flat duplication into input_corrected_RE failed: %s",
                                 cpl_error_get_message_default(my_error));
                    
                    cpl_image_delete(CCD_corrected_flat);
                    cpl_image_delete(output);
                    index++;
                    cpl_frameset_iterator_advance(iter, 1);
                    frame = cpl_frameset_iterator_get(iter);
                }
                cpl_frameset_iterator_delete(iter);
                
                real_llx = CCD_geom->exts[e].outputs[ox][oy].real_llx;
                real_lly = CCD_geom->exts[e].outputs[ox][oy].real_lly;
                real_urx = CCD_geom->exts[e].outputs[ox][oy].real_urx;
                real_ury = CCD_geom->exts[e].outputs[ox][oy].real_ury;
                used_real_nx = real_urx - real_llx + 1;
                used_real_ny = real_ury - real_lly + 1;
                
                espdr_msg("Computing espdr_stack_sigma_one_output for ext %d, output [%d, %d]", e, ox, oy);
                
                // Stacking frames by output.
                my_error = espdr_stack_sigma_one_output(corrected_imagelist,
                                                        used_real_nx, used_real_ny,
                                                        inst_config->flat_ksigma,
                                                        inst_config->flat_sigma_clipping_method,
                                                        nset - 1,
                                                        &master_flat_image,
                                                        RON_mean,
                                                        total_cosmics);
                
                espdr_ensure(master_flat_image == NULL,
                             CPL_ERROR_ILLEGAL_OUTPUT,
                             "Resulting master flat image is NULL");
                
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Stack sigma on one output failed");
                
                espdr_msg("espdr_stack_sigma_one_output for ext %d, output [%d, %d] computed", e, ox, oy);

                CONAD_FLAT[index_current_output] = conad;
                DARK_FLAT[index_current_output] = dark;
                DARK_RON_FLAT[index_current_output] = dark_ron;
                RON_FLAT[index_current_output] = ron;
                
                cpl_imagelist_set(mflat_outputs,
                                  master_flat_image,
                                  index_current_output);
                
                index_current_output++;
                
                my_error = cpl_error_get_code();
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Imagelist master flat images set failed");
                
                iy = ury + CCD_geom->exts[e].outputs[ox][oy].poscan_ny;
                
                cpl_imagelist_delete(corrected_imagelist);
            } /* end loop over y read-out */
            ix = urx;
        } /* end loop over x read-out */
    } /* end loop over extensions */
    
    /* end of total outputs processing */
    //for(int d = 0; d < CCD_geom->total_output_nb; d++) {
    //espdr_msg(">>> CONAD_FLAT[%d]=%f",d,CONAD_FLAT[d]);
    //espdr_msg(">>> DARK_FLAT[%d]=%f",d,DARK_FLAT[d]);
    //espdr_msg(">>> RON_FLAT[%d]=%f",d,RON_FLAT[d]);
    //}
    
    /* Merge all master flat output images into one extension,
     all the merged extensions are in the mflat_extensions */
    
    espdr_msg("Merging the MASTER FLAT image");
    cpl_imagelist *mflat_extensions = cpl_imagelist_new();
    my_error = espdr_image_merge_2_dim(mflat_outputs,
                                       CCD_geom,
                                       CPL_TYPE_DOUBLE,
                                       &mflat_extensions);
    
    cpl_imagelist *mflat_geom_corrected = cpl_imagelist_new();
    my_error = espdr_correct_geometry(mflat_extensions,
                                      CCD_geom,
                                      mflat_geom_corrected);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_correct_geometry failed: %s",
                 cpl_error_get_message_default(my_error));
    
    cpl_imagelist_delete(mflat_outputs);
    cpl_imagelist_delete(mflat_extensions);
    
    *master_flat_RE = cpl_imagelist_duplicate(mflat_geom_corrected);
    cpl_imagelist_delete(mflat_geom_corrected);
    
    espdr_msg("Master FLAT created");
    
    return (cpl_error_get_code());
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Compute flat relative error
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_compute_flat_rel_error(cpl_image *extracted_spectrum_table,
                                            cpl_image *spectrum_error_table,
                                            cpl_image *spectrum_error_relative) {
    
    double *flux_data = cpl_image_get_data_double(extracted_spectrum_table);
    double *error_data = cpl_image_get_data_double(spectrum_error_table);
    double *rel_error_data = cpl_image_get_data_double(spectrum_error_relative);
    int nx = cpl_image_get_size_x(extracted_spectrum_table);
    int ny = cpl_image_get_size_y(extracted_spectrum_table);

    for (int i = 0; i < nx*ny; i++) {
        if (flux_data[i] > 0.0) {
            rel_error_data[i] = error_data[i] / flux_data[i];
        } else {
            rel_error_data[i] = 0.0;
        }
    }
    
    return (cpl_error_get_code());
}




/*---------------------------------------------------------------------------*/
/**
 @brief    Check the flat QC
 @param    CCD_geom         CCD geometry
 @param    exposure_time    exposure time
 @param    inst_config      instrument configuration structure
 @param    qc_kws           structure holdin quality control, keys
 @param    keywords_RE      fits header to hold QC
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_flat_QC(espdr_CCD_geometry *CCD_geom,
                             espdr_inst_config *inst_config,
                             espdr_qc_keywords *qc_kws,
                             int nset,
                             double *max_flux,
                             cpl_imagelist *bkgr_measured,
                             int *cosmics_nb,
                             int orders_nb,
                             cpl_image *extracted_spectrum_table,
                             cpl_image *spectrum_error_table,
                             cpl_image *flat_spectrum,
                             cpl_propertylist **keywords_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    int i, j, k, index;
    char *new_keyword = NULL;
    int saturation_QC = 1;
    int bkgr_QC = 1;
    int snr_QC = 1;
    int rms_QC = 1;
    int global_QC = 1;
    double bkgr_mean;
    double bkgr_min;
    double bkgr_max;
    double *max_flux_ext;
    
    
    /* max flux KWs */
    
    index = 0;
    max_flux_ext = (double *)cpl_calloc(CCD_geom->ext_nb, sizeof(double));
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                new_keyword =
                espdr_add_ext_output_index_to_keyword(qc_kws->qc_max_flux_kw,
                                                      inst_config->prefix,
                                                      i, j, k);
                
                my_error = espdr_keyword_add_double(new_keyword,
                                                    max_flux[index],
                                                    "Max flux, raw image [ADU]",
                                                    keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword QC_MAX_FLUX_KW to the propertylist failed: %s",
                             cpl_error_get_message_default(my_error));
                
                cpl_free(new_keyword);
                
                if (max_flux[index] >= inst_config->satur_limit_adu) {
                    saturation_QC = 0;
                    global_QC = 0;
                }
                
                if (max_flux[index] > max_flux_ext[i]) {
                    max_flux_ext[i] = max_flux[index];
                }
                
                index++;
            }
        }
    }
    
    /* QC saturation CHECK KWs */
    
    my_error = espdr_keyword_add_int(qc_kws->qc_saturation_check_kw,
                                     saturation_QC,
                                     "Saturation [ADU] QC",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_WAVE_SATUR_CHECK_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    
    /* Background KWs */
    
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        new_keyword =
        espdr_add_output_index_to_keyword(qc_kws->qc_flat_bkgr_mean_kw_first,
                                          qc_kws->qc_flat_bkgr_mean_kw_last,
                                          i);
        
        bkgr_mean = cpl_image_get_mean(cpl_imagelist_get_const(bkgr_measured,i));
        
        my_error = espdr_keyword_add_double(new_keyword,
                                            bkgr_mean,
                                            "Bkgr mean in the extension [e-]",
                                            keywords_RE);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_BKGR_MEAN_KW to the propertylist failed: %s",
                     cpl_error_get_message_default(my_error));
        
        if (!isnan(max_flux_ext[i]) && (max_flux_ext[i]!=0.)) {
            if (bkgr_mean/nset/max_flux_ext[i] >= inst_config->flat_bkgr_limit) {
                bkgr_QC = 0;
            }
        }
        
        cpl_free(new_keyword);
        
        new_keyword =
        espdr_add_output_index_to_keyword(qc_kws->qc_flat_bkgr_min_kw_first,
                                          qc_kws->qc_flat_bkgr_min_kw_last,
                                          i);
        
        bkgr_min = cpl_image_get_min(cpl_imagelist_get_const(bkgr_measured,i));
        
        my_error = espdr_keyword_add_double(new_keyword,
                                            bkgr_min,
                                            "Bkgr min in the extension [e-]",
                                            keywords_RE);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_BKGR_MIN_KW to the propertylist failed: %s",
                     cpl_error_get_message_default(my_error));
        
        cpl_free(new_keyword);
        
        new_keyword =
        espdr_add_output_index_to_keyword(qc_kws->qc_flat_bkgr_max_kw_first,
                                          qc_kws->qc_flat_bkgr_max_kw_last,
                                          i);
        
        bkgr_max = cpl_image_get_max(cpl_imagelist_get_const(bkgr_measured,i));
        
        my_error = espdr_keyword_add_double(new_keyword,
                                            bkgr_max,
                                            "Bkgr max in the extension [e-]",
                                            keywords_RE);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_BKGR_MAX_KW to the propertylist failed: %s",
                     cpl_error_get_message_default(my_error));
        
        cpl_free(new_keyword);
        
    }
    
    my_error = espdr_keyword_add_int(qc_kws->qc_flat_bkgr_check_kw,
                                     bkgr_QC,
                                     "Background level [e-] QC",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_FLAT_BKGR_CHECK_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    
    /* Orders KWs */
    
    for (i = 1; i <= orders_nb; i++) {
        new_keyword =
        espdr_add_output_index_to_keyword(qc_kws->qc_flat_order_cosmic_nb_kw_first,
                                          qc_kws->qc_flat_order_cosmic_nb_kw_last,
                                          i);
        
        my_error = espdr_keyword_add_int(new_keyword,
                                         cosmics_nb[i-1],
                                         "Cosmics number in the order",
                                         keywords_RE);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_ORDER_COSMIC_NB_KW to the propertylist failed: %s",
                     cpl_error_get_message_default(my_error));
        
        cpl_free(new_keyword);
    }
    
    cpl_image *extr_spec;
    cpl_image *error_spec;
    double *extr_spec_data;
    double *error_spec_data;
    double snr_order;
    double rms_order;
    double snr_orders[orders_nb];
    double rms_orders[orders_nb];
    int central_point_extr_spec;
    int central_point_error_spec;
    cpl_size spectrum_size_x = cpl_image_get_size_x(extracted_spectrum_table);
    cpl_size error_size_x = cpl_image_get_size_x(spectrum_error_table);
    central_point_extr_spec = spectrum_size_x / 2;
    central_point_error_spec = error_size_x / 2;
    
    for (i = 0; i < orders_nb; i++) {
        
        extr_spec = cpl_image_extract(extracted_spectrum_table, 1, i+1, spectrum_size_x, i+1);
        extr_spec_data = cpl_image_get_data_double(extr_spec);
        
        error_spec = cpl_image_extract(spectrum_error_table, 1, i+1, error_size_x, i+1);
        error_spec_data = cpl_image_get_data_double(error_spec);
        
        new_keyword =
        espdr_add_output_index_to_keyword(qc_kws->qc_flat_order_snr_kw_first,
                                          qc_kws->qc_flat_order_snr_kw_last,
                                          i+1);
        
        snr_order = 0.;
        
        if (error_spec_data[central_point_error_spec] != 0. &&
            extr_spec_data[central_point_extr_spec] != 0.) {
            snr_order = extr_spec_data[central_point_extr_spec]/
                        error_spec_data[central_point_error_spec];
        }
        
        snr_orders[i] = snr_order;
        
        if (snr_order <= inst_config->flat_snr_limit) {
            espdr_msg("SNR too low for order %d: %f, limit: %f",
                      i+1, snr_orders[i], inst_config->flat_snr_limit);
            snr_QC = 0;
        }
        
        my_error = espdr_keyword_add_double(new_keyword,
                                            snr_order,
                                            "SNR in the order",
                                            keywords_RE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_ORDER_SNR_KW to the propertylist failed: %s",
                     cpl_error_get_message_default(my_error));
        
        cpl_image_delete(extr_spec);
        cpl_image_delete(error_spec);
        cpl_free(new_keyword);
    }
    
    my_error = espdr_keyword_add_int(qc_kws->qc_flat_snr_check_kw,
                                     snr_QC,
                                     "SNR QC",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_FLAT_SNR_CHECK_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    cpl_image *flat_spec = NULL;
    int central_point_flat_spec;
    int window = inst_config->blaze_flat_calc_slide_box;
    central_point_flat_spec = cpl_image_get_size_x(flat_spectrum) / 2;
    
    for (i = 0; i < orders_nb; i++) {
        
        flat_spec = cpl_image_extract(flat_spectrum,
                                      central_point_flat_spec-window,
                                      i+1,
                                      central_point_flat_spec+window,
                                      i+1);
        
        new_keyword =
        espdr_add_output_index_to_keyword(qc_kws->qc_flat_order_flat_rms_kw_first,
                                          qc_kws->qc_flat_order_flat_rms_kw_last,
                                          i+1);
        
        rms_order = 0.;
        rms_order = cpl_image_get_stdev(flat_spec);
        
        my_error = espdr_keyword_add_double(new_keyword,
                                            rms_order,
                                            "FLAT RMS in the order",
                                            keywords_RE);
        
        rms_orders[i] = rms_order;
        
        if (rms_order > inst_config->flat_rms_limit) {
            espdr_msg("RMS too high for order %d: %f, limit: %f",
                      i+1, rms_orders[i], inst_config->flat_rms_limit);
            rms_QC = 0;
        }
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_ORDER_RMS_KW to the propertylist failed: %s",
                     cpl_error_get_message_default(my_error));
        
        cpl_image_delete(flat_spec);
        cpl_free(new_keyword);
    }
    
    my_error = espdr_keyword_add_int(qc_kws->qc_flat_rms_check_kw,
                                     rms_QC,
                                     "RMS QC",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_FLAT_RMS_CHECK_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    if (saturation_QC == 0) {
        global_QC = 0;
    }
    if (bkgr_QC == 0) {
        global_QC = 0;
    }
    if (snr_QC == 0) {
        global_QC = 0;
    }
    if (rms_QC == 0) {
        global_QC = 0;
    }
    
    my_error = espdr_keyword_add_int(qc_kws->qc_flat_check_kw,
                                     global_QC,
                                     "FLAT global QC",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_FLAT_CHECK_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    cpl_free(max_flux_ext);
    
    return cpl_error_get_code();
}

